/*-----------------------------------------------------------------------------*
 * UDevManager.cpp                                                             *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file UDevManager.cpp
 *
 * \brief This file holds the implementatin of class CUDevManager.
 *
 * It cares notification by underlying platform and collects
 *        further information concerning properties of a connected device
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Koechling, Christian (Bosch), changes to gain that
 *          stateTable informs clients, further cleanup
 * \version 01 11.2011, Negi, Sunder (Montavista), parts of cleanup hannded over to C.K.
 * \version 01.11.2011, Negi, Sunder (Montavista),
 *          -# SD Card detection added
 *          -# Cleanup: Redundant functions removed
 *          -# Indentation and traces modified
 * \version 04 11.2011, Koechling, Christian (Bosch), Bugfix diagnosis,add more traces
 * \version 05.11.2011, Negi, Sunder (Montavista), Multiple partition support
 *          added for DOS partition tables
 * \version 06.11.2011, Negi, Sunder (Montavista), Changes incorporating disk
 *          storage details changed to unsigned long long from GENSTRING
 * \version 09.11.2011, Negi, Sunder (Montavista), Refactored some functions,
 *          Adjusted spaces, modified traces, Added some error handling
 * \version 17.11.2011, Koechling, Christian (Bosch),
 *          -# compilabilty enable switch off log mechanism used by MV - see LogTrace.h
 *          -# cleanup: 1st step to shift includes to Common.h
 * \version 19.11.2011, Negi, Sunder (Montavista), BugFix: Wrong USB mountpoint.
 *          HUB disconnection notification on USB removal.
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 21.11.2011, Negi, Sunder (Montavista), Added code for: USB drives
 *          which get mount as /dev/sdX and don't have partitions
 * \version 24.11.2011, Negi, Sunder (Montavista), Bugfix for multiple partition- part one
 * \version 24.11.2011, Koechling, Christian (Bosch), cleanup: singelton:semaphore
 *          and private contructor
 * \version 28.11.2011, Koechling, Christian (Bosch), cleanup - add headers
 * \version 28.11.2011, Negi, Sunder (Montavista),
 *          -# SD Card Reader media detection using DevKit-disks, some error handling
 *          -# Processing of 'change' action added
 *          -# Fix for crash while using blkid apis
 * \version 29.11.2011, Negi, Sunder (Montavista), Functions NOT to have
 *          vd_devicemanager_tclApp* as parameter, but will use the member instead
 * \version 29.11.2011, Negi, Sunder (Montavista), Modified DisconnectNotification calls
 * \version 05.12.2011, Negi, Sunder (Montavista), Added code to read product
 *          name for a usb_device from usb.ids file
 * \version 08.12.2011, Koechling, Christian (Bosch), Adjusted devicename as
 *          per requirement
 * \version 09.12.2011, Negi, Sunder (Montavista), Cleanup: Removed redundant
 *          blkid calls
 * \version 10.12.2011, Negi, Sunder (Montavista),
 *          -# Modified usage of m_pDev
 *          -# Enhancement: Support for multicard readers
 *          -# Cleanup: removed calls to getproductid
 *          -# Modification to add devices without SERIALIDs
 * \version 12.12.2011, Koechling, Christian (Bosch), cleanup singleton,
 *          further care for simulating EOL-values
 * \version 13.12.2011, Negi, Sunder (Montavista), RE-Fix USB devices that mount
 *          as a disk
 * \version 15.12.2011, Negi, Sunder (Montavista), cleanup
 * \version 05.01.2012, Koechling, Christian (Bosch), bugfix GMNGA-24665
 *          plug-unplug-sequence the stick gets a new device node /dev/sda,
 *          /dev/sdb,/dev/sdc ... /dev/sdg
 * \version 06.01.2012, Koechling, Christian (Bosch), cleanup FATAL traces
 * \version 14.01.2012, Negi, Sunder (Montavista), Cleanup of memory and open
 *          file descriptors
 * \version 16.01.2012, Negi, Sunder (Montavista),
 *          -# Fix for SD cards that does not contain a valid partition table,
 *          -# possible NULLpointer block probe GMNGA-21963,GMNGA-21955
 *          -# BUGFIX GMNGA-21958: device friendly
 * \version 17.01.2012, Negi, Sunder (Montavista), BUGFIX - for recognition of
 *          Jessica's test cards
 * \version 17.01.2012, Negi, Sunder (Montavista), Added: Code for hub detection
 * \version 17.01.2012, Koechling Christian (Bosch), test feedback GMNGA-21955,
 *          GMNGA-21963 are bugfixed
 * \version 18.01.2012, Negi, Sunder (Montavista), add helper function to detect
 *          if HUB is connected
 * \version 19.01.2012, Koechling Christian (Bosch),
 *          -# use helper function to detect if HUB is connected for diagnosis
 *          -# use 'DELAY_DIGANOSISRESULT_CYCLICDIAG'  see config.h
 * \version 19.01.2012, Negi, Sunder (Montavista), BUGFIX GMNGA-21958
 * \version 19.01.2012, Negi, Sunder (Montavista), Added: Code for UTF8
 *          conversion of disk label
 * \version 20.01.2012, Negi, Sunder (Montavista), cleanup to prevent possible
 *          reset replace strcpy by strncpy
 * \version 20.01.2012, Negi, Sunder (Montavista), Try converting disk label to
 *          UTF8 string, prevent reset
 * \version 20.01.2012, Koechling, Christian (Bosch), added retrial mechanism
 *          for SD cards to check if mount table /proc/mounts contains expected
 *          mount point
 * \version 23.01.2012, Koechling, Christian (Bosch), bugfix eol - SD-card
 *          disable/enable MTP enable/disable
 * \version 23.01.2012, Negi, Sunder (Montavista), prevent possible reset in
 *          GetPartitionDetails
 * \version 24.01.2012, Negi, Sunder (Montavista), prevent possible reset in
 *          helperStoreUdevEvts
 * \version 24.01.2012, Negi, Sunder (Montavista),
 *          -# Added: Minor changes to speed up device detection, storeudevevts
 *          disabled
 *          -# Added: Clear values in GetPartitionDetails
 * \version 26.01.2012, Negi, Sunder (Montavista),
 *          -# LOOKUP Enabled for fast device detection as startup,
 *          -# Disk details fetching delayed
 * \version 30.01.2012, Koechling, Christian (Bosch), EOL/Calibration based
 *          filter fixed and generalised
 * \version 30.01.2012, Negi, Sunder (Montavista), old apple device: not
 *          notified as apple device now.
 * \version 07.02.2012, Koechling, Christian (Bosch), Bugfix for USB2-present=0,
 *          maxports=7 and HUB_USB2_ENABLE = FALSE
 * \version 16.02.2012, Negi, Sunder (Montavista),
 *          -# BugFix: BUGFIX_LEXAR_DRIVE : Lexar drive not getting notified due
 *          to error while performing an MTP check on it
 *          -# Enhancement: USE_INOTIFY: Preparation for newer udev scripts
 *          -# Enhancement: EOL handling for MTP detection
 *          -# Cleanup: Removed some class members, renamed functions
 * \version 22.02.2012, Negi, Sunder (Montavista),
 *          -# BUGFIX: IF INOTIFY mechanism is used there was a bug concerning
 *          recognition of Apple devices
 *          -# Cleanup: Few macros cleaned up, unused code removed
 * \version 25.02.2012, Negi, Sunder (Montavista), Enhancement: Added code to
 *          list connected devices whether initialized or uninitialized.
 * \version 27.02.2012, Negi, Sunder (Montavista), Bugfix: iPod with empty
 *          mountpoint should not be notified.
 * \version 23.02.2012, Koechling, Christian (Bosch), add TTFIS command
 *          VD_DVM_GET_CONNETED_HUBS
 * \version 27.02.2012, Koechling, Christian (Bosch), cleanup fatal trace at startup
 * \version 29.02.2012, Koechling, Christian (Bosch), ENABLED NOTIFY : BUGFIX at
 *          STARTUP - see BUGFIX_STARTUPRECOGNITION_INOTIFY
 * \version 05.03.2012, Koechling, Christian (Bosch), cleanup defines; mechanism
 *          added to reach that device notification is not interrupted by cyclic
 *          diag
 * \version 07.03.2012, Koechling, Christian (Bosch), observe if there more than
 *          one inotify elements readable - after discussion with developer to
 *          prepare further improvement
 * \version 07.03.2012, Koechling, Christian (Bosch), merged MV-code Feb27 -
 *          contains issue for startup synchronisation
 * \version 10.03.2012, Koechling, Christian (Bosch), cleanup DELAY_STORAGEINFO
 *          -see config.h; cleanup history functions for analysis
 * \version 14.03.2012, Koechling, Christian (Bosch), cleanup USE_BUGFIX_HUB
 * \version 26.03.2012, Negi, Sunder (Montavista), modified the
 *          GetPartitionDetails() function to care for Latin device labels
 * \version 02.04.2012, Negi, Sunder (Montavista), IPOD recognition - retrial
 *          mechanism added see USE_IPOD_RECOGNITION_RETRIAL (c.k. added 200ms additional delay)
 * \version 02.04.2012, Negi, Sunder (Montavista), observer mechanism added -
 *          see USE_OBSERVER_AUTOMOUNTUSE_OBSERVER_AUTOMOUNT
 * \version 02.06.2012, Koechling, Christian (Bosch),
 *          -# added mechanisms to use central device list manager,
 *          -# improved critical voltage behaviour and
 *          -# state machine to send combined messages for output of the first
 *          state machine
 * \version 19.07.2012, Koechling, Christian (Bosch), add mechanism to log
 *          histories for analysis to USB-stick
 * \version 25.07.2012, Koechling, Christian (Bosch), rebase on Hotfix 51 - see
 *          Doc/Readme.txt
 * \version 25.07.2012, Negi, Sunder (Montavista), bugfix MTP crash, mtp check
 *          is not done on storage, apple and hub devices
 * \version 01.08.2012, Koechling, Christian (Bosch), shifted function header
 *          to *.h-file in doxygen style
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 08.08,2012, Negi, Sunder (MontaVista), fix for Sansa Fuze getting detected as DTY_USB
 * \version 09.08.2012, Negi, Sunder (MontaVista), usb and port no are made integer
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 11.08.2012, Negi, Sunder (MontaVista), general function moved to new
 *          location, see CLEANUP_UDEVMANAGER (utils.h)
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 * \version 14.08.2012, Negi, Sunder (MontaVista), updates for single device store
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup CLEANUP_UDEVMANAGER,
 *          HANDLE_GPT, CHECK_FOR_SERIALID, FOR_HASH
 * \version 24.08.2012, Negi, Sunder (MontaVista),
 *          -# added changes as per the newer libusb wrppers \see usbutils.h
 *          -# cleaned up as per updated \ref CDevice class
 *          -# cleanup USE_CHARGE_APPLE
 * \version 15.08.2012, Fiebing, Lars (Bosch), adjusted buffer from 256 to 2048 in CheckIfMounted
 *          to prevent crash in LSIM. @todo another change is info will be given to Montavista
 *          ,cleanup  register for notification of platform
 * \version 27.08.2012, Koechling, Christian (Bosch),
 *          -# changes to switch on/off extra debug info using USB-stick with special Key
 *          -# trigger FillStorage fct. by PDT-diag
 * \version 31.08.2012, Negi, Sunder (MontaVista), Fix for the issue when MTP
 *          device generates two events; udev and inotify(MTP_TWO_EVENTS)
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */



/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include <stdio.h>
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include <fstream>
#include <iostream>

#include "Enums.h"
#include "ETGTrace.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "IDeviceRecognizer.h" //see INTERMEDIATEDEFINE_REFACTORUDEV
#include "StateTable.h"
#include "UDevManager.h"
#include "DeviceManager_ErrorCodes.h"

#include "device/DeviceListManager.h"
#include "debug/HistoryManager.h"
#include "interface/DeviceManagerInterface.h"
#include "config/ConfigurationManager.h"

#include "common/utils.h"
#include "usbutils.h"
#include "StorageInfoProvider.h"


#include <sys/mount.h>
#include <sys/statfs.h>

#include <fcntl.h>
#include <unistd.h>

#include <errno.h>
#include <blkid/blkid.h>
#include <ftw.h>

#include <sys/inotify.h>
#include <dirent.h>
#include "Utils.h"

#ifndef _VD_DEVICEMANAGER_ROOTDAEMON_CLIENT_H_
#include "VD_DeviceManager_rootdaemon_client.h"
#endif
/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/UDevManager.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define INO_EVT            struct inotify_event
#define INOTIFY_MAXEVENTS  10 //space for 10 events
#define INOTIFY_SIZEOFBUF  ( (sizeof(INO_EVT) + NAME_MAX) * INOTIFY_MAXEVENTS)

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
#define NOTFOUND    -1
#define FOUND        0
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES

#define MAXSIZESTRING              1024
//@todo locate at another position
/*-----------------------------------------------------------------------------*
 * Enums                                                                     *
 *-----------------------------------------------------------------------------*/
/* THIS IS FOUND on TARGET at /tmp/.mountDB for Mass storage devices e.g. in sda1*/
/*typedef enum MOUNT_DATABASE_LINES_E
{
       LINE0_MASSSTORAGE_DEVICE             = 0,
       LINE1_MASSSTORAGE_MOUNTPOINT         = 1,
       LINE2_MASSSTORAGE_IDVENDOR           = 2,
       LINE3_MASSSTORAGE_SHORTSERIALNUMBER  = 3,
       LINE4_MASSSTORAGE_MANUFACTORER       = 4,
       LINE5_MASSSTORAGE_FSTYPE             = 5,
       LINE6_MASSSTORAGE_UUID               = 6,
       LINE7_MASSSTORAGE_LABEL              = 7,
       LINE8_MASSSTORAGE_MODEL              = 8,
       LINE9_MASSSTORAGE_PRODUCT            = 9,
       LINE10_MASSSTORAGE_VERSION           = 10,
       LINE_MASSSTORAGE_LASTELEM            = 11
}MOUNTDB_Lines;*/
#define LINE1_MASSSTORAGE_MOUNTPOINT 1    // temp. use till MOUNTDB_Lines could be activated
#define LINE_MASSSTORAGE_LASTELEM    11   // temp. use till MOUNTDB_Lines could be activated


/* THIS IS FOUND on TARGET at /tmp/.appledev e.g.  in  devicespci00000000000013.1usb11-11-12*/
/*typedef enum MOUNT_APPLEDEV_LINES_E
{
        LINE0_APPLEDEV_DEVICE        = 0,
        LINE1_APPLEDEV_SOUNDDEVNAME  = 1,
        LINE2_APPLEDEV_SOUNDDEVPATH  = 2,
        LINE3_APPLEDEV_MODEL         = 3,
        LINE4_APPLEDEV_MODEL_ID      = 4,
        LINE5_APPLEDEV_VENDOR        = 5,
        LINE6_APPLEDEV_VENDOR_ID     = 6,
        LINE7_APPLEDEV_SERIAL        = 7,
        LINE8_APPLEDEV_SERIAL_SHORT  = 8,
        LINE9_APPLEDEV_HIDDEVNAME    = 9,
        LINE10_APPLEDEV_HIDDEVPATH   = 10,
        LINE11_APPLEDEV_VERSION      = 11,
        LINE_APPLEDEV_LASTELEM       = 12
}APPLEDEV_Lines;*/
#define LINE_APPLEDEV_LASTELEM       12   // temp. use till APPLEDEV_Lines could be activated


/* THIS IS FOUND on TARGET at /tmp/.unsupported e.g.  in  sdb1*/
typedef enum MOUNT_UNSUPPORTED_LINES_E
{
    LINE0_UNSUPPORTEDDEV_DEVICE          = 0,
    LINE1_UNSUPPORTEDDEV_USB_CONNECTOR   = 1,
    LINE2_UNSUPPORTEDDEV_HUB_CONNECTED   = 2,
    LINE3_UNSUPPORTEDDEV_HUB_PORTNUM     = 3,
    LINE_UNSUPPORTEDDEV_LASTELEM         = 4
    //@tbd
}UNSUPPORTED_Lines;

#define PIPE_UNDEF -1

/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
DVMLOCK        CUDevManager::m_singelton;
CUDevManager  *CUDevManager::m_pUDevManager = NULL; // static pointer used to ensure a single instance of CUDevManager

const uint16_t AppleVendorID = 0x05ac;


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace udevmgr;
using namespace std;
using namespace statetbl;
using namespace configmgr;


/*-----------------------------------------------------------------------------*
 * Variables                                                                   *
 *-----------------------------------------------------------------------------*/
char*       strMountDBLine[LINE_MASSSTORAGE_LASTELEM];
const char* strMountDBLineKeyWord[LINE_MASSSTORAGE_LASTELEM] = {"device node ",
                                                                "mountpoint  ",
                                                                "idvendor    ",
                                                                "serial      ",
                                                                "manufacturer",
                                                                "fstype      ",
                                                                "uuid        ",
                                                                "label       ",
                                                                "model       ",
                                                                "product     ",
                                                                "version     "};
// for DTY_IPOD, DTY_IPAD, DTY_IPHONE
char*       strAppleDevLine[LINE_APPLEDEV_LASTELEM];
const char* strAppleDevLineKeyWord[LINE_APPLEDEV_LASTELEM] = {"device node  ",
                                                              "sounddevname ",
                                                              "sounddevpath ",
                                                              "model        ",
                                                              "model_id     ",
                                                              "vendor       ",
                                                              "vendor_id    ",
                                                              "serial       ",
                                                              "serial_short ",
                                                              "hiddevname   ",
                                                              "hiddevpath   ",
                                                              "version      "};

/*for unsupported devices*/
char*       strMountUnsupportedLine[LINE_UNSUPPORTEDDEV_LASTELEM];
const char* strMountUnsupportedLineKeyWord[LINE_UNSUPPORTEDDEV_LASTELEM] = {"device node  ",
                                                                            "usb_connector",
                                                                            "hub_connected",
                                                                            "hub_portnum"};

// Index LAST_ELEMS via NOTIFICATION_Type (-> Enums.h)             enInotify,   enInMscDev,                        enInUnsupDev,                        enInAppleDev,                  enInMtpDev,      enUnotify, enNotifyAll
tUInt        LINE_LAST_ELEMS[CGlobalEnumerations::enNotifyAll] = { 0,           (tUInt)LINE_MASSSTORAGE_LASTELEM,  (tUInt)LINE_UNSUPPORTEDDEV_LASTELEM, (tUInt)LINE_APPLEDEV_LASTELEM, 0,               0         };
char**       allLineDescript[CGlobalEnumerations::enNotifyAll] = { NULL,        strMountDBLine,                    strMountUnsupportedLine,             strAppleDevLine,               NULL,            NULL      };
const char*  INotWatch[CGlobalEnumerations::enNotifyAll]       = { INOTIFY_DIR, INOTIFY_MSC_DIR,                   INOTIFY_UNSUP_DIR,                   INOTIFY_APPLE_DIR,             INOTIFY_MTP_DIR, NULL      };
const char** allLineKeyWord[CGlobalEnumerations::enNotifyAll]  = { NULL,        strMountDBLineKeyWord,             strMountUnsupportedLineKeyWord,      strAppleDevLineKeyWord,        NULL,            NULL      };

//same like in Usbutilts
const uint16_t DigitalAuxProductID1 = 0x0101;
const uint16_t DigitalAuxProductID2 = 0x0122;
const uint16_t DigitalAuxProductID3 = 0x0112;

/*-----------------------------------------------------------------------------*
 * Parameterized constructor                                                   *
 *-----------------------------------------------------------------------------*/
//see INTERMEDIATEDEFINE_REFACTORUDEV
CUDevManager::CUDevManager(IDeviceRecognizer *pIDeviceRecognizer, IAutomounterAdapter *pIAutomounterAdapter)
{
    m_cAction          = "";
    m_pUdev            = NULL;
    m_pDev             = NULL;
    m_pBaseUSBDev      = NULL;
    m_pParentHUBDev    = NULL;
    m_pUdevMonitor     = NULL;
    m_bIsActive        = false;
    m_cParentUsb1Hub   = "";
    m_cParentHub1Node  = "";
    m_cParentUsb2Hub   = "";
    m_cParentHub2Node  = "";
    m_iUsb1HubPort     = 0;
    m_iUsb2HubPort     = 0;
    m_iUDevMonitorFD   = -1;

    m_iPipeFD[enPipeReader]          = PIPE_UNDEF; //RoadMap 14002 CDROMMounter (MixedModeDiag)
    m_iPipeFD[enPipeWriter]          = PIPE_UNDEF; //RoadMap 14002 CDROMMounter (MixedModeDiag)

    m_iAutomounterLibFD   = -1;

    m_iInotifyFD          = -1;
    m_iInotifyWatchFD     = -1;      // /tmp/.automount
    m_iInotifyWDMscDev    = -1;      // /tmp/.mountdb
    m_iInotifyWDUnsupDev  = -1;      // /tmp/.unsupported
    m_iInotifyWDAppleDev  = -1;      // /tmp/.appledev
    m_InotifyWDMtpDev     = -1;      // /tmp/.mtpdev

    m_cParentHub1Port  = 0;
    m_cParentHub2Port  = 0;

    m_cParentHubSerial = "";
    m_bConnectUDEV     = TRUE;

    m_pStateTable  = StateTable::GetInstance(); //@todo change this to be used within code

    m_pIDeviceRecognizer   = pIDeviceRecognizer; //INTERMEDIATEDEFINE_REFACTORUDEV
    m_pIAutomounterAdapter = pIAutomounterAdapter; //If UdevManager is no longer a singleton then this can be a reference

    m_pStorageInfoProvider = NULL;

    m_iNumOfDVDDriveNotification = 0;

    DEVICEPROCESSORHISTORY.ClearHistory();

#ifdef CHECK_LAST_DEVICE
    m_vSeenDevices.clear();
#endif // CHECK_LAST_DEVICE

    // initialize libusb
    LibUSBInit();

    SetIsDVDDriveMassStorageMode(false); // Default is in Backend mode(false)
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CUDevManager::~CUDevManager()
{
    // close libusb
    LibUSBExit();             //lint !e1551 Warning 1551;Function may throw exception

    // unref monitor
    if (NULL != m_pUdevMonitor)
    {
        udev_monitor_unref(m_pUdevMonitor);
        m_pUdevMonitor = NULL;
    }

    if (NULL != m_pUdev)
    {
        udev_unref(m_pUdev);
        m_pUdev = NULL;
    }

    if (-1 != m_iInotifyWatchFD)
    {
        inotify_rm_watch (m_iInotifyFD, m_iInotifyWatchFD);
    }
    m_iInotifyWatchFD = -1;

    if( m_iInotifyWDUnsupDev != -1 )
    {
        inotify_rm_watch( m_iInotifyFD, m_iInotifyWDUnsupDev );
    }
    m_iInotifyWDUnsupDev = -1;

    if( m_iInotifyWDAppleDev != -1 )
    {
        inotify_rm_watch( m_iInotifyFD, m_iInotifyWDAppleDev );
    }
    m_iInotifyWDAppleDev = -1;

    if( m_InotifyWDMtpDev != -1 )
    {
        inotify_rm_watch( m_iInotifyFD, m_InotifyWDMtpDev );
    }
    m_InotifyWDMtpDev = -1;

    if (-1 != m_iInotifyFD)
    {
        close (m_iInotifyFD);
    }

    if ((PIPE_UNDEF != m_iPipeFD[enPipeReader]))    //RoadMap 14002 CDROMMounter (MixedModeDiag)
    {
        close(m_iPipeFD[enPipeReader]);
        m_iPipeFD[enPipeReader] = PIPE_UNDEF;
    }
    if ((PIPE_UNDEF != m_iPipeFD[enPipeWriter]))    //RoadMap 14002 CDROMMounter (MixedModeDiag)
    {
        close(m_iPipeFD[enPipeWriter]);
        m_iPipeFD[enPipeWriter] = PIPE_UNDEF;
    }

    m_iInotifyFD = -1;


    unsigned int  l_uiVectorSize;        // stores vector size
    unsigned int  l_uiCount;             // loop control variable

    try
    {
        DeviceListManager::GetInstance()->deleteList(DEVLIST_UDEVMGR);
    }
    catch (const std::bad_alloc& badAllocException)
    {
        ETG_TRACE_ERR(("CUDevManager:Failed to delete UDEV Device List - Exception caught:%s", badAllocException.what()));
    }

    // Clear hub devices
    CGlobalEnumerations::HubDetails_t *l_HubDevice = NULL;
    l_uiVectorSize = m_vHubs.size();         //lint !e1551 Warning 1551;Function may throw exception

    for (l_uiCount = 0; l_uiCount < l_uiVectorSize; l_uiCount++)
    {
        l_HubDevice = m_vHubs[l_uiCount];    //lint !e1551 Warning 1551;Function may throw exception
        DEL_M(l_HubDevice);                  //lint !e1551 Warning 1551;Function may throw exception
    }

    m_vHubs.clear();                         //lint !e1551 Warning 1551;Function may throw exception

#ifdef INCOMPLETE_DEV
    m_vIncompleteDevices.clear();            //lint !e1551 Warning 1551;Function may throw exception
#endif // INCOMPLETE_DEV

    if( m_pIDeviceRecognizer->m_pMTPRawdevices )
    {
        free( m_pIDeviceRecognizer->m_pMTPRawdevices );
        m_pIDeviceRecognizer->m_pMTPRawdevices = NULL;
    }
    if(m_pIDeviceRecognizer)
    {
        m_pIDeviceRecognizer = NULL;
    }
    m_pStateTable = NULL;

    if(NULL != m_pDev)  // Free the udev device
    {
        udev_device_unref (m_pDev);
        m_pDev = NULL;
    }
    m_pBaseUSBDev = NULL;
    m_pParentHUBDev    = NULL;
    m_pStorageInfoProvider = NULL;

    ETG_TRACE_COMP(("~CUDevManager:APPLY UNLOCK"));
    m_tableLock.unlock();                       //lint !e1551 Warning 1551;Function may throw exception
    ETG_TRACE_COMP(("~CUDevManager:UNLOCK"));
}

/*-----------------------------------------------------------------------------*
 * CUDevManager* GetInstance(StateTable *f_pStateTable,                        *
 *          vd_devicemanager_tclApp *pMainApp)                                 *
 *-----------------------------------------------------------------------------*/
CUDevManager *CUDevManager::GetInstance()
{
    ETG_TRACE_USR4(("Begin: GetInstance"));

    CUDevManager::m_singelton.lock();
    if(NULL == m_pUDevManager)
    {
        m_pUDevManager = new CUDevManager();
    }
    CUDevManager::m_singelton.unlock();

    ETG_TRACE_USR4(("End  : GetInstance"));

    return m_pUDevManager;
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance()                                                      *
 *-----------------------------------------------------------------------------*/
void CUDevManager::DestroyInstance()
{
    ETG_TRACE_USR4(("Begin: DestroyInstance"));

    CUDevManager::m_singelton.lock();
    DEL_M(m_pUDevManager);
    CUDevManager::m_singelton.unlock();

    ETG_TRACE_USR4(("End  : DestroyInstance"));
}

/*-----------------------------------------------------------------------------*
 * int InitializeUDevMonitor()                                                 *
 *-----------------------------------------------------------------------------*/
int CUDevManager::InitializeUDevMonitor()
{
    ETG_TRACE_USR4(("Begin: InitializeUDevMonitor"));

    int l_iInitSuccess = DEVICEMANAGER_OK;

    /* Create a new Udev Context*/
    m_pUdev = udev_new();

    if(NULL == m_pUdev)
    {
        ETG_TRACE_ERR(("InitializeUDevMonitor: [ERROR] udev_new failed"));

        l_iInitSuccess = DEVICEMANAGER_ERROR_MEM_ALLOC;
    }
    else
    {
        ETG_TRACE_USR4(("InitializeUDevMonitor: udev context created"));
        struct udev_queue * queue = udev_queue_new (this->m_pUdev);

        if (NULL == queue)
        {
            ETG_TRACE_ERR (("InitializeUDevMonitor: [ERROR] udev_queue_new failed"));

            return DEVICEMANAGER_ERROR_MEM_ALLOC;
        }
        else
        {
            ETG_TRACE_USR4(("InitializeUDevMonitor: Queue creation successful"));

            while (!udev_queue_get_udev_is_active (queue))
            {
                usleep((useconds_t)(USE_STARTUP_SYNC_UDEV*1000));
                ETG_TRACE_USR4(("InitializeUDevMonitor: Waiting for Udev to become active..."));
            }
#ifdef BUGFIX_UDEV_SETTLE
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_UDEVADM, "settle");
#endif// BUGFIX_UDEV_SETTLE

#ifdef BUGFIX_EXTRA_STARTUPDELAY
            usleep((useconds_t)(1000*BUGFIX_EXTRA_STARTUPDELAY));
            ETG_TRACE_COMP(("InitializeUDevMonitor: extra startup delay: %d ms",BUGFIX_EXTRA_STARTUPDELAY));
#endif

            ETG_TRACE_USR4 (("InitializeUDevMonitor: Udev has started"));

            // Free the udev queue
            udev_queue_unref (queue);
        }
    }

    ETG_TRACE_USR4(("End  : InitializeUDevMonitor"));

    return l_iInitSuccess;
}

/*-----------------------------------------------------------------------------*
 * int InitializeInotifyMonitor()                                              *
 *-----------------------------------------------------------------------------*/
int CUDevManager::InitializeInotifyMonitor() const
{
    ETG_TRACE_USR4(("Begin: InitializeInotifyMonitor"));

    int err = 0;
    int l_iInitSuccess = DEVICEMANAGER_OK;
    int rc;

    for( int i=0; INotWatch[i] != NULL; ++i )
    {
        rc  = mkdir( INotWatch[i], (S_IRWXU | S_IRWXG | S_IRWXO) );
        if( rc == -1 )
        {
            err = errno;
            if( err != EEXIST )
            {
                //if it fails then platform has already created directory INotWatch[i] /tmp/.automount, /tmp/.unsupoorted, ...
                // that is ok - this function exists if VD_DVM would be executed faster then the platform would
                //establish its directory /tmp/.automount, /btmp/.unsupported - i.e. this function exists for
                //synchronisation
                ETG_TRACE_FATAL(( "InitializeInotifyMonitor: [FATAL] 'mkdir' failed: %s", strerror( err ) ));
                ETG_TRACE_FATAL(( "InitializeInotifyMonitor: [FATAL] 'mkdir' failed for dir: %s", INotWatch[i] ));
                l_iInitSuccess = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                ETG_TRACE_USR1(( "InitializeInotifyMonitor: setup success for dir: %s", INotWatch[i] ));
            }
        }
    }

    ETG_TRACE_USR4(( "End  : InitializeInotifyMonitor" ));

    return l_iInitSuccess;
}

/*-----------------------------------------------------------------------------*
 * int CreateUdevMonitor()                                                     *
 *-----------------------------------------------------------------------------*/
int CUDevManager::CreateUdevMonitor()
{
    ETG_TRACE_USR4(("Begin: CreateUdevMonitor"));

    int l_iCreateMonitorStatus = DEVICEMANAGER_OK;

    int l_iCreateMonitorStatus_usb     = DEVICEMANAGER_OK;
    int l_iCreateMonitorStatus_sound   = DEVICEMANAGER_OK;
    int l_iCreateMonitorStatus_block   = DEVICEMANAGER_OK;

    //Set up a monitor to monitor block devices
    m_pUdevMonitor = udev_monitor_new_from_netlink(m_pUdev, "udev");

    if (NULL == m_pUdevMonitor)
    {
        ETG_TRACE_ERR(("CreateUdevMonitor: [ERROR] udev_monitor_new_from_netlink failed"));

        l_iCreateMonitorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
    }
    else
    {

        l_iCreateMonitorStatus_usb   = udev_monitor_filter_add_match_subsystem_devtype(m_pUdevMonitor, "usb"  , NULL);
        l_iCreateMonitorStatus_sound = udev_monitor_filter_add_match_subsystem_devtype(m_pUdevMonitor, "sound", NULL);

        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
        {
            l_iCreateMonitorStatus_block  = udev_monitor_filter_add_match_subsystem_devtype(m_pUdevMonitor, "block" , NULL);
        }

        ETG_TRACE_COMP(("CreateUdevMonitor: l_iCreateMonitorStatus_usb   :%d",l_iCreateMonitorStatus_usb));
        ETG_TRACE_COMP(("CreateUdevMonitor: l_iCreateMonitorStatus_sound :%d",l_iCreateMonitorStatus_sound));
        ETG_TRACE_COMP(("CreateUdevMonitor: l_iCreateMonitorStatus_block  :%d",l_iCreateMonitorStatus_block));

        if (   (l_iCreateMonitorStatus_usb   < 0)
        #ifdef INCOMPLETE_DEV
            || (l_iCreateMonitorStatus_sound < 0)
        #endif // INCOMPLETE_DEV
            || (l_iCreateMonitorStatus_block  < 0)
           )
        {
            ETG_TRACE_ERR(("CreateUdevMonitor: [ERROR] udev_monitor_filter_add_match_subsystem_devtype failed"));

            l_iCreateMonitorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        else
        {
            if (udev_monitor_enable_receiving(m_pUdevMonitor) < 0)
            {
                ETG_TRACE_ERR(("CreateUdevMonitor: [ERROR] udev_monitor_enable_receiving failed"));

                l_iCreateMonitorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                /* Get the file descriptor (m_iUDevMonitorFD) for the monitor.
                 * This fd will get passed to select()
                 */
                m_iUDevMonitorFD = udev_monitor_get_fd(m_pUdevMonitor);

                ETG_TRACE_USR3(("CreateUdevMonitor: m_iUDevMonitorFD = %d  ", m_iUDevMonitorFD));
            } // if (udev_monitor_enable_receiving succeed)
        } // if (udev_monitor_filter_add_match_subsystem_devtype succeed)
    }// if (m_pUdevMonitor is not NULL)

    ETG_TRACE_USR4(("End  : CreateUdevMonitor"));

    return l_iCreateMonitorStatus;
}

/*-----------------------------------------------------------------------------*
 * int CreateInotifyMonitor()                                                  *
 *-----------------------------------------------------------------------------*/
int CUDevManager::CreateInotifyMonitor()
{
    ETG_TRACE_USR4 (("Begin : CreateInotifyMonitor"));

    int err = 0;
    int l_iError = DEVICEMANAGER_OK;

    // Initialize inotify
    m_iInotifyFD = inotify_init ();

    ETG_TRACE_COMP (("CreateInotifyMonitor: inotify instance created = %d", m_iInotifyFD));


    if (-1 == m_iInotifyFD)
    {
        err = errno;

        ETG_TRACE_ERR (("CreateInotifyMonitor: [ERROR] 'inotify_init' failed: %s", strerror (err)));

        l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
    }
    else
    {
        ETG_TRACE_COMP (("CreateInotifyMonitor: inotify instance created"));

        // Add path to be watched for events
        m_iInotifyWatchFD = inotify_add_watch (m_iInotifyFD, WATCH, IN_DELETE|IN_CREATE|IN_CLOSE_WRITE);
        ETG_TRACE_USR1(("CreateInotifyMonitor: watch: IN_DELETE | IN_CREATE | IN_CLOSE_WRITE: 0x%x | 0x%x | 0x%x",IN_DELETE,IN_CREATE,IN_CLOSE_WRITE));


        if (-1 == m_iInotifyWatchFD)
        {
            err = errno;

            ETG_TRACE_ERR (("CreateInotifyMonitor: [ERROR] 'inotify_add_watch' failed: %s", strerror (err)));

            l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        else
        {
            ETG_TRACE_COMP (("CreateInotifyMonitor: inotify watch added successfully = %d", m_iInotifyWatchFD));
        }

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        m_iInotifyWDUnsupDev = inotify_add_watch( m_iInotifyFD, WATCH_UNSUPPORTED, IN_DELETE|IN_CREATE|IN_CLOSE_WRITE );
        if( m_iInotifyWDUnsupDev == -1 )
        {
            err = errno;
            ETG_TRACE_ERR(( "CreateInotifyMonitor: [ERROR] 'm_iInotifyWDUnsupDev' failed: %s", strerror( err ) ));
            l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        else
        {
            ETG_TRACE_COMP(( "CreateInotifyMonitor: inotify watch added successfully = %d", m_iInotifyWDUnsupDev ));
        }
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
    }

    ETG_TRACE_USR4 (("End  : CreateInotifyMonitor"));

    // return the inotify descriptor
    return l_iError;
}

/*-----------------------------------------------------------------------------*
 * int CreateSendMountMonitor()                                                  *
 *-----------------------------------------------------------------------------*/
int CUDevManager::CreatePipeMonitor() //Roadmap 14002 MountCDROM (MixedModeDiag)
{
    ETG_TRACE_USR4 (("Begin : CreatePipeMonitor"));

    int l_iError = DEVICEMANAGER_OK;

    //create pipe
    if(pipe2(m_iPipeFD,O_NONBLOCK) == -1) //lint !e1055 !e746 Symbol undeclared although it is declared in unistd.h
    {
        ETG_TRACE_FATAL(("CreatePipeMonitor: nok: could not create pipe"));
        DVM_NORMAL_M_ASSERT_ALWAYS();
        l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
    }
    else
    {
        ETG_TRACE_COMP(("CreatePipeMonitor: ok: created pipe"));
    }

    ETG_TRACE_USR4 (("End   : CreatePipeMonitor"));

    return l_iError;
}



#ifdef USE_LOOKUP
/*-----------------------------------------------------------------------------*
 * int LookupConnectedDevices()                                                *
 *-----------------------------------------------------------------------------*/
int CUDevManager::LookupConnectedDevices()
{
    ETG_TRACE_USR4(("Begin: LookupConnectedDevices"));

    //ListConnectedDevices ();
    LibUSBShowAllDevices ();

    int err = 0;
    const char *l_pccPath = NULL;   // /sys path for the device

    struct udev_enumerate  *l_pUdevEnumerate   = udev_enumerate_new(m_pUdev);
    struct udev_list_entry *l_pDevices;
    struct udev_list_entry *l_pDeviceListEntry;  // Init (NULL) isn't necessary; Pointer will be set in udev_list_entry_foreach(...)

    if (NULL == l_pUdevEnumerate)
    {
        err = errno;

        ETG_TRACE_FATAL(("LookupConnectedDevices: [ERROR] 'udev_enumerate_new' failed: %s", strerror (err)));

        return DEVICEMANAGER_ERROR_MEM_ALLOC;
    }

    if (udev_enumerate_add_match_subsystem(l_pUdevEnumerate, "usb") < 0)
    {
        err = errno;
        udev_enumerate_unref(l_pUdevEnumerate);

        ETG_TRACE_ERR (("LookupConnectedDevices: [ERROR] Adding usb subsystem for enumeration: %s", strerror (err)));

        return DEVICEMANAGER_ERROR_UNKNOWN;
    }

    if (udev_enumerate_scan_devices(l_pUdevEnumerate) < 0)
    {
        err = errno;
        udev_enumerate_unref(l_pUdevEnumerate);

        ETG_TRACE_USR2(("LookupConnectedDevices: [ERROR] In scanning devices: %s", strerror (err)));

        return DEVICEMANAGER_ERROR_UNKNOWN;
    }

    l_pDevices = udev_enumerate_get_list_entry(l_pUdevEnumerate);

    udev_list_entry_foreach(l_pDeviceListEntry, l_pDevices)
    {
        /* Get the filename of the /sys entry for the device
         * and create a udev_device object (m_pDev) representing it
         */
        l_pccPath = udev_list_entry_get_name(l_pDeviceListEntry);
        m_pDev = udev_device_new_from_syspath(m_pUdev, l_pccPath);

        // Check for valid dev node
        GENSTRING l_cDevNode = GENSTRING(udev_device_get_devnode(m_pDev));
        GENSTRING l_cSysName = GENSTRING(udev_device_get_sysname(m_pDev));
        GENSTRING l_cDevType = GENSTRING(udev_device_get_devtype(m_pDev)); //Note this returns NULL sometimes then l_cDevType will be empty

        if (     GENSTRING("usb_device") != l_cDevType
                 || true == l_cDevNode.contains("/dev/loop")
                 || true == l_cDevNode.contains("/dev/ram")
                 || true == l_cDevNode.contains("/mmcb")
                 || true == l_cDevNode.contains("/mixer")
                 || true == l_cDevNode.contains("/audio")
                 || true == l_cDevNode.contains("/pcm")
                 || true == l_cDevNode.contains("dsp")
                 || true == l_cSysName.contains("usb"))
        {
            ETG_TRACE_USR1 (("LookupConnectedDevices: Not VALID Device : %s",
                             l_cDevNode.toStdString().c_str()));
        }
        else
        {
            usleep ((useconds_t)(100*1000));

            if ( TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) )
            {
                ETG_TRACE_USR1 (("LookupConnectedDevices: Device already connected : %s",
                                 l_cDevNode.toStdString().c_str()));
            }
            else
            {
                ETG_TRACE_USR1 (("LookupConnectedDevices: New device connected : %s",
                                 l_cDevNode.toStdString().c_str()));

                // Process the new device
                if (DEVICEMANAGER_DEVICE_DETECTED == AddDevice())
                {
                    ETG_TRACE_COMP (("LookupConnectedDevices: New device added"));
                }
#ifdef CHECK_LAST_DEVICE
                struct udev_list_entry *l_pListEntry; // Init (NULL) isn't necessary; Pointer will be set in udev_list_entry_foreach(...)

                udev_list_entry_foreach(l_pListEntry, udev_device_get_properties_list_entry(m_pDev))
                {
                    //NULL check for udev_list_entry_get_name(l_pListEntry)
                    if(NULL != udev_list_entry_get_name(l_pListEntry))
                    {
                        if(!strcmp(udev_list_entry_get_name(l_pListEntry),"ID_SERIAL"))
                        {
                            this->m_vSeenDevices.push_back(GENSTRING (udev_list_entry_get_value(l_pListEntry)));
                            break;
                        }
                    }
                }

#endif // CHECK_LAST_DEVICE
            } // if (IsAlreadyConnected)
        } // if (valid device)

        // Free the udev device
        udev_device_unref (m_pDev);
        m_pDev = NULL;
    }// udev_list_entry_foreach(l_pDeviceListEntry, m_pDevices)

    // Cleanup

    udev_enumerate_unref(l_pUdevEnumerate);

#ifdef CHECK_LAST_DEVICE
    ETG_TRACE_USR1 (("********************SEEN Devices******************************"));
    for ( unsigned int l_uiCount = 0; l_uiCount < m_vSeenDevices.size(); l_uiCount++)
    {
        ETG_TRACE_USR1 (("Device SEEN = %s", m_vSeenDevices[l_uiCount].toStdString().c_str()));
    }
    ETG_TRACE_USR1 (("****************************************************"));

#endif // CHECK_LAST_DEVICE

    ETG_TRACE_USR4(("End  : LookupConnectedDevices"));

    return DEVICEMANAGER_OK;
}
#endif // USE_LOOKUP

/*-----------------------------------------------------------------------------*
 * int LookupExistingMounts()                                                  *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::LookupExistingMounts( CGlobalEnumerations::NOTIFICATION_Type enWDevTyp/*=CGlobalEnumerations::enNotifyAll*/, tCString fileName/*=NULL*/ )
{
    //Lock to avoid multiple updates   //NCG3D-79558
    Locker locker(&m_UpdateDVDDriveInfoLock);
    int err = 0;
#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
    tU8 u8Offset = (tU8)AUTOMOUNTER_NAMEOFFSET;
#else
    tU8 u8Offset = 0;
#endif

#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
    IAutomounterAdapter::tenResIAutomounterAdapt_Type enAutoAdaptRes;
#endif //VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF

    DIR *automount = opendir( INOTIFY_DIR );
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
    DIR* unsupported = opendir( INOTIFY_UNSUP_DIR );
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
    if( (NULL == automount) || !unsupported )
#else // USE_EXTERNAL_UNSUPPORTED_DEVICES
    if( NULL == automount )
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
    {
        err = errno;
        ETG_TRACE_ERR (("LookupExistingMounts: [ERROR] 'opendir' failed: %s", strerror (err)));
    }
    else
    {
        GENSTRING l_cDevNode /*e.g. /dev/sda1*/, l_cPartitionNode /*e.d. sda1*/;
        tCString pName;
        struct dirent *device = NULL;
        errno = 0;
        CGlobalEnumerations::NOTIFICATION_Type l_enWNotiDevTyp;

        //======================================
        // care for supported devices
        // enNotifyAll: used at startup and for observer mechanism
        // enInotify  : used if single device is notified
        //======================================
        if(   enWDevTyp == CGlobalEnumerations::enNotifyAll   /*called at startup if lookup is done*/
              || enWDevTyp == CGlobalEnumerations::enInotify )  /* devices are notified in /tmp/.automount (old mount mechanism) */
        {
            l_enWNotiDevTyp = CGlobalEnumerations::enInotify;
            do
            {
                pName = NULL;                // reset pointer to file as indicator to don't process this directory any longer
                if( fileName )                  // file name (e.g. sda1) given?
                {
                    pName = fileName;            // copy given file name
                }
                else                            // else read director entries
                {
                    device = readdir(automount);
                    if( device )
                    {
                        pName = device->d_name;  // copy file names if there are entries
                    }
                }
                //->VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
                if( IsValidMscType( pName,u8Offset ) )   // Check for USB SD Card and CD ROM
                    //<-VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
                {
                    ETG_TRACE_SYS(( "LookupExistingMounts: Processing mount-point = %s", pName ));
#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
                    //@todo get rid of QTString in the course of QT-Replacement
                    std::string strOut = "/dev/";
                    std::string strIn  = pName;
                    std::string strPartitionNode;
                    strPartitionNode.clear();

                    NORMAL_M_ASSERT(m_pIAutomounterAdapter);
                    enAutoAdaptRes = IAutomounterAdapter::enRES_ERROR_GENERAL;
                    if(m_pIAutomounterAdapter)
                    {
                        enAutoAdaptRes = m_pIAutomounterAdapter->GetExtractBlockDevName(OUT strPartitionNode,IN strIn);
                    }


                    if(IAutomounterAdapter::enRES_OK == enAutoAdaptRes)  //@todo care for using error reult down in code
                    {
                        l_cPartitionNode = strPartitionNode.c_str(); //e.g. sda1

                        strOut.append(strPartitionNode);
                        l_cDevNode       = strOut.c_str();          //e.d. /dev/sda1


                    }

#else //VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
                    l_cPartitionNode = pName;                  //e.g. sda1
                    l_cDevNode = GENSTRING( "/dev/" );
                    l_cDevNode      += pName;//device->d_name;    //e.d. /dev/sda1


#endif //VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF

                    m_cAction = "add";

                    if ( TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) )
                    {
                        ETG_TRACE_SYS (("LookupExistingMounts: Device already connected : %s",
                                        l_cDevNode.toStdString().c_str()));
                    }
                    else
                    {
                        ETG_TRACE_SYS (("LookupExistingMounts: New device connected : %s",
                                        l_cDevNode.toStdString().c_str()));

                        // Process the new device
                        GENSTRING mountPoint; //@todo where does this come from???
                        if(DEVICEMANAGER_DEVICE_DETECTED == HandleMount( /*pName*/ l_cPartitionNode.toStdString().c_str(), mountPoint )) ///@todo care for error case with c_str();
                        {
                            ETG_TRACE_SYS (("LookupExistingMounts: New device added"));
                        }
                    } // if (IsAlreadyConnected)
                }
            } while (NULL != device);
        }

        if(   enWDevTyp == CGlobalEnumerations::enNotifyAll    // /*called at startup if lookup is done*/
              || enWDevTyp == CGlobalEnumerations::enInUnsupDev ) // or if only unsupported devices should be checked
        {
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
            l_enWNotiDevTyp = CGlobalEnumerations::enInUnsupDev;   // set watch descriptor notification type to UnsupportedDevices
            do
            {
                // @todo add check for file name like in 'enWDevTyp == CGlobalEnumerations::enInotify'
                pName = NULL;
                device = readdir( unsupported );
                if( device )
                {
                    pName = device->d_name;
                }
                if( IsValidMscType( pName,u8Offset ) )
                {
                    ETG_TRACE_SYS(( "LookupExistingMounts: Processing mount-point (unsupported device) = %s", pName ));
                    l_cDevNode = GENSTRING( "/dev/" ) + pName;      // e.g. /dev/sda1
                    m_cAction = "add";    // @todo check if a function could be used to set the member

                    if( DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) == TRUE )
                    {
                        ETG_TRACE_SYS (( "LookupExistingMounts: (unsupported) Device already connected : %s", l_cDevNode.toStdString().c_str() ));
                    }
                    else
                    {
                        ETG_TRACE_SYS (("LookupExistingMounts: New (unsupported) device connected : %s", l_cDevNode.toStdString().c_str()));
                        // Process the new device
                        GENSTRING mountPoint;     // @todo why is mountPoint necessary here??
                        if (DEVICEMANAGER_DEVICE_DETECTED == HandleMountUnsupported( pName, mountPoint, l_enWNotiDevTyp ))
                        {
                            ETG_TRACE_SYS (("LookupExistingMounts: New (unsupported) device added"));
                        }
                    } // if (IsAlreadyConnected)
                }
            }
            while (NULL != device);
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
        }

        (void)closedir( automount );
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        (void)closedir( unsupported );
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
        /*EXIT_SUCCESS*/
    }
}

/*-----------------------------------------------------------------------------*
 * int LookupRemovedMounts()                                                   *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::LookupRemovedMounts()
{
    //ETG_TRACE_SYS(("Begin: LookupRemovedMounts"));

    int err = 0;
    bool bFound = false;
    DIR *automount = opendir(WATCH);

    vector<GENSTRING> strExistingBlockDevices;
    vector<GENSTRING> strListofRemovedDevices;

    GENSTRING l_cDevNode; /*e.g. /dev/sda1*/
#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
    IAutomounterAdapter::tenResIAutomounterAdapt_Type enAutoAdaptRes;
    GENSTRING l_cPartitionNode /*e.d. sda1*/;
    std::string strOut;
    std::string strIn;
    std::string strPartitionNode;
#endif

    if (NULL == automount)
    {
        err = errno;

        ETG_TRACE_ERR (("LookupRemovedMounts: [ERROR] 'opendir' failed: %s", strerror (err)));
    }
    else
    {
        struct dirent *device = NULL;
        errno = 0;

        //-----------------------------
        //check content of automount
        //---------------------------
        do
        {
            device = readdir(automount);

            if (NULL != device && '.' != device->d_name[0])
            {
                ETG_TRACE_SYS (("LookupRemovedMounts: Processing mount-point = %s", device->d_name));



#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
                //@todo get rid of QTString in the course of QT-Replacement
                strOut = "/dev/";
                strIn  = device->d_name; //e.g. /dev/partition_dev_sdc1.info
                strPartitionNode.clear();

                NORMAL_M_ASSERT(m_pIAutomounterAdapter);
                enAutoAdaptRes = IAutomounterAdapter::enRES_ERROR_GENERAL;
                if(m_pIAutomounterAdapter)
                {
                    enAutoAdaptRes = m_pIAutomounterAdapter->GetExtractBlockDevName(OUT strPartitionNode,IN strIn);
                }

                if(IAutomounterAdapter::enRES_OK == enAutoAdaptRes)  //@todo care for using error reult down in code
                {
                    l_cPartitionNode = strPartitionNode.c_str(); //e.g. sda1

                    strOut.append(strPartitionNode);
                    l_cDevNode = strOut.c_str();          //e.d. /dev/sda1
                }
                else
                {
                    ETG_TRACE_FATAL(("LookupRemovedMounts:m_pIAutomounterAdapter->GetExtractBlockDevName returned !enRES_OK"));
                }
#else
                l_cDevNode = GENSTRING ("/dev/");
                l_cDevNode += device->d_name;

#endif
                strExistingBlockDevices.push_back(l_cDevNode);

            }
        } while (NULL != device);
        (void)closedir(automount);

        //-----------------------------------------------------
        //check which are missing and hence should be removed
        //-----------------------------------------------------
        CDevice oCurrentDevice;
        tBool bExists;

        unsigned int uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);

        for (unsigned int l_uiCount = 0; l_uiCount < uiSize;l_uiCount++)
        {
            bExists = DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount, oCurrentDevice);
            NORMAL_M_ASSERT(TRUE == bExists);
            if(TRUE == bExists)
            {
                ETG_TRACE_SYS(("m_vConnectedDevices[l_uiCount]->m_cMountPoint: %s",
                               oCurrentDevice.m_cMountPoint.toStdString().c_str()));
                bFound = false;
                for (unsigned int l_uiCnt = 0; l_uiCnt < strExistingBlockDevices.size();l_uiCnt++)
                {
                    ETG_TRACE_USR4(("strExistingBlockDevices[l_uiCnt]: %s",strExistingBlockDevices[l_uiCnt].toStdString().c_str()));
                    oCurrentDevice.Show();
#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
                    ETG_TRACE_USR4(("oCurrentDevice.m_cPartitionNode: %s",oCurrentDevice.m_cPartitionNode.toStdString().c_str()));
                    if(strExistingBlockDevices[l_uiCnt].contains(oCurrentDevice.m_cPartitionNode))
#else
                    ETG_TRACE_USR4(("oCurrentDevice.m_cDevnode: %s",oCurrentDevice.m_cDevnode.toStdString().c_str()));
                    if(strExistingBlockDevices[l_uiCnt].contains(oCurrentDevice.m_cDevnode))
#endif
                    {
                        bFound = true;
                        break;
                    }
                }
                if(bFound == false)
                {
                    //since it is triggered by mechanism INotify (/tmp/.automount) it is allowed to care for DTY_USB and DTY_SD only
                    //otherwhise we would disconnect apple devices here
                    if(  (CGlobalEnumerations::DTY_SD          == oCurrentDevice.m_eDeviceType)
                         ||(CGlobalEnumerations::DTY_USB         == oCurrentDevice.m_eDeviceType)
                         ||(CGlobalEnumerations::DTY_CDROM       == oCurrentDevice.m_eDeviceType)
                         ||(CGlobalEnumerations::DTY_SD_INTERNAL == oCurrentDevice.m_eDeviceType)
     #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                         ||(   ( oCurrentDevice.m_eDeviceType    == CGlobalEnumerations::DTY_NOT_SUPPORTED ) // ... and it's allowed for not supported devices
                               && ( oCurrentDevice.m_ePTType     != CGlobalEnumerations::PT_UNKNOWN ) // ... and it's allowed for not supported devices that are block devices(for block devices, the partition table will not be empty)
                               && ( oCurrentDevice.m_iVendorID   != AppleVendorID )                          // (but only if the vendor is not Apple...)
                               )
     #endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                         )
                    {
                        if(oCurrentDevice.m_cMountPoint.compare(GENSTRING(CDROMDIAG_MOUNTPOINT)) == 0)
                        {
                            ETG_TRACE_FATAL(("LookupRemovedMounts: Don't remove DiagCD with observer mechanism %s",CDROMDIAG_MOUNTPOINT));
                        }
                        else
                        {
                            ETG_TRACE_SYS(("m_vConnectedDevices[l_uiCount]->m_cDevnode:%s",
                                           oCurrentDevice.m_cDevnode.toStdString().c_str()));
                            strListofRemovedDevices.push_back(oCurrentDevice.m_cDevnode);
                        }
                    }
                }
            }
        }


        //--------------------
        //remove it
        //---------------------
        for (unsigned int l_uiCnt = 0; l_uiCnt < strListofRemovedDevices.size();l_uiCnt++)
        {
            HandleUnMount(strListofRemovedDevices[l_uiCnt]);
        }

        strExistingBlockDevices.clear();
        strListofRemovedDevices.clear();

        /*EXIT_SUCCESS*/
    }

    //ETG_TRACE_SYS(("End  : LookupRemovedMounts"));

}



/*-----------------------------------------------------------------------------*
 * int FillCurrDevice(CDevice* f_pCurDevice)                                   *
 *-----------------------------------------------------------------------------*/
int CUDevManager::FillCurrDevice(CDevice* f_pCurDevice) //adjust for Roadmap 13001_EXtendedUserInformation
{
    ETG_TRACE_USR4(("Begin: FillCurrDevice"));

    // fill usb device details
    if(   (f_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enUnotify)
          || (f_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enInotify )
          )
    {
        if (!GENSTRING(udev_device_get_devtype(m_pBaseUSBDev)).contains("usb_device"))
        {
            ETG_TRACE_USR1(( "FillCurrDevice:Device is NOT a usb_device" ));
            ETG_TRACE_USR4 (( "End  : FillCurrDevice with %i", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_ERROR_NODEVICE) ));
            return DEVICEMANAGER_ERROR_NODEVICE;
        }

        libusb_device *dev = LibUSBGetDevice( udev_device_get_devnode( m_pBaseUSBDev ) );
        int iRetrial = 0;
        int l_iRetVal = DEVICEMANAGER_OK;
        while(iRetrial < 3)
        {
            l_iRetVal = LibUSBFillDeviceDetails( dev, *f_pCurDevice );
            if(DEVICEMANAGER_OK == l_iRetVal)
            {
                ETG_TRACE_COMP(("FillCurrDevice: Filled the device details using the libusb apis"));
                //found the device details from the libusb apis. breaking the while loop
                break;
            }
            else
            {
              iRetrial++;
              usleep((useconds_t)(USE_DELAY_RETRIAL_LIBUSB_FILLDEVICE*1000));
              ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));
              ETG_TRACE_FATAL(("Retrial after %d ms - fill the device details using the  libusb apis",USE_DELAY_RETRIAL_LIBUSB_FILLDEVICE));
              ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));

            }
        }
#ifdef BUGFIX_EMPTYSERIALID_IPOD
        //if IPOD serial ID is empty then we try it again

        if ((AppleVendorID == f_pCurDevice->m_iVendorID)
                && (0x12 == (f_pCurDevice->m_iProductID >> 8)))
        {
            if(false == f_pCurDevice->m_cSerialID.contains("Apple"))
            {
                usleep(1000*1500);  //1,5sec
                ETG_TRACE_FATAL(("FillCurrDevice: Retrial after 1.5sec for Apple device"))
                        LibUSBFillDeviceDetails (dev, *f_pCurDevice);
            }
        }
#endif



        f_pCurDevice->m_cDevpath     = udev_device_get_syspath(m_pBaseUSBDev);
        f_pCurDevice->m_cDevnode     = udev_device_get_devnode(m_pBaseUSBDev);
        f_pCurDevice->m_cSysName     = udev_device_get_sysname(m_pBaseUSBDev);

        //care to sort to which USB-connector device belongs to

        ETG_TRACE_COMP(("FillCurrDevice: m_iUSBConnector     USB%1d",f_pCurDevice->m_iUSBConnector));
        ETG_TRACE_COMP(("FillCurrDevice: m_cDevpath          %s", f_pCurDevice->m_cDevpath.toStdString().c_str()));
        ETG_TRACE_COMP(("FillCurrDevice: m_cDevnode          %s",f_pCurDevice->m_cDevnode.toStdString().c_str()));
        ETG_TRACE_COMP(("FillCurrDevice: m_cSysName          %s",f_pCurDevice->m_cSysName.toStdString().c_str()));

        //RTC-669924: When Manufacturer,Serial and Product is empty, form SerialID from Vendorid, Productid and Sysname.
        if((DEVICEMANAGER_OK == l_iRetVal) && ((CGlobalEnumerations::DTY_UNKNOWN == f_pCurDevice->m_eDeviceType) || (CGlobalEnumerations::DTY_HUB == f_pCurDevice->m_eDeviceType)))
        {
            if(f_pCurDevice->m_cSerialID.isEmpty())
            {
                ETG_TRACE_USR4 (("FillCurrDevice: Serial Id formed from Manufacturer, Product and SerialNumber empty "));
                FormSerialIdFromVendorIdProductIdSysname(IN f_pCurDevice->m_iVendorID, IN f_pCurDevice->m_iProductID, IN f_pCurDevice->m_cSysName, OUT f_pCurDevice->m_cSerialID);
            }
            else if(f_pCurDevice->m_cShortSerial.isEmpty())//m_cSerialID not empty, but also not unique
            {
                ETG_TRACE_USR4 (("FillCurrDevice:SerialId formed doesnt has SerialNumber.Hence make the serialID unique"));
                f_pCurDevice->m_cSerialID += '_';
                makeSerialIdUnique(IN f_pCurDevice->m_cSysName, INOUT f_pCurDevice->m_cSerialID);
                f_pCurDevice->m_cSerialID.trimmed();
            }
        }

        //Opaque object representing one entry in a list. An entry contains contains a name, and
        //optionally a value
        struct udev_list_entry *l_pListEntry;

        udev_list_entry_foreach( l_pListEntry, udev_device_get_properties_list_entry(m_pBaseUSBDev) )
        {
            //NULL check for udev_list_entry_get_name(l_pListEntry)
            if( NULL != udev_list_entry_get_name(l_pListEntry) )
            {
                // Check for value of  ID_REVISION
                if( !strcmp( udev_list_entry_get_name(l_pListEntry), "ID_REVISION") )
                {
                    f_pCurDevice->m_cDeviceVersion = udev_list_entry_get_value( l_pListEntry );
                    ETG_TRACE_COMP(("FillCurrDevice: m_cDeviceVersion    %s", f_pCurDevice->m_cDeviceVersion.toStdString().c_str()));
                }
            }
        }
        ETG_TRACE_COMP(("FillCurrDevice: m_eDeviceType       %d", ETG_ENUM(DVM_DEVICE_TYPE, f_pCurDevice->m_eDeviceType) ));
    }

    DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, f_pCurDevice);

    // Don't proceed if serial id is missing for the device
#ifndef BUGFIX_NO_SERIALID
    if (f_pCurDevice->m_cSerialID.isEmpty())
#else
    if(   ( f_pCurDevice->m_cSerialID.isEmpty() )
          && ( f_pCurDevice->m_eDeviceType != CGlobalEnumerations::DTY_USB )   // Ignore missing SerialID only for DTY_USB
          )
#endif
    {
        ETG_TRACE_ERR (("FillCurrDevice: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        ETG_TRACE_ERR (("FillCurrDevice: !!!!    Serial ID is empty    !!!!"));
        ETG_TRACE_ERR (("FillCurrDevice: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        ETG_TRACE_USR4 (("End  : FillCurrDevice with %i", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_ERROR_UNKNOWN) ));

        return DEVICEMANAGER_ERROR_UNKNOWN;
    }

    if(   (f_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enUnotify)
          || (f_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enInotify )
          )
    {
        m_pParentHUBDev = udev_device_get_parent(m_pBaseUSBDev);

        if (NULL != m_pParentHUBDev)
        {
            GetParentSerialId(f_pCurDevice);

            // Reset m_pParentHUBDev to NULL
            m_pParentHUBDev = NULL;
        }
        // Parse USB port number
        ConfigurationManager::GetInstance()->vGetUSBConnectorFromSysPath(OUT f_pCurDevice->m_eUSB, IN f_pCurDevice->m_cDevpath.toStdString());
        ConfigurationManager::GetInstance()->iGetHUBConnectedFromSysName(OUT f_pCurDevice->m_bConnectedAtHub, OUT f_pCurDevice->m_iPortNumber, IN f_pCurDevice->m_cSysName);

        f_pCurDevice->m_iUSBConnector  = (int)f_pCurDevice->m_eUSB;

        if(eUSB1 == f_pCurDevice->m_eUSB)
        {
            f_pCurDevice->m_cUSBPort   = USBConnector1str;  //used for CCA output - should be shifted to that
        }
        else if(eUSB2 ==  f_pCurDevice->m_eUSB)
        {
            f_pCurDevice->m_cUSBPort = USBConnector2str;
        }
        else if (eUSB3 == f_pCurDevice->m_eUSB)
        {
            f_pCurDevice->m_cUSBPort = USBConnector3str;
        }
#ifdef USE_FOURTH_USB_CONNECTOR
        else if(eUSB4 ==  f_pCurDevice->m_eUSB)
        {
            f_pCurDevice->m_cUSBPort = USBConnector4str;
        }
#endif
        else
        {
            f_pCurDevice->m_cUSBPort       = USBConnectorInvalidstr;
            f_pCurDevice->m_iUSBConnector  = USBConnectorInvalid;
        }

        f_pCurDevice->m_iHubIndex = f_pCurDevice->m_iUSBConnector;

        // Parent hub swapping
        ParentHUBSwapping(f_pCurDevice);
    }

    //f_pCurDevice->Show();

    ETG_TRACE_USR4 (("End  : FillCurrDevice (returns with value %i", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_OK) ));

    return DEVICEMANAGER_OK;
}//lint !e429


/*-----------------------------------------------------------------------------*
 * int FillCurrDeviceSDInternal(CDevice* f_pCurDevice)                                   *
 *-----------------------------------------------------------------------------*/


int CUDevManager::FillCurrDeviceSDInternal(CDevice* f_pCurDevice, GENSTRING f_cDevNode) //adjust for Roadmap 13001_ExtendedUserInformation
{
    ///@todo check for memory leak

    int iRetVal = DEVICEMANAGER_ERROR_NODEVICE;

    tU64  u64SizeInBytes;

    const char *strSyspath;
    const char* strSysname;
    const char* strDevnode;
    const char *strDevnodeParent;

    const char *strID_NAME;
    const char *strID_SERIAL;
    const char *strID_FS_LABEL;
    //const char *strID_FS_LABEL_ENC;
    const char *strID_FS_TYPE;
    const char *strID_FS_UUID;
    const char *strUDISKS_PARTITION_SIZE;
    const char *strUDISKS_PARTITION_NUMBER;

    struct udev_device *l_device, *l_deviceParent;

    ETG_TRACE_USR4(("Begin: FillCurrDeviceSDInternal"));
    ETG_TRACE_USR4(("Begin: FillCurrDeviceSDInternal: f_cDevNode:%s",f_cDevNode.toStdString().c_str()));

    f_pCurDevice->m_eDeviceType      = CGlobalEnumerations::DTY_SD_INTERNAL;
    f_pCurDevice->m_bConnectedAtHub = false;

    if(m_pBaseUSBDev)
    {
        //init values which will be adjusted in this function
        f_pCurDevice->m_cDevpath          = "";
        f_pCurDevice->m_cSysName          = "";
        f_pCurDevice->m_cDevnode          = "";
        f_pCurDevice->m_cDeviceName       = "";
        f_pCurDevice->m_cSerialID         = "";
        f_pCurDevice->m_cShortSerial      = "";
        f_pCurDevice->m_cFSType           = "";
        f_pCurDevice->m_iValidPartitionNo = 0;
        f_pCurDevice->m_cAccessoryName    = "";

        strSyspath     = udev_device_get_syspath(m_pBaseUSBDev);
        strSysname     = udev_device_get_sysname(m_pBaseUSBDev);
        strDevnode     = udev_device_get_devnode(m_pBaseUSBDev);

        //map to CDevice-elements
        if(strSyspath){f_pCurDevice->m_cDevpath     = strSyspath;}
        if(strSysname){f_pCurDevice->m_cSysName     = strSysname;}
        if(strDevnode){f_pCurDevice->m_cDevnode     = strDevnode;}

        ETG_TRACE_USR3(("FillCurrDeviceSDInternal:strSyspath           :%s",f_pCurDevice->m_cDevpath.toStdString().c_str()));
        ETG_TRACE_USR3(("FillCurrDeviceSDInternal:strSysname           :%s",f_pCurDevice->m_cSysName.toStdString().c_str()));
        ETG_TRACE_USR3(("FillCurrDeviceSDInternal:strDevnode           :%s",f_pCurDevice->m_cDevnode.toStdString().c_str()));


        l_device = udev_device_new_from_syspath(m_pUdev /*udev-lib context*/,f_pCurDevice->m_cDevpath.toStdString().c_str());
        if(l_device)
        {

            strID_NAME                   = udev_device_get_property_value(l_device,"ID_NAME");
            strID_SERIAL                 = udev_device_get_property_value(l_device,"ID_SERIAL");
            strID_FS_LABEL               = udev_device_get_property_value(l_device,"ID_FS_LABEL");
            strID_FS_TYPE                = udev_device_get_property_value(l_device,"ID_FS_TYPE");
            strID_FS_UUID                = udev_device_get_property_value(l_device,"ID_FS_UUID");
            strUDISKS_PARTITION_SIZE     = udev_device_get_property_value(l_device,"UDISKS_PARTITION_SIZE");
            strUDISKS_PARTITION_NUMBER   = udev_device_get_property_value(l_device,"UDISKS_PARTITION_NUMBER");

            if(strID_NAME){                ETG_TRACE_USR3(("FillCurrDeviceSDInternal:ID_NAME                 :%s",strID_NAME));}
            if(strID_SERIAL){              ETG_TRACE_USR3(("FillCurrDeviceSDInternal:ID_SERIAL               :%s",strID_SERIAL));}
            if(strID_FS_LABEL){            ETG_TRACE_USR3(("FillCurrDeviceSDInternal:strID_FS_LABEL          :%s",strID_FS_LABEL));}
            if(strID_FS_TYPE){             ETG_TRACE_USR3(("FillCurrDeviceSDInternal:ID_FS_TYPE              :%s",strID_FS_TYPE));}
            if(strID_FS_UUID){             ETG_TRACE_USR3(("FillCurrDeviceSDInternal:ID_FS_UUID              :%s",strID_FS_UUID));}
            if(strUDISKS_PARTITION_SIZE){  ETG_TRACE_USR3(("FillCurrDeviceSDInternal:UDISKS_PARTITION_SIZE   :%s",strUDISKS_PARTITION_SIZE));}
            if(strUDISKS_PARTITION_NUMBER){ETG_TRACE_USR3(("FillCurrDeviceSDInternal:UDISKS_PARTITION_NUMBER :%s",strUDISKS_PARTITION_NUMBER));}

            //map to CDevice-elements
            //f_pCurDevice->m_cDeviceName  = strID_NAME;  SUZUKI-14757 demands FS_LABEL-Name to be used as cDeviceName
            if(strID_FS_LABEL)
            {
                f_pCurDevice->m_cDeviceName  = GENSTRING(strID_FS_LABEL);
            }
            else
            {
                ETG_TRACE_COMP(("FillCurrDeviceSDInternal: no ID_FS_LABEL use ID_NAME"));
                if(strID_NAME){f_pCurDevice->m_cDeviceName  = strID_NAME;}
            }
            if(strID_FS_TYPE){ f_pCurDevice->m_cFSType     = GENSTRING(strID_FS_TYPE);}

            //f_pCurDevice->m_cSerialID is constructed from different sub entries
            f_pCurDevice->m_cSerialID                      = GENSTRING("_NM:");          //device name
            if(strID_NAME){f_pCurDevice->m_cSerialID      += GENSTRING(strID_NAME) ;}
            f_pCurDevice->m_cSerialID                     += GENSTRING("_PRT:");         //partition name
            if(strID_FS_LABEL){f_pCurDevice->m_cSerialID  += GENSTRING(strID_FS_LABEL);}
            f_pCurDevice->m_cSerialID                     += GENSTRING("_SID:");          //serial id
            if(strID_SERIAL){  f_pCurDevice->m_cSerialID  += GENSTRING(strID_SERIAL);}
            f_pCurDevice->m_cSerialID                     += GENSTRING("_UUID:");        //uuid of partition
            if(strID_FS_UUID){ f_pCurDevice->m_cSerialID  += GENSTRING(strID_FS_UUID);}

            f_pCurDevice->m_cShortSerial = f_pCurDevice->m_cSerialID;

#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
            vMapFileSystemType(INOUT f_pCurDevice);
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
            FillStorageInfoSizeTUF(f_pCurDevice);
#endif //USE_TOTAL_USED_FREE_SIZE

            //--------------------
            // return success
            //--------------------
            iRetVal = DEVICEMANAGER_OK; //this is the minimum to let it go through to clients although other information demanded below might be missing

#ifdef USE_PARITIONNUMBER
            if(NULL != strUDISKS_PARTITION_NUMBER)
            {
                f_pCurDevice->m_iValidPartitionNo = (int)atoll(strUDISKS_PARTITION_NUMBER);
            }

            if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_PARTITIONS_ALL))
            {
                //-----------------------------------------------------------------------------------
                // do care that only a single partition is notified - first trial assuming that first notified is always first of device
                //Hypothesis for this f_pCurDevice->m_iValidPartitionNo = 0 means: sdcard has only a single partition
                //f_pCurDevice->m_iValidPartitionNo = 1 means there is one partition or even more
                //
                // if this does not work than we would have to go for reading the partition table as well
                //-----------------------------------------------------------------------------------

                if(f_pCurDevice->m_iValidPartitionNo > 1 /*num of partitions only 1*/)
                {
                    iRetVal = DEVICEMANAGER_ERROR_NODEVICE;
                }
            }


#endif //USE_PARITIONNUMBER

            if(NULL != strUDISKS_PARTITION_NUMBER)
            {
                u64SizeInBytes = (tU64)atoll( strUDISKS_PARTITION_SIZE );
            }
            else
            {
                u64SizeInBytes = 0;
            }
            f_pCurDevice->m_u64Size = u64SizeInBytes;

            l_deviceParent = udev_device_get_parent(l_device);
            if(l_deviceParent)
            {
                strDevnodeParent =  udev_device_get_devnode(l_deviceParent);
                f_pCurDevice->m_cAccessoryName = strDevnodeParent;

            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceSDInternal:l_deviceParent== NULL "));
            }

        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceSDInternal:l_device== NULL "));
        }

    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceSDInternal:m_pBaseUSBDev == NULL "));
    }


    // fill other entries with dummy //@todo cleanup
    f_pCurDevice->m_iVendorID      = 0xDEAD;
    f_pCurDevice->m_iProductID     = 0xBEAF;
    f_pCurDevice->m_iDevNUM        = 0xAA;
    f_pCurDevice->m_cDeviceVersion = "";

#ifdef BUGFIX_DECOUPLE_SDCARD_PRM
    f_pCurDevice->m_cUSBPort      = USBConnectorUndefStr;
    f_pCurDevice->m_iUSBConnector = USBConnectorUndef;
    f_pCurDevice->m_eUSB          = eUSBUndef;
    f_pCurDevice->m_iPortNumber   = 0;
#else
    f_pCurDevice->m_cUSBPort      = USBConnector1str;//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors
    f_pCurDevice->m_iUSBConnector = USBConnector1;//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors
    f_pCurDevice->m_eUSB          = eUSB1;
    f_pCurDevice->m_iPortNumber   = 1;
#endif

    //f_pCurDevice->m_cSerialID.replace (" ", "_");
    f_pCurDevice->m_cManufacturer = "Test:Manufacturer";//GENSTRING(l_sMfg).trimmed();

    f_pCurDevice->m_cParentSerialId = "-";
    f_pCurDevice->m_iHubIndex = f_pCurDevice->m_iUSBConnector; ///@todo check this

    //#ifdef HELPER_ANALYSIS_UDEVEVENTS //@todo think how to add this
    //    DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, f_pCurDevice);
    //#endif // HELPER_ANALYSIS_UDEVEVENTS

    f_pCurDevice->Show();
    ETG_TRACE_USR4 (("End  : FillCurrDeviceSDInternal (returns with value %i", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_OK) ));

    return iRetVal;
}//lint !e429

/*-----------------------------------------------------------------------------*
 * int FillCurrDeviceCDDA(...)                                   *
 *-----------------------------------------------------------------------------*/
int CUDevManager::FillCurrDeviceCDDA(CDevice* f_pCurDevice, GENSTRING f_cDevNode)
{
    ETG_TRACE_USR4(("Begin: FillCurrDeviceCDDA"));


    /* Example notification examined with udevadm monitor to get more background information

             //==============================================================
             //THIS IS WHAT WE GET IF WE JUST CONNECT THE DRIVE WITHOUT DISC INSERTED (Example)
             //==============================================================

             UDEV  [3758.980926] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host7/target7:0:0/7:0:0:0/block/sr0 (block)
             ACTION=change
             DEVLINKS=/dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0 /dev/cdrom /dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
             DEVNAME=/dev/sr0
             DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host7/target7:0:0/7:0:0:0/block/sr0
             DEVTYPE=disk
             DISK_MEDIA_CHANGE=1
             ID_BUS=usb
             ID_CDROM=1
             ID_CDROM_CD=1
             ID_CDROM_DVD=1
             ID_CDROM_MRW=1
             ID_CDROM_MRW_W=1
             ID_INSTANCE=0:0
             ID_MODEL=FD2-3500
             ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
             ID_MODEL_ID=0011
             ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
             ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
             ID_REVISION=0.01
             ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
             ID_SERIAL_SHORT=000000000000
             ID_TYPE=cd
             ID_USB_DRIVER=usb-storage
             ID_USB_INTERFACES=:080250:
             ID_USB_INTERFACE_NUM=00
             ID_VENDOR=JVCKWD
             ID_VENDOR_ENC=JVCKWD\x20\x20
             ID_VENDOR_ID=2166
             MAJOR=11
             MINOR=0
             SEQNUM=1883
             SUBSYSTEM=block
             SYSTEMD_READY=0
             TAGS=:systemd:
             UDISKS_PRESENTATION_NOPOLICY=0
             USEC_INITIALIZED=3758901740


            //==============================================================
            //THIS IS WHAT WE GET IF disc is inserted
            //==============================================================
            UDEV  [518.350487] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host3/target3:0:0/3:0:0:0/block/sr0 (block)
            ACTION=change
            DEVLINKS=/dev/cdrom /dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0 /dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0
            DEVNAME=/dev/sr0
            DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host3/target3:0:0/3:0:0:0/block/sr0
            DEVTYPE=disk
            DISK_MEDIA_CHANGE=1
            ID_BUS=usb
            ID_CDROM=1
            ID_CDROM_CD=1
            ID_CDROM_DVD=1
            ID_CDROM_MEDIA=1                                                  //bound to an inserted disc
            ID_CDROM_MEDIA_CD=1                                           //bound to an inserted disc
            ID_CDROM_MEDIA_TRACK_COUNT_AUDIO=16           // bound to an inserted disc type CDDA
            ID_CDROM_MRW=1
            ID_CDROM_MRW_W=1
            ID_INSTANCE=0:0
            ID_MODEL=FD2-3500
            ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
            ID_MODEL_ID=0011
            ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
            ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
            ID_REVISION=0.01
            ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
            ID_SERIAL_SHORT=000000000000
            ID_TYPE=cd
            ID_USB_DRIVER=usb-storage
            ID_USB_INTERFACES=:080250:
            ID_USB_INTERFACE_NUM=00
            ID_VENDOR=JVCKWD
            ID_VENDOR_ENC=JVCKWD\x20\x20
            ID_VENDOR_ID=2166
            MAJOR=11
            MINOR=0
            SEQNUM=1812
            SUBSYSTEM=block
            TAGS=:systemd:
            UDISKS_PRESENTATION_NOPOLICY=0
            USEC_INITIALIZED=120613824


            //==================================================
            //this is what we get if CD is removed
            //==================================================

            bash-3.2# udevadm monitor --property
            monitor will print the received events for:
            UDEV - the event which udev sends out after rule processing
            KERNEL - the kernel uevent

            KERNEL[1304.219254] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host0/target0:0:0/0:0:0:0/block/sr0 (block)
            ACTION=change
            DEVNAME=/dev/sr0
            DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host0/target0:0:0/0:0:0:0/block/sr0
            DEVTYPE=disk
            DISK_MEDIA_CHANGE=1
            MAJOR=11
            MINOR=0
            SEQNUM=1815
            SUBSYSTEM=block

            UDEV  [1304.307175] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host0/target0:0:0/0:0:0:0/block/sr0 (block)
            ACTION=change
            DEVLINKS=/dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0 /dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0 /dev/cdrom
            DEVNAME=/dev/sr0
            DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host0/target0:0:0/0:0:0:0/block/sr0
            DEVTYPE=disk
            DISK_MEDIA_CHANGE=1
            ID_BUS=usb
            ID_CDROM=1
            ID_CDROM_CD=1
            ID_CDROM_DVD=1
            ID_CDROM_MRW=1
            ID_CDROM_MRW_W=1
            ID_INSTANCE=0:0
            ID_MODEL=FD2-3500
            ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
            ID_MODEL_ID=0011
            ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
            ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
            ID_REVISION=0.01
            ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
            ID_SERIAL_SHORT=000000000000
            ID_TYPE=cd
            ID_USB_DRIVER=usb-storage
            ID_USB_INTERFACES=:080250:
            ID_USB_INTERFACE_NUM=00
            ID_VENDOR=JVCKWD
            ID_VENDOR_ENC=JVCKWD\x20\x20
            ID_VENDOR_ID=2166
            MAJOR=11
            MINOR=0
            SEQNUM=1815
            SUBSYSTEM=block
            SYSTEMD_READY=0
            TAGS=:systemd:
            UDISKS_PRESENTATION_NOPOLICY=0
            USEC_INITIALIZED=11784438
      */



    int iRet = DEVICEMANAGER_ERROR_NODEVICE;

    const char *strSyspath;
    const char* strSysname;
    const char* strDevnode;

    const char *strID_VENDOR;
    const char *strID_REVISION;
    const char *strID_MODEL;
    const char *strID_CDROM_MEDIA;
    const char *strID_CDROM_MEDIA_CD;
    const char *strID_CDROM_MEDIA_TRACK_COUNT_AUDIO;

    struct udev_device *l_device;


    if(m_pBaseUSBDev)
    {
        strSyspath     = udev_device_get_syspath(m_pBaseUSBDev);
        strSysname     = udev_device_get_sysname(m_pBaseUSBDev);
        strDevnode     = udev_device_get_devnode(m_pBaseUSBDev);

        ETG_TRACE_COMP(("FillCurrDeviceCDDA:strSyspath           :%s",strSyspath));
        ETG_TRACE_COMP(("FillCurrDeviceCDDA:strSysname           :%s",strSysname));
        ETG_TRACE_COMP(("FillCurrDeviceCDDA:strDevnode           :%s",strDevnode));



        l_device = udev_device_new_from_syspath(m_pUdev /*udev-lib context*/,GENSTRING(strSyspath).toStdString().c_str());
        if(l_device)
        {
            strID_VENDOR                        = udev_device_get_property_value(l_device,"ID_VENDOR");
            strID_MODEL                         = udev_device_get_property_value(l_device,"ID_MODEL");
            strID_REVISION                      = udev_device_get_property_value(l_device,"ID_REVISION");
            strID_CDROM_MEDIA                   = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA");
            strID_CDROM_MEDIA_CD                = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA_CD");
            strID_CDROM_MEDIA_TRACK_COUNT_AUDIO = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA_TRACK_COUNT_AUDIO");

            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_VENDOR                        :%s",strID_VENDOR));
            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_MODEL                         :%s",strID_MODEL));
            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_REVISION                      :%s",strID_REVISION));

            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_CDROM_MEDIA                    :%s",strID_CDROM_MEDIA));
            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_CDROM_MEDIA_CD                 :%s",strID_CDROM_MEDIA_CD));
            ETG_TRACE_COMP(("FillCurrDeviceCDDA:ID_CDROM_MEDIA_TRACK_COUNT_AUDIO  :%s",strID_CDROM_MEDIA_TRACK_COUNT_AUDIO));


            //-----------------------------------------
            //check if it is really notification of DTY_CDDA
            //-----------------------------------------
            tBool isCDDA = TRUE;

            //check if one of those is NULL
            if(isCDDA == TRUE) {if(true == GENSTRING(strID_CDROM_MEDIA).isEmpty())                 {isCDDA = FALSE;}}
            if(isCDDA == TRUE) {if(true == GENSTRING(strID_CDROM_MEDIA_CD).isEmpty() )             {isCDDA = FALSE;}}
            if(isCDDA == TRUE) {if(true == GENSTRING(strID_CDROM_MEDIA_TRACK_COUNT_AUDIO).isEmpty()){isCDDA = FALSE;}}


            if(TRUE == isCDDA)
            {
                ETG_TRACE_COMP(("FillCurrDeviceCDDA: CDDA inserted"));
                iRet = DEVICEMANAGER_OK;
            }
            else
            {
                 ETG_TRACE_COMP(("FillCurrDeviceCDDA: not a CDDA"));
            }


            if(iRet == DEVICEMANAGER_OK)
            {
                //map to CDevice-elements
                f_pCurDevice->m_eDeviceType   = CGlobalEnumerations::DTY_CDDA;
                f_pCurDevice->m_cDevpath      = strSyspath;
                f_pCurDevice->m_cSysName      = strSysname;
                f_pCurDevice->m_cDevnode      = strDevnode;
                f_pCurDevice->m_cBlockDevnode = f_pCurDevice->m_cDevnode; //bugfix: to prevent notification every 10 sec triggered by observer mechanism

                //--------------------------------------------
                //CREATE RANDOM ID - intermediate solution
                //--------------------------------------------
                //map to CDevice-elements
                srand((unsigned int)time(NULL));
                char cSerial[256]; //@todo intermediate solution - has to be replaced by scsi commands
                snprintf(cSerial,sizeof(cSerial),"TESTID:%d",rand()); //@todo intermediate solution
                f_pCurDevice->m_cSerialID = cSerial;

                f_pCurDevice->m_cShortSerial = f_pCurDevice->m_cSerialID;

                //f_pCurDevice->m_cDeviceVersion = GENSTRING("DRIVE: ") + GENSTRING("ID_VENDOR:") +GENSTRING(strID_VENDOR)+ GENSTRING("_ID_MODEL:") + GENSTRING(strID_MODEL) + GENSTRING("_ID_REVISION:") + GENSTRING(strID_REVISION);


                // fill other entries with dummy //@todo cleanup
                f_pCurDevice->m_u64Size = 0;
                f_pCurDevice->m_iVendorID      = 0xFEED;
                f_pCurDevice->m_iProductID     = 0xBEAF;
                f_pCurDevice->m_iDevNUM        = 0xAA;

                ConfigurationManager::GetInstance()->vGetUSBConnectorFromSysPath(OUT f_pCurDevice->m_eUSB, IN f_pCurDevice->m_cDevpath.toStdString());
                f_pCurDevice->m_iUSBConnector = (int)f_pCurDevice->m_eUSB;
                f_pCurDevice->m_iPortNumber   = (int)f_pCurDevice->m_eUSB;

                //find out to which USB ocnnector this is bound
                f_pCurDevice->m_cUSBPort      = GENSTRING((int)f_pCurDevice->m_eUSB).toStdString().c_str();//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors

                //f_pCurDevice->m_cSerialID.replace (" ", "_");
                f_pCurDevice->m_cManufacturer = "Test:Manufacturer";//GENSTRING(l_sMfg).trimmed();


                f_pCurDevice->m_cParentSerialId = "-";
                f_pCurDevice->m_iHubIndex = f_pCurDevice->m_iUSBConnector; ///@todo check this

                f_pCurDevice->m_cDeviceName = "CDDA";

                f_pCurDevice->m_cFSType = "ISO9660";
                f_pCurDevice->m_eFSType = CGlobalEnumerations::FSTY_ISO9660;

                //example: f_pCurDevice->m_cMountPoint = "/dev/sr0,/dev/sg0";
                std::string strCDDAMountPoint = "";
                std::string strSRx            = "";
                std::string strSGx            = "";
                if(TRUE == bFindScsiCDDrive(OUT strSRx, OUT strSGx))
                {
                    strCDDAMountPoint  = strSRx;
                    strCDDAMountPoint += ",";
                    strCDDAMountPoint += strSGx;
                    ETG_TRACE_USR4(("vFillCDDADevice: strCDDAMountPoint: %s ",strCDDAMountPoint.c_str()));
                }
                else
                {

                    ETG_TRACE_FATAL(("vFillCDDADevice: bFindScsiCDDrive(...) == FALSE!!"));
                }

                //#ifdef HELPER_ANALYSIS_UDEVEVENTS //@todo think how to add this
                //    DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, f_pCurDevice);
                //#endif // HELPER_ANALYSIS_UDEVEVENTS

                f_pCurDevice->Show();
            }

        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceCDDA:l_device== NULL "));
        }

    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceCDDA:m_pBaseUSBDev == NULL "));
    }



    ETG_TRACE_USR4 (("End  : FillCurrDeviceCDDA (returns with value %i", ETG_ENUM(DEVICEMANAGER_ERROR, iRet) ));

    return iRet;


}


/*-----------------------------------------------------------------------------*
 * int FillCurrDeviceCDDA(...)                                   *
 *-----------------------------------------------------------------------------*/
int CUDevManager::FillCurrDeviceDVDDrive(CDevice* f_pCurDevice, GENSTRING f_cDevNode)
{
    ETG_TRACE_USR4(("Begin: FillCurrDeviceDVDDrive"));


    /* udevadm monitor --property does show this

    //-----------------------------------------------
    Elements marked with '##'which play a role to recognize to drive:
    //-----------------------------------------------
    UDEV  [65061.112931] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host5/target5:0:0/5:0:0:0/block/sr0 (block)
## ACTION=change
    DEVLINKS=/dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0 /dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0 /dev/cdrom
    DEVNAME=/dev/sr0
## DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host5/target5:0:0/5:0:0:0/block/sr0   <----'218400' used to find out at which usb connector the drive is connected
## DEVTYPE=disk
## DISK_MEDIA_CHANGE=1
## ID_BUS=usb
## ID_CDROM=1
## ID_CDROM_CD=1
## ID_CDROM_DVD=1
## ID_CDROM_MRW=1
## ID_CDROM_MRW_W=1
    ID_INSTANCE=0:0
    ID_MODEL=FD2-3500
    ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
    ID_MODEL_ID=0011
    ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
    ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
    ID_REVISION=0.01
## ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
    ID_SERIAL_SHORT=000000000000
    ID_TYPE=cd
    ID_USB_DRIVER=usb-storage
    ID_USB_INTERFACES=:080250:
    ID_USB_INTERFACE_NUM=00
## ID_VENDOR=JVCKWD
    ID_VENDOR_ENC=JVCKWD\x20\x20
    ID_VENDOR_ID=2166
    MAJOR=11
    MINOR=0
    SEQNUM=1852
## SUBSYSTEM=block
    SYSTEMD_READY=0
    TAGS=:systemd:
    UDISKS_PRESENTATION_NOPOLICY=0
    USEC_INITIALIZED=65061036204


    //=========================================================================================
    if Optical disc CDROM is inserted we do see this: Elements marked with '***' which are used to differentiate this from DVD-Drive notification
    //=========================================================================================
    UDEV  [66253.045761] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host5/target5:0:0/5:0:0:0/block/sr0 (block)
    ACTION=change
    DEVLINKS=/dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0 /dev/disk/by-label/Kinderlieder_Ru /dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0 /dev/disk/by-uuid/2007-09-23-11-34-00-00 /dev/cdrom
    DEVNAME=/dev/sr0
    DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host5/target5:0:0/5:0:0:0/block/sr0
    DEVTYPE=disk
    DISK_MEDIA_CHANGE=1
    ID_BUS=usb
    ID_CDROM=1
    ID_CDROM_CD=1
    ID_CDROM_DVD=1
    ***ID_CDROM_MEDIA=1
    ***ID_CDROM_MEDIA_CD=1
    *** ID_CDROM_MEDIA_TRACK_COUNT_DATA=1
    ID_CDROM_MRW=1
    ID_CDROM_MRW_W=1
    ID_FS_APPLICATION_ID=NERO\x20BURNING\x20ROM
    ***ID_FS_LABEL=Kinderlieder_Ru
    ***ID_FS_LABEL_ENC=Kinderlieder_Ru
    ***ID_FS_TYPE=iso9660
    ***ID_FS_USAGE=filesystem
    ID_FS_UUID=2007-09-23-11-34-00-00
    ID_FS_UUID_ENC=2007-09-23-11-34-00-00
    ID_FS_VERSION=Joliet Extension
    ID_INSTANCE=0:0
    ID_MODEL=FD2-3500
    ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
    ID_MODEL_ID=0011
    ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
    ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
    ID_REVISION=0.01
    ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
    ID_SERIAL_SHORT=000000000000
    ID_TYPE=cd
    ID_USB_DRIVER=usb-storage
    ID_USB_INTERFACES=:080250:
    ID_USB_INTERFACE_NUM=00
    ID_VENDOR=JVCKWD
    ID_VENDOR_ENC=JVCKWD\x20\x20
    ID_VENDOR_ID=2166
    MAJOR=11
    MINOR=0
    SEQNUM=1854
    SUBSYSTEM=block
    TAGS=:systemd:
    UDISKS_PRESENTATION_NOPOLICY=0
    USEC_INITIALIZED=65061036204

    //==================================================================================
    //if CDDA is inserted we do see this: Elements marked with '***' which are used to differentiate this from DVD-Drive notification
    //==================================================================================
    UDEV  [518.350487] change   /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host3/target3:0:0/3:0:0:0/block/sr0 (block)
     ACTION=change
     DEVLINKS=/dev/cdrom /dev/disk/by-path/platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0 /dev/disk/by-id/usb-JVCKWD_FD2-3500_000000000000-0:0
     DEVNAME=/dev/sr0
     DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host3/target3:0:0/3:0:0:0/block/sr0
     DEVTYPE=disk
     DISK_MEDIA_CHANGE=1
     ID_BUS=usb
     ID_CDROM=1
     ID_CDROM_CD=1
     ID_CDROM_DVD=1
    *** ID_CDROM_MEDIA=1
    *** ID_CDROM_MEDIA_CD=1
    *** ID_CDROM_MEDIA_TRACK_COUNT_AUDIO=16
     ID_CDROM_MRW=1
     ID_CDROM_MRW_W=1
     ID_INSTANCE=0:0
     ID_MODEL=FD2-3500
     ID_MODEL_ENC=FD2-3500\x20\x20\x20\x20\x20\x20\x20\x20
     ID_MODEL_ID=0011
     ID_PATH=platform-ci_hdrc.0-usb-0:1:1.0-scsi-0:0:0:0
     ID_PATH_TAG=platform-ci_hdrc_0-usb-0_1_1_0-scsi-0_0_0_0
     ID_REVISION=0.01
     ID_SERIAL=JVCKWD_FD2-3500_000000000000-0:0
     ID_SERIAL_SHORT=000000000000
     ID_TYPE=cd
     ID_USB_DRIVER=usb-storage
     ID_USB_INTERFACES=:080250:
     ID_USB_INTERFACE_NUM=00
     ID_VENDOR=JVCKWD
     ID_VENDOR_ENC=JVCKWD\x20\x20
     ID_VENDOR_ID=2166
     MAJOR=11
     MINOR=0
     SEQNUM=1812
     SUBSYSTEM=block
     TAGS=:systemd:
     UDISKS_PRESENTATION_NOPOLICY=0
     USEC_INITIALIZED=120613824

    */

    int iRet = DEVICEMANAGER_ERROR_NODEVICE;

    const char *strSyspath;
    const char* strSysname;
    const char* strDevnode;

    //to recognize DVDDrive - see comment above
    const char* strDEVTYPE;
    const char* strDISK_MEDIA_CHANGE;
    const char* strID_BUS;
    const char* strID_CDROM;
    const char* strID_CDROM_CD;
    const char* strID_CDROM_DVD;
    const char* strID_CDROM_MRW;
    const char* strID_CDROM_MRW_W;
    const char *strID_VENDOR;
    const char *strID_VENDOR_ID;
    const char *strID_MODEL;
    const char *strID_MODEL_ID;

    //info to distinguish from CDROM inserted or DVD-Video,Audio,Rom
    const char *strID_CDROM_MEDIA;
    const char *strID_CDROM_MEDIA_CD;
    const char *strID_CDROM_MEDIA_TRACK_COUNT_DATA;

    const char *strID_FS_LABEL;
    const char *strID_FS_LABEL_ENC;
    const char *strID_FS_TYPE;
    const char *strID_FS_USAGE;

    //info to distinguish from CDDA inserted
    //const char *strID_CDROM_MEDIA; already done above
    //const char *strID_CDROM_MEDIA_CD; already done above
    const char *strID_CDROM_MEDIA_TRACK_COUNT_AUDIO;

    //additional info
    const char *strID_REVISION;
    const char *strID_SERIAL;

    struct udev_device *l_device;


    if(m_pBaseUSBDev)
    {
        strSyspath     = udev_device_get_syspath(m_pBaseUSBDev);
        strSysname     = udev_device_get_sysname(m_pBaseUSBDev);
        strDevnode     = udev_device_get_devnode(m_pBaseUSBDev);

        ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:strSyspath           :%s",strSyspath));
        ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:strSysname           :%s",strSysname));
        ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:strDevnode           :%s",strDevnode));

        l_device = udev_device_new_from_syspath(m_pUdev /*udev-lib context*/,GENSTRING(strSyspath).toStdString().c_str());
        if(l_device)
        {
            //to recognize DVDDrive

            strID_VENDOR_ID                     = udev_device_get_property_value(l_device,"ID_VENDOR_ID");
            strID_MODEL_ID                      = udev_device_get_property_value(l_device,"ID_MODEL_ID");

            //NCG3D-144806 :check if the drive is a KENWOOD drive.
            if(TRUE == ConfigurationManager::GetInstance()->bFilterIsInternalDvdDrive((uint32_t)GENSTRING(strID_VENDOR_ID).toInt(),(uint32_t)GENSTRING(strID_MODEL_ID).toInt()))
            {
                strDEVTYPE                          = udev_device_get_property_value(l_device,"DEVTYPE");
                strDISK_MEDIA_CHANGE                = udev_device_get_property_value(l_device,"DISK_MEDIA_CHANGE");
                strID_BUS                           = udev_device_get_property_value(l_device,"ID_BUS");
                strID_CDROM                         = udev_device_get_property_value(l_device,"ID_CDROM");
                strID_CDROM_CD                      = udev_device_get_property_value(l_device,"ID_CDROM_CD");
                strID_CDROM_DVD                     = udev_device_get_property_value(l_device,"ID_CDROM_DVD");
                strID_CDROM_MRW                     = udev_device_get_property_value(l_device,"ID_CDROM_MRW");
                strID_CDROM_MRW_W                   = udev_device_get_property_value(l_device,"ID_CDROM_MRW_W");
                strID_VENDOR                        = udev_device_get_property_value(l_device,"ID_VENDOR");
                strID_MODEL                         = udev_device_get_property_value(l_device,"ID_MODEL");
                //info to distinguish Drive from inserted disc
                strID_CDROM_MEDIA                   = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA");
                strID_CDROM_MEDIA_CD                = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA_CD");
                strID_CDROM_MEDIA_TRACK_COUNT_DATA  = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA_TRACK_COUNT_DATA");
                strID_FS_LABEL                      = udev_device_get_property_value(l_device,"ID_FS_LABEL");
                strID_FS_LABEL_ENC                  = udev_device_get_property_value(l_device,"ID_FS_LABEL_ENC");
                strID_FS_TYPE                       = udev_device_get_property_value(l_device,"ID_FS_TYPE");
                strID_FS_USAGE                      = udev_device_get_property_value(l_device,"ID_FS_USAGE");
                strID_CDROM_MEDIA_TRACK_COUNT_AUDIO = udev_device_get_property_value(l_device,"ID_CDROM_MEDIA_TRACK_COUNT_AUDIO");

                //additional info
                strID_REVISION                      = udev_device_get_property_value(l_device,"ID_REVISION");
                strID_SERIAL                        = udev_device_get_property_value(l_device,"ID_SERIAL");

                //show what we received from udev
                ETG_TRACE_COMP(("------------Info To Recognized DVDDrive----------------------------------------------------"));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:DEVTYPE                        :%s",strDEVTYPE));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:DISK_MEDIA_CHANGE              :%s",strDISK_MEDIA_CHANGE));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_BUS                         :%s",strID_BUS));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_CDROM                       :%s",strID_CDROM));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_CDROM_CD                    :%s",strID_CDROM_CD));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_CDROM_DVD                   :%s",strID_CDROM_DVD));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_CDROM_MRW                   :%s",strID_CDROM_MRW));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: DVDDrive:ID_CDROM_MRW_W                 :%s",strID_CDROM_MRW_W));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_VENDOR                               :%s",strID_VENDOR));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_VENDOR_ID                            :%s",strID_VENDOR_ID));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_MODEL                                :%s",strID_MODEL));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_MODEL_ID                             :%s",strID_MODEL_ID));



                ETG_TRACE_COMP(("------------Info(Expected to be empty - otherwhise it would be an inserted optical disc-----"));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_CDROM_MEDIA                           :%s",strID_CDROM_MEDIA));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_CDROM_MEDIA_CD                        :%s",strID_CDROM_MEDIA_CD));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_FS_LABEL                              :%s",strID_FS_LABEL));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_FS_LABEL_ENC                          :%s",strID_FS_LABEL_ENC));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_FS_TYPE                               :%s",strID_FS_TYPE));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_FS_USAGE                              :%s",strID_FS_USAGE));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_FS_TYPE                               :%s",strID_FS_TYPE));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_CDROM_MEDIA_TRACK_COUNT_DATA          :%s",strID_CDROM_MEDIA_TRACK_COUNT_DATA));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive:ID_CDROM_MEDIA_TRACK_COUNT_AUDIO         :%s",strID_CDROM_MEDIA_TRACK_COUNT_AUDIO));
                ETG_TRACE_COMP(("----------------------Info additional-----------------------------------------------"));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_REVISION                             :%s",strID_REVISION));
                ETG_TRACE_COMP(("FillCurrDeviceDVDDrive: ID_SERIAL                               :%s",strID_SERIAL));
                ETG_TRACE_COMP(("------------------------------------------------------------------------------------"));

                //Note if in future another drive has to be taken into account this check functionality should be bound to configuration manager
                //-----------------------------------------
                //check if it is really notification of DVDDrive
                //-----------------------------------------
                tBool isDVDDrive = TRUE;

                if(isDVDDrive == TRUE){ if(GENSTRING(strDEVTYPE)           != GENSTRING("disk"))    {isDVDDrive = FALSE;ETG_TRACE_USR4(("strDEVTYPE           - no match"))} }
                //if(isDVDDrive == TRUE){ if(GENSTRING(strDISK_MEDIA_CHANGE) != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strDISK_MEDIA_CHANGE - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_BUS)            != GENSTRING("usb"))     {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_BUS            - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_CDROM)          != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_CDROM          - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_CDROM_CD)       != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_CDROM_CD       - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_CDROM_DVD)      != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_CDROM_DVD      - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_CDROM_MRW)      != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_CDROM_MRW      - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_CDROM_MRW_W)    != GENSTRING("1"))       {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_CDROM_MRW_W    - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_VENDOR)         != GENSTRING("JVCKWD"))  {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_VENDOR         - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_VENDOR_ID)      != GENSTRING("2166"))    {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_VENDOR_ID      - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_MODEL)          != GENSTRING("FD2-3500")){isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_MODEL          - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_MODEL_ID)       != GENSTRING("0011"))    {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_MODEL_ID       - no match"))} }
                if(isDVDDrive == TRUE){ if(GENSTRING(strID_SERIAL).isEmpty() == true)               {isDVDDrive = FALSE;ETG_TRACE_USR4(("strID_SERIAL - not empty"))} }

                if(TRUE == isDVDDrive)
                {
                    ETG_TRACE_USR1(("FillCurrDeviceDVDDrive: Could be DVDDrive"));
                }
                else
                {
                    ETG_TRACE_USR4(("FillCurrDeviceDVDDrive: Could be DVDDrive"));
                }


                //check if one of those not NULL or empty in such case it relates to the disc inserted
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_CDROM_MEDIA).isEmpty())                   {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_CDROM_MEDIA not emtpy")) }}   //Hint: if String is Null GENSTRING is cleared to be emtpy
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_CDROM_MEDIA_CD).isEmpty())                {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_CDROM_MEDIA_CD not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_CDROM_MEDIA_TRACK_COUNT_DATA).isEmpty())  {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_CDROM_MEDIA_TRACK_COUNT_DATA not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_FS_LABEL).isEmpty())                      {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_FS_LABEL not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_FS_LABEL_ENC).isEmpty())                  {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_FS_LABEL_ENC not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_FS_TYPE).isEmpty())                       {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_FS_TYPE not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_FS_USAGE).isEmpty())                      {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_FS_USAGE not emtpy"))}}
                if(isDVDDrive == TRUE){ if(false == GENSTRING(strID_CDROM_MEDIA_TRACK_COUNT_AUDIO).isEmpty()) {isDVDDrive=FALSE;ETG_TRACE_USR4(("strID_CDROM_MEDIA_TRACK_COUNT_AUDIO not emtpy"))}}

                if(TRUE == isDVDDrive)
                {
                    m_iNumOfDVDDriveNotification++;
                    ETG_TRACE_USR1(("FillCurrDeviceDVDDrive: DVDDrive recognized: m_iNumOfDVDDriveNotification:%d",m_iNumOfDVDDriveNotification));
                    iRet = DEVICEMANAGER_OK;
                }
                else
                {
                    ETG_TRACE_USR4(("FillCurrDeviceDVDDrive: Information bound to inserted Disc: m_iNumOfDVDDriveNotification:%d",m_iNumOfDVDDriveNotification));
                    if(m_iNumOfDVDDriveNotification==0)
                    {
                        m_iNumOfDVDDriveNotification++;
                        iRet = DEVICEMANAGER_OK;
                    }

                }

                //Fill CDevice* f_pCurDevice,
                if(iRet == DEVICEMANAGER_OK)
                {
                    //fill in information
                    f_pCurDevice->m_eDeviceType   = CGlobalEnumerations::DTY_DVD_DRIVE;
                    f_pCurDevice->m_cDevpath      = strSyspath;
                    f_pCurDevice->m_cSysName      = strSysname;
                    f_pCurDevice->m_cDevnode      = strDevnode;
                    f_pCurDevice->m_cBlockDevnode = f_pCurDevice->m_cDevnode; //bugfix: to prevent notification every 10 sec triggered by observer mechanism
                    f_pCurDevice->m_cSerialID     = strID_SERIAL;
                    f_pCurDevice->m_cShortSerial  = strID_SERIAL;

                    f_pCurDevice->m_u64Size = 0;
                    f_pCurDevice->m_iVendorID      = (uint16_t)GENSTRING(strID_VENDOR_ID).toInt();
                    f_pCurDevice->m_iProductID     = (uint16_t)GENSTRING(strID_MODEL_ID).toInt();
                    f_pCurDevice->m_iDevNUM        = 0xAA;
                    f_pCurDevice->m_cDeviceName    = "DVD Drive (USB connected)";


                    ConfigurationManager::GetInstance()->vGetUSBConnectorFromSysPath(OUT f_pCurDevice->m_eUSB, IN f_pCurDevice->m_cDevpath.toStdString());
                    f_pCurDevice->m_iUSBConnector  = (int)f_pCurDevice->m_eUSB;
                    f_pCurDevice->m_iPortNumber    = (int)f_pCurDevice->m_eUSB;
                    f_pCurDevice->m_cUSBPort       = GENSTRING((int)f_pCurDevice->m_eUSB).toStdString().c_str();//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors
                    f_pCurDevice->m_iHubIndex      = f_pCurDevice->m_iUSBConnector; //check this
                    f_pCurDevice->m_cParentSerialId = "-";

                    GENSTRING strMountPoint = GENSTRING(f_pCurDevice->m_cDevnode);
                    f_pCurDevice->m_cMountPoint = strMountPoint;

                    f_pCurDevice->m_cAccessoryName = "Optical Disc Drive";
                    f_pCurDevice->m_cDeviceVersion = GENSTRING(strID_REVISION);

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
                    /* Work for sending the ID_REVISION as DriveVersion for DVD Drive... NCG3D-41791 */
                    trOpticalDiskCDInfo l_OpticalDiskCDInfo;
                    l_OpticalDiskCDInfo.OpticalDeviceState = DEVICE_READY;         // Set CD Device State DEFAULT (Not used by HMI for this instance)
                    ETG_TRACE_USR4(("FillCurrDeviceDVDDrive Filling the Optical drive CD Info DriveVersion with ID_REVISION"));
                    memset(l_OpticalDiskCDInfo.OpticalDriveVersion,0,DVMGR_MAX_DRIVEVERSIONLENGTH);
                    if(strID_REVISION)
                    {
                        strncpy(l_OpticalDiskCDInfo.OpticalDriveVersion,strID_REVISION,  sizeof( l_OpticalDiskCDInfo.OpticalDriveVersion)-1);   // Set Drive Version as strID_REVISION
                        l_OpticalDiskCDInfo.OpticalDriveVersion[sizeof( l_OpticalDiskCDInfo.OpticalDriveVersion)-1] = '\0'; // sim4hi: this line and the line above: fix for Coverity CID-128647
                    }
                    ETG_TRACE_USR4(("FillCurrDeviceDVDDrive Sending the Optical drive CD Info"));
                    m_pStateTable->vSetOpticalDiscCDInfoNotified(enOptDiscCDInfoNotified, l_OpticalDiskCDInfo);
#endif
                    f_pCurDevice->Show();

                }
            }
            else
            {
                ETG_TRACE_USR4(("FillCurrDeviceDVDDrive Information not bound to internal DVD drive"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceDVDDrive:l_device== NULL "));
        }

    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceDVDDrive:m_pBaseUSBDev == NULL "));
    }







    ETG_TRACE_USR4 (("End  : FillCurrDeviceDVDDrive (returns with value %i", ETG_ENUM(DEVICEMANAGER_ERROR, iRet) ));

    return iRet;


}



/*-----------------------------------------------------------------------------*
 * int FillCurrDeviceSDInternal(...)                                   *
 *-----------------------------------------------------------------------------*/
int CUDevManager::FillCurrDeviceCDROM(CDevice* f_pCurDevice, GENSTRING f_cDevNode) //adjust for Roadmap 13001_ExtendedUserInformation
{
    ///@todo check for memory leak

    tU64  u64SizeInBytes;

    const char *strSyspath;
    const char* strSysname;
    const char* strDevnode;

    const char *strName;
    const char *strFSType;
    const char *strID_VENDOR;
    const char *strID_REVISION;
    const char *strID_MODEL;
    const char *strID_VENDOR_ID;
    const char *strID_MODEL_ID;

    struct udev_device *l_device;



    ETG_TRACE_USR4(("Begin: FillCurrDeviceCDROM"));

    f_pCurDevice->m_eDeviceType     = CGlobalEnumerations::DTY_CDROM;
    f_pCurDevice->m_bConnectedAtHub = false;
    f_pCurDevice->m_cAccessoryName  = f_cDevNode.toStdString().c_str();

    ETG_TRACE_FATAL(("FillCurrDeviceCDROM:m_cBlockDevnode:%s",f_pCurDevice->m_cBlockDevnode.toStdString().c_str()));
    ETG_TRACE_FATAL(("FillCurrDeviceCDROM:f_cDevNode:%s",f_cDevNode.toStdString().c_str()));


//USING UDEV instead of BLK for filling m_cDeviceName and m_cFSType
/*
    ///@todo check if performance of blkid is ok
    blkid_probe pr = blkid_new_probe_from_filename(f_cDevNode.toStdString().c_str());
    const char *strName;
    const char *strData;
    size_t      lenOfStr;

    if(pr)
    {
        if(blkid_do_probe(pr) == 0 )
        {
            int nvals = blkid_probe_numof_values(pr);
            ETG_TRACE_FATAL(("nvals:%d",nvals));
            for (int n = 0; n < nvals; n++)
            {
                if (blkid_probe_get_value(pr, n, &strName, &strData, &lenOfStr) == 0)
                {
                    ETG_TRACE_FATAL(("FillCurrDeviceCDROM: strName:%s",strName));
                    ETG_TRACE_FATAL(("FillCurrDeviceCDROM: strData:%s",strData));

                    if(0 == strcmp(strName,"LABEL"))
                    {
                        f_pCurDevice->m_cDeviceName = strData;
                    }
                    else if(0 == strcmp(strName,"TYPE"))
                    {
                        f_pCurDevice->m_cFSType = strData;
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
                        vMapFileSystemType(INOUT f_pCurDevice);
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
                        FillStorageInfoSizeTUF(f_pCurDevice);
#endif //USE_TOTAL_USED_FREE_SIZE
#ifdef USE_PARITIONNUMBER
                        f_pCurDevice->m_iValidPartitionNo = 1; //Assume IS9660 and udf offer one partition
#endif


                    }
                }
            }

        }
        blkid_free_probe(pr);
    }
    else
    {
        ETG_TRACE_FATAL(("FillCurrDeviceCDROM:blkid_new_probe_from_filename failed"));
    }
*/

    if(m_pBaseUSBDev)
    {

        strSyspath     = udev_device_get_syspath(m_pBaseUSBDev);
        strSysname     = udev_device_get_sysname(m_pBaseUSBDev);
        strDevnode     = udev_device_get_devnode(m_pBaseUSBDev);

        ETG_TRACE_FATAL(("FillCurrDeviceCDROM:strSyspath           :%s",strSyspath));
        ETG_TRACE_FATAL(("FillCurrDeviceCDROM:strSysname           :%s",strSysname));
        ETG_TRACE_FATAL(("FillCurrDeviceCDROM:strDevnode           :%s",strDevnode));

        //map to CDevice-elements
        f_pCurDevice->m_cDevpath      = strSyspath;
        f_pCurDevice->m_cSysName      = strSysname;
        f_pCurDevice->m_cDevnode      = strDevnode;
        f_pCurDevice->m_cBlockDevnode = f_pCurDevice->m_cDevnode; //bugfix: to prevent notification every 10 sec triggered by observer mechanism



        l_device = udev_device_new_from_syspath(m_pUdev /*udev-lib context*/,f_pCurDevice->m_cDevpath.toStdString().c_str());
        if(l_device)
        {
            bool isDriveSupported = false;
            strID_VENDOR             = udev_device_get_property_value(l_device,"ID_VENDOR");
            strID_MODEL              = udev_device_get_property_value(l_device,"ID_MODEL");
            //NCG3D-144806 :check whether the CDROM disc is inserted in a KENWOOD internal DVD drive
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
            {
                ETG_TRACE_USR1 (("FillCurrDeviceCDROM: This variant supports internal dvd drive"));
                strID_VENDOR_ID             = udev_device_get_property_value(l_device,"ID_VENDOR_ID");
                strID_MODEL_ID              = udev_device_get_property_value(l_device,"ID_MODEL_ID");
                isDriveSupported = ConfigurationManager::GetInstance()->bFilterIsInternalDvdDrive((uint32_t)GENSTRING(strID_VENDOR_ID).toInt(),(uint32_t)GENSTRING(strID_MODEL_ID).toInt());
            }
            else//NCG3D-144806 :check whether the CDROM disc is inserted in a MASCA CD drive
            {
                ETG_TRACE_USR1 (("FillCurrDeviceCDROM: This variant supports CD drive"));
                isDriveSupported = ConfigurationManager::GetInstance()->bFilterIsCdDrive(strID_VENDOR,strID_MODEL);
            }

            //NCG3D-144806 :Allow CDROM discs only when it inserted on a KENWOOD internal DVD drive or MASCA CD drive
            if(isDriveSupported == TRUE)
            {
                strName= udev_device_get_property_value(l_device,"ID_TYPE");
                strFSType = udev_device_get_property_value(l_device,"ID_FS_TYPE");

                ETG_TRACE_FATAL(("FillCurrDeviceCDROM: FROM UDEV strFSType:%s",strFSType));
                f_pCurDevice->m_cDeviceName = strName;
                ETG_TRACE_FATAL(("FillCurrDeviceCDROM: FROM AUTOMOUNTER ID_FS_TYPE:%s",f_pCurDevice->m_cFSType.toStdString().c_str()));
                if(f_pCurDevice->m_cFSType.isEmpty())
                {
                    f_pCurDevice->m_cFSType = strFSType;
                    ETG_TRACE_FATAL(("FillCurrDeviceCDROM: FSType is empty from automounter. Filling FSType from UDEV call"));
                }
                ETG_TRACE_FATAL(("FillCurrDeviceCDROM: ID_TYPE:%s",f_pCurDevice->m_cDeviceName.toStdString().c_str()));
                ETG_TRACE_FATAL(("FillCurrDeviceCDROM: ID_FS_TYPE:%s",f_pCurDevice->m_cFSType.toStdString().c_str()));
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
                vMapFileSystemType(INOUT f_pCurDevice);
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
                FillStorageInfoSizeTUF(f_pCurDevice);
#endif //USE_TOTAL_USED_FREE_SIZE
#ifdef USE_PARITIONNUMBER
                f_pCurDevice->m_iValidPartitionNo = 1; //Assume IS9660 and udf offer one partition
#endif



                strID_REVISION           = udev_device_get_property_value(l_device,"ID_REVISION");

                ETG_TRACE_FATAL(("FillCurrDeviceCDROM:ID_VENDOR            :%s",strID_VENDOR));
                ETG_TRACE_FATAL(("FillCurrDeviceCDROM:ID_MODEL             :%s",strID_MODEL));
                ETG_TRACE_FATAL(("FillCurrDeviceCDROM:ID_REVISION          :%s",strID_REVISION));

#ifdef USE_RANDOMID_CDROM
                //map to CDevice-elements
                srand((unsigned int)time(NULL));
                char cSerial[256]; //@todo intermediate solution - has to be replaced by scsi commands
                snprintf(cSerial,sizeof(cSerial),"TESTID:%d",rand()); //@todo intermediate solution
                f_pCurDevice->m_cSerialID = cSerial;
#endif
                f_pCurDevice->m_cShortSerial = f_pCurDevice->m_cSerialID;

                f_pCurDevice->m_cDeviceVersion = GENSTRING("DRIVE: ") + GENSTRING("ID_VENDOR:") +GENSTRING(strID_VENDOR)+ GENSTRING("_ID_MODEL:") + GENSTRING(strID_MODEL) + GENSTRING("_ID_REVISION:") + GENSTRING(strID_REVISION);
            }
            else
            {
                ETG_TRACE_USR1 (("FillCurrDeviceCDROM: CD device is not pure CD device"));
                return DEVICEMANAGER_ERROR_NOTVALID;
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceCDROM:l_device== NULL "));
            return DEVICEMANAGER_ERROR_NOTVALID;
        }

    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]FillCurrDeviceCDROM:m_pBaseUSBDev == NULL "));
        return DEVICEMANAGER_ERROR_NOTVALID;
    }


    // fill other entries with dummy //@todo cleanup
    u64SizeInBytes = (tU64)0x0; //@todo no access to info via udev
    f_pCurDevice->m_u64Size = u64SizeInBytes;
    f_pCurDevice->m_iVendorID      = 0xFEED;
    f_pCurDevice->m_iProductID     = 0xBEAF;
    f_pCurDevice->m_iDevNUM        = 0xAA;
#ifdef BUGFIX_DECOUPLE_CDROM_PRM
    f_pCurDevice->m_cUSBPort      = USBConnectorUndefStr;
    f_pCurDevice->m_iUSBConnector = USBConnectorUndef;
    f_pCurDevice->m_eUSB          = eUSBUndef;
    f_pCurDevice->m_iPortNumber   = 0;
#else
    f_pCurDevice->m_cUSBPort      = USBConnector1str;//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors
    f_pCurDevice->m_iUSBConnector = USBConnector1;//@todo here I have to find a solution which is also allowed to differentiate it from USBconnectors
    f_pCurDevice->m_eUSB          = eUSB1;
    f_pCurDevice->m_iUSBConnector = (int)eUSB1;
    f_pCurDevice->m_iPortNumber   = 1;
#endif


    //f_pCurDevice->m_cSerialID.replace (" ", "_");
    f_pCurDevice->m_cManufacturer = "Test:Manufacturer";//GENSTRING(l_sMfg).trimmed();


    f_pCurDevice->m_cParentSerialId = "-";
    f_pCurDevice->m_iHubIndex = f_pCurDevice->m_iUSBConnector; ///@todo check this

    //#ifdef HELPER_ANALYSIS_UDEVEVENTS //@todo think how to add this
    //    DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, f_pCurDevice);
    //#endif // HELPER_ANALYSIS_UDEVEVENTS

    f_pCurDevice->Show();
    ETG_TRACE_USR4 (("End  : FillCurrDevice (returns with value %i", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_OK) ));

    return DEVICEMANAGER_OK;

}//lint !e429


/*-----------------------------------------------------------------------------*
 * void GetAllConnectedDevices(vector<CDevice *> &f_vConnectedDevices)         *
 *-----------------------------------------------------------------------------*/
void CUDevManager::GetAllConnectedDevices(vector<CDevice *> &f_vConnectedDevices) const
{
    ETG_TRACE_USR4(("Begin: GetAllConnectedDevices"));

    DeviceListManager::GetInstance()->getAllConnectedDevices(DEVLIST_UDEVMGR,f_vConnectedDevices);

    ETG_TRACE_USR4(("End  : GetAllConnectedDevices"));
}



/*-----------------------------------------------------------------------------*
 * int DetectBlockDevice(CDevice *f_pCurDevice)                                *
 *-----------------------------------------------------------------------------*/
int CUDevManager::DetectBlockDevice(CDevice* f_pCurDevice)
{
    GENSTRING emptyVal = "";
    return DetectBlockDevice(f_pCurDevice,CGlobalEnumerations::enUnotify,emptyVal/*empty GENSTRING f_cDevNode*/);
}

int CUDevManager::DetectBlockDevice(CDevice* f_pCurDevice,CGlobalEnumerations::NOTIFICATION_Type enNotiType,GENSTRING f_cDevNode)

{
    ETG_TRACE_USR4((" Begin:DetectBlockDevice"));

    // Variable to hold the return value
    int l_iISUSBBlokDeviceFound = DEVICEMANAGER_ERROR_NODEVICE;


    if (NULL != f_pCurDevice)
    {
        ETG_TRACE_USR4(("DetectBlockDevice:f_pCurDevice is not null"));

        // if m_pDev is an 'partition' generate the parent 'disk'
        if (GENSTRING("partition") ==  GENSTRING(udev_device_get_devtype(m_pDev)))
        {
            if(CGlobalEnumerations::enInotify == enNotiType)
            {
                /*f_cDevNode is e.g. sda1,sda2 or sdb1, or sdb2 - parent is sda or sdb*/
                ETG_TRACE_USR3 (("DetectBlockDevice: Generating 'disk' for partition: f_cDevNode:%s", f_cDevNode.toStdString().c_str()));
                /*check if /dev/sd exists*/
                int lastindex = f_cDevNode.lastIndexOf(GENSTRING("/dev/sd"),0);
                ETG_TRACE_COMP (("DetectBlockDevice: %d", lastindex));

                if(0 == lastindex)
                {
                    f_pCurDevice->m_cBlockDevnode = f_cDevNode.left(8); /*len of /dev/sd is 7 plus 1 gives the parent*/
                }
                else
                {
                    f_pCurDevice->m_cBlockDevnode = ""; /*empty*/
                }

            }
            else /*if enUnotify*/
            {

                ETG_TRACE_USR3 (("DetectBlockDevice: Generating 'disk' for partition %s", f_pCurDevice->m_cDevnode.toStdString().c_str()));

                struct udev_device *l_pDiskDev = udev_device_get_parent_with_subsystem_devtype(m_pDev, "block", "disk");

                f_pCurDevice->m_cBlockDevnode = GENSTRING(udev_device_get_devnode(l_pDiskDev));
            }
        }
        else
        {
            if(CGlobalEnumerations::enInotify == enNotiType)
            {
                /*f_cDevNode is e.g. sda,sdb it is not necessary to search the parent- parent is sda or sdb*/
                ETG_TRACE_USR3 (("DetectBlockDevice: Generating 'disk' for partition %s", f_cDevNode.toStdString().c_str()));
                f_pCurDevice->m_cBlockDevnode = f_cDevNode;

            }
            else /*if enUnotify*/
            {
                f_pCurDevice->m_cBlockDevnode = GENSTRING(udev_device_get_devnode(m_pDev));
            }
        } // if (device is 'partition')

        // Store the disk node i.e. /dev/sdX as the devnode
        f_pCurDevice->m_cDevnode = f_pCurDevice->m_cBlockDevnode;


        ETG_TRACE_USR3 (("DetectBlockDevice: Block Device node : %s",  f_pCurDevice->m_cBlockDevnode.toStdString().c_str()));
        ETG_TRACE_USR3 (("DetectBlockDevice: Device node       : %s",  f_pCurDevice->m_cDevnode.toStdString().c_str()));

        // Valid storage interface found
        // Find Partition table type
        f_pCurDevice->m_ePTType = FindPartitionTableType(f_pCurDevice->m_cBlockDevnode.toStdString().c_str());

        // Check for first valid partition

        l_iISUSBBlokDeviceFound = CheckForValidPartition(f_pCurDevice);
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        vMapFileSystemType(f_pCurDevice);
#endif

        if (DEVICEMANAGER_OK == l_iISUSBBlokDeviceFound)
        {
            ETG_TRACE_USR4 (("DetectBlockDevice: Valid Partition      : %s", f_pCurDevice->m_cDevnode.toStdString().c_str()));
            ETG_TRACE_USR4 (("DetectBlockDevice: Valid Partition type : %s", f_pCurDevice->m_cFSType.toStdString().c_str()));

#ifdef USE_PARITIONNUMBER
            // partition
#endif

            AddPartitionUUIDToSerialNumer(f_pCurDevice); //boudn to eConf_ENABLE_NOTIFY_PARTITIONS_ALL

        }
        else
        {
            ETG_TRACE_USR4 (("DetectBlockDevice: No valid partition found"));

            l_iISUSBBlokDeviceFound = DEVICEMANAGER_ERROR_NOTVALID;
        }

        // Check whether the partition is mounted
        if (DEVICEMANAGER_OK == l_iISUSBBlokDeviceFound)
        {

            l_iISUSBBlokDeviceFound = CheckIfMounted("/proc/mounts", f_pCurDevice);

            ETG_TRACE_USR4(("DetectBlockDevice: l_iISUSBBlokDeviceFound=0x%x",l_iISUSBBlokDeviceFound));
            ETG_TRACE_USR4(("DetectBlockDevice: f_pCurDevice->m_cDeviceUUID=%s",f_pCurDevice->m_cDeviceUUID.toStdString().c_str()));

            if (DEVICEMANAGER_OK != l_iISUSBBlokDeviceFound)
            {
                ETG_TRACE_USR1(("DetectBlockDevice: Unable to check if mounted"));

                l_iISUSBBlokDeviceFound = DEVICEMANAGER_ERROR_NOTVALID;
            }
            else
            {
                if (1 == f_pCurDevice->m_iIsMounted)
                {
                    ETG_TRACE_USR1(("DetectBlockDevice: Block Device found and Mounted"));

                    l_iISUSBBlokDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;

                    f_pCurDevice->m_u64Size = FillStorageInfoSize( m_pDev );


#ifdef USE_TOTAL_USED_FREE_SIZE
                    FillStorageInfoSizeTUF(f_pCurDevice);
#endif //USE_TOTAL_USED_FREE_SIZE


                }

            } // Fill CDevice details
        } // if (valid partition found)

        // Check whether the device is an SD Card
        if (DEVICEMANAGER_DEVICE_DETECTED == l_iISUSBBlokDeviceFound)
        {
            if ((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_SD)) && true == IsSDCard (this->m_pBaseUSBDev)) //call function to check if SDCard @HUB i.e. card reader is used. SDCard internal is treaated different
            {
                ETG_TRACE_USR4(("DetectBlockDevice: Device Detected is SD card"));

                f_pCurDevice->m_eDeviceType     = CGlobalEnumerations::DTY_SD;
                f_pCurDevice->m_bIsSdCardDevice = true;
            }
            else
            {
                ETG_TRACE_USR4(("DetectBlockDevice: Device Detected is not SD card"));

                f_pCurDevice->m_eDeviceType     = CGlobalEnumerations::DTY_USB;
                f_pCurDevice->m_bIsSdCardDevice = false;
            } // if (block device is SD card)
        } // if (block device is found and mounted)
    }
    else
    {
        ETG_TRACE_USR4(("DetectBlockDevice:f_pCurDevice is null"));

        l_iISUSBBlokDeviceFound = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if (f_pCurDevice is not NULL)

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
    // Change the device type to not supported is a problem was found -> to notify CCA clients
    if(   l_iISUSBBlokDeviceFound != DEVICEMANAGER_DEVICE_DETECTED
          && f_pCurDevice )
    {
        ETG_TRACE_USR4(("DetectBlockDevice: Not handle DTY_USB device detected. Change f_pCurDevice->m_eDeviceType to DTY_NOT_SUPPORTED and return value to DEVICEMANAGER_DEVICE_DETECTED" ));
        f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_NOT_SUPPORTED;
        l_iISUSBBlokDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
    }
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
    ETG_TRACE_USR4(("End  : DetectBlockDevice"));

    return l_iISUSBBlokDeviceFound;
}





/*-----------------------------------------------------------------------------*
 * int PushDeviceData(CDevice* f_pCurDevice)                                   *
 *-----------------------------------------------------------------------------*/
int CUDevManager::PushDeviceData( /*const*/ CDevice* f_pCurDevice)
{
    ETG_TRACE_USR3 (("PushDeviceData: Begin"));

    int  l_DeviceNotFound = DEVICEMANAGER_DEVICE_DETECTED;

    // Check for invalid parameters
    if (NULL != f_pCurDevice)
    {
        unsigned int l_uiCount;
        unsigned int uiSize;
        CDevice *l_poDevice = NULL;

        ETG_TRACE_USR2(("PushDeviceData: f_pCurDevice->m_cSerialID             = %s",
                        f_pCurDevice->m_cSerialID.toStdString().c_str()));

        CDevice oDevice;
        tBool bExists;

        uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);

        for (l_uiCount = 0; ((l_uiCount < uiSize)
                             && (DEVICEMANAGER_ERROR_NODEVICE != l_DeviceNotFound));l_uiCount++)
        {
            bExists = DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount, oDevice);
            NORMAL_M_ASSERT(TRUE==bExists);

            if (TRUE == bExists)
            {
                l_poDevice = &oDevice;

                ETG_TRACE_USR2(("PushDeviceData: m_vConnectedDevices[%2d]->m_cSerialID = %s",
                                l_uiCount, l_poDevice->m_cSerialID.toStdString().c_str()));

                //--------------------------------------------------------------------------------------
                //if serial id is the same then we assume the device is already in the list i.e. not a new one has been notified
                //-------------------------------------------------------------------------------------
                if (f_pCurDevice->m_cSerialID == l_poDevice->m_cSerialID)     //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_USR4 (("PushDeviceData: Serial found in list"));

                    l_DeviceNotFound = DEVICEMANAGER_ERROR_NODEVICE;

                    // If device is a SD card then match the mount-point as well
                    //i.e. the serial number might be the same because it could be the serial number of the card reader thus check the mount point as well ///@todo if automount from adit will be intergrated take care!
                    if ((CGlobalEnumerations::DTY_SD == f_pCurDevice->m_eDeviceType)
                            && (f_pCurDevice->m_cMountPoint != l_poDevice->m_cMountPoint))   //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
                    {
                        ETG_TRACE_USR4 (("PushDeviceData: Different SD card"));
                        l_DeviceNotFound = DEVICEMANAGER_DEVICE_DETECTED;
                    }
                }
                //special case DTY_AUX to prevent that two times same device to be pushed
                else if(CGlobalEnumerations::DTY_DIGIAUX == f_pCurDevice->m_eDeviceType) //to prevent
                {
                    ETG_TRACE_USR4(("PushDeviceData: is DTY_DIGIAUX"));

                    ETG_TRACE_USR4(("PushDeviceData: Device notified:"));
                    ETG_TRACE_USR4 (("PushDeviceData: f_pCurDevice->m_cSerialID             <%s>", f_pCurDevice->m_cSerialID.toStdString().c_str()));
                    ETG_TRACE_USR4 (("PushDeviceData: f_pCurDevice->m_iVendorID          <0x%04x>", f_pCurDevice->m_iVendorID));
                    ETG_TRACE_USR4 (("PushDeviceData: f_pCurDevice->m_iProductID         <0x%04x>", f_pCurDevice->m_iProductID));
                    ETG_TRACE_USR4 (("PushDeviceData: f_pCurDevice->m_cAccessoryName        <%s>", f_pCurDevice->m_cAccessoryName.toStdString().c_str()));

                    ETG_TRACE_USR4(("PushDeviceData: Device of list index=%d :",l_uiCount));
                    ETG_TRACE_USR4 (("PushDeviceData: l_poDevice->m_cSerialID             <%s>", l_poDevice->m_cSerialID.toStdString().c_str()));
                    ETG_TRACE_USR4 (("PushDeviceData: l_poDevice->m_iVendorID          <0x%04x>", l_poDevice->m_iVendorID));
                    ETG_TRACE_USR4 (("PushDeviceData: l_poDevice->m_iProductID         <0x%04x>", l_poDevice->m_iProductID));
                    ETG_TRACE_USR4 (("PushDeviceData: l_poDevice->m_cAccessoryName        <%s>", l_poDevice->m_cAccessoryName.toStdString().c_str()));


                    if( f_pCurDevice->m_iVendorID      == l_poDevice->m_iVendorID)
                    {
                        ETG_TRACE_USR4(("PushDeviceData: same m_iProductID "));
                        if( f_pCurDevice->m_cAccessoryName      == l_poDevice->m_cAccessoryName)
                        {
                            ETG_TRACE_USR4(("PushDeviceData: same m_cAccessoryName "));
                            if( f_pCurDevice->m_iProductID      == l_poDevice->m_iProductID)
                            {
                                ETG_TRACE_USR4(("PushDeviceData: same m_iProductID "));
                                ETG_TRACE_USR4(("PushDeviceData:  Don't push it: Found already  DTY_DIGIAUX device in the list "));
                                l_DeviceNotFound = DEVICEMANAGER_ERROR_NODEVICE; //mark it already inlist
                            }
                            else if ((f_pCurDevice->m_iProductID     == DigitalAuxProductID1)||
                                     (f_pCurDevice->m_iProductID      == DigitalAuxProductID2)||
                                     (f_pCurDevice->m_iProductID      == DigitalAuxProductID3))
                            {
                                ETG_TRACE_USR4(("PushDeviceData: m_iProductID of device in list is one of the ok ones"));
                                ETG_TRACE_USR4(("PushDeviceData:  Don't push it: Found already  DTY_DIGIAUX device in the list "));
                                l_DeviceNotFound = DEVICEMANAGER_ERROR_NODEVICE; //mark it already inlist
                            }

                        }

                    }
                }
                //extra check for to prevent that two de

            }

        }

        // Device not in the list
        if (DEVICEMANAGER_DEVICE_DETECTED == l_DeviceNotFound)
        {
            l_DeviceNotFound = ConfigurationManager::GetInstance()->FilterWithConfiguration(f_pCurDevice);
            ETG_TRACE_COMP(("PushDeviceData: f_pCurDevice->m_eDeviceUnsupportedReason = %d",f_pCurDevice->m_eDeviceUnsupportedReason));
            ETG_TRACE_COMP(("PushDeviceData: l_DeviceNotFound = %d",l_DeviceNotFound));
            if((DEVICE_OK != l_DeviceNotFound) && (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_UNKNOWN_DEVICES)))
            {
                if(DEVICE_NOK_APPLE_DEVICE_CONNECTION_WITHOUT_CUSTOMERHUB == l_DeviceNotFound)
                {
                    f_pCurDevice->m_eDeviceUnsupportedReason = AppleDeviceNotConnectedOnCustomerHub;
                }
                else
                {
                    f_pCurDevice->m_eDeviceUnsupportedReason = DisabledInConfiguration;
                }
                ETG_TRACE_COMP(("PushDeviceData: Set m_eDeviceUnsupportedReason to %d", ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,f_pCurDevice->m_eDeviceUnsupportedReason)));
                f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_NOT_SUPPORTED;
                ETG_TRACE_COMP(("PushDeviceData: Set m_eDeviceType to DTY_NOT_SUPPORTED as it is disabled in configuration"));
                l_DeviceNotFound = DEVICEMANAGER_DEVICE_DETECTED;
            }
            ConfigurationManager::GetInstance()->vCheckAdjustDeviceNameForRegion(INOUT f_pCurDevice);
            ConfigurationManager::GetInstance()->vCheckLSIMSDCardFake(INOUT f_pCurDevice);


            //+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            //THIS IS where handover to StateTable happens in principle i.e. next entity within VD_Devicemanager. StateTable decides to hand it over to Service(Interface)
            vCheckPushToList(l_DeviceNotFound, f_pCurDevice);
            //+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

            if (DEVICEMANAGER_DEVICE_DETECTED == l_DeviceNotFound)
            {
                ETG_TRACE_COMP(("------------PUSH2LIST------------/"));
                ETG_TRACE_COMP(("PushDeviceData: Device push successfully"));

                if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_GETMSC_FREE_USED_SIZE) && f_pCurDevice->m_iTotalValidPartitons  <=1 && 1 == f_pCurDevice->m_iIsMounted )
                {
                    if(0 == f_pCurDevice->m_u64TotalSize_KB)
                    {
                        ETG_TRACE_USR4(("PushDeviceData: Push the device to a vector to later fetch  the Storage info using the temporary thread"));
                        m_vDevicesWithOutStorageInfo.push_back(*f_pCurDevice);
                    }
                }


            }
            ETG_TRACE_USR2(("PushDeviceData: l_DeviceNotFound = 0x%x",l_DeviceNotFound));
            ETG_TRACE_USR2(("PushDeviceData: Device Serial ID   f_pCurDevice->m_cSerialID        = %s",
                            f_pCurDevice->m_cSerialID.toStdString().c_str()));
        }
        else
        {
            ETG_TRACE_USR2(("PushDeviceData: Device Present in list"));
        }
    }
    else
    {
        ETG_TRACE_ERR (("PushDeviceData: [ERROR] Invalid f_pCurDevice is NULL"));

        l_DeviceNotFound = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if (f_pCurDevice is NULL)

    ETG_TRACE_USR3 (("PushDeviceData: End"));

    return l_DeviceNotFound;
}

/*-----------------------------------------------------------------------------*
 * int GetParentSerialId(CDevice* f_pCurDevice)                                *
 *-----------------------------------------------------------------------------*/
int CUDevManager::GetParentSerialId(CDevice* f_pCurDevice)
{
    int l_iError = DEVICEMANAGER_OK;

    // Check for invalid parameters
    if (NULL != f_pCurDevice)
    {
        //Parent hub serial id
        GENSTRING l_cParentHubId = "";

        //Opaque object representing one entry in a list. An entry contains contains a name, and
        //optionally a value
        struct udev_list_entry *l_pListEntry;

        udev_list_entry_foreach(l_pListEntry, udev_device_get_properties_list_entry(m_pParentHUBDev))
        {
            if(!strcmp(udev_list_entry_get_name(l_pListEntry),"ID_SERIAL"))
            {
                l_cParentHubId = udev_list_entry_get_value(l_pListEntry);

                if(l_cParentHubId != f_pCurDevice->m_cSerialID &&        //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
                        false == f_pCurDevice->m_cSerialID.isEmpty())
                {
                    m_cParentHubSerial = udev_list_entry_get_value(l_pListEntry);
                    f_pCurDevice->m_cParentSerialId = m_cParentHubSerial;
                }
                else
                {
                    ETG_TRACE_USR4 (("Else m_cParentSerialId = %s  ", f_pCurDevice->m_cParentSerialId.toStdString().c_str()));
                }
            }
        }
        f_pCurDevice->m_cParentSerialId = m_cParentHubSerial;
    }
    else
    {
        ETG_TRACE_ERR (("GetParentSerialId: [ERROR] Invalid f_pCurDevice is NULL"));

        l_iError = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if (f_pCurDevice is NULL)

    return l_iError;
}//lint !e429


/*-----------------------------------------------------------------------------*
 * int DisconnectNotification(GENSTRING f_cDeviceNode,                           *
 *          CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction)           *
 *-----------------------------------------------------------------------------*/
int CUDevManager::DisconnectNotification(GENSTRING f_cDeviceNode,
                                         CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction)
{
    ETG_TRACE_USR4(("Begin: DisconnectNotification"));

    //pointer of class CDevice
    CDevice* l_pCurDevice = NULL;
    CDevice oDevice;

    //Check for device detected
    int l_DeviceNotFound = DEVICEMANAGER_ERROR_NODEVICE;

    //Pointer of class CDevice
    GENSTRING l_cDevNode = "";

#ifdef CHECK_LAST_DEVICE
    GENSTRING l_cSerialID = "";
#endif // CHECK_LAST_DEVICE

    tenDevManagerUSBHost eUsb;


    unsigned int l_uiCount = 0;

    ETG_TRACE_USR2(("DisconnectNotification: f_cDeviceNode: %s", f_cDeviceNode.toStdString().c_str()));

    //Switch case to check Device removed status
    switch(f_eAction)
    {
    case CGlobalEnumerations::SINGLE:
    {
        ETG_TRACE_USR2(("DisconnectNotification:Inside SINGLE"));

        //check if the removed device is there in the list of connected devices
        unsigned int uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);
        tBool bExists;

        for ( l_uiCount=0; l_uiCount< uiSize; l_uiCount++)
        {
            bExists=DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount, oDevice);
            if(FALSE == bExists)
            {
                ETG_TRACE_FATAL(("DisconnectNotification : bExists is False (Removed device is not present in the devicelistmanager) "));
            }
            else
            {
                l_cDevNode = oDevice.m_cDevnode;

                ETG_TRACE_USR2(("DisconnectNotification:l_cDevNode: %s",l_cDevNode.toStdString().c_str()));

                /* It may also happen that partition /dev/sdX1 removal notification comes,
                 * in that case the stored devnodes may contain the parent disk /dev/sdX.
                 * So a match is tried for the disk and the disk is removed if matched.
                 */
                //if (f_cDeviceNode == l_cDevNode)
                if (f_cDeviceNode == l_cDevNode || f_cDeviceNode.contains(l_cDevNode))    //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_USR2(("DisconnectNotification: Device node found in connected devices vector"));

                    //stack object to hold the Device information
                    l_pCurDevice = &oDevice;

#ifdef CHECK_LAST_DEVICE
                    l_cSerialID = l_pCurDevice->m_cSerialID;
#endif // CHECK_LAST_DEVICE

                    ETG_TRACE_SYS(("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -"));
                    ETG_TRACE_SYS(("DisconnectNotification:l_pCurDevice->m_iUSBConnector=USB%1d", l_pCurDevice->m_iUSBConnector));
                    ETG_TRACE_SYS(("DisconnectNotification:l_pCurDevice->m_eDeviceType=%d",       ETG_ENUM(DVM_DEVICE_TYPE,l_pCurDevice->m_eDeviceType)));
                    ETG_TRACE_SYS(("DisconnectNotification:l_pCurDevice->m_cMountPoint=%s",       l_pCurDevice->m_cMountPoint.toStdString().c_str()));
                    ETG_TRACE_SYS(("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -"));

                    DEVICEPROCESSORHISTORY.AddToHistory(m_cAction, l_pCurDevice);

                    if(TRUE == m_bConnectUDEV)
                    {
                        ETG_TRACE_USR2(("DisconnectNotification:Calling SendNotification to state machine"));

                        l_pCurDevice->eRemoveAction    = CGlobalEnumerations::SINGLE;
                        l_pCurDevice->cRemoveDeviceNode= f_cDeviceNode;
                        eUsb=m_pStateTable->GetConnector(l_pCurDevice);
                        l_pCurDevice->m_eUSB=eUsb;
                        m_pStateTable->setMount(enMountUdev,eUsb,SIG_FALSE,l_pCurDevice);
                    }
                    else
                    {
                        ETG_TRACE_COMP(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                        ETG_TRACE_COMP(("DisconnectNotification:FALSE == m_bConnectUDEV (CGlobalEnumerations::SINGLE)"));
                        ETG_TRACE_COMP(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                    }


                    ETG_TRACE_USR2(("DisconnectNotification:After SendNotification"));
                    // stop logging if enabled
                    ETG_TRACE_USR4 (("AddDevice: Check for logging"));

                    if (l_pCurDevice->m_bIsLogDump)
                    {
                        ETG_TRACE_USR1 (("DisconnectNotification: Stopping logging"));
                        HISTORYMANAGER->StopDumpHistoryToUSBStick();
                    }
                    else
                    {
                        ETG_TRACE_USR1 (("DisconnectNotification: Logging was enabled"));
                    }
                    DeviceListManager::GetInstance()->deleteElement(DEVLIST_UDEVMGR,l_uiCount); //no lock since this part is already locked
                    l_DeviceNotFound = DEVICEMANAGER_DEVICE_DETECTED;

                    break;
                }

            }

        }

        if(l_DeviceNotFound == DEVICEMANAGER_ERROR_NODEVICE)
        {
            for (l_uiCount=0; l_uiCount< m_vDisableDevices.size() ; l_uiCount++)
            {

                if (f_cDeviceNode == m_vDisableDevices[l_uiCount].m_cDevnode)   //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'

                {
                    //stack object to hold the Device information
                    l_pCurDevice = &m_vDisableDevices[l_uiCount];

#ifdef CHECK_LAST_DEVICE
                    l_cSerialID = l_pCurDevice->m_cSerialID;
#endif // CHECK_LAST_DEVICE

                    ETG_TRACE_USR2(("DisconnectNotification: Disconnection disable device"));

                    l_pCurDevice->Show();

                    if(m_cParentUsb1Hub == l_pCurDevice->m_cParentSerialId)   //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                    {
                        ETG_TRACE_USR2(("DisconnectNotification: Disconnection disable device From HUB 1"));
                        m_iUsb1HubPort--;
                    }
                    else if(m_cParentUsb2Hub == l_pCurDevice->m_cParentSerialId)   //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                    {
                        ETG_TRACE_USR2(("DisconnectNotification: Disconnection disable device From HUB 2"));
                        m_iUsb2HubPort--;
                    }

                    m_tableLock.lock();
                    m_vDisableDevices.erase(m_vDisableDevices.begin() + (int)l_uiCount );   //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'

                    m_tableLock.unlock();

                    break;
                }
            }
        }

        break;
    }

    default:
    {
        ETG_TRACE_USR2(("DisconnectNotification:Inside default"));
        break;
    }
    }

#ifdef CHECK_LAST_DEVICE
    // Remove device from vSeenDevice
    for (l_uiCount = 0; l_uiCount < m_vSeenDevices.size (); l_uiCount++)
    {
        if (l_cSerialID == m_vSeenDevices.at(l_uiCount))    //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
        {
            ETG_TRACE_USR3 (("DisconnectNotification: Removing from list of SEEN devices"));

            m_vSeenDevices.erase(m_vSeenDevices.begin()+(int)l_uiCount);     //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
        }
    }
#endif /// CHECK_LAST_DEVICE

    ETG_TRACE_USR4(("End  : DisconnectNotification"));

    return l_DeviceNotFound;
}

bool CUDevManager::IsHubAlreadyConnected(GENSTRING &sysName, CGlobalEnumerations::HUB_NUM_t &eHubNo)
{

    ETG_TRACE_USR4 (("Begin: IsHubAlreadyConnected: m_vHubs.size():%d; l_cSysName:%s ", m_vHubs.size(),sysName.toStdString().c_str() ));

    bool isHubConnected = false;

    if (sysName.isEmpty())
    {
        ETG_TRACE_ERR (("IsHubAlreadyConnected: [ERROR] sysName is EMPTY"));
    }
    else
    {
        if(m_vHubs.size() >0)
        {
            ETG_TRACE_USR4(("IsHubAlreadyConnected: -----------Check List-------------------"));
            for( unsigned int l_iDevCount = 0; l_iDevCount < m_vHubs.size(); l_iDevCount++)
            {
                if(!(m_vHubs[l_iDevCount]->sSysName.isEmpty()))
                {
                    if (sysName == m_vHubs[l_iDevCount]->sSysName)     //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                    {
                        ETG_TRACE_ERR (("IsHubAlreadyConnected: Device found in the list"));

                        eHubNo = m_vHubs[l_iDevCount]->eHubNo;
                        isHubConnected = true;
                    }
                    //else Continue checking the list
                }
                else
                {
                    ETG_TRACE_USR4(("IsHubAlreadyConnected:m_vHubs[l_iDevCount]->sysName is NULL"));
                }
            } // for (all devices in m_vHubs)
            ETG_TRACE_USR4(("IsHubAlreadyConnected: ----------------------------------------"));
        }
    } // if (f_cDeviceNode.isEmpty())

    ETG_TRACE_USR4 (("End  : IsHubAlreadyConnected isHubConnected:%d", isHubConnected));

    return isHubConnected;
}

/*-----------------------------------------------------------------------------*
 * int DisconnectNotificationHUB(GENSTRING f_cDeviceNode,                        *
 *          CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction)           *
 *-----------------------------------------------------------------------------*/
int CUDevManager::DisconnectNotificationHUB(GENSTRING f_cDeviceNode,
                                            CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction)
{
    ETG_TRACE_USR4(("Begin: DisconnectNotificationHUB"));

    int l_iRemovalSuccess = DEVICEMANAGER_OK;

    if (f_cDeviceNode.isEmpty())
    {
        ETG_TRACE_ERR (("DisconnectNotificationHUB: [ERROR] Invalid params"));

        l_iRemovalSuccess = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }
    else
    {
        int l_iHubIndex = 0;
        GENSTRING sysNameOfDeviceRemoved;

        if (CGlobalEnumerations::HUB1 == f_eAction)
        {
            l_iHubIndex = 1;
        }
        else if (CGlobalEnumerations::HUB2 == f_eAction)
        {
            l_iHubIndex = 2;
        }
        else
        {
            ETG_TRACE_ERR (("DisconnectNotificationHUB: [ERROR] Invalid Hub%d", (int)f_eAction));

            l_iRemovalSuccess = DEVICEMANAGER_ERROR_INVALID_PARAM;
        } // if (valid hub)

        if (DEVICEMANAGER_OK == l_iRemovalSuccess)
        {
            ETG_TRACE_SYS (("DisconnectNotificationHUB: Removed HUB%d", l_iHubIndex));

            CDevice *l_pCurDevice = NULL;
            unsigned int l_uiCount;
            tenDevManagerUSBHost eUsb;

            ETG_TRACE_USR2(("DisconnectNotificationHUB: Process connected devices list"));


            unsigned int uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);
            CDevice oDevice;
            tBool bExists;
            // check if the removed device is there in the list of connected devices
            for (l_uiCount=0; l_uiCount< uiSize ; l_uiCount++)
            {
                //stack object to hold the Device information
                bExists=DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount, oDevice);

                if(TRUE == bExists)
                {
                    l_pCurDevice = &oDevice;

                    if (l_iHubIndex == l_pCurDevice->m_iHubIndex && f_cDeviceNode == l_pCurDevice->m_cDevnode)
                    {

                        ETG_TRACE_SYS(("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -"));
                        ETG_TRACE_SYS(("DisconnectNotificationHUB:l_pCurDevice->m_eDeviceType=%d", ETG_ENUM(DVM_DEVICE_TYPE,l_pCurDevice->m_eDeviceType)));
                        ETG_TRACE_SYS(("DisconnectNotificationHUB:l_pCurDevice->m_cMountPoint=%s", l_pCurDevice->m_cMountPoint.toStdString().c_str()));
                        ETG_TRACE_SYS(("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -"));

                        DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, l_pCurDevice);


                        if(TRUE == m_bConnectUDEV)
                        {
                            ETG_TRACE_USR2(("DisconnectNotification:Calling SendNotification HUB%d to state machine", l_iHubIndex));

                            l_pCurDevice->eRemoveAction = f_eAction;
                            eUsb = m_pStateTable->GetConnector(l_pCurDevice);
                            l_pCurDevice->m_eUSB = eUsb;
                            l_pCurDevice->m_eConnectStatus = USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR;
                            m_pStateTable->setMount(enMountUdev, eUsb, SIG_FALSE, l_pCurDevice);
                        }
                        else
                        {
                            ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                            ETG_TRACE_FATAL(("DisconnectNotification:FALSE == m_bConnectUDEV (CGlobalEnumerations::HUB%d)", l_iHubIndex));
                            ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                        }

                        DeviceListManager::GetInstance()->deleteElement(DEVLIST_UDEVMGR,l_uiCount);
                        sysNameOfDeviceRemoved = l_pCurDevice->m_cSysName;

                    }

                }
                else
                {
                    ETG_TRACE_FATAL(("DisconnectNotificationHUB: EMPTY list: f_eAction %d f_cDeviceNode %s",
                                     ETG_CENUM(CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type, f_eAction),
                                     f_cDeviceNode.toStdString().c_str()));

                }

            }

            ETG_TRACE_USR2(("DisconnectNotificationHUB: Process disabled devices list"));

            for (l_uiCount = 0; l_uiCount < m_vDisableDevices.size(); l_uiCount++)
            {

                {
                    m_tableLock.lock();

                    m_vDisableDevices.erase(m_vDisableDevices.begin() + (int)l_uiCount );      //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'

                    m_tableLock.unlock();
                }
            }

            if (CGlobalEnumerations::HUB1 == f_eAction)
            {
                m_iUsb1HubPort = 0;
            }
            else if (CGlobalEnumerations::HUB2 == f_eAction)
            {
                m_iUsb2HubPort = 0;
            }
        }
        else
        {
            // Do nothing
        } // if (valid params)

        ETG_TRACE_USR1 (("DisconnectNotification: Removing Hub from m_vHubs"));

        CGlobalEnumerations::HubDetails_t *l_poHub = NULL;

        for (unsigned int l_uiCount = 0; l_uiCount < m_vHubs.size(); l_uiCount++)
        {
            l_poHub = m_vHubs[l_uiCount];

            if ((l_poHub != NULL) && ((int)f_eAction == (int)l_poHub->eHubNo) && (sysNameOfDeviceRemoved == l_poHub->sSysName))
            {
                ETG_TRACE_SYS (("DisconnectNotificationHUB: hub found in the list"));

                m_tableLock.lock();
                m_vHubs.erase(m_vHubs.begin() + (int)l_uiCount);    //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'

                delete l_poHub;
                l_poHub = NULL;
                m_tableLock.unlock();
            }
            else
            {
                // Continue checking the list
            }
        } // for (all devices in m_vHubs

    }

    ETG_TRACE_USR4(("End  : DisconnectNotificationHUB"));

    return l_iRemovalSuccess;
}

#ifdef USE_FOURTH_USB_CONNECTOR
/*-----------------------------------------------------------------------------*
 * void GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,         *
 *          CUSBDiagnosis *f_objUSBDiagnosisHUB1,                              *
 *          CUSBDiagnosis *f_objUSBDiagnosisHUB2,                             *
 *          CUSBDiagnosis *f_objUSBDiagnosisHUB3,                              *
 *          CUSBDiagnosis *f_objUSBDiagnosisHUB4)                              *
 *-----------------------------------------------------------------------------*/

//@todo improve to use vecto instead of 2 times object CUSBDiagnosis e.g.
//GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,vector<CUSBDiagnosis> &f_objUSBDiagnosisHUB2)
    void CUDevManager::GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,
                                          CUSBDiagnosis &f_objUSBDiagnosisHUB1,
                                          CUSBDiagnosis &f_objUSBDiagnosisHUB2,
                                          CUSBDiagnosis &f_objUSBDiagnosisHUB3,
                                          CUSBDiagnosis &f_objUSBDiagnosisHUB4)
#else
/*-----------------------------------------------------------------------------*
* void GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,         *
*          CUSBDiagnosis *f_objUSBDiagnosisHUB1,                              *
*          CUSBDiagnosis *f_objUSBDiagnosisHUB2,                             *
*          CUSBDiagnosis *f_objUSBDiagnosisHUB3)                              *
*-----------------------------------------------------------------------------*/

//@todo improve to use vecto instead of 2 times object CUSBDiagnosis e.g.
//GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,vector<CUSBDiagnosis> &f_objUSBDiagnosisHUB2)
    void CUDevManager::GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,
                                        CUSBDiagnosis &f_objUSBDiagnosisHUB1,
                                        CUSBDiagnosis &f_objUSBDiagnosisHUB2,
                                        CUSBDiagnosis &f_objUSBDiagnosisHUB3)
#endif
{
    ETG_TRACE_USR4(("Begin: GetDiagnosisValues"));

    CGlobalEnumerations::HubDetails_t* pHubDetails=NULL;

    CDevice  oCurDevice;
    unsigned long l_ulDeviceSize =0;
    unsigned int uiSize;

    CUSBDiagnosis *pObjUSBDiagnosis[ARRAYSIZEFORUSB];
    pObjUSBDiagnosis[0]    = NULL;
    pObjUSBDiagnosis[eUSB1] = &f_objUSBDiagnosisHUB1;
    pObjUSBDiagnosis[eUSB2] = &f_objUSBDiagnosisHUB2;
    pObjUSBDiagnosis[eUSB3] = &f_objUSBDiagnosisHUB3;
#ifdef USE_FOURTH_USB_CONNECTOR
    pObjUSBDiagnosis[eUSB4] = &f_objUSBDiagnosisHUB4;
#endif

    m_tableLock.lock();

    //------------------------------------------------------
    //delete udev related info of 'container' CUSBDiagnosis
    //------------------------------------------------------
    for(tInt i=(tInt)eUSB1;i<ARRAYSIZEFORUSB;i++)
    {
        ETG_TRACE_USR4(("USB%1d : GetDiagnosisValues:delete content",i));
        pObjUSBDiagnosis[i]->m_cVendor       ="";
        pObjUSBDiagnosis[i]->m_cSerialID     ="";
        pObjUSBDiagnosis[i]->m_cUSBPortNo    ="";       //e.g. either Hardwareport USB1 or 2
        pObjUSBDiagnosis[i]->m_cDeviceName   ="";
        pObjUSBDiagnosis[i]->m_cMountPoint   ="";
        pObjUSBDiagnosis[i]->m_cDeviceVersion="";
        pObjUSBDiagnosis[i]->m_ulSize          = 0;
        pObjUSBDiagnosis[i]->m_eDeviceType     = CGlobalEnumerations::DTY_UNKNOWN;

        pObjUSBDiagnosis[i]->m_eUSB_HUB_SEARCH         = CGlobalEnumerations::HUB_FOUND_NO;
        pObjUSBDiagnosis[i]->m_iUSBNoOfPortsFoundAtHub = 0;  //number of HUBs at port
        pObjUSBDiagnosis[i]->m_bHUBHasCardReader      = FALSE;
    }

    tBool bExists;

    uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);
    ETG_TRACE_USR1(("GetDiagnosisValues: m_vConnectedDevices.size()=%d",uiSize));

    //-----------------------
    //fill from list
    //-----------------------
    //check if the removed device is there in the list of connected devices
    for ( unsigned int l_uiCount=0; l_uiCount< uiSize ; l_uiCount++)
    {
        bExists=DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount,oCurDevice);    //@todo check validity
        NORMAL_M_ASSERT(TRUE == bExists);

        if(TRUE == bExists)
        {
            //production line diagnosis does need these values
            if(CGlobalEnumerations::DIAG_PDT_ONDEMAND==eDiagType)
            {
                // find the storage details
                StorageInfo_t sinfo;
#ifdef USE_BUGFIX_DIAGNOSIS_READSSIZE_MSC
                //prevent unnecessary error trace if not used
                if(TRUE == ConfigurationManager::GetInstance()->bIsMassStorageDevice(oCurDevice.m_eDeviceType))
                {
                    FillStorageInfo(oCurDevice.m_cMountPoint.toStdString().c_str(),sinfo); //tbd. check for redundant functions in this context
                }
                else
                {
                    //fill with 0
                    oCurDevice.m_lFreeSpace = 0;
                    oCurDevice.m_lUsedSpace = 0;
                    oCurDevice.m_lTotalSize = 0;
                    oCurDevice.m_lTotalSize = 0;
                }

#else
                FillStorageInfo(oCurDevice.m_cMountPoint.toStdString().c_str(),sinfo);
#endif
                // fill details into device object
                oCurDevice.m_lFreeSpace = sinfo.free_space; //without cast we get trouble i.e. cast from c to cpp
                oCurDevice.m_lUsedSpace = sinfo.used_space; //without cast we get trouble i.e. cast from c to cpp
                oCurDevice.m_lTotalSize = sinfo.total_size; //without cast we get trouble i.e. cast from c to cpp

                //size in MB (shifted to this position to keep previous behaviour after adjusting function FillStorageInfo to deliver KB)
                oCurDevice.m_lTotalSize = (unsigned long long)ceil((double) oCurDevice.m_lTotalSize / (1024 * 1024));
            }



            //@todo check for release 3 or 4 if such trace could be bound better to class CDevice
            ETG_TRACE_USR2(("-----------------------------GetDiagnosisValues- DeviceNo:%d-----------------------------------",l_uiCount+1));
            // device node like /dev/sda1 etc
            ETG_TRACE_USR2 (("Devices[%d]->m_cDevnode:        %s              ", l_uiCount, oCurDevice.m_cDevnode.toStdString().c_str()));
            // Kingston etc
            ETG_TRACE_USR2 (("Devices[%d]->m_iVendorID:       0x%04x          ", l_uiCount, oCurDevice.m_iVendorID));
            // serial number
            ETG_TRACE_USR2 (("Devices[%d]->m_cSerialID:       %s              ", l_uiCount, oCurDevice.m_cSerialID.toStdString().c_str()));
            // short serial number
            ETG_TRACE_USR2 (("Devices[%d]->m_cShortSerial:    %s              ", l_uiCount, oCurDevice.m_cShortSerial.toStdString().c_str()));
            // filesystem type i.e. vfat, ntfs etc
            ETG_TRACE_USR2 (("Devices[%d]->m_cFSType:         %s              ", l_uiCount, oCurDevice.m_cFSType.toStdString().c_str()));
            // device is mounted or not
            ETG_TRACE_USR2 (("Devices[%d]->m_iIsMounted:      %d              ", l_uiCount, oCurDevice.m_iIsMounted));
            // Mount point of the device
            ETG_TRACE_USR2 (("Devices[%d]->m_cMountPoint:     %s              ", l_uiCount, oCurDevice.m_cMountPoint.toStdString().c_str()));
            //Label of the device
            ETG_TRACE_USR2 (("Devices[%d]->m_cDeviceName:     %s              ", l_uiCount, oCurDevice.m_cDeviceName.toStdString().c_str()));
            //version of device
            ETG_TRACE_USR2 (("Devices[%d]->m_cDeviceVersion:  %s              ", l_uiCount, oCurDevice.m_cDeviceVersion.toStdString().c_str()));
            //Device Number
            ETG_TRACE_USR2 (("Devices[%d]->m_iDevNUM:         %d              ", l_uiCount, oCurDevice.m_iDevNUM));
            //Parent Hub Serial ID
            ETG_TRACE_USR2 (("Devices[%d]->m_cParentSerialId: %s              ", l_uiCount, oCurDevice.m_cParentSerialId.toStdString().c_str()));
            //Check for the Sd card
            ETG_TRACE_USR2 (("Devices[%d]->m_bIsSdCardDevice: 0x%x            ", l_uiCount, oCurDevice.m_bIsSdCardDevice));
            //Check for the device connection status
            ETG_TRACE_USR2 (("Devices[%d]->m_bIsConnected:    0x%x            ", l_uiCount, oCurDevice.m_bIsConnected));
            //Count for HUB
            ETG_TRACE_USR2 (("Devices[%d]->m_iHubIndex:       0x%x            ", l_uiCount, oCurDevice.m_iHubIndex));
            //USB Number
            ETG_TRACE_USR2 (("Devices[%d]->m_iUSBConnector:   %d              ", l_uiCount, oCurDevice.m_iUSBConnector));
            //Port NUmber
            ETG_TRACE_USR2 (("Devices[%d]->m_iPortNumber:     %d              ", l_uiCount, oCurDevice.m_iPortNumber));
            //Note: cast is necessary otherwhise ETG-Makro shows strange numbers
            //Freespace of usb device connected
            ETG_TRACE_USR2 (("Devices[%d]->m_lFreeSpace:      %u (KByte)      ", l_uiCount, (tU32)oCurDevice.m_lFreeSpace));
            //Usedspace of usb device connected
            ETG_TRACE_USR2 (("Devices[%d]->m_lUsedSpace:      %u (KByte)      ", l_uiCount, (tU32)oCurDevice.m_lUsedSpace));
            //Total sSize of USB device
            ETG_TRACE_USR2 (("Devices[%d]->m_lTotalSize:      %u (MByte)      ", l_uiCount, (tU32)oCurDevice.m_lTotalSize));
            //Device's UUID
            ETG_TRACE_USR2 (("Devices[%d]->m_cDeviceUUID:     %s              ", l_uiCount, oCurDevice.m_cDeviceUUID.toStdString().c_str()));
            //Type of device i.e. usb storage,sd card,ipod etc.
            ETG_TRACE_USR2 (("Devices[%d]->m_eDeviceType:     %d              ", l_uiCount, ETG_ENUM(DVM_DEVICE_TYPE, oCurDevice.m_eDeviceType)));
            //hardware port the device is connected to
            ETG_TRACE_USR2 (("Devices[%d]->m_cUSBPort:         %s             ", l_uiCount, oCurDevice.m_cUSBPort.toStdString().c_str()));
            //connect status of device
            ETG_TRACE_USR2 (("Devices[%d]->m_eConnectStatus:  %d              ", l_uiCount, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,
                                                                                                     oCurDevice.m_eConnectStatus)));
            //manufacturer name
            ETG_TRACE_USR2 (("Devices[%d]->m_cManufacturer:   %s",               l_uiCount, oCurDevice.m_cManufacturer.toStdString().c_str()));
            // product ID
            ETG_TRACE_USR2 (("Devices[%d]->m_iProductID:      0x%04x",           l_uiCount, oCurDevice.m_iProductID));
            //size
            ETG_TRACE_USR2 (("Devices[%d]->m_u64Size:         %u (KB)",          l_uiCount, (tU32)(oCurDevice.m_u64Size/1024) ));
            ETG_TRACE_USR2(("------------------------------------------------------------------------------------"));

            //------------------------------------------------------------------
            // copy diagnosis specific data to specific diag data container
            //------------------------------------------------------------------
            for(tInt i=(tInt)eUSB1;i<ARRAYSIZEFORUSB;i++)
            {
                ETG_TRACE_USR2(("i=%d oCurDevice.m_iUSBConnector.toInt=%d",i,oCurDevice.m_iUSBConnector ));
                if(i == oCurDevice.m_iUSBConnector)
                {
                    if (0!= oCurDevice.m_lTotalSize)
                    {
                        pObjUSBDiagnosis[i]->m_cVendor       =GENSTRING_NUMBER(oCurDevice.m_iVendorID);
                        pObjUSBDiagnosis[i]->m_cUSBPortNo    =GENSTRING_NUMBER(oCurDevice.m_iUSBConnector); //e.g. either Hardwareport USB1 or 2  still have to chk if this is ok
                        pObjUSBDiagnosis[i]->m_cSerialID     =oCurDevice.m_cSerialID;
                        pObjUSBDiagnosis[i]->m_cDeviceName   =oCurDevice.m_cDeviceName;
                        pObjUSBDiagnosis[i]->m_cMountPoint   =oCurDevice.m_cMountPoint;
                        pObjUSBDiagnosis[i]->m_cDeviceVersion=oCurDevice.m_cDeviceVersion;
                        pObjUSBDiagnosis[i]->m_eDeviceType   =oCurDevice.m_eDeviceType;

                        ETG_TRACE_USR3(("Get USB%d data",i));

                        l_ulDeviceSize = (unsigned long)(oCurDevice.m_lFreeSpace >> 20);
                        l_ulDeviceSize +=(unsigned long)(oCurDevice.m_lUsedSpace >> 20);

                        pObjUSBDiagnosis[i]->m_ulSize        =l_ulDeviceSize;
                    }
                    else
                    {
                    }
                }
                else
                {
                    ETG_TRACE_USR3(("Error No data for USB%d and HUB",i));
                }
            } // for (all USBs)
        }

    }

    for(tInt i=(tInt)eUSB1;i<ARRAYSIZEFORUSB;i++) ///@todo serves only two USB-connectors cleanup to make it more generic
    {
        ETG_TRACE_USR3(("USB%d: Check if HUB present",i));

        pHubDetails = GetHub((CGlobalEnumerations::HUB_NUM_t) i );

        if(NULL != pHubDetails)
        {

            pObjUSBDiagnosis[i]->m_eUSB_HUB_SEARCH     = pHubDetails->eHUBFound;

            if(CGlobalEnumerations::HUB_FOUND_YES == pObjUSBDiagnosis[i]->m_eUSB_HUB_SEARCH)
            {
                ETG_TRACE_USR1(("GetDiagnosisValues: USB%1d :HUB_FOUND_YES",i));

                pObjUSBDiagnosis[i]->m_bHUBHasCardReader        = pHubDetails->bHasCardReader;
                pObjUSBDiagnosis[i]->m_iUSBNoOfPortsFoundAtHub  = pHubDetails->uiPortCount;

                ETG_TRACE_USR1(("GetDiagnosisValues: USB%1d :Ports@HUB   =%u",  i,pObjUSBDiagnosis[i]->m_iUSBNoOfPortsFoundAtHub));
                ETG_TRACE_USR1(("GetDiagnosisValues: USB%1d :SDCardReader=0x%x",i,pObjUSBDiagnosis[i]->m_bHUBHasCardReader));
            }
        }
        else
        {
            ETG_TRACE_USR1(("GetDiagnosisValues: USB%1d :HUB_FOUND_NO",i));
        }
    }

    m_tableLock.unlock();

    ETG_TRACE_USR4(("End  : GetDiagnosisValues"));
}

/*-----------------------------------------------------------------------------*
 * void CUDevManager:: vCheckPushToList(...)
 *-----------------------------------------------------------------------------*/
void CUDevManager::vCheckPushToList(int f_iDevUsable,const CDevice* f_pCurDevice)
{
    if(f_pCurDevice)
    {
        //=================================
        //     push result to DeviceList
        //=================================
        if (DEVICE_OK == f_iDevUsable)
        {
            ETG_TRACE_COMP(("vCheckPushToList: DEVICE_OK"));

            ETG_TRACE_USR1(("vCheckPushToList:Pushing to m_vConnectedDevices"));
#ifdef IMPROVE_CRITVOLT
            if(TRUE == DeviceListManager::GetInstance()->getPlatformEventsLocked(f_pCurDevice->m_eUSB))
            {
                ETG_TRACE_COMP(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                ETG_TRACE_COMP(("vCheckPushToList:eUSB%1d locked - don't add to list",f_pCurDevice->m_eUSB));
                ETG_TRACE_COMP(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
            }
            else
            {
                DeviceListManager::GetInstance()->ListAddDevice(DEVLIST_UDEVMGR,*f_pCurDevice);
            }
#else
            DeviceListManager::GetInstance()->ListAddDevice(DEVLIST_UDEVMGR,*f_pCurDevice);
#endif

            ETG_TRACE_USR1(("vCheckPushToList:Valid Device type pushed"));
        }
        //=================================
        //     push it to DisabledDevices-List
        //=================================
        else
        {
            ETG_TRACE_COMP(("vCheckPushToList: DEVICE_NOK"));
            ETG_TRACE_USR1(("vCheckPushToList: Pushing to m_vDisableDevices"));

            m_vDisableDevices.push_back(*f_pCurDevice); //check for race conditions here

            ETG_TRACE_USR1(("vCheckPushToList:UNKNOW Device type, but port use."));
        }
    }
    else
    {
        ETG_TRACE_USR1(("[ERROR]: vCheckPushToList:f_pCurDevice == NULL"));
    }


}

/*-----------------------------------------------------------------------------*
 * void ParentHUBSwapping(CDevice* f_pCurDevice)                               *
 *-----------------------------------------------------------------------------*/
void CUDevManager::ParentHUBSwapping( const CDevice* f_pCurDevice)
{
    ETG_TRACE_USR4 (("Begin: ParentHubSwapping"));

    if (NULL != f_pCurDevice)
    {
        GENSTRING l_cHUBName = "";
        GENSTRING l_cHUBNode = "";
        int l_cHUBPort = 0;

        //Parent hub serial id in lower case
        GENSTRING l_cHubSerialLower = "";
        GENSTRING l_cHubSerial;

        l_cHubSerialLower = f_pCurDevice->m_cParentSerialId.toLower();
        l_cHubSerial = "hub";


        ETG_TRACE_USR4 (("ParentHubSwapping: f_pCurDevice->m_iUSBConnector  = %d  ",
                         f_pCurDevice->m_iUSBConnector));
        ETG_TRACE_USR4 (("ParentHubSwapping: f_pCurDevice->m_iPortNumber = %d  ",
                         f_pCurDevice->m_iPortNumber));

        if (true == l_cHubSerialLower.contains(l_cHubSerial))
        {
            if (1 == f_pCurDevice->m_iUSBConnector && f_pCurDevice->m_iPortNumber > 0)
            {
                ETG_TRACE_USR4 (("ParentHubSwapping: USB Port 1"));

                l_cHUBName = m_cParentUsb1Hub;
                l_cHUBNode = m_cParentHub1Node;
                l_cHUBPort = m_cParentHub1Port;
                if(f_pCurDevice->m_cParentSerialId != l_cHUBName && m_cParentUsb2Hub.isEmpty() == false)    //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_USR4 (("ParentHubSwapping: Swapping HUB1 details"));

                    m_cParentUsb1Hub  = m_cParentUsb2Hub;
                    m_cParentHub1Node = m_cParentHub2Node;
                    m_cParentHub1Port = m_cParentHub2Port;

                    m_cParentUsb2Hub  = l_cHUBName;
                    m_cParentHub2Node = l_cHUBNode;
                    m_cParentHub2Port = l_cHUBPort;
                }

            }

            else if (2 == f_pCurDevice->m_iUSBConnector && f_pCurDevice->m_iPortNumber > 0)
            {
                ETG_TRACE_USR4 (("ParentHubSwapping: USB Port 2"));

                l_cHUBName = m_cParentUsb2Hub;
                l_cHUBNode = m_cParentHub2Node;
                l_cHUBPort = m_cParentHub2Port;

                if (f_pCurDevice->m_cParentSerialId != l_cHUBName && m_cParentUsb1Hub.isEmpty() == false)      //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_USR4 (("ParentHubSwapping: Swapping HUB1 details"));

                    m_cParentUsb2Hub  = m_cParentUsb1Hub;
                    m_cParentHub2Node = m_cParentHub1Node;
                    m_cParentHub2Port = m_cParentHub1Port;

                    m_cParentUsb1Hub  = l_cHUBName;
                    m_cParentHub1Node = l_cHUBNode;
                    m_cParentHub1Port = l_cHUBPort;
                }
            }
            else
            {
                ETG_TRACE_ERR (("ParentHubSwapping: [ERROR] Swapping HUB2 details"));
            }
        }
    }
    else
    {
        ETG_TRACE_ERR (("ParentHUBSwapping: [ERROR] Invalid f_pCurDevice is NULL"));

    } // if (f_pCurDevice is NULL)

    ETG_TRACE_USR4 (("End  : ParentHubSwapping"));
}

tVoid CUDevManager::connectUDEV(tBool bConnect)
{
    ETG_TRACE_COMP(("connectUDEV: bConnect=0x%x",bConnect));
    m_bConnectUDEV=bConnect;
}


tVoid CUDevManager::AddPartitionUUIDToSerialNumer(CDevice* f_pCurDevice) const
{
    ETG_TRACE_USR4 (("Begin: AddPartitionUUIDToSerialNumer"));
    if(NULL != f_pCurDevice)
    {
        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL))
        {
            //The original serial id of the device
            //This will be used to identify all the partitions of multipartition  stick belongs to same device
            f_pCurDevice->m_cAccessoryName2 = f_pCurDevice->m_cSerialID;
            ETG_TRACE_USR3 (("AddPartitionUUIDToSerialNumer: f_pCurDevice->m_cAccessoryName2(Original Serial ID of the device): %s",f_pCurDevice->m_cAccessoryName2.toStdString().c_str()  ));
        }
        f_pCurDevice->m_cSerialID.append("_");
        f_pCurDevice->m_cSerialID.append(f_pCurDevice->m_cDeviceUUID);
        ETG_TRACE_COMP(("AddPartitionUUIDToSerialNumer:  f_pCurDevice->m_cSerialID: %s", f_pCurDevice->m_cSerialID.toStdString().c_str()));
    }
    else
    {
        ETG_TRACE_COMP(("AddPartitionUUIDToSerialNumer: f_pCurDevice"));
    }
    ETG_TRACE_USR4 (("End  : AddPartitionUUIDToSerialNumer"));
}

/*-----------------------------------------------------------------------------*
 * int CheckForValidPartition(CDevice* f_pCurDevice)                           *
 *-----------------------------------------------------------------------------*/
int CUDevManager::CheckForValidPartition(CDevice* f_pCurDevice)
{
    ETG_TRACE_USR4 (("Begin: CheckForValidPartition"));

    int l_iHasValidPartition = DEVICEMANAGER_ERROR_NODEVICE;

    if (NULL != f_pCurDevice)
    {
        /*
         * We have got the base device like '/dev/sda'
         * Then we get the partition table and find the first valid partition.
         * The partition number is then appended to the device i.e 'dev/sdaX', where
         * X is the partition number
         *
         * Then the other parameters are found and the p_cCurDevice is modified
         */

        /* find out the first valid partition */
        blkid_probe    pr;
        blkid_partlist ls;
        int nparts, i;
        int l_iValidPart = -1;
        blkid_partition par;
        int part_type;
        GENSTRING l_cPartitionNode;
        GENSTRING l_cFSType, l_cUUID, l_cLabel;

        ETG_TRACE_USR3 (("CheckForValidPartition: f_pCurDevice->m_cBlockDevnode = %s",
                         f_pCurDevice->m_cBlockDevnode.toStdString().c_str()));

        int fd = open (f_pCurDevice->m_cBlockDevnode.toStdString().c_str(), O_RDONLY);

        if (-1 != fd)
            //if (NULL != pr)
        {
            ETG_TRACE_USR3 (("CheckForValidPartition: Device opened successfully"));

            // Check if the device is one single partition
            l_iHasValidPartition = GetPartitionDetails (f_pCurDevice->m_cBlockDevnode, l_cFSType, l_cUUID, l_cLabel);

            if (DEVICEMANAGER_OK == l_iHasValidPartition)
            {
                f_pCurDevice->m_iValidPartitionNo = 0;

                ETG_TRACE_USR3 (("CheckForValidPartition: Whole device can be mounted"));
                ETG_TRACE_USR3 (("CheckForValidPartition: l_cFSType = %s", l_cFSType.toStdString().c_str()));

                if (!l_cFSType.isEmpty())
                {
                    // check whether the partition is supported
                    l_iHasValidPartition = ConfigurationManager::GetInstance()->IsSupportedFS(l_cFSType);
                    if (DEVICEMANAGER_OK == l_iHasValidPartition)
                    {
                        f_pCurDevice->m_cFSType = l_cFSType;

                        ETG_TRACE_USR3 (("CheckForValidPartition: Supported filesystem found : %s",
                                         l_cFSType.toStdString().c_str()));
                        l_iHasValidPartition = DEVICEMANAGER_DEVICE_DETECTED;
                    }
                    else
                    {
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                        f_pCurDevice->m_cFSType = l_cFSType;
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                        ETG_TRACE_USR1(( "CheckForValidPartition: Unsupported filesystem found : %s", l_cFSType.toStdString().c_str() ));
                    } // if (fstype is supported)
                }
                else
                {
                    // Do nothing
                } // if (fstype found)
            }
            else if (DEVICEMANAGER_ERROR_NOTVALID == l_iHasValidPartition)
            {
                // its a disk with partition table
                // Traverse partition list
                pr = blkid_new_probe_from_filename(f_pCurDevice->m_cBlockDevnode.toStdString().c_str());

                if (NULL != pr)
                {
                    ls = blkid_probe_get_partitions(pr);

                    if (NULL != ls)
                    {
                        nparts = blkid_partlist_numof_partitions(ls);

                        if ( -1 != nparts)
                        {
                            //@todo START: Bugfix before NISSAN LCN2KAI SOP2 -> Put this into a function!
                            // (check the code below and above for similar functionality and try to bring it into one function)
                            ETG_TRACE_USR3(( "CheckForValidPartition: Partition count = %d" , nparts )); //nparts 0BUGFIX_ANDROIDMSC
                            l_iHasValidPartition = DEVICEMANAGER_ERROR_NODEVICE;                         //BUGFIX_ANDROIDMSC this is why it returns no supported file system
                            ETG_TRACE_USR3(( "CheckForValidPartition: Traversing partition list" ));     //CheckForValidPartition: No supported filesystems this can bessen in trace
                            //special case for android
                            if( nparts == 0 )
                            {
                                ETG_TRACE_USR3(( "CheckForValidPartition: nparts = 0 - might happen with Android mobiles " ));
                                f_pCurDevice->m_iValidPartitionNo = 0;
                                ETG_TRACE_USR3(( "CheckForValidPartition: Whole device can be mounted" ));
                                ETG_TRACE_USR3(( "CheckForValidPartition: l_cFSType = %s", l_cFSType.toStdString().c_str() ));
                                l_cPartitionNode = f_pCurDevice->m_cBlockDevnode;                        // + GENSTRING((char)('0' + l_iValidPart));
                                //=======================
                                ETG_TRACE_USR3(( "CheckForValidPartition: -------------------------------------------------" ));
                                int l_iValidDetails = GetPartitionDetails( l_cPartitionNode, l_cFSType, l_cUUID, l_cLabel );
                                ETG_TRACE_USR3(( "CheckForValidPartition: -------------------------------------------------" ));
                                ETG_TRACE_USR3(( "CheckForValidPartition: l_cFSType = %s", l_cFSType.toStdString().c_str() ));
                                if(   (   (l_iValidDetails == DEVICEMANAGER_OK)
                                          || (l_iValidDetails == DEVICEMANAGER_ERROR_NOTVALID)
                                          )
                                      && ( l_cFSType.isEmpty() == FALSE )
                                      )
                                {
                                    // check whether the partition is supported
                                    l_iHasValidPartition = ConfigurationManager::GetInstance()->IsSupportedFS( l_cFSType );
                                    if( DEVICEMANAGER_OK == l_iHasValidPartition )
                                    {
                                        f_pCurDevice->m_cFSType = l_cFSType;
                                        f_pCurDevice->m_iValidPartitionNo = 0;
                                        ETG_TRACE_USR3(( "CheckForValidPartition: Supported filesystem found : %s", l_cFSType.toStdString().c_str() ));
                                        l_iHasValidPartition = DEVICEMANAGER_DEVICE_DETECTED;

                                        //break;
                                    }
                                    else
                                    {
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                                        f_pCurDevice->m_cFSType = l_cFSType;
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                                        ETG_TRACE_USR1(( "CheckForValidPartition: Unsupported filesystem found : %s", l_cFSType.toStdString().c_str() ));
                                    }
                                }
                                else
                                {
                                    ETG_TRACE_USR3(( "CheckForValidPartition: Empty fstype found" ));
                                }
                            }
                            //@todo END: Bugfix before NISSAN LCN2KAI SOP2 -> Put this into a function!
                            for (i = 0; i < nparts && DEVICEMANAGER_OK != l_iHasValidPartition; i++)
                            {
                                par = blkid_partlist_get_partition(ls, i);

                                if (NULL != par)
                                {
                                    l_iValidPart  = blkid_partition_get_partno(par);
                                    part_type     = blkid_partition_get_type(par);

                                    ETG_TRACE_USR3 (("CheckForValidPartition: ## Partition        : %d", i));
                                    ETG_TRACE_USR3 (("CheckForValidPartition:    Partition Number : %d", l_iValidPart));
                                    ETG_TRACE_USR4 (("CheckForValidPartition:    Part type        : %d", part_type));

                                    ETG_TRACE_USR4 (("CheckForValidPartition: Finding FS type for BLOCK device"));

                                    l_cPartitionNode = f_pCurDevice->m_cBlockDevnode + GENSTRING((char)('0' + l_iValidPart));//casting charensures constructor GENSTRING(char ) to be used)

                                    //=======================
                                    ETG_TRACE_USR3(("CheckForValidPartition: -------------------------------------------------"));
                                    ETG_TRACE_USR4(("CheckForValidPartition: l_cPartitionNode: %s",l_cPartitionNode.toStdString().c_str()));
                                    int l_iValidDetails = GetPartitionDetails (l_cPartitionNode, l_cFSType, l_cUUID, l_cLabel);
                                    ETG_TRACE_USR3(("CheckForValidPartition: -------------------------------------------------"));

                                    // Case when the disk is mac-formatted
                                    // The partition is shown as 'vfat' with label 'EFI' but
                                    // its not valid for processing, so it needs to be skipped
                                    if ((CGlobalEnumerations::PT_GPT == f_pCurDevice->m_ePTType) // Mac formatted
                                            && (GENSTRING("EFI") == l_cLabel))          // Label is "EFI"
                                    {
                                        // Skip the partition
                                        ETG_TRACE_USR1 (("CheckForValidPartition: 'EFI' partition found, skipping.."));
                                        continue;
                                    }

                                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_PARTITIONS_ALL))
                                    {

                                        char strData[4];
                                        snprintf(strData,sizeof(strData),"%d",l_iValidPart);
                                        l_cPartitionNode = f_pCurDevice->m_cDevnode + GENSTRING(strData);

                                        ETG_TRACE_USR1(("CheckForValidPartition:l_cPartitionNode        = %s",l_cPartitionNode.toStdString().c_str()));
                                        ETG_TRACE_USR1(("CheckForValidPartition:f_pCurDevice->m_cDenNode = %s",f_pCurDevice->m_cDevnode.toStdString().c_str()));
                                        if(f_pCurDevice->m_cPartitionNode == l_cPartitionNode)
                                        {
                                            ETG_TRACE_USR1 (("CheckForValidPartition: match for notified partition"));
                                        }
                                        else
                                        {
                                            // Skip the partition
                                            ETG_TRACE_USR1 (("CheckForValidPartition: no match yet - continue"));
                                            continue;
                                        }
                                    }

                                    ETG_TRACE_USR3(("CheckForValidPartition: l_cFSType = %s", l_cFSType.toStdString().c_str()));

                                    if (((DEVICEMANAGER_OK == l_iValidDetails)
                                         || (DEVICEMANAGER_ERROR_NOTVALID == l_iValidDetails))
                                            && (FALSE == l_cFSType.isEmpty()))
                                    {
                                        // check whether the partition is supported
                                        l_iHasValidPartition = ConfigurationManager::GetInstance()->IsSupportedFS(l_cFSType);
                                        if (DEVICEMANAGER_OK == l_iHasValidPartition)
                                        {
                                            f_pCurDevice->m_cFSType = l_cFSType;
                                            f_pCurDevice->m_iValidPartitionNo = l_iValidPart;
                                            ETG_TRACE_USR3 (("CheckForValidPartition: Supported filesystem found : %s", l_cFSType.toStdString().c_str()));
#ifdef MSD_BIGGEST_PARTITON_FEATURE
                                            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL))
                                            {

                                                    ETG_TRACE_USR3 (("CheckForValidPartition: eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL Incrementing the number of valid partitions" ));
                                                    int l_u8TotalValidPartitons = 0;
                                                    if(DEVICEMANAGER_OK == FindTotalValidPartitions(f_pCurDevice->m_cBlockDevnode.toStdString().c_str(),l_u8TotalValidPartitons))
                                                    {
                                                        f_pCurDevice->m_iTotalValidPartitons = l_u8TotalValidPartitons;
                                                    }

                                            }
#endif//MSD_BIGGEST_PARTITON_FEATURE

                                            l_iHasValidPartition = DEVICEMANAGER_DEVICE_DETECTED;

                                            break;
                                        }
                                        else
                                        {

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                                            f_pCurDevice->m_cFSType     = l_cFSType;
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                                            ETG_TRACE_USR1(( "CheckForValidPartition: Unsupported filesystem found : %s", l_cFSType.toStdString().c_str() ) );
                                            if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_FIRST_SUPPORTED_PARTITION) && nparts>1 )
                                            {
                                                ETG_TRACE_USR1(( "CheckForValidPartition: eCONF_ENABLE_NOTIFY_FIRST_SUPPORTED_PARTITION is OFF. No need to scan for valid partitions") )
                                                break;
                                            }
                                        }
                                    }
                                    else
                                    {
                                        ETG_TRACE_USR3(("CheckForValidPartition: Empty fstype found"));
                                    }
                                }
                                else
                                {
                                    ETG_TRACE_ERR (("CheckForValidPartition: [ERROR] blkid_partlist_get_partition failed"));

                                    l_iHasValidPartition = DEVICEMANAGER_ERROR_UNKNOWN;
                                } // if (blkid_partlist_get_partition failed)
                            } // for (whole partition list)
                        }
                        else
                        {
                            ETG_TRACE_ERR (("CheckForValidPartition: [ERROR] blkid_partlist_numof_partitions failed"));

                            l_iHasValidPartition = DEVICEMANAGER_ERROR_UNKNOWN;
                        } // if (blkid_partlist_numof_partitions success)
                    }
                    else
                    {
                        ETG_TRACE_ERR (("CheckForValidPartition: [ERROR] blkid_probe_get_partitions failed"));

                        l_iHasValidPartition = DEVICEMANAGER_ERROR_UNKNOWN;
                    } // if (NULL != ls)

                    blkid_free_probe(pr);
                }
                else
                {
                    ETG_TRACE_ERR (("CheckForValidPartition: [ERROR] blkid_new_probe_from_filename failed"));

                    l_iHasValidPartition = DEVICEMANAGER_ERROR_UNKNOWN;
                } // if (NULL != pr)
            } // if (check whole device is mounted)

            if (DEVICEMANAGER_DEVICE_DETECTED == l_iHasValidPartition)
            {
                // Modify CDevice attributes
                /* '0' means that the whole device is mounted and there are no
                 * partitions in it
                 */
                if (0 == f_pCurDevice->m_iValidPartitionNo)
                {
                    l_cPartitionNode = f_pCurDevice->m_cBlockDevnode;
                }
                else
                {
                    l_cPartitionNode = f_pCurDevice->m_cBlockDevnode+GENSTRING((char)('0'+f_pCurDevice->m_iValidPartitionNo));
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_FIRST_SUPPORTED_PARTITION))
                    {
                        ETG_TRACE_USR3 (("CheckForValidPartition: eCONF_ENABLE_NOTIFY_FIRST_SUPPORTED_PARTITION is ON, changing the partition number from %d to one ", l_iValidPart ));
                        f_pCurDevice->m_iValidPartitionNo = 1;
                    }
                    f_pCurDevice->m_cBlockDevnode = l_cPartitionNode;
                }

                f_pCurDevice->m_cDeviceUUID = l_cUUID;
                f_pCurDevice->m_cDeviceName = l_cLabel.trimmed();
                f_pCurDevice->m_cAccessoryName = f_pCurDevice->m_cDeviceName;

                ETG_TRACE_USR3 (("CheckForValidPartition: Partition UUID  = %s",
                                 f_pCurDevice->m_cDeviceUUID.toStdString().c_str()));
                ETG_TRACE_USR3 (("CheckForValidPartition: Partition Label = %s",
                                 l_cLabel.toStdString().c_str()));
                ETG_TRACE_USR3 (("CheckForValidPartition: Device Name     = %s",
                                 f_pCurDevice->m_cDeviceName.toStdString().c_str()));
                ETG_TRACE_USR3 (("CheckForValidPartition: Accessory Name  = %s",
                                 f_pCurDevice->m_cAccessoryName.toStdString().c_str()));

                l_iHasValidPartition = DEVICEMANAGER_OK;
            }
            else
            {
                ETG_TRACE_USR4 (("CheckForValidPartition: No supported filesystems"));
            } // if (valid partition is found)

            // Free up file descriptor
            close (fd);
        } // if (-1 != fd)
    }//if (NULL != f_pCurDevice)
    else
    {
        ETG_TRACE_USR4 (("CheckForValidPartition: Device is NULL"));

        l_iHasValidPartition = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if (NULL != f_pCurDevice)

    ETG_TRACE_USR4 (("End  : CheckForValidPartition"));

    return l_iHasValidPartition;
}


/*-----------------------------------------------------------------------------*
 * int AddDevice(CGlobalEnumerations::NOTIFICATION_Type eNotiType,             *
 *          GENSTRING f_cDevNode)                                                *
 *-----------------------------------------------------------------------------*/

int CUDevManager::AddDeviceDVDRelated( GENSTRING f_cDevNode)
{
    int iRetVal = DEVICEMANAGER_ERROR_NOTVALID;

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
    {
        ETG_TRACE_COMP(("AddDeviceDiscInDVDOrDVDDrive(...)"));
        CDevice l_DiscInDVDDriveUSBConneted;

        //find out if this is
        //DVD_Drive or
        //DTY_CDDA inserted into DVD_Drive
        //other disc types like CDROM, DVD-Rom are handled by autmounter as before
        //other disc types like DVD-Audio, DVD-Video are handled internally in DVDCtrl in Mediaplayer
       l_DiscInDVDDriveUSBConneted.m_eDeviceType = CGlobalEnumerations::DTY_DVD_DRIVE;

       iRetVal =   AddDevice(CGlobalEnumerations::enUnotify,f_cDevNode , l_DiscInDVDDriveUSBConneted );
    }
    return iRetVal;
}



int CUDevManager::AddDevice() //used by udev-notification
{
    CDevice DUMMY;
    return AddDevice(CGlobalEnumerations::enUnotify,"" /*empy GENSTRING f_cDevNode*/, DUMMY );
}


int CUDevManager::AddDevice( CGlobalEnumerations::NOTIFICATION_Type eNotiType, GENSTRING f_cDevNode, const CDevice& l_oCurAvailDevInfo ) //used by automount(er) - notification
{
    int l_iDevProcessingStatus = DEVICEMANAGER_OK;


    tenDevManagerUSBHost eUsb;

    //---------------------------------------------------------------------
    // get m_pBaseUSBDev used to calculate l_cNode
    //--------------------------------------------------------------------
    ETG_TRACE_USR4(( "AddDevice: NOTIFICATION_Type = %d", ETG_ENUM(NOTIFICATION_Type, eNotiType) ));
    if( eNotiType == CGlobalEnumerations::enInotify  ||  eNotiType == CGlobalEnumerations::enUnotify )
    {
        // Get the parent 'usb_device' for this block device
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
        if( f_cDevNode.contains("/dev/mmcblk") )
        {
            m_pBaseUSBDev = m_pDev;                      // for SDCardInternal take m_pDev as base USB dev
        }
        else
#endif //VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_CDROM_DEVICE
            if( f_cDevNode.contains("/dev/sr") )
            {
                m_pBaseUSBDev = m_pDev;                      // for CDRomInternal take m_pDev as base USB dev
            }
            else
#endif //VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_CDROM_DEVICE
            {
                m_pBaseUSBDev = FindBaseUSBDevice (m_pDev); //dor MTP,IPOD, IPHONE, HUBS, IPAD this is done
            }

        const char *l_cNode = udev_device_get_devnode(m_pBaseUSBDev);

        //MTP_TWO_EVENTS
        /* Added as when multiple devices are connected, for an MTP device
         * notifications from both; udev and inotify are coming up. udev is providing
         * devnode as /dev/bus/usb/00X/00Y and inotify as /dev/sdXY. Due to this
         * DeviceListManager::IsAlreadyConected() is failing and DeviceManager is
         * proceeding to add MTP again (for inotify event).
         */
        if (TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cNode))
        {
            ETG_TRACE_USR1 (("AddDevice: Device already connected"));

            // Reset m_pBaseUSBDev to NULL
            m_pBaseUSBDev = NULL;

            ETG_TRACE_USR4 (("End  : AddDevice with %i ", ETG_ENUM(DEVICEMANAGER_ERROR, DEVICEMANAGER_ERROR_NOTVALID) ));

            return DEVICEMANAGER_ERROR_NOTVALID;
        }

        ETG_TRACE_USR4 (("AddDevice: ---------NEW DEVICE TO ADD-----------------------------"));
        ETG_TRACE_USR4 (("AddDevice: | Node     : %s  ", l_cNode));
        ETG_TRACE_USR4 (("AddDevice: -------------------------------------------------------"));
    }

    //Create device object instance
    CDevice  oCurrentDevice;  //@todo later do use directly oCurrentDevice and not l_pCurDevice
    CDevice *l_pCurDevice = &oCurrentDevice;

    if (NULL == l_pCurDevice)//lint !e774
    {
        ETG_TRACE_FATAL(("AddDevice: m_cAction:'add' NULL == l_pCurDevice"));
        DVM_NORMAL_M_ASSERT_ALWAYS();

        l_iDevProcessingStatus = DEVICEMANAGER_ERROR_MEM_ALLOC;
    }
    else
    {
        //----------------------------------------------------------------------------------
        //current configuration  enInotify,enUnotify   ///@todo cleanup possible here!!
        //possible cleanup - please check: enInMscDev,enInAppleDev,enInMtpDev might not used anymore
        //----------------------------------------------------------------------------------
        // fill notification type
        switch( eNotiType )
        {
        case CGlobalEnumerations::enInMscDev:                     // case, no break
        case CGlobalEnumerations::enInAppleDev:                   // case, no break
        case CGlobalEnumerations::enInMtpDev:                     // case, no break
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        case CGlobalEnumerations::enInUnsupDev:                   // case, no break
            l_pCurDevice->FillDevice( l_oCurAvailDevInfo );       //HandleMount has extraced content which will be filled already into l_pCurDevice
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
            l_pCurDevice->m_eTypeOfNotify = eNotiType;            // set notify type
            break;
        case CGlobalEnumerations::enInotify:                      // case, no break
        case CGlobalEnumerations::enUnotify:
            l_pCurDevice->m_eTypeOfNotify = eNotiType;            // set notify type
            break;
        case CGlobalEnumerations::enNotifyAll:
            ETG_TRACE_ERR(( "AddDevice: CGlobalEnumerations::enNotifyAll is not expected!" ));
            l_pCurDevice->m_eTypeOfNotify = CGlobalEnumerations::enUnotify;  // UNOTIFY was the default in old if() else construct
            break;
        }

        bool bCallFillCurrDevice = true;

        l_pCurDevice->m_cPartitionNode = l_oCurAvailDevInfo.m_cPartitionNode;


        //-----------------------------------------------------------------------
        // internal SD-card and cdrom get's filled in a special way
        //-----------------------------------------------------------------------
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
        if (f_cDevNode.contains("/dev/mmcblk"))
        {
            l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NODEVICE;

            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_INTERNAL_SDCARD))
            {
                ETG_TRACE_USR4(("AddDevice: Check  DTY_SD_INTERNAL"));
                if(ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_SD_INTERNAL,f_cDevNode.toStdString()) )
                {
                    ETG_TRACE_USR4(("AddDevice: Found: DTY_SD_INTERNAL"));
                    l_pCurDevice->m_cMountPoint = l_oCurAvailDevInfo.m_cMountPoint;
                    l_iDevProcessingStatus = FillCurrDeviceSDInternal(l_pCurDevice,f_cDevNode);
                }
                ETG_TRACE_USR4(("DTY_SD_INTERNAL: %d ",ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus)));
            }

            bCallFillCurrDevice = false;
        }
#endif  //VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_SDCARD_DEVICE
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_CDROM_DEVICE //tbd.: check to replace by this if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
        if (f_cDevNode.contains("/dev/sr")) //@todo shift to config.h
        {
            tBool bCheckedSuccesful = FALSE;
            l_iDevProcessingStatus  = DEVICEMANAGER_ERROR_NODEVICE;

            //DTY_DVDDrive by udev - the connected drive - used by mediaplayer DVDCtrl to poll on disc notifications via SCSI
            if((FALSE == bCheckedSuccesful) && (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE)))
            {
                if(CGlobalEnumerations::DTY_DVD_DRIVE == l_oCurAvailDevInfo.m_eDeviceType)
                {
                    l_iDevProcessingStatus = FillCurrDeviceDVDDrive(l_pCurDevice,f_cDevNode);
                    bCheckedSuccesful = TRUE;
                }
                ETG_TRACE_USR4(("DTY_DVD_DRIVE: %d ",ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus)));
            }

#ifdef TEST_DVDDRIVE_CDDA_NOTIFICATION
            //DTY_CDDA notifies by udev  //Note previous optical disc this is received from VD_MEDIAMANAGER
            if((FALSE == bCheckedSuccesful) && (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE)))
            {
                ETG_TRACE_USR4(("AddDevice: Check  DTY_CDDA (DVD-Drive connected USB)"));
                if(CGlobalEnumerations::DTY_DVD_DRIVE == l_oCurAvailDevInfo.m_eDeviceType)
                {
                    ETG_TRACE_USR4(("AddDevice:Found: DTY_CDDA (DVD-Drive connected USB)"));
                    l_iDevProcessingStatus = FillCurrDeviceCDDA(l_pCurDevice,f_cDevNode);
                    bCheckedSuccesful = TRUE;
                }
                ETG_TRACE_USR4(("DTY_CDDA: %d ",ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus)));
            }
#endif
            //DTY_CDROM notifed by automounter
            if((FALSE == bCheckedSuccesful) &&  (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_INTERNAL_CDROM)) )
            {
                ETG_TRACE_USR4(("AddDevice: Check  DTY_CDROM INTERNAL"));
                if( ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_CDROM,f_cDevNode.toStdString()) )
                {
                    ETG_TRACE_USR4(("AddDevice: Found: DTY_CDROM (INTERNAL)"));
                    l_pCurDevice->m_cMountPoint = l_oCurAvailDevInfo.m_cMountPoint;
                    l_pCurDevice->m_cSerialID   = l_oCurAvailDevInfo.m_cSerialID;
                    l_pCurDevice->m_cFSType     = l_oCurAvailDevInfo.m_cFSType;
                    l_iDevProcessingStatus = FillCurrDeviceCDROM(l_pCurDevice,f_cDevNode);
                    bCheckedSuccesful = TRUE;
                 }
                 ETG_TRACE_USR4((" DTY_CDROM INTERNAL: %d ",ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus)));
            }

            bCallFillCurrDevice = false;


        }
#endif //VARIANT_S_FTR_ENABLE_DEVMGR_INTERNAL_CDROM_DEVICE



        // Fill the device details
        if(bCallFillCurrDevice) //lint !e774: Boolean within 'if' always evaluates to True  - currenly seen as easy way to cope with variants - refactoring planned
        {
            l_iDevProcessingStatus = FillCurrDevice(l_pCurDevice);
        }


#ifdef BUGFIX_DIAGGMHUB
        // serial id for GM hub is empty, so FillCurrDevice() returns error
        if (   (DEVICEMANAGER_OK == l_iDevProcessingStatus)
               || (CGlobalEnumerations::DTY_HUB == l_pCurDevice->m_eDeviceType)
               )
#else
        if (DEVICEMANAGER_OK == l_iDevProcessingStatus)
#endif
        {
            ETG_TRACE_USR4(( "AddDevice: Switch DeviceType (m_eDeviceType=%i)", ETG_ENUM(DVM_DEVICE_TYPE, l_pCurDevice->m_eDeviceType) ))

            switch (l_pCurDevice->m_eDeviceType)
            {
                //================================================
                case CGlobalEnumerations::DTY_USB: //this is either DTY_USB or DTY_SD
                //================================================
                    // process a storage device only if its notified by INOTIFY
                    if( l_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enInotify )
                    {
                        l_iDevProcessingStatus = DetectBlockDevice(l_pCurDevice,eNotiType,f_cDevNode);
#ifdef BUGFIX_NO_SERIALID
                        if( l_pCurDevice->m_cSerialID.isEmpty() )
                        {
                            l_pCurDevice->m_cSerialID = l_pCurDevice->m_cMountPoint;
                        }
#endif

                    }
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                    // process a storage device only if its notified by INOTIFY -> here for not supoorted devices
                    else if( l_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enInUnsupDev )
                    {
                        l_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_NOT_SUPPORTED;
                        ETG_TRACE_USR1 (("AddDevice: Set deviceType to DTY_NOT_SUPPORTED."));
                                l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    }
#endif
                    else if( l_pCurDevice->m_eTypeOfNotify == CGlobalEnumerations::enUnotify )
                    {
                        ETG_TRACE_USR1 (("AddDevice: Don't process USB device by udev."));
                    }

                    break;
                //================================================
                case CGlobalEnumerations::DTY_IPOD:
                //================================================

                    //RTC-497260(INF4CV): Detect apple devices only if apple device support is present
                    if(ConfigurationManager::GetInstance()->bIsAppleDeviceSupported())
                    {
                        l_iDevProcessingStatus = m_pIDeviceRecognizer->DetectDevice(IN CGlobalEnumerations::enAppleDev,INOUT l_pCurDevice);

                        if ((DEVICEMANAGER_DEVICE_DETECTED == l_iDevProcessingStatus) ||
                                (DEVICEMANAGER_ERROR_UNKNOWN   == l_iDevProcessingStatus))   //unknown apple device - will be charged but not used
                        {
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                            if( l_pCurDevice->m_eDeviceType == CGlobalEnumerations::DTY_NOT_SUPPORTED )
                            {
                                ETG_TRACE_COMP (("AddDevice: Not Supported Apple device detection SUCCESSFUL"));
                            }
                            else
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                            {
                                ETG_TRACE_COMP (("AddDevice: Apple device detection SUCCESSFUL"));
                            }
                        }
                        else
                        {
                            ETG_TRACE_COMP (("AddDevice: Apple device detection FAILED"));
                        }
                    }
                    else
                    {
                        l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NODEVICE;
                        ETG_TRACE_USR1 (("AddDevice: Apple devices are not supported.Disabled in config"));
                    }
                    break;
                //================================================
                case CGlobalEnumerations::DTY_MTP:
                case CGlobalEnumerations::DTY_MISCELLANEOUS: //Check for Miscellaneous also
                //================================================
                    // Check for MTP Device
                try
                {
                    // Detect MTP Device
                    l_iDevProcessingStatus = m_pIDeviceRecognizer->DetectDevice(IN CGlobalEnumerations::enMTPDevice, INOUT l_pCurDevice);

                    if (DEVICEMANAGER_DEVICE_DETECTED == l_iDevProcessingStatus)
                    {
                        ETG_TRACE_COMP (("AddDevice: MTP device detection SUCCESSFUL"));
                    }
                    else
                    {
                        ETG_TRACE_COMP (("AddDevice: MTP device detection FAILED"));
#ifdef USE_MYSPIN_CHANGE_PTP
                            ETG_TRACE_COMP (("AddDevice: MTP mapped to NOT SUPPORTED (MISCALLANEOUS) for Myspin"));
                            l_pCurDevice->m_cMountPoint     = "MISCELLANEOUS";
                            l_pCurDevice->m_cAccessoryName  = l_pCurDevice->m_cShortSerial;
                            if(l_pCurDevice->m_eDeviceType == CGlobalEnumerations::DTY_MTP)
                            {
                                l_pCurDevice->m_eDeviceUnsupportedReason = PossiblySupportedByOtherClient;
                            }
                            l_pCurDevice->m_eDeviceType     = CGlobalEnumerations::DTY_NOT_SUPPORTED;
                            l_iDevProcessingStatus          = DEVICEMANAGER_DEVICE_DETECTED;
#endif //USE_MYSPIN_CHANGE_PTP
                    }
                }
                catch(...)
                {
                    ETG_TRACE_ERR(("AddDevice: Exception in DetectMTPDevice"));
                    l_iDevProcessingStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                } // check for MTP device
                    break;
                //================================================
                case CGlobalEnumerations::DTY_SD_INTERNAL:
                //================================================
                    ETG_TRACE_COMP(("AddDevice: DTY_SD_INTERNAL"));
                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    break;
                //================================================
                case CGlobalEnumerations::DTY_CDROM:
                //================================================
                    ETG_TRACE_COMP(("AddDevice: DTY_CDROM"));
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
                    {
                        ETG_TRACE_COMP(("AddDevice: DTY_CDROM is in Internal DVD Drive"));
                        if(GetIsDVDDriveMassStorageMode())
                        {
                            ETG_TRACE_COMP(("AddDevice: DTY_CDROM is in Mass storage Mode. Notified by Mediaplayer"));
                            l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                        }
                        else
                        {
                            ETG_TRACE_COMP(("AddDevice: DTY_CDROM is NOT in Mass storage Mode. Or NOT Notified by Mediaplayer yet"));
                            l_iDevProcessingStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        }
                    }
                    else
                    {
                        ETG_TRACE_COMP(("AddDevice: DTY_CDROM is NOT in Internal DVD Drive"));
                        l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    }
                    break;

                //================================================
                case CGlobalEnumerations::DTY_CDDA:
                //================================================
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
                    {
                        ETG_TRACE_COMP(("AddDevice: DTY_CDDA"));
                        l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    }
                    break;
                //================================================
                case CGlobalEnumerations::DTY_DVD_DRIVE:
                //================================================
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
                    {
                        ETG_TRACE_COMP(("AddDevice: DTY_DVD_DRIVE"));
                        //Add _DRIVE as devicenode for DVD Drive to differentiate between DVD drive mounted and the CD device inserted in this drive having the same mountpoint
                        l_pCurDevice->m_cDevnode  += GENSTRING("_DRIVE");
                        l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    }
                    break;

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                //================================================
                case CGlobalEnumerations::DTY_NOT_SUPPORTED:
                //================================================
                    ETG_TRACE_COMP(("AddDevice: Unsupported Device detected"));
                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    break;
#endif

                    //DTY_HUB & DTY_DIGIAUX are both HUBs DetectHubDevices(...) makes it available to diagnosis in GM
                //================================================
                case CGlobalEnumerations::DTY_HUB:
                //================================================

                    if(DEVICEMANAGER_OK == DetectHubDevices(l_pCurDevice))
                    {
                        if((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_CUSTOMER_HUBS)))
                        {
                            ETG_TRACE_COMP(("AddDevice: DTY_HUB "));
                            if(TRUE == ConfigurationManager::GetInstance()->bFilterIsInternalHub(l_pCurDevice->m_iVendorID,l_pCurDevice->m_iProductID))  //is Internal Hub
                            {
                                ETG_TRACE_ERR (("AddDevice: Hub is internalHub.. No need to update to clients"));
                                l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NODEVICE;
                                break;
                            }
                            else
                            {
                                if(TRUE == ConfigurationManager::GetInstance()->isExternalHub(l_pCurDevice->m_eUSB,l_pCurDevice->m_iVendorID,l_pCurDevice->m_iProductID)) //is Customer Hub
                                {
                                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                                    l_pCurDevice->m_bIsConnected = true;
                                    l_pCurDevice->m_eDeviceUnsupportedReason = OK; // Customer Hub
                                }
                                else
                                {
                                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                                    l_pCurDevice->m_bIsConnected = true;
                                    l_pCurDevice->m_eDeviceUnsupportedReason = HubButNotCustomerHub;  //NOT a customer Hub
                                }
                                break;
                            }
                        }
                        else
                        {
                            l_iDevProcessingStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                            ETG_TRACE_ERR (("AddDevice: New device detected... Hub-unsupported Reason not updated"));
                            break;
                        }
                    }
                    else
                    {
                        l_iDevProcessingStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        ETG_TRACE_ERR (("AddDevice: Hub already connected"));
                        break;
                    }
                //================================================
                case CGlobalEnumerations::DTY_DIGIAUX:
                //================================================
                {
                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;

                    //DetectHubDevices (l_pCurDevice); extended detection in GetdeviceType in usbutils - now Uwired HUB is recognized as DTY_HUB that#s why this line is commented
                    ETG_TRACE_COMP(("AddDevice: DTY_DIGIAUX "));
                    GENSTRING l_pcAlsaHwName;
                    GENSTRING l_pcHidDevice;

                    int iRetrial = 0;
                    while(iRetrial < 2)
                    {
                        if(DEVICEMANAGER_OK == GetDeviceFileName( m_pDev,l_pcAlsaHwName,l_pcHidDevice))
                        {
                            l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                            if(iRetrial >0)
                            {
                                ETG_TRACE_FATAL(("AddDevice: DTY_DIGIAUX the alsa device name obtained %s",l_pcAlsaHwName.toStdString().c_str()));
                            }
                            ETG_TRACE_COMP(("AddDevice: DTY_DIGIAUX the alsa device name obtained %s",l_pcAlsaHwName.toStdString().c_str()));
                            l_pCurDevice->m_cAccessoryName = l_pcAlsaHwName;

                            break; //found leave while loop
                        }
                        else
                        {
                            iRetrial++;
                            l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NODEVICE;
                            usleep((useconds_t)(USE_DELAY_RETRIAL_SEARCH_SOUNDDEV*1000));
                            ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                            ETG_TRACE_FATAL(("Retrial after %d ms - looking for DTY_AUXIN-sound device",USE_DELAY_RETRIAL_SEARCH_SOUNDDEV));
                            ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));

                        }
                    }

                    l_pCurDevice->m_cMountPoint = "UNKNOWN";
                    l_pCurDevice->m_bIsConnected = true;
                }
                    break;
                //================================================
                case CGlobalEnumerations::DTY_BTA:
                //================================================
                {
                    ETG_TRACE_COMP(("AddDevice: DTY_BTA "));
                    l_iDevProcessingStatus = DEVICEMANAGER_DEVICE_DETECTED;
                    l_pCurDevice->m_cMountPoint = "UNKNOWN";
                    l_pCurDevice->m_bIsConnected = true;
                }
                    break;
                    /* Commented as DTY_MISCELLANEOUS is checked in DTY_MTP case (All miscellaneous devices are reported as MTP devices(if valid)
#ifdef USE_MYSPIN_CHANGE
                case CGlobalEnumerations::DTY_MISCELLANEOUS:
                {
                    ETG_TRACE_COMP(("AddDevice: DTY_MISCELLANEOUS "));
                    l_iDevProcessingStatus          = DEVICEMANAGER_DEVICE_DETECTED;
                    l_pCurDevice->m_cMountPoint     = "MISCELLANEOUS";
                    l_pCurDevice->m_cAccessoryName  = l_pCurDevice->m_cShortSerial;
                    l_pCurDevice->m_eDeviceType     = CGlobalEnumerations::DTY_NOT_SUPPORTED;


                }
                    break;
#endif*/
                //================================================
                case CGlobalEnumerations::DTY_UNKNOWN:
                //================================================
                {
                    ETG_TRACE_COMP(("AddDevice: DTY_UNKNOWN "));
                    if((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_UNKNOWN_DEVICES)))
                    {
                        ETG_TRACE_COMP(("AddDevice: l_pCurDevice->m_eDeviceClass = %d",l_pCurDevice->m_eDeviceClass));
                        if(l_pCurDevice->m_eDeviceClass != (CGlobalEnumerations::DEVICE_CLASS_MASS_STORAGE))
                        {
                            l_iDevProcessingStatus          = DEVICEMANAGER_DEVICE_DETECTED;
                            l_pCurDevice->m_cAccessoryName  = l_pCurDevice->m_cShortSerial;
                            l_pCurDevice->m_bIsConnected    = true;

                            if((CGlobalEnumerations::INTERFACE_DEVICE_CLASS_MTP == l_pCurDevice->m_eInterfaceDeviceClass) || (CGlobalEnumerations::INTERFACE_DEVICE_CLASS_AUDIO == l_pCurDevice->m_eInterfaceDeviceClass))
                            {
                                l_pCurDevice->m_eDeviceUnsupportedReason = PossiblySupportedByOtherClient;
                            }
                            else
                            {
                                l_pCurDevice->m_eDeviceUnsupportedReason  = NotSupportedByDVM;
                            }
                            l_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_NOT_SUPPORTED;
                            l_pCurDevice->m_cMountPoint = l_pCurDevice->m_cSerialID;  // Setting to avoid empty mountpoint
                        }
                        else
                        {
                            ETG_TRACE_ERR (("AddDevice: Unknown device connected is a Mass storage device. Will be pushed in Lookupnotmounted"));
                        }
                    }
                    else
                    {
                        l_iDevProcessingStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        ETG_TRACE_ERR (("AddDevice: Unknown device connected but NO configuration"));

                    }
                    ETG_TRACE_COMP(("AddDevice: l_pCurDevice->m_eDeviceUnsupportedReason = %d",l_pCurDevice->m_eDeviceUnsupportedReason));
                    break;
                }

                default:
                    break;
            }

            // if supported device is found
            if (DEVICEMANAGER_DEVICE_DETECTED == l_iDevProcessingStatus)
            {
                DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, l_pCurDevice);

                if (true != l_pCurDevice->m_cSerialID.isEmpty())
                {
                    switch (l_pCurDevice->m_eDeviceType)
                    {
                    case CGlobalEnumerations::DTY_USB:
                    case CGlobalEnumerations::DTY_SD:
                    case CGlobalEnumerations::DTY_SD_INTERNAL:
                    case CGlobalEnumerations::DTY_CDROM:
                    case CGlobalEnumerations::DTY_CDDA:
                    case CGlobalEnumerations::DTY_DVD_DRIVE:
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                    case CGlobalEnumerations::DTY_NOT_SUPPORTED:
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
                        l_pCurDevice->m_bIsConnected = true;
                        break;

                    case CGlobalEnumerations::DTY_IPOD:
                    case CGlobalEnumerations::DTY_IPHONE:
                    case CGlobalEnumerations::DTY_IPAD:

                        if((CGlobalEnumerations::DTY_IPAD == l_pCurDevice->m_eDeviceType)
                                && (DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_IPAD)))
                        {
                            ETG_TRACE_USR4 (("AddDevice: DTY_IPAD treated as UNKNOWN Device"));
                            break;
                        }

                        if (DEVICEMANAGER_OK ==
                                GetDeviceFileName (m_pDev,
                                                   l_pCurDevice->m_cAccessoryName,
                                                   l_pCurDevice->m_cMountPoint))
                        {

                            ETG_TRACE_USR2(("AddDevice: MountPoint    = %s",
                                            l_pCurDevice->m_cMountPoint.toStdString().c_str() ));
                            ETG_TRACE_USR2(("AddDevice: DeviceName    = %s",
                                            l_pCurDevice->m_cDeviceName.toStdString().c_str() ));
                            ETG_TRACE_USR2(("AddDevice: AccessoryName = %s",
                                            l_pCurDevice->m_cAccessoryName.toStdString().c_str() ));

                            l_pCurDevice->m_bIsConnected = true;
                        }
                        else
                        {
                            ETG_TRACE_ERR (("AddDevice: [ERROR] GetDeviceFileName failed"));

                            l_pCurDevice->m_bIsConnected = false;

                        } // if (GetDeviceFileName succeeded)
                        ETG_TRACE_USR2(("AddDevice:Serial Id will be filled as mount point"));
                        l_pCurDevice->m_cMountPoint = l_pCurDevice->m_cShortSerial;



#ifdef INCOMPLETE_DEV
                        if(false == l_pCurDevice->m_bIsConnected)
                        {
                            ETG_TRACE_USR1 (("AddDevice: Pushing to incomplete device list = %s",
                                             l_pCurDevice->m_cDevpath.toStdString().c_str()));
                            m_vIncompleteDevices.push_back (*l_pCurDevice);
                        }
#endif // INCOMPLETE_DEV

                        break;

                    case CGlobalEnumerations::DTY_MTP:
                        if ( 0xFF != l_pCurDevice->m_iDevNUM)
                        {
                            l_pCurDevice->m_bIsConnected = true;
                        }
                        break;



                    default:
                        ETG_TRACE_USR4 (("AddDevice: UNKNOWN Device"));

                        break;
                    } // switch (l_pCurDevice->m_eDeviceType)

                    l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NODEVICE;

                    // new device is connected
                    if (true == l_pCurDevice->m_bIsConnected)
                    {
                        ETG_TRACE_USR4 (("AddDevice: Pushing the device data"));

                        adjustDeviceName (l_pCurDevice);
                        l_iDevProcessingStatus = PushDeviceData(l_pCurDevice);
                    }

                    if (DEVICEMANAGER_DEVICE_DETECTED == l_iDevProcessingStatus)
                    {
                        ETG_TRACE_USR2(("AddDevice: Device pushed successfully"));

                        ETG_TRACE_USR2(("AddDevice: HUB : m_cParentUsb1Hub  = %s",
                                        m_cParentUsb1Hub.toStdString().c_str()));
                        ETG_TRACE_USR2(("AddDevice: HUB : m_cParentHub1Node = %s",
                                        l_pCurDevice->m_cParentSerialId.toStdString().c_str() ));


                        if (true == m_bConnectUDEV)

                        {
                            ETG_TRACE_SYS(("AddDevice: m_cMountPoint = %s",
                                           l_pCurDevice->m_cMountPoint.toStdString().c_str()));
                            ETG_TRACE_COMP(("AddDevice: m_cDeviceName = %s",
                                            l_pCurDevice->m_cDeviceName.toStdString().c_str()));

                            if( l_pCurDevice->m_eTypeOfNotify != CGlobalEnumerations::enInUnsupDev )
                            {
                                eUsb = m_pStateTable->GetConnector(l_pCurDevice);
                                l_pCurDevice->m_eUSB = eUsb;
                            }
                            l_pCurDevice->m_eConnectStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
                            m_pStateTable->setMount( enMountUdev, l_pCurDevice->m_eUSB, SIG_TRUE, l_pCurDevice );  // calls: bFillstructNotifyClients(...)

                            /* The call to statfs() in CheckIfMounted in case of an USB Storage device    //@todo it is not needed anymore device is notified if it is mounted
                                                         * takes about 1 second and this delays the notification for the new device
                                                         * Now, the storage info is fetched after notfication is sent.
                                                         */

                            // Check for storage device
                            if(   l_pCurDevice->m_eDeviceType == CGlobalEnumerations::DTY_USB
                                  || l_pCurDevice->m_eDeviceType == CGlobalEnumerations::DTY_SD
                                  /*|| l_pCurDevice->m_eDeviceType == CGlobalEnumerations::DTY_SD_INTERNAL*/) //@todo it is not yet clear to me if this is necessary has to be tested futhermore code cleanpu for unnecessary #define has to be done
                            {
                                ETG_TRACE_USR1 (("AddDevice: Filling disk details for Diagnosis"));

                                // Filling disk details for Diagnosis
                                unsigned int uiSize;
                                GENSTRING l_cDevNode = l_pCurDevice->m_cDevnode;
                                //l_pCurDevice = IsAlreadyConnected (l_cDevNode);

                                uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);
                                CDevice oDevice;
                                tBool bExists;

                                for (unsigned int l_iDevCount = 0; l_iDevCount < uiSize; l_iDevCount++)
                                {
                                    bExists = DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_iDevCount, oDevice);
                                    NORMAL_M_ASSERT(TRUE == bExists);

                                    if(TRUE == bExists)
                                    {
                                        ETG_TRACE_USR4(("AddDevice:l_cDevNode        :%s",l_cDevNode.toStdString().c_str()));
                                        ETG_TRACE_USR4(("AddDevice:oDevice.m_cDevnode:%s",oDevice.m_cDevnode.toStdString().c_str()));
                                        if (l_cDevNode.contains(oDevice.m_cDevnode, CASE_INSENSITIVE)
                                                || l_cDevNode == oDevice.m_cDevnode)        //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                                        {
                                            ETG_TRACE_USR4 (("AddDevice: Device found in the list"));


#ifdef WRITEHISTORYTOUSBSTICK  //check connected device for certain file with key if it exists then analysis information is written to the stick
                                            // check whether key file is present
                                            ETG_TRACE_USR4 (("AddDevice: Check for key log file"));

                                            if (CheckKeyFile(CGlobalEnumerations::KEY_LOGDGBINFO,oDevice.m_cMountPoint.toStdString().c_str()))

                                            {
                                                //switch off diag to prevent timout of semaphore
                                                //after next startup it is active again if it was
                                                //not switched off permanentely
                                                ETG_TRACE_FATAL(("AddDevice:disable diagnosis"));
                                                DeviceManagerInterface::GetInstance()->vDiagEnable(FALSE);
                                                sleep(4 /*sec*/);//some delay to ensure that diag has finished @todo solve better

                                                oDevice.m_bIsLogDump = true;
                                                HISTORYMANAGER->InitDumpHistoryToUSBStick(oDevice.m_cMountPoint.toStdString().c_str());
                                                HISTORYMANAGER->DumpHistoryToUSBStick();
                                                HISTORYMANAGER->DumpPlatformHistoryToUSBStick();
                                            }
                                            else
                                            {
                                                ETG_TRACE_USR1 (("AddDevice: Logging not requested"));
                                            }
#endif //WRITEHISTORYTOUSBSTICK

                                            l_pCurDevice->FillDevice(oDevice);

                                            //care that the additional size info is updated in central list
                                            bExists = DeviceListManager::GetInstance()->setElement(DEVLIST_UDEVMGR,l_iDevCount, oDevice);
                                            if(FALSE == bExists)
                                            {
                                                ETG_TRACE_FATAL(("[ERROR] AddDevice: bExists=FALSE"));
                                            }
                                        }
                                        else
                                        {
                                            // Continue checking the list
                                        }
                                    }
                                }


                            } // if (a storage device)

                            // Display the newly added device
                            l_pCurDevice->Show();
                        }
                        else
                        {
                            ETG_TRACE_SYS(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                            ETG_TRACE_SYS(("      AddDevice:FALSE == m_bConnectUDEV (m_bConnectUDEV=0x%x)  ",m_bConnectUDEV));
                            ETG_TRACE_SYS(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                        }
                    }
                }
                else
                {
                    ETG_TRACE_FATAL (("AddDevice: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                    ETG_TRACE_FATAL (("AddDevice: !!!!    Serial ID is empty    !!!!"));
                    ETG_TRACE_FATAL (("AddDevice: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));

                    l_iDevProcessingStatus = DEVICEMANAGER_ERROR_NOTVALID;
                }// if(true != l_pCurDevice->m_cShortSerial.isEmpty())
            } // if (supported device is found)
            else
            {

            }
        }
        else
        {
            ETG_TRACE_ERR (("AddDevice: Error filling device details"));
        } // if (FillCurDevice succeeds)
    } // if (new CDevice is allocated)

    // Reset m_pBaseUSBDev to NULL
    m_pBaseUSBDev = NULL;

    ETG_TRACE_USR4 (("End  : AddDevice (with return value %i)", ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus) ));

    return l_iDevProcessingStatus;
}

/*-----------------------------------------------------------------------------*
 * int RemoveDevice (GENSTRING f_cDevNode)                                       *
 *-----------------------------------------------------------------------------*/
int CUDevManager::RemoveDevice(GENSTRING f_cDevNode)
{
    ETG_TRACE_USR4(("Begin: RemoveDevice"));

    int l_iRemoveStatus = DEVICEMANAGER_ERROR_UNKNOWN;

    if (!f_cDevNode.isEmpty())
    {
        /* NOTE: UNLIKE 'add' action,  the parent 'usb_device' while device
         * removal is not generated as by the time we start to process
         * the removed device, the parent usb device is already removed.
         * And any call to get the parent usb_device gives us the parent
         * HUB device. And this causes wrong behaviour
         */
        ETG_TRACE_USR4 (("RemoveDevice: ---------DEVICE TO REMOVE------------------------------"));
        ETG_TRACE_USR4 (("RemoveDevice: | Node     : %s  ", udev_device_get_devnode(m_pBaseUSBDev)));
        ETG_TRACE_USR4 (("RemoveDevice: -------------------------------------------------------"));

        ETG_TRACE_USR4(("RemoveDevice: f_cDevNode = %s", f_cDevNode.toStdString().c_str()));
        ETG_TRACE_USR2(("RemoveDevice: HUB : m_cParentUsb1Hub  = %s", m_cParentUsb1Hub.toStdString().c_str()));
        ETG_TRACE_USR2(("RemoveDevice: HUB : m_cParentUsb2Hub  = %s", m_cParentUsb2Hub.toStdString().c_str()));
        ETG_TRACE_USR2(("RemoveDevice: HUB : m_cParentHub1Node = %s", m_cParentHub1Node.toStdString().c_str()));
        ETG_TRACE_USR2(("RemoveDevice: HUB : m_cParentHub2Node = %s", m_cParentHub2Node.toStdString().c_str()));

        CGlobalEnumerations::HUB_NUM_t eHubNo = CGlobalEnumerations::HUB_UNDEFINED;
        GENSTRING sysName = udev_device_get_sysname(m_pBaseUSBDev);
        bool isHubConnected = IsHubAlreadyConnected(sysName, eHubNo);

        if(isHubConnected && CGlobalEnumerations::HUB_1 == eHubNo)  //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
        {
            ETG_TRACE_USR2(("RemoveDevice: Removed Device is Hub 1 m_bConnectUDEV=0x%x", m_bConnectUDEV));

            m_cParentHub1Node = "";
            m_cParentUsb1Hub  = "";
            m_iUsb1HubPort    = 0;

            //TODO(kmv5cob):Check if this is really required, else use DisconnectNotification().
            DisconnectNotificationHUB(f_cDevNode, CGlobalEnumerations::HUB1);
        } // if (hub 1 node is removed)
        else if(isHubConnected && CGlobalEnumerations::HUB_2 == eHubNo)      //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
        {
            ETG_TRACE_USR2(("RemoveDevice: Removed Device is Hub 2  m_bConnectUDEV=0x%x",m_bConnectUDEV));

            m_cParentHub2Node = "";
            m_cParentUsb2Hub  = "";
            m_iUsb2HubPort    = 0;

            DisconnectNotificationHUB(f_cDevNode, CGlobalEnumerations::HUB2);
        } // if (hub 2 node is removed)
        else
        {
            ETG_TRACE_USR2(("RemoveDevice: Removed Single device from list m_bConnectUDEV=0x%x",m_bConnectUDEV));

            l_iRemoveStatus = DisconnectNotification(f_cDevNode, CGlobalEnumerations::SINGLE);
        } // if (its an individual device)
    }
    else
    {
        ETG_TRACE_USR2(("RemoveDevice: [ERROR] Invalid params"));

        l_iRemoveStatus = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if (valid params)

    ETG_TRACE_USR4 (("End  : RemoveDevice"));

    return l_iRemoveStatus;
}


/*-----------------------------------------------------------------------------*
 * tVoid adjustDeviceName(CDevice* f_pCurDevice)                               *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::adjustDeviceName(CDevice* f_pCurDevice) const
{
    ETG_TRACE_USR4(("Begin: adjustDeviceName"));

#ifdef WORKAROUND_MTP_NODEVICENAME
    GENSTRING l_tmpString;
#endif

    switch(f_pCurDevice->m_eDeviceType)
    {
    case CGlobalEnumerations::DTY_USB:
    case CGlobalEnumerations::DTY_SD:
        f_pCurDevice->m_cDeviceName = f_pCurDevice->m_cAccessoryName; //@todo why do previous processes don't write directly in the correct way - why this wired switching

        // send disk node for hash calculation
        f_pCurDevice->m_cAccessoryName = f_pCurDevice->m_cDevnode;

        break;
    case CGlobalEnumerations::DTY_SD_INTERNAL:
    case CGlobalEnumerations::DTY_CDROM:
        break;
    case CGlobalEnumerations::DTY_IPOD:
    case CGlobalEnumerations::DTY_IPAD:
    case CGlobalEnumerations::DTY_IPHONE:
        if((CGlobalEnumerations::DTY_IPAD == f_pCurDevice->m_eDeviceType) &&
                (DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_IPAD)))
        {
            ETG_TRACE_USR4(("adjustDeviceName: DTY_IPAD diasbled behaviour like default"));
            break;
        }
        f_pCurDevice->m_cDeviceName = "";
        break;
    case CGlobalEnumerations::DTY_MTP:
#ifdef WORKAROUND_MTP_NODEVICENAME
        l_tmpString = f_pCurDevice->m_cDeviceName;
        l_tmpString.remove(" "); //remove all spaces
        if(l_tmpString.isEmpty())
        {
            l_tmpString = f_pCurDevice->m_cShortSerial.left(6);
            f_pCurDevice->m_cDeviceName = f_pCurDevice->m_cManufacturer + GENSTRING(" ") + f_pCurDevice->m_cAccessoryName + GENSTRING("  SN") + l_tmpString + GENSTRING("...");
            ETG_TRACE_FATAL(("[INFO] adjustDeviceName (MTP) : %s", f_pCurDevice->m_cDeviceName.toStdString().c_str()));
        }
        break;
#endif
    case CGlobalEnumerations::DTY_BLUETOOTH:
    case CGlobalEnumerations::DTY_MSZUNE:
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
    case CGlobalEnumerations::DTY_NOT_SUPPORTED:
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
        //don't care
        break;
    default:
        break;
    }
    ETG_TRACE_USR4(("End    :adjustDeviceName"));
}

/*-----------------------------------------------------------------------------*
 * int DetectHubDevices(CDevice* f_pCurDevice)                                 *
 *-----------------------------------------------------------------------------*/
int CUDevManager::DetectHubDevices( const CDevice* f_pCurDevice)
{
    ETG_TRACE_USR4(("Begin: DetectHubDevices"));

    //variable to hold the return value
    int l_iHubFound = DEVICEMANAGER_ERROR_NODEVICE;


    //check pointer
    if (NULL != f_pCurDevice)
    {
        GENSTRING l_cSysName = GENSTRING(udev_device_get_sysname(m_pBaseUSBDev));

        ETG_TRACE_USR3(("DetectHubDevices l_cSysName = %s ",l_cSysName.toStdString().c_str()))
                // Check whether already added
        if (NULL == IsHubAlreadyConnected(l_cSysName))
        {
            ETG_TRACE_USR3 (("DetectHubDevices: New device found"));

            // Find whether the device is an hub or not
            // NOTE: need to find better way to detect whether the device is an hub or not
            GENSTRING l_cPath = udev_device_get_syspath(m_pBaseUSBDev);
            l_cPath.append("/");
            l_cPath.append(udev_device_get_sysname(m_pBaseUSBDev));
            l_cPath.append(":1.0");

            struct udev_device *dev = udev_device_new_from_syspath (m_pUdev, l_cPath.toStdString().c_str());

            if (NULL == dev)
            {
                // Device does not support an interface
                ETG_TRACE_USR1 (("DetectHubDevices: Device does not support an interface"));

                l_iHubFound = DEVICEMANAGER_ERROR_NODEVICE;
            }
            else
            {
                ETG_TRACE_USR4 (("DetectHubDevices: DRIVER: %s", udev_device_get_driver(dev)));

                if ((NULL != udev_device_get_driver(dev))
                        && (0 == strcmp (udev_device_get_driver(dev), "hub")))
                {
                    // Free the udev device
                    udev_device_unref (dev);
                    dev = NULL;

                    ETG_TRACE_USR1 (("DetectHubDevices: Device is HUB"));

                    CGlobalEnumerations::HubDetails_t *l_poHub = new(CGlobalEnumerations::HubDetails_t);

                    if (NULL != l_poHub)
                    {

                        tenDevManagerUSBHost l_eUsb       = eUSBUndef;
                        ConfigurationManager::GetInstance()->vGetUSBConnectorFromSysPath(OUT l_eUsb, IN f_pCurDevice->m_cDevpath.toStdString());


                        //if ((GENSTRING(udev_device_get_sysname(m_pBaseUSBDev))).contains(SUBSTRING_USBConnector1,CASE_SENSITIVE))
                        if(eUSB1 == l_eUsb)
                        {
                            ETG_TRACE_USR1 (("DetectHubDevices: Device is HUB_1"));

                            l_poHub->eHubNo = CGlobalEnumerations::HUB_1;

                            m_cParentHub1Node = GENSTRING(udev_device_get_sysname(m_pBaseUSBDev));

                            l_iHubFound = DEVICEMANAGER_OK;

                        }
                        else if(eUSB2 == l_eUsb)
                            //else if ((GENSTRING(udev_device_get_sysname(m_pBaseUSBDev))).contains(SUBSTRING_USBConnector2,CASE_SENSITIVE))
                        {
                            ETG_TRACE_USR1 (("DetectHubDevices: Device is HUB_2"));

                            l_poHub->eHubNo = CGlobalEnumerations::HUB_2;

                            m_cParentHub2Node = GENSTRING(udev_device_get_sysname(m_pBaseUSBDev));

                            l_iHubFound = DEVICEMANAGER_OK;
                        }
                        else if(eUSB3 == l_eUsb)
                            //else if ((GENSTRING(udev_device_get_sysname(m_pBaseUSBDev))).contains(SUBSTRING_USBConnector3,CASE_SENSITIVE))
                        {
                            ETG_TRACE_USR1 (("DetectHubDevices: Device is HUB_3"));

                            l_poHub->eHubNo = CGlobalEnumerations::HUB_3;

                            m_cParentHub2Node = GENSTRING(udev_device_get_sysname(m_pBaseUSBDev));

                            l_iHubFound = DEVICEMANAGER_OK;
                        }
#ifdef USE_FOURTH_USB_CONNECTOR
                        else if (eUSB4 == l_eUsb)
                            //else if ((GENSTRING(udev_device_get_sysname(m_pBaseUSBDev))).contains(SUBSTRING_USBConnector4,CASE_SENSITIVE))
                        {
                            ETG_TRACE_USR1(("DetectHubDevices: Device is HUB_4"));

                            l_poHub->eHubNo = CGlobalEnumerations::HUB_4;

                            m_cParentHub2Node = GENSTRING(udev_device_get_sysname(m_pBaseUSBDev));

                            l_iHubFound = DEVICEMANAGER_OK;
                        }
#endif//USE_FOURTH_USB_CONNECTOR
                        else
                        {
                            ETG_TRACE_USR1 (("DetectHubDevices: Cannot reach here"));

                            l_iHubFound = DEVICEMANAGER_ERROR_NOTVALID;
                        }

                        // If valid hub was found
                        if (DEVICEMANAGER_OK == l_iHubFound)
                        {
                            l_poHub->eHUBFound = CGlobalEnumerations::HUB_FOUND_YES;

                            // Get the port count
                            unsigned int l_uiPortCount = 0;
                            const char *l_cPortCount = udev_device_get_sysattr_value (m_pBaseUSBDev, "maxchild");

                            if (NULL != l_cPortCount)
                            {
                                ETG_TRACE_ERR (("DetectHubDevices: 'maxchild' = %s", l_cPortCount));

                                sscanf (l_cPortCount, "%u", &l_uiPortCount);
                            }
                            else
                            {
                                ETG_TRACE_ERR (("DetectHubDevices: [ERROR] can't get 'maxchild' property"));
                            }

                            l_poHub->uiPortCount = l_uiPortCount;

                            // Check whether the hub has a card reader
                            l_poHub->bHasCardReader = false;

                            if (0 == l_poHub->uiPortCount)
                            {
                                ETG_TRACE_USR1 (("DetectHubDevices: No ports are there. No need to check for card reader"));
                            }
                            else
                            {
                                ETG_TRACE_USR1 (("DetectHubDevices: Checking for card reader"));

                                l_cPath = udev_device_get_syspath(m_pBaseUSBDev);
                                l_cPath.append("/");
                                l_cPath.append(udev_device_get_sysname(m_pBaseUSBDev));

                                for( unsigned int iCount = 1; iCount <= l_poHub->uiPortCount; iCount++)
                                {
                                    GENSTRING l_cPath2 = l_cPath + GENSTRING(".") + GENSTRING_NUMBER(iCount);

                                    ETG_TRACE_USR4 (("DetectHubDevices: Port[%d] SysPath = %s", iCount, l_cPath2.toStdString().c_str()));

                                    dev = udev_device_new_from_syspath (m_pUdev, l_cPath2.toStdString().c_str());

                                    if (NULL != dev)
                                    {
                                        char product[1024];

                                        if (NULL != udev_device_get_sysattr_value(dev,"product"))
                                        {
                                            strncpy (product, udev_device_get_sysattr_value(dev,"product"), sizeof(product)-1); // sim4hi: fix for Coverity CID-18520
                                            product[sizeof(product)-1]= '\0';

                                            if ((strstr (product, "Reader")) || (strstr (product, "reader")))
                                            {
                                                ETG_TRACE_USR1 (("DetectHubDevices: Card Reader is connected"));

                                                l_poHub->bHasCardReader = true;

                                                break;
                                            }
                                        }

                                        // Free the udev device
                                        udev_device_unref (dev);
                                        dev = NULL;
                                    }
                                    else
                                    {
                                        // No device not attached to port
                                    }
                                } // for (all the ports on hub)
                            }

                            // Add to the list of Hubs
                            ETG_TRACE_USR1 (("DetectHubDevices: Hub pushed to list"));

                            //-------------------------
                            // add to HUB list
                            //-------------------------
                            l_poHub->sSysName = l_cSysName;
                            ETG_TRACE_USR4(("DetectHubDevices:-------------m_vHubs.push_back(l_poHub)-------------------------"));
                            ETG_TRACE_USR4(("DetectHubDevices:l_poHub->sSysName  :       %s",   l_poHub->sSysName.toStdString().c_str()));
                            ETG_TRACE_USR4(("DetectHubDevices:l_poHub->eHubNo     :       %d" ,  l_poHub->eHubNo));
                            ETG_TRACE_USR4(("DetectHubDevices:l_poHub->eHUBFound  :     0x%x", l_poHub->eHUBFound));
                            ETG_TRACE_USR4(("DetectHubDevices:l_poHub->uiPortCount:       %d",l_poHub->uiPortCount));
                            ETG_TRACE_USR4(("DetectHubDevices:l_poHub->bHasCardReader:  0x%x",l_poHub->bHasCardReader));
                            ETG_TRACE_USR4(("DetectHubDevices:---------------------------------------------------------------"));

                            m_tableLock.lock();
                            this->m_vHubs.push_back(l_poHub);
                            m_tableLock.unlock();
                        }
                        //if (NULL != l_poHub) delete l_poHub; // sim4hi: fix for Coverity CID-16672
                    }
                    else
                    {
                        ETG_TRACE_FATAL (("DetectHubDevices: [ERROR] Memory allocation failed"));

                        l_iHubFound = DEVICEMANAGER_ERROR_MEM_ALLOC;
                    }
                } //lint !e429 Custodial pointer 'l_poHub' (line xxx) has not been freed or returned
                // This OK because with m_vHubs.push_back( ) the pointer is copied and used. Later
                // the vector m_vHubs is freed at least in destructor with for loop, delete pointer
                // and m_vDevices.clear()
                else
                {
                    ETG_TRACE_USR1 (("DetectHubDevices: Not a 'hub' device"));

                    l_iHubFound = DEVICEMANAGER_ERROR_NODEVICE;
                }
            }
        }
        else
        {
            ETG_TRACE_USR3 (("DetectHubDevices: Hub already added"));
        } // if (hub already present)
    }
    else
    {
        ETG_TRACE_ERR(("DetectHubDevices: [ERROR] NULL Pointer f_pCurDevice"));
        l_iHubFound = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if( NULL != f_pCurDevice )

    ETG_TRACE_USR4 (("End  : DetectHubDevices"));

    return l_iHubFound;
}

/*-----------------------------------------------------------------------------*
 *CGlobalEnumerations::HubDetails_t* IsHubAlreadyConnected(GENSTRING l_cSysName)                     *
 *-----------------------------------------------------------------------------*/
CGlobalEnumerations::HubDetails_t* CUDevManager::IsHubAlreadyConnected(GENSTRING l_cSysName)
{

    ETG_TRACE_USR4 (("Begin: IsHubAlreadyConnected: m_vHubs.size():%d; l_cSysName:%s ", m_vHubs.size(),l_cSysName.toStdString().c_str() ));

    CGlobalEnumerations::HubDetails_t *l_pFoundDevice = NULL;

    if (l_cSysName.isEmpty())
    {
        ETG_TRACE_ERR (("IsHubAlreadyConnected: [ERROR] l_cSysName is EMPTY"));
        ETG_TRACE_ERRMEM(("IsHubAlreadyConnected: [ERROR] l_cSysName is EMPTY"));
    }
    else
    {
        if(m_vHubs.size() >0)
        {
            ETG_TRACE_USR4(("IsHubAlreadyConnected: -----------Check List-------------------"));
            for( unsigned int l_iDevCount = 0; l_iDevCount < m_vHubs.size(); l_iDevCount++)
            {
               if(!(m_vHubs[l_iDevCount]->sSysName.isEmpty()))
               {
                if (l_cSysName == m_vHubs[l_iDevCount]->sSysName)     //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_ERR (("IsHubAlreadyConnected: Device found in the list"));

                    l_pFoundDevice = m_vHubs[l_iDevCount];
                }
                else
                {
                    // Continue checking the list
                    }
                }
                else
                {
                    ETG_TRACE_USR4(("IsHubAlreadyConnected:m_vHubs[l_iDevCount]->sSysName is NULL"));
                }
            } // for (all devices in m_vHubs)
            ETG_TRACE_USR4(("IsHubAlreadyConnected: ----------------------------------------"));
        }
    } // if (f_cDeviceNode.isEmpty())

    ETG_TRACE_USR4 (("End  : IsHubAlreadyConnected"));

    return l_pFoundDevice;
}

/*-----------------------------------------------------------------------------*
 * CGlobalEnumerations::HubDetails_t* GetHub(CGlobalEnumerations::HUB_NUM_t f_eHubNum)              *
 *-----------------------------------------------------------------------------*/
CGlobalEnumerations::HubDetails_t* CUDevManager::GetHub(CGlobalEnumerations::HUB_NUM_t f_eHubNum)
{
    ETG_TRACE_USR4 (("Begin : GetHub"));

    CGlobalEnumerations::HubDetails_t *l_poHub = NULL;

    if (0 < m_vHubs.size())
    {

        if (!(CGlobalEnumerations::HUB_1 == f_eHubNum || CGlobalEnumerations::HUB_2 == f_eHubNum))
        {
            ETG_TRACE_ERR (("GetHub: Wrong Hub no specified"));
        }
        else
        {
            // Searh in the hub vector
            for( unsigned int l_uiCount  = 0; l_uiCount < m_vHubs.size(); l_uiCount++)
            {
                if (f_eHubNum == m_vHubs[l_uiCount]->eHubNo)
                {
                    l_poHub = m_vHubs[l_uiCount];

                    ETG_TRACE_USR1 (("GetHub: Hub found"));

                    break;
                }
                else
                {
                    // Continue searching the list
                }
            } // for (all hubs)
        } // if (valid hub number is queried)
    }
    else
    {
        ETG_TRACE_USR1 (("GetHub: No hubs present"));
    } // if (hub contains values)

    ETG_TRACE_USR4 (("End : GetHub"));

    return l_poHub;
}

/*-----------------------------------------------------------------------------*
 * void HandleMount(const char *f_cPartitionNode)                              *
 *-----------------------------------------------------------------------------*/
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
int CUDevManager::HandleMountUnsupported( const char *f_cPartitionNode, GENSTRING &mountPoint, CGlobalEnumerations::NOTIFICATION_Type enWNotiDevTyp )
{
    ETG_TRACE_USR4(("Begin: HandleMount"));

    CDevice oCurAvailDevInfo;
    int     l_iDevProcessingStatus = DEVICEMANAGER_OK;
    int     err = 0;
    char*   contentOfFile;

    tUInt iFirstLine   = 0;
    tUInt iLastElement;
    tUInt iLenPartitionNode;

    iLastElement = LINE_LAST_ELEMS[enWNotiDevTyp];

    //initialize memory
    contentOfFile = NULL;
    for( tUInt i=iFirstLine; i<iLastElement; i++ )
    {
        allLineDescript[enWNotiDevTyp][i] = NULL;
    }
    GENSTRING l_cFilePath;
    l_cFilePath = GENSTRING( INotWatch[enWNotiDevTyp] ) + GENSTRING( f_cPartitionNode );

    ETG_TRACE_USR4(( "HandleMount: l_cFilePath = %s", l_cFilePath.toStdString().c_str()) )


            if(CGlobalEnumerations::enInUnsupDev == enWNotiDevTyp)
    {
        FILE* fp = fopen( l_cFilePath.toStdString().c_str(), "r" );
        if( fp != NULL )
        {
            //cp 'device' i.e. /dev/sda1
            iLenPartitionNode = 0;
            if(f_cPartitionNode)
            {
                iLenPartitionNode = strlen( f_cPartitionNode ) + 1; /*string terminating '\0'*/
            }
            tUInt iLenDev = strlen( "/dev/" )+ iLenPartitionNode;

            allLineDescript[enWNotiDevTyp][iFirstLine] = (char*)OSAL_pvMemoryAllocate( iLenDev );
            snprintf( allLineDescript[enWNotiDevTyp][iFirstLine], iLenDev, "/dev/%s", f_cPartitionNode );   // /dev/sda
            //--------------------------------------------------------------------------
            //check size of file /tmp/.mountDB to allocate necessary memory to read it
            //------------------------------------------------------------------------
            fseek( fp, 0L, SEEK_END );
            int fileSize = ftell( fp );
            fseek( fp, 0L, SEEK_SET );
            if( fileSize != -1 )
            {
                contentOfFile  = (char*)OSAL_pvMemoryAllocate( (tU32)fileSize );
            }
            ETG_TRACE_USR4(( "HandleMount: fileSize=%i", fileSize ));
            //-----------------------------------------------------------------------------------
            //read file and examine line by line to extract content using fct.: vExtractContent
            //-----------------------------------------------------------------------------------
            if( contentOfFile )                                                                  // if contentOfFile isn't NULL also fileSize can't be -1
            {
                memset( contentOfFile, '\0', (tU32)fileSize );                         //lint !e671 Warning 671; Lint complains about fileSize could be '-1' -> but with fileSize==-1 contentOfFile stays NULL
                size_t size_read = fread( contentOfFile, 1 /*bytewhise*/, (size_t)fileSize, fp );//lint !e671 Warning 671; Lint complains about fileSize could be '-1' -> but with fileSize==-1 contentOfFile stays NULL
                ETG_TRACE_USR4(( "HandleMount: size_read = %d", size_read ));
                tU16    u16ByteCounter = 0;
                tU16    u16Offset      = 0;
                tU16    u16LenOfEntry  = 0;
                tUInt   lineNo         = iFirstLine;
                while( u16ByteCounter < size_read )  //Example index 0  size = 2 max index = 1 i.e. u16ByteCounter is allowed to be 1 which is ok
                {
                    if( contentOfFile[u16ByteCounter] == '\n' ) //end of line i.e. start to care for this line
                    {
                        u16LenOfEntry = u16ByteCounter - u16Offset;
                        //ETG_TRACE_USR4(( "HandleMount: Line %d u16LenOfEntry=%d", ETG_CENUM(MOUNTDB_Lines, (MOUNTDB_Lines)lineNo), u16LenOfEntry )); // @todo make function to get the right textes for the trace
                        ETG_TRACE_USR4(( "HandleMount: Line %d u16LenOfEntry=%d", ETG_CENUM( UNSUPPORTED_Lines, (UNSUPPORTED_Lines)lineNo), u16LenOfEntry ));
                        allLineDescript[enWNotiDevTyp][lineNo] = (char*)OSAL_pvMemoryAllocate( u16LenOfEntry + 1 );               // +1 to give space for '\0' i.e. End of String
                        memcpy( allLineDescript[enWNotiDevTyp][lineNo], &(contentOfFile[u16Offset]), u16LenOfEntry );  // example: u16LenOfEntry = 3 means index 0,1,2 exists
                        // for content since u16LenOfEntry+1 has been allocated
                        // index = 3 (=u16LenOfEntry) is reserved for '\0'
                        allLineDescript[enWNotiDevTyp][lineNo][u16LenOfEntry] = '\0';
                        switch( enWNotiDevTyp )       // @todo -> mit functions pointern machen??? -> oder lieber doch nicht
                        {
                        case CGlobalEnumerations::enInMscDev:
                            vExtractContentMountDB( oCurAvailDevInfo, strMountDBLine, lineNo, u16LenOfEntry );
                            break;
                        case CGlobalEnumerations::enInUnsupDev:
                            vExtractContentUnsupported( oCurAvailDevInfo, strMountUnsupportedLine, lineNo, u16LenOfEntry );
                            break;
                        default:
                            ETG_TRACE_FATAL(( "HandleMount: Unexpected Notification Type (%i)", ETG_ENUM(NOTIFICATION_Type, enWNotiDevTyp) ));
                        }
                        lineNo++;                         // increase line counter
                        u16Offset = u16ByteCounter + 1;  // @todo add comment (please)
                        ETG_TRACE_USR4(( "HandleMount:u16Offset = %d", u16Offset ));
                        if( lineNo == iLastElement )
                        {
                            ETG_TRACE_USR4(("HandleMount: LINE LASTELEM for notification: %i", ETG_ENUM(NOTIFICATION_Type, enWNotiDevTyp) ));
                            break;
                        }
                    }
                    u16ByteCounter++; //increment to go to next line
                } // while( u16ByteCounter < size_read )

                if( size_read != 0 )
                {
                    vPrintINotifyData( allLineKeyWord[enWNotiDevTyp], allLineDescript[enWNotiDevTyp], iLastElement );
                    if(   (enWNotiDevTyp == CGlobalEnumerations::enInMscDev)
                          && (strMountDBLine[LINE1_MASSSTORAGE_MOUNTPOINT] != NULL)     // @todo mountPoint only for MountDB notification????
                          )
                    {
                        mountPoint = GENSTRING( strMountDBLine[LINE1_MASSSTORAGE_MOUNTPOINT] );
                    }

                    GENSTRING l_cDevNode = GENSTRING("/dev/") + GENSTRING( f_cPartitionNode );
                    if( DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) == TRUE )
                    {
                        ETG_TRACE_USR1 (("HandleMount: Device already connected : %s", l_cDevNode.toStdString().c_str()));
                    }
                    else
                    {
                        ETG_TRACE_USR1(( "HandleMount: New device connected : %s", l_cDevNode.toStdString().c_str() ));
                        // Process the new device
                        l_iDevProcessingStatus = AddDevice( enWNotiDevTyp, l_cDevNode, oCurAvailDevInfo );
                        // Free the udev device
                        if( m_pDev != NULL )              // @todo wird m_pDev noch verwendet????
                        {
                            udev_device_unref( m_pDev );
                            m_pDev = NULL;
                        }
                    }
                    // free allLineDescript[enWNotiDevTyp][lineNo]
                    for( tUInt line=0; line<iLastElement; line++ )
                    {
                        if( allLineDescript[enWNotiDevTyp][line] )
                        {
                            OSAL_vMemoryFree( allLineDescript[enWNotiDevTyp][line] );
                        }
                    }
                } //if( size_read != -1 )
                else
                {
                    err = errno;
                    ETG_TRACE_ERR(( "[ERROR] HandleMount:'read' failed: %s", strerror( err ) ));
                    l_iDevProcessingStatus = DEVICEMANAGER_ERROR_FILE_READ;
                }
                //----------------------------------------------------------
                //delete memory which has been allocated in this function
                //----------------------------------------------------------
                OSAL_vMemoryFree( contentOfFile );
            }
            else
            {
                ETG_TRACE_ERR (("[ERROR] HandleMount:  memory allocation failed: %s", strerror (err)));
                l_iDevProcessingStatus = DEVICEMANAGER_ERROR_FILE_READ;
            }

            fclose(fp);
        }
        else
        {
            err = errno;
            ETG_TRACE_FATAL (("[ERROR] HandleMount: [ERROR] 'open' failed: %s", strerror (err)));
            l_iDevProcessingStatus = DEVICEMANAGER_OK;
        }

    }
    else
    {
        ETG_TRACE_FATAL (("[ERROR] HandleMount: enWNotiDevTyp!= enInUnsupDev")); //this function will become redundant if new automounter is fully introduced
        l_iDevProcessingStatus = DEVICEMANAGER_OK;
    }

    ETG_TRACE_USR4(( "End: HandleMount (returns with value %i)", ETG_ENUM(DEVICEMANAGER_ERROR, l_iDevProcessingStatus) ));

    return l_iDevProcessingStatus;
}
#endif //USE_EXTERNAL_UNSUPPORTED_DEVICES
int CUDevManager::HandleMount (const char *f_cPartitionNode, GENSTRING &mountPoint)
{
    ETG_TRACE_USR4(("Begin: HandleMount"));
    ETG_TRACE_USR4(("HandleMount: f_cPartitionNode:%s",f_cPartitionNode));
    ETG_TRACE_USR4(("HandleMount: mountPoint      :%s",mountPoint.toStdString().c_str()));

    int l_iDevProcessingStatus = DEVICEMANAGER_OK;

    CDevice oDevice;

    char    mnt[PATH_MAX];
    memset( mnt, '\0', sizeof(mnt) );

#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
    std::string strPathfilename = DIR_AUTOMOUNTER_MOUNTEDPARTITIONS_DB_FILENAMEBASE;
    strPathfilename.append(f_cPartitionNode);
    strPathfilename.append(DIR_AUTOMOUNTER_SUFFIX);
    ETG_TRACE_USR4(("HandleMount:strPathfilename:%s",strPathfilename.c_str()));

    IAutomounterAdapter::tenResIAutomounterAdapt_Type enRes = IAutomounterAdapter::enRES_ERROR_GENERAL;
    NORMAL_M_ASSERT(m_pIAutomounterAdapter);
    if(m_pIAutomounterAdapter)
    {
        IAutomounterAdapter::tenResIAutomounterAdapt_Type enRes = m_pIAutomounterAdapter->FillDevice(OUT oDevice, IN  strPathfilename);
    }

    ETG_TRACE_USR4(("HandleMount: ---filled by automounter----------------------->"));
    oDevice.Show(); //although automouner has filled it not everything has been used up till know because we trust old routines more
    ETG_TRACE_USR4(("HandleMount: <----------------------------------------------"));
    ETG_TRACE_USR4(("HandleMount: enRes: %d", ETG_ENUM(TENRESAUTOMOUNTERADAPTTYPE,enRes)));
    snprintf(mnt,sizeof(mnt),"%s",oDevice.m_cMountPoint.toStdString().c_str());
    {
#else
    int err = 0;
    ssize_t size;
    GENSTRING l_cFilePath   = GENSTRING (WATCH) + GENSTRING (f_cPartitionNode);
    int     fd = open(l_cFilePath.toStdString().c_str(), O_RDONLY);
    if (-1 != fd)
    {
        size = read(fd, mnt, sizeof(mnt));
        if (-1 != size)
#endif
        {
            ETG_TRACE_SYS (("HandleMount: >>>>>>>>>MOUNT>>>>>>>>>>>>>>"));
            ETG_TRACE_SYS (("HandleMount: device     :/dev/%s         ", f_cPartitionNode));
            ETG_TRACE_SYS (("HandleMount: mount point: %s             ", mnt));
            ETG_TRACE_SYS (("HandleMount: <<<<<<<<<<<<<<<<<<<<<<<<<<<<"));

            mountPoint = GENSTRING(mnt);
            GENSTRING l_cDevNode = GENSTRING("/dev/") + GENSTRING (f_cPartitionNode);

            if (TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode))

            {
                ETG_TRACE_USR1 (("HandleMount: Device already connected : %s",
                                 l_cDevNode.toStdString().c_str()));
            }
            else
            {
                ETG_TRACE_USR1 (("HandleMount: New device connected : %s",
                                 l_cDevNode.toStdString().c_str()));

                // Generate the udev_device object
                m_pDev = GenerateUDevDevice(m_pUdev, mnt);

                // Process the new device
                //#ifndef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF  //already filled
                oDevice.m_cMountPoint = mountPoint;
                //#endif
                oDevice.m_cPartitionNode = GENSTRING("/dev/") + f_cPartitionNode;
                l_iDevProcessingStatus = AddDevice( CGlobalEnumerations::enInotify, l_cDevNode, oDevice);

                // Free the udev device
                if(NULL != m_pDev)
                {
                    udev_device_unref (m_pDev);
                    m_pDev = NULL;
                }
            }
        }
#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
    }
#else //VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
        else
        {
            err = errno;

            ETG_TRACE_ERR (("HandleMount: [ERROR] 'read' failed: %s", strerror (err)));

            l_iDevProcessingStatus = DEVICEMANAGER_ERROR_FILE_READ;
        }
#ifndef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
        close(fd);
#endif
    }
    else
    {
        err = errno;
        ETG_TRACE_ERR (("HandleMount: [ERROR] 'open' failed: file : %s", l_cFilePath.toStdString().c_str()));
        ETG_TRACE_ERR (("HandleMount: [ERROR] 'open' failed: errno: %s", strerror (err)));

        l_iDevProcessingStatus = DEVICEMANAGER_OK;
    }
#endif //VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF

    ETG_TRACE_USR4(("End: HandleMount"));

    return l_iDevProcessingStatus;
}


/*-----------------------------------------------------------------------------*
 * void HandleUnMount(const char *f_cPartitionNode)                            *
 *-----------------------------------------------------------------------------*/
void CUDevManager::HandleUnMount (const char *f_cPartitionNode)
{
    ETG_TRACE_USR4(("Begin: HandleUnMount"));

    if (NULL != f_cPartitionNode)
    {
        ETG_TRACE_SYS(("HandleUnMount: >>>>>>>>>UNMOUNT>>>>>>>>>>>>>>"));
        ETG_TRACE_SYS(("HandleUnMount: device     :/dev/%s         ",f_cPartitionNode));
        ETG_TRACE_SYS(("HandleUnMount: <<<<<<<<<<<<<<<<<<<<<<<<<<<<"));

        GENSTRING l_cDevNode = GENSTRING("/dev/") + GENSTRING(f_cPartitionNode);

        DisconnectNotification(l_cDevNode, CGlobalEnumerations::SINGLE);
    }
    else
    {
        ETG_TRACE_ERR (("HandleUnMount: [ERROR] f_cPartitionNode is NULL"));
    }

    ETG_TRACE_USR4(("End: HandleUnMount"));
}

/*-----------------------------------------------------------------------------*
 * void HandleUnMount (GENSTRING cDevNode)                                       *
 *-----------------------------------------------------------------------------*/
void CUDevManager::HandleUnMount (GENSTRING cDevNode)
{
    ETG_TRACE_USR4(("Begin: HandleUnMount (Q)"));

    ETG_TRACE_SYS(("HandleUnMount: >>>>>>>>>UNMOUNT (Q)>>>>>>>>>>>>>>"));
    ETG_TRACE_SYS(("HandleUnMount: device     :%s         ",cDevNode.toStdString().c_str()));
    ETG_TRACE_SYS(("HandleUnMount: <<<<<<<<<<<<<<<<<<<<<<<<<<<<"));

    DisconnectNotification(cDevNode, CGlobalEnumerations::SINGLE);

    ETG_TRACE_USR4(("End: HandleUnMount (Q)"));
}

/*-----------------------------------------------------------------------------*
 * int LookupConnectedDevice(const GENSTRING f_cDeviceSerial,                    *
 *          CDevice& f_pCDevice)                                              *
 *-----------------------------------------------------------------------------*/
int CUDevManager::LookupConnectedDevice(const GENSTRING f_cDeviceSerial, CDevice& f_oCDevice)
{
    ETG_TRACE_USR4 (("Begin: LookupConnectedDevice"));

    int l_bDevicePresent = DEVICEMANAGER_ERROR_NODEVICE;

    if (!f_cDeviceSerial.isEmpty())
    {
        ETG_TRACE_USR4 (("LookupConnectedDevice: Unique ID: %s",
                         f_cDeviceSerial.toStdString().c_str()));

        unsigned int uiSize;
        CDevice *l_poDevice = NULL;
        unsigned int l_uiCount;
        bool l_cSerialFound = false;

        // Check the device in connected device list
        CDevice oDevice;
        tBool bExists;

        uiSize = DeviceListManager::GetInstance()->getSizeOfList(DEVLIST_UDEVMGR);

        for (l_uiCount = 0; l_uiCount < uiSize; l_uiCount++)
        {

            bExists = DeviceListManager::GetInstance()->getElement(DEVLIST_UDEVMGR,l_uiCount, oDevice);
            DVM_NORMAL_M_ASSERT(TRUE == bExists);

            if(TRUE == bExists)
            {
                l_poDevice = &oDevice;

                if (f_cDeviceSerial.contains (l_poDevice->m_cSerialID))
                {
                    ETG_TRACE_SYS (("LookupConnectedDevice: SerialID found in list"));

                    l_cSerialFound = true;
                    l_bDevicePresent = DEVICEMANAGER_DEVICE_DETECTED;

                    // the device could be formatted and then reconnected.
                    // In that case it needs to be processed as a new device
                    // match the mount-point as well
                    if (    (CGlobalEnumerations::DTY_SD          == l_poDevice->m_eDeviceType)
                            ||(CGlobalEnumerations::DTY_USB         == l_poDevice->m_eDeviceType)
                            ||(CGlobalEnumerations::DTY_SD_INTERNAL == l_poDevice->m_eDeviceType)) ///@todo what about cdrom here?
                    {
                        ETG_TRACE_USR4 (("LookupConnectedDevice: Checking mountpount: %s",
                                         l_poDevice->m_cMountPoint.toStdString().c_str()));

                        GENSTRING l_cAugSerial = l_poDevice->m_cSerialID + l_poDevice->m_cMountPoint;

                        if (!l_cAugSerial.contains(f_cDeviceSerial))
                        {
                            ETG_TRACE_SYS (("LookupConnectedDevice: Different SD card or device was formatted"));
                            l_bDevicePresent = DEVICEMANAGER_ERROR_NODEVICE;
                        }
                    }

                    if (DEVICEMANAGER_DEVICE_DETECTED == l_bDevicePresent)
                    {
                        ETG_TRACE_SYS (("LookupConnectedDevice: Device found"));

                        f_oCDevice.FillDevice(*l_poDevice);
                        f_oCDevice.m_eConnectStatus = USB_DEV_CONNECTED;

                        break;
                    }
                }

            }

        }

#ifdef CHECK_LAST_DEVICE
        if ((false == l_cSerialFound)
                && (DEVICEMANAGER_DEVICE_DETECTED != l_bDevicePresent))
        {
            // Check the device in seen devices list
            for (l_uiCount = 0; l_uiCount < m_vSeenDevices.size(); l_uiCount++)
            {
                if (f_cDeviceSerial.contains (m_vSeenDevices[l_uiCount]))
                {
                    ETG_TRACE_SYS (("LookupConnectedDevice: Device found in seen list"));

                    f_oCDevice.m_cSerialID      = m_vSeenDevices[l_uiCount];
                    f_oCDevice.m_eConnectStatus = USB_DEV_CONNECTED;

                    l_bDevicePresent = DEVICEMANAGER_DEVICE_DETECTED;
                }
            }
        }
#endif // CHECK_LAST_DEVICE
    }
    else
    {
        ETG_TRACE_ERR (("LookupConnectedDevice: [ERROR] Invalid parameters"));

        l_bDevicePresent = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    if (DEVICEMANAGER_DEVICE_DETECTED != l_bDevicePresent)
    {
        ETG_TRACE_USR1 (("LookupConnectedDevice: Device not connected"));
    }
    else
    {
        ETG_TRACE_USR1 (("LookupConnectedDevice: Device connected"));
    }

    ETG_TRACE_USR4 (("End  : LookupConnectedDevice"));

    return l_bDevicePresent;
}

/*-----------------------------------------------------------------------------*
 * void TraceConnectedHubs(tVoid)                                              *
 *-----------------------------------------------------------------------------*/
void  CUDevManager::TraceConnectedHubs(tVoid)
{
    ETG_TRACE_USR4(("Begin:TraceConnectedHubs"));

    m_tableLock.lock();

    ETG_TRACE_FATAL(("[ok] ================================->TraceConnectedHubs=================================================="));
    for( unsigned int l_iDevCount = 0; l_iDevCount < m_vHubs.size(); l_iDevCount++)
    {
        if(NULL != m_vHubs[l_iDevCount])
        {
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs:eHubNo          %d           ", ETG_ENUM( HUB_NUM_E, m_vHubs[l_iDevCount]->eHubNo ) ));
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs:eHUBFound       %d           ", ETG_CENUM(CGlobalEnumerations::HUB_SEARCHRESULT_Type,m_vHubs[l_iDevCount]->eHUBFound) ));
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs:uiPortCount     %d           ", m_vHubs[l_iDevCount]->uiPortCount ));
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs:sSysName        %s           ", m_vHubs[l_iDevCount]->sSysName.toStdString().c_str() ));
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs:bHasCardReader  0x%x          ", m_vHubs[l_iDevCount]->bHasCardReader ));
            ETG_TRACE_FATAL(("                                                                         "));
        }
        else
        {
            ETG_TRACE_FATAL(("[ok] TraceConnectedHubs: NULL == m_vHubs[%d]",l_iDevCount));
        }
    }

    ETG_TRACE_FATAL(("[ok] ================================TraceConnectedHubs<-=================================================="));

    m_tableLock.unlock();

    ETG_TRACE_USR4(("End  :TraceConnectedHubs"));
}

/*-----------------------------------------------------------------------------*
 * tVoid lockNotification(tInt iTimout)                                        *
 *-----------------------------------------------------------------------------*/
tBool CUDevManager::lockNotification(tInt iTimout)
{
    tBool bRet = FALSE;
    bool bResult = m_UdevLock.lock(iTimout);

    if(bResult)
    {
        bRet = TRUE;
    }

    return bRet;
}

/*-----------------------------------------------------------------------------*
 * tVoid lockNotification(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::lockNotification(tVoid)
{
    //ETG_TRACE_FATAL(("CUDevManager::lockNotification"));
    while(false == m_UdevLock.lock(DEFAULT_LOCK_TIMOUT_SEC))
    {
        ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        ETG_TRACE_FATAL(("[ERROR] CUDevManager::lockNotification()==false!"));
        ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        //NORMAL_M_ASSERT_ALWAYS();
        sleep(1);

    }
}

/*-----------------------------------------------------------------------------*
 * tVoid unlockNotification(tVoid)                                             *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::unlockNotification(tVoid)
{
    //ETG_TRACE_FATAL(("CUDevManager::unlockNotification"));
    m_UdevLock.unlock();
}


#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES

void CUDevManager::vPrintINotifyData( const char* strINotifyKeyWord[], char* strINotifyLine[], tUInt maxElem ) const
{
    ETG_TRACE_USR4 (("HandleMount: >>>>>>>>>MOUNT>>>>>>>>>>>>>>"));

    GENSTRING Out;
    for( tUInt line=1; (line < maxElem); line++ )
    {
        if( strINotifyLine[line] )
        {
            Out = GENSTRING(strINotifyKeyWord[line]) + ": " + GENSTRING(strINotifyLine[line]);
            ETG_TRACE_SYS(( "HandleMount: %s ", Out.toStdString().c_str() ));
        }
    }
    ETG_TRACE_USR4 (("HandleMount: <<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}//lint !e818 Info 818 -> compiler doesn't like a const pointer


//@todo add pointer checks
void CUDevManager::vExtractContentMountDB( CDevice &f_oCurAvailDevInfo, char* strMountDBLines[], tUInt mountDB_line, tInt u16LenOfEntry) const
{
    ETG_TRACE_USR4(("HandleMount: Line %s ", strMountDBLines[mountDB_line]));
    //example
    //mountpoint="/dev/media/20F1-24F7"        -> change to /dev/media/20F1-24F7
    //idvendor="0951"                          -> change to 0951
    //serial="0018F3D97163BC1121480BF8"        -> change to 0018F3D97163BC1121480BF8
    //manufacturer="Kingston"
    //fstype="vfat"
    //uuid="20F1-24F7"
    //label="K_CHLING"
    //model="DT R400         "
    //product="DT R400"
    //version=" 2.00"

    (void)f_oCurAvailDevInfo;// for future use
    (void)strMountDBLines;// for future use
    (void)mountDB_line;// for future use
    (void)u16LenOfEntry;// for future use

    // for future use -> GENSTRINGList causes LINT Prio 1 problems due to tooling problems
    /*
    char          l_sBcdUSB[32];
    tU16          bcdUSB;
    GENSTRING       strVersion;
    tInt          iIDVendor_hex;
    GENSTRING       IDVendor;
    bool          bOk;
    tInt iFoundQuoteCount = NOTFOUND;

    if( mountDB_line >= (tU8)LINE1_MASSSTORAGE_MOUNTPOINT )
    {
        for( tInt l=0; l<=u16LenOfEntry; l++ )
        {
            //copy number at the beginning after first " has found
            if(iFoundQuoteCount != NOTFOUND)
            {
                strMountDBLines[mountDB_line][iFoundQuoteCount] = strMountDBLines[mountDB_line][l];
                iFoundQuoteCount++;
            }
            if(strMountDBLines[mountDB_line][l] == 0x22)   // search for  "
            {
                //find first quote
                if(iFoundQuoteCount == NOTFOUND)
                {
                    iFoundQuoteCount = 0; //set offset at beginning
                }
                //find second quote finish copy and write end of string
                else
                {
                    strMountDBLines[mountDB_line][iFoundQuoteCount-1] = '\0'; // -1 because of iFoundQuoteCount above
                    break;
                }
            }
        }
        //if no " has been found then put /0 at the beginning of the srting to mark it empty
        if(iFoundQuoteCount == NOTFOUND)
        {
            strMountDBLines[mountDB_line][0] = '\0';
        }

        //'security hack' if something is wrong with the mountDB file care for end of string in any case even if this is redundant
        strMountDBLines[mountDB_line][u16LenOfEntry] = '\0';

        switch((MOUNTDB_Lines)mountDB_line)
        {
            case LINE0_MASSSTORAGE_DEVICE:
                //e.g. /dev/sda1
                f_oCurAvailDevInfo.m_cBlockDevnode = strMountDBLines[mountDB_line];
                break;
            case LINE1_MASSSTORAGE_MOUNTPOINT:
                f_oCurAvailDevInfo.m_cMountPoint = strMountDBLines[mountDB_line];
                break;
            case LINE2_MASSSTORAGE_IDVENDOR:
                IDVendor   = strMountDBLines[mountDB_line];
                IDVendor           = IDVendor.trimmed(); //eventual whitespaces removed
                iIDVendor_hex      = IDVendor.toInt(&bOk,16);
                if(bOk)
                {
                    f_oCurAvailDevInfo.m_iVendorID = iIDVendor_hex;
                }
                else
                {
                    f_oCurAvailDevInfo.m_iVendorID = 0x00;
                }
                break;
             case LINE3_MASSSTORAGE_SHORTSERIALNUMBER:
                 f_oCurAvailDevInfo.m_cShortSerial = strMountDBLines[mountDB_line];
                 break;
            case LINE4_MASSSTORAGE_MANUFACTORER:
                break;
            case LINE5_MASSSTORAGE_FSTYPE:
                f_oCurAvailDevInfo.m_cFSType = strMountDBLines[mountDB_line];
                break;
            case LINE6_MASSSTORAGE_UUID:
                f_oCurAvailDevInfo.m_cDeviceUUID = strMountDBLines[mountDB_line];
                break;
            case LINE7_MASSSTORAGE_LABEL:
                f_oCurAvailDevInfo.m_cDeviceName = GENSTRING::fromUtf8(strMountDBLines[mountDB_line]);
                break;
            case LINE8_MASSSTORAGE_MODEL:
                break;
            case LINE9_MASSSTORAGE_PRODUCT:
                //f_oCurAvailDevInfo.m_cDeviceName = strMountDBLines[mountDB_line];
                break;
            case LINE10_MASSSTORAGE_VERSION:
                strVersion = strMountDBLines[mountDB_line];
                strVersion = strVersion.trimmed();
                bcdUSB     = (tU16)strVersion.toInt(&bOk,10);
                if(bOk)
                {
                    snprintf (l_sBcdUSB, sizeof(l_sBcdUSB), "%2x.%02x", bcdUSB >> 8, bcdUSB & 0xff);
                    f_oCurAvailDevInfo.m_cDeviceVersion = GENSTRING(l_sBcdUSB);
                }
                else
                {
                    f_oCurAvailDevInfo.m_cDeviceVersion = "";
                }
                break;
            case LINE_MASSSTORAGE_LASTELEM:
                break;
            default:
                break;
        }
    }

    //calculate Serial ID from Manufacturer+Product+SerialID if this is the previous line
    if( (mountDB_line + 1) >= LINE_MASSSTORAGE_LASTELEM)
    {
        GENSTRING l_strProduct      = strMountDBLines[LINE9_MASSSTORAGE_PRODUCT];
        GENSTRING l_strManufacturer = strMountDBLines[LINE4_MASSSTORAGE_MANUFACTORER];
        GENSTRING l_strShortSerial  = strMountDBLines[LINE3_MASSSTORAGE_SHORTSERIALNUMBER];

        GENSTRING l_strMountPoint   = strMountDBLines[LINE1_MASSSTORAGE_MOUNTPOINT];
        GENSTRINGList   l_strMountPointList = l_strMountPoint.split("/"); //e.g. /dev/mount/123 -> "dev" . "mount" , "123"
        l_strMountPoint = l_strMountPointList.last();

        f_oCurAvailDevInfo.m_cSerialID = l_strManufacturer + " " + l_strProduct + " " + l_strShortSerial + " " + l_strMountPoint;
        f_oCurAvailDevInfo.m_cSerialID = f_oCurAvailDevInfo.m_cSerialID.trimmed();
        f_oCurAvailDevInfo.m_cSerialID.replace (" ", "_");
        f_oCurAvailDevInfo.m_eDeviceType = CGlobalEnumerations::DTY_USB;


        //workarounds
        if(f_oCurAvailDevInfo.m_cSerialID == "")
        {
            f_oCurAvailDevInfo.m_cSerialID = l_strMountPoint;
        }
        f_oCurAvailDevInfo.m_iProductID = 0xAFFE; //Dummy
        f_oCurAvailDevInfo.m_iDevNUM    = 3;      //Dummy
    }
*/
}//lint !e1764 Info 1764; could be removed once this function is used

void CUDevManager::vExtractContentUnsupported( CDevice& f_oCurAvailDevInfo, char* strUnsupportedLine[], tUInt lineNo, tInt u16LenOfEntry) const
{
    ETG_TRACE_USR4(("HandleMount: (unsupported Dev) Line no. %i: = %s ", lineNo, strUnsupportedLine[lineNo]));
    //example
    //root@bosch-nemid:/etc/udev/scripts# cat /tmp/.unsupported/sda1
    // /dev/sda1
    // usb_connector=1
    // hub_connected=no
    // hub_portnum=0

    tInt    iFoundQuoteCount = NOTFOUND;

    if( lineNo >= (tU8)LINE1_UNSUPPORTEDDEV_USB_CONNECTOR )
    {
        for( tInt l=0; l<=u16LenOfEntry; l++ )
        {
            //copy number at the beginning after first " has found
            if( iFoundQuoteCount != NOTFOUND )
            {
                strUnsupportedLine[lineNo][iFoundQuoteCount] = strUnsupportedLine[lineNo][l];
                iFoundQuoteCount++;
            }

            if(   (strUnsupportedLine[lineNo][l] == '=')        // search for '='
                  && (iFoundQuoteCount == NOTFOUND)                //find first quote
                  )
            {
                iFoundQuoteCount = FOUND;                       //set offset at beginning
            }
            else if(   (strUnsupportedLine[lineNo][l] == '\n')  //find second quote finish copy and write end of string
                       && (iFoundQuoteCount == FOUND )
                       )
            {
                strUnsupportedLine[lineNo][iFoundQuoteCount-1] = '\0';  // -1 because of iFoundQuoteCount above
                break; // end the for(...) loop
            }
        }
        //if no " has been found then put /0 at the beginning of the srting to mark it empty
        if( iFoundQuoteCount == NOTFOUND )
        {
            strUnsupportedLine[lineNo][0] = '\0';
        }

        //'security hack' if something is wrong with the mountDB file care for end of string in any case even if this is redundant
        strUnsupportedLine[lineNo][u16LenOfEntry] = '\0';

        switch( (UNSUPPORTED_Lines)lineNo )
        {
        case LINE0_UNSUPPORTEDDEV_DEVICE:
            // do nothing -> use LINE0_UNSUPPORTEDDEV_DEVICE to avoid lint
            //                //e.g. /dev/sda1
            //                f_oCurAvailDevInfo.m_cBlockDevnode = strUnsupportedLine[lineNo];
            break;
        case LINE1_UNSUPPORTEDDEV_USB_CONNECTOR:
            f_oCurAvailDevInfo.m_cUSBPort      = strUnsupportedLine[lineNo];
            f_oCurAvailDevInfo.m_eUSB          = (tenDevManagerUSBHost)atoi( strUnsupportedLine[lineNo] );
            f_oCurAvailDevInfo.m_iUSBConnector = atoi( strUnsupportedLine[lineNo] );
            break;
        case LINE2_UNSUPPORTEDDEV_HUB_CONNECTED:
            f_oCurAvailDevInfo.m_bConnectedAtHub =  (GENSTRING(strUnsupportedLine[lineNo]) == "no" ) ? false : true; //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
            break;
        case LINE3_UNSUPPORTEDDEV_HUB_PORTNUM:
            f_oCurAvailDevInfo.m_iPortNumber = atoi( strUnsupportedLine[lineNo] );
            break;
        default:
            break;
        }

        if( (lineNo + 1) >= (tInt)LINE_UNSUPPORTEDDEV_LASTELEM )
        {
            ETG_TRACE_USR4(( "HandleMount: vExtractContentUnsupported: Fill rest" ));
            f_oCurAvailDevInfo.m_cMountPoint    = "UNSUPPORTED_DEVICE";
            f_oCurAvailDevInfo.m_iVendorID      = 0x00;
            f_oCurAvailDevInfo.m_cShortSerial   = "";
            f_oCurAvailDevInfo.m_cFSType        = "vfat";
            f_oCurAvailDevInfo.m_cDeviceUUID    = "1F5B-1D4B";             // @todo check if this is necessary -> find a better dummy value
            f_oCurAvailDevInfo.m_cDeviceName    = "UnSupportedDevice";
            f_oCurAvailDevInfo.m_cDeviceVersion = "UnSupportedDevice";
            f_oCurAvailDevInfo.m_cSerialID      = "UnSupportedDevice";
            f_oCurAvailDevInfo.m_eDeviceType    = CGlobalEnumerations::DTY_NOT_SUPPORTED;
            f_oCurAvailDevInfo.m_iProductID     = 0x00;     //Dummy
            f_oCurAvailDevInfo.m_iDevNUM        = 3;        //Dummy
        }
    }
}//lint !e818 Info 818 -> compiler doesn't like a const pointer
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES


#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
void  CUDevManager::vMapFileSystemType(INOUT CDevice *f_pCurAvailDevInfo) const  ///@todo due to surrounding I can not use reference which I would like to do
{

    /*
please refer to this structure in ENUMS.h
    typedef enum FILESYSTEM_TYPE_E
        {
            FSTY_UNKNOWN       = 0,
            FSTY_FAT           = 1,
            FSTY_NTFS          = 2, //@HUB
            FSTY_EXT3_4        = 3,
            FSTY_HFS           = 4,
            FSTY_EXFAT         = 5,
            FSTY_ISO9660       = 6,
            FSTY_UDF           = 7,
            FSTY_LASTELEM      = 8
        }FILESYSTEM_TYPE_Type;
*/


    if(f_pCurAvailDevInfo->m_cFSType.contains("EXFAT",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_EXFAT;
    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("NTFS",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_NTFS;
    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("EXT",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_EXT3_4;
    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("HFSPLUS",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_HFS;
    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("FAT",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_FAT;
    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("ISO9660",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_ISO9660;

    }
    else if(f_pCurAvailDevInfo->m_cFSType.contains("UDF",CASE_INSENSITIVE))
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_UDF;

    }
    else
    {
        f_pCurAvailDevInfo->m_eFSType = CGlobalEnumerations::FSTY_UNKNOWN;
        ETG_TRACE_FATAL(("[ERROR]vMapFileSystemType:in bMapFileSystemType for %s",
                         f_pCurAvailDevInfo->m_cFSType.toStdString().c_str()));

    }

    ETG_TRACE_USR4(("vMapFileSystemType:  f_pCurAvailDevInfo->m_eFSType:%d",
                    ETG_CENUM(CGlobalEnumerations::FILESYSTEM_TYPE_Type,f_pCurAvailDevInfo->m_eFSType)))


}
#endif


void CUDevManager::vSendMountDevice(tBool bMount,GENSTRING devicename, GENSTRING mountPoint) const //Roadmap 14002 MountCDROM (DiagMixeMode)
{
    ETG_TRACE_USR4(("Begin: vSendMountedDevice"));

    tInt iRes = 0;
    trPipeParmsSendMountDev oPipeParams;

    if(PIPE_UNDEF != m_iPipeFD[enPipeWriter])  ///@todo check for correct return value
    {

        oPipeParams.bMount = bMount;

        //cp devicename
        if((tU32)(devicename.length()/*number of characters of string*/+1/*terminating '\0'*/) >=(tU32)sizeof(oPipeParams.cDeviceName))
        {
            ETG_TRACE_FATAL(("ERROR: vSendMountedDevice:[dest  ]sizeof(oPipeParams.cDeviceName)=%d  ",sizeof(oPipeParams.cDeviceName) ));
            ETG_TRACE_FATAL(("ERROR: vSendMountedDevice:[source]          devicename.length()-1=%d  ",devicename.length()-1    ));
            iRes++;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            strncpy(oPipeParams.cDeviceName,devicename.toStdString().c_str(),sizeof(oPipeParams.cDeviceName));
            oPipeParams.cDeviceName[sizeof(oPipeParams.cDeviceName)-1] = '\0'; // sim4hi: fix for Coverity CID-10070
        }

        // cp mountpoint
        if((tU32)(mountPoint.length()/*number of characters of string*/+1/*terminating '\0'*/) >=(tU32)sizeof(oPipeParams.cMountpoint))
        {
            ETG_TRACE_FATAL(("ERROR: vSendMountedDevice:[dest  ]sizeof(oPipeParams.cDeviceName)=%d  ",sizeof(oPipeParams.cMountpoint) ));
            ETG_TRACE_FATAL(("ERROR: vSendMountedDevice:[source]          mountPoint.length()-1=%d  ",mountPoint.length()-1    )); // sim4hi: fix for Coverity CID-10532
            iRes++;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            strncpy(oPipeParams.cMountpoint,mountPoint.toStdString().c_str(),sizeof(oPipeParams.cMountpoint)-1); //to care that destination string is possible to be terminated by '\0' using strncpy
            oPipeParams.cMountpoint[sizeof(oPipeParams.cMountpoint)-1]='\0'; // sim4hi: fix for Coverity similar to CID-10070
        }

        //send to pipe
        if(0 == iRes)
        {
            iRes = write(m_iPipeFD[enPipeWriter],&oPipeParams,sizeof(trPipeParmsSendMountDev));
            if(iRes == (tInt)sizeof(trPipeParmsSendMountDev))
            {
                ETG_TRACE_USR4(("vSendMountedDevice: send %d bytes to pipe ok!", iRes));
            }
        }
    }

    ETG_TRACE_USR4(("End  : vSendMountedDevice"));
}


#ifdef USE_TOTAL_USED_FREE_SIZE

tVoid CUDevManager::FillStorageInfoSizeTUF(INOUT CDevice* f_pCurDevice)
{
    GENSTRING strTotalSize_KB = "-";
    GENSTRING strUsedSize_KB = "-";
    GENSTRING strFreeSize_KB = "-";

    /*NCG3D-12724 GetSize called for RNAIVI project as well*/
    if((f_pCurDevice->m_iTotalValidPartitons > 1 && (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_MSD_BIGGEST_PARTITION) || DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_SORT_PARTITIONS_ALL)))
            ||DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_GETMSC_FREE_USED_SIZE))
    {
#ifdef USE_REPLACE_DF_WITH_STATFS
        GetSizeMSC_TUF(OUT strTotalSize_KB,OUT strUsedSize_KB,OUT strFreeSize_KB, IN f_pCurDevice->m_cMountPoint);
#else
        GetSizeMSC(OUT strTotalSize_KB,OUT strUsedSize_KB,OUT strFreeSize_KB, IN f_pCurDevice->m_cMountPoint);
#endif
    }

    strTotalSize_KB.trimmed();
    strUsedSize_KB.trimmed();
    strFreeSize_KB.trimmed();

    f_pCurDevice->m_u64TotalSize_KB = (tU64)atoll(strTotalSize_KB.toStdString().c_str());
    f_pCurDevice->m_u64UsedSize_KB  = (tU64)atoll(strUsedSize_KB.toStdString().c_str());
    f_pCurDevice->m_u64FreeSize_KB  = (tU64)atoll(strFreeSize_KB.toStdString().c_str());

    ETG_TRACE_USR4(("FillStorageInfoSizeTUF:f_pCurDevice->m_u64TotalSize_KB: %d",f_pCurDevice->m_u64TotalSize_KB));
    ETG_TRACE_USR4(("FillStorageInfoSizeTUF:f_pCurDevice->m_u64UsedSize_KB : %d",f_pCurDevice->m_u64UsedSize_KB));
    ETG_TRACE_USR4(("FillStorageInfoSizeTUF:f_pCurDevice->m_u64FreeSize_KB : %d",f_pCurDevice->m_u64FreeSize_KB));

}
tVoid  CUDevManager::GetSizeMSC_TUF(OUT GENSTRING& f_totalSize,OUT GENSTRING& f_usedSize,OUT GENSTRING &f_freeSize, IN GENSTRING f_mountPoint)
{
    int iRes;
    StorageInfo_t f_oInfo;

    //check if size is as expected:
    if(sizeof(unsigned long long) < sizeof(fsblkcnt_t))
    {
        ETG_TRACE_FATAL(("[WARNING]: unexpected difference in size of type fsblkcnt_t of struct statfs"));
        ETG_TRACE_FATAL(("[WARNING]: sizeof(unsigned long long)=%d",sizeof(unsigned long long)));
        ETG_TRACE_FATAL(("[WARNING]: sizeof(fsblkcnt_t)=%d",sizeof(fsblkcnt_t)));
    }


    ETG_TRACE_USR4(("GetSizeMSC_TUF:f_mountPoint.toStdString().c_str() = %s",f_mountPoint.toStdString().c_str()));
    iRes = FillStorageInfo(f_mountPoint.toStdString().c_str(),f_oInfo);

    ETG_TRACE_USR4(("GetSizeMSC_TUF:sizeof(unsigned long long) = %d",sizeof(unsigned long long))); //on GM it is 4  hence 4*8 = 32bit Maxvalue is 2^(32-1)-1 = 2147483647 i.e. printed as a string 10 +1 for end of lstring '\0'


    char strToltalSize[80]; //11 would be enough for 64bit max num would be 9223372036854775807 i.e. length 19+1 includign end of string '\0'
    char strUsedSpace[80];
    char strFreeSpace[80];

    if(iRes == DEVICEMANAGER_OK)
    {

        //improve check here
        if(sizeof(unsigned long long) > sizeof(unsigned long))
        {
            ETG_TRACE_USR4(("[WARNING]: sizeof(unsigned long long)=%d",sizeof(unsigned long long)));
            ETG_TRACE_USR4(("[WARNING]: sizeof(unsigned long int)=%d",sizeof(unsigned long int)));
        }
        //handover size in KB to string
        snprintf(strToltalSize, sizeof(strToltalSize),"%lu",(unsigned long)((f_oInfo.total_size)/1024)); //had trouble pritning unsigned long long since 4 bytes are used currently I will cast it. Which is not very nice
        snprintf(strUsedSpace, sizeof(strUsedSpace) , "%lu",(unsigned long)((f_oInfo.used_space)/1024));
        snprintf(strFreeSpace, sizeof(strFreeSpace) , "%lu",(unsigned long)((f_oInfo.free_space)/1024));

       f_totalSize = strToltalSize;
       f_usedSize  = strUsedSpace;
       f_freeSize  = strFreeSpace;


    }
    else
    {
        f_totalSize = "-";
        f_usedSize  = "-";
        f_freeSize  = "-";
    }

}

#ifndef USE_REPLACE_DF_WITH_STATFS
tVoid CUDevManager::GetSizeMSC(OUT GENSTRING& f_totalSize,OUT GENSTRING& f_usedSize,OUT GENSTRING &f_freeSize, IN GENSTRING f_mountPoint)
{
    ///@cleanup code next time

    char commandString[1024];
    char *pTmp = NULL;
    char totalSize[1024];
    char usedSize[1024];
    char freeSize[1024];
    char tmpSize[1025];
    char l;
    char nrOfString = 0;
    tU16 u16Count;
    tBool bCopy = FALSE;
    FILE *pFile;
    tInt  i,len;

    ETG_TRACE_USR1(("GetSizeMSC: Begin"));

    totalSize[0] = '\0';
    usedSize[0]  = '\0';
    freeSize[0]  = '\0';
    tmpSize[0]   = '\0';

    /* Calculate free disk space */
    snprintf(commandString,sizeof(commandString),"df %s -B %d > /tmp/GetSizeMSC.txt",f_mountPoint.toStdString().c_str(),TMP_BLOCKSIZE);
    ETG_TRACE_USR3(("commandString: %s", commandString));

    //SendLinuxCmd(commandString);
    GENSTRING sparameters;
    sparameters = f_mountPoint.toStdString() + GENSTRING(" -B 1024 > ") + GENSTRING("/tmp/GetSizeMSC.txt");
    execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_DFGETSIZE, sparameters.toStdString().c_str());

#define MAXREADSIZE 2056
    char data[MAXREADSIZE+1];
    size_t result = 0;
    pFile = fopen("/tmp/GetSizeMSC.txt","r");
    if(pFile)
    {
        result = fread(data,1,MAXREADSIZE,pFile);
        ETG_TRACE_USR2(("GetSizeMSC: result!=MAXREADSIZE: result:%d MAXREADSIZE:%d",result,MAXREADSIZE));

        data[result] = '\0'; //add string end

        len = (tInt)strlen_r(data); //strlen_r(NULL) prevented due to definition of data[]
        ETG_TRACE_COMP(("GetSizeMSC: len:%d data:%s",len,data));

        tBool bPreviousASpace   = TRUE;

        u16Count = 0;
        nrOfString = 0;
        for(i=len-1; i>=0;i--)
        {
            l = data[i];
            if(l == ' ')
            {
                bPreviousASpace = TRUE;
                //ETG_TRACE_COMP(("GetSizeMSC: SPACE found seperator"));
            }
            else
            {
                if(bPreviousASpace)
                {
                    bPreviousASpace = FALSE;
                    u16Count = 0;
                    nrOfString++;
                    //ETG_TRACE_COMP(("GetSizeMSC: FirstLetter of word found: u16Count:%d",u16Count));
                }
                else
                {
                    //ETG_TRACE_COMP(("GetSizeMSC: AnotherLetter of word found:u16Count:%d",u16Count));
                }


                if(nrOfString==5)
                {
                    pTmp = totalSize;
                    bCopy = TRUE;
                }
                else if(nrOfString==4)
                {
                    pTmp = usedSize;
                    bCopy = TRUE;
                }
                else if(nrOfString==3)
                {
                    pTmp = freeSize;
                    bCopy = TRUE;
                }
                else
                {
                    bCopy = FALSE;
                }

                if(bCopy && pTmp) ///@todo replace bCopy fully by Null check of pTmp
                {
                    pTmp[u16Count] = data[i];
                    u16Count++;
                    pTmp[u16Count] = '\0';

                    //ETG_TRACE_COMP(("GetSizeMSC: pTmp:%s",pTmp));
                }
            }


        }
    }

    //since we read from back strings have to be turned around
    len = (tInt)strlen_r(totalSize); //strlen_r(NULL) prevented due to definition of totalSize[]
    ETG_TRACE_COMP(("GetSizeMSC:len:%d",len));
    for(i=0; i<len;i++)
    {
        tmpSize[len-(i+1)] = totalSize[i];
    }
    tmpSize[len] = '\0';
    ETG_TRACE_COMP(("GetSizeMSC:----result------"));
    ETG_TRACE_COMP(("GetSizeMSC: tmpSize  : %s",tmpSize));
    ETG_TRACE_COMP(("GetSizeMSC: totalSize: %s",totalSize));
    f_totalSize = tmpSize;

    len = (tInt)strlen_r(usedSize);//strlen_r(NULL) prevented due to definition of usedSize[]
    ETG_TRACE_COMP(("GetSizeMSC:len:%d",len));
    for(i=0; i<len;i++)
    {
        tmpSize[len-(i+1)] = usedSize[i];
    }
    tmpSize[len] = '\0';
    ETG_TRACE_COMP(("GetSizeMSC:----result------"));
    ETG_TRACE_COMP(("GetSizeMSC: tmpSize  : %s",tmpSize));
    ETG_TRACE_COMP(("GetSizeMSC: usedSize : %s",usedSize));
    f_usedSize = tmpSize;


    len = (tInt)strlen_r(freeSize);//strlen_r(NULL) prevented due to definition of freeSize[]
    ETG_TRACE_COMP(("GetSizeMSC:len:%d",len));
    for(i=0; i<len;i++)
    {
        tmpSize[len-(i+1)] = freeSize[i];
    }
    tmpSize[len] = '\0';
    ETG_TRACE_COMP(("GetSizeMSC:----result------"));
    ETG_TRACE_COMP(("GetSizeMSC: tmpSize  : %s",tmpSize));
    ETG_TRACE_COMP(("GetSizeMSC: freeSize : %s",freeSize));
    f_freeSize = tmpSize;

    if(pFile) fclose(pFile); //I don't remove the file this is more easy for debugging

    ETG_TRACE_USR1(("GetSizeMSC: End  "));

}

#endif //USE_REPLACE_DF_WITH_STATFS
#endif //USE_TOTAL_USED_FREE_SIZE

/*-----------------------------------------------------------------------------*
 * int StartMonitorLooseBlocking()                                             *
 *-----------------------------------------------------------------------------*/
//@todo refactor this code subfunctions for the FDISSET's
int CUDevManager::StartMonitorLooseBlocking()
{
    ETG_TRACE_USR4(("Begin: StartMonitorLooseBlocking"));



#ifdef BUGFIX_STARTUPRECOGNITION_INOTIFY   //if we doe this here and in StartMonitorLooseblocing - do we loose something?
    // Monitor runs forever. Therefore this 'for(ever) loop
    LookupExistingMounts();
#endif

/* for future usage
    NORMAL_M_ASSERT(m_pIAutomounterAdapter);
    if(m_pIAutomounterAdapter)
    {
        ETG_TRACE_USR4(("Get Pipe from automounter adapter shared lib if to receive notifications"));
    }
*/



    //============================================================
    // Loop to care for notfication of devices (udev and  /tmp/automount or /tmp/automounter)
    //============================================================
    for(;/*ever*/;)
    {
        fd_set fds;  //set of FD's -set of  file descriptors
        struct timeval tv;
        int ret;

        FD_ZERO(&fds);                          //lint !e717 Info 717: do ... while(0);

        //========================
        //UDEV events added to FDSet
        //========================
        ETG_TRACE_USR4(("StartMonitorLooseBlocking: FD_SET:m_iUDevMonitorFD"));
        FD_SET((tUInt)m_iUDevMonitorFD, &fds);  //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        // cast (tUInt)m_iUDevMonitorFD is also OK if it's garantied that StartMonitorLooseBlocking() is only called if CreateInotifyMonitor() returns with DEVICEMANAGER_OK

        //================================
        // /tmp/.automounter  added to FDSet
        //===============================
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        if( -1 != m_iInotifyFD && -1 != m_iInotifyWatchFD  &&  m_iInotifyWDUnsupDev != -1 )
#else // #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        if( -1 != m_iInotifyFD && -1 != m_iInotifyWatchFD   )
#endif // #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        {
            ETG_TRACE_USR4(("StartMonitorLooseBlocking: FD_SET:m_iInotifyFD"));
            FD_SET( (tUInt)m_iInotifyFD, &fds );  //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
            // cast (tUInt)m_iInotifyFD is OK. Descriptors are checked so it's garantied that StartMonitorLooseBlocking() is only called if CreateInotifyMonitor() returns with DEVICEMANAGER_OK
        }

        //=====================
        //PIPE added to FDSet
        //=====================
        if(PIPE_UNDEF != m_iPipeFD[enPipeReader])//Roadmap 14002 StateTable sends command which should loose blocking in select function - i.e. CROMMixed mode is notified from StateTable to UdevManager
        {
            ETG_TRACE_USR4(("StartMonitorLooseBlocking: FD_SET:m_iPipeFD"));
            FD_SET(((tUInt)(m_iPipeFD[enPipeReader])), &fds);  //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        }

#ifdef USE_AUTMOUNTERLIB
        //===================================================
        //Autmounter shared lib notifications  added to FDSet
        //===================================================
        m_iAutomounterLibFD = -1;
        NORMAL_M_ASSERT(m_pIAutomounterAdapter);
        if(m_pIAutomounterAdapter)
        {
            m_iAutomounterLibFD = m_pIAutomounterAdapter->GetFileDescriptor();
        }

        if(-1 != m_iAutomounterLibFD )
        {
            ETG_TRACE_USR4(("StartMonitorLooseBlocking: FD_SET:m_iAutomounterLibFD"));
            FD_SET( ((tUInt)m_iAutomounterLibFD), &fds);
        }
#endif


        tv.tv_sec  =  OBSERVER_INTERVAL_TIME;  //Null means no possing
        tv.tv_usec =  0;


        //---------------------------------------------------------------------------------------------------------------------------
        //=======================================================
        // Wait for event from fd_set mechanism or timeout; otherwise blocking
        //=======================================================
        ret = select(FD_SETSIZE, &fds /*set of FD's*/, NULL, NULL, &tv /*timout*/);  //if one the FD's os ready for reading then this stops blocking

        ETG_TRACE_USR4(("StartMonitorLooseBlocking: select: ret:%d",ret));

#ifdef TMP_CDDA
        ETG_TRACE_FATAL(("StartMonitorLooseBlocking: ----------------------------"));
        ETG_TRACE_FATAL(("StartMonitorLooseBlocking: select: ret:%d",ret));
        ETG_TRACE_FATAL(("StartMonitorLooseBlocking: ----------------------------"));
#endif

        //==========================================================================================
        //event: UDEV: - used for MTP-and Appledevices
        //==========================================================================================
        // Check if our file descriptor has received data.
        if ( (ret > 0) && (FD_ISSET((tUInt)m_iUDevMonitorFD, &fds))) //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        {                                                       // cast (tUInt)m_iUDevMonitorFD is OK if it's garantied that StartMonitorLooseBlocking() is only called if CreateInotifyMonitor() returns with DEVICEMANAGER_OK
            vCASENotificationUdev();
        }//Roadmap 14002 MountCDROM
        else if ((ret > 0) && (FD_ISSET((tUInt)m_iPipeFD[enPipeReader], &fds)) ) //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        {
            trPipeParmsSendMountDev oPipeParams; //shift this to top of this function
            vCASENotificationPipe(oPipeParams);
        }
#ifdef USE_AUTMOUNTERLIB
        //====================================================================================================
        //event: autmounter shared lib - - for usb-sticks,sd-cards at HUB, sd-cards internal , cdrom internal  (planned to replace /tmp/automounter)
        //====================================================================================================
        if ( (ret > 0) && (FD_ISSET((tUInt)m_iAutomounterLibFD, &fds))) //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        {
            vCASENotificationAutomounterLib();
        }//Roadmap adjust to autmounter shared lib
#else //USE_AUTMOUNTERLIB
        //====================================================================================================
        //event: /tmp/automounter - used for usb-sticks,sd-cards at HUB, sd-cards internal , cdrom internal
        //====================================================================================================
        // Check if our file descriptor has received data.
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        else if( ret > 0 && (-1 != m_iInotifyFD && -1 != m_iInotifyWatchFD  &&  m_iInotifyWDUnsupDev != -1 )
         #else // #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        else if (ret > 0 && (-1 != m_iInotifyFD && -1 != m_iInotifyWatchFD)
         #endif // #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
                 && FD_ISSET((tUInt)m_iInotifyFD, &fds)) //lint !e703 Info 703: Shift left of signed quantity (long)  -> OK since the value in the fd_set array normaly declared as unsigned (don't know why it's signed for our fd_set!)
        {                                               // cast (tUInt) is OK. Descriptors are checked so it's garantied that StartMonitorLooseBlocking() is only called if CreateInotifyMonitor() returns with DEVICEMANAGER_OK
            vCASENotificationAutomount();
            //FD_CLR (g_inotify_fd, &fdset);
        }
#endif
        //====================================================================================================
        //event: timout - every 'OBSERVER_INTERVAL_TIME' it'll be checked for avaiilblility of mass storage devices
        //====================================================================================================
        else
        {
            ETG_TRACE_COMP(("StartMonitorLooseblocking: Timout (%dsec)",OBSERVER_INTERVAL_TIME));
            ETG_TRACE_COMP(("StartMonitorLooseblocking: run observer to check for connected/removed devices"));
            //check if anythink has to be   mounted which is not yet mounted
            LookupExistingMounts();
            //check for not mounted connected usb sticks here(usb sticks cannot be mounted due to various reasons. may be unsupported filesystem, no file system, or corrupted file system.
            LookupNotMountedDevices();
            //fill the storageinfo of the Mass storage devices in a temporary  thread.
            vFillStorageInfoMSC();
            //check if anythink has to be unmounted which is not yet mounted
#ifndef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF //intermediate to get in run first
            LookupRemovedMounts();
#endif

        }

    } //end of for(;/*ever*/;)

    // @todo: (next 2 lines) comment in again once the for(;/*ever*/;) could left be for shutdown
    // ETG_TRACE_USR4(("End  : StartMonitorLooseblocking"));

    // return DEVICEMANAGER_OK;
    ETG_TRACE_USR4(("Finished: StartMonitorLooseBlocking"));
    return 0;
}



tVoid CUDevManager::vCASENotificationUdev()
{
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->vCASENotificationUdev************************"));

    GENSTRING l_cDevNode;
    tenDevManagerUSBHost eUsb;

    m_pDev = NULL;
    m_pDev = udev_monitor_receive_device(m_pUdevMonitor);

    //--------------------------------------------------------------
    if (m_pDev)
    {
        m_cAction = udev_device_get_action(m_pDev);

        lockNotification();

        ETG_TRACE_SYS(("StartMonitorLooseBlocking: Action: %s", (const char*)m_cAction.toStdString().c_str()));

        /* check for valid dev node */
        l_cDevNode = GENSTRING(udev_device_get_devnode(m_pDev));
        GENSTRING l_cSysName = GENSTRING(udev_device_get_sysname(m_pDev));

        if (true == l_cDevNode.isEmpty()
                || true == l_cDevNode.contains("/dev/loop")
                || true == l_cDevNode.contains("/dev/ram")
                || true == l_cDevNode.contains("/mmcb")
                || true == l_cDevNode.contains("/mixer")
                || true == l_cDevNode.contains("/audio")
        #ifndef INCOMPLETE_DEV
                || true == l_cDevNode.contains("/pcm")
                || true == l_cDevNode.contains("/hiddev")
        #else
                || true == l_cDevNode.contains("/control")
        #endif // INCOMPLETE_DEV
                || true == l_cDevNode.contains("dsp")
                || true == l_cSysName.contains("usb"))
        {
            ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Not VALID Device : %s",
                             l_cDevNode.toStdString().c_str()));
        }
        else
        {
            ETG_TRACE_SYS(("StartMonitorLooseBlocking: ##### DevNode: %s",l_cDevNode.toStdString().c_str()));

            //--------------------------------------------------------------
            if (GENSTRING("add") == m_cAction)
            //--------------------------------------------------------------
            {
#ifdef INCOMPLETE_DEV
                if (true == l_cDevNode.contains("/pcm") || true == l_cDevNode.contains("/hiddev"))
                {
                    /*
                     * Check for apple devices that may be incomplete
                     */

                    ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Checking for incomplete apple devices"));

                    if (0 < m_vIncompleteDevices.size())
                    {
                        ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Incomplete apple devices present"));

                        CDevice *l_pCDevice = NULL;
                        // Get the parent 'usb_device' for this block device
                        m_pBaseUSBDev = FindBaseUSBDevice (m_pDev);

                        for (unsigned int l_uiCount = 0; l_uiCount < m_vIncompleteDevices.size(); l_uiCount++)
                        {
                            l_pCDevice = &m_vIncompleteDevices[l_uiCount];

                            if (GENSTRING (udev_device_get_syspath (m_pBaseUSBDev)) == l_pCDevice->m_cDevpath)    //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                            {
                                ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Incomplete apple device node = %s",
                                                 l_pCDevice->m_cDevnode.toStdString().c_str()));
                                if (l_cDevNode.contains("/hiddev") && l_pCDevice->m_cMountPoint.isEmpty())
                                {
                                    l_pCDevice->m_cMountPoint = l_cDevNode;

                                    ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Filled mountpoint = %s",
                                                     l_cDevNode.toStdString().c_str()));
                                }
                                else if (l_cDevNode.contains("/pcm") && l_pCDevice->m_cAccessoryName.isEmpty())
                                {
                                    const char *l_pcTemp;
                                    const char *l_cCardNumber   = NULL;
                                    const char *l_cDeviceNumber = NULL;
                                    char        l_sAlsaDeviceName[MAXSIZESTRING];
                                    l_sAlsaDeviceName[0] = '\0';

                                    ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Parsing for sound hardware device name"));

                                    l_pcTemp = strrchr(l_cDevNode.toStdString().c_str(), '/');
                                    if( l_pcTemp )
                                    {
                                        ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Device found is %s", l_pcTemp));
                                        l_cCardNumber   = strchr (l_pcTemp, 'C');
                                        l_cDeviceNumber = strchr (l_pcTemp, 'D');
                                    }
                                    if( l_cCardNumber && l_cDeviceNumber )
                                    {
                                        snprintf(l_sAlsaDeviceName,MAXSIZESTRING, "hw:%c,%c", *(l_cCardNumber + 1), *(l_cDeviceNumber+1));
                                    }
                                    l_pCDevice->m_cAccessoryName = l_sAlsaDeviceName;

                                    ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Filled Hardware Device = %s",
                                                     l_sAlsaDeviceName));
                                }

                                if (!(l_pCDevice->m_cMountPoint.isEmpty() || l_pCDevice->m_cAccessoryName.isEmpty()))
                                {
                                    ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Device is now complete"));

                                    l_pCDevice->m_bIsConnected = true;

                                    adjustDeviceName(l_pCDevice);

                                    ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Pushing the device data"));

                                    if (DEVICEMANAGER_DEVICE_DETECTED == PushDeviceData(l_pCDevice))
                                    {
                                        ETG_TRACE_USR2(("StartMonitorLooseBlocking: Device pushed successfully"));


                                        if (true == m_bConnectUDEV)
                                        {
                                            eUsb = m_pStateTable->GetConnector(l_pCDevice);
                                            l_pCDevice->m_eUSB = eUsb;
                                            l_pCDevice->m_eConnectStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
                                            m_pStateTable->setMount (enMountUdev, eUsb, SIG_TRUE, l_pCDevice);
                                        }

                                    }

                                    // Remove from incomplete device list
                                    m_vIncompleteDevices.erase (m_vIncompleteDevices.begin() + (int)l_uiCount);   //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
                                }
                                else
                                {
                                    ETG_TRACE_USR4 (("StartMonitorLooseBlocking: Device still INCOMPLETE"));
                                }

                                break;  // for (unsigned int l_uiCount = 0; l_uiCount < m_vIncompleteDevices.size(); l_uiCount++)
                            }
                            else
                            {
                                ETG_TRACE_USR4 (("StartMonitorLooseBlocking: No such incomplete device"));
                            }
                        }
                    }
                    else
                    {
                        ETG_TRACE_USR3 (("StartMonitorLooseBlocking: No Incomplete devices found"));
                    }
                }
                else
                {
#endif // INCOMPLETE_DEV
                    DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, l_cDevNode);

                    /* 'change' actions are generated for already connected devices as well.
                 * As such they may be already added to m_vConnectedDevices vector
                 *
                 * NOTE: 'change' actions are being handles because when an SD card is
                 * reinserted only 'change' event is generated for the disk i.e. /dev/sda
                 * and since individual partiions are not being processed the disk is prcocessed.
                 */
                    if ( TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) )
                    {
                        ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Device already connected : %s",
                                         l_cDevNode.toStdString().c_str()));
                    }
                    else
                    {
                        ETG_TRACE_USR1 (("StartMonitorLooseBlocking: New device connected : %s",
                                         l_cDevNode.toStdString().c_str()));
                        if(true == ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_DVD_DRIVE, l_cDevNode.toStdString()))
                        {
    #ifdef TMP_CDDA
                            ETG_TRACE_FATAL(("StartMonitorLooseBlocking: DVDDrive: Disc state changed"));
    #endif
                            //tbd. same like above could be done with extra function
                            if ( TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) )
                            {
    #ifndef INTERMEDIATE_PREVENT_REMOVE_DVD
                                ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Device already connected - remove it: %s",
                                                 l_cDevNode.toStdString().c_str()));
                                 this->m_pBaseUSBDev = m_pDev;

                                 RemoveDevice (l_cDevNode);

                                 this->m_pBaseUSBDev = NULL;
    #endif
                            }
                            else
                            {
                                ETG_TRACE_USR1 (("StartMonitorLooseBlocking: New device connected : %s",
                                                 l_cDevNode.toStdString().c_str()));

                                // Add the new device
                                if (DEVICEMANAGER_DEVICE_DETECTED == AddDeviceDVDRelated(l_cDevNode))
                                {
                                    ETG_TRACE_COMP (("StartMonitorLooseBlocking: New device added"));
                                }
                            } // if (IsAlreadyConnected)

                        }
                        // Add the new device
                        else if (DEVICEMANAGER_DEVICE_DETECTED == AddDevice())
                        {
                            ETG_TRACE_COMP (("StartMonitorLooseBlocking: New device added"));
                        }
                    } // if (IsAlreadyConnected)
#ifdef INCOMPLETE_DEV
                }
#endif // INCOMPLETE_DEV
            } // if ( action is 'add')

            //--------------------------------------------------------------
            else if ( GENSTRING("remove") == m_cAction )
            //--------------------------------------------------------------
            {
                /* NOTE: UNLIKE 'add' action,  the parent 'usb_device' while device
                 * removal is not generated as by the time we start to process
                 * the removed device, the parent usb device is already removed.
                 * And any call to get the parent usb_device gives us the parent
                 * HUB device. And this causes wrong behaviour
                 */
                DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, l_cDevNode);

                this->m_pBaseUSBDev = m_pDev;

                RemoveDevice (l_cDevNode);

                this->m_pBaseUSBDev = NULL;
            }
            //--------------------------------------------------------------
            else if (GENSTRING("change") == m_cAction)
            //--------------------------------------------------------------
            {
#ifdef TMP_CDDA
                ETG_TRACE_FATAL(("StartMonitorLooseBlocking: Action: %s", (const char*)m_cAction.toStdString().c_str()));
#endif
                //Used with new DVD-Drive connected to  USB
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
                {
                    //checks to filter just for external DVD-Drive not for other devices - this is to prevent unecessary traffic
                    //example udevadm monitor shows change of /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host5/target5:0:0/5:0:0:0/block/sr0 (block)
                    //l_cDevNode is /dev/sr0 then
                    if(true == ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_DVD_DRIVE, l_cDevNode.toStdString()))
                    {
#ifdef TMP_CDDA
                        ETG_TRACE_FATAL(("StartMonitorLooseBlocking: DVDDrive: Disc state changed"));
#endif
                        DEVICEPROCESSORHISTORY.AddToHistory (m_cAction, l_cDevNode);

                        //tbd. same like above could be done with extra function
                        if ( TRUE == DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) )
                        {
#ifndef INTERMEDIATE_PREVENT_REMOVE_DVD
                            ETG_TRACE_USR1 (("StartMonitorLooseBlocking: Device already connected - remove it: %s",
                                             l_cDevNode.toStdString().c_str()));
                             this->m_pBaseUSBDev = m_pDev;

                             RemoveDevice (l_cDevNode);

                             this->m_pBaseUSBDev = NULL;
#endif
                        }
                        else
                        {
                            ETG_TRACE_USR1 (("StartMonitorLooseBlocking: New device connected : %s",
                                             l_cDevNode.toStdString().c_str()));

                            // Add the new device
                            if (DEVICEMANAGER_DEVICE_DETECTED == AddDeviceDVDRelated(l_cDevNode))
                            {
                                ETG_TRACE_COMP (("StartMonitorLooseBlocking: New device added"));
                            }
                        } // if (IsAlreadyConnected)

                    }



                    this->m_pBaseUSBDev = m_pDev;
                }


            }
        } // if (valid device)

        // Free udev device
        if (NULL != m_pDev)
        {
            udev_device_unref(m_pDev);
            m_pDev = NULL;
        }

        unlockNotification(); //warranty to serialise UDevEvents and to care that notification is not interrupted by cyclic diag
    } //if (m_pDev)
    else
    {
        //No Device from receive_device(). An error occured.
    }

    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************vCASENotificationUdev<-************************"));
}
tVoid CUDevManager::vCASENotificationAutomount()
{
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->vCASENotificationAutomount************************"));

    GENSTRING l_cDevNode;

    ETG_TRACE_SYS(( "StartMonitorLooseblocking: Event from Inotify"));
    ETG_TRACE_SYS(( "StartMonitorLooseblocking: m_iInotifyFD                      = %d", m_iInotifyFD));
    ETG_TRACE_SYS(( "StartMonitorLooseblocking: m_iInotifyWatchFD                 = %d", m_iInotifyWatchFD));
    ETG_TRACE_SYS(( "StartMonitorLooseblocking: m_iInotifyWDUnsupDev              = %d", m_iInotifyWDUnsupDev ));
    ETG_TRACE_SYS(( "StartMonitorLooseblocking: m_iInotifyWDAppleDev              = %d", m_iInotifyWDAppleDev ));
    ETG_TRACE_SYS(( "StartMonitorLooseblocking: m_InotifyWDMtpDev                 = %d", m_InotifyWDMtpDev ));

    lockNotification();
    DEVICEPROCESSORHISTORY.AddToHistory ("inotify");


    ssize_t  size;
    char     buf[INOTIFY_SIZEOFBUF+1]; //space for 10 events

    ETG_TRACE_USR4(("StartMonitorLooseblocking: sizeof(buf)=%d",sizeof(buf)));
    memset( buf, '\0' , sizeof(buf) );
    size = read(m_iInotifyFD, buf, sizeof(buf)); //read all currently available events
    buf[INOTIFY_SIZEOFBUF] = '\0'; // sim4hi: fix for Coverity CID-17238

    if (-1 != size)
    {
        tBool bAllEventsChecked = FALSE;
        tInt iLenOfEvent = 0;
        tInt iOffset     = 0;

        while( bAllEventsChecked == FALSE )
        {
            INO_EVT* event  = (INO_EVT*)&buf[iOffset];       //cast to struct
            iLenOfEvent     = (tInt)(sizeof(INO_EVT) + event->len);
            CGlobalEnumerations::NOTIFICATION_Type enWNotiDevTyp;

            ETG_TRACE_COMP(("StartMonitorLooseblocking:event->wd     = %d",event->wd));
            ETG_TRACE_COMP(("StartMonitorLooseblocking:event->mask   = 0x%x",event->mask));
            ETG_TRACE_COMP(("StartMonitorLooseblocking:event->name[] = %s",event->name));

            //---------------------------------
            //type of device
            //---------------------------------
            //@todo recator to helper function  CGlobalEnumerations::NOTIFICATION_Type mapNotification(int watchDescriptor)
            if( event->wd == m_iInotifyWatchFD )
            {
                ETG_TRACE_COMP(("StartMonitorLooseblocking:USB or SDcard or CDRom are notified"));
                enWNotiDevTyp = CGlobalEnumerations::enInotify;
            }
            else if( event->wd == m_iInotifyWDUnsupDev )
            {
                enWNotiDevTyp = CGlobalEnumerations::enInUnsupDev;
                ETG_TRACE_COMP(("StartMonitorLooseblocking: unsupported USB device notified"));
            }
            else if( event->wd == m_iInotifyWDMscDev )
            {
                enWNotiDevTyp = CGlobalEnumerations::enInMscDev;
                ETG_TRACE_COMP(("StartMonitorLooseblocking: USB MSC device notified"));
            }
            else if( event->wd == m_iInotifyWDAppleDev )
            {
                enWNotiDevTyp = CGlobalEnumerations::enInAppleDev;
                ETG_TRACE_COMP(("StartMonitorLooseblocking: APPLE device notified"));
            }
            else if( event->wd == m_InotifyWDMtpDev )
            {
                enWNotiDevTyp = CGlobalEnumerations::enInMtpDev;
                ETG_TRACE_COMP(("StartMonitorLooseblocking: MTP device notified"));
            }
            else
            {
                enWNotiDevTyp = CGlobalEnumerations::enNotifyAll;
                ETG_TRACE_FATAL(("Unknown device notified"));
            }

            //-----------------------------------------------
            //check if more than one event has been received
            //-----------------------------------------------
            if( size > iLenOfEvent )
            {
                bAllEventsChecked = FALSE;
            }
            else
            {
                bAllEventsChecked = TRUE;
            }

            ETG_TRACE_USR1(( "StartMonitorLooseblocking: iLenOfEvent = %d  size = %d", iLenOfEvent, size ));

            l_cDevNode =  GENSTRING ("/dev/") + GENSTRING(event->name);

#ifdef VARIANT_S_FTR_ENABLE_AUTOMOUNTER_TXTIF
            //intermediate solution!
            if(( event->mask & IN_CREATE ))
#else
            if(( event->mask & IN_CLOSE_WRITE ))
#endif
            {
                ETG_TRACE_SYS(( "StartMonitorLooseBlocking: [INOTIFY_Type=%d] Device connected = %s", ETG_ENUM(NOTIFICATION_Type, enWNotiDevTyp), l_cDevNode.toStdString().c_str()) );
                m_cAction = "add";
                DEVICEPROCESSORHISTORY.AddToHistory ("add_i", l_cDevNode);
#ifdef BUGFIX_STARTUPRECOGNITION_INOTIFY
                {
                    ETG_TRACE_SYS(("StartMonitorLooseblocking: ADD thus LOOKUP MOUNTS"));
                    LookupExistingMounts( enWNotiDevTyp, event->name );      // e.g. enInotify, "sda1" -> here: file name in directory /tmp/.automount
                }
#endif
            }
            else if ((event->mask & IN_DELETE))
            {
                ETG_TRACE_SYS (("StartMonitorLooseBlocking: [INOTIFY] Device removed = %s",
                                l_cDevNode.toStdString().c_str()));
                m_cAction = "remove";

                DEVICEPROCESSORHISTORY.AddToHistory ("remove_i", l_cDevNode, GENSTRING("/dev/") + GENSTRING (event->name));
                //HandleUnMount (event->name);

                LookupRemovedMounts();

            }
            else if ((event->mask & IN_MODIFY))
            {
                ETG_TRACE_COMP(( "StartMonitorLooseBlocking: event->name= %s", event->name ))
                        //AddMountedDevice( "modify" ,enTypeOfNotify, GENSTRING(event->name) );

            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR]: EVENT UNKNOWN: StartMonitorLooseBlocking:event->mask = 0x%x", event->mask));
                ETG_TRACE_FATAL(("[ERROR]: INFO: KNOWN EVENT:CreateInotifyMonitor: watch: IN_DELETE | IN_CREATE | IN_CLOSE_WRITE: 0x%x | 0x%x | 0x%x",IN_DELETE,IN_CREATE,IN_CLOSE_WRITE));
            }

            if(bAllEventsChecked == FALSE)
            {
                iOffset  = iOffset   + iLenOfEvent;
                size     = size      - iLenOfEvent;
            }
        }//while bAllEventsChecked
    }//if -1!=size
    unlockNotification(); //warranty to serialise UDevEvents and to care that notification is not interrupted by cyclic diag

    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************vCASENotificationAutomount<-************************"));
}

tVoid CUDevManager::vCASENotificationPipe(trPipeParmsSendMountDev &f_oPipeParams)
{
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->CASENotificationPipe************************"));
    lockNotification();

    GENSTRING l_cDevNode;
    CDevice   l_oDevice;
    int       l_iDevProcessingStatus = DEVICEMANAGER_OK;

    tInt iLen = read(m_iPipeFD[enPipeReader], &f_oPipeParams, sizeof(trPipeParmsSendMountDev));
    f_oPipeParams.cDeviceName[sizeof(trPipeParmsSendMountDev::cDeviceName)-1]=0; // sim4hi: fix for Coverity CID-18516
    f_oPipeParams.cMountpoint[sizeof(trPipeParmsSendMountDev::cMountpoint)-1]=0; // sim4hi: see above
    ETG_TRACE_USR4(("vCASENotificationPipe: Read %d bytes from PIPE",iLen));
    if((tInt)sizeof(trPipeParmsSendMountDev) ==  iLen)
    {
        ETG_TRACE_USR4(("vCASENotificationPipe: Read PIPE: bMount     : 0x%x",f_oPipeParams.bMount));
        ETG_TRACE_USR4(("vCASENotificationPipe: Read PIPE: cDeviceName: %s"  ,f_oPipeParams.cDeviceName)); //expect e.g. ' /dev/sr0 here not sr0
        ETG_TRACE_USR4(("vCASENotificationPipe: Read PIPE: cMountpoint: %s"  ,f_oPipeParams.cMountpoint));

        //-----------
        //Mount
        //-----------
        if(true == f_oPipeParams.bMount)
        {
            //------------------------------
            // Generate the udev_device object
            //-----------------------------
            m_pDev = GenerateUDevDevice(m_pUdev,f_oPipeParams.cMountpoint);



            //add it if not connected
            l_cDevNode = f_oPipeParams.cDeviceName;
            if( DeviceListManager::GetInstance()->IsAlreadyConnected(l_cDevNode) == TRUE )
            {
                ETG_TRACE_USR1 (("vCASENotificationPipe: Device already connected : %s", l_cDevNode.toStdString().c_str()));
            }
            else
            {
                ETG_TRACE_USR1(( "vCASENotificationPipe: New device connected : %s", l_cDevNode.toStdString().c_str() ));
                // Process the new device
                l_oDevice.m_cMountPoint    = f_oPipeParams.cMountpoint;
                l_oDevice.m_cPartitionNode = f_oPipeParams.cDeviceName;

                //Bugfix for  NCG3D-9838
                struct udev_list_entry *l_pListEntry = NULL;
                udev_list_entry_foreach(l_pListEntry, udev_device_get_properties_list_entry(m_pDev))
                {
                    //NULL check for udev_list_entry_get_name(l_pListEntry)
                    if(NULL != udev_list_entry_get_name(l_pListEntry))
                    {
                        if(!strcmp(udev_list_entry_get_name(l_pListEntry),"ID_SERIAL"))
                        {
                            l_oDevice.m_cSerialID    =  udev_list_entry_get_value(l_pListEntry);
                            l_oDevice.m_cShortSerial = l_oDevice.m_cSerialID;
                            ETG_TRACE_USR1(( "vCASENotificationPipe: Got the property serial id from udev : %s", l_oDevice.m_cShortSerial.toStdString().c_str() ));
                            break;
                        }
                    }
                }
                l_iDevProcessingStatus     = AddDevice( CGlobalEnumerations::enInotify, l_cDevNode, l_oDevice );

                if(l_iDevProcessingStatus != DEVICEMANAGER_OK)
                {
                    ETG_TRACE_FATAL(("vCASENotificationPipe: l_iDevProcessingStatus     : 0x%x",l_iDevProcessingStatus));
                }
            }

            //----------------------------------------
            // Free the udev device
            //--------------------------------------
            if( m_pDev != NULL )              // @todo wird m_pDev noch verwendet????
            {
                udev_device_unref( m_pDev );
                m_pDev = NULL;
            }
        }
        //-----------------
        //Umount
        //-----------------
        else
        {
            l_cDevNode = f_oPipeParams.cDeviceName;
            ETG_TRACE_FATAL(("vCASENotificationPipe: Umount device: %s", l_cDevNode.toStdString().c_str()));
            DisconnectNotification(l_cDevNode.toStdString().c_str(), CGlobalEnumerations::SINGLE);
        }

    }
    unlockNotification(); //warranty to serialise UDevEvents and to care that notification is not interrupted by cyclic diag
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->CASENotificationPip*************************"));
}

tVoid CUDevManager::vCASENotificationAutomounterLib()
{
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->vCaseNotificationAutomounterLib************************"));
    lockNotification();
    unlockNotification(); //warranty to serialise UDevEvents and to care that notification is not interrupted by cyclic diag
    ETG_TRACE_SYS(("StartMonitorLooseblocking: ************->vCaseNotificationAutomounterLib*************************"));
}


void CUDevManager::setIDeviceRecognizer(IDeviceRecognizer *pIDeviceRecognizer)
{
    ETG_TRACE_USR1(("Begin: setIDeviceRecognizer"));
    m_pIDeviceRecognizer = pIDeviceRecognizer;

    if(m_pIDeviceRecognizer)
    {
        m_pIDeviceRecognizer->m_pMTPRawdevices   = NULL;
        m_pIDeviceRecognizer->m_iNumrawDevices   = 0;
        m_pIDeviceRecognizer->m_LIBMTP_Error     = LIBMTP_ERROR_NONE;
    }
    else
    {
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    ETG_TRACE_USR1(("End  : setIDeviceRecognizer"));
}

void CUDevManager::SetIAutomounterAdapter(IAutomounterAdapter *pIAutomounterAdapter)
{
    ETG_TRACE_USR1(("Begin: setIAutomounterAdapter"));
    m_pIAutomounterAdapter = pIAutomounterAdapter;
    ETG_TRACE_USR1(("End: setIAutomounterAdapter"));
}
/*-----------------------------------------------------------------------------*
 * int LookupNotMountedDevices()                           *
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::LookupNotMountedDevices()
{
    ETG_TRACE_USR1(("Begin: LookupUnMountedDevices"));
    int err = 0;
    tCString pName = NULL;

    DIR *automount = opendir( DIR_AUTOMOUNTER_MOUNTEDDEVICE_DB );
    if( NULL == automount )
    {
        err = errno;
        ETG_TRACE_ERR (("LookupNotMountedDevices: [ERROR] 'opendir' failed: %s", strerror (err)));
    }
    else
    {

        struct dirent *device = NULL;
        errno = 0;
        CGlobalEnumerations::NOTIFICATION_Type l_enWNotiDevTyp;


        l_enWNotiDevTyp = CGlobalEnumerations::enInotify;
        do
        {
            pName = NULL;                // reset pointer to file as indicator to don't process this directory any longer

            device = readdir(automount);
            if( device )
            {
                pName = device->d_name;  // copy file names if there are entries
                if(DT_DIR == device->d_type && IsValidMscType(pName,11))
                {
                    std::string devicedbdirstr = DIR_AUTOMOUNTER_MOUNTEDDEVICE_DB ;
                    devicedbdirstr += pName; //for example  /tmp/.automounter/device_db/
                    DIR *automountsubdir = opendir(devicedbdirstr.c_str());
                    if( NULL == automountsubdir )
                    {
                        err = errno;
                        ETG_TRACE_ERR (("LookupNotMountedDevices: [ERROR] 'opendir' failed: %s", strerror (err)));
                    }
                    else
                    {
                        struct dirent *deviceentry = NULL;
                        do
                        {
                            pName = NULL;
                            deviceentry = readdir(automountsubdir);
                            if(deviceentry)
                            {
                                pName = deviceentry->d_name;

                                if(( pName ) && ('.' != *pName ) )
                                {
                                    std::string devicedbfilnamestr = devicedbdirstr;//for example  /tmp/.automounter/device_db/device_dev_sda/
                                    devicedbfilnamestr +="/";
                                    devicedbfilnamestr += pName;//for example /tmp/.automounter/device_db/device_dev_sda/partition_dev_sda.info
                                    if(m_pIAutomounterAdapter)
                                    {
                                        CDevice oCurrentDevice;
                                        ETG_TRACE_USR1(("LookupNotMountedDevices Going to fill the data from the file %s",devicedbfilnamestr.c_str()));
                                        m_pIAutomounterAdapter->FillDevice(OUT oCurrentDevice,IN devicedbfilnamestr);

                                        tBool bSendUnsupported = TRUE;
                                        if(ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
                                        {
                                            ETG_TRACE_USR4(("Special case eConf_ENABLE_INTERNAL_DVD_DRIVE: oCurrentDevice.m_eDeviceUnsupportedReason=%d",ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,oCurrentDevice.m_eDeviceUnsupportedReason)));
                                            bSendUnsupported = FALSE;
                                        }


                                        if(    (TRUE == bSendUnsupported)
                                            && (1 != oCurrentDevice.m_iIsMounted))//not mounted
                                        {

                                            oCurrentDevice.m_cMountPoint = "UNKNOWN_";
                                            oCurrentDevice.m_cMountPoint += oCurrentDevice.m_cShortSerial;
                                            oCurrentDevice.m_bIsConnected = true;
                                            oCurrentDevice.m_iUSBConnector = USBConnectorUndef;
                                            if(oCurrentDevice.m_cBlockDevnode.size()>DIR_DEV_SUFFIX_LEN)
                                            {
                                                m_pDev = udev_device_new_from_subsystem_sysname(m_pUdev,"block",
                                                oCurrentDevice.m_cBlockDevnode.right(DIR_DEV_SUFFIX_LEN).toStdString().c_str());
                                                GENSTRING l_cDevNode;
                                                m_pBaseUSBDev = FindBaseUSBDevice(m_pDev);
                                                if(m_pBaseUSBDev)
                                                {
                                                    l_cDevNode = udev_device_get_devnode( m_pBaseUSBDev );
                                                    libusb_device *dev = LibUSBGetDevice( l_cDevNode.toStdString().c_str() );
                                                    LibUSBFillDeviceDetails( dev, oCurrentDevice );
                                                    ETG_TRACE_USR1(("LookupNotMountedDevices Device m_iDevNUM %d",oCurrentDevice.m_iDevNUM))
                                                    oCurrentDevice.m_cDevnode = l_cDevNode;
                                                    ETG_TRACE_USR1(("LookupNotMountedDevices Device l_cDevNode %s",l_cDevNode.toStdString().c_str()))
                                                    vMapFileSystemType(INOUT &oCurrentDevice);
                                                     //The original serial id of the device
                                                    //This will be used to identify all the partitions of multipartition  stick belongs to same device
                                                    oCurrentDevice.m_cAccessoryName2 = oCurrentDevice.m_cSerialID;
                                                    oCurrentDevice.m_cSerialID = oCurrentDevice.m_cSerialID + "_" + oCurrentDevice.m_cShortSerial;
                                                    if((!DeviceListManager::GetInstance()->isDeviceMultiPartitioned(DEVLIST_UDEVMGR, oCurrentDevice.m_cAccessoryName2))
                                                       && (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_UNKNOWN_DEVICES)))
                                                    {
                                                        oCurrentDevice.m_eDeviceUnsupportedReason = NotMountedbyAutomounter ;
                                                        ETG_TRACE_USR1(("LookupNotMountedDevices Set m_eDeviceUnsupportedReason to NotMountedDevice"));
                                                    }
                                                    else
                                                    {
                                                        oCurrentDevice.m_eDeviceUnsupportedReason = Multipart_UnSupportedPartition ;
                                                        ETG_TRACE_USR1(("LookupNotMountedDevices Multipartition device. Set m_eDeviceUnsupportedReason to Multipart_UnSupportedPartition"));
                                                    }

                                                    if(m_pStateTable)
                                                    {
                                                          int  l_iDevProcessingStatus = PushDeviceData(&oCurrentDevice);
                                                          if(DEVICEMANAGER_DEVICE_DETECTED == l_iDevProcessingStatus)
                                                          {
                                                              ETG_TRACE_USR2(("LookupNotMountedDevices: Device pushed successfully"));

                                                              ETG_TRACE_SYS(("LookupNotMountedDevices: m_cFSType = %s",
                                                                      oCurrentDevice.m_cFSType.toStdString().c_str()));
                                                              ETG_TRACE_COMP(("LookupNotMountedDevices: m_cDeviceName = %s",
                                                                      oCurrentDevice.m_cDeviceName.toStdString().c_str()));
                                                              oCurrentDevice.m_eConnectStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
                                                              //sending device to clients
                                                              m_pStateTable->setMount( enMountUdev, eUSBUndef, SIG_TRUE, &oCurrentDevice);

                                                          }
                                                    }// if(m_pStateTable)
                                                }//if(m_pBaseUSBDev)
                                            }//if(m_cBlockDevnode.size>DIR_DEV_SUFFIX_LEN)
                                        }//if(1 != oCurrentDevice.m_iIsMounted)//
                                    }//if(m_pIAutomounterAdapter)
                                }//if(( pName ) && ('.' != *pName ) )
                            }//if(deviceentry)
                        }//do whoile{..}  device entry
                        while(NULL!=deviceentry);
                        closedir(automountsubdir);
                    }//else of if( NULL == automountsubdir )
                }//if(DT_DIR == device->d_type && IsValidMscType(pName,11))
            }//if( device )


        }//do whoile{..}  device
        while (NULL != device);

    }

    (void)closedir( automount );

    ETG_TRACE_USR1(("End: LookupNotMountedDevices"));
}
/*-----------------------------------------------------------------------------*
*tVoid SetIStorageInfoProvider(IStorageInfoProvider *f_pIStorageInfoProvider)  *
*-----------------------------------------------------------------------------*/
tVoid CUDevManager::SetIStorageInfoProvider(IStorageInfoProvider *f_pIStorageInfoProvider)
{
    m_pStorageInfoProvider = (StorageInfoProvider*)f_pIStorageInfoProvider;
}

/*--------------------------------------------------*
tVoid vUpdateStorageInfo(IN CDevice &f_CurDevice)  *
*---------------------------------------------------*/
tVoid CUDevManager::vUpdateStorageInfo(IN const CDevice f_CurDevice)
{
    ETG_TRACE_USR1(("Begin: vUpdateStorageInfo"));
    if(TRUE == DeviceListManager::GetInstance()->isDeviceTypeConnected(DEVLIST_UDEVMGR,f_CurDevice.m_eUSB,f_CurDevice.m_eDeviceType))
    {
        //Device found.
        ETG_TRACE_USR1(("vUpdateStorageInfo Device found in the list. Sent the update device details"));
        if(m_pStateTable)
        {
            m_pStateTable->setMount( enMountUdev, f_CurDevice.m_eUSB, SIG_TRUE, &f_CurDevice);
            //Note that we are not updating the Storage info in the DEVLIST_UDEVMGR
        }

    }
    ETG_TRACE_USR1(("End: vUpdateStorageInfo"));
}
/*--------------------------------------------------*
tVoid vFillStorageInfoMSC                           *
*---------------------------------------------------*/
tVoid CUDevManager::vFillStorageInfoMSC()
{
    ETG_TRACE_USR1(("Begin: vFillStorageInfoMSC"));
    for(std::vector<CDevice>::iterator it = m_vDevicesWithOutStorageInfo.begin(); it != m_vDevicesWithOutStorageInfo.end(); ++it)
    {
        if(m_pStorageInfoProvider)
        {
            if(TRUE == m_pStorageInfoProvider->IsFreeSlotsAvailable())
            {
                ETG_TRACE_USR1((" vFillStorageInfoMSC Found a free slot to calculate the storage info for the device with Serail ID : %s",it->m_cSerialID.toStdString().c_str()));
                if(TRUE == DeviceListManager::GetInstance()->isDeviceTypeConnected(DEVLIST_UDEVMGR,it->m_eUSB,it->m_eDeviceType))
                {
                    m_pStorageInfoProvider->vGetSizeMSC(*it);
                }
                m_vDevicesWithOutStorageInfo.erase(it);
                break;
            }
            else
            {
                ETG_TRACE_USR1((" vFillStorageInfoMSC there is no free slots available to calculate the storage info"));
            }
        }

    }
    ETG_TRACE_USR1(("End: vFillStorageInfoMSC"));

}

/*--------------------------------------------------*
tVoid vDVDDriveInfoUpdated(IN trDVDDriveInfo f_DVDDriveInfo)  *
*---------------------------------------------------*/
tVoid CUDevManager::vDVDDriveInfoUpdated(IN trDVDDriveInfo f_DVDDriveInfo)
{
    ETG_TRACE_USR1(("Begin: vDVDDriveInfoUpdated"));

    ETG_TRACE_USR1(("vDVDDriveInfoUpdated: l_rDVDDriveInfo.DVDDriveMode = %d",ETG_ENUM(TENDVDDISCTYPE,f_DVDDriveInfo.DVDDriveMode)));
    ETG_TRACE_USR1(("vDVDDriveInfoUpdated: l_rDVDDriveInfo.DVDDiscType = %d",ETG_ENUM(TENDVDDRIVEMODE,f_DVDDriveInfo.DVDDiscType)));

    //Make use of f_DVDDriveInfo for future implementation where you need DVDdiscType or DVDDriveMode from Mediaplayer
    if((!GetIsDVDDriveMassStorageMode()/* If already set true, Do not spawn thread again for Mediaplayer redundant update*/)
        && (MASS_STORAGE_MODE == f_DVDDriveInfo.DVDDriveMode)
        && (DATA_DISC == f_DVDDriveInfo.DVDDiscType))
    {
        Locker locker(&m_UpdateDVDDriveInfoLock);
        SetIsDVDDriveMassStorageMode(true);
        ETG_TRACE_USR1(("DVD drivemode is in Mass storage mode. Retrigger device update for CDROM")); //NCG3D-79558
        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)CUDevManager::eThread_GetUdevManager, NULL); //starts thread eThread_GetUdevManager - see 'Do'-function
        ETG_TRACE_COMP(("CUDevManager: iThreadIndex:%d",iThreadIndex));
    }


    ETG_TRACE_USR1(("End: vDVDDriveInfoUpdated"));
}

/*--------------------------------------------------*
tVoid vThreadFunction                           *
*---------------------------------------------------*/
tVoid CUDevManager::vThreadFunction()
{
    ETG_TRACE_USR1(("Begin: vThreadFunction"));

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_INTERNAL_DVD_DRIVE))
    {
        ETG_TRACE_USR1(("Retriggering handlemount by calling LookupExistingMounts"));
        LookupExistingMounts();
    }

    ETG_TRACE_USR1(("End: vThreadFunction"));

}

/*-----------------------------------------------------------------------------*
tVoid CUDevManager::Do(...)
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do "));
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case CUDevManager::eThread_GetUdevManager: //execution tree for thread eThread_GetUdevManager
        {
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_GetUdevManager");
            vThreadFunction();
            break;
        }
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"));
}

/*-----------------------------------------------------------------------------*
tBool CUDevManager::GetIsDVDDriveMassStorageMode(...)
 *-----------------------------------------------------------------------------*/
tBool CUDevManager::GetIsDVDDriveMassStorageMode()
{
    Locker locker(&m_UpdateDVDDriveInfoLock);
    ETG_TRACE_USR4(("Returning m_bIsDVDDriveMassStorageMode"));
    return m_bIsDVDDriveMassStorageMode;
}

/*-----------------------------------------------------------------------------*
tVoid CUDevManager::SetIsDVDDriveMassStorageMode(...)
 *-----------------------------------------------------------------------------*/
tVoid CUDevManager::SetIsDVDDriveMassStorageMode(tBool f_bSetValue)
{
    Locker locker(&m_UpdateDVDDriveInfoLock);
    ETG_TRACE_USR4(("Setting m_bIsDVDDriveMassStorageMode to %d", f_bSetValue));
    m_bIsDVDDriveMassStorageMode = f_bSetValue;
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>
