/*-----------------------------------------------------------------------------*
 * PrmManager.h                                                                *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file PrmManager.h
 *
 * \brief This file holds the declaration for PrmManager
 *
 * PrmManager receives over current signals from USB HW. The prm signals are
 * used to trigger the mechanism to react on male function of a connector. E.g.
 * the VD_DeviceManager has to notify the clients about the MALFUNCTION and
 * OVER-CURRENT
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager
 *          changed
 * \version 24.11.2011, Koechling, Christian (Bosch), cleanup: care for singleton
 * \version 12.12.2011, Koechling, Christian (Bosch), cleanup singleton
 * \version 16.03.2012, Koechling, Christian (Bosch), added history for prmHW
 *          signals for special ULAB-Test
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __PRM_MANAGER_H__
#define __PRM_MANAGER_H__

/*-----------------------------------------------------------------------------*
 * include interfaces                                                                  *
 *-----------------------------------------------------------------------------*/
#include "IPrmManagerCard.h"
#include "IPrmManager.h"
#include "IHUBHardwareSignalReader.h"
#include "IStateTableHWSignals.h"     //old interface
#include "IStateTableHWMalfunction.h" //new interface
 


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace std;
using namespace prmmgr;
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/
typedef struct {
        tenSignalValue resultMounted; ///< result
        tInt           mediaChange;   ///< entry state 1
        tInt           mediaState;    ///< entry state 2
}mountPrmState;

/*-----------------------------------------------------------------------------*
 * forward declaration                                                           *
 *-----------------------------------------------------------------------------*/
 class StateTable;


/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class PrmManager
 */
class PrmManager : public IPrmManager, public IPrmManagerCard
{

    public:
    
 //=======================================
 // this part is added to make singleton of this object
 //=======================================

        /*! 
         * \brief Get PrmManager object
         *
         * Provides the singleton PrmManager object. First invocation of
         * this function creates the object
         *
         * \return [StateTable*] singleton object
         */

        static PrmManager* GetInstance();


      
        /*!
         * \brief Destroys the singleton object
         */
        static void  DestroyInstance();

        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'StateTable' which has no assignment
         * operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
        StateTable& operator=(const StateTable &oClientHandler);

//===================================================
      static tBool m_bConnectPRM;

       
        /*!
         * \brief Initializes the object
         *
         * \return [int] initialization status OSAL_OK or OSAL_ERROR
         */
         int Initialize();
#ifdef USE_FOURTH_USB_CONNECTOR
         /*!
          * \brief offers to diagnosis to access state of OC and PPON line - please refer to  fctCallbackHWSignals
          *
          * \param [out] f_objUSBDiagnosisHUB1 data container for USB1
          * \param [out] f_objUSBDiagnosisHUB2 data container for USB2
          * \param [out] f_objUSBDiagnosisHUB2 data container for USB3
          * \param [out] f_objUSBDiagnosisHUB2 data container for USB4
          * \todo: cleanup: replace parameters by reference to vector
          */
          virtual tVoid GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,
                 CUSBDiagnosis &f_objUSBDiagnosisHUB2,
                 CUSBDiagnosis &f_objUSBDiagnosisHUB3,
                 CUSBDiagnosis &f_objUSBDiagnosisHUB4) const;
#else
         /*!
         * \brief offers to diagnosis to access state of OC and PPON line - please refer to  fctCallbackHWSignals
         *
         * \param [out] f_objUSBDiagnosisHUB1 data container for USB1
         * \param [out] f_objUSBDiagnosisHUB2 data container for USB2
         * \param [out] f_objUSBDiagnosisHUB2 data container for USB3
         *
         * \todo: cleanup: replace parameters by reference to vector
         */
         virtual tVoid GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,
             CUSBDiagnosis &f_objUSBDiagnosisHUB2,
             CUSBDiagnosis &f_objUSBDiagnosisHUB3) const;
#endif
        /*!
         * \brief Set whether connect to Prm or not
         *
         * This function offers to enable/disable connection of PRM to state table.
         * This function is used for test purpose and triggered by TTFIS command.
         * E.g. for testing purpose it is possible to send simulated PRM signals
         * to the state machine. In such scenario real signals from prm would be
         * disturbing hence disconnection is offered for such test cases
         *
         * \param [in] bConnect: FALSE/TRUE = DISCONNECT/CONNECT
         */
         virtual tVoid connectPRM(tBool bConnect);


        /*!
        * \brief functoon to initialiseavailable USB connectors activateddeactivated in dependecy from values of ConfigurationManager
        *
        * \return TRUE/FALSE initialized with success or not
        */
         virtual tBool bInitUSBHardwareConnectors() const; //Roadmap 14003 Diagnosis2  activate/deactivate USB Hardware Connectors


        /*! 
        * \brief get if USB connector VBUS is switched on or off
        *        
        * \param [IN] eUSBConnectorNum: eUSB1, eUSB2 .... i.e. 
        * \param [out] eAvtivated SIG_TRUE means activated, SIG_FALSE disabled - variable won't be changed if function returns false 
        *
        * \return TRUE if result from prm could be caught. FALSE if an error occurred 
        */
         virtual tBool bGetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,OUT tenSignalValue &eActivated) const;//Roadmap 14003 ReadUSBPowerSupplay_Activated/Deactivated

                /*! 
        * \brief VBUS is switched on or off for certain USB connector
        *        
        * \param [IN] eUSBConnectorNum: eUSB1, eUSB2 .... i.e. 
        * \param [IN]TRUE means switch on FALSE means OFF
        *
        * \return TRUE if result from prm could be caught. FALSE if an error occurred 
        */
        virtual tBool bSetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,IN tBool bOn) const;//Roadmap 14003 ReadUSBPowerSupplay_Activated/Deactivated

        /*!
        * \brief Fill the device info from HUBHardwareSignalReader
        *
        *
        *
        */
        virtual  tVoid vFillDeviceSpeedinfo(OUT TelematicsBoxSpeedAndPortInfo f_TelematicsBoxSpeedAndPortInfo[ARRAYSIZEFORUSB]);

        /*!
        * \brief Set the Device speed of the device from ttfis command handler
        *
        *
        * \return pointer to HUBHardwareSignalReader
        */
        virtual  tVoid vsetDeviceSpeed(IN statetbl::tenDevManagerUSBHost eUsb,IN tBool bIsDevicePresent,IN tBool b_DeviceSpeed);

        /*!
        * \brief Set the interface of IHUBHardwareSignalReader
        *
        *
        *
        */
        virtual tVoid SetInterface(OUT IHUBHardwareSignalReader *pIHubHwSignalReader);

           /*!
        * \brief Set the interface IStateTableHWSignal to allow PRmManager to send info to StateTable
        *
        *
        *
        */
       tVoid SetInterface(IN IStateTableHWSignals *pIStateTableHWSignals); //old interface of statetable
    
        /*!
     * \brief Set the interface IStateTableHWMalfunction to allow PrmManager to send info to StateTable
     *
     *
     *
     */
       tVoid SetInterface(IN IStateTableHWMalfunction *pIStateTableHWMalfunction); //new interface of statetable


//===============================================
//interface implementation
//===============================================
    
    /*!
       * \brief see IPrmManagerCard
       */
     virtual  tBool bGetSDCardState(OUT CDeviceCard &oCard , IN tChar (&cUUIDDeviceName)[STRING_LENGTH]) const;


    private:

        IStateTableHWSignals     *m_pIStateTableHWSignals;     
        IStateTableHWMalfunction *m_pIStateTableHWMalfunction; 

//=======================================
// this part is added to make singleton of this object
//=======================================
        // static member variable declared for singleton class
               static PrmManager       *m_pPrmManager;
               static DVMLOCK           m_singelton;

        
        /*! 
         * \brief Private constructor for singleton
         *
         * \todo think about using directly singleton call instead of handing over pointer
         */
        PrmManager();

        /*!
         * \brief private destructor for singleton
         */
        ~PrmManager();
//==============================




        /*!
         * \brief Calback to register with the Prm 
         *
         * \return Response from Prm
         */
        tS32 s32PrmCallbackReg(tVoid);

        /*!
         * \brief used by s32PrmCallbackReg to register callbacks to receive HW-Signals i.e. PPON (VBUS nok/ok), OC (overcurrent) for USB-connectors 
         *
         * \return Response from Prm
         */
        tS32 s32PrmCallbackRegisterHWSignals();


        /*!
         * \brief used by s32PrmCallbackUnReg to unregister callbacks previously used to receive HW-Signals i.e. PPON (VBUS nok/ok), OC (overcurrent) for USB-connectors 
         *
         * \return Response from Prm
         */
        tS32 s32PrmCallbackUnRegisterHWSignals();

        /*!
         * \brief used by s32PrmCallbackReg to register callbacks to get info from internal SDCard reader if SDCard is connected
         *
         * \return Response from Prm
         */
        tS32 s32PrmCallbackRegisterSDCard();

           /*!
            * \brief used by s32PrmCallbackUnReg to unregister callbacks previoulsy used to get info from internal SDCard reader if SDCard is connected
            *
            * \return Response from Prm
            */

       tS32 s32PrmCallbackUnRegisterSDCard();

        /*!
         * \brief Calback to un-register with the Prm hardware
         *
         * \return Response from Prm
         */
        tS32 s32PrmCallbackUnReg(tVoid);


        /*!
         * \brief functions added because static function could not use non static member of interface injected inthe constructor
         *          Alternative is to check if it is possible to register non static function to the prm
         */
        tVoid SetMalfunctionUSBConnector(IN tBool bIsMalfunction, IN tenDevManagerUSBHost UsbConnectorNum);
        tVoid SetHWSignals(UsbPortState *pPortState);


    protected:

   

        /*! 
         * \brief Call back function of prm where it announces mount of a block device
         * 
         * Example deduced from traces:
         * Observed: HW signals are all posted before media ready again scenario
         * -# insert stick, no critvolt
         * OSAL_C_U16_NOTI_MEDIA_CHANGE,OSAL_C_U16_DATA_MEDIA
         * OSAL_C_U16_NOTI_MEDIA_STATE,OSAL_C_U16_MEDIA_READY
         *
         * -# concerning the timing
         * stick inserted, critvolt,then norm voltage again
         * OSAL_C_U16_NOTI_MEDIA_STATE,OSAL_C_U16_MEDIA_NOT_READY
         * OSAL_C_U16_NOTI_MEDIA_CHANGE,OSAL_C_U16_MEDIA_EJECTED
         *
         * USB1 OC:0; UV:2; PP:1;    0=UNDEF,1=FALSE,2=TRUE,
         * USB2 OC:0; UV:2; PP:1;
         * USB1 OC:2; UV:2; PP:1;
         * USB1 OC:2; UV:1; PP:1;
         * USB2 OC:0; UV:1; PP:1;
         * USB1 OC:2; UV:1; PP:2;    //PPON=TRUE means VBUS of USB switched on again
         * USB1 OC:1; UV:1; PP:2; ...// here things are ok again no OC1-Fault signal or UVolt anymore
         *
         * OSAL_C_U16_NOTI_MEDIA_CHANGE,OSAL_C_U16_DATA_MEDIA
         * OSAL_C_U16_NOTI_MEDIA_STATE,OSAL_C_U16_MEDIA_READY
         */
        static tVoid fctCallbackMount(tPCU32 pu32MediaChangeInfo,tPCU8 pu8Array);

        /*! 
         * \brief Call back function of prm where it announces hardware signals
         *        concerning USB connector
         *
         * e.g. we used MAX16944 and it provides a OC signal line to notify over
         * current. It has aPPON line which tells us if power supply of this
         * connector device is switched on or not
         *
         * \param [in] pu32ModeChangeInfo please refer to OSAL decription
         * \param [in] pVoid                this contains the information of signals
         */
        static tVoid fctCallbackHWSignals( tPCU32 pu32ModeChangeInfo, tVoid* pVoid );


        /*! 
          * \brief Call back function of prm where it announces SDCard inserted 
          *          this is used if the target has an internal SDCard reader e.g. like SUZSLN
          *
          * function is called by prm
          * \param [in] f_enConnected   SIG_TRUE: connected SIG_FALSE: disconnected
          * \param [in] f_strDeviceName reference on array which holds the UUID
          */
        static tVoid SetCardSignals(IN tenSignalValuePRM f_enConnected, IN tChar (&f_strDeviceName)[STRING_LENGTH]);


        /*! 
             * \brief Call back function of prm where it announces SDCard inserted 
             *          this is used if the target has an internal SDCard reader e.g. like SUZSLN
             *
             * function is called by prm
             * \param [in] pu32MediaChangeInfo to retriece media state and media type
             * \param [in] au8String contains name of mount point 
             */
        static tVoid fctCallbackSDCardInternal(tPCU32 pu32MediaChangeInfo,const tU8 au8String[]);

        static mountPrmState      m_rPrmMount[ARRAYSIZEFORUSB];
        static mountPrmState      PrmMountStateTable[];

        tU16                      m_u16MediaType;
        tU16                      m_u16MediaState;
        StateTable               *m_pStateTable;

        tBool                     m_CallbackSDCardRegistered;

        OSAL_trLoaderInfo         m_rLoaderInfo;
        OSAL_tIODescriptor        m_osalIODescriptorPRM;
        OSAL_trNotifyDataExt2     m_rNotifyData_HWSignals,m_rNotifyData_SDCardInternal;
        OSAL_trNotifyData         m_rNotifyData_Cards[ARRAYSIZEFORCARD+1]; // +1 because "/dev/crytcard" and "/dev/card" are for the same SD card slot. CF made two independet OSAL devices.

        IHUBHardwareSignalReader *m_pIHubHwSignalReader;



        /*!
        * \brief helper to copy UsbPortState variables
        *
        * \param [out] DestPortState holds data of SourcePortState after function has done its job
        * \param [in] SourcePortState
        *
        */
        static tVoid vCopyPortState(OUT UsbPortState &DestPortState, IN UsbPortState SourcePortState);

        /*!
        * \brief helper to fill parameter cmask for OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER of PRMDriver in vInitUSBHardwareConnectors()
        *
        * \param [out] return s cmask which is used as input parameter for OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER
        * \param [in] u8UsbConnectorNum: calculate cmask for  USB connector which should be activated/deactivated
        * \param [in] u8DemandedState:    calculate cmask  in with demanded state (activated/deactivated)
        *
        * \return mapped true/false
        */
        tBool bMapToMask(OUT /*usb_port_control*/ usb_port_control &rCmask, IN tU8 u8UsbConnectorNum, IN tU8 u8DemandedState) const /*const means function can't change members of class*/;


 

};

#endif //__PRM_MANAGER_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>
