/*-----------------------------------------------------------------------------*
 * PrmManager.cpp                                                              *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file PrmManager.cpp
 *
 * \brief This file holds the definition of PrmManager to receive over current
 *        signals from USB HW
 *
 * The prm signals are used to trigger the mechanism to react on male function
 * of a connector. E.g. the VD_DeviceManager has to notifiy the clients about
 * the MALFUNCTION and OVER-CURRENT.
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Keochling, Christain (Bosch),
 *          -# correct memset, correct strcpy in prm callback (prevent reset)
 *          -# care for compiler warnings,add traces
 * \version 17.11.2011, Koechling, Christian (Bosch),
 *          -# compilabilty enable switch off log mechanism used by MV - see LogTrace.h
 *          -# cleanup: 1st step to shift includes to Common.h
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 24.11.2011, Koechling, Christian (Bosch), cleanup: care for singleton
 * \version 28.11.2011, Koechling, Christian (Bosch), cleanup - add headers
 * \version 12.12.2011, Koechling, Christian (Bosch), cleanup singleton
 * \version 06.01.2012, Koechling, Christian (Bosch), cleanup FATAL traces
 * \version 30.01.2012, Koechling, Christian (Bosch), cleanup comment
 * \version 16.03.2012, Koechling, Christian (Bosch), added history for prmHW signals for special ULAB-Test
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */
/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
//#include <stdio.h>
#include "Config.h"


#define INCLUDE_VD_DVM_OSAL  //statetable uses ahl thread otherwhise this
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"
#include "PrmManager.h"
#include <unistd.h>
#include "debug/HistoryManager.h"
#include "config/ConfigurationManager.h"


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_PRMMANAGER
#include "trcGenProj/Header/PrmManager.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Defines & const                                                                     *
 *-----------------------------------------------------------------------------*/
#define UTIL_LOWORD(l)             ((tU16)((tU32)(l) & 0xffff))
#define UTIL_HIWORD(l)             ((tU16)((tU32)(l) >> 16))

//#define HIWORD( dwValue )   ((tU16)((((tU32)( dwValue )) >> 16) & 0xFFFF))

#define PRM_UNDEF -1

#define MOUNTED                     SIG_TRUE
#define UNMOUNTED                   SIG_FALSE
#define INTRANSITION                SIG_UNDEF

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * STATE table to evaluate from prm state mounted which is used as input       *
 *-----------------------------------------------------------------------------*/
/*state table to derive state mount from prm signals*/

//USE_CONNECT_PRMMOUNT this part might be cleaned please check

mountPrmState PrmManager::PrmMountStateTable[] =
{
/*       mounted        <-        OSAL_C_U16_NOTI_MEDIA_CHANGE.         |      OSAL_C_U16_NOTI_MEDIA_STATE              */
/*---------------------------------------------------------------------------------------------------------- */
/*No.  RESULT (04 entries):                                                                                                                                        */
/*01*/{  MOUNTED,                   OSAL_C_U16_DATA_MEDIA,          OSAL_C_U16_DEVICE_READY           },

/*02*/{  UNMOUNTED,                 OSAL_C_U16_DATA_MEDIA,          OSAL_C_U16_DEVICE_NOT_READY       },

/*03*/{  UNMOUNTED,                 OSAL_C_U16_MEDIA_EJECTED,       OSAL_C_U16_DEVICE_NOT_READY       },

/*04*/{  UNMOUNTED,                 OSAL_C_U16_MEDIA_EJECTED,       OSAL_C_U16_DEVICE_READY           },

/*05*/{  INTRANSITION,              OSAL_C_U16_DATA_MEDIA,          PRM_UNDEF                         }
/*----------------------------------------------------------------------------------------------------------- */
};

//set as if no device is connected at all
mountPrmState PrmManager::m_rPrmMount[] =
{ /*resultMounted      mediaChange                deviceReady */
  {SIG_FALSE ,       OSAL_NULL,                 OSAL_NULL                    },
  {SIG_FALSE,        OSAL_C_U16_MEDIA_EJECTED,  OSAL_C_U16_DEVICE_NOT_READY  },
  {SIG_FALSE,        OSAL_C_U16_MEDIA_EJECTED,  OSAL_C_U16_DEVICE_NOT_READY  }
};

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace prmmgr;
using namespace statetbl;
using namespace configmgr;

/*-----------------------------------------------------------------------------*
    static variable
 *-----------------------------------------------------------------------------*/
tBool PrmManager::m_bConnectPRM = TRUE; //can be set to FALSE using TTFIS command for debugging purpose


/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
// static pointer used to ensure a single instance of StateTable
PrmManager     *PrmManager::m_pPrmManager = NULL;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
DVMLOCK         PrmManager::m_singelton;
#endif

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
PrmManager::PrmManager()
{
    ETG_TRACE_USR4(("Begin: constructor PrmManager"));

    m_osalIODescriptorPRM  = (OSAL_tIODescriptor)OSAL_ERROR;
    m_u16MediaType = 0; //Bit Maske       OSAL_C_U16_UNKNOWN_MEDIA;
    m_u16MediaState= 0;

    memset(&m_rNotifyData_HWSignals     ,0x00,sizeof(OSAL_trNotifyDataExt2));
    memset(&m_rNotifyData_SDCardInternal,0x00,sizeof(OSAL_trNotifyDataExt2));
    memset(&m_rLoaderInfo               ,0x00,sizeof(OSAL_trLoaderInfo));

    m_bConnectPRM = TRUE;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
    PRMHISTORY.ClearHistory();
#endif

    for( int i=0; i<ARRAYSIZEFORCARD; ++i )
    {
        memset( &m_rNotifyData_Cards[i], 0x00, sizeof(OSAL_trNotifyData) );
    }

    m_CallbackSDCardRegistered = FALSE;

    m_pIHubHwSignalReader = NULL;
    m_pIStateTableHWSignals     = NULL;
    m_pIStateTableHWMalfunction = NULL;

    ETG_TRACE_USR4(("End  : constructor PrmManager"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
PrmManager::~PrmManager()
{
    try
    {
        s32PrmCallbackUnReg();
    }
    catch (const std::bad_alloc& badAllocException)
    {
        ETG_TRACE_ERR(("PrmManager:Failed to Unregister PRM callbacks - Exception caught:%s", badAllocException.what()));
    }

    m_pStateTable = NULL;
}


/*-----------------------------------------------------------------------------*
 * int Initialize()                                                            *
 *-----------------------------------------------------------------------------*/
int PrmManager::Initialize()
{
    tS32 s32RetVal;
    ETG_TRACE_USR4(("Begin: Initialize                                         "));
    s32RetVal= s32PrmCallbackReg();
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USR_PRM_HWCTRLUSB))
    {
#ifdef USE_HWCONFIGURATIONINIT
        bInitUSBHardwareConnectors();
#endif //USE_HWCONFIGURATIONINIT
    }

    ETG_TRACE_USR4(("End  : Initialize                                        "));

    return (int)s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackReg(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackReg(tVoid)
{
    tS32 s32RetVal = OSAL_ERROR;

    tS32 s32RetHWSignals = OSAL_OK;
    tS32 s32RetRegSDCard = OSAL_OK;
    tU32 u32ErrorCode;
    tU16 u16OpenPRM = 0;

    ETG_TRACE_USR4(("Begin: u32PrmCallbacksReg                                      "));

    u16OpenPRM += ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS);
    u16OpenPRM += ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_HWSIGNALS);

    if( (u16OpenPRM ==DVM_ON) || (u16OpenPRM ==DVM_ON+DVM_ON))
    {
        //------------------------------
        // OPEN /dev/prm
        //------------------------------
        ETG_TRACE_USR4(("s32PrmCallbackReg: OSAL_IOOpen for PRM"));
        //nok
        if(OSAL_ERROR==m_osalIODescriptorPRM)
        {
            m_osalIODescriptorPRM = OSAL_IOOpen("/dev/prm", OSAL_EN_READONLY);
            ETG_TRACE_USR4(("s32PrmCallbackReg: OSAL_IOOpen: m_osalIODescriptorPRM:%d (OSAL_ERROR=%d)",m_osalIODescriptorPRM, OSAL_ERROR));
        }

        //ok - prm open
        if(OSAL_ERROR !=m_osalIODescriptorPRM)
        {
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_HWSIGNALS))
            {
                s32RetHWSignals = s32PrmCallbackRegisterHWSignals();
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                HWMALFUNCTIONHISTORY.AddToHistory("[INFO] PrmManager::Initialize(): s32PrmCallbackRegisterHWSignals() ",NULL);
#endif
            }

            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
            {
                s32RetRegSDCard = s32PrmCallbackRegisterSDCard();
            }

            ETG_TRACE_USR4(("s32PrmCallbackReg:s32RetHWSignals:%d",s32RetHWSignals));
            ETG_TRACE_USR4(("s32PrmCallbackReg:s32RetRegSDCard:%d",s32RetRegSDCard));
        }
        //nok
        else
        {
            u32ErrorCode = OSAL_u32ErrorCode();
            ETG_TRACE_FATAL(( "OSAL_ERROR: Error Code: [0x%x], %s", u32ErrorCode, OSAL_coszErrorText(u32ErrorCode)  ));
            DVM_NORMAL_M_ASSERT(m_osalIODescriptorPRM!=OSAL_ERROR);
        }


        //ok
        if((s32RetHWSignals != OSAL_ERROR) && (s32RetRegSDCard != OSAL_ERROR) )
        {
            s32RetVal = OSAL_E_NOERROR;
        }

    }
    else
    {
        s32RetVal = OSAL_E_NOERROR;
    }

    ETG_TRACE_USR4(("End  : u32PrmCallbacksReg: s32RetVal %d",s32RetVal));
    return s32RetVal;

}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackRegisterHWSignals(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackRegisterHWSignals()
{
    tS32 s32RetVal;
    tU32 u32ErrorCode;
    //---------------------------
    //register for HWsignals
    //--------------------------
    m_rNotifyData_HWSignals.pu32Data            = 0;
    m_rNotifyData_HWSignals.ResourceName        = "/dev/usbpower" ;
    m_rNotifyData_HWSignals.u16AppID            = OSAL_C_U16_MP3CD_APPID ;
    m_rNotifyData_HWSignals.u8Status            = 0;
    m_rNotifyData_HWSignals.u16NotificationType = OSAL_C_U16_NOTI_LOW_POW;
    m_rNotifyData_HWSignals.pCallbackExt2       = (OSALCALLBACKFUNCEXT2)PrmManager::fctCallbackHWSignals;

    ETG_TRACE_USR4(("s32PrmCallbackReg: Register for HW-Signals"));
    s32RetVal = OSAL_s32IOControl(m_osalIODescriptorPRM, OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2, (intptr_t)&m_rNotifyData_HWSignals);

    if(OSAL_ERROR==s32RetVal)
    {
        u32ErrorCode = OSAL_u32ErrorCode();
        ETG_TRACE_FATAL(("ERROR: OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2 for HWsignals                        "));
        ETG_TRACE_FATAL(( "OSAL_ERROR: Error Code: [0x%x], %s", u32ErrorCode, OSAL_coszErrorText(u32ErrorCode)  ));
    }
    else
    {
        ETG_TRACE_USR1(("OK: OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2 for HWsignals                        "));
    }
    return s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackUnRegisterHWSignals(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackUnRegisterHWSignals()
{
    tS32 s32RetVal = OSAL_OK;

    //---------------------------
    //unregister HWSignals
    //--------------------------
    if(m_rNotifyData_HWSignals.pCallbackExt2==(OSALCALLBACKFUNCEXT2)PrmManager::fctCallbackHWSignals)
    {
        if(OSAL_ERROR==OSAL_s32IOControl(m_osalIODescriptorPRM, OSAL_C_S32_IOCTRL_PRM_UNREG_NOTIFICATION_EXT2, (intptr_t)&m_rNotifyData_HWSignals))
        {
            s32RetVal=OSAL_ERROR;
            ETG_TRACE_FATAL((" ERROR: u32PrmCallbacksUnReg - OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION   PrmManager::fctCallbackHWSignals"));
        }
        else
        {
            memset(&m_rNotifyData_HWSignals,0x00,sizeof(OSAL_trNotifyDataExt2));
            ETG_TRACE_COMP((" OK: u32PrmCallbacksUnReg - OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION   PrmManager::fctCallbackHWSignals"));
        }
    }
    else
    {
        s32RetVal=OSAL_ERROR;
        ETG_TRACE_FATAL((" ERROR: u32PrmCallbacksUnReg - m_rNotifyData_HWSignals.pCallback!=PrmManager::fctCallbackHWSignals"));
    }

    return s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackRegisterSDCard(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackRegisterSDCard()
{
    tS32 s32RetVal;
    tU32 u32ErrorCode;
    //---------------------------
    //register for SDCardinternal
    //--------------------------
    m_rNotifyData_SDCardInternal.u16AppID = OSAL_C_U16_DIAGNOSIS_APPID;
    m_rNotifyData_SDCardInternal.ResourceName = "/dev/media";
    m_rNotifyData_SDCardInternal.u16NotificationType = (OSAL_C_U16_NOTI_MEDIA_CHANGE);
    m_rNotifyData_SDCardInternal.pCallbackExt2 = (OSALCALLBACKFUNCEXT2)PrmManager::fctCallbackSDCardInternal;

    m_CallbackSDCardRegistered = TRUE; //to the control below triggers immendiately in SUZSLN the callback which calls bGetSDCardState- this is why this is set true before
    s32RetVal = OSAL_s32IOControl (m_osalIODescriptorPRM, OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2,(intptr_t)&m_rNotifyData_SDCardInternal);

    if(OSAL_ERROR==s32RetVal)
    {
        u32ErrorCode = OSAL_u32ErrorCode();
        ETG_TRACE_FATAL(("ERROR: OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2 for SDCardInternal                        "));
        ETG_TRACE_FATAL(( "OSAL_ERROR: Error Code: [0x%x], %s", u32ErrorCode, OSAL_coszErrorText(u32ErrorCode)  ));
        m_CallbackSDCardRegistered = FALSE; //set to false callback could not be registered successfully
    }
    else
    {
        ETG_TRACE_USR1(("OK: OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2 for SDCardInternal                        "));
    }
    return s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackUnRegisterSDCard(tVoid)                                               *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackUnRegisterSDCard()
{
    tS32 s32RetVal = OSAL_OK;
    //---------------------------
    //unregister SDCard internal
    //--------------------------
    if(m_rNotifyData_SDCardInternal.pCallbackExt2==(OSALCALLBACKFUNCEXT2)PrmManager::fctCallbackSDCardInternal)
    {
        if(OSAL_ERROR==OSAL_s32IOControl(m_osalIODescriptorPRM, OSAL_C_S32_IOCTRL_PRM_UNREG_NOTIFICATION_EXT2, (intptr_t)&m_rNotifyData_SDCardInternal))
        {
            s32RetVal=OSAL_ERROR;
            ETG_TRACE_FATAL((" ERROR: u32PrmCallbacksUnReg - OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION   PrmManager::fctCallbackSDCardInternal"));
        }
        else
        {
            memset(&m_rNotifyData_SDCardInternal,0x00,sizeof(OSAL_trNotifyDataExt2));
            ETG_TRACE_COMP((" OK: u32PrmCallbacksUnReg - OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION   PrmManager::fctCallbackSDCardInternal"));
        }
    }
    else
    {
        s32RetVal=OSAL_ERROR;
        ETG_TRACE_FATAL((" ERROR: u32PrmCallbacksUnReg - m_rNotifyData_SDCardInternal.pCallback!=PrmManager::fctCallbackSDCardInternal"));
    }
    return s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tS32 s32PrmCallbackUnReg(tVoid)                                             *
 *-----------------------------------------------------------------------------*/
tS32 PrmManager::s32PrmCallbackUnReg(tVoid)
{
    tS32 s32RetVal           = OSAL_OK;
    tS32 s32RetHWSignals     = OSAL_OK;
    tS32 s32RetSDCardSignals = OSAL_OK;

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_HWSIGNALS))
    {
        s32RetHWSignals  = s32PrmCallbackUnRegisterHWSignals();
    }

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
    {
        s32RetSDCardSignals  =s32PrmCallbackUnRegisterSDCard();
    }

    if( (s32RetHWSignals==OSAL_ERROR) || (s32RetSDCardSignals==OSAL_ERROR))
    {
        s32RetVal = OSAL_ERROR;
    }

    ETG_TRACE_COMP(("s32PrmCallbackUnReg: %d (OSAL_ERROR would be:%d)",s32RetVal,OSAL_ERROR));
    return s32RetVal;
}


tVoid PrmManager::vCopyPortState(OUT UsbPortState &DestPortState, IN UsbPortState SourcePortState)
{
    DestPortState.u8PortNr         = SourcePortState.u8PortNr;
    DestPortState.u8OC             = SourcePortState.u8OC;
    DestPortState.u8UV             = SourcePortState.u8UV;
    DestPortState.u8PPON           = SourcePortState.u8PPON;

    DestPortState.u32OCStartTime   = SourcePortState.u32OCStartTime;
    DestPortState.u32OCEndTime     = SourcePortState.u32OCEndTime;
    DestPortState.u32UVStartTime   = SourcePortState.u32UVStartTime;
    DestPortState.u32UVEndTime     = SourcePortState.u32UVEndTime;
    DestPortState.u32PPONStartTime = SourcePortState.u32PPONStartTime;
    DestPortState.u32PPONEndTime   = SourcePortState.u32PPONEndTime;
}

/*-----------------------------------------------------------------------------*
 * tVoid fctCallbackHWSignals(tPU32 pu32ModeChangeInfo, tPU8 pU8Array)         *
 *-----------------------------------------------------------------------------*/
 /*see unit tests VD_DeviceManager/utest/PrmManager for e.g. tests for such examples:
                                Example configuration a)
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT2_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_OFF;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

                                Example configuration b)
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_OFF;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

                                Example configuration c)
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT2_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_PRMPORT3_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

                                Example configuration d)
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT2_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_OFF;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

                                Example configuration e)
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT2_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT1_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_PRMPORT2_USED;
                                m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

*/

tVoid PrmManager::fctCallbackHWSignals(tPCU32 pu32ModeChangeInfo, tVoid* pVoid)
{
    UsbPortState        *pPortState = (UsbPortState*)pVoid;

    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>/fctCallbackHWSignals>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("fctCallbackHWSignals 0x%08x",*pu32ModeChangeInfo));
    ETG_TRACE_COMP(("                                                           "));
    ETG_TRACE_COMP(("  rPortState->u8PortNr: %d                                   ",ETG_ENUM(TENDEVMANAGERUSBHOST,pPortState->u8PortNr)));
    ETG_TRACE_COMP(("  rPortState->u8OC:     %d  [StartTime=%d   EndTime=%d]  ",ETG_ENUM(TENSIGNALVALUE,pPortState->u8OC),  pPortState->u32OCStartTime,  pPortState->u32OCEndTime));
    ETG_TRACE_COMP(("  rPortState->u8UV:     %d  [StartTime=%d   EndTime=%d]  ",ETG_ENUM(TENSIGNALVALUE,pPortState->u8UV),  pPortState->u32UVStartTime,  pPortState->u32UVEndTime));
    ETG_TRACE_COMP(("  rPortState->u8PPON:   %d  [StartTime=%d   EndTime=%d]  ",ETG_ENUM(TENSIGNALVALUE,pPortState->u8PPON),pPortState->u32PPONStartTime,pPortState->u32PPONEndTime));
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<fctCallbackHWSignals\<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));

    UsbPortState l_PortStateToSend;;
    UsbPortState l_PortState;

    vCopyPortState(OUT l_PortState, IN *pPortState); //decouple from driver pointer

    ConfigurationManager::GetInstance()->vPrintConfigurationUSBConnectorMapping();

    /*send to MsgQ received in StateTable::vThreadFunction */
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_HWSIGNALS))
    {
        if(TRUE == m_bConnectPRM)
        {

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
        PRMHISTORY.AddToHistory (pPortState);
#endif
            int l_iConfigVal;
            //---------------------------------------------------------------------------------------------------------------
            // temporarily not available mechanism t to cope with USB switched off by lower layer hard and software due to overcurrent or critical voltage
            //---------------------------------------------------------------------------------------------------------------



            tBool bIsMalfunction = FALSE;
            if((tU8)SIG_TRUE == l_PortState.u8OC )   //tbd.: we have to check with ulabe tests the behaviour to see if it is better to listen to (OC==TRUE) && (PPON==FALSE)
            {
                bIsMalfunction = TRUE;
            }

            ETG_TRACE_USR4(("HWMalfunction is for: eUSB%d",l_PortState.u8PortNr)); //e.g. eUSB1
            //l_iConfigVal = ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)(tUInt)(eCONF_HWSIGNAL_SOURCE_USB1+(l_PortState.u8PortNr-1))); //e.g. DVM_PRMPORT1 e.g. m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1+0] = DVM_PRMPORT1_USED;
            //check for all USBConnectors visible at the outside of the target if these ae also configured to use this value and send function to
            for(tInt i = (tInt)eUSB1; i < (tInt)eUSBAll; i++)
            {
                int l_iConfigValAtUSBConnector = ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)(tUInt)((tUInt)eCONF_HWSIGNAL_SOURCE_USB1+(i-1)));
                if(l_iConfigValAtUSBConnector == DVM_OFF)
                {
                    ETG_TRACE_COMP(("fctCallbackHWSignals: u16GetConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB%d] == DVM_OFF - HW signal PRMUSB%d ignored for USB%d",i,l_PortState.u8PortNr,i));
                }
                else
                {
                    tU16 u16Offset = DVM_PRMPORT1_USED;
                    if(((l_iConfigValAtUSBConnector - u16Offset)+1) == l_PortState.u8PortNr)
                    {
                        //--------------------------------------
                        //send Malfunction to StateTable
                        //--------------------------------------
                        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
                        {
                            ETG_TRACE_COMP(("fctCallbackHWSignals: eConf_ENABLE_MALFUNCTION_USBCONNECTORS is ON.  IStateTableHWMalfunction interface will be active"));
                            PrmManager::GetInstance()->SetMalfunctionUSBConnector(bIsMalfunction,(tenDevManagerUSBHost)i);
                        }
                        //--------------------------------------
                        //deprocated part: send HW-signals
                        //--------------------------------------
                        else
                        {
                            ETG_TRACE_COMP(("fctCallbackHWSignals: eConf_ENABLE_MALFUNCTION_USBCONNECTORS is OFF.  IStateTableHWSignals interface will be active"));

                            vCopyPortState(OUT l_PortStateToSend,IN l_PortState);
                            l_PortStateToSend.u8PortNr = i;

                            PrmManager::GetInstance()->SetHWSignals(&l_PortStateToSend);
                        }
                    }
                }
            }
        }
        else
        {
            ETG_TRACE_FATAL(("ERROR: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
            ETG_TRACE_FATAL(("ERROR: PRM:fctCallbackHWSignals: Using HW signals is not configured"));
            ETG_TRACE_FATAL(("ERROR: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        }
    }
}




/*

<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> [PrmManager.cpp(644)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :unknown Trace:57 44 04 64 10 FD 09 DB 22 7E 02 8B 2C 75 2D 80[PrmManager.cpp(651)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :fctCallbackSDCardInternal:Notification received: Media change [PrmManager.cpp(659)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< [PrmManager.cpp(676)]

<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> [PrmManager.cpp(644)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :unknown Trace:57 44 04 64 10 FD 09 DB 22 7E 02 8B 2C 75 2D 80[PrmManager.cpp(651)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :fctCallbackSDCardInternal:Notification received: Media change [PrmManager.cpp(659)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :fctCallbackSDCardInternal:Notification received: Data media [PrmManager.cpp(664)]
<dragon@cesock1>(core0)VD_DVM_MGR_PRM__CMP :<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< [PrmManager.cpp(676)]
*/
/*-----------------------------------------------------------------------------*
 * tVoid fctCallbackSDCardInternal( tPU32 pu32MediaChangeInfo, tU8 au8String[] )                            *
 *-----------------------------------------------------------------------------*/
tVoid PrmManager::fctCallbackSDCardInternal( tPCU32 pu32MediaChangeInfo, const tU8 au8String[])
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>/fctCallbackSDCardInternal>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));

    tChar cDeviceUUID[STRING_LENGTH];
    tenSignalValue l_enConnected = SIG_UNDEF;


    tU16 l_u16MediaType  = OSAL_C_U16_MEDIA_EJECTED;
    tU16 l_u16MediaState = OSAL_C_U16_DEVICE_NOT_READY;

    tU16 u16Parm1 = UTIL_HIWORD(*pu32MediaChangeInfo);
    tU16 u16Parm2 = UTIL_LOWORD(*pu32MediaChangeInfo);

    //-------------------------------------
    //care for parameter pu32MediaChangeInfo
    //-------------------------------------
    switch(u16Parm1)
    {
        //---------------------------------
        case OSAL_C_U16_NOTI_MEDIA_CHANGE:
        //---------------------------------
            l_u16MediaType = u16Parm2;
            ETG_TRACE_COMP(("---------------------------------------------------------------------------------------"));
            ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_MEDIA_CHANGEb(l_u16MediaType=ox%x)",l_u16MediaType));
            ETG_TRACE_COMP(("---------------------------------------------------------------------------------------"));
            switch(u16Parm2)
            {
               case OSAL_C_U16_DATA_MEDIA:
                  l_enConnected = SIG_TRUE;
                  ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_MEDIA_CHANGE->OSAL_C_U16_DATA_MEDIA"));
                  break;
               case OSAL_C_U16_MEDIA_EJECTED:
                  l_enConnected = SIG_FALSE;
                  ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_MEDIA_CHANGE->OSAL_C_U16_MEDIA_EJECTED"));
                  break;
               default:
                  ETG_TRACE_COMP(("fctCallbackSDCardInternal:UNUSED STATE: OSAL_C_U16_NOTI_MEDIA_CHANGE u16Parm2=0x%x",u16Parm2));
                  break;
            }
            break;
        //---------------------------------
        case OSAL_C_U16_NOTI_MEDIA_STATE:
        //---------------------------------
            l_u16MediaState = u16Parm2;
            ETG_TRACE_COMP(("------------------------------------------------------"));
            ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_MEDIA_STATE (l_u16MediaState=0x%x)",l_u16MediaState));
            ETG_TRACE_COMP(("------------------------------------------------------"));

            switch(u16Parm2)
            {
               case OSAL_C_U16_DEVICE_NOT_READY:
                   ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_DEVICE_READY->OSAL_C_U16_DEVICE_NOT_READY"));
                  break;
                case OSAL_C_U16_DEVICE_READY:
                   ETG_TRACE_COMP(("fctCallbackSDCardInternal:OSAL_C_U16_NOTI_DEVICE_READY->OSAL_C_U16_DEVICE_READY"));
                   break;
                 default:
                    ETG_TRACE_COMP(("fctCallbackSDCardInternal:UNUSED STATE: OSAL_C_U16_NOTI_MEDIA_CHANGE u16Parm2=0x%x",u16Parm2));
                    break;
            }
            break;
        //--------------------------------
        default:
        //--------------------------------
            break;
    }

    //-------------------------------------
    //care for parameter au8String[]
    //-------------------------------------
    ETG_TRACE_COMP(("fctCallbackSDCardInternal: STRING_MAXINDEX : %d",(int)STRING_MAXINDEX));
    memcpy(cDeviceUUID,au8String,STRING_MAXINDEX);
    cDeviceUUID[STRING_MAXINDEX] = '\0';

    ETG_TRACE_COMP(("fctCallbackSDCardInternal: cDeviceUUID: %s",cDeviceUUID));


    //------------------------------------------------
    // send to MsgQ received in StateTable::vThreadFunction
    //------------------------------------------------
    if( m_bConnectPRM == TRUE )
    {
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
       if(l_enConnected != SIG_UNDEF)
       {
           if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
           {
               StateTable::GetInstance()->SetCardSignals(IN l_enConnected, IN cDeviceUUID, IN l_u16MediaType);
           }

       }
#endif
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: fctCallbackCard: FALSE == m_pPrmManager->m_bConnectPRM"));
    }
    ETG_TRACE_COMP(("fctCallbackSDCardInternal: l_enConnected: %d",ETG_ENUM(TENSIGNALVALUE,l_enConnected)));
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>fctCallbackSDCardInternal\>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));

}

#ifdef USE_FOURTH_USB_CONNECTOR
/*-----------------------------------------------------------------------------*
 * tVoid GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,              *
 *          CUSBDiagnosis &f_objUSBDiagnosisHUB2,
 *          CUSBDiagnosis &f_objUSBDiagnosisHUB3,
 *          CUSBDiagnosis &f_objUSBDiagnosisHUB4)                              *
 *-----------------------------------------------------------------------------*/
tVoid PrmManager::GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,
        CUSBDiagnosis &f_objUSBDiagnosisHUB2,
        CUSBDiagnosis &f_objUSBDiagnosisHUB3,
        CUSBDiagnosis &f_objUSBDiagnosisHUB4) const
#else
/*-----------------------------------------------------------------------------*
* tVoid GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,              *
*          CUSBDiagnosis &f_objUSBDiagnosisHUB2,
*          CUSBDiagnosis &f_objUSBDiagnosisHUB3)                              *
*-----------------------------------------------------------------------------*/
tVoid PrmManager::GetDiagnosisValues(CUSBDiagnosis &f_objUSBDiagnosisHUB1,
    CUSBDiagnosis &f_objUSBDiagnosisHUB2,
    CUSBDiagnosis &f_objUSBDiagnosisHUB3) const
#endif
{
    ETG_TRACE_USR4(("Begin: GetDiagnosisValues"));

    voltageFailure_HWSignalsAll_t signal;
    voltageFailure_Result_t       msg2Clients;
    int dvmConfigValue[ARRAYSIZEFORUSB];

    dvmConfigValue[eUSB1] = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HWSIGNAL_SOURCE_USB1);
    dvmConfigValue[eUSB2] = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HWSIGNAL_SOURCE_USB2);
    dvmConfigValue[eUSB3] = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HWSIGNAL_SOURCE_USB3);
#ifdef USE_FOURTH_USB_CONNECTOR
    dvmConfigValue[eUSB4] = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HWSIGNAL_SOURCE_USB4);
#endif

    //DVM_PRM_USED         means signal has been send to stateTable using prm HW-signals lilek OC and PPON
    //DVM_HUBINTERNAL_USED means signal has been send to stateTable using HUBcommunication i.e. libusb communication for PortStatus is done

    //DVM_PRMPORT1_USED, DVM_PRMPORT2_USED,DVM_PRMPORT3_USED,DVM_PRMPORT4_USED means tha same like DVM_PRM_USED but helps for customer specific numbering of USBConnectors
    //adjust to DVM_PRM_USED if one of those used
    for(tInt i=eUSB1;i<ARRAYSIZEFORUSB;i++)
    {
        if(   (dvmConfigValue[i] == DVM_PRMPORT1_USED) ||
              (dvmConfigValue[i] == DVM_PRMPORT2_USED) ||
              (dvmConfigValue[i] == DVM_PRMPORT3_USED)
#ifdef USE_FOURTH_USB_CONNECTOR
              ||(dvmConfigValue[i] == DVM_PRMPORT4_USED)
#endif
          )
        {
            dvmConfigValue[i] = DVM_PRM_USED;
        }

    }



    if(DVM_PRM_USED == dvmConfigValue[eUSB1] || dvmConfigValue[eUSB1] == DVM_HUBINTERNAL_USED)
    {
        //------USB1-------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
        StateTable::GetInstance()->getSignals(eUSB1,&signal);
        StateTable::GetInstance()->getLastMsgSendtoClients(eUSB1,&msg2Clients);
#endif
        f_objUSBDiagnosisHUB1.m_enU60_drop      = signal.U60_drop;
        f_objUSBDiagnosisHUB1.m_enU50_USB_Reset = signal.U50_USB_Reset;
        f_objUSBDiagnosisHUB1.m_enOC            = signal.OC;
        f_objUSBDiagnosisHUB1.m_enPPON          = signal.PPON;
        f_objUSBDiagnosisHUB1.m_enMountedUSB    = signal.mountedUSB;
        f_objUSBDiagnosisHUB1.m_rStatus         = msg2Clients.resultUSB;
    }

    if(DVM_PRM_USED == dvmConfigValue[eUSB2] || DVM_HUBINTERNAL_USED == dvmConfigValue[eUSB2])
    {
        //------USB2-------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
        StateTable::GetInstance()->getSignals(eUSB2,&signal);
        StateTable::GetInstance()->getLastMsgSendtoClients(eUSB2,&msg2Clients);
#endif
        f_objUSBDiagnosisHUB2.m_enU60_drop      = signal.U60_drop;
        f_objUSBDiagnosisHUB2.m_enU50_USB_Reset = signal.U50_USB_Reset;
        f_objUSBDiagnosisHUB2.m_enOC            = signal.OC;
        f_objUSBDiagnosisHUB2.m_enPPON          = signal.PPON;
        f_objUSBDiagnosisHUB2.m_enMountedUSB    = signal.mountedUSB;
        f_objUSBDiagnosisHUB2.m_rStatus         = msg2Clients.resultUSB;
    }
    if(DVM_PRM_USED == dvmConfigValue[eUSB3] || DVM_HUBINTERNAL_USED == dvmConfigValue[eUSB3])
    {
    //------USB3-------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
        StateTable::GetInstance()->getSignals(eUSB3,&signal);
        StateTable::GetInstance()->getLastMsgSendtoClients(eUSB3,&msg2Clients);
#endif
        f_objUSBDiagnosisHUB3.m_enU60_drop      = signal.U60_drop;
        f_objUSBDiagnosisHUB3.m_enU50_USB_Reset = signal.U50_USB_Reset;
        f_objUSBDiagnosisHUB3.m_enOC            = signal.OC;
        f_objUSBDiagnosisHUB3.m_enPPON          = signal.PPON;
        f_objUSBDiagnosisHUB3.m_enMountedUSB    = signal.mountedUSB;
        f_objUSBDiagnosisHUB3.m_rStatus         = msg2Clients.resultUSB;
    }

#ifdef USE_FOURTH_USB_CONNECTOR
    if (DVM_PRM_USED == dvmConfigValue[eUSB4] || DVM_HUBINTERNAL_USED == dvmConfigValue[eUSB4])
    {
        //------USB4-------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
        StateTable::GetInstance()->getSignals(eUSB4, &signal);
        StateTable::GetInstance()->getLastMsgSendtoClients(eUSB4, &msg2Clients);
#endif
        f_objUSBDiagnosisHUB4.m_enU60_drop = signal.U60_drop;
        f_objUSBDiagnosisHUB4.m_enU50_USB_Reset = signal.U50_USB_Reset;
        f_objUSBDiagnosisHUB4.m_enOC = signal.OC;
        f_objUSBDiagnosisHUB4.m_enPPON = signal.PPON;
        f_objUSBDiagnosisHUB4.m_enMountedUSB = signal.mountedUSB;
        f_objUSBDiagnosisHUB4.m_rStatus = msg2Clients.resultUSB;
    }
#endif

    ETG_TRACE_USR4(("End  : GetDiagnosisValues"));
}

/*-----------------------------------------------------------------------------*
 * tVoid connectPRM(tBool bConnect)                                            *
 *-----------------------------------------------------------------------------*/
tVoid PrmManager::connectPRM(tBool bConnect)
{
    ETG_TRACE_COMP(("connectPRM: bConnect=0x%x",bConnect));
    m_bConnectPRM=bConnect;
}

/*-----------------------------------------------------------------------------*
 * tVoid connectPRM(tBool bConnect)                                            *
 *-----------------------------------------------------------------------------*/
tBool PrmManager::bGetSDCardState( OUT CDeviceCard &oCard , tChar (&cUUIDDeviceName)[STRING_LENGTH]) const
{
    ETG_TRACE_USR4(( "Begin: bGetSDCardState" ));

    tBool bRet = m_CallbackSDCardRegistered;
    //-------------------------
    // Set default/init values
    //-------------------------
    oCard.vSetErrorValues();

    //----------------
    //copy input variable
    //----------------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  /*intermediate solution to make unittests compilable*/
    oCard.m_strUUIDDeviceName = cUUIDDeviceName;
#endif

    if(m_CallbackSDCardRegistered)
    {
        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
        {
            //------------------------------------------------
            //filter if this is really a card otherwhise prevent calling it
            //------------------------------------------------
            bool bIsSDCard = ConfigurationManager::GetInstance()->bIsOsalNotifiedDeviceSDCardInternal(GENSTRING(cUUIDDeviceName));

            if(bIsSDCard)
            {
                //---------------------------------
                //care to receive card info using osal driver
                //---------------------------------
                OSAL_tIODescriptor      hOsalIODescriptorDevCard;
                OSAL_trIOCtrlCardState  State;
                tS32                    s32ReturnValue;

                hOsalIODescriptorDevCard = OSAL_IOOpen( STRING_CARD_INTERFACE, OSAL_EN_READONLY );

                ETG_TRACE_USR4(( "bGetSDCardState: hOsalIODescriptorDevCard: %d ", hOsalIODescriptorDevCard));
                ETG_TRACE_USR4(( "bGetSDCardState: device                  : %s ", STRING_CARD_INTERFACE ));
                ETG_TRACE_USR4(( "bGetSDCardState: cUUIDDeviceName         : %s ", cUUIDDeviceName ));
                ETG_TRACE_USR4(( "bGetSDCardState: size:%d, STRING_LENGTH: %d) ", sizeof(cUUIDDeviceName),STRING_LENGTH ));

                if( hOsalIODescriptorDevCard != OSAL_ERROR )
                {
                    strncpy(State.szDevicName,cUUIDDeviceName,sizeof(cUUIDDeviceName));

                    s32ReturnValue = OSAL_s32IOControl( hOsalIODescriptorDevCard, OSAL_C_S32_IOCTRL_CARD_STATE, (intptr_t)(&State) );
                    //ok
                    if( s32ReturnValue != OSAL_ERROR )
                    {
                        // Copy OSAL data to CDeviceCard
                        oCard.m_bHW_WriteProtected  = State.bHW_WriteProtected;
                        oCard.m_bSW_WriteProtected  = State.bSW_WriteProtected;
                        oCard.m_bMounted            = State.bMounted;
                        oCard.m_u8ManufactureId     = State.u8ManufactureId;
                        oCard.m_u32SerialNumber     = State.u32SerialNumber;
                        oCard.m_u8SDCardSpecVersion = State.u8SDCardSpecVersion;
                        oCard.m_u64CardSize         = State.u64CardSize;
                        oCard.m_u16MediaState       = m_u16MediaState;
                        for( tInt i=0; i<SizeCIDRegister; ++i )
                        {
                            oCard.m_u8CIDRegister[i] = State.u8CIDRegister[i];
                        }
                    }
                    //nok
                    else
                    {
                        tU32 u32ErrorCode = OSAL_u32ErrorCode();
                        ETG_TRACE_FATAL(( "bGetSDCardState: OSAL_ERROR: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode )  ));
                        //returns with default values
                    }
                    ETG_TRACE_USR4(( "bGetSDCardState: Close OSAL descriptor hOsalIODescriptorDevCard = %d", hOsalIODescriptorDevCard ));
                    OSAL_s32IOClose( hOsalIODescriptorDevCard );
                }
                ///nok
                else
                {
                    ETG_TRACE_USR3(( "bGetSDCardState: hOsalIODescriptorDevCard == OSAL_ERROR -> Load default values for Card State" ));
                    //returns with default values
                }
            }
            else
            {
                  ETG_TRACE_USR4(( "bGetSDCardState: No OSalCall cUUIDDeviceName %s does not belong to SDCard internal", cUUIDDeviceName));
            }

        }
        else
        {
            ETG_TRACE_FATAL(("--------------------------------------------------------------"));
            ETG_TRACE_FATAL(("[INFO] bGetSDCardState:eConf_ENABLE_USE_PRM_SDCARDSIGNALS disabled"));
            ETG_TRACE_FATAL(("--------------------------------------------------------------"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[WARNING] bGetSDCardState: m_CallbackSDCardRegistered==FALSE"));
    }

    ETG_TRACE_USR4(( "End: bGetSDCardState bRet=0x%x",bRet ));

    return bRet;
}


//@todo think about rCMans to be a INOUT
tBool  PrmManager::bMapToMask(OUT /*usb_port_control*/ usb_port_control &rCmask, IN tU8 u8UsbConnectorNum, IN tU8 u8DemandedState) const
{
    tBool bRet = FALSE;

    rCmask.PortMask  = 0x0; //check this!
    rCmask.ValueMask = 0x0;

    //simplify code
    if((tU8)eUSB1 == u8UsbConnectorNum)
    {
        rCmask.PortMask  |= (1<<0);
        if(DVM_ON == u8DemandedState)
        {
            rCmask.ValueMask |= (1<<0);
        }
        else
        {
            rCmask.ValueMask &= ~(1<<0);
        }
        bRet = TRUE;
    }
    else if((tU8)eUSB2 == u8UsbConnectorNum)
    {
        rCmask.PortMask |= (1<<1);
        if(DVM_ON == u8DemandedState)
        {
            rCmask.ValueMask |= (1<<1);
        }
        else
        {
            rCmask.ValueMask &= ~(0<<1);
        }
        bRet = TRUE;
    }
    else if((tU8)eUSB3 == u8UsbConnectorNum)
    {
        //TODO Change the logic for USB connector 3
        rCmask.PortMask |= (1<<2);
        if(DVM_ON == u8DemandedState)
        {
            rCmask.ValueMask |= (1<<2);
        }
        else
        {
            rCmask.ValueMask &= ~(0<<2);
        }
        bRet = TRUE;
    }
#ifdef USE_FOURTH_USB_CONNECTOR
    else if ((tU8)eUSB4 == u8UsbConnectorNum)
    {
        //TODO Change the logic for USB connector 4
        rCmask.PortMask |= (1 << 3);
        if (DVM_ON == u8DemandedState)
        {
            rCmask.ValueMask |= (1 << 3);
        }
        else
        {
            rCmask.ValueMask &= ~(0 << 3);
        }
        bRet = TRUE;
    }
#endif//USE_FOURTH_USB_CONNECTOR

/*
           LSB
             Bit 31 set
PortMask     1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32
ValueMask    0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32

Means USB1 is physically deactivated
----------------------------------------------------------------------------------
           LSB
PortMask     1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32
ValueMask    1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32

Means USB1 is physically activated
----------------------------------------------------------------------------------

           LSB
                     Bit 30 set
PortMask     0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32
ValueMask    0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32

Means USB2 is physically deactivated
----------------------------------------------------------------------------------
           LSB
PortMask     0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32
ValueMask    0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 �up to 32

Means USB2 is physically activated
*/

    ETG_TRACE_USR4(("bMapToMask: IN  u8UsbConnectorNum: USB%d",u8UsbConnectorNum));
    ETG_TRACE_USR4(("bMapToMask: IN  u8DemandedState  :    %d ",u8DemandedState));
    ETG_TRACE_USR4(("bMapToMask: OUT rCmask.PortMask  :    %d ",rCmask.PortMask));
    ETG_TRACE_USR4(("bMapToMask: OUT rCmask.ValueMask :    %d ",rCmask.ValueMask));
    ETG_TRACE_USR4(("bMapToMask: OUT bRet             :    %d ",ETG_ENUM(BOOL,bRet)));
    return bRet;
}


/* tbd.
s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT, (intptr_t)&s32PortCount);

what is the value for s32FunRetVal in OK-case and NOK-Case?
OSAL_s32IOControl always returns OSAL_OK on success and OSAL_ERROR on failure case. Error Code is set in the calling thread's context.
Here the possible error codes is OSAL_E_INVALIDVALUE incase the parameter (&s32PortCount)  is incorrect.

what is the value for s32PortCount in NOK-Case?
s32PortCount  will not modified in NOK case.

to get e.g. value PPON for USB1
rPortState.u8PortNr = 1
s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE, (intptr_t)&rPortState);
then rPortState.u8PPON contains the value for USB1 - correct?
Yes.
what is the value for s32FunRetVal in OK-case and NOK-Case?
OSAL_OK on success and OSAL_ERROR on failure case.
NOK case Error code:
OSAL_E_INVALIDVALUE ' in case of error in passed parameter
OSAL_E_UNKNOWN ' in case of internal error in getting the port state.

s32FunRetVal = OSAL_s32IOControl(hDevice, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&cmask_temp);
what is the value for s32FunRetVal in OK-case and NOK-Case?
OSAL_OK on success and OSAL_ERROR on failure case.
NOK case Error code will be set:
OSAL_E_INVALIDVALUE ' in case of error in passed parameter
OSAL_E_UNKNOWN ' in case of internal error.
OSAL_E_BUSY ' in case of active OC. Retry after OC end notification.
*/

tBool PrmManager::bInitUSBHardwareConnectors() const //Roadmap 14003 Diagnosis2  activate/deactivate USB Hardware Connectors
{
    ETG_TRACE_USR4(( "Begin: bInitUSBHardwareConnectors" ));
    tBool bRetVal = TRUE; //@todo this has to be adjusted see info for error handling above

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
    tBool bVal = TRUE ;
    if(bVal)
#else
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USR_PRM_HWCTRLUSB))
#endif

    {
         ETG_TRACE_COMP(("bInitUSBHardwareConnectors: eConf_ENABLE_USR_PRM_HWCTRLUSB = DVM_ON (ENABLED)"));

        tU32                u32ErrorCode = 0;
        tS32                s32FunRetVal = 0;
        tS32                s32PortCount = 0;
        UsbPortState        rPortState;  //@todo check if UsbPortState is defined now in OSAL in that case it is redundant code
        usb_port_control    rCmask; /*usb_port_control*/
        OSAL_tIODescriptor hOsalIODescriptorUSBPower = OSAL_IOOpen( OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READONLY );    ///@todo check if m_osalIODescriptorPRM can be used here!!!!
        //ok
        if(hOsalIODescriptorUSBPower != OSAL_ERROR)
        {
            s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT, (intptr_t)&s32PortCount);  ///@todo what does the function return for ERROR and SUCCESS
            //ok
            if(s32FunRetVal != OSAL_ERROR)
            {
                ETG_TRACE_COMP(("bInitUSBHardwareConnectors: USB-Ports available in system: %d", s32PortCount));

                //@todo check for error case here
                tU8 u8Offset;

                for(tU8 u8USBConnectorNum= 1;u8USBConnectorNum<=s32PortCount;u8USBConnectorNum++)
                {
                    u8Offset = u8USBConnectorNum -1;
                    rPortState.u8PortNr = u8USBConnectorNum;
                    s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE, (intptr_t)&rPortState);  ///@tod error case
                    //ok
                    if(s32FunRetVal != OSAL_ERROR)
                    {
                        ETG_TRACE_COMP(("bInitUSBHardwareConnectors: USB%d u8PPON=0x%x",u8USBConnectorNum,rPortState.u8PPON));

                        tU8 u8ConfValue = (tU8)eCONF_ENABLE_HW_USB1 + u8Offset;
                        DVM_NORMAL_M_ASSERT(u8ConfValue < (tU8)eCONF_LAST_ELEMENT);
                        DVM_NORMAL_M_ASSERT(u8ConfValue > (tU8)eCONF_USB_1_HUB_ENABLE /*=0*/);

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST //intermediate
                        tBool bVal = FALSE;
                        if(bVal)
#else
                        if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)u8ConfValue) )
#endif

                        {
                            if((tU8)SIG_FALSE != rPortState.u8PPON)
                            {
                                if(TRUE == bMapToMask(OUT rCmask,IN rPortState.u8PortNr,IN DVM_OFF))
                                {
                                    s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&rCmask);
                                    if(s32FunRetVal == OSAL_ERROR)
                                    {
                                        u32ErrorCode  = OSAL_u32ErrorCode();
                                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER"));
                                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                                    }
                                }
                            }
                        }
                        else
                        {
                            if((tU8)SIG_TRUE != rPortState.u8PPON)
                            {
                                if(TRUE == bMapToMask(OUT rCmask, IN rPortState.u8PortNr,IN DVM_ON))
                                {
                                    s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&rCmask);
                                    if(s32FunRetVal == OSAL_ERROR)
                                    {
                                        u32ErrorCode  = OSAL_u32ErrorCode();
                                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER"));
                                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                                    }
                                }
                            }
                        }
                    }
                    //nok
                    else
                    {
                        u32ErrorCode  = OSAL_u32ErrorCode();
                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE"));
                        ETG_TRACE_FATAL(("bInitUSBHardwareConnectors: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                    }
                }
                OSAL_s32IOClose(hOsalIODescriptorUSBPower);
            }
            //nok
            else
            {
                u32ErrorCode  = OSAL_u32ErrorCode();
                ETG_TRACE_FATAL(("bInitUSBHardwareConnectors:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT"));
                ETG_TRACE_FATAL(("bInitUSBHardwareConnectors: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
            }
        }
        //nok
        else
        {
            ETG_TRACE_FATAL(("[ERROR] bInitUSBHardwareConnectors: could not open OSAL_C_STRING_DEVICE_PRM:%s",OSAL_C_STRING_DEVICE_PRM));
            //bRetVal = FALSE;
        }

    }
    else
    {
        ETG_TRACE_COMP(("bInitUSBHardwareConnectors: eConf_ENABLE_USR_PRM_HWCTRLUSB = DVM_OFF (DISABLED)"));
    }

    ETG_TRACE_USR4(( "End  : bInitUSBHardwareConnectors" ));

    return bRetVal;
}


tBool PrmManager::bSetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,IN tBool bOn) const//Roadmap 14003 ReadUSBPowerSupplay_Activated/Deactivated
{
    ETG_TRACE_USR4(("Begin: bSetUSBHardwareConnector:"));

    ETG_TRACE_USR2(("bSetUSBHardwareConnector:IN eUSB%d, IN bOn=0x%x",eUSBConnectorNum,bOn));

    ConfigurationManager::GetInstance()->vPrintConfigurationUSBConnectorMapping();

    tBool bRetVal = TRUE; //@todo this has to be adjusted see info for error handling above


    trPortStates rPortStates;

    rPortStates.u8USBPortNo = eUSBConnectorNum;
    rPortStates.bUSBPowerON = TRUE;

    tU16  u16Index = (tU16)((tU16)eCONF_HWSIGNAL_SOURCE_USB1 +( (tU16)eUSBConnectorNum - 1) );

    if((u16Index >=eCONF_HWSIGNAL_SOURCE_USB1) && (u16Index <=eCONF_HWSIGNAL_SOURCE_USB4))
    {
        tU16 u16ConfigValue = ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)u16Index) ;    //to gain eUSB1...3//dependency decided in a phone call with USB Expert Martin Mueller (CM/ECO)

        if(eUSBConnectorNum != DVM_OFF)
        {
            eUSBConnectorNum = (tenDevManagerUSBHost)(u16ConfigValue + 1 - DVM_PRMPORT1_USED);

            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USR_PRM_HWCTRLUSB))
            {
                tU32                u32ErrorCode ;
                tS32                s32FunRetVal;
                tS32                s32PortCount = 0;
                UsbPortState        rUSBPortState;  //@todo check if UsbPortState is defined now in OSAL in that case it is redundant code
                usb_port_control    rCmask;


                OSAL_tIODescriptor  hOsalIODescriptorUSBPower = OSAL_IOOpen( OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READONLY );    ///@todo check if m_osalIODescriptorPRM can be used here!!!!
                if(hOsalIODescriptorUSBPower != OSAL_ERROR)
                {
                    s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT, (intptr_t)&s32PortCount);

                    //ok
                    if(s32FunRetVal != OSAL_ERROR)
                    {

                        ETG_TRACE_COMP(("bSetUSBHardwareConnector: USB-Ports available in system: %d", s32PortCount));

                        tBool bOk = TRUE;
                        if((s32PortCount > 0) &&((tU8)eUSBConnectorNum <= (tU8)s32PortCount) )
                        {
                            //bOk = FALSE;
                            ETG_TRACE_FATAL(("[WARNING]: bSetUSBHardwareConnector: s32PortCount:%d eUSBConnectorNum:%d",s32PortCount,eUSBConnectorNum));
                        }

                        if(bOk)
                        {
                            rUSBPortState.u8PortNr = (tU8)eUSBConnectorNum;

                            s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE, (intptr_t)&rUSBPortState);
                            if(s32FunRetVal != OSAL_ERROR)
                            {
                                ETG_TRACE_COMP(("bSetUSBHardwareConnector: USB%d u8PPON=%d",rUSBPortState.u8PortNr,ETG_ENUM(TENSIGNALVALUE,rUSBPortState.u8PPON)));
                                //-------------
                                //switch on
                                //------------
                                if(bOn == TRUE)
                                {
                                    //switch on - if not switched on yet
                                    if((tU8)SIG_TRUE != rUSBPortState.u8PPON) //if already switched off don't do this again
                                    {
                                        if(TRUE == bMapToMask(OUT rCmask, IN rUSBPortState.u8PortNr,IN DVM_ON))
                                        {
                                            //set vbus ON
                                            s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&rCmask);

                                            //nok:
                                            if(s32FunRetVal == OSAL_ERROR)
                                            {
                                                u32ErrorCode  = OSAL_u32ErrorCode();
                                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER"));
                                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                                            }
                                            //ok:
                                            else
                                            {
                                                ETG_TRACE_COMP(("-------------------OK: OSAL VBUS ON-----------------------------"));
                                                ETG_TRACE_COMP(("bSetUSBHardwareConnector: rCmask.PortMask  = %d",rCmask.PortMask));
                                                ETG_TRACE_COMP(("bSetUSBHardwareConnector: rCmask.ValueMask = %d",rCmask.ValueMask));
                                                ETG_TRACE_COMP(("----------------------------------------------------------------"));

                                            }
                                        }
                                    }
                                    else
                                    {
                                        ETG_TRACE_USR4(("bSetUSBHardwareConnector: USB%d already switched on",rUSBPortState.u8PortNr));
                                    }
                                }
                                //-------------------------
                                //switch off
                                //-------------------------
                                else
                                {
                                    //switch off - if not switched off yet
                                    if((tU8)SIG_FALSE != rUSBPortState.u8PPON) //if already switched ondon't do this again
                                    {
                                        if(TRUE == bMapToMask(OUT rCmask, IN rUSBPortState.u8PortNr,IN DVM_OFF))
                                        {
                                            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TESTMODE_SETUSBHARDWARECONNECTOR))
                                            {
                                                ETG_TRACE_FATAL(("[TESTMODE] bSetUSBHardwareConnector:switched off - will be switched on after 5sec"));
                                                sleep(1); //to reach that above trace is visible before in testmode this is switched off
                                            }
                                             //set vbus OFF
                                            s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&rCmask);
                                            if(s32FunRetVal == OSAL_ERROR)
                                            {
                                                u32ErrorCode  = OSAL_u32ErrorCode();
                                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER"));
                                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                                            }
                                            else
                                            {
                                                rPortStates.bUSBPowerON = FALSE;
                                                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TESTMODE_SETUSBHARDWARECONNECTOR))
                                                {
                                                    //-----------------------------------------------------------------------------------
                                                    //done in test only:
                                                    //suzuki has only one USBConnector if it is switched off then we don't get traces again
                                                    //psa has 3 connectors but one is
                                                    //-----------------------------------------------------------------------------------
                                                    //for test purpose switch it on again after couple of seconds otherwhise we don't have trace
                                                    ETG_TRACE_FATAL(("[TESTMODE] bSetUSBHardwareConnector:wait 5 sec  - then switched VBUS on"));
                                                    sleep(5/*sec*/);
                                                    if(TRUE == bMapToMask(OUT rCmask,IN  rUSBPortState.u8PortNr,IN  DVM_ON))
                                                    {
                                                        //set vbus ON
                                                        s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER, (intptr_t)&rCmask);

                                                        //nok
                                                        if(s32FunRetVal == OSAL_ERROR)
                                                        {
                                                            u32ErrorCode  = OSAL_u32ErrorCode();
                                                            ETG_TRACE_FATAL(("bSetUSBHardwareConnector:OSAL_ERROR==OSAL_s32IOControl(....OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER"));
                                                            ETG_TRACE_FATAL(("bSetUSBHardwareConnector: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                                                        }

                                                        else
                                                        {
                                                            ETG_TRACE_COMP(("--------------------OK:OSAL VBUS OFF ---------------------------"));
                                                            ETG_TRACE_COMP(("bSetUSBHardwareConnector: rCmask.PortMask  = %d",rCmask.PortMask));
                                                            ETG_TRACE_COMP(("bSetUSBHardwareConnector: rCmask.ValueMask = %d",rCmask.ValueMask));
                                                            ETG_TRACE_COMP(("----------------------------------------------------------------"));
                                                            rPortStates.bUSBPowerON = TRUE;
                                                        }
                                                    }
                                                    ETG_TRACE_FATAL(("[TESTMODE] bSetUSBHardwareConnector:switch on after 20sec again"));
                                                }

                                            }
                                        }
                                    }
                                    else
                                    {
                                        ETG_TRACE_USR4(("bSetUSBHardwareConnector: USB%d already switched OFF",rUSBPortState.u8PortNr));
                                    }
                                }
                            }
                            //nok
                            else
                            {
                                u32ErrorCode  = OSAL_u32ErrorCode();
                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector: OSAL_ERROR == OSAL_s32IOControl(...OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE"));
                                ETG_TRACE_FATAL(("bSetUSBHardwareConnector: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));
                            }
                        }
                        else
                        {
                            ETG_TRACE_FATAL(("[INFO]: bSetUSBHardwareConnector: eUSB%d but s32PortCount:%d ",(tU8)eUSBConnectorNum,s32PortCount));
                        }
                        OSAL_s32IOClose(hOsalIODescriptorUSBPower);
                    }
                    //nok
                    else
                    {
                        u32ErrorCode  = OSAL_u32ErrorCode();
                        ETG_TRACE_FATAL(("bSetUSBHardwareConnector: OSAL_ERROR == OSAL_s32IOControl(...OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT"));
                        ETG_TRACE_FATAL(("bSetUSBHardwareConnector: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode ) ));

                    }
                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] bInitUSBHardwareConnectors: could not open OSAL_C_STRING_DEVICE_PRM:%s",OSAL_C_STRING_DEVICE_PRM));
                    //bRetVal = FALSE;
                }
            }
            else
            {
                ETG_TRACE_FATAL(("bGetUSBHardwareConnector: eConf_ENABLE_USR_PRM_HWCTRLUSB not enabled - connection to HW not enabled"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]: eUSBConnectorNum == DVM_OFF"));


        }
    }
    else
    {
        ETG_TRACE_USR4(("bGetUSBHardwareConnector: Wrong : u16Index: %d  eCONF_HWSIGNAL_SOURCE_USB1:%d eCONF_HWSIGNAL_SOURCE_USB4:%d",
                    u16Index,(int)eCONF_HWSIGNAL_SOURCE_USB1,(int)eCONF_HWSIGNAL_SOURCE_USB4));
    }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
    //set the Power info of the port
    StateTable::GetInstance()->vSetPortPowerStateNotified(enPortPowerInfoNotified,rPortStates);
#endif//VARIANT_S_FTR_ENABLE_UNITTEST
    ETG_TRACE_USR2(("bSetUSBHardwareConnector:bRetVal=0x%x",bRetVal));
    ETG_TRACE_USR4(("End  : bGetUSBHardwareConnector"));
    return bRetVal;

}



tBool PrmManager::bGetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,OUT tenSignalValue &eAvtivated) const//Roadmap 14003 ReadUSBPowerSupply_Activated/Deactivated
{
    ETG_TRACE_USR4(("Begin: bGetUSBHardwareConnector:"));
    ETG_TRACE_USR2(("bGetUSBHardwareConnector:IN eUSB%d, OUT eAvtivated=%d",eUSBConnectorNum,ETG_ENUM(TENSIGNALVALUE,eAvtivated)));

    tBool        bRet = FALSE;


    ConfigurationManager::GetInstance()->vPrintConfigurationUSBConnectorMapping();


    tU16  u16Index = (tU16)((tU16)eCONF_HWSIGNAL_SOURCE_USB1 +( (tU16)eUSBConnectorNum - 1) );

    if((u16Index >=eCONF_HWSIGNAL_SOURCE_USB1) && (u16Index <=eCONF_HWSIGNAL_SOURCE_USB4))
    {
        tU16 u16ConfigValue = ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)u16Index) ;    //to gain eUSB1...3//dependency decided in a phone call with USB Expert Martin Mueller (CM/ECO)


        if(u16ConfigValue != DVM_OFF)
        {
            eUSBConnectorNum = (tenDevManagerUSBHost)(u16ConfigValue + 1 - DVM_PRMPORT1_USED);

            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USR_PRM_HWCTRLUSB))
            {
                tS32         s32FunRetVal;
                UsbPortState rPortState;  //@todo check if UsbPortState is defined now in OSAL in that case it is redundant code
                OSAL_tIODescriptor  hOsalIODescriptorUSBPower;

                OUT eAvtivated = SIG_UNDEF;//in error case this will be returned


                hOsalIODescriptorUSBPower = OSAL_IOOpen( OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READONLY );    ///@todo check if m_osalIODescriptorPRM can be used here!!!!
                if(hOsalIODescriptorUSBPower != OSAL_ERROR)
                {
                    rPortState.u8PortNr = (tU8)eUSBConnectorNum;

                    s32FunRetVal = OSAL_s32IOControl(hOsalIODescriptorUSBPower, OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE, (intptr_t)&rPortState);  ///@tod error case
                    if(s32FunRetVal != OSAL_ERROR)
                    {
                        ETG_TRACE_USR4(("bGetUSBHardwareConnector: rPortState.u8PortNr: %d",rPortState.u8PortNr));
                        ETG_TRACE_USR4(("bGetUSBHardwareConnector: rPortState.u8OC    : %d",rPortState.u8OC));
                        ETG_TRACE_USR4(("bGetUSBHardwareConnector: rPortState.u8UV    : %d",rPortState.u8UV));
                        ETG_TRACE_USR4(("bGetUSBHardwareConnector: rPortState.u8PPON  : %d",rPortState.u8PPON));

                        ETG_TRACE_COMP(("-------------------------------------------------------------------"));
                        ETG_TRACE_COMP(("bGetUSBHardwareConnector: USB%d u8PPON=%d",rPortState.u8PortNr,ETG_ENUM(TENSIGNALVALUE,rPortState.u8PPON)));
                        ETG_TRACE_COMP(("-------------------------------------------------------------------"));

                        if((rPortState.u8PPON == (tU8)SIG_FALSE) || (rPortState.u8PPON == (tU8)SIG_TRUE))
                        {
                            //----------
                            // success
                            //----------
                            bRet       = TRUE; //FUNCTION RETURNS:OK
                            eAvtivated = (tenSignalValue)rPortState.u8PPON;
                        }
                        else
                        {
                            ETG_TRACE_FATAL(( "[ERROR] bGetUSBHardwareConnector: rPortState.u8PPON=%d -Unexpected value",ETG_ENUM(TENSIGNALVALUE,rPortState.u8PPON)));
                        }
                    }
                    else
                    {
                        tU32 u32ErrorCode  = OSAL_u32ErrorCode();
                        ETG_TRACE_FATAL(( "[ERROR] bGetUSBHardwareConnector: OSAL_ERROR: Error Code: [0x%x], %s", u32ErrorCode , OSAL_coszErrorText(u32ErrorCode )  ));
                    }

                    OSAL_s32IOClose(hOsalIODescriptorUSBPower);
                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] bGetUSBHardwareConnector: could not open OSAL_C_STRING_DEVICE_PRM:%s",OSAL_C_STRING_DEVICE_PRM));
                }
            }
            else
            {
                eAvtivated = SIG_UNDEF;
                ETG_TRACE_FATAL(("bGetUSBHardwareConnector: eConf_ENABLE_USR_PRM_HWCTRLUSB not enabled - connection to HW not enabled"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]: eUSBConnectorNum == DVM_OFF"));
        }
    }
    else
    {
         ETG_TRACE_FATAL(("[ERROR] bGetUSBHardwareConnector: Wrong : u16Index: %d  eCONF_HWSIGNAL_SOURCE_USB1:%d eCONF_HWSIGNAL_SOURCE_USB4:%d",
                                u16Index,(int)eCONF_HWSIGNAL_SOURCE_USB1,(int)eCONF_HWSIGNAL_SOURCE_USB4));

    }
    ETG_TRACE_USR2(("bGetUSBHardwareConnector:IN eUSB%d, OUT bAvtivated=0x%x, OUT bRet=0x%x (FALSE=0x%x,TRUE=0x%x)",
                    eUSBConnectorNum,
                    ETG_ENUM(TENSIGNALVALUE,eAvtivated),bRet,FALSE,TRUE));
    ETG_TRACE_USR4(("End  : bGetUSBHardwareConnector"));
    return bRet;
}


tVoid PrmManager::SetInterface(IHUBHardwareSignalReader *pIHubHwSignalReader)
{
    m_pIHubHwSignalReader = pIHubHwSignalReader;
}

tVoid PrmManager::SetInterface(IN IStateTableHWSignals *pIStateTableHWSignals) //old interface of statetable
{
    m_pIStateTableHWSignals = pIStateTableHWSignals;
}

tVoid PrmManager::SetInterface(IN IStateTableHWMalfunction *pIStateTableHWMalfunction) //new interface of statetable
{
    m_pIStateTableHWMalfunction = pIStateTableHWMalfunction;
}


tVoid PrmManager::vFillDeviceSpeedinfo(OUT TelematicsBoxSpeedAndPortInfo f_TelematicsBoxSpeedAndPortInfo[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin  : vFillDeviceSpeedinfo"));
    //get the device speed of each of the usb port from HubHwSignalReader
    if(m_pIHubHwSignalReader)
    {
        m_pIHubHwSignalReader->vFillDeviceSpeedinfo(f_TelematicsBoxSpeedAndPortInfo);
    }
    ETG_TRACE_USR4(("End  : vFillDeviceSpeedinfo"));
}


tVoid PrmManager::vsetDeviceSpeed(IN statetbl::tenDevManagerUSBHost eUsb,IN tBool bIsDevicePresent,IN tBool b_DeviceSpeed)
{
    ETG_TRACE_USR4(("Begin  : vsetDeviceSpeed"));
    if(m_pIHubHwSignalReader)
    {
        m_pIHubHwSignalReader->vsetDeviceSpeed(eUsb,bIsDevicePresent,b_DeviceSpeed);
    }
    ETG_TRACE_USR4(("End  : vsetDeviceSpeed"));

}

tVoid PrmManager::SetMalfunctionUSBConnector(IN tBool bIsMalfunction, IN tenDevManagerUSBHost UsbConnectorNum)
{
    ETG_TRACE_USR4(("Begin  : SetMalfunctionUSBConnector"));
    if(m_pIStateTableHWMalfunction) //function added to hide this Null pointer check from calling parts in the code
    {
        m_pIStateTableHWMalfunction->SetMalfunctionUSBConnector(bIsMalfunction,UsbConnectorNum);
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: SetMalfunctionUSBConnector:m_pIStateTableHWMalfunction=NULL"));
    }
    ETG_TRACE_USR4(("End  : SetMalfunctionUSBConnector"));
}

tVoid PrmManager::SetHWSignals(UsbPortState *pPortState)
{
    ETG_TRACE_USR4(("Begin  : SetHWSignals"));
    if(m_pIStateTableHWSignals) //function added to hide this Null pointer check from calling parts in the code
    {
        m_pIStateTableHWSignals->setHWSignals(pPortState);
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: SetHWSignals:m_pIStateTableHWSignals=NULL"));
    }
    ETG_TRACE_USR4(("End  : SetHWSignals"));
}


/*-----------------------------------------------------------------------------*
 * PrmManager* GetInstance()                 *
 *-----------------------------------------------------------------------------*/
PrmManager *PrmManager::GetInstance()
{
    ETG_TRACE_USR4(("Begin: GetInstance"));

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //to disable dependency with LockDVM and thus Lock which is in framework_so. framework_so uses osal if build for gen3armmake and gen3x86make since it uses etg-trace
    PrmManager::m_singelton.lock();
#endif
    if(NULL == m_pPrmManager)
    {
        m_pPrmManager = new PrmManager();
    }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    PrmManager::m_singelton.unlock();
#endif
    ETG_TRACE_USR4(("End  : GetInstance"));

    return m_pPrmManager;
}


/*-----------------------------------------------------------------------------*
 * void DestroyInstance()                                                      *
 *-----------------------------------------------------------------------------*/
void PrmManager::DestroyInstance()
{
    ETG_TRACE_USR4(("Begin: DestroyInstance"));

    //---------/Finish Thread of ThreadFactory----------
    ETG_TRACE_USR4(("DestroyInstance: ask to stop thread"));

    //------------------------------------------------
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    PrmManager::m_singelton.lock();
#endif
    if(NULL != m_pPrmManager)
    {
        if(m_pPrmManager)
        {
            delete m_pPrmManager;
            m_pPrmManager = NULL;
        }
    }
    else
    {
        //do nothing
    }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    PrmManager::m_singelton.unlock();
#endif
    ETG_TRACE_USR4(("End  : DestroyInstance"));
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>
