/*-----------------------------------------------------------------------------*
 *PlatformCtrl.cpp                                                              *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file voltageManager.cpp
 *
 * \brief This file holds the implementation of PlatformCtrl. It hides specific
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 11.06.2014, Christian Koechling (Bosch) 
 *          -# add drv_volt to react on low or critical voltage -some projects switch off voltage supply for 
*                USB at critical voltage and not at 5Volt furthermore cd drive has to b taken into account
 *
.*
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */


#include <stdio.h>
#include "Config.h"


#include <fstream>  //for AutomounterAdapter
#include <iostream> //for AutomounterAdapter

#define INCLUDE_VD_DVM_OSAL  //statetable uses ahl thread otherwhise this 
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "PrmManager.h"
#include "VoltageManager.h"
#include "HUBHardwareSignalReader.h"
#include "AutomounterAdaptSharedLibIF.h"
#include "AutomounterAdaptTextIF.h"
#include "AutomounterAdapter.h"
#include "StorageInfoProvider.h"

#include "config/ConfigurationManager.h"

#include "PlatformCtrl.h"

/*-----------------------------------------------------------------------------*
 * namespace                                                                     *
 *-----------------------------------------------------------------------------*/
using namespace prmmgr;
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_PLATFORMCTRL
#include "trcGenProj/Header/PlatformCtrl.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * defines                                                                 *
 *-----------------------------------------------------------------------------*/


//PrmManager needs interfaces of StateTable 
PlatformCtrl::PlatformCtrl(IN IStateTableHWSignals *pIStateTableHWSignals,IN IStateTableHWMalfunction *pIStateTableHWMalfunction)
{
    m_pPrmManager               = NULL;
    m_pVoltageManager           = NULL;
    m_pHUBHardwareSignalReader  = NULL;
    m_pStorageInfoProvider      = NULL;


    m_pIStateTableHWSignals     = pIStateTableHWSignals;
    m_pIStateTableHWMalfunction = pIStateTableHWMalfunction;

    for(tInt i=0;i<enAllInterfaces;i++)
    {
        m_bInitialised[i] = FALSE;
    }

#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED
 
    m_pPrmManager = PrmManager::GetInstance();
    if(m_pPrmManager)
    {
        ETG_TRACE_USR4(("PlatformCtrl: PrmManager created"));
    }
    else
    {
       ETG_TRACE_FATAL(("[ERROR]PlatformCtrl: PrmManager not created"));
       ETG_TRACE_ERRMEM(("[ERROR]PlatformCtrl: PrmManager not created"));
       DVM_NORMAL_M_ASSERT_ALWAYS(); //even FATA_M_ASSERT_ALWAYS would be ok 
    }

    if(DVM_OFF != ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HUB_HWSIGNAL_MONITOR_INTERVAL))
    {
        m_pHUBHardwareSignalReader = new HUBHardwareSignalReader(); //<---needs to be checked 
    }
#endif

#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_USE_DRV_VOLT
    m_pVoltageManager = new VoltageManager(*m_pIStateTableHWMalfunction);
#endif

    m_pAutomounterAdapter = new AutomounterAdapter();

    m_pStorageInfoProvider = new StorageInfoProvider();

}

PlatformCtrl::~PlatformCtrl()
{
    //delete it
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED
    PrmManager::DestroyInstance();;
#endif
    delete m_pVoltageManager;
    delete m_pHUBHardwareSignalReader;
    delete m_pAutomounterAdapter;
    delete m_pStorageInfoProvider;

    m_pPrmManager              = NULL;
    m_pVoltageManager          = NULL;
    m_pHUBHardwareSignalReader = NULL;
    m_pAutomounterAdapter      = NULL;
    m_pStorageInfoProvider     = NULL;
}


/*-----------------------------------------------------------------------------*
 * int Initialize()                                                            *
 *-----------------------------------------------------------------------------*/
int PlatformCtrl::Initialize()
{
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED   //keep switches to suppress code dependency


    //---------------------------
    //HWSignalReader
    //--------------------------- 
    if(DVM_OFF != ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_HUB_HWSIGNAL_MONITOR_INTERVAL))
    {
        if(FALSE == m_bInitialised[enPrmManagerIF])
        {
            if(m_pHUBHardwareSignalReader == NULL)
            {
                ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pHUBHardwareSignalReader = NULL"));
                DVM_FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                if (m_pHUBHardwareSignalReader->Initialize() == OSAL_ERROR)
                {
                    ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pHUBHardwareSignalReader->Initialize()  == OSAL_ERROR"));
                    DVM_FATAL_M_ASSERT_ALWAYS();
                }
                else
                {
                    m_bInitialised[enHUBHardwareSignalreader]     = TRUE;
                    ETG_TRACE_COMP(("[WorkThread] Initialize: OK : m_pHUBHardwareSignalReader->Initialize() ))"));
                }
            }
        }
    }


    //---------------------------
    //PrmManager
    //--------------------------- 
    if(FALSE == m_bInitialised[enPrmManagerIF])
    {
        if(m_pPrmManager == NULL)
        {
            ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pPrmManager = NULL"));
            DVM_FATAL_M_ASSERT_ALWAYS();
        }
        else
        {
            //Inject Interface
            ETG_TRACE_COMP(("[WorkThread] Inject Interface : IHUBHardwareSignalReader to PrmManager"));
            m_pPrmManager->SetInterface((IHUBHardwareSignalReader*)m_pHUBHardwareSignalReader);
            ETG_TRACE_COMP(("[WorkThread] Inject Interface : IStateTableHWSignals     to PrmManager"));
            m_pPrmManager->SetInterface(m_pIStateTableHWSignals);
            ETG_TRACE_COMP(("[WorkThread] Inject Interface : IStateTableHWMalfunction to PrmManager"));
            m_pPrmManager->SetInterface(m_pIStateTableHWMalfunction);

            //initialize PrmManager
            if (m_pPrmManager->Initialize() == OSAL_ERROR)
            {
                ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pPrmManager->Initialize()  == OSAL_ERROR"));
                DVM_FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                m_bInitialised[enPrmManagerIF]     = TRUE;
                m_bInitialised[enPrmManagerCardIF] = TRUE;
                ETG_TRACE_COMP(("[WorkThread] Initialize: OK : m_pPrmManager->Initialize() ))"));
            }
        }
    }
#endif //VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED

#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_USE_DRV_VOLT //keep switches to suppress code dependency 
    //---------------------------
    //VoltageManager
    //---------------------------
    if(FALSE == m_bInitialised[enVoltageManagerIF])
    {
         if(m_pVoltageManager == NULL)
        {
            ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pVoltageManager = NULL"));
            DVM_FATAL_M_ASSERT_ALWAYS();
        }
        else
        {
            if (m_pVoltageManager->Initialize() == OSAL_ERROR)
            {
                ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pVoltageManager->Initialize() == OSAL_ERROR"));
                ETG_TRACE_ERRMEM(("ERROR: Initialize: ERROR m_pVoltageManager->Initialize() == OSAL_ERROR"));
                DVM_FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                m_bInitialised[enVoltageManagerIF] = TRUE;
                ETG_TRACE_COMP(("[WorkThread] Initialize: OK : m_pVoltageManager->Initialize()"));
            }
        }
    }
#endif //VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED
    //---------------------------
    //AutomounterAdapter
    //---------------------------
    if(FALSE == m_bInitialised[enAutomounterAdapterIF])
    {
        if(m_pAutomounterAdapter == NULL)
        {
            ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pAutomounterAdapter = NULL"));
            FATAL_M_ASSERT_ALWAYS();
        }
        else
        {
            if (m_pAutomounterAdapter->Initialize() == OSAL_ERROR)
            {
                ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pAutomounterAdapter->Initialize() == OSAL_ERROR"));
                FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                m_bInitialised[enAutomounterAdapterIF] = TRUE;
                ETG_TRACE_COMP(("[WorkThread] Initialize: OK : m_pAutomounterAdapter->Initialize()"));
            }
        }
    }

    //---------------------------
    //StorageInfoProvider
    //---------------------------
    if(FALSE == m_bInitialised[enStorageInfoProviderIF])
    {
        if(m_pStorageInfoProvider == NULL)
        {
            ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pStorageInfoProvider = NULL"));
            FATAL_M_ASSERT_ALWAYS();
        }
        else
        {
            if (m_pStorageInfoProvider->Initialize() == OSAL_ERROR)
            {
                ETG_TRACE_FATAL(("ERROR: Initialize: ERROR m_pStorageInfoProvider->Initialize() == OSAL_ERROR"));
                FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                m_bInitialised[enStorageInfoProviderIF] = TRUE;
                ETG_TRACE_COMP(("[WorkThread] Initialize: OK : m_pStorageInfoProvider->Initialize()"));
            }
        }
    }

    return OSAL_OK; //it won't get here due to the FATAL_M_ASSERTS above if something goes wrong
}

/*-----------------------------------------------------------------------------*
 * IPrmManager GetInterfacePrmManager(...)                                                            *
 *-----------------------------------------------------------------------------*/
IPrmManager  *PlatformCtrl::GetInterfacePrmManager()
{
    IPrmManager *pIPrmManager = NULL;
    if(TRUE == m_bInitialised[enPrmManagerIF])
    {
         pIPrmManager = (IPrmManager*)m_pPrmManager;
    }
    return pIPrmManager;
}

/*-----------------------------------------------------------------------------*
 * IPrmManagerCard GetInterfacePrmManagerCard(...)                                                            *
 *-----------------------------------------------------------------------------*/
IPrmManagerCard  *PlatformCtrl::GetInterfacePrmManagerCard()
{
    IPrmManagerCard *pIPrmManagerCard = (IPrmManagerCard*)m_pPrmManager;
   
    return pIPrmManagerCard;
}

/*-----------------------------------------------------------------------------*
 * IPrmManager GetInterfacePrmManager(...)                                                            *
 *-----------------------------------------------------------------------------*/
IVoltageManager  *PlatformCtrl::GetInterfaceVoltageManager()
{
    IVoltageManager *pIVoltageManager = NULL;
    if(TRUE == m_bInitialised[enVoltageManagerIF])
    {
         pIVoltageManager = (IVoltageManager*)m_pVoltageManager;
    }
    return pIVoltageManager;
}

/*-----------------------------------------------------------------------------*
 * IAutomounterAdapter  GetInterfaceAutomounterAdapter(...)                                                            *
 *-----------------------------------------------------------------------------*/
IAutomounterAdapter  *PlatformCtrl::GetInterfaceAutomounterAdapter()
{
    IAutomounterAdapter *pIAutomounterAdapter = NULL;
    if(TRUE == m_bInitialised[enAutomounterAdapterIF])
    {
         pIAutomounterAdapter = (IAutomounterAdapter*)m_pAutomounterAdapter;
    }
    return pIAutomounterAdapter;
}

/*-----------------------------------------------------------------------------*
 * IStorageInfoProvider  *GetInterfaceStorageInfoProvider(...)                                                            *
 *-----------------------------------------------------------------------------*/

IStorageInfoProvider  *PlatformCtrl::GetInterfaceStorageInfoProvider()
{
    IStorageInfoProvider *l_pStorageInfoProvider = NULL;
    if(TRUE == m_bInitialised[enStorageInfoProviderIF])
    {
        l_pStorageInfoProvider = (IStorageInfoProvider*)m_pStorageInfoProvider;
    }
    return l_pStorageInfoProvider;

}

////////////////////////////////////////////////////////////////////////////////
// <EOF>
