/*-----------------------------------------------------------------------------*
 * AutomounterAdapter.h                                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file AutomounterAdapter.h
 *
 * \brief new automounter provides a slightly different interface like the previously used mount.sh
 *         in /tmp/.automount a new structure for information can be found
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 * \version Initial Version 03.12.2013, Koechling, Christian (Bosch)
 * \version For Roadmap 1615 Internal DVD-Drive added 'extracting unsupported reason' 
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2013-2020
 */

#ifndef __AUTOMOUNTERADAPTER_TEXTIF_H__
#define __AUTOMOUNTERADAPTER_TEXTIF_H__

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"
#include "IAutomounterAdapter.h" //for some typedefinitions

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define AUTOMOUNTER_LENLINE 1024
#define FILE_HAS_NOENTRIES 0
#define NUMOFARRAYENTRIES AutomounterAdaptTextIF::EN_PART_ENTRY_INVALID
#define PARTITIONDEC 2 //means 0..99
#define PARTITIONDECMAX 99 //depends on PARTITIONDEC





#define CDROM__INTERNAL_BLOCK_DEVNAME_PART "/dev/sr"
#define SDCARD_INTERNAL_BLOCK_DEVNAME_PART "/dev/mmcblk"  
#define USB_SD_EXTERNAL_BLOCK_DEVNAME_PART "/dev/sd"      //no distinction beween sd card external and usb-stick possible at this point

// Roadmap 1615 Internal DVD-Drive 
#define PARTITION_UNSUPPORTED_REASON__SUPPORTED              "_SUPPORTED"   
#define PARTITION_UNSUPPORTED_REASON__UNSUPPORTED_FILESYSTEM "UNSUPPORTED_FILESYSTEM"
#define PARTITION_UNSUPPORTED_REASON__NO_FILESYSTEM          "NO_FILESYSTEM"
#define PARTITION_UNSUPPORTED_REASON__AUDIO_CD               "AUDIO_CD"
#define PARTITION_UNSUPPORTED_REASON__AUDIO_MIXED_MODE       "AUDIO_MIXED_MODE"
#define PARTITION_UNSUPPORTED_REASON__BLACKLISTED            "BLACKLISTED"


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace std;

/*-----------------------------------------------------------------------------*
 * Forward declaration                                                                  *
 *-----------------------------------------------------------------------------*/
class device_info_t;
class partition_info_t;

//class CDevice;
/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class CDevice
 *
 * \brief class to hold the properties of the device
 */

//Example output of new automounter interface
/*
/tmp/.automount/dev-sda.info
/tmp/.automount/dev-cr0.info
/tmp/.automount/dev-sda/dev-sda1.info
/tmp/.automount/dev-sda/dev-sda2.info
/tmp/.automount/dev-sda/dev-cr0.info
/tmp/.automount/mount_points/dev-sda2.info -> ../dev-sda/dev-sda2.info

dev-sda.info:
-----------------
INTERFACE_ID=/dev/sda
IDENTIFIER=KINGSTON_ad4fd-7z78s-asdkd-www99
TYPE=USB_MASS_STORAGE
PART_CNT=2
MEDIA=NOT_REMOVABLE

dev-cr0.info:
-----------------
INTERFACE_ID=/dev/cr0
IDENTIFIER=CDR_aewr-734es-assad-w9324
TYPE=CDROM
PART_CNT=1
MEDIA=REMOVABLE_PRESENT

dev-sda1.info:
-----------------
INTERFACE_ID=/dev/sda1
MOUNT_POINT=
IDENTIFIER=EXT3_ad4fd-7z78s-asada-ww99s
STATUS=UNSUPPORTED_FS
PART_NO=1
FS_TYPE=ext3

dev-sda2.info:
-----------------
INTERFACE_ID=/dev/sda2
MOUNT_POINT=/media/ADIT_random_x2xxse2_sda2 (:)
IDENTIFIER=VFAT_assdw-7e34s-as23a-9s342
STATUS=MOUNTED_RO
PART_NO=2
FS_TYPE=vfat

dev-cr0.info:
-----------------
INTERFACE_ID=/dev/cr0
MOUNT_POINT=/media/ADIT_random_se231se2_cr0 (:)
IDENTIFIER= CDR_aewr-734es-assad-w9324
STATUS=UNMOUNTED
PART_NO=1
FS_TYPE=iso9660

*/




class AutomounterAdaptTextIF
{

public:


    typedef enum tenResAutomounterAdapt_E{   //add to class without namespace helps to let ETG_CENNUM work
        enRES_OK                   = 0,
        enRES_ERROR_FILE_          = 1,
        enRES_ERROR_FILE_EMPTY     = 2,
        enRES_ERROR_FILE           = 3,
        enRES_ERROR_EMPTY_LINE     = 4,
        enRES_ERROR_INPUT_STRING_LEN   = 5,
        enRES_ERROR_INPUT_STRING_EMTPY = 6,
        enRES_ERROR_GENERAL            = 7
    }tenResAutomounterAdapt_Type;


    typedef enum tenParmAutomounterAdaptFile_Type_E{
        EN_TYP_DEVICE    = 0,
        EN_TYP_PARTITION = 1
    }tenParmAutomounterAdaptFile_Type;


    ///@todo check to hide these enums from public!!!
    
    //if a device is attached:
    //automounter creates e.g entry for device in /tmp/.automounter/device_db/device_dev_sdb there a file  partition_dev_sdb1.info appears
    //a softlink of partition_dev_sdb1.info appears can be found in  /tmp/.automounter/mounted/_partitions_db
    ///tmp/.automounter/device_db/device_dev_sdb.info holds info about the device

    enum tenEntryDevice{               //Example of entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
        EN_DEV_INTERFACE_ID         = 0, //e.g INTERFACE_ID=/dev/sdb
        EN_DEV_IDENTIFIER           = 1, //IDENTIFIER=TOSHIBA_TransMemory_001D92AD6768C030A396072A-0:0
        EN_DEV_STATE                = 2, //STATE=DEVICE_AUTOMOUNTED
        EN_DEV_PARTITION_CNT        = 3, //PARTITION_CNT=1
        EN_DEV_TYPE                 = 4, //TYPE=disk
        EN_DEV_AM_DEVICE_HANDLER    = 5, //AM_DEVICE_HANDLER=GENERAL_MSD
        EN_DEV_ENTRY_INVALID        = 6
    };
    
    //Note: EN_PART_INTERFACE_ID is with 6 one after EN_DEV_AM_DEVICE_HANDLER - that's on purpose
    
    enum tenEntryPartition{               //Example of entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
        EN_PART_INTERFACE_ID           = 7, //e.g INTERFACE_ID=/dev/sdb1
        EN_PART_IDENTIFIER             = 8, //IDENTIFIER=3494-040F
        EN_PART_STATE                  = 9, //STATE=PARTITION_MOUNTED
        EN_PART_MOUNT_POINT            = 10, //MOUNT_POINT=/media/AAM_dev_sdb1
        EN_PART_MOUNT_SRC              = 11, //MOUNT_SRC=/dev/sdb1
        EN_PART_FILESYSTEM             = 12, //FILESYSTEM=vfat
        EN_PART_PARTITION_NO           = 13, //PARTITION_NO=1
        EM_PART_UNSUPPORTED_REASON     = 14, //UNSUPPORTED_REASON=
        EN_PART_MOUNTED_WRITABLE       = 15, //MOUNTED_WRITABLE=0
        EN_PART_ENTRY_INVALID          = 16
    };


    //Example of entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
    enum tenFileStatus{
        EN_FILE_NOT_OPEN       = 0,
        EN_FILE_EOF            = 1,
        EN_FILE_OK             = 2,
        EN_FILE_ERROR          = 3
    };

    
    //Example of entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
    enum tenCalcDevice{
        EN_CLCDEV_OK                   = 0,  //  e.g. /dev/sda12 resulting in /dev/sda  partition number is 12
        EN_CLCDEV_NONUMBER             = 1,  // e.g /dev/sda  resulting in /dev/sda partition number is 0
        EN_CLCDEV_ERROR_PARTNUMTOHIGH  = 2,
        EN_CLCDEV_ERROR                = 3   // error in calculation
    };


    /*!
         * \brief Constructor
         */
    AutomounterAdaptTextIF();

    /*!
         * \brief Destructor
         */
    ~AutomounterAdaptTextIF();


    AutomounterAdaptTextIF::tenResAutomounterAdapt_Type FillDevice(OUT CDevice& device, IN  std::string strPathfilename);
    AutomounterAdaptTextIF::tenResAutomounterAdapt_Type FillDevice(OUT CDevice& device, IN  std::string strPathfilename, tenParmAutomounterAdaptFile_Type enFileType);

    /*!
     * \Get the state of device mount state by parsing the device db file.
     * \param [in] strBlockDevName block device name of the device such as /dev/sda or /dev/sdcc
     * \param [in] strPartitionNode partition node of device  such as /dev/sda1 or /dev/sdcc1
     * \param [out] True if mount is successful
     */
    tBool GetDeviceMountState(IN std::string strBlockDevName,IN std::string strPartitionNode);


    // EXAMPLE:
    // IN    strAutomounterPartitionFileName: partition_dev_sda1.info
    // OUT strPathFileNameDevDB              : /dev/sda1
    AutomounterAdaptTextIF::tenResAutomounterAdapt_Type GetExtractBlockDevName(OUT std::string &strDevName, IN std::string strAutomounterPartitionFileName) const;

protected:

    ///@todo add comments!!!
    void GetNextLineOfFile(OUT std::string &f_strLine,OUT AutomounterAdaptTextIF::tenFileStatus &eFileStatus, IN std::ifstream &f_inFile) const;
    void GetTypeOfEntryDevice(OUT AutomounterAdaptTextIF::tenEntryDevice &enTypeOfEntryDevice ,OUT std::string &strEntrySubString,IN const std::string &strLine) const;
    void GetTypeOfEntryPartition(OUT AutomounterAdaptTextIF::tenEntryPartition &enTypeOfEntryPartition ,OUT std::string &strEntrySubString,IN const std::string &strLine) const;
    bool SetDeviceEntry(OUT  CDevice &device,IN tUInt iTypeOfEntry,IN const std::string &strEntrySubString) const;
    
    AutomounterAdaptTextIF::tenCalcDevice CalcDeviceDbNameOfDevice(OUT std::string &strPathFileNameDevDB,IN std::string strBlockDevName /*e.g. /dev/sdb1,2,3... or /dev/sda1,2,3..*/,IN tU8 u8PartitionNum) const;  ///@todo care for const


    AutomounterAdaptTextIF::tenResAutomounterAdapt_Type subfctFillDevicePartitionInfo(OUT CDevice& device,std::ifstream &ifStreamFile, tenParmAutomounterAdaptFile_Type enFileType) const;

    void GetDeviceTypeFromMountSrc(OUT  CDevice &device, IN std::string strBlockDevName /*e.g. /dev/sdb1,2,3... or /dev/sda1,2,3..*/) const;
	void GetUnsupportedReasonType(OUT  CDevice &device, IN std::string strUnsupportedReason) const;



private:
    std::ifstream  m_inFile;
    std::string    m_ArrayOfEntries[NUMOFARRAYENTRIES];

    int   m_FileDescrAutomounter;

    void subfctGetTypeOfEntry(OUT tUInt &iTypeOfEntry ,OUT std::string &strEntrySubString,IN const std::string &strLine,IN tenParmAutomounterAdaptFile_Type enType) const;


};



#endif // __AUTOMOUNTERADAPTER_TEXTIF_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>

