//@todo: This cold has access to text interface of automounter and shared lib interface
// Recommendation is to put different behaviour into its own class. Decorator pattern is used to give either or bheaviour to the
// VD_DVM.

/*-----------------------------------------------------------------------------*
 * AutomounterAdaptTextIF.cpp                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM Gen3                                                   *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file AutomounterAdaptTextIF.cpp
 *
 * helper class to connect VD_Devicemanager to new automounter mechanism of Adit


 * \version Initial Version
 * \version 03.12.2013, Koechling, Christian (Bosch)
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include <stdexcept>
#include <string>
#include <unistd.h>
#include <iostream>
#include <fstream>
#include <unistd.h>

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"

#include "AutomounterAdaptTextIF.h"
#include "config/ConfigurationManager.h"

#include <signal.h>
#include <sys/epoll.h>
#include <errno.h>
#include <poll.h>
//#include <sys/inotify.h>
//#include <unistd.h>


/*-----------------------------------------------------------------------------*
* ETG Tracing                                                                 *
*-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/AutomounterAdaptTextIF.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

const int cDevDirStrLen = 5;//strlen("/dev/")



/*-----------------------------------------------------------------------------*
* name spaces                                                                 *
*-----------------------------------------------------------------------------*/
//using namespace AutomounterAdaptTextIF;


/*-----------------------------------------------------------------------------*
* static variables of class                                                                 *
*-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
AutomounterAdaptTextIF::AutomounterAdaptTextIF()
{
    //Init DeviceInfo  //Example of key entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_INTERFACE_ID]      = "INTERFACE_ID=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_IDENTIFIER]        = "IDENTIFIER=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_STATE]             = "STATE=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_PARTITION_CNT]     = "PARTITION_CNT=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_TYPE]              = "TYPE=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_AM_DEVICE_HANDLER] = "AM_DEVICE_HANDLER=";
    //Init PartitionInfo //Example of key entries in /tmp/.automounter/mounted/_partitions_db/ parttition_dev_sdb1.info
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_INTERFACE_ID]     = "INTERFACE_ID=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_IDENTIFIER]       = "IDENTIFIER=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_STATE]            = "STATE=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_MOUNT_POINT]      = "MOUNT_POINT=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_MOUNT_SRC]        = "MOUNT_SRC=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_FILESYSTEM]       = "FILESYSTEM=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_PARTITION_NO]     = "PARTITION_NO=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EM_PART_UNSUPPORTED_REASON] = "UNSUPPORTED_REASON=";
    m_ArrayOfEntries[AutomounterAdaptTextIF::EN_PART_MOUNTED_WRITABLE] = "MOUNTED_WRITABLE=";


    m_FileDescrAutomounter = -1;
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                 *
 *-----------------------------------------------------------------------------*/
AutomounterAdaptTextIF::~AutomounterAdaptTextIF()
{
}

/*-----------------------------------------------------------------------------*
  * FillDevice  - Alternative based on shared lib callbacks                                                               *
  *-----------------------------------------------------------------------------*/
AutomounterAdaptTextIF::tenResAutomounterAdapt_Type AutomounterAdaptTextIF::FillDevice(OUT CDevice& device, IN  std::string strPathfilename)
{
    tenResAutomounterAdapt_Type enResult =  enRES_ERROR_GENERAL;

    //------------------------------------------------------------------
    //looks e.g. at tmp/.automounter/mounted_partitions_db/partition_dev_sda1.info
    //------------------------------------------------------------------
    ETG_TRACE_USR4(("EN_TYP_PARTITION: strPathfilename: %s\n",strPathfilename.c_str()));
    enResult = FillDevice(OUT device, IN  strPathfilename,  AutomounterAdaptTextIF::EN_TYP_PARTITION);

    if(enResult == AutomounterAdaptTextIF::enRES_OK)
    {
        std::string strPathFileNameDevDB;
        std::string strBlockDevnode       = device.m_cBlockDevnode.toStdString().c_str();

        CalcDeviceDbNameOfDevice(OUT strPathFileNameDevDB,IN strBlockDevnode,IN (tU8)device.m_iValidPartitionNo); ///@todo care for ERROR case here!!!!!

        ETG_TRACE_USR4(("FillDevice:strBlockDevnode              :%s",strBlockDevnode.c_str()));
        ETG_TRACE_USR4(("FillDevice:m_iValidPartitionNo          :%d",device.m_iValidPartitionNo));
        ETG_TRACE_USR4(("FillDevice:strPathFileNameDevDB.c_str():%s\n",strPathFileNameDevDB.c_str()));

        ETG_TRACE_USR4(("EN_TYP_DEVICE: strPathFileNameDevDB: %s\n",strPathFileNameDevDB.c_str()));
        //---------------------------------------------------------
        //looks e.g. at /tmp/.automounter/device_db/device_dev_sda.info
        //---------------------------------------------------------
        enResult = FillDevice(OUT device, IN  strPathFileNameDevDB,  AutomounterAdaptTextIF::EN_TYP_DEVICE);
    }
    //---------------------------------------------
    //some entries are a combination of existing entries
    //------------------------------------------
    if(enResult == AutomounterAdaptTextIF::enRES_OK)
    {
        //e.g. USB SerialID plus UUID of partition
        device.m_cSerialID = device.m_cSerialID + GENSTRING("_") + device.m_cShortSerial;
    }

    ETG_TRACE_COMP(("FillDevice: enResult:%d", ETG_CENUM(AutomounterAdaptTextIF::tenResAutomounterAdapt_Type, enResult)))
            return enResult;
}


AutomounterAdaptTextIF::tenResAutomounterAdapt_Type AutomounterAdaptTextIF::FillDevice(OUT CDevice& device, IN  std::string strPathfilename, tenParmAutomounterAdaptFile_Type enFileType)
{
    ETG_TRACE_USR4(("Begin: FillDevice"));

    tenResAutomounterAdapt_Type enResult =  enRES_ERROR_GENERAL;

    ETG_TRACE_USR1(("FillDevice: strPathfilenamed: %s",strPathfilename.data()))
            m_inFile.open(strPathfilename.data());

    //Partition file entry
    if (m_inFile.is_open())
    {
        ETG_TRACE_USR1(("FillDevice: FILE open: %s",strPathfilename.c_str()));
        enResult = subfctFillDevicePartitionInfo(device,m_inFile, enFileType);
        m_inFile.close();
    }
    else
    {
        ETG_TRACE_USR1(("FillDevice: FILE empty: %s",strPathfilename.c_str()));
        enResult = enRES_ERROR_FILE_EMPTY;
    }

    ETG_TRACE_USR4(("End  : FillDevice"));
    return enResult;

}






AutomounterAdaptTextIF::tenResAutomounterAdapt_Type AutomounterAdaptTextIF::GetExtractBlockDevName(OUT std::string &strDevName, IN std::string strAutomounterPartitionFileName) const
{
    tenResAutomounterAdapt_Type enResult =  enRES_ERROR_INPUT_STRING_EMTPY;

    size_t len,lenPayload = 0;
    size_t offset    = (size_t)AUTOMOUNTER_NAMEOFFSET;
    size_t chopAtEnd = (size_t)DIR_AUTOMOUNTER_SUFFIX_LEN;

    if(!strAutomounterPartitionFileName.empty())
    {
        len = strAutomounterPartitionFileName.size();
        if(len > (offset+chopAtEnd))
        {
            lenPayload = (len - (offset + chopAtEnd));
            strDevName.append( strAutomounterPartitionFileName.substr(offset,lenPayload) );
            enResult = enRES_OK;
        }
        else
        {
            OUT strDevName.clear();
            enResult = enRES_ERROR_INPUT_STRING_LEN;
        }
    }
    else
    {
        OUT strDevName.clear();
    }

    return enResult;
}


AutomounterAdaptTextIF::tenCalcDevice AutomounterAdaptTextIF::CalcDeviceDbNameOfDevice(OUT std::string &strPathFileNameDevDB,
                                                                                       IN std::string strBlockDevName, /*e.g. /dev/sdb1,2,3... or /dev/sda1,2,3.. or /dev/sr0 */
                                                                                       IN tU8         u8PartitionNum) const
{

    AutomounterAdaptTextIF::tenCalcDevice enRetVal = AutomounterAdaptTextIF::EN_CLCDEV_ERROR;

    bool bFoundNum = false;
    tU8  u8LenOfString;

    std::string str_dec;
    std::string strBlockDevNameWithoutSlashDev;

    char cPartitionNumber[PARTITIONDEC+1];

    //printf("strBlockDevName: %s\n",strBlockDevName.c_str());


    // for /dev/sr[0-9] i.e. internal CDROM
    if(true == ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_CDROM,strBlockDevName))
    {
        bFoundNum = true;
        //NCG3D-144806: Block device name for CDROM is not constant(/dev/sr[0-9]), hence use the strBlockDevName input parameter
        //printf("DTY_CDROM");
    }
    //for /dev/mmcblk0p1 i.e. internal SDCard
    else if(true == ConfigurationManager::GetInstance()->bCheckHWSpecificBlockDevName(CGlobalEnumerations::DTY_SD_INTERNAL,strBlockDevName))
    {
        bFoundNum = true;
        ConfigurationManager::GetInstance()->vGetSpecificBlockDevName(IN CGlobalEnumerations::DTY_SD_INTERNAL, OUT strBlockDevName); //returns e.g. mmcbkl0
        //printf("DTY_SD_INTERNAL");
    }
    //for  SDCard@ExternalHUB and USB-Stick
    else
    {
        if(u8PartitionNum == 0)
        {
            //strBlockDevName = strBlockDevName; //e.g. /dev/sda or /dev/sdb have partition number 0
            bFoundNum = true;
            //-------------------------------------------------
            enRetVal = AutomounterAdaptTextIF::EN_CLCDEV_NONUMBER;
            //-------------------------------------------------
        }
        else if(u8PartitionNum <=PARTITIONDECMAX)
        {
            snprintf(cPartitionNumber, sizeof(cPartitionNumber),"%d",u8PartitionNum);
            //printf("CalcDeviceDbNameOfDevice u8PartitionNum:%d\n",u8PartitionNum);
            //printf("CalcDeviceDbNameOfDevice cPartitionNumber:%s\n",cPartitionNumber);

            str_dec = cPartitionNumber;
            //printf("CalcDeviceDbNameOfDevice %s\n",str_dec.c_str());

            u8LenOfString = (tU8)str_dec.size();
            //Extract substring at the End
            str_dec = strBlockDevName.substr(strBlockDevName.size()- u8LenOfString,strBlockDevName.size());
            //printf("CalcDeviceDbNameOfDevice: %s\n",str_dec.c_str());

            //ending contains correct number
            if(0 == str_dec.compare(cPartitionNumber))
            {
                bFoundNum = true;
                strBlockDevName = strBlockDevName.substr(0,strBlockDevName.size()-u8LenOfString); //e.g. instead of /dev/sda12 we should get /dev/sda here
                //printf("CalcDeviceDbNameOfDevice: strBlockDevName:%s\n",strBlockDevName.c_str());
                //-------------------------------------------
                enRetVal = AutomounterAdaptTextIF::EN_CLCDEV_OK;
                //-------------------------------------------
            }
        }
        else
        {
            //------------------------------------------------------------
            enRetVal = AutomounterAdaptTextIF::EN_CLCDEV_ERROR_PARTNUMTOHIGH;
            //------------------------------------------------------------
        }


    }

    //compare to partition number received as parameter - should be the same
    if(bFoundNum == true)
    {
        strPathFileNameDevDB  = DIR_AUTOMOUNTER_MOUNTEDDEVICE_DB_FILENAMEBASE;
        strBlockDevNameWithoutSlashDev = strBlockDevName.substr(5,strBlockDevName.size());       //extract e.g. /dev/sda extract /dev/ i.e. first 5 elements
        strPathFileNameDevDB += strBlockDevNameWithoutSlashDev;
        strPathFileNameDevDB += DIR_AUTOMOUNTER_SUFFIX;
        //printf("CalcDeviceDbNameOfDevice: %s\n", strPathFileNameDevDB.c_str());
    }

    return enRetVal;

}


AutomounterAdaptTextIF::tenResAutomounterAdapt_Type AutomounterAdaptTextIF::subfctFillDevicePartitionInfo(OUT CDevice& device,std::ifstream &ifStreamFile, tenParmAutomounterAdaptFile_Type enFileType) const
{
    ETG_TRACE_USR4(("Begin: subfctFillDevicePartitionInfo: enFileType:%d",ETG_CENUM(AutomounterAdaptTextIF::tenParmAutomounterAdaptFile_Type, enFileType)));

    tenResAutomounterAdapt_Type enResult =  enRES_ERROR_GENERAL;
    AutomounterAdaptTextIF::tenEntryDevice    enTypeOfEntryDevice;
    AutomounterAdaptTextIF::tenEntryPartition enTypeOfEntryPartition;


    tUInt numOfLines = 0;
    AutomounterAdaptTextIF::tenFileStatus eFileStatus;
    std::string                       strLineOfAutomounter,strEntrySubString;


    eFileStatus =  AutomounterAdaptTextIF::EN_FILE_OK;
    while(eFileStatus == AutomounterAdaptTextIF::EN_FILE_OK)
    {
        GetNextLineOfFile(OUT strLineOfAutomounter,OUT eFileStatus, IN ifStreamFile);
        //ok
        if(eFileStatus == AutomounterAdaptTextIF::EN_FILE_OK)
        {
            switch(enFileType)
            {
            //----------------------------------------------
            case EN_TYP_PARTITION:
                //----------------------------------------------
            {
                GetTypeOfEntryPartition(OUT enTypeOfEntryPartition ,OUT strEntrySubString,IN strLineOfAutomounter);
                if(enTypeOfEntryPartition != AutomounterAdaptTextIF::EN_PART_ENTRY_INVALID)
                {
                    //nok
                    if(false == SetDeviceEntry(OUT device,IN (tUInt)enTypeOfEntryPartition,IN strEntrySubString))
                    {
                        ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: SetDeviceEntry returned false"));
                    }
                    //ok
                    else
                    {
                        enResult = enRES_OK;
                        numOfLines++;
                    }

                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: GetTypeOfEntry returned EN_ENTRY_INVALID: line ignored is: %s",strLineOfAutomounter.data()));
                }
            }
                break;
                //-----------------------------------------------
            case EN_TYP_DEVICE:
                //-----------------------------------------------
            {
                GetTypeOfEntryDevice(OUT enTypeOfEntryDevice ,OUT strEntrySubString,IN strLineOfAutomounter);
                if(enTypeOfEntryDevice != AutomounterAdaptTextIF::EN_DEV_ENTRY_INVALID)
                {
                    //nok
                    if(false == SetDeviceEntry(OUT device,IN (tUInt)enTypeOfEntryDevice,IN strEntrySubString))
                    {
                        ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: SetDeviceEntry returned false"));
                    }
                    //ok
                    else
                    {
                        enResult = enRES_OK;
                        numOfLines++;
                    }

                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: GetTypeOfEntry returned EN_ENTRY_INVALID: line ignored is: %s",strLineOfAutomounter.data()));
                }
            }
                break;
                //--------------------------------------------------
            default:
                //-------------------------------------------------
                ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: not expected: enFileType: %d",enFileType));
                NORMAL_M_ASSERT_ALWAYS();
                break;

            }

        }
        else if(eFileStatus == AutomounterAdaptTextIF::EN_FILE_EOF)
        {
            if(numOfLines == FILE_HAS_NOENTRIES)
            {
                enResult = enRES_ERROR_FILE_EMPTY;
                ETG_TRACE_FATAL(("[ERROR]: AutomounterAdaptTextIF::FillDevice: EN_ERROR_FILE_EMPTY "));
            }

        }
        //nok
        else if(eFileStatus == AutomounterAdaptTextIF::EN_FILE_ERROR)
        {
            enResult = enRES_ERROR_FILE;

            ETG_TRACE_USR4(("AutomounterAdaptTextIF::FillDevice: EN_FILE_ERROR"));
        }
        else
        {
            NORMAL_M_ASSERT_ALWAYS();
            ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptTextIF::FillDevice: SHOULD NEWVER BE EXECUTED"));
        }


    }

    return enResult;

}

/*-----------------------------------------------------------------------------*
  * void GetLineOfFile                                                                 *
  *-----------------------------------------------------------------------------*/
void  AutomounterAdaptTextIF::GetNextLineOfFile(OUT std::string &f_strLine,OUT AutomounterAdaptTextIF::tenFileStatus &eFileStatus, IN std::ifstream &f_inFile) const
{
    char cLine[AUTOMOUNTER_LENLINE];

    if(f_inFile.is_open())
    {
        eFileStatus = AutomounterAdaptTextIF::EN_FILE_OK;
        f_inFile.getline(cLine,AUTOMOUNTER_LENLINE);
        //printf("GetNextLineOfFile :%s\n",cLine);

        f_strLine = cLine;
        if(f_inFile.eof())
        {
            eFileStatus = AutomounterAdaptTextIF::EN_FILE_EOF;
        }
        else
        {
            if(f_inFile.good())
            {
                eFileStatus = AutomounterAdaptTextIF::EN_FILE_OK;
            }
            else
            {
                eFileStatus = AutomounterAdaptTextIF::EN_FILE_ERROR;
            }
        }
    }
    else
    {
        eFileStatus = AutomounterAdaptTextIF::EN_FILE_NOT_OPEN;
    }
}

/*-----------------------------------------------------------------------------*
* void GetTypeOfEntry                                                                 *
*-----------------------------------------------------------------------------*/
void AutomounterAdaptTextIF::subfctGetTypeOfEntry(OUT tUInt &uTypeOfEntry ,OUT std::string &strEntrySubString,IN const std::string &strLine,IN tenParmAutomounterAdaptFile_Type enType) const
{
    tUInt uiBegin, uiLast;

    std::size_t found, lenOfKey, lenOfEntry;
    std::string stringTofind;

    //prepare dedicate search
    switch(enType)
    {
    case EN_TYP_DEVICE:
        uiBegin = (tUInt) AutomounterAdaptTextIF::EN_DEV_INTERFACE_ID;
        uiLast  = (tUInt) AutomounterAdaptTextIF::EN_DEV_ENTRY_INVALID;
        break;
    case EN_TYP_PARTITION:
        uiBegin = (tUInt) AutomounterAdaptTextIF::EN_PART_INTERFACE_ID;
        uiLast  = (tUInt) AutomounterAdaptTextIF::EN_PART_ENTRY_INVALID;
        break;
    default:
        uiBegin = 0;
        uiLast  = 0;
        NORMAL_M_ASSERT_ALWAYS(); //should never happen
        break;
    }

    //printf("subfctGetTypeOfEntry:iBegin  %d\n",iBegin);
    //printf("subfctGetTypeOfEntry:iLast   %d\n",iLast);

    //now search it
    for(tUInt i= uiBegin; i<uiLast; i++)
    {
        stringTofind = m_ArrayOfEntries[i];
        //printf("subfctGetTypeOfEntry:stringTofind %s\n",stringTofind.c_str());
        //printf("subfctGetTypeOfEntry:strLine      %s\n",strLine.c_str());

        found = strLine.find(stringTofind);

        if (found!=std::string::npos)
        {
            uTypeOfEntry = i;
            //Example: INTERFACE_ID=/dev/sda2, lenOfKey = 13 including '=', lenOfEntry: 9,
            lenOfKey = stringTofind.length();
            lenOfEntry = strLine.length() - lenOfKey;
            strEntrySubString = strLine.substr(lenOfKey,lenOfEntry);

            //printf("GetTypeOfEntry:strLine           :%s\n",strLine.data());
            //printf("GetTypeOfEntry:stringTofind      :%s\n",stringTofind.data());
            //1printf("GetTypeOfEntry:strEntrySubString :%s\n",strEntrySubString.data());
            break;
        }
    }
}



void AutomounterAdaptTextIF::GetTypeOfEntryDevice(OUT AutomounterAdaptTextIF::tenEntryDevice &enTypeOfEntryDevice ,OUT std::string &strEntrySubString,IN const std::string &strLine) const
{
    tUInt uTypeOfEntry;
    //return values if no match found
    enTypeOfEntryDevice = AutomounterAdaptTextIF::EN_DEV_ENTRY_INVALID; //used as return value if nothing is found
    strEntrySubString = "";
    uTypeOfEntry = (tUInt)enTypeOfEntryDevice;


    subfctGetTypeOfEntry(OUT uTypeOfEntry ,OUT strEntrySubString,
                         IN strLine,
                         IN EN_TYP_DEVICE);

    enTypeOfEntryDevice = (AutomounterAdaptTextIF::tenEntryDevice)uTypeOfEntry;

}


void AutomounterAdaptTextIF::GetTypeOfEntryPartition(OUT AutomounterAdaptTextIF::tenEntryPartition &enTypeOfEntryPartition ,OUT std::string &strEntrySubString,IN const std::string &strLine) const
{
    tUInt uTypeOfEntry;
    //return values if not match fount
    enTypeOfEntryPartition = AutomounterAdaptTextIF::EN_PART_ENTRY_INVALID; //used as return value if nothing is found
    strEntrySubString = "";
    uTypeOfEntry = (tUInt)enTypeOfEntryPartition;

    subfctGetTypeOfEntry(OUT uTypeOfEntry ,OUT strEntrySubString,
                         IN strLine,
                         IN EN_TYP_PARTITION);


    enTypeOfEntryPartition = (AutomounterAdaptTextIF::tenEntryPartition)uTypeOfEntry;


}

/*-----------------------------------------------------------------------------*
* void GetDeviceTypeFromMountSrc                                                                 *
*-----------------------------------------------------------------------------*/
void AutomounterAdaptTextIF::GetDeviceTypeFromMountSrc(OUT  CDevice &device, IN std::string strBlockDevName /*e.g. /dev/sdb1,2,3... or /dev/sda1,2,3..*/) const
{
    if(strBlockDevName.empty())
    {
        device.m_eDeviceType = CGlobalEnumerations::DTY_UNKNOWN;
    }
    else if(string::npos != strBlockDevName.find(CDROM__INTERNAL_BLOCK_DEVNAME_PART))
    {
        device.m_eDeviceType = CGlobalEnumerations::DTY_CDROM;
    }
    else if(string::npos != strBlockDevName.find(SDCARD_INTERNAL_BLOCK_DEVNAME_PART))
    {
        device.m_eDeviceType = CGlobalEnumerations::DTY_SD_INTERNAL;
    }
    else if(string::npos != strBlockDevName.find(USB_SD_EXTERNAL_BLOCK_DEVNAME_PART))
    {
        device.m_eDeviceType = CGlobalEnumerations::DTY_USB;
    }
    else
    {
        device.m_eDeviceType = CGlobalEnumerations::DTY_UNKNOWN;
    }
}


/*-----------------------------------------------------------------------------*
* void GetDeviceTypeFromMountSrc                                                                 *
*-----------------------------------------------------------------------------*/
void AutomounterAdaptTextIF::GetUnsupportedReasonType(OUT  CDevice &device, IN std::string strUnsupportedReason /*e.g.'AUDIO_CD' see partition_unsupported_reason_t in automounter_types.h*/ ) const // Roadmap 1615 Internal DVD-Drive
{
    if(strUnsupportedReason.empty())
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__SUPPORTED))
    {
        device.m_eDeviceUnsupportedReason = OK;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__UNSUPPORTED_FILESYSTEM))
    {
        device.m_eDeviceUnsupportedReason =NotMountedbyAutomounter_UNSUPPORTED_FILESYSTEM;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__NO_FILESYSTEM))
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter_UNSUPPORTED_NO_FILESYSTEM;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__AUDIO_CD))
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter_UNSUPPORTED_AUDIO_CD;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__AUDIO_MIXED_MODE))
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter_UNSUPPORTED_AUDIO_MIXED_MODE;
    }
    else if(string::npos != strUnsupportedReason.find(PARTITION_UNSUPPORTED_REASON__BLACKLISTED))
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter_UNSUPPORTED_BLACKLISTED;
    }
    else //case never expected!
    {
        device.m_eDeviceUnsupportedReason = NotMountedbyAutomounter;
    }
}


/*-----------------------------------------------------------------------------*
* void SetDeviceEntry                                                                 *
*-----------------------------------------------------------------------------*/
bool AutomounterAdaptTextIF::SetDeviceEntry(OUT  CDevice &device,IN tUInt iTypeOfEntry,IN const std::string &strEntrySubString) const
{
    bool bRetVal = true;

    switch(iTypeOfEntry)
    {
    //DeviceEntries
    case EN_DEV_INTERFACE_ID:
        device.m_cAccessoryName = strEntrySubString.c_str();
        device.m_cDevnode       = strEntrySubString.c_str();
        break;
    case EN_DEV_IDENTIFIER:
        device.m_cSerialID = strEntrySubString.c_str(); //has to be combined with device.m_cShortSerial afterwards
        break;
    case EN_DEV_STATE:
        break;
    case EN_DEV_PARTITION_CNT:
        break;
    case EN_DEV_TYPE:
        break;
    case EN_DEV_AM_DEVICE_HANDLER:
        break;
        //Partitionentries
    case EN_PART_INTERFACE_ID:
        device.m_cBlockDevnode = strEntrySubString.c_str(); //check if m_cDevnode or m_cDevpath have to be used here
        break;
    case EN_PART_IDENTIFIER:
        device.m_cShortSerial = strEntrySubString.c_str();
        break;
    case EN_PART_STATE:
        if("PARTITION_MOUNTED" == strEntrySubString)
        {
            device.m_iIsMounted = 1;
        }
        break;
    case EN_PART_MOUNT_POINT:
        device.m_cMountPoint = strEntrySubString.c_str();
        break;
    case EN_PART_MOUNT_SRC:
        GetDeviceTypeFromMountSrc(OUT device, IN strEntrySubString);
        break;
    case EN_PART_FILESYSTEM:
        device.m_cFSType = strEntrySubString.c_str();
        break;
    case EN_PART_PARTITION_NO:
        device.m_iValidPartitionNo = atoi(strEntrySubString.c_str());  //check this for a better solution
        break;
    case EM_PART_UNSUPPORTED_REASON:
        GetUnsupportedReasonType(OUT device, IN strEntrySubString);
        #ifdef TMP_CDDA
        device.Show();
        #endif
        break;
    case EN_PART_MOUNTED_WRITABLE:
        break;
        // ERROR cases
    case EN_DEV_ENTRY_INVALID:
    case EN_PART_ENTRY_INVALID:
        bRetVal = false;
        break;
    default:
        bRetVal = false;
        break;
    }

    return bRetVal;
}


/*-----------------------------------------------------------------------------*
 * AutomounterAdaptTextIF GetDeviceMountState(...)                                                            *
 *-----------------------------------------------------------------------------*/
tBool AutomounterAdaptTextIF::GetDeviceMountState(IN std::string strBlockDevName,IN std::string strPartitionNode)
{
    tBool bRes = TRUE;
    std::string strPathFileNameDevDB="";


    //printf("strBlockDevName %s\n",strBlockDevName.c_str());
    //printf("strPartitionNode %s\n",strPartitionNode.c_str());
    //ETG_TRACE_USR4(("GetDeviceMountState : strBlockDevName %s",strBlockDevName.c_str()));
    //ETG_TRACE_USR4(("GetDeviceMountState : strPartitionNode %s",strPartitionNode.c_str()));

    strPathFileNameDevDB  = DIR_AUTOMOUNTER_MOUNTEDDEVICE_DB_FILENAMEBASE;
    strPathFileNameDevDB += strBlockDevName.substr(cDevDirStrLen,strBlockDevName.size());
    strPathFileNameDevDB += "/partition_dev_";
    strPathFileNameDevDB += strPartitionNode.substr(cDevDirStrLen,strPartitionNode.size());
    strPathFileNameDevDB +=".info";

    ETG_TRACE_USR4(("GetDeviceMountState: File to be searched strPathFileNameDevDB : %s",strPathFileNameDevDB.c_str()));

    //printf("strPathFileNameDevDB %s",strPathFileNameDevDB.c_str());

    m_inFile.open(strPathFileNameDevDB.data());
    //Partition file entry
    if (m_inFile.is_open())
    {
        AutomounterAdaptTextIF::tenFileStatus eFileStatus;
        std::string                       strLineOfAutomounter,strEntrySubString;
        eFileStatus =  AutomounterAdaptTextIF::EN_FILE_OK;
        while(eFileStatus == AutomounterAdaptTextIF::EN_FILE_OK)
        {
            GetNextLineOfFile(OUT strLineOfAutomounter,OUT eFileStatus, IN m_inFile);
            if(std::string::npos != strLineOfAutomounter.find(m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_STATE]))
            {
                //printf("strLineOfAutomounter %s\n",strLineOfAutomounter.c_str());
                ETG_TRACE_USR4(("GetDeviceMountState strLineOfAutomounter %s",strLineOfAutomounter.c_str()));
                size_t indexofStr = m_ArrayOfEntries[AutomounterAdaptTextIF::EN_DEV_STATE].size();
                //printf("substr %s\n",strLineOfAutomounter.substr(indexofStr).c_str());
                if("PARTITION_MOUNTED" == strLineOfAutomounter.substr(indexofStr))
                {
                    //printf("Found that the strBlockDevName %s is automounted",strBlockDevName.c_str());
                    ETG_TRACE_USR4(("GetDeviceMountState Found that the strPartitionNode %s is automounted",strPartitionNode.c_str()));
                }
                else
                {
                    //printf("Found that the strBlockDevName %s is not automounted",strBlockDevName.c_str());
                    ETG_TRACE_USR4(("GetDeviceMountState Found that the strPartitionNode %s is not automounted ",strPartitionNode.c_str()));
                    bRes = FALSE;
                }
                //no need to parse next entries
                break;
            }

        }
        m_inFile.close();
    }
    else
    {
        ETG_TRACE_USR1(("GetDeviceMountState: FILE empty: %s",strPathFileNameDevDB.c_str()));
        bRes = FALSE;

    }

    return bRes;
}
////////////////////////////////////////////////////////////////////////////////
// <EOF>



