/*-----------------------------------------------------------------------------*
 * AutomounterAdapter.h                                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file AutomounterAdapter.h
 *
 * \brief new automounter provides a slightly different interface like the previously used mount.sh
 *         in /tmp/.automount a new structure for information can be found
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 * \version Initial Version 03.12.2013, Koechling, Christian (Bosch)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2013-2020
 */

#ifndef __AUTOMOUNTERADAPTER_SHAREDLIBIF_H__
#define __AUTOMOUNTERADAPTER_SHAREDLIBIF_H__

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"
#include "IAutomounterAdapter.h"
#include "automounter_api.h"   //see "%_SWBUILDROOT%/generated/staging/$(COMPILERENV)/usr/include/automounter"/>


/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace std;

/*-----------------------------------------------------------------------------*
 * Forward declaration                                                                  *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/


class AutomounterAdaptSharedLibIF: public TFThread 
{

    public:

    typedef enum
    {
       eThread_AutomounterAdapter = 0,
       eThread_Last       = 1
    }tenThreadFunction;

        
   

   /*!
         * \brief Constructor
         */
     AutomounterAdaptSharedLibIF();

    /*!
         * \brief Destructor
         */
    ~AutomounterAdaptSharedLibIF();



    /*!
         * \brief after a new device has been notified this function is called to read from 'pathfilename' corresponding data' to fill f_CDevice - uses shared lib interface of automounter
         *
         * \param [in] f_device device to be copied
         */
    IAutomounterAdapter::tenResIAutomounterAdapt_Type FillDevice(OUT CDevice& f_device);

    IAutomounterAdapter::tenResIAutomounterAdapt_Type FillDevice(OUT CDevice& device, IN  std::string strPathfilename);


    // EXAMPLE:
    // IN    strAutomounterPartitionFileName: partition_dev_sda1.info
    // OUT strPathFileNameDevDB              : /dev/sda1
    IAutomounterAdapter::tenResIAutomounterAdapt_Type GetExtractBlockDevName(OUT std::string &strDevName, IN std::string strAutomounterPartitionFileName) const;


    tVoid AutomounterLibTriggerCallbacks();

    
    int Initialize(tVoid);


    /**
     * \brief Thread function -
     *
     * Thread to listen to autmounter notification i.e. if new devices are mounted this received within this thread
     * clientHandler_spm and timerExp
     */
    virtual tVoid vThreadFunction();


    /**
    * This function derives from TFThread and starts a user function in a thread from the thread factory's thread pool.
    * After leaving while loop used thread is stopped and released for other users
    *
    * @return void
    */
    virtual void Do(int functionID, void *ptr);


    //---------------------------------------------------------------
    //callbacks functions which will be registered with automounter
    //---------------------------------------------------------------
    static void establish_connection_success();

    static void establish_connection_failure();
    
    static void connection_lost();

    static void device_detected(const device_info_t *device_info);

    static void device_nomedia(const device_info_t *device_info);

    static void device_automounting(const device_info_t *device_info);

    static void device_automounted(const device_info_t *device_info);

    static void device_unmounting(const device_info_t *device_info);

    static void device_unmounted(const device_info_t *device_info);

    static void device_invalid(const device_info_t *device_info);

    static void partition_detected(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_unsupported(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_mounting(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_mounted(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_mount_err(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_remounting(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_unmounting(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_unmounted(const partition_info_t *partition_info, const device_info_t *device_info);

    static void partition_invalid(const partition_info_t *partition_info, const device_info_t *device_info);

    static void update_device_info(const device_info_t *device_info, int request_id);

    static void update_partition_info(const partition_info_t *partition_info,const device_info_t *device_info, int request_id);
    
    static void snapshot_complete(int request_id);


    //------------------------------------------
    // helper functions to proint output of automounter
    //------------------------------------------
    static void printElemOfStruct(const char* strItemName,const char* strItem );
    
    static void printiElemOfStruct(const char* strItemName,int iValue );
    
    static void printbElemOfStruct(const char* strItemName,bool bValue );
    
    static void print_device_info(const device_info_t *device_info);
    
    static void print_part_info(const partition_info_t *part_info);


};

#endif //__AUTOMOUNTERADAPTER_SHAREDLIBIF_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>

