/*-----------------------------------------------------------------------------*
 * WorkThread.cpp                                             *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2014 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file WorkThread.cpp
 *
 * \brief Worker thread for device manager
 *
 * \version 18.06.2014, Koechling, Christian (Bosch),
 *          -#changed to work with threadFactory-threads
 *          -# shifted disgnosis functions to DevicemanagerInterface
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_BASICS
#define INCLUDE_VD_DVM_OSAL         ///* Include OSAL Interface */
#include "Common.h"


#include <iostream>
#include <fstream>

#include "GenString.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"
#include "PrmManager.h"
#ifdef USE_ENABLE_DEVMGR_GPIOIF
    #include "GpioIf.h"
#endif //#ifdef USE_ENABLE_DEVMGR_GPIOIF
#include "UDevManager.h"
#include "debug/TraceCmdManager.h"
#include "device/DeviceListManager.h"
#include "ports/PortListManager.h"
#include "interface/DeviceManagerInterface.h"
#include "DeviceManager_ErrorCodes.h"
#include "config/ConfigurationManager.h"
#include "VoltageManager.h"
#include "debug/HistoryManager.h"
#include "PlatformCtrl.h"
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
#include "CyclicDiagThread.h"
#endif
#include "WorkThread.h"

#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_SD_NOTIFY
#include <systemd/sd-daemon.h>
#endif

//see INTERMEDIATEDEFINE_REFACTORUDEV
#include "IDeviceRecognizer.h"
#include "DeviceRecognizer.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_WORKERTHREAD
#include "trcGenProj/Header/WorkThread.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
WorkThread::WorkThread()
{
    ETG_TRACE_USR4(("Begin: constructor WorkThread"));

    m_bActiveThread = FALSE;

    ETG_TRACE_USR4(("End  : constructor WorkThread"));

}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
WorkThread::~WorkThread()
{

    //@todo care about deleting all singletons
    //---------------------------
    //disable cyclic diagnosis
    //---------------------------
#ifndef USE_SHIFT_CYCLICDIAGNOSIS
    DeviceManagerInterface::GetInstance()->vDiagEnable(FALSE);   //lint !e1551 Warning 1551;Function may throw exception
#endif
}

/*-----------------------------------------------------------------------------*
 * tVoid vThreadFunction()                                                     *
 *-----------------------------------------------------------------------------*/
tVoid WorkThread::vThreadFunction()
{
    ETG_TRACE_USR4(("Begin: WorkThread::vThreadFunction"));

    //-----------------------------------------------------------
    //DeviceListManager
    //-----------------------------------------------------------
    DeviceListManager *l_pDeviceListApp = DeviceListManager::GetInstance();

    if (l_pDeviceListApp->Initialize() == OSAL_ERROR)
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pDeviceListApp->Initialize() == OSAL_ERROR"));

        DeviceListManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pDeviceListApp->Initialize()"));
    }

    //-----------------------------------------------------------
    //PortListManager
    //-----------------------------------------------------------
    PortListManager *l_pPortlistList = PortListManager::GetInstance();

    if (!l_pPortlistList)
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: l_pPortlistList == NULL"));

        PortListManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pPortlistList "));
    }
    PortListManager::GetInstance();

    //-----------------------------------------------------------
    //ConfigurationManager
    //-----------------------------------------------------------
    ConfigurationManager *l_pConfigurationApp = ConfigurationManager::GetInstance();

    if (l_pConfigurationApp)
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pConfigurationApp->Initialized"));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pConfigurationApp->Initialize() == OSAL_ERROR"));
    }


    //-----------------------------------------------------------
    //state table
    //-----------------------------------------------------------
    StateTable *l_pStateTableApp = StateTable::GetInstance();



    if (l_pStateTableApp->Initialize() == OSAL_ERROR)
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pStateTableApp->Initialize() == OSAL_ERROR"));

        StateTable::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pStateTableApp->Initialized()"));
        TraceCmdManager::GetInstance()->SetStateTableMountIF(l_pStateTableApp);
    }

    //---------------------------------------------------------------
    // PlatformCtrl
    //--------------------------------------------------------------
    //this pointer to object mapping is ok since a singleton is used - i.e. during lifetime of PlatformCtrl Interface is not deleted
    //because of reference usage there is no null pointer check in PlatformCtrl and PrmManager which is using the intererface these use references
    IStateTableHWSignals     *pIStateTableHWSignals     = (IStateTableHWSignals*)(StateTable::GetInstance());
    IStateTableHWMalfunction *pIStateTableHWMalfunction = (IStateTableHWMalfunction*)(StateTable::GetInstance());

    PlatformCtrl l_oPlatformCtrl(pIStateTableHWSignals,pIStateTableHWMalfunction);

    //interface injection: PrmManager provides interface which is called by StateTable if PrmManager has registered callback for SDcard at prm
    //and receives a callback and calls a function of StateTable to add payload of callback to queue
    //if StateTable reads from the queue if calls function of CardIF.
    StateTable::GetInstance()->SetPrmManagerCardIF(l_oPlatformCtrl.GetInterfacePrmManagerCard());

    //reisters callbacks for SDCard and HW-signals
    if (l_oPlatformCtrl.Initialize() == OSAL_ERROR)
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pVoltageManagerApp->Initialize() == OSAL_ERROR"));
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        //-----------------------------------------------------------
        //PRM interface - inject interface pointer
        //-----------------------------------------------------------

        DeviceManagerInterface::GetInstance()->SetPrmManagerIF(l_oPlatformCtrl.GetInterfacePrmManager());

        TraceCmdManager::GetInstance()->SetPrmManagerIF(l_oPlatformCtrl.GetInterfacePrmManager());
        TraceCmdManager::GetInstance()->SetPrmManagerCardIF(l_oPlatformCtrl.GetInterfacePrmManagerCard());

        //-----------------------------------------------------------
        //VoltageManager inject interface pointer
        //-----------------------------------------------------------
        DeviceManagerInterface::GetInstance()->SetVoltageManagerIF(l_oPlatformCtrl.GetInterfaceVoltageManager());
        TraceCmdManager::GetInstance()->SetVoltageManagerIF(l_oPlatformCtrl.GetInterfaceVoltageManager());

        //@todo for completion: has to be continued for other platform e.g. GPIOIF elements see director platform

    }

    //-----------------------------------------------------------
    //UDEV (uses StateTable)
    //-----------------------------------------------------------
    //Pointer of CUDevManager Class
    CUDevManager *l_pUDevManagerApp = CUDevManager::GetInstance();

    //see INTERMEDIATEDEFINE_REFACTORUDEV
    DeviceRecognizer *l_pDeviceRecognizer = new DeviceRecognizer();
    CUDevManager::GetInstance()->setIDeviceRecognizer((IDeviceRecognizer*)l_pDeviceRecognizer); //tbd.: take care that UdevManager is not a singleton anymore then IDeviceRecognizer will be handed over to UdevManager using the contructor (already prepared). Currently I don't want to add to many changes here before refactoring has been fully done

    if (l_pUDevManagerApp->InitializeUDevMonitor() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in UDEV monitor initialization"));

        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        l_pUDevManagerApp->SetIAutomounterAdapter(l_oPlatformCtrl.GetInterfaceAutomounterAdapter());
        l_pUDevManagerApp->SetIStorageInfoProvider(l_oPlatformCtrl.GetInterfaceStorageInfoProvider());

        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: Injected IAutmounterAdapter to UdevManager"));

        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pUDevManagerApp->InitializeUDevMonitor()"));

    }

    if (l_pUDevManagerApp->InitializeInotifyMonitor() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in INOTIFY monitor initialization"));

        //l_pUDevManagerApp->DestroyInstanceUDevManager();
        //exit(1);
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pUDevManagerApp->InitializeInotifyMonitor()"));
    }


#ifdef USE_ENABLE_DEVMGR_GPIOIF
    //-----------------------------------------------------------
    // GPIO Interface
    //-----------------------------------------------------------
    GpioIf* l_pGpioIf = GpioIf::GetInstance( l_pStateTableApp );

    if( l_pGpioIf->Initialize() == OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "vThreadFunction: ERROR l_pGpioIf->Initialize() == OSAL_ERROR" ));
        GpioIf::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(( "[WorkThread] vThreadFunction: OK : l_pGpioIf->Initialize()"));
    }
#endif //#ifdef USE_ENABLE_DEVMGR_GPIOIF

    //-----------------------------------------------------------
    //TraceCmdManager
    //-----------------------------------------------------------
    TraceCmdManager *l_pTraceCmdManagerApp = TraceCmdManager::GetInstance();

    if (l_pTraceCmdManagerApp->Initialize() == OSAL_ERROR)
    {
        ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pTraceCmdManagerApp->Initialize() == OSAL_ERROR"));

        TraceCmdManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pTraceCmdManagerApp->Initialize()"));
    }

    //---------------------------------------------------------------------
    //create monitor - to enable events like add,change and remove to be send by udev
    // see function StartMonitorLooseblocking
    //---------------------------------------------------------------------
    if(l_pUDevManagerApp->CreateUdevMonitor() != DEVICEMANAGER_OK)
    {
        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pUDevManagerApp->CreateUdevMonitor())"));
    }

    //---------------------------------------------------------------------
    // create inotify monitor - to enable processing block device notification events
    // as per the newer udev scripts
    //---------------------------------------------------------------------
    if(l_pUDevManagerApp->CreateInotifyMonitor() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in INOTIFY monitor creation"));

        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pUDevManagerApp->CreateInotifyMonitor())"));
    }
    //---------------------------------------------------------------------
    // create pipe monitor - to enable processing block device notification events //Roadmap 14002 Mountdiag
    // as per the newer udev scripts
    //---------------------------------------------------------------------
    if(l_pUDevManagerApp->CreatePipeMonitor() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in pipe monitor creation"));

        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pUDevManagerApp->CreatePipeMonitor())"));
    }


    //==========================================================================================
    // trigger udev to send all. lib/systemd/system/devicemanager.servce in configured for this
    //==========================================================================================

//VD_DeviceManager sends notifiy in RNAIVI,PSARCC if it is up ready.Tthis is used by systemd to trigger udev notification of all devices. Due to statup changes this is necessary to catch Appledevice and its sound card after e.g. 16s and not 50sec
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_SD_NOTIFY
    sd_notify(0, "READY=1");
    ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK :sd_notify done"));
#endif

#ifndef BUGFIX_STARTUPRECOGNITION
    // Scan for already mounted devices
    l_pUDevManagerApp->LookupExistingMounts();
#endif

#ifdef USE_LOOKUP
    //-----------------------------------------------------------
    //UDEV: Lookup devices plugged in
    //-----------------------------------------------------------

    if (l_pUDevManagerApp->LookupConnectedDevices() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in connected devices lookup"));

        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }

    //------------------------------------------------------------------------------------
    //Lookup fills state table - now allow next input to state table to trigger execution time to be started
    //-----------------------------------------------------------------------------------
    l_pStateTableApp->enableExecutionTimer(TRUE);
    l_pStateTableApp->vChkTriggerTimer(FALSE); //@todo check what happens if nothing is connected at startup
#else //USE_LOOKUP
    l_pStateTableApp->enableExecutionTimer(TRUE); //trigger will be done if device detected by monitor
#endif //USE_LOOKUP


    //----------------------------------------------------------------------------
    //after UDEV-LookupConnectedDevices it's allowed to make service available now
    //----------------------------------------------------------------------------
    FILE *l_pStartupDisableFile;
    l_pStartupDisableFile = fopen (STARTUP_SERVICEAVAILABLE_DISABLEFILE, "r");
    if (l_pStartupDisableFile)
    {
       ETG_TRACE_FATAL(("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!/Test case enabled!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
       ETG_TRACE_FATAL(("[ok] STARTUP: SERVICE AVAILABILITY: PERMANENT DISABLED (Found file %s:) ",STARTUP_SERVICEAVAILABLE_DISABLEFILE));
       ETG_TRACE_FATAL(("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!Test case enabled/!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
       fclose(l_pStartupDisableFile);
    }
    else
    {
        DeviceManagerInterface::GetInstance()->vSetServiceAvailable(TRUE);
    }

    //----------------------------------------------
    //enable cyclic diagnosis
    //----------------------------------------------
#ifdef USE_SHIFT_CYCLICDIAGNOSIS
    if(m_pCyclicDiagThread)
    {
        m_pCyclicDiagThread->Initialize(); //this starts the thread. VD_DVM_main
        ETG_TRACE_COMP(("[WorkThread] vThreadFunction: m_pCyclicDiagThread->Initialize() called"));
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: m_pCyclicDiagthread missing pointer "));
    }

#else

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_CYCLIC_DIAGNOSIS))
    {
        FILE *l_pCyclicDiagDisableFile;
        l_pCyclicDiagDisableFile = fopen (CYLCIC_DIAGNOSIS_DISABLEFILE, "r");
        if (l_pCyclicDiagDisableFile)
        {
           ETG_TRACE_FATAL(("Found file %s: hence cyclic diagnosis is disabled",CYLCIC_DIAGNOSIS_DISABLEFILE));
           DeviceManagerInterface::GetInstance()->vDiagEnable(FALSE);   //lint !e1551 Warning 1551;Function may throw exception
           fclose(l_pCyclicDiagDisableFile);
        }
        else
        {
            DeviceManagerInterface::GetInstance()->vDiagEnable(TRUE);   //lint !e1551 Warning 1551;Function may throw exception
        }
    }
#endif

    //==========================================================
    //UDEV:start endless loop to wait on new devices
    //==========================================================
    if (l_pUDevManagerApp->StartMonitorLooseBlocking() != DEVICEMANAGER_OK)
    {
        ETG_TRACE_FATAL (("[WorkThread] vThreadFunction: Exiting due to error in start monitor"));

        CUDevManager::DestroyInstance();
        DVM_FATAL_M_ASSERT_ALWAYS();
    }

    //============NOTE: currently VD_DVM is designed not to go to OFF-State - hence this code part is not reached yet at all==========
    ETG_TRACE_USR4(("End  : WorkThread::vThreadFunction"));
}

/*-----------------------------------------------------------------------------*
 *tBool WorkThread::bActivate()
 *-----------------------------------------------------------------------------*/
tBool WorkThread::bActivate()
{
    ETG_TRACE_USR4(("Begin: bActivate"));

    tBool bRes = FALSE;
    if(!m_bActiveThread)
    {

        //-----------------------------------------------------------
        //Thread factory starts all threads necessary for the none  CCA-Part
        //-----------------------------------------------------------

        ThreadFactoryDVM *l_pThreadFactoryApp = ThreadFactoryDVM::GetInstance();

        if (l_pThreadFactoryApp->Initialize() == OSAL_ERROR)
        {
            ETG_TRACE_FATAL(("ERROR: vThreadFunction: ERROR l_pThreadFactoryApp->Initialize() == OSAL_ERROR"));

            ThreadFactoryDVM::DestroyInstance();

            DVM_FATAL_M_ASSERT_ALWAYS();
        }
        else
        {
            ETG_TRACE_COMP(("[WorkThread] vThreadFunction: OK : l_pThreadFactoryApp->Initialize()"));
        }

        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)WorkThread::eThread_WorkThread, NULL); //starts thread

        ETG_TRACE_COMP(("WorkThread: iThreadIndex:%d",iThreadIndex));
        bRes = TRUE;
    }
    else
    {
        ETG_TRACE_FATAL(("bActivate: Thread has been started already"));
    }
    ETG_TRACE_USR4(("End  : bActivate"));
    return bRes;
}


tBool WorkThread:: bDeActivate()
{
    //to be done
    return TRUE;
}

/*-----------------------------------------------------------------------------*
 *void WorkThread::Do(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
void WorkThread::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do"))

    (void)ptr; //antilint - not used
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case WorkThread::eThread_WorkThread: //execution tree for thread eThread_SystemVolt
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_WorkThread");
            vThreadFunction();
            m_bActiveThread = FALSE;
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"))
}

#ifdef USE_DECOUPLE_CYCLICDIAGCCA
/*-----------------------------------------------------------------------------*
 *void WorkThread::SetInterface(...)
 *-----------------------------------------------------------------------------*/
void WorkThread::SetInterface(CyclicDiagThread *pCyclicDiagThread)
{
    m_pCyclicDiagThread = pCyclicDiagThread;
    ETG_TRACE_USR4(("SetInterface"));
}
#endif //USE_DECOUPLE_CYCLICDIAGCCA
////////////////////////////////////////////////////////////////////////////////
// <EOF>
