/*-----------------------------------------------------------------------------*
 * CyclicDiagThread.cpp                                             *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2014 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file CyclicDiagThread.cpp
 *
 * \brief Worker thread for device manager
 *
 * \version 18.06.2014, Koechling, Christian (Bosch),
 *          -#changed to work with threadFactory-threads
 *          -# shifted disgnosis functions to DevicemanagerInterface
 */
 
/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_BASICS
#define INCLUDE_VD_DVM_OSAL         ///* Include OSAL Interface */
#include "Common.h"

#include "unistd.h" //for sleep if timout of ms are necessary thread has to listen to queue and timer has to be used to wake up loop
#include "ICyclicDiagnosis.h"
#include "ICyclicDiagReceiver.h"
#include "CyclicDiagThread.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_DIAGLOG
#include "trcGenProj/Header/CyclicDiagThread.cpp.trc.h"
#endif
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

#include "ETGTrace.h"

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CyclicDiagThread::CyclicDiagThread()
{
    ETG_TRACE_USR4(("Begin: constructor CyclicDiagThread"));

    m_eThreadState = eThread_off;

    m_StartDelay_sec = VD_CLIENTHANDLER_DIAGLOG_TIMER_TIME_UNTIL_FIRST_TICK_SEC_DEFAULT;
    m_Interval_sec   = VD_CLIENTHANDLER_DIAGLOG_TIMER_TICK_INTERVAL_SEC_DEFAULT;

    m_bDiagEnabled        = FALSE;
    m_bFirstDelayFinished = FALSE;

    m_pICyclicDiagnosis    = NULL;
    m_pICyclicDiagReveiver = NULL;


    ETG_TRACE_USR4(("End  : constructor CyclicDiagThread"));

}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CyclicDiagThread::~CyclicDiagThread()
{
}


tVoid CyclicDiagThread::SetTimeout(unsigned int StartDelay_sec, unsigned int Interval_sec)
{
   m_StartDelay_sec = StartDelay_sec;
   m_Interval_sec   = Interval_sec;
   if(Interval_sec == 0)
   {
       ETG_TRACE_USR4(("[Warning]: setTimout: Interval_sec: %d - thread won't loo - is started once with delay %d ",Interval_sec,StartDelay_sec));
   }
}

/*-----------------------------------------------------------------------------*
 * tVoid vThreadFunction()                                                     *
 *-----------------------------------------------------------------------------*/
tVoid CyclicDiagThread::vThreadFunction()
{
    ETG_TRACE_USR4(("Begin: [CyclicDiagThread]:vThreadFunction"));
    m_eThreadState = eThread_running;
    m_LockThreadActive.lock();

    vector<trDtcCodeResult> ListCyclicDiagResults;


    ETG_TRACE_COMP(("[CyclicDiagThread]:vThreadFunction Cyclic Diagnosis: m_eThreadState: %d m_StartDelay_sec:%d m_Interval_sec:%d",
                      ETG_CENUM(CyclicDiagThread::tenThreadState,m_eThreadState),m_StartDelay_sec,m_Interval_sec));

    sleep(m_StartDelay_sec);
    ETG_TRACE_COMP(("[CyclicDiagThread]: After Start delay: %d have passed by",m_StartDelay_sec));
    m_bFirstDelayFinished = TRUE;
    do
    {
        //----------------------------------
        // error case
        //0: no looping once executed
        //----------------------------------
        if(m_Interval_sec == 0)
        {
            m_eThreadState = eThread_stopcalled;
            ETG_TRACE_FATAL(("[ERROR]: Wrong parameter: vThreadFunction is thought do run intervalls not once (m_eThreadState set:%d)", 
                                     ETG_CENUM(CyclicDiagThread::tenThreadState,m_eThreadState)));

        }
       
        //diag routine call can be enabled or disabled
        if(m_bDiagEnabled)
        {
            ETG_TRACE_USR4(("[CyclicDiagThread]:Round begins"));

            if(m_pICyclicDiagnosis)
            {
                m_pICyclicDiagnosis->ProcessDiag(OUT ListCyclicDiagResults);
                
            }
            if(m_pICyclicDiagReveiver)
            {
                m_pICyclicDiagReveiver->ReceiveDiagResults(IN ListCyclicDiagResults);
            }

            ETG_TRACE_USR4(("[CyclicDiagThread]:Round ends"));
        }

        sleep(m_Interval_sec);  
        ETG_TRACE_COMP(("[CyclicDiagThread]: Alive - Interval: %d m_bDiagEnabled:0x%x",m_Interval_sec,m_bDiagEnabled));
    }while(m_eThreadState == eThread_running);

    m_eThreadState = eThread_off;
    m_LockThreadActive.unlock();

    ETG_TRACE_USR4(("End  : [CyclicDiagThread]:vThreadFunction"));
}

/*-----------------------------------------------------------------------------*
 *void CyclicDiagThread::Do(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
void CyclicDiagThread::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do"))
 

    (void)ptr; //antilint - not used
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case CyclicDiagThread::eThread_CyclicDiagThread: //execution tree for thread eThread_SystemVolt 
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_CyclicDiagThread");
            vThreadFunction();
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"))
}



void CyclicDiagThread::Initialize()
{
    ETG_TRACE_USR4(("Begin: initialize"));

    if(m_eThreadState == eThread_off)
    {
        m_eThreadState = eThread_started;
        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)CyclicDiagThread::eThread_CyclicDiagThread, NULL); //starts thread 

        ETG_TRACE_COMP(("CyclicDiagThread: iThreadIndex:%d",iThreadIndex));
      
    }
      
   
    ETG_TRACE_USR4(("End  : initialize"));

}


void CyclicDiagThread:: Shutdown()
{
    //to be done
    SetEnableCyclicDiag(FALSE);

    m_eThreadState = eThread_stopcalled;

    while(FALSE == m_LockThreadActive.lock(TIMOUT_LOCKCYCLICTHREAD_TIMOUT))
    {
        ETG_TRACE_FATAL(("[ERROR]: Shutdown: no succede to get lock with timout TIMOUT_LOCKCYCLICTHREAD_TIMOUT:%d",(int)TIMOUT_LOCKCYCLICTHREAD_TIMOUT))
        DVM_NORMAL_M_ASSERT_ALWAYS();
    }

    if(m_eThreadState == eThread_off)
    {
        ETG_TRACE_FATAL(("bDeActivate: m_threadState:%d",ETG_CENUM(CyclicDiagThread::tenThreadState,m_eThreadState)));
    }

    m_LockThreadActive.unlock();
}

tVoid CyclicDiagThread::SetEnableCyclicDiag(tBool bEnable) //is not used yet is that a merge probem feature used by diagnosis?
{
    m_bDiagEnabled = bEnable;
    ETG_TRACE_USR4(("SetEnableCyclicDiag :0x%x", m_bDiagEnabled));
}

tVoid CyclicDiagThread::SetInterface(ICyclicDiagnosis* pICyclicDiagnosis)
{
    m_pICyclicDiagnosis = pICyclicDiagnosis;
}
tVoid CyclicDiagThread::SetInterface(ICyclicDiagReceiver* pICyclicDiagReceiver)
{
    m_pICyclicDiagReveiver = pICyclicDiagReceiver;
}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
