/*-----------------------------------------------------------------------------*
 * DeviceManagerInterface                                                              *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     :G3G                                                     *
 * COPYRIGHT   : (c) 2014  Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DeviceManagerInterface.cpp
 *
 * \brief decouple form special CCA-Solution
 *
 * objective is to connect it to other IF-mechanism then CCA
 *
 * \version Initial Version: Koechling Christian (CM-AI/ECD1)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */
/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_READREG
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#define DIAGLIB_INCLUDE_GENERIC
#include <diaglib_if.h>
using namespace diaglib;

#include <fstream>
#include <iostream>
#include <unistd.h> //for GCC 4.7.3 

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "UDevManager.h"
#include "IPrmManager.h"
#include "StateTable.h"
#include "device/DeviceListManager.h"
#include "ports/PortListManager.h"
#include "debug/HistoryManager.h"
#include "debug/TraceCmdManager.h"
#include "DeviceManager_ErrorCodes.h"


#include "interface/DeviceManagerInterface.h"

#include "WorkThread.h"

#ifdef USE_MEASUREMENT_RESISTANCE_USBCONNECTOR
#include "diag/DiagTestMeasureResistance.h"
#endif
#include "config/ConfigurationManager.h"


#include "VD_DeviceManager_main.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_SERVICE_DEVICEMANAGER
#include "trcGenProj/Header/DeviceManagerInterface.cpp.trc.h"
#endif  //VARIANT_S_FTR_ENABLE_TRC_GEN
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;
using namespace configmgr;

/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
DeviceManagerInterface *DeviceManagerInterface::m_pDeviceManagerInterface = NULL; // static pointer used to ensure a single instance of VD_DeviceManager
DVMLOCK                 DeviceManagerInterface::m_singelton;

/*-----------------------------------------------------------------------------*
 * private Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
DeviceManagerInterface::DeviceManagerInterface()
{
    ETG_TRACE_USR4(("Begin: constructor DeviceManagerInterface"));

    //clear array
    for(tU8 i=0; i<=NUMOFUSBCONNECTORS;i++)
    {
        m_rArrayDiagTestMeasureResistance[i].u32SendID      = 0;
        m_rArrayDiagTestMeasureResistance[i].eUSBConnector  = eUSBUndef;
        m_rArrayDiagTestMeasureResistance[i].pThreadPointer = NULL;
    }

    m_pIPrmManager     = NULL;
    m_bIsBTAAlive      = FALSE;
    m_bOverTemperature = FALSE;
    m_pIVoltageManager = NULL;

    ETG_TRACE_USR4(("End  : constructor DeviceManagerInterface"));
}

/*-----------------------------------------------------------------------------*
 * private Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
DeviceManagerInterface::~DeviceManagerInterface()
{
    m_pIPrmManager = NULL;
}

/*-----------------------------------------------------------------------------*
 * DeviceManagerInterface* GetInstance(StateTable *pStateTable)                            *
 *-----------------------------------------------------------------------------*/
DeviceManagerInterface *DeviceManagerInterface::GetInstance()
{
    ETG_TRACE_USR4(("Begin: GetInstance"));
    DeviceManagerInterface::m_singelton.lock();
    if(NULL == m_pDeviceManagerInterface)
    {
        m_pDeviceManagerInterface = new DeviceManagerInterface();
    }
    DeviceManagerInterface::m_singelton.unlock();
    ETG_TRACE_USR4(("End  : GetInstance"));

    return m_pDeviceManagerInterface;
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance()                                                      *
 *-----------------------------------------------------------------------------*/
void DeviceManagerInterface::DestroyInstance()
{
    DeviceManagerInterface::m_singelton.lock();
    ETG_TRACE_USR4(("Begin: DestroyInstance"));
    DEL_M(m_pDeviceManagerInterface);
    DeviceManagerInterface::m_singelton.unlock();
    ETG_TRACE_USR4(("End  : DestroyInstance"));
}


//=====================================================================================
// IN: these functions are used by VD_DVM to receive input  from the 'outside world'
//=====================================================================================

/*-----------------------------------------------------------------------------*
 *tVoid DeviceManagerInterface::vDiagMountCDRom(...)
 *-----------------------------------------------------------------------------*/
IN tVoid DeviceManagerInterface::vDiagMountCDRom(IN tBool bMount, OUT const std::string &strDeviceName /*e.g. /dev/sr0*/) const//Roadmap 14002 MountCDROM (diagMixeMode)
{
    ETG_TRACE_USR4(("Begin: IN vDiagMountCDRom"));
    StateTable::GetInstance()->vDiagMountCDRom(bMount,strDeviceName);
    ETG_TRACE_USR4(("End  : IN vDiagMountCDRom"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceManagerInterface:vSetOverTemperatureCDDrive(...)
 *-----------------------------------------------------------------------------*/
IN tVoid DeviceManagerInterface::vSetOverTemperatureCDDrive(IN tBool bOverTemperature)  //Roadmap 13035 Over temperature
{
    ETG_TRACE_USR4(("Begin: IN vSetOverTemperatureCDDrive"));
    m_bOverTemperature = bOverTemperature;
    StateTable::GetInstance()->vSetOverTemperatureCDDrive(bOverTemperature);
    ETG_TRACE_USR4(("End  : IN vSetOverTemperatureCDDrive"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceManagerInterface:vSetTelematicsBoxApplicationLayerAlive(...)
 *-----------------------------------------------------------------------------*/
IN tVoid DeviceManagerInterface::vSetTelematicsBoxApplicationLayerAlive(IN tBool bisBTAAlive,IN tU32 u32TimoutBeforeSendingITC)  //PSA Project
{
    ETG_TRACE_USR4(("Begin: IN vSetTelematicsBoxApplicationLayerAlive"));
    m_bIsBTAAlive = bisBTAAlive;
    if(u32TimoutBeforeSendingITC)
    {
        ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_TELEMATICSBOX_MONITORING,u32TimoutBeforeSendingITC);
    }
    ETG_TRACE_USR4(("End  : IN vSetTelematicsBoxApplicationLayerAlive"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceManagerInterface:vIsBTABoxAlive(...)
 *-----------------------------------------------------------------------------*/
OUT tVoid  DeviceManagerInterface::vIsBTABoxAlive(INOUT BTAMonitorStates &eBTAMonitoringState,OUT bool &bisBTAAlive) const
{
    ETG_TRACE_USR4(("Begin: bIsBTABoxAlive"));
    static time_t starttime;
    time_t elapsedttime;

    int u32BTABoxConfigValue = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TELEMATICSBOX_MONITORING);

    if( DVM_OFF != u32BTABoxConfigValue  )
    {
        //As per the discussion with Klimanek Martin (BSOT/PJ-SI), The BTA is alive should have a more timeout. So devicemanager configures it as follows
        //the BTABox is alive time out should be eCONF_ENABLE_TELEMATICSBOX_MONITORING + eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT
        u32BTABoxConfigValue += ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT);
        bisBTAAlive = m_bIsBTAAlive;
        if(BTA_MONITOR_NOT_STARTED == eBTAMonitoringState)
        {
            eBTAMonitoringState = BTA_MONITOR_STARTED;
            ETG_TRACE_USR4(("bIsBTABoxAlive: Starting the monitoring.VD_DVM expects the input from FC_ATB with in %d seconds",u32BTABoxConfigValue));
            starttime = ConfigurationManager::GetInstance()->GetEclapsedTime_ms();
        }
        else
        {
            elapsedttime = ConfigurationManager::GetInstance()->GetEclapsedTime_ms();
            int u32difftime = elapsedttime - starttime;
            if(u32difftime >= (u32BTABoxConfigValue*1000))
            {
                eBTAMonitoringState = BTA_MONITOR_FINISHED;
                ETG_TRACE_USR4(("bIsBTABoxAlive: BTA Box is %s",(bisBTAAlive?"Alive":"Not alive")));
            }

        }
    }
    else
    {
        ETG_TRACE_USR4(("!!!!!!!!BTA Monitor mechanism is not enabled in the configuration!!!!!!"));
    }
    ETG_TRACE_USR4(("End: bIsBTABoxAlive"));

}


//=====================================================================================
// OUT: these functions are used by VD_DVM to send to the 'outside world'
//=====================================================================================

/*-----------------------------------------------------------------------------*
 * tVoid  DeviceManagerInterface::vInformClientsCDDriveOvertemp(...)
 *-----------------------------------------------------------------------------*/
OUT tVoid DeviceManagerInterface::vInformClientsCDDriveOvertemp(structNotifyClients &f_oDevice) const //Roadmap_13035_Overtemp
{
    ETG_TRACE_USR4(("Begin: OUT vInformClientsCDDriveOvertemp"));
    vInformClients(f_oDevice, FALSE /*=bTest*/);
    ETG_TRACE_USR4(("End  : OUT vInformClientsCDDriveOvertemp"));
}


/*-----------------------------------------------------------------------------*
 *  OUT tBool DeviceManagerInterface::bGetUSBHardwareConnector(...)
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::bGetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,OUT tenSignalValue &eActivated) const//Roadmap 14003 ReadUSBPowerSupplay_Activated/Deactivated
{
    ETG_TRACE_USR4(("Begin: bGetUSBHardwareConnectors"));

    tBool bRetVal = FALSE;

    tU8   l_u8IndexConfigValue;
    tU16  l_u16Byte19_Bit6And7,l_u16ConfigValue;

    ETG_TRACE_USR2(("bGetUSBHardwareConnector: eUSBConnectorNum=0x%x,eAvtivated=%d",eUSBConnectorNum,ETG_ENUM(TENSIGNALVALUE,eActivated)));

    if(m_pIPrmManager)
    {
        if( ((tU8)eUSBConnectorNum >= (tU8)eUSB1) && ((tU8)eUSBConnectorNum <= (tU8)NUMOFUSBCONNECTORS) )
        {
            bRetVal = m_pIPrmManager->bGetUSBHardwareConnector(IN eUSBConnectorNum,OUT eActivated);

            //extra check to see how configuration is. this is thought to be helpful for upcoming trace analysis
            l_u8IndexConfigValue  = (tU8)((tU8)eCONF_ENABLE_HW_USB1 + (tU8)eUSBConnectorNum  - 1);
            l_u16ConfigValue      = ConfigurationManager::GetInstance()->u16GetConfigurationValue((tenDevManagerConfigValue)l_u8IndexConfigValue);
            l_u16Byte19_Bit6And7  = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_SYSTEM_OFFERING_BYTE1);

            ETG_TRACE_USR2(("bGetUSBHardwareConnector: eAvtivated:          %d",ETG_ENUM(TENSIGNALVALUE,eActivated) ));
            ETG_TRACE_USR2(("bGetUSBHardwareConnector: l_u16ConfigValue:    %d",l_u16ConfigValue));
            ETG_TRACE_USR2(("bGetUSBHardwareConnector: l_u16Byte19_Bit6And7:%d",l_u16Byte19_Bit6And7));
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] bGetUSBHardwareConnector:eUSBConnectorNUM out of range"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] bGetUSBHardwareConnector: m_pIPrmManager = NULL"));
    }

    ETG_TRACE_USR4(("End  : bGetUSBHardwareConnector bRetVal=0x%x",bRetVal));
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 *  IN tBool DeviceManagerInterface::bSetUSBHardwareConnector(...)
 *-----------------------------------------------------------------------------*/
IN tBool DeviceManagerInterface::bSetUSBHardwareConnector(IN tenDevManagerUSBHost eUSBConnectorNum,IN tBool bOn) const//Roadmap 14003 SetUSBPowerSupply_Activated/Deactivated
{
    ETG_TRACE_USR4(("Begin: bSetUSBHardwareConnector"));

    tBool bRetVal = FALSE;
    if(m_pIPrmManager)
    {
        if( ((tU8)eUSBConnectorNum >= (tU8)eUSB1) && ((tU8)eUSBConnectorNum <= (tU8)NUMOFUSBCONNECTORS) )
        {
            bRetVal = m_pIPrmManager->bSetUSBHardwareConnector(IN eUSBConnectorNum,IN bOn);
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] bGetUSBHardwareConnector:eUSBConnectorNUM out of range"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] bSetUSBHardwareConnector: m_pIPrmManager = NULL"));
    }

    ETG_TRACE_USR4(("End  : bSetUSBHardwareConnector: bRetVal=0x%x",bRetVal));
    return bRetVal;
}


/*-----------------------------------------------------------------------------*
 *  IN  tBool DeviceManagerInterface::bInitUSBHardwareConnectors(...)
 *-----------------------------------------------------------------------------*/
IN tBool DeviceManagerInterface::DeviceManagerInterface::bInitUSBHardwareConnectors() const
{
    ETG_TRACE_USR4(("Begin: bInitUSBHardwareConnectors"));

    tBool bRetVal = TRUE;
    if(m_pIPrmManager)
    {
        bRetVal = m_pIPrmManager->bInitUSBHardwareConnectors();
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] bInitUSBHardwareConnectors: m_pIPrmManager = NULL "));
    }

    ETG_TRACE_USR4(("End  : bInitUSBHardwareConnectors"));
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 *   IN tBool bStartMeasureResistanceUSBConnector(...)
 *-----------------------------------------------------------------------------*/
IN tBool DeviceManagerInterface::bStartMeasureResistanceUSBConnector(IN tenDevManagerUSBHost enUSBConnector, IN tU32 u32SendID) //Roadnap 14003  diagnosis measure resistance USB
{
    ETG_TRACE_USR4(("Begin: bStartMeasureResistanceUSBConnector"));

    tBool bRetVal = FALSE;

    //----------------------------------------------------
    //check if USB port is switched on as premisses for measurement
    //----------------------------------------------------
    tenSignalValue eActivated = SIG_UNDEF;

    if(m_pIPrmManager)
    {
        tBool bErrorDriver = m_pIPrmManager->bGetUSBHardwareConnector(IN enUSBConnector,OUT eActivated);
        if(TRUE == bErrorDriver)
        {
            bRetVal = FALSE;
            if(SIG_TRUE == eActivated)
            {
                if(NULL == m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer)
                {
                    //---------------
                    //success
                    //---------------
                    m_rArrayDiagTestMeasureResistance[enUSBConnector].u32SendID = u32SendID;
#ifdef USE_MEASUREMENT_RESISTANCE_USBCONNECTOR    //thread has to be adjusted to use thread factory since feature not used yet (thought for VAGMIBE)
                    m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer = new DiagTestMeasureResistance(IN m_poMainApp,IN enUSBConnector, IN u32SendID); //start thread
#else
                    ETG_TRACE_FATAL(("bStartMeasureResistanceUSBConnector:USE_MEASUREMENT_RESISTANCE_USBCONNECTOR diasbled"));
#endif //USE_MEASUREMENT_RESISTANCE_USBCONNECTOR

                    bRetVal = TRUE;
                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] bStartMeasureResistanceUSBConnector: Test at eUSB%d already executing",enUSBConnector));
                }
            }
        }
        else
        {
            bRetVal = FALSE;
        }

        ETG_TRACE_USR2(("bStartMeasureResistanceUSBConnector: eActivated = %d",ETG_ENUM(TENSIGNALVALUE,eActivated)));
        ETG_TRACE_USR2(("bStartMeasureResistanceUSBConnector: bRetVal    = %d",ETG_ENUM(TBOOL,bRetVal)));
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] bStartMeasureResistanceUSBConnector: m_pIPrmManager = NULL"));
    }
    ETG_TRACE_USR4(("End bStartMeasureResistanceUSBConnector"));
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 *   OUT tVoid vSendResistanceUSBConnector_Result(...)
 *-----------------------------------------------------------------------------*/
OUT tVoid DeviceManagerInterface::vSendResistanceUSBConnector_Result(IN DeviceManagerInterface::tenResultResistance enResultResistance, IN tenDevManagerUSBHost enUSBConnector,IN tU32 u32SendID) //Roadmap 14003 Measure resistance of USB-connector (VAGMIBE)
{
    ETG_TRACE_USR4(("Begin: vSendResistanceUSBConnector_Result"));
    ETG_TRACE_USR2(("vSendResistanceUSBConnector_Result: u32SendID:%d enResultResistance:%d",u32SendID, ETG_CENUM(DeviceManagerInterface::tenResultResistance,enResultResistance)));

    //-----------------
    //send to diaglog here
    //-----------------

    //delete Testthread from list ///@todo think if Interface should be locked by a semaphore to sequentialize usage for Input/output

    if(m_rArrayDiagTestMeasureResistance[enUSBConnector].u32SendID == u32SendID)
    {
        if(m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer)
        {
            if(m_rArrayDiagTestMeasureResistance[enUSBConnector].eUSBConnector == enUSBConnector)
            {
#ifdef USE_MEASUREMENT_RESISTANCE_USBCONNECTOR
                m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer->vWaitFinishThread();
                delete m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer;   //@todo eventually trhow exception here
#else  //USE_MEASUREMENT_RESISTANCE_USBCONNECTOR
                ETG_TRACE_FATAL(("bStartMeasureResistanceUSBConnector:USE_MEASUREMENT_RESISTANCE_USBCONNECTOR disbled"));
#endif //USE_MEASUREMENT_RESISTANCE_USBCONNECTOR
                m_rArrayDiagTestMeasureResistance[enUSBConnector].pThreadPointer = NULL;
            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR] vSendResistanceUSBConnector_Result: m_rArrayDiagTestMeasureResistance[%d].eUSBConnector != enUSBConnector  %d!=%d",
                                 enUSBConnector,m_rArrayDiagTestMeasureResistance[enUSBConnector].eUSBConnector,enUSBConnector));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] vSendResistanceUSBConnector_Result: m_rArrayDiagTestMeasureResistance[%d].u32SendID != u32SendID  %d!=%d",
                             enUSBConnector,m_rArrayDiagTestMeasureResistance[enUSBConnector].u32SendID,u32SendID));
        }
    }

    ETG_TRACE_USR4(("End  : vSendResistanceUSBConnector_Result"));
}

/*-----------------------------------------------------------------------------*
 *   IN tVoid DeviceManagerInterface::vConnectClients(...)
 *-----------------------------------------------------------------------------*/
OUT bool DeviceManagerInterface::IsCriticalVoltage()
{
    ETG_TRACE_USR4(("Begin: GetDiagnosis"));
    bool bRet = false;
    if(m_pIVoltageManager)
    {
        bRet = m_pIVoltageManager->IsCriticalVoltage();
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO]: IsCriticalVoltage: m_pIVoltageManager = NULL"));
    }
    return bRet;
}

#ifdef USE_FOURTH_USB_CONNECTOR
/*-----------------------------------------------------------------------------*
 * tBool bGetDiagnosis(CUSBDiagnosis &f_objUSBDiagnosisHUB1,                   *
 *          CUSBDiagnosis &f_objUSBDiagnosisHUB2,
 *          CUSBDiagnosis &f_objUSBDiagnosisHUB3,
            CUSBDiagnosis &f_objUSBDiagnosisHUB4)                              *
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::bGetDiagnosis(
        CGlobalEnumerations::Diag_Type_e eDiagType,
        CUSBDiagnosis &f_objUSBDiagnosisHUB1,
        CUSBDiagnosis &f_objUSBDiagnosisHUB2,
        CUSBDiagnosis &f_objUSBDiagnosisHUB3,
        CUSBDiagnosis &f_objUSBDiagnosisHUB4) const
#else
/*-----------------------------------------------------------------------------*
* tBool bGetDiagnosis(CUSBDiagnosis &f_objUSBDiagnosisHUB1,                   *
*          CUSBDiagnosis &f_objUSBDiagnosisHUB2,
*          CUSBDiagnosis &f_objUSBDiagnosisHUB3)                              *
*-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::bGetDiagnosis(
    CGlobalEnumerations::Diag_Type_e eDiagType,
    CUSBDiagnosis &f_objUSBDiagnosisHUB1,
    CUSBDiagnosis &f_objUSBDiagnosisHUB2,
    CUSBDiagnosis &f_objUSBDiagnosisHUB3) const
#endif
{
    ETG_TRACE_USR4(("Begin: GetDiagnosis"));

    tBool bDiagOk = FALSE;

    if(TRUE == CUDevManager::GetInstance()->lockNotification(TIMOUT_LOCK_DIAG_MS)) 
    {
        bDiagOk = TRUE;
        ETG_TRACE_COMP(("++++++++/bGetDiagnosis:do diag (begin)++++++++++"));
        //-------------------------------------------
        //information from PRM
        //-------------------------------------------
        if(m_pIPrmManager)
        {
#ifdef USE_FOURTH_USB_CONNECTOR
            m_pIPrmManager->GetDiagnosisValues(f_objUSBDiagnosisHUB1, f_objUSBDiagnosisHUB2, f_objUSBDiagnosisHUB3, f_objUSBDiagnosisHUB4);
#else
            m_pIPrmManager->GetDiagnosisValues(f_objUSBDiagnosisHUB1, f_objUSBDiagnosisHUB2, f_objUSBDiagnosisHUB3);
#endif

        }
        else
        {
            ETG_TRACE_FATAL(("[INFO]: bGetDiagnosis: m_pIPrmManager = NULL"));
        }

        //-------------------------------------------
        //information from UDEV for USB1 and USB2
        //-------------------------------------------
#ifdef USE_FOURTH_USB_CONNECTOR
        CUDevManager::GetInstance()->GetDiagnosisValues(eDiagType,
                                                        f_objUSBDiagnosisHUB1,
                                                        f_objUSBDiagnosisHUB2,
                                                        f_objUSBDiagnosisHUB3,
                                                        f_objUSBDiagnosisHUB4);
#else

        CUDevManager::GetInstance()->GetDiagnosisValues(eDiagType,
            f_objUSBDiagnosisHUB1,
            f_objUSBDiagnosisHUB2,
            f_objUSBDiagnosisHUB3);
#endif

        ETG_TRACE_COMP(("++++++++bGetDiagnosis:do diag (end)/++++++++++"));
        CUDevManager::GetInstance()->unlockNotification();
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: bGetDiagnosis: lockNotification == FALSE (Timout:%d)",TIMOUT_LOCK_DIAG_MS));
    }

    ETG_TRACE_USR4(("End   : bGetDiagnosis"));

    return bDiagOk;
}


/*-----------------------------------------------------------------------------*
 * bool getDeviceProperties(...)
 *-----------------------------------------------------------------------------*/
bool DeviceManagerInterface::getDeviceProperties(const GENSTRING f_ucDeviceSerial,
                                                 CDevice& f_objDevice) const
{
    ETG_TRACE_USR4 (("Begin: getDeviceProperties"));

    bool l_bRetVal = false;

    if (DEVICEMANAGER_DEVICE_DETECTED == CUDevManager::GetInstance()->LookupConnectedDevice(f_ucDeviceSerial, f_objDevice))
    {
        ETG_TRACE_SYS (("getDeviceProperties: Device detected"));
        l_bRetVal = true;
    }
    else
    {
        ETG_TRACE_SYS (("getDeviceProperties: Device NOT detected"));
    }

    ETG_TRACE_USR4 (("End  : getDeviceProperties"));

    return l_bRetVal;
}


/*-----------------------------------------------------------------------------*
 * getAllConnectedDevices(...)
 *-----------------------------------------------------------------------------*/
StateTable* DeviceManagerInterface::getAllConnectedDevices(
        vector<structNotifyClients> &f_vConnectedDevices) const
{
    ETG_TRACE_USR4(("Begin: getAllConnectedDevices"));

    StateTable::GetInstance()->GetAllConnectedDevices(f_vConnectedDevices);

    ETG_TRACE_USR4(("End  : getAllConnectedDevices"));

    return (StateTable::GetInstance());
}




tBool DeviceManagerInterface::bGetSdcDiagnosis( tenSlot eSdcSlot, const tBool& bSdcOcActive ) const
{
    ETG_TRACE_USR4(( "Begin: bGetSdcDiagnosis" ));
    tBool bDiagOk = TRUE;

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
    {
#ifdef USE_ENABLE_DEVMGR_GPIOIF
        StateTable* l_pStateTableManagerApp;
        GpioIf*     l_pGpioIf = NULL;
#else
        (tVoid)eSdcSlot;        // for future use
        (tVoid)bSdcOcActive;    // for future use
#endif //#ifdef USE_ENABLE_DEVMGR_GPIOIF

#ifdef USE_ENABLE_DEVMGR_GPIOIF
        l_pStateTableManagerApp = StateTable::GetInstance();
        if( l_pStateTableManagerApp )
        {
            l_pGpioIf = GpioIf::GetInstance( l_pStateTableManagerApp );
        }
        //information from GPIO interface
        if( l_pStateTableManagerApp  && l_pGpioIf )
        {
            l_pGpioIf->GetDiagnosisValues( eSdcSlot, bSdcOcActive );
        }
        else
        {
            bDiagOk = FALSE;
        }
#endif //#ifdef USE_ENABLE_DEVMGR_GPIOIF
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] bGetSdcDiagnosis: INTERNAL SDCARD DIAGNOSIS switched off"));
    }
    ETG_TRACE_USR4(( "End: bGetSdcDiagnosis" ));
    return bDiagOk;
}

/*-----------------------------------------------------------------------------*
 * tVoid SetPrmManagerIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid DeviceManagerInterface::SetPrmManagerIF(IPrmManager *f_pIPrmManager)
{
    m_pIPrmManager = f_pIPrmManager;
}

/*-----------------------------------------------------------------------------*
 * tVoid SetVoltageManagerIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid DeviceManagerInterface::SetVoltageManagerIF(IVoltageManager *f_pIVoltageManager)
{
    m_pIVoltageManager = f_pIVoltageManager;
}

/*-----------------------------------------------------------------------------*
 * void StoreConnectionNotificationInHistory(...)
 *-----------------------------------------------------------------------------*/
IN void DeviceManagerInterface::StoreConnectionNotificationInHistory(tBool bSend) const
{
    ETG_TRACE_FATAL(("Begin: StoreConnectionNotificationInHistory"));
    vector<CDevice> f_ListOfSendDevicestServices;
    trPortStates    f_ListOfSendArrayPortStates[ARRAYSIZEFORUSB];

    DeviceListManager::GetInstance()->GetCopyDeviceListOfService(OUT f_ListOfSendDevicestServices);
    PortListManager::GetInstance()->GetCopyArrayPortStates(OUT f_ListOfSendArrayPortStates);

    INTERFACEHISTORY.AddToHistory(IN bSend, IN f_ListOfSendDevicestServices, IN f_ListOfSendArrayPortStates);

    ETG_TRACE_FATAL(("End  : StoreConnectionNotificationInHistory"));

}

/*-----------------------------------------------------------------------------*
 * void StoreDiagSDCardNotificationInHistory(...)
 *-----------------------------------------------------------------------------*/
IN void DeviceManagerInterface::StoreDiagSDCardNotificationInHistory(tBool bSend) const
{
    (void) bSend;
    //tbd.:
}

/*-----------------------------------------------------------------------------*
 * tVoid vFillDeviceSpeedinfo(...)                                             *
 *-----------------------------------------------------------------------------*/
tVoid DeviceManagerInterface::vFillDeviceSpeedinfo(INOUT TelematicsBoxSpeedAndPortInfo f_TelematicsBoxSpeedAndPortInfo[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin  : FillDeviceSpeedinfo"));
    //get the device speed of each of the usb port from PRM manager
    m_pIPrmManager->vFillDeviceSpeedinfo(f_TelematicsBoxSpeedAndPortInfo);
    ETG_TRACE_USR4(("End  : FillDeviceSpeedinfo"));
}
/*-----------------------------------------------------------------------------*
 * tBool bGetOverTemperatureCDDriveStatus()                                    *
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::bGetOverTemperatureCDDriveStatus()
{
    return m_bOverTemperature;
}

/*-----------------------------------------------------------------------------*
 * tBool bGetTelematicsBoxApplicationLayerAliveStatus()                        *
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::bGetTelematicsBoxApplicationLayerAliveStatus()
{
    return m_bIsBTAAlive;
}

/*-----------------------------------------------------------------------------*
 * tBool AddTelematicsBox(...)                        *
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::AddTelematicsBox(IN trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo, IN tBool bAdd)
{
    tBool bRetVal = FALSE;

    if(bAdd)  //Add/Update TelematicsBoxDevice Info to datapool
    {
        bRetVal = ConfigurationManager::GetInstance()->AddTelematicsBox(l_uiTelematicsBoxDeviceInfo);
    }
    else  //Remove TelematicsBoxDevice Info from datapool
    {
        tBool ClearAllfromDP = FALSE; //Clear all data from DP NOT needed
        bRetVal = ConfigurationManager::GetInstance()->RemoveTelematicsBox(l_uiTelematicsBoxDeviceInfo, ClearAllfromDP);
    }
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tBool AddExternalHub(...)                        *
 *-----------------------------------------------------------------------------*/
OUT tBool DeviceManagerInterface::AddExternalHub(IN trExternalHubInfo l_uiExternalHubInfo, IN tBool bAdd)
{
    tBool bRetVal = FALSE;

    if(bAdd)  //Add/Update ExternalHub info to datapool
    {
        bRetVal = ConfigurationManager::GetInstance()->AddExternalHub(l_uiExternalHubInfo);
    }
    else  //Remove ExternalHub info from datapool
    {
        tBool ClearAllfromDP = FALSE;  //Clear all data from DP NOT needed
        bRetVal = ConfigurationManager::GetInstance()->RemoveExternalHub(l_uiExternalHubInfo, ClearAllfromDP);
    }

    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceManagerInterface:vSetDVDDriveInfo(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceManagerInterface::vSetDVDDriveInfo(IN trDVDDriveInfo f_rDVDDriveInfo) //NCG3D-79558
{
    ETG_TRACE_USR4(("Begin: IN vSetDVDDriveInfo"));
    CUDevManager::GetInstance()->vDVDDriveInfoUpdated(f_rDVDDriveInfo);
    ETG_TRACE_USR4(("End  : IN vSetDVDDriveInfo"));
}
