/************************************************************************
* FILE:         DiaglibHandlerGen3.h
* PROJECT:      
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: Example implementation of a system set listener.
* This handler sets the demodata value used for IncreaseDecrease demo
* back to default on a DefSet request.
* Take this class as a template for your own listener classes and
* adjust the parts marked with
* // +++ comment +++
*
* - ccademo_diaghandler.x              => // +++ Comment +++
* - ccademo_IncreaseDecreaseHandler.x  => // +++ Comment +++
* - ccademo_tclRoutineControlHandler.x            => // +++ Comment +++
* - ccademo_main.x                     => // +++diaglib+++ comment
* - ccademo_msgtypes.h                 => // +++diaglib+++ comment
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author                 | Modification
* 02.10.08  | CM-AI/PJ-FO45 Evers    | Inital version
*
*************************************************************************/

/*-----------------------------------------------------------------------------*
 * DiaglibHandlerGen3.cpp                                  
 *-----------------------------------------------------------------------------*
 *                                                                             
 * SW-COMPONENT: VD_DeviceManager                                    
 * PROJECT     : GM Gen3                                               
 * COPYRIGHT   : (c) 2014 Robert Bosch GmbH, Hildesheim                  
 *                                                                           
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                   
 *-----------------------------------------------------------------------------*/
/*!
 * \file DiaglibHandlerGen3.cpp
 *
 * \brief Implementation of a Dignosis DiaglibHandlerGen3 for VAGMIBE
 *
 * \version 31.03.2014, Christian koechling (CM-AI/ECD1)
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2014-2020
 */

#include "Config.h"

#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"
#include "WorkThread.h"

#define DIAGLIB_INCLUDE_ROUTINE_CONTROL
#define DIAGLIB_INCLUDE_IO_CONTROL
#define DIAGLIB_INCLUDE_DIAGNOSIS_DATA
#define DIAGLIB_INCLUDE_GENERIC
#include <diaglib_if.h>
using namespace diaglib;

#include "diag/DiagLibHandlerGen2.h"
#include "client/VD_DeviceManager_clienthandler_Diaglog.h"
#include "service/VD_DeviceManager_service_DeviceManager.h"
#include "VD_DeviceManager_main.h"


#include "diag/IDiagLibHandler.h"
#include "diag/DiagLibHandlerGen3.h"

#include "interface/DeviceManagerInterface.h"
#include "diag/DiagTestMeasureResistance.h"
#include "config/ConfigurationManager.h"

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace diaglib;

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_DIAGLOG
#include "trcGenProj/Header/DiagLibHandlerGen3.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#if !defined(ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST) || !defined(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL) || !defined(DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS)
#undef USE_GEN3_DIAGLIBHANDLER
#warning "ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST missing      and/or "
#warning "IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL   and/or "
#warning "DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS missing"
#else
#define USE_GEN3_DIAGLIBHANDLER
#define U32_EMPTY_VALUE (tU32)0x0
#endif



/*-----------------------------------------------------------------------------*
 * Enums                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * constructor
 *-----------------------------------------------------------------------------*/
DiaglibHandlerGen3::DiaglibHandlerGen3(vd_devicemanager_tclApp *poMainApp, diaglib::tclServiceDiaglib* pDiaglibService): m_pDiaglibService(pDiaglibService)
{
    ETG_TRACE_USR4(("Begin:DiaglibHandlerGen3"));
#ifdef USE_GEN3_DIAGLIBHANDLER
    if(m_pDiaglibService != NULL)
    {
        ETG_TRACE_USR1(("DiaglibHandlerGen3:vRegisterListener(ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST              (=%d)) ", ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST));
        ETG_TRACE_USR1(("DiaglibHandlerGen3:vRegisterListener(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL   (=%d)) ", IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL));
        ETG_TRACE_USR1(("DiaglibHandlerGen3:vRegisterListener(DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS (=%d)) ", DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS));

        m_pDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST, this);
        m_pDiaglibService->poGetIoCtrl()->vRegisterListener(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL, this);
        m_pDiaglibService->poGetDiagData()->vRegisterListener(DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS, this);
    }
#else
    ETG_TRACE_FATAL(("VD_DVM:DiaglibHandler_Gen3:dependency DIA_COMMON_GEN3_FEA_DIAGLIB_FIX_01 not found"));
#endif //USE_GEN3_DIAGLIBHANDLER

    m_poMainApp = poMainApp;

    m_u32SendID = 0;
    for(tU8 i=(tU8)eUSB1; i<ARRAYSIZEFORUSB; i++)
    {
       m_eArrUSBPowerActivated[i] = SIG_UNDEF;
       m_bArrUSBPowerActivated[i] = FALSE;
       m_bRetUSBPowerActivated[i] = FALSE;
    }

    ETG_TRACE_USR4(("End  :DiaglibHandlerGen3"));
}

/*-----------------------------------------------------------------------------*
 * destructor
 *-----------------------------------------------------------------------------*/
DiaglibHandlerGen3::~DiaglibHandlerGen3()
{
   ETG_TRACE_USR4(("Begin:~DiaglibHandlerGen3"));
   m_pDiaglibService = NULL;
   m_poMainApp       = NULL;
   ETG_TRACE_USR4(("End  :~DiaglibHandlerGen3"))
}


//======================tclRoutineControlListenerIF=================================================================================================================



/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnRoutineControlStart (...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnRoutineControlStart ( 
                                                                 tU32 u32RoutineId, 
                                                                 tU32 u32Bitmask, 
                                                                 const diaglib::tclParameterVector& oParameterVector,
                                                                 diaglib::tContext MsgContext
                                                               )
{
   ETG_TRACE_USR4(("Begin:vOnRoutineControlStart"));

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32RoutineId );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Bitmask );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( oParameterVector );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext );


   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
#ifdef USE_GEN3_DIAGLIBHANDLER
   tBool bTestIt = FALSE;
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenRoutineCtrlResult oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;

    ETG_TRACE_USR2(("vOnIoControl: u32RoutineId = %d",u32RoutineId));
    switch(u32RoutineId)
    {
      //-----------------------------------------------------------------------
      case ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_u32DevicesToTest :
      //-----------------------------------------------------------------------
      {
         ETG_TRACE_FATAL(("[WARNING] vOnRoutineControlStart: CASE: ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_u32DevicesToTest has been implemented for VAGMIBE and is thus not fully tested"));
        
         // create object for response
         if(diaglib::bCheckParameterSignature(oParameterVector,
                                              ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_LEN , 
                                              ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_PARAM1 ) == FALSE)
         {
            //------------
            //NOK
            //-----------
            ETG_TRACE_USR4(("vOnIoControl: ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK; // Control failed

            diaglib::trParameter paramError;
            paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value        = (tU8)diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramError);
         } 
         else
         {
            // get Data
            //tenDiaglibMediaStatusBits enDiaglibMediaStatusBits;
            tU32 u32DevicesToTest;
            u32DevicesToTest = static_cast<tU32> (oParameterVector[ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_u32DevicesToTest ].u32Value);

            //-------------------------------------------------
            //call of internal function of VD_DeviceManager
            //-------------------------------------------------
                 /*
                enum tenDiaglibMediaStatusBits
                {
                   EN_MediaStatusBitsUSB1 = 1,       // USB1;       1=On, 0=Off     //001
                   EN_MediaStatusBitsUSB2 = 2,       // USB2;       1=On, 0=Off     //010
                   EN_MediaStatusBitsSD_CARD = 4 // SD_CARD; 1=On, 0=Off    //100       Note: For SDCard we don't care hersince Connector can't be switched off elictrically
                };
                */
            ETG_TRACE_USR4(("vOnIoControl: u32DevicesToTest = 0x%x",u32DevicesToTest));
            tBool bGotDeviceToTest = FALSE;
            for(tU8 i=(tU8)eUSB1;i<=(tU8)NUMOFUSBCONNECTORS;i++) 
            {
                 bGotDeviceToTest = bGetUSBDeviceToTest(IN (tenDevManagerUSBHost)i, IN u32DevicesToTest);
                 if(TRUE == bGotDeviceToTest)
                 {
                      bTestIt = DeviceManagerInterface::GetInstance()->bStartMeasureResistanceUSBConnector(IN (tenDevManagerUSBHost)i, ++m_u32SendID); 
                      
                 }
                 else
                 {
                    ETG_TRACE_FATAL(("[ERROR] vOnIoControl:eUSB%d could not be tested",i));
                 }
            }
#if 0  //think about taking this from the code totally
            //check if all hasbeen set as demanded
            if(u32DiaglibMediaStatusBits_Result == u32DiaglibMediaStatusBits)
            {
                //-------------------
                //SUCCESS
                //-------------------
                oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;  
                diaglib::trParameter paramResult;
                paramResult.enType   = static_cast<diaglib::tenParameterType>(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_RESULT_u32MediaStatusBitField);
                paramResult.u32Value = static_cast<tU32> (u32DiaglibMediaStatusBits_Result);
                oTmpVec.push_back(paramResult);
            }
            else
            {
                //-----------------------------------------------------------------------
                //NOK - no fully set as demanded - diagnosis can check u32DiaglibMediaStatusBits_Result
                //-----------------------------------------------------------------------
                ETG_TRACE_FATAL(("[WARNING] vOnIoControl: u32DiaglibMediaStatusBits:0x%x u32DiaglibMediaStatusBits_Result:0x%x - no set as demanded",u32DiaglibMediaStatusBits,u32DiaglibMediaStatusBits_Result));
                oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;  
                diaglib::trParameter paramResult;
                paramResult.enType   = static_cast<diaglib::tenParameterType>(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_RESULT_u32MediaStatusBitField);
                paramResult.u32Value = static_cast<tU32> (u32DiaglibMediaStatusBits_Result);
                oTmpVec.push_back(paramResult);
            }
#endif
         } 
         //bSendResponseToDiaglib = TRUE;
         break;
      } 
      //------------------------------------
      default:
      //------------------------------------
      {
         break;
      }// default:
    }// switch(u32IoControlId)


   if(diaglib::bCheckParameterSignature(oParameterVector, ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_LEN,
                                                          ROUTINEID_DEVICEMAMAGER_RUN_DTC_TEST_START_PARAM1) == FALSE)
   {
      oCtrlResult             = diaglib::EN_ROUTINE_CONTROL_NOT_OK; // Routine Control failed
      oTmpVec[0].u8ErrorValue = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE; // choose one Error
   }// if(diaglib::bCheckParameterSignature(oParamVector, ROUTINEID_NAV_ROUTE_CALC, ...
   else
   {
      // get Data
      tU8 u8Data   = oParameterVector[ROUTINEID_DEMO_RC_START_u8Value].u8Value;
      tU16 u16Data = oParameterVector[ROUTINEID_DEMO_RC_START_u16Value].u16Value;
      tU32 u32Data = oParameterVector[ROUTINEID_DEMO_RC_START_u32Value].u32Value;
      ETG_TRACE_COMP_THR(("--- ccademo_tclDiaglibHandlerGen3::vOnRoutineControlStart => U8=%d, U16=%d, U32=%d",u8Data, u16Data,u32Data));
   }
   /* Excample end */

   // +++ Set values back to default, then send positive response +++
   
   /*
      Response can be one of the following:
      EN_SYSTEMSET_OK            -> Empty error vector

      Actually the diaglib already checks if ID is supported, so we don't need
      to do it here again.
   */


   if(m_pDiaglibService != OSAL_NULL)
   {
      if(TRUE == m_pDiaglibService->poGetRoutineCtrl()->bSendRoutineCtrlMethodResult( oCtrlResult, //@tofo if this is sometimes used it is necessary to care for return values
                                                                                     oTmpVec, 
                                                                                     MsgContext) )
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
         ETG_TRACE_USR4(("--- ccademo_tclDiaglibHandlerGen3::vOnRoutineControlStart => bSendRoutineCtrlMethodResult successfull"));
      }
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_USR4(("!!! ccademo_tclDiaglibHandlerGen3::vOnRoutineControlStart => ERROR : bSendRoutineCtrlMethodResult failed"));
      }
   }
   else
   {
      u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
      ETG_TRACE_USR4(("!!! ccademo_tclDiaglibHandlerGen3::vOnRoutineControlStart => ERROR : m_pDiaglibService == OSAL_NULL"));
   }

   ETG_TRACE_COMP(("<-- ccademo_tclDiaglibHandlerGen3::vOnRoutineControlStart(%x)",u32Return)); 

   ETG_TRACE_USR4(("End  :vOnRoutineControlStart u32Return:%d, bTestIt:0x%x",u32Return, bTestIt));
#else
    
#endif
   return u32Return;
}


/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnRoutineControlAbort ( ...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnRoutineControlAbort ( 
                                                                 tU32 u32RoutineId, 
                                                                 diaglib::tContext MsgContext
                                                              )


{
   ETG_TRACE_USR4(("Begin:vOnRoutineControlAbort"));

   // +++ Reallow access to values that needed default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32RoutineId );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext );

   ETG_TRACE_USR2(("vOnIoControl: u32RoutineId = %d",u32RoutineId));

   ETG_TRACE_USR4(("End  :vOnRoutineControlAbort"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}


/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnGetRoutineResult (( ...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnGetRoutineResult ( 
                                                              tU32 u32RoutineId,
                                                              diaglib::tContext MsgContext
                                                            )
{
   ETG_TRACE_USR4(("Begin:vOnGetRoutineResult"));
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32RoutineId );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext );

   ETG_TRACE_USR2(("vOnIoControl: u32RoutineId = %d",u32RoutineId));

   ETG_TRACE_USR4(("End  :vOnGetRoutineResult"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}


//======================tclIoControlListenerIF=================================================================================================================


/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnIoControl ( ...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnIoControl (tU32 u32IoControlId,diaglib::tenIoControlAction enActionId, const diaglib::tclParameterVector& oParameterVector,diaglib::tContext MsgContext)
{
    ETG_TRACE_USR4(("Begin:vOnIoControl"));

    tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
#ifdef USE_GEN3_DIAGLIBHANDLER
    tBool bOn;
    diaglib::tenIoCtrlResult oCtrlResult = diaglib::EN_IO_CONTROL_NOT_OK;
    tBool bSendResponseToDiaglib = FALSE;
    diaglib::tclParameterVector oTmpVec;

    OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( enActionId );
    OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext );
    // check for expected Routine ID


    ETG_TRACE_USR2(("vOnIoControl: u32IoControlId = %d",u32IoControlId));
    switch(u32IoControlId)
    {
      //-----------------------------------------------------------------------
      case IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL:
      //-----------------------------------------------------------------------
      {
         ETG_TRACE_USR4(("vOnIoControl: CASE: IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField"));

         // create object for response
         if(diaglib::bCheckParameterSignature(oParameterVector,    /* ///@todo no clear how IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField could be checked */
                                              IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_LEN, 
                                              IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_PARAM1) == FALSE)
         {
            //------------
            //NOK
            //-----------
            ETG_TRACE_USR4(("vOnIoControl: ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_IO_CONTROL_NOT_OK; // Control failed

            diaglib::trParameter paramError;
            paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value        = (tU8)diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramError);
         } 
         else
         {
            // get Data
            //tenDiaglibMediaStatusBits enDiaglibMediaStatusBits;
            tU32 u32DiaglibMediaStatusBits_Result = U32_EMPTY_VALUE;
            tU32 u32DiaglibMediaStatusBits        = static_cast<tU32> (oParameterVector[IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField].u32Value);

            //-------------------------------------------------
            //call of internal function of VD_DeviceManager
            //-------------------------------------------------
                 /*
                enum tenDiaglibMediaStatusBits
                {
                   EN_MediaStatusBitsUSB1 = 1,       // USB1;       1=On, 0=Off     //001
                   EN_MediaStatusBitsUSB2 = 2,       // USB2;       1=On, 0=Off     //010
                   EN_MediaStatusBitsSD_CARD = 4 // SD_CARD; 1=On, 0=Off    //100       Note: For SDCard we don't care hersince Connector can't be switched off elictrically
                };
                */
            ETG_TRACE_USR4(("vOnIoControl: enDiaglibMediaStatusBits = 0x%x",u32DiaglibMediaStatusBits));

            /*//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            # Diag Change request: RTC: 78209
               Use Tristate logic to determine if the status of a particular port has to be changed or not. Then change it(ON/OFF) according to request.
               Eg: 00000000 00000000 00000001 00000001 means USB1 is made ON
                   00000000 00000000 00000001 00000000 means USB1 is made OFF
                   00000000 00000000 00000000 00000000 means Dont care

                Similarly done for other ports..
             //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            */
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TRISTATE_DIAGNOSIS))
            {
                ETG_TRACE_USR4(("##############vOnIoControl: USING TRISTATE_DIAGNOSIS method##############"));
                for(tU8 i=(tU8)eUSB1;i<=(tU8)NUMOFUSBCONNECTORS;i++)
                {
                    if(bCheckUSBChangeStatus(IN (tenDevManagerUSBHost)i, IN u32DiaglibMediaStatusBits))     //checks is USBi should be CHANGED OR NOT according to u32DiaglibMediaStatusBits
                    {
                        bOn = bGetUSBActive(IN (tenDevManagerUSBHost)i, IN u32DiaglibMediaStatusBits); //checks is USBi should be active according to u32DiaglibMediaStatusBits
                        if(TRUE == DeviceManagerInterface::GetInstance()->bSetUSBHardwareConnector(IN (tenDevManagerUSBHost)i,bOn))  //if yes then set it
                        {
                           ETG_TRACE_USR4(("vOnIoControl: Set eUSB%d to 0x%x",i,bOn));
                           SetUSBActive(IN (tenDevManagerUSBHost)i,IN bOn, INOUT u32DiaglibMediaStatusBits_Result); //accumulate result into bitfield for later check and return of result
                           break;  // Set One port and exit.. Later remove to check status for all ports
                        }
                        else
                        {
                           ETG_TRACE_FATAL(("[ERROR] vOnIoControl:eUSB%d could not be set 0x%x",i,bOn));
                        }
                    }
                    else
                    {
                       ETG_TRACE_FATAL(("vOnIoControl:eUSB%d is DONT CARE. Doing Nothing...",i));
                    }
                }
            }
            else // If tristate is not be used, use the old implementation for backward compatibility
            {
                ETG_TRACE_USR4(("##############vOnIoControl: USING old method..Backward compatible##############"));
                for(tU8 i=(tU8)eUSB1;i<=(tU8)NUMOFUSBCONNECTORS;i++)
                {
                     bOn = bGetUSBActive(IN (tenDevManagerUSBHost)i, IN u32DiaglibMediaStatusBits); //checks is USBi should be active according to u32DiaglibMediaStatusBits
                     if(TRUE == DeviceManagerInterface::GetInstance()->bSetUSBHardwareConnector(IN (tenDevManagerUSBHost)i,bOn))  //if yes then set it
                     {

                        SetUSBActive(IN (tenDevManagerUSBHost)i,IN bOn, INOUT u32DiaglibMediaStatusBits_Result); //accumulate result into bitfield for later check and return of result
                     }
                     else
                     {
                        ETG_TRACE_FATAL(("[ERROR] vOnIoControl:eUSB%d could not be set 0x%x",i,bOn));
                     }
                }
            }

            //check if all hasbeen set as demanded
            if(u32DiaglibMediaStatusBits_Result == u32DiaglibMediaStatusBits)
            {
                //-------------------
                //SUCCESS
                //-------------------
                ETG_TRACE_USR4(("vOnIoControl:SUCCESS"));
                oCtrlResult = diaglib::EN_IO_CONTROL_OK;  
                diaglib::trParameter paramResult;
                paramResult.enType   = static_cast<diaglib::tenParameterType>(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_RESULT_u32MediaStatusBitField);
                paramResult.u32Value = static_cast<tU32> (u32DiaglibMediaStatusBits_Result);
                oTmpVec.push_back(paramResult);
            }
            else
            {
                //-----------------------------------------------------------------------
                //NOK - no fully set as demanded - diagnosis can check u32DiaglibMediaStatusBits_Result
                //-----------------------------------------------------------------------
                ETG_TRACE_FATAL(("[WARNING] vOnIoControl: u32DiaglibMediaStatusBits:0x%x u32DiaglibMediaStatusBits_Result:0x%x - no set as demanded",u32DiaglibMediaStatusBits,u32DiaglibMediaStatusBits_Result));
                oCtrlResult = diaglib::EN_IO_CONTROL_NOT_OK;
                diaglib::trParameter paramResult;
                paramResult.enType   = static_cast<diaglib::tenParameterType>(IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_RESULT_u32MediaStatusBitField);
                paramResult.u32Value = static_cast<tU32> (u32DiaglibMediaStatusBits_Result);
                oTmpVec.push_back(paramResult);
            }
         } 
         bSendResponseToDiaglib = TRUE;
         break;
      } 
      //------------------------------------
      default:
      //------------------------------------
      {
         ETG_TRACE_FATAL(("vOnIoControl: Not expected to receive u32IoControlId. %d ",u32IoControlId));
         break;
      }// default:
    }// switch(u32IoControlId)


    //-----------------------------------
    //send to diagnosis
    //-----------------------------------
    if(bSendResponseToDiaglib)
    {
        // send message back to diagnosis
        if(m_pDiaglibService != OSAL_NULL)
        {
            if(TRUE == m_pDiaglibService->poGetIoCtrl()->bSendIoCtrlMethodResult( oCtrlResult,enActionId,oTmpVec,MsgContext) ) //sends formerly set EN_ERROR_FATAL_INTERNAL
            {
                u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
                ETG_TRACE_USR4(("vOnIoControl: bSendIoCtrlMethodResult successfull"));
            } 
            else
            {
                u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
                ETG_TRACE_ERR_THR(("[ERROR] vOnIoControl: bSendRoutineCtrlMethodResult failed"));
            } 
        } 
        else
        {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_USR4(("[ERROR] vOnIoControl: m_pDiaglibService == OSAL_NULL"));
        } 
    } 
    else
    {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_USR4(("[ERROR] vOnIoControl: bSendResponseToDiaglib == FALSE"));
    }

#else
    (tVoid)u32IoControlId;
    (tVoid)enActionId;
    (tVoid)oParameterVector;
    (tVoid)MsgContext;

#endif //#ifdef USE_GEN3_DIAGLIBHANDLER
    ETG_TRACE_USR4(("End:  vOnIoControl:u32Return=0x%x",u32Return));
    return u32Return;
}

/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnIoControlFreeze ( ...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnIoControlFreeze (tU32 u32IoControlId,diaglib::tContext MsgContext)
{
    ETG_TRACE_USR4(("Begin:vOnIoControlFreeze"));

   tU32                       u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;

#ifdef USE_GEN3_DIAGLIBHANDLER
   diaglib::tenIoCtrlResult oCtrlResult = diaglib::EN_IO_CONTROL_OK;
   tBool bSendResponseToDiaglib = FALSE;


   ETG_TRACE_USR2(("vOnIoControl: u32IoControlId = %d",u32IoControlId));
   switch(u32IoControlId)
   {
      //--------------------------------------------------------------------
      case IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField:
      //--------------------------------------------------------------------
      {
         ETG_TRACE_USR4(("vOnIoControlFreeze: CASE: IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField"));

         //----------------
         //success
         //----------------
         // store all relevant current values for this IOCONTROL now (not persistent!!! only to restore at unfreeze): This is not necessary since VD_DeviceManager::ConfigurationManager 
         // already has the correct values to restore it
         oCtrlResult = diaglib::EN_IO_CONTROL_OK;
         bSendResponseToDiaglib = TRUE;
         break;
      }
      default:
      {
         break;
      }
   }

    //----------------------------
    //send to diagnosis
    //----------------------------
    if(bSendResponseToDiaglib)
    {
        if(m_pDiaglibService != OSAL_NULL)
        {
         if(TRUE == m_pDiaglibService->poGetIoCtrl()->bSendIoCtrlFreezeResult( oCtrlResult, 
                                                                              MsgContext) )
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
            ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: bSendIoCtrlFreezeResult successfull"));
         }
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: ERROR : bSendRoutineCtrlMethodResult failed"));
         }
        }
        else
        {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze:  ERROR : m_pDiaglibService == OSAL_NULL"));
        }
    } 
    else
    {
        u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
        ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze:  ERROR : bSendResponseToDiaglib == FALSE"));
    }

    ETG_TRACE_USR4(("End  :vOnIoControlFreeze: vOnIoControlFreeze u32Return:0x%x",u32Return));
#else //USE_GEN3_DIAGLIBHANDLER
    (tVoid)MsgContext;
    (tVoid)u32IoControlId;
#endif //#ifdef USE_GEN3_DIAGLIBHANDLER
    return u32Return;
}


/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnIoControlUnfreeze (...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnIoControlUnfreeze (tU32 u32IoControlId,diaglib::tContext MsgContext)
{
    ETG_TRACE_USR4(("Begin:vOnIoControlUnfreeze"));

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
#ifdef USE_GEN3_DIAGLIBHANDLER
   diaglib::tenIoCtrlResult oCtrlResult  = diaglib::EN_IO_CONTROL_OK;
   tBool bSendResponseToDiaglib = FALSE;

   ETG_TRACE_USR2(("vOnIoControl: u32IoControlId = %d",u32IoControlId));
   switch(u32IoControlId)
   {
      //--------------------------------------------------------------------
      case IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField:
      //--------------------------------------------------------------------
      {
         ETG_TRACE_USR4(("vOnIoControlUnfreeze: CASE: IOCONTROLID_DEVICEMAMAGER_USB_DETECTION_CONTROL_SET_u32MediaStatusBitField"));
         if(TRUE == DeviceManagerInterface::GetInstance()->bInitUSBHardwareConnectors())
         {
            oCtrlResult = diaglib::EN_IO_CONTROL_OK; // data valid   
         }
         else
         {
             oCtrlResult = diaglib::EN_IO_CONTROL_NOT_OK; // data valid   
         }
         bSendResponseToDiaglib = TRUE;
         break;
      }
      //------------
      default:
      //------------
      {
         break;
      }
   }// switch(u32IoControlId)

   //-------------------------------
   //send to diagnosis
   //-------------------------------
   if(bSendResponseToDiaglib)
   {
      if(m_pDiaglibService != OSAL_NULL)
      {
         if(TRUE == m_pDiaglibService->poGetIoCtrl()->bSendIoCtrlFreezeResult(oCtrlResult,MsgContext) )
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
            ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: bSendIoCtrlFreezeResult successfull"));
         }
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: bSendRoutineCtrlMethodResult failed"));
         }
      }
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: m_pDiaglibService == OSAL_NULL"));
      }
   }
   else
   {
        u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
        ETG_TRACE_USR4(("[ERROR] vOnIoControlFreeze: bSendResponseToDiaglib == FALSE"));
   }
#else
    (tVoid)MsgContext;
    (tVoid)u32IoControlId;
#endif //#ifdef USE_GEN3_DIAGLIBHANDLER
   ETG_TRACE_USR4(("vOnIoControlFreeze: u32Return=0x%x",u32Return));
   return u32Return;

}


//==================================================origin in tclDiagnosisDataListenerIF=====================================================================================

/*-----------------------------------------------------------------------------*
 * tU32 DiaglibHandlerGen3::vOnDiagnosisDataRead( ...)
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen3::vOnDiagnosisDataRead(tU32 u32DiagDataId, diaglib::tContext MsgContext)
{
    ETG_TRACE_USR4(("Begin:vOnDiagnosisDataRead"));
    tU32           u32Return  =  diaglib::U32_DIAGLIB_RETURN_NOT_OK;
#ifdef USE_GEN3_DIAGLIBHANDLER
    tBool          bActivate;
    tBool          bRetDriver = FALSE;
    tenSignalValue eActivated = SIG_UNDEF;

    ETG_TRACE_USR2(("vOnDiagnosisDataRead: u32DiagDataId = %d",u32DiagDataId));
    switch(u32DiagDataId)
    {
        //----------------------------------------------------------------------------------
        case DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS:
        //----------------------------------------------------------------------------------
            ETG_TRACE_USR2(("vOnDiagnosisDataRead:DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS"));
            vClearArray();
            for(tU8 i=(tU8)eUSB1; i <= (tU8)NUMOFUSBCONNECTORS; i++)
            {
                m_bRetUSBPowerActivated[i] = DeviceManagerInterface::GetInstance()->bGetUSBHardwareConnector(IN (tenDevManagerUSBHost)i,OUT eActivated);
                if(SIG_TRUE == eActivated)
                {
                    bActivate = TRUE;
                }
                else
                {   
                    bActivate = FALSE;
                }
                vAddPartialResult(IN (tenDevManagerUSBHost)i, bActivate); //add if activeated/deactivated for certain connector
            }
            //in VAGMIBE & Suzuki  currently eUSB1 is supported - hence we do care for this only
            if(m_bRetUSBPowerActivated[eUSB1] == TRUE)
            {
                bRetDriver = TRUE;
            }
            // Send message back to diagnosis
            
            if (    (TRUE == bRetDriver)
                 && (OSAL_NULL != m_pDiaglibService) 
                 && (OSAL_NULL != m_pDiaglibService->poGetDiagData()) 
                 /*&& (OSAL_NULL != m_poCpIdInstance)*/) //@todo check for this in example)
            {
                diaglib::tenDiagDataResult  oDataResult =  diaglib::EN_DIAGDATA_OK;
                diaglib::tclParameterVector oTmpVec;
                diaglib::trParameter        paramResult;

                paramResult.enType = static_cast<diaglib::tenParameterType>(DATAID_DEVICEMANAGER_USB_DETECTION_CONTROL_STATUS_READ_RESULT_u32MediaStatusBitField );
   
                paramResult.u32Value        = static_cast<tU32> (m_bArrUSBPowerActivated[eUSB1]); // SIG_UNDEF   = 0x00,SIG_FALSE   = 0x01,SIG_TRUE    = 0x02,

                oTmpVec.push_back(paramResult);
                if (TRUE == m_pDiaglibService->poGetDiagData()->bSendDiagnosisDataResult(oDataResult,oTmpVec, MsgContext))
                {
                   //-----------
                   //success
                   //-----------
                   u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
                   ETG_TRACE_USR4(("vOnDiagnosisDataRead successfull"));
                }  
                else
                {
                   ETG_TRACE_ERR(("[ERROR] vOnDiagnosisDataRead: bSendDiagnosisDataResult failed"));
                }  // End of if-else; if (TRUE == m_poDataIDIf->bSendDiagnosisDataResul..
            }
            else
            {
                //ERROR - SUMMARY
                if(OSAL_NULL == m_pDiaglibService)
                {
                    ETG_TRACE_FATAL(("[ERROR] vOnDiagnosisDataRead:m_pDiaglibService == OSAL_NULL"));
                }
                else if(OSAL_NULL == m_pDiaglibService->poGetDiagData())
                {
                    ETG_TRACE_FATAL(("[ERROR] vOnDiagnosisDataRead: m_pDiaglibService->poGetDiagData() == OSAL_NULL"));
                }
                if(bRetDriver == FALSE)
                {
                    ETG_TRACE_FATAL(("[ERROR] vOnDiagnosisDataRead: bRetDriver == FALSE"));
                }
            }
            break;
         //-----------------------------------------------------------------
         default:
         //-----------------------------------------------------------------
            ETG_TRACE_FATAL(("[ERROR] vOnDiagnosisDataRead: bRetDriver == FALSE"));
            break;
          
    }

#else
    (tVoid)MsgContext;
    (tVoid)u32DiagDataId;
#endif //USE_GEN3_DIAGLIBHANDLER
    ETG_TRACE_USR4(("End:  vOnDiagnosisDataRead: u32Return=0x%x",u32Return));
    return u32Return;

}


tU32 DiaglibHandlerGen3::vOnDiagnosisDataWrite(tU32 /*u32DiagDataId*/,const diaglib::tclParameterVector& /*rfcoParamVector*/,diaglib::tContext /*MsgContext*/)
{
    tU32 u32Return  =  diaglib::U32_DIAGLIB_RETURN_NOT_OK;
    ETG_TRACE_FATAL(("vOnDiagnosisDataWrite: Not supported"));
 
    return u32Return;
}




//================= internal functions==================================================

/*-----------------------------------------------------------------------------*
 * tVoid DiaglibHandlerGen3::vAddPartialResult(...)
 *-----------------------------------------------------------------------------*/
tVoid DiaglibHandlerGen3::vAddPartialResult(IN tenDevManagerUSBHost i, IN tBool bActivated)
{
    if(bActivated == TRUE)
    {
        m_eArrUSBPowerActivated[i] = SIG_TRUE;
        m_bArrUSBPowerActivated[i] = TRUE;
    }
    else
    {
        m_eArrUSBPowerActivated[i] = SIG_FALSE;
        m_bArrUSBPowerActivated[i] = FALSE;
    }
}

/*-----------------------------------------------------------------------------*
 * tVoid DiaglibHandlerGen3::vClearArray()
 *-----------------------------------------------------------------------------*/
tVoid DiaglibHandlerGen3::vClearArray()
{
    for(tU8 i=(tU8)eUSB1; i<ARRAYSIZEFORUSB; i++)
    {
       m_eArrUSBPowerActivated[i] = SIG_UNDEF;
       m_bRetUSBPowerActivated[i] = FALSE;
    }
}

/*-----------------------------------------------------------------------------*
 * tBool DiaglibHandlerGen3::bGetUSBDeviceToTest(...)
 *-----------------------------------------------------------------------------*/
tBool DiaglibHandlerGen3::bGetUSBDeviceToTest(IN tenDevManagerUSBHost eUsbConnector, IN tU32 u32DevicesToTestBits) const
{
    return bGetUSBActive(IN eUsbConnector, IN u32DevicesToTestBits);
}

/*-----------------------------------------------------------------------------*
 * tBool DiaglibHandlerGen3::bGetUSBActive(...)
 *-----------------------------------------------------------------------------*/
tBool DiaglibHandlerGen3::bGetUSBActive(IN tenDevManagerUSBHost eUsbConnector, IN tU32 u32DiaglibMediaStatusBits) const
{
    ETG_TRACE_USR4(("Begin: bGetUSBActive"));
    ETG_TRACE_USR4(("bGetUSBActive: eUsbConnector             : USB%d",eUsbConnector));
    ETG_TRACE_USR4(("bGetUSBActive: u32DiaglibMediaStatusBits : 0x%x",u32DiaglibMediaStatusBits));
    tBool bRetVal;
    /*
    enum tenDiaglibMediaStatusBits
    {
       EN_MediaStatusBitsUSB1 = 1, // USB1; 1=On, 0=Off
       EN_MediaStatusBitsUSB2 = 2, // USB2; 1=On, 0=Off
       EN_MediaStatusBitsSD_CARD = 4 // SD_CARD; 1=On, 0=Off
    };
    */
                                                         //lsb
     /*USB1 ON: MASK: 00000000 00000000 00000000 00000001 = 0x01*/
     /*USB2 ON: MASK: 00000000 00000000 00000000 00000010 = 0x02*/
     /*SD     ON: MASK: 00000000 00000000 00000000 00000010 = 0x04*/ //Note used in this context

    switch(eUsbConnector)
    {
        case eUSB1:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x01);
            break;
        case eUSB2:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x02) >> 1;
            break;
        case eUSB3:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x04) >> 1;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case eUSB4:
            bRetVal = ((u32DiaglibMediaStatusBits)& 0x08) >> 1;
            break;
#endif//USE_FOURTH_USB_CONNECTOR
        default:
            ETG_TRACE_FATAL(("bGetUSBActive: Unexpected eUSBConnector: %d",eUsbConnector));
            bRetVal = FALSE;
            break;
    }
  
    ETG_TRACE_USR4(("bGetUSBActive: bRetVal:0x%x",bRetVal));
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tBool DiaglibHandlerGen3::SetUSBActive(...)
 *-----------------------------------------------------------------------------*/
tBool DiaglibHandlerGen3::SetUSBActive(IN tenDevManagerUSBHost eUsbConnector, tBool IN bOn, INOUT  tU32 &u32DiaglibMediaStatusBits_Result) const
{
    ETG_TRACE_USR4(("Begin:SetUSBActive"));
    ETG_TRACE_USR4(("SetUSBActive: IN    eUsbConnector                    :USB%d",eUsbConnector));
    ETG_TRACE_USR4(("SetUSBActive: IN    bOn                              :0x%x",bOn));
    ETG_TRACE_USR4(("SetUSBActive: INOUT u32DiaglibMediaStatusBits_Result :0x%x",u32DiaglibMediaStatusBits_Result));
    tBool bRetVal = TRUE;
    tU32 bitfield = 0;

    /*
    enum tenDiaglibMediaStatusBits
    {
       EN_MediaStatusBitsUSB1 = 1, // USB1; 1=On, 0=Off
       EN_MediaStatusBitsUSB2 = 2, // USB2; 1=On, 0=Off
       EN_MediaStatusBitsSD_CARD = 4 // SD_CARD; 1=On, 0=Off
    };
    */

     /*USB1 ON: MASK: 00000000 00000000 00000000 00000001 = 0x01*/
     /*USB2 ON: MASK: 00000000 00000000 00000000 00000010 = 0x02*/
     /*USB3 ON: MASK: 00000000 00000000 00000000 00000100 = 0x04*/ //Note used in this context

    switch(eUsbConnector)
    {
        case eUSB1:
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TRISTATE_DIAGNOSIS)) //Use Tristate logic
            {
                if(bOn)
                {

                    bitfield = 0x101; //00000000 00000000 00000001 00000001
                }
                else
                {
                    bitfield = 0x100; //00000000 00000000 00000001 00000000
                }
                u32DiaglibMediaStatusBits_Result |= bitfield ;
            }
            else //Old implementation - backward compatibility
            {
                if(bOn)
                {

                    bitfield = 0x01; //00000000 00000000 00000000 00000001
                    u32DiaglibMediaStatusBits_Result |= bitfield ;
                }
            }

            break;
        case eUSB2:
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TRISTATE_DIAGNOSIS)) //Use Tristate logic
            {
                if(bOn)
                {

                    bitfield = 0x202; //00000000 00000000 00000010 00000010
                }
                else
                {
                    bitfield = 0x200; //00000000 00000000 00000010 00000000
                }
                u32DiaglibMediaStatusBits_Result |= bitfield ;
            }
            else //Old implementation - backward compatibility
            {
                if(bOn)
                {
                    bitfield = 0x02;  //00000000 00000000 00000000 00000010
                    u32DiaglibMediaStatusBits_Result |= bitfield ;
                }
            }

            break;

        case eUSB3:
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TRISTATE_DIAGNOSIS)) //Use Tristate logic
            {
                if(bOn)
                {

                    bitfield = 0x404; //00000000 00000000 00000100 00000100
                }
                else
                {
                    bitfield = 0x400; //00000000 00000000 00000100 00000000
                }
                u32DiaglibMediaStatusBits_Result |= bitfield ;
            }
            else //Old implementation - backward compatibility
            {
                if(bOn)
                {
                    bitfield = 0x04;  //00000000 00000000 00000000 00000100
                    u32DiaglibMediaStatusBits_Result |= bitfield ;
                }
            }
            break;

#ifdef USE_FOURTH_USB_CONNECTOR

        case eUSB4:
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TRISTATE_DIAGNOSIS)) //Use Tristate logic
            {
                if(bOn)
                {

                    bitfield = 0x808; //00000000 00000000 00001000 00001000
                }
                else
                {
                    bitfield = 0x800; //00000000 00000000 00001000 00000000
                }
                u32DiaglibMediaStatusBits_Result |= bitfield ;
            }
            else //Old implementation - backward compatibility
            {
                if(bOn)
                {
                    bitfield = 0x08;  //00000000 00000000 00000000 00001000
                    u32DiaglibMediaStatusBits_Result |= bitfield ;
                }
            }
            break;
#endif//USE_FOURTH_USB_CONNECTOR
        default:
            bRetVal = FALSE;
            ETG_TRACE_FATAL(("SetUSBActive: Unexpected eUSBConnector: %d",eUsbConnector));
            break;
    }
    ETG_TRACE_USR4(("SetUSBActive: OUT u32DiaglibMediaStatusBits_Result :0x%x",u32DiaglibMediaStatusBits_Result));
    ETG_TRACE_USR4(("End  :SetUSBActive"));
    return bRetVal;
}


/*-----------------------------------------------------------------------------*
 * tBool DiaglibHandlerGen3::bCheckUSBChangeStatus(...)
 *-----------------------------------------------------------------------------*/
tBool DiaglibHandlerGen3::bCheckUSBChangeStatus(IN tenDevManagerUSBHost eUsbConnector, IN tU32 u32DiaglibMediaStatusBits) const
{
    ETG_TRACE_USR4(("Begin: bCheckUSBChangeStatus"));
    ETG_TRACE_USR4(("bCheckUSBChangeStatus: eUsbConnector             : USB%d",eUsbConnector));
    ETG_TRACE_USR4(("bCheckUSBChangeStatus: u32DiaglibMediaStatusBits : 0x%x",u32DiaglibMediaStatusBits));
    tBool bRetVal;
                                                         //lsb
     /*USB1 ON: MASK: 00000000 00000000 00000001 00000000 = 0x100*/
     /*USB2 ON: MASK: 00000000 00000000 00000010 00000000 = 0x200*/
     /*USB3 ON: MASK: 00000000 00000000 00000100 00000000 = 0x400*/
	 /*USB4 ON: MASK: 00000000 00000000 00001000 00000000 = 0x800*/

    switch(eUsbConnector)
    {
        case eUSB1:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x100) >> 8;
            break;
        case eUSB2:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x200) >> 9;
            break;
        case eUSB3:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x400) >> 10;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case eUSB4:
            bRetVal = ((u32DiaglibMediaStatusBits) & 0x800) >> 11;
            break;
#endif//USE_FOURTH_USB_CONNECTOR
        default:
            ETG_TRACE_FATAL(("bCheckUSBChangeStatus: Unexpected eUSBConnector: %d",eUsbConnector));
            bRetVal = FALSE;
            break;
    }

    ETG_TRACE_USR4(("bCheckUSBChangeStatus: bRetVal:0x%x",bRetVal));
    return bRetVal;
}


