/*-----------------------------------------------------------------------------*
 * DeviceListManager.cpp                                                       *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DeviceListManager.cpp
 *
 * \brief This file holds implementation for class DeviceListManager
 *
 * \version 25.08.2011,Koechling, Christian (Bosch), Initial Version
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"
#include <unistd.h> //for sleep
#include "Utils.h"
#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"

//#include "StateTable.h"
#include "device/DeviceListManager.h"
#include "debug/TraceCmdManager.h"
#include "debug/HistoryManager.h"
#include "DeviceManager_ErrorCodes.h"

#include "config/ConfigurationManager.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_LISTMANAGER
#include "trcGenProj/Header/DeviceListManager.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;
using namespace devlistmgr;

/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
DeviceListManager *DeviceListManager::m_pDeviceListManager = NULL; // static pointer used to ensure a single instance of PrmManager

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
DVMLOCK     DeviceListManager::m_singelton;
#endif

#ifdef CODE_CLEANUP_SEPT
tBool DeviceListManager::bPlatformEventsLocked[((tInt)eUSBAll+1)];
#else
//@todo: linked to size of tenDevManagerUSBHost
//for possible future generic extension it seems to be better to init this in the contructor
tBool DeviceListManager::bPlatformEventsLocked[] ={FALSE,FALSE,FALSE,FALSE};
#endif

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
DVMLOCK     DeviceListManager::m_lock;
#endif

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
DeviceListManager::DeviceListManager()
{
    ETG_TRACE_USR4(("Begin: constructor DeviceListManager"));

    for(tInt i= 0; i<=(tInt)eUSBAll;i++)
    {
        bPlatformEventsLocked[i] = FALSE;
    }

    // Init with error values.
    for( tInt i=0; i<ARRAYSIZEFORCARD; ++i )
    {
        m_rDevicesCard[i].vSetErrorValues();
    }

    ETG_TRACE_USR4(("End  : constructor DeviceListManager"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
DeviceListManager::~DeviceListManager()
{
    ETG_TRACE_USR4(("Begin: ~DeviceListManager"));


    ETG_TRACE_USR4(("End  : ~DeviceListManager"));
}

/*-----------------------------------------------------------------------------*
 * DeviceListManager *GetInstance(tVoid)                                       *
 *-----------------------------------------------------------------------------*/
DeviceListManager *DeviceListManager::GetInstance(tVoid)
{

    ETG_TRACE_USR4(("Begin: GetInstance"));
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    DeviceListManager::m_singelton.lock();
#endif
    if(NULL == m_pDeviceListManager)
    {
        m_pDeviceListManager = new DeviceListManager();
    }

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    DeviceListManager::m_singelton.unlock();
#endif
    ETG_TRACE_USR4(("End  : GetInstance"));
    return m_pDeviceListManager;
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance(tVoid)                                                 *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::DestroyInstance()
{
    ETG_TRACE_USR4(("Begin: DestroyInstance"));

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    DeviceListManager::m_singelton.lock();
#endif
    DEL_M(m_pDeviceListManager);
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    DeviceListManager::m_singelton.unlock();
#endif

    ETG_TRACE_USR4(("End  : DestroyInstance"));
}

/*-----------------------------------------------------------------------------*
 * int Initialize()                                                            *
 *-----------------------------------------------------------------------------*/
int DeviceListManager::Initialize() const
{
    tS32 s32RetVal = OSAL_OK;
    ETG_TRACE_USR4(("Begin: Initialize                                         "));

    ETG_TRACE_USR4(("End  : Initialize                                        "));

    return (int)s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * tenDevManagerUSBHost mapMountPoint(DEVICELIST_Type eDevListType,char*cMount)*
 *-----------------------------------------------------------------------------*/
tenDevManagerUSBHost DeviceListManager::mapMountPoint(DEVICELIST_Type eDevListType,
        tPCChar cMount)
{
    DeviceListManager::lockList();

    GENSTRING strMount = cMount;
    tenDevManagerUSBHost usbConnector = eUSB1;//@todo care for invalid value
    ETG_TRACE_USR4(("Begin: mapMountPoint(cMount)"));

    if(DEVLIST_STATETABLE == eDevListType)
    {
        if(NULL != cMount)
        {
            for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
            {
                ETG_TRACE_USR4(("mapMountPoint: USB%d:m_vConnectedDevices_StateTable[%2d].cMountPoint:%s       ",
                        m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                        l_uiCount,
                        m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                if(strMount ==  m_vConnectedDevices_StateTable[l_uiCount].cMountPoint)    //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                {
                    ETG_TRACE_COMP(("mapMountPoint: USB%d:m_vConnectedDevices_StateTable[%2d].cMountPoint:%s       ",
                            m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                            l_uiCount,
                            m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));

                    usbConnector=m_vConnectedDevices_StateTable[l_uiCount].eUSB;
                    break;
                }
            }
        }
        else
        {
            ETG_TRACE_FATAL(("ERROR: NULL == cMount"));
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: mapMountPoint: DEVLIST_STATETABLE != eDevListType"));
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : mapMountPoint(cMount)"));
    return usbConnector; //@todo care for invalid value
}

/*-----------------------------------------------------------------------------*
 * tVoid checkNumOfDevicesAtConnector(DEVICELIST_Type eDevListType,            *
 *              tenDevManagerUSBHost usb, tInt &iNumOfConnectedDevices,        *
 *              tInt &iAnyApplyFor)                                            *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::checkNumOfDevicesAtConnector(DEVICELIST_Type eDevListType,
        tenDevManagerUSBHost usb, OUT tInt &iNumOfConnectedDevices, OUT tInt &iAnyApplyFor)
{

    ETG_TRACE_USR4(("Begin: checkNumOfDevicesAtConnector: eDevListType=%d usb=%d",eDevListType,usb));

    DeviceListManager::lockList();
    if(DEVLIST_STATETABLE == eDevListType)
    {
        iAnyApplyFor                          = 0;
        iNumOfConnectedDevices                = 0;

        ETG_TRACE_USR4(("checkNumOfDevicesAtConnector: m_vConnectedDevices_StateTable.size()=%d",
                m_vConnectedDevices_StateTable.size()));
        //--------------------------------------------------------------------------------------
        //check if any device is set 'APPLY' - see in 'StateTable::StartMonitorLooseBlocking' case enMountUdev
        //--------------------------------------------------------------------------------------
        for (unsigned int l_uiCount=0; l_uiCount < m_vConnectedDevices_StateTable.size() ; l_uiCount++)
        {
            if(usb == m_vConnectedDevices_StateTable[l_uiCount].eUSB)
            {
                //count connected devices
                if(  (USB_DEV_REMOVED_BY_USR != m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
                   &&(USB_DEV_UNDEFINED      != m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus))
                {
                    iNumOfConnectedDevices++;
                }
                //count devices 'apply for connect or remove'
                if( (m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus == USB_DEV_INTERNAL_APPLY_CONNECT)
                        || (m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus == USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR)
                )
                {
                    ++iAnyApplyFor;
                }
            }
        }

        ETG_TRACE_COMP(("checkNumOfDevicesAtConnector: USB all     : NumOfDevices: %d",m_vConnectedDevices_StateTable.size()));
        ETG_TRACE_COMP(("checkNumOfDevicesAtConnector: USB%1d      : NumOfDevices: %d",usb, iNumOfConnectedDevices));
        ETG_TRACE_COMP(("checkNumOfDevicesAtConnector: USB%1d Apply: iAnyApplyFor: %d",usb, iAnyApplyFor));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: checkNumOfDevicesAtConnector: DEVLIST_STATETABLE != eDevListType"));
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End:  checkNumOfDevicesAtConnector"));
}

/*-----------------------------------------------------------------------------*
 * tInt ListAddDevice(DEVICELIST_Type eDevListType, tenDevManagerUSBHost eUsb, *
 *              structNotifyClients oNewCurDevice)                             *
 *-----------------------------------------------------------------------------*/
tInt DeviceListManager::ListAddDevice(DEVICELIST_Type eDevListType,
        tenDevManagerUSBHost eUsb,structNotifyClients oNewCurDevice)
{
    ETG_TRACE_USR4(("Begin: ListAddDevice"));

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST //To avoid compiler warnings in linuxx86_64 build
    (void)eUsb;
#endif

    int  l_DeviceFound = DEVICEMANAGER_ERROR_NODEVICE;

    if(DEVLIST_STATETABLE == eDevListType)
    {
        //----------------------------------
        //check if device already exists in device list
        //----------------------------------
        for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
        {
            ETG_TRACE_USR4(("ListAddDevice: f_oCurDevice.cMountPoint                         %s",
                    oNewCurDevice.cMountPoint));
            ETG_TRACE_USR4(("ListAddDevice: m_vConnectedDevices_StateTable[%2d].cMountPoint                         %s",
                    l_uiCount,
                    m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));


            int l_iResCompareMountpoint = strcmp(oNewCurDevice.cMountPoint,m_vConnectedDevices_StateTable[l_uiCount].cMountPoint);

            if(((strlen(oNewCurDevice.cMountPoint) != 0) && (0 == l_iResCompareMountpoint)) || (0 == strcmp(oNewCurDevice.cSerialID,m_vConnectedDevices_StateTable[l_uiCount].cSerialID)))
            {
                ETG_TRACE_USR2(("ListAddDevice:Device Present in list                                   "));
                if(0 != l_iResCompareMountpoint)
                {
                    ETG_TRACE_U64_USR4((char*)"ListAddDevice: Updating the mount point. May be it is changed during the malfunction. Malfunction time stamp = %d",m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction);
                    strncpy(m_vConnectedDevices_StateTable[l_uiCount].cMountPoint,   oNewCurDevice.cMountPoint,   sizeof(m_vConnectedDevices_StateTable[l_uiCount].cMountPoint)   -1);
                }
                if( 0 != oNewCurDevice.u64TotalSize_KB  && 0 == m_vConnectedDevices_StateTable[l_uiCount].u64TotalSize_KB)
                {
                    m_vConnectedDevices_StateTable[l_uiCount].u64TotalSize_KB = oNewCurDevice.u64TotalSize_KB;
                    m_vConnectedDevices_StateTable[l_uiCount].u64UsedSize_KB  = oNewCurDevice.u64UsedSize_KB;
                    m_vConnectedDevices_StateTable[l_uiCount].u64FreeSize_KB  = oNewCurDevice.u64FreeSize_KB;
                    ETG_TRACE_USR4(("ListAddDevice: Updating the Storage Info for the device."));

                }
                m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ConnectTimeStampForRetry = oNewCurDevice.u64TimeStamp_ConnectTimeStampForRetry;
                l_DeviceFound = DEVICEMANAGER_DEVICE_DETECTED;  //device detected in list
                break;
            }
            else
            {
                ETG_TRACE_USR2(("ListAddDevice:searching: device not yet found in list                            "));
            }
        }

        //----------------------------------
        //device not yet found in list - hence add it
        //----------------------------------
        if(DEVICEMANAGER_ERROR_NODEVICE == l_DeviceFound)
        {
            ETG_TRACE_COMP(("------------ADD2LIST-USB%1d,%d,mount:%s-----------",
                    eUsb,
                    ETG_ENUM(DVM_DEVICE_TYPE,oNewCurDevice.eDeviceType),
                    oNewCurDevice.cMountPoint));

            oNewCurDevice.s32MalfunctionElapsedTime_sec = MALFUNCTION_NO;
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
            oNewCurDevice.u64TimeStamp_ApplyForRemove         = U64TIMESTAMP_NOTSET;
            oNewCurDevice.u64TimeStamp_ApplyForConnect        = U64TIMESTAMP_NOTSET;
            oNewCurDevice.u64TimeStamp_Malfunction            = U64TIMESTAMP_NOTSET;
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
            oNewCurDevice.u64TimeStamp_HighRiskForMalfunction = U64TIMESTAMP_NOTSET;
#endif //USE_HIGHRISK4USB_HWMALFUNCTION
#endif //TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
            HWMALFUNCTIONHISTORY.AddToHistory("ListAddDevice(...)",&oNewCurDevice);
#endif

            m_vConnectedDevices_StateTable.push_back(oNewCurDevice);
            l_DeviceFound =DEVICEMANAGER_OK; //no error added to list
        }

        ETG_TRACE_COMP(("ListAddDevice: connected devices: %d",m_vConnectedDevices_StateTable.size()));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: ListAddDevice: DEVLIST_STATETABLE != eDevListType"));
    }

    ETG_TRACE_USR4(("End  : ListAddDevice"));
    return l_DeviceFound;
}

/*-----------------------------------------------------------------------------*
 * tInt ListChkDeleteDevice(DEVICELIST_Type eDevListType,                      *
 *              tenDevManagerUSBHost eUsb)                                     *
 *-----------------------------------------------------------------------------*/
tInt DeviceListManager::ListChkDeleteDevice(DEVICELIST_Type eDevListType,
        tenDevManagerUSBHost eUsb)
{
    ETG_TRACE_USR4(("Begin: ListChkDeleteDevice"));
    tBool bLoop  = TRUE;
    tBool bFound = FALSE;
    tInt iNumOfDeletedDevices = 0;

    if(DEVLIST_STATETABLE == eDevListType)
    {
        ETG_TRACE_COMP(("ListChkDeleteDevice: connected devices: %d",
                m_vConnectedDevices_StateTable.size()));

        while(TRUE == bLoop)
        {
            for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
            {
                if(eUsb == m_vConnectedDevices_StateTable[l_uiCount].eUSB)
                {
                    if(USB_DEV_REMOVED_BY_USR == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
                    {
                        ETG_TRACE_COMP(("ListChk4DeleteDevice:------------DELETE FROM LIST: USB%1d,mount:%s --------------------------",
                                eUsb,m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                        //this changes the order in the vector hence next elem with 'USB_DEV_REMOVED_BY_USR'
                        //has to be searched from the beginning again
                        m_vConnectedDevices_StateTable.erase(m_vConnectedDevices_StateTable.begin() + (tInt)l_uiCount );      //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
                        iNumOfDeletedDevices++;
                        bFound = TRUE;
                        break; //stop for-loop
                    }
                    else
                    {
                        ETG_TRACE_COMP(("ListChkDeleteDevice:m_vConnectedDevices_StateTable[l_uiCount].cMountPoint=%s",
                                m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                    }
                }
                else
                {
                    ETG_TRACE_USR4(("ListChk4DeleteDevice:          NOT DELETE: USB%d:m_vConnectedDevices_StateTable[%2d].cMountPoint       %s",eUsb,l_uiCount,
                            m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                }
            }
            ETG_TRACE_COMP(("ListChk4DeleteDevice:bFound=0x%x",bFound));
            if(TRUE == bFound)   ///@todo try it more elegant
            {
                bFound = FALSE;
                bLoop = TRUE; //continue to check if another elem with 'USB_DEV_REMOVED_BY_USR' exists since the erase has changed the list

            }
            else
            {
                bLoop = FALSE; //no device found anymore hence no further check is necessary
            }
        }

        ETG_TRACE_USR2(("ListChk4DeleteDevice:DeletedDevices=%d",iNumOfDeletedDevices));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: ListChk4DeleteDevice: DEVLIST_STATETABLE != eDevListType"));
    }
    ETG_TRACE_USR4(("End  : ListChkDeleteDevice"));

    return iNumOfDeletedDevices;
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceAllDevice(DEVICELIST_Type eDevListType,tenDevManagerUSBHost eUsb)*
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::TraceAllDevice(DEVICELIST_Type eDevListType,
        tenDevManagerUSBHost eUsb)
{
    ETG_TRACE_USR4(("Begin: TraceAllDevice"));

    int  iNumDevs = 0;
    tUInt l_uiCount;

    DeviceListManager::lockList();
    //m_lock.lock();
    //------------------------
    // StateTable
    //------------------------
    if(DEVLIST_STATETABLE == eDevListType)
    {
        ETG_TRACE_FATAL(("[ok]: ==================================->StateTable======================================================="));
        for ( l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                ETG_TRACE_FATAL(("[ok]: TraceAllDevice: m_vConnectedDevices_StateTable[%2d]:USB%d eDeviceType:%d (%s)",
                        l_uiCount,
                        m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                        ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType),
                        m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_StateTable[l_uiCount].eUSB)
                {
                    ETG_TRACE_FATAL(("[ok]: TraceAllDevice: m_vConnectedDevices_StateTable[%2d]:USB%d eDeviceType:%d (%s)",
                            l_uiCount,
                            m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                            ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType),
                            m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                    iNumDevs++;
                }
            }
        }
        ETG_TRACE_FATAL(("[ok]: TraceAllDevice: => %d:   %d   devices connected",ETG_ENUM(SHOW_FOR_CONNECTOR,eUsb),iNumDevs));
        ETG_TRACE_FATAL(("[ok]: ====================================StateTable<-====================================================="));
    }

    //------------------------
    // Service
    //------------------------
    else if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        ETG_TRACE_FATAL(("[ok] ================================->vd_devicemanager_tclService_DeviceManager=================================================="));
        for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_Service.size();
                l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                ETG_TRACE_FATAL(("[ok] TraceAllDevice: m_vConnectedDevices_Service[%2d]:USB%1d eDeviceType:%d (%s)",
                        l_uiCount,
                        m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                        ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_Service[l_uiCount].m_eDeviceType),
                        m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_Service[l_uiCount].m_eUSB)
                {
                    ETG_TRACE_FATAL(("[ok] TraceAllDevice: m_vConnectedDevices_Service[%2d]:USB%1d eDeviceType:%d (%s)",
                            l_uiCount,
                            m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                            ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_Service[l_uiCount].m_eDeviceType),
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));
                    iNumDevs++;
                }
            }
        }

        ETG_TRACE_FATAL(("[ok] TraceAllDevice: => %d:   %d   devices connected",ETG_ENUM(SHOW_FOR_CONNECTOR,eUsb),iNumDevs));
        ETG_TRACE_FATAL(("[ok] ==================================vd_devicemanager_tclService_DeviceManager<-========================================================="));
    }
    //------------------------
    // UdevManager
    //------------------------
    else if(DEVLIST_UDEVMGR == eDevListType)
    {
        ETG_TRACE_FATAL(("[ok] ================================->UDevManager=================================================="));

        for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_UdevMgr.size();
                l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                ETG_TRACE_FATAL(("[ok] TraceAllDevice: m_vConnectedDevices_UdevMgr[%2d]:USB%1d eDeviceType:%d (%s)",
                        l_uiCount,
                        m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                        ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType),
                        m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint.toStdString().c_str()));
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB)
                {
                    ETG_TRACE_FATAL(("[ok] TraceAllDevice: m_vConnectedDevices_UdevMgr[%2d]:USB%1d eDeviceType:%d (%s)",
                            l_uiCount,
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                            ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType),
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint.toStdString().c_str()));
                    iNumDevs++;
                }
            }
        }

        ETG_TRACE_FATAL(("[ok] TraceAllDevice: => %d:   %d   devices connected",ETG_ENUM(SHOW_FOR_CONNECTOR,eUsb),iNumDevs));
        ETG_TRACE_FATAL(("[ok] ================================->UDevManager<-========================================================="));

    }
    DeviceListManager::unlockList();
    //m_lock.unlock();
    ETG_TRACE_USR4(("End  : TraceAllDevice"));

}

/*-----------------------------------------------------------------------------*
 * void getAllConnectedDevices(DEVICELIST_Type eDevListType,                   *
 *              vector<CDevice *> &f_vConnectedDevices)                        *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::getAllConnectedDevices(DEVICELIST_Type eDevListType,
        vector<CDevice *> &f_vConnectedDevices)
{
    ETG_TRACE_USR4(("Begin: getAllConnectedDevices"));
    DeviceListManager::lockList();

    //Pointer Of Class CDevice
    CDevice* l_pCurDevice = NULL;
    unsigned int l_uiCount = 0;

    if(DEVLIST_STATETABLE == eDevListType)
    {
        //delete elements from the vector, if there are any
        while (!f_vConnectedDevices.empty())
        {
            f_vConnectedDevices.pop_back();
        }



        for (l_uiCount=0; l_uiCount < m_vConnectedDevices_UdevMgr.size() ; l_uiCount++)
        {
            ETG_TRACE_USR4(("getAllConnectedDevices:Device serial Id %s",
                    m_vConnectedDevices_UdevMgr[l_uiCount].m_cShortSerial.toStdString().c_str()));

            l_pCurDevice = &m_vConnectedDevices_UdevMgr[l_uiCount];

            f_vConnectedDevices.push_back(l_pCurDevice);
        }
        ETG_TRACE_COMP(("getAllConnectedDevices:l_uiCount:%d ",l_uiCount));


    }
    else
    {
        ETG_TRACE_FATAL((""));
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : getAllConnectedDevices"));
}

/*-----------------------------------------------------------------------------*
 * tVoid getAllConnectedDevices(DEVICELIST_Type eDevListType,                  *
 *              vector<structNotifyClients> &f_vConnectedDevices)              *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::getAllConnectedDevices(DEVICELIST_Type eDevListType,
        vector<structNotifyClients> &f_vConnectedDevices)
{
    ETG_TRACE_USR4(("Begin: GetAllConnectedDevices"));
    DeviceListManager::lockList();
    //Pointer Of Class CDevice
    structNotifyClients l_oCurDevice;

    unsigned int l_uiCount = 0;

    //------------------------
    // StateTable
    //------------------------
    if(DEVLIST_STATETABLE == eDevListType)
    {
        //delete elements from the vector, if there are any
        while (!f_vConnectedDevices.empty())
        {
            f_vConnectedDevices.pop_back();
        }

        ETG_TRACE_COMP(("GetAllConnectedDevices: m_vConnectedDevices.size()=%d",m_vConnectedDevices_StateTable.size()));
        //fill vector f_vConnectedDevices
        for (l_uiCount=0; l_uiCount < m_vConnectedDevices_StateTable.size() ; l_uiCount++)
        {
            l_oCurDevice = m_vConnectedDevices_StateTable[l_uiCount];

            if(   (USB_DEV_INTERNAL_APPLY_CONNECT        == l_oCurDevice.eConnectStatus)
                    ||(USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR == l_oCurDevice.eConnectStatus) )
            {
                ETG_TRACE_USR4(("GetAllConnectedDevices: intermediate internal state"));
            }
            else
            {
                ETG_TRACE_USR4(("GetAllConnectedDevices:%d,%d,%s",
                        ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType),
                        ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus),
                        m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));

                f_vConnectedDevices.push_back(l_oCurDevice);
            }
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : GetAllConnectedDevices"));
}

/*-----------------------------------------------------------------------------*
 * tVoid ListApplyConnectDevice(DEVICELIST_Type eDevListType,                  *
 *              tenDevManagerUSBHost eUsb,DEVICE_CONNECTSTATUS_Type eApply,    *
 *              structNotifyClients &f_oCurDevice)                             *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::ListApplyConnectDevice(DEVICELIST_Type eDevListType,
        tenDevManagerUSBHost eUsb,DEVICE_CONNECTSTATUS_Type eApply,
        structNotifyClients &f_oCurDevice)
{
    ETG_TRACE_USR4(("Begin: ListApplyConnectDevice"));
    DeviceListManager::lockList();
    unsigned int l_uiFoundIndex = 0;
    int  l_DeviceFound = DEVICEMANAGER_ERROR_NODEVICE;

    DVM_NORMAL_M_ASSERT(eUsb==f_oCurDevice.eUSB);

    //------------------------
    // StateTable
    //------------------------
    if(DEVLIST_STATETABLE == eDevListType)
    {
        for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
        {
            ETG_TRACE_USR4(("ListApplyConnectDevice: f_oCurDevice.cMountPoint                         %s",
                    f_oCurDevice.cMountPoint));
            ETG_TRACE_USR4(("ListApplyConnectDevice: m_vConnectedDevices[%2d].cMountPoint       %s",l_uiCount,
                    m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
            ETG_TRACE_USR4(("ListApplyConnectDevice: m_vConnectedDevices[%2d].eUSB    0x%x",l_uiCount,
                    (tInt)m_vConnectedDevices_StateTable[l_uiCount].eUSB));
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
            ETG_TRACE_SYS(("ListApplyConnectDevice: m_vConnectedDevices[%2d].m_eFSType         : %d",l_uiCount,
                    ETG_CENUM(CGlobalEnumerations::FILESYSTEM_TYPE_Type,m_vConnectedDevices_StateTable[l_uiCount].eFSType) ));
#endif


            //--------------------------
            //search to match mount point
            //--------------------------
            if(0 == strcmp(f_oCurDevice.cMountPoint,m_vConnectedDevices_StateTable[l_uiCount].cMountPoint) )
            {
                if(f_oCurDevice.eUSB == m_vConnectedDevices_StateTable[l_uiCount].eUSB)
                {
                    ETG_TRACE_USR2(("ListApplyConnectDevice:Device Present in list                                   "));
                    l_uiFoundIndex = l_uiCount;
                    l_DeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
                }
                else
                {
                    ETG_TRACE_USR4(("ListApplyConnectDevice: DEVICE in list but eUSB=USB%1d (f_oCurDevice.eUSB=%1d) does not match",(tInt)eUsb,(tInt)f_oCurDevice.eUSB));
                }
            }
            /*count number of connected devices independent if mount point matches or not
                    eDisconnect should not appear at all since such an element would be deleted from list
                    furthermore this thread waits on a queue thus it should be thread save*/
        }
        if(DEVICEMANAGER_DEVICE_DETECTED == l_DeviceFound)
        {
            ETG_TRACE_COMP(("------------%d------------",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eApply)));
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
            if (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
            {
                //-----------------------------------------
                //take previously set state malfunction in to account
                //-----------------------------------------
                if(  (USB_DEV_UNAVAIL_HW_MALFUNC           == m_vConnectedDevices_StateTable[l_uiFoundIndex].eConnectStatus)
                   ||(USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT == m_vConnectedDevices_StateTable[l_uiFoundIndex].eConnectStatus)
                  )
                {
                    if(eApply == USB_DEV_INTERNAL_APPLY_CONNECT)
                    {
                         ETG_TRACE_USR4(("[INFO]: ListApplyConnectDevice: USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC "));
                         eApply = USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC;
                    }

                }
            }
#endif
            m_vConnectedDevices_StateTable[l_uiFoundIndex].eConnectStatus = eApply;

            //---------------------
            //add to history (backtrace)
            //---------------------
            if(USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC == m_vConnectedDevices_StateTable[l_uiFoundIndex].eConnectStatus)
            {
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                HWMALFUNCTIONHISTORY.AddToHistory("ListApplyConnectDevice(...)",&(m_vConnectedDevices_StateTable[l_uiFoundIndex]));
#endif
            }

            f_oCurDevice.eConnectStatus = eApply; //to reach that trace in calling function shows result
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : ListApplyConnectDevice"));
}

/*-----------------------------------------------------------------------------*
 * unsigned int getSizeOfList(DEVICELIST_Type eDevListType)                    *
 *-----------------------------------------------------------------------------*/
unsigned int DeviceListManager::getSizeOfList(DEVICELIST_Type eDevListType) const
{
    unsigned int iSize = 0;

    DeviceListManager::lockList();
    switch(eDevListType)
    {

        case DEVLIST_STATETABLE:
            iSize=(unsigned int)m_vConnectedDevices_StateTable.size();
            break;
        case DEVLIST_UDEVMGR:
            iSize=(unsigned int)m_vConnectedDevices_UdevMgr.size();
            break;
        case DEVLIST_DEVICEMGRSERVICE:
            iSize=(unsigned int)m_vConnectedDevices_Service.size();
            break;
        case DEVLIST_UNDEFINED:     // case, no break
        default:
            // nothing to do. iSize is already initialized with 0
            break;

    }

    ETG_TRACE_COMP(( "getSizeOfList:  eDevListType =%d : len=%d", ETG_ENUM( DEVICELISTTYP, eDevListType ), iSize ));
    DeviceListManager::unlockList();
    return iSize;
}

/*---------------------------------------------------------------------------------------*
 * tBool isMassStorageDevice(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType)     *
 *---------------------------------------------------------------------------------------*/
tBool DeviceListManager::isMassStorageDevice(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType)
{
    tBool bRetVal = FALSE;

    switch(f_eDeviceType)
    {
    case CGlobalEnumerations::DTY_USB:
    case CGlobalEnumerations::DTY_SD:
    case CGlobalEnumerations::DTY_SD_INTERNAL:
        bRetVal = TRUE;
        break;
    default:
        break;
    }
    return bRetVal;

}

/*---------------------------------------------------------------------------------------*
 * tBool isDeviceConnected(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const *
 *---------------------------------------------------------------------------------------*/
tBool DeviceListManager::isDeviceConnected(IN uint16_t f_iVendorID,IN uint16_t f_iProductID,IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const
{
    //---------------------------------
    //bound to DEVLIST_DEVICEMGRSERVICE only!!
    //--------------------------------
    tBool bFound = FALSE;
    if(m_vConnectedDevices_Service.size() == 0)
    {
        ETG_TRACE_COMP(("isDeviceConnected:DEVLIST_DEVICEMGRSERVICE: empty list"));
    }
    else
    {

        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_Service.size()); i++)
        {

            if(f_eDeviceType == m_vConnectedDevices_Service[i].m_eDeviceType)
            {
                if(f_iVendorID == m_vConnectedDevices_Service[i].m_iVendorID &&
                  f_iProductID == m_vConnectedDevices_Service[i].m_iProductID)
                {
                    bFound = TRUE;
                }
            }
        }
    }
    return bFound;
}

/*-----------------------------------------------------------------------------*
 * tBool getElement(tUInt iIndex, structNotifyClients &device)                 *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::getElement(tUInt iIndex, structNotifyClients &device)
{
    tBool bRetVal = FALSE;
    DeviceListManager::lockList();
    if (iIndex < m_vConnectedDevices_StateTable.size())
    {
        bRetVal=bFillstructNotifyClients(&device,&m_vConnectedDevices_StateTable[iIndex]);

    }
    DeviceListManager::unlockList();
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tBool getElement(DEVICELIST_Type eDevListType,tInt iIndex, CDevice &device) *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::getElement(DEVICELIST_Type eDevListType,tUInt iIndex,
        CDevice &device)
{
    tBool bRetVal = FALSE;

    DeviceListManager::lockList();
    ETG_TRACE_USR4(("getElement: eDevListType=%d ",eDevListType));

    if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        if (iIndex < m_vConnectedDevices_Service.size())
        {
            device.FillDevice(m_vConnectedDevices_Service[iIndex]); //object
            bRetVal = TRUE;
        }
    }
    else if(DEVLIST_UDEVMGR== eDevListType)
    {
        if (iIndex < m_vConnectedDevices_UdevMgr.size())
        {
            device.FillDevice(m_vConnectedDevices_UdevMgr[iIndex]); //pointer
            bRetVal = TRUE;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: getElement: eDevListType=%d is not supported",eDevListType));
    }
    DeviceListManager::unlockList();
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tBool setElement(DEVICELIST_Type eDevListType,tInt iIndex, CDevice &device) *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::setElement(DEVICELIST_Type eDevListType,tUInt iIndex, const CDevice &device)
{
    tBool bRetVal = FALSE;

    DeviceListManager::lockList();
    ETG_TRACE_USR4(("setElement: eDevListType=%d ",eDevListType));

    if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        if (iIndex < m_vConnectedDevices_Service.size())
        {
            m_vConnectedDevices_Service[iIndex].FillDevice(device); //object
            bRetVal = TRUE;
        }
    }
    else if(DEVLIST_UDEVMGR== eDevListType)
    {
        if (iIndex < m_vConnectedDevices_UdevMgr.size())
        {
            m_vConnectedDevices_UdevMgr[iIndex].FillDevice(device); //pointer
            bRetVal = TRUE;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: setElement: eDevListType=%d is not supported",eDevListType));
    }
    DeviceListManager::unlockList();
    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tBool bFillstructNotifyClients(structNotifyClients *pNotifyClients_Dest,    *
 *              const structNotifyClients *pNotifyClients_Src)                 *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::bFillstructNotifyClients(
        structNotifyClients *pNotifyClients_Dest,
        const structNotifyClients *pNotifyClients_Src) const
{
    tBool bRet = TRUE;
    memset(pNotifyClients_Dest,0,sizeof(structNotifyClients)); //intialize
    if( (NULL!=pNotifyClients_Dest) && (NULL!=pNotifyClients_Src) )
    {
        pNotifyClients_Dest->eUSB           = pNotifyClients_Src->eUSB;
        pNotifyClients_Dest->eDeviceType    = pNotifyClients_Src->eDeviceType;
        pNotifyClients_Dest->eConnectStatus = pNotifyClients_Src->eConnectStatus;
        pNotifyClients_Dest->iVendorID      = pNotifyClients_Src->iVendorID;
        pNotifyClients_Dest->iProductID     = pNotifyClients_Src->iProductID;
        pNotifyClients_Dest->u64Size        = pNotifyClients_Src->u64Size;
        pNotifyClients_Dest->eDeviceSpeedUSB = pNotifyClients_Src->eDeviceSpeedUSB;
        ETG_TRACE_USR4(("bFillstructNotifyClients: pNotifyClients_Src->eDeviceUnsupportedReason=%d ",pNotifyClients_Src->eDeviceUnsupportedReason));
        pNotifyClients_Dest->eDeviceUnsupportedReason = pNotifyClients_Src->eDeviceUnsupportedReason;
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        pNotifyClients_Dest->eFSType          = pNotifyClients_Src->eFSType;
#endif //USE_FILESYSTEMTYPE_ALIAS_FSTY
#ifdef USE_TOTAL_USED_FREE_SIZE
        pNotifyClients_Dest->u64TotalSize_KB = pNotifyClients_Src->u64TotalSize_KB;
        pNotifyClients_Dest->u64UsedSize_KB  = pNotifyClients_Src->u64UsedSize_KB;
        pNotifyClients_Dest->u64FreeSize_KB  = pNotifyClients_Src->u64FreeSize_KB;
#endif
#ifdef USE_PARITIONNUMBER
       pNotifyClients_Dest->u8ValidPartitionNo = pNotifyClients_Src->u8ValidPartitionNo;
#endif

        /*additional error check to keep an eye on memory faults or to see if array size of dest string to slow*/
        if((unsigned int)strlen_r(pNotifyClients_Src->cSerialID)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cSerialID))  //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cSerialID)=%d  ",sizeof(pNotifyClients_Dest->cSerialID) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cSerialID)+1=%d  ",strlen_r(pNotifyClients_Src->cSerialID)+1    ));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cUSBPort)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cUSBPort))  //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cUSBPort)=%d  ",sizeof(pNotifyClients_Dest->cUSBPort) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cUSBPort)+1=%d  ",strlen_r(pNotifyClients_Src->cUSBPort)+1));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cDeviceVersion)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cDeviceVersion)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cDeviceVersion)=%d  ",sizeof(pNotifyClients_Dest->cDeviceVersion) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cDeviceVersion)+1=%d  ",strlen_r(pNotifyClients_Src->cDeviceVersion)+1));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cDeviceName)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cDeviceName)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cDeviceName)=%d  ",sizeof(pNotifyClients_Dest->cDeviceName) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cDeviceName)+1=%d  ",strlen_r(pNotifyClients_Src->cDeviceName)+1));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cAccessoryName)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cAccessoryName)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cAccessoryName)=%d  ",sizeof(pNotifyClients_Dest->cAccessoryName) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:strlen_r(pNotifyClients_Src->cAccessoryName)+1=%d  ",strlen_r(pNotifyClients_Src->cAccessoryName)+1));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cMountPoint)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cMountPoint)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cMountPoint)=%d  ",sizeof(pNotifyClients_Dest->cMountPoint) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cMountPoint)+1=%d  ",strlen_r(pNotifyClients_Src->cMountPoint)+1    ));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cManufacturer)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cManufacturer)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cManufacturer)=%d  ",sizeof(pNotifyClients_Dest->cManufacturer) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cManufacturer)+1=%d  ",strlen_r(pNotifyClients_Src->cManufacturer)+1    ));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cDevPath)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cDevPath)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cDevPath)=%d  ",sizeof(pNotifyClients_Dest->cDevPath) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[source]strlen_r(pNotifyClients_Src->cDevPath)+1=%d  ",strlen_r(pNotifyClients_Src->cDevPath)+1    ));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        if((unsigned int)strlen_r(pNotifyClients_Src->cAccessoryName2)/*number of characters of string*/+1 /*terminating '\0'*/ >=sizeof(pNotifyClients_Dest->cAccessoryName2)) //strlen: No Nullpointercheck ok because struct has array - see common/enums.h
        {
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:[dest  ]sizeof(pNotifyClients_Dest->cAccessoryName2)=%d  ",sizeof(pNotifyClients_Dest->cAccessoryName2) ));
            ETG_TRACE_FATAL(("ERROR: bFillstructNotifyClients:strlen_r(pNotifyClients_Src->cAccessoryName2)+1=%d  ",strlen_r(pNotifyClients_Src->cAccessoryName2)+1));
            bRet = FALSE;
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        /*copy strings*/
        /*target string*/                  /*source string*/                            /*max size of target string*/         /*1 byte space kept to be padded by srtncopy with '\0' to terminate string*/
        strncpy(pNotifyClients_Dest->cSerialID,     pNotifyClients_Src->cSerialID,     sizeof(pNotifyClients_Dest->cSerialID)     -1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cUSBPort,      pNotifyClients_Src->cUSBPort,      sizeof(pNotifyClients_Dest->cUSBPort)      -1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cDeviceVersion,pNotifyClients_Src->cDeviceVersion,sizeof(pNotifyClients_Dest->cDeviceVersion)-1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cDeviceName,   pNotifyClients_Src->cDeviceName,   sizeof(pNotifyClients_Dest->cDeviceName)   -1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cAccessoryName,pNotifyClients_Src->cAccessoryName,sizeof(pNotifyClients_Dest->cAccessoryName)-1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cMountPoint,   pNotifyClients_Src->cMountPoint,   sizeof(pNotifyClients_Dest->cMountPoint)   -1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        strncpy(pNotifyClients_Dest->cManufacturer, pNotifyClients_Src->cManufacturer, sizeof(pNotifyClients_Dest->cManufacturer) -1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        if((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_DEVPATH)))
        {
            strncpy(pNotifyClients_Dest->cDevPath,  pNotifyClients_Src->cDevPath,      sizeof(pNotifyClients_Dest->cDevPath)      -1);
        }
        strncpy(pNotifyClients_Dest->cDevNode,  pNotifyClients_Src->cDevNode,      sizeof(pNotifyClients_Dest->cDevNode)      -1);
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL))
        {
            pNotifyClients_Dest->u8TotalValidPartitons =(tU8)pNotifyClients_Src->u8TotalValidPartitons;
            strncpy(pNotifyClients_Dest->cAccessoryName2,pNotifyClients_Src->cAccessoryName2,sizeof(pNotifyClients_Dest->cAccessoryName2)-1);//lint !e522: (or direkt call to strncpy): Warning 522: prio3: Highest operation, a 'cast', lacks side-effects
        }
#endif//MSD_BIGGEST_PARTITON_FEATURE
    }
    else
    {
        bRet = FALSE;
        DVM_NORMAL_M_ASSERT_ALWAYS();
    }
    return bRet;
}

#ifdef REFACTOR_STATETABLE_OUTPUT
/*-----------------------------------------------------------------------------*
 * tVoid DeviceListManager:: UpdateList(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager:: UpdateList(OUT  DEVICELIST_Type eDevListTypeTarget,
                                         IN  DEVICELIST_Type eDevListTypeSource)
{
    ETG_TRACE_USR4(("Begin: UpdateList"));
    DeviceListManager::lockList();
    ETG_TRACE_USR4(("UpdateList: eDevListTypeTarget=%d eDevListTypeSource=%d",
                                ETG_ENUM(DEVICELISTTYP,eDevListTypeTarget),
                                ETG_ENUM(DEVICELISTTYP,eDevListTypeSource)));

    //serves StateTable to Service only
    if( (DEVLIST_DEVICEMGRSERVICE == eDevListTypeTarget) || (DEVLIST_STATETABLE == eDevListTypeSource) )
    {
        ETG_TRACE_FATAL(("UpdateList: Currently implemented to copy from StateTableList to ServiceList only "));


        CDevice oStateTableElem;
        unsigned int l_uiCount;

        //m_vConnectedDevices_Service = m_vConnectedDevices_StateTable; //currentl ynot possible because of different data types

        m_vConnectedDevices_Service.clear(); //check possible race conditions here

        for ( l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++) //content in state table
        {
            oStateTableElem.clear();
            if(bFillDevice(OUT &oStateTableElem, IN &(m_vConnectedDevices_StateTable[l_uiCount]) ))
            {
                m_vConnectedDevices_Service.push_back(oStateTableElem);
            }
            else
            {
                ETG_TRACE_FATAL(("UpdateList: bFillCDevice returned error"));
            }
        }



        ETG_TRACE_COMP(("UpdateList: updated: devices in list are: %d",m_vConnectedDevices_Service.size()));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: DEVLIST_DEVICEMGRSERVICE != eDevListType"));
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : UpdateList"));
}

/*-----------------------------------------------------------------------------*
 * tBool DeviceListManager::bCompareList(...)
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::bGetIsListEqual(IN  DEVICELIST_Type eDevListTypeOne,IN  DEVICELIST_Type eDevListTypeTwo)
{
    ETG_TRACE_FATAL(("Begin: bGetIsListEqual"));
    DeviceListManager::lockList();

    ETG_TRACE_USR4(("bGetIsListEqual: eDevListTypeOne=%d eDevListTypeTwo=%d",
                                ETG_ENUM(DEVICELISTTYP,eDevListTypeOne),
                                ETG_ENUM(DEVICELISTTYP,eDevListTypeTwo)));

    unsigned int l_uiCountService,l_uiCountStateTable, i;
    tBool bListEqual      = FALSE;
    tBool bCompareDetails = FALSE;
    tBool bCompareElement;
    tBool bElemsFoundInSecondList;
    size_t  iElemFoundInSecondList;
    tInt  iElementsEqualCompare;
    tInt  iElementsEqual;

    CDevice oStateTableElem;
    vector<unsigned int>  l_AlreadyFoundIndexArray;

    //serves StateTable to Service only
    if(   ((DEVLIST_DEVICEMGRSERVICE == eDevListTypeOne) || (DEVLIST_STATETABLE == eDevListTypeOne))
        &&((DEVLIST_DEVICEMGRSERVICE == eDevListTypeTwo) || (DEVLIST_STATETABLE == eDevListTypeTwo )))
    {

        //--------------------------------
        //if size not equal  then lists are different
        //--------------------------------
        if((m_vConnectedDevices_StateTable.size() > 0) &&(m_vConnectedDevices_Service.size()    > 0))
        {
            if(m_vConnectedDevices_StateTable.size() == m_vConnectedDevices_Service.size())
            {
                bCompareDetails = TRUE;
            }
        }
    }

    if(bCompareDetails)
    {
        iElemFoundInSecondList = 0;
        iElementsEqualCompare = 0;

        //---------------------------------------------------------------------------------
        //iterate each element of state table and check if the elemetn can be found in the Service list as well
        //---------------------------------------------------------------------------------
        for ( l_uiCountStateTable=0; l_uiCountStateTable< m_vConnectedDevices_StateTable.size();l_uiCountStateTable++)
        {
            if(bFillDevice(OUT &oStateTableElem, IN &(m_vConnectedDevices_StateTable[l_uiCountStateTable]) ))
            {
                bElemsFoundInSecondList = FALSE;
                for ( l_uiCountService=0; l_uiCountService< m_vConnectedDevices_Service.size();l_uiCountService++)
                {
                     iElementsEqual = 0;
                     //------------------------------
                     //check if elem has been found already
                     //------------------------------
                     bCompareElement = TRUE;
                     for(i = 0; i < l_AlreadyFoundIndexArray.size(); i++)
                     {
                         if(l_AlreadyFoundIndexArray[i] == l_uiCountService)
                         {
                             bCompareElement = FALSE;
                         }
                     }
                     //-----------------------------
                     //compare  element
                     //-----------------------------
                     if(bCompareElement)
                     {
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cSerialID         == oStateTableElem.m_cSerialID)      {iElementsEqual++;}/*1*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cUSBPort          == oStateTableElem.m_cUSBPort)       {iElementsEqual++;}/*2*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_eDeviceType       == oStateTableElem.m_eDeviceType)    {iElementsEqual++;}/*3*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cDeviceVersion    == oStateTableElem.m_cDeviceVersion) {iElementsEqual++;}/*4*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cDeviceName       == oStateTableElem.m_cDeviceName)    {iElementsEqual++;}/*5*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cAccessoryName    == oStateTableElem.m_cAccessoryName) {iElementsEqual++;}/*6*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cMountPoint       == oStateTableElem.m_cMountPoint)    {iElementsEqual++;}/*7*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_eConnectStatus    == oStateTableElem.m_eConnectStatus) {iElementsEqual++;}/*8*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_cManufacturer     == oStateTableElem.m_cManufacturer)  {iElementsEqual++;}/*9*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_iVendorID         == oStateTableElem.m_iVendorID)      {iElementsEqual++;}/*10*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_iProductID        == oStateTableElem.m_iProductID)     {iElementsEqual++;}/*11*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_u64Size           == oStateTableElem.m_u64Size)        {iElementsEqual++;}/*12*/
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
                         if(m_vConnectedDevices_Service[l_uiCountService].m_eFSType           == oStateTableElem.m_eFSType)        {iElementsEqual++;}/*13*/
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
                         if(m_vConnectedDevices_Service[l_uiCountService].m_u64TotalSize_KB   == oStateTableElem.m_u64TotalSize_KB){iElementsEqual++;}/*14*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_u64UsedSize_KB    == oStateTableElem.m_u64UsedSize_KB) {iElementsEqual++;}/*15*/
                         if(m_vConnectedDevices_Service[l_uiCountService].m_u64FreeSize_KB    == oStateTableElem.m_u64FreeSize_KB) {iElementsEqual++;}/*16*/
#endif
#ifdef USE_PARITIONNUMBER
                         if(m_vConnectedDevices_Service[l_uiCountService].m_iValidPartitionNo == oStateTableElem.m_iValidPartitionNo){iElementsEqual++;}/*17*/
#endif
                         if(m_vConnectedDevices_Service[l_uiCountService].m_eUSB              == oStateTableElem.m_eUSB)             {iElementsEqual++;}/*18*/

                         //List equal?
                         iElementsEqualCompare = 18;
     #ifndef USE_FILESYSTEMTYPE_ALIAS_FSTY
                         iElementsEqualCompare--;
     #endif
     #ifndef USE_TOTAL_USED_FREE_SIZE
                         iElementsEqualCompare--;
     #endif
     #ifndef USE_PARITIONNUMBER
                         iElementsEqualCompare--;
     #endif
                     }
                     //---------------------------------------------------------
                     //check all elements have been compared to equal and leave inner loop
                     //--------------------------------------------------------
                     if(iElementsEqual == iElementsEqualCompare)
                     {
                        bElemsFoundInSecondList = TRUE;
                        break;
                     }
                }

                /*elem of first list elem found in second list*/
                if(bElemsFoundInSecondList)
                {
                    iElemFoundInSecondList++;
                    //-----------------------------------------------------------------
                    //store index of already found element to prevent using it again in search above
                    //----------------------------------------------------------------
                    l_AlreadyFoundIndexArray.push_back(l_uiCountService); //mark already found element

                }
                /*elem of first list NOT found in second list*/
                else
                {
                    break;
                }

            }

        }

        //-----------------------------------------------
        //deduce result: all eelments of first list found in second one
        //-----------------------------------------------
        if(iElemFoundInSecondList == m_vConnectedDevices_StateTable.size())
        {
            bListEqual = TRUE;
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_FATAL(("End  : bGetIsListEqual: bListEqual = 0x%x",bListEqual));
    return bListEqual;
}


/*-----------------------------------------------------------------------------*
 * tBool bFillDevice(...)                           *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::bFillDevice( OUT CDevice* pCurDevice_Dest, IN const structNotifyClients* pCurDevice_Src) const
{
    tBool bRet = TRUE;
    if( (NULL!=pCurDevice_Dest) && (NULL!=pCurDevice_Src) )
    {
        pCurDevice_Dest->clear(); //intialize
        pCurDevice_Dest->m_cSerialID             = pCurDevice_Src->cSerialID;
        pCurDevice_Dest->m_cUSBPort              = pCurDevice_Src->cUSBPort;
        pCurDevice_Dest->m_eDeviceType           = pCurDevice_Src->eDeviceType;
        pCurDevice_Dest->m_cDeviceVersion        = pCurDevice_Src->cDeviceVersion;
        pCurDevice_Dest->m_cDeviceName           = pCurDevice_Src->cDeviceName;
        pCurDevice_Dest->m_cAccessoryName        = pCurDevice_Src->cAccessoryName;
        pCurDevice_Dest->m_cMountPoint           = pCurDevice_Src->cMountPoint;

        if((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_DEVPATH)))
        {
            pCurDevice_Dest->m_cDevpath          = pCurDevice_Src->cDevPath;
        }

        pCurDevice_Dest->m_cDevnode          = pCurDevice_Src->cDevNode;
        pCurDevice_Dest->m_eConnectStatus        = pCurDevice_Src->eConnectStatus;
        pCurDevice_Dest->m_cManufacturer         = pCurDevice_Src->cManufacturer;
        pCurDevice_Dest->m_iVendorID             = pCurDevice_Src->iVendorID;
        pCurDevice_Dest->m_iProductID            = pCurDevice_Src->iProductID;
        pCurDevice_Dest->m_u64Size               = pCurDevice_Src->u64Size;
        pCurDevice_Dest->m_DeviceSpeedUSB        = pCurDevice_Src->eDeviceSpeedUSB;
        pCurDevice_Dest->m_eDeviceUnsupportedReason = pCurDevice_Src->eDeviceUnsupportedReason;

#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        pCurDevice_Dest->m_eFSType = pCurDevice_Src->eFSType;
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
        pCurDevice_Dest->m_u64TotalSize_KB      = pCurDevice_Src->u64TotalSize_KB;
        pCurDevice_Dest->m_u64UsedSize_KB       = pCurDevice_Src->u64UsedSize_KB;
        pCurDevice_Dest->m_u64FreeSize_KB       = pCurDevice_Src->u64FreeSize_KB;
#endif
#ifdef USE_PARITIONNUMBER
        pCurDevice_Dest->m_iValidPartitionNo    = pCurDevice_Src->u8ValidPartitionNo;
#endif
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
        pCurDevice_Dest->m_iTotalValidPartitons = pCurDevice_Src->u8TotalValidPartitons;
        pCurDevice_Dest->m_cAccessoryName2      = pCurDevice_Src->cAccessoryName2;
#endif
        //care that connector is shown if VD_DVM_GET_CONNECTED ALL is called
        pCurDevice_Dest->m_eUSB          = pCurDevice_Src->eUSB;
    }
    else
    {
        bRet = FALSE;
        DVM_NORMAL_M_ASSERT_ALWAYS();
    }
    return bRet;
}


tBool DeviceListManager::bFillDevice( OUT structNotifyClients* pCurDevice_Dest, IN const CDevice* pCurDevice_Src) const
{
    tBool bRet = TRUE;
    memset(pCurDevice_Dest,0,sizeof(structNotifyClients)); //intialize
    if( (NULL!=pCurDevice_Dest) && (NULL!=pCurDevice_Src) )
    {
        snprintf (pCurDevice_Dest->cSerialID,      sizeof(pCurDevice_Dest->cSerialID),      "%s",  pCurDevice_Src->m_cSerialID.toStdString().c_str());
        snprintf (pCurDevice_Dest->cUSBPort,       sizeof(pCurDevice_Dest->cUSBPort),       "%s",  pCurDevice_Src->m_cUSBPort.toStdString().c_str());
        snprintf (pCurDevice_Dest->cDeviceVersion, sizeof(pCurDevice_Dest->cDeviceVersion), "%s",  pCurDevice_Src->m_cDeviceVersion.toStdString().c_str());
        snprintf (pCurDevice_Dest->cDeviceName,    sizeof(pCurDevice_Dest->cDeviceName),    "%s",  pCurDevice_Src->m_cDeviceName.toStdString().c_str());
        snprintf (pCurDevice_Dest->cAccessoryName, sizeof(pCurDevice_Dest->cAccessoryName), "%s",  pCurDevice_Src->m_cAccessoryName.toStdString().c_str());
        snprintf (pCurDevice_Dest->cMountPoint,    sizeof(pCurDevice_Dest->cMountPoint),    "%s",  pCurDevice_Src->m_cMountPoint.toStdString().c_str());
        snprintf (pCurDevice_Dest->cManufacturer,  sizeof(pCurDevice_Dest->cManufacturer),  "%s",  pCurDevice_Src->m_cManufacturer.toStdString().c_str());
        if((DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_NOTIFY_DEVPATH)))
        {
            snprintf (pCurDevice_Dest->cDevPath,  sizeof(pCurDevice_Dest->cDevPath),  "%s",  pCurDevice_Src->m_cDevpath.toStdString().c_str());
        }
        snprintf (pCurDevice_Dest->cDevNode,  sizeof(pCurDevice_Dest->cDevNode),  "%s",  pCurDevice_Src->m_cDevnode.toStdString().c_str());
        pCurDevice_Dest->eDeviceType     = pCurDevice_Src->m_eDeviceType;
        pCurDevice_Dest->eConnectStatus  = pCurDevice_Src->m_eConnectStatus;
        pCurDevice_Dest->iVendorID       = pCurDevice_Src->m_iVendorID;
        pCurDevice_Dest->iProductID      = pCurDevice_Src->m_iProductID;
        pCurDevice_Dest->u64Size         = pCurDevice_Src->m_u64Size;
        pCurDevice_Dest->eDeviceSpeedUSB  = pCurDevice_Src->m_DeviceSpeedUSB;
        pCurDevice_Dest->eDeviceUnsupportedReason = pCurDevice_Src->m_eDeviceUnsupportedReason;

#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        pCurDevice_Dest->eFSType = pCurDevice_Src->m_eFSType;
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
        pCurDevice_Dest->u64TotalSize_KB = pCurDevice_Src->m_u64TotalSize_KB;
        pCurDevice_Dest->u64UsedSize_KB  = pCurDevice_Src->m_u64UsedSize_KB;
        pCurDevice_Dest->u64FreeSize_KB  = pCurDevice_Src->m_u64FreeSize_KB;
#endif
#ifdef USE_PARITIONNUMBER
        pCurDevice_Dest->u8ValidPartitionNo = (tU8)pCurDevice_Src->m_iValidPartitionNo;
#endif
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
        pCurDevice_Dest->u8TotalValidPartitons = (tU8)pCurDevice_Src->m_iTotalValidPartitons;
        snprintf (pCurDevice_Dest->cAccessoryName2,      sizeof(pCurDevice_Dest->cAccessoryName2),      "%s",  pCurDevice_Src->m_cAccessoryName2.toStdString().c_str());

#endif

        //care that connector is shown if VD_DVM_GET_CONNECTED ALL is called
        pCurDevice_Dest->eUSB          = pCurDevice_Src->m_eUSB;
    }
    else
    {
        bRet = FALSE;
        DVM_NORMAL_M_ASSERT_ALWAYS();
    }
    return bRet;
}


tVoid DeviceListManager::vEraseFromListRemovedDevices(DEVICELIST_Type eDevListType)
{
    ETG_TRACE_USR4(("vEraseFromListRemovedDevices: eDevListType=%d", ETG_ENUM(DEVICELISTTYP,eDevListType)));

    unsigned int l_uiCount;
    tInt  uiIndexOfElem  = 0;
    tBool bSearch       = FALSE;
    tBool bRemovedEntry;

    //@todo: possible maintenance: one can reduce size of code here if only one list would be used
    if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        if( m_vConnectedDevices_Service.size() >0)
        {
            bSearch = TRUE;
        }
        while(bSearch)
        {
            bRemovedEntry = FALSE;
            for ( l_uiCount=0; l_uiCount< m_vConnectedDevices_Service.size();l_uiCount++)
            {
                if(USB_DEV_REMOVED_BY_USR == m_vConnectedDevices_Service[l_uiCount].m_eConnectStatus)
                {
                    uiIndexOfElem = (tInt)l_uiCount;
                    m_vConnectedDevices_Service.erase(m_vConnectedDevices_Service.begin() + uiIndexOfElem );
                    bRemovedEntry = TRUE;
                    break; //iter again since vector has changed
                }
            }

            if(!bRemovedEntry)
            {
                bSearch = FALSE;
            }
        }

    }
    else if(DEVLIST_STATETABLE == eDevListType)
    {
        if( m_vConnectedDevices_StateTable.size() > 0)
        {
            bSearch = TRUE;
        }
        while(bSearch)
        {
            bRemovedEntry = FALSE;
            for ( l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
            {
                if(USB_DEV_REMOVED_BY_USR == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
                {
                    uiIndexOfElem = (tInt)l_uiCount;
                    m_vConnectedDevices_StateTable.erase(m_vConnectedDevices_StateTable.begin() + uiIndexOfElem );
                    bRemovedEntry = TRUE;
                    break; //iter again since vector has changed

                }
            }

            if(!bRemovedEntry)  //if whole list has passed without erasing an element then work is done
            {
                bSearch = FALSE;
            }
        }

    }
    else if(DEVLIST_UDEVMGR == eDevListType)
    {
        ETG_TRACE_FATAL(("vEraseFromListRemovedDevices: Not implemented for DEVLIST_UDEVMGR"));
    }
    else
    {
        ETG_TRACE_FATAL(("vEraseFromListRemovedDevices: Not implemented"));
    }

    ETG_TRACE_USR4(("Begin: vEraseFromListRemovedDevices"));
}


#endif
/*-----------------------------------------------------------------------------*
 * tVoid vListAddRemoveDevice(DEVICELIST_Type eDevListType,CDevice l_oDevice)  *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::vListAddRemoveDevice(DEVICELIST_Type eDevListType,
        CDevice l_oDevice)
{
    ETG_TRACE_USR4(("Begin: vListAddRemoveDevice"));
    DeviceListManager::lockList();

    ETG_TRACE_USR4(("vListAddRemoveDevice: eDevListType=%d", ETG_ENUM(DEVICELISTTYP,eDevListType)));

    if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {


        unsigned int l_uiCount;
        tBool bFoundInList = FALSE;

        for ( l_uiCount=0; l_uiCount< m_vConnectedDevices_Service.size();l_uiCount++)
        {
            ETG_TRACE_USR4(("ListAddDevice: l_oDevice.m_cMountPoint                         %s",
                    l_oDevice.m_cMountPoint.toStdString().c_str()));
            ETG_TRACE_USR4(("ListAddDevice: m_vConnectedDevices_Service[%2d].m_cMountPoint                         %s",l_uiCount,
                    m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));

            if(l_oDevice.m_cMountPoint == m_vConnectedDevices_Service[l_uiCount].m_cMountPoint)    //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
            {
                ETG_TRACE_USR2(("ListAddDevice:Device Present in list                                   "));
                bFoundInList = TRUE;  //device detected in list
                break;
            }
            else
            {
                ETG_TRACE_USR2(("ListAddDevice:searching: device not yet found in list                            "));
            }
        }
        //-------------------------------------
        //device not found yet hence add it
        //-------------------------------------
        if (    (FALSE                    == bFoundInList)
                &&  (USB_DEV_REMOVED_BY_USR != l_oDevice.m_eConnectStatus))
        {
            ETG_TRACE_COMP(("------------ADD2LIST-USB%1d,%d,mount:%s-----------",
                    l_oDevice.m_eUSB,
                    ETG_ENUM(DVM_DEVICE_TYPE, l_oDevice.m_eDeviceType),
                            l_oDevice.m_cMountPoint.toStdString().c_str()));
            m_vConnectedDevices_Service.push_back(l_oDevice);

        }
        //-------------------------------------
        //remove it
        //-------------------------------------
        else if (    (TRUE                   == bFoundInList)
                && (USB_DEV_REMOVED_BY_USR == l_oDevice.m_eConnectStatus))
        {
            ETG_TRACE_COMP(("------------REMOVE from LIST-USB%1d,%d,mount:%s-----------",
                    l_oDevice.m_eUSB,
                    ETG_ENUM(DVM_DEVICE_TYPE, m_vConnectedDevices_Service[l_uiCount].m_eDeviceType),
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));
            m_vConnectedDevices_Service.erase(m_vConnectedDevices_Service.begin() + (tInt)l_uiCount );      //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
        }

        //-------------------------------------
        //adjust  connection state
        //-------------------------------------
        else if (    (TRUE                      == bFoundInList)
                  && (
                         (USB_DEV_UNAVAIL_BAT_LOWVOLT          == l_oDevice.m_eConnectStatus)
                       ||(USB_DEV_UNAVAIL_HW_MALFUNC           == l_oDevice.m_eConnectStatus)
                       ||(USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT == l_oDevice.m_eConnectStatus)
                       ||(USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE  == l_oDevice.m_eConnectStatus)
                     )
                )
        {
            ETG_TRACE_COMP(("------------ADJUST in  LIST-USB%1d,%d,mount:%s-----------",
                    l_oDevice.m_eUSB,
                    ETG_ENUM(DVM_DEVICE_TYPE, m_vConnectedDevices_Service[l_uiCount].m_eDeviceType),
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));

            m_vConnectedDevices_Service[l_uiCount].m_eConnectStatus = l_oDevice.m_eConnectStatus;

        }


        //m_lock.unlock();
        ETG_TRACE_COMP(("ListAddRemoveDevice: connected devices: %d",m_vConnectedDevices_Service.size()));
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: DEVLIST_DEVICEMGRSERVICE != eDevListType"));
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : ListAddRemoveDevice"));
}

/*-----------------------------------------------------------------------------*
 * tVoid lockList()                                                            *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::lockList()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    while(false == m_lock.lock(DEFAULT_LOCK_TIMOUT_SEC))
    {
         ETG_TRACE_FATAL(("[ERROR] DeviceListManager::lockList() timout:%d sec",DEFAULT_LOCK_TIMOUT_SEC));
         sleep(1);
    }

#endif
}

/*-----------------------------------------------------------------------------*
 * tVoid unlockList()                                                          *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::unlockList()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
    m_lock.unlock();
#endif
}

/*-----------------------------------------------------------------------------*
 * tVoid deleteElement(DEVICELIST_Type eDevListType,tInt iIndex)               *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::deleteElement(DEVICELIST_Type eDevListType,tUInt iIndex)
{
    DeviceListManager::lockList();

    tBool bRetVal = FALSE;

    ETG_TRACE_USR4(("deleteElement: eDevListType=%d ",eDevListType));

    if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        if (iIndex < m_vConnectedDevices_Service.size())
        {
            m_vConnectedDevices_Service.erase(m_vConnectedDevices_Service.begin() + (tInt)iIndex);    //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
            bRetVal = TRUE;
        }
    }
    else if(DEVLIST_UDEVMGR == eDevListType)
    {
        if (iIndex < m_vConnectedDevices_UdevMgr.size())
        {
            m_vConnectedDevices_UdevMgr.erase(m_vConnectedDevices_UdevMgr.begin() + (tInt)iIndex);    //lint !e1702 Info 1702: operator 'operator+' is both an ordinary function '...' and a member function '...'
            bRetVal = TRUE;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: deleteElement: eDevListType=%d is not supported",eDevListType));
    }


    DeviceListManager::unlockList();

    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tVoid deleteLists(tenDevManagerUSBHost eUsb)                                *
 *-----------------------------------------------------------------------------*/
tVoid  DeviceListManager::deleteLists(tenDevManagerUSBHost eUsb)
{
    tInt i,j;

    ETG_TRACE_COMP(("deleteLists eUsb = %d",eUsb));

    DeviceListManager::lockList();

    switch(eUsb)
    {

        case eUSB1:
        case eUSB2:
        case eUSBUndef:
        case eUSB3:
#ifdef USE_FOURTH_USB_CONNECTOR
        case eUSB4:
#endif
            for(i=((tInt)DEVLIST_UNDEFINED+1);i<(tInt)DEVLIST_ALL;i++)
            {
                _deleteList((DEVICELIST_Type)i, eUsb);
            }
            break;

        case eUSBAll:
            for(i=(tInt)eUSBUndef;i<(tInt)eUSBAll;i++)
            {
                for( j=(tInt)DEVLIST_UDEVMGR; j<(tInt)DEVLIST_ALL; j++)
                {
                    _deleteList((DEVICELIST_Type)j, (tenDevManagerUSBHost)i);
                }
            }
            break;
        default:
            break;
    }

    DeviceListManager::unlockList();
}

/*-----------------------------------------------------------------------------*
 * tVoid deleteList(DEVICELIST_Type eListType)                                 *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::deleteList(DEVICELIST_Type eListType)
{
    DeviceListManager::lockList();
    ETG_TRACE_COMP(("deleteList eListType = %d",eListType));
    for(tInt i = (tInt)eUSBUndef;i < ARRAYSIZEFORUSB; i++)
    {
        _deleteList(eListType, (tenDevManagerUSBHost)i);
    }

    DeviceListManager::unlockList();
}

/*-----------------------------------------------------------------------------*
 * tVoid deleteList(DEVICELIST_Type eListType, tenDevManagerUSBHost eUsb)      *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::_deleteList(DEVICELIST_Type eListType, tenDevManagerUSBHost eUsb)
{
    unsigned int l_uiCount;
    vector<CDevice>    vConnectedDevices_UdevMgr;
    vector<CDevice>    vConnectedDevices_Service;
    vector<structNotifyClients>    vConnectedDevices_StateTable;

    /*note here are similar passages if Lists can be reduced to a single one then this could be solved*/
    ETG_TRACE_USR4(("Begin: _deleteList"));
    ETG_TRACE_COMP(("_deleteList: eListType = %d USB%1d",eListType,eUsb));
    switch(eListType)
    {
        //-----------------------
        //UdevMgr
        //-----------------------
        case DEVLIST_UDEVMGR:

            //fill new list with those elements which should not be deleted
            for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_UdevMgr.size(); l_uiCount++)
            {
                /*store those which don't match*/
                if(m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB != eUsb)
                {
                    vConnectedDevices_UdevMgr.push_back(m_vConnectedDevices_UdevMgr[l_uiCount]);
                }
                else
                {
                    ETG_TRACE_COMP(("_deleteList:USB%1d MountPoint:%s",
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint.toStdString().c_str()));
                }
            }
            /*erase original*/
            m_vConnectedDevices_UdevMgr.clear();

            /*add previously stored elements which did not match the criteria to delete it*/
            for (l_uiCount = 0; l_uiCount < vConnectedDevices_UdevMgr.size(); l_uiCount++)
            {
                m_vConnectedDevices_UdevMgr.push_back(vConnectedDevices_UdevMgr[l_uiCount]);
            }

            /*erase content of temporaly vector*/
            vConnectedDevices_UdevMgr.clear();



            break;
            //-----------------------
            //StateTable
            //-----------------------
        case DEVLIST_STATETABLE:


            for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_StateTable.size(); l_uiCount++)
            {
                /*store those which don't match*/
                if(m_vConnectedDevices_StateTable[l_uiCount].eUSB != eUsb)
                {
                    vConnectedDevices_StateTable.push_back(m_vConnectedDevices_StateTable[l_uiCount]);
                }
                else
                {
                    ETG_TRACE_COMP(("_deleteList:USB%1d MountPoint:%s",
                            m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                            m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                }
            }
            /*erase original*/
            m_vConnectedDevices_StateTable.clear();

            /*add previously stored elements which did not match the criteria to delete it*/
            for (l_uiCount = 0; l_uiCount < vConnectedDevices_StateTable.size(); l_uiCount++)
            {
                m_vConnectedDevices_StateTable.push_back(vConnectedDevices_StateTable[l_uiCount]);
            }

            /*erase content of temporaly vector*/
            vConnectedDevices_StateTable.clear();
            break;

            //-----------------------
            //Service
            //-----------------------
        case DEVLIST_DEVICEMGRSERVICE:
            for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_Service.size(); l_uiCount++)
            {
                /*store those which don't match*/
                if(m_vConnectedDevices_Service[l_uiCount].m_eUSB != eUsb)
                {
                    vConnectedDevices_Service.push_back(m_vConnectedDevices_Service[l_uiCount]);
                }
                else
                {
                    ETG_TRACE_COMP(("_deleteList:USB%1d MountPoint:%s",
                            m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));
                }
            }
            /*erase original*/
            m_vConnectedDevices_Service.clear();

            /*add previously stored elements which did not match the criteria to delete it*/
            for (l_uiCount = 0; l_uiCount < vConnectedDevices_Service.size(); l_uiCount++)
            {
                m_vConnectedDevices_Service.push_back(vConnectedDevices_Service[l_uiCount]);
            }

            /*erase content of temporaly vector*/
            vConnectedDevices_Service.clear();

            break;

        case DEVLIST_UNDEFINED:
            break;
        default:
            break;
    }

    ETG_TRACE_USR4(("End  : _deleteList"));
}

/*-----------------------------------------------------------------------------*
 * tVoid ListAddDevice(DEVICELIST_Type eDevListType, CDevice  &device)         *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::ListAddDevice(DEVICELIST_Type eDevListType,
        const CDevice  &device)
{

    DeviceListManager::lockList();

    if(DEVLIST_UDEVMGR == eDevListType)
    {
        ETG_TRACE_USR4(("ListAddDevice: DEVLIST_UDEVMGR: m_cMountPoint:%s",device.m_cMountPoint.toStdString().c_str()));
        m_vConnectedDevices_UdevMgr.push_back(device);
    }
    else if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        ETG_TRACE_USR4(("ListAddDevice: DEVLIST_UDEVMGR: m_cMountPoint:%s",device.m_cMountPoint.toStdString().c_str()));
        m_vConnectedDevices_Service.push_back(device);
    }
    else if(DEVLIST_STATETABLE == eDevListType)
    {
        structNotifyClients oCurDevice_Dest;
        //memset(&oCurDevice_Dest,0,sizeof(structNotifyClients));
        bFillDevice(OUT &oCurDevice_Dest,IN &device);

        ETG_TRACE_USR4(("ListAddDevice: DEVLIST_STATETABLE: m_cMountPoint:%s",device.m_cMountPoint.toStdString().c_str()));
        m_vConnectedDevices_StateTable.push_back(oCurDevice_Dest);
    }

    DeviceListManager::unlockList();
}

/*-----------------------------------------------------------------------------*
 * tBool IsAlreadyConnected(GENSTRING f_cDeviceNode)                             *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::IsAlreadyConnected(GENSTRING f_cDeviceNode)
{
    //ETG_TRACE_USR4 (("Begin: IsAlreadyConnected"));
    tBool bFound = FALSE;

    DeviceListManager::lockList();

    if (f_cDeviceNode.isEmpty())
    {
        ETG_TRACE_ERR (("IsAlreadyConnected: [ERROR] f_cDeviceNode is EMPTY"));
    }
    else
    {
        ETG_TRACE_USR4(("IsAlreadyConnected: f_cDeviceNode: %s",f_cDeviceNode.toStdString().c_str()));
        for( unsigned int l_iDevCount = 0; l_iDevCount < m_vConnectedDevices_UdevMgr.size(); l_iDevCount++)
        {

            if (f_cDeviceNode.contains(m_vConnectedDevices_UdevMgr[l_iDevCount].m_cDevnode, CASE_INSENSITIVE)   //e.g. f_cDeviceNode: /dev/sda1 m_cDevnode: /dev/sda
                    || f_cDeviceNode == m_vConnectedDevices_UdevMgr[l_iDevCount].m_cDevnode)       //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'    //single partition  f_cDeviceNode: /dev/sda m_cDevnode: /dev/sda
            {

                ETG_TRACE_COMP(("IsAlreadyConnected: Device found in the list  l_iDevCount:%d/%d ", l_iDevCount,m_vConnectedDevices_UdevMgr.size()-1 ));
                ETG_TRACE_USR4(("IsAlreadyConnected: m_cDevnode (device)       :%s",m_vConnectedDevices_UdevMgr[l_iDevCount].m_cDevnode.toStdString().c_str()));
                ETG_TRACE_USR4(("IsAlreadyConnected: m_cBlockDevnode(partition):%s",m_vConnectedDevices_UdevMgr[l_iDevCount].m_cBlockDevnode.toStdString().c_str()));

                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_PARTITIONS_ALL))
                {

                    //e.g. f_cDeviceNode: /dev/sda1  m_cBlockDevnode: /dev/sda1   or else case /dev/sda2
                    //RTC-669924: If m_cBlockDevnode is empty then consider the device is present in list, since devicenode is already matching
                    if(m_vConnectedDevices_UdevMgr[l_iDevCount].m_cBlockDevnode.isEmpty() || (f_cDeviceNode == m_vConnectedDevices_UdevMgr[l_iDevCount].m_cBlockDevnode))
                    {
                        ETG_TRACE_COMP (("IsAlreadyConnected: BlockDevnode(partition) already in list or it is empty, m_cBlockDevnode = %s", m_vConnectedDevices_UdevMgr[l_iDevCount].m_cBlockDevnode.toStdString().c_str()));
                        bFound = TRUE;
                        break; //if matches don't continue to  check if already connected
                    }
                    else
                    {
                        ETG_TRACE_COMP(("IsAlreadyConnected: BlockDevnode(partition) not yet found in list - other partition"));
                        bFound = FALSE;
                    }
                }
                else
                {
                    bFound = TRUE;
                }

                //l_pFoundDevice = &m_vConnectedDevices_UdevMgr[l_iDevCount];
            }
            else
            {
                // Continue checking the list
            }
        } // for (all devices in m_vConnectedDevices)

        ETG_TRACE_COMP(("IsAlreadyConnected: %s  ", (bFound?"yes":"no")));
    } // if (f_cDeviceNode.isEmpty())

    //ETG_TRACE_USR4 (("End  : IsAlreadyConnected"));

    DeviceListManager::unlockList();

    //return l_pFoundDevice;
    return bFound;
}

/*-----------------------------------------------------------------------------*
 * void Set_eConnectStatus(DEVICELIST_Type eDevListType,                       *
 *              structNotifyClients *l_pCurDevice)                             *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::Set_eConnectStatus(DEVICELIST_Type eDevListType,   //@todo thing about changein gto reference
        structNotifyClients *l_pCurDevice)
{
    ETG_TRACE_USR4(("Begin: Set_eConnectStatus"));

    ETG_TRACE_COMP(("Set_eConnectStatus eDevListType = %d",ETG_ENUM(DEVICELIST_TYPE,eDevListType)));

    DeviceListManager::lockList();

    GENSTRING      f_cMountPoint;
    GENSTRING      l_curMountPoint;
    unsigned int l_iDevCount    = 0;

    if (NULL !=l_pCurDevice)
    {
        f_cMountPoint = l_pCurDevice->cMountPoint;

        switch(eDevListType)
        {
            case DEVLIST_UDEVMGR:

                /*search device in list with same mount point ofl_pCurDevice*/
                for (l_iDevCount = 0; l_iDevCount < m_vConnectedDevices_UdevMgr.size(); l_iDevCount++)
                {
                    if (m_vConnectedDevices_UdevMgr[l_iDevCount].m_cMountPoint.contains(f_cMountPoint,  CASE_INSENSITIVE))
                    {
                        ETG_TRACE_COMP(("Set_eConnectStatus:[UdevManager] Device found in the list (%s)",l_pCurDevice->cMountPoint));
                        ETG_TRACE_COMP(("Set_eConnectStatus:[UdevManager] l_pFoundDevice->m_eConnectStatus= %d (%s)",
                                ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,l_pCurDevice->eConnectStatus),l_pCurDevice->cMountPoint) );
                        m_vConnectedDevices_UdevMgr[l_iDevCount].m_eConnectStatus = l_pCurDevice->eConnectStatus;
                    }
                    else
                    {
                        // Do nothing
                    }
                } // for (all devices in m_vConnectedDevices)
                break;
            case DEVLIST_STATETABLE:
                /*search device in list with same mount point ofl_pCurDevice*/
                for (l_iDevCount = 0; l_iDevCount < m_vConnectedDevices_StateTable.size(); l_iDevCount++)
                {
                    l_curMountPoint = m_vConnectedDevices_StateTable[l_iDevCount].cMountPoint;
                    if (l_curMountPoint.contains(f_cMountPoint,  CASE_INSENSITIVE) && !strcmp(m_vConnectedDevices_StateTable[l_iDevCount].cSerialID,l_pCurDevice->cSerialID) )
                    {
                        ETG_TRACE_COMP(("Set_eConnectStatus:[StateTable] Device found in the list (%s)",l_pCurDevice->cMountPoint));
                        ETG_TRACE_COMP(("Set_eConnectStatus:[StateTable] l_pFoundDevice->m_eConnectStatus= %d (%s)",
                                ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,l_pCurDevice->eConnectStatus),l_pCurDevice->cMountPoint) );
                        m_vConnectedDevices_StateTable[l_iDevCount].eConnectStatus = l_pCurDevice->eConnectStatus;

#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
                        ETG_TRACE_SYS(("Set_eConnectStatus:[StateTable] m_eFSType         : %d",
                                ETG_CENUM(CGlobalEnumerations::FILESYSTEM_TYPE_Type, m_vConnectedDevices_StateTable[l_iDevCount].eFSType) ));
#endif

                    }
                    else
                    {
                        // Do nothing
                    }
                } // for (all devices in m_vConnectedDevices)
                break;
            case DEVLIST_DEVICEMGRSERVICE:                        //Roadmap 13035 OverTemperature
                /*search device in list with same mount point ofl_pCurDevice*/
                for (l_iDevCount = 0; l_iDevCount < m_vConnectedDevices_Service.size(); l_iDevCount++)
                {
                    if (m_vConnectedDevices_Service[l_iDevCount].m_cMountPoint.contains(f_cMountPoint,  CASE_INSENSITIVE))
                    {
                        ETG_TRACE_COMP(("Set_eConnectStatus:[Service] Device found in the list (%s)",l_pCurDevice->cMountPoint));
                        ETG_TRACE_COMP(("Set_eConnectStatus:[Service] l_pFoundDevice->m_eConnectStatus= %d (%s)",
                                ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,l_pCurDevice->eConnectStatus),l_pCurDevice->cMountPoint) );
                        m_vConnectedDevices_Service[l_iDevCount].m_eConnectStatus = l_pCurDevice->eConnectStatus;
                    }
                    else
                    {
                        // Do nothing
                    }
                } // for (all devices in m_vConnectedDevices)
                break;
            default:
                ETG_TRACE_FATAL(("ERROR: Set_eConnectStatus:[default]:not implemented"));
                break;
        }
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  : Set_eConnectStatus"));
}

/*-----------------------------------------------------------------------------*
 * void lockPlatformEvents(tenDevManagerUSBHost eUsb)                          *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::lockPlatformEvents(tenDevManagerUSBHost eUsb) const
{
    ETG_TRACE_COMP(("Begin:lockPlatformEvents"));
    DeviceListManager::lockList();
    bPlatformEventsLocked[eUsb] = TRUE;
    DeviceListManager::unlockList();
    ETG_TRACE_COMP(("End  :lockPlatformEvents"));
}

/*-----------------------------------------------------------------------------*
 * void unlockPlatformEvents(tenDevManagerUSBHost eUsb)                        *
 *-----------------------------------------------------------------------------*/
void DeviceListManager::unlockPlatformEvents(tenDevManagerUSBHost eUsb) const
{
    ETG_TRACE_COMP(("Begin:unlockPlatformEvents %1d",eUsb));
    DeviceListManager::lockList();
    bPlatformEventsLocked[eUsb] = FALSE;
    DeviceListManager::unlockList();
    ETG_TRACE_COMP(("End  :unlockPlatformEvents"));
}

/*-----------------------------------------------------------------------------*
 * tBool getPlatformEventsLocked(tenDevManagerUSBHost eUsb)                    *
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::getPlatformEventsLocked(tenDevManagerUSBHost eUsb) const
{
    tBool bRetVal = FALSE;
    ETG_TRACE_COMP(("Begin:getPlatformEventsLocked USB%1d",eUsb));

    DeviceListManager::lockList();

    if((eUsb >= eUSB1) && ((tInt)eUsb <ARRAYSIZEFORUSB))
    {
        bRetVal= bPlatformEventsLocked[eUsb];
    }
    //to observe traces in unervolt lab which are bound to low volt scenario
    if(TRUE == bRetVal)
    {
        ETG_TRACE_COMP(("getPlatformEventsLocked: TRUE"));
    }
    else
    {
        ETG_TRACE_USR4(("getPlatformEventsLocked: FALSE"));
    }

    DeviceListManager::unlockList();

    ETG_TRACE_COMP(("End  :getPlatformEventsLocked"));

    return bRetVal;
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceAllDevice2Logfile(FILE *l_pFileLog, DEVICELIST_Type eDevListType,*
 *              tenDevManagerUSBHost eUsb)                                     *
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::TraceAllDevice2Logfile(FILE *l_pFileLog,
        DEVICELIST_Type eDevListType,tenDevManagerUSBHost eUsb)
{
    ETG_TRACE_USR4(("Begin: TraceAllDevice2Logfile"));

    GENSTRING myText;
    int  iNumDevs = 0;


    //has to match to DEVICE_TYPE_Type!!
    GENSTRING deviceTypes[] =
           {"DTY_UNKNOWN",
            "DTY_USB",
            "DTY_SD",
            "DTY_IPOD",
            "DTY_IPHONE",
            "DTY_MTP",
            "DTY_BLUETOOTH",
            "DTY_MSZUNE",
            "DTY_IPAD",
            "DTY_HUB",
            "DTY_NOT_SUPPORTED",
            "DTY_CDROM",
            "DTY_SD_INTERNAL",
            "DTY_LASTELEM",
            };


    GENSTRING DevListType[] =
          { "DEVLIST_UNDEFINED",
            "DEVLIST_UDEVMGR",
            "DEVLIST_STATETABLE",
            "DEVLIST_DEVICEMGRSERVICE",
            "DEVLIST_ALL"};


    ETG_TRACE_FATAL(("[ok]: TraceAllDevice2Logfile: eUsb = %d eDevListType = %s",eUsb,DevListType[eDevListType].toStdString().c_str()));

    DeviceListManager::lockList();
    //m_lock.lock();
    //------------------------
    // StateTable
    //------------------------
    if(DEVLIST_STATETABLE == eDevListType)
    {
        fprintf(l_pFileLog,"[ok] ==================================->StateTable=======================================================\n");
        for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                myText = deviceTypes[ m_vConnectedDevices_StateTable[l_uiCount].eDeviceType ] +
                        " (" +
                        GENSTRING(m_vConnectedDevices_StateTable[l_uiCount].cMountPoint) +
                        ")";
                fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_StateTable[%2u]:USB%d eDeviceType:%s\n",
                        l_uiCount,
                        m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                        myText.toStdString().c_str());
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_StateTable[l_uiCount].eUSB)
                {
                    myText = deviceTypes[ m_vConnectedDevices_StateTable[l_uiCount].eDeviceType ] +
                            " (" +
                            GENSTRING(m_vConnectedDevices_StateTable[l_uiCount].cMountPoint) +
                            ")";

                    fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_StateTable[%2u]:USB%d eDeviceType:%s\n",
                            l_uiCount,
                            m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                            myText.toStdString().c_str());
                    iNumDevs++;
                }
            }
        }
        fprintf(l_pFileLog,"[ok] TraceAllDevice: => eUsb=%d:   %d   devices connected\n",eUsb,iNumDevs);
        fprintf(l_pFileLog,"[ok] ====================================StateTable<-=====================================================\n");
    }

    //------------------------
    // Service
    //------------------------
    else if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        fprintf(l_pFileLog,"[ok] ================================->vd_devicemanager_tclService_DeviceManager==================================================\n");
        for (tUInt l_uiCount = 0; l_uiCount < m_vConnectedDevices_Service.size();
                l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                myText = deviceTypes[m_vConnectedDevices_Service[l_uiCount].m_eDeviceType] +
                        " (" +
                        m_vConnectedDevices_Service[l_uiCount].m_cMountPoint +
                        ")";
                fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_Service[%2u]:USB%1d eDeviceType:%s\n",
                        l_uiCount,
                        m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                        myText.toStdString().c_str());
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_Service[l_uiCount].m_eUSB)
                {
                    myText = deviceTypes[m_vConnectedDevices_Service[l_uiCount].m_eDeviceType] +
                            " (" +
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint +
                            ")";

                    fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_Service[%2u]:USB%1d eDeviceType:%s\n",
                            l_uiCount,
                            m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                            myText.toStdString().c_str());
                    iNumDevs++;
                }
            }
        }

        fprintf(l_pFileLog,"[ok] TraceAllDevice: => eUSB=%d:   %d   devices connected\n",eUsb,iNumDevs);
        fprintf(l_pFileLog,"[ok] ==================================vd_devicemanager_tclService_DeviceManager<-=========================================================\n");

    }
    //------------------------
    // UdevManager
    //------------------------
    else if(DEVLIST_UDEVMGR == eDevListType)
    {
        fprintf(l_pFileLog,"[ok] ================================->UDevManager==================================================\n");

        for (unsigned int l_uiCount = 0; l_uiCount < m_vConnectedDevices_UdevMgr.size();
                l_uiCount++)
        {
            if(eUsb==eUSBUndef)
            {
                myText = deviceTypes[m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType] +
                        " (" +
                        m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint +
                        ")";
                fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_UdevMgr[%2u]:USB%1d eDeviceType:%s\n",
                        l_uiCount,
                        m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                        myText.toStdString().c_str());
                iNumDevs++;
            }
            else
            {
                if(eUsb==m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB)
                {
                    myText = deviceTypes[m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType] +
                            " (" +
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint +
                            ")";

                    fprintf(l_pFileLog,"[ok] TraceAllDevice: m_vConnectedDevices_UdevMgr[%2u]:USB%1d eDeviceType:%s\n",
                            l_uiCount,
                            m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                            myText.toStdString().c_str());
                    iNumDevs++;
                }
            }
        }

        fprintf(l_pFileLog,"[ok] TraceAllDevice: => eUsb=%d:   %d   devices connected\n",eUsb,iNumDevs);
        fprintf(l_pFileLog,"[ok] ================================UDevManager<-=========================================================\n");

    }
    DeviceListManager::unlockList();
    //m_lock.unlock();
    ETG_TRACE_USR4(("End  : TraceAllDevice2Logfile\n"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::setCardInfo(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::setCardInfo( IN const CDeviceCard& oDeviceCard )
{
    ETG_TRACE_USR4(("Begin: setCardInfo"));
    ETG_TRACE_USR4(("setCardInfo: Copy Card for Card %d into m_rDevicesCard", ETG_ENUM(DVVM_CARD_SLOT, oDeviceCard.m_eSlot) ));

    m_rDevicesCard[oDeviceCard.m_eSlot].FillDevice( oDeviceCard );
    ETG_TRACE_USR4(("End: setCardInfo"));
}

/*-----------------------------------------------------------------------------*
 * tVoid DeviceListManager::getCardInfo(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::getCardInfo( OUT CDeviceCard* poDeviceCard, IN tenSlot eSlot ) const
{
    ETG_TRACE_USR4(("Begin: getCardInfo"));
    ETG_TRACE_USR4(("getCardInfo: Copy Card info for Card %d from m_rDevicesCard[eSlot]", ETG_ENUM(DVVM_CARD_SLOT, poDeviceCard->m_eSlot) ));
    poDeviceCard->FillDevice( m_rDevicesCard[eSlot] );
    ETG_TRACE_USR4(("End: getCardInfo"));
}
#ifdef REFACTOR_STATETABLE_OUTPUT
/*-----------------------------------------------------------------------------*
 * tVoid DeviceListManager::SetStateAtConnectorDueToApply(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::CheckSetStateAtConnectorsDueToApplyAllDevices()
{
    ETG_TRACE_USR4(("Begin: SetStateAtConnectorDueToApply"));
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------

    //printf("CheckSetStateAtConnectorsDueToApplyAllDevices:m_vConnectedDevices_StateTable.size():%d\n",m_vConnectedDevices_StateTable.size());

    DeviceListManager::lockList();

    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
        switch(m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {
            case USB_DEV_INTERNAL_APPLY_CONNECT:
                m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_CONNECTED;
                m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForConnect = 0; //state changed hence reset then corresponding time
                break;
            case USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR:
                m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_REMOVED_BY_USR;
                m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForRemove = 0; //state changed hence reset then corresponding time
                break;
            //unused cases
            case USB_DEV_UNDEFINED:
            case USB_DEV_WARNING:
            case USB_DEV_CONNECTED:
            case USB_DEV_REMOVED_BY_USR:
            case USB_DEV_UNAVAIL_BAT_LOWVOLT:
            case USB_DEV_UNAVAIL_HW_MALFUNC:
            case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:
            case USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE:
                break;
            default:
                break;
        }
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  : SetStateAtConnectorDueToApply"));
}

/*-----------------------------------------------------------------------------*
 * tVoid DeviceListManager::SetStateAtConnectorDueToHWSignals(...)
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::SetStateAtConnectorsDueToHWSignalsAndVoltage(IN voltageFailure_Result_t (&f_result)[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin: SetStateAtConnectorDueToHWSignals"));
    tenDevManagerUSBHost eUsb;
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------

    //f_result might have one of these values:
    //USB_DEV_UNDEFINED                       (ignored)
    //USB_DEV_REMOVED_BY_USR            (ignored)
    //USB_DEV_CONNECTED                      (ignored)
    //USB_DEV_UNAVAIL_HW_MALFUNC    (used )
    //USB_DEV_UNAVAIL_BAT_LOWVOLT   (used)

    DeviceListManager::lockList();

    //printf("SetStateAtConnectorsDueToHWSignalsAndVoltage m_vConnectedDevices_StateTable.size():%d\n",m_vConnectedDevices_StateTable.size());

    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
#if defined(BUGFIX_DECOUPLE_SDCARD_PRM) || defined(BUGFIX_DECOUPLE_CDROM_PRM)
        if(eUSBUndef == m_vConnectedDevices_StateTable[l_uiCount].eUSB)
        {
            ETG_TRACE_USR4(("SetStateAtConnectorDueToHWSignals: Do not modify the connection status according to PRM signals, as eUSBUndef is not bond to PRM "));
            continue;
        }
#endif//defined(BUGFIX_DECOUPLE_SDCARD_PRM) || defined(BUGFIX_DECOUPLE_CDROM_PRM)
        switch(m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {
            //map state
            case USB_DEV_CONNECTED:
            case USB_DEV_UNAVAIL_BAT_LOWVOLT:
            case USB_DEV_UNAVAIL_HW_MALFUNC:
                eUsb = m_vConnectedDevices_StateTable[l_uiCount].eUSB;
                //printf("SetStateAtConnectorsDueToHWSignalsAndVoltage: eUSB:%d\n",eUsb);
                m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = f_result[eUsb].resultUSB;
                //printf("SetStateAtConnectorsDueToHWSignalsAndVoltage: f_result[eUsb].resultUSB:%d\n",f_result[eUsb].resultUSB);
                break;
            //nothing to do
            case USB_DEV_WARNING:                        /*unused state*/
            case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:   /*unused state*/
                ETG_TRACE_FATAL(("SetStateAtConnectorsDueToHWSignalsAndVoltage: unused state: %d",m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus ));
                break;
            case USB_DEV_INTERNAL_APPLY_CONNECT:         /*due to previous call of CheckSetStateAtConnectorsDueToApplyAllDevices this can't happen*/
            case USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR:  /*due to previous call of CheckSetStateAtConnectorsDueToApplyAllDevices this can't happen*/
                ETG_TRACE_FATAL(("SetStateAtConnectorsDueToHWSignalsAndVoltage: unexpected: %d",m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus ));
                break;
            case USB_DEV_UNDEFINED:
            case USB_DEV_REMOVED_BY_USR:
            case USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE:
                break;
            default:
                break;
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : SetStateAtConnectorDueToHWSignals"));

}

bool DeviceListManager::IsDeviceAtConnector(tenDevManagerUSBHost eUsb,DEVICELIST_Type eDevListType,const bool onlyPlayableDevice)
{
    ETG_TRACE_USR4(("Begin: IsDeviceAtConnector: eUSB%d",(int)eUsb));

    DeviceListManager::lockList();

    bool bRes = false;
    unsigned int uiNumOfdevsAtConnector = 0;
    if(DEVLIST_STATETABLE == eDevListType)
    {
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {
            if(eUsb == m_vConnectedDevices_StateTable[i].eUSB)
            {
                if(false == onlyPlayableDevice || ((CGlobalEnumerations::DTY_DVD_DRIVE != m_vConnectedDevices_StateTable[i].eDeviceType) && (CGlobalEnumerations::DTY_HUB != m_vConnectedDevices_StateTable[i].eDeviceType)))
                {
                    switch(m_vConnectedDevices_StateTable[i].eConnectStatus)
                    {
                        case USB_DEV_UNAVAIL_BAT_LOWVOLT:
                        case USB_DEV_UNAVAIL_HW_MALFUNC:
                        case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:
                        case USB_DEV_CONNECTED:
                            uiNumOfdevsAtConnector++;
                            break;
                        case USB_DEV_INTERNAL_APPLY_CONNECT:
                        case USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR:
                            ETG_TRACE_FATAL(("IsDeviceAtConnector: unexpected message"));
                            break;
                        case USB_DEV_UNDEFINED:
                        case USB_DEV_WARNING:
                        case USB_DEV_REMOVED_BY_USR:
                        case USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE: //CDDrive is not  bound to Usb Connectors
                            break;
                        default:
                            break;//nothing to do
                    }
                }
            }
        }
    }
    else if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_Service.size()); i++)
        {
            if(eUsb == m_vConnectedDevices_Service[i].m_eUSB)
            {
                switch(m_vConnectedDevices_Service[i].m_eConnectStatus)
                {
                    case USB_DEV_UNAVAIL_BAT_LOWVOLT:
                    case USB_DEV_UNAVAIL_HW_MALFUNC:
                    case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:
                    case USB_DEV_CONNECTED:
                        uiNumOfdevsAtConnector++;
                        break;
                    case USB_DEV_INTERNAL_APPLY_CONNECT:
                    case USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR:
                         ETG_TRACE_FATAL(("IsDeviceAtConnector: unexpected message"));
                         break;
                    case USB_DEV_UNDEFINED:
                    case USB_DEV_WARNING:
                    case USB_DEV_REMOVED_BY_USR:
                    case USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE: //CDDrive is not  bound to Usb Connectors
                        break;
                    default:
                        break;//Nothing to do
                }
            }
        }
    }

    if(uiNumOfdevsAtConnector > 0)
    {
        bRes = true;
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("IsDeviceAtConnector: 0x%x",bRes));

    return bRes;


}
void DeviceListManager::ShowList(DEVICELIST_Type eDevListType)
{
    tInt i;
    ETG_TRACE_USR4(("ShowList:Begin ---------------------------------------------------------------------------------------------------------------"));

    DeviceListManager::lockList();

    switch(eDevListType)
    {
        case DEVLIST_UDEVMGR:
            if(m_vConnectedDevices_UdevMgr.size() == 0)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList:DEVLIST_UDEVMGR: empty list"));
            }
            for (i = 0; i < static_cast<int>(m_vConnectedDevices_UdevMgr.size()); i++)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_cSerialID : %s",    i,m_vConnectedDevices_UdevMgr[i].m_cSerialID.toStdString().c_str()));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_cMountPoint: %s",   i,m_vConnectedDevices_UdevMgr[i].m_cMountPoint.toStdString().c_str()));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_eConnectStatus: %d",i,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,m_vConnectedDevices_UdevMgr[i].m_eConnectStatus)));
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_iNofValidPartitons   : %d",i,m_vConnectedDevices_UdevMgr[i].m_iTotalValidPartitons));
#endif
                ETG_TRACE_COMP(("======="));
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                /*
                printf("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_cSerialID : %s\n",    i,m_vConnectedDevices_UdevMgr[i].m_cSerialID.toStdString().c_str() );
                printf("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_cMountPoint: %s\n",   i,m_vConnectedDevices_UdevMgr[i].m_cMountPoint.toStdString().c_str());
                printf("DeviceListManagerShowList: m_vConnectedDevices_UdevMgr[%d].m_eConnectStatus: %d\n",i,m_vConnectedDevices_UdevMgr[i].m_eConnectStatus);
                printf("=======\n");
                */

#endif
            }
            break;
        case DEVLIST_STATETABLE:
            if(m_vConnectedDevices_StateTable.size() == 0)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList:DEVLIST_STATETABLE: empty list"));
            }
            for (i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cSerialID : %s",    i,m_vConnectedDevices_StateTable[i].cSerialID ));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cMountPoint: %s",   i,m_vConnectedDevices_StateTable[i].cMountPoint));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eConnectStatus: %d",i,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,m_vConnectedDevices_StateTable[i].eConnectStatus)));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eUSB          : %d",i,m_vConnectedDevices_StateTable[i].eUSB));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cDevPath: %s",   i,m_vConnectedDevices_StateTable[i].cDevPath));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cDevNode: %s",   i,m_vConnectedDevices_StateTable[i].cDevNode));
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u8ValidPartitionNo   : %d",i,m_vConnectedDevices_StateTable[i].u8TotalValidPartitons));
#endif

#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect           : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForRemove            : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_Malfunction               : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction));
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_HighRiskForMalfunction     : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_HighRiskForMalfunction));
#endif
#endif //TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].s32MalfunctionElapsedTime_sec              : %d",i,m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ConnectTimeStampForRetry              : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_ConnectTimeStampForRetry));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eDeviceType                               : %d",i,ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[i].eDeviceType)));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eDeviceUnsupportedReason   : %d",i,ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,m_vConnectedDevices_StateTable[i].eDeviceUnsupportedReason)));
                ETG_TRACE_COMP(("======="));
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cSerialID : %s\n",    i,m_vConnectedDevices_StateTable[i].cSerialID );
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].cMountPoint: %s\n",   i,m_vConnectedDevices_StateTable[i].cMountPoint);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eConnectStatus: %d\n",i,m_vConnectedDevices_StateTable[i].eConnectStatus);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].eUSB          : %d\n",i,m_vConnectedDevices_StateTable[i].eUSB);
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect           : %lu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForRemove            : %lu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_Malfunction               : %lu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction);
#else
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect           : %llu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForRemove            : %llu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove);
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_Malfunction               : %llu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction);

#endif
#endif
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_HighRiskForMalfunction    : %lu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_HighRiskForMalfunction);
#else
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].u64TimeStamp_HighRiskForMalfunction    : %llu\n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_HighRiskForMalfunction);
#endif
#endif
                printf("DeviceListManagerShowList: m_vConnectedDevices_StateTable[%d].s32MalfunctionElapsedTime_sec              : %d\n"  ,i,m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec);

                printf("=======\n");


#endif
            }
            break;
        case DEVLIST_DEVICEMGRSERVICE:
            if(m_vConnectedDevices_Service.size() == 0)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList:DEVLIST_DEVICEMGRSERVICE: empty list"));
            }
            for (i = 0; i < static_cast<int>(m_vConnectedDevices_Service.size()); i++)
            {
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_cSerialID : %s",     i,m_vConnectedDevices_Service[i].m_cSerialID.toStdString().c_str() ));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_cMountPoint: %s",    i,m_vConnectedDevices_Service[i].m_cMountPoint.toStdString().c_str()));
                ETG_TRACE_COMP(("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_eConnectStatus: %d",i,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,m_vConnectedDevices_Service[i].m_eConnectStatus)));
                ETG_TRACE_COMP(("======="));
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                /*
                printf("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_cSerialID : %s\n",     i,m_vConnectedDevices_Service[i].m_cSerialID.toStdString().c_str() );
                printf("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_cMountPoint: %s\n",    i,m_vConnectedDevices_Service[i].m_cMountPoint.toStdString().c_str());
                printf("DeviceListManagerShowList: m_vConnectedDevices_Service[%d].m_eConnectStatus: %d\n ",i,m_vConnectedDevices_Service[i].m_eConnectStatus);
                printf("=======\n");
                */


#endif
            }
            break;
        default:
            break;
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("ShowList:  End---------------------------------------------------------------------------------------------------------------\n"));
}


tVoid DeviceListManager::GetCopyDeviceListOfService(vector<CDevice>  &f_ListOfSendDevicestServices)
{
    for(int i = 0; i<static_cast<int>(m_vConnectedDevices_Service.size()); i++)
    {
        f_ListOfSendDevicestServices.push_back(m_vConnectedDevices_Service[i]);
    }
}




/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::UpdateMalfunctionInfoWithTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::UpdateMalfunctionInfoWithTimeStamp(IN tenDevManagerUSBHost eUsb,IN tS32 s32MalfunctionElapsedTime_sec)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //--------------------------------
    DeviceListManager::lockList();
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp:DEVLIST_STATETABLE: empty list"));
    }
    else
    {

        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {

            if(eUsb == m_vConnectedDevices_StateTable[i].eUSB)
            {

                ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp: m_vConnectedDevices_StateTable[%d].cSerialID : %s",  i,  m_vConnectedDevices_StateTable[i].cSerialID ));
                ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp: Hardware malfunction Elapsed time        : %d",s32MalfunctionElapsedTime_sec));
                m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec = s32MalfunctionElapsedTime_sec;
                ETG_TRACE_COMP(("======="));

            }
        }

    }
    DeviceListManager::unlockList();
}
/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::UpdateMalfunctionInfoWithTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::UpdateMalfunctionInfoWithTimeStamp(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,IN tS32 s32MalfunctionElapsedTime_sec)
{
    DeviceListManager::lockList();
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //--------------------------------
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp:DEVLIST_STATETABLE: empty list"));
    }
    else
    {
        switch(f_eDeviceType)
        {
        case CGlobalEnumerations::DTY_CDROM:
        case CGlobalEnumerations::DTY_CDDA:
        case CGlobalEnumerations::DTY_SD_INTERNAL:

            for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {
                if(eUSBUndef == m_vConnectedDevices_StateTable[i].eUSB)
                {

                    switch(m_vConnectedDevices_StateTable[i].eDeviceType)
                    {
                    case CGlobalEnumerations::DTY_CDROM:
                    case CGlobalEnumerations::DTY_CDDA:
                    case CGlobalEnumerations::DTY_SD_INTERNAL:
                        if(f_eDeviceType == m_vConnectedDevices_StateTable[i].eDeviceType)
                        {

                            m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec = s32MalfunctionElapsedTime_sec;
                            ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp: m_vConnectedDevices_StateTable[%d].cSerialID : %s",  i,  m_vConnectedDevices_StateTable[i].cSerialID ));
                            ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp: Hardware malfunction Elapsed time for OpticalDisc or SD internal          : %d",s32MalfunctionElapsedTime_sec));
                            ETG_TRACE_COMP(("======="));

                        }
                        break;
                    default:
                        break;
                    }
                }
            }

            break;
        default:
           ETG_TRACE_COMP(("UpdateMalfunctionInfoWithTimeStamp:Unexpected Device type %d ",ETG_ENUM(DVM_DEVICE_TYPE,f_eDeviceType)));
           break;
        }

    }
    DeviceListManager::unlockList();

}

/*-----------------------------------------------------------------------------*
 *tBool DeviceListManager::bIsRemovedCausedByMalfunction(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::bIsRemovedCausedByMalfunction(IN tenDevManagerUSBHost eUsb)
{
    tBool bHasMalfunction = FALSE;
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  bIsRemovedCausedByMalfunction"));
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:DEVLIST_STATETABLE: empty list"));
    }
    else
    {
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {

            if(eUsb == m_vConnectedDevices_StateTable[i].eUSB)
            {
                ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:eUsb : %d",ETG_ENUM(tenDevManagerUSBHost,m_vConnectedDevices_StateTable[i].eUSB)));
                ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:s32MalfunctionElapsedTime_sec: %d",m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec));
                if(MALFUNCTION_NO != m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec)
                {
                    bHasMalfunction = TRUE;
                    //if one device connected to USB port eusb has malfunction, we are assuming the other devices
                    //connected to the same port will have the same.
                    break;
                }
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

                //in this case, after the remove the a voltage disturbance is occurring for a very short time.
                else if(m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction)
                {
                    bHasMalfunction = TRUE;
                    ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:bHasMalfunction is TRUE for the port %d",ETG_ENUM(tenDevManagerUSBHost,eUsb)));
                    break;
                }
                ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove : %d",m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove));
                ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction : %d",m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction));
                ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect : %d",m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect));

#endif //TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
            }


        }
    }
    ETG_TRACE_COMP(("bIsRemovedCausedByMalfunction:m_vConnectedDevices_StateTable[%d] :Malfunction Status : %d",eUsb,ETG_ENUM(BOOL,bHasMalfunction) ));
    ETG_TRACE_USR4(("End:  bIsRemovedCausedByMalfunction"));
    return bHasMalfunction;
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::SetStateAtConnectorsDueToMalfunction(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::SetStateAtConnectorsDueToMalfunction(IN tenDevManagerUSBHost eusb,IN DEVICE_CONNECTSTATUS_E f_eConnectStatus)
{

    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  SetStateAtConnectorsDueToMalfunction eusb %d, %d",eusb,f_eConnectStatus))
    DeviceListManager::lockList();

        if(m_vConnectedDevices_StateTable.size() == 0)
        {
            ETG_TRACE_COMP(("SetStateAtConnectorsDueToMalfunction:DEVLIST_STATETABLE: empty list"));
        }
        else
        {
            for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {
                if(eusb == m_vConnectedDevices_StateTable[i].eUSB)
                {

                    switch(m_vConnectedDevices_StateTable[i].eConnectStatus)
                    {
                    case USB_DEV_CONNECTED:
                    case USB_DEV_UNAVAIL_HW_MALFUNC:
                    case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:

                        ETG_TRACE_USR4((" SetStateAtConnectorsDueToMalfunction Updating the  connection status  %d ",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,f_eConnectStatus)))
                        m_vConnectedDevices_StateTable[i].eConnectStatus = f_eConnectStatus;
                        break;
                    case USB_DEV_REMOVED_BY_USR:
                        if(USB_DEV_CONNECTED != f_eConnectStatus && USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED != f_eConnectStatus)
                        {
                            ETG_TRACE_USR4((" SetStateAtConnectorsDueToMalfunction USB_DEV_REMOVED_BY_USR Updating the  connection status  %d ",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,f_eConnectStatus)))
                            m_vConnectedDevices_StateTable[i].eConnectStatus     = f_eConnectStatus;
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                            //Clearing the Remove timestamp
                            m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove = 0;


#endif //TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                        }

                        break;



                    default:
                        break;

                    }


                }

            }
        }


    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  :  SetStateAtConnectorsDueToMalfunction"));

}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::SetStateAtConnectorsDueToMalfunction(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::SetStateAtConnectorsDueToMalfunction(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,IN DEVICE_CONNECTSTATUS_E f_eConnectStatus)
{
    ETG_TRACE_USR4(("Begin:  SetStateAtConnectorsDueToMalfunction"));
    DeviceListManager::lockList();
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    switch(f_eDeviceType)
    {
       case CGlobalEnumerations::DTY_CDROM:
       case CGlobalEnumerations::DTY_CDDA:
       case CGlobalEnumerations::DTY_SD_INTERNAL:
           ETG_TRACE_USR4(("Begin:  SetStateAtConnectorsDueToMalfunction"));
           if(m_vConnectedDevices_StateTable.size() == 0)
           {
               ETG_TRACE_COMP(("SetStateAtConnectorsDueToMalfunction:DEVLIST_STATETABLE: empty list"));
           }
           else
           {
               for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
               {
                   if(f_eDeviceType == m_vConnectedDevices_StateTable[i].eDeviceType)
                   {

                       switch(m_vConnectedDevices_StateTable[i].eConnectStatus)
                       {
                       case USB_DEV_CONNECTED:
                       case USB_DEV_UNAVAIL_HW_MALFUNC:
                       case USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT:

                           ETG_TRACE_USR4((" SetStateAtConnectorsDueToMalfunction Updating the  connection status  %d ",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,f_eConnectStatus)))
                           m_vConnectedDevices_StateTable[i].eConnectStatus = f_eConnectStatus;

                           break;
                       default:
                           break;
                        }
                     }
               }
            }
           break;
       default:
           ETG_TRACE_COMP(("SetStateAtConnectorsDueToMalfunction: Unexpected device type. %d: ",ETG_ENUM(DVM_DEVICE_TYPE,f_eDeviceType)))
                   break;

    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  :  SetStateAtConnectorsDueToMalfunction"));

}


/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::GetMalfunctionTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::GetMalfunctionTimeStampForDevices(IN tenDevManagerUSBHost eusb,OUT tS32 &s32MalfunctionElapsedTime_sec)
{
    tBool bFound = FALSE;
    //initializing timestamp
    s32MalfunctionElapsedTime_sec = MALFUNCTION_NO;
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
       ETG_TRACE_COMP(("GetMalfunctionTimeStampForDevices:DEVLIST_STATETABLE: empty list"));
    }
    else if(eusb != eUSBUndef)
    {
        //traverse and find out the right usb connector
       for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
       {
           if(eusb == m_vConnectedDevices_StateTable[i].eUSB)
           {
               s32MalfunctionElapsedTime_sec = m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec;
               bFound = TRUE;
               break;
           }
       }
    }
    else
    {
        ETG_TRACE_COMP(("GetMalfunctionTimeStampForDevices: Unexpected USB Connector %d ",ETG_ENUM(tenDevManagerUSBHost,eusb)));
    }
    return bFound;
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::GetMalfunctionTimeStampForDevices(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tBool DeviceListManager::GetMalfunctionTimeStampForDevices(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,OUT tS32 &s32MalfunctionElapsedTime_sec)
{
    tBool bFound = FALSE;
    //initializing timestamp
    s32MalfunctionElapsedTime_sec = MALFUNCTION_NO;
    switch(f_eDeviceType)
    {
       //is only applicable to optical disc devices and SD internal
       case CGlobalEnumerations::DTY_CDROM:
       case CGlobalEnumerations::DTY_CDDA:
       case CGlobalEnumerations::DTY_SD_INTERNAL:
           //---------------------------------
           //bound to DEVLIST_STATETABLE only!!
           //---------------------------------
           if(m_vConnectedDevices_StateTable.size() == 0)
           {
               ETG_TRACE_COMP(("GetMalfunctionTimeStampForDevices:DEVLIST_STATETABLE: empty list"));
           }
           else
           {
               for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
               {
                   //traverse and find out the right device type
                   switch(m_vConnectedDevices_StateTable[i].eDeviceType)
                   {
                   case CGlobalEnumerations::DTY_CDROM:
                   case CGlobalEnumerations::DTY_CDDA:
                   case CGlobalEnumerations::DTY_SD_INTERNAL:
                       if(f_eDeviceType == m_vConnectedDevices_StateTable[i].eDeviceType)
                       {
                           s32MalfunctionElapsedTime_sec = m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec;
                           bFound = TRUE;
                           ETG_TRACE_COMP(("GetMalfunctionTimeStampForDevices:u64MalfunctionDuration:m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec :%d",m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec));
                           break;
                       }
                   default:
                       break;
                   }
               }
           }
           break;
       default:
           ETG_TRACE_COMP(("GetMalfunctionTimeStampForDevices:Unexpected Device type %d ",ETG_ENUM(DVM_DEVICE_TYPE,f_eDeviceType)));
           break;
    }
    return bFound;
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::GetOpticalDiscDeviceType(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::GetOpticalDiscDeviceType(OUT CGlobalEnumerations::DEVICE_TYPE_Type &f_eDeviceType)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    f_eDeviceType = CGlobalEnumerations::DTY_UNKNOWN;
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("GetOpticalDiscDeviceType:DEVLIST_STATETABLE: empty list"));
    }
    else
    {
        //traverse and find out the right device type for Optical disc
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {
            switch(m_vConnectedDevices_StateTable[i].eDeviceType)
            {
            case CGlobalEnumerations::DTY_CDDA:
            case CGlobalEnumerations::DTY_CDROM:
                f_eDeviceType = m_vConnectedDevices_StateTable[i].eDeviceType;
                ETG_TRACE_COMP(("GetOpticalDiscDeviceType: Got the device type = %d",ETG_ENUM(DVM_DEVICE_TYPE,f_eDeviceType)));
                break;
            default:
                break;
            }
        }
    }


}
/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::vStoreMalfunctionTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::vStoreMalfunctionTimeStamp(IN tenDevManagerUSBHost eusb, IN tU64 f_u64MalfunctionTimeStamp)
{
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
    printf("vStoreMalfunctionTimeStamp: Begin\n");
#endif
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  vStoreMalfunctionTimeStamp "));

    DeviceListManager::lockList();
        if(m_vConnectedDevices_StateTable.size() == 0)
        {
            ETG_TRACE_COMP(("vStoreMalfunctionTimeStamp:DEVLIST_STATETABLE: empty list"));
        }
        else
        {
            ETG_TRACE_USR4(("vStoreMalfunctionTimeStamp m_vConnectedDevices_StateTable.size() %d",m_vConnectedDevices_StateTable.size()))
            for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {
                ETG_TRACE_USR4(("vStoreMalfunctionTimeStamp m_vConnectedDevices_StateTable[i].eConnectStatus %d",m_vConnectedDevices_StateTable[i].eConnectStatus))
                ETG_TRACE_USR4(("vStoreMalfunctionTimeStamp m_vConnectedDevices_StateTable[i].eUSB %d",m_vConnectedDevices_StateTable[i].eUSB))

                if(eusb == m_vConnectedDevices_StateTable[i].eUSB)
                {
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                    m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction = f_u64MalfunctionTimeStamp; //latest overwrites previous - ulabe tests will proove if this is ok
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                    HWMALFUNCTIONHISTORY.AddToHistory("vStoreMalfunctionTimeStamp(...)",&(m_vConnectedDevices_StateTable[i]));
#endif

#endif// TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

                }
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
               printf("vStoreMalfunctionTimeStamp: m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction : %lu \n",m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction);
#else
               printf("vStoreMalfunctionTimeStamp: m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction : %llu \n",m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction);
#endif
               printf("vStoreMalfunctionTimeStamp: m_vConnectedDevices_StateTable[i].eUSB : %d \n",m_vConnectedDevices_StateTable[i].eUSB);
               printf("vStoreMalfunctionTimeStamp: m_vConnectedDevices_StateTable[i].eConnectStatus : %d \n",m_vConnectedDevices_StateTable[i].eConnectStatus);
               printf("vStoreMalfunctionTimeStamp: ----------------------------------------------------- \n");
#else
               ETG_TRACE_USR4(("vStoreMalfunctionTimeStamp: m_vConnectedDevices_StateTable[%d].u64TimeStamp_Malfunction : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction));
               ETG_TRACE_USR4(("vStoreMalfunctionTimeStamp: -----------------------------------------------------"));
#endif

            }
        }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  :  vStoreMalfunctionTimeStamp"));


}


/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::vStoreHighRiskForMalfunctionTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
//Note: currently Optical disc and SD_Card internal do not used such mechanism hence this is bound to USB only
tVoid DeviceListManager::vStoreHighRiskForMalfunctionTimeStamp(IN tU64 f_u64HighRiskForMalfunctionTimeStamp)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  vStoreHighRiskForMalfunctionTimeStamp "));

    DeviceListManager::lockList();

    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("vStoreHighRiskForMalfunctionTimeStamp:DEVLIST_STATETABLE: empty list"));
    }
    else
    {
        ETG_TRACE_USR4(("vStoreHighRiskForMalfunctionTimeStamp m_vConnectedDevices_StateTable.size() %d",m_vConnectedDevices_StateTable.size()))
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {
            ETG_TRACE_USR4(("vStoreHighRiskForMalfunctionTimeStamp m_vConnectedDevices_StateTable[i].eConnectStatus %d",m_vConnectedDevices_StateTable[i].eConnectStatus))
            ETG_TRACE_USR4(("vStoreHighRiskForMalfunctionTimeStamp m_vConnectedDevices_StateTable[i].eUSB %d",m_vConnectedDevices_StateTable[i].eUSB))


#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
            //Note: eUSBUndef are bound to devices which arenot connected to USB
            //this function is currently use for device connected to USB only
            if(((int)eUSB1<= (int)m_vConnectedDevices_StateTable[i].eUSB) && ((int)m_vConnectedDevices_StateTable[i].eUSB < (int)eUSBAll))
            {
                m_vConnectedDevices_StateTable[i].u64TimeStamp_HighRiskForMalfunction = f_u64HighRiskForMalfunctionTimeStamp;//latest overwrites previous - ulabe tests will proove if this is ok
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                HWMALFUNCTIONHISTORY.AddToHistory("vStoreHighRiskForMalfunctionTimeStamp(...)",&(m_vConnectedDevices_StateTable[i]));
#endif
            }
#endif// TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

        }
    }

   DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  :  vStoreHighRiskForMalfunctionTimeStamp"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::vStoreConnectTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::vStoreConnectTimeStamp(const structNotifyClients& oCurDevice)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  vStoreConnectTimeStamp "));

    DeviceListManager::lockList();
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("vStoreConnectTimeStamp:DEVLIST_STATETABLE: empty list"));
    }
    else
    {

        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {
#ifdef USE_BUGFIX_AFTERMALFUNCTION_NEWMOUNTPOINT
            tBool bFoundInList = FALSE;
            if(0 == strcmp(oCurDevice.cSerialID,m_vConnectedDevices_StateTable[i].cSerialID))
            {
                if(0 == strcmp(oCurDevice.cMountPoint,m_vConnectedDevices_StateTable[i].cMountPoint))
                {
                    bFoundInList = TRUE;
                }
                else
                {
                   //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                   // workaround due to platform behaviour covered also in unit test
                   //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                   if(USB_DEV_UNAVAIL_HW_MALFUNC == m_vConnectedDevices_StateTable[i].eConnectStatus)
                   {
                       bFoundInList = TRUE;
                       ETG_TRACE_FATAL(("[WARNING] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                       ETG_TRACE_FATAL(("[WARNING] vStoreConnectTimeStamp(): after USB_DEV_UNAVAIL_HW_MALFUNC: New Mountpoint"));
                       ETG_TRACE_FATAL(("[WARNING] vStoreConnectTimeStamp(): old: %s",m_vConnectedDevices_StateTable[i].cMountPoint));
                       ETG_TRACE_FATAL(("[WARNING] vStoreConnectTimeStamp(): new: %s",oCurDevice.cMountPoint));
                       ETG_TRACE_FATAL(("[WARNING] vStoreConnectTimeStamp(): =>replace old by new in list "));
                       ETG_TRACE_FATAL(("[WARNING] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                       memcpy(m_vConnectedDevices_StateTable[i].cMountPoint,oCurDevice.cMountPoint,sizeof(m_vConnectedDevices_StateTable[i].cMountPoint));

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                      HWMALFUNCTIONHISTORY.AddToHistory("[WARNING] vStoreConnectTimeStamp(): after USB_DEV_UNAVAIL_HW_MALFUNC: New Mountpoint ",NULL);
#endif
                   }
                }
            }

            if(bFoundInList)
#else
            if(0 == strcmp(oCurDevice.cMountPoint,m_vConnectedDevices_StateTable[i].cMountPoint))
#endif
            {
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
                printf("vStoreConnectTimeStamp: oCurDevice.u64TimeStamp_ApplyForConnect: %lu \n",oCurDevice.u64TimeStamp_ApplyForConnect);
#else
                printf("vStoreConnectTimeStamp: oCurDevice.u64TimeStamp_ApplyForConnect: %llu \n",oCurDevice.u64TimeStamp_ApplyForConnect);
#endif
#else
                ETG_TRACE_USR4(("vStoreConnectTimeStamp: oCurDevice.u64TimeStamp_ApplyForConnect: %d",(tU32)oCurDevice.u64TimeStamp_ApplyForConnect));
#endif
                m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect = oCurDevice.u64TimeStamp_ApplyForConnect;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                HWMALFUNCTIONHISTORY.AddToHistory("vStoreConnectTimeStamp(...)",&(m_vConnectedDevices_StateTable[i]));
#endif
#endif// TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

                break;
            }
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
               printf("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect : %lu \n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect);
#else
               printf("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect : %llu \n",i,m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect);
#endif
               printf("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].eUSB : %d \n",i,m_vConnectedDevices_StateTable[i].eUSB);
               printf("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].eConnectStatus : %d \n",i,m_vConnectedDevices_StateTable[i].eConnectStatus);
#else
               ETG_TRACE_USR4(("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForConnect : (tU32)%d",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForConnect));
               ETG_TRACE_USR4(("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].eUSB                         : eUSB%d",i,m_vConnectedDevices_StateTable[i].eUSB));
               ETG_TRACE_USR4(("vStoreConnectTimeStamp: m_vConnectedDevices_StateTable[%d].eConnectStatus               : %d",i,ETG_CENUM(DEVICE_CONNECTSTATUS_Type,m_vConnectedDevices_StateTable[i].eConnectStatus)));
#endif

        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  :  vStoreConnectTimeStamp"));
}

/*-----------------------------------------------------------------------------*
 *tVoid DeviceListManager::vStoreRemoveTimeStamp(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
tVoid DeviceListManager::vStoreRemoveTimeStamp(const structNotifyClients& oCurDevice)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  vStoreRemoveTimeStamp "));
    DeviceListManager::lockList();
        if(m_vConnectedDevices_StateTable.size() == 0)
        {
            ETG_TRACE_COMP(("vStoreRemoveTimeStamp:DEVLIST_STATETABLE: empty list"));
        }
        else
        {
            for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {

                if(0 == strcmp(oCurDevice.cMountPoint,m_vConnectedDevices_StateTable[i].cMountPoint))
                {

#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                    m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove = oCurDevice.u64TimeStamp_ApplyForRemove;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                    HWMALFUNCTIONHISTORY.AddToHistory("vStoreRemoveTimeStamp(...)",&(m_vConnectedDevices_StateTable[i]));
#endif
#endif// TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
                   printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove : %lu ms\n",m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove);
#else
                   printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove : %llu ms\n",m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove);
#endif
                   printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].eUSB : %d \n",m_vConnectedDevices_StateTable[i].eUSB);
                   printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].eConnectStatus : %d \n",m_vConnectedDevices_StateTable[i].eConnectStatus);
#else
                   ETG_TRACE_USR4(("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[%d].u64TimeStamp_ApplyForRemove : (tU32)%d ms",i,(tU32)m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove));
                   ETG_TRACE_USR4(("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[%d].eUSB                            :eUSB%d",i,m_vConnectedDevices_StateTable[i].eUSB));
                   ETG_TRACE_USR4(("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[%d].eConnectStatus                 : %d",i,ETG_CENUM(DEVICE_CONNECTSTATUS_Type,m_vConnectedDevices_StateTable[i].eConnectStatus)));
#endif
                    break;
                }


            }
        }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  :  vStoreMalfunctionTimeStamp"));

}



#ifdef MSD_BIGGEST_PARTITON_FEATURE
/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::isAllPartitionsRetried(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tBool DeviceListManager::isAllPartitionsRetried()
{
    //store the Serial ids which are already searched
    vector<GenString>  l_vSerialIDs;
    vector<GenString>::const_iterator it;
    ETG_TRACE_COMP(("Inside isAllPartitionsRetried"));
    DeviceListManager::lockList();
    tBool bAllPartitionsRetried = FALSE;
    tU8 u8TotalValidMountedPartitions = 0;
    tBool bNeedSearch = TRUE;
    GenString l_cSerialID = "";
    for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
    {
        ETG_TRACE_COMP(("isAllPartitionsRetried..For cDeviceName = %s",m_vConnectedDevices_StateTable[i].cDeviceName));
       //traverse and find out the right device type
       if((TRUE == isMassStorageDevice(m_vConnectedDevices_StateTable[i].eDeviceType)) && !(getPlatformEventsLocked(m_vConnectedDevices_StateTable[i].eUSB)) && (m_vConnectedDevices_StateTable[i].eUSB != eUSBUndef))
       {
           if(m_vConnectedDevices_StateTable[i].u8TotalValidPartitons>1)
           {
                bNeedSearch = TRUE;
                 for(it = l_vSerialIDs.begin();it<l_vSerialIDs.end();it++)
                 {
                     if(*it == m_vConnectedDevices_StateTable[i].cAccessoryName2)
                     {
                         ETG_TRACE_COMP(("AccessoryName : %s is already searched. So no need to search it",m_vConnectedDevices_StateTable[i].cAccessoryName2));
                         bNeedSearch = FALSE;
                         break;
                     }

                 }
                 l_cSerialID = m_vConnectedDevices_StateTable[i].cAccessoryName2;
                 if(TRUE == bNeedSearch)
                 {
                     l_vSerialIDs.push_back(l_cSerialID);
                 }

                 if(GenString("") != l_cSerialID)
                 {
                     if(l_cSerialID == m_vConnectedDevices_StateTable[i].cAccessoryName2)
                     {
                         ETG_TRACE_COMP(("isAllPartitionsRetried..inside"));
                         if(m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus == RETRIAL_NEEDED)
                         {
                             m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus = RETRIAL_DONE;
                             ETG_TRACE_COMP(("isAllPartitionsRetried..m_vConnectedDevices_StateTable[i].cDeviceName = %s",m_vConnectedDevices_StateTable[i].cDeviceName));
                             ETG_TRACE_COMP(("isAllPartitionsRetried..m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus = %d",m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus));
                         }
                         else
                         {
                             m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus = RETRIAL_NEEDED;
                         }

                     }

                 }

           }
       }

    }
    /* Update the totalvalid partitions count based on mounted partitions(RETRIAL_DONE for all partitions)*/
       for(it = l_vSerialIDs.begin();it<l_vSerialIDs.end();it++)
       {
           ETG_TRACE_COMP(("isAllPartitionsRetried..Check if Update partitions count needed"));

           for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
           {
               if((*it == m_vConnectedDevices_StateTable[i].cAccessoryName2)  && (m_vConnectedDevices_StateTable[i].eUSB != eUSBUndef))
               {
                   ETG_TRACE_COMP(("isAllPartitionsRetried..For cDeviceName = %s",m_vConnectedDevices_StateTable[i].cDeviceName));
                   if(m_vConnectedDevices_StateTable[i].eMultiPartRetrialStatus == RETRIAL_DONE)
                   {
                       bAllPartitionsRetried = TRUE;
                       u8TotalValidMountedPartitions++;
                       l_cSerialID = m_vConnectedDevices_StateTable[i].cAccessoryName2;
                       ETG_TRACE_COMP(("isAllPartitionsRetried..u8TotalValidMountedPartitions = %d",u8TotalValidMountedPartitions));
                   }
                   else
                   {
                       ETG_TRACE_COMP(("isAllPartitionsRetried..Not all partitions mounted cDeviceName = %s",m_vConnectedDevices_StateTable[i].cDeviceName));
                       bAllPartitionsRetried = FALSE;
                       break;
                   }
               }
           }
           if(bAllPartitionsRetried)
            {
               ETG_TRACE_COMP(("isAllPartitionsRetried..All partitions mounted for l_cSerialID = %s",l_cSerialID));
               UpdatePartitionsCount(l_cSerialID,u8TotalValidMountedPartitions);
               u8TotalValidMountedPartitions = 0;
            }
       }

    ETG_TRACE_COMP(("Leaving isAllPartitionsRetried"));
    DeviceListManager::unlockList();
    return bAllPartitionsRetried;

}


/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::UpdatePartitionsCount(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tVoid DeviceListManager::UpdatePartitionsCount(GenString f_cSerialID,tU8 u8TotalValidMountedPartitions)
{

    ETG_TRACE_COMP(("Inside UpdatePartitionsCount"));

    for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
    {
        if((m_vConnectedDevices_StateTable[i].u8TotalValidPartitons>1) &&  (f_cSerialID == m_vConnectedDevices_StateTable[i].cAccessoryName2))
        {
            m_vConnectedDevices_StateTable[i].u8TotalValidPartitons = u8TotalValidMountedPartitions;
            if(1 == m_vConnectedDevices_StateTable[i].u8TotalValidPartitons)
            {
                if(USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED == m_vConnectedDevices_StateTable[i].eConnectStatus )
                {
                    m_vConnectedDevices_StateTable[i].eConnectStatus = USB_DEV_CONNECTED;
                    ETG_TRACE_COMP(("UpdatePartitionsCount..Updating connection status for SerialID: %s ",m_vConnectedDevices_StateTable[i].cAccessoryName2));
                }
            }
            ETG_TRACE_COMP(("UpdatePartitionsCount..m_vConnectedDevices_StateTable[i].u8TotalValidPartitons = %d",m_vConnectedDevices_StateTable[i].u8TotalValidPartitons));
        }

    }
    ETG_TRACE_COMP(("Leaving UpdatePartitionsCount"));
}

/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::PlatformHasAllPartitionsofDeviceNotified(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tBool DeviceListManager::PlatformHasAllPartitionsofDeviceNotified()
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    //Used to check whether all partitions of USB stick is notified from the platform
    //This is applicable for all USB based mass storage devices
    tBool bAllPartitionsNotified = TRUE;
    GenString cSerialID = "";
    //store the Serail ids which are already searched
    vector<GenString>  l_vSerialIDs;
    vector<GenString>::const_iterator it;
    tU8 u8TotalValidPartitons = 0;
    tBool bNeedSearch = TRUE;
    ETG_TRACE_COMP(("Inside PlatformHasAllPartitionsofDeviceNotified"));

    for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
    {
       //traverse and find out the right device type
       if(TRUE == isMassStorageDevice(m_vConnectedDevices_StateTable[i].eDeviceType))
       {

           u8TotalValidPartitons = 0;

           if(m_vConnectedDevices_StateTable[i].u8TotalValidPartitons>1)
           {
               bNeedSearch = TRUE;
               for(it = l_vSerialIDs.begin();it<l_vSerialIDs.end();it++)
               {
                   if(*it == m_vConnectedDevices_StateTable[i].cAccessoryName2)
                   {
                       ETG_TRACE_COMP(("AccessoryName : %s is already searched. So no need to search it",m_vConnectedDevices_StateTable[i].cAccessoryName2));
                       bNeedSearch = FALSE;
                       break;
                   }

               }
               if(FALSE == bNeedSearch)
               {
                  //We have already search for the alias name m_vConnectedDevices_StateTable[i].cAccessoryName2(which is the correct serial id of the device), so no need to search again
                  continue;
               }
               cSerialID = m_vConnectedDevices_StateTable[i].cAccessoryName2;
               //storing the accessory names into vector
               l_vSerialIDs.push_back(cSerialID);
               if(GenString("") != cSerialID)
               {
                   u8TotalValidPartitons = 1;
                   //searching from the next element in the list
                   for (int j = i+1; j < static_cast<int>(m_vConnectedDevices_StateTable.size()); j++)
                   {
                          if(isMassStorageDevice(m_vConnectedDevices_StateTable[j].eDeviceType) && cSerialID == m_vConnectedDevices_StateTable[j].cAccessoryName2)
                          {
                              u8TotalValidPartitons++;
                              ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified found %d partitions of device %s notified ",u8TotalValidPartitons,m_vConnectedDevices_StateTable[i].cAccessoryName2));

                          }
                   }
                   ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified (calculated) u8TotalValidPartitons %d ",u8TotalValidPartitons));
                    ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified (from BLK) m_vConnectedDevices_StateTable[i].u8TotalValidPartitons %d",m_vConnectedDevices_StateTable[i].u8TotalValidPartitons));
                   //Checking the partitions are notified or not
                   if(u8TotalValidPartitons >= m_vConnectedDevices_StateTable[i].u8TotalValidPartitons)
                   {
                       ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified All the %d partitions of device %s notified ",u8TotalValidPartitons,m_vConnectedDevices_StateTable[i].cAccessoryName2));

                       if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_MSD_BIGGEST_PARTITION))
                       {
                           ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified Calling the Biggest partition function for the device with the original Serial ID %s ",m_vConnectedDevices_StateTable[i].cAccessoryName2));
                           //call the update list function of the device with SerialID
                           FindBiggestPartitionfDeviceAndUpdateList(cSerialID);
                       }
                       if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_SORT_PARTITIONS_ALL))
                       {
                           ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified Calling the sorting function for the device with the original Serial ID %s ",m_vConnectedDevices_StateTable[i].cAccessoryName2));
                           SortPartitionsAndUpdateList(cSerialID);
                       }
                       break;

                       }
                   else
                   {
                       ETG_TRACE_COMP(("PlatformHasAllPartitionsofDeviceNotified not all the partitions notified %d",u8TotalValidPartitons));
                       bAllPartitionsNotified = FALSE;
                       break;

                   }
               }

           }
       }

    }
    ETG_TRACE_COMP(("Leaving PlatformHasAllPartitionsofDeviceNotified"));
    return bAllPartitionsNotified;
}

/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::FindBiggestPartitionfDeviceAndUpdateList(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tVoid DeviceListManager::FindBiggestPartitionfDeviceAndUpdateList(GenString f_cSerialID)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    //find the biggest partition with in the same USB stick from the list of devices
    //Update connect status of biggest partition to USB_DEV_CONNECTED
    //Also helps during the removal of the usb sticks with multiple partitions
    tU64 u64TotalSize_KB = 0;
    unsigned int l_uiCount;

    DEVICE_CONNECTSTATUS_E eConnectionStatus = USB_DEV_UNDEFINED;
    ETG_TRACE_COMP(("Inside FindBiggestPartitionfDeviceAndUpdateList"));

    DeviceListManager::lockList();

    if(GenString("") != f_cSerialID)
    {
        for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
        {
           //only care if device has more than one valid partitions
           if(m_vConnectedDevices_StateTable[i].u8TotalValidPartitons>1 &&  f_cSerialID == m_vConnectedDevices_StateTable[i].cAccessoryName2)
           {
               //Search for biggest partitions starts here.
               u64TotalSize_KB = m_vConnectedDevices_StateTable[i].u64TotalSize_KB;
               for (int j = i+1; j < static_cast<int>(m_vConnectedDevices_StateTable.size()); j++)
               {
                   if(f_cSerialID == m_vConnectedDevices_StateTable[j].cAccessoryName2)
                   {
                       if(USB_DEV_REMOVED_BY_USR == m_vConnectedDevices_StateTable[j].eConnectStatus)
                       {
                           //store the connection status so in the next iteration the connect status will be applied to all devices with same cAccessoryName2(which is the correct serial id of the device)
                           eConnectionStatus = USB_DEV_REMOVED_BY_USR;
                       }
                       if(u64TotalSize_KB < m_vConnectedDevices_StateTable[j].u64TotalSize_KB)
                       {
                           ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList Updating the Bigger partition  %d",m_vConnectedDevices_StateTable[j].u64TotalSize_KB));
                           ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList: m_vConnectedDevices_StateTable[%d].cMountPoint: %s",   j,m_vConnectedDevices_StateTable[j].cMountPoint));
                           u64TotalSize_KB = m_vConnectedDevices_StateTable[j].u64TotalSize_KB;
                       }
                   }
               }
               //no need to search again for the Serial ID f_cSerialID
               break;
           }
           ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList Biggest partition size after traversing the list of devices %u",u64TotalSize_KB));
        }

        //update the list
        //Devices with partitions in the same USB stick will marked as USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED if it is not the biggest partition
        //Devices with partitions in the same USB stick will marked as USB_DEV_REMOVED_BY_USER if either of partitions in the stick is having the USB_DEV_REMOVED_BY_USR connect status
        for (l_uiCount = 0; l_uiCount < m_vConnectedDevices_StateTable.size(); l_uiCount++)
        {
            //Check either of partitions in the same mass storage device is removed by the user. This information  is to update connect status of the other partitions
            if(USB_DEV_REMOVED_BY_USR == eConnectionStatus && f_cSerialID == m_vConnectedDevices_StateTable[l_uiCount].cAccessoryName2)
            {
              ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList: Updating the connect status to %d for the device with mount point %s",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eConnectionStatus),m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
              m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_REMOVED_BY_USR;
            }
            if(GenString(m_vConnectedDevices_StateTable[l_uiCount].cAccessoryName2) != f_cSerialID || u64TotalSize_KB == m_vConnectedDevices_StateTable[l_uiCount].u64TotalSize_KB )
            {
                //these are devices with either not part of Mass storage device with multiple partitions or other devices connected to the system
                if(u64TotalSize_KB == m_vConnectedDevices_StateTable[l_uiCount].u64TotalSize_KB && USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
                {
                    ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList: Device with biggest partition USB%1d has MountPoint:%s",
                        m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                        m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                    //update the Partition number to as this is the biggest supported partition.
                    m_vConnectedDevices_StateTable[l_uiCount].u8ValidPartitionNo =1;
                    m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_CONNECTED;

                }
            }
            else
            {
                //These are the partitions that are part of the same Mass storage device but has less storage size
                ETG_TRACE_COMP(("FindBiggestPartitionfDeviceAndUpdateList: The device with mount point %s will not be send to clients",m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
            }
        }
    }
    DeviceListManager::unlockList();
    ETG_TRACE_COMP(("Leaving FindBiggestPartitionfDeviceAndUpdateList"));
}
//TODO optimize the algorithm to sort the vector
/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::SortPartitionsAndUpdateList(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tVoid DeviceListManager::SortPartitionsAndUpdateList(GenString f_cSerialID)
{
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    //Sort the partitions of the device(if the device is a multi partition mass storage device such as SD and USB stick) with the same serial ID and change the connect status
    //Update connect status of partitions to USB_DEV_CONNECTED
    //Also helps during the removal of the usb sticks with multiple partitions
    DEVICE_CONNECTSTATUS_E eConnectionStatus = USB_DEV_UNDEFINED;
    //tU64 u64TotalSize_KB = 0;
    structNotifyClients l_TempDevice;
    ETG_TRACE_COMP(("Inside SortPartitionsAndUpdateList"));
    tU8 u8Maxindex = 0;

    DeviceListManager::lockList();
    if(GenString("") != f_cSerialID)
    {
        for (int i = 0; i < (static_cast<int>(m_vConnectedDevices_StateTable.size())); i++)
        {
           //only care if device has more than one valid partitions
           if(m_vConnectedDevices_StateTable[i].u8TotalValidPartitons>1 &&  f_cSerialID == m_vConnectedDevices_StateTable[i].cAccessoryName2)
           {
               //Search for biggest partitions starts here.
               //u64TotalSize_KB = m_vConnectedDevices_StateTable[i].u64TotalSize_KB;
               u8Maxindex = (tU8)i;

               for (int j = i+1; j < static_cast<int>(m_vConnectedDevices_StateTable.size()); j++)
               {
                   if(f_cSerialID == m_vConnectedDevices_StateTable[j].cAccessoryName2)
                   {
                       if(USB_DEV_REMOVED_BY_USR == m_vConnectedDevices_StateTable[j].eConnectStatus)
                       {
                           //store the connection status so in the next iteration the connect status will be applied to all devices with same cAccessoryName2(which is the correct serial id of the device)
                           eConnectionStatus = USB_DEV_REMOVED_BY_USR;
                       }
                       if(m_vConnectedDevices_StateTable[u8Maxindex].u64TotalSize_KB < m_vConnectedDevices_StateTable[j].u64TotalSize_KB)
                       {
                           ETG_TRACE_COMP(("SortPartitionsAndUpdateList Updating the Bigger partition  %d",m_vConnectedDevices_StateTable[j].u64TotalSize_KB));
                           ETG_TRACE_COMP(("SortPartitionsAndUpdateList: m_vConnectedDevices_StateTable[%d].cMountPoint: %s",   j,m_vConnectedDevices_StateTable[j].cMountPoint));
                           u8Maxindex = (tU8)j;
                       }
                   }
               }
           }
           if(i != u8Maxindex)
           {
                l_TempDevice    = m_vConnectedDevices_StateTable[i];
                m_vConnectedDevices_StateTable[i] = m_vConnectedDevices_StateTable[u8Maxindex];
                m_vConnectedDevices_StateTable[u8Maxindex] = l_TempDevice;
                ETG_TRACE_COMP(("SortPartitionsAndUpdateList m_vConnectedDevices_StateTable[u8Maxindex] index %d", u8Maxindex));
           }
        }

        //Devices with partitions in the same USB stick will marked as USB_DEV_REMOVED_BY_USER if either of partitions in the stick is having the USB_DEV_REMOVED_BY_USR connect status
        for (unsigned int l_uiCount = 0; l_uiCount < m_vConnectedDevices_StateTable.size(); l_uiCount++)
        {
            if(f_cSerialID == m_vConnectedDevices_StateTable[l_uiCount].cAccessoryName2)
            {
                if(USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus )
                {
                    m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_CONNECTED;
                }
                //Check either of partitions in the same mass storage device is removed by the user. This information  is to update connect status of the other partitions
                if(USB_DEV_REMOVED_BY_USR == eConnectionStatus)
                {
                    ETG_TRACE_COMP(("SortPartitionsAndUpdateList: Updating the connect status to %d for the device with mount point %s",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eConnectionStatus),m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
                    m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_REMOVED_BY_USR;
                }
            }
        }

    }

    DeviceListManager::unlockList();
    ETG_TRACE_COMP(("Leaving SortPartitionsAndUpdateList"));


}

#endif//MSD_BIGGEST_PARTITON_FEATURE

#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::ResetMalfunctionTimeStamp(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tVoid DeviceListManager::ResetMalfunctionTimeStamp(IN tenDevManagerUSBHost eusb )
{
    DeviceListManager::lockList();
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  ResetMalfunctionTimeStamp "));


        if(m_vConnectedDevices_StateTable.size() == 0)
        {
            ETG_TRACE_COMP(("ResetMalfunctionTimeStamp:DEVLIST_STATETABLE: empty list"));
        }
        else
        {
            for (int i = 0; i < static_cast<int>(m_vConnectedDevices_StateTable.size()); i++)
            {
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
      /*
      printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec : %d \n",m_vConnectedDevices_StateTable[i].s32MalfunctionElapsedTime_sec);
      printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].eUSB : %d \n",m_vConnectedDevices_StateTable[i].eUSB);
      printf("vStoreRemoveTimeStamp: m_vConnectedDevices_StateTable[i].eConnectStatus : %d \n",m_vConnectedDevices_StateTable[i].eConnectStatus);
      */
#endif

                if(eusb == m_vConnectedDevices_StateTable[i].eUSB)
                {

#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
                    if(m_vConnectedDevices_StateTable[i].u64TimeStamp_ApplyForRemove)
                    {
                        //reset the malfunction timestamp for the devices connected to port
                        ETG_TRACE_COMP(("ResetMalfunctionTimeStamp:Reset the malfunction time stamp for the device : %s",m_vConnectedDevices_StateTable[i].cSerialID));
                        m_vConnectedDevices_StateTable[i].u64TimeStamp_Malfunction = 0;
                    }
#endif// TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION


                }

            }
        }


    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  :  ResetMalfunctionTimeStamp"));

}
#endif//TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION


#endif
/*--------------------------------------------------------------------------------------*
 *tVoid DeviceListManager::SetStateAtConnectorsDueToOverTemperature(...) THREAD FUNCTION
 *--------------------------------------------------------------------------------------*/
tVoid DeviceListManager::SetStateAtConnectorsDueToOverTemperature()
{
    DeviceListManager::lockList();
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    ETG_TRACE_USR4(("Begin:  SetStateAtConnectorsDueToOverTemperature "));
    if(m_vConnectedDevices_StateTable.size() == 0)
    {
        ETG_TRACE_COMP(("SetStateAtConnectorsDueToOverTemperature:DEVLIST_STATETABLE: empty list"));
    }
    else
    {
        //Devices with Device type CDROM or CDDA the connect state will be set to USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE
        //if the connect status is USB_DEV_INTERNAL_APPLY_CONNECT
        //USB_DEV_INTERNAL_APPLY_CONNECT ==> USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE
        for (unsigned int l_uiCount = 0; l_uiCount < m_vConnectedDevices_StateTable.size(); l_uiCount++)
        {
            switch(m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)
            {
            case CGlobalEnumerations::DTY_CDDA:
            case CGlobalEnumerations::DTY_CDROM:
                if(USB_DEV_INTERNAL_APPLY_CONNECT == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
                {
                    ETG_TRACE_COMP(("SetStateAtConnectorsDueToOverTemperature:Update the connect state to USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE for the device %d",ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)));
                    m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE;

                }
            default:
                break;//nothing to do.
            }

        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End:  SetStateAtConnectorsDueToOverTemperature "));
}

#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
tVoid DeviceListManager::CheckSetStateTemporaryNotAvailableAllDevices(tU64 u64TestValue)
{
    INFORM_IF_TESTMODE((char*)"CheckSetStateTemporaryNotAvailableAllDevices",u64TestValue);

    ETG_TRACE_USR4(("Begin:  CheckSetStateTemporaryNotAvailableAllDevices "));

    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------

    DeviceListManager::lockList();

    tBool bCheckToAdjust,bMalfunctUsable,bHighRiskUsable;


    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
         bCheckToAdjust  = FALSE;
         bMalfunctUsable = FALSE;
         bHighRiskUsable = FALSE;

        //======
        //stage1
        //======
        //check for apply remove
        if(USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                printf("CheckSetStateTemporaryNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR\n",l_uiCount);
#else
                ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR",l_uiCount));
#endif

                bCheckToAdjust = TRUE;
        }

        //======
        //stage2
        //======
        //check if malfunction has happened so far
        if(bCheckToAdjust)
        {
             bCheckToAdjust = FALSE;
             if(0 < m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction)
             {
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                 printf("CheckSetStateTemporaryNotAvailableAllDevices: u64TimeStamp_Malfunction\n");
#else
                 ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: u64TimeStamp_Malfunction"));
#endif
                 bCheckToAdjust  = TRUE;
                 bMalfunctUsable = TRUE;
             }
             if(0 < m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction)
             {
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                 printf("CheckSetStateTemporaryNotAvailableAllDevices: u64TimeStamp_HighRiskForMalfunction");
#else
                 ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: u64TimeStamp_HighRiskForMalfunction"));
#endif
                 bCheckToAdjust  = TRUE;
                 bHighRiskUsable = TRUE;
             }
        }

        //======
        //stage3
        //======
        if(bCheckToAdjust)
        {
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
            HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateTemporaryNotAvailableAllDevices_1(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
#endif
            // please refer to section 5.1 in https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-730309/DESIGN_15003_HandlingOfTemporaryUnavailableDevices_05.doc
            //
            //                       r
            //                       e
            //                       m
            //                       o
            //                       v
            //                       e
            //     zerotime    |           nowtime
            //          |           |           |
            //      ->| Tprev | Tpost |<---
            //          |           |           |
            //
            // if Tmalf1 or Tmal2 has happened within above interval then set to temporary not available
            tU64  u64Tprev   = (tU64)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_USBCOMPARETIME_MALFUNCTION_OR_REMOVE); //remove at 0ms Tprev e.g. -400ms before
            tU64  u64Tpost   = (tU64)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_STATETABLE_EXECUTION_TIME_MS);
            tU64  u64Tnow    = ConfigurationManager::GetInstance()->GetEclapsedTime_ms(); if(u64TestValue>0){u64Tnow = u64TestValue;}//adjusted intest mode
            tU64  u64Tzero   = u64Tnow - u64Tprev - u64Tpost;

            if((u64Tnow > 0) && (u64Tzero > 0))
            {
                tU64  u64Tmal1   =  m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction;              //Note: if value is not set it would be 0
                tU64  u64Tmal2   =  m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction;   //Note: if value is not set it would be 0
                tU64  u64Remove  =  m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForRemove;

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
                printf("u64Tprev : %lu\n",u64Tprev);
                printf("u64Tpost : %lu\n",u64Tpost);
                printf("u64Tzero : %lu\n",u64Tzero);
                printf("u64Remove: %lu\n",u64Remove);
                printf("u64Tmal1 : %lu\n",u64Tmal1);
                printf("u64Tmal2 : %lu\n",u64Tmal2);
                printf("u64Tnow  : %lu\n",u64Tnow);
#else
                printf("u64Tprev : %llu\n",u64Tprev);
                printf("u64Tpost : %llu\n",u64Tpost);
                printf("u64Tzero : %llu\n",u64Tzero);
                printf("u64Remove: %llu\n",u64Remove);
                printf("u64Tmal1 : %llu\n",u64Tmal1);
                printf("u64Tmal2 : %llu\n",u64Tmal2);
                printf("u64Tnow  : %llu\n",u64Tnow);
#endif
#else
                ETG_TRACE_U64_COMP((char*)"u64Tprev : ",u64Tprev);
                ETG_TRACE_U64_COMP((char*)"u64Tpost : ",u64Tpost);
                ETG_TRACE_U64_COMP((char*)"u64Tzero : ",u64Tzero);
                ETG_TRACE_U64_COMP((char*)"u64Remove: ",u64Remove);
                ETG_TRACE_U64_COMP((char*)"u64Tmal1 : ",u64Tmal1);
                ETG_TRACE_U64_COMP((char*)"u64Tmal2 : ",u64Tmal2);
                ETG_TRACE_U64_COMP((char*)"u64Tnow  : ",u64Tnow);
#endif

                //normlize to zerotime - see comment above
                int Tzero   = 0;
                int Tremove = (int) (m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForRemove - u64Tzero);
                int Tmal1   = (int) (u64Tmal1 -u64Tzero); //expected to be >=0
                int Tmal2   = (int) (u64Tmal2 -u64Tzero); //expected to be >=0
                int Tnow    = (int) (u64Tprev + u64Tpost);

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                printf("Tzero  : %d\n" ,Tzero);
                printf("Tremove: %d\n",Tremove);
                printf("Tmal1  : %d\n",Tmal1);
                printf("Tmal2  : %d\n",Tmal2);
                printf("Tnow   : %d\n",Tnow);
#else
                ETG_TRACE_COMP(("Tzero  : %d",Tzero));
                ETG_TRACE_COMP(("Tremove: %d",Tremove));
                ETG_TRACE_COMP(("Tmal1  : %d",Tmal1));
                ETG_TRACE_COMP(("Tmal2  : %d",Tmal2));
                ETG_TRACE_COMP(("Tnow   : %d",Tnow));
#endif
                //error checks
                tBool bValid = TRUE;

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                if(Tzero   < 0){printf("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tzero:   %d\n",Tzero);}
                if(Tremove < 0){printf("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tremove: %d\n",Tremove);}
                if(Tnow    < 0){printf("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tnow:    %d\n",Tnow);}
                if((Tmal1  <=0) && (TRUE == bMalfunctUsable)){printf("[INFO] CheckSetStateTemporaryNotAvailableAllDevices:Tmal1:   %d\n",Tmal1);}
                if((Tmal2  <=0) && (TRUE == bHighRiskUsable)){printf("[INFO] CheckSetStateTemporaryNotAvailableAllDevices:Tmal2:   %d\n",Tmal2);}
#endif

                if(Tzero   < 0){bValid = FALSE; ETG_TRACE_FATAL(("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tzero:   %d",Tzero));}
                if(Tremove < 0){bValid = FALSE; ETG_TRACE_FATAL(("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tremove: %d",Tremove));}
                if(Tnow    < 0){bValid = FALSE; ETG_TRACE_FATAL(("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tnow:    %d",Tnow));}
                if((Tmal1  <=0)&& (TRUE == bMalfunctUsable)){bMalfunctUsable = FALSE; ETG_TRACE_FATAL(("[INFO] CheckSetStateTemporaryNotAvailableAllDevices:Tmal1:   %d",Tmal1));}
                if((Tmal2  <=0)&& (TRUE == bHighRiskUsable)){bHighRiskUsable = FALSE; ETG_TRACE_FATAL(("[INFO] CheckSetStateTemporaryNotAvailableAllDevices:Tmal2:   %d",Tmal2));}

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                if(Tzero    >= Tnow){ printf("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tzero   (%d) >= Tnow (%d)\n",Tzero,Tnow);}
                if(Tremove  >  Tnow){ printf("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tremove (%d) >= Tnow (%d)\n",Tremove,Tnow);}
#endif
                if(Tzero    >= Tnow){bValid = FALSE; ETG_TRACE_FATAL(("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tzero   (%d) >= Tnow (%d)",Tzero,Tnow));}
                if(Tremove  >  Tnow){bValid = FALSE; ETG_TRACE_FATAL(("[ERROR] CheckSetStateTemporaryNotAvailableAllDevices:Tremove (%d) >= Tnow (%d)",Tremove,Tnow));}

                //-------------
                //evaluation
                //-------------
                ETG_TRACE_USR4(("-------------------------------------------------------"));
                ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: bValid: 0x%x",bValid));

                if(bValid)
                {
                   ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: Tzero : %d",Tzero));
                   ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: Tnow  : %d",Tnow));
                   ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: Tmal1 : %d",Tmal1));
                   ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: Tmal2 : %d",Tmal2));

                   //--------------------------------
                   //evaluation malfunction
                   //--------------------------------
                   tBool bMalfunctionHasSet = FALSE;
                   if(TRUE == bMalfunctUsable)
                   {
                       if((Tmal1 >= Tzero) && (Tmal1 <= Tnow))
                       {
                           m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_UNAVAIL_HW_MALFUNC; //temporary not available
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForRemove         = 0; //since it has mapped to HW-MALFCT i.e. temporyry not available
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction            = 0; //reset time stamp
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction = 0; //reset time stamp
                           //=====================================
                           //start monitoring to map
                           // HW_MALFUNC to HW_MALFUNC_PERMANENT  i.e
                           //  'temp. not available'  to 'perm. not available'
                           //=====================================
                           m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec   = MALFUNCTION_YES; //now vMonitoringForMalfunctionPermanent increases this time to check for PERM not available
    #ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                           HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateTemporaryNotAvailableAllDevices_2(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
    #endif
                           bMalfunctionHasSet = TRUE;
                       }
                       else
                       {
                           ETG_TRACE_FATAL(("[WARNING]: Malfunction  not within Tprev+Tpos"));
                       }
                   }

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                   printf("vCASEenTimerExp(): bMalfunctUsable    : 0x%x\n",bMalfunctUsable);
                   printf("vCASEenTimerExp(): bHighRiskUsable    : 0x%x\n",bHighRiskUsable);
                   printf("vCASEenTimerExp(): bMalfunctionHasSet : 0x%x\n",bMalfunctionHasSet);
#else
                   ETG_TRACE_USR4(("vCASEenTimerExp(): bMalfunctUsable    : 0x%x",bMalfunctUsable));
                   ETG_TRACE_USR4(("vCASEenTimerExp(): bHighRiskUsable    : 0x%x",bHighRiskUsable));
                   ETG_TRACE_USR4(("vCASEenTimerExp(): bMalfunctionHasSet : 0x%x",bMalfunctionHasSet));
#endif



                   //--------------------------------
                   //evaluation high risk for malfunction
                   //--------------------------------
                   if((FALSE == bMalfunctionHasSet) && (TRUE == bHighRiskUsable)) //use check Tmal2 if Tmal1 did not set malfunction yet
                   {
                       if((Tmal2 >= Tzero) && (Tmal2 <= Tnow))
                       {
                           m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_UNAVAIL_HW_MALFUNC; //temporary not available
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ApplyForRemove         = 0; //since it has mapped to HW-MALFCT i.e. temporyry not available
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction            = 0; //reset time stamp
                           m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction = 0; //reset time stamp
                           //=====================================
                           //start monitoring to map
                           // HW_MALFUNC to HW_MALFUNC_PERMANENT  i.e
                           //  'temp. not available'  to 'perm. not available'
                           //=====================================
                           m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec           = MALFUNCTION_YES; //now vMonitoringForMalfunctionPermanent increases this time to check for PERM not available
    #ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                           HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateTemporaryNotAvailableAllDevices_3(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
    #endif
                       }
                       else
                       {
                           ETG_TRACE_FATAL(("[WARNING]: High Risc for Malfunction  not within Tprev+Tpos"));
                       }
                   }
                }
                else
                {
                    //hints to check configuration
                    //eCONF_USBCOMPARETIME_MALFUNCTION_OR_REMOVE &
                    //eCONF_STATETABLE_EXECUTION_TIME_MS
                    //or that platform events which notify malfunction ser send to slow in dependency with the 'received removed'
                    if(Tmal1<0)
                    {
                        ETG_TRACE_FATAL(("[WARNING]: CheckSetStateTemporaryNotAvailableAllDevices:Malfunct. before Tzero - reset u64TimeStamp_Malfunction"));
                        m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction    = 0;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                        HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateTemporaryNotAvailableAllDevices_4(...) Tmal1<0",&(m_vConnectedDevices_StateTable[l_uiCount]));
#endif
                    }
                    if(Tmal2<0)
                    {
                        if(bHighRiskUsable)
                        {
                            ETG_TRACE_FATAL(("[WARNING]: CheckSetStateTemporaryNotAvailableAllDevices:Malfunct. before Tzero - reset u64TimeStamp_HighRiskForMalfunction"));
                            m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction = 0;
    #ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                            HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateTemporaryNotAvailableAllDevices_5(...) Tmal2<0",&(m_vConnectedDevices_StateTable[l_uiCount]));
    #endif
                        }
                    }

                }
            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR] condition not valid. u64Tnow: %d u64Tzero: %d)",(unsigned int)u64Tnow, (unsigned int)u64Tzero ));
            }
            ETG_TRACE_USR4(("-------------------------------------------------------"));


        }
        else
        {
    #ifdef VARIANT_S_FTR_ENABLE_UNITTEST
            printf("CheckSetStateTemporaryNotAvailableAllDevices: No malfunction found");
    #else
            ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices: No malfunction found"));
    #endif
        }
    }


    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  : CheckSetStateTemporaryNotAvailableAllDevices"));
}
tVoid DeviceListManager::CheckSetStateConnectAfterTemporaryNotAvailableAllDevices()
{
    ETG_TRACE_USR4(("Begin:  CheckSetStateConnectAfterTemporaryNotAvailableAllDevices "));
    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    DeviceListManager::lockList();

    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
        //check for apply connect
        if(USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {
                ETG_TRACE_USR4(("CheckSetStateTemporaryNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC",l_uiCount));
                m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus            = USB_DEV_INTERNAL_APPLY_CONNECT;
                m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec = MALFUNCTION_NO; //stop monitor for permanent malfunction

                //Note:
                //these values have been resetted to 0 in the moment when a removed has been mapped to HW_MALUNCTION i.e. state temporary not available:
                //m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_Malfunction
                //m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_HighRiskForMalfunction
                // i.e. in case these values are set again a removed has not happened yet because then USB_DEV_INTERNAL_APPLY_REMOVED would be set in this list
                // but signals above either or both would be set then to a values > 0. Either or both becsuse this depends on the trigger from platfrom
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                HWMALFUNCTIONHISTORY.AddToHistory("vCASEenTimerExp(): CheckSetStateConnectAfterTemporaryNotAvailableAllDevices(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
#endif
        }
    }

    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  : CheckSetStateConnectAfterTemporaryNotAvailableAllDevices"));
}


tVoid DeviceListManager::CheckSetStatePermanentlyNotAvailableAllDevices()
{
    ETG_TRACE_USR4(("Begin:  CheckSetStatePermanentlyNotAvailableAllDevices "));

    //---------------------------------
    //bound to DEVLIST_STATETABLE only!!
    //---------------------------------
    DeviceListManager::lockList();

    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
        //check for apply connect
        if(USB_DEV_UNAVAIL_HW_MALFUNC == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {

            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: cSerialID: %s",m_vConnectedDevices_StateTable[l_uiCount].cSerialID));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: eDeviceType: %d",ETG_CENUM(CGlobalEnumerations::DEVICE_TYPE_Type,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType))); //ETG_CENUM works if no #ifdefs are in DEVICE_TYPE_Type anymore
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: cMountPoint: %s",m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d",m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_UNAVAIL_HW_MALFUNC",l_uiCount));

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
           printf("-------------------------------------------------------------------------------------------------------------------\n");
           printf("CheckSetStatePermanentlyNotAvailableAllDevices: cSerialID: %s\n",m_vConnectedDevices_StateTable[l_uiCount].cSerialID);
           printf("CheckSetStatePermanentlyNotAvailableAllDevices: eDeviceType: %d\n",m_vConnectedDevices_StateTable[l_uiCount].eDeviceType); //ETG_CENUM works if no #ifdefs are in DEVICE_TYPE_Type anymore
           printf("CheckSetStatePermanentlyNotAvailableAllDevices: cMountPoint: %s\n",m_vConnectedDevices_StateTable[l_uiCount].cMountPoint);
           printf("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d\n",m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec);
           printf("CheckSetStatePermanentlyNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_UNAVAIL_HW_MALFUNC\n",l_uiCount);
#endif


            tU32 u32MalfunctionMaxTime     = (tU32)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_TIMOUT_MALFUNCTION_SET_PERMANENT);
            tS32 s32MalfunctionElapsedTime_sec = m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec;

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
            printf("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d  (u32MalfunctionMaxTime: %d, MALFUNCTION_YES=%d)\n",s32MalfunctionElapsedTime_sec,u32MalfunctionMaxTime,MALFUNCTION_YES);
#endif

            if(s32MalfunctionElapsedTime_sec >= MALFUNCTION_YES)
            {
                ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d  (u32MalfunctionMaxTime: %d)",s32MalfunctionElapsedTime_sec,u32MalfunctionMaxTime));

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                printf("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d  (u32MalfunctionMaxTime: %d)\n",s32MalfunctionElapsedTime_sec,u32MalfunctionMaxTime);
#endif
                if(s32MalfunctionElapsedTime_sec > (tS32)u32MalfunctionMaxTime)
                {
                    m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus = USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
                   HWMALFUNCTIONHISTORY.AddToHistory ("vCASEenTimerExp(): CheckSetStatePermanentlyNotAvailableAllDevices(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
#endif
                }
            }
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("End  : CheckSetStatePermanentlyNotAvailableAllDevices"));
}


tBool DeviceListManager::bCheckIncrementMalfunctionEclapsedTimeAllDevices(tU16 u16Increment_sec)
{
    ETG_TRACE_USR4(("Begin:  bCheckIncrementMalfunctionEclapsedTimeAllDevices "));

    tBool bIncremented = FALSE;
    //----------------------------------------------------
    //function is bound to be used with DEVLIST_STATETABLE only!!
    //----------------------------------------------------
    DeviceListManager::lockList();

    //Interate all devices
    for (unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
        //check for those with USB_DEV_UNAVAIL_HW_MALFUNC
        if(USB_DEV_UNAVAIL_HW_MALFUNC == m_vConnectedDevices_StateTable[l_uiCount].eConnectStatus)
        {
        #ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
            HWMALFUNCTIONHISTORY.AddToHistory ("Monitor: bCheckIncrementMalfunctionEclapsedTimeAllDevices(...)",&(m_vConnectedDevices_StateTable[l_uiCount]));
        #endif
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: cSerialID: %s",m_vConnectedDevices_StateTable[l_uiCount].cSerialID));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: eDeviceType: %d",ETG_CENUM( CGlobalEnumerations::DEVICE_TYPE_Type,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType))); //ETG_CENUM works if no #ifdefs are in DEVICE_TYPE_Type anymore
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: cMountPoint: %s",m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d",m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec));
            ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_UNAVAIL_HW_MALFUNC",l_uiCount));

/*
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
            printf("------------------------------------------------------------------------------------\n");
            printf("CheckSetStatePermanentlyNotAvailableAllDevices: cSerialID: %s\n",m_vConnectedDevices_StateTable[l_uiCount].cSerialID);
            printf("CheckSetStatePermanentlyNotAvailableAllDevices: eDeviceType: %d\n",m_vConnectedDevices_StateTable[l_uiCount].eDeviceType); //ETG_CENUM works if no #ifdefs are in DEVICE_TYPE_Type anymore
            printf("CheckSetStatePermanentlyNotAvailableAllDevices: cMountPoint: %s\n",m_vConnectedDevices_StateTable[l_uiCount].cMountPoint);
            printf("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d\n",m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec);
            printf("CheckSetStatePermanentlyNotAvailableAllDevices m_vConnectedDevices_StateTable[%d]==USB_DEV_UNAVAIL_HW_MALFUNC\n",l_uiCount);
#endif
*/


            tU32 u32MalfunctionMaxTime_sec     = (tU32)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_TIMOUT_MALFUNCTION_SET_PERMANENT);
            tS32 s32MalfunctionElapsedTime_sec = (tU32)m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec;
            if(s32MalfunctionElapsedTime_sec >= MALFUNCTION_YES)
            {

                m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec += (tS32)u16Increment_sec; //reset value
                ETG_TRACE_USR4(("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %d  (u32MalfunctionMaxTime: %d)",
                                m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec,u32MalfunctionMaxTime_sec));
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
                printf("CheckSetStatePermanentlyNotAvailableAllDevices: s32MalfunctionElapsedTime_sec: %i  (u32MalfunctionMaxTime: %d)\n",
                                m_vConnectedDevices_StateTable[l_uiCount].s32MalfunctionElapsedTime_sec,u32MalfunctionMaxTime_sec);
#endif
                //------------------
                //Return value
                //------------------
                bIncremented = TRUE; //

            }
        }
    }


    DeviceListManager::unlockList();

    ETG_TRACE_USR4(("End  : bCheckIncrementMalfunctionEclapsedTimeAllDevices"));

    return bIncremented;
}

#endif


/*--------------------------------------------------------------------------------------*
*tBool DeviceListManager::isDeviceTypeConnected(...) THREAD FUNCTION
*--------------------------------------------------------------------------------------*/
tBool DeviceListManager::isDeviceTypeConnected(IN DEVICELIST_Type eDevListType,IN tenDevManagerUSBHost eUsb,IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const
{
    ETG_TRACE_USR4(("Begin:  isDeviceTypeConnected "));
    tBool bIsFound = FALSE;

    DeviceListManager::lockList();

    //------------------------
    // UdevNanger
    //------------------------
    if(DEVLIST_UDEVMGR == eDevListType)
    {
        for (unsigned int l_uiCount = 0; l_uiCount < m_vConnectedDevices_UdevMgr.size();l_uiCount++)
        {

           if(eUsb==m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB && f_eDeviceType == m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType)
           {
               ETG_TRACE_USR4(("[ok] isDeviceTypeConnected: m_vConnectedDevices_UdevMgr[%2d]:USB%1d eDeviceType:%d (%s)",
                           l_uiCount,
                           m_vConnectedDevices_UdevMgr[l_uiCount].m_eUSB,
                           ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_UdevMgr[l_uiCount].m_eDeviceType),
                           m_vConnectedDevices_UdevMgr[l_uiCount].m_cMountPoint.toStdString().c_str()));
               bIsFound = TRUE;
               break;
           }

        }
    }
    //------------------------
    // Statetable
    //------------------------
    else if(DEVLIST_STATETABLE == eDevListType)
    {
        for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
        {
            if(eUsb ==   m_vConnectedDevices_StateTable[l_uiCount].eUSB && f_eDeviceType == m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)
            {
                ETG_TRACE_USR4(("[ok] isDeviceTypeConnected: USB%d:m_vConnectedDevices_StateTable[%2d].cMountPoint:%s       ",
                        m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                        l_uiCount,
                        m_vConnectedDevices_StateTable[l_uiCount].cMountPoint));

                bIsFound = TRUE;
                break;
            }
        }

    }
    //------------------------
    // Service
    //------------------------
    else if(DEVLIST_DEVICEMGRSERVICE == eDevListType)
    {

        for (unsigned int  l_uiCount = 0; l_uiCount < m_vConnectedDevices_Service.size();l_uiCount++)
        {

                if(eUsb==m_vConnectedDevices_Service[l_uiCount].m_eUSB && f_eDeviceType ==  m_vConnectedDevices_Service[l_uiCount].m_eDeviceType)
                {
                    ETG_TRACE_USR4(("[ok] isDeviceTypeConnected: m_vConnectedDevices_Service[%2d]:USB%1d eDeviceType:%d (%s)",
                            l_uiCount,
                            m_vConnectedDevices_Service[l_uiCount].m_eUSB,
                            ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_Service[l_uiCount].m_eDeviceType),
                            m_vConnectedDevices_Service[l_uiCount].m_cMountPoint.toStdString().c_str()));
                    bIsFound = TRUE;
                    break;
                }
            }
        }


    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("isDeviceTypeConnected  %d",ETG_ENUM(BOOL,bIsFound)));


    ETG_TRACE_USR4(("End:  isDeviceTypeConnected  "));
    return bIsFound;

}

/*--------------------------------------------------------------------------------------*
*tBool DeviceListManager::isDeviceMultiPartitioned(...) THREAD FUNCTION
*--------------------------------------------------------------------------------------*/
tBool DeviceListManager::isDeviceMultiPartitioned(IN DEVICELIST_Type eDevListType, IN GENSTRING f_AccessoryName2) const
{
    ETG_TRACE_USR4(("Begin:  isDeviceMultiPartitioned "));
    tBool bIsFound = FALSE;

    DeviceListManager::lockList();

    if(DEVLIST_UDEVMGR == eDevListType)
    {
        int nPartitionCount = 1;
        for (unsigned int l_uiCount = 0; l_uiCount < m_vConnectedDevices_UdevMgr.size();l_uiCount++)
        {
            if(m_vConnectedDevices_UdevMgr[l_uiCount].m_cAccessoryName2 == f_AccessoryName2)
            {
                nPartitionCount++;
            }
            if(nPartitionCount > 1)
            {
                ETG_TRACE_USR4(("isDeviceMultiPartitioned ... Device is Multipartitioned"));
                bIsFound = TRUE;
                break;
            }
        }
    }

    DeviceListManager::unlockList();
    ETG_TRACE_USR4(("isDeviceMultiPartitioned  %d",ETG_ENUM(BOOL,bIsFound)));


    ETG_TRACE_USR4(("End:  isDeviceMultiPartitioned  "));
    return bIsFound;

}

/*--------------------------------------------------------------------------------------*
*tBool DeviceListManager::UpdateDeviceTypeIfRetryCompleted(...) THREAD FUNCTION
*--------------------------------------------------------------------------------------*/
tBool DeviceListManager::UpdateDeviceTypeIfRetryCompleted(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType, IN tU64 u64RetryElapsedTime_sec)
{
    ETG_TRACE_USR4(("Begin:  UpdateDeviceTypeIfRetryCompleted "));
    tBool bIsFound = FALSE;

    for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {
        ETG_TRACE_USR4(("[ok] %lu u64RetryElapsedTime_sec: %d  u64TimeStamp_ConnectTimeStampForRetry : %d ", ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType),u64RetryElapsedTime_sec,m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ConnectTimeStampForRetry));
        if( f_eDeviceType == m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)
        {
            if(m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ConnectTimeStampForRetry && u64RetryElapsedTime_sec)
            {
                if((u64RetryElapsedTime_sec - m_vConnectedDevices_StateTable[l_uiCount].u64TimeStamp_ConnectTimeStampForRetry) > (DEVICETYPE_RETRY_ELAPSED_TIMESEC*1000))
                {
                    DeviceListManager::lockList();
                    m_vConnectedDevices_StateTable[l_uiCount].eDeviceType = CGlobalEnumerations::DTY_MTP;
                    m_vConnectedDevices_StateTable[l_uiCount].eDeviceUnsupportedReason = OK;

                    ETG_TRACE_USR4(("[ok] UpdateDeviceTypeIfRetryCompleted: USB%d:m_vConnectedDevices_StateTable[%2d].eDeviceType:%d       ",
                            m_vConnectedDevices_StateTable[l_uiCount].eUSB,
                            l_uiCount,
                            ETG_ENUM(DVM_DEVICE_TYPE,m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)));
                    bIsFound = TRUE;
                    DeviceListManager::unlockList();

                }
            }
        }
    }

    ETG_TRACE_USR4(("UpdateDeviceTypeIfRetryCompleted  %d",ETG_ENUM(BOOL,bIsFound)));
    ETG_TRACE_USR4(("End:  UpdateDeviceTypeIfRetryCompleted  "));
    return bIsFound;
}

/*--------------------------------------------------------------------------------------*
*tBool DeviceListManager::vStopRetryForDevice(...) THREAD FUNCTION
*--------------------------------------------------------------------------------------*/
tBool DeviceListManager::vStopRetryForDevice(IN GENSTRING fSerialID)
{
    ETG_TRACE_USR4(("Begin:  vStopRetryForDevice "));
    ETG_TRACE_USR4(("vStopRetryForDevice  fSerialID to be stop retrying : %s ", fSerialID.toStdString().c_str()));
    tBool bIsFound = FALSE;

    for ( unsigned int l_uiCount=0; l_uiCount< m_vConnectedDevices_StateTable.size();l_uiCount++)
    {

        if(CGlobalEnumerations::DTY_NOT_SUPPORTED == m_vConnectedDevices_StateTable[l_uiCount].eDeviceType)
        {
            if( fSerialID.length() > 0  && (NULL != strcasestr(m_vConnectedDevices_StateTable[l_uiCount].cMountPoint,fSerialID.toStdString().c_str())))
            {
                DeviceListManager::lockList();
                m_vConnectedDevices_StateTable[l_uiCount].eDeviceType = CGlobalEnumerations::DTY_MTP;
                m_vConnectedDevices_StateTable[l_uiCount].eDeviceUnsupportedReason = OK;
                bIsFound = TRUE;
                ETG_TRACE_USR4(("[ok] vStopRetryForDevice: Stop retry done for device with serialID:%s ",
                                              m_vConnectedDevices_StateTable[l_uiCount].cSerialID));
                DeviceListManager::unlockList();

            }
        }

    }

    ETG_TRACE_USR4(("vStopRetryForDevice  %d",ETG_ENUM(BOOL,bIsFound)));
    ETG_TRACE_USR4(("End:  vStopRetryForDevice  "));
    return bIsFound;

}
////////////////////////////////////////////////////////////////////////////////
// EOF
