/*-----------------------------------------------------------------------------*
 * Device.h                                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file Device.h
 *
 * \brief data container to hold device information i.e. properties of the
 *        connected devices and it is used for notification to clients.
 *
 * \version Initial Version
 * \version 01.11.2011, Koechling, Christian (Bosch), statemachine supports
 *          several devices at one hub, product ID to charge apple and cleanup
 * \version 01.11.2011, Sunder Negi (Montavista), Disk storage details changed
 *          to unsigned long long from GENSTRING
 * \version 17.11.2011, Koechling, Christian (Bosch), cleanup: 1st step to shift
 *          includes to Common.h, merge of latest MV bugfixes from Sunder 9.11.2011
 * \version 06.12.2011, Koechling, Christian (Bosch), add send counter
 * \version 16.12.2011, Sunder Negi (Montavista), force mount added (some factory
 *          formated SD-cards)
 * \version 18.12.2011, Koechling, Christian (Bosch), cleanup config.h for
 *          'USE_NEW_MAPUSBSTRING'
 * \version 14.03.2012, Koechling, Christian (Bosch), cleanup define USE_BUGFIX_HUB
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 *          fixed traces again
 * \version 09.08.2012, Negi, Sunder (MontaVista), usb and port no are made integer
 * \version 13.08.2012, Negi, Sunder (MontaVista), added member to check if
 *          this device is being used for storing the traces
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup HANDLE_GPT
 * \version 28,08,2012, Negi, Sunder (MontaVista), change vendorid, productid to integers
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *         -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __CDEVICE_H__
#define __CDEVICE_H__

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define SUBSTRING_USBConnector1 "-1"
#define SUBSTRING_USBConnector2 "-2"

#define SUBSTRING_USBHub1 "/1-1."
#define SUBSTRING_USBHub2 "/1-2."

//->BUGFIX_DECOUPLE_SDCARD_PRM and BUGFIX_DECOUPLE_CDROM_PRM
#define USBConnectorUndef       0
#define USBConnectorUndefStr   "Undef"
//<-

#define USBConnector1   1
#define USBConnector2   2
#define USBConnector3   3
#ifdef USE_FOURTH_USB_CONNECTOR // Added to control the fourth USB connector
#define USBConnector4   4
#endif

#define USBConnector1str   "1"  /*means 1-1*/
#define USBConnector2str   "2"  /*means 1-2*/
#define USBConnector3str   "3"  /*means 1-3*/
#ifdef USE_FOURTH_USB_CONNECTOR // Added to control the fourth USB connector
#define USBConnector4str   "4"  /*means 1-4*/
#endif

#ifdef USE_FOURTH_USB_CONNECTOR // Added to control the fourth USB connector
#define USBConnectorLast USBConnector4
#else
#define USBConnectorLast USBConnector3
#endif

#define USBConnectorInvalid      -1
#define USBConnectorInvalidstr  "NONE"

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace std;
using namespace statetbl;


/*-----------------------------------------------------------------------------*
 * Forward declaration                                                                  *
 *-----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class CDevice
 *
 * \brief class to hold the properties of the device
 */

//lint -sem(CDevice::clear,initializer)
class CDevice
{
    private:

    public:
        // those marde with '[SENDTOCLIENTS]' will be send to clients please compare with FI
        GENSTRING m_cBlockDevnode;     ///< partition node like /dev/sdXY
        GENSTRING m_cDevpath;          ///< sys path
        GENSTRING m_cDevnode;          ///< device node
        GENSTRING m_cSysName;          ///< the usb name 1-2.3
        uint16_t  m_iVendorID;        ///< vendor ID                                                                                                           [SENDTOCLIENTS]
        uint16_t  m_iProductID;       ///< product ID                                                                                                          [SENDTOCLIENTS]
        GENSTRING m_cSerialID;         ///< serial number                                                                                                   [SENDTOCLIENTS]
        GENSTRING m_cShortSerial;      ///< short serial number
        GENSTRING m_cFSType;           ///< filesystem type i.e. vfat, ntfs etc
        int       m_iIsMounted;        ///< device is mounted or not
        GENSTRING m_cMountPoint;       ///< Mount point of the device                                                                                   [SENDTOCLIENTS]
        GENSTRING m_cPartitionNode;    ///< device e.g. /dev/sda0 /dev/sda1 /dev/sda2 or /dev/sdb0 /dev/sdb1 etc.
        GENSTRING m_cAccessoryName;    ///< Alsa DeviceName                                                                                              [SENDTOCLIENTS]
        GENSTRING m_cDeviceName;       ///< disk label                                                                                                         [SENDTOCLIENTS]
        GENSTRING m_cDeviceVersion;    ///< version of device                                                                                              [SENDTOCLIENTS]
        uint8_t   m_iDevNUM;           ///< Device Number, used for MTP
        GENSTRING m_cParentSerialId;   ///< Parent Hub Serial ID
        bool      m_bIsSdCardDevice;   ///< is an Sd card?
        bool      m_bIsConnected;      ///< check for the device connection status
        int       m_iHubIndex;         ///< the hub index
        int       m_iPortNumber;       ///< port number
        GENSTRING m_cDeviceUUID;       ///< device UUID
        int       m_iValidPartitionNo; ///< valid partition number                                                                                       [SENDTOCLIENTS]
#ifdef MSD_BIGGEST_PARTITON_FEATURE
        int       m_iTotalValidPartitons; ///< valid partition number                                                                                       [SENDTOCLIENTS] // bound to eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL
        //if config param eConf_ENABLE_NOTIFY_PARTITIONS_ALL is enabled the serial id of each partitions will be serialid + uuid of the device
        //this additional m_cAccessoryName2 This will be used to identify all the partitions of multipartition  stick belongs to same device
        GENSTRING m_cAccessoryName2;//Used to store the original serial id of the device
		GENSTRING m_cBcdDevice; // Used to differentiate ES2 & ES4 hub based on Version ID

#endif

        unsigned long long m_lFreeSpace; ///< free space, if an storage device                                                                     [SENDTOCLIENTS]
        unsigned long long m_lUsedSpace; ///< used space, if an storage device                                                                    [SENDTOCLIENTS]
        unsigned long long m_lTotalSize; ///< total space, if an storage device                                                                    [SENDTOCLIENTS]


#ifdef USE_TOTAL_USED_FREE_SIZE
        ///@todo check if above values might be replaced by this depends on the speed df versus stat
        tU64 m_u64TotalSize_KB;
        tU64 m_u64UsedSize_KB;
        tU64 m_u64FreeSize_KB;
#endif      
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        CGlobalEnumerations::FILESYSTEM_TYPE_Type m_eFSType; //gained from m_cFSType                                                     [SENDTOCLIENTS]
#endif

        tenDeviceSpeedUSB m_DeviceSpeedUSB;  ///<Device Speed of USB device
        GENSTRING m_cManufacturer;                                                                            // [SENDTOCLIENTS]
        tU64    m_u64Size;               ///< size via udev_device_get_sysattr_value(..."size") in byte                                   [SENDTOCLIENTS]

        GENSTRING m_cUSBPort;              ///< hardware USB port                                                                                      [SENDTOCLIENTS]
        tenDevManagerUSBHost  m_eUSB;    ///< Connector - same as cUSBPort but as enum
        DEVICE_CONNECTSTATUS_Type m_eConnectStatus;          ///< connect status of device                                               [SENDTOCLIENTS]
        CGlobalEnumerations::DEVICE_TYPE_Type m_eDeviceType; ///< device type                                                                 [SENDTOCLIENTS]

        //used for remove
        CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type eRemoveAction;
        GENSTRING                                     cRemoveDeviceNode;
        tInt m_iUSBConnector;    ///< is is connected to 1, 2 or 3
        int m_iSendNo; ///< after startup each notification to clients gets its number

        OSAL_tMSecond m_rSendTime_ms;                           ///< after startup each notification to clients gets its number
        CGlobalEnumerations::NOTIFICATION_Type m_eTypeOfNotify; ///< shows how information has been give to clients

        CGlobalEnumerations::PT_Type_e m_ePTType; ///< Partition table type
        CGlobalEnumerations::eDeviceRecognitionStatusType_t m_eDeviceStatus;

        bool m_bIsLogDump; ///< debug information is being stored in this device
        bool m_bConnectedAtHub; ///<shows if device is connected at HUB or not
        tenDevice_UnsupportedReason  m_eDeviceUnsupportedReason;     /* USB Compliance - UnsupportedReason for device connected*/
        CGlobalEnumerations::tenDeviceClass m_eDeviceClass; /*Device class*/
        CGlobalEnumerations::tenInterfaceDeviceClass m_eInterfaceDeviceClass; /*Interface Device class*/
        /*!
         * \brief Constructor
         */
        CDevice();

        /*!
         * \brief Destructor
         */
        ~CDevice();

        /*Public methods*/

        /*!
         * \brief Copies f_pCDevice into this
         *
         * \param [in] f_pCDevice device to be copied
         */
        void FillDevice( const CDevice& f_pCDevice);

        /*!
         * \brief Prints device properties
         */
        void Show();

        void clear();
};




#endif // __CDEVICE_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>

