/*-----------------------------------------------------------------------------*
 * Device.cpp                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file Device.cpp
 *
 * Container to hold device information i.e. properties of the connected devices
 * and it is used for notification to clients.
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Koechling, Christian (Bosch),
 *          -# add 'ProductID to charge Apple devices'
 *          -# add AccessoryName for IPOD and FC_MediaPlayer
 *          -# Statemachine cares for >1 devices connected to HUB
 *          -# cleanup
 * \version 01.11.2011, Negi, Sunder (Montavista), Traces modified
 * \version 17.11.2011, Koechling, Christian (Bosch),
 *          -# cleanup: 1st step to shift includes to Common.h
 *          -# merge of latest MV bugfixes from Sunder 9.11.2011
 * \version 28.11.2011, Koechling, Christian (Bosch), cleanup - add headers
 * \version 06.12.2011, Koechling, Christian (Bosch), enable ETGTrace, add send counter
 * \version 16.11.2011, Negi, Sunder (Montavista), force mount added
 * \version 14.03.2012, Koechling, Christian (Bosch), cleanup define USE_BUGFIX_HUB
 * \version Negi, Sunder (Montavista), Modify as per device type
 * \version Negi, Sunder (Montavista), Update traces
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 *          fixed traces again
 * \version 09.08.2012, Negi, Sunder (MontaVista), usb and port no are made integer
 * \version 14.08.2012, Negi, Sunder (MontaVista), added member to check if
 *          this device is being used for storing the traces
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup HANDLE_GPT
 * \version 28,08,2012, Negi, Sunder (MontaVista), modified class members
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/Device.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CDevice::CDevice()
{
    clear(); 
}

void CDevice::clear() 
{
    m_cBlockDevnode         = "";
    m_cDevpath              = "";
    m_cDevnode              = "";
    m_cSysName              = "";
    m_iVendorID             = 0x0;
    m_iProductID            = 0x0;
    m_cSerialID             = "";
    m_cShortSerial          = "";
    m_cFSType               = "";
    m_iIsMounted            = 0;
    m_cMountPoint           = "";
    m_cPartitionNode        = "";
    m_cDeviceName           = "";
    m_cAccessoryName        = "";
    m_cDeviceVersion        = "";
    m_iDevNUM               = 0xFF;
    m_cParentSerialId       = "";
    m_iPortNumber           = -1;
    m_cDeviceUUID           = "";
    m_cUSBPort              = "";
    m_iIsMounted            = 0;
    m_iHubIndex             = 0;
    m_lUsedSpace            = 0;
    m_lFreeSpace            = 0;
    m_lTotalSize            = 0;
    m_bIsConnected          = false;
    m_bIsSdCardDevice       = false;
    m_eUSB                  = eUSBUndef;
    m_eConnectStatus        = USB_DEV_UNDEFINED;
    eRemoveAction           = CGlobalEnumerations::SINGLE;
    cRemoveDeviceNode       = "";
    m_eDeviceType           = CGlobalEnumerations::DTY_UNKNOWN;
    m_iUSBConnector         = -1;
    m_iValidPartitionNo     = -1;
    m_iSendNo = -1;
    m_eDeviceStatus         = CGlobalEnumerations::DEVICE_INVALID;
    m_ePTType               = CGlobalEnumerations::PT_UNKNOWN;
    m_cManufacturer         = "";
    m_u64Size               = 0;
#ifdef MSD_BIGGEST_PARTITON_FEATURE
    m_iTotalValidPartitons  = 0;
    m_cAccessoryName2       = "";
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
        ///@todo check if above values might be replaced by this depends on the speed df versus stat
      m_u64TotalSize_KB     = 0;
      m_u64UsedSize_KB      = 0;
      m_u64FreeSize_KB      = 0;
#endif
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
    m_eFSType               = CGlobalEnumerations::FSTY_UNKNOWN;
#endif

    m_rSendTime_ms          = 0;
    m_eTypeOfNotify         = CGlobalEnumerations::enNotifyAll;
    m_bIsLogDump            = false;
    m_bConnectedAtHub       = false;
    m_eDeviceUnsupportedReason = OK;
    m_eDeviceClass          = CGlobalEnumerations::DEVICE_CLASS_HID;
    m_eInterfaceDeviceClass = CGlobalEnumerations::INTERFACE_DEVICE_CLASS_OTHERS;
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CDevice::~CDevice()
{
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * void Show()                                                                 *
 *-----------------------------------------------------------------------------*/
void CDevice::Show()
{
    ETG_TRACE_USR4 (("/---------------CDEVICE STRUCTURE--------------------"));

    ETG_TRACE_USR4(("m_cBlockDevnode     = %s  ", m_cBlockDevnode.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cDevpath          = %s  ", m_cDevpath.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cDevnode          = %s  ", m_cDevnode.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cSysName          = %s  ", m_cSysName.toStdString().c_str()));
    ETG_TRACE_USR4(("m_iVendorID         = 0x%04x ", m_iVendorID));
    ETG_TRACE_USR4(("m_iProductID        = 0x%04x ", m_iProductID));
    ETG_TRACE_USR4(("m_cSerialID         = %s  ", m_cSerialID.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cShortSerial      = %s  ", m_cShortSerial.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cFSType           = %s  ", m_cFSType.toStdString().c_str()));
    ETG_TRACE_USR4(("m_iIsMounted        = 0x%x ", m_iIsMounted));
    ETG_TRACE_USR4(("m_cMountPoint       = %s  ", m_cMountPoint.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cPartitionNode    = %s  ", m_cPartitionNode.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cDeviceName       = %s  ", m_cDeviceName.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cAccessoryName    = %s  ", m_cAccessoryName.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cDeviceVersion    = %s  ", m_cDeviceVersion.toStdString().c_str()));
    ETG_TRACE_USR4(("m_eDeviceType       = %d  ", ETG_ENUM(DVM_DEVICE_TYPE, m_eDeviceType)));
    ETG_TRACE_USR4(("m_iDevNUM           = %d  ",m_iDevNUM));
    ETG_TRACE_USR4(("m_cParentSerialId   = %s  ", m_cParentSerialId.toStdString().c_str()));
    ETG_TRACE_USR4(("m_bIsSdCardDevice   = 0x%x ", m_bIsSdCardDevice));
    ETG_TRACE_USR4(("m_iHubIndex         = %d  ",m_iHubIndex));
    ETG_TRACE_USR4(("m_iPortNumber       = %d  ",m_iPortNumber));
    ETG_TRACE_USR4(("m_lUsedSpace        = %u (KB) ", m_lUsedSpace));
    ETG_TRACE_USR4(("m_lFreeSpace        = %u (KB) ", m_lFreeSpace));
    ETG_TRACE_USR4(("m_lTotalSize        = %u (MB) ", m_lTotalSize));
    ETG_TRACE_USR4(("m_cDeviceUUID       = %s  ", m_cDeviceUUID.toStdString().c_str()));
    ETG_TRACE_USR4(("m_cUSBPort          = %s  ", m_cUSBPort.toStdString().c_str()));
    ETG_TRACE_USR4(("m_eUSB              = %d  ", m_eUSB));
    ETG_TRACE_USR4(("m_eConnectStatus    = %d",ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,m_eConnectStatus)));
    ETG_TRACE_USR4(("m_iUSBConnector     = %d",m_iUSBConnector));
    ETG_TRACE_USR4(("m_iValidPartitionNo = %d",m_iValidPartitionNo));
#ifdef MSD_BIGGEST_PARTITON_FEATURE
    ETG_TRACE_USR4(("m_iTotalValidPartitons = %d",m_iTotalValidPartitons));
#endif
    ETG_TRACE_USR4(("m_iSendNo           = %d",m_iSendNo));
    ETG_TRACE_USR4(("m_eDeviceStatus     = %d  ", ETG_CENUM(CGlobalEnumerations::eDeviceRecognitionStatusType_t,m_eDeviceStatus)));
    ETG_TRACE_USR4(("m_ePTType           = %d  ",ETG_CENUM(CGlobalEnumerations::PT_Type_e,m_ePTType)));
    ETG_TRACE_USR4(("m_cManufacturer     = %s  ", m_cManufacturer.toStdString().c_str()));
    ETG_TRACE_USR4(("m_u64Size (kB)      = %d  ", (m_u64Size/1024) ));

#ifdef USE_TOTAL_USED_FREE_SIZE
    ETG_TRACE_USR4(("m_u64TotalSize_KB   = %u ", m_u64TotalSize_KB));
    ETG_TRACE_USR4(("m_u64UsedSize_KB    = %u ", m_u64UsedSize_KB));
    ETG_TRACE_USR4(("m_u64FreeSize_KB    = %u ", m_u64FreeSize_KB));
#endif
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
    ETG_TRACE_USR4(("m_eFSType           = %d ",ETG_CENUM(CGlobalEnumerations::FILESYSTEM_TYPE_Type,m_eFSType)));
#endif

    ETG_TRACE_USR4(("m_rSendTime_ms      = %d  ", m_rSendTime_ms));
    ETG_TRACE_USR4(("m_eTypeOfNotify     = %d ",ETG_CENUM(CGlobalEnumerations::FILESYSTEM_TYPE_Type,m_eFSType)));

    ETG_TRACE_USR4(("m_bIsLogDump        = 0x%x ", m_bIsLogDump));
    ETG_TRACE_USR4(("m_bConnectedAtHub   = 0x%x ", m_bConnectedAtHub));
    ETG_TRACE_USR4(("m_eDeviceUnsupportedReason = %d",ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,m_eDeviceUnsupportedReason)));
    ETG_TRACE_USR4(("m_eDeviceClass = %d",ETG_ENUM(TENDEVICECLASS,m_eDeviceClass)));
    ETG_TRACE_USR4(("m_eInterfaceDeviceClass = %d",ETG_ENUM(TENINTERFACEDEVICECLASS,m_eInterfaceDeviceClass)));
}

/*-----------------------------------------------------------------------------*
 * void FillDevice(CDevice& f_pCDevice)                                        *
 *-----------------------------------------------------------------------------*/
void CDevice::FillDevice( const CDevice& f_pCDevice)
{
    m_cBlockDevnode            = f_pCDevice.m_cBlockDevnode;
    m_cDevpath                 = f_pCDevice.m_cDevpath;
    m_cDevnode                 = f_pCDevice.m_cDevnode;
    m_cSysName                 = f_pCDevice.m_cSysName;
    m_iVendorID                = f_pCDevice.m_iVendorID;
    m_iProductID               = f_pCDevice.m_iProductID;
    m_cSerialID                = f_pCDevice.m_cSerialID;
    m_cShortSerial             = f_pCDevice.m_cShortSerial;
    m_cFSType                  = f_pCDevice.m_cFSType;
    m_iIsMounted               = f_pCDevice.m_iIsMounted;
    m_cMountPoint              = f_pCDevice.m_cMountPoint;
    m_cPartitionNode           = f_pCDevice.m_cPartitionNode;
    m_cDeviceName              = f_pCDevice.m_cDeviceName;
    m_cAccessoryName           = f_pCDevice.m_cAccessoryName;
    m_cDeviceVersion           = f_pCDevice.m_cDeviceVersion;
    m_eDeviceType              = f_pCDevice.m_eDeviceType;
    m_iDevNUM                  = f_pCDevice.m_iDevNUM;
    m_cParentSerialId          = f_pCDevice.m_cParentSerialId;
    m_bIsSdCardDevice          = f_pCDevice.m_bIsSdCardDevice;
    m_iHubIndex                = f_pCDevice.m_iHubIndex;
    m_iPortNumber              = f_pCDevice.m_iPortNumber;
    m_lUsedSpace               = f_pCDevice.m_lUsedSpace;
    m_lFreeSpace               = f_pCDevice.m_lFreeSpace;
    m_lTotalSize               = f_pCDevice.m_lTotalSize;
    m_cDeviceUUID              = f_pCDevice.m_cDeviceUUID;
    m_cUSBPort                 = f_pCDevice.m_cUSBPort;             //check if this new entry is redundant
    m_eUSB                     = f_pCDevice.m_eUSB;
    m_eConnectStatus           = f_pCDevice.m_eConnectStatus;
    m_iUSBConnector            = f_pCDevice.m_iUSBConnector;
    m_iValidPartitionNo        = f_pCDevice.m_iValidPartitionNo;
#ifdef MSD_BIGGEST_PARTITON_FEATURE
    m_iTotalValidPartitons     = f_pCDevice.m_iTotalValidPartitons;//bound to eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL
    m_cAccessoryName2          = f_pCDevice.m_cAccessoryName2;//bound to eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL
#endif
    m_iSendNo                  = f_pCDevice.m_iSendNo;
    m_eDeviceStatus            = f_pCDevice.m_eDeviceStatus;
    m_ePTType                  = f_pCDevice.m_ePTType;
    m_cManufacturer            = f_pCDevice.m_cManufacturer;
    m_u64Size                  = f_pCDevice.m_u64Size;

#ifdef USE_TOTAL_USED_FREE_SIZE
    m_u64TotalSize_KB          = f_pCDevice.m_u64TotalSize_KB;
    m_u64UsedSize_KB           = f_pCDevice.m_u64UsedSize_KB;
    m_u64FreeSize_KB           = f_pCDevice.m_u64FreeSize_KB;
#endif
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
    m_eFSType                  = f_pCDevice.m_eFSType;
#endif

    m_rSendTime_ms             = f_pCDevice.m_rSendTime_ms;
    m_eTypeOfNotify            = f_pCDevice.m_eTypeOfNotify;

    m_bIsLogDump               = f_pCDevice.m_bIsLogDump;
    m_bConnectedAtHub          = f_pCDevice.m_bConnectedAtHub; ///<shows if device is connected at HUB or not
    m_DeviceSpeedUSB           = f_pCDevice.m_DeviceSpeedUSB; ///< USB device speed
    m_eDeviceUnsupportedReason = f_pCDevice.m_eDeviceUnsupportedReason;  //USB Compliance
    m_eDeviceClass             = f_pCDevice.m_eDeviceClass;  // Device class
    m_eInterfaceDeviceClass    = f_pCDevice.m_eInterfaceDeviceClass;  // Interface Device class
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>
