/*-----------------------------------------------------------------------------*
 * TraceCmdManager.cpp                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \class TraceCmdManager 
 *
 * \brief This file contains TraceCmdManager definition. It is used to receive
 * TTFIS commands.
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include <stdio.h>
#include "Config.h"
/* #include "usb.h" */
#include <unistd.h>
#include <dirent.h>

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include <fstream>
#include <iostream>

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "IPrmManager.h"
#include "IPrmManagerCard.h"
#include "IVoltageManager.h"
#include "StateTable.h"
#include "UDevManager.h"
#include "device/DeviceListManager.h"
#include "debug/TraceCmdManager.h"
#include "debug/HistoryManager.h"
#include "interface/DeviceManagerInterface.h"
#include "usbutils.h"
#include "client/AdapterOpticalDiscNotifications.h"

#include "config/ConfigurationManager.h"

#ifndef _VD_DEVICEMANAGER_ROOTDAEMON_CLIENT_H_
#include "VD_DeviceManager_rootdaemon_client.h"
#endif

#include "utils.h"
#include <fcntl.h>



/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_TRACECMDMANAGER
#include "trcGenProj/Header/TraceCmdManager.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
OSAL_tMSecond    TraceCmdManager::m_Testdelay_ms =0;

// static pointer used to ensure a single instance of TraceCmdManager
DVMLOCK          TraceCmdManager::m_singelton;
TraceCmdManager *TraceCmdManager::m_pTraceCmdManager   = NULL;

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;
using namespace devlistmgr;

/*-----------------------------------------------------------------------------*
 * Parameterized contructor                                                    *
 *-----------------------------------------------------------------------------*/
TraceCmdManager::TraceCmdManager()
{
    ETG_TRACE_USR1(("Begin: constructor TraceCmdManager"));

    m_TraceHandle           = OSAL_ERROR;
    m_pStateTableApp = StateTable::GetInstance();
    m_poDumpFile = NULL;

    memset( &m_signal, 0x00, sizeof(voltageFailure_HWSignalsAll_t) );

    //Interfaces
    m_pIPrmManager     = NULL;
    m_pIPrmManagerCard = NULL;
    m_pIVoltageManager = NULL;

    ETG_TRACE_USR1(("End  : constructor TraceCmdManager"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
TraceCmdManager::~TraceCmdManager()
{
    m_poDumpFile        = NULL;
    m_pStateTableApp    = NULL;
    m_pIPrmManager      = NULL;
    m_pIPrmManagerCard  = NULL;
    m_pIVoltageManager  = NULL;
    m_pIStateTableMount = NULL;
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * PrmManager* GetInstance(vd_devicemanager_tclApp *poMainApp)                 *
 *-----------------------------------------------------------------------------*/
TraceCmdManager *TraceCmdManager::GetInstance()
{
    ETG_TRACE_USR4(("Begin: GetInstance"));
    TraceCmdManager::m_singelton.lock();

    if(NULL == m_pTraceCmdManager)
    {
        m_pTraceCmdManager = new TraceCmdManager();
    }

    TraceCmdManager::m_singelton.unlock();
    ETG_TRACE_USR4(("End  : GetInstance"));

    return m_pTraceCmdManager;
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance()                                                      *
 *-----------------------------------------------------------------------------*/
void TraceCmdManager::DestroyInstance()
{
    ETG_TRACE_USR4(("Begin: DestroyInstance"));
    TraceCmdManager::m_singelton.lock();
    DEL_M(m_pTraceCmdManager);
    TraceCmdManager::m_singelton.unlock();
    ETG_TRACE_USR4(("End  : DestroyInstance"));
}

/*-----------------------------------------------------------------------------*
 * int Initialize()                                                            *
 *-----------------------------------------------------------------------------*/
int TraceCmdManager::Initialize() const
{
    ETG_TRACE_USR4(("Begin: Initialize"));

    m_pTraceCmdManager->TraceOpen();

    ETG_TRACE_USR4(("End  : Initialize"));

    return OSAL_OK;
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceOpen(void)                                                       *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TraceOpen(void)
{
    tS32                        s32Error;
    OSAL_trIOCtrlLaunchChannel  oTraceChannel;
    ETG_TRACE_USR4(("Begin: TraceOpen"));

    if (m_TraceHandle == OSAL_ERROR)
    {
         m_TraceHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);
         if(OSAL_ERROR == m_TraceHandle)
         {
             ETG_TRACE_FATAL((" [ERROR]: TraceOpen: OSAL_ERROR == m_TraceHandle"));
             ETG_TRACE_ERRMEM(("[ERROR]: TraceOpen: OSAL_ERROR == m_TraceHandle"));
             DVM_NORMAL_M_ASSERT_ALWAYS(); 
         }
    }
    if(m_TraceHandle != OSAL_ERROR)
    {
        ETG_TRACE_USR2((" TraceOpen: Trace is open"));
        oTraceChannel.enTraceChannel = TR_TTFIS_DEVICEMANAGER;
        oTraceChannel.pCallback = (OSAL_tpfCallback)vChannelMgr;

        s32Error = OSAL_s32IOControl (m_TraceHandle, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t) &oTraceChannel);
        if (OSAL_OK != s32Error)
        {
            ETG_TRACE_FATAL(("[ERROR: TraceOpen:Can't register Tracechannel"));
        }
        else
        {
            ETG_TRACE_USR2((" Trace channel established"));
        }

    }
    ETG_TRACE_USR4(("End  : TraceOpen"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceClose(void)                                                      *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TraceClose(void)
{
   tS32 s32Error;
   OSAL_trIOCtrlLaunchChannel oTraceChannel;

   oTraceChannel.enTraceChannel = TR_TTFIS_DEVICEMANAGER;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vChannelMgr;

   s32Error = OSAL_s32IOControl (m_TraceHandle, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t) &oTraceChannel);
   if (OSAL_OK != s32Error)
   {
      ETG_TRACE_FATAL(("Error: TraceClose::Can't unregister trace channel"));
   }
   else
   {
       ETG_TRACE_USR2(("TraceClose::unregistered trace channel"));
       if (OSAL_OK != OSAL_s32IOClose(m_TraceHandle))
       {
          ETG_TRACE_FATAL((" Error: TraceClose:Can't close Trace Device"));
       }
       else
       {
          ETG_TRACE_USR2(("TraceClose::ok!"));
          m_TraceHandle = OSAL_ERROR;
       }
   }
}

/*-----------------------------------------------------------------------------*
 * tVoid vChannelMgr (const tUChar* puchData)                                  *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::vChannelMgr (const tUChar* puchData)
{
  tU8 u8MsgCode = 0;

  ETG_TRACE_USR4(("Begin: vChannelMgr                                             "));

  if ((NULL != puchData))
  {
      u8MsgCode = puchData[1]; //byte 1 -> start of transmitted data
      ETG_TRACE_USR2(("vChannelMgr u8MsgCode=0x%x", u8MsgCode));
  }

  if( m_pTraceCmdManager && u8MsgCode==DevManagerTTFIScmd_ValHeaderFlag_1st && puchData )
  {
      m_pTraceCmdManager->TTFIScmdHandler(&(puchData[1]));
  }
  else
  {
      ETG_TRACE_FATAL(("ERROR:vChannelMgr: u8MsgCode!=DevManagerTTFIScmd_ValHeaderFlag_1st"));
  }

  ETG_TRACE_USR4(("end  : vChannelMgr                                              "));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler(tPCU8 pU8Data)                                        *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler(tPCU8 pU8Data)
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHandler                                             "));
    if( pU8Data && (OSAL_ERROR != m_TraceHandle) )
    {
        if(
           (pU8Data[index_header1] == DevManagerTTFIScmd_ValHeaderFlag_1st) &&
           (pU8Data[index_header2] == DevManagerTTFIScmd_ValHeaderFlag_2nd)
           )
        {
            ETG_TRACE_FATAL(("[ok] vChannelMgr header=0x%x,0x%x",
                               pU8Data[index_header1],
                               pU8Data[index_header2]));
            TTFIScmdHandler_cmd(pU8Data);
        }
        else
        {
            ETG_TRACE_FATAL(("[ok] TTFIScmdHandler: Wrong header 1st:0x%x 2nd:0x%x",
                               pU8Data[index_header1],
                               pU8Data[index_header2]));
        }
    }
    ETG_TRACE_USR4(("End  : TTFIScmdHandler                                             "));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler_cmd(tPCU8 pU8Data)                                    *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler_cmd(tPCU8 pU8Data)
{
    tenDevManagerTTFIScmd enCmd=(tenDevManagerTTFIScmd)pU8Data[(tU8)index_cmd];
    ETG_TRACE_FATAL(("->#############################################################################"));
    ETG_TRACE_FATAL(("[ok] TTFIScmdHandler_cmd: cmd %d",ETG_CENUM(tenDevManagerTTFIScmd, enCmd)));

    switch (enCmd)
    {
            case TTFIScmd_Help:
                TTFIScmdShowHelpManual();
                break;
            case TTFIScmd_PrintConfiguration:
                TTFIScmdPrintConfiguration();   //see ConfigurationManager
                break;
            case TTFIScmd_SimPrmSignal:
                TTFIScmdHandler_signals(pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_U60_drop:
                TTFIScmdHandler_signal(U60_drop,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_U50_USB_Reset:
                TTFIScmdHandler_signal(U50_USB_Reset,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_OC1:
                TTFIScmdHandler_signal(OC1,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_OC2:
                TTFIScmdHandler_signal(OC2,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_PPON1:
                TTFIScmdHandler_signal(PPON1,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_PPON2:
                TTFIScmdHandler_signal(PPON2,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_mountedUSB1_prm:
                TTFIScmdHandler_signal(mountedUSB1_prm,pU8Data);
                break;
            case TTFIScmd_SimPrmSignal_mountedUSB2_prm:
                TTFIScmdHandler_signal(mountedUSB2_prm,pU8Data);
                break;
           case TTFIScmd_SimPrmSignal_mountedUSB1_udev:
                TTFIScmdHandler_signal(mountedUSB1_udev,pU8Data);
                break;
           case TTFIScmd_SimPrmSignal_mountedUSB2_udev:
                TTFIScmdHandler_signal(mountedUSB2_udev,pU8Data);
                break;
            case TTFIScmd_SimMSG2Client:
                TTFIScmdHandler_SimSendMsgtoClients(pU8Data);
                break;
            case TTFIScmd_GetSignals:
                TTFIScmdHandler_getSignals();
                break;
            case TTFIScmd_GetLastMsg2clients:
                TTFIScmdGetLastMsg2clients();
                break;
            case TTFIScmd_SetTimerInterval_sec:
                TTFIScmdSetTimerInterval_ms((OSAL_tMSecond)pU8Data[(tInt)index_cmd+1]   *1000);
                break;
            case TTFIScmd_SetTimerInterval_msec:
                TTFIScmdSetTimerInterval_ms(((OSAL_tMSecond)pU8Data[(tInt)index_cmd+1]) *100);
                break;
            case TTFIScmd_GetConnectedDevices:
                TTFIScmdGetConnectedDevices(pU8Data);
                break;
            case TTFIScmd_InterfaceConnect:
                TTFIScmdInterfaceConnect(pU8Data);
                break;
            case TTFIScmd_CyclicDiagnosis:
                TTFIScmdCyclicDiagnosis(pU8Data);
                break;
            case TTFIScmd_ShowConfiguration:
                TTFIScmdShowConfiguration(); //see ai_mediaplayer_prod.xml used by configurationManager
                break;
            case TTFIScmd_Sendtesttime_sec:
                TTFIScmdSendtesttime_ms(((OSAL_tMSecond)pU8Data[(tInt)index_cmd+1]) *1000);
                break;
            case TTFIScmd_HistoryUdevEvts:
                TTFIScmdHistoryUdevEvts();
                break;
            case TTFIScmd_HistoryUdevEvts_Clear:
                TTFIScmdHistoryUdevEvts_Clear();
                break;
            case TTFIScmd_HistoryLastMsgsSend2Clients:
                TTFIScmdHistoryLastMsgsSend2Clients();
               break;
            case TTFIScmd_HistoryLastMsgsSend2Clients_Clear:
                TTFIScmdHistoryLastMsgsSend2Clients_Clear();
                break;
            case TTFIScmd_SetServiceAvailable:
                TTFIScmdSetServiceAvailable(pU8Data);
                break;
            case TTFIScmd_SimEOLValues:
                TTFIScmdSimEOLValues(pU8Data);
                break;
            case TTFIScmd_TriggerReadEOLValuesFromFile:
                TTFIScmdTriggerReadEOLValuesFromFile();
                break;
            case TTFIScmd_SimulateOverCurrentPlatformDTY_USB:
                TTFIScmdSimulateOverCurrentPlatformDTY_USB();
                break;
            case TTFIScmd_SimulateUndervoltagePlatformDTY_USB:
                TTFIScmdSimulateUndervoltagePlatformDTY_USB();
                break;
            case TTFIScmd_GetConnectedHubs:
                TTFIScmdGetConnectedHubs();
                break;
            case TTFIScmd_CyclicDiagnosis_Permanent:
                TTFIScmdCyclicDiagnosis_Permanent(pU8Data);
                break;
            case TTFIScmd_HistoryLastMsgsSend2Diag:
                TTFIScmdHistoryLastMsgsSend2Diag();
                break;
            case TTFIScmd_HistoryLastMsgsSend2Diag_Clear:
                TTFIScmdHistoryLastMsgsSend2Diag_Clear();
                break;
            case TTFIScmd_HistoryCallbacksPrm:
                TTFIScmdHistoryCallbacksPrm();
            break;
            case TTFIScmd_HistoryCallbacksPrm_Clear:
                TTFIScmdHistoryCallbacksPrm_Clear();
                break;
            case TTFIScmd_HistoryCallbacksVolt:
                TTFIScmdHistoryCallbacksVolt();
               break;
            case TTFIScmd_HistoryCallbacksVolt_Clear:
                TTFIScmdHistoryCallbacksVolt_Clear();
                break;
            case TTFIScmd_CheckAutomountDir:
                TTFIScmdCheckAutomountDir();
               break;
            case TTFIScmd_ServiceUnavailable_Permanent:
                TTFIScmdServiceUnavailable_Permanent(pU8Data);
                break;
            case TTFIScmd_BlockMonitor_Permanent:
                TTFIScmdBlockMonitor_Permanent(pU8Data);
                break;
            case TTFIScmd_HistoryStateTable:
                TTFIScmdHistoryStateTable();
                break;

        #ifdef TTFISCMD_PRODUCTIONLINETEST
            case TTFIScmd_ProductionLineTest:
                TTFIScmdProductionLineTest();
                break;
        #endif // TTFISCMD_PRODUCTIONLINETEST

        #ifdef USE_HUB_CONNECTED_TO_HUB
            case TTFIScmd_EnableHubConnectedToHub_Permanent:
                 TTFIScmdEnableHubConnectedToHubPermanent(pU8Data);
                 break;
        #endif
            case TTFIScmd_SimulateCDDriveOverTemperature:
                 TTFIScmdSimulateCDDriveOverTemperature(pU8Data);
                 break;

            case TTFIScmd_SimulateMixeModeCDRom_Mount:
                 TTFIScmd_SimulateMixeModeCDRomMount(pU8Data);
                 break;

            case TTFIScmd_SimulateMixeModeCDRom_UnMount:
                 TTFIScmd_SimulateMixeModeCDRomUnMount(pU8Data);
                 break;

            case TTFIScmd_TriggerPrmConfiguration:
                 TTFIScmdTriggerPrmConfiguration();
                 break;
            case TTFIScmd_PRMMANAGER_vGetSDCardState:
                 TTFIScmd_PRMMANAGERvGetSDCardState();
                 break;
            case TTFiscmd_PRMMANAGER_bGetUSBHardwareConnector:
                 TTFiscmd_PRMMANAGERbGetUSBHardwareConnector(pU8Data);
                 break;
            case TTFiscmd_PRMMANAGER_bSetUSBHardwareConnectorON:
                 TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON(pU8Data);
                 break;
            case TTFiscmd_PRMMANAGER_bSetUSBHardwareConnectorOFF:
                 TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF(pU8Data);
                 break;
            case TTFiscmd_VoltageManager_vGetSystemVoltage:
                 TTFiscmd_VoltageManagervGetSystemVoltage();
                 break;
            case TTFiscmd_Voltagemanager_vGetBoardVoltage:
                 TTFiscmd_VoltagemanagervGetBoardVoltage();
                 break;
          case TTFiscmd_Voltagemanager_vSendVoltageWarning:
                 TTFiscmd_VoltagemanagervSendVoltageWarning(pU8Data);
                 break;
          case TTFiscmd_LinuxCmd_DbgAnalysis:
                TTFiscmd_LinuxCmdDbgAnalysis();
                break;
          case TTFIScmd_InternalHUB_Test:
                TTFIScmd_InternalHUBTest();
                break;
          case TTFIScmd_BTABoxisAlive_Test:
               TTFIScmd_BTABoxisAliveTest(pU8Data);
               break;
          case TTFIScmd_BTABoxDeviceSpeed_Test:
               TTFIScmd_BTABoxDeviceSpeedTest(pU8Data);
               break;
          case TTFIScmd_CDDAConnection:
               TTFIScmdCDDAConnection(pU8Data);
               break;
          case TTFIScmd_CDDAConnection_srsgWithUdev:
               TTFIScmdCDDAConnectionSrSgWithUdev(pU8Data);
               break;
          case TTFIScmd_EjectOpticalDisc:
               TTFIScmdEjectOpticalDisc(pU8Data);
               break;
          case TTFIScmd_OpticalDiscSlotStatus:
               TTFIScmdOpticalDiscSlotStatus(pU8Data);
               break;
          case TTFIScmd_GetConfigurationValue:
               TTFIScmdGetConfigurationValue(pU8Data);
               break;
          case TTFIScmd_SetConfigurationValue:
               TTFIScmdSetConfigurationValue(pU8Data);
               break;
          case TTFIScmd_SimulateMalfunctionUSBConnector:
               TTFIScmdSimulateMalfunctionUSBConnector(pU8Data);
               break;
          case TTFIScmd_HistoryHardwareMalfunction:
               TTFIScmdHistoryHardwareMalfunction();
               break;
         case TTFIScmd_HistoryHardwareMalfunction_Clear:
               TTFIScmdHistoryHardwareMalfunction_Clear();
               break;
         case TTFIScmd_ExternalHub_DetailsAdd:
             TTFIScmdExternalHubDetailsAdd(pU8Data);
               break;
         case TTFIScmd_ExternalHub_DetailsRemove:
             TTFIScmdExternalHubDetailsRemove(pU8Data);
               break;
         case TTFIScmd_ExternalHub_Print:
             TTFIScmdExternalHubPrint();
               break;
         case TTFIScmd_ExternalHub_Clear:
             TTFIScmdExternalHubDPClear();
               break;
         case TTFIScmd_TelematicsBox_DetailsAdd:
             TTFIScmdTelematicsBoxDetailsAdd(pU8Data);
               break;
         case TTFIScmd_TelematicsBox_DetailsRemove:
             TTFIScmdTelematicsBoxDetailsRemove(pU8Data);
               break;
         case TTFIScmd_TelematicsBox_Print:
             TTFIScmdTelematicsBoxPrint();
               break;
         case TTFIScmd_TelematicsBox_Clear:
             TTFIScmdTelematicsBoxDPClear();
               break;
         case TTFIScmd_ShowGenericKDS_DeviceManager:
            TTFIScmdShowGenericKDSDeviceManager();
            break;
         case TTFIScmd_ShowGenericKDS_Mediaplayer:
            TTFIScmdShowGenericKDSMediaplayer();
            break;
         case TTFIScmd_ShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer:
            TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer();
            break;
         case TTFIScmd_RootDaemon_Execute_Mkdir:
             TTFIScmdRootDaemonExecute_Mkdir();
            break;
         case TTFIScmd_RootDaemon_Execute_Cmd1:
             TTFIScmdRootDaemonExecute_Cmd1();
            break;
         case TTFIScmd_RootDaemon_Execute_Cmd2:
             TTFIScmdRootDaemonExecute_Cmd2();
            break;
        default:
            break;
    }

    ETG_TRACE_FATAL(("[ok] #############################################################################<-"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler_signals(tPCU8 pU8Data)                                *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler_signals(tPCU8 pU8Data)
{
    tU8 u8USBNo;
    UsbPortState f_oPortState;

    //voltageFailure_HWSignalsAll_t signal;

    ETG_TRACE_USR4(("Begin: TTFIScmdHandler_signals"));

    u8USBNo=pU8Data[(tInt)index_cmd+1];
    ETG_TRACE_USR1(("USBNo: 0x%x",u8USBNo));

    if((u8USBNo < (tU8)eUSB1)|| (u8USBNo >=ARRAYSIZEFORUSB))
    {
        ETG_TRACE_FATAL(("[ok] TTFIScmdHandler_signals: Warning this SW supports USB1 and USB2 not more!!"));
    }
#ifndef WORKAROUND_PRMBUG
    else
#endif
    {
        //get current signals
        if(NULL==m_pStateTableApp)
        {
             ETG_TRACE_FATAL((" [ERROR]: TTFIScmdHandler_signals: NULL==m_pStateTableApp"));
             ETG_TRACE_ERRMEM(("[ERROR]: TTFIScmdHandler_signals: NULL==m_pStateTableApp"));
             DVM_NORMAL_M_ASSERT_ALWAYS(); 
        }
        else
        {
            if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
            {
                //set signals according to TTFIS command
                f_oPortState.u8PortNr = (tU8)pU8Data[(tInt)index_cmd+1];
                f_oPortState.u8UV     = (tU8)pU8Data[(tInt)index_cmd+2];
                f_oPortState.u8OC       = (tU8)pU8Data[(tInt)index_cmd+3];
                f_oPortState.u8PPON     = (tU8)pU8Data[(tInt)index_cmd+4];

                f_oPortState.u32OCStartTime   = 0;
                f_oPortState.u32OCEndTime     = 0;
                f_oPortState.u32UVStartTime   = 0;
                f_oPortState.u32UVEndTime     = 0;
                f_oPortState.u32PPONStartTime = 0;
                f_oPortState.u32PPONEndTime   = 0;

                ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
                ETG_TRACE_FATAL(("[ok]         TTFIScmdHandler_signals (set all signals) USB%d      ", u8USBNo));
                ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
                ETG_TRACE_FATAL(("[ok] u8PortNr          :%d                                 ", f_oPortState.u8PortNr));
                ETG_TRACE_FATAL(("[ok] u8UV              :%d                                 ",ETG_ENUM(TENSIGNALVALUE, f_oPortState.u8UV)));
                ETG_TRACE_FATAL(("[ok] OC                :%d                                 ",ETG_ENUM(TENSIGNALVALUE, f_oPortState.u8OC)));
                ETG_TRACE_FATAL(("[ok] PPON              :%d                                 ",ETG_ENUM(TENSIGNALVALUE, f_oPortState.u8PPON)));
                ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
                ETG_TRACE_FATAL(("[ok]                                                        "));
                //sends it to StateTable but for diagnosis also
                m_pStateTableApp->setHWSignals(&f_oPortState);
            }
            else
            {
                ETG_TRACE_FATAL(("TTFIScmdHandler_signals Cannot be executed as the  feature temporary not available is enabled!!!    "));
            }

        }

        
    }

    ETG_TRACE_USR4(("End  : TTFIScmdHandler_signals"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler_signal(tenDevManagerSignals enSignal,tPCU8 pU8Data)   *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler_signal(tenDevManagerSignals enSignal,tPCU8 pU8Data)
{
    //voltageFailure_HWSignalsAll_t signal;

    ETG_TRACE_USR4(("Begin: TTFIScmdHandler_signal"));

    ETG_TRACE_FATAL(("[ok] -------------------------------------------------------------------------------"));
    ETG_TRACE_FATAL(("[ok] TTFIScmdHandler_signal %d",ETG_ENUM(TENDEVMANAGERSIGNALS, enSignal)));
    ETG_TRACE_FATAL(("[ok] -------------------------------------------------------------------------------"));

    // those which keep  to be set 'SIG_ANY' won't be changed
    m_signal.U60_drop       =SIG_ANY;
    m_signal.U50_USB_Reset  =SIG_ANY;
    m_signal.OC             =SIG_ANY;
    m_signal.PPON           =SIG_ANY;
    m_signal.mountedUSB     =SIG_ANY;     //send by prm which gets ti from udev
    m_signal.mountedUSBref  =SIG_ANY;   //send by udev (reference)

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]          TTFIScmdHandler_signal  (set a signal)        "));

    switch (enSignal)
    {
        case OC1:
            m_signal.usb=eUSB1;
            m_signal.OC           =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: OC               :0x%x (%d)                           ",m_signal.usb,m_signal.OC,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.OC)));
            break;
        case OC2:
            m_signal.usb=eUSB2;
            m_signal.OC           =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: OC               :0x%x (%d)                           ",m_signal.usb,m_signal.OC,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.OC)));
            break;
        case OC3:
            m_signal.usb = eUSB3;
            m_signal.OC = (tenSignalValue)pU8Data[(tInt)index_cmd + 1];
            ETG_TRACE_FATAL(("[ok] USB%1d: OC               :0x%x (%d)                           ", m_signal.usb, m_signal.OC,
                ETG_ENUM(TENSIGNALVALUE, m_signal.OC)));
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case OC4:
             m_signal.usb=eUSB4;
             m_signal.OC           =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
             ETG_TRACE_FATAL(("[ok] USB%1d: OC               :0x%x (%d)                           ",m_signal.usb,m_signal.OC,
                                                                                        ETG_ENUM(TENSIGNALVALUE,m_signal.OC)));
             break;
#endif//USE_FOURTH_USB_CONNECTOR
        case PPON1:
            m_signal.usb=eUSB1;
            m_signal.PPON         =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: PPON             :0x%x (%d)                           ",m_signal.usb,m_signal.PPON,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.PPON)));
            break;
        case PPON2:
            m_signal.usb=eUSB2;
            m_signal.PPON         =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: PPON             :0x%x (%d)                           ",m_signal.usb,m_signal.PPON,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.PPON)));
            break;
        case PPON3:
            m_signal.usb = eUSB3;
            m_signal.PPON = (tenSignalValue)pU8Data[(tInt)index_cmd + 1];
            ETG_TRACE_FATAL(("[ok] USB%1d: PPON             :0x%x (%d)                           ", m_signal.usb, m_signal.PPON,
                ETG_ENUM(TENSIGNALVALUE, m_signal.PPON)));
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case PPON4:
            m_signal.usb=eUSB4;
            m_signal.PPON         =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: PPON             :0x%x (%d)                           ",m_signal.usb,m_signal.PPON,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.PPON)));
            break;
#endif//USE_FOURTH_USB_CONNECTOR
        case U60_drop:
            m_signal.U60_drop      =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB01&USB02: U60_drop          :0x%x (%d)                           ",m_signal.U60_drop,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.U60_drop)));
            break;
        case mountedUSB1_prm:
            m_signal.usb=eUSB1;
            m_signal.mountedUSB   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSB       :0x%x (%d)                           ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
        case mountedUSB2_prm:
            m_signal.usb=eUSB2;
            m_signal.mountedUSB   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSB       :0x%x (%d)                           ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
        case mountedUSB3_prm:
            m_signal.usb = eUSB3;
            m_signal.mountedUSB = (tenSignalValue)pU8Data[(tInt)index_cmd + 1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSB       :0x%x (%d)                           ", m_signal.usb, m_signal.mountedUSB,
                ETG_ENUM(TENSIGNALVALUE, m_signal.mountedUSB)));
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case mountedUSB4_prm:
            m_signal.usb=eUSB4;
            m_signal.mountedUSB   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSB       :0x%x (%d)                           ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
#endif//USE_FOURTH_USB_CONNECTOR
        case mountedUSB1_udev:
            m_signal.usb=eUSB1;
            m_signal.mountedUSBref   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSB       :0x%x (%d)                            ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
        case mountedUSB2_udev:
            m_signal.usb=eUSB2;
            m_signal.mountedUSBref   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSBref       :0x%x (%d)                            ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
        case mountedUSB3_udev:
            m_signal.usb = eUSB3;
            m_signal.mountedUSBref = (tenSignalValue)pU8Data[(tInt)index_cmd + 1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSBref       :0x%x (%d)                            ", m_signal.usb, m_signal.mountedUSB,
                ETG_ENUM(TENSIGNALVALUE, m_signal.mountedUSB)));
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case mountedUSB4_udev:
            m_signal.usb=eUSB4;
            m_signal.mountedUSBref   =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB%1d: mountedUSBref       :0x%x (%d)                            ",m_signal.usb,m_signal.mountedUSB,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.mountedUSB)));
            break;
#endif//USE_FOURTH_USB_CONNECTOR

        case U50_USB_Reset:
            m_signal.U50_USB_Reset =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok] USB01&USB02: U50_USB_Reset     :0x%x (%d)                           ",m_signal.U50_USB_Reset,
                                                                                       ETG_ENUM(TENSIGNALVALUE,m_signal.U50_USB_Reset)));
            break;
        default:
            ETG_TRACE_FATAL(("[ok] TTFIScmdHandler_signal:Unknown enSignal"));
            break;
    }

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));

    if(NULL!=m_pStateTableApp)
    {
        if( (U50_USB_Reset==enSignal) || (U60_drop==enSignal))
        {
            m_signal.usb=eUSB1;
            ETG_TRACE_FATAL(("[ok] m_signal.usb = 0x%x",m_signal.usb));
            m_pStateTableApp->sendSignalsTest(m_signal);
            
            m_signal.usb=eUSB2;
            ETG_TRACE_FATAL(("[ok] m_signal.usb = 0x%x",m_signal.usb));
            m_pStateTableApp->sendSignalsTest(m_signal);
            
            m_signal.usb = eUSB3;
            ETG_TRACE_FATAL(("[ok] m_signal.usb = 0x%x",m_signal.usb));
            m_pStateTableApp->sendSignalsTest(m_signal);
#ifdef USE_FOURTH_USB_CONNECTOR
            m_signal.usb=eUSB4;
            ETG_TRACE_FATAL(("[ok] m_signal.usb = 0x%x",m_signal.usb));
            m_pStateTableApp->sendSignalsTest(m_signal);
#endif//USE_FOURTH_USB_CONNECTOR

        }
        else
        {
            m_pStateTableApp->sendSignalsTest(m_signal);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("Error: TTFIScmdHandler_signal: Could not send signal"));
    }

    ETG_TRACE_USR4(("End  : TTFIScmdHandler_signal"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler_SimSendMsgtoClients(tPCU8 pU8Data)                    *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler_SimSendMsgtoClients(tPCU8 pU8Data)
{
    tenDevManagerUSBHost       usb;
    DEVICE_CONNECTSTATUS_Type  eConnectionStatus;

    ETG_TRACE_USR4(("Begin: TTFIScmdHandler_SimSendMsgtoClients"));

    if( (NULL!=pU8Data) && (NULL != m_pStateTableApp))
    {
        usb               = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+1];
        eConnectionStatus = (DEVICE_CONNECTSTATUS_Type)pU8Data[(tInt)index_cmd+2];
        ETG_TRACE_FATAL(("[ok] TTFIScmdHandler_SimSendMsgtoClients: USB%1d eConnectionStatus=%d",usb,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,eConnectionStatus)));

        m_pStateTableApp->send2ClientsTest(usb,eConnectionStatus);
    }
    else
    {
        ETG_TRACE_FATAL(("ERROR: TTFIScmdHandler_SimSendMsgtoClients:NULL==pU8Data"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmdHandler_SimSendMsgtoClients"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHandler_getSignals(tVoid)                                     *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHandler_getSignals(tVoid)
{
    voltageFailure_HWSignalsAll_t voltageFailuresignal;
    ETG_TRACE_USR4(("Begin: TTFIScmdHandler_getSignals"));

    if(NULL!=m_pStateTableApp)
    {
        if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
        {
            for(tInt i=(tInt)eUSB1;i<ARRAYSIZEFORUSB;i++)
            {
                m_pStateTableApp->getSignals((tenDevManagerUSBHost)i,&voltageFailuresignal);

#ifdef CODE_CLEANUP_SEPT  //this was missing
                ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
                ETG_TRACE_FATAL((" getSignal (show current value):                        "));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].U60_drop           %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.U60_drop)));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].U50_USB_Reset      %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.U50_USB_Reset) ));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].OC                 %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.OC )));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].PPON               %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.PPON )));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].mountedUSB  (prm)  %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.mountedUSB ) ));
                ETG_TRACE_FATAL((" getSignal:signal[USB%1d].mountedUSB  (udev) %d                 ",i, ETG_ENUM(TENSIGNALVALUE,voltageFailuresignal.mountedUSBref ) ));
                ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
#endif//CODE_CLEANUP_SEPT
            }
        }
        else
        {
            ETG_TRACE_FATAL(("TTFIScmdHandler_getSignals Cannot be executed as the feature temporary not available is enabled!!!"));
        }
    }
    ETG_TRACE_USR4(("End  : TTFIScmdHandler_getSignals"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdGetLastMsg2clients(tVoid)                                     *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdGetLastMsg2clients(tVoid) const
{
    voltageFailure_Result_t oLastMsgSendtoClients[ARRAYSIZEFORUSB];

    if(NULL!=m_pStateTableApp)
    {
        if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
        {
            m_pStateTableApp->getLastMsgSendtoClients(eUSB1,&oLastMsgSendtoClients[eUSB1]);
            m_pStateTableApp->getLastMsgSendtoClients(eUSB2,&oLastMsgSendtoClients[eUSB2]);
            m_pStateTableApp->getLastMsgSendtoClients(eUSB3, &oLastMsgSendtoClients[eUSB3]);
#ifdef USE_FOURTH_USB_CONNECTOR
            m_pStateTableApp->getLastMsgSendtoClients(eUSB4,&oLastMsgSendtoClients[eUSB4]);
#endif

            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~TTFIScmdGetLastMsg2clients~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok] ......................................................."));
            ETG_TRACE_FATAL(("[ok] oLastMsgSendtoClients[USB%1d].resultUSB  %d         ",eUSB1,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,oLastMsgSendtoClients[eUSB1].resultUSB)));
            ETG_TRACE_FATAL(("[ok] oLastMsgSendtoClients[USB%1d].resultUSB  %d         ",eUSB2,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,oLastMsgSendtoClients[eUSB2].resultUSB)));
            ETG_TRACE_FATAL(("[ok] oLastMsgSendtoClients[USB%1d].resultUSB  %d         ", eUSB3, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE, oLastMsgSendtoClients[eUSB3].resultUSB)));
#ifdef USE_FOURTH_USB_CONNECTOR
            ETG_TRACE_FATAL(("[ok] oLastMsgSendtoClients[USB%1d].resultUSB  %d         ",eUSB4,ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,oLastMsgSendtoClients[eUSB4].resultUSB)));
#endif
            ETG_TRACE_FATAL(("[ok]                                                        "));

            SERVICEHISTORY.ShowLastMsgSend2Clients();

            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        }
        else
        {
            ETG_TRACE_FATAL(("TTFIScmdGetLastMsg2clients Cannot be executed as the  feature temporary not available is enabled!!!"));
        }
    }
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSetTimerInterval_ms(OSAL_tMSecond millisec)                   *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSetTimerInterval_ms(OSAL_tMSecond millisec)
{
    ETG_TRACE_USR4(("Begin: TTFIScmdSetTimerInterval_ms"));

    ETG_TRACE_FATAL(("[ok]       TTFIScmdSetTimerInterval_ms set to: %d milliseconds",millisec));

    if( m_pStateTableApp )
    {
        m_pStateTableApp->setIntervalTime(millisec);
    }

    ETG_TRACE_USR4(("End  : TTFIScmdSetTimerInterval_ms"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSendtesttime_ms(OSAL_tMSecond millisec)                       *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSendtesttime_ms(OSAL_tMSecond millisec) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_Sendtesttime_ms"));

    ETG_TRACE_FATAL(("[ok]       TTFIScmd_Sendtesttime_ms set to: %d milliseconds",millisec));

    TraceCmdManager::m_Testdelay_ms= millisec;

    ETG_TRACE_USR4(("End  : TTFIScmd_Sendtesttime_ms"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdShowHelpManual(tVoid)                                         *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdShowHelpManual(tVoid) const
{
    ETG_TRACE_FATAL(("[ok] ------------------------------------------------------------------------------------------------------"));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  HELP - MANUAL: possible TTFIS commands VD_DeviceManager:                                             "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  VD_DEVICEMANAGER_SHOW_HELP_MANUAL:                  explains commands                                "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  to set all signals use:"));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_SIGNAL <USBNo> <U60_drop> <U50_USB_Reset> <OC> <PPON> <mountedUSB_prm>  <mountedUSB_udev> "));
    ETG_TRACE_FATAL(("[ok]  Example:                                                                                             "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_SIGNAL USB1 U60_drop_TRUE U50_USB_Reset_TRUE OC_TRUE PPON_TRUE mountedUSB_prm_TRUE mountedUSB_udev_TRUE  "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  to set a single signal:                                                                              "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_SIGNAL_U60_drop  <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_U50_USB_Reset    <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_OC1              <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_OC2              <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_PPON1            <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_PPON2            <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_mountedUSB1_prm  <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_mountedUSB2_prm  <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_mountedUSB1_udev <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_PRM_mountedUSB2_udev <SIG_FALSE or SIG_TRUE>                                         "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  VD_DVM_SIMULATE_MSG2CLIENTS  <MSG related to USB1 and USB2>                                          "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  to get current signal setting received by PRM and last message send to clients:                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_GET_SIGNALS                                                                               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_GET_LASTMSG_SENDTOCLIENTS                                                                 "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper to analyse what udev has send to VD_DeviceManager                                             "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_UDEV_EVTS                                                                         "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_UDEV_EVTS_CLEAR                                                                   "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper to analyse what VD_DeviceManager  has send to clients                                         "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_MSGS_SEND2CLIENTS                                                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_MSGS_SEND2CLIENTS_CLEAR                                                           "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper to analyse what VD_DeviceManager  has send to diagnosis (DTC)                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_MSGS_SEND2DIAGNOSIS                                                               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_MSGS_SEND2DIAGNOSIS_CLEAR                                                         "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper to analyse what prm  has send to VD_DeviceManager                                             "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_CALLBACK_PRM                                                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_CALLBACK_PRM_CLEAR                                                                "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper to analyse what drv_volt  has send to VD_DeviceManager                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_CALLBACK_VOLT                                                                   "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_HISTORY_CALLBACK_VOLT_CLEAR                                                             "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  to manipulate  for runtime the time interval                                                         "));
    ETG_TRACE_FATAL(("[ok]  used to collect prm signals before deducing and sending result:                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_TIMER_INTERVAL_sec  <parms to choose time in seconds >                                "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_TIMER_INTERVAL_ms   <parms to choose time in milliseconds us>                         "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  to get current EOL settings:                                                                         "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_PRINT_EOL_DATA                                                                            "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  show configuration i.e. those set config.h                                                           "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SHOW_CONFIGURATION   .............                                                        "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  enable and disable cyclic diagnosis - useful to better observe other traces                          "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_CYCLIC_DIAG <ENABLE,DISABLE>   .............                                              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_CYCLIC_DIAG_PERMANENT <ENABLE,DISABLE>   .............                                    "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  helper for tests to disconnect VD_DeviceManager from clients and drivers                             "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_INTERFACE_CONNECT <PRM,UDEV,CCAClients>  <CONNECT,DISCONNECT>                           "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  manipulate service available - see also define TEST_START_WITH_NO_SERVICEAVAILABLE in config.h       "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SETSERVICE <AVAILABLE,UNAVAILABLE>                                                        "));
    ETG_TRACE_FATAL(("[ok]    alternatively                                                                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SETSERVICE_AVAILABLE                                                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SETSERVICE_UNAVAILABLE                                                                    "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  show current connected devices                                                                       "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_GET_CONNECTED_DEVICES   <ALL,USB1,USB2>                                                   "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]  set EOL values for test                                                                            "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES  INFO_USB_VOLTAGE_RECOVERY_AUTOPLAY_DELAY_TIME  <delaytime>            "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES  INFO_USB_1_HUB_MAX_PORTS  <No of ports>                               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES  INFO_USB_2_HUB_MAX_PORTS  <No of ports>                               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_ENABLE_MTP_DEVICES <DISABLE/ENABLE>                               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_OFFERING_BYTE1 <value of byte>                             "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_BYTE1Bit3 <DISABLE/ENABLE>                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_BYTE1Bit4 <DISABLE/ENABLE>                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_BYTE1Bit5 <DISABLE/ENABLE>                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_BYTE1Bit6 <DISABLE/ENABLE>                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_EOLVALUES_INFO_SYSTEM_BYTE1Bit7 <DISABLE/ENABLE>                                 "));
    ETG_TRACE_FATAL(("[ok]                                                                                                       "));
    ETG_TRACE_FATAL(("[ok]   set configuration values generic                                                                    ")); 
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_SD            <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_USB           <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_MTP           <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_IPOD          <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_IPAD          <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_IPHONE        <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_INTERNAL_CDROM         <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_INTERNAL_SDCARD        <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_EXTERNAL_UNSUPPORTED   <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]                                                                                                      "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_USB1                   <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_USB2                   <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_USB3                   <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_USB4                   <DISABLE/ENABLE>              "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HUBatUSB1              <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HUBatUSB2              <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HUBatUSB3              <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HUBatUSB4              <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_observeDTC_HUBatUSB1   <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_observeDTC_HUBatUSB2   <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_observeDTC_HUBatUSB3   <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_observeDTC_HUBatUSB4   <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HW_USB1                <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HW_USB2                <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HW_USB3                <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HW_USB4                <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_FAT         <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_NTFS        <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_EXTFAT      <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_HFSPLUS     <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_EXT_3_4_FS  <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_ISO9660     <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FILESYSTEM_UDF         <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_LIMIT_SIZE_MSD         <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_CHARGE_APPLEDEVS       <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_UNSUPPORTED_DEVTYPE    <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_UNSUPPORTED_FILESYSTEM <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_UNSUPPORTED_HUB        <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_FAKE_SDCARD            <DISABLE/ENABLE>               "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  set EOL values in file e.g. using TTFIS MCNet commands - then trigger to read this                    "));
    ETG_TRACE_FATAL(("[ok]      CMD VD_DVM_TRIGGER_READ_EOLVALUES_FROMFILE                                                        "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  simulate platform behaviour for undervoltage for USB-sticks and SD-cards                              "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_UNDERVOLTAGE_PLATFORM_DTY_USB  (not yet for new automounter)                      "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  allow to connect HUB at HUB for development - e.g. used in VAGMIBE where USBCable for IPOD            "));
    ETG_TRACE_FATAL(("[ok]  is a HUB with one connector to connect a device plus Trace at it wnother HUB has to be connected      "));
    ETG_TRACE_FATAL(("[ok]  to it for development purpose. I.e. to receive traces while connecting a device                       "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_PERMANENT_HUBCONNECTEDHUB <ENABLE,DISABLE>                                                 "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  simulate FC_HeatControl sending overtemperature for CDDRIVE                                           "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_CDDRIVE_OVERTEMPERATURE <TRUE/FALSE>                                              "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  simulate FC_Diagnosis sending Mount/Unmount command to mount/unmount a MixedMode CD                   "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_MIXMODE_CDROM_MOUNT <EMPTY/SR0/SR1/SR2>                                           "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_SIMULATE_MIXMODE_CDROM_UNMOUNT <EMPTY/SR0/SR1/SR2>                                         "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  for targets with internal SDCard reader - receive information about the card normally used by diag    "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_PRMMANAGER_vGetSDCardState                                                            "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  check if power supply of USB connector is switched off/on                                             "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_PRMMANAGER_bGetUSBHardwareConnector     <USB1/USB2>                                   "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  power supply of USB connector: switched on                                                            "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_PRMMANAGER_bSetUSBHardwareConnector_ON  <USB1/USB2>                                   "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  power supply of USB connector: switched off                                                           "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_PRMMANAGER_bGetUSBHardwareConnector_OFF <USB1/USB2>                                   "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  System voltage e.g.   CRITICAL_LOW,OPERATING (NORMAL)                                                 "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_VOLTAGEMANAGER_vGetSystemVoltage                                                      "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  Board voltage in milliVolt                                                                            "));
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_VOLTAGEMANAGER_vGetBoardVoltage                                                       "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  Send voltage warning triggered TTFIS send to StateTable                                               "))
    ETG_TRACE_FATAL(("[ok]      VD_DVM_TEST_VOLTAGEMANAGER_vSendVoltageWarning  <SIG_FALSE/SIG_TRUE>                              "));
    ETG_TRACE_FATAL(("[ok]                                                                                                        "));
    ETG_TRACE_FATAL(("[ok]  ------------------------------------------------------------------------------------------------------"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdInterfaceConnect(tPCU8 pU8Data)                               *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdInterfaceConnect(tPCU8 pU8Data) const
{
    tInt interfaceDVM;
    tBool bConnect;

    ETG_TRACE_USR4(("Begin: TTFIScmd_InterfaceConnect"));

    if(NULL!=pU8Data)
    {
        interfaceDVM  =                pU8Data[(tInt)index_cmd+1];
        bConnect     = mapConnectParm(pU8Data[(tInt)index_cmd+2]);
        switch(interfaceDVM)
        {
            case TTFIScmd_InterfaceConnect_PRM:
                if(m_pIPrmManager)
                {
                    m_pIPrmManager->connectPRM(bConnect);
                }
                else
                {
                    ETG_TRACE_FATAL(("[INFO] TTFIScmdInterfaceConnect: m_pIPrmManager == NULL"));
                }
                break;

            case TTFIScmd_InterfaceConnect_UDEV:
                CUDevManager::GetInstance()->connectUDEV(bConnect);
                break;

            case TTFIScmd_InterfaceConnect_CCAclients:
                DeviceManagerInterface::GetInstance()->vConnectClients(bConnect);
                break;

            default:
                ETG_TRACE_FATAL(("ERROR: TTFIScmd_InterfaceConnect: This IF does not exist!"));
        }
    }
  
    ETG_TRACE_USR4(("End  : TTFIScmd_InterfaceConnect"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdCyclicDiagnosis(tPCU8 pU8Data)                                *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdCyclicDiagnosis(tPCU8 pU8Data) const
{
    tInt enable;
    ETG_TRACE_USR4(("Begin: TTFIScmd_CyclicDiagnosis"));

    if(NULL!=pU8Data)
    {
        enable=pU8Data[(tInt)index_cmd+1];
        switch(enable)
        {
            case TTFIScmd_Disable:
                DeviceManagerInterface::GetInstance()->vDiagEnable(FALSE);
                break;
            case TTFIScmd_Enable:
                DeviceManagerInterface::GetInstance()->vDiagEnable(TRUE);
                break;
            default:
                ETG_TRACE_FATAL(("ERROR: TTFIScmd_CyclicDiagnosis: This IF does not exist!"));
                break;
        }
    }
    ETG_TRACE_USR4(("End  : TTFIScmd_CyclicDiagnosis"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSimUSBDevice(tPCU8 pU8Data)                                   *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSimUSBDevice(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdSimUSBDevice"));
    (void)pU8Data;
    ETG_TRACE_USR4(("End  : TTFIScmdSimUSBDevice"));
}

/*-----------------------------------------------------------------------------*
 * tVoid mapConnectParm(tPCU8 pU8Data)                                         *
 *-----------------------------------------------------------------------------*/
tBool TraceCmdManager::mapConnectParm(tU8 u8Value) const
{
    tBool bResult;
    tenDevManagerTTFIScmd_InterfaceConnectParm2 eConnect;

    eConnect = (tenDevManagerTTFIScmd_InterfaceConnectParm2)u8Value;
    switch(eConnect)
    {
        case TTFIScmd_InterfaceConnect_Disconnect:
            bResult=FALSE;
            break;
        case TTFIScmd_InterfaceConnect_Connect:
            bResult=TRUE;
            break;
        default:
            ETG_TRACE_FATAL(("ERROR:TraceCmdManager::mapConnectParm: WRONG VALUE: 0x%x",u8Value));
            bResult = TRUE;
            break;
    }

    return bResult;
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdGetConnectedDevices(tPCU8 pU8Data)                            *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdGetConnectedDevices(tPCU8 pU8Data)
{
    tenDevManagerUSBHost usb;
    ETG_TRACE_USR4(("Begin: TTFIScmdGetConnectedDevices"));
    if(NULL == pU8Data)
    {
        usb = eUSBUndef; //if pU8Data=NULL then print all
    }
    else
    {
        usb =(tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+1];
    }
    if((eUSBUndef<=usb) && ((tInt)usb <ARRAYSIZEFORUSB) && (NULL!=m_pStateTableApp))
    {
        DeviceListManager::GetInstance()->TraceAllDevice(DEVLIST_STATETABLE,      usb);
        DeviceListManager::GetInstance()->TraceAllDevice(DEVLIST_UDEVMGR,         usb);
        DeviceListManager::GetInstance()->TraceAllDevice(DEVLIST_DEVICEMGRSERVICE,usb);

        // dump to file, if available
        if(NULL != m_poDumpFile)
       {
            DeviceListManager::GetInstance()->TraceAllDevice2Logfile(m_poDumpFile,DEVLIST_STATETABLE,      usb);
            DeviceListManager::GetInstance()->TraceAllDevice2Logfile(m_poDumpFile,DEVLIST_UDEVMGR,         usb);
            DeviceListManager::GetInstance()->TraceAllDevice2Logfile(m_poDumpFile,DEVLIST_DEVICEMGRSERVICE,usb);
       }

    }

    ETG_TRACE_USR4(("End  : TTFIScmdGetConnectedDevices"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryUdevEvts(tVoid)                                        *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryUdevEvts(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryUdevEvts"));

    DEVICEPROCESSORHISTORY.TraceHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryUdevEvts"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryUdevEvts(tVoid)                                        *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryUdevEvts_Clear(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryStoredUdevEvts_Clear"));

    DEVICEPROCESSORHISTORY.ClearHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryStoredUdevEvts_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryLastMsgsSend2Clients(tVoid)                            *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryLastMsgsSend2Clients(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryLastMsgsSend2Clients"));
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_NO_REDUNDAND_MSGS))
    {
        INTERFACEHISTORY.TraceHistory();
    }
    else
    {
        SERVICEHISTORY.TraceHistory();
    }

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryLastMsgsSend2Clients"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryLastMsgsSend2Clients_Clear(tVoid)                      *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryLastMsgsSend2Clients_Clear(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryLastMsgsSend2Clients_Clear"));

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_NO_REDUNDAND_MSGS))
    {
        INTERFACEHISTORY.ClearHistory();
    }
    else
    {
        SERVICEHISTORY.ClearHistory();
    }
  

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryLastMsgsSend2Clients_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSetServiceAvailable(tPCU8 pU8Data)                            *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSetServiceAvailable(tPCU8 pU8Data) const
{
    tU8 u8Val = 0;
    tBool bAvailable = FALSE;
    ETG_TRACE_USR4(("Begin: TTFIScmdSetService"));

    if(NULL!=pU8Data)
    {
        u8Val = pU8Data[(tInt)index_cmd+1];
    }
    if(u8Val ==1)
    {
        bAvailable = TRUE;
    }
    else
    {
        bAvailable = FALSE;
    }

    DeviceManagerInterface::GetInstance()->vSetServiceAvailable(bAvailable);
    
    ETG_TRACE_FATAL(("[ok] TTFIScmdSetServiceAvailable 0x%x",bAvailable));
    ETG_TRACE_USR4(("End  : TTFIScmdSetService"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSimEOLValues(tPCU8 pU8Data)                                   *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSimEOLValues(tPCU8 pU8Data) const
{
    tU8   u8Parm1,u8Parm2,u8Parm3;
    tU16  u16Val;

    ETG_TRACE_USR4(("Begin: TTFIScmdSimEOLValues"));

    if(NULL != pU8Data)
    {
        u8Parm1 = pU8Data[(tInt)index_cmd+1];
        u8Parm2 = pU8Data[(tInt)index_cmd+2];

        if( u8Parm1<(tU8)eCONF_LAST_ELEMENT )
        {
            //care for 2byte parameters
            if((tenDevManagerConfigValue)u8Parm1==eCONF_USB_VOLTAGE_RECOVERY_AUTOPLAY_DELAY_TIME)
            {
                u8Parm3 = pU8Data[(tInt)index_cmd+3];
                u16Val  = (u8Parm2 <<8) + u8Parm3;
            }
            else
            {
                u16Val = (tU16)u8Parm2;
            }

            ETG_TRACE_FATAL(("[ok] TTFIScmdSimEOLValues:type:%d value:%d",
                                 ETG_CENUM(tenDevManagerConfigValue,u8Parm1),u16Val));

            ConfigurationManager::GetInstance()->vSetConfigurationValue( (tenDevManagerConfigValue)u8Parm1,u16Val);

            //care that after the value has been set that it'll be used in the StateTable
            if((tenDevManagerConfigValue)u8Parm1==eCONF_STATETABLE_EXECUTION_TIME_MS)
            {
                if( m_pStateTableApp )
                {
                    m_pStateTableApp->updateIntervalTimeFromConfiguration();
                }
            }

            //care for dependent value  of eCONF_ENABLE_NOTIFY_UNSUPPORTED_DEVTYPE
            if(     (tenDevManagerConfigValue)u8Parm1 == eCONF_ENABLE_NOTIFY_UNSUPPORTED_DEVTYPE) 
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_NOTIFY_UNSUPPORTED_HUB,u16Val);
    

            //care for special GM values. Code of VD_DVM uses complementary generic values
            //the values could be taken from code currently kept for traditional reason
            if(     (tenDevManagerConfigValue)u8Parm1 == eCONF_SYSTEM_OFFERING_BYTE1Bit5_SDPresent) 
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_EXTERNAL_SD,u16Val);

            else if((tenDevManagerConfigValue)u8Parm1 == eCONF_ENABLE_MTP_DEVICES)
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_EXTERNAL_MTP,u16Val);

            else if((tenDevManagerConfigValue)u8Parm1 == eCONF_SYSTEM_OFFERING_BYTE1Bit7_USB1Present)
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_USB1,u16Val);

            else if((tenDevManagerConfigValue)u8Parm1 == eCONF_SYSTEM_OFFERING_BYTE1Bit6_USB2Present)
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_USB2,u16Val);


            else if((tenDevManagerConfigValue)u8Parm1 == eCONF_SYSTEM_OFFERING_BYTE1Bit4_USB1HubPresent)
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB1,u16Val);

            else if((tenDevManagerConfigValue)u8Parm1 == eCONF_SYSTEM_OFFERING_BYTE1Bit3_USB2HubPresent)
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB2,u16Val);

            else if(((tenDevManagerConfigValue)u8Parm1 == eCONF_USB_1_HUB_MAX_PORTS) && (u16Val >=DVM_ON))
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB1,u16Val);

            else if(((tenDevManagerConfigValue)u8Parm1 == eCONF_USB_2_HUB_MAX_PORTS) && (u16Val >=DVM_ON))
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB2,u16Val);

#ifdef DISABLED_OBSOLETE_CODE
            //I think this is already covered 
            else if(((tenDevManagerConfigValue)u8Parm1 == eCONF_ENABLE_TELEMATICSBOX_MONITORING))
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_TELEMATICSBOX_MONITORING,u16Val);

            else if(((tenDevManagerConfigValue)u8Parm1 == eCONF_ENABLE_TESTMODE_SETUSBHARDWARECONNECTOR))
                ConfigurationManager::GetInstance()->vSetConfigurationValue(eCONF_ENABLE_TESTMODE_SETUSBHARDWARECONNECTOR,u16Val);
#endif


            else
            {
                ETG_TRACE_FATAL(("[ok] TTFIScmdSimEOLValues: not a GM specificvalue"));
            }
  
        }
    }

    ETG_TRACE_USR4(("End  : TTFIScmdSimEOLValues"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdPrintConfiguration(tVoid)                                           *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdPrintConfiguration(tVoid)
{
    ETG_TRACE_USR4(("Begin: TTFIScmdPrintConfiguration"));
    ConfigurationManager *pConfigurationManager = ConfigurationManager::GetInstance();
    if(pConfigurationManager)
    {
        //if dump file is NULL then 
        pConfigurationManager->vPrintConfiguration(m_poDumpFile,TRUE /*modefor analysis and testing*/); //@todo ocleanup 
    }
    ETG_TRACE_USR4(("End  : TTFIScmdPrintConfiguration"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdTriggerReadEOLValuesFromFile(tVoid)                           *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdTriggerReadEOLValuesFromFile(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTriggerReadEOLValuesFromFile"));

    ConfigurationManager::GetInstance()->vInitCustomerValues();

    ETG_TRACE_USR4(("End  : TTFIScmdTriggerReadEOLValuesFromFile"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFISsimulateUndervoltage(tVoid)                                             *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdSimulateOverCurrentPlatformDTY_USB(tVoid) const 
{
    UsbPortState rPortState;

    ETG_TRACE_USR4(("Begin: TTFIScmdSimulateOverCurrentPlatformDTY_USB"));

    ETG_TRACE_FATAL(("[ok]: =============================================="));
    ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: "));
    ETG_TRACE_FATAL(("[ok]: =============================================="));

    if(NULL==m_pStateTableApp)
    {
        ETG_TRACE_FATAL((" [ERROR]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: NULL==m_pStateTableApp"));
        ETG_TRACE_ERRMEM(("[ERROR]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: NULL==m_pStateTableApp"));
        DVM_NORMAL_M_ASSERT_ALWAYS(); 
    }
    else
    {
        if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
        {
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: ------------------"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: SET HW SIGNALS:NOK"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: ------------------"));

            for(tU8 i =(tU8)eUSB1;i<= (tU8)ARRAYSIZEFORUSB;i++)
            {
                rPortState.u8PortNr     = (tU8)eUSB1;
                rPortState.u8UV         = (tU8)SIG_FALSE;
                rPortState.u8OC         = (tU8)SIG_TRUE;  //<-------------------------overcurrent
                rPortState.u8PPON       = (tU8)SIG_FALSE; //<-------------------------overcurrent

                rPortState.u32OCStartTime  =0;
                rPortState.u32OCEndTime    =0;
                rPortState.u32UVStartTime  =0;
                rPortState.u32UVEndTime    =0;
                rPortState.u32PPONStartTime=0;
                rPortState.u32PPONEndTime  =0;

                m_pStateTableApp->setHWSignals(&rPortState);
            }
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: REMOVE devices from /tmp/.automount"));

            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_MKDIR, "/tmp/tmp_automount");
            GENSTRING sparameters;
            sparameters = GENSTRING("/tmp/.automount/* ") + GENSTRING("/tmp/tmp_automount/");
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_CP, sparameters.toStdString().c_str());
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_RM, "/tmp/.automount/*");
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: WAIT 5 seconds"));
            sleep(5);

            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: ------------------"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: SET HW SIGNALS: OK"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: ------------------"));
            for(tU8 i =(tU8)eUSB1;i<= (tU8)ARRAYSIZEFORUSB;i++)
            {
                rPortState.u8PortNr     = (tU8)eUSB1;
                rPortState.u8UV         = (tU8)SIG_FALSE;
                rPortState.u8OC         = (tU8)SIG_FALSE;//<-------------------------normal
                rPortState.u8PPON       = (tU8)SIG_TRUE; //<-------------------------normal

                rPortState.u32OCStartTime  =0;
                rPortState.u32OCEndTime    =0;
                rPortState.u32UVStartTime  =0;
                rPortState.u32UVEndTime    =0;
                rPortState.u32PPONStartTime=0;
                rPortState.u32PPONEndTime  =0;

                m_pStateTableApp->setHWSignals(&rPortState);
            }

            sleep(1); ///@todo without this pause this will not work - this must be checked!!!!!!!!!!!!!!!!!!!!!!!!!
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateOverCurrentPlatformDTY_USB: add devices to /tmp/.automount"));

            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_MKDIR, "/tmp/tmp_automount");
            sparameters = GENSTRING("/tmp/tmp_automount/* ") + GENSTRING("/tmp/.automount/");
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_CP, sparameters.toStdString().c_str());
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_RMDIR, "/tmp/tmp_automount");
        }
        else
        {
            ETG_TRACE_FATAL(("TTFIScmdSimulateOverCurrentPlatformDTY_USB Cannot be executed as the  feature temporary not available is enabled!!!    "));
        }

    }
    ETG_TRACE_FATAL(("TTFIScmdSimulateOverCurrentPlatformDTY_USB: Not yet implemented")); //Roamap 130024 UndervoltageHandling - placeholder

    ETG_TRACE_USR4(("End  : TTFIScmdSimulateOverCurrentPlatformDTY_USB"));

}

/*-----------------------------------------------------------------------------*
 * tVoid TTFISsimulateUndervoltage(tVoid)                                             *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdSimulateUndervoltagePlatformDTY_USB(tVoid) const 
{
    UsbPortState rPortState;

    ETG_TRACE_USR4(("Begin: TTFIScmdSimulateUndervoltagePlatformDTY_USB"));

    ETG_TRACE_FATAL(("[ok]: =============================================="));
    ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: "));
    ETG_TRACE_FATAL(("[ok]: =============================================="));

    if(NULL==m_pStateTableApp)
    {
        ETG_TRACE_FATAL((" [ERROR]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: NULL==m_pStateTableApp"));
        ETG_TRACE_ERRMEM(("[ERROR]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: NULL==m_pStateTableApp"));
        DVM_NORMAL_M_ASSERT_ALWAYS(); 
    }
    else
    {
        if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MALFUNCTION_USBCONNECTORS))
        {
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: ------------------"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: SET HW SIGNALS:NOK"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: ------------------"));

            for(tU8 i =(tU8)eUSB1;i<= (tU8)ARRAYSIZEFORUSB;i++)
            {
                rPortState.u8PortNr     = (tU8)eUSB1;
                rPortState.u8UV         = (tU8)SIG_TRUE;//<-------------------------under voltage - USB connector disabled by HW
                rPortState.u8OC         = (tU8)SIG_FALSE;   //any is possible
                rPortState.u8PPON       = (tU8)SIG_TRUE;    //any is possible

                rPortState.u32OCStartTime  =0;
                rPortState.u32OCEndTime    =0;
                rPortState.u32UVStartTime  =0;
                rPortState.u32UVEndTime    =0;
                rPortState.u32PPONStartTime=0;
                rPortState.u32PPONEndTime  =0;

                m_pStateTableApp->setHWSignals(&rPortState);
            }
            //@todo this part will give trouble with new autmounter
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: REMOVE devices from /tmp/.automount"));

            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_MKDIR, "/tmp/tmp_automount");
            GENSTRING sparameters;
            sparameters = GENSTRING("/tmp/.automount/* ") + GENSTRING("/tmp/tmp_automount/");
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_CP, sparameters.toStdString().c_str());
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_RM, "/tmp/.automount/*");
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: WAIT 5 seconds"));
            sleep(5);

            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: ------------------"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: SET HW SIGNALS: OK"));
            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: ------------------"));
            for(tU8 i =(tU8)eUSB1;i<= (tU8)ARRAYSIZEFORUSB;i++)
            {
                rPortState.u8PortNr     = (tU8)eUSB1;
                rPortState.u8UV         = (tU8)SIG_FALSE;//<-------------------------normal
                rPortState.u8OC         = (tU8)SIG_FALSE;   //any is possible
                rPortState.u8PPON       = (tU8)SIG_TRUE;  //any is possible

                rPortState.u32OCStartTime  =0;
                rPortState.u32OCEndTime    =0;
                rPortState.u32UVStartTime  =0;
                rPortState.u32UVEndTime    =0;
                rPortState.u32PPONStartTime=0;
                rPortState.u32PPONEndTime  =0;

                m_pStateTableApp->setHWSignals(&rPortState);
            }

            sleep(1); ///@todo without this pause this will not work - this must be checked!!!!!!!!!!!!!!!!!!!!!!!!!

            ETG_TRACE_FATAL(("[ok]: TTFIScmdSimulateUndervoltagePlatformDTY_USB: add devices to /tmp/.automount"));

            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_MKDIR, "/tmp/tmp_automount");
            sparameters = GENSTRING("/tmp/tmp_automount/* ") + GENSTRING("/tmp/.automount/");
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_CP, sparameters.toStdString().c_str());
            execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_RMDIR, "/tmp/.automount/");
        }
        else
        {
            ETG_TRACE_FATAL(("TTFIScmdSimulateUndervoltagePlatformDTY_USB Cannot be executed as the  feature temporary not available is enabled!!!    "));
        }
        

    }
    ETG_TRACE_FATAL(("TTFIScmdSimulateUndervoltagePlatformDTY_USB: Not yet implementer")); //Roamap 130024 UndervoltageHandling - placeholder

    ETG_TRACE_USR4(("End  : TTFIScmdSimulateUndervoltagePlatformDTY_USB"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdGetConnectedHubs(tVoid)                                       *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdGetConnectedHubs(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdGetConnectedHubs"));

    CUDevManager::GetInstance()->TraceConnectedHubs();

    ETG_TRACE_USR4(("End  : TTFIScmdGetConnectedHubs"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryLastMsgsSend2Diag(tVoid)                               *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdHistoryLastMsgsSend2Diag(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryLastMsgsSend2Diag"));

    DIAGCLIENTHISTORY.TraceHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryLastMsgsSend2Diag"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryLastMsgsSend2Diag_Clear(tVoid)                         *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdHistoryLastMsgsSend2Diag_Clear(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryLastMsgsSend2Diag_Clear"));

    DIAGCLIENTHISTORY.ClearHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryLastMsgsSend2Diag_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdBlockMonitor_Permanent(tPCU8 pU8Data)                         *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdBlockMonitor_Permanent(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdBlockMonitor_Permanent"));

    permanentMechanismFileBased(pU8Data,STARTUP_DEVICEMONITOR_DISABLEFILE,STARTUP_DEVICEMONITOR_DISABLEFILETEXT);

    ETG_TRACE_USR4(("End  : TTFIScmdBlockMonitor_Permanent"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdServiceUnavailable_Permanent(tPCU8 pU8Data)                   *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdServiceUnavailable_Permanent(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_ServiceUnavailable_Permanent"));

    permanentMechanismFileBased(pU8Data,STARTUP_SERVICEAVAILABLE_DISABLEFILE,STARTUP_SERVICEAVAILABLE_DISABLEFILETEXT);

    ETG_TRACE_USR4(("End  : TTFIScmd_ServiceUnavailable_Permanent"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdCyclicDiagnosis_Permanent(tPCU8 pU8Data)                      *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdCyclicDiagnosis_Permanent(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdCyclicDiagnosis_Permanent"));

    TTFIScmdCyclicDiagnosis(pU8Data);
    permanentMechanismFileBased(pU8Data,CYLCIC_DIAGNOSIS_DISABLEFILE,CYCLIC_DIAGNOSIS_DISABLEFILETEXT);

    ETG_TRACE_USR4(("End  : TTFIScmdCyclicDiagnosis_Permanent"));
}

/*-----------------------------------------------------------------------------*
 * tVoid permanentMechanismFileBased(tPCU8 pU8Data,tPCChar pFileName,          *
 *          tPCChar pFileText)                                                 *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::permanentMechanismFileBased(tPCU8 pU8Data,tPCChar pFileName,tPCChar pFileText) const
{
    ETG_TRACE_USR4(("Begin: permanentMechanismFileBased"));

    tInt enable;
     FILE *l_pDisableFile = NULL;

    if( (NULL!=pU8Data) && (NULL!=pFileName) && (NULL!=pFileText))
    {
        enable=pU8Data[(tInt)index_cmd+1];
        switch(enable)
        {
            case TTFIScmd_Enable:
                /*additionally write */
                if(remove(pFileName) != 0)
                {
                    ETG_TRACE_FATAL(("permanentMechanismFileBased: Error delete File %s",pFileName));

                }
                else
                {
                   ETG_TRACE_FATAL(("permanentMechanismFileBased: ok delete File %s",pFileName));
                }
                break;
            case TTFIScmd_Disable:
                if (NULL != (l_pDisableFile = fopen (pFileName, "w")))
                {
                    ETG_TRACE_FATAL(("permanentMechanismFileBased: File created: %s",pFileName));
                    fputs(pFileText,l_pDisableFile);
                    fclose(l_pDisableFile);


                }
                else
                {
                    ETG_TRACE_ERR (("TTFIScmdCyclicDiagnosis_Permanent: ERROR: fopen failed"));
                }
                break;
            default:
                ETG_TRACE_FATAL(("ERROR: permanentMechanismFileBased: This IF does not exist!"));
        }
    }
    ETG_TRACE_USR4(("End  : permanentMechanismFileBased"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryCallbacksPrm(tVoid)                                    *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryCallbacksPrm(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryCallbacksPrm"));

    PRMHISTORY.TraceHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryCallbacksPrm"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryCallbacksPrm_Clear(tVoid)                              *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryCallbacksPrm_Clear(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryCallbacksPrm_Clear"));

    PRMHISTORY.ClearHistory();

    ETG_TRACE_USR4(("End  : TTFIScmdHistoryCallbacksPrm_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryNotificationsSpm(tVoid)                                *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryCallbacksVolt(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryCallbackVolt"));
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_VOLTAGEMANAGER))
    {
        VOLTCLIENTHISTORY.TraceHistory();
    }
    else
    {
        ETG_TRACE_FATAL(("TTFIScmdHistoryCallbackVolt; VoltageManager not connected to state table"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmdHistoryCallbackVolt"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryNotificationsSpm_Clear(tVoid)                          *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryCallbacksVolt_Clear(tVoid) const
{
     ETG_TRACE_USR4(("Begin: TTFIScmdHistoryCallbacksVolt_Clear"));
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_VOLTAGEMANAGER))
    {
        VOLTCLIENTHISTORY.ClearHistory();
    }
    else
    {
        ETG_TRACE_FATAL(("TTFIScmdHistoryCallbacksVolt_Clear; VoltageManager not connected to state table"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmdHistoryCallbacksVolt_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdCheckAutomountDir(tVoid)                                      *
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdCheckAutomountDir(tVoid)
{
    ETG_TRACE_FATAL(("[ok] TTFIScmdCheckAutomountDir"));

    DIR *automount = opendir(INOTIFY_DIR);
    GENSTRING l_cDevplusMPoint;

    if (NULL == automount)
    {
        ETG_TRACE_FATAL(("TTFIScmdCheckAutomountDir: [ERROR] opendir failed:"));
    }
    else
    {
        GENSTRING l_cDevNode;
        struct dirent *device = NULL;
        do
        {
            device = readdir(automount);

            if (NULL != device)
            {
                l_cDevNode = GENSTRING ("/dev/") + GENSTRING(device->d_name);

                if(l_cDevNode.contains("/dev/sd"))
                {
                    char    mnt[PATH_MAX];
                    memset( mnt, '\0', PATH_MAX);
                    mnt[PATH_MAX-1] = '\0';
                    GENSTRING l_cFilePath   = GENSTRING (INOTIFY_DIR) + GENSTRING (device->d_name);
                    int     fd            = open(l_cFilePath.toStdString().c_str(), O_RDONLY);

                    if (-1 != fd)
                    {
                        ssize_t size = read(fd, mnt, PATH_MAX);
                        mnt[PATH_MAX-1] = '\0'; // sim4hi: fix for Coverity CID-18517
                        if (-1 != size)
                        {
                            l_cDevplusMPoint = GENSTRING("device :") + l_cDevNode + GENSTRING("   mount point:") + GENSTRING(mnt);
                            ETG_TRACE_FATAL(("[ok] TTFIScmdCheckAutomountDir: %s",l_cDevplusMPoint.toStdString().c_str()));

                            // dump to file, if available
                            if (NULL != m_poDumpFile)
                            {
                                fprintf(m_poDumpFile,
                                        "[ok] TTFIScmdCheckAutomountDir: %s\n",
                                        l_cDevplusMPoint.toStdString().c_str());
                            }
                        }
                        close(fd);
                    }
                }
            }
        } while (NULL != device);

        (void)closedir(automount);
    }
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryStateTable(tVoid)                                      *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryStateTable(tVoid) const
{
    // Do nothing
}


#ifdef TTFISCMD_PRODUCTIONLINETEST
/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdProductionLineTest(tVoid)                                     *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdProductionLineTest(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdProductionLineTest"));

    DeviceManagerInterface::GetInstance()->vOnDiagnosisProductionLineTEST();

    ETG_TRACE_USR4(("End  : TTFIScmdProductionLineTest"));
}
#endif




#ifdef USE_HUB_CONNECTED_TO_HUB
/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdProductionLineTest(tVoid)                                     *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdEnableHubConnectedToHubPermanent(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdEnableHubConnectedToHubPermanent pU8Data[(tInt)index_cmd+1 %d",pU8Data[(tInt)index_cmd+1]));
    tBool bEnable = FALSE;
    if(NULL!=pU8Data)
    {
        //inverse logic, TTFIScmd_Disable means we have to enable the HubConnectedToHubPermanent feature
        if(TTFIScmd_Disable == pU8Data[(tInt)index_cmd+1])
        {
            bEnable = TRUE;
        }
        ConfigurationManager::GetInstance()->vEnableHubConnectedToHubPermanent(bEnable);
    }

    permanentMechanismFileBased(pU8Data,USE_HUB_CONNECTED_TO_HUB_ALLOW_IT_FILE,USE_HUB_CONNECTED_TO_HUB_ALLOW_IT_FILETEXT);

    ETG_TRACE_USR4(("End  : TTFIScmdEnableHubConnectedToHubPermanent"));
}
#endif

tVoid TraceCmdManager::TTFIScmdSimulateCDDriveOverTemperature(tPCU8 pU8Data) const //Roadmap 13035 Overtemperature
{
    ETG_TRACE_USR4(("Begin: TTFIScmdSimulateCDDriveOverTemperature"));

    tBool bOverTemperature;

    if(m_pStateTableApp)
    {
        if(pU8Data)
        {
            bOverTemperature=(tBool)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("[ok]  bOverTemperature=0x%x",bOverTemperature));
            m_pStateTableApp->vSetOverTemperatureCDDrive(bOverTemperature);
        }
        else
        {
           ETG_TRACE_FATAL(("[ERROR]: TTFIScmdSimulateCDDriveOverTemperature: NULL == m_pStateTableApp"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: TTFIScmdSimulateCDDriveOverTemperature: NULL == m_pStateTableApp"));
    }

    ETG_TRACE_USR4(("End  : TTFIScmdSimulateCDDriveOverTemperature"));
}

tVoid TraceCmdManager::TTFIScmd_SimulateMixeModeCDRomMount(tPCU8 pU8Data) const     //Roadmap_14002_MountCDRom (MixeMode)
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_SimulateMixeModeCDRomMount"));

    tU8 u8DeviceNameType;
    std::string strDeviceName;
    if(m_pStateTableApp)
    {
        if(pU8Data)
        {
            u8DeviceNameType=(tU8)pU8Data[(tInt)index_cmd+1];
       /* see VD_DeviceManager.trc
            ARRAY DEVICENAME = 0 : EMPTY
            ARRAY DEVICENAME = 1 : SR0
            ARRAY DEVICENAME = 2 : SR1
            ARRAY DEVICENAME = 3 : SR2
            */
            switch(u8DeviceNameType)
            {
                case 0:
                    strDeviceName.clear();
                    break;
                case 1:
                    strDeviceName = "/dev/sr0";
                    break;
                case 2:
                    strDeviceName = "/dev/sr1";
                    break;
                case 3:
                    strDeviceName = "/dev/sr2";
                    break;
                case 4:
                    strDeviceName = "/dev/sda1";
                    break;
                case 5:
                    strDeviceName = "/dev/sdb1";
                    break;
                default:
                    strDeviceName.clear();
                    break;
            }
            ETG_TRACE_FATAL(("[ok]  TTFIScmd_SimulateMixeModeCDRomMount: MOUNT:=%s",strDeviceName.c_str()));
            m_pStateTableApp->vDiagMountCDRom(TRUE /*TRUE=MOUNT*/, strDeviceName);
        }
        else
        {
           ETG_TRACE_FATAL(("[ERROR]: TTFIScmd_SimulateMixeModeCDRomMount: NULL == pU8Data"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: TTFIScmdSimulateCDDriveOverTemperature: NULL == m_pStateTableApp"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmd_SimulateMixeModeCDRomMount"));
}


tVoid TraceCmdManager::TTFIScmd_SimulateMixeModeCDRomUnMount(tPCU8 pU8Data) const //Roadmap_14002_MountCDRom (MixeMode)
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_SimulateMixeModeCDRomUnMount"));

    tU8 u8DeviceNameType;
    std::string strDeviceName;
    if(m_pStateTableApp)
    {
        if(pU8Data)
        {
            u8DeviceNameType=(tU8)pU8Data[(tInt)index_cmd+1];
       /* see VD_DeviceManager.trc
            ARRAY DEVICENAME = 0 : EMPTY
            ARRAY DEVICENAME = 1 : SR0
            ARRAY DEVICENAME = 2 : SR1
            ARRAY DEVICENAME = 3 : SR2
            */
            switch(u8DeviceNameType)
            {
                case 0:
                    strDeviceName.clear();
                    break;
                case 1:
                    strDeviceName = "/dev/sr0";
                    break;
                case 2:
                    strDeviceName = "/dev/sr1";
                    break;
                case 3:
                    strDeviceName = "/dev/sr2";
                    break;
                case 4:
                    strDeviceName = "/dev/sda1";
                    break;
                case 5:
                    strDeviceName = "/dev/sdb1";
                    break;
                default:
                    strDeviceName.clear();
                    break;
            }
            ETG_TRACE_FATAL(("[ok]  TTFIScmd_SimulateMixeModeCDRomUnMount: UNMOUNT:=%s",strDeviceName.c_str()));
            m_pStateTableApp->vDiagMountCDRom(FALSE /*FALSE=UNMOUNT*/, strDeviceName);
        }
        else
        {
           ETG_TRACE_FATAL(("[ERROR]: TTFIScmd_SimulateMixeModeCDRomUnMount: NULL == pU8Data"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: TTFIScmd_SimulateMixeModeCDRomUnMount: NULL == m_pStateTableApp"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmd_SimulateMixeModeCDRomUnMount"));
}

/*-----------------------------------------------------------------------------*
 * void TTFIScmdTriggerPrmConfiguration(const FILE*& poDumpFile)                                   *
 *-----------------------------------------------------------------------------*/
void TraceCmdManager::TTFIScmdTriggerPrmConfiguration(tVoid) const //Roadmap_14002_MountCDRom (MixeMode)
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTriggerPrmConfiguration"));


    if(m_pIPrmManager)
    {
        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USR_PRM_HWCTRLUSB))
        {
            m_pIPrmManager->bInitUSBHardwareConnectors();
        }
        else
        {
            ETG_TRACE_FATAL(("[WARNING] TTFIScmdTriggerPrmConfiguration:eConf_ENABLE_USR_PRM_HWCTRLUSB DIASABLED"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFIScmdTriggerPrmConfiguration: m_pIPrmManager == NULL"));
    }


    ETG_TRACE_USR4(("End  : TTFIScmdTriggerPrmConfiguration"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceCmdManager::TTFIScmd_PRMMANAGERvGetSDCardState(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmd_PRMMANAGERvGetSDCardState(tVoid) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_PRMMANAGERvGetSDCardState"));

    if(m_pIPrmManagerCard)
    {
        tChar cUUIDDEviceName[STRING_LENGTH];
        cUUIDDEviceName[0] = '\0';

        CDeviceCard oCard;
        oCard.m_bHW_WriteProtected = FALSE;
        oCard.m_bSW_WriteProtected = FALSE;
        oCard.m_bMounted           = FALSE;
        oCard.m_u8ManufactureId    = 0xFF;
        oCard.m_u32SerialNumber    = 0xFFFFFFFF;
        oCard.m_u8SDCardSpecVersion= 0xFF;
        oCard.m_u64CardSize        = OSAL_C_U64_MAX;
        for( tInt i=0; i<SizeCIDRegister; ++i )
        {
            oCard.m_u8CIDRegister[i] = 0; 
        }

        if(FALSE == m_pIPrmManagerCard->bGetSDCardState(oCard,cUUIDDEviceName))
        {
            ETG_TRACE_FATAL(("[WARNIGN] TFALSE==bGetSDCardState(...) - PRmManager not initialised yet"));
        }
    
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] m_pIPrmManager == NULL"));
    }
    ETG_TRACE_USR4(("End  : TTFIScmd_PRMMANAGERvGetSDCardState"));
}
 
/*-----------------------------------------------------------------------------*
 * tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbGetUSBHardwareConnector(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbGetUSBHardwareConnector(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_PRMMANAGERbGetUSBHardwareConnector"));

    tBool          bSuccess;
    tU8            u8USBConnectorNum;
    tenSignalValue eAvtivated;

    if(m_pIPrmManager)
    {
        if(pU8Data)
        {
            u8USBConnectorNum =(tU8)pU8Data[(tInt)index_cmd+1];
            bSuccess = m_pIPrmManager->bGetUSBHardwareConnector(IN (tenDevManagerUSBHost)u8USBConnectorNum,OUT eAvtivated);
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbGetUSBHardwareConnector: [RET] bSuccess          : 0x%x", bSuccess));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbGetUSBHardwareConnector: [IN ] u8USBConnectorNum : USB%d", u8USBConnectorNum));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbGetUSBHardwareConnector: [OUT] eAvtivated        :    %d", ETG_ENUM(TENSIGNALVALUE,eAvtivated)));
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] TTFiscmd_PRMMANAGERbGetUSBHardwareConnector: pU8Data == NULL"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_PRMMANAGERbGetUSBHardwareConnector: m_pIPrmManager == NULL"));
    }
    ETG_TRACE_USR4(("End  : TTFiscmd_PRMMANAGERbGetUSBHardwareConnector"));
}
  
/*-----------------------------------------------------------------------------*
 * tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON"));
   
    tBool bSuccess;

    if(m_pIPrmManager)
    {
        if(pU8Data)
        {
            tU8 u8USBConnectorNum =(tU8)pU8Data[(tInt)index_cmd+1];
            bSuccess = m_pIPrmManager->bSetUSBHardwareConnector(IN (tenDevManagerUSBHost) u8USBConnectorNum,IN TRUE);
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON: [RET] bSuccess          : 0x%x", bSuccess));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON: [IN ] u8USBConnectorNum : USB%d", u8USBConnectorNum));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON: [IN ] bOn               :    %d", TRUE));
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON: pU8Data == NULL"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON: m_pIPrmManager == NULL"));
    }

    ETG_TRACE_USR4(("End  : TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF"));

    tBool bSuccess;
    if(m_pIPrmManager)
    {
        if(pU8Data)
        {
            tU8 u8USBConnectorNum =(tU8)pU8Data[(tInt)index_cmd+1];
            bSuccess = m_pIPrmManager->bSetUSBHardwareConnector(IN (tenDevManagerUSBHost) u8USBConnectorNum,IN FALSE);
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF: [RET] bSuccess          : 0x%x", bSuccess));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF: [IN ] u8USBConnectorNum : USB%d", u8USBConnectorNum));
            ETG_TRACE_FATAL(("TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF: [IN ] bOn               :    %d", TRUE));
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF: pU8Data == NULL"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF: m_pIPrmManager == NULL"));
    }
    ETG_TRACE_USR4(("End  : TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF"));
}
   

/*-----------------------------------------------------------------------------*
 * TraceCmdManager::TTFiscmd_VoltageManagervGetSystemVoltage(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_VoltageManagervGetSystemVoltage(void) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_VoltageManagervGetSystemVoltage"));
  
    if(m_pIVoltageManager)
    {
        IVoltageManager::tenSystemVoltage l_eSystemVoltage = IVoltageManager::eVolt_Undefined;
        m_pIVoltageManager->vGetSystemVoltage(OUT l_eSystemVoltage);
        ETG_TRACE_FATAL(("TTFiscmd_VoltageManagervGetSystemVoltage: l_eSystemVoltage: %d",ETG_CENUM(VoltageManager::tenSystemVoltage,l_eSystemVoltage)));
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_VoltageManagervGetSystemVoltage:m_pIVoltageManager = NULL"));
    }

    ETG_TRACE_USR4(("End  : TTFiscmd_VoltageManagervGetSystemVoltage"));
}
 
/*-----------------------------------------------------------------------------*
 * TraceCmdManager::TTFiscmd_VoltagemanagervGetBoardVoltage(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_VoltagemanagervGetBoardVoltage(void) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_VoltagemanagervGetBoardVoltage"));

    if(m_pIVoltageManager)
    {
        tU32 l_u32SystemVoltage = 0;
        m_pIVoltageManager->vGetBoardVoltage(OUT l_u32SystemVoltage);
        ETG_TRACE_FATAL(("TTFiscmd_VoltagemanagervGetBoardVoltage: l_u32SystemVoltage: %d",l_u32SystemVoltage));
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_VoltagemanagervGetBoardVoltage:m_pIVoltageManager = NULL"));
    }
    ETG_TRACE_USR4(("End  : TTFiscmd_VoltagemanagervGetBoardVoltage"));
   
}

/*-----------------------------------------------------------------------------*
 * TraceCmdManager::TTFiscmd_VoltagemanagervGetBoardVoltage(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFiscmd_VoltagemanagervSendVoltageWarning(tPCU8 pU8Data)
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_VoltagemanagervSendVoltageWarning"));
    if(m_pIVoltageManager)
    {
        if(pU8Data)
        {
            tenSignalValue l_eU60drop =(tenSignalValue)pU8Data[(tInt)index_cmd+1];
            ETG_TRACE_FATAL(("TTFiscmd_VoltagemanagervSendVoltageWarning: VoltageManager::GetInstance()->vSendVoltageWarning(/*l_eU60drop*/=%d",ETG_ENUM(TENSIGNALVALUE,l_eU60drop) ));
            m_pIVoltageManager->vSendVoltageWarning(IN l_eU60drop);
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[INFO] TTFiscmd_VoltagemanagervSendVoltageWarning:m_pIVoltageManager = NULL"));
    }
    ETG_TRACE_USR4(("End  : TTFiscmd_VoltagemanagervSendVoltageWarning"));
}


tVoid TraceCmdManager::TTFiscmd_LinuxCmdDbgAnalysis(void) const
{
    ETG_TRACE_USR4(("Begin: TTFiscmd_VoltagemanagerCmddbgAnalysis"));

    //----------------
    //call root daemon
    //----------------
        execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_DBG_ANALYSIS,"/tmp/cmd_dbgAnalysis.result");
    //----------------
    //read resulting file
    //----------------
    ETG_TRACE_FATAL(("=========================Begin Read /tmp/cmd_dbgAnalysis.result=================================="));

    //-----------------------------
    // trace output which has been created in /tmp/cmd_dbgAnalysis.result by abovve command
    //-----------------------------
    int counter = 0;
    std::ifstream inputStream;
    std::string line;
    inputStream.open("/tmp/cmd_dbgAnalysis.result");
    //strVariableParms[0].c_str(); /*mount point*/
    while(!inputStream.eof())
    {
        getline(inputStream,line);
        ETG_TRACE_FATAL(("cmd_dbgAnalysis [line %d]:%s",++counter, line.c_str()));
    }
    inputStream.close();

   
    ETG_TRACE_FATAL(("=========================End Read /tmp/cmd_dbgAnalysis.result=================================="));

    ETG_TRACE_USR4(("End  : TTFiscmd_VoltagemanagerCmddbgAnalysis"));
}



tVoid TraceCmdManager::TTFIScmd_InternalHUBTest(void) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_InternalHUBTest"));

#ifdef USE_EXPERIMENTAL_PSA_HUBCOMMUNICATION

    LibUsbScanForInternalHUB();
#endif

    ETG_TRACE_USR4(("End  : TTFIScmd_InternalHUBTest"));
}



tVoid TraceCmdManager::TTFIScmd_BTABoxisAliveTest(tPCU8 pU8Data ) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_BTABoxisAliveTest"));
    bool isBTAAlive;
    tU32 u32TimoutBeforeSendingITC;
#ifdef USE_PSA_BTA_TELEMATICSBOX_ITCS
    isBTAAlive = (tBool)pU8Data[(tInt)index_cmd+1];
    u32TimoutBeforeSendingITC = (tU32)pU8Data[(tInt)index_cmd+2];
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]         TTFIScmd_BTABoxisAliveTest" ));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok] isBTAAlive                 :%d",isBTAAlive));
    ETG_TRACE_FATAL(("[ok] u32TimoutBeforeSendingITC  :%d",u32TimoutBeforeSendingITC));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]                                                        "));;
    DeviceManagerInterface::GetInstance()->vSetTelematicsBoxApplicationLayerAlive(isBTAAlive,u32TimoutBeforeSendingITC);
#endif
    ETG_TRACE_USR4(("End  : TTFIScmd_BTABoxisAliveTest"));

}

tVoid TraceCmdManager::TTFIScmd_BTABoxDeviceSpeedTest(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmd_BTABoxDeviceSpeedTest"));
    tBool b_DeviceSpeed;//to store the device speed obtained after the scanning
    b_DeviceSpeed = (tBool)pU8Data[(tInt)index_cmd+1];

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok] b_DeviceSpeed              :%d",b_DeviceSpeed));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]                                                        "));
    ConfigurationManager::GetInstance()->vSetTelematicsBoxDeviceSpeed(b_DeviceSpeed);
    ETG_TRACE_USR4(("End  : TTFIScmd_BTABoxDeivceSpeedTest"));
}

/*-----------------------------------------------------------------------------*
 * void setDumpFile(const FILE*& poDumpFile)                                   *
 *-----------------------------------------------------------------------------*/
void TraceCmdManager::setDumpFile(FILE* poDumpFile)
{
    m_poDumpFile = poDumpFile;
}//lint !e1764 Info Reference parameter 'poDumpFile' could be declared const ref

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdShowConfiguration(tVoid)                                      *
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdShowConfiguration(tVoid)   //adjust for Roadmap: 13001_ExtendedUserInformation  VariantSwitch use KDS
{
    ConfigurationManager::GetInstance()->vShowVariantSwitches(m_poDumpFile);
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdCDDAConnection(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdCDDAConnection(tPCU8 pU8Data) const 
{
    ETG_TRACE_USR4(("Begin: TTFIScmdCDDAConnection"));
    std::string strPlatformNotification[4];

    strPlatformNotification[2] = "PLAYABLE_CONNECTED";
    strPlatformNotification[3] = "PLAYABLE_REMOVED";


    //---------------------------------------------------------------------------------------------
    //if ttfis command used with parameters for SRx and SGx (i.e. VD_DVM_SIMULATE_CDDA_PLATFORMNOTIFICATION)
    //---------------------------------------------------------------------------------------------
   
    std::string strCDDAMountPointPartOne[5];
    std::string strCDDAMountPointPartTwo[5];
    std::string strCDDAMountPoint;
    strCDDAMountPointPartOne[0] = "/dev/sr0";
    strCDDAMountPointPartOne[1] = "/dev/sr1";
    strCDDAMountPointPartOne[2] = "/dev/sr2";
    strCDDAMountPointPartOne[3] = "/dev/sr3";
    strCDDAMountPointPartOne[4] = "/dev/sr4";

    strCDDAMountPointPartTwo[0] = "/dev/sg0";
    strCDDAMountPointPartTwo[1] = "/dev/sg1";
    strCDDAMountPointPartTwo[2] = "/dev/sg2";
    strCDDAMountPointPartTwo[3] = "/dev/sg3";
    strCDDAMountPointPartTwo[4] = "/dev/sg4";

    tU32 u32PlatformNotification;
    tU32 u32CDDAMountPointPartOne;
    tU32 u32CDDAMountPointPartTwo;

    DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus = USB_DEV_UNDEFINED;
    tBool b_SendResult = FALSE;

    if(pU8Data)
    {
        u32PlatformNotification = (tU32)pU8Data[(tInt)index_cmd+1];
        u32CDDAMountPointPartOne = (tU32)pU8Data[(tInt)index_cmd+2];
        u32CDDAMountPointPartTwo = (tU32)pU8Data[(tInt)index_cmd+3];

        if(u32PlatformNotification < 4)
        {
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok]         TTFIScmdCDDAConnection "  ));
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok] uiPlatformNotification: %s",strPlatformNotification[u32PlatformNotification].c_str()));
            ETG_TRACE_FATAL(("[ok] u32CDDAMountPointPartOne: %d",u32CDDAMountPointPartOne));
            ETG_TRACE_FATAL(("[ok] u32CDDAMountPointPartTwo: %d",u32CDDAMountPointPartTwo));
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok]                                                        "));
            //here the extended IStateTableMount-Interface for Mount should be called

            //----------------
            //first parameter
            //----------------
            if(0 == strPlatformNotification[u32PlatformNotification].compare("PLAYABLE_CONNECTED"))
            {
                eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
                b_SendResult = TRUE;
            }
            else if(0 == strPlatformNotification[u32PlatformNotification].compare("PLAYABLE_REMOVED"))
            {
                eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR;
                b_SendResult = TRUE;
            }
            else
            {
                ETG_TRACE_FATAL(("vSimulateCDDAConnection the connection state of optical disc is not handled Currently"));
            }

            //----------------
            //second parameter
            //----------------
            if(u32CDDAMountPointPartOne >= 0 && u32CDDAMountPointPartOne < 5)
            {
                strCDDAMountPoint = strCDDAMountPointPartOne[u32CDDAMountPointPartOne];
            }
            if(u32CDDAMountPointPartTwo >= 0 && u32CDDAMountPointPartTwo < 5)
            {
                strCDDAMountPoint +=",";
                strCDDAMountPoint += strCDDAMountPointPartTwo[u32CDDAMountPointPartTwo];

            }

            //----------------------------
            //send to adapter
            //----------------------------
            if(b_SendResult)
            {
                AdapterOpticalDiscNotifications l_AdapterOpticalDiscNotifications;
                l_AdapterOpticalDiscNotifications.vSimulateCDDAConnection(eCDDAConnectionStatus,strCDDAMountPoint);
            }
            else
            {
                ETG_TRACE_FATAL(("vSimulateCDDAConnection: ERROR sending this"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]: TTFIScmdCDDAConnection: Not expected value:uiPlatformNotification(=%d)>=4 ",u32PlatformNotification));
        }
    }
    ETG_TRACE_USR4(("End : TTFIScmdCDDAConnection"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdCDDAConnectionSrSgWithUdev(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdCDDAConnectionSrSgWithUdev(tPCU8 pU8Data) const 
{
    ETG_TRACE_USR4(("Begin: TTFIScmdCDDAConnectionSrSgWithUdev"));
    std::string strPlatformNotification[4];
 
    strPlatformNotification[2] = "PLAYABLE_CONNECTED";
    strPlatformNotification[3] = "PLAYABLE_REMOVED";

    tU32 u32PlatformNotification;
    tU32 u32CDDAMountPointPartOne;
    tU32 u32CDDAMountPointPartTwo;

    DEVICE_CONNECTSTATUS_E eCDDAConnectionStatus = USB_DEV_UNDEFINED;
    tBool b_SendResult = FALSE;

    if(pU8Data)
    {
        u32PlatformNotification = (tU32)pU8Data[(tInt)index_cmd+1];

        if(u32PlatformNotification < 4)
        {
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok]         TTFIScmdCDDAConnectionSrSgWithUdev "  ));
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok] uiPlatformNotification: %s",strPlatformNotification[u32PlatformNotification].c_str()));

            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok]                                                        "));
            //here the extended IStateTableMount-Interface for Mount should be called

            //----------------
            //first parameter
            //----------------
            if(0 == strPlatformNotification[u32PlatformNotification].compare("PLAYABLE_CONNECTED"))
            {
                eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_CONNECT;
                b_SendResult = TRUE;
            }
            else if(0 == strPlatformNotification[u32PlatformNotification].compare("PLAYABLE_REMOVED"))
            {
                eCDDAConnectionStatus = USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR;
                b_SendResult = TRUE;
            }
            else
            {
                ETG_TRACE_FATAL(("TTFIScmdCDDAConnectionSrSgWithUdev the connection state of optical disc is not handled Currently"));
            }
            if(b_SendResult)
            {
                AdapterOpticalDiscNotifications l_AdapterOpticalDiscNotifications;
                l_AdapterOpticalDiscNotifications.vSimulateCDDAConnection(eCDDAConnectionStatus);
            }
            else
            {
                ETG_TRACE_FATAL(("TTFIScmdCDDAConnectionSrSgWithUdev: ERROR sending this"));
            }

        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR]: TTFIScmdCDDAConnectionSrSgWithUdev: Not expected value:uiPlatformNotification(=%d)>=4 ",u32PlatformNotification));
        }
    }
    ETG_TRACE_USR4(("End : TTFIScmdCDDAConnectionSrSgWithUdev"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdEjectOpticalDisc(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdEjectOpticalDisc(tPCU8 pU8Data) const 
{
    ETG_TRACE_USR4(("Begin: TTFIScmdEjectOpticalDisc"));

    if(pU8Data)
    {
       
        tU32 u32CmdParam = (tU32)pU8Data[(tInt)index_cmd+1];
        
        switch(u32CmdParam)
        {
            case DM_OPTDISC_CMD_EJECT:
                DeviceManagerInterface::GetInstance()->vSendOpticalDiscEjectRequest(DM_OPTDISC_CMD_EJECT);
                break;
            case DM_OPTDISC_CMD_INSERT:
                DeviceManagerInterface::GetInstance()->vSendOpticalDiscEjectRequest(DM_OPTDISC_CMD_INSERT);
                break;
            default:
                ETG_TRACE_FATAL(("TTFIScmdEjectOpticalDisc: Unexpected parameter %d", u32CmdParam));
                break;

        }
    
    } 
    ETG_TRACE_USR4(("End : TTFIScmdEjectOpticalDisc"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdOpticalDiscSlotStatus(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdOpticalDiscSlotStatus(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdOpticalDiscSlotStatus"));

  tU32 u32PlatformNotification;
  std::string strPlatformNotification[18];
  strPlatformNotification[0]    = "SLOTSTATUS_INITIALISING";
  strPlatformNotification[2]    = "INITIALISED";
  strPlatformNotification[3]    = "LASTMODE_EMTPY";
  strPlatformNotification[4]    = "LASTMODE_INSERTED_CDDA";
  strPlatformNotification[5]    = "LASTMODE_INSERTED_CDROM";
  strPlatformNotification[6]    = "LASTMODE_INSERTED_CDERROR";
  strPlatformNotification[7]    = "INSERTING";
  strPlatformNotification[8]    = "INSERTED_CDAUDIO";
  strPlatformNotification[9]    = "INSERTED_CDROM";
  strPlatformNotification[10]   = "INSERTED_CDERROR";
  strPlatformNotification[11]   = "INSERTED_AUTOMATIC_CDAUDIO";
  strPlatformNotification[12]   = "INSERTED_AUTOMATIC_CDROM";
  strPlatformNotification[13]   = "INSERTED_AUTOMATIC_CDERROR";
  strPlatformNotification[14]   = "EJECTING";
  strPlatformNotification[15]   = "EJECTED_READY_TO_REMOVE";
  strPlatformNotification[16]   = "EJECTED_EMPTY";
  strPlatformNotification[17]   = "UNDEFINED_STATE";

  if(pU8Data)
  {
        u32PlatformNotification = (tU32)pU8Data[(tInt)index_cmd+1];
        if(u32PlatformNotification == 0xFF)
        {
            u32PlatformNotification = 17;
        }

        if(u32PlatformNotification < 18)
        {
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok]         TTFIScmdOpticalDiscSlotStatus "  ));
            ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[ok] SlotStatus: %s",strPlatformNotification[u32PlatformNotification].c_str()));
            tDMOpticalDiscSlotState eSlotState = (tDMOpticalDiscSlotState)u32PlatformNotification;
            m_pStateTableApp->vSetOpticalDiscSlotStateNotified(enOptDiscSlotStateNotified,eSlotState);
        }
    }
    ETG_TRACE_USR4(("End  : TTFIScmdOpticalDiscSlotStatus"));
}
 
/*-----------------------------------------------------------------------------*
 * tVoid SetPrmManagerIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid TraceCmdManager::SetPrmManagerIF(IPrmManager *f_pIPrmManager)
{
    m_pIPrmManager = f_pIPrmManager;
}
/*-----------------------------------------------------------------------------*
 * tVoid SetPrmManagerIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid TraceCmdManager::SetPrmManagerCardIF(IPrmManagerCard *f_pIPrmManagerCard)
{
    m_pIPrmManagerCard = f_pIPrmManagerCard;
}
/*-----------------------------------------------------------------------------*
 * tVoid SetVoltageManagerIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid TraceCmdManager::SetVoltageManagerIF(IVoltageManager *f_pIVoltageManager)
{
    m_pIVoltageManager = f_pIVoltageManager;
}

/*-----------------------------------------------------------------------------*
 * tVoid SetStateTableMountIF(...)
 *-----------------------------------------------------------------------------*/
IN tVoid TraceCmdManager::SetStateTableMountIF(IStateTableMount *f_pIStateTableMount)
{
    m_pIStateTableMount = f_pIStateTableMount;
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdGetConfigurationValue(...)
 *-----------------------------------------------------------------------------*/
IN IN tVoid TraceCmdManager::TTFIScmdGetConfigurationValue(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdGetConfigurationValue"));
    const char *l_strConfigParam = "";
    tU16 u16ConfigValue;
    tU16 u16ConfigParam;

    if(pU8Data)
    {
        u16ConfigParam = (tU32)pU8Data[(tInt)index_cmd+1];

        switch(u16ConfigParam)
        {
            case 00:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1";
                break;
            case 01:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2";
                break;
            case 02:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3";
                break;
            case 03:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4";
                break;
            default:
                ETG_TRACE_FATAL(("Wrong Configuration value"  ))
                break;
        }

        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdGetConfigurationValue "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] Configuration parameter: %s",l_strConfigParam));
        if(TRUE == ConfigurationManager::GetInstance()->vGetConfigurationValue( l_strConfigParam,u16ConfigValue))
        {
            ETG_TRACE_FATAL(("[ok] Configuration value obtained: %d",u16ConfigValue));
        }
    }

    ETG_TRACE_USR4(("End: TTFIScmdGetConfigurationValue"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSetConfigurationValue(...)
 *-----------------------------------------------------------------------------*/
IN tVoid TraceCmdManager::TTFIScmdSetConfigurationValue(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdSetConfigurationValue"));
    const char *l_strConfigParam = "";
    tU16 u16ConfigParam;
    tU16 u16ConfigValue;
    tenSettingsMode eSettingsMode;

    if(pU8Data)
    {
        u16ConfigParam = (tU32)pU8Data[(tInt)index_cmd+1];
        u16ConfigValue = (tU32)pU8Data[(tInt)index_cmd+2];
        eSettingsMode  = (tenSettingsMode)pU8Data[(tInt)index_cmd+3];

        switch(u16ConfigParam)
        {
            case 00:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1";
                break;
            case 01:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2";
                break;
            case 02:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3";
                break;
            case 03:
                l_strConfigParam = "eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4";
                break;
            default:
                ETG_TRACE_FATAL(("Wrong Config value"  ))
                break;
        }




        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdSetConfigurationValue "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] Configuration parameter: %s",l_strConfigParam));
        ETG_TRACE_FATAL(("[ok] Configuration value: %d",u16ConfigValue));
        ETG_TRACE_FATAL(("[ok] Settings Mode: %d",eSettingsMode));
        if(TRUE == ConfigurationManager::GetInstance()->vSetConfigurationValue( l_strConfigParam,u16ConfigValue,eSettingsMode))
        {
            ETG_TRACE_FATAL(("[ok] Configuration value Stored Successfully"));
        }

    }
    ETG_TRACE_USR4(("End: TTFIScmdSetConfigurationValue"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdSimulateMalfunctionUSBConnector(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdSimulateMalfunctionUSBConnector(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdSimulateMalfunctionUSBConnector"));
    tenDevManagerUSBHost       eUsbConnector;
    tBool           bIsMalfunction;

    if(pU8Data)
    {
        eUsbConnector  = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+1];
        bIsMalfunction = (tBool)pU8Data[(tInt)index_cmd+2];

        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdSimulateMalfunctionUSBConnector "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] USB Connector: %d",ETG_ENUM(TENDEVMANAGERUSBHOST,eUsbConnector)));
        ETG_TRACE_FATAL(("[ok] HasMalfunction : %d",ETG_ENUM(BOOL,bIsMalfunction)));
        if(NULL==m_pStateTableApp)
        {
            ETG_TRACE_FATAL((" [ERROR]: TTFIScmdSimulateMalfunctionUSBConnector: NULL==m_pStateTableApp"));
            ETG_TRACE_ERRMEM(("[ERROR]: TTFIScmdSimulateMalfunctionUSBConnector: NULL==m_pStateTableApp"));
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            m_pStateTableApp->SetMalfunctionUSBConnector(bIsMalfunction,eUsbConnector);
        }

    }

    ETG_TRACE_USR4(("End: TTFIScmdSimulateMalfunctionUSBConnector"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryHardwareMalfunction(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryHardwareMalfunction() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryHardwareMalfunction"));

    HWMALFUNCTIONHISTORY.TraceHistory();

    ETG_TRACE_USR4(("End:   TTFIScmdHistoryHardwareMalfunction"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdHistoryHardwareMalfunction(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdHistoryHardwareMalfunction_Clear() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdHistoryHardwareMalfunction_Clear"));

    HWMALFUNCTIONHISTORY.ClearHistory();

    ETG_TRACE_USR4(("End:   TTFIScmdHistoryHardwareMalfunction_Clear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdExternalHubDetailsAdd(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdExternalHubDetailsAdd(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdExternalHubDetailsAdd"));

    tenDevManagerUSBHost       eUsbConnector;
    tBool           bIsMalfunction;
    trExternalHubInfo l_uiExternalHubInfo;
    uint16_t high_byte, low_byte;
    if(pU8Data)
    {
        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+1];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+2];
        l_uiExternalHubInfo.iVendorID  = (high_byte<<8) + low_byte;

        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+3];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+4];
        l_uiExternalHubInfo.iProductID  = (high_byte<<8) + low_byte;
        
        l_uiExternalHubInfo.eUsb  = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+5];

        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdExternalHubDetailsAdd "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] USB Connector: %d",ETG_ENUM(TENDEVMANAGERUSBHOST,eUsbConnector)));

        if(TRUE != ConfigurationManager::GetInstance()-> AddExternalHub(IN l_uiExternalHubInfo))
        {
            ETG_TRACE_FATAL(("[Error] AddExternalHub.. Addition Failed"));
        }

    }

    ETG_TRACE_USR4(("End: TTFIScmdExternalHubDetailsAdd"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdExternalHubDetailsRemove(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdExternalHubDetailsRemove(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdExternalHubDetailsRemove"));

    tenDevManagerUSBHost       eUsbConnector;
    tBool           bIsMalfunction;
    trExternalHubInfo l_uiExternalHubInfo;
    uint16_t high_byte, low_byte;
    
    if(pU8Data)
    {
        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+1];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+2];
        l_uiExternalHubInfo.iVendorID  = (high_byte<<8) + low_byte;

        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+3];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+4];
        l_uiExternalHubInfo.iProductID  = (high_byte<<8) + low_byte;
       
        l_uiExternalHubInfo.eUsb  = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+5];


        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdExternalHubDetailsRemove "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] USB Connector: %d",ETG_ENUM(TENDEVMANAGERUSBHOST,eUsbConnector)));

        tBool ClearAllfromDP = FALSE; //Clear all data from DP NOT needed
        if(TRUE != ConfigurationManager::GetInstance()-> RemoveExternalHub(IN l_uiExternalHubInfo, IN ClearAllfromDP))
        {
            ETG_TRACE_FATAL(("[Error] RemoveExternalHub.. Removal Failed"));
        }

    }

    ETG_TRACE_USR4(("End: TTFIScmdExternalHubDetailsRemove"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdExternalHubPrint()
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdExternalHubPrint() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdExternalHubPrint"));
    trExternalHubInfo l_uiExternalHubInfo [iMaxSizeOfArray];
    tU8 l_uiNumberOfExternalHubsUsed;

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]         TTFIScmdExternalHubPrint "  ));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    if(TRUE != ConfigurationManager::GetInstance()->ReadExternalHub(l_uiExternalHubInfo,l_uiNumberOfExternalHubsUsed))
    {
        ETG_TRACE_FATAL(("[Error] ReadExternalHub.. Printing Failed"));
    }
    else
    {
        ETG_TRACE_USR4(("***********************ReadExternalHub ExternalHub details in Array(read from Datapool)***********************"));
        for(int i=0 ; i<l_uiNumberOfExternalHubsUsed ; i++)
        {
            ETG_TRACE_USR4(("ExternalHub in array: l_uiExternalHubInfo count: %d",i));
            ETG_TRACE_USR4(("ExternalHub in array: l_uiExternalHubInfo[i].iVendorID = %d",l_uiExternalHubInfo[i].iVendorID));
            ETG_TRACE_USR4(("ExternalHub in array: l_uiExternalHubInfo[i].iProductID = %d",l_uiExternalHubInfo[i].iProductID));
            ETG_TRACE_USR4(("ExternalHub in array: l_uiExternalHubInfo[i].eUsb = %d",l_uiExternalHubInfo[i].eUsb));
            ETG_TRACE_USR4(("---------------------------------------------------------------------------------------"));
        }
    }
    ETG_TRACE_USR4(("End: TTFIScmdExternalHubPrint"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdExternalHubDPClear()
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdExternalHubDPClear() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdExternalHubDPClear"));

    trExternalHubInfo l_uiExternalHubInfo;

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]         TTFIScmdExternalHubDPClear "  ));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));

    ETG_TRACE_USR4(("***********************ExternalHub details: EMPTYING FROM DATAPOOL***********************"));
    tBool ClearAllfromDP = TRUE;  //Set ClearAllfromDP as TRUE to empty the datapool
    if(TRUE != ConfigurationManager::GetInstance()-> RemoveExternalHub(IN l_uiExternalHubInfo, IN ClearAllfromDP))
    {
        ETG_TRACE_FATAL(("[Error] AddExternalHub.. Clearing Failed"));
    }


    ETG_TRACE_USR4(("End: TTFIScmdExternalHubDPClear"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdTelematicsBoxDetailsAdd(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdTelematicsBoxDetailsAdd(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTelematicsBoxDetailsAdd"));

    tenDevManagerUSBHost       eUsbConnector;
    tBool           bIsMalfunction;
    trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo;
    uint16_t high_byte, low_byte;
    if(pU8Data)
    {
        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+1];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+2];
        l_uiTelematicsBoxDeviceInfo.iVendorID  = (uint16_t)pU8Data[(tInt)index_cmd+1];

        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+3];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+4];
        l_uiTelematicsBoxDeviceInfo.iProductID  = (uint16_t)pU8Data[(tInt)index_cmd+2];

        
        l_uiTelematicsBoxDeviceInfo.bIsHighSpeedExpected  = (tBool)pU8Data[(tInt)index_cmd+5];
        l_uiTelematicsBoxDeviceInfo.eUsb  = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+6];

        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"))
        ETG_TRACE_FATAL(("[ok]         TTFIScmdTelematicsBoxDetailsAdd "  ))
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"))
        ETG_TRACE_FATAL(("[ok] USB Connector: %d",ETG_ENUM(TENDEVMANAGERUSBHOST,eUsbConnector)));

        if(TRUE != ConfigurationManager::GetInstance()-> AddTelematicsBox(IN l_uiTelematicsBoxDeviceInfo))
        {
            ETG_TRACE_FATAL(("[Error] AddTelematicsBox.. Addition Failed"));
        }
    }

    ETG_TRACE_USR4(("End: TTFIScmdTelematicsBoxDetailsAdd"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdTelematicsBoxDetailsRemove(...)
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdTelematicsBoxDetailsRemove(tPCU8 pU8Data) const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTelematicsBoxDetailsRemove"));

    tenDevManagerUSBHost       eUsbConnector;
    tBool           bIsMalfunction;
    trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo;
    uint16_t high_byte, low_byte;
    
    if(pU8Data)
    {
        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+1];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+2];
        l_uiTelematicsBoxDeviceInfo.iVendorID  = (uint16_t)pU8Data[(tInt)index_cmd+1];

        high_byte = (uint16_t)pU8Data[(tInt)index_cmd+3];
        low_byte  = (uint16_t)pU8Data[(tInt)index_cmd+4];
        l_uiTelematicsBoxDeviceInfo.iProductID  = (uint16_t)pU8Data[(tInt)index_cmd+2];


        l_uiTelematicsBoxDeviceInfo.bIsHighSpeedExpected  = (tBool)pU8Data[(tInt)index_cmd+5];
        l_uiTelematicsBoxDeviceInfo.eUsb  = (tenDevManagerUSBHost)pU8Data[(tInt)index_cmd+6];


        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok]         TTFIScmdTelematicsBoxDetailsRemove "  ));
        ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
        ETG_TRACE_FATAL(("[ok] USB Connector: %d",ETG_ENUM(TENDEVMANAGERUSBHOST,eUsbConnector)));

        tBool ClearAllfromDP = FALSE; //Clear all data from DP NOT needed
        if(TRUE != ConfigurationManager::GetInstance()-> RemoveTelematicsBox(IN l_uiTelematicsBoxDeviceInfo, IN ClearAllfromDP))
        {
            ETG_TRACE_FATAL(("[Error] RemoveTelematicsBox.. Removal Failed"));
        }
    }

    ETG_TRACE_USR4(("End: TTFIScmdTelematicsBoxDetailsRemove"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdTelematicsBoxPrint()
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdTelematicsBoxPrint() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTelematicsBoxPrint"));
    trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo [iMaxSizeOfArray];
    tU8 l_uiNumberOfTelematicBoxesUsed;

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"))
    ETG_TRACE_FATAL(("[ok]         TTFIScmdTelematicsBoxPrint "  ))
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"))
    if(TRUE != ConfigurationManager::GetInstance()->ReadTelematicsBox(l_uiTelematicsBoxDeviceInfo,l_uiNumberOfTelematicBoxesUsed))
    {
        ETG_TRACE_FATAL(("[Error] ReadTelematicsBox.. Printing Failed"));
    }
    else
    {
        ETG_TRACE_USR4(("***********************ReadTelematicsBox TelematicsBox details in Array(read from Datapool)***********************"));
        for(int i=0 ; i<l_uiNumberOfTelematicBoxesUsed ; i++)
        {
            ETG_TRACE_USR4(("TelematicsBox in array: l_uiTelematicsBoxDeviceInfo count: %d",i));
            ETG_TRACE_USR4(("TelematicsBox in array: l_uiTelematicsBoxDeviceInfo[i].iVendorID = %d",l_uiTelematicsBoxDeviceInfo[i].iVendorID));
            ETG_TRACE_USR4(("TelematicsBox in array: l_uiTelematicsBoxDeviceInfo[i].iProductID = %d",l_uiTelematicsBoxDeviceInfo[i].iProductID));
            ETG_TRACE_USR4(("TelematicsBox in array: l_uiTelematicsBoxDeviceInfo[i].bIsHighSpeedExpected = %d",l_uiTelematicsBoxDeviceInfo[i].bIsHighSpeedExpected));
            ETG_TRACE_USR4(("TelematicsBox in array: l_uiTelematicsBoxDeviceInfo[i].eUsb = %d",l_uiTelematicsBoxDeviceInfo[i].eUsb));
            ETG_TRACE_USR4(("---------------------------------------------------------------------------------------"));
        }
    }
    ETG_TRACE_USR4(("End: TTFIScmdTelematicsBoxPrint"));
}

/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdTelematicsBoxDPClear()
 *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdTelematicsBoxDPClear() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdTelematicsBoxDPClear"));

    trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo;

    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
    ETG_TRACE_FATAL(("[ok]         TTFIScmdTelematicsBoxDPClear "  ));
    ETG_TRACE_FATAL(("[ok] ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));

    ETG_TRACE_USR4(("***********************TelematicsBox details: EMPTYING FROM DATAPOOL***********************"));
    tBool ClearAllfromDP = TRUE;  //Set ClearAllfromDP as TRUE to empty the datapool
    if(TRUE != ConfigurationManager::GetInstance()-> RemoveTelematicsBox(IN l_uiTelematicsBoxDeviceInfo, IN ClearAllfromDP))
    {
        ETG_TRACE_FATAL(("[Error] AddTelematicsBox.. Clearing Failed"));
    }


    ETG_TRACE_USR4(("End: TTFIScmdTelematicsBoxDPClear"));
}


/*-----------------------------------------------------------------------------*
 * tVoid TTFIScmdShowGenericKDSDeviceManager()
 *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdShowGenericKDSDeviceManager() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdShowGenericKDSDeviceManager"));

    ConfigurationManager::GetInstance()->vPrintGenericKdsValuesDevicemanager();

    ETG_TRACE_USR4(("Begin: TTFIScmdShowGenericKDSDeviceManager"));
}


/*-----------------------------------------------------------------------------*
  * tVoid TTFIScmdShowGenericKDSMediaplayer()
  *-----------------------------------------------------------------------------*/
tVoid  TraceCmdManager::TTFIScmdShowGenericKDSMediaplayer() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdShowGenericKDSMediaplayer"));

    ConfigurationManager::GetInstance()->vPrintGenericKdsValuesMediaplayer();

    ETG_TRACE_USR4(("Begin: TTFIScmdShowGenericKDSMediaplayer"));

}
/*-----------------------------------------------------------------------------*
  * tVoid TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer()
  *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer"));

    //these two function read the same value: First one directly KDS value - second does it using Datapool function. But it should do the same
    ConfigurationManager::GetInstance()->vPrintRNAIVKdsValuesMediaplayerFeaDVD();
    ConfigurationManager::GetInstance()->vPrintRNAIVDatapoolValuesMediaplayerFeaDVD();
    
    //these two function read the same value: First one directly KDS value - second does it using Datapool function. But it should do the same
    ConfigurationManager::GetInstance()->vPrintRNAIVKdsValuesMediaplayerReagionJapan();
    ConfigurationManager::GetInstance()->vPrintRNAIVDatapoolValuesMediaplayerReagionJapan();

    ETG_TRACE_USR4(("Begin: TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer"));

}

/*-----------------------------------------------------------------------------*
  * tVoid TTFIScmdRootDaemonExecute_Mkdir()
  *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdRootDaemonExecute_Mkdir() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdRootDaemonExecute_Mkdir"));
    CmdData result;
    result = execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME, CMD_MKDIR, "/tmp/tmp_ganny");
    ETG_TRACE_USR4(("TTFIScmdRootDaemonExecute_Mkdir result.errorNo %d",result.errorNo));
    ETG_TRACE_USR4(("End: TTFIScmdRootDaemonExecute_Mkdir"));

}

/*-----------------------------------------------------------------------------*
  * tVoid TTFIScmdRootDaemonExecute_Cmd1()
  *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdRootDaemonExecute_Cmd1() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdRootDaemonExecute_Cmd1"));

    CmdData result;
    result = execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_LS2FILE,"/tmp/.automounter >> /tmp/cmd_dbgAnalysis.result");
    ETG_TRACE_USR4(("TTFIScmdRootDaemonExecute_Cmd1 result.errorNo %d",result.errorNo));
    ETG_TRACE_USR4(("End: TTFIScmdRootDaemonExecute_Cmd1"));

}

/*-----------------------------------------------------------------------------*
  * tVoid TTFIScmdRootDaemonExecute_Cmd2()
  *-----------------------------------------------------------------------------*/
tVoid TraceCmdManager::TTFIScmdRootDaemonExecute_Cmd2() const
{
    ETG_TRACE_USR4(("Begin: TTFIScmdRootDaemonExecute_Cmd2"));

    CmdData result;
    result = execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_MOUNT2FILE," >> /tmp/cmd_dbgAnalysis.result");
    ETG_TRACE_USR4(("TTFIScmdRootDaemonExecute_Cmd2 result.errorNo %d",result.errorNo));
    ETG_TRACE_USR4(("End: TTFIScmdRootDaemonExecute_Cmd2"));

    ETG_TRACE_USR4(("End: TTFIScmdRootDaemonExecute_Cmd2"));

}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
