/*-----------------------------------------------------------------------------*
 * ServiceHistory.cpp                                                          *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file ServiceHistory.cpp
 *
 * \brief This file holds the implementation for CServiceHistory
 *
 * \version 02.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "debug/ObjectHistory.h"
#include "debug/InterfaceHistory.h"
#include "config/ConfigurationManager.h" //for eclapsed time

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_HISTORYMANAGER
#include "trcGenProj/Header/InterfaceHistory.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * namespace                                                                   *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
InterfaceHistory::InterfaceHistory()
    :CObjectHistory() 
{
    Init ();
}//lint !e1566  
//Warning 1566: prio3: member 'CServiceHistory::m_currIndex'       might have been initialized by a separate function but no '-sem(CServiceHistory::ClearHistory,initializer)' was seen
//Warning 1566: prio3: member 'CServiceHistory::m_iSendNo' might have been initialized by a separate function but no '-sem(CServiceHistory::ClearHistory,initializer)' was seen
//see function ClearHistory

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
InterfaceHistory::~InterfaceHistory()
{
    // Do nothing
}


/*-----------------------------------------------------------------------------*
 * void TraceHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void InterfaceHistory::TraceHistory()
{
    ETG_TRACE_USR3 (("TraceHistory: Begin"));
    ETG_TRACE_FATAL (("[ok] ===============================->Service==============================="));

    tU64 u64StartTimeAbsoluteDeviceManager = ConfigurationManager::GetInstance()->GetStartTime_ms();
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (ms) : ",u64StartTimeAbsoluteDeviceManager);
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (sec): ",u64StartTimeAbsoluteDeviceManager/1000);

    tBool        bUsed = FALSE;
    unsigned int itr;

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,"\n[ok] ===============================->Service===============================");
    }

    //======================================================
    // print details: device lits and connector list
    //======================================================
    ETG_TRACE_FATAL(("[ok]                                                                                                    "));
    ETG_TRACE_FATAL(("[ok]                                                                                                    "));
    ETG_TRACE_FATAL(("[ok] ***************************************************************************************************"));
    ETG_TRACE_FATAL(("[ok]         HISTORY: DETAILS: DeviceLists and ConnectorLists "));
    ETG_TRACE_FATAL(("[ok] ***************************************************************************************************"));
    for (itr = 0;itr < HELPER_ANALYSIS_SENDINTERFACESERVICE;itr++)
    {
       
        //---------------------
        //trace device list - summary
        //---------------------
        TraceHistoryDeviceList(IN itr, OUT bUsed);

        //---------------------
        //trace connector list
        //---------------------
        TraceHistoryConnectorList(IN itr);
    }

    if (FALSE == bUsed)
    {
        ETG_TRACE_FATAL (("[ok] ***************/TraceStoredLastMsgs2Clients ***********************"));
        ETG_TRACE_FATAL (("[ok] *       No message send to clients yet                            *"));
        ETG_TRACE_FATAL (("[ok] ***************TraceStoredLastMsgs2Clients ************************"));

        // dump to file, if available
        if (NULL != m_poDumpFile)
        {
            fprintf (m_poDumpFile, "\n[ok] ***************/TraceStoredLastMsgs2Clients ***********************");
            fprintf (m_poDumpFile, "\n[ok]        No message send to clients yet                              ");
            fprintf (m_poDumpFile, "\n[ok] ***************TraceStoredLastMsgs2Clients ************************");
        }
    }

    //======================================================
    // print summary send device list
    //======================================================
    ETG_TRACE_FATAL(("[ok]                                                                                                    "));
    ETG_TRACE_FATAL(("[ok]                                                                                                    "));
    ETG_TRACE_FATAL(("[ok] ***************************************************************************************************"));
    ETG_TRACE_FATAL(("[ok]         HISTORY: SUMMARY DEVICE CONNECTION"));
    ETG_TRACE_FATAL(("[ok] ***************************************************************************************************"));
    for (itr = 0;itr < HELPER_ANALYSIS_SENDINTERFACESERVICE;itr++)
    {
        //---------------------
        //trace device list - summary
        //---------------------
        TraceHistoryDeviceListSummary(IN itr);
    }

    ETG_TRACE_FATAL (("[ok] ================================Service<-=============================="));

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,
                "\n[ok] ================================Service<-==============================");
    }

    ETG_TRACE_USR3 (("TraceHistory: End"));
}

/*-----------------------------------------------------------------------------*
 * void Init ()                                                        *
 *-----------------------------------------------------------------------------*/
void InterfaceHistory::Init()
{
    ClearHistory();
}

/*-----------------------------------------------------------------------------*
 * void ClearHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void InterfaceHistory::ClearHistory()
{
    int itr, i;
    for (itr = 0; itr < HELPER_ANALYSIS_SENDINTERFACESERVICE; itr++)
    {
        m_rSendConnectionNotificationInterface[itr].bSend     = FALSE;
        m_rSendConnectionNotificationInterface[itr].send_nr   = -1;
        m_rSendConnectionNotificationInterface[itr].sent_time = 0;
        m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.clear();
        for(i = (int)eUSB1; i< ARRAYSIZEFORUSB; i++)
        {
             m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].u8USBPortNo  =  eUSBUndef;
             m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOverCurrent  = FALSE;
             m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUndervoltage = FALSE;
             m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bHubConnected = FALSE;
             m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOpenCircuit  = FALSE;
        }
    }

    m_iSendNo   = 0;
    m_currIndex = 0;
}   //lint !e1565 Warning 1565: member 'CObjectHistory::m_poDumpFile' (VD_DeviceManager/inc/debug/ObjectHistory.h) not assigned by initializer function


void InterfaceHistory::AddToHistory (IN tBool bSendSuccess, IN vector<CDevice> &f_ListOfSendDevicestServices, IN trPortStates (&f_ListOfSendArrayPortStates)[ARRAYSIZEFORUSB])
{
    int i;

    ETG_TRACE_USR4(("InterfaceHistory::AddToHistory(...)"));

    m_LockHistory.lock();

    ++m_iSendNo;

    m_rSendConnectionNotificationInterface[m_currIndex].bSend     = bSendSuccess;
    m_rSendConnectionNotificationInterface[m_currIndex].send_nr   = m_iSendNo;
    m_rSendConnectionNotificationInterface[m_currIndex].sent_time = (tU32)ConfigurationManager::GetInstance()->GetEclapsedTime_ms(); 

    
    for(i=0;i<static_cast<int>(f_ListOfSendDevicestServices.size());i++)
    {
        m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendDevicestServices.push_back(f_ListOfSendDevicestServices[i]);
    }

    for(i = (int)eUSB1; i< ARRAYSIZEFORUSB; i++)
    {
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].u8USBPortNo            = f_ListOfSendArrayPortStates[i].u8USBPortNo;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bOverCurrent           = f_ListOfSendArrayPortStates[i].bOverCurrent;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bUndervoltage          = f_ListOfSendArrayPortStates[i].bUndervoltage;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bHubConnected          = f_ListOfSendArrayPortStates[i].bHubConnected;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bOpenCircuit           = f_ListOfSendArrayPortStates[i].bOpenCircuit;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bElectricalFailure     = f_ListOfSendArrayPortStates[i].bElectricalFailure;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bUSBPortConfiguredUsed = f_ListOfSendArrayPortStates[i].bUSBPortConfiguredUsed;
         m_rSendConnectionNotificationInterface[m_currIndex].ListOfSendArrayPortStates[i].bUSBPowerON            = f_ListOfSendArrayPortStates[i].bUSBPowerON;

    }

    m_currIndex++;

    if (m_currIndex >= HELPER_ANAYLSIS_SEND2CLIENTS) //is a ringbuffer
    {
        m_currIndex = 0;
    }

    m_LockHistory.unlock();
}

void InterfaceHistory::TraceHistoryDeviceListSummary(IN unsigned int itr)
{
    //print only if send_nr not cleared i.e. != -1
    if ((itr < HELPER_ANALYSIS_SENDINTERFACESERVICE) && (-1 != m_rSendConnectionNotificationInterface[itr].send_nr) )
    {
       ETG_TRACE_FATAL (("[ok] =>------------------------------------------------------------------------------------------------------------------------------"));

        //ETG_TRACE_FATAL (("[ok] [%2d]: Send ok by Service : %d",    itr, m_rSendConnectionNotificationInterface[itr].bSend)); //e.g. third notification since startup
        ETG_TRACE_FATAL (("[ok] [%2d]: Send No           : %d",    itr, m_rSendConnectionNotificationInterface[itr].send_nr)); //e.g. third notification since startup
        ETG_TRACE_FATAL (("[ok] [%2d]: Send Time (ms)    : %d ms", itr, m_rSendConnectionNotificationInterface[itr].sent_time));
        if (NULL != m_poDumpFile)
        {
            fprintf (m_poDumpFile, "\n\n\n[ok] ====/TraceHistoryDeviceListShort (SUMMARY)==============================================================================================");

            fprintf (m_poDumpFile, "\n[ok] [%2d]: Send No       : %d",itr,m_rSendConnectionNotificationInterface[itr].send_nr); //e.g. third notification since startup
            fprintf (m_poDumpFile, "\n[ok] [%2d]: Send Time (ms): %u ms",itr, (tUInt)m_rSendConnectionNotificationInterface[itr].sent_time);
        }

        //---------------------
        //trace devicelist
        //---------------------
        if(m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size()== 0)
        {
            ETG_TRACE_FATAL(("[ok] [%2d]: ListOfSendDevices: Empty",itr));
            if (NULL != m_poDumpFile)
            {
                 fprintf (m_poDumpFile, "\n[ok] [%2d]: ListOfSendDevices: Empty",itr);
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ok] [%2d]: ListOfSendDevices: Elems: %d",itr,m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size()));
            if (NULL != m_poDumpFile)
            {
                 fprintf (m_poDumpFile,"\n[ok] [%2d]: ListOfSendDevices: Elems: %d",itr, m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size());
            }
        }
        for(int i = 0; i <static_cast<int>(m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size());i++)
        {
            ETG_TRACE_FATAL (("[ok] ========= Index=%d: =============================================",i));
            ETG_TRACE_FATAL (("[ok] [%2d]: Serial ID     : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cSerialID.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Type   : %d",     itr, ETG_ENUM(DVM_DEVICE_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceType)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Name   : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceName.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Mount Point   : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cMountPoint.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Connect Status: %d",     itr, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eConnectStatus)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Unsupported Reason: %d", itr, ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceUnsupportedReason)));
            ETG_TRACE_FATAL (("[ok] ================================================================="));

            // dump to file, if available
            if (NULL != m_poDumpFile)
            {
                fprintf (m_poDumpFile, "\n[ok] ========= Index=%d: =============================================",i);
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Serial ID     : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cSerialID.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Type   : %d", itr,     ETG_ENUM(DVM_DEVICE_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceType));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Name   : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceName.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Mount Point   : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cMountPoint.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Connect Status: %d", itr,     ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,   m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eConnectStatus));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Unsupported Reason: %d", itr, ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceUnsupportedReason));
                fprintf (m_poDumpFile, "[ok] =================================================================");
           }
        }

        ETG_TRACE_FATAL (("[ok] -----------------------------------------------------------------------------------------------------------------------------<="));
        ETG_TRACE_FATAL (("[ok]                                                                                                                               "));
    }
}

/*-----------------------------------------------------------------------------*
 * void TraceHistoryDeviceList ()                                                        *
 *-----------------------------------------------------------------------------*/
void InterfaceHistory::TraceHistoryDeviceList(IN unsigned int itr, OUT tBool &f_bUsed)
{
    //print only if send_nr not cleared i.e. != -1
    if ((itr < HELPER_ANALYSIS_SENDINTERFACESERVICE) && (-1 != m_rSendConnectionNotificationInterface[itr].send_nr) )
    {
        ETG_TRACE_FATAL (("[ok] ====/TraceHistoryDeviceList=============================================================================================="));

        f_bUsed = TRUE;
        //ETG_TRACE_FATAL (("[ok] [%2d]: Send ok by Service : %d",    itr, m_rSendConnectionNotificationInterface[itr].bSend)); //e.g. third notification since startup
        ETG_TRACE_FATAL (("[ok] [%2d]: Send No            : %d",   itr, m_rSendConnectionNotificationInterface[itr].send_nr)); //e.g. third notification since startup
        ETG_TRACE_FATAL (("[ok] [%2d]: Send Time (ms)    : %d ms", itr, m_rSendConnectionNotificationInterface[itr].sent_time));
        //ETG_TRACE_FATAL (("[ok] [%2d]: Type Of Notify: %i", itr, ETG_ENUM(NOTIFICATION_Type, m_aoLastNotifiedDevices[itr].m_eTypeOfNotify) ));
        if (NULL != m_poDumpFile)
        {
            fprintf (m_poDumpFile, "\n[ok] [%2d]: Send No       : %d",   itr,m_rSendConnectionNotificationInterface[itr].send_nr); //e.g. third notification since startup
            fprintf (m_poDumpFile, "\n[ok] [%2d]: Send Time (ms): %u ms",itr, (tUInt)m_rSendConnectionNotificationInterface[itr].sent_time);
        }

        //---------------------
        //trace devicelist
        //---------------------
        if(m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size()== 0)
        {
            ETG_TRACE_FATAL(("[ok] [%2d]: ListOfSendDevices: Empty", itr));
            if (NULL != m_poDumpFile)
            {
                 fprintf (m_poDumpFile, "\n[ok] [%2d]: ListOfSendDevices: Empty", itr);
            }
        }
        else
        {
            ETG_TRACE_FATAL(("[ok] +++++++++++++++++++++++"));
            ETG_TRACE_FATAL(("[ok] [%2d]: ListOfSendDevices: Elems: %d",itr,m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size()));
            if (NULL != m_poDumpFile)
            {
                 fprintf (m_poDumpFile,"[ok] +++++++++++++++++++++++");
                 fprintf (m_poDumpFile,"\n[ok] [%2d]: ListOfSendDevices: Elems: %d",itr,m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size());
            }
        }
        for(int i = 0; i <static_cast<int>(m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices.size());i++)
        {
            ETG_TRACE_FATAL (("[ok] ========= Index=%d: =============================================",i));
            ETG_TRACE_FATAL (("[ok] [%2d]: Serial ID     : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cSerialID.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: USB Port      : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cUSBPort.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Type   : %d",     itr, ETG_ENUM(DVM_DEVICE_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceType)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Version: %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceVersion.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Name   : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceName.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Acessory Name : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cAccessoryName.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Mount Point   : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cMountPoint.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Connect Status: %d",     itr, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eConnectStatus)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Manufacturer  : %s",     itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cManufacturer.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: iVendorID     : 0x%04x", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_iVendorID));
            ETG_TRACE_FATAL (("[ok] [%2d]: iProductID    : 0x%04x", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_iProductID));
            ETG_TRACE_FATAL (("[ok] [%2d]: u64Size [KB]  : %u",     itr, (tU32)( m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_u64Size/1024) ));
            ETG_TRACE_FATAL (("[ok] [%2d]: Unsupported Reason : %d",itr, ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceUnsupportedReason)));

            ETG_TRACE_FATAL (("[ok] ====================================================================="));

            // dump to file, if available
            if (NULL != m_poDumpFile)
            {
                 ETG_TRACE_FATAL (("[ok] ========= Index=%d: =============================================",i));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Serial ID     : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cSerialID.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: USB Port      : %s",itr,      m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cUSBPort.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Type   : %d", itr,     ETG_ENUM(DVM_DEVICE_TYPE,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceType));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Version: %s",itr,      m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceVersion.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Name   : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cDeviceName.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Acessory Name : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cAccessoryName.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Mount Point   : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cMountPoint.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Connect Status: %d", itr,     ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,   m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eConnectStatus));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Manufacturer  : %s", itr,     m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_cManufacturer.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok]  [%2d]: iVendorID    : 0x%04x", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_iVendorID);
                fprintf (m_poDumpFile, "\n[ok]  [%2d]: iProductID   : 0x%04x", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_iProductID);
                fprintf (m_poDumpFile, "\n[ok] [%2d]: u64Size [KB]  : %u", itr, (tUInt)( m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_u64Size/1024) );
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Unsupported Reason : %d",itr,  ETG_ENUM(TENDEVICEUNSUPPORTEDREASON,  m_rSendConnectionNotificationInterface[itr].ListOfSendDevicestServices[i].m_eDeviceUnsupportedReason) );
                fprintf (m_poDumpFile, "\n[ok] =====================================================================");
            }
        }
        ETG_TRACE_FATAL (("[ok] ====TraceHistoryDeviceList\=============================================================================================="));
        ETG_TRACE_FATAL (("[ok]                                                                                                                               "));

    }

}


/*-----------------------------------------------------------------------------*
 * void TraceHistoryDeviceList ()                                                        *
 *-----------------------------------------------------------------------------*/
void InterfaceHistory::TraceHistoryConnectorList(IN unsigned int itr)
{

    tInt iNumOfUSBConnectors = ARRAYSIZEFORUSB - 1;

    //print only if send_nr not cleared i.e. != -1
    if((itr < HELPER_ANALYSIS_SENDINTERFACESERVICE) && (-1 != m_rSendConnectionNotificationInterface[itr].send_nr))
    {
        ETG_TRACE_FATAL (("[ok] ====/TraceHistoryConnectorList=============================================================================================="));

        ETG_TRACE_FATAL(("[ok] +++++++++++++++++++++++"));
        ETG_TRACE_FATAL(("[ok] ListOfSendArrayPortStates: Elems: %d",iNumOfUSBConnectors));
        
        if (NULL != m_poDumpFile)
        {
            fprintf (m_poDumpFile, "[ok] +++++++++++++++++++++++");
            fprintf (m_poDumpFile, "[ok] ListOfSendArrayPortStates: Elems: %d",iNumOfUSBConnectors);
        }
        for(int i = (int)eUSB1; i< ARRAYSIZEFORUSB; i++)
        {
             ETG_TRACE_FATAL (("[ok] ========= eUsb%d: =============================================",i));
             ETG_TRACE_FATAL (("[ok] [%2d]:  u8USBPortNo                      : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].u8USBPortNo));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bOverCurrent                     : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOverCurrent));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bUndervoltage                    : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUndervoltage));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bHubConnected                    : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bHubConnected));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bOpenCircuit                     : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOpenCircuit));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bElectricalFailure               : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bElectricalFailure));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bUSBPortConfiguredUsed           : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUSBPortConfiguredUsed));
             ETG_TRACE_FATAL (("[ok] [%2d]:  bUSBPowerON                      : %d", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUSBPowerON));

            if (NULL != m_poDumpFile)
            {
                fprintf (m_poDumpFile, "[ok] ========= eUsb%d: =============================================",i);
                fprintf (m_poDumpFile, "[ok] [%2d]:  u8USBPortNo              : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].u8USBPortNo);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bOverCurrent             : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOverCurrent);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bUndervoltage            : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUndervoltage);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bHubConnected            : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bHubConnected);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bOpenCircuit             : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bOpenCircuit);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bElectricalFailure       : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bElectricalFailure);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bUSBPortConfiguredUsed   : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUSBPortConfiguredUsed);
                fprintf (m_poDumpFile, "[ok] [%2d]:  bUSBPowerON              : %d\n", itr,  m_rSendConnectionNotificationInterface[itr].ListOfSendArrayPortStates[i].bUSBPowerON);
            }
        }
        ETG_TRACE_FATAL (("[ok] ====TraceHistoryConnectorList\=============================================================================================="));
        ETG_TRACE_FATAL (("[ok]                                                                                                                               "));

    }


}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
