/*-----------------------------------------------------------------------------*
 * HistoryManager.cpp                                                          *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2010 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file HistoryManager.cpp
 *
 * \brief This file holds the implementation for CHistoryManager
 *
 * \version 02.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 13.08.2012, Negi, Sunder (MontaVista), changes to dump traces to file
 * \version 13.08.2012, Koechling, Christian (Bosch), changes to switch on/off extra debug info 
 *                                                   using USB-stick with special Key
 * \version 29.08.2012, Koechling, Christian (Bosch), added traces and changed order of commands
 *
 *-----------------------------------------------------------------
 *									development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *		  -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_AILAHL //dependency with diaglib
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "StateTable.h"
#include "DeviceManager_ErrorCodes.h"
#include "Diagnosis.h"

#include "debug/TraceCmdManager.h"
#include "debug/HistoryManager.h"
#include "utils.h"
//#include <stdio.h>antilint: repeatedly included but does not have a standard include guard
#include <sys/stat.h>
//#include <errno.h>
#include <sys/mount.h>

#include <sys/types.h>
#include <dirent.h>

#ifndef _VD_DEVICEMANAGER_ROOTDAEMON_CLIENT_H_
#include "VD_DeviceManager_rootdaemon_client.h"
#endif

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_HISTORYMANAGER
#include "trcGenProj/Header/HistoryManager.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Singleton                                                                   *
 *-----------------------------------------------------------------------------*/
DVMLOCK           CHistoryManager::m_singelton;
CHistoryManager  *CHistoryManager::m_poHistoryManager = NULL;

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CHistoryManager::CHistoryManager ()
{
    m_cPlatformLogfile = NULL;
    m_cLogFile = NULL;
    m_poDumpFile = NULL;
    
    ClearDumpFile();
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CHistoryManager::~CHistoryManager ()
{
    // cleanup memory
    if (NULL != m_cPlatformLogfile)
    {
        free (m_cPlatformLogfile);
        m_cPlatformLogfile = NULL;
    }
    if (NULL != m_cLogFile)
    {
        free (m_cLogFile);
        m_cLogFile = NULL;
    }

    m_poDumpFile = NULL;
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * CHistoryManager *GetInstance ()                                             *
 *-----------------------------------------------------------------------------*/
CHistoryManager *CHistoryManager::GetInstance ()
{
    ETG_TRACE_USR3 (("GetInstance: Begin"));

    CHistoryManager::m_singelton.lock();
    if (NULL == m_poHistoryManager)
    {
        m_poHistoryManager = new CHistoryManager();
    }
    CHistoryManager::m_singelton.unlock();

    ETG_TRACE_USR3 (("GetInstance: End"));

    return m_poHistoryManager;
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance ()                                                     *
 *-----------------------------------------------------------------------------*/
void CHistoryManager::DestroyInstance()
{
    ETG_TRACE_USR3 (("DestroyInstance: Begin"));

    CHistoryManager::m_singelton.lock();
    DEL_M(m_poHistoryManager);
    CHistoryManager::m_singelton.unlock();

    ETG_TRACE_USR3 (("DestroyInstance: End"));
}

/*-----------------------------------------------------------------------------*
 * CDeviceProcessorHistory&  getDevHistory ()                                  *
 *-----------------------------------------------------------------------------*/
CDeviceProcessorHistory& CHistoryManager::getDevHistory()
{
    return m_oDevHistory;
}

/*-----------------------------------------------------------------------------*
 * CServiceHistory&  getServiceHistory ()                                      *
 *-----------------------------------------------------------------------------*/
CServiceHistory& CHistoryManager::getServiceHistory()
{
    return m_oServiceHistory;
}


InterfaceHistory& CHistoryManager::getInterfaceHistory()
{
    return m_oInterfaceHistory;
}
/*-----------------------------------------------------------------------------*
 * CPrmHistory& getPrmHistory ()                                               *
 *-----------------------------------------------------------------------------*/
CPrmHistory& CHistoryManager::getPrmHistory()
{
    return m_oPrmHistory;
}

/*-----------------------------------------------------------------------------*
 * VoltClientHistory& getVoltClientHistory ()                                   *
 *-----------------------------------------------------------------------------*/
VoltClientHistory& CHistoryManager::getVoltClientHistory()
{
    return m_oVoltClientHistory;
}

/*-----------------------------------------------------------------------------*
 * CDiagClientHistory& getDiagClientHistory ()                                 *
 *-----------------------------------------------------------------------------*/
CDiagClientHistory& CHistoryManager::getDiagClientHistory()
{
    return m_oDiagClientHistory;
}

/*-----------------------------------------------------------------------------*
 * CDiagClientHistory& getHwMalfunctionHistory()                                 *
 *-----------------------------------------------------------------------------*/
HWMalfunctionHistory& CHistoryManager::getHwMalfunctionHistory()
{
    return m_oHWMalfunctionHistory;
}


/*-----------------------------------------------------------------------------*
 * void setDumpFile( FILE*poDumpFile)                                   *
 *-----------------------------------------------------------------------------*/
void CHistoryManager::setDumpFile(FILE* poDumpFile)
{
    m_poDumpFile = poDumpFile;

    // set dump file for debug history objects
    m_oDevHistory.setDumpFile(m_poDumpFile);
    m_oServiceHistory.setDumpFile(m_poDumpFile);
    m_oPrmHistory.setDumpFile(m_poDumpFile);
    m_oVoltClientHistory.setDumpFile(m_poDumpFile);
    m_oDiagClientHistory.setDumpFile(m_poDumpFile);
}

/*-----------------------------------------------------------------------------*
 * void ClearDumpFile()                                                        *
 *-----------------------------------------------------------------------------*/
void CHistoryManager::ClearDumpFile()
{
    m_poDumpFile = NULL;

    // set dump file for debug history objects
    m_oDevHistory.setDumpFile(m_poDumpFile);
    m_oServiceHistory.setDumpFile(m_poDumpFile);
    m_oInterfaceHistory.setDumpFile(m_poDumpFile);
    m_oPrmHistory.setDumpFile(m_poDumpFile);
    m_oVoltClientHistory.setDumpFile(m_poDumpFile);
    m_oDiagClientHistory.setDumpFile(m_poDumpFile);
}

/*-----------------------------------------------------------------------------*
 * int InitDumpHistoryToUSBStick(const GENSTRING f_cUSBMountPoint)               *
 *-----------------------------------------------------------------------------*/
int CHistoryManager::InitDumpHistoryToUSBStick(const char *f_cUSBMountPoint)
{
    ETG_TRACE_USR3 (("InitDumpHistoryToUSBStick: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    // generate log directory name
    time_t     rawtime;
    struct tm *timeinfo;
    char       l_cLogDirName [80];
    char slogfile[512], splatformlogfile[512];

    time (&rawtime);
    timeinfo = localtime (&rawtime);
    strftime (l_cLogDirName, 80, "%Y-%m-%d_%H%M%S", timeinfo);
    ETG_TRACE_USR4 (("[ok] InitDumpHistoryToUSBStick: Log dir = %s", l_cLogDirName));

    // generate log file name including path
    snprintf (slogfile, 512, "%s/%s/%s",f_cUSBMountPoint, l_cLogDirName, "log.txt");

    // make the usb read/write
    execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_MOUNT_RW,f_cUSBMountPoint);
    // create log directory
    char l_cCmd[128];
    snprintf (l_cCmd, sizeof (l_cCmd), "%s/%s", f_cUSBMountPoint, l_cLogDirName);
    ETG_TRACE_USR4 (("[ok] InitDumpHistoryToUSBStick: Cmd: %s", l_cCmd));
   
    execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_MKDIR,l_cCmd);
    execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_CHMODFULL,l_cCmd);

    //check if dir exists
    DIR *pDir = opendir (l_cCmd);
    if (pDir != NULL)
    {
        (void) closedir (pDir);

        ETG_TRACE_USR3 (("InitDumpHistoryToUSBStick: log file = %s", slogfile));
        m_cLogFile = strdup (slogfile);

        // generate platform lig file name including path
        snprintf (splatformlogfile, 512, "%s/%s/%s",f_cUSBMountPoint, l_cLogDirName, "platformlog.txt");
        ETG_TRACE_USR3 (("InitDumpHistoryToUSBStick: platform log file = %s", splatformlogfile));
        m_cPlatformLogfile = strdup (splatformlogfile);
    }

    ETG_TRACE_USR3 (("InitDumpHistoryToUSBStick: End"));

    return l_iError;
}

/*-----------------------------------------------------------------------------*
 * int DumpPlatformHistoryToUSBStick()                                         *
 *-----------------------------------------------------------------------------*/
int CHistoryManager::DumpPlatformHistoryToUSBStick() const
{
    ETG_TRACE_USR3 (("DumpPlatformHistoryToUSBStick: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    if (NULL != m_cPlatformLogfile)
    {
        execRootCommand(VD_DEVICEMANAGER_CLIENT_NAME,CMD_DBG_ANALYSIS,m_cPlatformLogfile);
        ETG_TRACE_FATAL(("===================================="));
        ETG_TRACE_FATAL(("DumpPlatformHistoryToUSBStick: done!"));
        ETG_TRACE_FATAL(("===================================="));
    }
    else
    {
        l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
        ETG_TRACE_ERR (("DumpPlatformHistoryToUSBStick: [ERROR] dump file path is invalid"));
    }

    ETG_TRACE_USR3 (("DumpPlatformHistoryToUSBStick: End"));

    return l_iError;
}

/*-----------------------------------------------------------------------------*
 * int DumpHistoryToUSBStick()                                                 *
 *-----------------------------------------------------------------------------*/
int CHistoryManager::DumpHistoryToUSBStick()
{
    ETG_TRACE_USR3 (("DumpHistoryToUSBStick: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    if (NULL != m_cLogFile)
    {
        FILE *l_pFileLog;

        if (NULL != (l_pFileLog = fopen (m_cLogFile, "w")))
        {
            TraceCmdManager *l_poTraceClient = TraceCmdManager::GetInstance();

            setDumpFile(l_pFileLog);
            l_poTraceClient->setDumpFile(l_pFileLog);

            // dump the history
            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdShowConfiguration\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            l_poTraceClient->TTFIScmdShowConfiguration();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdPrintConfiguration\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            l_poTraceClient->TTFIScmdPrintConfiguration();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdCheckAutomountDir\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            l_poTraceClient->TTFIScmdCheckAutomountDir();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdGetConnectedDevices\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            l_poTraceClient->TTFIScmdGetConnectedDevices(NULL);

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdHistoryLastMsgsSend2Clients\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            SERVICEHISTORY.TraceHistory(); //depricated if Interface history has replaced it
            INTERFACEHISTORY.TraceHistory();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdHistoryUdevEvts\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            DEVICEPROCESSORHISTORY.TraceHistory();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: TTFIScmdHistoryLastMsgsSend2Diag\n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            DIAGCLIENTHISTORY.TraceHistory();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: VD_DVM_HISTORY_CALLBACK_PRM \n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            PRMHISTORY.TraceHistory();

            fprintf (l_pFileLog, "\n\n");
            fprintf (l_pFileLog, "===================================================================================\n");
            fprintf (l_pFileLog, "  cmd: VD_DVM_HISTORY_NOTIFICATION_VOLTAGE \n");
            fprintf (l_pFileLog, "===================================================================================\n\n");
            VOLTCLIENTHISTORY.TraceHistory();

            // cleanup
            ClearDumpFile ();
            l_poTraceClient->setDumpFile(m_poDumpFile); //koe2hi: placed it here because ClearDumpFile is called in init phase and pointer 
                                                        //l_poTraceClient seems to be not available

            ETG_TRACE_FATAL(("===================================="));
            ETG_TRACE_FATAL(("DumpHistoryToUSBStick: done!        "));
            ETG_TRACE_FATAL(("===================================="));

            fclose(l_pFileLog);
        }
        else
        {
            l_iError = DEVICEMANAGER_ERROR_FILE_OPEN;
            ETG_TRACE_ERR (("DumpHistoryToUSBStick: [ERROR] 'fopen' failed"));
        }
    }
    else
    {
        l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
        ETG_TRACE_ERR (("DumpHistoryToUSBStick: [ERROR] dump file path is invalid"));
    }

    ETG_TRACE_USR3 (("DumpHistoryToUSBStick: End"));

    return l_iError;
}

/*-----------------------------------------------------------------------------*
 * void StopDumpHistoryToUSBStick()                                            *
 *-----------------------------------------------------------------------------*/
void CHistoryManager::StopDumpHistoryToUSBStick()
{
    ETG_TRACE_USR3 (("StopDumpHistoryToUSBStick: Begin"));

    // cleanup FILE object
    if (NULL != m_poDumpFile)
    {
        fflush (m_poDumpFile);
        fclose (m_poDumpFile);
        m_poDumpFile = NULL;
    }

    // cleanup memory
    if (NULL != m_cPlatformLogfile)
    {
        free (m_cPlatformLogfile);
        m_cPlatformLogfile = NULL;
    }
    if (NULL != m_cLogFile)
    {
        free (m_cLogFile);
        m_cLogFile = NULL;
    }
}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
