/*-----------------------------------------------------------------------------*
 * DeviceProcessorHistory.cpp                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DeviceProcessorHistory.cpp
 *
 * \brief This file holds the implementation for CDeviceProcessorHistory
 *
 * \version 02.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb
 *          (WRITEHISTORYTOUSBSTICK)
 * \version 30.08.2012, Koechling, Christian (Bosch), fix for crash in TraceHistory
 *
 *-----------------------------------------------------------------
 *									development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *		  -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "debug/ObjectHistory.h"
#include "debug/DeviceProcessorHistory.h"
#include "config/ConfigurationManager.h" //for eclapsed time

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_HISTORYMANAGER
#include "trcGenProj/Header/DeviceProcessorHistory.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CDeviceProcessorHistory::CDeviceProcessorHistory()
    :CObjectHistory()
{
    CDeviceProcessorHistory::ClearHistory ();

}//lint !e1566 
//Warning 1566: prio3: member 'CDeviceProcessorHistory::m_uiCurrent'      might have been initialized by a separate function but no '-sem(CDeviceProcessorHistory::ClearHistory,initializer)' was seen
//Warning 1566: prio3: member 'CDeviceProcessorHistory::m_uiTotalCounter' might have been initialized by a separate function but no '-sem(CDeviceProcessorHistory::ClearHistory,initializer)' was seen


/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CDeviceProcessorHistory::~CDeviceProcessorHistory()
{
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * void AddToHistory (GENSTRING f_cAction, CDevice *f_pCurDevice)                *
 *-----------------------------------------------------------------------------*/
void CDeviceProcessorHistory::AddToHistory (GENSTRING f_cAction, const CDevice *f_pCurDevice)
{
    ETG_TRACE_USR3 (("AddToHistory: Begin"));
    m_LockHistory.lock();

    if (NULL != f_pCurDevice)
    {
        if (m_uiCurrent >= HELPER_ANALYSIS_UDEVEVENTS)
        {
            m_uiCurrent = 0;
        }

        m_aoDevEvts[m_uiCurrent].u32TimeStamp = (tU32)ConfigurationManager::ConfigurationManager::GetInstance()->GetEclapsedTime_ms();
        m_aoDevEvts[m_uiCurrent].action       = f_cAction;
        m_aoDevEvts[m_uiCurrent].serial_id    = f_pCurDevice->m_cSerialID;
        m_aoDevEvts[m_uiCurrent].mount_point  = f_pCurDevice->m_cMountPoint;
        m_aoDevEvts[m_uiCurrent].dev_node     = f_pCurDevice->m_cDevnode;
        m_aoDevEvts[m_uiCurrent].dev_type     = f_pCurDevice->m_eDeviceType;
        m_aoDevEvts[m_uiCurrent].rcv_counter  = (int)m_uiTotalCounter;


        ETG_TRACE_USR4 (("AddToHistory: Event for %d", ETG_ENUM(DVM_DEVICE_TYPE, m_aoDevEvts[m_uiCurrent].dev_type)));
    }
    else
    {
        DVM_NORMAL_M_ASSERT_ALWAYS();
    }

    m_uiCurrent++;

    ETG_TRACE_USR4 (("AddToHistory: TotalCounter[%d] Current[%d]", m_uiTotalCounter, m_uiCurrent));

    m_LockHistory.unlock();

    ETG_TRACE_USR3 (("AddToHistory: End"));
}

/*-----------------------------------------------------------------------------*
 * void AddToHistory (GENSTRING f_cAction, GENSTRING f_cDevNode,                   *
 *                    GENSTRING f_cMountPoint)                                   *
 *-----------------------------------------------------------------------------*/
void CDeviceProcessorHistory::AddToHistory (GENSTRING f_cAction, GENSTRING f_cDevNode,
        GENSTRING f_cMountPoint)
{
    ETG_TRACE_USR3 (("AddToHistory: Begin"));
    m_LockHistory.lock();

    if (GENSTRING("") == f_cMountPoint)
    {
        m_uiTotalCounter++;
    }

    if (m_uiCurrent >= HELPER_ANALYSIS_UDEVEVENTS)
    {
        m_uiCurrent = 0;
    }

    m_aoDevEvts[m_uiCurrent].u32TimeStamp = (tU32)ConfigurationManager::ConfigurationManager::GetInstance()->GetEclapsedTime_ms();
    m_aoDevEvts[m_uiCurrent].action      = f_cAction;
    m_aoDevEvts[m_uiCurrent].rcv_counter = (int)m_uiTotalCounter;
    m_aoDevEvts[m_uiCurrent].dev_node    = (f_cDevNode != "")?f_cDevNode:"  EMPTY  " ;       //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
    m_aoDevEvts[m_uiCurrent].mount_point = (f_cMountPoint != "")?f_cMountPoint:"  EMPTY  " ; //lint !e1702 Info 1702: operator 'operator!=' is both an ordinary function '...' and a member function '...'
    m_aoDevEvts[m_uiCurrent].dev_type    = CGlobalEnumerations::DTY_UNKNOWN;

    m_uiCurrent++;

    ETG_TRACE_USR4 (("AddToHistory: TotalCounter[%d] Current[%d]",
            m_uiTotalCounter, m_uiCurrent));

    m_LockHistory.unlock();

    ETG_TRACE_USR3 (("AddToHistory: End"));
}

/*-----------------------------------------------------------------------------*
 * void TraceHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void CDeviceProcessorHistory::TraceHistory () //called triggered by TTFIS cmd only
{
    GENSTRING strActionMountPointAndId;

    static GENSTRING strDeviceTypes[(int)CGlobalEnumerations::DTY_LASTELEM];
    strDeviceTypes[CGlobalEnumerations::DTY_UNKNOWN]       = "DTY_UNKNOWN";
    strDeviceTypes[CGlobalEnumerations::DTY_USB]           = "DTY_USB    ";
    strDeviceTypes[CGlobalEnumerations::DTY_SD]            = "DTY_SD     ";
    strDeviceTypes[CGlobalEnumerations::DTY_IPOD]          = "DTY_IPOD   ";
    strDeviceTypes[CGlobalEnumerations::DTY_IPHONE]        = "DTY_IPHONE ";
    strDeviceTypes[CGlobalEnumerations::DTY_MTP]           = "DTY_MTP    ";
    strDeviceTypes[CGlobalEnumerations::DTY_BLUETOOTH]     = "DTY_BLUETOO";
    strDeviceTypes[CGlobalEnumerations::DTY_MSZUNE]        = "DTY_MSZUNE ";
    strDeviceTypes[CGlobalEnumerations::DTY_IPAD]          = "DTY_IPAD   ";
    strDeviceTypes[CGlobalEnumerations::DTY_HUB]           = "DTY_HUB    ";
    strDeviceTypes[CGlobalEnumerations::DTY_NOT_SUPPORTED] = "DTY_NOT_SUPPORTED";
	strDeviceTypes[CGlobalEnumerations::DTY_SD_INTERNAL]   = "DTY_SD_INTERNAL";
	strDeviceTypes[CGlobalEnumerations::DTY_CDROM]         = "DTY_CDROM";

    ETG_TRACE_USR3 (("TraceHistory: Begin"));
    ETG_TRACE_FATAL (("[ok] ===============================->DeviceProcessor==============================="));

    tU64 u64StartTimeAbsoluteDeviceManager = ConfigurationManager::GetInstance()->GetStartTime_ms();
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (ms) : ",u64StartTimeAbsoluteDeviceManager);
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (sec): ",u64StartTimeAbsoluteDeviceManager/1000);

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,
                "\n[ok] ===============================->DeviceProcessor===============================");
        fprintf (m_poDumpFile,"u64StartTimeAbsoluteDeviceManager (ms) : %llu\n",u64StartTimeAbsoluteDeviceManager);
        fprintf (m_poDumpFile,"u64StartTimeAbsoluteDeviceManager (sec): \n",u64StartTimeAbsoluteDeviceManager);
        fprintf (m_poDumpFile,"u64StartTimeRelativeDeviceManager (ms) : 0\n",u64StartTimeAbsoluteDeviceManager);
    }

    for (int i = 0;i < HELPER_ANALYSIS_UDEVEVENTS;i++)
    {
        if (-1 != m_aoDevEvts[i].rcv_counter)
        {
            strActionMountPointAndId = "";
            //ETG supports one string only
            strActionMountPointAndId.append(strDeviceTypes[m_aoDevEvts[i].dev_type].toStdString().c_str());
            strActionMountPointAndId.append(" , ");
            strActionMountPointAndId.append(m_aoDevEvts[i].action);
            strActionMountPointAndId.append(" , ");
            strActionMountPointAndId.append(m_aoDevEvts[i].mount_point);
            strActionMountPointAndId.append(" , ");
            strActionMountPointAndId.append(m_aoDevEvts[i].dev_node);
            strActionMountPointAndId.append(" , ");
            strActionMountPointAndId.append(m_aoDevEvts[i].serial_id);
            ETG_TRACE_FATAL (("[ok] StoredAdds[%2d]:SendNo %3d, Time %12d ms : %s",
                    m_aoDevEvts[i].rcv_counter,
                    i,
                    m_aoDevEvts[i].u32TimeStamp,
                     strActionMountPointAndId.toStdString().c_str()));

            // dump to file, if available
            if (NULL != m_poDumpFile)
            {
                fprintf (m_poDumpFile, "\n[ok] StoredAdds[%2d]:No %3d,Time %12d ms : %s",
                        m_aoDevEvts[i].rcv_counter,
                        i, 
                        m_aoDevEvts[i].u32TimeStamp,
                        strActionMountPointAndId.toStdString().c_str());
             }
        }
    }

    ETG_TRACE_FATAL (("[ok] ================================DeviceProcessor<-=============================="));

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,
                "\n[ok] ================================DeviceProcessor<-==============================");
    }

    ETG_TRACE_USR3 (("TraceHistory: End"));
}

/*-----------------------------------------------------------------------------*
 * void ClearHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void CDeviceProcessorHistory::ClearHistory ()
{
    ETG_TRACE_USR3 (("ClearHistory: Begin"));

    m_uiCurrent      = 0;
    m_uiTotalCounter = 0;

    for (int i = 0;i < HELPER_ANALYSIS_UDEVEVENTS;i++)
    {
        m_aoDevEvts[i].serial_id   = "  EMPTY  ";
        m_aoDevEvts[i].action      = "  EMPTY  ";
        m_aoDevEvts[i].mount_point = "  EMPTY  ";
        m_aoDevEvts[i].dev_node    = "  EMPTY  ";
        m_aoDevEvts[i].dev_type    = CGlobalEnumerations::DTY_UNKNOWN;
        m_aoDevEvts[i].rcv_counter = -1;
    }

    ETG_TRACE_USR3 (("ClearHistory: End"));
}  //lint !e1565 Warning 1565: member 'CObjectHistory::m_poDumpFile' (VD_DeviceManager/inc/debug/ObjectHistory.h) not assigned by initializer function

////////////////////////////////////////////////////////////////////////////////
// <EOF>

