/*-----------------------------------------------------------------------------*
 * IConfigurationPersistent.h                                                      *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM G3G                                                   *
 * COPYRIGHT   : (c) 2014-2016  Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file IConfigurationPersistent.h
 *
 * \brief Implementation to open, read and close persistent configuration
 *
 * \version 05.03.2014, Koechling Christian (CM-AI/ECD1), version 1.0
 
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef INTERFACE_CONFIGURATION_PERSISTENT_H
#define INTERFACE_CONFIGURATION_PERSISTENT_H  

#include "Config.h"
#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_EOL
#define INCLUDE_VD_DVM_EOLLIB
#endif // #ifdef VARIANT_S_FTR_ENABLE_DEVMGR_EOL
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

#include "Common.h"
#include "Enums.h"



/*!
 * \namespace generickds
 *
 * \brief enums and structs provided in context of Generic KDS
 */
namespace generic_kds 
{


    typedef enum
    {
        eDVM_UNDEFINED = 0x00,
        eDVM_ON        = 0x01,
        eDVM_OFF       = 0x02
    }tenKDSValue;

	typedef enum
	{
		ePRM_PORTS_UDEFINED = 0x00,
		ePRM_PORT1_USED     = 0x01,
		ePRM_PORT2_USED     = 0x02,
		ePRM_PORT3_USED     = 0x03,
		ePRM_PORT4_USED     = 0x04
	}tenKDSValueHWSignal;

/*
	  Byte	  Bit	  Name	  Conversion 
	  0 - 31  RegistryFile	  ASCII[32] 
	  see https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-893134/Gen3g_CMDS_1v17RC1.pdf  
*/
	typedef struct
	{
	  tU8 RegistryFile[32]; //i.e. Subvariants //Note:KDS entry might not hold end of string i.e. /0

	}generickds_mediaplayer; 


	typedef struct
	{
		tU8 RegistryFile[32+1];; //i.e. Subvariants
	}generickds_mediaplayer_decoded; 


	
/*
	Byte	Bit 	Name	Conversion 
	0 - 7	Tag Parameter set identification tag.	ASCII[8] 
	8	Major version see (7)	ASCII[1] 
	9 - 10	Minor version see (8)	ASCII[2] 
	11 - 42 	Subvariant	ASCII[32] 
	43	Settings	HexDump (1 Byte) 
	=>reads currently to this 
	0 - 3	IndividualSettingsFollows	DVM On/Off 
	etc.
*/
	typedef struct
    {
	    tU8 Tag[8];               
		tU8 MajorVersion[1];   //Note: diagnosis has nothing reserved for end of string i.e no /0 
		tU8 MinorVersion[2];
		tU8 Subvariants[32];	
		tU8 IndividualSettings[GENERIC_KDS_ADDRESS_DEVICEMANAGER_ARRAYOFVALUES_BYTELEN-(8+1+2+32)];
	}generickds_devicemanager; 


	typedef struct
	{

		tU8 Tag[8+1];               
		tU8 MajorVersion[1+1];   //Note: diagnosis has nothing reserved for end of string i.e no /0 
		tU8 MinorVersion[2+1];
		tU8 Subvariants[32+1];	

        //FOR FIRST IMPLEMENTATION THIS PART IS NOT TREATED - FOCUS IS ON SUBVARIANTS
        //43: IndividualSettings
		tenKDSValue eIndividualSettingsFollows;
		//44-45: chipconnectors
		tenKDSValue eChipConnector1;
		tenKDSValue eChipConnector2;
		tenKDSValue eChipConnector3;
		tenKDSValue eChipConnector4;
		//46-47: HW USB Ports
		tenKDSValue eHwUSB1;
		tenKDSValue eHwUSB2;
		tenKDSValue eHwUSB3;
		tenKDSValue eHwUSB4;
		//48-49: USB Ports
		tenKDSValue eUSB1;
		tenKDSValue eUSB2;
		tenKDSValue eUSB3;
		tenKDSValue eUSB4;
		//50-51 ChipconnectorAllowHub
		tenKDSValue eChipConnector1AllowHub;
		tenKDSValue eChipConnector2AllowHub;
		tenKDSValue eChipConnector3AllowHub;
		tenKDSValue eChipConnector4AllowHub;
        //=>
		//koe2hi: Need for clarification HubAndPorts and HubConnected and how these are bound to existing configuration values of VD_DVM
		//52-53 HubAndPorts (for none customer hub?)
		tU8 iDVMEnableHubAndPorts_USB1; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubAndPorts_USB2; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubAndPorts_USB3; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubAndPorts_USB4; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		//53-54 HubConnected - (for customer HUB?)
		tU8 iDVMEnableHubConnected_USB1; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubConnected_USB2; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubConnected_USB3; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
		tU8 iDVMEnableHubConnected_USB4; //0 is off values >0 means on and how many ports are usable  //4bits are used 1111 i.e. max value is 15
        //<=
        
		//56-57 ObserveDTCForHubs means for customer hubs DTC-diagnosis is enabeld
		tenKDSValue eObserveDTCForHubs_USB1;
		tenKDSValue eObserveDTCForHubs_USB2;
		tenKDSValue eObserveDTCForHubs_USB3;
		tenKDSValue eObserveDTCForHubs_USB4;
		//58-60 TelematicsBox
		tenKDSValue eTelematicsBox_USB1;
		tenKDSValue eTelematicsBox_USB2;
		tenKDSValue eTelematicsBox_USB3;
		tenKDSValue eTelematicsBox_USB4;
		//61-62 HwSignalSource  means prm notifies signals e.g. telling that IMX1 orIMX2 connector has electrical failure - due to demanded customer mapping and possible internal hubs this has to mapped corretly
		tenKDSValueHWSignal eHWSignalSourceForUSB1;
		tenKDSValueHWSignal eHWSignalSourceForUSB2;
		tenKDSValueHWSignal eHWSignalSourceForUSB3;
		tenKDSValueHWSignal eHWSignalSourceForUSB4;
		//63-64 USBEnumeration gives hint about how this external USBConnector number is bound to the internal one
		/* koe2hi teStringMatchers needs to be shifted from ConfigurationBase - further look hast to be done for this
		teStringMatchers  eUSBEnumeration_USB1;
		teStringMatchers  eUSBEnumeration_USB2;
		teStringMatchers  eUSBEnumeration_USB3;
		teStringMatchers  eUSBEnumeration_USB4;
		*/
		//65-68 FileSystem
		tenKDSValue eFAT;
		tenKDSValue NTFS;
		tenKDSValue EXTFAT;
		tenKDSValue HFSPlus;
		tenKDSValue Ext3_4_FS;
		tenKDSValue ISO9660;
		tenKDSValue UDF;
		//69-71 DeviceNotify
		tenKDSValue eUnsupportedDeviceType;
		tenKDSValue eUnsupportedFileSystem;
		tenKDSValue eUnsupportedHub;
	    tenKDSValue eAllPartitions;
		tenKDSValue eMSDBiggestPArtition;
		tenKDSValue eSortAllPartitions;
		//72-75 Configure Device1 - enable/disabled certain devices external means devices connected externally to the target, internal means build into the deive
		tenKDSValue eExternal_SD;
		tenKDSValue eExternal_USB;
		tenKDSValue eExternal_MTP;
		tenKDSValue eExternal_IPOD;
		tenKDSValue eExternal_IPAD;
		tenKDSValue eExternal_IPHONE;
		tenKDSValue eInternal_CDROM;
		tenKDSValue eInternal_CDDA;
		//72-75 Configure Device2 - enable/disabled certain devices external means devices connected externally to the target, internal means build into the deive
		tenKDSValue eInternal_SDCARD; //special reader bound to mms e.g. used in NissanLCN2Kai
		tenKDSValue eExternal_DIGIAUX;
		tenKDSValue eExternal_BTA;
		tenKDSValue eExternal_UNSUPPORTED;
		//78 Configure Others
		tenKDSValue eChargeAppledevs; //apple devices can be charged with >500mA if certain libusb command is send to it
		tU16        u16CRC; //could be used to add additional check to this set of values
		
		
	}generickds_devicemanager_dedcoded; 






  
};


//#include "GenString.h"
/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class IConfigurationPersistent
 */

class IConfigurationPersistent
{
    public:


    enum enTypeOfPeristentReader
    {
		eReader_Undefined       = 0x00,
		eReader_KDS    	        = 0x01,
		eReader_EOL             = 0x02,
		eReader_WriterDataPool  = 0x03,
		eReader_ReaderDataPool  = 0x04
    };

        /*!
         * \brief Default constructor
         */
        IConfigurationPersistent(){};

        /*!
         * \brief Destructor
         */
        virtual ~IConfigurationPersistent(){};



		virtual int GetTypeOfReader(){return (int)eReader_Undefined;}

        /*!
         * \brief   Read the EOL data from the file
         *
         * \param [in]  cu8Table   table type
         * \param [in]  cu16Offset offset value
         * \param [out] rfu8Value  output value
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bReadEOL(IN tCU8 cu8Table, IN tCU16 cu16Offset, tU8 &rfu8Value){return FALSE;}; //EOL specific

        /*!
         * \brief   Read the KDS data from the file
         *
         * \param [in]  u16EntryID   Key to get value
         * \param [out] rfu8Value  output value
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bReadKDS(IN tU16 u16EntryID, IN tU16 u16EntryLength, OUT  tU8 (&f_u8Array)[KDS_MAX_ENTRY_LENGTH]){return FALSE;}; //KDS
		
        /*!
         * \brief   Read the Persistent data from datapool
         *
         * \param [in]  u16EntryID   Key to get value
         * \param [out] rfu8Value  output value
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bReadDataPool(IN tU16 eTypeOfConfigValue, OUT tU16 &u16ConfigValue  ){return FALSE;}; //datapool

        /*!
         * \brief   Read the Persistent data from datapool
         *
         * \param [in]  u16EntryID   Key to get value
         * \param [out] rfu8Value  output value
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bWriteDataPool(IN tU16 eTypeOfConfigValue, IN tU16 u16ConfigValue  ){return FALSE;}; //datapool

        /*!
         * \brief   Read the Persistent data from datapool - Overloaded for ExternalHub(RNAIVI)
         *
         * \param [out]  f_uiExternalHubInfo ExternalHub structure
         * \param [out]  f_uiNumberOfExternalHubsUsed ExternalHub Count
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bReadDataPool(OUT trExternalHubInfo f_uiExternalHubInfo [] , OUT tU8 &f_uiNumberOfExternalHubsUsed){return FALSE;}; //datapool

        /*!
         * \brief   Read the Persistent data from datapool - Overloaded for ExternalHub(RNAIVI)
         *
         * \param [in]  f_uiExternalHubInfo ExternalHub structure
         * \param [in]  f_uiNumberOfExternalHubsUsed ExternalHub Count
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bWriteDataPool(IN trExternalHubInfo f_uiExternalHubInfo [] ,IN tU8 f_uiNumberOfExternalHubsUsed){return FALSE;}; //datapool

        /*!
         * \brief   Read the Persistent data from datapool - Overloaded for ExternalHub(RNAIVI)
         *
         * \param [out]  f_uiTelematicsBoxDeviceInfo ExternalHub structure
         * \param [out]  f_uiNumberOfTelematicBoxesUsed ExternalHub Count
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bReadDataPool(OUT trTelematicsBoxDeviceInfo f_uiTelematicsBoxDeviceInfo [] , OUT tU8 &f_uiNumberOfTelematicBoxesUsed){return FALSE;}; //datapool

        /*!
         * \brief   Read the Persistent data from datapool - Overloaded for ExternalHub(RNAIVI)
         *
         * \param [in]  f_uiTelematicsBoxDeviceInfo ExternalHub structure
         * \param [in]  f_uiNumberOfTelematicBoxesUsed ExternalHub Count
         *
         * \return TRUE/FALSE  success/failure
         */
        virtual tBool bWriteDataPool(IN trTelematicsBoxDeviceInfo f_uiTelematicsBoxDeviceInfo [] ,IN tU8 f_uiNumberOfTelematicBoxesUsed){return FALSE;}; //datapool





};
// class IConfigurationPersistent

#endif // INTERFACE_CONFIGURATION_PERSISTENT_H

////////////////////////////////////////////////////////////////////////////////
// <EOF>




