/*-----------------------------------------------------------------------------*
 * includes                                                                  *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include <fstream>

#include "config/ConfigurationBase.h"
#include "config/ConfigurationProduct.h"
#include "config/IConfigurationPersistent.h"
#include "config/ReaderKDS.h"
#include "CommonReadRoutinesKds.h"
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_DATAPOOL
#include "config/ReaderDatapool.h" //for reading the persistent data
#endif//VARIANT_S_FTR_ENABLE_DEVMGR_DATAPOOL
// Includes all header files for FI of DP-component
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_devicemanager_if.h"
/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CONFIGMANAGER
  #include "trcGenProj/Header/ConfigurationProduct_Gen3_INF4CV.cpp.trc.h"
  #endif
  #include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * namespace                                                                 *
 *-----------------------------------------------------------------------------*/
using namespace configmgr;

#define USB_DEVICES_VID_PID_CONFIG_FILE "/opt/bosch/whitelist/usb_supported_devices.txt"
#define USB_DEVICETYPE_EXTERNALHUB "EHUB"

/*-----------------------------------------------------------------------------*
 * constructor()
 *-----------------------------------------------------------------------------*/
ConfigurationProduct::ConfigurationProduct()
{
    ETG_TRACE_USR4(("Begin: Constructor: ConfigurationProduct() (SUZUKI)"));
    m_pIFConfPersistent = NULL;
    bInterfaceConfPersistentFactoryMethod();
    vInitCustomerValues();
    TracestrSysPathEntryUSBPortAdressForUSBConnector();

    //if the HubConnectedToHubPermanent is Enabled, then the project specific configuration will be overloaded by
    //calling the function vEnalbleHubConnectedToHubPermanent
    if(TRUE == isHubConnectedToHubPermanentEnabled())
    {
        //this will set internal configuration value eCONF_ENABLE_HUBconnectedToHUBPermanent, which influences some of the project specific configuration
        //values in ConfigurationProduct class. See the file ConfigurationProduct_Gen3_JACCII.cpp for example
        vEnableHubConnectedToHubPermanent(TRUE);
    }
    ETG_TRACE_USR4(("End  : Constructor: ConfigurationProduct() (SUZUKI)"));
}

/*-----------------------------------------------------------------------------*
 * InterfaceConfPersistentFactoryMethod()
 *-----------------------------------------------------------------------------*/
tBool ConfigurationProduct::bInterfaceConfPersistentFactoryMethod()
{
    tBool bRetVal = FALSE;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    if(!m_pIFConfPersistent)
    {
        m_pIFConfPersistent = (IConfigurationPersistent*)new ReaderKDS();

        if(NULL == m_pIFConfPersistent)
        {
            bRetVal = TRUE;
            ETG_TRACE_ERRMEM(("[ERROR]: bInterfaceConfPersistentFactoryMethod (Product GMMY16) failed to create new ReaderKDS "));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR] bInterfaceConfPersistentFactoryMethod: (Product GMMY16) ReaderKDS already exists"));
    }
#endif

    return bRetVal;
}

//--------------------------------------------------------------
//see fct. GetSubversion(),teSubversionType,trSubVersionValueDescription
//see https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-893141/Nissan_A-IVI_PD_DeviceManager.xlsx received from RNAIVI FO-Mediaplayer
//the table can be found also in
//TODO : Once new KDS have been invented for INF4CV, Remove the subversion related code part completely
trSubVersionValueDescription ConfigurationProduct::m_rSubVersionValueDescriptions[eSubTypeLastElem] =
{
        //enum                                                Registry name                               Description                                                                                                  Use Registry (alternative is to harcoded feature set for this subversion (level1 implementation))
        {eSubType_GMP_Undefined                 , "-"                           ,"KDS-entry has value 'undefined' hence default setting is used"    , FALSE , -1 /*info existence internal HUB*/},
        {eSubType_GMP_NisScope1                 , "GMP_NisScope1"               ,"KDS-entry has value GMP_NisScope1.reg"                            , FALSE , -1 /*undefined*/},
        {eSubType_GMP_NisScope2                 , "GMP_NisScope2"               ,"KDS-entry has value GMP_NisScope2.reg"                            , FALSE , -1 },
        {eSubType_GMP_NisScope2_1AIVI           , "GMP_NisScope2_1AIVI"         ,"KDS-entry has value GMP_NisScope2_1AIVI.reg"                      , FALSE , -1},
        {eSubType_GMP_NisScope2_1RENAULT        , "GMP_NisScope2_1RENAULT"      ,"KDS-entry has value GMP_NisScope2_1RENAULT.reg"                   , FALSE , -1},
        {eSubType_GMP_NIS_PIVI                  , "GMP_Nis_PIVI"                ,"KDS-entry has value GMP_Nis_PIVI.reg"                             , FALSE , -1},
        {eSubType_GMP_IVI_SC2_Nis_1P_nH         , "GMP_IVI_SC2_Nis_1P_nH"       ,"KDS-entry has value GMP_IVI_SC2_Nis_1P_nH.reg"                    , FALSE ,  0 /*nH*/},
        {eSubType_GMP_IVI_SC2_Nis_2P_nH_TCU     , "GMP_IVI_SC2_Nis_2P_nH_TCU"   ,"KDS-entry has value GMP_IVI_SC2_Nis_2P_nH_TCU.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_Nis_2P_wH_TCU     , "GMP_IVI_SC2_Nis_2P_wH_TCU"   ,"KDS-entry has value GMP_IVI_SC2_Nis_2P_wH_TCU.reg"                , FALSE ,  1 /*wH*/},
        {eSubType_GMP_IVI_SC2_Nis_2P_nH_ETC     , "GMP_IVI_SC2_Nis_2P_nH_ETC"   ,"KDS-entry has value GMP_IVI_SC2_Nis_2P_nH_ETC.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_Nis_2P_wH_ETC     , "GMP_IVI_SC2_Nis_2P_wH_ETC"   ,"KDS-entry has value GMP_IVI_SC2_Nis_2P_wH_ETC.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_Nis_3P_wH         , "GMP_IVI_SC2_Nis_3P_wH"       ,"KDS-entry has value GMP_IVI_SC2_Nis_3P_wH.reg"                    , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Nis_1P_nH       , "GMP_IVI_SC2.1_Nis_1P_nH"     ,"KDS-entry has value GMP_IVI_SC2.1_Nis_1P_nH.reg"                  , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Nis_2P_nH_TCU   , "GMP_IVI_SC2.1_Nis_2P_nH_TCU" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_2P_nH_TCU.reg"              , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Nis_2P_wH_TCU   , "GMP_IVI_SC2.1_Nis_2P_wH_TCU" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_2P_wH_TCU.reg"              , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Nis_2P_nH_ETC   , "GMP_IVI_SC2.1_Nis_2P_nH_ETC" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_2P_nH_ETC.reg"              , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Nis_2P_wH_ETC   , "GMP_IVI_SC2.1_Nis_2P_wH_ETC" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_2P_wH_ETC.reg"              , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Nis_2P_nH_ATB   , "GMP_IVI_SC2.1_Nis_2P_nH_ATB" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_2P_nH_ATB"                  , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Nis_3P_wH       , "GMP_IVI_SC2.1_Nis_3P_wH"     ,"KDS-entry has value GMP_IVI_SC2.1_Nis_3P_wH.reg"                  , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Nis_3P_wH_ATB   , "GMP_IVI_SC2.1_Nis_3P_wH_ATB" ,"KDS-entry has value GMP_IVI_SC2.1_Nis_3P_wH_ATB"                  , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Ren_1P_nH       , "GMP_IVI_SC2.1_Ren_1P_nH"     ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_nH_TCU.reg"              , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Ren_2P_nH_TCU   , "GMP_IVI_SC2.1_Ren_2P_nH_TCU" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_nH_TCUreg"               , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Ren_2P_wH_TCU   , "GMP_IVI_SC2.1_Ren_2P_wH_TCU" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_wH_TCU.reg"              , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Ren_2P_nH_ETC   , "GMP_IVI_SC2.1_Ren_2P_nH_ETC" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_nH_ETC.reg"              , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Ren_2P_wH_ETC   , "GMP_IVI_SC2.1_Ren_2P_wH_ETC" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_wH_ETC.reg"              , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Ren_2P_nH_ATB   , "GMP_IVI_SC2.1_Ren_2P_nH_ATB" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_2P_nH_ATB.reg"              , FALSE ,  0},
        {eSubType_GMP_IVI_SC2_1_Ren_3P_wH       , "GMP_IVI_SC2.1_Ren_3P_wH"     ,"KDS-entry has value GMP_IVI_SC2.1_Ren_3P_wH.reg"                  , FALSE ,  1},
        {eSubType_GMP_IVI_SC2_1_Ren_3P_wH_ATB   , "GMP_IVI_SC2.1_Ren_3P_wH_ATB" ,"KDS-entry has value GMP_IVI_SC2.1_Ren_3P_wH_ATB.reg"              , FALSE ,  1},
        {eSubType_GMP_IVI_PIVI_Nis_1P_nH        , "GMP_IVI_PIVI_Nis_1P_nH"      ,"KDS-entry has value GMP_IVI_PIVI_Nis_1P_nH.reg"                   , FALSE ,  0},
        {eSubType_GMP_IVI_PIVI_Nis_2P_nH_TCU    , "GMP_IVI_PIVI_Nis_2P_nH_TCU"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_2P_nH_TCU.reg"               , FALSE ,  0},
        {eSubType_GMP_IVI_PIVI_Nis_2P_wH_TCU    , "GMP_IVI_PIVI_Nis_2P_wH_TCU"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_2P_wH_TCU.reg"               , FALSE ,  1},
        {eSubType_GMP_IVI_PIVI_Nis_2P_nH_ETC    , "GMP_IVI_PIVI_Nis_2P_nH_ETC"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_2P_nH_ETC.reg"               , FALSE ,  0},
        {eSubType_GMP_IVI_PIVI_Nis_2P_wH_ETC    , "GMP_IVI_PIVI_Nis_2P_wH_ETC"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_2P_wH_ETC.reg"               , FALSE ,  1},
        {eSubType_GMP_IVI_PIVI_Nis_2P_nH_ATB    , "GMP_IVI_PIVI_Nis_2P_nH_ATB"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_2P_nH_ATB.reg"               , FALSE ,  0},
        {eSubType_GMP_IVI_PIVI_Nis_3P_wH        , "GMP_IVI_PIVI_Nis_3P_wH"      ,"KDS-entry has value GMP_IVI_PIVI_Nis_3P_wH.reg"                   , FALSE ,  1},
        {eSubType_GMP_IVI_PIVI_Nis_3P_wH_ATB    , "GMP_IVI_PIVI_Nis_3P_wH_ATB"  ,"KDS-entry has value GMP_IVI_PIVI_Nis_3P_wH_ATB.reg"               , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Nis_1P_nH         , "GMP_IVI_SC3_Nis_1P_nH"       ,"KDS-entry has value GMP_IVI_SC3_Nis_1P_nH.reg"                    , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Nis_2P_nH_TCU     , "GMP_IVI_SC3_Nis_2P_nH_TCU"   ,"KDS-entry has value GMP_IVI_SC3_Nis_2P_nH_TCU.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Nis_2P_wH_TCU     , "GMP_IVI_SC3_Nis_2P_wH_TCU"   ,"KDS-entry has value GMP_IVI_SC3_Nis_2P_wH_TCU.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Nis_2P_nH_ETC     , "GMP_IVI_SC3_Nis_2P_nH_ETC"   ,"KDS-entry has value GMP_IVI_SC3_Nis_2P_nH_ETC.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Nis_2P_wH_ETC     , "GMP_IVI_SC3_Nis_2P_wH_ETC"   ,"KDS-entry has value GMP_IVI_SC3_Nis_2P_wH_ETC.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Nis_2P_nH_ATB     , "GMP_IVI_SC3_Nis_2P_nH_ATB"   ,"KDS-entry has value GMP_IVI_SC3_Nis_2P_nH_ATB.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Nis_3P_wH         , "GMP_IVI_SC3_Nis_3P_wH"       ,"KDS-entry has value GMP_IVI_SC3_Nis_3P_wH.reg"                    , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Nis_3P_wH_ATB     , "GMP_IVI_SC3_Nis_3P_wH_ATB"   ,"KDS-entry has value GMP_IVI_SC3_Nis_3P_wH_ATB.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Ren_1P_nH         , "GMP_IVI_SC3_Ren_1P_nH"       ,"KDS-entry has value GMP_IVI_SC3_Ren_1P_nH.reg"                    , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Ren_2P_nH_TCU     , "GMP_IVI_SC3_Ren_2P_nH_TCU"   ,"KDS-entry has value GMP_IVI_SC3_Ren_2P_nH_TCU.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Ren_2P_wH_TCU     , "GMP_IVI_SC3_Ren_2P_wH_TCU"   ,"KDS-entry has value GMP_IVI_SC3_Ren_2P_wH_TCU.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Ren_2P_nH_ETC     , "GMP_IVI_SC3_Ren_2P_nH_ETC"   ,"KDS-entry has value GMP_IVI_SC3_Ren_2P_nH_ETC.reg"                , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Ren_2P_wH_ETC     , "GMP_IVI_SC3_Ren_2P_wH_ETC"   ,"KDS-entry has value GMP_IVI_SC3_Ren_2P_wH_ETC.reg"                , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Ren_2P_nH_ATB     , "GMP_IVI_SC3_Ren_2P_nH_ATB"   ,"KDS-entry has value GMP_IVI_SC3_Ren_2P_nH_ATB"                    , FALSE ,  0},
        {eSubType_GMP_IVI_SC3_Ren_3P_wH         , "GMP_IVI_SC3_Ren_3P_wH"       ,"KDS-entry has value GMP_IVI_SC3_Ren_3P_wH.reg"                    , FALSE ,  1},
        {eSubType_GMP_IVI_SC3_Ren_3P_wH_ATB     , "GMP_IVI_SC3_Ren_3P_wH_ATB"   ,"KDS-entry has value GMP_IVI_SC3_Ren_3P_wH_ATB.reg"                , FALSE ,  1},
};
//--------------------------------------------------------------


/*-----------------------------------------------------------------------------*
 * vInitCustomerValues                                                                 *
 *-----------------------------------------------------------------------------*/
void ConfigurationProduct::vInitCustomerValues()
{

    m_eCustomer                                                                 = eRNAIVI;
    //TODO fill the connectors where the Telematics box will be connected
    //TODO also check whether to send ITC for Telematics box
    /*used to filter for Telematic Boxes used by this customer*/
    tBool bRetVal =FALSE;
    /*Read TelematicsBoxDevice from Datapool(if any)*/

    //internal HUB
    m_uiInternalHubVendorID  = 0x0424;
    m_uiInternalHubProductID = 0x2514;


    trTelematicsBoxDeviceInfo l_uiTelematicsBoxDeviceInfo [iMaxSizeOfArray];
    memset(l_uiTelematicsBoxDeviceInfo, 0, sizeof(l_uiTelematicsBoxDeviceInfo));
    tU8 l_uiNumberOfTelematicBoxesUsed = 0;
    bRetVal = ReadTelematicsBox(l_uiTelematicsBoxDeviceInfo,l_uiNumberOfTelematicBoxesUsed);

    m_uiNumberOfTelematicBoxesUsed                                              = 4;
    //1st Telematic Box - former TCU3GVendorID used in LCN2Kai//1st Telematic Box - former TCU3GVendorID used in LCN2Kai
    m_uiTelematicsBoxDeviceInfo[0].iVendorID                                    = 0x1199;
    m_uiTelematicsBoxDeviceInfo[0].iProductID                                   = 0x68ab;
    m_uiTelematicsBoxDeviceInfo[0].bIsHighSpeedExpected                         = TRUE;
    //2nd Telematic Box - former TCU3GVendorID used in LCN2Kai//2nd Telematic Box - former TCU3GVendorID used in LCN2Kai
    m_uiTelematicsBoxDeviceInfo[1].iVendorID                                    = 0x114F;
    m_uiTelematicsBoxDeviceInfo[1].iProductID                                   = 0x1234;
    m_uiTelematicsBoxDeviceInfo[1].bIsHighSpeedExpected                         = TRUE;

    //3rd Telematic Box - For 2.1 and PIVI (NCG3D-70717) ##A-IVC
    m_uiTelematicsBoxDeviceInfo[2].iVendorID                                    = 0x05C6;
    m_uiTelematicsBoxDeviceInfo[2].iProductID                                   = 0x9025;
    m_uiTelematicsBoxDeviceInfo[2].bIsHighSpeedExpected                         = TRUE;
    //4th Telematic Box - For 2.1 and PIVI (NCG3D-70717) ##A-IVC SW7
    m_uiTelematicsBoxDeviceInfo[3].iVendorID                                    = 0x1E3A;
    m_uiTelematicsBoxDeviceInfo[3].iProductID                                   = 0x10B4;
    m_uiTelematicsBoxDeviceInfo[3].bIsHighSpeedExpected                         = TRUE;

    if(l_uiNumberOfTelematicBoxesUsed && bRetVal)
    {
        for(int i=0; i<l_uiNumberOfTelematicBoxesUsed ;i++)
        {
            m_uiTelematicsBoxDeviceInfo[i+m_uiNumberOfTelematicBoxesUsed].iVendorID = l_uiTelematicsBoxDeviceInfo[i].iVendorID;
            m_uiTelematicsBoxDeviceInfo[i+m_uiNumberOfTelematicBoxesUsed].iProductID = l_uiTelematicsBoxDeviceInfo[i].iProductID;
            m_uiTelematicsBoxDeviceInfo[i+m_uiNumberOfTelematicBoxesUsed].eUsb = l_uiTelematicsBoxDeviceInfo[i].eUsb;
        }
        m_uiNumberOfTelematicBoxesUsed+=l_uiNumberOfTelematicBoxesUsed;
    }
    ETG_TRACE_USR4(("*********************** TelematicsBoxDeviceInfo m_uiTelematicsBoxDeviceInfo***********************"));
    ETG_TRACE_USR4(("TelematicsBoxDeviceInfo in array: Total Number of Telematics Box Devices = %d",m_uiNumberOfTelematicBoxesUsed));
    for(unsigned int i=0 ; i<m_uiNumberOfTelematicBoxesUsed ; i++)
    {
        ETG_TRACE_USR4(("TelematicsBoxDevice in array: m_uiTelematicsBoxDeviceInfo[i].u16VendorID = 0x%x",m_uiTelematicsBoxDeviceInfo[i].iVendorID));
        ETG_TRACE_USR4(("TelematicsBoxDevice in array: m_uiTelematicsBoxDeviceInfo[i].u16ProductID = 0x%x",m_uiTelematicsBoxDeviceInfo[i].iProductID));
        ETG_TRACE_USR4(("TelematicsBoxDevice in array: m_uiTelematicsBoxDeviceInfo[i].bIsHighSpeedExpected = %d",m_uiTelematicsBoxDeviceInfo[i].bIsHighSpeedExpected));
        ETG_TRACE_USR4(("TelematicsBoxDevice in array: m_uiTelematicsBoxDeviceInfo[i].u16USBConnector = %d",m_uiTelematicsBoxDeviceInfo[i].eUsb));
        ETG_TRACE_USR4(("---------------------------------------------------------------------------------------"));
    }
    /*Read ExternalHub from Datapool(if any)*/
    bRetVal = FALSE;
    trExternalHubInfo l_uiExternalHubInfo [iMaxSizeOfArray];
    memset(l_uiExternalHubInfo, 0, sizeof(l_uiExternalHubInfo));
    tU8 l_uiNumberOfExternalHubsUsed = 0;
    bRetVal = ReadExternalHub(l_uiExternalHubInfo, l_uiNumberOfExternalHubsUsed);

    //TODO(KMV5COB):Read the vid/pid of Externalhub from file USB_DEVICES_VID_PID_CONFIG_FILE, instead of doing it this way.
    m_uiNumberOfExternalHubsUsed                                                = 5;
    m_uiExternalHubInfo[0].iVendorID                                            = 0x2996;  //Customer specified VendorID ..NCG3D-166910
    m_uiExternalHubInfo[0].iProductID                                           = 0x015B;  //Customer specified ProductID ..NCG3D-166910
    m_uiExternalHubInfo[0].eUsb                                                 = eUSB2;

    m_uiExternalHubInfo[1].iVendorID                                            = 0x2996;  //Customer specified VendorID ..(NCG3D-166910
    m_uiExternalHubInfo[1].iProductID                                           = 0x013F;  //Customer specified ProductID ..NCG3D-166910
    m_uiExternalHubInfo[1].eUsb                                                 = eUSB2;

    m_uiExternalHubInfo[2].iVendorID                                            = 0x2996;  //Customer specified VendorID ..NCG3D-166910
    m_uiExternalHubInfo[2].iProductID                                           = 0x0112;   //Customer specified ProductID ..NCG3D-166910
    m_uiExternalHubInfo[2].eUsb                                                 = eUSB2;

    m_uiExternalHubInfo[3].iVendorID                                            = 0x0424;  //Customer specified VendorID ..RTC-533645
    m_uiExternalHubInfo[3].iProductID                                           = 0x4913;  //Customer specified ProductID ..RTC-533645
    m_uiExternalHubInfo[3].eUsb                                                 = eUSB2;

    m_uiExternalHubInfo[4].iVendorID                                            = 0x0424;  //Customer specified VendorID ..RTC-533645
    m_uiExternalHubInfo[4].iProductID                                           = 0x4914;  //Customer specified ProductID ..RTC-533645
    m_uiExternalHubInfo[4].eUsb                                                 = eUSB2;


    if(l_uiNumberOfExternalHubsUsed && bRetVal)
    {
        for(int i=0; i<l_uiNumberOfExternalHubsUsed ;i++)
        {
            m_uiExternalHubInfo[i+m_uiNumberOfExternalHubsUsed].iVendorID = l_uiExternalHubInfo[i].iVendorID;
            m_uiExternalHubInfo[i+m_uiNumberOfExternalHubsUsed].iProductID = l_uiExternalHubInfo[i].iProductID;
            m_uiExternalHubInfo[i+m_uiNumberOfExternalHubsUsed].eUsb = l_uiExternalHubInfo[i].eUsb;
        }
        m_uiNumberOfExternalHubsUsed+=l_uiNumberOfExternalHubsUsed;
    }
    ETG_TRACE_USR4(("*********************** ExternalHubInfo m_uiExternalHubInfo***********************"));
    ETG_TRACE_USR4(("ExternalHubInfo in array: Total Number of External Hubs = %d",m_uiNumberOfExternalHubsUsed));
    for(unsigned int i=0 ; i<m_uiNumberOfExternalHubsUsed ; i++)
    {
        ETG_TRACE_USR4(("ExternalHubInfo in array: m_uiExternalHubInfo[i].u16VendorID = %d",m_uiExternalHubInfo[i].iVendorID));
        ETG_TRACE_USR4(("ExternalHubInfo in array: m_uiExternalHubInfo[i].u16ProductID = %d",m_uiExternalHubInfo[i].iProductID));
        ETG_TRACE_USR4(("ExternalHubInfo in array: m_uiExternalHubInfo[i].u16USBConnector = %d",m_uiExternalHubInfo[i].eUsb));
        ETG_TRACE_USR4(("---------------------------------------------------------------------------------------"));
    }

    ifstream usbDeviceInfoFile(USB_DEVICES_VID_PID_CONFIG_FILE);
    if(usbDeviceInfoFile.is_open())
    {
        ETG_TRACE_USR4(("vInitCustomerValues (RNAIVI) Filling usb supported devices vid/pid from file = %s",USB_DEVICES_VID_PID_CONFIG_FILE));
        ReadUsbDeviceInfoFromIstream(usbDeviceInfoFile, externalHubAssociatedDeviceInfo, USB_DEVICETYPE_EXTERNALHUB);
        usbDeviceInfoFile.close();
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR] vInitCustomerValues (RNAIVI): Failed to open usb supported devices vid,pid config file. errno = %d",errno));
    }

    ETG_TRACE_USR4(("*********************** externalHubAssociatedDeviceInfo***********************"));
    ETG_TRACE_USR4(("externalHubAssociatedDeviceInfo in array: Total Number of usb devices = %d",externalHubAssociatedDeviceInfo.size()));
    for(unsigned int i=0 ; i<externalHubAssociatedDeviceInfo.size() ; i++)
    {
        ETG_TRACE_USR4(("externalHubAssociatedDeviceInfo in array: externalHubAssociatedDeviceInfo[i].u16VendorID = %d",externalHubAssociatedDeviceInfo[i].iVendorID));
        ETG_TRACE_USR4(("externalHubAssociatedDeviceInfo in array: externalHubAssociatedDeviceInfo[i].u16ProductID = %d",externalHubAssociatedDeviceInfo[i].iProductID));
        ETG_TRACE_USR4(("---------------------------------------------------------------------------------------"));
    }

    /*enable USB connector */
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB1]                               = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]                               = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB3]                               = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB4]                               = DVM_OFF;
    /*if disabled VD_DVM supporesses notification to clients*/
    m_u16ConfigurationValue[eCONF_ENABLE_USB1]                                  = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_USB2]                                  = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_USB3]                                  = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_USB4]                                  = DVM_OFF;

    /*if enabled HUBs will be observed for open circuit and overcurrent*/
    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB1]                   = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB2]                   = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB3]                   = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB4]                   = DVM_OFF;


#ifdef  MSD_BIGGEST_PARTITON_FEATURE
    //For biggest partition support
    m_u16ConfigurationValue[eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL]            = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_SORT_PARTITIONS_ALL]                    = DVM_ON;
    m_u16ConfigurationValue[eConf_MAXIMUM_SUPPORTED_PARTITIONS_MSD]              = DVM_ALL_PARTITIONS; //No limit in the number of partitions. The configuration number of partitons is applied in mediaplayer
#endif
    m_u16ConfigurationValue[eCONF_ENABLE_GETMSC_FREE_USED_SIZE]                  = DVM_OFF;

    // Enable the tristate logic to check mediastatusbits based on three states for ports tried to be switched on/off from diagnosis
    // Check DiagLibhandlerGen3::vOnIoControl
    m_u16ConfigurationValue[eCONF_ENABLE_TRISTATE_DIAGNOSIS]                     = DVM_ON;
//!!!!!!!!!!!!!!!!!!!!!currently only available for RNAIVI, PSA  has to be enabled for SUZUKI - Short discussion with Nils Bannow 23.03.2016 15:52!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#ifdef VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED
    m_u16ConfigurationValue[eConf_ENABLE_MALFUNCTION_USBCONNECTORS]         = DVM_ON;
    m_u16ConfigurationValue[eConf_ENABLE_MALFUNCTION_CDROMINTERNAL]         = DVM_ON;
    m_u16ConfigurationValue[eCONF_TIMOUT_MALFUNCTION_SET_PERMANENT]         = DVM_MALFUNCTION_MONITOR_INTERVAL_SEC;
    m_u16ConfigurationValue[eCONF_USBCOMPARETIME_MALFUNCTION_OR_REMOVE]     = DVM_COMPARETIME_MALFUNCTION_OR_REMOVE_MSEC; //Tprev - see 15003_HandlingOfTemporaryUnavailableDevices
    m_u16ConfigurationValue[eCONF_ENABLE_HIGHRISK4USB_HWMALFUNCTION]        = DVM_ON; //Tprev - see 15003_HandlingOfTemporaryUnavailableDevices
#endif//VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED

//tmp
    m_u16ConfigurationValue[eCONF_ENABLE_SUPPRESS_APPLE_DEVICE_WITHOUT_CUSTOMERHUB]         = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_CHARGE_APPLEDEVS]                                  = DVM_OFF;//Disable charging, since it will be handled in clients(Mediaplayer)
    m_u16ConfigurationValue[eConf_ENABLE_EXTERNAL_HUB]                                      = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_NOTIFY_CUSTOMER_HUBS]                              = DVM_ON;  //Device(Hub) USB Compliance - Check and update UnsupportedReason based on Customer Hubs or not
    m_u16ConfigurationValue[eCONF_ENABLE_NOTIFY_UNKNOWN_DEVICES]                            = DVM_ON;
//
    m_u16ConfigurationValue[eCONF_ENABLE_MEDIAMANAGER_CLIENTHANDLER]          = DVM_ON;  // Enable Mediamanager clienthandler

    m_u16ConfigurationValue[eCONF_ENABLE_NOTIFY_DEVPATH]                                    = DVM_ON;  // Notify clients of the devicepath(syspath) for connector  CRQ     CMG3G-12660
    //--------------------------------------------------------
    //adjustments for diagnosis and temporary not available/Malfunction
    //-------------------------------------------------------
    //Note: this mapping is currently used to serve Scope1 and will be adjusted by later call of kds dependent vLoadCustomerValuesBySubVersion() - see below
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                       = DVM_PRMPORT2_USED; //For the USB port 1(black port) the hardware signal is provided from PRM2
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                       = DVM_PRMPORT1_USED; //For the USB port 2(black port) the hardware signal is provided from PRM1
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                       = DVM_OFF;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                       = DVM_OFF;

    //port numbering will be based on the below array values
    //caf expects a different port numbering than the other G3g projects
    //see information here for port numbering required by https://hi-cmts.apps.intranet.bosch.com:8443/browse/CAFI06-1533
    //looking at the back of the device
    //USB2: gray   connector ,position left  - thought to be used with USB mass storage devices    | bound IMX Connector: 2184200 |example of syspath: /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:2.2/0003:05AC:12A8.0003/hidraw/hidraw0 (hidraw)
    //USB1: black connector  , position right - thought to be used with Apple devices               | bound IMX Connector: 2184000 |example of syspath:/devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1:2.2/0003:05AC:12A8.0004/hidraw/hidraw0 (hidraw)

#ifdef USE_OLD_GETSYSPATH
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB1]               = "/2184000.usb/";  //IMXConnectorUSB1  //OTG-PORT
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB2]               = "/2184200.usb/";  //IMXConnectorUSB2  //HOST-PORT
#else
    //USB1
    m_rCustomerUSBConnectorConfig[eUSB1].eConnectorInfo = eConnectorRealisedDirect;

#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxOtgConnector];
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = "/2184000.usb/";
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector   = "";
#endif


    //USB2
    m_rCustomerUSBConnectorConfig[eUSB2].eConnectorInfo = eConnectorRealisedDirect;
#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = "/2184200.usb/";
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector   = "";
#endif

#endif //USE_OLD_GETSYSPATH

    //----------------------------------------------------------

#ifndef INTERMEDIATELY_DISABLED_USINGKDSVALUES_INF4CV
    //special order for this reason not added to contructor!
    //=====================PERSISTENT STORAGE========================================================

     //---------------------------------------
    // Read G3G values from KDS (Persistent storage)
    //--------------------------------------
    CommonReadRoutinesKds oCommonReadRoutinesKds;
    oCommonReadRoutinesKds.vInitCustomerValues_G3G_USBExtendedInfo(IN m_pIFConfPersistent, OUT m_u16ConfigurationValue);

    /* Load Configuration values based on the Subvariants*/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
    vLoadCustomerValuesBySubVersion();
#endif

    vInitCustomerValuesPostPersistent(); //has not been put behind vInitCustomerValuesPersistent to keep customer configuration stronger then g3g configuration!!!
                                         //with Scope2 this has to be adjusted values like "eCONF_HWSIGNAL_SOURCE..." & "eCONF_ENABLE_HWInternalHUBRealizesUSB..." have to be set

    //-------------------------------------------------
    // Read Customer specific values from KDS (Persistent storage)
    //-------------------------------------------------
    vInitCustomerValuesPersistent();//tbd.: shift to constructor

    //============================================================================================
#endif



}



/*-----------------------------------------------------------------------------*
 * vInitCustomerValuesPersistent                                                                 *
 *-----------------------------------------------------------------------------*/
//sequence is vInitCustomerValues() calls
//    1. vInitCustomerValues_G3G_USBExtendedInfo()      (for KDS USBExtendedInfo to set eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//    2. vLoadCustomerValuesBySubVersion()                  (for  KDS eSubType_GMP_IVI_SC2_Nis_1P_nH etc. to set variants)
//    3. vInitCustomerValuesPostPersistent()                    (to use eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//->4. vInitCustomerValuesPersistent()                          (care for customer HUB )
void ConfigurationProduct::vInitCustomerValuesPersistent()
{

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
  teSubversionType eSubversion = eSubType_GMP_NisScope1; //mSubVersion;
#else
  std::string strSubversion;
  teSubversionType eSubversion = (teSubversionType)GetSubversion(strSubversion);
#endif
  int iHardwareHasInternalHub = iGetHardwareHasInternalHubFromSubversion(eSubversion);


  /*
    reference see:
        Alliance_IVI_DiagnosticServices.pdf  (https://hi-dms.de.bosch.com/docushare/dsweb/View/Collection-262754)


    KDS-Address: 0xA110
    Identifier: Identifier ($DAB1)
    byte index 29 of 64 byte array - assume index starts with 0
    Bit 2-3


    AllianceMCH1MountType
    [0x00] No MCH1 mounted in car
    [0x01] AUX + 1 USB port (MCH1, w/o HUB)  -1Aux+1Device DVM_ON_2PORTS
    [0x02] AUX + 2 USB ports (MCH1, with HUB) -1Aux+2Devices DVM_ON_3PORTS
    [0x03] AUX + 2 USB ports + SD slot (3rd USB MSC device) (MCH1, with HUB -1Aux+2Devices DVM_ON_3PORTS
*/


    tU8 l_u8Array[KDS_MAX_ENTRY_LENGTH];
    tU8 l_u8PayloadByte29;
    tU8 l_Byte29_Bit2And3 = 0;
    tBool bFound = TRUE;

    if(m_pIFConfPersistent)
    {
        if(TRUE==m_pIFConfPersistent->bReadKDS(0xA110      /*address*/,
                                               64          /*len of bytes to read - we start to count with 0 as first byte  */,
                                               l_u8Array))
        {
            l_u8PayloadByte29   = l_u8Array[29]; //byte index 29
            l_Byte29_Bit2And3 = (tU8)(((l_u8PayloadByte29) & 0x0C)>>2);/*Mask 00001100=0x0C =  3*/


            ETG_TRACE_COMP(("vInitCustomerValues(INF4CV): KDSvalue 0xA110 read,    l_u8PayloadByte29 :  0x%x", l_u8PayloadByte29));
            ETG_TRACE_COMP(("vInitCustomerValues(INF4CV):                          l_Byte29_Bit2And3 :  0x%x", l_Byte29_Bit2And3));

            switch(l_Byte29_Bit2And3)
            {
                case 0x00:
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_HUBandPORTSatUSB1 and eCONF_ENABLE_HUBandPORTSatUSB2  DVM_OFF (for external hub support"));
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED DVM_OFF (for external hub support"));

                    //in theory we would have a
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]                     = DVM_OFF;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]                     = DVM_OFF;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_OFF;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;

                    //no HUB connected- hence it makes no sense to send ITC_USB2_HUB_NOT_DETECTED
                    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB1]                  = DVM_OFF; //Black connector is currently bound to USB2 since mapping feature has not been configured to be used
                    m_u16ConfigurationValue[eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED]            = DVM_OFF;//To send LINK Disabled ITC if the configured hub details are not present.
                    if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBPermanent])
                    {
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4]   = DVM_OFF;
                    }
                    break;
                case 0x01:
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_HUBandPORTSatUSB1 and eCONF_ENABLE_HUBandPORTSatUSB2  DVM_ON_2PORTS (for external hub support"));
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED DVM_OFF (for external hub support"));
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]                     = DVM_ON_2PORTS; //it is not a real HUB but a cable so a ticket might arise for this
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]                     = DVM_ON_2PORTS; //t is not a real HUB but a cable so a ticket might arise for this
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_OFF;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;

                     //new subvariants give info if internal HUB is used
                    if(iHardwareHasInternalHub == 1)
                    {
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_ON_2PORTS;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;
                    }

                    //this is not a real HUB so it won't be detected as HUB at all - hence it makes no sense to send ITC_USB2_HUB_NOT_DETECTED
                    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB1]                  = DVM_OFF; //Black connector is currently bound to USB2 since mapping feature has not been configured to be used
                    m_u16ConfigurationValue[eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED]            = DVM_OFF;//To send LINK Disabled ITC if the configured hub details are not present.
                    if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBPermanent])
                    {
                        /*RTC-932208 : supress notfication of devices connected to a HUB which is connected to another hub always*/
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4]   = DVM_OFF;

                         //new subvariants give info if internal HUB is used
                        if(iHardwareHasInternalHub == 1)
                        {
                            m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_ON_2PORTS;
                            m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;
                        }
                    }
                    break;
                case 0x02:
                case 0x03:
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_HUBandPORTSatUSB1 and eCONF_ENABLE_HUBandPORTSatUSB2  DVM_ON_3PORTS (for external hub support"));
                    ETG_TRACE_USR4(("ConfigurationProduct (INF4CV): eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED DVM_ON (for external hub support"));
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]                     = DVM_ON_3PORTS;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]                     = DVM_ON_3PORTS;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_OFF;
                    m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;

                    //new subvariants give info if internal HUB is used
                    if(iHardwareHasInternalHub == 1)
                    {
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_ON_3PORTS;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;
                    }

                    //this is a real HUB  - hence it makes sense to send ITC_USB2_HUB_NOT_DETECTED if it is not connected
                    //------------------------------------------------------------------------------
                    //in INF4CV we expect customer HUB to b connected only to IMX_HOST_PORT i.e.USB2
                    //------------------------------------------------------------------------------
                    m_u16ConfigurationValue[eCONF_ENABLE_observeDTC_HUBatUSB2]                  = DVM_ON;

                    m_u16ConfigurationValue[eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED]            = DVM_ON;//To send LINK Disabled ITC if the configured hub details are not present.

                    if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBPermanent])
                    {
                        /*RTC-932208 : supress notfication of devices connected to a HUB which is connected to another hub always*/
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3]   = DVM_OFF;
                        m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4]   = DVM_OFF;

                         //new subvariants give info if internal HUB is used
                        if(iHardwareHasInternalHub == 1)
                        {
                            m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_ON_3PORTS;
                            m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;
                        }
                    }
                    break;
                //Note: default: can't happen since we have only 2 Bits - all 4 possibilities are mentioned above already
                default:
                    bFound = FALSE;
                    ETG_TRACE_FATAL(("[ERROR] vInitCustomerValues (INF4CV):Unexpected value in KDS for Addr.: 0xA110 l_Byte29_Bit2And3 : 0x%x", l_Byte29_Bit2And3));
                    break;
            }




        }
        else
        {
            //@todo change charge apple to use
            bFound = FALSE;
            ETG_TRACE_FATAL(("vInitCustomerValues(INF4CV): failed to Read configuration 0xA110 from KDS"));
            ETG_TRACE_ERRMEM(("vInitCustomerValues(INF4CV): failed to Read configuration 0xA110 from KDS"));
        }


    }
    else
    {
        bFound = FALSE;
        ETG_TRACE_FATAL(("vInitCustomerValues(INF4CV):m_pIFConfPersistent = NULL"));
    }

    if(FALSE == bFound)
    {
        ETG_TRACE_FATAL(("vInitCustomerValues(INF4CV): failed to Read configuration 0xA110 from KDS so setting default  configuration  for external hubs"));
        /*if disabled VD_DVM supresses notfication of devices connected to a HUB*/
        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]                     = DVM_ON_1PORT;
        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]                     = DVM_ON_1PORT;
        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB3]                     = DVM_OFF;
        m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB4]                     = DVM_OFF;

        //if the eCONF_ENABLE_HUBconnectedToHUBPermanent internal config value is enabled the project dependent configuration will  not be taken into account
        if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBPermanent])
        {
            /*RTC-932208 : supress notfication of devices connected to a HUB which is connected to another hub always*/
            m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1]   = DVM_OFF;
            m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2]   = DVM_OFF;
            m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3]   = DVM_OFF;
            m_u16ConfigurationValue[eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4]   = DVM_OFF;
        }

    }

     //===============================================================================================================================================
     //RTC-497260(INF4CV) :This part has been added to find out if Apple devices are supported or not based on MFICertified KDS value

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
     tU8 isMFiCertified = 0;
     tS32 s32Res = DP_s32GetConfigItem("AIVIVariantCoding", "MFICertified", &isMFiCertified, sizeof(tU8));

     if ( s32Res == DP_S32_NO_ERR )
     {
         ETG_TRACE_COMP(("=========================================================================================="));
         ETG_TRACE_COMP(("vInitCustomerValuesPersistent_AIVIVariantCoding_MFICertified(INF4CV):%d ", isMFiCertified));
         ETG_TRACE_COMP(("=========================================================================================="));
         if(!isMFiCertified)//If the variant is not MFiCertified, disable apple device support
         {
             ETG_TRACE_USR4(("vInitCustomerValuesPersistent_AIVIVariantCoding_MFICertified (INF4CV): Disable apple device support"));
             vEnableAppleDeviceSupport(FALSE);//Disable apple device support
         }
     }
     else
     {
         ETG_TRACE_FATAL(("[ERROR] vInitCustomerValuesPersistent_GetMFiCertificationStatus: Could not read MFICertified with 'DP_s32GetConfigItem'"))
     }
#endif

}

/*-----------------------------------------------------------------------------*
 * vInitCustomerValuesPostPersistent                                                                 *
 *-----------------------------------------------------------------------------*/
//sequence is vInitCustomerValues() calls
//    1. vInitCustomerValues_G3G_USBExtendedInfo()      (for KDS USBExtendedInfo to set eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//    2. vLoadCustomerValuesBySubVersion()                  (for  KDS eSubType_GMP_IVI_SC2_Nis_1P_nH etc. to set variants)
//->3. vInitCustomerValuesPostPersistent()                    (to use eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//    4. vInitCustomerValuesPersistent()                          (care for customer HUB )

void ConfigurationProduct::vInitCustomerValuesPostPersistent()
{

    //tbd.:  up till now there is no mapping implemented. Waiting for further Feedback from HW collegues. I.e. Behaviour is

    //=============================================================
    //set these values in dependency of:
    //m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB1]
    //m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB2]
    // &
    // m_u16ConfigurationValue[eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB]
    //m_u16ConfigurationValue[eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB]
    // gained from kds
    //=============================================================

    //-------------------------------
    //IMX1
    //-------------------------------
    if(DVM_ON == m_u16ConfigurationValue[eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB])   //means USB at IMX-with addr in syspath: 2184200
    {
         m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]         = DVM_ON_4PORTS;
    }
    else if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB])
    {
         m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB1]         = DVM_OFF;
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: Unexpected value vInitCustomerValuesPostPersistent: m_u16ConfigurationValue[eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB]:%d",m_u16ConfigurationValue[eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB]));
    }

    //-------------------------------
    //IMX2
    //-------------------------------
    if(DVM_ON == m_u16ConfigurationValue[eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB] )   //means USB at IMX-with addr in syspath: 2184000
    {
         m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]         = DVM_ON_4PORTS;
    }
    else if(DVM_OFF == m_u16ConfigurationValue[eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB])
    {
         m_u16ConfigurationValue[eCONF_ENABLE_HUBandPORTSatUSB2]         = DVM_OFF;
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR]: Unexpected value vInitCustomerValuesPostPersistent: m_u16ConfigurationValue[eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB]:%d",m_u16ConfigurationValue[eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB]));
    }
}

/*-----------------------------------------------------------------------------*
 * vLoadCustomerValuesBySubVersion                                                                 *
 *-----------------------------------------------------------------------------*/
 //sequence is vInitCustomerValues() calls
//    1. vInitCustomerValues_G3G_USBExtendedInfo()      (for KDS USBExtendedInfo to set eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//->2. vLoadCustomerValuesBySubVersion()                  (for  KDS eSubType_GMP_IVI_SC2_Nis_1P_nH etc. to set variants)
//    3. vInitCustomerValuesPostPersistent()                    (to use eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB & eCONF_ENABLE_IMX_OTG__USB_2184200_ALLOWHUB)
//    4. vInitCustomerValuesPersistent()                          (care for customer HUB )
void ConfigurationProduct::vLoadCustomerValuesBySubVersion()
{
    std::string strSubversion;
    mSubVersion = (teSubversionType)GetSubversion(OUT strSubversion);

    switch(mSubVersion)
    {

        case eSubType_GMP_NisScope1:
            vLoadCustomerValues_SubType_NisScope1();
            break;
        case eSubType_GMP_NisScope2_1RENAULT: //project: planned not to be called anymore
            vLoadCustomerValues_SubType_NisScope2_1RENAULT();
            break;
        case eSubType_GMP_NIS_PIVI: //project: planned not to be called anymore
            vLoadCustomerValues_SubType_NIS_PIVI();
            break;
        case eSubType_GMP_IVI_SC2_Nis_1P_nH:
        case eSubType_GMP_IVI_SC2_1_Nis_1P_nH:
        case eSubType_GMP_IVI_SC2_1_Ren_1P_nH:
        case eSubType_GMP_IVI_SC3_Nis_1P_nH:
        case eSubType_GMP_IVI_SC3_Ren_1P_nH:
        case eSubType_GMP_IVI_PIVI_Nis_1P_nH:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/1,/*internal Hub*/0, /*TCU/BTA*/0,/*ETC/DSRN*/0,/*AT-Box*/0);
            break;
        case eSubType_GMP_NisScope2:       //Decided in Meeting with Thomas Beul 15/05/2017 10:30 a.m. at place koe2hi - in future will be replaced by other subtypes
        case eSubType_GMP_NisScope2_1AIVI: //same like eSubType_GMP_NisScope2
        case eSubType_GMP_IVI_SC2_Nis_2P_nH_TCU:
        case eSubType_GMP_IVI_SC2_1_Nis_2P_nH_TCU:
        case eSubType_GMP_IVI_SC2_1_Ren_2P_nH_TCU:
        case eSubType_GMP_IVI_SC3_Nis_2P_nH_TCU:
        case eSubType_GMP_IVI_SC3_Ren_2P_nH_TCU:
        case eSubType_GMP_IVI_PIVI_Nis_2P_nH_TCU:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/0, /*TCU/BTA*/1,/*ETC/DSRN*/0,/*AT-Box*/0);
            break;
        case eSubType_GMP_IVI_SC2_Nis_2P_wH_TCU:
        case eSubType_GMP_IVI_SC2_1_Nis_2P_wH_TCU:
        case eSubType_GMP_IVI_SC2_1_Ren_2P_wH_TCU:
        case eSubType_GMP_IVI_SC3_Nis_2P_wH_TCU:
        case eSubType_GMP_IVI_SC3_Ren_2P_wH_TCU:
        case eSubType_GMP_IVI_PIVI_Nis_2P_wH_TCU:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/1, /*TCU/BTA*/1,/*ETC/DSRN*/0,/*AT-Box*/0);
            break;
        case eSubType_GMP_IVI_SC2_Nis_2P_nH_ETC:
        case eSubType_GMP_IVI_SC2_1_Nis_2P_nH_ETC:
        case eSubType_GMP_IVI_SC2_1_Ren_2P_nH_ETC:
        case eSubType_GMP_IVI_SC3_Nis_2P_nH_ETC:
        case eSubType_GMP_IVI_SC3_Ren_2P_nH_ETC:
        case eSubType_GMP_IVI_PIVI_Nis_2P_nH_ETC:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/0, /*TCU/BTA*/0,/*ETC/DSRN*/1,/*AT-Box*/0);
            break;
        case eSubType_GMP_IVI_SC2_Nis_2P_wH_ETC:
        case eSubType_GMP_IVI_SC2_1_Nis_2P_wH_ETC:
        case eSubType_GMP_IVI_SC2_1_Ren_2P_wH_ETC:
        case eSubType_GMP_IVI_SC3_Nis_2P_wH_ETC:
        case eSubType_GMP_IVI_SC3_Ren_2P_wH_ETC:
        case eSubType_GMP_IVI_PIVI_Nis_2P_wH_ETC:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/1, /*TCU/BTA*/0,/*ETC/DSRN*/1,/*AT-Box*/0);
            break;
        case eSubType_GMP_IVI_SC2_1_Nis_2P_nH_ATB://todo(kmv5cob): Add special handling for AT Box support
        case eSubType_GMP_IVI_SC2_1_Ren_2P_nH_ATB:
        case eSubType_GMP_IVI_SC3_Nis_2P_nH_ATB:
        case eSubType_GMP_IVI_SC3_Ren_2P_nH_ATB:
        case eSubType_GMP_IVI_PIVI_Nis_2P_nH_ATB:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/0, /*TCU/BTA*/0,/*ETC/DSRN*/0,/*AT-Box*/1);
            break;
        case eSubType_GMP_IVI_SC2_Nis_3P_wH:
        case eSubType_GMP_IVI_SC2_1_Nis_3P_wH:
        case eSubType_GMP_IVI_SC2_1_Ren_3P_wH:
        case eSubType_GMP_IVI_SC3_Nis_3P_wH:
        case eSubType_GMP_IVI_SC3_Ren_3P_wH:
        case eSubType_GMP_IVI_PIVI_Nis_3P_wH:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/3,/*internal Hub*/1, /*TCU/BTA*/1,/*ETC/DSRN*/1,/*AT-Box*/0);//special case 3Port devices support TCU and ETC
           break;
        case eSubType_GMP_IVI_SC2_1_Nis_3P_wH_ATB://todo(kmv5cob): Add special handling for AT Box support
        case eSubType_GMP_IVI_SC2_1_Ren_3P_wH_ATB:
        case eSubType_GMP_IVI_SC3_Nis_3P_wH_ATB:
        case eSubType_GMP_IVI_SC3_Ren_3P_wH_ATB:
        case eSubType_GMP_IVI_PIVI_Nis_3P_wH_ATB:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/3,/*internal Hub*/1, /*TCU/BTA*/1,/*ETC/DSRN*/0,/*AT-Box*/1);
            break;
        case eSubType_GMP_Undefined:
            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - project has to write KDS configuration for VD_DeviceManager!"));
            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - project has to write KDS configuration for VD_DeviceManager! "));

            unsigned int numOfPorts;
            unsigned int withInternalHub;
            unsigned int withTelematicsBox;
            unsigned int withEletronicTollCollectBox;

            vUseGenericDefaultSetting(IN strSubversion,OUT numOfPorts,OUT withInternalHub,OUT withTelematicsBox,OUT withEletronicTollCollectBox);//TODO(kmv5cob):Extend this function when ATBox support is provided

            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - Use Default  derived from unknown generic kds entry"));
            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - numOfPorts                 : %d" ,numOfPorts));
            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withInternalHub            : %d" ,withInternalHub));
            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withTelematicsBox          : %d" ,withTelematicsBox));
            ETG_TRACE_ERRMEM(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withEletronicTollCollectBox: %d" ,withEletronicTollCollectBox));

            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - Use Default: derived from unknown generic kds entry"));
            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - numOfPorts                 : %d" ,numOfPorts));
            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withInternalHub            : %d" ,withInternalHub));
            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withTelematicsBox          : %d" ,withTelematicsBox));
            ETG_TRACE_FATAL(("vLoadCustomerValuesBySubVersion: eSubType_GMP_Undefined - withEletronicTollCollectBox: %d" ,withEletronicTollCollectBox));

            vLoadCustomerValues_SubType_Generic(numOfPorts,withInternalHub, withTelematicsBox,withEletronicTollCollectBox,/*AT-Box*/0);

            break;
        case eSubTypeLastElem:
        default:
            vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/0, /*TCU/BTA*/0,/*ETC/DSRN*/0,/*AT-Box*/0);
            break;
    }
}

/*-----------------------------------------------------------------------------*
 * GetHardwareHasInternalHubFromSubversion                                                                 *
 *-----------------------------------------------------------------------------*/
int  ConfigurationProduct::iGetHardwareHasInternalHubFromSubversion(teSubversionType eSubversion)
{
    int iRes = -1;

    ETG_TRACE_USR4(("GetHardwareHasInternalHubFromSubversion(%d)",eSubversion));
    iRes = m_rSubVersionValueDescriptions[eSubversion].iHardwareHasInternalHUB;

    return iRes;
}



void  ConfigurationProduct::vUseGenericDefaultSetting(IN std::string strSubversion,
                                                          OUT unsigned int &numOfPorts,
                                                          OUT unsigned int &withInternalHub,
                                                          OUT unsigned int &withTelematicsBox,
                                                          OUT unsigned int &withEletronicTollCollectBox)
{
    tBool bRet=TRUE;

    ETG_TRACE_USR4(("strSubversion(%s)",strSubversion.c_str()));


      /*
        Possible values in string taken from m_rSubVersionValueDescriptions

        numOfPorts:                  _1P, _2P, _3P
        withInternalHub;             _wH, _nH
        withTelematicsBox;           _TCU
        withEletronicTollCollectBox; _ETC
        */

    //set numOfPorts
    if (bRet == TRUE)
    {
        if(strSubversion.find("_1P") != string::npos)
        {
           numOfPorts=1;
        }
        else if (strSubversion.find("_2P") != string::npos)
        {
           numOfPorts=2;
        }
        else if (strSubversion.find("_3P") != string::npos)
        {
           numOfPorts=3;
        }
        else
        {
            bRet=FALSE;
        }
    }

    //set internal HUB
    if (bRet == TRUE)
    {
        if(strSubversion.find("_nH") != string::npos)
        {
           withInternalHub=0;
        }
        else if (strSubversion.find("_wH") != string::npos)
        {
           withInternalHub=1;
        }
        else
        {
            bRet=FALSE;
        }
    }

    // set TCU
    if (bRet == TRUE)
    {
        if(strSubversion.find("_TCU") != string::npos)
        {
           withTelematicsBox=1;
        }
        else
        {
            withTelematicsBox=0;
        }
    }

    // set ETC
    if (bRet == TRUE)
    {
        if(strSubversion.find("_ETC") != string::npos)
        {
           withEletronicTollCollectBox=1;
        }
        else
        {
            withEletronicTollCollectBox=0;
        }
    }


    if(bRet == TRUE) //special case 3P always supports TCU and ETC
    {
        if(numOfPorts==3) //poossible with internal HUB only
        {
            withTelematicsBox=1;
            withEletronicTollCollectBox=1;
        }
    }

    if(bRet==FALSE)
    {
        ETG_TRACE_USR4(("COULD NOT DERIVE INFO FROM SUBSTRING - USE FINAL DEFAULT"));
        numOfPorts=2;
        withInternalHub=0;
        withTelematicsBox=0;
        withEletronicTollCollectBox=0;
    }

    ETG_TRACE_USR4(("numOfPorts                 : %d",numOfPorts));
    ETG_TRACE_USR4(("withInternalHub            : %d",withInternalHub));
    ETG_TRACE_USR4(("withTelematicsBox          : %d",withTelematicsBox));
    ETG_TRACE_USR4(("withEletronicTollCollectBox: %d",withEletronicTollCollectBox));

    return;
}



/*-----------------------------------------------------------------------------*
 * GetSubversion                                                                 *
 *-----------------------------------------------------------------------------*/
unsigned int  ConfigurationProduct::GetSubversion(std::string &strSubversion)
{
    unsigned int iter;
    unsigned int res = (unsigned int)eSubType_GMP_Undefined;

    strSubversion="ENTRY EMPTY IN KDS";

#define VD_DVM_SUBVERSION_ADDRESS 0xA0B0
#define VD_DVM_SUBVERSION_STRING_START  11 //holds a string registry name :Starting byte
#define VD_DVM_SUBVERSION_STRING_LEN  42 //holds a string registry name  :Ending byte

    tU8 l_u8ArrayTmp[KDS_MAX_ENTRY_LENGTH];
    tU8 l_u8ArraySubversion[KDS_MAX_ENTRY_LENGTH];
    if(m_pIFConfPersistent)
    {
        if(TRUE==m_pIFConfPersistent->bReadKDS(IN VD_DVM_SUBVERSION_ADDRESS, VD_DVM_SUBVERSION_STRING_LEN, l_u8ArrayTmp))
            {
                memcpy(l_u8ArraySubversion,l_u8ArrayTmp+VD_DVM_SUBVERSION_STRING_START,VD_DVM_SUBVERSION_STRING_LEN);
                ETG_TRACE_USR2(("GetSubversion: l_u8ArraySubversion: %s",l_u8ArraySubversion));

                strSubversion=(char*)l_u8ArraySubversion;

                for (iter = eSubType_GMP_Undefined ;iter < eSubTypeLastElem; iter++ )
                {
                    if (!strncmp((const char*)l_u8ArraySubversion, m_rSubVersionValueDescriptions[iter].strBoundRegistryValue.c_str(),sizeof(l_u8ArrayTmp)))
                    {
                        ETG_TRACE_USR2(("GetSubversion: eSubVersion: %d",ETG_CENUM(teSubversionType,iter)));
                        return iter;
                        //break;

                    }
                }
            }

            else
            {
                ETG_TRACE_FATAL (("GetSubversion: failed to read KDS value"));
                ETG_TRACE_ERRMEM(("GetSubversion: failed to read KDS value"));
            }
    }

    return res;

}

/*-----------------------------------------------------------------------------*
 * vLoadCustomerValues_SubType_NisScope1                                                                 *
 *-----------------------------------------------------------------------------*/
void ConfigurationProduct::vLoadCustomerValues_SubType_NisScope1()
{

    /*
     * Has 1 connector:  USB_A(BLACK)
     * USB_A bound to IMX_OTG_CONNECTOR (2184000.usb,usable for carplay)
     * USB_A electrical failure is notified by PRM  at port2 (DVM_PRMPORT2_USED)
     *
     * Set USB1 == USB_A
     */
    ETG_TRACE_FATAL(("vLoadCustomerValues_SubType_NisScope1: Customer values for Nissan Scope1"));


#ifdef USE_WORKAROUND_WRONG_PROJECT_CONFIGURATION
    ETG_TRACE_FATAL(("vLoadCustomerValues_SubType_NisScope1(): Call vLoadCustomerValues_SubType_NisScope2()"));
    vLoadCustomerValues_SubType_Generic(/*numOfPorts:*/2,/*internal Hub*/0, /*TCU/BTA*/1,/*ETC/DSRN*/0,/*AT-Box*/0);
#else //USE_WORKAROUND_WRONG_PROJECT_CONFIGURATION



#ifdef USE_OLD_GETSYSPATH
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB1]               = "/2184000.usb/";  //IMXConnectorUSB1  //OTG-PORT
#else

    //USB1 //Connected direct
    m_rCustomerUSBConnectorConfig[eUSB1].eConnectorInfo                         = eConnectorRealisedDirect;
#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxOtgConnector];
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = "/2184000.usb/";
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector   = "";
#endif
#endif

    /*enable USB connector */
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB1]                               = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]                               = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB3]                               = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB4]                               = DVM_OFF;

    /*if disabled VD_DVM suppresses notification to clients*/
    m_u16ConfigurationValue[eCONF_ENABLE_USB1]                                  = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_USB2]                                  = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_USB3]                                  = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_USB4]                                  = DVM_OFF;

    /*Electrical failure is notified by PRM*/
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                         = DVM_PRMPORT2_USED;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                         = DVM_OFF;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                         = DVM_OFF;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                         = DVM_OFF;

#endif //USE_WORKAROUND_WRONG_PROJECT_CONFIGURATION
}

/*-----------------------------------------------------------------------------*
 * vLoadCustomerValues_SubType_NisScope2_1RENAULT                                                                 *
 *-----------------------------------------------------------------------------*/
void ConfigurationProduct::vLoadCustomerValues_SubType_NisScope2_1RENAULT()
{
    /*
     * Has 3 connector: USB_A(BROWN), USB_B(VIOLET), USB_C(FRONT)
     * //One connector is not used ? needs to be checked with hardware team\\
     * USB_A bound to IMX_HOST_CONNECTOR (2184200.usb, 2-1)
     * USB_B not connected
     * USB_C bound to IMX_OTG_CONNECTOR (2184000.usb, 1-1)
     *
     * USB_A electrical failure is notified by PRM  at port21(DVM_PRMPORT1_USED)
     * USB_B Not connected internally //Clarification from HW needed\\
     * USB_C electrical failure is notified by PRM  at port11(DVM_PRMPORT1_USED)
     *
     * Set USB1 == USB_C
     * Set USB2 == USB_A
     */

    ETG_TRACE_FATAL(("vLoadCustomerValues_SubType_NisScope2_1RENAULT: Customer values for Nissan Scope2.1 Renault"));

#ifdef USE_OLD_GETSYSPATH
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB1] = "/2184000.usb/";  //IMXConnectorUSB1      //OTG (name used in platform)
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB2] = "/2184200.usb/";  //IMXConnectorUSB2      //Host without external HUB (name used in platform)
    m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB3] = "IS#NOT#USED#HERE";  //Connector is not used ? needs to be checked with hardware colleagues if an intermediate sample has been given to us or if is planned like this
#else
    //USB1
    m_rCustomerUSBConnectorConfig[eUSB1].eConnectorInfo = eConnectorRealisedDirect;
#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxOtgConnector];  //IMXConnectorUSB1      //OTG (name used in platform)
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector   =  strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = "/2184000.usb/";  //IMXConnectorUSB1      //OTG (name used in platform)
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector   = "";
#endif //USE_SYSPATH_MATCHER

    //USB2
    m_rCustomerUSBConnectorConfig[eUSB2].eConnectorInfo = eConnectorRealisedDirect;
#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB2     //Host without external HUB (name used in platform)
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = "/2184200.usb/";  //IMXConnectorUSB2     //Host without external HUB (name used in platform)
    m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector   = "";
#endif

    //USB3 - not connected //One connector is not used ? needs to be checked with hardware colleagues if an intermediate sample has been given to us or if is planned like this
    m_rCustomerUSBConnectorConfig[eUSB3].eConnectorInfo = eConnectorUnused;
#ifdef USE_SYSPATH_MATCHER
    m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherNotUsedText];  //not connected //One connector is not used ? needs to be checked with hardware colleagues if an intermediate sample has been given to us or if is planned like this
    m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];
#else
    m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortAdressForUSBConnector = "IS#NOT#USED#HERE";  //not connected //One connector is not used ? needs to be checked with hardware colleagues if an intermediate sample has been given to us or if is planned like this
    m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortHubForUSBConnector    = "";
#endif

#endif

    /*enable USB connector */
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB1]                               = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]                               = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB3]                               = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB4]                               = DVM_OFF;

    /*if disabled VD_DVM suppresses notification to clients*/
    m_u16ConfigurationValue[eCONF_ENABLE_USB1]                                  = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_USB2]                                  = DVM_ON;
    m_u16ConfigurationValue[eCONF_ENABLE_USB3]                                  = DVM_OFF;
    m_u16ConfigurationValue[eCONF_ENABLE_USB4]                                  = DVM_OFF;

    /*Electrical failure is notified by PRM*/
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1]                         = DVM_PRMPORT1_USED;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2]                         = DVM_PRMPORT1_USED;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3]                         = DVM_OFF;
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4]                         = DVM_OFF;
}

/*-----------------------------------------------------------------------------*
 * vLoadCustomerValues_SubType_NIS_PIVI                                                                 *
 *-----------------------------------------------------------------------------*/
void ConfigurationProduct::vLoadCustomerValues_SubType_NIS_PIVI()
{
    /*
       Project support needed
    */

    ETG_TRACE_FATAL(("vLoadCustomerValues_SubType_NIS_PIVI: Customer values for Nissan PIVI"));
    //Config values to be added
    /*Asked for project support*/

    ///////////Temporary/////////////////////////////////////////////////////////
    /*Call vLoadCustomerValues_SubType_NisScope2_1RENAULT() for the interim*/
    ETG_TRACE_FATAL(("vLoadCustomerValues_SubType_NIS_PIVI: Call vLoadCustomerValues_SubType_NisScope2_1RENAULT() for interim"));
    vLoadCustomerValues_SubType_NisScope2_1RENAULT();
    ////////////////////////////////////////////////////////////////////////////
}

//koe2hi: 02.03.2016
//
//used scope 0 hardware with software AI_PRJ_RN_AIVI_16.1V09 to test
//looking at the back of the device
//(in putty used /bin/udevadm monitor)
//
//=================
//Scope0-Hardware
//=================
//USBx:black   connector, position right: thought to be used with Apple devices  (host mode for carplay)  | bound IMX Connector: 2184000 | : example syspath: add      /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1:1.0/host2/target2:0:0/2:0:0:0/block/sda
//USBx:green  connector, position middle: no output!!
//USBx:brown connector, position left:                                                                                                         | bound IMX Connector: 2184200 |      /devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1:1.0/host4/target4:0:0/4:0:0:0/block/sda/sda1 (block)

//=> this mapping is used - see ConfigurationBase.cpp
//  m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB1] = "/2184200.usb/";  //IMXConnectorUSB1
// m_strSysPathEntryUSBPortAdressForUSBConnector[(tUInt)eUSB2] = "/2184000.usb/";  //IMXConnectorUSB2
// devices connected to the black connector a named as USB2 currently - which is not perfect for Scope1 HArdware with a single connector only
//Note: Otherwhise vGetUSBConnectorFromSysPath  would have been to provided for the project - see PSA-project


//How about overcurrent messages from PRM:
//scenario: short circuit between plus and GND line of black USBConnector:
//
//prm informs about overcurrent at USB2 which maps ok OC=SIG_TRUE means overcurrent/failure of USB
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[ok] ===============================->PrmManager=============================== [PrmHistory.cpp(129)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :>>>>>>>>>>>>>>>>>>>>>>>>/PRM: fctCallbackHWSignals [1]>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> [PrmHistory.cpp(144)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  m_uiSendNo:           1 [PrmHistory.cpp(146)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  m_sendTime:           904991 ms [PrmHistory.cpp(148)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8PortNr: eUSB2 (2)                                    [PrmHistory.cpp(150)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8OC:     SIG_TRUE (2)  [StartTime=904990   EndTime=0]   [PrmHistory.cpp(152)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8UV:     SIG_UNDEF (0)  [StartTime=0   EndTime=0]   [PrmHistory.cpp(155)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8PPON:   SIG_FALSE (1)  [StartTime=0   EndTime=904990]   [PrmHistory.cpp(158)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :<<<<<<<<<<<<<<<<<<<<<<<<<PRM: fctCallbackHWSignals/<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< [PrmHistory.cpp(161)]

//scenario: short circuit between plus and GND line of brown USBConnector:
//
//prm informs about overcurrent at USB1 which maps ok OC=SIG_TRUE means overcurrent/failure of USB
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :>>>>>>>>>>>>>>>>>>>>>>>>/PRM: fctCallbackHWSignals [1]>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> [PrmHistory.cpp(144)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  m_uiSendNo:           1 [PrmHistory.cpp(146)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  m_sendTime:           1242560 ms [PrmHistory.cpp(148)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8PortNr: eUSB1 (1)                                    [PrmHistory.cpp(150)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8OC:     SIG_TRUE (2)  [StartTime=1242560   EndTime=0]   [PrmHistory.cpp(152)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8UV:     SIG_UNDEF (0)  [StartTime=0   EndTime=0]   [PrmHistory.cpp(155)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :[1]  rPortState->u8PPON:   SIG_FALSE (1)  [StartTime=0   EndTime=1242560]   [PrmHistory.cpp(158)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_HSTR_FAT :<<<<<<<<<<<<<<<<<<<<<<<<<PRM: fctCallbackHWSignals/<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< [PrmHistory.cpp(161)]

//=================
// Scope1 Hardware
//=================
//see: https://hi-dms.de.bosch.com/docushare/dsweb/View/Collection-309409

//=================
//Scope2 Hardware
//=================
//see: https://hi-dms.de.bosch.com/docushare/dsweb/View/Collection-309409


void ConfigurationProduct::vLoadCustomerValues_SubType_Generic(unsigned int numOfPorts, unsigned int withInternalHub,unsigned int withTelematicsBox,unsigned int withEletronicTollCollectBox,unsigned int withATBox)
{

    (void)withATBox;
    ETG_TRACE_COMP(("vLoadCustomerValues_SubType_Generic: numOfPorts: %d, withInternalHub:0x%x, withTelematicsBox:0x%x, withEletronicTollCollectBox:%d",numOfPorts,withInternalHub,withTelematicsBox,withEletronicTollCollectBox));

    //Error behaviour  numOfPorts: Max
    if(numOfPorts >= (unsigned int)eUSBAll)
    {
        ETG_TRACE_FATAL(("[ERROR]: vLoadCustomerValues_SubType_Generic: numOfPorts (%d) >= eUSBAll (%d) adjust it to max eUSBAll-1)",numOfPorts,(unsigned int)eUSBAll));
        numOfPorts = ((unsigned int)eUSBAll) - 1;
    }

    if(withTelematicsBox > 0)
    {
        m_u16ConfigurationValue[eConf_ENABLE_EXTERNAL_BTA] = DVM_ON;

            ETG_TRACE_COMP(("vLoadCustomerValues_SubType_Generic: eConf_ENABLE_EXTERNAL_BTA set due to kds (VARIANT_S_FTR_ENABLE_DEVMGR_EXTERNAL_BTA_DEVICE not set)"));
            m_u16ConfigurationValue[eCONF_ENABLE_TELEMATICSBOX_MONITORING]            = DVM_TELEMATICSBOX_MONITOR_INTERVAL_SEC;

            //Disable this config. Because for inf4cv USB2 is MCH port and the config is not required.
            //m_u16ConfigurationValue[eConf_ENABLE_DTC_MAP_TCU_OC_PORT3]                = DVM_ON;

        }
    else
    {
       m_u16ConfigurationValue[eConf_ENABLE_EXTERNAL_BTA] = DVM_OFF;
    }
    if(withEletronicTollCollectBox > 0)
    {
        ETG_TRACE_COMP(("vLoadCustomerValues_SubType_Generic:withEletronicTollCollectBox - nothing defined yet to be set"));
        //m_u16ConfigurationValue[eConf_ENABLE_DTC_MAP_ETC_OC_PORT2]                    = DVM_ON; //prepared for future already
    }

    //--------------
    //USB1 - direct (IMX_OTG_USB always USB1, this port is used for carplay)
    //--------------
    m_u16ConfigurationValue[eCONF_ENABLE_HW_USB1]       = DVM_ON;            /*enable USB connector */
    m_u16ConfigurationValue[eCONF_ENABLE_USB1]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
    m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB1] = DVM_PRMPORT1_USED; /*Electrical failure is notified by PRM*/

    m_rCustomerUSBConnectorConfig[eUSB1].eConnectorInfo = eConnectorRealisedDirect;
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxOtgConnector];  //IMXConnectorUSB1      //OTG (name used in platform)
    m_rCustomerUSBConnectorConfig[eUSB1].m_strSysPathEntryUSBPortHubForUSBConnector   =  strMatchers[eMatcherEmpty]; //no internal HUB used

    if(numOfPorts == 2)
    {
        if(withInternalHub == 0)
        {
            //-------------
            //USB2 - direct
            //-------------
            m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]       = DVM_ON;            /*enable USB connector */
            m_u16ConfigurationValue[eCONF_ENABLE_USB2]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
            m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2] = DVM_PRMPORT2_USED; /*Electrical failure is notified by PRM*/

            m_rCustomerUSBConnectorConfig[eUSB2].eConnectorInfo = eConnectorRealisedDirect;
            m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB2     //Host without external HUB (name used in platform)
            m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherEmpty];//no internal HUB used
        }
        else
        {
            //----------------------------
            //USB2  - Connected Internal Hub
            //----------------------------
            m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]       = DVM_ON;            /*enable USB connector */
            m_u16ConfigurationValue[eCONF_ENABLE_USB2]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
            m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2] = DVM_PRMPORT2_USED; /*Electrical failure is notified by PRM - same is used*/

            m_u16ConfigurationValue[eCONF_ENABLE_HWInternalHUBRealizesUSB2]             = DVM_ON; /*enable USB connector connected through internal hub*/
            m_rCustomerUSBConnectorConfig[eUSB2].eConnectorInfo = eConnectorRealisedIntHub;
            m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB2     //Host without external HUB (name used in platform)
            m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherGen3UsbImxHostConnectorIntHubPort1];
        }
    }
    else if(numOfPorts == 3)
    {
        //----------------------------
        //USB2  - Connected Internal Hub
        //----------------------------
        m_u16ConfigurationValue[eCONF_ENABLE_HW_USB2]       = DVM_ON;            /*enable USB connector */
        m_u16ConfigurationValue[eCONF_ENABLE_USB2]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
        m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB2] = DVM_PRMPORT2_USED; /*Electrical failure is notified by PRM - same is used*/

        m_u16ConfigurationValue[eCONF_ENABLE_HWInternalHUBRealizesUSB2]             = DVM_ON; /*enable USB connector connected through internal hub*/
        m_rCustomerUSBConnectorConfig[eUSB2].eConnectorInfo = eConnectorRealisedIntHub;
        m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB2     //Host without external HUB (name used in platform)
        m_rCustomerUSBConnectorConfig[eUSB2].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherGen3UsbImxHostConnectorIntHubPort1];

        //----------------------------
        //USB3  - Connected Internal Hub
        //----------------------------
        m_u16ConfigurationValue[eCONF_ENABLE_HW_USB3]       = DVM_ON;            /*enable USB connector */
        m_u16ConfigurationValue[eCONF_ENABLE_USB3]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
        m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB3] = DVM_PRMPORT2_USED; /*Electrical failure is notified by PRM - same is used*/

        m_u16ConfigurationValue[eCONF_ENABLE_HWInternalHUBRealizesUSB3]             = DVM_ON;/*enable USB connector connected through internal hub*/
        m_rCustomerUSBConnectorConfig[eUSB3].eConnectorInfo = eConnectorRealisedIntHub;
        m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB3     //Host without external HUB (name used in platform)
        m_rCustomerUSBConnectorConfig[eUSB3].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherGen3UsbImxHostConnectorIntHubPort2];


         /* Added USB4 support in RNAIVI despite the hardware having only 3 ports maximum. This is because Kenwood variants has DVD Drive internally connected through usb port of the internal hub on port2.
         * So, this is shown as fourth port in Nissan2.1(Japan) even though there is no fourth port support.
         ***********************************************************************************
         * The syspath address of the DVD drive connected differs between PIVI and Nissan2.1(Japan).
         *      In PIVI:             /sys/devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.1/2-1.1:1.0/host0/target0:0:0/0:0:0:0/block/sr0
         *      In Nissan2.1(Japan): /sys/devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.3/2-1.3:1.0/host1/target1:0:0/1:0:0:0/block/sr0

         * It seems in the Nissan2.1 variant, as per syspath the DVD drive is connected in 3rd port of the internal hub.In PIVI,DVD drive is connected on the 1st port of the internal hub
         * The target has 3 ports(1 OTG, 2 ports internally connected through hubb).
         * Hence For PIVI as per DVM config, in USB2(2-1.1) DVD drive is connected(not visible externally) and USB3(2-1.2) and USB4(2-1.3) are visible externally
         * For Nissan 2.1 as per dvm config, USB2(2-1.1) and USB3(2-1.2) are externally visible and in USB4(2-1.3) DVD_DRIVE is connected(not visible externally)
         * The current strMatchers are inline with Nissan 2.1 where USB2 and USB3 are externally visible and we will not be notifying DTC'S for USB4 where DVD_DRIVE is connected.This is fine.
         * TODO : PIVI also uses the same strMatchers which disables notifying DTC's for USB4 and enables notifying DTC'S for USB2 where DVD_DRIVE is connected. This is not fine, have a different config for PIVI by assigning 2-1.1 to USB4
         * TODO : For scope2.1 Nissan US variants or for variants(Renault...) where DVDDrive is not present, disable USB4.
         ***********************************************************************************
         */
        //----------------------------
        //USB4  - Connected Internal Hub
        //----------------------------
        m_u16ConfigurationValue[eCONF_ENABLE_HW_USB4]       = DVM_ON;            /*enable USB connector */
        m_u16ConfigurationValue[eCONF_ENABLE_USB4]          = DVM_ON;            /*if disabled VD_DVM suppresses notification to clients*/
        m_u16ConfigurationValue[eCONF_HWSIGNAL_SOURCE_USB4] = DVM_PRMPORT2_USED; /*Electrical failure is notified by PRM - same is used*/

        m_u16ConfigurationValue[eCONF_ENABLE_HWInternalHUBRealizesUSB4]             = DVM_ON;/*enable USB connector connected through internal hub*/
        m_rCustomerUSBConnectorConfig[eUSB4].eConnectorInfo = eConnectorRealisedIntHub;
        m_rCustomerUSBConnectorConfig[eUSB4].m_strSysPathEntryUSBPortAdressForUSBConnector = strMatchers[eMatcherGen3UsbImxHostConnector];  //IMXConnectorUSB3     //Host without external HUB (name used in platform)
        m_rCustomerUSBConnectorConfig[eUSB4].m_strSysPathEntryUSBPortHubForUSBConnector    = strMatchers[eMatcherGen3UsbImxHostConnectorIntHubPort3];  //Internal hub port 3

    }
    else if(numOfPorts == 4)
    {
        //tbd.: Extend for 4 Ports
        ETG_TRACE_FATAL(("UNEXPECTED PORTNUMBER: numOfPorts: %d",numOfPorts));
    }
    else
    {
        ETG_TRACE_FATAL(("UNEXPECTED PORTNUMBER: numOfPorts: %d",numOfPorts));
    }


}


////////////////////////////////////////////////////////////////////////////////
// <EOF>


