/*-----------------------------------------------------------------------------*
 * ConfigurationBase.h                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                           *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                   *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file ConfigurationBase.h
 *
 * \brief Implementation of Configuration interface bound to either EOL, KDS, DATAPOOL
 *          Note: if writing and reading does not appear at the same time hence there is no extra sempahore spend
 *          see VD_DeviceManager_WorkThread.cpp
 *
 * \version 06.11.2013, Christian K�chling (Bosch), version 1.0
 *          Initial version

 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef CONFIGURATIONMANAGER_H
#define CONFIGURATIONMANAGER_H

/*-----------------------------------------------------------------------------*
 * includes of configurationManager                                                        *
 *-----------------------------------------------------------------------------*/
#include "config/ConfigurationBase.h"
#include "config/ConfigurationProduct.h"
#include "config/IConfigurationManagerFilter.h"

//used to get time stamps since it is fairly accurate
#include <time.h>       /* clock_t, clock, CLOCKS_PER_SEC */


/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class Info_tclDmEolConfSettings
 */

//@todo after it works rename Info_tclDmEolConfSettings to ConfigurationManager
class ConfigurationManager : public ConfigurationProduct, public IConfigurationManagerFilter   //@todo check if it makes sense to make another child which does care for singleton behaviour only
{
    public:


        /*!
         * \brief Get singleton object
         *
         * \return Info_tclDmEolConfSettings*
         */
        static ConfigurationManager * GetInstance();

        /*!
         * \brief Destroys singleton object
         */
        static void  DestroyInstance();

        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'fc_spmslv_tclClientHandler_Spm' which has
         * no assignment operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
        ConfigurationManager& operator=
                (const ConfigurationManager &oClientHandler);


        /*!
         * \brief Sets the EOL value
         *
         * \param [in] tenDevManagerConfigValue  index of configuration values to be updated
         * \param [in] u16Value   value to be set
         */
        tVoid vSetConfigurationValue(tenDevManagerConfigValue eTypeOfConfigValue,tU16 u16Value);

        /*!
         * \brief Returns the EOL value
         *
         * \param [in]  tenDevManagerConfigValue  index of configuration values to be returned
         *
         * \return tU16 configuration value
         */
        tU16  u16GetConfigurationValue(IN tenDevManagerConfigValue eTypeOfConfigValue) const;


        /*!
         * \brief Returns description for configuration value
         *
         * \param [in] tenDevManagerConfigValue  index of configuration values
         * \param [in] rConfigValueDescription with description about the configuration values
         *
         * \return if description could be returned ok or nok
         */
        tBool bGetConfigurationValueDescription(IN tenDevManagerConfigValue eTypeOfConfigValue, OUT  trConfigValueDescription &rConfigValueDescription);

        /*!
         * \brief Reads the all the required configuration values from class variable and
         *        prints it out to TTFis. This function is triggered via TTFIS
         *
         * \param [in] poDumpFile the file pointer used to trace to USB-stick
         * \param [in] bTest : to prevent that EOL values are printed at each startup.
         *             'fatal' analysis and test mode for easy using without having to know
         *             trace classes
         * \see TraceCmdManager.cpp
         */
        tVoid vPrintConfiguration(FILE *poDumpFile,tBool bTest) const;


         /*!
         * \brief prints traces related to usb connectors. E.g. covers HW-Signal mapping, Hub allowed at certain ports, VBUS-switch on off mapping
         *
         * \see TraceCmdManager.cpp
         */
        tVoid vPrintConfigurationUSBConnectorMapping() const;

         /*!
        * \brief function to check for specific BlockDevNode of inernal CDROM or SDCard
        *        example Internal SDCard is: /dev/mmcblk0p1 in VAGMIBE this function helps to use mmcblk1p1
        *        which might be the internal mmc as SDCard Internal. I.e. clients of VD_DVM should not get internal MMC notified
        *        Note for future alternative: /tmp/.automountner mechanism of adit also provides a blacklist not to notify such devices to VD_DVM
        */
        tBool bCheckHWSpecificBlockDevName(IN CGlobalEnumerations::DEVICE_TYPE_Type eDeviceType, IN std::string strBlockDeviceName) const;


        /*!
        * \brief function to check for specific BlockDevNode of inernal CDROM or SDCard
        *        example Internal SDCard is: /dev/mmcblk0p1 in VAGMIBE this function helps to use mmcblk1p1
        *        which might be the internal mmc as SDCard Internal. I.e. clients of VD_DVM should not get internal MMC notified
        *        Note for future alternative: /tmp/.automoutner mechanism of adit also provides a blacklist not to notify such devices to VD_DVM
        */
        tVoid vGetSpecificBlockDevName(IN CGlobalEnumerations::DEVICE_TYPE_Type eDeviceType, OUT std::string &strBlockDeviceName) const; //Roadmap 14002 MountCDROM (MixeModeCD)

        /*!
        * \brief function to Set the device speed of telematicsbox
        * \param [in] b_DeviceSpeed of Telematics Box, if the parameter is TRUE, then devicemanager expecting high speed device at the usb ports meant for telematics box
        *
        */
        tVoid vSetTelematicsBoxDeviceSpeed(IN tBool b_DeviceSpeed);


        /*!
        * \brief see IConfigurationManagerFilter
        */
        virtual int  FilterWithConfiguration( INOUT CDevice* f_pCurDevice) const;


        /*!
           * \brief Check the given fs is in the list of supported filesystems or not.
           *
           * \param [in] f_cFSType filesystem type
           *
           * \return [int] Returns true if the fs type is in the list of supported
           *         filesystems
           */
         int IsSupportedFS( GENSTRING f_cFSType);


        /*!
        * \brief see IConfigurationManagerFilter
        */

        virtual bool bIsOsalNotifiedDeviceSDCardInternal(IN GENSTRING f_strcUUIDDeviceName) const;


    /*!
        * \brief check if in dependency of the marketing reagion the device name has to be adjusted.
        *
        * has been added to support device names in chinese language 'gpk'
        * \param [in] f_pCurDevice  holds device properties
        *
        */
       void vCheckAdjustDeviceNameForRegion(CDevice* f_pCurDevice) const;


       /*!
            * \brief check if DTY_USB has to be adjusted to be DTY_SD because of existing KeyFile SDCard on USB-stick plus LSIM environment
            *
            * \param [in] f_pCurDevice with mount point and device type - device type is updated DTY_SD if currently DTY_USB and Key file SDCARD available
            */
       void vCheckLSIMSDCardFake(INOUT CDevice *f_pCurDevice) const;

       /*!
            * \brief get USBConnector from syspath if a HUB is connected  e.g. syspath: /sys/devices/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.2
            *          from last entry 2-1.2 especially 1.2 it can be deduced that the device is connected via a HUB
            *
            * \param [in] syspath of device
            * \param [OUT] eUsb. either eUsb1 or eUsb2
            *
            *
            */
       int iGetHUBConnectedFromSysName(OUT bool &f_bConnectedWithHUB, OUT int &f_iPortNum, IN const  GENSTRING &f_strSysName) const;

       /*!
              * \brief show variant switches set in configuration/..._prod.xml
              *
              * \param [in] syspath of device
              * \param [OUT] eUsb. either eUsb1 or eUsb2
              *
              *
              */
       tVoid vShowVariantSwitches(FILE* f_poDumpFile);

       /*!
                  * \brief in configurationProduct_Gen3_<customer>.cpp for project name has to be set this function returns it
                  *
                  * \param [OUT] strProjectName - name of project
                  *
                  *
                  */
       tVoid vGetProjectName(OUT std::string &strProjectName);


       /*!
        * \brief  check if hub connected to hub is allowed as per the configuration. If yes, function dont filter it out.
        * Otherwise this will check the device is connected via a hub which is connected to another hub.
        *
        * \param [in]  f_cSysName: sysname of the device (like 1-1.1)
        * \param [out] f_iDevUsable: enEnabled,enDisabled
        *
        * \return
        */
       void vFilterForHubConnectedToHub(INOUT int &f_iDevUsable, IN  GENSTRING f_cSysName) const;


           /*!
        * \brief  function to check if due to configuration of temporarily not available interface Core/IStateTableHWMalfunction
        *           is allowed to be used. Has to be adde to all functions of the corresponding interface
        *
        * \return true: yes allowed, fals: not allowed to be used
        */
        bool bIsEnabled_IStateTableHWMalfunction();

        /*!
           * \brief  function to check if due to configuration of temporarily not available interface Core/IStateTableHWSignals
           *           is allowed to be used. Has to be adde to all functions of the corresponding interface
           *
           * \return true: yes allowed, fals: not allowed to be used
           */

        bool bIsEnabled_IStateTableHWSignals();

        /*!
       * \brief  function to fetch the value of configuration param. This will be used by  Device manager Service.
       * This function is used to get the configuration value of certain configuration parameters. This will be used by service device manager,
       *  to expose the value of configuration parameter to outside world. Only certain configuration parameters are allowed to be fetched by the outside world.
       *
       * \param [in]  f_strConfigParam: the Configuration Parameter, whose value to be fetched
       * \param [in]  uConfigValue: the Configuration Value
       * \return TRUE: if the configuration parameter is allowed to be fetched, FALSE: if the configuration parameter is not allowed to be fetched from outside world.
       */

        tBool vGetConfigurationValue(IN const char* f_strConfigParam,OUT tU16 &u16ConfigValue) const;

        /*!
       * \brief  function to set the value of configuration param. This will be used by  Device manager Service.
       * This function is used to set the configuration value of certain configuration parameters. This will be used by service device manager,
       *  to set the value of configuration parameter from the outside world. Only certain configuration parameters are allowed to be fetched by the outside world.
       *
       * \param [in]  f_strConfigParam: the Configuration Parameter, whose value to be fetched
       * \param [in]  uConfigValue: the Configuration Value
       * \param [in]  eSettingsMode: the Setting mode(Permanent, Temporary, default)
       *
       * \return TRUE: if the configuration parameter is allowed to be fetched, FALSE: if the configuration parameter is not allowed to be fetched from outside world.
       */

        tBool vSetConfigurationValue(IN const char* f_strConfigParam,IN tU16 u16ConfigValue,IN tenSettingsMode eSettingsMode);
    /*!
       * \brief  check if certain device type is allowed to be used concerning the customer configuration
       *
       * \param [out] f_enResult: enEnabled,enDisabled
       * \param [in]  check if device type is allowed to be used with this customer configuration
       * \param [in]  check if USB Portnumber is allowed to be used with this customer configuration
       *
       * \return
       */
      void vFilterForDeviceType(OUT tenResultFilter &f_enResult, IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType, IN int f_iUsbNumber) const;




        /*!
            * \brief  find out if device is mass storage device e.g. SD, USB etc.
            *
            * \param[IN] eDeviceType: type of device
            *
            *\returns TRUE/FALSE  TRUE: is mass storage device
            */

        tBool bIsMassStorageDevice(CGlobalEnumerations::DEVICE_TYPE_Type eDeviceType) const;


        /*!
                * \brief  time from system boot
                *
                *\returns time in millisec
                */

        tU64 GetEclapsedTime_ms();


        /*!
                * \brief  returns time initialised with InitEclapsedTime_ms (macros in config.h: INFORM_IF_TESTMODE helps to use it)
                *
                *\returns time in millisec
                */

        tU64 GetStartTime_ms();


        /*!
                * \brief  some functions got additional test values to inject parameters in testmode. This function is used to inform if software uses this testmode
                *           it is bound to configuration manager since it is available as singleton from all places in the code
                *
                *\returns time in millisec
                */

        void InformIfTestmode(char* strFunctionName, tU64 u64TestValue);

            /*!
                * \brief  ETG does not support tU64 (only u32) this function
                *           it is bound to configuration manager since it is available as singleton from all places in the code
                *
                *\returns time in millisec
                */

        void EtgTraceU64Value(char* strDescriptionText_80characters, tU64 u64TestValue, tU8 u8Level=LEVEL_USR4);


        /*!
        * \brief  read generic kds values of devicemanager and print it
        */
        void vPrintGenericKdsValuesDevicemanager();

        /*!
        * \brief  read generic kds values of mediaplayer and print it
        */
        void vPrintGenericKdsValuesMediaplayer();

        /*!
            * \brief  read kds values of mediaplayer related to project RNAIVI feature dvd
            */
        void vPrintRNAIVKdsValuesMediaplayerFeaDVD();

        /*!
              * \brief  read via Datapool function the kds values of mediaplayer related to project RNAIVI feature dvd
        */
        void vPrintRNAIVDatapoolValuesMediaplayerFeaDVD();

        /*!
              * \brief  read kds values of mediaplayer related to project RNAIVI feature musicbox related to region japan
            */
        void vPrintRNAIVKdsValuesMediaplayerReagionJapan();

        /*!
          * \brief    read via Datapol function values of mediaplayer related to project RNAIVI feature musicbox related to region japan
        */
        void vPrintRNAIVDatapoolValuesMediaplayerReagionJapan();

    protected:

         /*!
                  * \brief  called once if contructor of ConfigurationManager is started. Holds the starttime of VD_DVM - see GetEclapsedTime_ms
                  *
                  *\returns time in millisec
                  */

      tU64 InitEclapsedTime_ms();

        /*!
         * \brief This function counts devices at available USB connectors
         *
         * \param [OUT] returns result into f_arrUSBPortsUsed e.g. f_arrUSBPortsUsed[eUSB1] = 3
         *
         * \returns
         */
        void vCountDevicesAtUSBConnectors(OUT int (&f_arrUSBPortsUsed)[ARRAYSIZEFORUSB]) const;

        void  vAdjustValuesForGM(tenDevManagerConfigValue eTypeOfConfigValue);

        /*!
             * \brief  check with configuration manager if apple device is connected on a customer hub
             *
             * \param [in]  usbNumber USBConnectors to which the device is connected
             * \param [in]  deviceType type of device connected
             * \param [out] deviceUsable If device is allowed to be used concerning customer configuration
             *
             * \return void
             */
        void vFilterForAppleDeviceWithoutCustomerHub(INOUT int &deviceUsable, IN const int &usbNumber, const IN CGlobalEnumerations::DEVICE_TYPE_Type &deviceType) const;

    /*!
         * \brief  check with configuration manager if device is allowed to be used in dependency of its current connection.
         *           device might be connected at USB1 or 2 connector, via a HUB, only a certain number of devices is allowed to be used at a HUB
         *
         * \param [in]  USBConnectors of target
         * \param [in]  USBconnector number current device is connected to
         * \param [in]  HUB index if it is connected to a HUB
         * \param [in]  if it is connected to a HUB at all
         * \param [out] f_iDevUsable: If device is allowed to be used concerning customer configuration
         *
         * \return
         */
        void vFilterForConnectorsAndHubs(INOUT int &f_iDevUsable, IN const int (&f_arrUSBPortsUsed)[ARRAYSIZEFORUSB], IN int f_iUsbNumber, IN int f_iHubIndex, IN tBool f_bConnectedAtHub,IN  GENSTRING f_cSysName, IN CGlobalEnumerations::DEVICE_TYPE_Type deviceType) const;


        /*!
        * \brief  check if certain filesystem is allowed to be used concerning the customer configuration
        *
        * \param [in]  filesystem of device
        * \param [out] f_enResult: enEnabled,enDisabled
        *
        * \return
        */

        void vFilterForFilsystemType(OUT tenResultFilter &f_enResult,IN CGlobalEnumerations::FILESYSTEM_TYPE_Type f_eFileSystemType) const;


        /*!
        * \brief  check if size of a certain connected device exceeds a certain size
        *
        * \param [in]  device type
        * \param [in]  size
        *
        * \param [out] f_enResult: enEnabled,enDisabled
        *
        * \return
        */
        void vFilterForMassStorageDeviceSize(OUT tenResultFilter &f_enResult, IN CGlobalEnumerations::DEVICE_TYPE_Type eDeviceType,IN tU16 f_u16TotalSize_GB) const;



        void vFilterForDeviceAppleUnsupported(OUT tenResultFilter &f_enResult, IN int f_iVendorID,IN int f_iProductID) const;






    private:

        tU64 m_u64StartTime;

        static ConfigurationManager *m_pConfigurationManager; ///< singleton object


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
        static DVMLOCK    m_singelton;   ///< lock
#endif

        /*!
         * \brief Default constructor
         *
         * \note This is private since its a singleton class
         */
        ConfigurationManager();

        /*!
         * \brief Destructor
         *
         * \note This is private since its a singleton class
         */
        virtual ~ConfigurationManager();

        /*!
         * \brief function will be used by ConfigurationManager to check whether the Configuration Parameter is allowed to be fetched from the outside world
         * \param[IN] f_strConfigParam: the Configuration parameter need to be searched
         * \param[IN] enConfigValue: the Configuration parameter
         *
         * \return  true if the Configuration Parameter is allowed to be fetched from the outside world.
         */
        tBool isConfigParamAllowedTobeFetched(IN const char* f_strConfigParam, OUT tenDevManagerConfigValue &enConfigValue) const;


}; // class Info_tclDmEolConfSettings




#endif // CONFIGURATIONMANAGER_H

////////////////////////////////////////////////////////////////////////////////
// <EOF>
