/*-----------------------------------------------------------------------------*
 * ConfigurationBase.h                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                           *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                   *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file ConfigurationBase.h
 *
 * \brief Implementation of Configuration interface bound to either EOL, KDS, DATAPOOL
 *
 * \version 06.11.2013, Christian K�chling (Bosch), version 1.0
 *          Initial version

 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef CONFIGURATION_BASE_H
#define CONFIGURATION_BASE_H


/*-----------------------------------------------------------------------------*
 * Enums/structs   here added to namespace                                                            *
 *-----------------------------------------------------------------------------*/

namespace configmgr
{
/*!
 * \enum tenDevManagerEOLValue
 * !!!!!!!!!!!IF YOU ADD AN ELEMENT HERE ADJUST ALSO:!!!!!!!!!!!!!!!!!!!!
 * ConfigurationBase.cpp trInfoAboutConfigValues ConfigurationBase::m_rConfigValueDescription[eCONF_LAST_ELEMENT]
 * and correspodning values in vd_devicemanager.trc to offer configuration of new value with TTFIS
 */
typedef enum
{
    //-------------GM specific---------------------------- //although it has been fully replaced I'll keep this in order not to confuse GM-Testers who are used to corresponding TTFIS command their are used to
    eCONF_USB_1_HUB_ENABLE                                       = 0x00,
    eCONF_USB_2_HUB_ENABLE                                       = 0x01,
    eCONF_USB_1_HUB_MAX_PORTS                                    = 0x02,
    eCONF_USB_2_HUB_MAX_PORTS                                    = 0x03,
    eCONF_SYSTEM_OFFERING_BYTE1                                  = 0x04,  //this is used also to store l_Byte19_Bit6And7 of VAGMIBE project
    eCONF_SYSTEM_OFFERING_BYTE1Bit3_USB2HubPresent               = 0x05,
    eCONF_SYSTEM_OFFERING_BYTE1Bit4_USB1HubPresent               = 0x06,
    eCONF_SYSTEM_OFFERING_BYTE1Bit5_SDPresent                    = 0x07,
    eCONF_SYSTEM_OFFERING_BYTE1Bit6_USB2Present                  = 0x08,
    eCONF_SYSTEM_OFFERING_BYTE1Bit7_USB1Present                  = 0x09,
    eCONF_ENABLE_MTP_DEVICES                                     = 0x0A,
    eCONF_USB_VOLTAGE_RECOVERY_AUTOPLAY_DELAY_TIME               = 0x0B,
    eCONF_MARKETING_REGION                                       = 0x0C,
    eCONF_USB_UNUSED_BITS                                        = 0x0D,
    //-------------generic part ---------------------------- //here the part begins which also replaces the above poart but serves all projects since it is more generic
    eCONF_ENABLE_EXTERNAL_SD                                     = 0x0E,
    eCONF_ENABLE_EXTERNAL_USB                                    = 0x0F,
    eCONF_ENABLE_EXTERNAL_MTP                                    = 0x10,
    eCONF_ENABLE_EXTERNAL_IPOD                                   = 0x11,
    eCONF_ENABLE_EXTERNAL_IPAD                                   = 0x12,
    eCONF_ENABLE_EXTERNAL_IPHONE                                 = 0x13,
    eCONF_ENABLE_INTERNAL_CDROM                                  = 0x14,
    eCONF_ENABLE_INTERNAL_CDDA                                   = 0x15,//Roadmap 15002 CD Audio device Detection
    eCONF_ENABLE_INTERNAL_SDCARD                                 = 0x16,
    eConf_ENABLE_EXTERNAL_DIGIAUX                                = 0x17,
    eConf_ENABLE_EXTERNAL_BTA                                    = 0x18,
    eConf_ENABLE_EXTERNAL_HUB                                    = 0x19,//USB Compliance.. Hub as Device
    eConf_ENABLE_INTERNAL_DVD_DRIVE                                 = 0x1A,//1615 Internal DVD Drive connected to USB
    eCONF_ENABLE_EXTERNAL_UNSUPPORTED                            = 0x1B,

    /*if disabled VD_DVM supporesses notification to clients*/
    eCONF_ENABLE_USB1                                            = 0x1C, //similar to eCONF_SYSTEM_OFFERING_BYTE1Bit7_USB1Present
    eCONF_ENABLE_USB2                                            = 0x1D, //similar to eCONF_SYSTEM_OFFERING_BYTE1Bit6_USB2Present
    eCONF_ENABLE_USB3                                            = 0x1E,
    eCONF_ENABLE_USB4                                            = 0x1F,
    /*if disabled VD_DVM supresses notfication of devices connected to a HUB*/
    eCONF_ENABLE_HUBandPORTSatUSB1                               = 0x20,  //number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBandPORTSatUSB2                               = 0x21,  //number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBandPORTSatUSB3                               = 0x22,  //number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBandPORTSatUSB4                               = 0x23,  //number >0 means enabled for number of max ports
    /*if enabled HUBs will be observed for open circuit and overcurrent*/
    eCONF_ENABLE_observeDTC_HUBatUSB1                            = 0x24,
    eCONF_ENABLE_observeDTC_HUBatUSB2                            = 0x25,
    eCONF_ENABLE_observeDTC_HUBatUSB3                            = 0x26,
    eCONF_ENABLE_observeDTC_HUBatUSB4                            = 0x27,
    /*if diabled physically USB connector will be switched off*/      ///@todo question to Tomas Mehring has been posted if this should be done here or more down in the code
    eCONF_ENABLE_HW_USB1                                         = 0x28,
    eCONF_ENABLE_HW_USB2                                         = 0x29,
    eCONF_ENABLE_HW_USB3                                         = 0x2A,
    eCONF_ENABLE_HW_USB4                                         = 0x2B,

    //If the values are enabled, the hardware signals from the PRM will be used for the USB ports
    eCONF_HWSIGNAL_SOURCE_USB1                                   = 0x2C,
    eCONF_HWSIGNAL_SOURCE_USB2                                   = 0x2D,
    eCONF_HWSIGNAL_SOURCE_USB3                                   = 0x2E,
    eCONF_HWSIGNAL_SOURCE_USB4                                   = 0x2F,

    //If the value is enabled. It is assumed that there is an internal HUB. It'll be asked based on a time intervall if disturbance like over current has happened
    eCONF_HUB_HWSIGNAL_MONITOR_INTERVAL                          = 0x30,

    //filesystems - note normal philosophy is to allow as most as possible and supported by platform but in Mediaplayer certain will be suppressed
    eCONF_ENABLE_FILESYSTEM_FAT                                  = 0x31,
    eCONF_ENABLE_FILESYSTEM_NTFS                                 = 0x32,
    eCONF_ENABLE_FILESYSTEM_EXTFAT                               = 0x33,
    eCONF_ENABLE_FILESYSTEM_HFSPLUS                              = 0x34,
    eCONF_ENABLE_FILESYSTEM_EXT_3_4_FS                           = 0x35,
    eCONF_ENABLE_FILESYSTEM_ISO9660                              = 0x36,
    eCONF_ENABLE_FILESYSTEM_UDF                                  = 0x37,

    //-------------others---------------------------
    eCONF_ENABLE_LIMIT_SIZE_MSD                                  = 0x38,  // limit the size of Massstorage devices GigaByte
    eCONF_ENABLE_CHARGE_APPLEDEVS                                = 0x39, //+500mA i.e. 1A

    eCONF_ENABLE_NOTIFY_UNSUPPORTED_DEVTYPE                      = 0x3A,
    eCONF_ENABLE_NOTIFY_UNSUPPORTED_FILESYSTEM                   = 0x3B,
    eCONF_ENABLE_NOTIFY_UNSUPPORTED_HUB                          = 0x3C,  //under contruction

    eCONF_STATETABLE_EXECUTION_TIME_MS                           = 0x3D,  //statetable has a time window where it waits that HW signals and SPM to settle to final state

    eConf_ENABLE_NOTIFY_PARTITIONS_ALL                           = 0x3E,  //all partitions of a usb stick  or sd card will be notified
    eConf_ENABLE_NOTIFY_IPAD_AS_IPOD                             = 0x3F,  //IPAD will be notified a to clients as IPOD
    eConf_ENABLE_NOTIFY_SDINT_AS_SD                              = 0x40,  //internal SD Card will be notifiede to clients like an SDCard in external Card Reader connected via USB
    eConf_ENABLE_NOTIFY_CDROM_AS_USB                             = 0x41,  //cdrom will be notified as USB-device up till now used for test only
    eConf_ENABLE_CYCLIC_DIAGNOSIS                                = 0x42,  //cyclic check of connected devices and HUBS and Overcurrent to inform FC_diagnosis if something goes wrong

    eConf_ENABLE_USE_VOLTAGEMANAGER                              = 0x43, //use events of voltage manager - other customer products then GM do switch off USBConnetors at e.g. 6.5Volt
    eConf_ENABLE_USE_PRM_HWSIGNALS                               = 0x44, //use callbacks of prm telling if there is overcurrent etc.
    eConf_ENABLE_USE_PRM_SDCARDSIGNALS                           = 0x45, //for FC-diagnosis we need special information about internal SDCard (internal reader not connected via USB) that#s why we rely on PRM notification of card
    eConf_ENABLE_USR_PRM_HWCTRLUSB                               = 0x46, //it is possible to switch on off the voltage (VBUS) of USBConnectors this can be defined here
    eConf_ENABLE_USE_GPIO_INTERFACE                              = 0x47, //e.g. for getting overcurrent at internal SDCard reader it is neccessary to check a certain GPIO
    eConf_ENABLE_DIAGLOG_DTC_TRIGGERED_HWSIGNALS                 = 0x48, //in GM these are triggered due to cyclic diagnosis this offers to send it directly //@todo check what happens if both are switched on eventuelly don't allow it
    eCONF_ENABLE_FAKE_SDCARD                                     = 0x49, //For LSIM environment it is possible to fake a SDCard with an USB-stick with a FILE 'SDCARD'
    eConf_ENABLE_DEVICES_ALWAYS_AT_USB1                          = 0x4A, //For LSIM environemnt it is possible that all devices are connected to the same port even devices connected to a hub
    eConf_ENABLE_USE_ROOTDAEMON                                  = 0x4B, //due to new security concept Linux commands have to go via Rootdaemon
    eConf_ENABLE_NOTIFY_UNKNOWNDEV_AT_STARTUP                    = 0x4C, //it is possible to notifyinstead of empty vector an unknown element at startup - used in GM only
    eConf_ENABLE_NOTIFY_NO_REDUNDAND_MSGS                        = 0x4D, //output of state table adjusted in order not to send same messages several times. This hs an effect at the interface of VD_DVM thus its made configurable
    eConf_ENABLE_SUPPRESS_DIAGNOSISHWSIGNAL                      = 0x4E, //GM specific: Intermediate solution until eConf_ENABLE_NOTIFY_NO_REDUNDAND_MSGS is fully used and tested
    eConf_ENABLE_ITC_OVERCURRENT_SUPPRESSED_IFCRITVOLT           = 0x4F, //Sending ITC for OVERCURRENT via diaglog is suppressed if system has critical voltage. Reason because of the Hardware it is not really possible to make a distinction between overcurrent and critical voltage.
    eConf_ENABLE_ITC_OVERCURRENT_TRIGGERS_ITC_POWER_SUPPLY_FAULT = 0x50, //special solution currently used in PSA all ITC's which mark that USB HW does not work should be send - this is for ITC_POWER_SUPPLY_FAULT
    eConf_ENABLE_MISSINGTELEMATICSBOX_TRIGGERS_ITC_USB_OPEN      = 0x51, //special solution currently used in PSA all ITC's which mark that USB HW does not work should be send - this is for ITC_USB_OPEN
    eConf_ENABLE_TELEMATICSBOX_USB1                              = 0x52, //PSA and RNAIVI use a TelematicsBox at one USB connector - mark it to be bound to USB1
    eConf_ENABLE_TELEMATICSBOX_USB2                              = 0x53, //PSA and RNAIVI use a TelematicsBox at one USB connector - mark it to be bound to USB2
    eConf_ENABLE_TELEMATICSBOX_USB3                              = 0x54, //PSA and RNAIVI use a TelematicsBox at one USB connector - mark it to be bound to USB3
    eConf_ENABLE_TELEMATICSBOX_USB4                              = 0x55, //PSA and RNAIVI use a TelematicsBox at one USB connector - mark it to be bound to USB4
    eCONF_ENABLE_TELEMATICSBOX_MONITORING                        = 0x56, //PSA and RNAIVI use a TelematicsBox. The FC_ATB will update the VD_DVM if the connection to the BTA Box to application layer is active
    eCONF_ENABLE_TESTMODE_SETUSBHARDWARECONNECTOR                = 0x57, //function PrmManager::bSetUSBHardwareConnector() is used to enable/disable VBUS of correspodning USB connector. In PSA e.g. one chip provides VBUS for all 3 connectors. I.e. if switched off also ttfis is switched off but with this value enabled it switches on again and traces can be analysed
    eCONF_ENABLE_NOTIFY_FIRST_SUPPORTED_PARTITION                = 0x58, //Sending the first supported partition to the clients provided the USB supports more than one partitions.
    eConf_ENABLE_MALFUNCTION_USBCONNECTORS                       = 0x59, //Used to block/unblock usage of corresponding functions of IStateTableHWMalfunction and it is bound to VARIANT_S_FTR_ENABLE_DEVMGR_PRM_USED
    eCONF_TIMOUT_MALFUNCTION_SET_PERMANENT                       = 0x5A, //If a device is set MALFUNCTION then mechanism to monitor if the malfunction is started to check if time of malfunction exceeds this configurable value. In this manner  MALFUNCTION_PERMANENT could be send to clients
    eCONF_USBCOMPARETIME_MALFUNCTION_OR_REMOVE                   = 0x5B, //Occurrences malfunction are send to DeviceManager which stores it with time stamp. If it receives a ‘device removed’ it uses this configurable ‘compare time’ to decide if the remove has to be adjusted to MALFUNCTION to be send to clients
    eConf_ENABLE_MALFUNCTION_SDCARDINTERNAL                      = 0x5C, //Used to block/unblock usage of corresponding functions of IStateTableHWMalfunction
    eConf_ENABLE_MALFUNCTION_CDROMINTERNAL                       = 0x5D, //Used to block/unblock usage of corresponding functions of IStateTableHWMalfunction
    eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL                     = 0x5E, //Find the All the partitions of the MSD devices such as USB sticks and SD card.And sort the list of devices based on size of partition
    eConf_MAXIMUM_SUPPORTED_PARTITIONS_MSD                       = 0x5F, //Number of partitions supported by MSD devices such as USB sticks and SD card
    eCONF_ENABLE_HWInternalHUBRealizesUSB1                       = 0x60,//if USB1 is connected through the Internal hub
    eCONF_ENABLE_HWInternalHUBRealizesUSB2                       = 0x61,//if USB2 is connected through the Internal hub
    eCONF_ENABLE_HWInternalHUBRealizesUSB3                       = 0x62,//if USB3 is connected through the Internal hub
    eCONF_ENABLE_HWInternalHUBRealizesUSB4                       = 0x63,//if USB3 is connected through the Internal hub
    eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB1                 = 0x64,//number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB2                 = 0x65,//number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB3                 = 0x66,//number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBconnectedToHUBandPORTSatUSB4                 = 0x67,//number >0 means enabled for number of max ports
    eCONF_ENABLE_HUBconnectedToHUBPermanent                      = 0x68,//This is an internal configuration value set by configuration base
    eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT                           = 0x69,//Time out used to decide whether to change the value of ITC USB - Circuit open
    eCONF_ENABLE_MSD_BIGGEST_PARTITION                           = 0x6A,//Find the Biggest of all the partitions of the MSD devices such as USB sticks and SD card.
    eCONF_ENABLE_SORT_PARTITIONS_ALL                             = 0x6B,//Sorts all the partitions of the MSD devices such as USB sticks and SD card based on its size.
    eCONF_ENABLE_HWSIGNALS_SWAPPING                              = 0x6C,//OBSOLETE!!! Replaced by eCONF_HWSIGNAL_SOURCE_USB (Formerly used for Swaps the source of hardware signals for example CAF projects needs different naming for ports.If the HW signal coming for port 1 in CAF project, we have to map it to port2 to match with the naming convention. Refer https://hi-cmts.apps.intranet.bosch.com:8443/browse/CAFI06-1533 for details of port numbering in CAF project)
    eCONF_ENABLE_GETMSC_FREE_USED_SIZE                           = 0x6D,//RNAIVI startup could be speeded up if statfs (or df) is not called to get free, used and total size. If it is switched off that a temporary parallel thread is started to fetch it and update it later
    eCONF_ENABLE_AUTOMOUNTER_SHAREDLIBIF                         = 0x6E,//autmounter shared lib is used instead of the text based interface to automounter in /tmp/.autmounter
    eCONF_ENABLE_PERSISTENT_STORAGE_FOR_CONFIG_PARAMS            = 0x6F,//PSA they need to store the configuration values in persistent memory to enable the hub to hub connection. DVM using datapool to write the values to persistent memory.
    eCONF_ENABLE_observeDTC_INTERVALL                            = 0x70,//A thread is executing in dependency of this time intervall to collect diagnosis values to send it to diaglog. Dependency: see  eCONF_ENABLE_observeDTC for USB1, USB2 ...
    eCONF_ENABLE_DTC_HUB_NUMOFPORTS_MISMATCH_VARIANT2            = 0x71,//in GM customer writes num of ports directly to configuration. If ports detected are less or equal then PASSED. For RNAIVI: KDS used but not directly num of ports are written
                                                                        //but either 1 or 2 is set. Here PASSED is send if num of found ports of HUB is >= number of configured ports. Since some HUBS have internal ports not seen at the outside this is the best way to prevent unneccessary tickets and discussions
                                                                        //common sense: customer wants to know if ports of a hub don't work anymore. But GM is used to the current implementation but can incluence it directly to prevent unneccessary FAILS

    eCONF_ENABLE_IMX_HOST_USB_2184200_ALLOWHUB                   = 0x72,//Routines in CommonReadTourinesKDS refer to IMX connectors not to Connectors visible at the outside -for this reason an intermediate setting of these values help to set others - used with gen3 architectur
    eCONF_ENABLE_IMX_OTG__USB_2184000_ALLOWHUB                   = 0x73,//Routines in CommonReadTourinesKDS refer to IMX connectors not to Connectors visible at the outside -for this reason an intermediate setting of these values help to set others - used with gen3 architectur
    eCONF_ENABLE_HIGHRISK4USB_HWMALFUNCTION                      = 0x74, //beside using directly HW signals which tell �f USB connectors fail or not it is possible to use critical voltage as input to set HWMALFUNCTION (temporary not available) for connected devices to Mediaplayer (RNAIVI is first customer)
    eCONF_ENABLE_ITC_DATA_LINE_FAULT                             = 0x75, //PSA needs to set the ITC ITC_DATA_LINE_FAULT for telematic boxes, if vd_devicemanager identifies a wrond device(older telematics box not present in the configurationn) or an invalid device like USB or iPod devices to configured usb port
    eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED                      = 0x76,//RNAIVI needs the ITC LINK_DISABLED to be set when the hub is not present. The hub details are stored statically in the configuration files or in the persistent files. The dependency for setting the ITC LINK_DISABLED is eCONF_ENABLE_observeDTC_HUBatUSB1, eCONF_ENABLE_observeDTC_HUBatUSB2,eCONF_ENABLE_observeDTC_HUBatUSB3 and eCONF_ENABLE_observeDTC_HUBatUSB4
    eCONF_ENABLE_NOTIFY_CUSTOMER_HUBS                            = 0x77,//Device(Hub) USB Compliance - Check and update UnsupportedReason based on Cusomter Hubs or not
    eCONF_ENABLE_NOTIFY_UNKNOWN_DEVICES                          = 0x78,//USB Compliance - Check and update UnsupportedReason based on Supported USB devices or not
    eCONF_ENABLE_LIBUSB_ATSTARTUP_SHOWALLDEVICES                 = 0x79,//Formerly always part of the software to see inconsistency between libusb & udev. Now disabled  to reduce load. Disabled: Less communication at usb-bus and prevent access to devices where root is demanded
    eCONF_ENABLE_USE_SPM_HISTORYTRIGGER_FOR_EARLYEJECT           = 0x7A,//if VD_DVM and VD_MMGR are fully started but user has pressed eject before start of these components. VD_DVM checks at startup if such eject and does care for it
    eCONF_ENABLE_NOTIFY_DEVPATH                                  = 0x7B,//Notify clients about the devpath,i.e., syspath of the device connceted at the port
    eCONF_ENABLE_DELAY_NOTIFY_MTP_DEVICES                        = 0x7C,//PSARCC - This will delay the notification  of MTP devices till the SPI updates that  delaying the MTP devices  is not needed for a device or the configured timeout(DEVICETYPE_RETRY_ELAPSED_TIMESEC) is reached.
    eConf_ENABLE_DTC_MAP_TCU_OC_PORT3                            = 0x7D, //RNAIVI expects TCU  (Telematics box) always to be bound to USB3. This is valid also for targets which do have only two external USB connectors. Hence a mapping is necessary.
    eConf_ENABLE_DTC_MAP_ETC_OC_PORT2                            = 0x7E, //RNAIVI expects ETC  (Eletronic toll collect ) always to be bound to USB2.
    eCONF_ENABLE_TRISTATE_DIAGNOSIS                              = 0x7F,//This will enable the tristate logic in checking mediastatusbits in Diagnosis IOControl. If not enabled, backward compatible code is used.
    eCONF_ENABLE_MEDIAMANAGER_CLIENTHANDLER                      = 0x80,//This will enable the MediaManager Cliethandler to receive CD related updates from Mediamanager
    eCONF_STATETABLE_CONNECT_EXECUTION_TIME_MS                   = 0x81,//To accumulate multiple connection events
    eCONF_ENABLE_SUPPRESS_APPLE_DEVICE_WITHOUT_CUSTOMERHUB       = 0x82,//This is to suppress apple device connection, which are not connected on a customer hub
   //----------------------------------------------------------------
    eCONF_LAST_ELEMENT                                           = 0x83 //last values
} tenDevManagerConfigValue;

//Note: check if these configurations also make sense to be shifted into dynamic configuration
//USE_ENABLE_DEVMGR_GPIOIF
//USE_SENT_ALL_CHANGES
//USE_EXTERNAL_UNSUPPORTED_DEVICES
//USE_HWCONFIGURATIONINIT
//VARIANT_S_FTR_ENABLE_DEVMGR_CONFIGDEFAULT_GEN3_GMMY16 in config hsould be made redundant also for GM //@todo after ok test on GM this is obsolete


typedef enum
{
    enUnknown  = 0x00,
    enEnabled  = 0x01,
    enDisabled = 0x02,
}tenResultFilter;


#define eNO   "   "
#define eEOL  "[EOL]"
#define eKDS  "[KDS]"


typedef enum
{
    //Products
    eG3G          = 0x00,
    eGMMY16       = 0x01,
    eJACCII       = 0x02,
    ePSARCC       = 0x03,
    eRNAIVI       = 0x04,
    eSUZSLN       = 0x05,
    eSPINUP       = 0x06,
    eCAF          = 0x07,
    eCHERY        = 0x08,
    //experimental Products
    eVAGMIBE      = 0x09,
    eWHITELABEL   = 0x0A,
    eLESS         = 0x0B,
    eLastCustomer = 0x0C
}tenCustomer;




typedef struct
{
    tenDevManagerConfigValue          enConfigValue;
    std::string                       strInternal;
    std::string                       strRange;
    std::string                       strConfigValue;
    std::string                       strConfigurableProductionLine;
    std::string                       strDescriptionOfValues;
    std::string                       strVarianSwitchToWhichItIsbound;
    tBool                             bIsConfigurableFromOutside;//if the value is TRUE, then this tenDevManagerConfigValue can be configured from the external CCA interface
} trConfigValueDescription;

};


//predefined configuration values
#define DVM_OFF     (tU16)0
#define DVM_ON                  (tU16)1
// Added for Filtering based on Device Type and Portnumber ... Changed for CAFI06-1830
#define DVM_ON_USB1             (tU16)2
#define DVM_ON_USB2             (tU16)3
#define DVM_ON_USB3             (tU16)4
#define DVM_ON_USB1_USB2        (tU16)5
#define DVM_ON_USB2_USB3        (tU16)6
#define DVM_ON_USB3_USB1        (tU16)7
#define DVM_INVALID (tU16)0xFFFF

#define DVM_PRM_USED         0xA0   //plain mapping: OC1,PPON1 for USB1, OC2,PPON3 for USB2, OC3,PPON3 for USB3
#define DVM_HUBINTERNAL_USED 0xB0

//if HW mapping is enabled see eCONF_ENABLE_HWSIGNALS_MAPPING this means customer required numbering of USB connector is used
//this means that e.g. OC1,PPON1 might e.g. map to USB2 etc.
#define DVM_PRMPORT1_USED 0xC1
#define DVM_PRMPORT2_USED 0xC2
#define DVM_PRMPORT3_USED 0xC3
#define DVM_PRMPORT4_USED 0xC4

#define DVM_HUB_HWSIGNAL_MONITOR_INTERVAL_SEC       10 /*seconds*/

#define DVM_DIAG_TIMEOUT_USB_OPEN_SEC               60 /*seconds*/

#define DVM_TELEMATICSBOX_MONITOR_INTERVAL_SEC      10 /*seconds*/
#define DVM_MALFUNCTION_MONITOR_INTERVAL_SEC        30 /*seconds*/
#define DVM_COMPARETIME_MALFUNCTION_OR_REMOVE_MSEC  DVM_Tprev_ms /*milli seconds*/
#define DVM_LIMITEDSIZE_32GB  32 + 2 /*2 is offset*/
#define DVM_ON_1PORT (tU16)1
#define DVM_ON_2PORTS (tU16)2
#define DVM_ON_3PORTS (tU16)3
#define DVM_ON_4PORTS (tU16)4
#define DVM_ON_8PORTS (tU16)8
#define DVM_FIVE_PARTITIONS (tU16)5
#define DVM_ALL_PARTITIONS (tU16)0xff /*max is 255*/

const unsigned int iMaxSizeOfArray = 10;





/*-----------------------------------------------------------------------------*
 * unsing namespace                                                               *
 *-----------------------------------------------------------------------------*/
using namespace statetbl; //for eUSBAll
using namespace configmgr;

/*!
 * \struct Info_rDmEolConfSettings
 *
 * \brief Data structure to store the calibration values.
 */


/*-----------------------------------------------------------------------------*
 * cforward declaration                                                                *
 *-----------------------------------------------------------------------------*/
class IConfigurationPersistent;

/*-----------------------------------------------------------------------------*
 * class declaration                                                                *
 *-----------------------------------------------------------------------------*/
class ConfigurationBase
{
public:

    //-----------functions-------------------------------

    ConfigurationBase();
    virtual ~ConfigurationBase(){};



    /*!
            * \brief get USBConnector from syspath e.g. syspath: /sys/devices/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.2
            *          from USBPortAdress 2184200.usb it can be deduced which USBConnector it is.
            *
            * \param [in] syspath of device
            * \param [OUT] eUsb. either eUsb1 or eUsb2
            *
            */
    virtual tVoid vGetUSBConnectorFromSysPath(OUT tenDevManagerUSBHost &eUsb, IN const std::string &strSysPath) const;
    /*!
          * \brief function to check whether the BTA (TCU) device supported or not.
          * \return  true if the BTA device is enabled.
          */
    tBool bFilterIsTelematicsBox(IN unsigned int iVendorID, IN unsigned int iProductID);


       /*!
            * \brief function to check whether DigitalAuxIn Device is suppored or detected
            *   used for GM who have an auxin connector in their hub and a unit in the hub converts the analogue input to digital output send via usb to our target

            * \return  true if the AUXIN device matches iVendorID and ProductID and is supported at all
            */
      tBool bFilterIsDigitalAuxInDevice(IN unsigned int iVendorID, IN unsigned int iProductID);


    /*!
          * \brief function to check if BTA (Telematics box) is expected to be connector at connector eUsb
          * \param [IN] Port number
          * \param [INOUT] expected device speed. TRUE means the device will work at high speed
          * \return  true if the BTA device is expected at connector eUsb
          */
    tBool bFilterTelematicsBoxExpectedAtConnector(IN tenDevManagerUSBHost eUsb);
    /*!
          * \brief Fills the expected device speed info at a connector eUsb
          * \param [IN] Port number
          * \param [INOUT] expected device speed. TRUE means the device will work at high speed
          * \return  true if the BTA device is connected at connector eUsb
          */
    tBool vFillDeviceSpeed(IN tenDevManagerUSBHost eUsb,INOUT tBool &bHighSpeedDeviceExpected);

    /*!
         * \brief function to check whether the Internal Hub is found or not.
         * \return  true if the Internal hub found
         */
    tBool bFilterIsInternalHub(IN unsigned int iVendorID, IN unsigned int iProductID);

    /*!
         * \brief function to check whether the internal DVD drive(KENWOOD) is supported or not.
         * \param [IN] iVendorID vendor id of internal dvd drive
         * \param [IN] iProductID product id of internal dvd drive
         * \return  true if the Internal DVD drive is supported
         *          false otherwise
         */
    tBool bFilterIsInternalDvdDrive(IN unsigned int iVendorID, IN unsigned int iProductID);

    /*!
         * \brief function to check whether the CD drive(MASCA) is supported or not.
         * \param [IN] sVendor vendor name of cd drive
         * \param [IN] sProduct product name of cd drive
         * \return  true if the CD drive is supported
         *          false otherwise
         */
    tBool bFilterIsCdDrive(IN std::string strVendor, IN std::string strModel);

    /*!
     * \brief function to check whether apple devices are supported or not.
     * \return  true if apple devices are supported
     *          false otherwise
     */
    tBool bIsAppleDeviceSupported();

    /*!
     * \brief function will be called by ConfigurationProduct to enable or disable apple device support
     * \param [IN] bEnable TRUE or FALSE
     */
    tVoid vEnableAppleDeviceSupport(tBool bEnable);

    /*!
    * \brief function will be called by TraceCmdManager and ConfigurationProduct to enable or disable the hub connected to hub
    * \param [IN] bEnable TRUE or FALSE
    */
    tVoid vEnableHubConnectedToHubPermanent(tBool bEnable);

    /*!
     * \brief function will be used by ConfigurationBase to check the hub connected to hub permanent is enabled or not
     * \return  true if the hub connected to hub permanent is enabled
     */
    tBool isHubConnectedToHubPermanentEnabled();
    /*!
        * \brief function will be used by ConfigurationManager to store ExternalHub details in datapool
        * \param[IN] f_AddExternalHubInfo ExternalHub structure
        *
        * \return  true if Datapool is updated/idle.
        */
    tBool AddExternalHub(IN trExternalHubInfo f_AddExternalHubInfo) ;

    /*!
        * \brief function will be used by ConfigurationManager to remove ExternalHub details from datapool
        * \param[IN] f_RemoveExternalHubInfo ExternalHub structure
        * \param[IN] tBool ClearAllfromDP Flag to check if clearing of Datapool is needed or not
        *
        * \return  true if Datapool is updated/idle.
        */
    tBool RemoveExternalHub(IN trExternalHubInfo f_RemoveExternalHubInfo,IN tBool ClearAllfromDP) ;

    /*!
    * \brief function will be used by ConfigurationManager to read ExternalHub details from datapool
    *
    * \return  true if reading is successful.
    */
    tBool ReadExternalHub(OUT trExternalHubInfo f_uiExternalHubInfo [], OUT tU8 &f_uiNumberOfExternalHubsUsed) ;

    /*!
        * \brief function will be used by ConfigurationManager to store TelematicsBoxDevice details in datapool
        * \param[IN] f_AddTelematicsBoxDeviceInfo TelematicsBoxDevice structure
        *
        * \return  true if Datapool is updated/idle.
        */
    tBool AddTelematicsBox(IN trTelematicsBoxDeviceInfo f_AddTelematicsBoxDeviceInfo) ;

    /*!
        * \brief function will be used by ConfigurationManager to remove TelematicsBoxDevice details from datapool
        * \param[IN] f_RemoveTelematicsBoxDeviceInfo TelematicsBoxDevice structure
        * \param[IN] tBool ClearAllfromDP Flag to check if clearing of Datapool is needed or not
        *
        * \return  true if Datapool is updated/idle.
        */
    tBool RemoveTelematicsBox(IN trTelematicsBoxDeviceInfo f_RemoveTelematicsBoxDeviceInfo,IN tBool ClearAllfromDP) ;

    /*!
    * \brief function will be used by ConfigurationManager to read TelematicsBoxDevice details from datapool
    *
    * \return  true if reading is successful.
    */
    tBool ReadTelematicsBox(OUT trTelematicsBoxDeviceInfo f_uiTelematicsBoxDeviceInfo [], OUT tU8 &f_uiNumberOfTelematicBoxesUsed) ;

    /*!
    *\brief function will be used by ConfigurationProduct to read Usb device details from a stream
    *\param[IN] input stream from which we need to read usb device info
    *\param[INOUT] usbDeviceInfo vector of UsbDeviceInfo, to be filled
    *\param[IN] deviceType usb device type to be read
    *
    * \return  true if reading is successful.
    *          false otherwise
    */
    tBool ReadUsbDeviceInfoFromIstream(std::istream &input, vector<UsbDeviceInfo> &usbDeviceInfo, const std::string &deviceType);

    /*!
    * \brief function will be used by ConfigurationManager to find if a usb device is part of External hub
    * \param[IN] iVendorID Vendor-id of usb device
    * \param[IN] iProductID Product-id of usb device
    * \return true if the iVendorID,iProductID is part of External hub
    *         false otherwise
    */
    tBool isDeviceAssociatedWithHub(const uint16_t &vendorID, const uint16_t &productID) const;

   /*!
    * \brief function will be used by Diaglog to check the customer expected external hub is connected or not.
    *  Details about the hub will be stored corresponding project configuration file or in the persistent memory. Based on this function output daiglog has to set the LINK_DISABLED ITC to diaglog
    *  \param [IN] eUsb the connector where the HUB is expected.
    * \return  true if the customer expected external hub is connected
    */
    tBool bExpectedExternalHubIsConnected(IN tenDevManagerUSBHost eUsb);

    /*!
     * \brief function to check whether the hub connected is supported or not.
     * \param [IN] eUsb the connector where the HUB is connected.
     * \param [IN] iVendorID vendor id of hub connected
     * \param [IN] iProductID product id of hub connected
     * \return  true if the hub connected is supported
     *          false otherwise
     */
    bool isExternalHub(IN const tenDevManagerUSBHost& eUsb, IN const unsigned int& iVendorID, IN const unsigned int& iProductID);

    //bound to eConf_ENABLE_EXTERNAL_BTA (Box Telematic Automous) - see e.g. ConfigurationProduct for PSAARCC and RNAIVI
    //as long as the utest is using the singleton we can not derive a testable configurationManager class and set protected to public here - hence its made public for now
    unsigned int m_uiNumberOfTelematicBoxesUsed;
    trTelematicsBoxDeviceInfo m_uiTelematicsBoxDeviceInfo[iMaxSizeOfArray];

    //bound to eConf_ENABLE_EXTERNAL_DIGIAUX (digital auxin) - see e.g. ConfigurationProduct for GM
    //as long as the utest is using the singleton we can not derive a testable configurationManager class and set protected to public here - hence its made public for now
    unsigned int m_uiNumberOfDigitalAuxinDevicesUsed;
    trDigitalAuxinDeviceInfo m_uiDigitalAuxinDeviceInfo[iMaxSizeOfArray];


   //bound to eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED  - see e.g. ConfigurationProduct for RNAIVI
   //as long as the utest is using the singleton we can not derive a testable configurationManager class and set protected to public here - hence its made public for now
   //This will be storing the number an details of external hubs expected by the RNAIVI project
   unsigned int m_uiNumberOfExternalHubsUsed;
   trExternalHubInfo m_uiExternalHubInfo[iMaxSizeOfArray];

   vector<UsbDeviceInfo> externalHubAssociatedDeviceInfo;



    //for detecting the internal hub in PSARCC project
    unsigned int m_uiInternalHubProductID;
    unsigned int m_uiInternalHubVendorID;

    //NCG3D-144806 : For detecting KENWOOD internal DVD drive
    unsigned int m_uiInternalDvdDriveProductID;
    unsigned int m_uiInternalDvdDriveVendorID;

    //NCG3D-144806 : For detecting MASCA CD drive
    std::string m_strCdDriveModel;
    std::string m_strCdDriveVendor;


    std::string m_strAllCustomerNames[eLastCustomer];

    static trConfigValueDescription m_rConfigValueDescription[eCONF_LAST_ELEMENT];

protected:

#ifndef USE_OLD_GETSYSPATH

    //these matchers are used to map kds values later to configure different USB connector configurations
    //see prepared code in CommonReadroutinesKds.h
    typedef enum StringMatchers_E
    {
        eMatcherEmpty                               = 0,
        eMatcherGen3UsbImxOtgConnector              = 1,
        eMatcherGen3UsbImxHostConnector             = 2,
        eMatcherGen3UsbImxHostConnectorIntHubPort1  = 3,
        eMatcherGen3UsbImxHostConnectorIntHubPort2  = 4,
        eMatcherGen3UsbImxHostConnectorIntHubPort3  = 5,
        eMatcherNotUsedText                         = 6,
        eMatchersAll                                = 7,
    }teStringMatchers;

    typedef enum ConnectorInfo_E
    {
        eConnectorUnused           = 0,
        eConnectorRealisedDirect   = 1,
        eConnectorRealisedIntHub   = 2,
    }teConnectorInfo;

    //syspath e.g. /sys/devices/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.2
    //contains USBPortAdress e.g. 2184200.usb this is fixed bound to certain USBConnector
    typedef struct{
          teConnectorInfo eConnectorInfo;
          std::string     m_strSysPathEntryUSBPortAdressForUSBConnector;
          std::string     m_strSysPathEntryUSBPortHubForUSBConnector;
     }trCustomerUSBConnectorConfig;
#endif



    //-----------variables-------------------------------



    tenCustomer m_eCustomer;
    tU16 m_u16ConfigurationValue[eCONF_LAST_ELEMENT];

#ifdef USE_OLD_GETSYSPATH
    std::string m_strSysPathEntryUSBPortAdressForUSBConnector[eUSBAll];
#else
    trCustomerUSBConnectorConfig m_rCustomerUSBConnectorConfig[eUSBAll];

    std::string strMatchers[eMatchersAll];
#endif


    //SDCard is mapped to fixed devices this name  is set here and can be adjusted in ConfiguratioProduct_Gen3_xyz if necessary
    std::string m_strBlockDevNameSDCardInternal;

    //NCG3D-144806 :CDROM and KENWOOD Internal DVD drive is mapped to ex./dev/sr[0-9]. Hence part(ex./dev/sr) is stored and can be adjusted in ConfiguratioProduct_Gen3_xyz if necessary
    std::string m_strBlockDevNameCDROMInternalPart;
    std::string m_strBlockDevNameDVDInternalUSBConnectedPart;

    std::string m_strBlockDevNameSDCardInternalWithoutDev;
    std::string m_strBlockDevNameCDROMInternalWithoutDev;
    std::string m_strBlockDevNameDVDInternalUSBConnectedWithoutDev;


    //helpers for filter configuration filter in derived claass ConfigurationManager
    tenDevManagerConfigValue m_eHUBAndPortsEnable[eUSBAll];
    tenDevManagerConfigValue m_eHUBconnectedToHUBAndPortsEnable[eUSBAll];
    tenDevManagerConfigValue m_eUSBPresent[eUSBAll];

    CGlobalEnumerations::DEVICE_TYPE_Type     m_eDeviceType[eCONF_ENABLE_EXTERNAL_UNSUPPORTED +1]; //brute force programming - arrays bigger but easier to use
    CGlobalEnumerations::FILESYSTEM_TYPE_Type m_eFileSystemType[eCONF_ENABLE_FILESYSTEM_UDF +1];

    tBool bUseMArketingRegion;



    //-----------functions-------------------------------
    //lint -sem(ConfigurationBase::vClear,initializer)
    void vClear();
    //lint -sem(ConfigurationBase::vInitBaseConfig,initializer)
    void vInitBaseConfig();  //harcoded default values plus values based on Variant-switches


    /*!
    * \brief function helps to check consistency between tenDevManagerConfigValue and m_rConfigValueDescription
    * \return FALSE means  ERROR inconsistent, TRUE means ok
    */
    tBool bCheckConsistencyOfConfigValuesDescription();

     /*!
    * \brief function helps to check consistency between values set. If certain values are set other dependent values have to be disabled or vice versa
    *         this function does FATAL_TRACES, and ERMEM_Entries if it finds such contradictions. ERRMEM_Entries are important because the project has infueuence on some values
    *         using e.g. KDS or EOL this might lead to inconsistency and could cause tickets. By checking the ERRMEM one could easily check for this.
    *         THIS FUNCTION IS A PROTECTED FUNCTION AND CALLED FROM CONSTRUCTOR OF ConfigurationPRoduct after all values have been set.
    * \return FALSE means  ERROR inconsistent, TRUE means ok
    */
    tBool bCheckConsistencyOfConfigValues();


   /*!
     * \brief trace configured address to USB-Port setting of this configuration
     */

   void TracestrSysPathEntryUSBPortAdressForUSBConnector();


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
    public:
        void setCustomerUSBConnectorConfig(tenDevManagerUSBHost eUsb,int iConnector,std::string strSysPathEntryUSBPortAdressForUSBConnector,std::string strSysPathEntryUSBPortHubForUSBConnector);
#endif



};

#endif // CONFIGURATIONBASE_H

////////////////////////////////////////////////////////////////////////////////
// <EOF>
