/*-----------------------------------------------------------------------------*
 * usbutils.h                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2010 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------
 *
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *         -# start tp replace QStrings by replacing QString by define GENSTRING
 *
  * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file usbutils.h
 *
 * \brief This file holds declaration of libusb wrapper functions
 *
 * \version 27.08.2012, Negi, Sunder (MontaVista), initial version
 *
 * \copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef USBUTILS_H_
#define USBUTILS_H_

#include <libusb-1.0/libusb.h>

typedef libusb_device_handle libUsbHandle;
/*!
 * Initializes libusb
 *
 * \return 0 on success, or libusb error code on error
 */
int LibUSBInit(void);

/*!
 * De-initializes libusb
 *
 * \return 0 on success, or libusb error code on error
 */
int LibUSBExit(void);

/**
 * \brief Generates libusb device object corresponding to a usb device path
 *
 * \param [in] f_cUsbDevPath path to the usb device e.g. /dev/bus/usb/00X/00Y
 *
 * \return [libusb_device *] the usb device on success, NULL on error
 */
libusb_device *LibUSBGetDevice(const char *f_cUsbDevPath);

/*!
 * \brief Fetches the string descriptor at index f_iId
 *
 * \param [in]     f_poDev the libusb device handle
 * \param [in]     f_iId   the string descriptor index
 * \param [in,out] f_cBuffer the buffer to store the string
 * \param [in]     f_iBufSize the buffer size
 *
 * \return [int] the string length
 */
int LibUSBGetString(libusb_device_handle *f_poDev, uint8_t f_iId,
        unsigned char *f_cBuffer, size_t f_iBufSize);

/*!
 * \brief Prints details of the usb device
 *
 * \param [in] f_poDev the libusb device
 *
 * \return [int] error status
 *
 * \note libusb needs to be initialized first using \ref LibUSBInit()
 */
int LibUSBShowDeviceDetails (libusb_device *f_poDev);

/*!
 * \brief Lists all usb devices connected to the system
 *
 * \return [int] error status
 *
 * \note libusb needs to be initialized first using \ref LibUSBInit()
 */
int LibUSBShowAllDevices (void);

/*!
 * \brief Get the device type
 *
 * \param [in] f_poDev the usb device
 *
 * \return [CGlobalEnumerations::DEVICE_TYPE_Type] device type
 */
CGlobalEnumerations::DEVICE_TYPE_Type LibUSBGetDeviceType (libusb_device *f_poDev);

/*!
 * \brief Fill device details
 *
 * \param [in] f_poDev the usb device
 * \param [out] f_poDevice the device to fill
 *
 * \return [int] error status
 */
int LibUSBFillDeviceDetails (libusb_device *f_poDev, CDevice& f_poDevice);

/*!
 * \brief Initiate apple device charging
 *
 * \param [in] f_iVendorID device vendor id
 * \param [in] f_iProductID device product id
 */
void ChargeAppleDevice(uint16_t f_iVendorID, uint16_t f_iProductID);



#ifdef USE_EXPERIMENTAL_PSA_HUBCOMMUNICATION

/*!
 * Get the status of hub connected to the target using the libusb communication
 *
 * \return TRUE on success
 */
bool GetHubStatus(libUsbHandle* f_poHandle);

/*!
 * Get the status of port connected to the target using the libusb communication
 *
 * \return TRUE on success
 */
bool GetPortStatus(libUsbHandle* f_poHandle,tenDevManagerUSBHost eUSB);

/*!
 * Get the handle to internal hub by scanning the connected devices in the target.
 *
 * \return 0 on success, or libusb error code on error
 */
libUsbHandle* GetInternalHubHandle(void);

/*!
 * Scan for the connected internal hub and get status of hub and ports connected to it.
 *
 */
void LibUsbScanForInternalHUB(void);

/*!
 * Scan for the connected internal hub and get the speed of Telematics box connected to it.
 * \param [in] eUsb port number of the telematics box
 * return TRUE if the device is connected and working at a full speed
 */
tBool bIsDeviceAtFullSpeed(tenDevManagerUSBHost eUsb,tBool &bIsDevicePresent);

/*!
 * Check whether the device with specified vendor and product info  is connected or not
 * \param [in] f_iVendorID device vendor id
 * \param [in] f_iProductID device product id
 * return TRUE if the device is connected
 */
tBool isDeviceConnected(uint16_t f_iVendorID, uint16_t f_iProductID);



#endif//USE_EXPERIMENTAL_PSA_HUBCOMMUNICATION

#endif // USBUTILS_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
