/*-----------------------------------------------------------------------------*
 * TimerDVM.cpp                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                   *
 * COPYRIGHT   : (c) 2012 - 2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file TimerDVM.cpp
 *
 * \brief Adapter class (Wrapper) to ease replacing osal timer functions it uses gmp frame work timer
 *
  * \version 22.12.2014, Koechling Christian (CM-AI/ECD1), version 1.0 - first implementation to hand over
 *  this to Rajeev Narayanan Sambhu (RBEI/ECS1)
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#include "Common.h"

#include "Timer.h"
#include "TimerDVM.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/TimerDVM.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * define                                                                 *
 *-----------------------------------------------------------------------------*/

//init static vector of class TimerDVM it exists to return uniqID

unsigned int TimerDVM::m_uiUsedTimerIDs = 0;

//==========================public functions of class===========================

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
TimerDVM::TimerDVM()
{
    mpCallback       = NULL;
    m_isTimerCreated = false;
    m_isTimerStarted = false;
    mTimerID         = 0;
    m_hTimerDVMID    = 0;
    m_uiUsedTimerIDs = 0;

}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
TimerDVM::~TimerDVM()
{
    mpCallback       = NULL;
    m_isTimerCreated = false;
    m_isTimerStarted = false;
    mTimerID         = 0;
    m_hTimerDVMID    = 0;
    m_uiUsedTimerIDs = 0;

}


/*-----------------------------------------------------------------------------*
 * iTimerCreate(...)                                                                  *
 *-----------------------------------------------------------------------------*/
//this function is used to handover the name only if OSLA is not used anymore further improvement is possible
int TimerDVM::iTimerCreate(tpfCallback pCallback,void* pvArg ,std::string strNameOfTimer)
{
    (void)pvArg;

    int iRet = TimerDVM::ERROR;

    m_isTimerCreated = false;

    if(m_hTimerDVMID == 0)
    {
        //-------------------------------------------------
        //name of timer: given name + address of callback function
        //-------------------------------------------------
        char numberDerivedFromCallback[100];
        strcpy(numberDerivedFromCallback,"");
        if(pCallback)
        {
            snprintf(numberDerivedFromCallback,sizeof(numberDerivedFromCallback),"_fctAddr:%zd",(intptr_t)pCallback); //z     For integer types, causes printf to expect a size_t-sized integer argument.
        }

        m_strNameOfTimer  = strNameOfTimer;
        m_strNameOfTimer += numberDerivedFromCallback;

        ETG_TRACE_USR1(("iTimerCreate: %s",m_strNameOfTimer.c_str()));

        if(true == bCalcNewUniqID(INOUT m_uiUsedTimerIDs,OUT m_hTimerDVMID))
        {
            ETG_TRACE_USR4(("iTimerCreate(OSAL): OK "));
            mpCallback       = NULL; //callback not used here is set with the set function
            m_isTimerCreated = true;
            iRet             = TimerDVM::OK;
        }
    }
    else
    {
        iRet = TimerDVM::ERROR_ALREADY_CREATED;
        ETG_TRACE_FATAL(("iTimerCreate: Already Created m_hTimerDVMID:%d",m_hTimerDVMID));
    }

     return iRet;


}


/*-----------------------------------------------------------------------------*
 * iTimerDelete(...)                                                                  *
 *-----------------------------------------------------------------------------*/
int TimerDVM::iTimerDelete()
{
    int iRet = TimerDVM::ERROR;
    if(m_hTimerDVMID != 0)
    {
        if(m_isTimerCreated)
        {
            iRet = TimerDVM::OK;
            mTimer.CancelTimer(mTimerID);
            mTimerID = 0;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("iTimerStop: Wrong handle: m_hTimerDVMID=%d ",(int)m_hTimerDVMID)); //cast to int for ETG
    }
    return iRet;
}



/*-----------------------------------------------------------------------------*
 * iTimerStop(...)                                                                  *
 *-----------------------------------------------------------------------------*/
int TimerDVM::iTimerStop()
{
    int iRet = TimerDVM::ERROR;

    if(m_hTimerDVMID != 0) //iTimerCreate has to be called first
    {
        if(m_isTimerCreated)
        {
            iRet = TimerDVM::OK;
            mTimer.CancelTimer(mTimerID);
            ETG_TRACE_FATAL(("iTimerStop: OK, mTimerID:%p",mTimerID));
        }
    }
    m_isTimerStarted = false;

    return iRet;
}



bool TimerDVM::bCalcNewUniqID(INOUT unsigned int &f_iUsedTimerIDs, OUT long int &f_hTimerDVMID)
{
    bool bRet = false;

    if(f_iUsedTimerIDs >= MAXNUM_DVMTIMERS)
    {
        ETG_TRACE_FATAL(("bCalcNewUniqID: Exceeeded max number of allowed timers"));
    }
    else
    {
        f_iUsedTimerIDs++;
        f_hTimerDVMID = (long int)f_iUsedTimerIDs;
        bRet = true;
    }

    return bRet;
}


/*-----------------------------------------------------------------------------*
 * iTimerSet(...)                                                                  *
 *-----------------------------------------------------------------------------*/
int TimerDVM::iTimerStart(long timeMilliseconds,long intervalMilliseconds, void *pObject,Timer::tCallBackFn pCallBackFn, void *userData)
{
     (void)userData;
     ETG_TRACE_USR4(("Begin: iTimerStart"));
     int iRet = TimerDVM::ERROR;

     ETG_TRACE_USR4(("Begin: iTimerStart: timeMilliseconds:%d, intervalMilliseconds:%d",timeMilliseconds,intervalMilliseconds));
    if(m_hTimerDVMID != 0)
    {
        if((0 == timeMilliseconds) && (0 == intervalMilliseconds))
        {
            ETG_TRACE_USR4(("iTimerStart: stop timer"));
            iTimerStop();
            iRet = TimerDVM::OK;
        }
        else
        {
            ETG_TRACE_USR4(("iTimerStart: start timer (m_isTimerStarted: 0x%x)",m_isTimerStarted));
            if(m_isTimerStarted)
            {
                ETG_TRACE_USR4(("iTimerStart: stop timer"));
                iTimerStop();
            }
            char name[20];
            snprintf(name,sizeof(name),"%s",m_strNameOfTimer.c_str());
            if(mTimer.StartTimer(OUT  mTimerID, timeMilliseconds , intervalMilliseconds, pObject , pCallBackFn, (void*)name))
            {
                ETG_TRACE_USR4(("iTimerStart: OK : mTimerID:%p name:%s",mTimerID,name));
                m_isTimerStarted = true;
                iRet             = TimerDVM::OK;
            }
            else
            {
                ETG_TRACE_USR4(("iTimerStart: NOK"));
                iRet = TimerDVM::ERROR;
            }

        }
    }
    ETG_TRACE_USR4(("End  : iTimerStart"));
    return iRet;

}


