/************************************************************************
* FILE:         vd_devicemanager_clienthandler_SPI.cpp
* PROJECT:      G3g
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD Devicemanager SPI client handler
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 17.02.17  | Rajeev Narayanan Sambhu (RBEI/ECO31) | initial version
*
*************************************************************************/




//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <vector>

#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_FIDIAG
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#define DIAGLIB_INCLUDE_GENERIC
#include <diaglib_if.h>
using namespace diaglib;



#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_SPI
#include "trcGenProj/Header/VD_DeviceManager_clienthandler_SPI.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST






#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"

#include "VD_DeviceManager_clienthandler_SPI.h"
#include "VD_DeviceManager_main.h"

#include <unistd.h>
#include "Utils.h"


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;


/* ************************************************************************** */
/* INITIALIZE STATIC CLASS VARIABLES                                          */
/* ************************************************************************** */


//uses hardcoded binding
#define SPI_SERVICE_FI_MAJOR_VERSION  MIDW_SMARTPHONEINTFI_C_U16_SERVICE_MAJORVERSION
#define SPI_SERVICE_FI_MINOR_VERSION  MIDW_SMARTPHONEINTFI_C_U16_SERVICE_MINORVERSION


// +++ MESSAGE MAP: enter the function IDs (FID) and the corresponding functions here.
//      the function will be called when a message with the corresponding FID arrives +++
BEGIN_MSG_MAP(vd_devicemanager_clienthandler_SPI, ahl_tclBaseWork)
   ON_MESSAGE( MIDW_SMARTPHONEINTFI_C_U16_DEVICESTATUSINFO,vHandleDeviceStatusInfoUpdate) // CCA property
   ON_MESSAGE_SVCDATA(MIDW_SMARTPHONEINTFI_C_U16_GETDEVICEINFOLIST,AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandleReceiveDeviceInfoList) // CCA  method
END_MSG_MAP()


/*************************************************************************
*
* FUNCTION: vd_devicemanager_clienthandler_SPI::vd_devicemanager_clienthandler_SPI(vd_devicemanager_tclApp* m_poMainAppApp)
*
* DESCRIPTION: constructor, creates object vd_devicemanager_clienthandler_SPI - object
*
* PARAMETER: vd_devicemanager_tclApp* m_poMainAppApp: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
vd_devicemanager_clienthandler_SPI::vd_devicemanager_clienthandler_SPI( vd_devicemanager_tclApp* poMainAppl )
    : ahl_tclBaseOneThreadClientHandler(
            /* Application Pointer          */ poMainAppl,
            /* ID of used Service           */ CCA_C_U16_SRV_SMARTPHONEINTEGRATION,
            /* MajorVersion of used Service */ SPI_SERVICE_FI_MAJOR_VERSION,
            /* MinorVersion of used Service */ SPI_SERVICE_FI_MINOR_VERSION )
      , m_poMainApp(poMainAppl)

{
    ETG_TRACE_USR4(("Begin  : vd_devicemanager_clienthandler_SPI()"));


    ETG_TRACE_USR4(("vd_devicemanager_clienthandler_SPI: Expected: SPI_SERVICE_FI_MAJOR_VERSION:%d ",(int)SPI_SERVICE_FI_MAJOR_VERSION));
    ETG_TRACE_USR4(("vd_devicemanager_clienthandler_SPI: Expected: SPI_SERVICE_FI_MINOR_VERSION:%d ",(int)SPI_SERVICE_FI_MINOR_VERSION));

    m_isServiceAvailble             = FALSE;
    m_isRetryDone = FALSE;

    vAddAutoRegisterForProperty(MIDW_SMARTPHONEINTFI_C_U16_DEVICESTATUSINFO);


    ETG_TRACE_USR4(("End  : vd_devicemanager_clienthandler_SPI()"));

}



/*************************************************************************/
/*
* FUNCTION:    vd_devicemanager_clienthandler_SPI::~vd_devicemanager_clienthandler_SPI( )
*
* DESCRIPTION: destructor: deletes the thread
*
* PARAMETER:   void
*
* RETURNVALUE: none
*/
/*************************************************************************/
vd_devicemanager_clienthandler_SPI::~vd_devicemanager_clienthandler_SPI( )
{
   tS32 s32Success;
   // Reset pointer to main application
   m_poMainApp = OSAL_NULL;
}



/*-----------------------------------------------------------------------------*
 * tVoid vOnServiceAvailable()                                                 *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::vOnServiceAvailable()
{
    ETG_TRACE_USR4(("Running: vd_devicemanager_clienthandler_SPI::vOnServiceAvailable() entered. AppID = %u.",
                    u16GetServerAppID()));
    m_isServiceAvailble = TRUE;
}

/*-----------------------------------------------------------------------------*
 * tVoid vOnServiceUnavailable()                                               *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::vOnServiceUnavailable()
{

    ETG_TRACE_USR4(("Running: vd_devicemanager_clienthandler_SPI::vOnServiceUnavailable() entered. AppID = %u.",
                    u16GetServerAppID()));
    m_isServiceAvailble = FALSE;
}


/*-----------------------------------------------------------------------------*
 * tVoid vHandleDeviceStatusInfoUpdate(...)                                               *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::vHandleDeviceStatusInfoUpdate( IN amt_tclServiceData *poMessage )
{
    ETG_TRACE_USR4(("Begin  : vHandleDeviceStatusInfoUpdate"));
    switch( poMessage->u8GetOpCode() )
    {
       case CCA_C_U8_OPCODE_STATUS:
       {
           m_isRetryDone = false;
           // extract payload
           midw_smartphoneintfi_tclMsgDeviceStatusInfoStatus oStatus;
           (tVoid)fi_tclVisitorMessage(poMessage).s32GetData( oStatus );

           tU8 u8DeviceConnectStatus     = (tU8)oStatus.DeviceStatus.enType ;         // Get connect status of the device
           ETG_TRACE_USR4(( "vHandleDeviceStatusInfoUpdate( ): Get device connect status %d", u8DeviceConnectStatus ));
           //request the device info list from SPI
           vRequestDeviceInfoList();
           break;
       }
       case CCA_C_U8_OPCODE_ERROR:
       {
          ETG_TRACE_ERR(( "vHandleDeviceStatusInfoUpdate: ERROR  OpCode error state received" ));
          break;
       }
       default:
       {
          ETG_TRACE_ERR(( "vHandleDeviceStatusInfoUpdate: ERROR: Unhandled OpCode received" ));
          break;
       }
    }
    if(! poMessage->bDelete())
    {
       ETG_TRACE_ERR(( "vHandleDeviceStatusInfoUpdate: ERROR: deleting message failed" ));
    }
    ETG_TRACE_USR4(("End  : vHandleDeviceStatusInfoUpdate"));

}

/*-----------------------------------------------------------------------------*
 * tVoid vHandleReceiveDeviceInfoList()                                               *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::vHandleReceiveDeviceInfoList( IN amt_tclServiceData *poMessage )
{
    ETG_TRACE_USR3(( "Begin ::vHandleReceiveDeviceInfoList" ));

    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oResultMsg( poMessage );
    // Create the (message related) FI data object
    midw_smartphoneintfi_tclMsgGetDeviceInfoListMethodResult oResultData;
    oResultMsg.s32GetData(oResultData);

    for ( unsigned int l_uiCount=0; l_uiCount < oResultData.DeviceInfoList.size();l_uiCount++)
    {
        tChar cSerialID[LEN_cSerialID];
        strncpy(cSerialID,(tPChar)oResultData.DeviceInfoList[l_uiCount].szSerialNumber,sizeof(cSerialID));
        cSerialID[sizeof(cSerialID)-1] = '\0'; // sim4hi: fix for Coverity CID-83541
        ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Device name: %s", (tPChar)oResultData.DeviceInfoList[l_uiCount].szDeviceName));
        ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Serial number: %s", cSerialID));
        ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Connect status: %d", oResultData.DeviceInfoList[l_uiCount].enDeviceConnectionStatus.enType));
        ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Device Category: %d", oResultData.DeviceInfoList[l_uiCount].enDeviceCategory.enType));


        //The below conditions added to fix the issue PSARCC21-1683
        //In the ticket I have attached the mail communication I have recieved from SPI. In addition to that some more additions done
        if( midw_fi_tcl_e8_DeviceConnectionStatus::FI_EN_DEV_CONNECTED == oResultData.DeviceInfoList[l_uiCount].enDeviceConnectionStatus.enType)
        {
            if(midw_fi_tcl_e8_DeviceCategory::FI_EN_DEV_TYPE_DIPO != oResultData.DeviceInfoList[l_uiCount].enDeviceCategory.enType)
            {
                midw_fi_tcl_e8_DeviceSelectionState eDeviceSelectionState = oResultData.DeviceInfoList[l_uiCount].rDeviceSelectionInfo.enDeviceSelectionState;
                ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Device Selection State: %d", eDeviceSelectionState.enType));
                if( midw_fi_tcl_e8_DeviceSelectionState::FI_EN_DEVICE_NOT_SELECTED == eDeviceSelectionState.enType )
                {
                    //if the FI_EN_DEVICE_NOT_SELECTED is received for the first time,devicemanager will request device list again
                    if(!m_isRetryDone)
                    {
                        if(!m_RetryDeviceListLock.locked())
                        {
                            tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)vd_devicemanager_clienthandler_SPI::eThread_SPIRETRYDEVICELIST, (void*)this);
                            ETG_TRACE_COMP(("vHandleReceiveDeviceInfoList: iThreadIndex:%d",iThreadIndex));
                            m_isRetryDone = true;
                        }
                        else
                        {
                            ETG_TRACE_COMP(("vHandleReceiveDeviceInfoList: Cannot spawn thread since another thread already active in geting the device list from SPI"));
                        }
                    }
                    else
                    {
                        //if the FI_EN_DEVICE_NOT_SELECTED is received even after requestdevicelist, then no need to wait for any updates and proceed with the MTP detection
                        //proceed for MTP detection
                        ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Selection state is still FI_EN_DEVICE_NOT_SELECTED : "));
                        if(0 != strlen_r(cSerialID))
                        {
                            StateTable::GetInstance()->vStopRetryForDevice(cSerialID);
                        }
                    }

                }
                else if(     ( midw_fi_tcl_e8_DeviceSelectionState::FI_EN_DEVICE_SELECTED != eDeviceSelectionState.enType ) &&
                        ( midw_fi_tcl_e8_DeviceSelectionState::FI_EN_DEVICE_SELECTION_INPROGRESS != eDeviceSelectionState.enType) &&
                        ( midw_fi_tcl_e8_DeviceSelectionState::FI_EN_DEVICE_SELECTION_ERROR != eDeviceSelectionState.enType )
                   )
                {
                    ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Stop the retrial for the device: %s",cSerialID));

                    //proceed for MTP detection
                    tChar *l_cTempSerialID = new tChar[LEN_cSerialID];
                    strncpy(l_cTempSerialID,cSerialID,LEN_cSerialID);
                    if(!m_WaitForStopRetrialLock.locked())
                    {
                        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)vd_devicemanager_clienthandler_SPI::eThread_WAITBEFORESTOPPINGRETRIAL, (void*)l_cTempSerialID);
                        ETG_TRACE_COMP(("vHandleReceiveDeviceInfoList: iThreadIndex:%d",iThreadIndex));
                    }
                    else
                    {
                        ETG_TRACE_COMP(("vHandleReceiveDeviceInfoList: Cannot spawn thread since another thread already active in stopping the retrial of device"));
                    }

                }
                else if(midw_fi_tcl_e8_DeviceSelectionState::FI_EN_DEVICE_SELECTION_ERROR == eDeviceSelectionState.enType)
                {
                    //proceed for MTP detection
                    ETG_TRACE_USR3(( "vHandleReceiveDeviceInfoList Stop the retrial for the device: %s",cSerialID));
                    if(0 != strlen_r(cSerialID))
                    {
                        StateTable::GetInstance()->vStopRetryForDevice(cSerialID);
                    }
                }



            }
        }
    }
    oResultData.vDestroy();

    ETG_TRACE_USR3(( "End ::vHandleReceiveDeviceInfoList" ));

}

/*-----------------------------------------------------------------------------*
 * tVoid vRequestDeviceInfoList()                                              *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::vRequestDeviceInfoList()
{
    ETG_TRACE_USR4(("Begin  : vRequestDeviceInfoList"));
    if(m_isServiceAvailble)
    {
        // Create object of midw_devicemanagerfi_tclMsgDevManagerGetAllConnectedDevicesMethodStart
        midw_smartphoneintfi_tclMsgGetDeviceInfoListMethodStart oFiDataObject;

        // Create FI VisitorMessage. (The FI data object will be streamed (each
        // parameter is copied individually) to shared memory.)
        fi_tclVisitorMessage oMsg(oFiDataObject,SPI_SERVICE_FI_MAJOR_VERSION);

        // Always destroy the (message related) FI data object (before leaving
        // its creation scope)
        oFiDataObject.vDestroy();

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData (oMsg,                                                   // ServiceDataMsg
                          0,                                                          // CmdCounter
                          MIDW_SMARTPHONEINTFI_C_U16_GETDEVICEINFOLIST,// Function ID
                          AMT_C_U8_CCAMSG_OPCODE_METHODSTART);                        // Opcode

        // Post the message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

        if (AIL_EN_N_NO_ERROR != enResult)
        {
            // Posting of the message has failed,
            // but it is not necessary to call 'oMsg2.bDelete()' here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that the
            // message is deleted in a case of an error.
            ETG_TRACE_ERR(( "vRequestDeviceInfoList: enPostMessage() failed for 'GetDeviceInfoList \
                        - GET' message, 'ail_tenCommunicationError' = %u", (tU32)enResult ));
        }

    }
    else
    {
        ETG_TRACE_USR3(( "vRequestDeviceInfoList: !!!!!!!SPI Service not ready!!!!!!" ));
        //tbd send the error message to servide vd_devicemanager
    }
    ETG_TRACE_USR4(("End  : vRequestDeviceInfoList"));

}

/*-----------------------------------------------------------------------------*
tVoid vd_devicemanager_clienthandler_SPI::Do(...)
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_clienthandler_SPI::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do "));
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case vd_devicemanager_clienthandler_SPI::eThread_SPIRETRYDEVICELIST: //execution tree for thread eThread_SystemVolt
        {
            Locker m_RetryDeviceListLocker(&m_RetryDeviceListLock);
            vd_devicemanager_clienthandler_SPI *l_clienthandler_SPI = (vd_devicemanager_clienthandler_SPI*)ptr;
            if(l_clienthandler_SPI)
            {
                ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_SPIRETRYDEVICELIST");
                ETG_TRACE_USR3(( "Do: Get the device list in a GMP thread" ));

                sleep(cDelaySpiRetryDeviceList);
                l_clienthandler_SPI->vRequestDeviceInfoList();

            }
            break;
        }
        case vd_devicemanager_clienthandler_SPI::eThread_WAITBEFORESTOPPINGRETRIAL: //execution tree for thread eThread_SystemVolt
        {
            Locker m_WaitForStopRetrialLocker(&m_WaitForStopRetrialLock);
            const char *cSerialID =(const char*)ptr;
            if(cSerialID)
            {
                ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_WAITBEFORESTOPPINGRETRIAL");
                ETG_TRACE_USR3(( "Do: Wait for the re enumeration of MTP device in a separate GMP thread" ));

                //proceed for MTP detection
                if(0 != strlen_r(cSerialID))
                {
                    sleep(cWaitBeforeStoppingRetrial);
                    StateTable::GetInstance()->vStopRetryForDevice(cSerialID);
                }
                delete[] cSerialID;
            }
            break;
        }
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"));
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>

