/*-----------------------------------------------------------------------------*
 * VD_DeviceManager_clienthandler_Diaglog.cpp                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file VD_DeviceManager_clienthandler_Diaglog.cpp
 *
 * \brief CCA client-handler for the service Diaglog
 *
 * \version 12.07.2011, Ramkumar Muniraj (Bosch), version 1.0 - Initial version
 * \version 12.07.2011, Ramkumar Muniraj (Bosch), version 1.1 - Added the
 *          required logic to perform DTC tests
 * \version 12.07.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager
 *          changed, ETG-traces improved, cleanup diagnosis
 * \version 19.08.2011, Ramkumar Muniraj (Bosch)   , version 1.3 - bugfixes
 * \version 14.09.2011, Ramkumar Muniraj (Bosch)   , version 1.4
 *          -# Results of enUSB_HUB_NUMBER_OF_PORT_MISMATCH and enUSB_HUB_NOT_DETECTED was
 *             sent NO_RESULT when no of ports is equal to zero, this is corrected.
 *          -# When overcurrent is undefined  NO_Result will be sent for enUSB_OVERCURRENT
 *          -# Added additional comments for trace.
 *          -# Assigned default value to u16DtcCode= 0, before using it.
 * \version 04.11.2011, Koechling, Christian (Bosch), trace adjusted
 * \version 17.11.2011, Koechling, Christian (Bosch), cleanup: 1st step to shift
 *          includes to Common.h
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 28.11.2011, Koechling, Christian (Bosch), cleanup - add headers
 * \version 28.11.2011, Sunder Negi (Montavista), Removed qDebug calls
 * \version 06.12.2011, Koechling, Christian (Bosch)
 * \version 12.12.2011, Koechling, Christian (Bosch), further care for simulating EOL-values
 * \version 06.01.2012, Koechling, Christian (Bosch), cleanup FATAL traces
 * \version 16.01.2012, Koechling, Christian (Bosch),
 *          -# in case of over current DTC check for HUB and ports not done anymore
 *          -# result of discussion with diagnosis collegues
 *          -# use OC for over current now since PPON is switched on cyclic if
 *          permanent shortcut
 * \version 18.01.2012, Koechling, Christian (Bosch),bDoUSBCircuitTest for
 *          DTC-diagnosis (GMNGA-19307)
 * \version 19.01.2012, Koechling, Christian (Bosch), use
 *          'DELAY_DIGANOSISRESULT_CYCLICDIAG  see config spec.fix due to code
 *          review with Lars Plischke
 * \version 24.01.2012, Koechling, Christian (Bosch), support DIAG_PORT_VICEVERSA
 * \version 25.01.2012, Koechling, Christian (Bosch), adjust diagnosis: sends
 *          always pass at first call; sends HUB related results even if over
 *          current variable still undefined
 * \version 30.01.2012, Koechling, Christian (Bosch), cleanup: expected HUB
 *          directly notified if recognized by UDEV
 * \version 07.02.2012, Koechling, Christian (Bosch), added semaphore care for
 *          possible race condition for clearAll diagnosis
 * \version 15.02.2012, Koechling, Christian (Bosch), Diagnosis:immediate response
 *          on clear all function call (MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT)
 * \version 27.02.2012, Koechling, Christian (Bosch), cleanup: adjusted code not
 *          to use m_bTestResultChanged
 * \version 28.02.2012, Koechling, Christian (Bosch), BUGFIX diagnopsis after test
 * \version 05.03.2012, Koechling, Christian (Bosch), cleanup: function calls and defines
 * \version 07.03.2012, Koechling, Christian (Bosch), merged MV-code Feb27 -
 *          contains issue for startup synchronisation
 * \version 10.03.2012, Koechling, Christian (Bosch), for analysis: add history
 *          of messages send to diaglog (DTC-only)
 * \version 14.03.2012, Koechling, Christian (Bosch), cleanup DIAG_PORT_VICEVERSA and BUGFIX_DIAG
 * \version 19.07.2012, Koechling, Christian (Bosch), add mechanism to log
 *          histories for analysis to USB-stick
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 * \version 15.08.2012, Fiebing, Lars (Bosch), enable Linux based simulation (LSIM)
 * \version 27.08.2012, Koechling, Christian (Bosch), trigger FillStorage fct. by PDT-diag
 * \version 25.09.2012, Koechling, Christian (Bosch), clear function of DTC-doagnosis did not work due to wrong define
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include <stdio.h>
#include "Config.h"

#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_FIDIAG
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"



#define DIAGLIB_INCLUDE_GENERIC
#include <diaglib_if.h>
using namespace diaglib;

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"
#include "debug/HistoryManager.h"
#include "interface/DeviceManagerInterface.h"


#ifdef USE_ENABLE_DEVMGR_GPIOIF
#include "GpioIf.h"
#endif //#ifdef USE_ENABLE_DEVMGR_GPIOIF

#include "WorkThread.h"
#include "diag/DiagLibHandlerGen2.h"
#include "client/VD_DeviceManager_clienthandler_Diaglog.h"
#include "service/VD_DeviceManager_service_DeviceManager.h"
#include "VD_DeviceManager_main.h"
#include "usbutils.h"

#include "config/ConfigurationManager.h"


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_DIAGLOG
#include "trcGenProj/Header/VD_DeviceManager_clienthandler_Diaglog.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * define                                                                 *
 *-----------------------------------------------------------------------------*/
/*
GM Gen3 (MY16)
ITC_USB1_OVERCURRENT       ///< DTC B124B_0B USB1 Signal Current Over Threshold
ITC_USB1_HUB_NOT_DETECTED             ///< DTC B124B_04 USB HUB 1 Open Circuit
ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH      ///< DTC B124B_11 USB HUB 1 Above Maximum Threshold
ITC_USB2_OVERCURRENT       ///< DTC B124C_0B USB2 Signal Current Over Threshold
ITC_USB2_HUB_NOT_DETECTED             ///< DTC B124C_04 USB HUB 2 Open Circuit
ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH      ///< DTC B124C_11 USB HUB 2 Above Maximum Threshold
*/


//Fault Memory Clear - vHandle_DiaglogSendNextTestResult_Status will be called from FC_Diagnosis [VD_DVM_DOC]

//for the defines look at di_middleware_server/components/vd_diaglog/common/framework/vd_diaglog_its_definitions_common.h [VD_DVM_DOC]


#ifndef ITC_MEMORY_CARD_OVERCURRENT        
#warning "ITC_MEMORY_CARD_OVERCURRENT missing - not provided by FC_DIAGNOSIS label yet"
#define ITC_MEMORY_CARD_OVERCURRENT ITC_MEMORY_CARD_OVER_CURERENT  // SD Card 1 Signal Current Over Threshold
#endif


// Version defines for used service
#define DIAGLOG_FI_MAJOR_VERSION  1
#define DIAGLOG_FI_MINOR_VERSION  0

/*-----------------------------------------------------------------------------*
 * CCA_Message MAP                                                             *
 *-----------------------------------------------------------------------------*/
BEGIN_MSG_MAP(vd_devicemanager_tclClientHandler_Diaglog, ahl_tclBaseWork)
ON_MESSAGE_SVCDATA( MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,
                    AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
                    vHandle_SaveTestResult_MethodResult)
ON_MESSAGE_SVCDATA( MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,
                    AMT_C_U8_CCAMSG_OPCODE_STATUS,
                    vHandle_DiaglogSendNextTestResult_Status)
END_MSG_MAP()

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;



/*static function because of dependencies this has to be solved later*/
/*-----------------------------------------------------------------------------*
 * void bMapToDiagHistory (...)           *
 *-----------------------------------------------------------------------------*/
static tBool  bMapToDiagHistory (OUT trDiagEvt &f_rDiagEvts, IN midw_fi_tcl_TestResultList& f_oTestResultList)
{
    ETG_TRACE_USR3 (("AddToHistory: Begin"));
    tBool bRes = TRUE;

    //init result parameter
    f_rDiagEvts.send_nr   = 0;
    f_rDiagEvts.sent_time = 0;
    for(int i=0;i<NUM_OF_DTC_ERRORS;i++)
    {
        f_rDiagEvts.last_msg[i].s8DtcResult = -1; //for analysis
    }

    ETG_TRACE_USR4 (("AddToHistory: f_oTestResultList.TestResultList.size()=%d",
                     f_oTestResultList.TestResultList.size()));

    for (unsigned int itr = 0; itr < f_oTestResultList.TestResultList.size(); itr++)
    {
        ETG_TRACE_USR4 (("AddToHistory: oTestResult.TroubleCode=0x%x",
                         f_oTestResultList.TestResultList[itr].TroubleCode));

        switch(f_oTestResultList.TestResultList[itr].TroubleCode)
        {
        /* overcurrent */
        case ITC_USB1_OVERCURRENT:
            f_rDiagEvts.last_msg[enITC_USB1_SIGNAL_OVERCURRENT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_OVERCURRENT:
            f_rDiagEvts.last_msg[enITC_USB2_SIGNAL_OVERCURRENT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_OVERCURRENT:
            f_rDiagEvts.last_msg[enITC_USB3_SIGNAL_OVERCURRENT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_OVERCURRENT:
            f_rDiagEvts.last_msg[enITC_USB4_SIGNAL_OVERCURRENT].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
            /*hub missing*/
        case ITC_USB1_HUB_NOT_DETECTED:
            f_rDiagEvts.last_msg[enITC_USB1_HUB_NOT_DETECTED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_HUB_NOT_DETECTED:
            f_rDiagEvts.last_msg[enITC_USB2_HUB_NOT_DETECTED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_HUB_NOT_DETECTED:
            f_rDiagEvts.last_msg[enITC_USB3_HUB_NOT_DETECTED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_HUB_NOT_DETECTED:
            f_rDiagEvts.last_msg[enITC_USB4_HUB_NOT_DETECTED].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
            /*ports of hub mismatch*/
        case ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH:
            f_rDiagEvts.last_msg[enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH:
            f_rDiagEvts.last_msg[enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_HUB_NUMBER_OF_PORTS_MISMATCH:
            f_rDiagEvts.last_msg[enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_HUB_NUMBER_OF_PORTS_MISMATCH:
            f_rDiagEvts.last_msg[enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
            /*SDcard*/
        case ITC_MEMORY_CARD_OVERCURRENT:
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
            {
                f_rDiagEvts.last_msg[enITC_SDC1_SIGNAL_OVERCURRENT].s8DtcResult = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            }
            else
            {
                ETG_TRACE_FATAL(("[INFO] bMapToDiagHistory: eConf_ENABLE_USE_PRM_SDCARDSIGNALS disabled"));
                        bRes = FALSE;
            }
            break;
        case ITC_USB1_POWER_SUPPLY_FAULT:
            f_rDiagEvts.last_msg[enITC_USB1_POWER_SUPPLY_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_POWER_SUPPLY_FAULT:
            f_rDiagEvts.last_msg[enITC_USB2_POWER_SUPPLY_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_POWER_SUPPLY_FAULT:
            f_rDiagEvts.last_msg[enITC_USB3_POWER_SUPPLY_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_POWER_SUPPLY_FAULT:
            f_rDiagEvts.last_msg[enITC_USB4_POWER_SUPPLY_FAULT].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
        case ITC_USB1_OPEN:
            f_rDiagEvts.last_msg[enITC_USB1_OPEN].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_OPEN:
            f_rDiagEvts.last_msg[enITC_USB2_OPEN].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_OPEN:
            f_rDiagEvts.last_msg[enITC_USB3_OPEN].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_OPEN:
            f_rDiagEvts.last_msg[enITC_USB4_OPEN].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
        case ITC_USB1_LINK_DISABLED:
            f_rDiagEvts.last_msg[enITC_USB1_LINK_DISABLED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_LINK_DISABLED:
            f_rDiagEvts.last_msg[enITC_USB2_LINK_DISABLED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_LINK_DISABLED:
            f_rDiagEvts.last_msg[enITC_USB3_LINK_DISABLED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_LINK_DISABLED:
            f_rDiagEvts.last_msg[enITC_USB4_LINK_DISABLED].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
        case ITC_USB1_SPEED_REDUCED:
            f_rDiagEvts.last_msg[enITC_USB1_SPEED_REDUCED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_SPEED_REDUCED:
            f_rDiagEvts.last_msg[enITC_USB2_SPEED_REDUCED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_SPEED_REDUCED:
            f_rDiagEvts.last_msg[enITC_USB3_SPEED_REDUCED].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_SPEED_REDUCED:
            f_rDiagEvts.last_msg[enITC_USB4_SPEED_REDUCED].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
        case ITC_USB1_DATA_LINE_FAULT:
            f_rDiagEvts.last_msg[enITC_USB1_DATA_LINE_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB2_DATA_LINE_FAULT:
            f_rDiagEvts.last_msg[enITC_USB2_DATA_LINE_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
        case ITC_USB3_DATA_LINE_FAULT:
            f_rDiagEvts.last_msg[enITC_USB3_DATA_LINE_FAULT].s8DtcResult
                    = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#ifdef USE_FOURTH_USB_CONNECTOR
        case ITC_USB4_DATA_LINE_FAULT:
            f_rDiagEvts.last_msg[enITC_USB4_DATA_LINE_FAULT].s8DtcResult
                = (tS8)f_oTestResultList.TestResultList[itr].Result.enType;
            break;
#endif
        default:
            bRes = FALSE;
            ETG_TRACE_FATAL (("ERROR: AddToHistory: oTestResult.TroubleCode=0x%x",f_oTestResultList.TestResultList[itr].TroubleCode));
        }
    }



    ETG_TRACE_USR3 (("AddToHistory: End"));

    return bRes;
}



/*-----------------------------------------------------------------------------*
 * vd_devicemanager_tclClientHandler_Diaglog(                                  *
 *       vd_devicemanager_tclApp* poMainAppl)                                  *
 *-----------------------------------------------------------------------------*/
vd_devicemanager_tclClientHandler_Diaglog::
vd_devicemanager_tclClientHandler_Diaglog(vd_devicemanager_tclApp* poMainAppl)
    : ahl_tclBaseOneThreadClientHandler(
          /* Application Pointer          */ poMainAppl,
          /* ID of used Service           */ CCA_C_U16_SRV_DIAGLOG,
          /* MajorVersion of used Service */ DIAGLOG_FI_MAJOR_VERSION,
          /* MinorVersion of used Service */ DIAGLOG_FI_MINOR_VERSION )
    , m_poMainApp(poMainAppl)
{
    ETG_TRACE_USR4(("Begin: vd_devicemanager_tclClientHandler_Diaglog() entered"));
    m_bSendFirstMessageToDiagLog = TRUE;
    m_u32CountSecondsFromStartup   = 0;
    m_u32CountSecondsUSBOpen       = 0;//bound to USB Open circuit ITC
    m_eBTAMonitorStates            = BTA_MONITOR_NOT_STARTED;
    m_u8UsbOverCurrentCount[0] = 0; // not used
    m_u8UsbOverCurrentCount[eUSB1] = 0;
    m_u8UsbOverCurrentCount[eUSB2] = 0;
    m_u8UsbOverCurrentCount[eUSB3] = 0;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_u8UsbOverCurrentCount[eUSB4] = 0;
#endif

    m_rDtc_error[enITC_USB1_SIGNAL_OVERCURRENT].u16ErrorCode = ITC_USB1_OVERCURRENT ;
    m_rDtc_error[enITC_USB1_SIGNAL_OVERCURRENT].s8DtcResult = -1;
    m_rDtc_error[enITC_USB2_SIGNAL_OVERCURRENT].u16ErrorCode = ITC_USB2_OVERCURRENT;
    m_rDtc_error[enITC_USB2_SIGNAL_OVERCURRENT].s8DtcResult =  -1;
    m_rDtc_error[enITC_USB3_SIGNAL_OVERCURRENT].u16ErrorCode = ITC_USB3_OVERCURRENT;
    m_rDtc_error[enITC_USB3_SIGNAL_OVERCURRENT].s8DtcResult =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_SIGNAL_OVERCURRENT].u16ErrorCode = ITC_USB4_OVERCURRENT;
    m_rDtc_error[enITC_USB4_SIGNAL_OVERCURRENT].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_HUB_NOT_DETECTED].u16ErrorCode = ITC_USB1_HUB_NOT_DETECTED;
    m_rDtc_error[enITC_USB1_HUB_NOT_DETECTED].s8DtcResult =  -1;
    m_rDtc_error[enITC_USB2_HUB_NOT_DETECTED].u16ErrorCode = ITC_USB2_HUB_NOT_DETECTED;
    m_rDtc_error[enITC_USB2_HUB_NOT_DETECTED].s8DtcResult =  -1;
    m_rDtc_error[enITC_USB3_HUB_NOT_DETECTED].u16ErrorCode = ITC_USB3_HUB_NOT_DETECTED;
    m_rDtc_error[enITC_USB3_HUB_NOT_DETECTED].s8DtcResult =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_HUB_NOT_DETECTED].u16ErrorCode = ITC_USB4_HUB_NOT_DETECTED;
    m_rDtc_error[enITC_USB4_HUB_NOT_DETECTED].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH].u16ErrorCode = ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH;
    m_rDtc_error[enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult =  -1;
    m_rDtc_error[enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH].u16ErrorCode = ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH;
    m_rDtc_error[enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult =  -1;
    m_rDtc_error[enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH].u16ErrorCode = ITC_USB3_HUB_NUMBER_OF_PORTS_MISMATCH;
    m_rDtc_error[enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH].u16ErrorCode = ITC_USB4_HUB_NUMBER_OF_PORTS_MISMATCH;
    m_rDtc_error[enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
#endif


    m_rDtc_error[enITC_USB1_POWER_SUPPLY_FAULT].u16ErrorCode = ITC_USB1_POWER_SUPPLY_FAULT ;
    m_rDtc_error[enITC_USB1_POWER_SUPPLY_FAULT].s8DtcResult  = -1;
    m_rDtc_error[enITC_USB2_POWER_SUPPLY_FAULT].u16ErrorCode = ITC_USB2_POWER_SUPPLY_FAULT;
    m_rDtc_error[enITC_USB2_POWER_SUPPLY_FAULT].s8DtcResult  =  -1;
    m_rDtc_error[enITC_USB3_POWER_SUPPLY_FAULT].u16ErrorCode = ITC_USB3_POWER_SUPPLY_FAULT;
    m_rDtc_error[enITC_USB3_POWER_SUPPLY_FAULT].s8DtcResult  =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_POWER_SUPPLY_FAULT].u16ErrorCode = ITC_USB4_POWER_SUPPLY_FAULT;
    m_rDtc_error[enITC_USB4_POWER_SUPPLY_FAULT].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_OPEN].u16ErrorCode = ITC_USB1_OPEN ;
    m_rDtc_error[enITC_USB1_OPEN].s8DtcResult  = -1;
    m_rDtc_error[enITC_USB2_OPEN].u16ErrorCode = ITC_USB2_OPEN;
    m_rDtc_error[enITC_USB2_OPEN].s8DtcResult  =  -1;
    m_rDtc_error[enITC_USB3_OPEN].u16ErrorCode = ITC_USB3_OPEN;
    m_rDtc_error[enITC_USB3_OPEN].s8DtcResult  =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_OPEN].u16ErrorCode = ITC_USB4_OPEN;
    m_rDtc_error[enITC_USB4_OPEN].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_LINK_DISABLED].u16ErrorCode = ITC_USB1_LINK_DISABLED ;
    m_rDtc_error[enITC_USB1_LINK_DISABLED].s8DtcResult  = -1;
    m_rDtc_error[enITC_USB2_LINK_DISABLED].u16ErrorCode = ITC_USB2_LINK_DISABLED;
    m_rDtc_error[enITC_USB2_LINK_DISABLED].s8DtcResult  =  -1;
    m_rDtc_error[enITC_USB3_LINK_DISABLED].u16ErrorCode = ITC_USB3_LINK_DISABLED;
    m_rDtc_error[enITC_USB3_LINK_DISABLED].s8DtcResult  =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_LINK_DISABLED].u16ErrorCode = ITC_USB4_LINK_DISABLED;
    m_rDtc_error[enITC_USB4_LINK_DISABLED].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_SPEED_REDUCED].u16ErrorCode = ITC_USB1_SPEED_REDUCED ;
    m_rDtc_error[enITC_USB1_SPEED_REDUCED].s8DtcResult  = -1;
    m_rDtc_error[enITC_USB2_SPEED_REDUCED].u16ErrorCode = ITC_USB2_SPEED_REDUCED;
    m_rDtc_error[enITC_USB2_SPEED_REDUCED].s8DtcResult  =  -1;
    m_rDtc_error[enITC_USB3_SPEED_REDUCED].u16ErrorCode = ITC_USB3_SPEED_REDUCED;
    m_rDtc_error[enITC_USB3_SPEED_REDUCED].s8DtcResult  =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_SPEED_REDUCED].u16ErrorCode = ITC_USB4_SPEED_REDUCED;
    m_rDtc_error[enITC_USB4_SPEED_REDUCED].s8DtcResult = -1;
#endif

    m_rDtc_error[enITC_USB1_DATA_LINE_FAULT].u16ErrorCode = ITC_USB1_DATA_LINE_FAULT;
    m_rDtc_error[enITC_USB1_DATA_LINE_FAULT].s8DtcResult  = -1;
    m_rDtc_error[enITC_USB2_DATA_LINE_FAULT].u16ErrorCode = ITC_USB2_DATA_LINE_FAULT;
    m_rDtc_error[enITC_USB2_DATA_LINE_FAULT].s8DtcResult  =  -1;
    m_rDtc_error[enITC_USB3_DATA_LINE_FAULT].u16ErrorCode = ITC_USB3_DATA_LINE_FAULT;
    m_rDtc_error[enITC_USB3_DATA_LINE_FAULT].s8DtcResult  =  -1;
#ifdef USE_FOURTH_USB_CONNECTOR
    m_rDtc_error[enITC_USB4_DATA_LINE_FAULT].u16ErrorCode = ITC_USB4_DATA_LINE_FAULT;
    m_rDtc_error[enITC_USB4_DATA_LINE_FAULT].s8DtcResult = -1;
#endif

    vAddAutoRegisterForProperty(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT);

    DIAGCLIENTHISTORY.ClearHistory();

    m_bTimerStarted      = FALSE; 
    m_bTimerSdcOcStarted = FALSE; 

    m_Timer.bInit( this );
    m_TimerSdcOc.bInit( this );

    m_bFirstTransmission = TRUE;

    ETG_TRACE_USR4(("End  : vd_devicemanager_tclClientHandler_Diaglog()"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
vd_devicemanager_tclClientHandler_Diaglog::~vd_devicemanager_tclClientHandler_Diaglog(tVoid)
{
    ETG_TRACE_USR4(("Running: ~vd_devicemanager_tclClientHandler_Diaglog() entered."));

    m_Timer.bStop();               //lint !e1551 Warning 1551;Function may throw exception
    m_Timer.bDestroy();            //lint !e1551 Warning 1551;Function may throw exception

    m_TimerSdcOc.bStop();          //lint !e1551 Warning 1551;Function may throw exception
    m_TimerSdcOc.bDestroy();       //lint !e1551 Warning 1551;Function may throw exception

    m_poMainApp = NULL;
}

/*-----------------------------------------------------------------------------*
 * tVoid vOnServiceAvailable()                                                 *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vOnServiceAvailable()
{
    ETG_TRACE_USR4(("Running: vOnServiceAvailable() entered. AppID = %u.",
                    u16GetServerAppID()));
}

/*-----------------------------------------------------------------------------*
 * tVoid vOnServiceUnavailable()                                               *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vOnServiceUnavailable()
{
    ETG_TRACE_USR4(("Running: vOnServiceUnavailable() entered. AppID = %u.",
                    u16GetServerAppID()));
}

/*-----------------------------------------------------------------------------*
 * tVoid vSendResult(midw_diaglogfi_tclMsgSaveTestResultMethodStart &ms)       *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vSendResult(
        OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart &ms)
{
    ETG_TRACE_USR4(("Begin: vSendResult"));

    if (m_poMainApp != NULL) // Send only if DTC result is changed
    {
        ETG_TRACE_USR4 ((" vSendResult - Change in Result, Sending the message to Diaglog "));

        
        if(m_bFirstTransmission==FALSE)
        {
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_DTC_MAP_TCU_OC_PORT3))
            {
				//SPECIAL MAPPING FOR AIVI
                ETG_TRACE_COMP(("vSendResult: eConf_ENABLE_DTC_MAP_TCU_OC_PORT3:DVM_ON: MAP USB2 to ITC_USB3_OVERCURRENT"));
                unsigned int numOfUSB3OVercurrent = 0;
                for (unsigned int itr = 0; itr < ms.TestResultList.TestResultList.size(); itr++)
                {
                     if(ms.TestResultList.TestResultList[itr].TroubleCode == ITC_USB2_OVERCURRENT)
                     {
                         ETG_TRACE_COMP(("vDoUSBCircuitTest: ITC_USB2_OVERCURRENT changed to ITC_USB3_OVERCURRENT"));
                         ms.TestResultList.TestResultList[itr].TroubleCode = ITC_USB3_OVERCURRENT;
                     }
                     if(ms.TestResultList.TestResultList[itr].TroubleCode == ITC_USB3_OVERCURRENT)
                     {
                         numOfUSB3OVercurrent++;
                     }
                }
                if(numOfUSB3OVercurrent > 1)
                {
                     ETG_TRACE_FATAL(("[ERROR] vSendResult:numOfUSB3OVercurrent: %d ",numOfUSB3OVercurrent));
                }
            }
        }
        else
        {
            m_bFirstTransmission = FALSE;
        }
		
	
        //-> add to history
        trDiagEvt rDiagEvts;
        if(bMapToDiagHistory (OUT rDiagEvts, IN ms.TestResultList))
        {
            DIAGCLIENTHISTORY.AddToHistory(rDiagEvts);
        }
        //<-

		fi_tclVisitorMessage oMethodStart(ms);
        // Destroy the FI data object before leaving its creation scope
        ms.vDestroy();

        vInitServiceData( oMethodStart,            // ServiceDataMsg
                          0,                                     // CmdCounter
                          MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,   // Function ID
                          AMT_C_U8_CCAMSG_OPCODE_METHODSTART     // Opcode
                          );

        // Post message
        ail_tenCommunicationError enResult
                = m_poMainApp->enPostMessage(&oMethodStart, TRUE);

        if (enResult == AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_USR4 ((" vSendToDiag - Sending message to Diaglog is OK"));
        }
        else
        {
            ETG_TRACE_USR4 ((" vSendToDiag - Error Sending message to Diaglog is NOT OK"));
        }

    }  // if((m_bTestResultChanged)&&(m_poMainApp!=NULL))
    else
    {
        ETG_TRACE_USR4 ((" vSendResult - No Change in Result, message is not sent."));
    }  // if-else if((m_bTestResultChanged)&&(m_poMainApp!=NULL))
    ETG_TRACE_USR4(("End  : vSendToDiag"));
}



/*-----------------------------------------------------------------------------*
 * tVoid vProcessTimer(tU16 u16TimerId)                                        *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vProcessTimer(IN tU16 u16TimerId)
{
    ETG_TRACE_USR4(("Begin: vProcessTimer() entered. TimerID = %u.",u16TimerId));
    midw_diaglogfi_tclMsgSaveTestResultMethodStart ms;
/*
     midw_fi_tcl_TestResult     oTestResult;
     ms.TroubleCode = 7936;
     ms.Re<sult.enType = (midw_fi_tcl_e8_TestResult::tenType)2;

    //add the oTestResult into TestResultList
    ms.TestResultList.TestResultList.push_back(oTestResult);

    midw_diaglogfi_tclMsgSaveTestResultMethodStart ms;
*/
    //===============================================
    //initial transmission at startup
    //===============================================
    if(TRUE == m_bSendFirstMessageToDiagLog)
    {
        m_eBTAMonitorStates = BTA_MONITOR_NOT_STARTED;
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
        vDoSendFirstMessageToDiagLog();
#else
        vDoSendFirstMessageToDiagLog(&ms);
#endif
        m_bSendFirstMessageToDiagLog = FALSE;
    }
    //===============================================
    //Messages after initial transmission
    //===============================================
    else
    {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
        vDoCircuitTestAndSendMessageToDiagLog(u16TimerId);
#else
        vDoCircuitTestAndSendMessageToDiagLog(&ms,u16TimerId);
#endif
    }

    ETG_TRACE_USR4(("End  : vProcessTimer      "));

}  //  tVoid vd_devicemanager_tclClientHandler_Diaglog::vProcessTimer(tU16 u16T..



//extra function added to prevent changing code at so many locations within this file. Refactoring done with defines to mark it and to be able to release it disabled for upcoming release if partial finished
tU8 vd_devicemanager_tclClientHandler_Diaglog::u8SetDiagLogDTCCode(
        IN tenDevManagerUSBHost usb,
        IN tenUsbTestType enDtcTestType,
        IN tU8 u8Result
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
        ,OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData
#endif
        )
{
    tBool bServiceIsAvailable=FALSE;
    tU8   u8Changed  = 0; //o means not changed 1 means changed
    if(bIfServiceAvailable())
    {
         bServiceIsAvailable=TRUE;
    }
 

#ifdef USE_SHIFT_PARTOFDIAG_TO_ADAPTER
    m_DiagnosisTroubleCoder.u8SetDiagLogDTCCode(usb,enDtcTestType,u8Result,bServiceIsAvailable,m_rDtc_error,m_ListCyclicDiagResults);
#else //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
    u8Changed = u8SetDiagLogDTCCode(usb,enDtcTestType,u8Result,
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
                        poTestData,
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                        bServiceIsAvailable);

#endif //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
    return u8Changed;
}

#ifndef USE_SHIFT_PARTOFDIAG_TO_ADAPTER
/*-----------------------------------------------------------------------------*
 * tU8 u8SetDiagLogDTCCode(tenDevManagerUSBHost usb,                           *
 *          tenUsbTestType enDtcTestType, tU8 u8Result,                        *
 *          midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData)      *
 *-----------------------------------------------------------------------------*/
tU8 vd_devicemanager_tclClientHandler_Diaglog::u8SetDiagLogDTCCode(
        IN tenDevManagerUSBHost usb,
        IN tenUsbTestType enDtcTestType,
        IN tU8 u8Result
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
        ,OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData
#endif
        ,tBool bServiceIsAvailable )
{
    tU8   u8Index    = 0; ///@todo check if there is a better way to avoid: Variable 'u8Index' may not have been initialized
    tU8   u8Changed  = 0; //o means not changed 1 means changed
    tU16  u16DtcCode = 0;
    tBool bvalid     = TRUE;
    tBool bTestResultChanged = FALSE;

    ETG_TRACE_USR4 (("Begin: u8SetDiagLogDTCCode:: USB%1d DTCTestType %d u8Result=%d",
                     (tInt)usb, ETG_CENUM(tenUsbTestType,enDtcTestType), u8Result));

#ifdef USE_DECOUPLE_CYCLICDIAGCCA
    if ( bServiceIsAvailable == TRUE) // check for VD_Diaglog is availability
#else
    if ((bServiceIsAvailable == TRUE)&& (poTestData!=NULL)) // check for VD_Diaglog is availability
#endif //USE_DECOUPLE_CYCLICDIAGCCA
    {
#ifdef USE_FOURTH_USB_CONNECTOR
        if((usb<=ARRAYSIZEFORUSB) &&  (eUSB1 <=usb) && (usb<=eUSB4))
#else
        if((usb<=ARRAYSIZEFORUSB)&& (eUSB1 <=usb) && (usb<=eUSB3))
#endif
        {
            switch(enDtcTestType)
            {
            //=================================
            case enUSB_OVERCURRENT:
                //=================================

                //@todo replace switch later-----------------------
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_SIGNAL_OVERCURRENT;
                    u16DtcCode =   ITC_USB1_OVERCURRENT;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_SIGNAL_OVERCURRENT;
                    u16DtcCode =   ITC_USB2_OVERCURRENT;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_SIGNAL_OVERCURRENT;
                    u16DtcCode =   ITC_USB3_OVERCURRENT;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_SIGNAL_OVERCURRENT;
                    u16DtcCode = ITC_USB4_OVERCURRENT;
                    break;
#endif
                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break ;
                //=================================
            case enUSB_HUB_NOT_DETECTED:
                //=================================
                //@todo replace switch later-----------------------
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_HUB_NOT_DETECTED;
                    u16DtcCode =   ITC_USB1_HUB_NOT_DETECTED;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_HUB_NOT_DETECTED;
                    u16DtcCode =   ITC_USB2_HUB_NOT_DETECTED;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_HUB_NOT_DETECTED;
                    u16DtcCode =   ITC_USB3_HUB_NOT_DETECTED;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_HUB_NOT_DETECTED;
                    u16DtcCode = ITC_USB4_HUB_NOT_DETECTED;
                    break;
#endif
                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;
                //=================================
            case enUSB_HUB_NUMBER_OF_PORT_MISMATCH:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH;
                    u16DtcCode =   ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH;
                    u16DtcCode =   ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH;
                    u16DtcCode =   ITC_USB3_HUB_NUMBER_OF_PORTS_MISMATCH;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH;
                    u16DtcCode = ITC_USB4_HUB_NUMBER_OF_PORTS_MISMATCH;
                    break;
#endif
                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;

                //=================================
            case enUSB_POWER_SUPPLY_FAULT:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_POWER_SUPPLY_FAULT;
                    u16DtcCode =   ITC_USB1_POWER_SUPPLY_FAULT;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_POWER_SUPPLY_FAULT;
                    u16DtcCode =   ITC_USB2_POWER_SUPPLY_FAULT;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_POWER_SUPPLY_FAULT;
                    u16DtcCode =   ITC_USB3_POWER_SUPPLY_FAULT;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_POWER_SUPPLY_FAULT;
                    u16DtcCode = ITC_USB4_POWER_SUPPLY_FAULT;
                    break;
#endif //USE_FOURTH_USB_CONNECTOR

                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;
                //=================================
            case enUSB_OPEN:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_OPEN;
                    u16DtcCode =   ITC_USB1_OPEN;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_OPEN;
                    u16DtcCode =   ITC_USB2_OPEN;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_OPEN;
                    u16DtcCode =   ITC_USB3_OPEN;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_OPEN;
                    u16DtcCode = ITC_USB4_OPEN;
                    break;
#endif //USE_FOURTH_USB_CONNECTOR

                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;


                //=================================
            case enUSB_LINK_DISABLED:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_LINK_DISABLED;
                    u16DtcCode =   ITC_USB1_LINK_DISABLED;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_LINK_DISABLED;
                    u16DtcCode =   ITC_USB2_LINK_DISABLED;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_LINK_DISABLED;
                    u16DtcCode =   ITC_USB3_LINK_DISABLED;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_LINK_DISABLED;
                    u16DtcCode = ITC_USB4_LINK_DISABLED;
                    break;
#endif //USE_FOURTH_USB_CONNECTOR

                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;

                //=================================
            case enUSB_SPEED_REDUCED:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_SPEED_REDUCED;
                    u16DtcCode =   ITC_USB1_SPEED_REDUCED;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_SPEED_REDUCED;
                    u16DtcCode =   ITC_USB2_SPEED_REDUCED;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_SPEED_REDUCED;
                    u16DtcCode =   ITC_USB3_SPEED_REDUCED;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_SPEED_REDUCED;
                    u16DtcCode = ITC_USB4_SPEED_REDUCED;
                    break;
#endif //USE_FOURTH_USB_CONNECTOR

                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;

            case enITC_DATA_LINE_FAULT:
                //=================================
                switch(usb)
                {
                case eUSB1:
                    u8Index    = enITC_USB1_DATA_LINE_FAULT;
                    u16DtcCode = ITC_USB1_DATA_LINE_FAULT;
                    break;
                case eUSB2:
                    u8Index    = enITC_USB2_DATA_LINE_FAULT;
                    u16DtcCode = ITC_USB2_DATA_LINE_FAULT;
                    break;
                case eUSB3:
                    u8Index    = enITC_USB3_DATA_LINE_FAULT;
                    u16DtcCode = ITC_USB3_DATA_LINE_FAULT;
                    break;
#ifdef USE_FOURTH_USB_CONNECTOR
                case eUSB4:
                    u8Index = enITC_USB4_DATA_LINE_FAULT;
                    u16DtcCode = ITC_USB4_DATA_LINE_FAULT;
                    break;
#endif //USE_FOURTH_USB_CONNECTOR

                default:
                    ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d", (tInt)usb));
                    DVM_NORMAL_M_ASSERT_ALWAYS();
                    break;
                }
                //------------------------------------------------
                break;


                //=================================
            default:
                //=================================
                bvalid = FALSE;
                ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode enDtcTestType %d is not recognized",enDtcTestType));
                ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode enDtcTestType %d is not recognized",enDtcTestType));
                DVM_NORMAL_M_ASSERT_ALWAYS();
            } // switch(enDtcTestType)
            if(TRUE ==bvalid)
            {
                if (m_rDtc_error[u8Index].s8DtcResult != u8Result)
                {
                    m_rDtc_error[u8Index].s8DtcResult = (tS8)u8Result;     // cast is OK because u8Result has values from 0 'FI_EN_PASSED' to 7 'FI_EN_ENABLE'
                    bTestResultChanged = TRUE;
                }
            }
        }
        else   // if( (eUSB1 <=usb) && (usb<=eUSB2))
        {
            ETG_TRACE_FATAL((" [ERROR]: u8SetDiagLogDTCCode usb:%d out of bound", (tInt)usb));
            ETG_TRACE_ERRMEM(("[ERROR]: u8SetDiagLogDTCCode usb:%d out of bound", (tInt)usb));
            DVM_NORMAL_M_ASSERT_ALWAYS();
        }

        if (bTestResultChanged)  //@todo cleanup this variable with later release not necessary anymore
        {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
#ifdef USE_SHIFT_PARTOFDIAG_TO_ADAPTER
            m_DiagnosisTroubleCoder.vStoreResultInVector(u16DtcCode,u8Result,m_ListCyclicDiagResults)
#else //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
            vStoreResultInVector(u16DtcCode,u8Result,m_ListCyclicDiagResults);
#endif //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
#else
            midw_fi_tcl_TestResult     oTestResult;
            oTestResult.TroubleCode = u16DtcCode;
            //fill this oTestResult with result type from data
            oTestResult.Result.enType = (midw_fi_tcl_e8_TestResult::tenType)u8Result;

            //add the oTestResult into TestResultList
            poTestData->TestResultList.TestResultList.push_back(oTestResult);
#endif

            u8Changed = 1;
            ETG_TRACE_COMP(("*******************************************************************************"));
            ETG_TRACE_COMP(("u8SetDiagLogDTCCode: USB%1d DTCTestType=%d u8Result=%d",
                            (tInt)usb,
                            ETG_CENUM(tenUsbTestType,enDtcTestType),
                            u8Result
                            /*ETG_CENUM(midw_fi_tcl_e8_TestResult::tenType,u8Result)*/));
            switch(u8Index)
            {
            /*overcurrent*/
            case enITC_USB1_SIGNAL_OVERCURRENT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_SIGNAL_OVERCURRENT : u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_SIGNAL_OVERCURRENT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_SIGNAL_OVERCURRENT : u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_SIGNAL_OVERCURRENT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB3_SIGNAL_OVERCURRENT : u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_SIGNAL_OVERCURRENT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB4_SIGNAL_OVERCURRENT : u8Result=0x%x ", u8Result));
                break;
#endif

                /*hub missing or not*/
            case enITC_USB1_HUB_NOT_DETECTED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_HUB_NOT_DETECTED u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_HUB_NOT_DETECTED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_HUB_NOT_DETECTED u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_HUB_NOT_DETECTED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB3_HUB_NOT_DETECTED u8Result=0x%x ",u8Result)); //the codes came from Gen2 check if this is still valid
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_HUB_NOT_DETECTED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB4_HUB_NOT_DETECTED u8Result=0x%x ", u8Result)); 
                break;
#endif

                /*hub number of port mismatch*/
            case enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH for  u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH for  u8Result=0x%x ", u8Result));
                break;
#endif

                /*power supply fault*/
            case enITC_USB1_POWER_SUPPLY_FAULT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_POWER_SUPPLY_FAULT : u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_POWER_SUPPLY_FAULT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_POWER_SUPPLY_FAULT : u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_POWER_SUPPLY_FAULT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB3_POWER_SUPPLY_FAULT : u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_POWER_SUPPLY_FAULT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeenITC_USB4_POWER_SUPPLY_FAULT : u8Result=0x%x ", u8Result));
                break;
#endif

                /*open circuit*/
            case enITC_USB1_OPEN:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:ITC_USB1_OPEN : u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_OPEN:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:ITC_USB2_OPEN : u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_OPEN:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeITC_USB3_OPEN : u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_OPEN:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCodeITC_USB4_OPEN : u8Result=0x%x ", u8Result));
                break;
#endif

                /*Link disabled*/
            case enITC_USB1_LINK_DISABLED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_LINK_DISABLED : u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_LINK_DISABLED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_LINK_DISABLED : u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_LINK_DISABLED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB3_LINK_DISABLED : u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_LINK_DISABLED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB4_LINK_DISABLED : u8Result=0x%x ", u8Result));
                break;
#endif

                /*Speed reduced*/
            case enITC_USB1_SPEED_REDUCED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_SPEED_REDUCED : u8Result=0x%x",u8Result));
                break;
            case enITC_USB2_SPEED_REDUCED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_SPEED_REDUCED : u8Result=0x%x",u8Result));
                break;
            case enITC_USB3_SPEED_REDUCED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB3_SPEED_REDUCED : u8Result=0x%x ",u8Result));
                break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_SPEED_REDUCED:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB4_SPEED_REDUCED : u8Result=0x%x ", u8Result));
                break;
#endif

                /*data line fault*/
            case enITC_USB1_DATA_LINE_FAULT:
                  ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB1_DATA_LINE_FAULT : u8Result=0x%x",u8Result));
                  break;
            case enITC_USB2_DATA_LINE_FAULT:
                  ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB2_DATA_LINE_FAULT : u8Result=0x%x",u8Result));
                  break;
            case enITC_USB3_DATA_LINE_FAULT:
                  ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB3_DATA_LINE_FAULT : u8Result=0x%x ",u8Result));
                  break;
#ifdef USE_FOURTH_USB_CONNECTOR
            case enITC_USB4_DATA_LINE_FAULT:
                ETG_TRACE_COMP(("u8SetDiagLogDTCCode:enITC_USB4_DATA_LINE_FAULT : u8Result=0x%x ", u8Result));
                break;
#endif

            default:
                ETG_TRACE_ERR(("u8SetDiagLogDTCCode: Unknow u8Index to ITC u8Result=0x%x",u8Result));
                break;
            }
            ETG_TRACE_COMP(("*******************************************************************************"));

        }
        else
        {
            ETG_TRACE_USR4 ((" u8SetDiagLogDTCCode Test result has not changed"));
        }
    }
    else
    {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
        ETG_TRACE_ERR (("Error u8SetDiagLogDTCCode Service is Not Available"));
#else
        ETG_TRACE_ERR (("Error u8SetDiagLogDTCCode Service is Not Available/poTestData==NULL "));
#endif
    }

    ETG_TRACE_USR4 (("End  : u8SetDiagLogDTCCode"));

    return u8Changed;
}  //  tVoid vd_devicemanager_tclClientHandler_Diaglog::vSetDiagLogDTCCode(tU..

#endif //#ifndef USE_SHIFT_PARTOFDIAG_TO_ADAPTER

/*-----------------------------------------------------------------------------*
 * tU8 u8SetDiagLogItcSdc(tenSlot eSdcSlot, tU8 u8Result                       *
 *                midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData)*
 *-----------------------------------------------------------------------------*/
tU8 vd_devicemanager_tclClientHandler_Diaglog::u8SetDiagLogItcSdc( tenSlot eSdcSlot, tU8 u8Result
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
, midw_diaglogfi_tclMsgSaveTestResultMethodStart* poTestData 
#endif
)
{
    tU8   u8Index            = 0; ///@todo check if there is a better way to avoid: Variable 'u8Index' may not have been initialized
    tU16  u16DtcCode         = 0;
    tBool bvalid             = TRUE;
    tU8   u8Changed          = 0; //o means not changed 1 means changed

    ETG_TRACE_USR4(( "Begin: u8SetDiagLogItcSdc:: SD Card: %d, u8Result=%d", ETG_ENUM(DVM_CARD_SLOT, eSdcSlot), u8Result ));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
    if(  bIfServiceAvailable()) // check for VD_Diaglog is availability
#else
    if( ( bIfServiceAvailable() ) && poTestData ) // check for VD_Diaglog is availability
#endif
    {
        // Switch SD Card slot to get the index to ITC and the right ITC code
        switch( eSdcSlot )
        {
        case eCard0:
            u8Index    = enITC_SDC1_SIGNAL_OVERCURRENT;
            u16DtcCode =   ITC_MEMORY_CARD_OVERCURRENT;
            break;
        default:
            bvalid = FALSE;
            ETG_TRACE_FATAL(( "u8SetDiagLogItcSdc: ERROR: (SD Card: %d) wrong value - check code should not happen at all", eSdcSlot ));
            break;
        }
        // check with bvalid ITC and SD Csrd slot is consistent and if new test result is different to previous one
        if( bvalid && ( u8Result != m_rDtc_error[u8Index].s8DtcResult ) )
        {
            m_rDtc_error[u8Index].s8DtcResult = (tS8)u8Result;     // cast is OK because u8Result has values from 0 'FI_EN_PASSED' to 7 'FI_EN_ENABLE'
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
#ifdef USE_SHIFT_PARTOFDIAG_TO_ADAPTER
            m_DiagnosisTroubleCoder.vStoreResultInVector(u16DtcCode,u8Result, m_ListCyclicDiagResults);
#else //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
            vStoreResultInVector(u16DtcCode,u8Result, m_ListCyclicDiagResults);
#endif //USE_SHIFT_PARTOFDIAG_TO_ADAPTER
#else //USE_DECOUPLE_CYCLICDIAGCCA
            // Prepare CCA message to VD DiagLog
            midw_fi_tcl_TestResult     oTestResult;
            oTestResult.TroubleCode = u16DtcCode;
            //fill this oTestResult with result type from data
            oTestResult.Result.enType = (midw_fi_tcl_e8_TestResult::tenType)u8Result;
            //add the oTestResult into TestResultList
            poTestData->TestResultList.TestResultList.push_back(oTestResult);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
            u8Changed = 1;    // set return value to signal that there is something to send to VD DiagLog
            ETG_TRACE_COMP(( "u8SetDiagLogItcSdc: SD Card: %d, u8Result=%d", ETG_ENUM(DVM_CARD_SLOT, eSdcSlot), ETG_ENUM(DIAGLOG_STATUS, u8Result) ));
        }
        else
        {
            ETG_TRACE_USR4(( " u8SetDiagLogItcSdc Test result has not changed" ))
        }
    }
    else
    {
        ETG_TRACE_ERR (( "u8SetDiagLogItcSdc: Error: Service is Not Available/poTestData==NULL " ));
    }
    ETG_TRACE_USR4(( "End: u8SetDiagLogItcSdc" ));
    return u8Changed;
}

/*-----------------------------------------------------------------------------*
 * tVoid vDoUSBCircuitTest(tenDevManagerUSBHost usb,                           *
 *          midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData,      *
 *          tU8 &u8NumOfChanges)                                               *
 *-----------------------------------------------------------------------------*/
//@todo functions needs to be refactored - is to long - dividing it to sub functions would be helpful
tVoid vd_devicemanager_tclClientHandler_Diaglog::vDoUSBCircuitTest(
        IN tenDevManagerUSBHost usb,
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
        OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart   *poTestData,
#endif
        OUT tU8 &u8NumOfChanges)
{
    tBool bDiagOK   = FALSE;
    tBool bTestHUB  = FALSE;
    tBool bSuppressITCOverCurrentOfCritVolt  = FALSE;    
    tBool bTelematicsBoxPresent = FALSE;
    bool bIsBTAAlive = false;

    ETG_TRACE_USR4 (("Begin: bDoUSBCircuitTest USB%d",usb));
    ETG_TRACE_USR1(("------------------------------------------------------------------------"));
    ETG_TRACE_USR1(("                     USB%d CYCLIC DIAGNOSIS                              ",usb));
    ETG_TRACE_USR1(("------------------------------------------------------------------------"));



    //---------------------------------------------
    //wait until Hub has been  has been recognized
    //---------------------------------------------
    if(m_u32CountSecondsFromStartup<= DELAY_DIGANOSISRESULT_CYCLICDIAG)
    {
        m_u32CountSecondsFromStartup++;
    }
    if(FALSE == m_bSendFirstMessageToDiagLog){
#ifdef USE_FOURTH_USB_CONNECTOR
        //Class variable to hold HUB1,HUB2,HUB3,HUB4 Diagnosis data
        CUSBDiagnosis oUSBDiagnosisHUB1,oUSBDiagnosisHUB2,oUSBDiagnosisHUB3,oUSBDiagnosisHUB4;
#else
        //Class variable to hold HUB1,HUB2,HUB3 Diagnosis data
        CUSBDiagnosis oUSBDiagnosisHUB1,oUSBDiagnosisHUB2,oUSBDiagnosisHUB3;
#endif
        CUSBDiagnosis *poUSBDiagnosisHUB[ARRAYSIZEFORUSB];
        poUSBDiagnosisHUB[0]=NULL;
        poUSBDiagnosisHUB[eUSB1]=&oUSBDiagnosisHUB1;
        poUSBDiagnosisHUB[eUSB2]=&oUSBDiagnosisHUB2;
        poUSBDiagnosisHUB[eUSB3]=&oUSBDiagnosisHUB3;
#ifdef USE_FOURTH_USB_CONNECTOR
        poUSBDiagnosisHUB[eUSB4]=&oUSBDiagnosisHUB4;
#endif

        if (m_poMainApp == NULL) ///@ todo pointer check not necessary
        {
            ETG_TRACE_ERR ((" DTC Test is not possible m_poMainApp==NULL "));
            ETG_TRACE_USR4 (("<--- Exit bDoUSBCircuitTest"));
        }
        else
        {
            if (NULL != m_poMainApp->m_poWorkThread) ///@todo
            {
                /*configuration dependency*/
                tBool bGetDiag = TRUE;

                //see ConfigurationBase.h description of eConf_ENABLE_DIAGLOG_DTC_TRIGGERED_HWSIGNALS and eConf_ENABLE_CYCLIC_DIAGNOSIS
                if(DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_DIAGLOG_DTC_TRIGGERED_HWSIGNALS))
                {
                    bGetDiag = DeviceManagerInterface::GetInstance()->bGetDiagEnable(); //with a ttfis command cyclic diag can be switched off
                }

                //see ConfigurationBase.h description of eConf_ENABLE_ITC_OVERCURRENT_SUPPRESSED_IFCRITVOLT
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_ITC_OVERCURRENT_SUPPRESSED_IFCRITVOLT) )
                {
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_VOLTAGEMANAGER) )
                    {
                        if(DeviceManagerInterface::GetInstance()->IsCriticalVoltage())
                        {
                            bSuppressITCOverCurrentOfCritVolt = TRUE;
                        }
                    }
                }
                ETG_TRACE_USR4(("bGetDiag                         : 0x%x",bGetDiag));
                ETG_TRACE_USR4(("bSuppressITCOverCurrentOfCritVolt: 0x%x",bSuppressITCOverCurrentOfCritVolt));

                if(TRUE == bGetDiag)
                {
                    //====================================================================
                    // get values for sending it
                    //====================================================================
#ifdef USE_FOURTH_USB_CONNECTOR
                    bDiagOK = DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                                                   oUSBDiagnosisHUB1,
                                                                                   oUSBDiagnosisHUB2,
                                                                                   oUSBDiagnosisHUB3,
                                                                                   oUSBDiagnosisHUB4);



                    if(TRUE == ConfigurationManager::GetInstance()->bFilterTelematicsBoxExpectedAtConnector(usb))
                    {
                        tBool l_bIsHighSpeedExpected;
                        TelematicsBoxSpeedAndPortInfo l_TelematicsBoxSpeedAndPortInfo[ARRAYSIZEFORUSB];
                        //Get the device speed and connection status using the libusb communication
                        DeviceManagerInterface::GetInstance()->vFillDeviceSpeedinfo(l_TelematicsBoxSpeedAndPortInfo);

                        if(TRUE == l_TelematicsBoxSpeedAndPortInfo[usb].bIsDevicePresent)
                        {
                            //Get the expected  device speed of telematics box  from the configuration
                            //Also whether the device is expected as per configuration
                            //if vFillDeviceSpeed is TRUE that means the telematics box is connected. Otherwise a wrong device is connected to the port usb
                            //We are using the same info to set the ITC enITC_DATA_LINE_FAULT. That means that  vFillDeviceSpeed is returing TRUE means enITC_DATA_LINE_FAULT should be PASSED

                            if(TRUE == ConfigurationManager::GetInstance()->vFillDeviceSpeed((tenDevManagerUSBHost)usb,l_bIsHighSpeedExpected))
                            {
                                bTelematicsBoxPresent = TRUE;
                                ETG_TRACE_USR4(("vDoUSBCircuitTest: TelematicsBoxSpeed vFillDeviceSpeed %d ",l_bIsHighSpeedExpected));
                                ETG_TRACE_USR4(("vDoUSBCircuitTest: TelematicsBoxSpeed vFillDeviceSpeedinfo %d",l_TelematicsBoxSpeedAndPortInfo[usb].b_DeviceSpeed));

                                //Both are equal means the expected and configured speed limits are matching
                                if(l_TelematicsBoxSpeedAndPortInfo[usb].b_DeviceSpeed == l_bIsHighSpeedExpected)
                                {
                                    poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk = SIG_TRUE;
                                }
                                else
                                {
                                    poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk = SIG_FALSE;
                                }

                            }
                            else
                            {
                                ETG_TRACE_FATAL(("vDoUSBCircuitTest: TelematicsBoxSpeed set to SIG_UNDEF as the device with expected vendor id and product id not present"));

                                poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk = SIG_UNDEF;
                            }


                        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_ITC_DATA_LINE_FAULT))
                        {

                            if(TRUE == bTelematicsBoxPresent)
                            {
                                ETG_TRACE_USR4(("vDoUSBCircuitTest: Set the Data lines DTC  for the usb port %d as  FI_EN_PASSED",ETG_ENUM(USBCONNECTOR,usb)));
    #ifdef USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enITC_DATA_LINE_FAULT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
    #else
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enITC_DATA_LINE_FAULT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
    #endif

                            }
                            else
                            {
                                ETG_TRACE_USR4(("vDoUSBCircuitTest: Set the Data lines DTC for the usb port %d as  FI_EN_FAILED",ETG_ENUM(USBCONNECTOR,usb)));
    #ifdef USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enITC_DATA_LINE_FAULT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
    #else
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enITC_DATA_LINE_FAULT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);

    #endif

                                }
                            }
                        }
                        else
                        {
                            ETG_TRACE_USR4(("vDoUSBCircuitTest: TelematicsBoxSpeed set to SIG_UNDEF as the device not present"));
                            poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk = SIG_UNDEF;

                        }
                    }






                    int u32BTABoxConfigValue = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_TELEMATICSBOX_MONITORING);
                    if(DVM_OFF != u32BTABoxConfigValue )
                    {

                        DeviceManagerInterface::GetInstance()->vIsBTABoxAlive(m_eBTAMonitorStates,bIsBTAAlive);


                        //if the bIsBTAAlive status is positive then we can send the status
                        if(bIsBTAAlive)
                        {
                            m_eBTAMonitorStates = BTA_MONITOR_FINISHED;
                        }
                        //if the bIsBTAAlive status received is false when the timer is running we have to reset the timer
                        else if(m_bBTAAliveStatus != bIsBTAAlive && BTA_MONITOR_STARTED == m_eBTAMonitorStates)
                        {
                            ETG_TRACE_USR4(("vDoUSBCircuitTest: The BTA Box is Non Alive status received during the timer running.Reset the timer and start monitoring again" ));
                            m_eBTAMonitorStates = BTA_MONITOR_NOT_STARTED;
                        }



                        ETG_TRACE_USR4(("vDoUSBCircuitTest: bIsBTABoxAlive %d",bIsBTAAlive ));
                        ETG_TRACE_USR4(("vDoUSBCircuitTest: m_eBTAMonitorStates %d",m_eBTAMonitorStates ));
                        
                        if(BTA_MONITOR_FINISHED == m_eBTAMonitorStates)
                        {
                            ETG_TRACE_USR4(("vDoUSBCircuitTest: The BTA Box is Alive time out has reached. BTA Box is %s",(bIsBTAAlive?"Alive":"Not alive")));
                            ETG_TRACE_USR4(("vDoUSBCircuitTest: Sending Link Disabled" ));
                        }
                        m_bBTAAliveStatus = bIsBTAAlive;

                    }

#else
                    bDiagOK = DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                                                   oUSBDiagnosisHUB1,
                                                                                   oUSBDiagnosisHUB2,
                                                                                   oUSBDiagnosisHUB3);
#endif

                    //Note: if OC2 (Over current signal) happens then PPON2 is used to switch on VBUS2
                    //      i.e. PPON lasts longer due to timout in prm hence we take that signal
                    //
                    //
                    //                       OC1
                    //                   -----------
                    //                   |         |
                    //                   |         |
                    //      --------------.........----------------
                    //
                    //                             <--delay time to switch on-->
                    //
                    //                                   PPON1..................
                    //                    --------------------------------------
                    //                    |                                    |
                    //                    |                                    |
                    //     ----------------                                    -------------------------
                    //
                    //
                    vTraceInfo(usb,poUSBDiagnosisHUB[usb]);
                    ETG_TRACE_COMP(("bDoUSBCircuitTest: bDiagOK;0x%x",bDiagOK));
                }
            }
            else
            {
                ETG_TRACE_ERR ((" DTC Test is not possible m_poMainApp->m_poDeviceManagerWorkThread==NULL "));
                ETG_TRACE_USR4 (("<--- Exit bDoUSBCircuitTest"));
            }
        }
#ifdef USE_FOURTH_USB_CONNECTOR
        if (!((eUSB4 == usb) || (eUSB3 == usb) || (eUSB2 == usb) || (eUSB1 == usb)))
#else
        if (!((eUSB3 == usb) || (eUSB2 == usb) || (eUSB1 == usb)))
#endif
        {
            ETG_TRACE_ERR ((" DTC Test is not possible for requested USB HOST %d ",usb));
            ETG_TRACE_USR4 (("<--- Exit bDoUSBCircuitTest"));

            bDiagOK = FALSE;
        }

        if((TRUE == bDiagOK) && (FALSE == bSuppressITCOverCurrentOfCritVolt))
        {
            //====================================================
            // send enUSB_OVERCURRENT,enUSB_POWER_SUPPLY_FAULT,enUSB_OPEN
            //====================================================
            switch (poUSBDiagnosisHUB[usb]->m_enOC)
            {
            case SIG_TRUE:
            {
                ETG_TRACE_USR4 (("USB%d port is OverCurrent",usb));
                ETG_TRACE_USR4 (("DTC USB%d port OverCurrent Set Status = FAIL",usb));

                m_u8UsbOverCurrentCount[usb] = 0;
                //need not send ITC of BTA telematics box during Over current
                m_eBTAMonitorStates = BTA_MONITOR_NOT_STARTED;

#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OVERCURRENT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OVERCURRENT       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION //send only if PSA
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_ITC_OVERCURRENT_TRIGGERS_ITC_POWER_SUPPLY_FAULT))
                {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }

#endif
                break;
            }
            case SIG_FALSE:
            {
                ETG_TRACE_USR4 (("Current status in USB%d port is normal for %d secs",usb,m_u8UsbOverCurrentCount[usb]));

                // Send DTC PASS
                ETG_TRACE_USR4 (("Current status in USB%d port is normal continuosly for %d secs",usb,m_u8UsbOverCurrentCount[usb]));
                ETG_TRACE_USR4 (("DTC USB%d port OverCurrent Set Status = PASS",usb));
                m_u8UsbOverCurrentCount[usb] = 0;
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_OVERCURRENT           ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_OVERCURRENT           ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA

#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION //added for PSA
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_ITC_OVERCURRENT_TRIGGERS_ITC_POWER_SUPPLY_FAULT))
                {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MISSINGTELEMATICSBOX_TRIGGERS_ITC_USB_OPEN))
                {
                    if(TRUE == ConfigurationManager::GetInstance()->bFilterTelematicsBoxExpectedAtConnector((tenDevManagerUSBHost)usb))
                    {
                        if(TRUE == bTelematicsBoxPresent )
                        {
                            ETG_TRACE_USR4(("vDoUSBCircuitTest: TelematicsBox USB OPEN Circuit  Sending the  ITC as FI_EN_PASSED"));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                            m_u32CountSecondsUSBOpen = 0;//reseting the USB open counter
                        }
                        else
                        {
                            m_u32CountSecondsUSBOpen++;
                            if(m_u32CountSecondsUSBOpen > ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT))
                            {
                                ETG_TRACE_USR4(("vDoUSBCircuitTest: TelematicsBox USB OPEN Circuit  Sending the  ITC as FI_EN_FAILED m_u32CountSecondsUSBOpen %d",m_u32CountSecondsUSBOpen));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                                m_u32CountSecondsUSBOpen = 0;//resetting the USB open counter
                            }
                        }
                    }
                }
#endif

                bTestHUB = TRUE;
                break;
            }  // if-else (bUsb1VbusOverCurrentPinStatus)
            case SIG_UNDEF:
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_OVERCURRENT      ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_OVERCURRENT      ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA

#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION //send only if PSA
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_ITC_OVERCURRENT_TRIGGERS_ITC_POWER_SUPPLY_FAULT))
                {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_POWER_SUPPLY_FAULT,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_MISSINGTELEMATICSBOX_TRIGGERS_ITC_USB_OPEN))
                {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_OPEN              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
#endif
                ETG_TRACE_USR4 (("DTC USB%d port OverCurrent undefined Set Status = NO_RESULT",usb));
                ETG_TRACE_ERR(("    SIG_UNDEF==poUSBDiagnosisHUB[usb]->m_enOC USB%d ",usb));

                bTestHUB = TRUE; //at startup oc1 might be undefined and thus nothing is send to HUB hence we allow it
                break;
            case SIG_UNUSED:
            case SIG_ANY:
                ETG_TRACE_FATAL(("ERROR: (USB%d) wrong value - check code should not happen at all",usb));
                DVM_NORMAL_M_ASSERT_ALWAYS();
                break;
            default:
                ETG_TRACE_FATAL(("[ERROR]:  vDoUSBCircuitTest poUSBDiagnosisHUB[usb]->m_enOC: %d", (tInt)poUSBDiagnosisHUB[usb]->m_enOC));
                ETG_TRACE_ERRMEM(("[ERROR]: vDoUSBCircuitTest poUSBDiagnosisHUB[usb]->m_enOC: %d", (tInt)poUSBDiagnosisHUB[usb]->m_enOC));
                DVM_NORMAL_M_ASSERT_ALWAYS();
                break;
            }


            //--------------------------------
        }//if(TRUE == bRetVal)
        //No. of USB ports in HUB

        //========================================================
        // send enUSB_HUB_NOT_DETECTED,enUSB_HUB_NUMBER_OF_PORT_MISMATCH
        //========================================================
        if( (TRUE == bTestHUB) && (TRUE == bDiagOK) && (FALSE == bSuppressITCOverCurrentOfCritVolt) ) 
        {
            //--------------------------------
            // Read Data Stored in EOL
            tU8   u8EolUsbHubPortsConfigured[ARRAYSIZEFORUSB];
            tBool bEolUsbHubEnabled[ARRAYSIZEFORUSB];
#ifdef USE_FOURTH_USB_CONNECTOR
            //todo care for format
            bEolUsbHubEnabled[eUSB1] = FALSE;
            bEolUsbHubEnabled[eUSB2] = FALSE;
            bEolUsbHubEnabled[eUSB3] = FALSE;
            bEolUsbHubEnabled[eUSB4] = FALSE;
            u8EolUsbHubPortsConfigured[eUSB1] = 0;
            u8EolUsbHubPortsConfigured[eUSB2] = 0;
            u8EolUsbHubPortsConfigured[eUSB3] = 0;
            u8EolUsbHubPortsConfigured[eUSB4] = 0;
#else
            //todo care for format
            bEolUsbHubEnabled[eUSB1] = FALSE;
            bEolUsbHubEnabled[eUSB2] = FALSE;
            bEolUsbHubEnabled[eUSB3] = FALSE;
            u8EolUsbHubPortsConfigured[eUSB1] = 0;
            u8EolUsbHubPortsConfigured[eUSB2] = 0;
            u8EolUsbHubPortsConfigured[eUSB3] = 0;
#endif
            if(  (usb == eUSB1) &&
                 (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_USB1)) &&
                 (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB1)) &&
                 (DVM_ON <= ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB1          /*USB1 HUB ENABLED with ports > 0*/))
                 )
            {
                bEolUsbHubEnabled[eUSB1] = TRUE;
                u8EolUsbHubPortsConfigured[eUSB1] = (tU8)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB1);

                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d : EOL values say: HUB expected",usb));
            }
            else if(  (usb == eUSB2) &&
                      (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_USB2)) &&
                      (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB2)) &&
                      (DVM_ON <= ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB2          /*USB2 HUB ENABLED with ports >0*/))
                      )
            {
                bEolUsbHubEnabled[eUSB2] = TRUE;
                u8EolUsbHubPortsConfigured[eUSB2] = (tU8)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB2);

                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d : EOL values say: HUB expected",usb));
            }
            else if(  (usb == eUSB3) &&
                      (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_USB3)) &&
                      (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB3)) &&
                      (DVM_ON <= ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB3          /*USB3 HUB ENABLED with ports >0*/))
                      )
            {
                bEolUsbHubEnabled[eUSB3] = TRUE;
                u8EolUsbHubPortsConfigured[eUSB3] = (tU8)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB3);

                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d : EOL values say: HUB expected",usb));
            }
#ifdef USE_FOURTH_USB_CONNECTOR
            else if ((usb == eUSB4) &&
                (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_USB4)) &&
                (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB4)) &&
                (DVM_ON <= ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB4          /*USB4 HUB ENABLED with ports >0*/))
                )
            {
                bEolUsbHubEnabled[eUSB4] = TRUE;
                u8EolUsbHubPortsConfigured[eUSB4] = (tU8)ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_HUBandPORTSatUSB4);

                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d : EOL values say: HUB expected", usb));
            }
#endif
            else
            {
                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d : EOL values say no HUB expected",usb));
            }


            if (TRUE == bEolUsbHubEnabled[usb])
            {

                // Hub Expected
                if (    (m_u32CountSecondsFromStartup< DELAY_DIGANOSISRESULT_CYCLICDIAG)
                        && (poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH == CGlobalEnumerations::HUB_FOUND_NO) //if HUB found then ignore timout
                        )
                {   // Hub is present, send PASS result
                    ETG_TRACE_COMP(("bDoUSBCircuitTest:USB%1d: m_u32CountSecondsFromStartup=%d< DELAY_DIGANOSISRESULT_CYCLICDIAG (%d)",usb,m_u32CountSecondsFromStartup,(int)DELAY_DIGANOSISRESULT_CYCLICDIAG));
                    //No need to send the PASSED for the Hub because we already sent midw_fi_tcl_e8_TestResult::FI_EN_NORESULT
#if 0
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else  //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED       ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
#endif
                }
                else
                {
                    if (poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH == CGlobalEnumerations::HUB_FOUND_YES)
                    {
                        // No of ports detected is less than or equal configured value
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB connected    => set enUSB_HUB_NOT_DETECTED        = PASSED",usb));
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: ports configured at HUB: %d (due to EOL)                  ",usb,u8EolUsbHubPortsConfigured[usb]));
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: ports found      at HUB: %u (udev check)                  ",usb,poUSBDiagnosisHUB[usb]->m_iUSBNoOfPortsFoundAtHub));
                        // DTC Test Passed
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA

                        //====================================================================================================================================
                        //=====================================================================================================================================
                        // Variant2: If number of ports found with connected HUB >=configured ports then PASSED  (Default: e.g. RNAIVI: num of ports not directly written by customer - depend indirectly form KDS)
                        //====================================================================================================================================
                        //====================================================================================================================================
                        //Note: Keep in mind customer might use a HUB which has e.g. 2 ports at the outside but there might be hidden ports inside. I.e. their might be more and we don't want to have unnecessary DTC#s fopr this
                        if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_DTC_HUB_NUMOFPORTS_MISMATCH_VARIANT2))
                        {
                        //----------------------
                        //PASS (no of ports ok)
                        //----------------------
                        // use >= means Ports found must be equal or more then those set per EOL value
                        //I cannot use == because e.g. Gen1HUB has internally 3 ports but only 2 are usable
                            if (poUSBDiagnosisHUB[usb]->m_iUSBNoOfPortsFoundAtHub>=u8EolUsbHubPortsConfigured[usb])
                            {
                                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d:                  => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = PASSED",usb));
    #ifdef USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
    #else //USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
    #endif //USE_DECOUPLE_CYCLICDIAGCCA
                            } // if (poUSBDiagnosisHUB[usb]->m_iNoofaUSBPorts<=b8EolUsbHubM..
                            else
                            {
                                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d:                  => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = FAILED",usb));
                                // No of ports detected is More than expected value
                                // Test fails
    #ifdef USE_DECOUPLE_CYCLICDIAGCCA
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
    #else
                                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
    #endif
                            }  // if-else(poUSBDiagnosisHUB[usb]->m_iNoofaUSBPorts<=b8EolUs..
                        }
                        //=================================================================================================================
                        //=================================================================================================================
                        // Variant1: if number of ports found with connected HUB <= configured ports  then PASSED  (used by GM - GM directly configures values and thus has full access)
                        //=================================================================================================================
                        //=================================================================================================================
                        else
                        {
                            //----------------------
                            //PASS (no of ports ok)
                            //----------------------
                            // use >= means Ports found must be equal or less then those set per EOL value
                            //I cannot use == because e.g. Gen1HUB has internally 3 ports but only 2 are usable
                        if (poUSBDiagnosisHUB[usb]->m_iUSBNoOfPortsFoundAtHub<=u8EolUsbHubPortsConfigured[usb])
                        {
                            ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d:                  => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = PASSED",usb));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                        } // if (poUSBDiagnosisHUB[usb]->m_iNoofaUSBPorts<=b8EolUsbHubM..
                        else
                        {
                            ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d:                  => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = FAILED",usb));
                            // No of ports detected is More than expected value
                            // Test fails
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else
                            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif
                        }  // if-else(poUSBDiagnosisHUB[usb]->m_iNoofaUSBPorts<=b8EolUs..
                        }
                    }   // if (poUSBDiagnosisHUB[usb]->m_eUSB_HUB_ENABLE == CGlobalEnu..
                    else
                    {
                        // Hub is not present, send FAIL result
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not connected                                   ",usb));
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: ports found      at HUB= none                       ",usb));
                        // DTC Test Failed
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB expected => set enUSB_HUB_NOT_DETECTED        = FAILED",usb));
                        ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB expected => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = NORESULT",usb));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED  );
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED  , poTestData);
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                    }  // if -else if (poUSBDiagnosisHUB[usb]->m_eUSB_HUB_ENABLE == CG..

                }  // (m_u32CountSecondsFromStartup< DELAY_DIGANOSISRESULT_CYCLICDIAG)


            } // if (bEolUsbHubEnabled[usb])

            else
            {
                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not expected => set enUSB_HUB_NOT_DETECTED        = NORESULT",usb));
                ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not expected => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = NORESULT",usb));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
                u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
            }  // if-else (bEolUsbHubEnabled[usb])
        }
        //---------------------------------
        //not checked because over current
        //---------------------------------
        else //FALSE == bTestHUB i.e. overcurrent
        {
            ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not checked due to overcurrent!                            ",usb));
            ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not expected => set enUSB_HUB_NOT_DETECTED        = NORESULT",usb));
            ETG_TRACE_USR4(("bDoUSBCircuitTest:USB%1d: HUB not expected => set enUSB_HUB_NUMBER_OF_PORT_MISMATCH = NORESULT",usb));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#else //USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_HUB_NOT_DETECTED,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA

        }  // if-else (bEolUsbHubEnabled[usb])


        //====================================================================
        // send  those bound to BTA - Telematicsbox  enUSB_LINK_DISABLED,  enUSB_SPEED_REDUCED
        //====================================================================

        if( (TRUE == bDiagOK) && (FALSE == bSuppressITCOverCurrentOfCritVolt) )
        {
            if(TRUE == ConfigurationManager::GetInstance()->bFilterTelematicsBoxExpectedAtConnector(usb) )
            {
                if(BTA_MONITOR_FINISHED == m_eBTAMonitorStates)
                {
                    if(m_bBTAAliveStatus)
                    {
                        poUSBDiagnosisHUB[usb]->m_enTelematicsBoxConnected = SIG_TRUE;
                    }
                    else
                    {
                        poUSBDiagnosisHUB[usb]->m_enTelematicsBoxConnected = SIG_FALSE;
                    }
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: poUSBDiagnosisHUB[usb]->m_enTelematicsBoxConnected: %d",ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enTelematicsBoxConnected)));
                    switch(poUSBDiagnosisHUB[usb]->m_enTelematicsBoxConnected)
                    {
                    case SIG_TRUE:
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                        break;
                    case SIG_FALSE:
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                        u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                        break;
                    default:
                        break;
                    }
                }

                ETG_TRACE_USR4(("bDoUSBCircuitTest: poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk: %d",ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk)));
                switch(poUSBDiagnosisHUB[usb]->m_enTelematicsBoxSpeedOk)
                {
                case SIG_TRUE:
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_SPEED_REDUCED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_SPEED_REDUCED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                    break;
                case SIG_FALSE:
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_SPEED_REDUCED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_SPEED_REDUCED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                    break;
                default:
                    break;
                }


            }

        }

        //====================================================================
        // send  LINK_DISABLED if it is bound to an external HUB -  enUSB_LINK_DISABLED
        //====================================================================
        if((TRUE == bDiagOK) &&
           (FALSE == bSuppressITCOverCurrentOfCritVolt) &&
           (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED))
          )
        {
            tBool bRes = ConfigurationManager::GetInstance()->bExpectedExternalHubIsConnected(usb);
            ETG_TRACE_USR4(("bDoUSBCircuitTest: eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED configuration parameter is enabled."));
            if((eUSB1 == usb) &&
               (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB1))
              )
            {
                if(TRUE == bRes && CGlobalEnumerations::HUB_FOUND_YES == poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH )
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is found sending USB_LINK_DISABLED as FI_EN_PASSED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                else
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is not found sending USB_LINK_DISABLED as FI_EN_FAILED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
            }//if((eUSB1 == usb)
            if((eUSB2 == usb) &&
               (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB2))
              )
            {
                if(TRUE == bRes && CGlobalEnumerations::HUB_FOUND_YES == poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH)
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is found sending USB_LINK_DISABLED as FI_EN_PASSED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                else
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is not found sending USB_LINK_DISABLED as FI_EN_FAILED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
            }// if((eUSB2 == usb)
            if((eUSB3 == usb) &&
               (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB3))
              )
            {
                if(TRUE == bRes && CGlobalEnumerations::HUB_FOUND_YES == poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH)
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is found sending USB_LINK_DISABLED as FI_EN_PASSED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                else
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is not found sending USB_LINK_DISABLED as FI_EN_FAILED for the port %d ",ETG_ENUM(TENDEVMANAGERUSBHOST,usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb,enUSB_LINK_DISABLED              ,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
            }//if((eUSB3 == usb)
#ifdef USE_FOURTH_USB_CONNECTOR
            if ((eUSB4 == usb) &&
                (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_observeDTC_HUBatUSB4))
                )
            {
                if (TRUE == bRes && CGlobalEnumerations::HUB_FOUND_YES == poUSBDiagnosisHUB[usb]->m_eUSB_HUB_SEARCH)
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is found sending USB_LINK_DISABLED as FI_EN_PASSED for the port %d ", ETG_ENUM(TENDEVMANAGERUSBHOST, usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_LINK_DISABLED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_LINK_DISABLED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
                else
                {
                    ETG_TRACE_USR4(("bDoUSBCircuitTest: expected external hub is not found sending USB_LINK_DISABLED as FI_EN_FAILED for the port %d ", ETG_ENUM(TENDEVMANAGERUSBHOST, usb)));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_LINK_DISABLED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
                    u8NumOfChanges += u8SetDiagLogDTCCode(usb, enUSB_LINK_DISABLED, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED, poTestData);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
                }
            }//if((eUSB4 == usb)
#endif//USE_FOURTH_USB_CONNECTOR
        }//if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_DIAG_ITC_USB_LINK_DISABLED))


    }//if(FALSE == m_bSendFirstMessageToDiagLog)





    ETG_TRACE_USR4 (("End  : bDoUSBCircuitTest"));
} // tBool vd_devicemanager_tclClientHandler_Diaglog::bDoUSBCircuitTest(

/*--------------------------------------------------------------------------------*
 * tVoid vvDoSendFirstMessageToDiagLog(...)
 *--------------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vDoSendFirstMessageToDiagLog(
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart *poTestData
#endif
)
{


    ETG_TRACE_USR4(("================================================"));
    ETG_TRACE_USR4(("vDoUSBCircuitTest: DIAGLOG: Initial Transmission"));
    ETG_TRACE_USR4(("================================================"));
    tenDevManagerUSBHost usbConnector;
    tU8 u8NumOfChanges = 0;
    for(int l = (int)eUSB1;l < (int)eUSBAll; l++) //set values for
    {
        usbConnector = (tenDevManagerUSBHost)l;
        ETG_TRACE_USR4(("vDoUSBCircuitTest: -----------------------------------------"));
        ETG_TRACE_USR4(("vDoUSBCircuitTest: Initial Transmission eUSB%d", usbConnector));
        ETG_TRACE_USR4(("vDoUSBCircuitTest: -----------------------------------------"));

        //send all available ITCs passed first - evaluation of a failed  would take some time for some ITC's in this way FC_Diagnosis gets a value in time at any case at startup
//------------------------------
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
//------------------------------
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_OVERCURRENT,                (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_POWER_SUPPLY_FAULT,         (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_OPEN,                       (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
#endif
#ifdef USE_PSA_BTA_TELEMATICSBOX_ITCS
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_LINK_DISABLED,              (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_SPEED_REDUCED,              (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enITC_DATA_LINE_FAULT,            (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);

#endif //USE_PSA_BTA_TELEMATICSBOX_ITCS
//------------------------------
#else  //USE_DECOUPLE_CYCLICDIAGCCA
//------------------------------
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_OVERCURRENT,                (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_HUB_NOT_DETECTED,           (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_HUB_NUMBER_OF_PORT_MISMATCH,(tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_POWER_SUPPLY_FAULT,         (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_OPEN,                       (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
#endif
#ifdef USE_PSA_BTA_TELEMATICSBOX_ITCS
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_LINK_DISABLED,              (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enUSB_SPEED_REDUCED,              (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);
        u8NumOfChanges += u8SetDiagLogDTCCode(usbConnector, enITC_DATA_LINE_FAULT,            (tU8)midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, poTestData);

#endif //USE_PSA_BTA_TELEMATICSBOX_ITCS
//------------------------------
#endif //USE_DECOUPLE_CYCLICDIAGCCA
//------------------------------
        
    }
    ETG_TRACE_USR4(("vDoUSBCircuitTest: u8NumOfChanges: %d",u8NumOfChanges));
    if(u8NumOfChanges > 0)
    {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
        //caller of function cares to send it - see usage of function vCopyResultToVector(...) for details
#else
        vSendResult(*poTestData);
#endif

    }

}
/*--------------------------------------------------------------------------------*
 * tVoid vd_devicemanager_tclClientHandler_Diaglog::vDoSdcTest( tenSlot eSdcSlot, *
 *  midw_diaglogfi_tclMsgSaveTestResultMethodStart* poTestData, u8ValuesChanged ) *
 *--------------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vDoSdcTest( IN tenSlot eSdcSlot, 
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
OUT midw_diaglogfi_tclMsgSaveTestResultMethodStart* poTestData,
#endif
OUT tU8& u8NumOfChanges )
{
    ETG_TRACE_USR4 (("Begin: vDoSdcTest for SD Card slot: %d", ETG_ENUM(DVM_CARD_SLOT, eSdcSlot) ));

    tBool   bDiagOK = FALSE;
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
    {
        if( m_u32CountSecondsFromStartup == 0 )
        {
            //send no over current i.e. passed
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogItcSdc( eSdcSlot, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
#else //USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogItcSdc( eSdcSlot, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED, poTestData );
#endif //USE_DECOUPLE_CYCLICDIAGCCA
        }

        if((tUInt)eSdcSlot < ARRAYSIZEFORCARD)
        {
            tBool bSdcOcAvtice = FALSE;
            tU8   u8TestResult;
            // get diagnosis value for given SD Card slot
            bDiagOK = DeviceManagerInterface::GetInstance()->bGetSdcDiagnosis( eSdcSlot, bSdcOcAvtice );
            // care for over current
            u8TestResult = ( bSdcOcAvtice ) ? (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
            ETG_TRACE_USR4(( "SD Card slot: %d port OverCurrent Set Status", ETG_ENUM(DVM_CARD_SLOT, eSdcSlot) ));
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogItcSdc( eSdcSlot, u8TestResult);
#else //USE_DECOUPLE_CYCLICDIAGCCA
            u8NumOfChanges += u8SetDiagLogItcSdc( eSdcSlot, u8TestResult, poTestData );
#endif //USE_DECOUPLE_CYCLICDIAGCCA
            m_u8SdcOverCurrentCount[eSdcSlot] = 0;
        }
        else
        {
            ETG_TRACE_ERR(( "vDoSdcTest: DTC Test is not possible for requested SD Card slot: %d ", ETG_ENUM(DVM_CARD_SLOT, eSdcSlot) ));
            bDiagOK = FALSE;
        }
        ETG_TRACE_USR4(( "End: vDoSdcTest with bDiagOK=%d", ETG_ENUM(BOOL, bDiagOK) ));
    }
    else
    {
        u8NumOfChanges=0;;
        
        ETG_TRACE_FATAL(("[INFO] vDoSdcTest: Diagnosis of SDCard-Internal is disabled - see eConf_ENABLE_USE_PRM_SDCARDSIGNALS"));
    }

    ETG_TRACE_USR4 (("End: vDoSdcTest (bDiagOK:%d)",bDiagOK ));
} 

/*-----------------------------------------------------------------------------*
 * Handler functions for FI messages                                           *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * tVoid vHandle_SaveTestResult_MethodResult(amt_tclServiceData* poMessage)    *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog
::vHandle_SaveTestResult_MethodResult(OUT amt_tclServiceData* poMessage) const
{
    ETG_TRACE_USR4 (("Begin: vHandle_SaveTestResult_MethodResult"));
    ETG_TRACE_SYS(("***************/vHandle_SaveTestResult_MethodResult**********"));
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oResultMsg( poMessage );

    // Create the (message related) FI data object
    midw_diaglogfi_tclMsgSaveTestResultMethodResult oResultData;
    oResultMsg.s32GetData(oResultData);

    // if control setting accepted, composing positive response
    if (midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED ==
            oResultData.Status.enType)
    {
        ETG_TRACE_USR4 (("vHandle_DiagLogDTCCode :: Accepted"));
    }
    else
    {
        ETG_TRACE_ERR ((" vHandle_DiagLogDTCCode ::Rejected"));
    }

    // Always destroy the FI data object before leaving its creation scope
    oResultData.vDestroy();
    ETG_TRACE_SYS(("***************vHandle_SaveTestResult_MethodResult/**********"));
    ETG_TRACE_USR4 (("End  : vHandle_SaveTestResult_MethodResult"));

}   // tVoid vd_devicemanager_tclClientHandler_Diaglog ::vHandle_SaveTestResu..

/*-----------------------------------------------------------------------------*
 * tVoid vHandle_DiaglogSendNextTestResult_Status(amt_tclServiceData* poMessage)*
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog::vHandle_DiaglogSendNextTestResult_Status(OUT amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4 (("Begin: vHandle_DiaglogSendNextTestResult_Status : Retest and Send results "));
    ETG_TRACE_SYS(("***************/vHandle_DiaglogSendNextTestResult_Status**********"));

    midw_diaglogfi_tclMsgSaveTestResultMethodStart ms;
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg( poMessage );

    // Create the FI data object
    midw_diaglogfi_tclMsgSendNextTestResultStatus oResultStatus;

    if (oVisitorMsg.s32GetData(oResultStatus) != OSAL_ERROR)
    {
        if (oResultStatus.NextTestResultStruct.Status == TRUE) //TRUE :client is required to send next test result to DiagLog
            //(FALSE if this is a confirmation message from the Diaglog that thel UpReg has been succesful)
        {
            //-----------------------------------------------
            //diagnosis asks for all ITCs
            //-----------------------------------------------
            if (oResultStatus.NextTestResultStruct.ITCList.empty()) //if the List of DTC's  is empty and "NextTestResultStruct.Status == True" then the client should report all the DTCs for which he is reponsible.

            { /* All Test result is empty, hence clear all the local test results.
              so the latest test results can be sent when timer slot is available*/
                ETG_TRACE_COMP (("vHandle_DiaglogSendNextTestResult_Status: Reset all test results for resending"));

                m_rDtc_error[enITC_USB1_SIGNAL_OVERCURRENT].s8DtcResult  = -1;
                m_rDtc_error[enITC_USB2_SIGNAL_OVERCURRENT].s8DtcResult  = -1;
                m_rDtc_error[enITC_USB3_SIGNAL_OVERCURRENT].s8DtcResult  = -1;

                m_rDtc_error[enITC_USB1_HUB_NOT_DETECTED].s8DtcResult        = -1;
                m_rDtc_error[enITC_USB2_HUB_NOT_DETECTED].s8DtcResult        = -1;
                m_rDtc_error[enITC_USB3_HUB_NOT_DETECTED].s8DtcResult        = -1;

                m_rDtc_error[enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;

                m_rDtc_error[enITC_USB1_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_POWER_SUPPLY_FAULT].s8DtcResult = -1;

                m_rDtc_error[enITC_USB1_OPEN].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_OPEN].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_OPEN].s8DtcResult = -1;

                m_rDtc_error[enITC_USB1_LINK_DISABLED].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_LINK_DISABLED].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_LINK_DISABLED].s8DtcResult = -1;

                m_rDtc_error[enITC_USB1_SPEED_REDUCED].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_SPEED_REDUCED].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_SPEED_REDUCED].s8DtcResult = -1;

                m_rDtc_error[enITC_USB1_DATA_LINE_FAULT].s8DtcResult = -1;
                m_rDtc_error[enITC_USB2_DATA_LINE_FAULT].s8DtcResult = -1;
                m_rDtc_error[enITC_USB3_DATA_LINE_FAULT].s8DtcResult = -1;


                // Set to zero so that test can restart for required time(DEVICE_MANAGER_enUSB_OVERCURRENT_TIME)
                m_u8UsbOverCurrentCount[eUSB1] = 0;
                m_u8UsbOverCurrentCount[eUSB2] = 0;
                m_u8UsbOverCurrentCount[eUSB3] = 0;
#ifdef USE_FOURTH_USB_CONNECTOR
                m_u8UsbOverCurrentCount[eUSB4] = 0;
#endif
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
                {
                    m_rDtc_error[enITC_SDC1_SIGNAL_OVERCURRENT].s8DtcResult  = -1;
                    m_u8SdcOverCurrentCount[eCard0] = 0;         // Set to zero so that test can restart
                }
                tU16 u16ConfigVal = ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT);

                if(DVM_OFF != u16ConfigVal)
                {
                    //Set to timeout value eCONF_TIMEOUT_ITC_USB_OPEN_CIRCUIT
                    //refer PSARCCB-7297
                    //Suppose the BTA is removed and if the fault memory is cleared
                    //Then the result ITC_USB_OPEN_CIRCUIT FAILED will not be send if the m_u32CountSecondsUSBOpen is not reached.
                    // the below assignment will make the ITC will be send immediatly once the fault memory is cleared.
                    m_u32CountSecondsUSBOpen = u16ConfigVal;
                }


            }  // if (oResultStatus.NextTestResultStruct.ITCList.empty())
            //-----------------------------------------------
            //diagnosis asks for specific ITCs
            //-----------------------------------------------
            else
            {
                for(tU16 i=0; i < oResultStatus.NextTestResultStruct.ITCList.size(); i++)
                {
                    ETG_TRACE_USR4 (("Reset test DTC %d result for resending",
                                     oResultStatus.NextTestResultStruct.ITCList[i]));
                    switch(oResultStatus.NextTestResultStruct.ITCList[i])
                    {
                    /*overcurrent*/
                    case ITC_USB1_OVERCURRENT :
                        m_rDtc_error[enITC_USB1_SIGNAL_OVERCURRENT].s8DtcResult = -1;
                        m_u8UsbOverCurrentCount[eUSB1] = 0;// Set to zero so that test can restart for required time(DEVICE_MANAGER_enUSB_OVERCURRENT_TIME)
                        break ;
                    case ITC_USB2_OVERCURRENT :
                        m_rDtc_error[enITC_USB2_SIGNAL_OVERCURRENT ].s8DtcResult = -1;
                        m_u8UsbOverCurrentCount[eUSB2] = 0;// Set to zero so that test can restart for required time(DEVICE_MANAGER_enUSB_OVERCURRENT_TIME)
                        break ;
                    case ITC_USB3_OVERCURRENT :
                        m_rDtc_error[enITC_USB3_SIGNAL_OVERCURRENT ].s8DtcResult = -1;
                        m_u8UsbOverCurrentCount[eUSB3] = 0;// Set to zero so that test can restart for required time(DEVICE_MANAGER_enUSB_OVERCURRENT_TIME)
                        break ;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_OVERCURRENT:
                        m_rDtc_error[enITC_USB4_SIGNAL_OVERCURRENT].s8DtcResult = -1;
                        m_u8UsbOverCurrentCount[eUSB4] = 0;// Set to zero so that test can restart for required time(DEVICE_MANAGER_enUSB_OVERCURRENT_TIME)
                        break;
#endif
                        /*hub not detected*/
                    case ITC_USB1_HUB_NOT_DETECTED :
                        m_rDtc_error[enITC_USB1_HUB_NOT_DETECTED].s8DtcResult = -1;
                        break;
                    case ITC_USB2_HUB_NOT_DETECTED :
                        m_rDtc_error[enITC_USB2_HUB_NOT_DETECTED].s8DtcResult = -1;
                        break;
                    case ITC_USB3_HUB_NOT_DETECTED :
                        m_rDtc_error[enITC_USB3_HUB_NOT_DETECTED].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_HUB_NOT_DETECTED:
                        m_rDtc_error[enITC_USB4_HUB_NOT_DETECTED].s8DtcResult = -1;
                        break;
#endif
                        /*hub number of port mismatch*/
                    case ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH :
                        m_rDtc_error[enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                        break;
                    case ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH :
                        m_rDtc_error[enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                        break;
                    case ITC_USB3_HUB_NUMBER_OF_PORTS_MISMATCH :
                        m_rDtc_error[enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_HUB_NUMBER_OF_PORTS_MISMATCH:
                        m_rDtc_error[enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH].s8DtcResult = -1;
                        break;
#endif

                    case ITC_USB1_POWER_SUPPLY_FAULT :
                        m_rDtc_error[enITC_USB1_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                        break;
                    case ITC_USB2_POWER_SUPPLY_FAULT :
                        m_rDtc_error[enITC_USB2_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                        break;
                    case ITC_USB3_POWER_SUPPLY_FAULT :
                        m_rDtc_error[enITC_USB3_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_POWER_SUPPLY_FAULT:
                        m_rDtc_error[enITC_USB4_POWER_SUPPLY_FAULT].s8DtcResult = -1;
                        break;
#endif
                    case ITC_USB1_OPEN :
                        m_rDtc_error[enITC_USB1_OPEN].s8DtcResult = -1;
                        break;
                    case ITC_USB2_OPEN :
                        m_rDtc_error[enITC_USB2_OPEN].s8DtcResult = -1;
                        break;
                    case ITC_USB3_OPEN :
                        m_rDtc_error[enITC_USB3_OPEN].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_OPEN:
                        m_rDtc_error[enITC_USB4_OPEN].s8DtcResult = -1;
                        break;
#endif

                    case ITC_USB1_LINK_DISABLED :
                        m_rDtc_error[enITC_USB1_LINK_DISABLED].s8DtcResult = -1;
                        break;
                    case ITC_USB2_LINK_DISABLED :
                        m_rDtc_error[enITC_USB2_LINK_DISABLED].s8DtcResult = -1;
                        break;
                    case ITC_USB3_LINK_DISABLED :
                        m_rDtc_error[enITC_USB3_LINK_DISABLED].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_LINK_DISABLED:
                        m_rDtc_error[enITC_USB4_LINK_DISABLED].s8DtcResult = -1;
                        break;
#endif
                    case ITC_USB1_SPEED_REDUCED :
                        m_rDtc_error[enITC_USB1_SPEED_REDUCED].s8DtcResult = -1;
                        break;
                    case ITC_USB2_SPEED_REDUCED :
                        m_rDtc_error[enITC_USB2_SPEED_REDUCED].s8DtcResult = -1;
                        break;
                    case ITC_USB3_SPEED_REDUCED :
                        m_rDtc_error[enITC_USB3_SPEED_REDUCED].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_SPEED_REDUCED:
                        m_rDtc_error[enITC_USB4_SPEED_REDUCED].s8DtcResult = -1;
                        break;
#endif

                    case ITC_USB1_DATA_LINE_FAULT :
                        m_rDtc_error[enITC_USB1_DATA_LINE_FAULT].s8DtcResult = -1;
                        break;
                    case ITC_USB2_DATA_LINE_FAULT :
                        m_rDtc_error[enITC_USB2_DATA_LINE_FAULT].s8DtcResult = -1;
                        break;
                    case ITC_USB3_DATA_LINE_FAULT :
                        m_rDtc_error[enITC_USB3_DATA_LINE_FAULT].s8DtcResult = -1;
                        break;
#ifdef USE_FOURTH_USB_CONNECTOR
                    case ITC_USB4_DATA_LINE_FAULT:
                        m_rDtc_error[enITC_USB4_DATA_LINE_FAULT].s8DtcResult = -1;
                        break;
#endif
                    default:
                        ETG_TRACE_ERR (("u16DtcCode is not recognized %d",
                                        oResultStatus.NextTestResultStruct.ITCList[i]));
                        break;
                    }  // switch(oResultStatus.NextTestResultStruct.ITCList[i])
                }  // for(tU16 i=0; i < oResultStatus.NextTestResultStruct.ITCList..
            }  // if-else (oResultStatus.NextTestResultStruct.ITCList.empty())



            //-------------------------------------------
            //get values and care for sending to diaglog
            //-------------------------------------------
            //since ProcessTimer which triggers bDoUSBCircuitTest at other parts fo the code does it from the same thread context
            //we can also send it immediately
            //For sending the SD card ITCs
            tU16 u16TimerId = 0;
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
            {
                ETG_TRACE_USR4(("vHandle_DiaglogSendNextTestResult_Status: eConf_ENABLE_USE_PRM_SDCARDSIGNALS is enabled"));
                if( m_bTimerSdcOcStarted )
                {
                    u16TimerId = DEVICE_MANAGER_TIMERID_OC_SDC;
                }
            }
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
            vDoCircuitTestAndSendMessageToDiagLog(u16TimerId); //calls a function which uses vStoreResultInVector
            vector<trDtcCodeResult> ListCyclicDiagResults;
            ListCyclicDiagResults.clear();
            vCopyResultToVector(ListCyclicDiagResults);
            vSendResultOfCyclicDiag(ListCyclicDiagResults);
#else //USE_DECOUPLE_CYCLICDIAGCCA
            vDoCircuitTestAndSendMessageToDiagLog(&ms,u16TimerId);
#endif //USE_DECOUPLE_CYCLICDIAGCCA
        }  // if (oResultStatus.NextTestResultStruct.Status == TRUE)
        else
        {
            /* this is only a confirmation message from the last upreg, can be ignored */
        }  // if-else (oResultStatus.NextTestResultStruct.Status == TRUE)
    }  // if (oVisitorMsg.s32GetData(oResultStatus) != OSAL_ERROR)
    else
    {
        ETG_TRACE_ERR(( " Received message is invalid."));
    }  // if-else (oVisitorMsg.s32GetData(oResultStatus) != OSAL_ERROR)

    // Always destroy the FI data object before leaving its creation scope
    oResultStatus.vDestroy();
    ETG_TRACE_SYS(("***************vHandle_DiaglogSendNextTestResult_Status/**********"));
    ETG_TRACE_USR4 (("End  : vHandle_DiaglogSendNextTestResult_Status : Retest and Send results "));

} // tVoid vd_devicemanager_tclClientHandler_Diaglog::vHandle_DiaglogSendNextT..


/*-----------------------------------------------------------------------------*
 * tVoid vDoCircuitTestAndSendMessageToDiagLog(amt_tclServiceData* poMessage)*
 *-----------------------------------------------------------------------------*/

tVoid vd_devicemanager_tclClientHandler_Diaglog::vDoCircuitTestAndSendMessageToDiagLog(
#ifndef USE_DECOUPLE_CYCLICDIAGCCA
IN midw_diaglogfi_tclMsgSaveTestResultMethodStart *poTestData, 
#endif
IN tU16 u16TimerId )
{

#ifdef USE_DECOUPLE_CYCLICDIAGCCA
    ETG_TRACE_USR4 (("Begin  : vDoCircuitTestAndSendMessageToDiagLog "));
    m_LockDiagFct.lock();
#endif
    ETG_TRACE_USR4 (("vDoCircuitTestAndSendMessageToDiagLog: locked() "));

    tU8 u8ValuesChanged = 0;
    for(tInt i=(tInt)eUSB1;i<ARRAYSIZEFORUSB;i++)
    {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
        vDoUSBCircuitTest((tenDevManagerUSBHost)i,u8ValuesChanged);
#else
        vDoUSBCircuitTest((tenDevManagerUSBHost)i,poTestData,u8ValuesChanged);
#endif
    }

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
    {
        if( u16TimerId == DEVICE_MANAGER_TIMERID_OC_SDC )
        {
            for( tInt i=(tInt)eCard0; i<ARRAYSIZEFORCARD; i++ )
            {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
                vDoSdcTest( (tenSlot)i, u8ValuesChanged );
#else
                vDoSdcTest( (tenSlot)i, poTestData, u8ValuesChanged );
#endif
            }
        }
    }
    //==========================
    // sending
    //==========================
    ETG_TRACE_USR4(("vDoUSBCircuitTest: u8NumOfChanges: %d",u8ValuesChanged));
    if( u8ValuesChanged > 0 )
    {
#ifdef USE_DECOUPLE_CYCLICDIAGCCA
         //caller of function cares to send it - see usage of function vCopyResultToVector(...) for details
#else
        vSendResult(*poTestData);
#endif
    }

#ifdef USE_DECOUPLE_CYCLICDIAGCCA
    ETG_TRACE_USR4 (("vDoCircuitTestAndSendMessageToDiagLog: m_LockDiagFct.unlock() "));
    m_LockDiagFct.unlock();
#endif
    ETG_TRACE_USR4 (("End  : vDoCircuitTestAndSendMessageToDiagLog "));

 
}

/*-----------------------------------------------------------------------------*
 * tVoid vTraceInfo(tenDevManagerUSBHost usb,CUSBDiagnosis* poUSBDiag)         *
 *-----------------------------------------------------------------------------*/
tVoid vd_devicemanager_tclClientHandler_Diaglog
::vTraceInfo(IN tenDevManagerUSBHost usb,IN const CUSBDiagnosis* poUSBDiag) const
{
    //------------------------------------------
    //show info USB1
    //------------------------------------------
    ETG_TRACE_USR2(("*******************************************************"));

    if(NOTCONNECTED == poUSBDiag->m_ulSize)
    {
        ETG_TRACE_USR2(("* [   <---********** DISCONNECTED: USB%d ***************",usb));
        ETG_TRACE_USR2(("******** **********************************************"));
    }
    else
    {
        ETG_TRACE_USR2((" [<----- CONNECTED USB%d********************************",usb));
        ETG_TRACE_USR2(("*******************************************************"));

        //-------------------
        //UDEV related info
        //-------------------
        ETG_TRACE_USR2(("poUSBDiag->m_ulSize:          %u  MByte         ",poUSBDiag->m_ulSize));
        ETG_TRACE_USR2(("poUSBDiag->m_cVendor:         %s                ",poUSBDiag->m_cVendor.toStdString().c_str()));
        ETG_TRACE_USR2(("poUSBDiag->m_iVersion:        %d                ",poUSBDiag->m_iVersion));
        ETG_TRACE_USR2(("poUSBDiag->m_iUSB_HUB_SEARCH: %d                ",ETG_CENUM(CGlobalEnumerations::HUB_SEARCHRESULT_Type,
                                                                                     poUSBDiag->m_eUSB_HUB_SEARCH)));
        ETG_TRACE_USR2(("poUSBDiag->m_iUSBNoOfPortsFoundAtHub:  %u       ",poUSBDiag->m_iUSBNoOfPortsFoundAtHub));
        ETG_TRACE_USR2(("poUSBDiag->m_cSerialID:       %s                ",poUSBDiag->m_cSerialID.toStdString().c_str()));
        ETG_TRACE_USR2(("poUSBDiag->m_cUSBPortNo:      %s                ",poUSBDiag->m_cUSBPortNo.toStdString().c_str()));
        ETG_TRACE_USR2(("poUSBDiag->m_eDeviceType:     %d                ",ETG_ENUM(DVM_DEVICE_TYPE, poUSBDiag->m_eDeviceType)));
        ETG_TRACE_USR2(("poUSBDiag->m_cDeviceName:     %s                ",poUSBDiag->m_cDeviceName.toStdString().c_str()));
        ETG_TRACE_USR2(("poUSBDiag->m_cMountPoint:     %s                ",poUSBDiag->m_cMountPoint.toStdString().c_str()));
        ETG_TRACE_USR2(("poUSBDiag->m_cDeviceVersion:  %s                ",poUSBDiag->m_cDeviceVersion.toStdString().c_str()));
    }

    //----------------------
    //prm related info
    //----------------------
    ETG_TRACE_USR2(("poUSBDiag->m_enOC:            0x%x (%d)                ",poUSBDiag->m_enOC,ETG_ENUM(TENSIGNALVALUE,poUSBDiag->m_enOC)));
    ETG_TRACE_USR2(("poUSBDiag->m_enPPON:          0x%x (%d)                ",poUSBDiag->m_enPPON,ETG_ENUM(TENSIGNALVALUE,poUSBDiag->m_enPPON)));
    ETG_TRACE_USR2(("poUSBDiag->m_enU60_drop:      0x%x (%d)                ",poUSBDiag->m_enU60_drop,ETG_ENUM(TENSIGNALVALUE,poUSBDiag->m_enU60_drop)));
    ETG_TRACE_USR2(("poUSBDiag->m_enMountedUSB:    0x%x (%d)                ",poUSBDiag->m_enMountedUSB,ETG_ENUM(TENSIGNALVALUE,poUSBDiag->m_enMountedUSB)));
    ETG_TRACE_USR2(("poUSBDiag->m_enU50_USB_Reset: 0x%x (%d)                ",poUSBDiag->m_enU50_USB_Reset,ETG_ENUM(TENSIGNALVALUE,poUSBDiag->m_enU50_USB_Reset)));
    ETG_TRACE_USR2(("poUSBDiag->m_rStatus:         0x%x (%d)                ",poUSBDiag->m_rStatus,ETG_ENUM(DEVICE_CONNECTSTATUS_Type,poUSBDiag->m_rStatus)));

}

//bound to a GPIO telling if OVercurrent for SD Card happens or not
//Note of analysis: m_TimerSdcOc and m_Timer are both intialised
//both use function vExecute() as expiration callback function
//m_Timer is started with DEVICE_MANAGER_TIMEOUT_OC_ITC_USB
//if vExecute is called then a mapping to DEVICE_MANAGER_TIMERID_OC_SDC in dependency of the configuration (eConf_ENABLE_USE_PRM_SDCARDSIGNALS)
//does trigger the call of function vDoSdcTest which checks state of GPIO
//but m_bTimerSdcOcStarted has to be TRUE to call vDoSdcTest and this is not the case
//=>currently the functionality is disabled
//a good way to continue with refactoring would be a) put call of vDoSDCTest to its own function to be called
//change cyclicDiagthread to use Timouts instead of sleeps and put trigger mechanism to CyclicDiagThread
tVoid vd_devicemanager_tclClientHandler_Diaglog::vStartUsbOcTimer(tVoid)
{
    ETG_TRACE_USR4(( "vStartUsbOcTimer: Begin" ));

    // Restart the timer for every change to get the actual status after 3s of no change
    m_bTimerStarted = TRUE;    //@todo use counter to replace this
    m_Timer.bStart( DEVICE_MANAGER_TIMEOUT_OC_ITC_USB, /*(OSAL_tMSecond) interval = */0 ); // wait 3s before chekcing for errors
    ETG_TRACE_COMP(("vStartUsbOcTimer: TIMER started!"));
    ETG_TRACE_USR4(( "vStartUsbOcTimer: End" ));
}

#ifdef MARKER_UNUSED_CODE
//this code is never called 
tVoid vd_devicemanager_tclClientHandler_Diaglog::vStopUsbOcTimer(tVoid)
{
    ETG_TRACE_USR4(( "vStopUsbOcTimer: Begin" ));
    if( m_bTimerStarted == TRUE )
    {
        m_bTimerStarted = FALSE;    //@todo use counter to replace this
        m_Timer.bStop( );
        ETG_TRACE_COMP(("vStopUsbOcTimer: TIMER stopped!"));
    }
    ETG_TRACE_USR4(( "vStopUsbOcTimer: End" ));
}
#endif

#ifdef MARKER_UNUSED_CODE
//koe2hi could not find any usage of this function in the code please check
//idea is to check overcurrent in a sd card reader. OC is notified via GPIO but since NissanLcn2Kai (Gen2) this has not been realised
tVoid vd_devicemanager_tclClientHandler_Diaglog::vStartSdcOcTimer(tVoid) //is not used yet is that a merge probem feature used by diagnosis?
{
    ETG_TRACE_USR4(( "vStartSdcOcTimer: Begin" ));
    // Restart the timer for every change to get the actual status after 3s of no change
    m_bTimerSdcOcStarted = TRUE;    //@todo use counter to replace this
    m_TimerSdcOc.bStart( DEVICE_MANAGER_TIMEOUT_OC_ITC_SDC, /*(OSAL_tMSecond) interval = */0 ); // wait 3s before chekcing for errors
    ETG_TRACE_COMP(("vStartSdcOcTimer: TIMER started!"));
    ETG_TRACE_USR4(( "vStartSdcOcTimer: End" ));
}
tVoid vd_devicemanager_tclClientHandler_Diaglog::vStopSdcOcTimer(tVoid)
{
    ETG_TRACE_USR4(( "vStopSdcOcTimer: Begin" ));
    if( m_bTimerSdcOcStarted == TRUE )
    {
        m_bTimerSdcOcStarted = FALSE;    //@todo use counter to replace this
        m_TimerSdcOc.bStop( );
        ETG_TRACE_COMP(("vStopSdcOcTimer: TIMER stopped!"));
    }
    ETG_TRACE_USR4(( "vStopSdcOcTimer: End" ));
}
#endif //MARKER_UNUSED_CODE


/*-----------------------------------------------------------------------------*
 * void vExecute(tVoid)                                                        *
 * \brief function called if evaluation timer expires it calls setTimerExpired *
 *-----------------------------------------------------------------------------*/
void vd_devicemanager_tclClientHandler_Diaglog::vExecute(tVoid)
{
#ifdef MARKER_UNUSED_CODE
    //if function vProcessTimer and underlying functions are called with this TimerID in the current code this is ignore
    tU16    u16TimerId = DEVICE_MANAGER_TIMERID_OC_USB;
#else
    tU16    u16TimerId = 1;
#endif

    m_u32CountSecondsFromStartup = 11;       // @todo: check check if startup at least x seconds ago. If not start timer again
    ETG_TRACE_USR4(( "vExecute: Begin:(callbackTimerExpired)" ));

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_USE_PRM_SDCARDSIGNALS))
    {
        ETG_TRACE_USR4(("vExecute: eConf_ENABLE_USE_PRM_SDCARDSIGNALS is enabled"));
        if( m_bTimerSdcOcStarted )
        {
            u16TimerId = DEVICE_MANAGER_TIMERID_OC_SDC;
        }
    }

    vProcessTimer( u16TimerId );
    ETG_TRACE_USR4(( "vExecute: End (callbackTimerExpired) "));
}

/*-----------------------------------------------------------------------------*
 * void ProcessDiag(tVoid)     (ICyclicDiagnosis)                                                   *
 *-----------------------------------------------------------------------------*/
void vd_devicemanager_tclClientHandler_Diaglog::ProcessDiag(vector<trDtcCodeResult> &ListCyclicDiagResults)
{
#ifdef USE_SHIFT_CYCLICDIAGNOSIS
    vProcessTimer(VD_CLIENTHANDLER_DIAGLOG_TIMER_ID);
    vCopyResultToVector(ListCyclicDiagResults);
#endif
}

#ifdef USE_DECOUPLE_CYCLICDIAGCCA

#ifndef USE_SHIFT_PARTOFDIAG_TO_ADAPTER
void vd_devicemanager_tclClientHandler_Diaglog::vStoreResultInVector(IN tU16 u16DtcCode,IN tU8 u8Result,INOUT vector<trDtcCodeResult> &ListCyclicDiagResults)
{
    trDtcCodeResult oResult;

    oResult.u8Result        = u8Result;
    oResult.u16DTroubleCode = u16DtcCode;

    m_LockList.lock();
    ListCyclicDiagResults.push_back(oResult);
    m_LockList.unlock();

}
#endif //USE_SHIFT_PARTOFDIAG_TO_ADAPTER

void vd_devicemanager_tclClientHandler_Diaglog::vCopyResultToVector(OUT vector<trDtcCodeResult> &ListCyclicDiagResults)
{
     
    if(TRUE == m_LockList.lock(TIMOUT_LOCKDIAGLOG_LIST_SEC)) //we use timout here because it could block 
    {
    //clear
    ListCyclicDiagResults.clear();

    for (vector<trDtcCodeResult>::iterator it = m_ListCyclicDiagResults.begin() ; it != m_ListCyclicDiagResults.end(); ++it)
    {
        ListCyclicDiagResults.push_back(*it);
    }
    //clear vector since 
    m_ListCyclicDiagResults.clear();
       m_LockList.unlock();
    }

}



void  vd_devicemanager_tclClientHandler_Diaglog::vSendResultOfCyclicDiag(IN vector<trDtcCodeResult> ListCyclicDiagResults)
{
     ETG_TRACE_USR4(("Begin: vSendResultOfCyclicDiag ListCyclicDiagResults.size():%d",ListCyclicDiagResults.size()));
     midw_diaglogfi_tclMsgSaveTestResultMethodStart ms;
     midw_fi_tcl_TestResult    oTestResult;

     if(ListCyclicDiagResults.size()>0) //if elements in list
     {
         //-----------------------------
         //unpack and map to diangosis type
         //-----------------------------
        for (vector<trDtcCodeResult>::iterator it = ListCyclicDiagResults.begin() ; it != ListCyclicDiagResults.end(); ++it)
        {
            ETG_TRACE_COMP(("vSendResultOfCyclicDiag : it->u16DTroubleCode: %d  , it->u8Result: %d",it->u16DTroubleCode,it->u8Result));
            oTestResult.TroubleCode   = it->u16DTroubleCode;
            oTestResult.Result.enType = (midw_fi_tcl_e8_TestResult::tenType)it->u8Result;
            ms.TestResultList.TestResultList.push_back(oTestResult);
        }
         //send it
         //--------------
         vSendResult(ms);
     }
     else
     {
         ETG_TRACE_COMP(("[INFO]: ListCyclicDiagResults.size() == 0 i.e. empty vextor"));
     }
     ETG_TRACE_USR4(("End : vSendResultOfCyclicDiag"));

} 

#endif //USE_DECOUPLE_CYCLICDIAGCCA

#ifdef USE_ENABLE_VD_DVM_CLH_DIAGLOG_ICYCLICDIAGRECEIVER
OUT /*internal*/void vd_devicemanager_tclClientHandler_Diaglog::ReceiveDiagResults(vector<trDtcCodeResult> ListCyclicDiagResults)
{
    ETG_TRACE_USR4(("Begin: ReceiveDiagResults"));

    vSendResultOfCyclicDiag(IN ListCyclicDiagResults);

    ETG_TRACE_USR4(("End  : ReceiveDiagResults"));

}
#endif

////////////////////////////////////////////////////////////////////////////////
// <EOF>
