////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Robert Bosch Car Multimedia GmbH, Hildesheim
//
// Author: Stephan Pieper
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
////////////////////////////////////////////////////////////////////////////////

/*lint -save -e737 -e774 -e776 -e845 -e831 -e849 -e1786 -e1415 */

////////////////////////////////////////////////////////////////////////////////

#if !defined(CPRA_MPEG_H_INCLUDED)
#define CPRA_MPEG_H_INCLUDED

////////////////////////////////////////////////////////////////////////////////

#include <string.h>

////////////////////////////////////////////////////////////////////////////////

namespace mpeg {

////////////////////////////////////////////////////////////////////////////////

typedef void void_t;
typedef bool bool_t;
typedef unsigned long long ulong_t;
typedef unsigned long uint_t;
typedef long long_t;
typedef long int_t;
typedef char char_t;
typedef unsigned char uchar_t;
typedef short short_t;
typedef unsigned short ushort_t;

////////////////////////////////////////////////////////////////////////////////

static bool_t  const overlap   = true;
static ulong_t const timescale = 1000000;    // [us];
static uint_t  const i         = (uint_t)-1; // invalid;

////////////////////////////////////////////////////////////////////////////////
//                                  reserved layer3 layer2 layer1
static uint_t const slotbytes[4] = {       i,     1,     1,     4 };

////////////////////////////////////////////////////////////////////////////////

static uint_t const kbitrates[2][4][16] = { // kbps (10^3 bit/second)
   {  // mpeg1; lsf = 0;
      {  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i}, // reserved
      {  0, 32, 40, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,  i}, // layer3
      {  0, 32, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,384,  i}, // layer2
      {  0, 32, 64, 96,128,160,192,224,256,288,320,352,384,416,448,  i}  // layer1
   } ,
   {  // mpeg2/2.5; lsf = 1;
      {  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i,  i}, // reserved
      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160,  i}, // layer3
      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160,  i}, // layer2
      {  0, 32, 48, 56, 64, 80, 96,112,128,144,160,176,192,224,256,  i}  // layer1
   }
};

////////////////////////////////////////////////////////////////////////////////

static uint_t const samplerates[4][4] = {
   {11025, 12000,  8000,  i}, //mpeg2.5
   {    i,     i,     i,  i}, //reserved
   {22050, 24000, 16000,  i}, //mpeg2
   {44100, 48000, 32000,  i}  //mpeg1
};

////////////////////////////////////////////////////////////////////////////////

static uint_t const allowedmodes[2][16] = { // 1 = allowed; 0 = forbidden;
   // mpeg1/layer2: allowed bitrate/mode combinations ;
   // no restriction for mpeg 2/2.5 or layer 1/2;
   {  1,  0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1,  i}, // stereo
   {  1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,  i}  // mono
};

////////////////////////////////////////////////////////////////////////////////

// bits/frame to slots/frame (samples/frame / bits/slot ~ 1/frame);
static uint_t const bit2slots[2][4] = {
   // mpeg1 (lsf = 0)           samples/frame;
   {    i,    144,    144,     12 },  //  1152   1152    384
   // mpeg2/2.5 (lsf = 1)
   {    i,     72,    144,     12 }   //   576   1152    384
   // res  layer1  layer2  layer3       layer1 layer2 layer3
};

////////////////////////////////////////////////////////////////////////////////

// crc protected data:2nd 16 bits of frame header plus start bits of audio_data;
static uint_t const protected_audiobits[6][2] = {
//mode: stereo    mono;   layer
       {   256,    128 }, //   I (bit alloc)
       {   284,    142 }, //  II (bit alloc 3-B.2a, scalefact select)
       {   308,    154 }, //  II (bit alloc 3-B.2b, scalefact select)
       {    84,     42 }, //  II (bit alloc 3-B.2c, scalefact select)
       {   124,     62 }, //  II (bit alloc 3-B.2d, scalefact select)
       {   256,    136 }  // III (side info)
};

////////////////////////////////////////////////////////////////////////////////
// layer 1/2 intensity stereo bands 4-31  8-31 12-31 16-31 (MODE_JOINT_STEREO)
static uint_t const boundary[4] = {    4,    8,   12,   16 };

////////////////////////////////////////////////////////////////////////////////

enum alloctab_e {
   TAB_INVALID =  i,
   TAB_A       =  0,
   TAB_B,
   TAB_C,
   TAB_D
};

////////////////////////////////////////////////////////////////////////////////

class Decoder {

public:

   /////////////////////////////////////////////////////////////////////////////

   static uint_t const headersize = 4;

   enum version_e {
      VERSION_MPEG_25  = 0, // 00 - MPEG 2 extension; lsf;
      VERSION_RESERVED = 1, // 01
      VERSION_MPEG_2   = 2, // 10 - (ISO/IEC 13818-3); lsf;
      VERSION_MPEG_1   = 3  // 11 - (ISO/IEC 11172-3)
   };
   enum layer_e {
      LAYER_RESERVED = 0, // 00
      LAYER_III      = 1, // 01
      LAYER_II       = 2, // 10
      LAYER_I        = 3  // 11
   };
   enum mode_e {
      MODE_STEREO         = 0, // 00
      MODE_JOINT_STEREO   = 1, // 01; I/II:intensity; III:intensity and/or ms;
      MODE_DUAL_CHANNEL   = 2, // 10; 2 mono channels;
      MODE_SINGLE_CHANNEL = 3  // 11; mono;
   };
   enum emphasis_e {
      EMPH_NONE     = 0, // 00
      EMPH_50_15MS  = 1, // 01
      EMPH_RESERVED = 2, // 10
      EMPH_CCIT_J17 = 3  // 11
   };

   /////////////////////////////////////////////////////////////////////////////

   enum direction_e {
      backward = -1,
      forward  =  1
   };

   struct context_r { // e.g. file or superblock;
      uint_t      overall_bytes;   // file size;
      ulong_t     overall_samples; // file samples;
      ulong_t     overall_time;    // file duration;

      uint_t      start_offset;    // e.g. leading id3v2.3 bytes;
      uint_t      start_bytes;     // slider;
      ulong_t     start_samples;

      direction_e drctn;     // ffwd/bwd
      int_t       factor;    // ffwd/bwd speed;

      int_t       frame_offset_bytes;
      int_t       frame_offset_samples;

      int_t       block_offset_bytes;
      int_t       block_offset_samples;

      int_t       total_offset_bytes;
      long_t      total_offset_samples;

      uint_t      total_bytes;
      ulong_t     total_samples;

      ulong_t     clt_frame_send_time; // cummulative;
      ulong_t     clt_block_send_time;
      ulong_t     rdt_frame_send_time; // absolute;
      ulong_t     rdt_block_send_time;

      context_r() {
         memset(this, 0, sizeof(context_r));
      }
   };

   /////////////////////////////////////////////////////////////////////////////

   struct state_r {
      uint_t    last_block_pos;
      uint_t    block_frames;
      char_t   *block;
      uint_t    block_bytes; // scanned bytes;
      uint_t    block_sum_kbitrate;
      uint_t    block_avg_kbitrate;
      uint_t    total_sum_kbitrate;
      uint_t    total_avg_kbitrate;
      bool_t    complete;
      uint_t    frag_pos;
      uint_t    frag_size;
      uint_t    offset;
      uint_t    total_blocks;
      uint_t    total_frames;
      uint_t    total_bytes; // content bytes
      uint_t    block_time;
      ulong_t   total_time;
      ulong_t   frame_send_time;
      ulong_t   block_send_time;
      uint_t    samplerate;
      uint_t    block_samples;
      ulong_t   total_samples;

      bool_t    variable_version;
      version_e version; // @ cbr;
      bool_t    variable_layer;
      layer_e   layer; // @ cbr;
      bool_t    variable_bitrate;
      uint_t    bitrate; // @ cbr;

      context_r context;

      uchar_t   crc;

      state_r() {
         memset(this, 0, sizeof(state_r));
         version = VERSION_RESERVED;
      }
   };

   /////////////////////////////////////////////////////////////////////////////

   struct framedesc_r {
      uint_t     id;
      char_t*    begin;
      uint_t     block_pos;
      uint_t     distance;
      uint_t     header[headersize]; // ssssssss sssvvllc bbbbssp. ccmmcoee
      version_e  version;
      bool_t     lsf;
      layer_e    layer;
      uint_t     bytesperslot;
      bool_t     crc;
      uint_t     bitrateindex;
      uint_t     kbitrate;
      uint_t     bitrate;
      bool_t     padded;
      bool_t     priv;
      mode_e     mode;
      bool_t     mono;
      bool_t     stereo;
      bool_t     allowed;
      uint_t     modeext;
      bool_t     copyright;
      bool_t     original;
      emphasis_e emphasis;
      bool_t     verisimilar;
      bool_t     sync;
      uint_t     bit2slot;
      uint_t     framesize;
      uint_t     matches;
      ushort_t   checksum;
      alloctab_e allocationtable;
      uint_t     protectedaudiobits;
      uint_t     protectedbits;
      uint_t     framesamples;
      uint_t     frametime;

      framedesc_r() {
         memset(this, 0, sizeof(framedesc_r));
      }
   };

   /////////////////////////////////////////////////////////////////////////////

   typedef void_t (*fptr_t)(framedesc_r const&, state_r const&);

   enum callback_e {
      never,
      per_frame,
      per_block
   };

   struct config_r {
      fptr_t     callback;
      callback_e call;
      uint_t     checks;
   };

   /////////////////////////////////////////////////////////////////////////////

   Decoder(fptr_t fptr = 0, callback_e const call = per_block, uint_t const checks = 0) {
      memset(&config_, 0, sizeof(config_r));
      config(fptr, call, checks);
   }
   ~Decoder()
   {}

   /////////////////////////////////////////////////////////////////////////////

   void_t config(fptr_t fptr = 0, callback_e const call = per_block, uint_t const checks = 0) {
      config_.callback = fptr;
      config_.call     = call;
      config_.checks   = checks ? checks : 3;
   }

   /////////////////////////////////////////////////////////////////////////////
   // provide state access for persistability;

   state_r const& state() const {
      return state_;
   }

   void_t state(state_r const& stat0) {
      state_ = stat0;
   }

   /////////////////////////////////////////////////////////////////////////////

   state_r const& scan(char_t *block, uint_t const size, bool_t const seq, bool_t const fix) {

      //////////////////////////////////////////////////////////////////////////

      char_t* p = block;

      if(seq) {

         p += state_.offset;
      }

      //////////////////////////////////////////////////////////////////////////

      char_t* frbegin = 0;
      uint_t  frsize  = 0;

      //////////////////////////////////////////////////////////////////////////
      // state;

      state_.last_block_pos = 0;

      state_.block_frames = 0;

      state_.block       = block;
      state_.block_bytes = size;

      state_.block_sum_kbitrate = 0;

      state_.complete     = true;
      state_.frag_pos     = 0;
      state_.frag_size    = 0;

      state_.block_samples = 0;
      state_.block_time    = 0;

      //////////////////////////////////////////////////////////////////////////
      // context;

      state_.context.block_offset_bytes   = 0;
      state_.context.block_offset_samples = 0;

      //////////////////////////////////////////////////////////////////////////

      framedesc_r frd;

      while(p < block + size - 4) {

         uint_t matches = match(p, block + size, config_.checks);
         if(1 < matches) {

            frd = handle_frame(p, matches, fix); // matching headers found;
            p += framesize(p);
            frbegin = 0;

         } else { // 1 or no match;

            if(frbegin && p == frbegin + frsize) {

               frd = handle_frame(frbegin, 1, fix); // complete frame found;
               frbegin = 0;
            }
            if(1 == matches && !frbegin) { // header found;
               frbegin = p;
               frsize  = framesize(p);
            }
            ++p;
         }
      }

      //////////////////////////////////////////////////////////////////////////

      if(frbegin) {

         state_.complete  = false;
         state_.frag_pos  = (uint_t)(frbegin - block);
         state_.frag_size = (size - state_.frag_pos) - 1;

         frd = handle_frame(frbegin, 1, fix); // incomplete frame found;
      }

      //////////////////////////////////////////////////////////////////////////

      state_.total_blocks++;

      //////////////////////////////////////////////////////////////////////////

      if(config_.callback && per_block == config_.call) {

         (*config_.callback)(frd, state_);
      }

      //////////////////////////////////////////////////////////////////////////

      return state_;
   }

   /////////////////////////////////////////////////////////////////////////////

   void_t context(uint_t const size, uint_t const offset = i, uint_t const pos = i, int_t const fac = 0) {

      // any calculated/cummulated context entries relate to
      // frame data found in scanned blocks; detected obsolete
      // data is not considered; with offset = id3v2.3 length set pos
      // to the current file position (ftell);

      //////////////////////////////////////////////////////////////////////////
      // preserve unforgiven;

      if(i != offset) {

         // force out of bounds offset to 0; // constant within file;
         state_.context.start_offset = (offset < size) ? offset : 0;
      }

      if(size) {

         state_.context.overall_bytes = size - state_.context.start_offset;
      }

      //////////////////////////////////////////////////////////////////////////

      if(i != pos) {

         position(pos, fac);

      } else {

         factor(fac);
      }

      //////////////////////////////////////////////////////////////////////////
   }

   /////////////////////////////////////////////////////////////////////////////

   void_t position(uint_t const pos, int_t const fac = 0) {

      //////////////////////////////////////////////////////////////////////////

      state_.context.block_offset_bytes = 0;
      state_.context.total_offset_bytes = 0;

      state_.context.block_offset_samples = 0;
      state_.context.total_offset_samples = 0;

      //////////////////////////////////////////////////////////////////////////

      if(pos >= state_.context.start_offset) {

         // constant during scan;
         state_.context.start_bytes = pos - state_.context.start_offset;
      }

      factor(fac);
   }

   /////////////////////////////////////////////////////////////////////////////

   void_t factor(int_t const fac) {

      state_.context.factor = fac ? fac : 1;

      state_.context.drctn = (fac < 0 ? backward : forward);
   }

   /////////////////////////////////////////////////////////////////////////////

   void_t reset() {

     state_ = state_r();
   }

   /////////////////////////////////////////////////////////////////////////////

private:

   ///////////////////////////////////////////////////////////////////////////////

   config_r config_;
   state_r  state_;

   ///////////////////////////////////////////////////////////////////////////////

   framedesc_r handle_frame(char_t* p, uint_t const matches, bool_t const fix) {

      //////////////////////////////////////////////////////////////////////////

      framedesc_r frd;

      frd.id        = state_.block_frames;
      frd.begin     = p;
      frd.block_pos = (uint_t)(p - state_.block);
      frd.distance  = frd.block_pos - state_.last_block_pos;

      memcpy(frd.header, p, headersize);

      frd.version      = version(p);
      frd.lsf          = lsf(frd.version);
      frd.layer        = layer(p);
      frd.bytesperslot = bytesperslot(p);
      frd.crc          = crc(p);
      frd.bitrateindex = bitrateindex(p);

      frd.kbitrate   = kbitrate(p);
      frd.bitrate    = bitrate(p);

      state_.samplerate = samplerate(p); // TODO: error check; may not change in file;
      frd.padded     = padded(p);
      frd.priv       = priv(p);
      frd.mode       = mode(p);

      frd.mono      = mono(p);
      frd.stereo    = stereo(p);
      frd.allowed   = allowed(p);
      frd.modeext   = modeext(p);
      frd.copyright = copyright(p);
      frd.original  = original(p);

      frd.emphasis     = emphasis(p);
      frd.verisimilar  = verisimilar(p);
      frd.sync         = sync(p);
      frd.bit2slot     = bit2slot(p);

      frd.framesamples = framesamples(p);

      frd.framesize = framesize(p);
      frd.matches   = matches;

      frd.checksum           = (ushort_t)checksum(p);
      frd.allocationtable    = allocationtable(p);;
      frd.protectedaudiobits = protectedaudiobits(p);
      frd.protectedbits      = protectedbits(p);

      //////////////////////////////////////////////////////////////////////////

      state_.last_block_pos = frd.block_pos;

      //////////////////////////////////////////////////////////////////////////
      // frames;

      ++state_.block_frames;
      ++state_.total_frames;

      //////////////////////////////////////////////////////////////////////////
      // average kbitrates;

      state_.block_sum_kbitrate += frd.kbitrate;
      state_.block_avg_kbitrate  = state_.block_sum_kbitrate / state_.block_frames;

      state_.total_sum_kbitrate += frd.kbitrate;
      state_.total_avg_kbitrate  = state_.total_sum_kbitrate / state_.total_frames;

      //////////////////////////////////////////////////////////////////////////
      // offset of next frame from beginning of next block (in seq-mode);

      state_.offset = state_.complete ? 0 : (frd.framesize - state_.frag_size) - 1;

      //////////////////////////////////////////////////////////////////////////
      // bytes/samples;

      if(state_.complete || overlap) {

         // bytes;
         state_.total_bytes += frd.framesize;

         // samples;
         state_.block_samples += frd.framesamples;
         state_.total_samples += frd.framesamples;

      } else {

         // bytes;
         state_.total_bytes += frd.framesize - state_.frag_size;

         // samples;
         state_.block_samples += (         state_.frag_size * frd.framesamples) / frd.framesize;
         state_.total_samples += ((ulong_t)state_.frag_size * frd.framesamples) / frd.framesize;
      }

      //////////////////////////////////////////////////////////////////////////
      // time;

      // frame;
      if(state_.complete || overlap) {

         frd.frametime = frametime(p); // [us];

      } else {

         frd.frametime = (state_.frag_size * frametime(p)) / frd.framesize;
     }

      // after block/total_samples frame increment;

      // block;
      state_.block_time = (uint_t)((timescale * state_.block_samples) / state_.samplerate);

      // total;
      if(state_.samplerate) {

         state_.total_time = (timescale * state_.total_samples) / state_.samplerate;
      }

      //////////////////////////////////////////////////////////////////////////
      // send times;

      // context free send_times always count positively (even when a block
      // of some context is scanned multiple times); all scanned blocks are
      // considered sequentially (even when their origin is distributed
      // regarding source context);

      if(state_.samplerate) {

         // after block/total_samples frame increment;
         state_.frame_send_time  =

            (timescale * (state_.total_samples - frd.framesamples    )) / state_.samplerate;

         state_.block_send_time =

            (timescale * (state_.total_samples - state_.block_samples)) / state_.samplerate;
      }

      //////////////////////////////////////////////////////////////////////////

      if(1 < state_.block_frames) { // ignore changes from blocks frame0 to frame1;

         ///////////////////////////////////////////////////////////////////////
         // detect variable version (true on error);

         if(false == state_.variable_version) {

            if(VERSION_RESERVED != state_.version && frd.version != state_.version) {

               state_.variable_version = true;

            } else {

               state_.version = frd.version;
            }
         }

         ///////////////////////////////////////////////////////////////////////
         // detect variable layer (true on error);

         if(false == state_.variable_layer && 1 < matches) {

            if(LAYER_RESERVED != state_.layer && frd.layer != state_.layer) {

               state_.variable_layer = true;

            } else {

               state_.layer = frd.layer;
            }
         }

         ///////////////////////////////////////////////////////////////////////
         // detect variable bitrate;

         if(false == state_.variable_bitrate) {

            if(state_.bitrate && frd.bitrate != state_.bitrate) {

               state_.variable_bitrate = true;
               state_.bitrate = 0;

            } else {

               state_.bitrate = frd.bitrate;
            }
         }

         ///////////////////////////////////////////////////////////////////////
      }

      //////////////////////////////////////////////////////////////////////////
      // overall context; precision increases with amount of decoded blocks;

      // samples;
      if(state_.total_bytes) {

         state_.context.overall_samples =

            (state_.total_samples * state_.context.overall_bytes) / state_.total_bytes;

      } else {

         // when total_bytes is 0, total_samples should be 0 as well;
         state_.context.overall_samples = 0;
      }

      // time (duration);
      if(state_.samplerate && !fix) {

         state_.context.overall_time = // [us];

            (timescale * state_.context.overall_samples) / state_.samplerate;
      }

      //////////////////////////////////////////////////////////////////////////
      // context start samples;

      // update regularly since overall bytes/samples give a more precies estimate
      // by time; start_bytes given by client (rdt); this is an estimate just
      // beeing exact in special cases (e.g. context begin/end or on other
      // positions in context depending on its data content);
      // will be used for rdt_send_times;

      if(state_.total_bytes) {

         state_.context.start_samples =

            (state_.total_samples * state_.context.start_bytes) / state_.total_bytes;

      } else {

         // when total_bytes is 0, total_samples should be 0 as well;
         state_.context.start_samples = 0;
      }


      //////////////////////////////////////////////////////////////////////////
      // context bytes/samples;

      // sample counts are interpolated; here we deviate from real sample count
      // since just a subset of overall context (file) is available to
      // parser/decoder; byte count is precise since we know exactly how many
      // context bytes are skipped/left out (determined by client pos/factor);

      if(state_.complete || overlap) {

         // bytes (exact);
         state_.context.frame_offset_bytes = state_.context.factor * (int_t)frd.framesize;

         state_.context.block_offset_bytes += state_.context.frame_offset_bytes;
         state_.context.total_offset_bytes += state_.context.frame_offset_bytes;

         // samples (interpolated);
         state_.context.frame_offset_samples = state_.context.factor * (int_t)frd.framesamples;

         state_.context.block_offset_samples += state_.context.frame_offset_samples;
         state_.context.total_offset_samples += state_.context.frame_offset_samples;

      } else {

         // bytes (exact);
         state_.context.frame_offset_bytes = state_.context.factor * (int_t)(frd.framesize - state_.frag_size);

         state_.context.block_offset_bytes += state_.context.frame_offset_bytes;
         state_.context.total_offset_bytes += state_.context.frame_offset_bytes;

         // samples (interpolated);
         state_.context.frame_offset_samples = (state_.context.factor * (int_t)(state_.frag_size * frd.framesamples)) / (int_t)frd.framesize;

         state_.context.block_offset_samples += state_.context.frame_offset_samples;
         state_.context.total_offset_samples += state_.context.frame_offset_samples;
      }

      // bytes (exact);
      state_.context.total_bytes = state_.context.start_bytes + state_.context.total_offset_bytes;
      //                                                ^this is constant;

      // samples (interpolated);
      state_.context.total_samples = state_.context.start_samples + state_.context.total_offset_samples;
      //                                                  ^this varies;

      //////////////////////////////////////////////////////////////////////////
      // context send times;

      // context "send_time" members opposed to respective state members
      // take starting position, direction and factor given by client into account;
      // they are counted positive or negative and may vary within a range
      // determined by source context;

      // cummulative send times (linear but not converging towards context limits);
      // based on scanned frame/block content increments;
      if(state_.samplerate) {

         // after context samples frame increment;
         state_.context.clt_frame_send_time =

            (timescale * (state_.context.total_samples - state_.context.frame_offset_samples)) / state_.samplerate;

         state_.context.clt_block_send_time =

            (timescale * (state_.context.total_samples - state_.context.block_offset_samples)) / state_.samplerate;

      } else {

         state_.context.clt_frame_send_time = 0;
         state_.context.clt_block_send_time = 0;
      }

      // relative send times (not linear but converging towards context limits;
      // regula de tribus on total context; varying with bitrate versus byte axis @ vbr);

      if(state_.context.overall_bytes) {

         uint_t const last_frame_total_bytes = state_.context.total_bytes - state_.context.frame_offset_bytes;
         uint_t const last_block_total_bytes = state_.context.total_bytes - state_.context.block_offset_bytes;

         // check 64 bit overflow;
         bool_t const overflow = last_frame_total_bytes &&
                                 state_.context.overall_time > 0xffffffffffffffffull / last_frame_total_bytes;

         if(overflow) {

            // reduce significant digits;
            state_.context.overall_time >>= 3;
         }

         state_.context.rdt_frame_send_time =

             (state_.context.overall_time * last_frame_total_bytes) /  state_.context.overall_bytes;

         state_.context.rdt_block_send_time =

             (state_.context.overall_time * last_block_total_bytes) /  state_.context.overall_bytes;

         if(overflow) {

            // restore scaling;
            state_.context.overall_time        <<= 3;
            state_.context.rdt_frame_send_time <<= 3;
            state_.context.rdt_block_send_time <<= 3;
         }

      } else {

         state_.context.rdt_frame_send_time = 0;
         state_.context.rdt_block_send_time = 0;
      }

      //////////////////////////////////////////////////////////////////////////

      if(config_.callback && per_frame == config_.call) {

         (*config_.callback)(frd, state_);
      }

      //////////////////////////////////////////////////////////////////////////

     return frd;
   }

   /////////////////////////////////////////////////////////////////////////////
   // extract len bits from position bit within a byte @ offset bytes from p;

   template<uint_t const offset, uint_t const bit, uint_t const len>
   struct Bits
   {
      enum {
         shift = (8  -  bit) - len,
         mask  = (2 << (len - 1)) - 1
      };
      static uint_t val(char_t* p) {
         return ((*(p + offset)) >> shift) & mask;
      }
   };

   /////////////////////////////////////////////////////////////////////////////
   // sync (byte1:11 bit, byte2:3 bit);

   bool_t sync(char_t* p) const {
      return (0xff == Bits<0, 0, 8>::val(p)) &&
             (0x07 == Bits<1, 0, 3>::val(p)) &&
              verisimilar(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // version (byte2:2 bit from 3);

   version_e version(char_t* p) const {
      return static_cast<version_e>(Bits<1, 3, 2>::val(p));
   }

   /////////////////////////////////////////////////////////////////////////////
   // lower sample frequency;

   bool_t lsf(version_e const ver) const {
      return (VERSION_MPEG_1 == ver) ? false : true;
   }

   /////////////////////////////////////////////////////////////////////////////
   // layer (byte2:2 bit from 5);

   layer_e layer(char_t* p) const {
      return static_cast<layer_e>(Bits<1, 5, 2>::val(p));
   }

   /////////////////////////////////////////////////////////////////////////////
   // bits/bytes per slot within frames (1 optional padding slot);

   uint_t bytesperslot(char_t* p) const {
      return slotbytes[layer(p)];
   }

   /////////////////////////////////////////////////////////////////////////////
   // protection (byte2:1 bit from 7);

   bool_t crc(char_t* p) const {
      return !Bits<1, 7, 1>::val(p); // 16bit crc follows header
   }

   short_t checksum(char_t const* p) const { // checksum word
      return (short_t)((uchar_t)(*(p + 4)) << 8) + ((uchar_t)(*(p + 5)) & 0x00ff);
   }

   /////////////////////////////////////////////////////////////////////////////
   // bitrate (byte3:4 bit from 0); kbps; 0 = free;  i = bad;

   uint_t bitrateindex(char_t* p) const {
      return Bits<2, 0, 4>::val(p);
   }
   uint_t kbitrate(char_t* p) const {
      return kbitrates[lsf(version(p))][layer(p)][bitrateindex(p)];
   }
   uint_t bitrate(char_t* p) const {
      return 1000 * kbitrate(p);
   }
   /////////////////////////////////////////////////////////////////////////////
   // mpeg1/layer2 allocation table;

   //            C     A           B
   // 44.1 kHz: 32 48 56, 64, 80, 96, 112, 128, 160, 192, free
   //            C     A
   // 48   kHz: 32 48 56, 64, 80, 96, 112, 128, 160, 192, free
   //            D     A           B
   // 32   kHz: 32 48 56, 64, 80, 96, 112, 128, 160, 192, free

   alloctab_e allocationtable(char_t* p) const {
      if(VERSION_MPEG_1 != version(p) || LAYER_II != layer(p)) {
         return TAB_INVALID;
      }
      uint_t const kbpc = kbitrate(p) >> 1;// kilobits per channel
      switch(samplerate(p)) {
      case 44100:
         return 80 < kbpc ? TAB_B : (48 < kbpc ? TAB_A : TAB_C);
      case 48000:
         return 48 < kbpc ? TAB_A : TAB_C;
      case 32000:
      default:
         return 80 < kbpc ? TAB_B : (48 < kbpc ? TAB_A : TAB_D);
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // crc protected data:2nd 16 bits of header plus varying bits of audio_data;

   uint_t protectedaudiobits(char_t* p) const {
      switch(layer(p)) {
      case LAYER_I:
         return protected_audiobits[0][mono(p)];
      case LAYER_II:
         return protected_audiobits[1 + (int)allocationtable(p)][mono(p)];
      case LAYER_III:
      default:
         return protected_audiobits[5][mono(p)];
      }
   }
   uint_t protectedbits(char_t* p) const {
      return 16 + protectedaudiobits(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // sampling frequency (byte3: 2 bit from 4);

   uint_t samplerate(char_t* p) const {
      return samplerates[version(p)][Bits<2, 4, 2>::val(p)];
   }

   /////////////////////////////////////////////////////////////////////////////
   // padding slot in frame (byte3: 1 bit from 6);

   bool_t padded(char_t* p) const {
      return Bits<2, 6, 1>::val(p); // layer1:32bit slot; layer 2/3: 8bit slot;
   }

   /////////////////////////////////////////////////////////////////////////////
   // private bit (byte3: 1 bit from 7);

   bool_t priv(char_t* p) const {
      return Bits<2, 7, 1>::val(p); // nonsense;
   }

   /////////////////////////////////////////////////////////////////////////////
   // channel mode (byte4: 2 bit from 0);

   mode_e mode(char_t* p) const {
      return static_cast<mode_e>(Bits<3, 0, 2>::val(p));
   }
   bool_t mono(char_t* p) const {
      return (MODE_SINGLE_CHANNEL == mode(p)) ? true : false;
   }
   bool_t stereo(char_t* p) const {
      return !mono(p);
   }
   bool_t allowed(char_t* p) const {
      return (VERSION_MPEG_1 == version(p) && LAYER_II == layer(p)) ?
             allowedmodes[mono(p)][bitrateindex(p)] : true;
   }

   /////////////////////////////////////////////////////////////////////////////
   // mode extension (byte4: 2 bit from 2);

   //        layer 1/2   layer3
   // stereo intensity   int m/s
   // 00     bands  4-31  0   0
   // 01     bands  8-31  1   0
   // 10     bands 12-31  0   1
   // 11     bands 16-31  1   1

   uint_t modeext(char_t* p) const {
      return (MODE_JOINT_STEREO == mode(p)) ? Bits<3, 2, 2>::val(p) :  i;
   }

   uint_t intensitybound(char_t* p) const {
      return (LAYER_I == layer(p) || LAYER_II == layer(p)) ? boundary[modeext(p)] : 32;
   }
   uint_t intensity(char_t* p) const {
      char_t modeextension = (char_t)modeext(p);
      return (LAYER_III == layer(p)) ? Bits<3, 7, 1>::val(&modeextension) :  i;
   }
   uint_t ms(char_t* p) const {
      char_t modeextension = (char_t)modeext(p);
      return (LAYER_III == layer(p)) ? Bits<3, 6, 1>::val(&modeextension) :  i;
   }

   /////////////////////////////////////////////////////////////////////////////
   // copyrighted audio (byte4: 1 bit from 4);

   bool_t copyright(char_t* p) const {
      return Bits<3, 4, 1>::val(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // original media (byte4: 1 bit from 5);

   bool_t original(char_t* p) const {
      return Bits<3, 5, 1>::val(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // emphasis (byte4: 2 bit from 6);

   emphasis_e emphasis(char_t* p) const {
      return static_cast<emphasis_e>(Bits<3, 6, 2>::val(p));
   }

   /////////////////////////////////////////////////////////////////////////////
   // framesize calculation; not directly extracted from header;

   uint_t bit2slot(char_t* p) const {
      return bit2slots[lsf(version(p))][layer(p)];
   }
   uint_t framesamples(char_t* p) const {
      return bit2slot(p) * (bytesperslot(p) << 3);
   }
   uint_t framesize(char_t* p) const { // bytes
      // framebits  = framesamples * bitrate / samplerate;
      // frameslots = bit2slot * framebits + padded;
      // mpeg(mpegx/layerx):
      // framesize = int(bit2slot  * bitrate / samplerate) + padded)    * slotsize
      // [bytes]         [1/frame]   [bps]     [sps]         [slots/frame][bytes]
      // mp3 (mpeg1/layer3):
      // framesize = int(144       * bitrate / samplerate) + padded)
      // slots = bits/sample;
      return uint_t((bit2slot(p) * bitrate(p) / samplerate(p)) + padded(p)) * bytesperslot(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // frametime calculation (microseconds);

   uint_t frametime(char_t* p) const {
      return (uint_t)((timescale * framesamples(p)) / samplerate(p));
   }

   /////////////////////////////////////////////////////////////////////////////
   // check plausibility;

   bool_t verisimilar(char_t* p) const {
      return i != samplerate(p)
             &&
             0 != bitrateindex(p)
             &&
             i != kbitrate(p)
             //&&
             //LAYER_III == layer(p)
             &&
             allowed(p);
   }

   /////////////////////////////////////////////////////////////////////////////
   // detect matching headers;

   uint_t match(char_t* p, char_t const* end, uint_t const checks) const {
      uint_t matches = sync(p);
      for(uint_t j = matches ? 1 : checks;
          (j < checks) &&
          ((uint_t)(end - p) >= framesize(p)) &&
          (matches += sync(p += framesize(p))) == j + 1;
          j++)
      {};
      return matches;
   }

   /////////////////////////////////////////////////////////////////////////////
};

////////////////////////////////////////////////////////////////////////////////

} // namespace mpeg;

////////////////////////////////////////////////////////////////////////////////

#endif // !defined CPRA_MPEG_H_INCLUDED

///////////////////////////////////////////////////////////////////////////////

/*lint -restore */

///////////////////////////////////////////////////////////////////////////////

