/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine mecodec_mp3.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   Mp3 codec.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_MP3_HPP_INCLUDED)
#define ME_CODEC_MP3_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "1_common/megstcodec.hpp"

#include "id3.hpp"
#include "mpeg.hpp"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct codec_mp3_t : public gstcodec_t,
                     public single_t<codec_mp3_t> {

   /* ----------------------------------------------------------------------- */

   friend class single_t<codec_mp3_t>;

   /* ----------------------------------------------------------------------- */

   virtual strings_t onexts() const {
      trace("codec_mp3:onexts()");
      static strings_t fts;
      if(fts.empty()) {
         //fts.append(string_t("mp3"));
      }
      return fts;
   }

   virtual int_t onisaudio(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 1; // file extension *.mp3 always supports audio;
   }
   virtual int_t onisvideo(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 0; // file extension *.mp3 never video;
   }

   /* ----------------------------------------------------------------------- */

   codec_mp3_t() : loop(0), src(0), line(0),
      /*dmx(0),*/ dec(0), conv(0), smpl(0), vol(0), sink(0) {

      pump = true;
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t oninit() {

      /* -------------------------------------------------------------------- */
       /*gst_init() call is moved to mediaengineserver::init  to avoid the multiple calls to platform  
         Fix for NCG3D-118451*/
      // setup pipeline;

      // mainloop;
      trace("me:codec_mp3:oninit:mainloop");
      loop = g_main_loop_new(0, false);
      asrt(NULL != loop);

      // pipeline;
      trace("me:codec_mp3:oninit:pipe");
      line = gst_pipeline_new("pipe");
      asrt(NULL != line);

      // elements;
      trace("me:codec_mp3:oninit:appsrc");
      src = (GstAppSrc*)gst_element_factory_make("appsrc", "src"); asrt(NULL != src);
      if(0 == src) {
         trace("me:codec_bt:oninit:new:appsrc:failed");
      }

      //trace("me:codec_mp3:oninit:mpegaudioparse");
      //dmx  = gst_element_factory_make("mpegaudioparse", "dmx"); asrt(NULL != dmx);
      //if(0 == dmx) {
      //   trace("me:codec_bt:oninit:new:mpegaudioparse:failed");
      //}

      /* -------------------------------------------------------------------- */
      // decoder

      dec = gst_element_factory_make("beepdec", "dec");

      if(dec) {

        trace("me:codec_mp3:oninit:dec:beepdec");

      } else {

          dec = gst_element_factory_make("flump3dec", "dec");

          if(dec) {

             trace("me:codec_mp3:oninit:dec:flump3dec");
          }
      }
      asrt(NULL != dec);
      if(0 == dec) {
         trace("me:codec_bt:oninit:new:beepdec/flump3dec:failed");
      }

      /* -------------------------------------------------------------------- */

      trace("me:codec_mp3:oninit:audioconvert");
      conv = gst_element_factory_make("audioconvert", "conv"); asrt(NULL != conv);
      if(0 == conv) {
         trace("me:codec_bt:oninit:new:audioconvert:failed");
      }

      trace("me:codec_mp3:oninit:audioresample");
      smpl = gst_element_factory_make("audioresample", "smpl"); asrt(NULL != smpl);
      if(0 == smpl) {
         trace("me:codec_bt:oninit:new:audioresample:failed");
      }

      trace("me:codec_mp3:oninit:volume");
      vol = gst_element_factory_make("volume", "vol"); asrt(NULL != vol);
      if(0 == vol) {
         trace("me:codec_bt:oninit:new:volume:failed");
      }

      trace("me:codec_mp3:oninit:alsasink");
      sink = gst_element_factory_make("alsasink", "alsa"); asrt(NULL != sink);
      if(0 == sink) {
         trace("me:codec_bt:oninit:new:alsasink:failed");
      }

      // reference;
      gst_bin_add_many(GST_BIN(line), (GstElement*)src, /*dmx,*/ dec, conv, smpl, vol, sink, NULL);

      // link;
      gst_element_link_many((GstElement*)src, /*dmx,*/ dec, conv, smpl, vol, sink, NULL);

      // callbacks;
      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
      gst_bus_add_watch(bus, cb_bus_call, this);
#if defined (ENABLE_GSTREAMER_1_0)
      gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL); // gst_bus_set_sync_handler(bus, cb_bus_sync, this) is modified to gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL)
#else
      gst_bus_set_sync_handler(bus, cb_bus_sync, this);
#endif

      gst_object_unref(bus);


      // configure;
      g_object_set(src,  "block", true, NULL); // block appsrc push_data when queue is full;
      g_object_set(sink, "sync", false, NULL); // decouple alsasink from timestamps;

      // input buffer size;
      int_t max_bytes = gst_app_src_get_max_bytes(src);

      trace("me:codec_mp3:oninit:max_bytes0:", max_bytes);

      //gst_app_src_set_max_bytes(src, 20480);
      //max_bytes = (int_t)gst_app_src_get_max_bytes(src);
      //trace("me:codec_mp3:oninit:max_bytes1:", max_bytes)

   }

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {

      /* -------------------------------------------------------------------- */

      trace("me:codec_mp3:mainloop");

      if(NULL != loop)
      {  
         g_main_loop_run(loop); // main routine;
      }
      trace("me:codec_mp3:mainloop:finished");

      /* -------------------------------------------------------------------- */

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
#if defined (ENABLE_GSTREAMER_1_0)
      gst_bus_set_sync_handler(bus, 0, 0, NULL); // gst_bus_set_sync_handler(bus, 0, 0) is modified to gst_bus_set_sync_handler(bus, 0, 0, NULL)
#else
      gst_bus_set_sync_handler(bus, 0, 0);
#endif
      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      setstate(line, GST_STATE_NULL);

      /* -------------------------------------------------------------------- */
      // cleanup;

      // delete pipeline;
      gst_element_unlink_many((GstElement*)src, /*dmx,*/ dec, conv, smpl, vol, sink, NULL);

      /* -------------------------------------------------------------------- */

      if(line) {
         gst_object_unref(line); // let line-container unref its content elements;
         line = NULL;
      }

      if(loop) {
         g_main_loop_unref(loop);
         loop = NULL;
      }

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t onfini() {

      trace("me:codec_mp3:onfini");

      if(loop) {

         trace("me:codec_mp3:onfini:mainloop:quit");

         g_main_loop_quit(loop);
      }

      trace("me:codec_mp3:onfini:gst:deinit");
      //gst_deinit(); // do not gst_deinit since subsequent gst_init fails;
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t onctrl(state_t const &state) {

      playstate_e const ps = state.playstate;

      trace("me:codec_mp3:onctrl:", estr2(ps));

      /* -------------------------------------------------------------------- */

      int_t const res = gstcodec_t::configlog(state.logconfig);

      trace("me:codec_mp3:onctrl:configlog:res:", res);

      /* -------------------------------------------------------------------- */

      switch(ps) {
         case PLAYSTATE_STOP:

            setstate(line, GST_STATE_NULL);

            break;
         case PLAYSTATE_PAUSE:

            setstate(line, GST_STATE_PAUSED);

            break;
         case PLAYSTATE_PLAY: {

            /* -------------------------------------------------------------- */
            // dev;

            trace("me:codec_mp3:onctrl:dev:", in.dev);

            g_object_set(sink, "device", out.dev().at(), NULL); // e.g. "hw:1,0";

            /* -------------------------------------------------------------- */
            // vol;

            flt_t volume = in.vol;
            volume /= 100;
            trace("me:codec_mp3:oninit:volume:", volume);

            g_object_set(vol, "volume", volume, NULL); // 0.0 - 1.0;

            /* -------------------------------------------------------------- */

            setstate(line, GST_STATE_PLAYING);

            break;
         }
         default: break;
      }
      return 0;
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t read(file_t &file, int_t const& space, buffer_t &buf) {

      /* -------------------------------------------------------------------- */
      // parse file header length and duration;

      if(out.dur().empty()) {

         Id3Tag tag;

         id3.parse_file(file, &tag);

         trace("me:codec_mp3:read:tag:title:", tag.title);
         trace("me:codec_mp3:read:tag:artist:", tag.artist);
         trace("me:codec_mp3:read:tag:album:", tag.album);

         int_t const taglength = file.tell();

         trace("me:codec_mp3:read:taglength:", taglength);

         int_t const scanlength = 100 * 1024;

         vector_t<byte_t> data(scanlength);

         int_t const scanlength1 = file.read(data.at(), data.size());

         if(0 > scanlength1) {

            trace("me:codec_mp3:read:error:", scanlength1);

         } else {

            mpeg.context(file.size(), taglength, taglength, 1); // file.tell()

            mds = mpeg.scan(data.at(), data.size(), false, false);

            int_t const avg_bitrate  = mds.total_avg_kbitrate   * 1000;
            int_t const dur_ms       = mds.context.overall_time / 1000;

            trace("me:codec_mp3:read:avg_bitrate:", avg_bitrate, " duration:", dur_ms, "ms");

            out.dur().bytes = file.size() - taglength;
            out.dur().ms    = dur_ms;

            file.seek(taglength, file_t::origin_set);
            mpeg.position(taglength);
         }
      }

      /* -------------------------------------------------------------------- */
      // derive buffer length from mp3 frame info;

      flt_t const factor_at_speed_1  =  80.0; // 400.0; // 25.0; // 50.0; //100.0; // 400.0;
      flt_t const factor_at_speed_10 =  80.0; //   5.0; // 10.0; // 20.0; // 80.0;

      flt_t const grad = (factor_at_speed_1 - factor_at_speed_10) / 9;

      trace("me:codec_mp3:read:grad:", grad);

      // linear function deriving buffer length from avg kbitrate;
      int_t factor = (factor_at_speed_1 + grad - grad * M1_ABS(out.speed()));

      // apply lower bound;
      if(factor < 40) {
         factor = 40;
      }
      trace("me:codec_mp3:read:factor:", factor);

      // apply linear function and round to multiple of 8;
      int_t bytes = rndup8(mds.block_avg_kbitrate * factor);

      // limit buffer length to a value less than total queue size;
      int_t const max_bytes = 0x10000;
      if(max_bytes < bytes) {
         bytes = max_bytes;
      }

      trace("me:codec_mp3:read:bytes:", bytes);

      /* -------------------------------------------------------------------- */

      // check space;
      if(space < bytes) {

         trace("me:codec_mp3:read:bytes:", bytes, " space:", space, " low");
         return 0;
      }

      /* -------------------------------------------------------------------- */

      // fill header;
      buf.header().pos().bytes = file.tell();
      buf.header().pos().pct   = rdt(100, buf.header().pos().bytes, file.size());

      buf.header().speed = out.speed();

      // allocate data;
      buf.data().resize(bytes);

      // read data;
      int_t const bytes1 = file.read(buf.data().at(), buf.data().size());

      if(bytes1 != bytes) {

         trace("me:codec_mp3:read:eof");

         buf.data().resize(bytes1);

         buf.header().reason = REASON_END_OF_FILE;

      } else if(bytes1 < 0) {

         buf.header().reason = REASON_READ_ERROR;
      }

      buf.header().dur().bytes = bytes1;

      /* -------------------------------------------------------------------- */
      // parse-calculate playtime of buffer;

      mpeg.position(buf.header().pos().bytes, buf.header().speed);

      int_t const seq = (1 == out.speed) ? true : false;

      mds = mpeg.scan(buf.data().at(), buf.data().size(), seq, true);

      buf.header().pos().ms = mds.context.rdt_block_send_time / 1000; // us -> ms;
      buf.header().dur().ms = mds.block_time / 1000; // us -> ms;

      /* -------------------------------------------------------------------- */
      // seek in fast mode;

      if(out.speed && 1 != out.speed) {

         int_t const offset = sign3(out.speed()) * (M1_ABS(out.speed()) - 1) * buf.header().dur().bytes;

         int_t const res = file.seek(offset, file_t::origin_cur);

         trace("me:codec_mp3:read:seek:offset:", offset, " res:", res);
      }

      /* -------------------------------------------------------------------- */

      trace("me:codec_mp3:read:file:size:", file.size(), "b buf:", buf.str());

      return bytes1;
   }

   virtual int_t write(buffer_t const &buf) {

      trace("me:codec_mp3:write:buf:", buf.str());

      /* -------------------------------------------------------------------- */
      // push buffer downstream;

      // gst queues copied buffers;
      byte_t* data = (byte_t*)malloc(buf.data().size());
      mcpy(data, buf.data().at(), buf.data().size());

      GstBuffer *gbuf = gst_buffer_new();

      #if defined(ENABLE_GSTREAMER_1_0)
         // GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)data; //GST_BUFFER_MALLOCDATA Removed from Gstreamer-1.0
         // GST_BUFFER_DATA(gbuf)       = (guint8*)data; //GST_BUFFER_DATA Removed from Gstreamer-1.0
         // GST_BUFFER_SIZE(gbuf)       = (guint  )buf.data.size(); //GST_BUFFER_SIZE Removed from Gstreamer-1.0
         gbuf = gst_buffer_new_wrapped ((guint8*)data,(guint  )buf.data().size());
         // Replacing GST_BUFFER_MALLOCDATA,GST_BUFFER_DATA,GST_BUFFER_SIZE with gst_buffer_new_wrapped ((guint8*)data,(guint  )buf.data.size()) API call
      #else
         GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)data;
         GST_BUFFER_DATA(gbuf)       = (guint8*)data;
         GST_BUFFER_SIZE(gbuf)       = (guint  )buf.data().size();
      #endif

      GstFlowReturn res = gst_app_src_push_buffer(src, gbuf); // do not unref;

      if(res) {

         trace("me:codec_mp3:write:res:", res);
      }

      /* -------------------------------------------------------------------- */

      if(REASON_END_OF_FILE == buf.header().reason) {

         out.reason = buf.header().reason;

         trace("me:codec_mp3:write:eof");

         gst_app_src_end_of_stream(src);
      }

      /* -------------------------------------------------------------------- */

      out.pos = buf.header().pos;

      forward(out);

      /* -------------------------------------------------------------------- */

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_bus_call(GstBus *bus, GstMessage *msg, gpointer that) {

       trace("me:codec_mp3:cb_bus_call");

       asrt(NULL != that);
       codec_mp3_t &codec = *(codec_mp3_t*)that;

       GstMessageType const type = GST_MESSAGE_TYPE(msg);

       trace("me:codec_mp3:cb_bus_call:type:", gst_message_type_get_name(type));

       switch(type) {
          case GST_MESSAGE_EOS: {

             trace("me:codec_mp3:cb_bus_call:EOS");

             codec.out.reason = REASON_END_OF_STREAM;

             codec.out.pos().bytes = codec.out.dur().bytes;
             codec.out.pos().ms    = codec.out.dur().ms;
             codec.out.pos().pct   = 100;

             codec.forward(codec.out); // signal owner;

             break;
          }
          case GST_MESSAGE_STATE_CHANGED: {

             GstState state0, state1;
             gst_message_parse_state_changed(msg, &state0, &state1, 0);

             trace("me:codec_mp3:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                     " state:", gst_element_state_get_name(state0),
                                        " -> ", gst_element_state_get_name(state1));
             break;
          }
          case GST_MESSAGE_STREAM_STATUS: {

             GstStreamStatusType status;
             gst_message_parse_stream_status(msg, &status, 0);

             trace("me:codec_mp3:cb_bus_call:", GST_OBJECT_NAME(msg->src), " status:", status, ":", streamstatus(status));
             break;
          }
          case GST_MESSAGE_TAG: {

             GstTagList *tags = 0;

             gst_message_parse_tag(msg, &tags);
             trace("me:codec_mp3:cb_bus_call:", GST_OBJECT_NAME(msg->src), " received:taglist");

             gst_tag_list_foreach(tags, cb_handle_tag, 0);
             gst_tag_list_free(tags);
             break;
          }
          case GST_MESSAGE_INFO: {

             GError *info = 0;
             gchar  *txt  = 0;
             gst_message_parse_info(msg, &info, &txt);
             gchar  *str  = gst_error_get_message(info->domain, info->code);

             trace("me:codec_mp3:cb_bus_call:INFO:element:name:", GST_OBJECT_NAME(msg->src));
             trace("me:codec_mp3:cb_bus_call:msg0:", info->message);
             trace("me:codec_mp3:cb_bus_call:msg1:", (str ? str : "none"));
             trace("me:codec_mp3:cb_bus_call:msg2:", (txt ? txt : "none"));

             g_error_free(info);
             g_free(str);
             g_free(txt);
             break;
          }
          case GST_MESSAGE_WARNING: {

             GError *warntxt = 0;
             gchar  *txt  = 0;
             gst_message_parse_warning(msg, &warntxt, &txt);
             gchar  *str  = gst_error_get_message(warntxt->domain, warntxt->code);

             trace("me:codec_mp3:cb_bus_call:WARN:element:name:", GST_OBJECT_NAME(msg->src));
             trace("me:codec_mp3:cb_bus_call:msg0:", warntxt->message);
             trace("me:codec_mp3:cb_bus_call:msg1:", (str ? str : "none"));
             trace("me:codec_mp3:cb_bus_call:msg2:", (txt ? txt : "none"));

             g_error_free(warntxt);
             g_free(str);
             g_free(txt);
             break;
          }
          case GST_MESSAGE_ERROR: {

             GError *error = 0;
             gchar  *txt   = 0;
             gst_message_parse_error(msg, &error, &txt);
             gchar  *str   = gst_error_get_message(error->domain, error->code);

             trace("me:codec_mp3:cb_bus_call:ERR:element:name:", GST_OBJECT_NAME(msg->src),
                                                           " domain:", gstdomain(error->domain),
                                                             " code:", error->code);
             trace("me:codec_mp3:cb_bus_call:msg0:", error->message);
             trace("me:codec_mp3:cb_bus_call:msg1:", (str ? str : "none"));
             trace("me:codec_mp3:cb_bus_call:msg2:", (txt ? txt : "none"));
             if(GST_RESOURCE_ERROR            == error->domain &&
                GST_RESOURCE_ERROR_OPEN_WRITE == error->code) {

                codec.out.reason = REASON_DEVICE_ERROR;
                codec.forward(codec.out); // signal owner;
             }
             g_error_free(error);
             g_free(str);
             g_free(txt);
             break;
          }
          default: break;
       }
       return true;

   }

   /* ----------------------------------------------------------------------- */

   GMainLoop  *loop;
   GstAppSrc  *src;
   GstElement *line, /**dmx,*/ *dec, *conv, *smpl, *vol, *sink;

   /* ----------------------------------------------------------------------- */

   id3::Id3 id3;
   mpeg::Decoder mpeg;
   mpeg::Decoder::state_r mds;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

#ifdef __cplusplus
   extern "C" {
#endif

codec_t *instance() {
   return &codec_mp3_t::inst();
}

#ifdef __cplusplus
   }
#endif

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#include "1_common/mecodecif.hpp"

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_MP3_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

