////////////////////////////////////////////////////////////////////////////////
///
/// (C) Copyright 2005 Blaupunkt GmbH, Hildesheim.
///
/// \class Id3 id3.hpp.
///
/// Parse/extract ID3 tags from mp3 files.
///
/// http://www.id3.org/
///
////////////////////////////////////////////////////////////////////////////////
///
/// \date   2005-06-20.
/// \author julian rose, martin mueller, stephan pieper.
///
/// <hr>
///
////////////////////////////////////////////////////////////////////////////////

/*lint -save -e1736 -e1704 -e801 -e506 -e774 */

////////////////////////////////////////////////////////////////////////////////

#if !defined(CPRA_ID3_H_INCLUDED)
#define CPRA_ID3_H_INCLUDED

////////////////////////////////////////////////////////////////////////////////

#include "1_common/metrace.hpp"
using namespace me;

typedef bool bool_t;
typedef unsigned long long ulong_t;
typedef unsigned long uint_t;
typedef long long_t;
typedef char char_t;
typedef unsigned char uchar_t;
typedef short short_t;
typedef unsigned short ushort_t;

enum strenc_e {
   STRENC_INVALID,
   STRENC_ISO8859_1,
   STRENC_UTF16,
   STRENC_UTF8,
   STRENC_LAST
};

struct Id3Tag
{
   Id3Tag() : valid(false), titenc(STRENC_INVALID), titsize(0), artenc(STRENC_INVALID), artsize(0),
      albenc(STRENC_INVALID), albsize(0), genenc(STRENC_INVALID), gensize(0), length(0), id(0),
      version0(0), major0(0), minor0(0) {
   }
   ~Id3Tag() {
   }

   //void_t trace() const {
   //   trace("me:id3:-------------------------"; tr.trace(level);
   //   trace("me:id3:valid:  ", valid();
   //   trace("me:id3:titenc: ", titenc().str(), "(" << titsize() << ")";
   //   trace("me:id3:title:  ", title().str(bytes_t::CHR, 1);
   //   trace("me:id3:artenc: ", artenc().str(), "(" << artsize() << ")";
   //   trace("me:id3:artist: ", artist().str(bytes_t::CHR, 1);
   //   trace("me:id3:albenc: ", albenc().str(), "(" << albsize() << ")";
   //   trace("me:id3:album:  ", album().str(bytes_t::CHR, 1);
   //   trace("me:id3:genenc: ", genenc().str(), "(" << gensize() << ")";
   //   trace("me:id3:genre:  ", genre().str(bytes_t::CHR, 1);
   //   trace("me:id3:length: ", length();
   //   trace("me:id3:id:name:", id(), ":", name().str(bytes_t::CHR, 1);
   //   trace("me:id3:version:", version0(), ".", major0(), "." << minor0();
   //   trace("me:id3:------------------------------";
   //}

   bool_t   valid;

   string_t title;
   strenc_e titenc;
   uint_t   titsize;

   string_t  artist;
   strenc_e artenc;
   uint_t   artsize;

   string_t  album;
   strenc_e albenc;
   uint_t   albsize;

   string_t  genre;
   strenc_e genenc;
   uint_t   gensize;

   uint_t   length;

   uint_t   id;
   string_t  name;

   uint_t   version0;
   uint_t   major0;
   uint_t   minor0;
};

typedef unsigned char   UTF8; /* typically 8 bits */
typedef unsigned short  UTF16;   /* at least 16 bits */

#define MAX_BROWSE_STRING 512

enum tenCpraActiveSource
{
   CPRA_NONE,
   CPRA_CD,
   CPRA_SD,
   CPRA_USB,
   CPRA_FFS
};

enum trCpraFileType
{
   CPRA_FILE_TYPE_DEFAULT,
   CPRA_FILE_TYPE_MP3,
   CPRA_FILE_TYPE_WMA,
   CPRA_FILE_TYPE_OGG,
   CPRA_FILE_TYPE_M3U,
   CPRA_FILE_TYPE_PLS,
   CPRA_FILE_TYPE_WPL,
   CPRA_FILE_TYPE_WAV
};

enum trCpraFileError
{
   CPRA_FILE_ERROR_NONE,
   CPRA_FILE_ERROR_DRM_PROTECTED,
   CPRA_FILE_ERROR_INEXISTENT,
   CPRA_FILE_ERROR_INVALID_WMA,
   CPRA_FILE_ERROR_DECODER_SKIP,
   CPRA_FILE_ERROR_READ,
   CPRA_FILE_ERROR_FORMAT
};

//CPRA_FID_FILEINFO
struct trCpraFileInfoStatus
{
   trCpraFileInfoStatus() {
      memset((void*)this, 0, sizeof(trCpraFileInfoStatus));
   }

   bool_t bIsID3; // file contains metadata;

   uchar_t aru8Title    [MAX_BROWSE_STRING];
   uchar_t aru8Artist   [MAX_BROWSE_STRING];
   uchar_t aru8Album    [MAX_BROWSE_STRING];
   uchar_t aru8File     [MAX_BROWSE_STRING]; // not fully qualified
   uchar_t aru8Directory[MAX_BROWSE_STRING]; // not fully qualified

   tenCpraActiveSource enSource; // 1 = CD, 2 = SD, 3 = USB;

   uchar_t aru8Genre[MAX_BROWSE_STRING];

   trCpraFileType rFileType; // default/mp3/wma

   uint_t tParentDirID; // absolute id of directory ../
   uint_t tDirID;       // absolute id of directory ./
   uint_t tFileID;      // relative id of file within directory ./
                      // (not counting directories or ./ or ../ as elements)

   bool_t bIsVBR; // file has variable bitrate;

   trCpraFileError enError; // file related error;

   uint_t tTitleEnc; // string encoding;
   uint_t tArtistEnc;
   uint_t tAlbumEnc;
   uint_t tGenreEnc;

   uint_t tLanguage; // iso8859 language mapping;

   uchar_t aru8TitleEnc [MAX_BROWSE_STRING];
   uchar_t aru8ArtistEnc[MAX_BROWSE_STRING];
   uchar_t aru8AlbumEnc [MAX_BROWSE_STRING];
   uchar_t aru8GenreEnc [MAX_BROWSE_STRING];

   uint_t tTitleSize; // string sizes;
   uint_t tArtistSize;
   uint_t tAlbumSize;
   uint_t tGenreSize;

   bool_t bDecoded;

   uint_t  tID3Size;
   uint_t  tFileSize;
};

////////////////////////////////////////////////////////////////////////////////

namespace id3 {

////////////////////////////////////////////////////////////////////////////////

//class Id3Tracer : public Tracer
//{
//public:
//Id3Tracer() : Tracer(TracerDefs::Parse::ComponentId, TracerDefs::Parse::ID3, "Id3::") {}
//};

////////////////////////////////////////////////////////////////////////////////
// definitions

#define SUPPORT_ID3V2_2 1        // support for ID3 V2.2.x tags
#define ID3_MAXPARSE    0x100000 // maximum parse size
#define RETRY_COUNT     3        // read attempts from the CD drive

#define UTF16BE_BOM     0xFEFF
#define UTF16LE_BOM     0xFFFE

#define IDV1_STRING_SZ  30

#define TAG_3C_ID3_v1   0x544147  // "TAG"
#define TAG_3C_ID3_v2   0x494433  // "ID3"

#define ID3_TITLE       0x01
#define ID3_ARTIST      0x02
#define ID3_ALBUM       0x04
#define ID3_GENRE       0x08
#define ID3_PICTURE     0x10

////////////////////////////////////////////////////////////////////////////////

#define MAX_STRING     512 // Joliet; was 256

#define UNSYNC_MASK    0x80
#define EXTEND_MASK    0x40
#define V22_COMPR_MASK 0x40
#define EXPERM_MASK    0x20
#define FOOTER_MASK    0x10

////////////////////////////////////////////////////////////////////////////////
// MP3 tag names as 24-bit integers (ID3v2 2.2)

enum {
   TAG_C_TT2 = 0x545432, TAG_C_TP1 = 0x545031, TAG_C_TAL = 0x54414c,
   TAG_C_TCO = 0x54434f
};

////////////////////////////////////////////////////////////////////////////////
// MP3 tag names as 32-bit integers (ID3v2 2.3 and greater)

enum {
   /*
   TAG_C_AENC = 0x41454f43,
   */
   TAG_C_APIC = 0x41504943, // embedded picture
   /*
   TAG_C_ASPI = 0x41535049, TAG_C_COMM = 0x434f4d4d, TAG_C_COMR = 0x434f4d52,
   TAG_C_ENCR = 0x454e4352, TAG_C_EQU2 = 0x45515532, TAG_C_EQUA = 0x45515541,
   TAG_C_ETCO = 0x4554434f, TAG_C_GEOB = 0x47454f42, TAG_C_GRID = 0x47524944,
   TAG_C_IPLS = 0x49504c53, TAG_C_LINK = 0x4c494f4b, TAG_C_MCDI = 0x4d434449,
   TAG_C_MLLT = 0x4d4c4c54, TAG_C_OWNE = 0x4f574e45, TAG_C_PRIV = 0x50524956,
   TAG_C_PCNT = 0x50434e54, TAG_C_POPM = 0x504f504d, TAG_C_POSS = 0x504f5353,
   TAG_C_RBUF = 0x52425546, TAG_C_RVA2 = 0x52564132, TAG_C_RVAD = 0x52564144,
   TAG_C_RVRB = 0x52565242, TAG_C_SEEK = 0x5345454b, TAG_C_SIGN = 0x5349474e,
   TAG_C_SYLT = 0x53594c54, TAG_C_SYTC = 0x53595443,
   */
   TAG_C_TALB = 0x54414c42, // album;
   /*
   TAG_C_TBPM = 0x5442504d, TAG_C_TCOM = 0x54434f4d,
   */
   TAG_C_TCON = 0x54434f4e, // genre;
   /*
   TAG_C_TCOP = 0x54434f50, TAG_C_TDAT = 0x54444154, TAG_C_TDEN = 0x5444454e,
   TAG_C_TDLY = 0x54444c59, TAG_C_TDOR = 0x54444f52, TAG_C_TDRC = 0x54445243,
   TAG_C_TDRL = 0x5444524c, TAG_C_TDTG = 0x54445447, TAG_C_TENC = 0x54454e43,
   TAG_C_TEXT = 0x54455854, TAG_C_TFLT = 0x54464c54, TAG_C_TIME = 0x54494d45,
   TAG_C_TIPL = 0x5449504c, TAG_C_TIT1 = 0x54495431,
   */
   TAG_C_TIT2 = 0x54495432, // title;
   /*
   TAG_C_TIT3 = 0x54495433, TAG_C_TKEY = 0x544b4559, TAG_C_TLAN = 0x544c414e,
   TAG_C_TLEN = 0x544c454e, TAG_C_TMCL = 0x544d434c, TAG_C_TMED = 0x544d4544,
   TAG_C_TMOO = 0x544d4f4f, TAG_C_TOAL = 0x544f414c, TAG_C_TOFN = 0x544f464e,
   TAG_C_TOLY = 0x544f4c59, TAG_C_TOPE = 0x544f5045, TAG_C_TORY = 0x544f5259,
   TAG_C_TOWN = 0x544f574e,
   */
   TAG_C_TPE1 = 0x54504531  // artist;
   /*
   TAG_C_TPE2 = 0x54504532, TAG_C_TPE3 = 0x54504533, TAG_C_TPE4 = 0x54504534,
   TAG_C_TPOS = 0x54504f53, TAG_C_TPRO = 0x5450524f, TAG_C_TPUB = 0x54505542,
   TAG_C_TRCK = 0x5452434b, TAG_C_TRDA = 0x54524441, TAG_C_TRSN = 0x5452534e,
   TAG_C_TRSO = 0x5452534f, TAG_C_TSIZ = 0x5453495a, TAG_C_TSOA = 0x54534f41,
   TAG_C_TSOP = 0x54534f50, TAG_C_TSOT = 0x54534f54, TAG_C_TSRC = 0x54535243,
   TAG_C_TSSE = 0x54535345, TAG_C_TSST = 0x54535354, TAG_C_TXXX = 0x54585858,
   TAG_C_TYER = 0x54594552, TAG_C_UFID = 0x55464944, TAG_C_USER = 0x55534552,
   TAG_C_USLT = 0x55534c54, TAG_C_WCOM = 0x57434f4d, TAG_C_WCOP = 0x57434f50,
   TAG_C_WOAF = 0x574f4146, TAG_C_WOAR = 0x574f4152, TAG_C_WOAS = 0x574f4153,
   TAG_C_WORS = 0x574f5253, TAG_C_WPAY = 0x57504159, TAG_C_WPUB = 0x57505542,
   TAG_C_WXXX = 0x57585858
   */
};

////////////////////////////////////////////////////////////////////////////////
// MP3 tag names as 24-bit integers (ID3v2 2.0 through 2.2)

/*
enum {
   TAG_C_BUF0 = 0x42555800, TAG_C_CDM0 = 0x43444d00, TAG_C_CNT0 = 0x434e5400,
   TAG_C_COM0 = 0x434f4d00, TAG_C_CRA0 = 0x43524100, TAG_C_CRM0 = 0x43524d00,
   TAG_C_ETC0 = 0x45544300, TAG_C_EQU0 = 0x45585500, TAG_C_GEO0 = 0x47454f00,
   TAG_C_IPL0 = 0x49504c00, TAG_C_LNK0 = 0x4c4e4b00, TAG_C_MCI0 = 0x4d434900,
   TAG_C_MLL0 = 0x4d4c4c00, TAG_C_PIC0 = 0x50494300, TAG_C_POP0 = 0x504f5000,
   TAG_C_REV0 = 0x52455600, TAG_C_RVA0 = 0x52564100, TAG_C_SLT0 = 0x534c5400,
   TAG_C_STC0 = 0x53544300, TAG_C_TAL0 = 0x54414c00, TAG_C_TBP0 = 0x54425000,
   TAG_C_TCM0 = 0x54434d00, TAG_C_TCO0 = 0x54434f00, TAG_C_TCR0 = 0x54435200,
   TAG_C_TDA0 = 0x54444100, TAG_C_TDY0 = 0x54445800, TAG_C_TEN0 = 0x54454e00,
   TAG_C_TFT0 = 0x54465400, TAG_C_TIM0 = 0x54494d00, TAG_C_TKE0 = 0x544b4500,
   TAG_C_TLA0 = 0x544c4100, TAG_C_TLE0 = 0x544c4500, TAG_C_TMT0 = 0x544d5400,
   TAG_C_TOA0 = 0x544f4100, TAG_C_TOF0 = 0x544f4600, TAG_C_TOL0 = 0x544f4c00,
   TAG_C_TOR0 = 0x544f5200, TAG_C_TOT0 = 0x544f5400, TAG_C_TP10 = 0x54503100,
   TAG_C_TP20 = 0x54503200, TAG_C_TP30 = 0x54503300, TAG_C_TP40 = 0x54503400,
   TAG_C_TPA0 = 0x54504100, TAG_C_TPB0 = 0x54504200, TAG_C_TRC0 = 0x54524300,
   TAG_C_TRD0 = 0x54524400, TAG_C_TRK0 = 0x54524b00, TAG_C_TSI0 = 0x54534900,
   TAG_C_TSS0 = 0x54535300, TAG_C_TT10 = 0x54543100, TAG_C_TT20 = 0x54543200,
   TAG_C_TT30 = 0x54543300, TAG_C_TXT0 = 0x54585400, TAG_C_TXX0 = 0x54585800,
   TAG_C_TYE0 = 0x54594500, TAG_C_UFI0 = 0x55464900, TAG_C_ULT0 = 0x554c5400,
   TAG_C_WAF0 = 0x57414600, TAG_C_WAR0 = 0x57415200, TAG_C_WAS0 = 0x57415300,
   TAG_C_WCM0 = 0x57434d00, TAG_C_WCP0 = 0x57435000, TAG_C_WPB0 = 0x57504200,
   TAG_C_WXX0 = 0x57585800
};
*/

////////////////////////////////////////////////////////////////////////////////

#define GENRE_MAX_NUM 80
#define GENRE_MAX_SIZ 18

static unsigned char genres[GENRE_MAX_NUM][GENRE_MAX_SIZ] = {
   "Blues",             "Classic Rock",      "Country",           "Dance",
   "Disco",             "Funk",              "Grunge",            "Hip-Hop",
   "Jazz",              "Metal",             "New Age",           "Oldies",
   "Other",             "Pop",               "R&B",               "Rap",
   "Reggae",            "Rock",              "Techno",            "Industrial",
   "Alternative",       "Ska",               "Death Metal",       "Pranks",
   "Soundtrack",        "Euro-Techno",       "Ambient",           "Trip-Hop",
   "Vocal",             "Jazz+Funk",         "Fusion",            "Trance",
   "Classical",         "Instrumental",      "Acid",              "House",
   "Game",              "Sound Clip",        "Gospel",            "Noise",
   "AlternRock",        "Bass",              "Soul",              "Punk",
   "Space",             "Meditative",        "Instrumental Pop",  "Instrumental Rock",
   "Ethnic",            "Gothic",            "Darkwave",          "Techno-Industrial",
   "Electronic",        "Pop-Folk",          "Eurodance",         "Dream",
   "Southern Rock",     "Comedy",            "Cult",              "Gangsta",
   "Top 40",            "Christian Rap",     "Pop/Funk",          "Jungle",
   "Native American",   "Cabaret",           "New Wave",          "Psychadelic",
   "Rave",              "Showtunes",         "Trailer",           "Lo-Fi",
   "Tribal",            "Acid Punk",         "Acid Jazz",         "Polka",
   "Retro",             "Musical",           "Rock & Roll",       "Hard Rock"
};

////////////////////////////////////////////////////////////////////////////////

class Id3
{

public:

   /////////////////////////////////////////////////////////////////////////////

   Id3() : frameflags_(0), id3tag_(0),
      putf16buf_(0), putf8buf_(0) {
      memset(utf16buf_,0, sizeof(utf16buf_));
      memset(utf8buf_,0, sizeof(utf8buf_));
   };
   ~Id3() {
      id3tag_    = 0;
      putf16buf_ = 0;
      putf8buf_  = 0;
   };

   /////////////////////////////////////////////////////////////////////////////
   // parse id3;

   bool_t parse_file(file_t const &file, Id3Tag *const id3tag) {

      trace("me:id3:parse_file");

      uint_t tag = 0;
      file_      = file;
      id3tag_    = id3tag;

      // clear used parts of info structure;
      fileinfo_.bIsID3 = false;
      memset(&fileinfo_.aru8Title[0],  0, MAX_BROWSE_STRING);
      memset(&fileinfo_.aru8Artist[0], 0, MAX_BROWSE_STRING);
      memset(&fileinfo_.aru8Album[0],  0, MAX_BROWSE_STRING);
      memset(&fileinfo_.aru8Genre[0],  0, MAX_BROWSE_STRING);

      frameflags_ = 0;
      memset((void_t*)&id3header_, 0, sizeof(id3header_r));

      // go to beginning of file;
      if(parse_seek(0, SEEK_SET) < 0) goto read_error;

      // start parsing id3v1 at the right position;
      if(get_int24((uchar_t*)&tag)) goto read_error;

      // parse id3v1;
      parse_id3v1();
      id3tag_->version0 = 1;

      // go to beginning of file;
      if(parse_seek(0, SEEK_SET) < 0) goto read_error;

      if(get_int24((uchar_t*)&tag)) goto read_error;

      // parse id3v2;
      if(TAG_3C_ID3_v2 == tag) {

         trace("me:id3:parse_file:ID3V2 first");

         parse_id3v2();
         id3tag_->version0 = 2;

         // check for another id3v2 tag;
         if(get_int24((uchar_t*)&tag)) goto read_error;
         if(TAG_3C_ID3_v2 == tag) {

            trace("me:id3:parse_file:ID3V2 second");

            parse_id3v2();
         }
         // rewind 24 bit;
         if(parse_seek(-3, SEEK_CUR) < 0) goto read_error;

      } else {

         // go to beginning of file;
         if(parse_seek(0, SEEK_SET) < 0) goto read_error;
      }

      id3tag_->valid  = fileinfo_.bIsID3;
      //id3tag_->title (cpra::bytes_t(fileinfo_.aru8Title,  id3tag_->titsize() + 4));
      //id3tag_->artist(cpra::bytes_t(fileinfo_.aru8Artist, id3tag_->artsize() + 4));
      //id3tag_->album (cpra::bytes_t(fileinfo_.aru8Album,  id3tag_->albsize() + 4));
      //id3tag_->genre (cpra::bytes_t(fileinfo_.aru8Genre,  id3tag_->gensize() + 4));

      id3tag_->title  = string_t((char*)fileinfo_.aru8Title);
      id3tag_->artist = string_t((char*)fileinfo_.aru8Artist);
      id3tag_->album  = string_t((char*)fileinfo_.aru8Album);
      id3tag_->genre  = string_t((char*)fileinfo_.aru8Genre);

      id3tag_->major0 = id3header_.version_.major_;
      id3tag_->minor0 = id3header_.version_.minor_;

      return true;

   read_error:
      trace("me:id3:parse_file:read error");

      return false;
   }

   /////////////////////////////////////////////////////////////////////////////

private:

   struct id3header_r {
      id3header_r() : size_(0), flags_(0), tagsfound_(0) {}
      int size_;
      struct id3version_r {
         id3version_r() : major_(0), minor_(0) {}
         uchar_t major_;
         uchar_t minor_;
      } version_;
      // arm compiler whines about char bit fields, though it eats them;
      // to remove all warnings, changed to use bit constants;
      uchar_t flags_;  // bits 7-0: unsync extend experiment footer 0 0 0 0;
      uint_t  tagsfound_;
   };


   /////////////////////////////////////////////////////////////////////////////

   int sync_safe(uchar_t const *p, uchar_t const test, int *size) const {

      trace("me:id3:sync_safe:test:", test, " size:", *size);

      uint_t i;
      // sync safe lengths for every tag after 2.4.0, but only for ID3 header length up to 2.3.0;
      int const doSync   = (test) || (id3header_.version_.major_ > 3);
      uint_t mask  = 0xff;
      uint_t shift = 8;
      *size = 0;
      if(doSync) {
         mask  = 0x7f;
         shift = 7;
      }
      for(i = 0; i<4; i++) {
         uint_t b = p[i] & mask;
         if(p[i] & (0xff ^ mask)) return -1;
         b <<= (shift * i);
         *size += (int)b;
      }
      return 0;
   }

   /////////////////////////////////////////////////////////////////////////////

   size_t test_hdr_flags() {

      trace("me:id3:test_hdr_flags:id3 header flags set");

      size_t skip = 0;

      if(id3header_.flags_ & UNSYNC_MASK) {

         // this could lead to parsing or playout failure;
         trace("me:id3:test_hdr_flags:unsynchronisation header bit");
      }

      if((id3header_.version_.major_ == 2) && (id3header_.flags_ & V22_COMPR_MASK)) {
         // not defined, should be ignored;
         trace("me:id3:test_hdr_flags:id3v2.2 compression header bit");
         return 0;
      }

      if(id3header_.flags_ & EXTEND_MASK) { // discard any extended header;
         int extendSz = 0;
         uchar_t exHdrSize[sizeof(int)];
         if(get_int32(&exHdrSize[0])) goto read_error;
         // decode 7-bit (synchsafe) ID3 extended header size;
         if(sync_safe(exHdrSize, 0, &extendSz)) goto frame_error;
         trace("me:id3:test_hdr_flags:extended header bit, size:", extendSz);
         skip += (size_t)extendSz - sizeof(exHdrSize);
      }

      if(id3header_.flags_ & FOOTER_MASK) {
         // can just ignore this, "3DI" will be treated as an unknown tag;
         trace("me:id3:test_hdr_flags:footer header bit");
      }

      if(id3header_.flags_ & EXPERM_MASK) {
         // not interested in supporting experimental format files;
         trace("me:id3:test_hdr_flags:experimental header bit");
      }
      return skip;

   frame_error:
      trace("me:id3:test_hdr_flags:frame error");
      return 0;
   read_error:
      trace("me:id3:test_hdr_flags:read error");
      return 0;
   }

   /////////////////////////////////////////////////////////////////////////////

   void_t test_frame_hdr_flags(ushort_t const flags) const {

      trace("me:id3:test_frame_hdr_flags:id3 frame flags set:flags:", flags);

      switch(id3header_.version_.major_) {
         case 3 :
            {
               if(flags & 0x8000) {
                  trace("me:id3:tag alter preservation");
               }
               if(flags & 0x4000) {
                  trace("me:id3:file alter preservation");
               }
               if(flags & 0x2000) {
                  trace("me:id3:read only");
               }
               if(flags & 0x0080) {
                  trace("me:id3:tag compression");
               }
               if(flags & 0x0040) {
                  trace("me:id3:tag encryption");
               }
               if(flags & 0x0020) {
                  trace("me:id3:group identity");
               }
            }
            break;
         case 4 :
            {
               if(flags & 0x4000) {
                  trace("me:id3:tag alter preservation");
               }
               if(flags & 0x2000) {
                  trace("me:id3:file alter preservation");
               }
               if(flags & 0x1000) {
                  trace("me:id3:read only");
               }
               if(flags & 0x0040) {
                  trace("me:id3:group identity");
               }
               if(flags & 0x0008) {
                  trace("me:id3:tag compression");
               }
               if(flags & 0x0004) {
                  trace("me:id3:tag encryption");
               }
               if(flags & 0x0002) {
                  trace("me:id3:tag unsynchronisation");
               }
               if(flags & 0x0001) {
                  trace("me:id3:data length added");
               }
            }
            break;
         case 2 :
         default :
            break;
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // low level parse;

   int parse_read(uchar_t *const buf, uint_t const len) {

      uint_t cnt   = 0;
      uint_t retry = 0;
      int  error = -1;

      while(retry < RETRY_COUNT && cnt != len) {
         retry++;
         error = (int)file_.read((char*)(char_t*)(buf + cnt), (int_t)(len - cnt));
         if(-1 != error) {
            cnt += (uint_t)error;
         }
      }
      if(cnt != len) {
         trace("me:id3:parse_read:non recoverable read error:want:", (int_t)len, " get:", (int_t)cnt);
      }
      return (int)cnt;
   }

   ///

   int parse_seek(int offs, int const rel) {

      int fpos0 = (int)file_.tell();

      trace("me:id3:parse_seek:rel:", rel, " offs:", offs, " fpos0:", fpos0);

      if(SEEK_CUR == rel) {

         if(0 > fpos0 + offs) {

            offs = -fpos0;
            trace("me:id3:parse_seek:underrun:modified offs:", offs);
         }
      }

      int noffs = 0;
      int  error = (int)file_.seek((int_t)offs, (int_t)rel);
      if(-1 != error) {
         noffs = (int)file_.tell();
         if((long_t)-1 == noffs) {
            noffs = -2l;
         }
      } else {
         noffs = -1l;
      }

      trace("me:id3:parse_seek:offs:", offs, " rel:", rel, " noffs:", noffs);

      return noffs;
   }

   ///

   int parse_tell() {

      int noffs = -1l;

      if(file_.fd) {
         noffs = (int)file_.tell();

         if(-1 == noffs) {
            noffs = -1l;
         }
      }

      trace("me:id3:parse_tell:noffs:", noffs);

      return(noffs);
   }

   ///

   #define ENABLE_UNSYNCHRONISATION true //false

   // false:
   //
   // we are aware of leading and terminating id3 tags and do not
   // want our mpeg parser to scan id3 data there;

   // true:
   //
   // we do support mp3 files containing id3 data in the middle that
   // contain mpeg frame headers;

   // unsynchronisation may spoil performance though; e.g. with embedded pictures;

   int parse_sync_seek(int const offs, int const rel) {

      trace("me:id3:parse_sync_seek:offs:", offs, " rel:", rel);

      int noffs = -1l;
      int error = -1;

      bool unsync = ((3 == id3header_.version_.major_) && (id3header_.flags_ & UNSYNC_MASK)) ||
                  ((4 == id3header_.version_.major_) && (frameflags_       & 0x0002));

      trace("me:id3:parse_sync_seek:major:", id3header_.version_.major_,
                                  " flags:", id3header_.flags_,
                             " frameflags:", frameflags_,
                                 " unsync:", unsync);

      if(ENABLE_UNSYNCHRONISATION && unsync && SEEK_CUR == rel) {

         // crazy data seek for brain dead unsynchronization;
         int     cnt     = 0;
         uint_t  retry   = 0;
         uchar_t buffer[2048];
         uchar_t lbuf    = 0;
         uint_t  tempcnt = 0;
         int     counter = 0;

         while(RETRY_COUNT > retry && cnt != offs && 0 != error) {
            retry++;
            tempcnt = (offs - cnt) > 2048 ? (uint_t)2048 : (uint_t)(offs - cnt);
            error = (int)file_.read((char*)(char_t *)buffer, (int_t)tempcnt);

            trace("me:id3:parse_sync_seek:retry:", (int_t)retry,
                                      " tempcnt:", (int_t)tempcnt,
                                         " offs:", (int_t)offs,
                                      " counter:", (int_t)counter,
                                        " error:", (int_t)error);
            if(-1 != error) {
               cnt  += error;
               retry = 0;
               for(counter = 0; counter < error; counter++) {

                  trace("me:id3:parse_sync_seek:retry:", (int_t)retry,
                                                " cnt:", (int_t)cnt,
                                            " tempcnt:", (int_t)tempcnt,
                                               " offs:", (int_t)offs,
                                            " counter:", (int_t)counter);

                  if((0xff == lbuf) && (0x00 == buffer[counter])) cnt--;
                  lbuf = buffer[counter];
               }
            }
         }
         if(cnt != offs) {
            trace("me:id3:parse_sync_seek:non recoverable seek/read error:want:", offs, " get:", cnt);
         }
         return cnt;

      } else {

         // normal seek for normal data;
         error = (int)file_.seek((int_t)offs, (int_t)rel);

         if(-1 != error) {
            noffs = (int)file_.tell();
            if((long_t)-1 == noffs) {
               noffs = -2l;
            }
         } else {
            noffs = -1l;
         }

         trace("me:id3:parse_sync_seek:offs:", offs, " rel:", rel, " error:", error, " noffs:", noffs);

         return noffs;
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // util;

   void_t remove_white_space(char_t *buf, int const n) const {

      int cnt = 0;

      // find string end;
      while((0x00 != *(buf + cnt)) && (cnt < n)) {
         cnt++;
      }

      // find white spaces;
      while((0x20 == *(buf + cnt) || 0x00 == *(buf + cnt)) && cnt >= 0) {
         *(buf + cnt--) = 0;
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // formatted get (pod);

   int get_int32(uchar_t *buf) {

      trace("me:id3:get_int32:4");

      int cnt = 0;

      // little endian requires conversion when reading an int from MP3 file;
      if(file_.fd) {
         for(int i = 0; i < 4; i++) {
            cnt = parse_read(&buf[3 - i], 1);
            if(1 != cnt) return -1;
         }
      } else return -1;
      return 0;
   }

   ///

   int get_int24(uchar_t *buf) {

     trace("me:id3:get_int24:3");

      int cnt = 0;

      // little endian requires conversion when reading an int from MP3 file;
      if(file_.fd) {
         for(int i = 0; i < 3; i++) {
            cnt = parse_read(&buf[2 - i], 1);
            if(1 != cnt) return -1;
         }
      } else return -1;
      return 0;
   }

   ///

   int get_int16(uchar_t *buf) {

      trace("me:id3:get_int16:2");

      int cnt = 0;

      // little endian requires conversion when reading an int from MP3 file;
      if(file_.fd) {
         for(int i = 0; i < 2; i++) {
            cnt = parse_read(&buf[1 - i], 1);
            if(1 != cnt) return -1;
         }
      } else return -1;
      return 0;
   }

   ///

   int get_char(char_t *buf) {

      trace("me:id3:get_char:1");

      int cnt = 0;

      if(file_.fd) {
         cnt = parse_read((uchar_t*)buf, 1);
         if(1 != cnt) return -1;
      } else return -1;
      return 0;
   }

   /////////////////////////////////////////////////////////////////////////////
   // formatted get (UTF, id3v1);

   int get_utf_from_id3v1(UTF8 *ibuf) {

      UTF8*  buf = ibuf;
      size_t cnt = (size_t)parse_read((uchar_t*)utf8buf_, (uint_t)IDV1_STRING_SZ);
      if(IDV1_STRING_SZ != cnt) goto read_error;

      // id3v1 contains up to 30 not neccessarily zeroterminated characters;
      // http://www.id3.org/ID3v1
      *(utf8buf_ + IDV1_STRING_SZ) = 0;

      remove_white_space((char_t*)utf8buf_, IDV1_STRING_SZ);

      sprnt((char*)buf,  MAX_STRING - 1, "%s", (char const*)utf8buf_);

     trace("me:id3:get_utf_from_id3v1:buf:", (char*)buf);

      return 0;

   read_error:
      return -1;
   }

   ///

   int get_utfgenre_from_id3v1(UTF8 *ibuf) {
      UTF8* buf = ibuf;
      uchar_t cbuf;

      // strings are not limited, force that;
      *(buf) = 0;

      if(parse_seek(34, SEEK_CUR) < 0) goto read_error;

      if(get_char((char_t*)&cbuf)) goto read_error;

      if(GENRE_MAX_NUM > cbuf) {

       putf8buf_ = genres[cbuf];
       sprnt((char*)buf,  MAX_STRING - 1, "%s", (char const*)putf8buf_);
      }

      return 0;

   read_error:
      return -1;
   }

   /////////////////////////////////////////////////////////////////////////////
   // formatted get (UTF, id3v2);

   int get_utf_from_id3v2(UTF8 *ibuf, size_t const isize, uint_t& encoding, uint_t& strsize) {

      size_t  cnt        = 0;
      size_t  size       = isize;
      size_t  size_trunc = 0;
      uchar_t cbuf;
      UTF8*   buf = ibuf;

      int curroffs = parse_tell();
      if(curroffs < 0) goto read_error;

      if(file_.fd) {

         // encoding; 1 byte;
         if(get_char((char_t*)&cbuf)) goto read_error;
         size--;

         trace("me:id3:get_utf_from_id3v2:isize:", (int_t)isize, " size:", (int_t)size, " cbuf:", (int_t)cbuf);

         if(0 == cbuf) {

            // ISO-8859-1 encoded, terminated with 0x00, tU8[MAX_STRING]

            size_trunc = size > MAX_STRING - 1 ? MAX_STRING - 1 : size;
            cnt = (size_t)parse_read((uchar_t*)utf8buf_, size_trunc);
            if(cnt != size_trunc) goto read_error;

            // most strings are not limited, force that
            *(utf8buf_ + size_trunc) = 0;

            putf8buf_ = utf8buf_;

            encoding = 1; // ISO-8859-1;
            strsize  = size_trunc + 1;
            memcpy((char_t*)buf, (char_t const*)putf8buf_, MAX_STRING - 1);

         } else if(1 == cbuf) {

            // UTF-16 encoded, endianess encoded in BOM, terminated with 0x00 0x00, tU16[MAX_STRING]

            UTF16 sbuf;

            // bom; 2 byte;
            if(get_int16((uchar_t*)&sbuf)) goto read_error;

            //#define HEURISTIC

            #if defined(HEURISTIC)
            if(UTF16BE_BOM != sbuf && UTF16LE_BOM != sbuf) {
               // missing bom; assume 1st UTF-16 character right behind encoding byte;
               parse_seek(-2, SEEK_CUR);
            } else
            #endif
               size -= 2;

            size_trunc = size > (MAX_STRING * 2 - 2) ? (MAX_STRING * 2 - 2) : size;
            cnt = (size_t)parse_read((uchar_t*)utf16buf_, size_trunc);

            // most strings are not limited, force that
            *(utf16buf_ + size_trunc / 2) = 0;

            if(cnt != size_trunc) goto read_error;

            if(UTF16BE_BOM == sbuf) {
               // stream is BE, convert it;
               size_t i;
               for(i = 0; i < (cnt / 2); i++) {
                  // flip bytes
                  utf16buf_[i] = ((utf16buf_[i] & 0xFF) << 8) | ((utf16buf_[i] >> 8) & 0xFF);
               }
            } else if(UTF16LE_BOM == sbuf) {
               // stream is LE, nothing to do
            } else {
               // endianess unknown; assume LE;
               trace("me:id3:get_utf_from_id3v2:missing bom:curoffs:", (int_t)curroffs, " isize:", (int_t)isize);

               #if !defined(HEURISTIC)
                  // accept id3 standard conformant tags only;
                  // return global false on missing bom;
                  goto frame_error;
               #endif
            }

            putf16buf_ = utf16buf_;

            encoding = 2; // UTF-16
            strsize  = size_trunc;
            memcpy((char_t*)buf,  (char_t const*)putf16buf_, MAX_STRING - 1);

         } else if((2 == cbuf) && (4 == id3header_.version_.major_)) {

            // UTF-16 big endian encoded, terminated with 0x00 0x00, tU16[MAX_STRING];

            size_t i;

            size_trunc = size > (MAX_STRING * 2 - 2) ? (MAX_STRING * 2 - 2) : size;
            cnt = (size_t)parse_read((uchar_t*)utf16buf_, size_trunc);

            if(cnt != size_trunc) goto read_error;

            // most strings are not limited, force that;
            *(utf16buf_ + size_trunc / 2) = 0;

            // stream is BE, convert it;
            for(i = 0; i < (cnt / 2); i++) {
               // flip bytes
               utf16buf_[i] = ((utf16buf_[i] & 0xFF) << 8) | ((utf16buf_[i] >> 8) & 0xFF);
            }

            putf16buf_ = utf16buf_;

            encoding = 2; // UTF-16
            strsize  = size_trunc;

            memcpy((char_t*)buf,  (char_t const*)putf16buf_, MAX_STRING - 1);

         } else if((3 == cbuf) && (4 == id3header_.version_.major_)) {

            // UTF-8 encoded, terminated with 0x00, tU8[MAX_STRING];

            size_trunc = size > MAX_STRING - 1 ? MAX_STRING - 1 : size;
            cnt = (size_t)parse_read(buf, size_trunc);

            // most strings are not limited, force that;
            *(utf8buf_ + size_trunc) = 0;

            encoding = 3; // UTF-8
            strsize  = size_trunc;

            if(cnt != size_trunc) goto read_error;

         } else goto frame_error;

      } else {
         encoding = 0; // invalid
         return -1;
      }

      remove_white_space((char_t*)buf, MAX_STRING);

      if(size != size_trunc) {
         // skip over non-read data
         if(parse_seek((int)(size - size_trunc), SEEK_CUR) < 0) goto read_error;
      }

      return 0;

   read_error:

      trace("me:id3:get_utf_from_id3v2:read error in id3v2 read");

      return -1;

   frame_error:

      trace("me:id3:get_utf_from_id3v2:frame size error in id3v2 read:curoffs:", (int_t)curroffs, " size:", (int_t)size);

      // set file position to where it is supposed to be;
      parse_seek(curroffs + (int)size, SEEK_SET);
      return 0;
   }

   ///

   int convert_utf_id3v2_genre(UTF8 *ibuf) {

      UTF8* buf = ibuf;
      int genreval;

      if((buf[0] == '(') &&
         (buf[1] >= '0') && (buf[1] <= '9') &&
         (((buf[2] >= '0') && (buf[2] <= '9') && (buf[3] <= ')')) || (buf[2] == ')'))) {
         genreval = atoi((char const*)ibuf + 1);
         if((0 <= genreval) && (GENRE_MAX_NUM > genreval)) {
            putf8buf_ = genres[genreval];
            //id3tag_->genenc(1); // ISO8859-1;
            sprnt((char*)buf, MAX_STRING - 1, "%s", (char const*)putf8buf_);
         }
      }
      return 0;
   }

   /////////////////////////////////////////////////////////////////////////////
   // parse id3v1;

   void_t parse_id3v1() {

      trace("me:id3:parse_id3v1");

      uint_t tag = 0;

      if(file_.fd) {

         if(parse_seek(-128, SEEK_END) < 0) goto read_error;
         // FIXME - files < 128?

         if(get_int24((uchar_t*)&tag)) goto read_error;

         trace("me:id3:parse_id3v1:tag:", (int_t)tag);

         switch(tag) {
            case TAG_3C_ID3_v1:
               {
                  trace("me:id3:parse_id3v1:ID3V1");

                  fileinfo_.bIsID3 = true;

                  // ISO-8859-1 encoded, not terminated, tU8[IDV1_STRING_SZ];
                  get_utf_from_id3v1(fileinfo_.aru8Title);
                  id3tag_->titenc = (strenc_e)1;
                  id3tag_->titsize = IDV1_STRING_SZ + 1;
                  get_utf_from_id3v1(fileinfo_.aru8Artist);
                  id3tag_->artenc = (strenc_e)1;
                  id3tag_->artsize = IDV1_STRING_SZ + 1;
                  get_utf_from_id3v1(fileinfo_.aru8Album);
                  id3tag_->albenc = (strenc_e)1;
                  id3tag_->albsize = IDV1_STRING_SZ + 1;
                  get_utfgenre_from_id3v1(fileinfo_.aru8Genre);
                  id3tag_->genenc = (strenc_e)1;
                  id3tag_->gensize = GENRE_MAX_SIZ + 1;
               }
               break;

            default :
               break;
         }

         // set file position back to the beginning of file;
         parse_seek(0, SEEK_SET);
         return;

      read_error:
         trace("me:id3:parse_id3v1:read error in id3v1");

         // set file position back to the beginning of file;
         parse_seek(0, SEEK_SET);
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // parse id3v2;

   void_t parse_id3v2() {

      int   done = 0, skipcnt = 0;
      uchar_t headersize[sizeof(int)];
      int  curroffs, id3endoffs, parseendoffs;

      if(get_char((char_t*)&id3header_.version_.major_)) goto read_error;
      if(get_char((char_t*)&id3header_.version_.minor_)) goto read_error;
      if(get_char((char_t*)&id3header_.flags_))          goto read_error;
      if(get_int32(&headersize[0]))                      goto read_error;

      curroffs = parse_tell();
      if(0 > curroffs) goto read_error;

      // ID3 version determines which tags are accepted;

      // decode 7-bit (synchsafe) ID3 header size;
      if(sync_safe(headersize, 1, &id3header_.size_)) goto frame_error;

      id3endoffs = curroffs + id3header_.size_;

      trace("me:id3:parse_id3v2:version:2.", id3header_.version_.major_, ".", id3header_.version_.minor_,
                            " headersize:", id3header_.size_,
                                 " flags:", id3header_.flags_);

      if(id3header_.flags_) {
         skipcnt = (int)test_hdr_flags();
      }

      // read and discard extended header;
      if(parse_seek(skipcnt, SEEK_CUR) < 0) goto read_error;

      curroffs = parse_tell();
      if(curroffs < 0) goto read_error;

      // we have a hard limit - if not all the frames are found we will stop parsing;
      parseendoffs = id3endoffs > ID3_MAXPARSE ? ID3_MAXPARSE : id3endoffs;

      if((3 == id3header_.version_.major_) ||
         (4 == id3header_.version_.major_)) {

         while((curroffs < parseendoffs) &&
               (id3header_.tagsfound_ != (ID3_TITLE | ID3_ARTIST | ID3_ALBUM | ID3_GENRE)) &&
               !done) {
            uint_t frame = 0;
            uint_t size  = 0;
            uint_t   tagSizeI;
            uchar_t *tagSize = (uchar_t*)&tagSizeI;

            if(get_int32((uchar_t*)&frame)) goto read_error;

            // reached the padding?
            if(0 == frame) break;
            if(get_int32(tagSize)) goto read_error;
            if(sync_safe(tagSize, 0, (int*)&size)) goto frame_error;

            trace("me:id3:parse_id3v2:", (char)((frame & 0xFF000000) >> 24),
                                        (char)((frame & 0xFF0000) >> 16),
                                        (char)((frame & 0xFF00) >> 8),
                                        (char)((frame & 0xFF)),
                         " framesize:", (int_t)tagSizeI,
                              " size:", (int_t)size);

            if(size > (uint_t)id3header_.size_) goto frame_error;

            if(get_int16((uchar_t*)&frameflags_)) goto read_error;

            if(frameflags_) {
               test_frame_hdr_flags(frameflags_);
            }

            uint_t encoding = 0;
            uint_t strsize  = 0;

            switch(frame) {
               case TAG_C_TIT2:
                  trace("me:id3:parse_id3v2:TAG_C_TIT2");

                  id3header_.tagsfound_ |= ID3_TITLE;
                  if(get_utf_from_id3v2(fileinfo_.aru8Title, size, encoding, strsize)) goto read_error;
                  id3tag_->titenc = (strenc_e)encoding;
                  id3tag_->titsize = strsize;
                  break;

               case TAG_C_TPE1 :
                  trace("me:id3:parse_id3v2:TAG_C_TPE1");

                  id3header_.tagsfound_ |= ID3_ARTIST;
                  if(get_utf_from_id3v2(fileinfo_.aru8Artist, size, encoding, strsize)) goto read_error;
                  id3tag_->artenc = (strenc_e)encoding;
                  id3tag_->artsize = strsize;
                  break;

               case TAG_C_TALB :
                  trace("me:id3:parse_id3v2:TAG_C_TALB");
                  id3header_.tagsfound_ |= ID3_ALBUM;
                  if(get_utf_from_id3v2(fileinfo_.aru8Album, size, encoding, strsize)) goto read_error;
                  id3tag_->albenc = (strenc_e)encoding;
                  id3tag_->albsize = strsize;
                  break;

               case TAG_C_TCON :
                  trace("me:id3:parse_id3v2:TAG_C_TCON");
                  id3header_.tagsfound_ |= ID3_GENRE;
                  if(get_utf_from_id3v2(fileinfo_.aru8Genre, size, encoding, strsize)) goto read_error;
                  id3tag_->genenc = (strenc_e)encoding;
                  id3tag_->gensize = strsize;
                  convert_utf_id3v2_genre(fileinfo_.aru8Genre);
                  break;

               case TAG_C_APIC :
                  trace("me:id3:parse_id3v2:TAG_C_APIC");
                  // skip picture; temp solution;
                  id3header_.tagsfound_ |= ID3_PICTURE;
                  if(parse_sync_seek((int)size, SEEK_CUR) < 0) goto read_error;
                  break;

               case 0 :
                  trace("me:id3:parse_id3v2:end of frames in this tag");
                  done = 1;
                  break;

               default:
                  trace("me:id3:parse_id3v2:skip over data:size:", (int_t)size);
                  if(parse_sync_seek((int)size, SEEK_CUR) < 0) goto read_error;
                  break;
            }
            curroffs = parse_tell();
            if(0 > curroffs) goto read_error;
         }

         trace("me:id3:parse_id3v2:set file position to end of id3 tag");
         if(parse_seek(id3endoffs, SEEK_SET) < 0) goto read_error;

      } else

         #ifdef SUPPORT_ID3V2_2

            if(2 == id3header_.version_.major_) {

               trace("me:id3:parse_id3v2:officially unsupported id3 version ID3V2.2.x");

               while((curroffs < parseendoffs) &&
                     (id3header_.tagsfound_ != (ID3_TITLE | ID3_ARTIST | ID3_ALBUM | ID3_GENRE)) &&
                     !done) {

                  uint_t frame = 0;
                  uint_t size  = 0;
                  uint_t   tagSizeI;
                  uchar_t *tagSize = (uchar_t*)&tagSizeI;

                  if(get_int24((uchar_t*)&frame)) goto read_error;

                  // reached the padding?
                  if(0 == frame) break;

                  if(get_int24(tagSize)) goto read_error;

                  size = *tagSize & 0xFFFFF;

                  trace("me:id3:parse_id3v2:", (int_t)((frame & 0xFF0000) >> 16),
                                           (int_t)((frame & 0xFF00)   >>  8),
                                           (int_t)((frame & 0xFF)          ),
                                    " size:", (int_t)size);

                  if(size > (uint_t)id3header_.size_) goto frame_error;

                  uint_t encoding = 0;
                  uint_t strsize  = 0;

                  switch(frame) {
                     case TAG_C_TT2 :
                        id3header_.tagsfound_ |= ID3_TITLE;
                        if(get_utf_from_id3v2(fileinfo_.aru8Title, size, encoding, strsize)) goto read_error;
                        id3tag_->titenc = (strenc_e)encoding;
                        id3tag_->titsize = strsize;
                        break;

                     case TAG_C_TP1 :
                        id3header_.tagsfound_ |= ID3_ARTIST;
                        if(get_utf_from_id3v2(fileinfo_.aru8Artist, size, encoding, strsize)) goto read_error;
                        id3tag_->artenc = (strenc_e)encoding;
                        id3tag_->artsize = strsize;
                        break;

                     case TAG_C_TAL :
                        id3header_.tagsfound_ |= ID3_ALBUM;
                        if(get_utf_from_id3v2(fileinfo_.aru8Album, size, encoding, strsize)) goto read_error;
                        id3tag_->albenc = (strenc_e)encoding;
                        id3tag_->albsize = strsize;
                        break;

                     case TAG_C_TCO :
                        id3header_.tagsfound_ |= ID3_GENRE;
                        if(get_utf_from_id3v2(fileinfo_.aru8Genre, size, encoding, strsize)) goto read_error;
                        id3tag_->genenc = (strenc_e)encoding;
                        id3tag_->gensize = strsize;
                        convert_utf_id3v2_genre(fileinfo_.aru8Genre);
                        break;

                     case 0 :
                        trace("me:id3:parse_id3v2:end of frames in this tag");
                        done = 1;
                        break;

                     default:
                        trace("me:id3:parse_id3v2:skip over data:size:", (int_t)size);
                        if(parse_seek((int)size, SEEK_CUR) < 0) goto read_error;
                        break;
                  }
                  curroffs = parse_tell();
                  if(curroffs < 0) goto read_error;
               }

               // set file position to the end of the ID3 tag;
               if(parse_seek(id3endoffs, SEEK_SET) < 0) goto read_error;

            } else

         #endif // SUPPORT_ID3V2_2
            goto unsupp_id3;

      if(id3header_.tagsfound_) fileinfo_.bIsID3 = true;

      return;

   unsupp_id3:
      // set file position back to the end of id3 tag;
      parse_seek(id3header_.size_, SEEK_SET);
      trace("me:id3:parse_id3v2:unsupported ID3 version");
      return;
   read_error:
      // set file position back to the end of id3 tag;
      parse_seek(id3header_.size_, SEEK_SET);
      trace("me:id3:parse_id3v2:read error in id3v2 read");
      return;
   frame_error:
      // set file position back to the end of id3 tag;
      parse_seek(id3header_.size_, SEEK_SET);
      trace("me:id3:parse_id3v2:frame size error in id3v2 read");
      return;
   }

   /////////////////////////////////////////////////////////////////////////////

private:

   /////////////////////////////////////////////////////////////////////////////

   file_t file_;

   ushort_t frameflags_;
   Id3Tag*  id3tag_;

   id3header_r id3header_;
   trCpraFileInfoStatus  fileinfo_;

   UTF16 utf16buf_[MAX_STRING];
   UTF16 const *putf16buf_;

   UTF8 utf8buf_[MAX_STRING];
   UTF8 const *putf8buf_;

   /////////////////////////////////////////////////////////////////////////////

   Id3(Id3 const&);
   Id3& operator=(Id3 const&);
};

////////////////////////////////////////////////////////////////////////////////

} // namespace cpra

////////////////////////////////////////////////////////////////////////////////

#endif // !defined CPRA_ID3_H_INCLUDED

////////////////////////////////////////////////////////////////////////////////

