/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine mecodec_bt.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2014
 *
 *   Bluetooth streaming codec.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_BT_HPP_INCLUDED)
#define ME_CODEC_BT_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "1_common/megstcodec.hpp"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct codec_bt_t : public gstcodec_t,
                    public single_t<codec_bt_t> {

   /* ----------------------------------------------------------------------- */

   friend class single_t<codec_bt_t>;

   /* ----------------------------------------------------------------------- */

   virtual strings_t onexts() const {
      trace("codec_bt:onexts()");
      static strings_t fts;
      if(fts.empty()) {
         fts.append(string_t("bt"));
      }
      return fts;
   }

   virtual int_t onisaudio(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 1; // file extension *.bt always supports bt audio;
   }
   virtual int_t onisvideo(string_t const &ext0) const {
      if(0 == onexts().find_f(ext0)) return -1;
      return 0; // file extension *.bt never supports bt video;
   }

   /* ----------------------------------------------------------------------- */

   codec_bt_t() : loop(0), line(0), src(0), caps_sbc(0), caps_aac(0), filter(0), is_aac(0),buffer_unavailable(0),
      queue(0), bin(0), aconv(0), asmpl(0), vol(0), sink(0), read_tid(0), fd(-1){

      pump = false;
#ifndef ENABLE_GSTREAMER_1_0
      csource = 0;
      controller = 0;
#endif
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t oninit() {

      trace("me:codec_bt:oninit");
      /* -------------------------------------------------------------------- */
      buffer_unavailable = false;
      /*gst_init() call is moved to mediaengineserver::init  to avoid the multiple calls to platform  
         Fix for NCG3D-118451*/
      /* -------------------------------------------------------------------- */
      // setup pipeline;

      // mainloop;
      loop = g_main_loop_new(0, false); asrt(NULL != loop);
        if(0 == loop) {
         trace("me:codec_bt:oninit:new:loop:failed");
      }

      // pipeline;
      line = gst_pipeline_new("line"); asrt(NULL != line);
        if(0 == line) {
         trace("me:codec_bt:oninit:new:line:failed");
        }

      trace("me:codec_bt:oninit:pipeline");

      /* -------------------------------------------------------------------- */

      src = gst_element_factory_make("appsrc", "src"); asrt(NULL != src);
        if(0 == src) {
         trace("me:codec_bt:oninit:new:appsrc:failed");
        }

      /* -------------------------------------------------------------------- */
#if defined (ENABLE_GSTREAMER_1_0)
      caps_sbc = gst_caps_new_simple("audio/x-sbc", NULL, NULL);
#else
    caps_sbc = gst_caps_new_simple("audio/x-sbc", NULL);
#endif
        if(0 == caps_sbc) {
         trace("me:codec_bt:oninit:new:audio/x-sbc:failed");
        }

      caps_aac = gst_caps_new_simple("audio/mpeg", "mpegversion", G_TYPE_INT, 2,
                                     "stream-format", G_TYPE_STRING, "adts",
                                     NULL);
        if(0 == caps_aac) {
         trace("me:codec_bt:oninit:new:audio/mpeg:failed");
        }

      filter = gst_element_factory_make("capsfilter", "filter");
        if(0 == filter) {
         trace("me:codec_bt:oninit:new:capsfilter:failed");
        }

      /* -------------------------------------------------------------------- */

      // decoder bin created dynamically in onctrl;

      /* -------------------------------------------------------------------- */

      aconv = gst_element_factory_make("audioconvert", "aconv"); asrt(NULL != aconv);
        if(0 == aconv) {
         trace("me:codec_bt:oninit:new:audioconvert:failed");
        }

      asmpl = gst_element_factory_make("audioresample", "asmpl"); asrt(NULL != asmpl);
        if(0 == asmpl) {
         trace("me:codec_bt:oninit:new:audioresample:failed");
        }

      vol  = gst_element_factory_make("volume", "vol" ); asrt(NULL != vol);
        if(0 == vol) {
         trace("me:codec_bt:oninit:new:audioresample:failed");
        }
        else
        {
#ifndef ENABLE_GSTREAMER_1_0

            controller = gst_object_control_properties ((GObject*)vol, "volume", NULL);

            if(controller != NULL)
            {
                 csource = (GstControlSource*)gst_interpolation_control_source_new();
                 if(NULL != csource)
                 {
                     gst_interpolation_control_source_set_interpolation_mode((GstInterpolationControlSource*)csource, GST_INTERPOLATE_LINEAR);
                     gst_controller_set_control_source (controller, "volume", csource);
                 }
            }
#endif
        }
      /* -------------------------------------------------------------------- */
      // queue;

      queue = gst_element_factory_make("queue2", "queue2"); asrt(NULL != queue);
        if(0 == queue) {
         trace("me:codec_bt:oninit:new:queue2:failed");
        }

      /* -------------------------------------------------------------------- */
      // alsasink;

      sink = gst_element_factory_make("alsasink", "sink"); asrt(NULL != sink);
        if(0 == sink) {
         trace("me:codec_bt:oninit:new:alsasink:failed");
        }

      /* -------------------------------------------------------------------- */

      g_object_set((GObject*)src,  "block", true,  NULL);

      g_object_set((GObject*)sink, "sync" ,             false,          NULL);
      g_object_set((GObject*)sink, "can-activate-pull", true,          NULL);
      g_object_set((GObject*)sink, "buffer-time",       (gint64)400000, NULL);

      /* -------------------------------------------------------------------- */
      // link;

      // elements added and linked dynamically in onctrl;

      /* -------------------------------------------------------------------- */
      // callbacks;

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));
      gst_bus_add_watch(bus, cb_bus_call, this);

      #if defined(ENABLE_GSTREAMER_1_0)
         gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL);
         // gst_bus_set_sync_handler(bus, cb_bus_sync, this) is modified to gst_bus_set_sync_handler(bus, cb_bus_sync, this, NULL)
      #else
         gst_bus_set_sync_handler(bus, cb_bus_sync, this);
      #endif
      gst_object_unref(bus);

      // no hotlinking;

      /* -------------------------------------------------------------------- */

      trace("me:codec_bt:oninit:finished");
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {

      /* -------------------------------------------------------------------- */

      trace("me:codec_bt:mainloop");

      if(NULL != loop)
      {
         g_main_loop_run(loop); // main routine;
      }
      trace("me:codec_bt:mainloop:finished");

      /* -------------------------------------------------------------------- */

      GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(line));

      #if defined(ENABLE_GSTREAMER_1_0)
         gst_bus_set_sync_handler(bus, 0, 0, NULL);
         // gst_bus_set_sync_handler(bus, 0, 0) is modified to gst_bus_set_sync_handler(bus, 0, 0, NULL)
      #else
         gst_bus_set_sync_handler(bus, 0, 0);
      #endif
      gst_object_unref(bus);

      /* -------------------------------------------------------------------- */

      GstStateChangeReturn res = gst_element_set_state(line, GST_STATE_NULL);

      trace("me:codec_bt:mainloop:GST_STATE_NULL:res:", res);

      /* -------------------------------------------------------------------- */
      // handle async statechange;

      int_t seconds = 0;
      while(GST_STATE_CHANGE_ASYNC == res) {

         GstState st0, st1;
         st1 = st0 = GST_STATE_NULL;
         GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);

         trace("me:codec_bt:state:async:wait:", (int_t)timeout);

         res = gst_element_get_state(line, &st0, &st1, timeout);

         trace("me:codec_bt:state:async:", st0, " -> ", st1, " res:", res);

         if(GST_STATE_CHANGE_ASYNC == res) {

            trace("me:codec_bt:state:still:async:seconds:", ++seconds);
         }
      }

      /* -------------------------------------------------------------------- */

      stoptimers();

      /* -------------------------------------------------------------------- */

      if(-1 != fd) {

         close(fd);
         fd = -1;
      }

      /* -------------------------------------------------------------------- */
      // cleanup;

      // delete pipeline;

      if(1 == is_aac) {
          gst_element_unlink_many((GstElement*)src, filter, bin, aconv, vol, queue, sink, NULL);
      } else {
          gst_element_unlink_many((GstElement*)src, filter, bin, aconv, asmpl, queue, sink, NULL);
      }

      if(caps_sbc) {
         gst_object_unref(caps_sbc);
         caps_sbc = NULL;
      }
      if(caps_aac) {
         gst_object_unref(caps_aac);
         caps_aac = NULL;
      }
      if(line) {
         gst_object_unref(line);
         line = NULL;
      }

      if(loop) {
         g_main_loop_unref(loop);
         loop = NULL;
      }
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t onfini() {

      trace("me:codec_bt:onfini");

      if(loop) {

         trace("me:codec_bt:onfini:mainloop:quit");

         g_main_loop_quit(loop);
      }

      trace("me:codec_bt:onfini:gst:deinit");
      //gst_deinit();
   }


   /* ----------------------------------------------------------------------- */

   virtual int_t onctrl(state_t const &state) {

      playstate_e const ps = state.playstate;

      trace("me:codec_bt:onctrl:", estr2(ps));

      /* -------------------------------------------------------------------- */

      int_t const res = gstcodec_t::configlog(state.logconfig);

      trace("me:codec_bt:onctrl:configlog:res:", res);

      /* -------------------------------------------------------------------- */

      switch(ps) {
         case PLAYSTATE_STOP: {

            if(1 == is_aac) {

                 mute(vol, true);
             }
             stoptimers();
             gst_element_set_state(line, GST_STATE_NULL);
             break;
         }
         case PLAYSTATE_PAUSE: {

            stoptimers();
            gst_element_set_state(line, GST_STATE_PAUSED);
            break;
         }
         case PLAYSTATE_PLAY: {

            /* -------------------------------------------------------------- */
            buffer_unavailable = false;
            GstState st0 = GST_STATE_NULL;
            gst_element_get_state(line, &st0, 0, GST_CLOCK_TIME_NONE);
            if(GST_STATE_PLAYING == st0) {
               gst_element_set_state(line, GST_STATE_NULL);
            }

            /* -------------------------------------------------------------- */
            // output device dev;

            trace("me:codec_bt:onctrl:output:dev:", out.dev());

            g_object_set(sink, "device", out.dev().at(), NULL); // e.g. "hw:1,0";

            /* -------------------------------------------------------------- */
            // caps filter & decoder;

            string_t const url = in.url; // e.g.: url=/dev/btaac.aac.bt;

            trace("me:codec_bt:onctrl:url:", url);

            string_t const ft0 = url.from(url.find_f('.') + 1); // e.g. ft0=aac.bt;
            string_t const fmt = ft0.upto(ft0.find_f('.') - 1); // e.g. fmt=aac;

            trace("me:codec_bt:onctrl:fmt:", fmt);


            if(string_t("sbc") == fmt) {

               trace("me:codec_bt:onctrl:caps:sbc");

               // attach caps to filter;
               g_object_set(G_OBJECT(filter), "caps", caps_sbc, NULL);

               // create decoder;
               bin = gst_element_factory_make("sbcdec", "bin"); asrt(NULL != bin);
               if(0 == bin) {
                  trace("me:codec_bt:oninit:new:sbcdec:failed");
               }

               g_object_set((GObject*)queue, "ring-buffer-max-size", (gint64)20000, NULL); // fix for NCG3D-139406

               is_aac = 0;

            } else if(string_t("aac") == fmt) {

               trace("me:codec_bt:onctrl:caps:aac");

               // attach caps to filter;
               g_object_set(G_OBJECT(filter), "caps", caps_aac, NULL);

               // create decoder;
               bin = gst_element_factory_make("beepdec.aac", "bin"); asrt(NULL != bin);
               if(0 == bin) {
                  trace("me:codec_bt:oninit:new:beepdec.aac:failed");
               }

               g_object_set((GObject*)queue, "ring-buffer-max-size", (gint64)20000, NULL);// fix for NCG3D-139406

               // vol;
               flt_t volume = out.vol;
               volume /= 100;
               trace("me:codec_bt:onctrl:volume:", volume);

               g_object_set(vol, "volume", volume, NULL); // 0.0 - 1.0;

               is_aac = 1;

            } else {
               trace("me:codec_bt:onctrl:format:unsupported:", fmt);
               return 0;
            }

            /* -------------------------------------------------------------- */
            // add and link;

            if(string_t("aac") == fmt) {

                gst_bin_add_many(GST_BIN(line), src, filter, bin, aconv, vol, queue, sink, NULL);
                gst_element_link_many(src, filter, bin, aconv, vol, queue, sink, NULL);

            } else {

                gst_bin_add_many(GST_BIN(line), src, filter, bin, aconv, asmpl, queue, sink, NULL);
                gst_element_link_many(src, filter, bin, aconv, asmpl, queue, sink, NULL);
            }

            /* -------------------------------------------------------------- */

            gst_element_set_state(line, GST_STATE_PLAYING);

#ifndef ENABLE_GSTREAMER_1_0
             GValue vol_min = { 0, };
             GValue vol_max = { 0, };
             g_value_init (&vol_min, G_TYPE_DOUBLE);
             g_value_init (&vol_max, G_TYPE_DOUBLE);

             g_value_set_double (&vol_min, 0.0);
             g_value_set_double (&vol_max, 1.0);

             trace("codec_av:onctrl:RAMP applied on Volume");
             if(NULL != csource)
             {
                 gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,0 * GST_MSECOND, &vol_min);
                 gst_interpolation_control_source_set ((GstInterpolationControlSource*)csource,in.sourceswitchramp() * GST_MSECOND, &vol_max);
             }
#endif
            /* -------------------------------------------------------------- */

            if(0 == read_tid) { // start reading bt data;

               trace("me:codec_bt:onctrl:start:read_tid");

               if(string_t("aac") == fmt) {
                   read_tid = g_timeout_add(100, (GSourceFunc)cb_read, this);

               } else {
                   read_tid = g_timeout_add(10, (GSourceFunc)cb_read, this);
               }
            }

            /* -------------------------------------------------------------- */

            break;
         }
         default:break;
      }
      trace("me:codec_bt:onctrl:pos:", in.pos().str());

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   void_t stoptimers() {

      if(read_tid) {
         trace("me:codec_bt:stoptimers:stop:read_tid");
         g_source_remove(read_tid); // stop playtime;
         read_tid = 0;
      }
   }

   /* ----------------------------------------------------------------------- */

   static gboolean cb_bus_call(GstBus *bus, GstMessage *msg, gpointer that) {

      trace("me:codec_bt:cb_bus_call");

      asrt(NULL != that);
      codec_bt_t &codec = *(codec_bt_t*)that;

      GstMessageType const type = GST_MESSAGE_TYPE(msg);

      trace("me:codec_bt:cb_bus_call:type:", gst_message_type_get_name(type));

      switch(type) {
         case GST_MESSAGE_EOS: {

            trace("me:codec_bt:cb_bus_call:EOS");
            break;
         }
         case GST_MESSAGE_STATE_CHANGED: {

            GstState state0, state1;
            gst_message_parse_state_changed(msg, &state0, &state1, 0);

            trace("me:codec_bt:cb_bus_call:", GST_OBJECT_NAME(msg->src),
                                   " state:", gst_element_state_get_name(state0),
                                      " -> ", gst_element_state_get_name(state1));
            break;
         }
         case GST_MESSAGE_STREAM_STATUS: {

            GstStreamStatusType status;
            gst_message_parse_stream_status(msg, &status, 0);

            trace("me:codec_bt:cb_bus_call:", GST_OBJECT_NAME(msg->src), " status:", status, ":", streamstatus(status));
            break;
         }
         case GST_MESSAGE_INFO: {

            GError *info = 0;
            gchar  *txt  = 0;
            gst_message_parse_info(msg, &info, &txt);
            gchar  *str  = gst_error_get_message(info->domain, info->code);

            trace("me:codec_bt:cb_bus_call:INFO:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_bt:cb_bus_call:msg0:", info->message);
            trace("me:codec_bt:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_bt:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(info);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_WARNING: {

            GError *warntxt = 0;
            gchar  *txt  = 0;
            gst_message_parse_warning(msg, &warntxt, &txt);
            gchar  *str  = gst_error_get_message(warntxt->domain, warntxt->code);

            trace("me:codec_bt:cb_bus_call:WARN:element:name:", GST_OBJECT_NAME(msg->src));
            trace("me:codec_bt:cb_bus_call:msg0:", warntxt->message);
            trace("me:codec_bt:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_bt:cb_bus_call:msg2:", (txt ? txt : "none"));

            g_error_free(warntxt);
            g_free(str);
            g_free(txt);
            break;
         }
         case GST_MESSAGE_ERROR: {

            GError *error = 0;
            gchar  *txt   = 0;
            gst_message_parse_error(msg, &error, &txt);
            gchar  *str   = gst_error_get_message(error->domain, error->code);

            trace("me:codec_bt:cb_bus_call:ERR:element:name:", GST_OBJECT_NAME(msg->src), " domain:", gstdomain(error->domain), " code:", error->code);
            trace("me:codec_bt:cb_bus_call:msg0:", error->message);
            trace("me:codec_bt:cb_bus_call:msg1:", (str ? str : "none"));
            trace("me:codec_bt:cb_bus_call:msg2:", (txt ? txt : "none"));

            if(GST_RESOURCE_ERROR == error->domain &&
               GST_RESOURCE_ERROR_OPEN_WRITE == error->code) {

               codec.out.reason = REASON_DEVICE_ERROR;
               codec.forward(codec.out); // signal owner;
            }
            g_error_free(error);
            g_free(str);
            g_free(txt);
            break;
         }
         default: break;
      }
      return true;
   }

   /* ----------------------------------------------------------------------- */

   //#define DEBUG_TRACE_AAC

   #define PATTERN_SIZE     5
   #define ADTS_HEADER_SIZE 7

   /* ----------------------------------------------------------------------- */

   #define SWAP_ENDIANESS(value) ((((int)(value) & 0x000000ff) << 24) | (((int)(value) & 0x0000ff00) <<  8) | \
                                  (((int)(value) & 0x00ff0000) >>  8) | (((int)(value) & 0xff000000) >> 24) )

   /* ----------------------------------------------------------------------- */

   static int_t sync_aac(byte_t * const buf0, int_t const size0) {

      byte_t *buf1 = buf0;
      int_t found  = 1;

      while(false == (0xB0 == buf1[0] && 0x90 == buf1[1] &&
                      0x80 == buf1[2] && 0x03 == buf1[3] && 0x00 == buf1[4])) {

         //trace("me:codec_bt:sync:0x:", string_t::hed, buf1[0], " ",

         if((size0 - 5) == (++buf1 - buf0)) {
            found = 0;
            break;
         }
      }
      return found ? buf1 - buf0 : -1;  // offset;
   }

   static gboolean cb_read(gpointer *that) {

      asrt(NULL != that);

      codec_bt_t &codec = *(codec_bt_t*)that;

       if(0 == codec.read_tid) {
          return false;
       }

      /* -------------------------------------------------------------------- */

      // set name to this thread visible in top;
      //thr_t::setname(string_t("gt_bt"));

      /* -------------------------------------------------------------------- */

      static string_t fmt;

      static int adtsheader   [2] = { 0 };
      static char muxeddata[8192] = { 0 };
      static int_t remaining = 0;

      /* -------------------------------------------------------------------- */

      if(-1 == codec.fd) {

         /* ----------------------------------------------------------------- */
         // open bluetooth pipe;

         // url -> input device;
         string_t const url = codec.in.url; // e.g.: url=/dev/btaac.aac.bt;
         string_t const dev = url.upto(url.find_f('.') - 1); // e.g. dev=/dev/btaac;

         trace("me:codec_bt:cb_read:url:", url, " input:dev:", dev);

         codec.fd = open(dev.at(), O_RDONLY | O_NONBLOCK);

         if(0 > codec.fd) {

            trace("me:codec_bt:cb_read:err:", err_str());
            asrt(false);
         }

         string_t const ft0 = url.from(url.find_f('.') + 1); // e.g. ft0=aac.bt;
                        fmt = ft0.upto(ft0.find_f('.') - 1); // e.g. fmt=aac;

         trace("me:codec_bt:cb_read:fmt:", fmt);

         /* ----------------------------------------------------------------- */
         // initialize aac frame parser;

         if(string_t("aac") == fmt) {

            remaining = 0;

            trace("me:codec_bt:create adts aac header");
            // http://wiki.multimedia.cx/index.php?title=ADTS

            adtsheader[0] = 0xffful << 20 | // 12 sync;
                                1ul << 19 | //  1 container: 1=mpeg2, 2=mpeg4;
                                0ul << 17 | //  2 layer
                                1ul << 16 | //  1 protection absent
                                1ul << 14 | //  2 aac low complexity profile;
                                4ul << 10 | //  4 sampling rate index 44100;
                                0ul <<  9 | //  1 private
                                2ul <<  6 | //  3 channels: 2
                                0ul <<  5 | //  1 original
                                0ul <<  4 | //  1 home
                                0ul <<  3 | //  1 copyright ID
                                0ul <<  2 | //  1 copyright ID start
                                0ul <<  0;  //  2 aac frame length (crc ? 7 : 9) + size(aac frame);

            adtsheader[1] =     0ul << 21 | // 11 aac frame length (crc ? 7 : 9) + size(aac frame) continued;
                            0x7fful << 10 | // 11 aac buffer fullness
                                0ul <<  8 | //  2 number of aac frames in adts frame minus 1;
                                0ul <<  0;  //  8 of 16 crc; 0 if unprotected;
         }

         /* ----------------------------------------------------------------- */
      }

      /* -------------------------------------------------------------------- */
      // wait for data;

      fd_set fds;
      FD_ZERO(&fds);
      FD_SET(codec.fd, &fds);
      static int buffer_unavailable_count = 0;
      struct timeval timeout;
      timeout.tv_sec  = 2; //3;
      timeout.tv_usec = 0;
      const int_t AAC_MAX_BUFFER_COUNT = 30;
      const int_t SBC_MAX_BUFFER_COUNT = 300;
      int const res = select(codec.fd + 1, &fds, 0, 0, &timeout);

      if(-1 == res) {

         trace("me:codec_bt:cb_read:select:err:", err_str());
         return true; // pipe error; retry;

      } else if(0 == res) {

         buffer_unavailable_count++;
         if(string_t("aac") == fmt){
             if(buffer_unavailable_count > AAC_MAX_BUFFER_COUNT && false == codec.buffer_unavailable)
             {
               codec.out.reason = REASON_BUFFER_UNAVAILABLE;
               codec.forward(codec.out); // signal owner;
               codec.buffer_unavailable = true;
               buffer_unavailable_count = 0;
             }
         }
         else{
              if(buffer_unavailable_count > SBC_MAX_BUFFER_COUNT && false == codec.buffer_unavailable)
             {
               codec.out.reason = REASON_BUFFER_UNAVAILABLE;
               codec.forward(codec.out); // signal owner;
               codec.buffer_unavailable = true;
               buffer_unavailable_count = 0;
             }
         }
         return true; // got no data in timeout interval; retry;
      }
      asrt(0 < res);

      /* ----------------------------------------------------------------- */
      // read;

      int_t const want = 4096;

      byte_t *const data = (byte_t*)malloc(want);

      int_t const got = ::read(codec.fd, data, want);

      trace("me:codec_bt:cb_read:want:", want, " got:", got, " fmt:", fmt);
      trace("me:codec_bt:cb_read:data:", hed, data[0], " ", data[1], " ", data[2], " ", data[3], " ",
                                              data[4], " ", data[5], " ", data[6], " ", data[7]);

      if(0 > got) {

         trace("me:codec_bt:cb_read:read:err:", err_str());
         free(data);
         buffer_unavailable_count =0;
         codec.buffer_unavailable = false;
         return false;

      } else if(0 == got) {

         buffer_unavailable_count =0;
         codec.buffer_unavailable = false;
         trace("me:codec_bt:cb_read:read:eof:0");
         free(data);
         return true;
      }
      asrt(0 < got);

      /* -------------------------------------------------------------------- */
      // write sbc;

      if(0 == fmt.compare(string_t("sbc"))) { // format sbc;
         buffer_unavailable_count =0;
         #if 0

            file_t orig("/var/opt/bosch/dynamic/media/db/orig.sbc", "ab+");

            orig.open();

            int_t const res = orig.write(data, got);

            trace(warn, "pe:codec_bt:cb_read:write:sbc:orig:", res);

         #endif

         #if defined(ENABLE_GSTREAMER_1_0)
            GstBuffer *gbuf = gst_buffer_new(); //Constant variable to normal variable
            // GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)data; // receiver frees data; // GST_BUFFER_MALLOCDATA(gbuf) removed
            // GST_BUFFER_DATA(gbuf)       = (guint8*)data; // GST_BUFFER_DATA(gbuf) Removed
            // GST_BUFFER_SIZE(gbuf)       = (guint  )got; // GST_BUFFER_SIZE(gbuf) Removed
            gbuf = gst_buffer_new_wrapped ((guint8*)data,(guint  )got);

            // GST_BUFFER_MALLOCDATA,GST_BUFFER_DATA,GST_BUFFER_SIZE
              //are replaced with gst_buffer_new_wrapped ((guint8*)data,(guint  )got) call
         #else
            GstBuffer *const gbuf = gst_buffer_new();
            GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)data; // receiver frees data;
            GST_BUFFER_DATA(gbuf)       = (guint8*)data;
            GST_BUFFER_SIZE(gbuf)       = (guint  )got;
         #endif

         GstFlowReturn const flow = gst_app_src_push_buffer((GstAppSrc*)codec.src, gbuf); // do not unref;
         buffer_unavailable_count =0;
         if(true == codec.buffer_unavailable)
         {
             codec.out.reason = REASON_BUFFER_AVAILABLE;
             codec.forward(codec.out); // signal owner;
             codec.buffer_unavailable = false;
         }
         if(GST_FLOW_OK != flow) {

            trace("me:codec_bt:cb_read:flow:", flow);
         }
         return true;
      }

      /* -------------------------------------------------------------------- */
      // write aac;

      if(0 == fmt.compare(string_t("aac"))) { // format aac;

         int_t used = 0;
         buffer_unavailable_count =0;
         if(true == codec.buffer_unavailable)
         {
             codec.out.reason = REASON_BUFFER_AVAILABLE;
             codec.forward(codec.out); // signal owner;
             codec.buffer_unavailable = false;
         }
         #if defined(DEBUG_TRACE_AAC)
            trace("me:codec_bt:cb_read:before mux:remaining: ", remaining);
         #endif

         mcpy((void *)(muxeddata + remaining), (const void *)data, got);

         remaining += got;

         #if defined(DEBUG_TRACE_AAC)
            trace("me:codec_bt:cb_read:after mux:remaining:", remaining, " got:", got);
         #endif

         while(0x10 < remaining) {

            int_t const offset = sync_aac(muxeddata + used, remaining);

            if(-1 == offset) {
               trace("me:codec_bt:cb_read:sync not found 1");
               break;
            }

            used += offset + PATTERN_SIZE;

            int_t   usedperframe = offset    + PATTERN_SIZE;
            byte_t* buf          = muxeddata + used;
            int_t   framesize    = 0;
            int_t   lengthsize   = 0;

            while(0xff == *buf) {

               ++buf;
               framesize += 0xff;
               ++lengthsize;
            }
            framesize += *buf++;

            #if defined(DEBUG_TRACE_AAC)
                trace(warn, "pe:codec_bt:cb_read:sync found:offset:", offset,
                                                      " remaining: ", remaining, " framesize:", framesize);
            #endif

            ++lengthsize;

            if((offset + PATTERN_SIZE + lengthsize + framesize) > remaining) {

               used -= (offset + PATTERN_SIZE);
               break;
            }

            used += lengthsize;
            usedperframe += lengthsize;

            int_t const framelength = framesize + ADTS_HEADER_SIZE;
            byte_t * const aacframe = (byte_t*)malloc(framelength);
            asrt(NULL != aacframe);

            /* -------------------------------------------------------------- */
            // add size field to header;

            int adtsheader1[2] = { 0 };

            adtsheader1[0] = SWAP_ENDIANESS(adtsheader[0] | (framelength >> 11));
            adtsheader1[1] = SWAP_ENDIANESS(adtsheader[1] | (framelength << 21));

            mcpy((void *)aacframe, (const void *)adtsheader1, ADTS_HEADER_SIZE);

            /* -------------------------------------------------------------- */

            mcpy(aacframe + ADTS_HEADER_SIZE, buf, framesize);

            used         += framesize;
            usedperframe += framesize;
            remaining    -= usedperframe;

            #if defined(DEBUG_TRACE_AAC)
               trace(warn, "pe:codec_bt:cb_read:frame found:used:", used,  " perframe:", usedperframe,
                                                     " remaining:", remaining, " push:", framelength);
            #endif

            /* -------------------------------------------------------------- */
            // record;

            #if 0

               file_t orig("/var/opt/bosch/dynamic/media/db/orig.aac", "ab+");
               file_t adts("/var/opt/bosch/dynamic/media/db/adts.aac", "ab+");

               orig.open();
               adts.open();

               int_t const res1 = orig.write(data, got);
               int_t const res2 = adts.write((byte_t*)aacframe, framelength);

               trace("me:codec_bt:cb_read:write:aac:orig:", res1, " adts:", res2);

            #endif

            /* -------------------------------------------------------------- */
            // push;

            #if defined(ENABLE_GSTREAMER_1_0)
               GstBuffer *gbuf = gst_buffer_new(); // Constant variable to normal variable

               // GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)aacframe; // receiver frees data;//GST_BUFFER_MALLOCDATA Removed from Gstreamer-1.0
               // GST_BUFFER_DATA(gbuf)       = (guint8*)aacframe; //GST_BUFFER_DATA Removed from Gstreamer-1.0
               // GST_BUFFER_SIZE(gbuf)       = (guint  )framelength; //GST_BUFFER_SIZE Removed from Gstreamer-1.0

               gbuf = gst_buffer_new_wrapped ((guint8*)aacframe,(guint  )framelength);
               // Replacing GST_BUFFER_MALLOCDATA, GST_BUFFER_DATA,GST_BUFFER_SIZE with gst_buffer_new_wrapped ((guint8*)aacframe,(guint  )framelength) call
            #else
               GstBuffer *const gbuf = gst_buffer_new();
               GST_BUFFER_MALLOCDATA(gbuf) = (guint8*)aacframe; // receiver frees data;
               GST_BUFFER_DATA(gbuf)       = (guint8*)aacframe;
               GST_BUFFER_SIZE(gbuf)       = (guint  )framelength;
            #endif

            GstFlowReturn const flow = gst_app_src_push_buffer((GstAppSrc*)codec.src, gbuf); // do not unref;

            if(GST_FLOW_OK != flow) {

               trace("me:codec_bt:cb_read:aac:flow:", flow);
            }

            /* -------------------------------------------------------------- */
         }

         // sim4hi 180629: inserted free of data as fix for coverity CID-184133
         if(data) {

            free(data);
         }

         if(remaining + used >= 8192) {

            trace("me:codec_bt:cb_read:sync not found 2");
            return false;

         } else if(remaining) {

            mcpy(muxeddata, muxeddata + used, remaining);
            trace("me:codec_bt:cb_read:copy remaining:", remaining);
            return true;
         }
         return true;
      }

      /* -------------------------------------------------------------------- */
      // format unknown;

      if(data) {

         free(data);
      }
      trace("me:codec_bt:cb_read:unknown fmt:", fmt);
      return false;

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   GMainLoop  *loop;

   GstCaps *caps_sbc, *caps_aac;

   // pipeline;
   GstElement *line, *src, *filter, *queue, *bin, *aconv, *asmpl, *vol, *sink;

   // timers;
   guint read_tid;

   int_t fd, is_aac;
   bool buffer_unavailable;
#ifndef ENABLE_GSTREAMER_1_0
   GstController* controller;
   GstControlSource* csource;
#endif
   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

#ifdef __cplusplus
   extern "C" {
#endif

codec_t *instance() {
   return &codec_bt_t::inst();
}

#ifdef __cplusplus
   }
#endif

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

#include "1_common/mecodecif.hpp"

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_BT_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
