/*
 * MediaEngineRipper.cpp
 *
 *  Created on: Mar 25, 2014
 *      Author: thm3hi
 */

//#include "rootdaemon/perdclient.hpp"

#include <sys/dir.h>
#include <dlfcn.h>

#include "MediaEngineRipper.h"

static int defaultCodecFn()
{
    //ENTRY
    me::traces("server:ripper");
    return -1;
}

void MediaEngineRipper::SaveContext()
{
    //ENTRY_INTERNAL

    FILE *fp = fopen("/tmp/MediaEngineRipper.context", "w");
    if (fp) {
        fprintf(fp, "%s\n", mObservingStateMachineName);
        fclose(fp);
    }
}

void MediaEngineRipper::LoadContext()
{
    //ENTRY_INTERNAL

    FILE *fp = fopen("/tmp/MediaEngineRipper.context", "r");
    if (fp) {
        fscanf(fp, "%s", mObservingStateMachineName);
        fclose(fp);
    } else {
        mObservingStateMachineName[0] = 0;
    }
}

int MediaEngineRipper::GetTargetPlaybackState(tPEPlaybackState& playbackState)
{
    //ENTRY_INTERNAL

    int res = 0;

    /* get the target state in state machine */
    tState state;
    res = GetTargetState(OUT state);
    if (res) {
        me::trace("GetTargetState failed:", res);
    }

    /* map state machine state to playback state */
    if (!strcmp(state.name, "stop")) {
        playbackState = PE_PBS_STOPPEDSTATE;
    }
    else if (!strcmp(state.name, "pause")) {
        playbackState = PE_PBS_PAUSEDSTATE;
    }
    else if (!strcmp(state.name, "play")) {

        playbackState = PE_PBS_PLAYINGSTATE;
    }
    else {
        me::trace("GetTargetPlaybackState unknown state:", state.name);
        playbackState = PE_PBS_UNKNOWN;
    }

    return res;
}

MediaEngineRipper::MediaEngineRipper()
{
    mPEState = me::state_t();
    mPEState.handle = HANDLE_NONE;
    mPEState.local = 1;
    mComponentId = 0;
    mNULLCodec.exts = (tFnExts *)defaultCodecFn;
    mNULLCodec.config = (tFnConfig *)defaultCodecFn;
    mNULLCodec.init = (tFnInit *)defaultCodecFn;
    mNULLCodec.fini = (tFnFini *)defaultCodecFn;
    mNULLCodec.ctrl = (tFnCtrl *)defaultCodecFn;
    strncpy(mNULLCodec.soName, "noCodec", sizeof(mNULLCodec.soName));
    mNULLCodec.soHandle = NULL;
    memset(&mNotificationState, 0, sizeof(mNotificationState));
    mResendPlaybackStatus   = 0;
    mResendNowPlayingStatus = 0;
    mOutputDevice[0] = 0;

    /* restore last observer (if possible) */
    LoadContext();
}

MediaEngineRipper::~MediaEngineRipper()
{

}

int MediaEngineRipper::Init()
{
    //ENTRY

    /* Init the state machine */
    MediaEngineRipperSM::Init();

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);

    return 0;
}

int MediaEngineRipper::InitSM()
{
    me::traces("server:ripper");

    /* scan directory for files with prefix: "libmenginecodec" */
    DIR *dir = 0;
    string_t searchDirectory;

    #ifdef TARGET_BUILD

       #if defined(MENGINE_CODEC_PATH)

          searchDirectory = (char *)M1_EXPAND_AND_QUOTE(MENGINE_CODEC_PATH);

       #else
          me::trace("server:InitSM:err:missing codec path");

          //searchDirectory = (char *)getenv("GMP_LIBS");
          //searchDirectory = (char *)getenv("MEDIAENGINE_PATH");  // /opt/bosch/mediaplayer/bin;
       #endif
    #else

       #if defined(MENGINE_CODEC_PATH)

          searchDirectory = (char *)M1_EXPAND_AND_QUOTE(MENGINE_CODEC_PATH);

       #else

          searchDirectory = "./MediaEngine/4_codecs";

          //searchDirectory = (char *)".";
       #endif

    #endif

    me::trace("server:InitSM:searchDirectory:", searchDirectory);

    dir = opendir(searchDirectory.at());
    if(!dir) {
       me::trace("server:InitSM:could not open directory:err:", err_str());
       return -1;
    }
    /* read out all entries in the directory */
    while(1) {

        /* read one entry */
        struct dirent *directoryEntry;
        directoryEntry = readdir(dir);
        if (!directoryEntry) {
           me::trace("MediaEngineRipper::InitSM:searchDirectory:error:no codec dlls found");
           break; // end of directory
        }

        /* is the name matching? */
        // test trace("readdir: %s", directoryEntry->d_name);
        if (strstr(directoryEntry->d_name, "libmenginecodec_rip")) {

            me::trace("found codec: ", directoryEntry->d_name);

            mCodec.closed = 0;

            /* create a working path */
            strncpy(mCodec.soName, searchDirectory.at(), 256);
            strcat(mCodec.soName, "/");
            strcat(mCodec.soName, directoryEntry->d_name);
            // sim4hi 180629: placed null character at the very end of soName as fix for Coverity CID-148171
            mCodec.soName[255]=0;

            /* open the shared object */
            mCodec.soHandle = dlopen(mCodec.soName, RTLD_LAZY);
            if (!mCodec.soHandle) {
                me::trace("dlopen(%40s): %s", mCodec.soName, dlerror());
                continue;
            }

            /*lint -save -e611 */

            /* get symbol references */
            mCodec.exts = (tFnExts *)dlsym(mCodec.soHandle, "exts");
            if (!mCodec.exts) {
                me::trace("exts() not defined", mCodec.soName);
                continue;
            }
            mCodec.config = (tFnConfig *)dlsym(mCodec.soHandle, "config");
            if (!mCodec.config) {
                me::trace("config() not defined", mCodec.soName);
                continue;
            }
            mCodec.init = (tFnInit *)dlsym(mCodec.soHandle, "init");
            if (!mCodec.init) {
                me::trace("init() not defined", mCodec.soName);
                continue;
            }
            mCodec.fini = (tFnFini *)dlsym(mCodec.soHandle, "fini");
            if (!mCodec.fini) {
                me::trace("fini() not defined", mCodec.soName);
                continue;
            }
            mCodec.ctrl = (tFnCtrl *)dlsym(mCodec.soHandle, "ctrl");
            if (!mCodec.ctrl) {
                me::trace("ctrl() not defined", mCodec.soName);
                continue;
            }
            /*lint –restore */

            /* get the supported extensions from the codec */
            char extensions[512];
            int res;
            res = (mCodec.exts)(OUT extensions, IN sizeof(extensions));
            if (res) {
                me::trace("error in extensions query", mCodec.soName);
                continue;
            }
            me::trace("codec for extensions: ", extensions);
            me::trace("codec loaded: ", mCodec.soName);
        }
    }

    /* close the directory */
    closedir(dir);
    return 0;
}

void MediaEngineRipper::Create()
{
    //ENTRY
    MediaEngineRipperSM::Create();
}

void MediaEngineRipper::Do(int functionID, void *ptr)
{
    //ENTRY
    (void)ptr;
    (void)functionID;

    switch(functionID)
    {
        case 0:
        {
            ThreadFactory::SetName(string_t("me_sm_rip").at());
            while((MediaEngineRipperSM::STATE_MACHINE_FINISHED != MediaEngineRipperSM::StateMachine_Main())){}
            break;
        }

    }
}

int MediaEngineRipper::StopEventProcessed()
{
    //ENTRY
    int res;

    /* call fini for current codec */

    res = (mCodec.fini)();
    if (res) {
        me::trace("server:codec:", mCodec.str(), " fini() failed:", res);
    }

    /* send back desired answer */
    tAllParameters parameter;
    Marshal((char *)parameter, sizeof(parameter)-1, "i", mComponentId);
    Dispatcher::GetInstance().SendMessage("LocalSPMStopSM::STOP_READY", parameter);

    /* no errors */
    return 0;
}

int MediaEngineRipper::DoneEventProcessed()
{
    //ENTRY

    /* close the codec shared object */
    if(!mCodec.closed){
        dlclose(mCodec.soHandle);
        mCodec.closed = 1;
    }
    /* de-register state machine from dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* no errors */
    return 0;
}

int MediaEngineRipper::Setup(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    mPEState.pos().bytes = -1;
    mPEState.pos().ms = -1;
    mPEState.pos().pct = -1;

    /* configure the codec */
    res = (mCodec.config)(IN mCodec.soName, IN NULL, IN &CallbackRipper, IN (void *)this, IN mPEState.local);
    if (res) {
        me::trace("server:Setup:codec:", mCodec.str(), " config() failed:", res);
        return 0; // setup not ok
    }

    /* init the codec */
    res = (mCodec.init());
    if (res) {
        me::trace("server:Setup:mCodec.soNameinit() failed:", res);
        mPEState.reason = (reason_e)res;
        return 0; // setup not ok
    }

    /* send the ramp data */
    mPEState.playstate = PLAYSTATE_STOP;
    res = (mCodec.ctrl(&mPEState));
    if (res) {
        me::trace("server:Setup:codec:", mCodec.str(), " ctrl(PLAYSTATE_NOP) for ramp data failed:", res);
    }

    return 1; // setup ok
}

int MediaEngineRipper::SetupOk(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send myself a PLAY to continue in state machien */
    res = SendEvent(PLAY, PEStateString);
    if (res) {
        me::trace("SendMessage failed:", res);
        return 1; // setup not ok
    }

    return 0;
}

int MediaEngineRipper::ActionFailed(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed:", res);
    }

    reason_e reason = REASON_ACTION_ERROR;

    tURL url;
    int speed;
    int position;
    tU64 handle;
    UnMarshal((char *)PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiil", url, &speed, &position, &handle);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lill",
            handle,
            (int)playbackState,
            reason,
            speed);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed:", res);
    }

    return 0;
}

int MediaEngineRipper::Play(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send play command */
    state_t PEState = me::state_t();

    //PEState.ramp().numsteps = 0;
    //memset(PEState.ramp.steps, 0, sizeof(PEState.ramp.steps);

    tURL url;
    int speed;
    int position;
    tU64 handle;
    tAudioOutputDevice dev;
    tU64 samplerate;
    speedstate_e speedstate;
    ripconf_t ripconf;

    UnMarshal((char *)PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiiltlllll", url, &speed, &position, &handle,
                      dev, &samplerate, &speedstate, &ripconf.tracknumber(), &ripconf.encodeformat(),
                      &ripconf.encodequality());

    if(speedstate && (1 == speed || -1 == speed ))
    {
        PEState.speed = (speed_e)(speed * 100);
    }
    PEState.speed = (speed_e)speed;
    PEState.speedstate = speedstate;
    PEState.pos().ms = position;
    PEState.handle = handle;
    PEState.url = url;
    PEState.dev = dev;
    PEState.fmt().samplerate = samplerate;
    PEState.ripconf = ripconf;
    PEState.playstate = PLAYSTATE_PLAY;

    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:Play:codec:", mCodec.str(), " ctrl(PLAYSTATE_PLAY) failed:", res);
        mPEState.reason = PEState.reason;
        return 0; // control failed
    }

    /* force sending a new playback status and now playing status */
    mResendPlaybackStatus   = 1;
    mResendNowPlayingStatus = 1;

    return 1; // control worked
}

int MediaEngineRipper::FatalError(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;

    /* discard the current codec */
    res = (mCodec.fini)();
    if (res) {
        me::trace("server:FatalError:codec:", mCodec.str(), " fini() failed:", res);
    }

    /* set NULL codec as current one */
    mCodec = mNULLCodec;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed: %d", res);
    }

    tURL url;
    int speed;
    int position;
    tU64 handle;
    speedstate_e speedstate;
    UnMarshal((char *)PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiill", url, &speed, &position, &handle, &speedstate);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lilll",
            handle,
            (int)playbackState,
            (tS64)REASON_ERROR,
            speed,
            speedstate);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed: ", res);
    }
    return 0;
}

int MediaEngineRipper::ActionOk(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed: ", res);
    }

    tURL url;
    int speed;
    int position;
    tU64 handle;
    speedstate_e speedstate;
    UnMarshal((char *)PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiill", url, &speed, &position, &handle, &speedstate);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lilll",
            handle,
            (int)playbackState,
            (tS64)REASON_OK,
            (tS64)speed,
            (tS64)speedstate);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed: ", res);
    }

    return 0;
}

void_t MediaEngineRipper::CallbackRipper(IN me::state_t const& state, void_t *context)
{
    //ENTRY
    int res;
    MediaEngineRipper *mediaEngineRipper = (MediaEngineRipper *)context;

    if(me::REASON_ACK_LOGCONFIG == state.reason()) {
        return;
    }

    me::state_t &state0 = *(me::state_t*)&mediaEngineRipper->mNotificationState;

    if (state0.playstate != state.playstate ||
            state0.speed     != state.speed ||
            state0.reason    != state.reason) {

        tPEPlaybackState playbackState;
        switch(state.playstate()) {
            case PLAYSTATE_STOP:
                playbackState = PE_PBS_STOPPEDSTATE;
                break;
            case PLAYSTATE_PAUSE:
                playbackState = PE_PBS_PAUSEDSTATE;
                break;
            case PLAYSTATE_PLAY:
                playbackState = PE_PBS_PLAYINGSTATE;
                break;
            case PLAYSTATE_NOP:
                playbackState = PE_PBS_LOADINGSTATE;
                break;
            default:
                playbackState = PE_PBS_ERRORSTATE;
                break;
        }

        tGeneralString messageName;
        strncpy_r(OUT messageName, "CDRipperControlSM", IN sizeof(messageName));
        strncat_r(OUT messageName, "::RIPPER_STATUS", IN sizeof(messageName));

        me::trace("MediaEngine::ForwardPlaybackStatus:PLAYBACK_STATUS:reason",state.reason());
        me::trace("MediaEngine::ForwardPlaybackStatus:PLAYBACK_STATUS:status",state.playstate());

        tAllParameters args;
        Marshal((char *)args, sizeof(args)-1, "lil",
                state.handle(),
                (int)playbackState,
                state.reason());

        /* send the PLAYBACK_STATUS to observer */
        Dispatcher::GetInstance().SendMessage(messageName, args);
        state0 = state;
    }
}

int MediaEngineRipper::PlayToStop(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send stop command */
    state_t PEState;
    PEState.playstate = PLAYSTATE_STOP;
    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:PlayToStop:codec:", mCodec.str(), " ctrl(PLAYSTATE_STOP) failed:", res);
        mPEState.reason = PEState.reason;
    }

    /* send the STOP event to myself to advance the current state to stop */
    res = SendEvent(STOP, PEStateString);
    if (res) {
        me::trace("SendEvent(STOP) failed: ", res);
    }
    return 0;
}

int MediaEngineRipper::Stop(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;

    /* discard the current codec */
    res = (mCodec.fini)();
    if (res) {
        me::trace("server:Stop:codec:", mCodec.str(), " fini() failed:", res);
        mPEState.reason = (reason_e)res;
    }

    /* set NULL codec as current one */
    //mCodec = mNULLCodec;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed:", res);
    }

    tURL url;
    int speed;
    int position;
    tS64 handle;
    UnMarshal(PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiil", url, &speed, &position, &handle);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lill",
            handle,
            (int)playbackState,
            (tS64)REASON_OK,
            speed);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed: ", res);
    }
    return 0;
}

int MediaEngineRipper::Pause(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send pause command */
    state_t PEState;

    PEState.playstate = PLAYSTATE_PAUSE;
    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:Pause:codec:", mCodec.str(), " ctrl(PLAYSTATE_PAUSE) failed:", res);
        mPEState.reason = PEState.reason;
    }
    return 1; // no error
}

int MediaEngineRipper::RePlay(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send play command */
    state_t PEState;

    /* in the case the media engine was restarted and the current state is play,
     * the current codec is set to null codec.
     * try to set the correct one for the new file */
    me::trace("codec:name:", mCodec.soName, " handle:", mCodec.soHandle,
         " null:name:", mNULLCodec.soName, " handle:", mNULLCodec.soHandle);
    if (mCodec.soHandle == mNULLCodec.soHandle) {
        Setup(PEStateString);
    }

    /* do the replay */
    tURL url;
    int speed;
    int position;
    tU64 handle;
    tAudioOutputDevice dev;
    tU64 samplerate;
    speedstate_e speedstate;
    ripconf_t ripconf;

    UnMarshal((char *)PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiiltlllll", url, &speed, &position, &handle, dev, &samplerate, &speedstate,
       &ripconf.tracknumber(), &ripconf.encodeformat(), &ripconf.encodequality());

    PEState.pos().ms = position;
    PEState.handle = handle;
    PEState.url = url;
    PEState.dev = dev;
    PEState.fmt().samplerate = samplerate;
    PEState.speed = (speed_e)speed;
    PEState.speedstate = speedstate;
    PEState.ripconf = ripconf;


    PEState.playstate = PLAYSTATE_PLAY;
    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:RePlay:codec:", mCodec.str(), " ctrl(PLAYSTATE_PLAY) failed:", res);
        mPEState.reason = PEState.reason;
        return 0; // control failed
    }

    /* force sending a new playback status and now playing status */
    mResendPlaybackStatus   = 1;
    mResendNowPlayingStatus = 1;

    /* go back to play state */
    return 1;
}

int MediaEngineRipper::PlayToPause(const tPEStateString PEStateString)
{
    //ENTRY
    int res;

    /* send stop command; codec if finalized implicitly */
    state_t PEState;
    PEState.playstate = PLAYSTATE_STOP;
    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:PlayToPause:codec:", mCodec.str(), " ctrl(PLAYSTATE_STOP) failed:", res);
    }

    /* start playing of next song */
    res = SendEvent(PLAY, PEStateString);
    if (res) {
        me::trace("SendEventAnswerByName failed:", res);
    }
    return 0;
}

int MediaEngineRipper::ReStop(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed:", res);
    }

    tURL url;
    int speed;
    int position;
    tU64 handle;
    UnMarshal(PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiil", url, &speed, &position, &handle);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lill",
            handle,
            (int)playbackState,
            (tS64)REASON_OK,
            mPEState.speed());

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed:", res);
    }
    return 0;
}

int MediaEngineRipper::RePause(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed:", res);
    }

    tURL url;
    int speed;
    int position;
    tU64 handle;
    UnMarshal(PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiil", url, &speed, &position, &handle);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lill",
            handle,
            (int)playbackState,
            (tS64)REASON_OK,
            speed);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed:", res);
    }
    return 0;
}

int MediaEngineRipper::SetFlush(const tPEStateString PEStateString)
{
    //ENTRY
    (void)PEStateString;
    int res;
    state_t PEState;

    /* send buffer command */
    PEState.playstate = PLAYSTATE_PLAY;
    PEState.reason = reason_e::REASON_FLUSH;
    res = (mCodec.ctrl(&PEState));
    if (res) {
        me::trace("server:SetFlush:codec:", mCodec.str(), " ctrl(PLAYSTATE_PLAY) failed:", res);
    }

    /* get the target playback state */
    tPEPlaybackState playbackState = PE_PBS_UNKNOWN;
    res = GetTargetPlaybackState(OUT playbackState);
    if (res) {
        me::trace("GetTargetPlaybackState failed:", res);
    }

    tURL url;
    int speed;
    int position;
    tU64 handle;
    UnMarshal(PEStateString, DOUBLE_MARSHAL_SEPARATOR, "tiil", url, &speed, &position, &handle);

    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "lill",
            handle,
            (int)playbackState,
            (tS64)REASON_OK,
            mPEState.speed());

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed:", res);
    }
    return 0;
}

int MediaEngineRipper::switchObserver(const tStateMachineName stateMachineName)
{
    //ENTRY
    int res;

    /* remember the state machine name which will get the status updates from now on (see ForwardPlaybackStatus) */
    strncpy_r(OUT mObservingStateMachineName, IN stateMachineName, IN sizeof(mObservingStateMachineName));

    /* store current observer to temp file */
    SaveContext();

    tReturnValue returnValue = true;
    tAllParameters args;
    Marshal((char *)args, sizeof(args)-1, "i", (int)returnValue);

    /* send the answer to the caller */
    res = SendAnswer(args);
    if (res) {
        me::trace("SendAnswer failed:", res);
    }
    return 0;
}

int MediaEngineRipper::SetOutputDevice(const tAudioOutputDevice outputDevice)
{
    //ENTRY

#ifdef TARGET_BUILD
    /* remember it */
    me::trace("set mOutputDevice to %s", outputDevice);
    strncpy_r(OUT mOutputDevice, IN outputDevice, IN sizeof(mOutputDevice));
#else
    me::trace("uBuntu tests: mOutputDevice not set to %s", outputDevice);
#endif

    return 0;
}

int MediaEngineRipper::ForwardRipperStatus(const tPEStateString PEStateString)
{
    //ENTRY

    /* no observer registered -> do not send status updates */
    if (mObservingStateMachineName[0] == 0) {
        return 0;
    }

    me::state_t &state0 = *(me::state_t*)&mNotificationState;
    me::state_t  state1;

   tGeneralString url, dev;

    UnMarshal(PEStateString, DOUBLE_MARSHAL_SEPARATOR, "lltlllllltli",
            &state1.playstate, &state1.reason,
            url,
            &state1.handle(),
            &state1.pos().ms(), &state1.pos().ms(), &state1.pos().pct(),
            &state1.dur().bytes(), &state1.dur().ms(),
            &state1.dev(),
            &state1.speed(),
            &state1.speedstate());

   state1.url() = url;
   state1.dev() = dev;

    bool update = false;

    /* check the contents of the notification to figure out what event to send */

    // PLAYBACK_STATUS;
    if (state0.playstate != state1.playstate ||
        state0.speed     != state1.speed ||
        state0.reason    != state1.reason ||
        mResendPlaybackStatus) {

        me::trace("MediaEngine::ForwardPlaybackStatus:PLAYBACK_STATUS");

        tPEPlaybackState playbackState;
        switch(state1.playstate()) {
        case PLAYSTATE_STOP:
            playbackState = PE_PBS_STOPPEDSTATE;
            break;
        case PLAYSTATE_PAUSE:
            playbackState = PE_PBS_PAUSEDSTATE;
            break;
        case PLAYSTATE_PLAY:
            if(0 > state1.speed) {
                playbackState = PE_PBS_FASTREVERSESTATE;
            } else if(1 < state1.speed) {
                playbackState = PE_PBS_FASTFORWARDSTATE;
            } else {
                playbackState = PE_PBS_PLAYINGSTATE;
            }
            break;
        case PLAYSTATE_NOP:
            playbackState = PE_PBS_LOADINGSTATE;
            break;
        default:
            playbackState = PE_PBS_ERRORSTATE;
            break;
        }

        tGeneralString messageName;
        strncpy_r(OUT messageName, IN mObservingStateMachineName, IN sizeof(messageName));
        strncat_r(OUT messageName, "::RIPPER_STATUS", IN sizeof(messageName));

        tAllParameters args;
        Marshal((char *)args, sizeof(args)-1, "lil",
                state1.handle(),
                (int)playbackState,
                state1.reason());

        /* send the PLAYBACK_STATUS to observer */
        Dispatcher::GetInstance().SendMessage(messageName, args);

        /* reset the extra trigger for sending playback status */
        mResendPlaybackStatus = 0;

        update = true; // save notification state as current;
    }

    if(update) {
        state0 = state1;
    }

    /* TODO: ACTIVITY_STATUS? */
    /* there is no activity status in the state */

    return 0;
}
