/*
 * MediaEngine_rootdaemon_client.cpp
 *
 *  This file implements IClientSharedlib.h (the client side of the root daemon)
 *  it should be compiled as shared library and copied to the sharedlibs loading area to be loadable from the root daemon server side
 *
 *  Created on: Jan 16, 2018
 *      Author: Ganesh Rengasamy
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/mount.h>
#include <syslog.h>
#include <errno.h>

#include <glob.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <string>
#include <vector>
#include <istream>
#include <iostream>
#include <fstream>

#include <IClientSharedlib.h>
#include <MediaEngine_rootdaemon_client.h>
#include "1_common/metrace.hpp"
#include <unistd.h> // for fork(), execve()
#include <sys/wait.h>  // for waitpid()

//---------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------
int ExecuteSystemCommand(char *argv[])
{
    if (!argv)       /* just checking... */
        return(1);

    pid_t pid;
    int status = -1;
    sig_t intsave, quitsave;
    sigset_t mask, omask;
    int pstat;

    sigemptyset(&mask);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_BLOCK, &mask, &omask);
    switch (pid = vfork()) {
        case -1:        /* error */
            syslog(LOG_INFO, "vfork error: %d", pid);
            sigprocmask(SIG_SETMASK, &omask, NULL);
            return(-1);
        case 0:             /* child */
            syslog(LOG_INFO, "child execve");
            sigprocmask(SIG_SETMASK, &omask, NULL);
            execv(argv[0], argv);
            syslog(LOG_INFO, "execve failed %d", errno);
            _exit(127);
    }

    syslog(LOG_INFO, "parent wait for %d", pid);
    intsave = (sig_t)  signal(SIGINT, SIG_IGN);
    quitsave = (sig_t) signal(SIGQUIT, SIG_IGN);
    pid = waitpid(pid, (int *)&pstat, 0);

    syslog(LOG_INFO, "waitpid returned: pid %d", pid);
    syslog(LOG_INFO, "waitpid returned: pstat %d", pstat);
    syslog(LOG_INFO, "waitpid returned: WIFEXITED %d", WIFEXITED(pstat));
    syslog(LOG_INFO, "waitpid returned: WEXITSTATUS %d", WEXITSTATUS(pstat));

    if(pid != -1 && WIFEXITED(pstat))
    {
        //0-256 valid range, LSB 8 bit exit code of child process
        char cstatus = WEXITSTATUS(pstat);
        //cast to signed integer return value
        status = (cstatus & 0x80) ? (0xffffff00 | cstatus) : cstatus;
    }

    sigprocmask(SIG_SETMASK, &omask, NULL);
    (void)signal(SIGINT, intsave);
    (void)signal(SIGQUIT, quitsave);

    syslog(LOG_INFO, "ExecuteSystemCommand: returned %d", status);
    return status;
}

#ifdef __cplusplus
extern "C"
{
#endif

    const char * getClientName()
   {
      return "Mediaengine";//TODO put it in header file
   }

    const char * getClientGroupName()
   {
       return (char *)"aid_mediaplayer";
   }

   const char * getClientUserName()
   {
       return (char *)"aid_mediaplayer";
   }

   CmdData command(const int cmdNum, std::string args)
   {
      CmdData result;

      switch(cmdNum)
      {
          /*cmdNum = 0*/
          case rpcs_chrt:  //call in MediaEngineServer::Configure //
          {
               //command : chrt -r -p 30 pid
               if((!args.empty()) && (stoi(args.c_str())> 0)) //Expect NON-EMPTY arguments
               {
                   char *command[6]={};
                   command[0] = (char *)"/usr/bin/chrt";
                   command[1] = (char *)"-r";
                   command[2] = (char *)"-p";
                   command[3] = (char *)"30";
                   command[4] = (char *)args.c_str();

                   // change realtime attributes of current process to round-robin with given priority;
                   syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),executing[%s]", (char *)command);// Fix for CID-944934
                   if(0 == ExecuteSystemCommand(command))
                   {
                      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),Successful operation [%s]", (char *)command);// Fix for CID-944934
                      result.errorNo = ERR_NONE;
                   }
                   else
                   {
                      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),unsuccessful operation [%s]", (char *)command);// Fix for CID-944934
                      result.errorNo = ERR_CANNOT_EXEC_CMD;
                   }
               }
               else
                  result.errorNo = ERR_INVALID_CMD_ARGS;

               break;
          }

           /*cmdNum = 1*/
          case rpcs_renice:  //call in MediaEngineServer::Configure //
          {
               //command : renice -4 -p pid
               if((!args.empty()) && (stoi(args.c_str())> 0)) //Expect NON-EMPTY arguments
               {
                   // renice current process;
                   char *command[5]={};
                   command[0] = (char *) "/usr/bin/renice";
                   command[1] = (char *) "-4";
                   command[2] = (char *) "-p";
                   command[3] = (char *)args.c_str();

                   syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),executing[%s]", (char *)command);// Fix for CID-944934
                   if(0 == ExecuteSystemCommand(command))
                   {
                      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),Successful operation [%s]", (char *)command);// Fix for CID-944934
                      result.errorNo = ERR_NONE;
                   }
                   else
                   {
                      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),unsuccessful operation [%s]", (char *)command);// Fix for CID-944934
                      result.errorNo = ERR_CANNOT_EXEC_CMD;
                   }
               }
               else
               {
                  result.errorNo = ERR_INVALID_CMD_ARGS;
               }
               break;
         }

         default:
            syslog(LOG_INFO, "MediaEngine_rootdaemon_client:command(),default");
            result.errorNo = ERR_UNKNOWN_CMD;
      }

      return result;
   }

   #ifdef __cplusplus
}
#endif


CmdData execRootCommand(const char * clientName, const int cmdNum, const char * args)
{
   uid_t euid = geteuid();

   if(0 == euid)
   {
      //User is root: execute root operation directly
      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:execRootCommand() calling command(), clientname[%s],cmdnum[%d], args[%s]",clientName, cmdNum, args);
      return command(cmdNum, std::string(args));
   }
   else
   {
      //use root daemon to perform root operation
      syslog(LOG_INFO, "MediaEngine_rootdaemon_client:execRootCommand() calling calling performRootOp(), clientname[%s],cmdnum[%d], args[%s]",clientName, cmdNum, args);
      return RootDaemonHelper::performRootOp(clientName, cmdNum, std::string(args));
   }
}
