/* -------------------------------------------------------------------------- */
/**
 *   @defgroup meengine megstcodec.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2014
 *
 *   Common part of GStreamer codecs.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_GSTCODEC_HPP_INCLUDED)
#define ME_GSTCODEC_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "gst/gst.h"
#include "glib.h"
#include "gst/app/gstappsrc.h"

#include <gio/gio.h>

#include "mebase.hpp"
#ifndef ENABLE_GSTREAMER_1_0
#include <gst/controller/gstcontroller.h>
#endif
/* -------------------------------------------------------------------------- */

#define ME_GSTCODEC_GETSTATE_TIMEOUT 2000000000 // ns;
#define PE_GST_CODEC_MAXIMUM_PLAYBACKSPEED 3200
#define PE_GST_CODEC_MICRO_TO_NANO_SECONDS 1000000
/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct gstcodec_t : public codec_t {


   /* ----------------------------------------------------------------------- */

   gstcodec_t() : dcon(0), dproxy(0) {

      /* -------------------------------------------------------------------- */
      // gst debug settings;

      #if defined(ENABLE_GST_LOG)

         int_t res = setenv("GST_DEBUG_FILE", "/dev/null", 1);

         if(res) {
            trace("gstcodec:gstcodec_t:setenv:GST_DEBUG_FILE:error:", err_str());
         }

         /* ----------------------------------------------------------------- */

         gst_debug_set_active(false);
         gst_debug_set_default_threshold((GstDebugLevel)0);

      #endif

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   static void gstlog(GstDebugCategory *category,
                      GstDebugLevel level,
                      const gchar *file,
                      const gchar *function,
                      gint line,
                      GObject *object,
                      GstDebugMessage *message,
                      gpointer data) {

      if(0 == message || 0 == data || 0 == file || 0 == function || 0 == object) {
      
         return;
      }
      
      gstcodec_t& codec = *(gstcodec_t*)data;
      
      if(level != /* > */ codec.out.logconfig().lvl) {

         // gst does not filter according to api settings 
         // so we apply loglevel (threshold) filtering here;

         return;
      }
      
      string_t const msg = gst_debug_message_get(message);
      
      if(false == msg.empty()) {
         
         gchar *name = 0;
         g_object_get(G_OBJECT(object), "name", &name, NULL);
         
         if(0 == name) {
            
            return;
         }

         if(     codec.out.logconfig().filter() && 
            0 == codec.out.logconfig().match(name)) {
            return;
         }
         
         string_t filename(string_t((byte_t*)file).name(), " l.", (int_t)line);
         string_t funcname((byte_t*)name, ":", (byte_t*)function);

         g_free(name);
         
         trace(codec.codecname, ":gst:", (int_t)level, ":", filename.pad(27), 
                                                            funcname.pad(55), msg);
      
         if(codec.log.file.fd) {
      
            codec.log.write("", codec.codecname, ":gst:", msg);
         }
      }
   }

   /* ----------------------------------------------------------------------- */

   int_t configlog(logconfig_t const& lc) {

      trace("gstcodec:configlog:", lc.str());

      /* -------------------------------------------------------------------- */

      static int_t registered = 0;
      if(0 == registered) {
      
	  #if defined (ENABLE_GSTREAMER_1_0)
	   gst_debug_add_log_function(gstlog, (gpointer)this, NULL);
	  #else
		  gst_debug_add_log_function(gstlog, (gpointer)this);
	  #endif
         
      
         registered = 1;
      }
      
      /* -------------------------------------------------------------------- */
      // handle toggle;
      
      if(TOGGLE_NONE != lc.tog) {
      
         if(TOGGLE_ON == lc.tog) {
      
            trace("gstcodec:configlog:tog:gst_debug_set_active:true");
            gst_debug_set_active(true);

            // attach callback;
			#if defined (ENABLE_GSTREAMER_1_0)
			gst_debug_add_log_function(gstlog, (gpointer)this, NULL);            
			#else
				gst_debug_add_log_function(gstlog, (gpointer)this);            
				#endif
            
            //#define ME_GSTCODEC_USE_GST_DEBUG
            
            #if defined(ME_GSTCODEC_USE_GST_DEBUG)
            
               string_t url = lc.file.url;
      
               if(url.empty()) {
      
                  url = "/var/opt/bosch/dynamic/media/me.log";
               }
            
               int_t res = setenv("GST_DEBUG_FILE", url.at(), 1);
            
               trace("gstcodec:configlog:tog:GST_DEBUG_FILE:url:", url.at(), " res:", res);
               
            #endif
            
         } else if(TOGGLE_OFF == lc.tog) {
      
            trace("gstcodec:configlog:tog:gst_debug_set_active:false");
            gst_debug_set_active(false);

            // above gst api function does not work;
            // -> detach callback;
			#if defined (ENABLE_GSTREAMER_1_0)
            gst_debug_add_log_function(gstlog, (gpointer)0, NULL);            
			#else
			gst_debug_add_log_function(gstlog, (gpointer)0);            
			#endif
            
            #if defined(ME_GSTCODEC_USE_GST_DEBUG)          
            
               string_t const url = "/dev/null";
            
               int_t res = setenv("GST_DEBUG_FILE", url.at(), 1);
            
               trace("gstcodec:configlog:tog:GST_DEBUG_FILE:url:", url.at(), " res:", res);
               
            #endif
      
         } else {
      
            trace("gstcodec:configlog:tog:", lc.tog(), ":", estr2(lc.tog), " unhandled");
         }
      }
      
      /* -------------------------------------------------------------------- */
      // handle level;
      
      if(-1 != lc.lvl)  {
      
         if(GST_LEVEL_NONE <= lc.lvl && lc.lvl < GST_LEVEL_COUNT) {
      
            gstlevel_e const lvl = (gstlevel_e)lc.lvl();
      
            trace("gstcodec:configlog:lvl:gst_debug_set_default_threshold:", estr2(lvl));
      
            gst_debug_set_default_threshold((GstDebugLevel)lvl);
      
         } else {
      
            trace("gstcodec:configlog:lvl:", lc.lvl(), " unhandled");
         }
      }
      
      /* -------------------------------------------------------------------- */
      // handle logpatterns;
      
      if(0 != lc.dels().size())  {
      
         for(int_t i = 0; i < lc.dels().size(); ++i) {
      
            trace("gstcodec:configlog:pats:unset_threshold:", lc.dels()[i].pat());
            gst_debug_unset_threshold_for_name(lc.dels()[i].pat().at());
         }
      }
      
      if(0 != lc.pats().size())  {
      
         for(int_t i = 0; i< lc.pats().size(); ++i) {
      
            trace("gstcodec:configlog:pats:set_threshold:pat:", lc.pats()[i].pat(), " lvl:", lc.pats()[i].lvl());
            gst_debug_set_threshold_for_name(lc.pats()[i].pat().at(), (GstDebugLevel)lc.pats()[i].lvl());
         }
      }
      
      //TODO: consider level on composing pats & dels;
      
      /* -------------------------------------------------------------------- */
      // handle logfile url;
      
      log.init(lc.file().url().size() ? lc.file().url() : string_t("/tmp/me.log"));      
         
      trace("gstcodec:configlog:file:url:", log.file.url);         
      
      /* -------------------------------------------------------------------- */
      // handle logfile limit;
      
      int_t limit = lc.file().limit;
      
      // limit;
      if(-1 != lc.file().limit) {
      
         static const int_t limit_max = 0x40000000; // 1GiB;
      
         if(limit_max < limit) {
      
            trace("gstcodec:configlog:file:limit:max:", limit_max);
      
            limit = limit_max;
         }
         log.limit(limit);
         
         trace("gstcodec:configlog:file:limit:", limit);         
      }
      
      /* -------------------------------------------------------------------- */
      // handle logfile toggle;
      
      if(TOGGLE_NONE != lc.file().tog) {
      
         if(TOGGLE_ON == lc.file().tog && 0 == log.file.fd) {
      
            trace("gstcodec:configlog:file:tog:", estr2(lc.file().tog), 
                                          " clear:", log.file.url);
            log.clear();
            log.open();
         }
      
         if(TOGGLE_OFF == lc.file().tog && log.file.fd) {
      
            trace("gstcodec:configlog:file:tog:", estr2(lc.file().tog), 
                                          " close:", log.file.url);
            log.close();
         }
      }

      /* -------------------------------------------------------------------- */

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   int_t setstate(GstElement *line, GstState const state1) {

      GstStateChangeReturn res = gst_element_set_state(line, state1);

      trace("gstcodec:setstate:", state1, ":", gststate(state1), " res:", gststatechangereturn(res));

      return res;

      #if 0
         GstStateChangeReturn res1 = gst_element_set_state(line, state1);

         trace("gstcodec:setstate:1:", state1, ":", gststate(state1), " res:", gststatechangereturn(res1));

         GstState state2;

         GstStateChangeReturn res2 = (GstStateChangeReturn)getstate(line, state2, 10);

         trace("gstcodec:setstate:2:", state2, ":", gststate(state2), " res:", gststatechangereturn(res2));

         return res2;
      #endif
   }

   /* ----------------------------------------------------------------------- */

   int_t getstate(GstElement *line, GstState &state1, int_t const timeout_sec) {

      int_t res = GST_STATE_CHANGE_ASYNC;
      int_t seconds = 0;
      while(GST_STATE_CHANGE_ASYNC == res && seconds < timeout_sec) {

         GstState st0, st1;
         st1 = st0 = GST_STATE_NULL;
         GstClockTime const timeout = 1000000000ull; // nanoseconds (1 second);

         res = gst_element_get_state(line, &st0, &st1, timeout);

         trace("gstcodec:getstate:", st0, " -> ", st1, " res:", res);

         if(GST_STATE_CHANGE_ASYNC == res) {

            trace("gstcodec:getstate:async:seconds:", ++seconds);
         }
      }
      if(seconds == timeout_sec) {

         trace("gstcodec:getstate:exit:async:", seconds);
         return -1;
      }

      trace("gstcodec:getstate:exit:", seconds);
      return res;
   }

   /* ----------------------------------------------------------------------- */

   void_t mute(GstElement *vol, int_t const &mute0) {

      trace("gstcodec:mute:", mute0 ? "on" : "off");

      g_object_set(G_OBJECT(vol), "mute", mute0 ? true : false, (GstElement*)NULL);
   }

   /* ----------------------------------------------------------------------- */

   int_t slowmotion(GstElement *line, int_t const rate, GstElement *vsink0, GstElement *vsink1) {

      trace("gstcodec:slowmotion:rate:", rate);

      if(vsink0 || vsink1) {

         int_t cur_pos = position_ms(line);
         trace("gstcodec:slowmotion:cur_pos:", cur_pos);

         GstEvent *event = 0;

         if(100 == rate) { // normal rate;

            trace("gstcodec:slowmotion:normal"); 

            event = gst_event_new_seek(1.0, GST_FORMAT_TIME, 
                                       GST_SEEK_FLAG_FLUSH , GST_SEEK_TYPE_SET,
                                       cur_pos * PE_GST_CODEC_MICRO_TO_NANO_SECONDS, GST_SEEK_TYPE_NONE, 0);
         } else {

            if(0 < rate) {

               trace("gstcodec:slowmotion:ffwd");

               asrt(   0 < rate && rate <= PE_GST_CODEC_MAXIMUM_PLAYBACKSPEED); // ffwd;
               event = gst_event_new_seek (((gfloat)rate)/100.0, GST_FORMAT_TIME,
                                               GST_SEEK_FLAG_FLUSH , GST_SEEK_TYPE_SET,
                                               cur_pos * PE_GST_CODEC_MICRO_TO_NANO_SECONDS, GST_SEEK_TYPE_NONE, 0);

            } else if(rate < 0){

               trace("gstcodec:slowmotion:frev");
               asrt(-(PE_GST_CODEC_MAXIMUM_PLAYBACKSPEED) <= rate && rate <   0); // frev;
               event = gst_event_new_seek (((gfloat)rate) / 100.0, GST_FORMAT_TIME,
                                           GST_SEEK_FLAG_FLUSH , GST_SEEK_TYPE_SET,
                                           0, GST_SEEK_TYPE_NONE, cur_pos * PE_GST_CODEC_MICRO_TO_NANO_SECONDS);

            } else {

               trace("gstcodec:slowmotion:unsupported");
               return -1;
            }
         }

         gst_element_send_event(vsink0, event);

         //g_free(event);

         return 0;

      } else {

         trace("gstcodec:no videosink");

         return -1;
      }
   }

   int_t normalmotion(GstElement *line, GstElement *vsink0, GstElement *vsink1) {

      trace("gstcodec:normalmotion");

      int_t const res = slowmotion(line, 100, vsink0, vsink1);

      if(0 != res) {

         trace("gstcodec:normalmotion:failed");
      }
      return res;
   }

   /* ----------------------------------------------------------------------- */

   static string_t gstlevel(GstDebugLevel const& level0) {
      switch(level0) {
         case GST_LEVEL_NONE:    return "none";
         case GST_LEVEL_ERROR:   return "error";
         case GST_LEVEL_WARNING: return "warning";
         case GST_LEVEL_INFO:    return "info";
         case GST_LEVEL_DEBUG:   return "debug";
         case GST_LEVEL_LOG:     return "log";
         case GST_LEVEL_FIXME:   return "fixme";
         case GST_LEVEL_TRACE:   return "trace";
         case GST_LEVEL_MEMDUMP: return "memdump";
         case GST_LEVEL_COUNT:   return "count";
         default: return "?";
      }
   }
   static string_t streamstatus(GstStreamStatusType const& status0) {
      switch(status0) {
         case GST_STREAM_STATUS_TYPE_CREATE:  return "create";
         case GST_STREAM_STATUS_TYPE_ENTER:   return "enter";
         case GST_STREAM_STATUS_TYPE_LEAVE:   return "leave";
         case GST_STREAM_STATUS_TYPE_DESTROY: return "destroy";
         case GST_STREAM_STATUS_TYPE_START:   return "start";
         case GST_STREAM_STATUS_TYPE_PAUSE:   return "pause";
         case GST_STREAM_STATUS_TYPE_STOP:    return "stop";
         default: return "?";
      }
   }
   static string_t gststate(GstState const& state0) {
      switch(state0) {
         case GST_STATE_VOID_PENDING: return "void_pending";
         case GST_STATE_NULL:         return "null";
         case GST_STATE_READY:        return "ready";
         case GST_STATE_PAUSED:       return "paused";
         case GST_STATE_PLAYING:      return "playing";
         default: return "?";
      }
   }
   static string_t gststatechangereturn(GstStateChangeReturn const &ret0) {
      switch(ret0) {
         case GST_STATE_CHANGE_FAILURE:    return "failure";
         case GST_STATE_CHANGE_SUCCESS:    return "success";
         case GST_STATE_CHANGE_ASYNC:      return "async";
         case GST_STATE_CHANGE_NO_PREROLL: return "no_preroll";
         default: return "?";
      }
   }

   static string_t gstdomain(GQuark const &domain0) {
          if(GST_CORE_ERROR     == domain0) return "core";
     else if(GST_LIBRARY_ERROR  == domain0) return "library";
     else if(GST_RESOURCE_ERROR == domain0) return "resource";
     else if(GST_STREAM_ERROR   == domain0) return "stream";
     else                                   return "system"; // GST_ERROR_SYSTEM
   }

   /* ----------------------------------------------------------------------- */

   static void cb_handle_tag(GstTagList const *tags, gchar const* tag, gpointer /*user*/) {

       trace("gstcodec:cb_handle_tag");

       GType type = gst_tag_get_type(tag);

       if(G_TYPE_STRING != type) {

          trace("gstcodec:cb_handle_tag:name:", tag, " type:", (int_t)type);
          return;
       }
       gchar* val = 0;
       if(gst_tag_list_get_string(tags, tag, &val)) {

          trace("gstcodec:cb_handle_tag:name:", tag, " val:", val);
          g_free(val);

       }  else {

          trace("gstcodec:cb_handle_tag:name:", tag, " desc:", gst_tag_get_description(tag));
       }
   }

   /* ----------------------------------------------------------------------- */

   static GstBusSyncReply cb_bus_sync(GstBus* /*bus*/, GstMessage *msg, gpointer /*that*/) {

       if(GST_MESSAGE_STREAM_STATUS == GST_MESSAGE_TYPE(msg)) {

          GstStreamStatusType type;
          GstElement *owner;
          gst_message_parse_stream_status(msg, &type, &owner);

          gchar *path = gst_object_get_path_string(GST_MESSAGE_SRC(msg));

          //#define DEBUG_BUS_SYNC
          #if defined(DEBUG_BUS_SYNC)
             trace("gstcodec:cb_bus_sync:type:", type, " path:", path);
             trace("gstcodec:cb_bus_sync:cur:", GST_STATE(owner), " tgt:", GST_STATE_TARGET(owner));
          #endif

          g_free(path);

          GValue const *val = gst_message_get_stream_status_object(msg);

          if(0 == val) {
             return GST_BUS_PASS;
          }

          #if defined(DEBUG_BUS_SYNC)
             trace("gstcodec:cb_bus_sync:obj:name:", G_VALUE_TYPE_NAME(val));
          #endif

          if(GST_TYPE_TASK == G_VALUE_TYPE(const_cast<GValue*>(val))) {

             GstTask *task = (GstTask*)g_value_get_object(val);

             switch(type) {
                case GST_STREAM_STATUS_TYPE_CREATE: {
                   GThreadPriority const prio1 = G_THREAD_PRIORITY_LOW;
                   trace("gstcodec:cb_bus_sync:create:prio:", prio1);
#if defined (ENABLE_GSTREAMER_1_0)
                            //gst_task_set_priority(task, prio1); //gst_task_set_priority() call is removed in GST 1.0 and
                            //there is no replacement for this call as it doesn't do anythig.
#else
                            gst_task_set_priority(task, prio1);
#endif
                   break;
                }
                case GST_STREAM_STATUS_TYPE_ENTER:
                   #if defined(DEBUG_BUS_SYNC)
                      trace("gstcodec:cb_bus_sync:enter");
                   #endif
                   break;
                case GST_STREAM_STATUS_TYPE_LEAVE:
                   #if defined(DEBUG_BUS_SYNC)
                      trace("gstcodec:cb_bus_sync:leave");
                   #endif
                   break;
                default:break;
             }
          }
       }
       return GST_BUS_PASS;
   }

   /* ----------------------------------------------------------------------- */

   int_t position_ms(GstElement *line) { // ms;
       int_t res = -1;
       if(NULL != line) {
           GstFormat fmt  = GST_FORMAT_TIME;
           gint64    pos = -1;
#if defined (ENABLE_GSTREAMER_1_0)
           res = gst_element_query_position(line, fmt, &pos) ? pos / 1000000 : -1;// gst_element_query_position(line, &fmt, &pos) is modified to gst_element_query_position(line, fmt, &pos)
#else
           res = gst_element_query_position(line, &fmt, &pos) ? pos / 1000000 : -1;
#endif
       }
       return res;
   }
   int_t position_bytes(GstElement *line) { // ms;
       int_t res = -1;
       if(NULL != line) {
           GstFormat fmt  = GST_FORMAT_BYTES;
           gint64    pos = -1;
#if defined (ENABLE_GSTREAMER_1_0)
           res = gst_element_query_position(line, fmt, &pos) ? pos : -1;// gst_element_query_position(line, &fmt, &pos) is modified to gst_element_query_position(line, fmt, &pos)
#else
           res = gst_element_query_position(line, &fmt, &pos) ? pos : -1;
#endif
       }
       return res;
   }

   int_t duration_ms(GstElement *line) { // ms;
       int_t res = -1;
       if(NULL != line) {
           GstFormat fmt = GST_FORMAT_TIME;
           gint64 dur0 = -1;
#if defined (ENABLE_GSTREAMER_1_0)
           res = gst_element_query_duration(line, fmt, &dur0) ? dur0 / 1000000 : -1; // gst_element_query_duration(line, &fmt, &dur0) is modified to gst_element_query_duration(line, fmt, &dur0)
#else
           res = gst_element_query_duration(line, &fmt, &dur0) ? dur0 / 1000000 : -1;
#endif
       }
       return res;
   }
   int_t duration_bytes(GstElement *line) { // ms;
       int_t res = -1;
       if(NULL != line) {
           GstFormat fmt = GST_FORMAT_BYTES;
           gint64 dur0 = -1;
#if defined (ENABLE_GSTREAMER_1_0)
           res = gst_element_query_duration(line, fmt, &dur0) ? dur0 : -1;// gst_element_query_duration(line, &fmt, &dur0) is modified to gst_element_query_duration(line, fmt, &dur0)
#else
           res = gst_element_query_duration(line, &fmt, &dur0) ? dur0 : -1;
#endif
       }
       return res;
   }


   /* ----------------------------------------------------------------------- */

   void_t dconnect() {

      trace("gstcodec:dconnect");

      GError *error = 0;

      // connect dbus session bus;
      if(0 == dcon) {

         dcon = g_bus_get_sync(G_BUS_TYPE_SESSION, 0, &error);

         trace("gstcodec:dconnect:session:", dcon ? string_t("success") : string_t("err:") << error->message);
      }

      string_t const service   = "service";
      string_t const path      = "path";
      string_t const interface = "interface";

      // create dbus proxy;
      if(dcon) {

         dproxy = g_dbus_proxy_new_sync(dcon, G_DBUS_PROXY_FLAGS_NONE, 0,
                                        service.at(), path.at(), interface.at(), 0, &error);

         trace("gstcodec:dconnect:proxy:", dproxy ? string_t("success") : string_t("err:") << error->message);
      }

      if(error) g_error_free(error);
   }

   void_t disolate() {

      trace("gstcodec:disolate");

      if(dproxy) {
          g_object_unref(dproxy);
          dproxy = 0;
      }
      if(dcon) {
         g_object_unref(dcon);
         dcon = 0;
      }
   }

   void_t dviewon() {

      trace("gstcodec:dviewon");

      asrt(dcon && dproxy);

      GError *error = 0;

      int_t const timeout = 60000;

      g_dbus_proxy_call_sync(dproxy, "viewStatusDCReplySlot", g_variant_new ("(i)", 1),
                             G_DBUS_CALL_FLAGS_NONE, (int)timeout, 0, &error);

      trace("gstcodec:dviewon:", error ? string_t("err:") << error->message : string_t("success"));

      if(error) g_error_free(error);
   }
   
   int_t is_stream_seekable(GstElement *line) {

       GstQuery *seekquery = NULL;
       int_t res = false;

       seekquery = gst_query_new_seeking(GST_FORMAT_TIME);

       if(seekquery) {
           res = gst_element_query (line, seekquery);
           if (res)
           {
               gboolean stream_seekable=false;
               trace("me:codec: parsing seeking query");
               gst_query_parse_seeking (seekquery, NULL, &stream_seekable, NULL, NULL);

               if (stream_seekable)
               {
                   trace("me:codec:seekable stream");
                   res = true;
               }
               else
               {
                   trace("me:codec: not a seekable stream");
                   res = false;
               }
           }
       }
       return res;
   }

   /* ----------------------------------------------------------------------- */   
   
   #if 0
   static string_t fetch_encoder(encfmt_e const enc, string_t &ext)
   {
       trace("gstcodec:fetch_encoder:", enc);
       switch(enc)
       {
           case ME_EF_MP3:
           {
               ext = "mp3";
               #ifndef TARGET_BUILD
                  return "lame";
               #else
                  return "mfw_mp3encoder";
               #endif
           }
           case ME_EF_WAV:
           case ME_EF_AAC:
           case ME_EF_M4A:
           case ME_EF_WMA:
           case ME_EF_OGG:
           case ME_EF_FLAC:
           case ME_EF_AIF:
           case ME_EF_AIFF:
           case ME_EF_UNDEFINED:
           default: return "?";
       }
   }
   #endif
   
   /* ----------------------------------------------------------------------- */

   GDBusConnection *dcon;
   GDBusProxy *dproxy;

   m1::log_t log;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

 #endif // ME_GSTCODEC_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
