/* -------------------------------------------------------------------------- */
/**
 *   @defgroup mecodecif mecodecif.h
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2014
 *
 *   Common codec interface.
 */
 /* ------------------------------------------------------------------------- */

#if !defined(ME_CODECIF_H_INCLUDED)
#define ME_CODECIF_H_INCLUDED

/* -------------------------------------------------------------------------- */

#include "meprotocol.hpp"

/* -------------------------------------------------------------------------- */
// https://gcc.gnu.org/wiki/Visibility

#if defined _WIN32 || defined __CYGWIN__
  #ifdef BUILDING_DLL
    #ifdef __GNUC__
      #define DLL_PUBLIC __attribute__ ((dllexport))
    #else
      #define DLL_PUBLIC __declspec(dllexport) // Note: actually gcc seems to also supports this syntax.
    #endif
  #else
    #ifdef __GNUC__
      #define DLL_PUBLIC __attribute__ ((dllimport))
    #else
      #define DLL_PUBLIC __declspec(dllimport) // Note: actually gcc seems to also supports this syntax.
    #endif
  #endif
  #define DLL_LOCAL
#else
  #if __GNUC__ >= 4
    #define DLL_PUBLIC __attribute__ ((visibility ("default")))
    #define DLL_LOCAL  __attribute__ ((visibility ("hidden")))
  #else
    #define DLL_PUBLIC
    #define DLL_LOCAL
  #endif
#endif

/* -------------------------------------------------------------------------- */

extern "C" {

/* -------------------------------------------------------------------------- */
/**
 * Return formats supported by loaded codec dll as file extension
 * list separated by semicolons; e.g. "wav;mp3;wma;".
 *
 * param[in] extensions     - character array
 * param[in] extensionsSize - character array size
 * @return number of written bytes;
 */
DLL_PUBLIC int_t exts(byte_t *exts2, int_t size2) {

   strings_t exts0 = me::instance()->exts();

   me::trace("codecif:exts:size:", exts0.size(),
                         " proc:", execname().name(),
                          " dll:", dllname().name());

   string_t exts1;
   for(auto r = exts0.range(); r.x; ++r) {
      exts1 << *r.x;
      if(r.x != r.z) exts1 << ";";
   }

   sprnt(exts2, size2, "%s", exts1.at());
   me::trace("codecif:exts:exts:", exts2);
   return 0;
}

/* -------------------------------------------------------------------------- */
/**
 * Configure loaded codec dll; call once on load;
 *
 * param[in] ext      - file extension to use this codec dll for; used to check support;
 * param[in] dllname  - name of codec-dll binary (e.g. "libmenginecodec_av_so.so");
 * param[in] server   - unique server name (cp. ip-address); defaults to "~mengine" when 0;
 * param[in] callBack - client fptr;
 * param[in] context  - clients this;
 * @return 0;
 */
DLL_PUBLIC int_t config(byte_t* dllname, byte_t *server, me::state_call_t call, void_t *context, int_t local) {

   me::trace("codecif:config");

   string_t const server1 = server ? server : "~mengine";
   
   me::instance()->config(string_t(dllname), server1, call, context, local);
   return 0;
}

/* -------------------------------------------------------------------------- */
/**
 * Allocate resources and basic GStreamer pipeline setup;
 *
 * @return 0;
 */
DLL_PUBLIC int_t init() {

   me::trace("codecif:init");

   me::instance()->init();
   return 0;
}

/* -------------------------------------------------------------------------- */
/**
 * Finish GStreamer pipeline and de-allocate resources;
 *
 * @return 0;
 */
DLL_PUBLIC int_t fini() {

   me::trace("codecif:fini");

   me::instance()->fini();
   return 0;
}

/* -------------------------------------------------------------------------- */
/**
 * Control playback through loaded, configured an init'ed codec-dll;
 *
 * @return < 0:error;
 */
DLL_PUBLIC int_t ctrl(me::state_t *state) {

   me::trace("codecif:ctrl:", state->str());

   int_t const res = me::instance()->ctrl(*state, 1);

   me::trace("codecif:ctrl:res:", res);
   return res;
}

/* -------------------------------------------------------------------------- */

} // extern "C"

/* -------------------------------------------------------------------------- */

#endif // ME_CODECIF_H_INCLUDED

/* -------------------------------------------------------------------------- */

