/* -------------------------------------------------------------------------- */
/**
 *   @defgroup mecodec mecodec.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   Baseclass for specialised codecs.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_CODEC_HPP_INCLUDED)
#define ME_CODEC_HPP_INCLUDED

/* -------------------------------------------------------------------------- */

#include "metrace.hpp"
#include "Utils.h"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

enum cmd_e : int_t { off  = 0, on, flush };

/* -------------------------------------------------------------------------- */

struct cycle_t : nif_t<cycle_t> {
   static string_t name() {
      return "cycle_t";
   }
   enum : int_t { cid = 0 };
   cycle_t() : cmd(off) {
   }
   cycle_t(cmd_e const &cmd0) :
      cmd(cmd0) {
   }
   if_t<cmd_e> cmd;
   nif_t<>    done;
   string_t str() const {
      return string_t(off == cmd ? "off" : (on == cmd ? "on" : "flush"));
   }
};

/* -------------------------------------------------------------------------- */

struct codec_t : public thr_t {

   codec_t() : ini(false), call(0), context(0) {
   }
   codec_t(string_t const &dll_name0, string_t const  &server0, state_call_t call0, void_t *context0, int_t const local) :
      ini(true), call(0), context(0) {
      config(dll_name0, server0, call0, context0, local);
   }
   ~codec_t() {
      if(ini) fini();
   }

   /* ----------------------------------------------------------------------- */
   // abstract interface;

   virtual strings_t onexts() const = 0;
   virtual int_t     onisaudio(string_t const &ext0) const = 0;
   virtual int_t     onisvideo(string_t const &ext0) const = 0;

   virtual void_t    oninit() = 0;
   virtual void_t    onfini() = 0;
   virtual int_t     onctrl(state_t const &state0) = 0;

   /* ----------------------------------------------------------------------- */

   virtual void_t mainloop() {
      trace("codec:mainloop:default");
   }

   /* ----------------------------------------------------------------------- */

   virtual int_t read(file_t &file, int_t const& space, buffer_t &buf) {
      trace("codec:read:default");
      return 0;
   }
   virtual int_t write(buffer_t const &buf) {
      trace("codec:write:default");
      return 0;
   }

   /* ----------------------------------------------------------------------- */

   void_t config(string_t const &dll_name0, string_t const &server0, state_call_t call0, void_t *context0, int_t const local) {

      trace("codec:config:dll:", dll_name0, " server:", server0, " local:", local);

      /* -------------------------------------------------------------------- */

      //asrt(-1 == reader.nid);
      if(-1 != reader.nid) {

         trace("codec:config:dll:", dll_name, " already configured");
         return;
      }

      /* -------------------------------------------------------------------- */

      dll_name = dll_name0.name();
      trace("codec:config:dll_name:", dll_name);

      byte_t *const b0 = dll_name.find_f(string_t("codec"));
      byte_t *const b1 = dll_name.find_b(string_t("_so.so"));

      codecname = dll_name.range(b0 + 6, b1 - 1);
      trace("codec:config:codecname:", codecname);

      /* -------------------------------------------------------------------- */
      // server name;

      if(server0.empty()) {
         in.server() = "~mengine";
         trace("codec:config:default:server:", in.server);
      } else {
         in.server() = server0;
         trace("codec:config:server:", server0);
      }

      /* -------------------------------------------------------------------- */

      trace("codec:config:local:", local);

      in.local = local;

      /* -------------------------------------------------------------------- */
      // cleanup shared resources;

      if(0 == call0) {

         trace("codec:config:cleanup");
         cleanup();
      }

      /* -------------------------------------------------------------------- */

      call    = call0;    // attach callback;
      context = context0; // callback context;

      /* -------------------------------------------------------------------- */
      // default cwd;

      string_t cwd((byte_t*)getcwd(0, 0));

      trace("codec:config:cwd:", cwd);

      /* -------------------------------------------------------------------- */
      // default alsa device path;

      string_t dev = in.dev;
      if(dev.empty()) {

         dev = envvar("MENGINE_AUDIO_OUT_DEV"); // from cppunit_mediaengine.sh;

         if(dev.empty()) {

            #if defined(MENGINE_ALSA_DEVICE)

               dev <<  MENGINE_ALSA_DEVICE; // e.g. 0 for linuxx86make; 3 for gen3armmake;

               if(dev.isuint()) {

                   dev.clear();
                   dev << "hw:" << MENGINE_ALSA_DEVICE << ",0"; // from ai_mediaplayer_prod.xml;

                   trace("codec:config:default:xml:number:dev:", dev);

               } else {

                   trace("codec:config:default:xml:string:dev:", dev);
               }

            #else

               //dev = string_t("hw:3,0"); // default;
               dev = string_t("AdevEnt1Out"); // default;

               trace("codec:config:default:fallback:dev:", dev);

            #endif

         } else {

            trace("codec:config:default:environment:dev:", dev);
         }

         in.dev = dev;
      }

      /* -------------------------------------------------------------------- */
      // default playback speed;

      if(0 == in.speed) {

         in.speed = ME_SPEED_NORMAL;
         trace("codec:config:default:speed:", in.speed());
      }

      /* -------------------------------------------------------------------- */
      // default playback volume;

      if(-1 == in.vol) {

         in.vol = 100; // %;
         trace("codec:config:default:volume:", in.vol());
      }

      /* ----------------------------------------------------------------- */
      // default display;

      in.display().num() = 1;

      screen_t& screen0 = in.display().screens()[0]; // all projects;

      screen0().rsl().width  = 1280;
      screen0().rsl().height =  720;
      screen0().layer        =   20;
      screen0().srf          =   22;
      trace("codec:config:default:screen:0:", screen0.str());

      screen_t& screen1 = in.display().screens()[1]; // JAC;

      screen1().rsl().width  = 1280;
      screen1().rsl().height =  800;
      screen1().layer        = 2600;
      screen1().srf          =   31;
      trace("codec:config:default:screen:1:", screen1.str());

      screen_t& screen2 = in.display().screens()[2]; // no project yet;

      screen2().rsl().width  =  800;
      screen2().rsl().height =  480;
      screen2().layer        = 3000;
      screen2().srf          =   22;
      trace("codec:config:default:screen:2:", screen2.str());

      /* -------------------------------------------------------------------- */

      in.playstate = PLAYSTATE_STOP;
      //default video properties

      in.vprops().brightness = 100;
      in.vprops().hue = 0;
      in.vprops().saturation = 100;
      in.vprops().contrast = 100;
      in.vprops().brightnessoffset = 0;
      in.vprops().saturationoffset = 0;
      in.vprops().hueoffset = 0;

      //out = in;
      out = state_t();

      /* -------------------------------------------------------------------- */

      trace("codec:config:in:", in.str(0));
      trace("codec:config:in:", in.str(1));
      trace("codec:config:in:", in.str(2));

      trace("codec:config:out:", out.str(0));
      trace("codec:config:out:", out.str(1));
      trace("codec:config:out:", out.str(2));

      /* -------------------------------------------------------------------- */

      trace("codec:config:exit");

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   void_t cleanup() {

       //Remove only matching files from /dev/shm directory
       int retStatus = RemoveDirectory("/dev/shm",false,"sem.smp:");
       trace ("codec:cleanup:RemoveDirectory() for 'sem.smp:'. retStatus:", retStatus);

       retStatus = RemoveDirectory("/dev/shm",false,"shm:");
       trace ("codec:cleanup:RemoveDirectory() for 'shm:'. retStatus:", retStatus);
   }

   /* ----------------------------------------------------------------------- */

   strings_t exts() {

      //trace("codec:exts:proc:", execname().name(), " dll:", dllname().name());

      return onexts();
   }

   /* ----------------------------------------------------------------------- */

   void_t init() {

      /* -------------------------------------------------------------------- */

      trace("codec:init");

      /* -------------------------------------------------------------------- */

      if(-1 != reader.nid) {

          trace("codec:init:dll:", dll_name, " already initialised");
          return;
      }

      /* -------------------------------------------------------------------- */

      asrt(dll_name.size()); // not configured;

      string_t const server = in.server;

      trace("codec:init:dll:", dll_name, " server:", server);

      /* -------------------------------------------------------------------- */
      // pump;

      int_t const timeout0 = smp_t::forever;

      /* -------------------------------------------------------------------- */
      // create reader;

      asrt(-1 == reader.nid);

      // ctrl queue: loop control (client, reader & writer);
      // data queue: not used;
      // mode: listen in separate thread;
      // dispatch: local function pointer;
      // timeout: none;

      string_t tmpname = server;
      tmpname << ".pump.read";

      reader.init(string_t("read", (int_t)thr_t::whoami()), 0x1000, 0, node_t::mode_listen, &dispatch_read, this, timeout0);

      trace("codec:init:reader:", reader.str());

      /* -------------------------------------------------------------------- */
      // create reader ctrl;

      asrt(-1 == rctrl.nid);

      rctrl.init(string_t("rctrl", (int_t)thr_t::whoami()), 0x1000, 0x1000, node_t::mode_init);

      /* -------------------------------------------------------------------- */
      // create writer;

      asrt(-1 == writer.nid);

      int_t const size_data = 0x100000;

      // ctrl queue: loop control (reader & writer);
      // data queue: read data from filesystem (media object blocks);
      // mode: listen in separate thread;
      // dispatch: local function pointer;
      // timeout: none;
      writer.init(string_t("write", (int_t)thr_t::whoami()), 0x1000, size_data, node_t::mode_listen, &dispatch_write, this, timeout0);

      trace("codec:init:writer:", writer.str());

      /* -------------------------------------------------------------------- */
      // create writer ctrl;

      asrt(-1 == wctrl.nid);

      wctrl.init(string_t("wctrl", (int_t)thr_t::whoami()), 0x1000, 0x1000, node_t::mode_init);

      /* -------------------------------------------------------------------- */

      int_t const timeout_connect = 10000;

      /* -------------------------------------------------------------------- */
      // connect reader to reader;

      int_t const cid0 = reader.connect(reader.name, timeout_connect);

      trace("codec:init:reader->reader:connect:", (cid0 == reader.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */
      // connect reader to writer;

      int_t const cid1 = reader.connect(writer.name, timeout_connect);

      trace("codec:init:reader->writer:connect:", (cid1 == writer.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */
      // connect writer to writer;

      int_t const cid2 = writer.connect(writer.name, timeout_connect);

      trace("codec:init:writer->writer:connect:", (cid2 == writer.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */
      // connect writer to reader;

      int_t const cid3 = writer.connect(reader.name, timeout_connect);

      trace("codec:init:writer->reader:connect:", (cid3 == reader.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */
      // connect rctrl to reader;

      int_t const cid4 = rctrl.connect(reader.name, timeout_connect);

      trace("codec:init:rctrl->reader:connect:", (cid4 == reader.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */
      // connect wctrl to writer;

      int_t const cid5 = wctrl.connect(writer.name, timeout_connect);

      trace("codec:init:wctrl->writer:connect:", (cid5 == writer.nid ? "succeeded" : "failed"));

      /* -------------------------------------------------------------------- */

      oninit_int();

      /* -------------------------------------------------------------------- */

      trace("codec:init:exit");
   }

   /* ----------------------------------------------------------------------- */

   void_t oninit_int() {

      trace("codec:oninit_int:dll:", dll_name);

      /* -------------------------------------------------------------------- */

      if((smp_t::handle_t)invalid_handle != join.fd) {

         trace("codec:oninit_int:dll:", dll_name, " already initialised");
         return;
      }

      /* -------------------------------------------------------------------- */

      #if defined(GEN3ARM)

         string_t const xdgdir = "/tmp";

         int_t res = setenv("XDG_RUNTIME_DIR", xdgdir.at(), 0);

         if(res) {
            trace("codec:oninit:setenv:XDG_RUNTIME_DIR:error:", err_str());
         }

         res = setenv("LD_LIBRARY_PATH", "/usr/lib/gstreamer-0.10", 0);
         if(res) {
            trace("codec:oninit:setenv:LD_LIBRARY_PATH:error:", err_str());
         }

      #endif

      /* -------------------------------------------------------------------- */

      string_t const server = in.server;

      trace("codec:oninit:server:", server, " dll:", dll_name);

      join.init(string_t("<") << server << ".join>." << dll_name);

      /* -------------------------------------------------------------------- */

      oninit(); // implementation in derived codec;

      /* -------------------------------------------------------------------- */
      // start mainloop;

      thr_t::init(string_t("srv", (int_t)thr_t::whoami())); // runs work() paralelly;

      thr_t::wait(30); // prevent rare delayed/asynchronous/blocking pipeline
                       // statechange due to statechange-request before mainloop runs;

      /* -------------------------------------------------------------------- */

      trace("codec:oninit_int:exit");
   }

   /* ----------------------------------------------------------------------- */

   void_t fini() {

      /* -------------------------------------------------------------------- */

      if(-1 == reader.nid) {

         trace("codec:fini:exit");
         return;
      }

      /* -------------------------------------------------------------------- */

      trace("codec:fini:dll:", dll_name);

      /* -------------------------------------------------------------------- */

      if(PLAYSTATE_PLAY  == out.playstate ||
         PLAYSTATE_PAUSE == out.playstate ||
         PLAYSTATE_STOP  == out.playstate) {

         state_t state;
         state.playstate = PLAYSTATE_STOP;

         int_t const res = ctrl(state, 1);

         /* ----------------------------------------------------------------- */

         onfini_int();
      }

      /* -------------------------------------------------------------------- */
      // cleanup;

      wctrl.fini();
      writer.fini();

      rctrl.fini();
      reader.fini();

      if(file.fd) {
         file.close();
      }

      /* -------------------------------------------------------------------- */

      out /*= in*/ = state_t();

      states.clear();

      /* -------------------------------------------------------------------- */

      trace("codec:fini:exit");
   }

   /* ----------------------------------------------------------------------- */

   void_t onfini_int() {

      /* -------------------------------------------------------------------- */

      if((smp_t::handle_t)invalid_handle == join.fd) {

         trace("codec:onfini_int:already finished");
         return;
      }

      /* -------------------------------------------------------------------- */

      trace("codec:onfini_int:dll:", dll_name);

      /* -------------------------------------------------------------------- */

      onfini();

      /* -------------------------------------------------------------------- */

      int_t const timeout = 2000;

      trace("codec:onfini_int:join:wait...");
      int_t const res = join.wait(timeout);
      trace("codec:onfini_int:join:...", (smp_t::timeout == res ? "timeout" : (-1 == res ? "failed" : "succeeded")));

      join.fini();
      trace("codec:onfini_int:join:finished");

      /* -------------------------------------------------------------------- */

      thr_t::fini();

      /* -------------------------------------------------------------------- */

      trace("codec:onfini_int:exit");
   }

   /* ----------------------------------------------------------------------- */

   int_t ctrl(state_t state, int_t const &sync) {

      trace("codec:ctrl:", estr2(state.playstate), " sync:", sync);

      /* -------------------------------------------------------------------- */

      if(-1 == reader.nid){
          trace("codec:ctrl:reader uninitialized");
          return -1;
      }

      /* -------------------------------------------------------------------- */

      playstate_e ips = PLAYSTATE_NOP; // interrupt playstate;

      reason_e const rs = input(state, ips);

      trace("codec:ctrl:reason:", estr2(rs), " interrupt:", estr2(ips));

      /* -------------------------------------------------------------------- */

      if(REASON_ARGUMENT_ERROR == rs ||
         REASON_FORMAT_ERROR   == rs) {
         trace("codec:ctrl:", estr2(rs));
         return -1;
      }
      if(REASON_BUFFER == rs) {
         trace("codec:ctrl:", estr2(rs));
         return 0;
      }

      /* -------------------------------------------------------------------- */

      transfer();

      if(PLAYSTATE_STOP == ips) {

         ctl(PLAYSTATE_STOP, sync);
         out.pos().clear();

      } else if(PLAYSTATE_PAUSE == ips) {

         ctl(PLAYSTATE_PAUSE, sync);

      } else {

         trace("codec:ctrl:interrupt:", estr2(ips), " unhandled");
      }

      in.playstate = state.playstate;
      ctl(in.playstate, sync);

      /* -------------------------------------------------------------------- */

      trace("codec:ctrl:exit");

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   int_t ctl(playstate_e const ps, int_t const &sync) {

      trace("codec:ctl:playstate:", estr2(ps), " sync:", sync);

      ctrl_t ctl(ps, sync);

      msg_t msg(rctrl.nid, reader.nid, ctrl_t::cid);
      msg.set(ctl);

      if(sync) {

         trace("codec:ctl:fetch:", estr2(ps), " msg:", msg.str());

         int_t const res = rctrl.fetch(node_t::queue_ctrl, msg);

         trace("codec:ctl:fetch:res:", (-1 == res ? "failed" : "succeeded"));

      } else {

         trace("codec:ctl:post:", estr2(ps), " msg:", msg.str());
         rctrl.post(node_t::queue_ctrl, msg);
      }
      return 0;
   }

   /* ----------------------------------------------------------------------- */

   virtual void_t work(void_t * /* arg0 */) {  // overload from thr_t;

      trace("codec:work");

      /* -------------------------------------------------------------------- */

      thr_t::setname(string_t("codec_", codecname));

      /* -------------------------------------------------------------------- */

      mainloop();

      /* -------------------------------------------------------------------- */

      trace("codec:work:exit");

      /* -------------------------------------------------------------------- */

      int_t const res = join.post();
      if(-1 == res){
          trace("codec:work:sem_t post failed");
      }
   }

   /* ----------------------------------------------------------------------- */

   reason_e input(state_t state, playstate_e &ips) {

      trace("codec:input:ramp:", state.ramp().str());

      trace("codec:input0:", state.str(0));
      trace("codec:input1:", state.str(1));
      trace("codec:input2:", state.str(2));
      trace("codec:input3:", state.str(3));
      trace("codec:input4:", state.str(4));

      /* -------------------------------------------------------------------- */
      // check request to be buffered;

      if(REASON_BUFFER == state.reason) {

         state.reason = REASON_OK; // clear flag;
         states.append(state);
         trace("codec:input:buffered:", states.size());
         return REASON_BUFFER;
      }

      /* -------------------------------------------------------------------- */
      // check format support;

      string_t const url = state.url;
      string_t ext = url.ext();
      ext.lower();

      trace("codec:input:format:url:", url, " ext:", ext);

      int_t const isvideo = onisvideo(ext);

      trace("codec:input:handle:url:isvideo:", isvideo);

      if(url.size() && 0 == exts().find_f(ext)) {

          trace("codec:input:format:ext:", ext, " not supported");

          state.reason() = REASON_FORMAT_ERROR;
          forward(state);
          return REASON_FORMAT_ERROR;
      }

      /* -------------------------------------------------------------------- */
      // handle reason;

      if(REASON_OK != state.reason()) {

         trace("codec:input:reason:", estr2(state.reason()));
         forward(state);
      }

      /* -------------------------------------------------------------------- */
      // handle ramp;

      int_t const ramp_size = state.ramp().steps().size();

      trace("codec:input:handle:ramp:", state.ramp().str(), " size:", ramp_size);

      if(ramp_size) {

         in.ramp().steps().resize(ramp_size);

         if(state.ramp() != in.ramp()) {

            trace("codec:input:handle:ramp:", in.ramp().str(), "->", state.ramp().str());

            for(int_t i = 0; i < state.ramp().steps().size(); ++i) {

               if(state.ramp().steps()[i].empty()) {
                  continue;
               }

               if(state.ramp().steps()[i].lvl() < 5ll || 100ll < state.ramp().steps()[i].lvl()) {

                  trace("codec:input:handle:ramp:step:", i, " level:", state.ramp().steps()[i].lvl(), " out of bounds:[5;100]%");
                  state.reason = REASON_ARGUMENT_ERROR;
                  forward(state);
                  return REASON_ARGUMENT_ERROR;
               }
               if(state.ramp().steps()[i].dur().ms < 1000ll) {

                  trace("codec:input:handle:ramp:step:", i, " dur:", state.ramp().steps()[i].dur().ms(), " ms out of bounds: > 1000 ms");
                  state.reason = REASON_ARGUMENT_ERROR;
                  forward(state);
                  return REASON_ARGUMENT_ERROR;
               }

               trace("codec:input:handle:ramp:step:", i, ":", state.ramp().steps()[i].str());
               out.ramp().steps()[i] = in.ramp().steps()[i] = state.ramp().steps()[i];
            }

            //in.ramp() = state.ramp();

            ips = PLAYSTATE_STOP;
            trace("codec:input:handle:ramp:interrupt:", estr2(ips));
         }
      }

      /* -------------------------------------------------------------------- */
      // handle display;

      if(state.display().screens().size()) {

         out.display = in.display = state.display;
      }

      /* -------------------------------------------------------------------- */
      // handle pcm sample format;

      if(-1 != state.fmt().channels()) {

         if(in.fmt().channels() != state.fmt().channels()) {

            trace("codec:input:handle:fmt:channels:", in.fmt().channels(),
                                              " -> ", state.fmt().channels());

            in.fmt().channels() = state.fmt().channels();
         }
      }

      /* -------------------------------------------------------------------- */
      // handle pcm samplerate;

      if(-1 != state.fmt().samplerate() && 0 != state.fmt().samplerate()) {

          if(state.fmt().samplerate() != in.fmt().samplerate()) {
             ips = PLAYSTATE_STOP;
             trace("codec:input:handle:fmt:interrupt:", estr2(ips));
          }
          trace("codec:input:handle:fmt:samplerate:", in.fmt().samplerate(),
                                              " -> ", state.fmt().samplerate());

          out.fmt().samplerate() = in.fmt().samplerate() = state.fmt().samplerate();
      }

      /* -------------------------------------------------------------------- */
      // handle pcm samplewidth;

      if(-1 != state.fmt().samplewidth) {

         if(state.fmt().samplewidth != in.fmt().samplewidth) {

            trace("codec:input:handle:fmt:samplewidth:", in.fmt().samplewidth(),
                                                 " -> ", state.fmt().samplewidth());

            in.fmt().samplewidth() = state.fmt().samplewidth();
         }
      }

      /* -------------------------------------------------------------------- */
      // handle pcm sampledepth;

      if(-1 != state.fmt().sampledepth) {

         if(state.fmt().sampledepth() != in.fmt().sampledepth()) {

            trace("codec:input:handle:fmt:sampledepth:", in.fmt().sampledepth(),
                                                 " -> ", state.fmt().sampledepth());

            in.fmt().sampledepth() = state.fmt().sampledepth();
         }
      }

      /* -------------------------------------------------------------------- */
      // handle pcm sample endianess;

      if(0 == state.fmt().endianess() || 1 == state.fmt().endianess()) {

         if(state.fmt().endianess() != in.fmt().endianess()) {

            trace("codec:input:handle:fmt:endianess:", in.fmt().endianess(),
                                               " -> ", state.fmt().endianess());

            in.fmt().endianess() = state.fmt().endianess();
         }
      }

      /* -------------------------------------------------------------------- */
      // handle pcm sample signedness;

      if(0 == state.fmt().signedness() || 1 == state.fmt().signedness()) {

         if(state.fmt().signedness() != in.fmt().signedness()) {

            trace("codec:input:handle:fmt:signedness:", in.fmt().signedness(),
                                                " -> ", state.fmt().signedness());

            in.fmt().signedness = state.fmt().signedness;
         }
      }

      /* -------------------------------------------------------------------- */
      // handle device;

      if(state.dev().size()) {

         string_t const dev0 = in.dev();
         string_t const dev1 = state.dev();

         if(dev0 != dev1) {

             trace("codec:input:handle:dev:", in.dev(),
                     " -> ", state.dev());

             in.dev = state.dev();

             ips = PLAYSTATE_STOP;
             trace("codec:input:handle:dev:interrupt:", estr2(ips));
         }
      }
      /* -------------------------------------------------------------------- */
      // handle playbackspeed state;
      if(-1 != state.speedstate)
      {
          trace("codec:input:handle:speestate:", in.speedstate(), " -> ", state.speedstate());
          in.speedstate = state.speedstate;
      }
      /* -------------------------------------------------------------------- */
      // handle sourceswitchramp
      if(-1 != state.sourceswitchramp)
      {
          trace("codec:input:handle:sourceswitchramp:", in.sourceswitchramp(), " -> ", state.sourceswitchramp());
          in.sourceswitchramp = state.sourceswitchramp;
      }

      /* -------------------------------------------------------------------- */
      // handle videosupport state;
      if(-1 != state.videosupport)
      {
          trace("codec:input:handle:videosupport:", in.videosupport(), " -> ", state.videosupport());
          in.videosupport = state.videosupport;
      }

      // handle videopropertycontrol
      if(-1 != state.videopropertycontrol)
      {
          trace("codec:input:handle:videopropertycontrol:", in.videopropertycontrol(), " -> ", state.videopropertycontrol());
          in.videopropertycontrol = state.videopropertycontrol;
      }

      /* -------------------------------------------------------------------- */
      // handle speed;

      if(0 != state.speed) {

         if(state.speed != in.speed) {

            // range check;
            if(state.speedstate){

            if(-3200 > state.speed) {

               trace("codec:input:handle:speed:", state.speed(), " out of range [-3200;3200]");

               state.speed = ME_SPEED_FREV3200;
            }
            if(3200 < state.speed) {

               trace("codec:input:handle:speed:", state.speed(), " out of range [-20;20]");

               state.speed = ME_SPEED_FFWD3200;
            }

            }
            else {

            if(-20 > state.speed) {

               trace("codec:input:handle:speed:", state.speed(), " out of range [-20;20]");

               state.speed = ME_SPEED_FREV20;
            }
            if(20 < state.speed) {

               trace("codec:input:handle:speed:", state.speed(), " out of range [-20;20]");

               state.speed = ME_SPEED_FFWD20;
            }
            }

            trace("codec:input:handle:speed:", in.speed(),
                                       " -> ", state.speed());

            in.speed = state.speed;

            // gst pipeline can change position on-the-fly;
            //ips = PLAYSTATE_STOP;
         }
      }

      /* -------------------------------------------------------------------- */
      // handle volume;

      if(-1 != state.vol) {

         if(state.vol != in.vol) {

            trace("codec:input:handle:vol:", in.vol(), " -> ", state.vol());

            in.vol = state.vol;

            // gst pipeline can change volume on-the-fly;
            //ips = PLAYSTATE_STOP;
         }
      }

      /* -------------------------------------------------------------------- */
      // handle handle;

      if(-1 != state.handle) {

         if(state.handle != in.handle) {

            trace("codec:input:handle:handle:", in.handle(),
                                        " -> ", state.handle());

            in.handle = state.handle;
         }
      }

      /* -------------------------------------------------------------------- */
      // handle server name;

      if(state.server().size()) {

         string_t const server0 = in.server;
         string_t const server1 = state.server;

         if(server0 != server1) {

            trace("codec:input:handle:server:", in.server,
                                        " -> ", state.server);
            in.server = state.server;
         }
      }

      /* -------------------------------------------------------------------- */
      // handle logconfig;

      if(0 != mcmp(&in.logconfig, &state.logconfig, sizeof(logconfig_t))) {

         trace("codec:input:handle:logconfig:", in.logconfig().str(),
                                        " -> ", state.logconfig().str());
         in.logconfig = state.logconfig;
      }

      /* -------------------------------------------------------------------- */
      // handle position;

      if(false == state.pos().empty()) {

         //trace("codec:input:handle:position:", state.pos().str());

         /* ----------------------------------------------------------------- */
         // plausibility check;

         int_t const cnt = (-1 != state.pos().bytes) +
                           (-1 != state.pos().ms)    +
                           (-1 != state.pos().pct);
         if(1 < cnt) {

            trace("codec:input:handle:position:conflicting position request:", cnt);
            state.reason = REASON_FORMAT_ERROR;
            forward(state);
            return REASON_FORMAT_ERROR;
         }

         if(-1 != state.pos().pct && (state.pos().pct < 0 || 100 < state.pos().pct)) {

             trace("codec:input:handle:position:percentage out of rage [0:100]:", state.pos().pct());
             state.reason = REASON_ARGUMENT_ERROR;
             forward(state);
             return REASON_ARGUMENT_ERROR;
         }

         /* ----------------------------------------------------------------- */

         trace("codec:input:handle:position:", in.pos().str(), " -> ", state.pos().str());

         in.pos = state.pos;
      }

      /* -------------------------------------------------------------------- */
      // handle duration;

      // clients can propose a duration for the given url which will
      // be overwritten if the codec dll is able to parse it internally;

      dur_t dur1 = state.dur;

      if(false == dur1.empty()) {

         dur_t dur0 = in.dur;

         trace("codec:input:handle:dur:", dur0().str(), " -> ", dur1().str());

         in.dur = dur1;
      }

      /* -------------------------------------------------------------------- */
      // handle url;

      if(!state.url().empty() && state.reason() != REASON_CODECS &&
         (PLAYSTATE_PLAY  == state.playstate() ||
          PLAYSTATE_PAUSE == state.playstate())) {

         trace("codec:input:handle:url:", state.url());

         string_t const url0 = in.url;
         string_t const url1 = state.url;

         string_t ext0 = url0.ext();
         string_t ext1 = url1.ext();

         ext0.lower();
         ext1.lower();

         if(ext1.empty()) { // check existing extension;

            trace("codec:input:handle:url:no extension");
            state.reason = REASON_ARGUMENT_ERROR;
            forward(state);
            return REASON_ARGUMENT_ERROR;
         }

         if(url0 == url1 && 0 != state.speed) {

            trace("codec:input:handle:url:speed change with identical url");

            // pipeline can handle speedchange on the fly;
            //ips = PLAYSTATE_NOP;

         } else {

            in.url = state.url;

            int_t const isvideo = onisvideo(ext1);

            trace("codec:input:handle:url:isvideo:", isvideo);

            if(ext0 != ext1 || 1 == isvideo ) {

               ips = PLAYSTATE_STOP;

            } else if(string_t("pcm") == ext1) {

               ips = PLAYSTATE_NOP ;

            } else {
               ips = PLAYSTATE_PAUSE;
            }
            trace("codec:input:handle:url:interrupt:", estr2(ips));

            if(REASON_OK == state.reason) {

               trace("codec:input:handle:url:states:clear");
               states.clear();
            }
         }
      }

      /* -------------------------------------------------------------------- */

      if(state.url().empty() && states.size() && PLAYSTATE_PLAY == state.playstate) {

         state_t const next = states.a->obj;
         states.remove(states.a);

         string_t const url = next.url;
         pos_t    const pos = next.pos;

         in.url = url;

         in.pos = pos;

         ips = PLAYSTATE_PAUSE;

         trace("codec:input:handle:url:next:", url, " pos:", pos.str(), " buffered:", states.size());

      } else if(state.url().empty() &&
                0 == state.speed  &&
                in.url().size() &&
                PLAYSTATE_PLAY == state.playstate &&
                PLAYSTATE_PAUSE != out.playstate) {

         ips = PLAYSTATE_PAUSE;
         trace("codec:input:handle:url:interrupt:", estr2(ips));

         trace("codec:input:handle:url:repeat:", in.url());
      }

      /* -------------------------------------------------------------------- */

      if(isvideo) {

         /* ----------------------------------------------------------------- */

         if(in.vprops() != state.vprops()) {

            trace("codec:input:handle:vprops:", in.vprops().str(), " -> ", state.vprops().str());
         }

         /* ----------------------------------------------------------------- */
         // handle video brightness;

         if((  0 <= state.vprops().brightness) &&
            (200 >= state.vprops().brightness)){

            out.vprops().brightness = in.vprops().brightness =
                                   state.vprops().brightness;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

         /* ----------------------------------------------------------------- */
         // handle video hue;

         if((-180 <= state.vprops().hue) &&
            ( 180 >= state.vprops().hue)) {

            out.vprops().hue = in.vprops().hue =
                            state.vprops().hue;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

         /* ----------------------------------------------------------------- */
         // handle video saturation;

         if((  0 <= state.vprops().saturation) &&
            (200 >= state.vprops().saturation)){

            out.vprops().saturation = in.vprops().saturation =
                                   state.vprops().saturation;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

         /* -------------------------------------------------------------------- */
         // handle video contrast;

         if((  0 <= state.vprops().contrast) &&
            (200 >= state.vprops().contrast)) {

            out.vprops().contrast = in.vprops().contrast =
                                 state.vprops().contrast;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

          /* -------------------------------------------------------------------- */
          // handle video brightnessoffset;

          if((-100 <= state.vprops().brightnessoffset) &&
             ( 100 >= state.vprops().brightnessoffset)){

            out.vprops().brightnessoffset = in.vprops().brightnessoffset =
                                         state.vprops().brightnessoffset;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

         /* -------------------------------------------------------------------- */
         // handle video saturationoffset;

         if((  0 <= state.vprops().saturationoffset) &&
            (200 >= state.vprops().saturationoffset)) {

            out.vprops().saturationoffset = in.vprops().saturationoffset =
                                         state.vprops().saturationoffset;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }

         /* -------------------------------------------------------------------- */
         // handle video hueoffset;

         if((-180 <= state.vprops().hueoffset) &&
            ( 180 <= state.vprops().hueoffset)) {

            out.vprops().hueoffset = in.vprops().hueoffset =
                                  state.vprops().hueoffset;

            if(PLAYSTATE_SET == state.playstate) {

               ips = PLAYSTATE_SET;
            }
         }
      }

        /* -------------------------------------------------------------------- */
        // handle ripconf;

         if((0 != state.ripconf().tracknumber) && (PLAYSTATE_PLAY == state.playstate || PLAYSTATE_PAUSE == state.playstate)) {

            if(state.ripconf().tracknumber != in.ripconf().tracknumber) {

               trace("me:codec:input:handle:ripconf:tracknumber:", in.ripconf().tracknumber(), " -> ", state.ripconf().tracknumber());

               in.ripconf().tracknumber = state.ripconf().tracknumber;

               ips = PLAYSTATE_STOP;
            }
         }

         if((ME_EF_UNDEFINED != state.ripconf().encodeformat) && (PLAYSTATE_PLAY == state.playstate || PLAYSTATE_PAUSE == state.playstate)) {

            if(state.ripconf().encodeformat != in.ripconf().encodeformat) {

               trace("me:codec:input:handle:ripconf:encodeformat:", in.ripconf().encodeformat(), " -> ", state.ripconf().encodeformat());

               in.ripconf().encodeformat = state.ripconf().encodeformat;

               ips = PLAYSTATE_STOP;
            }
         }

         if((ME_EQ_UNDEFINED != state.ripconf().encodequality) && (PLAYSTATE_PLAY == state.playstate || PLAYSTATE_PAUSE == state.playstate)){

            if(state.ripconf().encodequality != in.ripconf().encodequality) {

               trace("me:codec:input:handle:ripconf:encodequality:", in.ripconf().encodequality(), " -> ", state.ripconf().encodequality());

               in.ripconf().encodequality = state.ripconf().encodequality;

               ips = PLAYSTATE_STOP;
            }
         }

         if(-1 != state.v4lprop().capturewidth) {

             trace("codec:input:handle:v4lprop:capture width ", state.v4lprop().capturewidth());

             out.v4lprop().capturewidth = in.v4lprop().capturewidth = state.v4lprop().capturewidth;
         }

         if(-1 != state.v4lprop().captureheight) {

             trace("codec:input:handle:v4lprop:capture height ", state.v4lprop().captureheight());

             out.v4lprop().captureheight = in.v4lprop().captureheight = state.v4lprop().captureheight;
         }

         if(-1 != state.v4lprop().fpsnumerator) {

             trace("codec:input:handle:v4lprop:fps numerator ", state.v4lprop().fpsnumerator());

             out.v4lprop().fpsnumerator = in.v4lprop().fpsnumerator = state.v4lprop().fpsnumerator;
         }

         if(-1 != state.v4lprop().fpsdenominator) {

             trace("codec:input:handle:v4lprop:fps denominator ", state.v4lprop().fpsdenominator());

             out.v4lprop().fpsdenominator = in.v4lprop().fpsdenominator = state.v4lprop().fpsdenominator;
         }

         if(-1 != state.v4lprop().motion) {

             trace("codec:input:handle:v4lprop:motion ", state.v4lprop().motion());

             out.v4lprop().motion = in.v4lprop().motion = state.v4lprop().motion;
         }

         if(10 > state.v4lprop().pixformat) {

             trace("codec:input:handle:v4lprop:pixformat ", state.v4lprop().pixformat());

             out.v4lprop().pixformat = in.v4lprop().pixformat = state.v4lprop().pixformat;
         }

         if(!state.v4lprop().v4linput().empty()){

             trace("codec:input:handle:v4linput:", state.v4lprop().v4linput());

             out.v4lprop().v4linput = in.v4lprop().v4linput = state.v4lprop().v4linput();
         }

         me::trace("codec:input:handle:v4lprop::", out.v4lprop().str());
         /* -------------------------------------------------------------------- */
         // handle playstate;

         trace("codec:input:handle:playstate:", estr2(in.playstate),
                 " -> ", estr2(state.playstate));

         in.playstate = state.playstate;

      /* -------------------------------------------------------------------- */

      return REASON_OK;
   }

   /* ----------------------------------------------------------------------- */

   void_t transfer() {

      /* -------------------------------------------------------------------- */
      // url;

      if(in.url().size()) {

         trace("codec:transfer:out:url:", out.url(), " -> ", in.url());

         out.url = in.url;

         /* ----------------------------------------------------------------- */
         // clear duration for new url;
         trace("codec:transfer:Before clearing :dur:", out.dur().str());
         out.dur().clear();

         trace("codec:transfer:clear:dur:", out.dur().str());

         /* ----------------------------------------------------------------- */
      }

      /* -------------------------------------------------------------------- */
      // handle;

      if(in.handle() != -1) {

         trace("codec:transfer:out:handle:", out.handle(), " -> ", in.handle());

         out.handle = in.handle;
      }

      /* -------------------------------------------------------------------- */
      // dev;

      if(in.dev().size()) {

         trace("codec:transfer:out:dev:", out.dev(), " -> ", in.dev());

         out.dev = in.dev;
      }

      /* -------------------------------------------------------------------- */
      // vol;

      if(0 <= in.vol && in.vol <= 100) {

         trace("codec:transfer:out:vol:", out.vol(), " -> ", in.vol());

         out.vol = in.vol;
      }

      /* -------------------------------------------------------------------- */
      // pos;

      if((-1 != in.pos().bytes && 0 != in.pos().bytes) ||
         (-1 != in.pos().ms    && 0 != in.pos().ms   ) ||
         (-1 != in.pos().pct   && 0 != in.pos().pct  )) {

         if(PLAYSTATE_PAUSE == out.playstate) {

            trace("codec:transfer:out:pos:", out.pos().str(), " -> ", in.pos().str());

            out.pos = in.pos;

            branch(out, true);
         }
      }

      /* -------------------------------------------------------------------- */
      // dur;

      dur_t dur1 = in.dur;

      if(false == dur1.empty()) {

         dur_t dur0 = out.dur;

         trace("codec:transfer:handle:dur:", dur0.str(), " -> ", dur1.str());

         out.dur = dur1;

         if(PLAYSTATE_PAUSE == out.playstate) {

             branch(out, true);
         }
      }

      /* -------------------------------------------------------------------- */
      // logconfig;

      if(in.logconfig() != out.logconfig()) {

         trace("codec:transfer:handle:logconfig:", out.logconfig().str(), " -> ", in.logconfig().str());

         out.logconfig = in.logconfig;
      }

      /* -------------------------------------------------------------------- */
      // reason;

      //if(REASON_OK != in.reason) {
      //
      //   trace("codec:transfer:out:reason:", estr2(out.reason()), " -> ", estr2(in.reason()));
      //
      //   out.reason = in.reason;
      //}


      /* -------------------------------------------------------------------- */
      // ripconf;

      if(in.ripconf().tracknumber) {

         trace("me:codec:transfer:out:tracknumber:", out.ripconf().tracknumber(), " -> ", in.ripconf().tracknumber());

         out.ripconf().tracknumber = in.ripconf().tracknumber;
      }

      if(in.ripconf().encodeformat) {

         trace("me:codec:transfer:out:encodeformat:", out.ripconf().encodeformat(), " -> ", in.ripconf().encodeformat());

         out.ripconf().encodeformat = in.ripconf().encodeformat;
      }

      if(in.ripconf().encodequality) {

         trace("me:codec:transfer:out:encodequality:", out.ripconf().encodequality(), " -> ", in.ripconf().encodequality());

         out.ripconf().encodequality = in.ripconf().encodequality;
      }

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   void_t branch(state_t &state, int_t const force = false) {

      me::traces("codec:branch");

      static int_t const sample_length = 300; // 200 ms;

      static int_t pos_ms = 0;

      /* -------------------------------------------------------------------- */

      int_t sample = 0;

      // stream position changed;
      if(pos_ms / sample_length != state.pos().ms() / sample_length) {
         sample = 1;
      }

      // stream position increased during frev;
      if(0 > state.speed() && pos_ms < state.pos().ms()) {
         sample = 0;
      }

      if(REASON_OK != state.reason()) {
         sample = 1;
      }

      /* -------------------------------------------------------------------- */

      pos_ms = state.pos().ms();

      if(force || sample) {

         trace("codec:branch:dll:", codecname, " force:", force, " ", state.str());

         //if(mecallback) {
         //   mecallback(mecontext, &state);
         //}
         if(call && context) {
            call(state, context);
         }
         if(/*0 == mecallback && */0 == call || 0 == context) {
            trace("codec:branch:call:missing");
         }
      }
   }

   /* ----------------------------------------------------------------------- */

   void_t forward(state_t &state) {

      me::traces("codec:forward");

      /* -------------------------------------------------------------------- */

      trace("codec:forward:reason:", estr2(state.reason));

      /* -------------------------------------------------------------------- */

      if(REASON_OK != state.reason()) {

         if(0 > state.reason()) {

            trace("codec:forward:reason:error:", estr2(state.reason()));

            if(0 /*REASON_PIPELINE_ERROR == state.reason || REASON_DEVICE_ERROR == state.reason*/) {

               // update client before shutdown;
               //branch(state);

               trace("codec:forward:force shutdown");
               cleanup();
               proc_t::shut();

               // this is the end.
            }
         }
      }

      /* -------------------------------------------------------------------- */

      branch(state);

      /* -------------------------------------------------------------------- */

      if(REASON_END_OF_STREAM  == state.reason ||
         REASON_OK             >  state.reason) {

         out.pos().clear();
         out.dur().clear();

         if(states.size()) {

            state_t next = states.a->obj;

            trace("codec:forward:next:url:", next.url, " buffered:", states.size());

            state_t play(PLAYSTATE_PLAY);

            ctrl(play, 0);

         } else {

            trace("codec:forward:buffered:zero");

            state_t stop(PLAYSTATE_STOP);

            stop.handle = state.handle;

            if(REASON_OK > state.reason) {

               stop.reason = REASON_OK;

            } else {

               stop.reason = REASON_EMPTY;
            }

            ctrl(stop, 0);
         }
         state.reason = REASON_OK;

         trace("codec:forward:exit");
      }
   }

   /* ----------------------------------------------------------------------- */
   // pump;

   static void_t dispatch_read(msg_t& msg, node_t::res_e res, void_t* that) {

      codec_t& codec = *(codec_t*)that;

      switch(res) {
         case node_t::res_first: {

            trace("codec:dispatch_read:first");

            break;
         }
         case node_t::res_ok: {

            trace("codec:dispatch_read:ok");

            if(cycle_t::cid == msg.cid()) {

               trace("codec:dispatch_read:cycle_t");

               cycle_t cycle;
               msg.get(cycle);

               if(flush == cycle.cmd)  {

                  trace("codec:dispatch_read:flush");
                  codec.reader.flush();
                  // stop cycling;

               } else if(on == cycle.cmd) {

                  trace("codec:dispatch_read:cycle:on");
                  if(0 == codec.oncycle_read(msg)) { // payload;

                     codec.cycle_read(on); // next cycle;
                  }
               } else if(off == cycle.cmd) {
                  trace("codec:dispatch_read:cycle:off");
               }
            } else if(ctrl_t::cid == msg.cid()){

               trace("codec:dispatch_read:ctrl_t");

               ctrl_t ctrl1;
               msg.get(ctrl1);

               codec.onctrl_read(ctrl1);

               if(ctrl1.sync) {
                  msg.swap();
                  codec.reader.post(node_t::queue_ctrl, msg);
               }
            }
            break;
         }
         case node_t::res_timeout: {

            static int_t i = -1;
            trace("codec:dispatch_read:timeout:#:", ++i);

            break;
         }
         case node_t::res_request_con: {

            trace("codec:dispatch_read:connect:request");
            break;
         }
         case node_t::res_confirm_con: {

            trace("codec:dispatch_read:connect:confirm");
            break;
         }
         case node_t::res_last: {

            trace("codec:dispatch_read:last");

            // custom cleanup here;

            trace("codec:dispatch_read:last:exit");
            break;
         }
         default: {
            trace("codec:dispatch_read:", res, ":unhandled");
            break;
         }
      }
   }

   /* ----------------------------------------------------------------------- */

   int_t cycle_read(cmd_e const &cmd0) {

      cycle_t cycle(cmd0);
      int_t res = 0;

      switch(cmd0) {
         //case flush:
         case off: {

            msg_t msg(reader.nid, reader.nid, cycle_t::cid);
            msg.set(cycle);

            trace("codec:cycle_read:post:", cycle.str());
            res = reader.post(node_t::queue_ctrl, msg);
            trace("codec:cycle_read:post:", cycle.str(), -1 == res ? " failed" : " succeeded");
            break;
         }
         case on: {

            msg_t msg(reader.nid, reader.nid, cycle_t::cid);
            msg.set(cycle);

            trace("codec:cycle_read:post:", cycle.str());
            res = reader.post(node_t::queue_ctrl, msg);
            trace("codec:cycle_read:post:", cycle.str(), -1 == res ? " failed" : " succeeded");
         }
         default: {
            trace("codec:cycle_read:not handled");
         }
      }
      return res;
   }

   /* ----------------------------------------------------------------------- */

   static void_t dispatch_write(msg_t& msg, node_t::res_e res, void_t* that) {

      codec_t& codec = *(codec_t*)that;

      switch(res) {
         case node_t::res_first: {

            trace("codec:dispatch_write:first");
            break;
         }
         case node_t::res_ok: {

            trace("codec:dispatch_write:ok");

            if(cycle_t::cid == msg.cid()) {

               cycle_t cycle;
               msg.get(cycle);

               if(off == cycle.cmd) {

                  trace("codec:dispatch_write:cycle:off");
                  msg.swap();
                  codec.writer.post(node_t::queue_ctrl, msg);

               } else if(on == cycle.cmd) {

                  trace("codec:dispatch_write:cycle:on");
                  if(0 == codec.oncycle_write(msg)) { // payload;

                     codec.cycle_write(on); // next cycle;
                  }
               } else if(flush == cycle.cmd)  {

                  trace("codec:dispatch_write:flush");
                  codec.writer.flush();
                  msg.swap();
                  codec.writer.post(node_t::queue_ctrl, msg);
               }
            } else if(buffer_t::cid == msg.cid()) {

               buffer_t buf;
               msg.get(buf);

               codec.onmsg_write(buf);
            }
            break;
         }
         case node_t::res_timeout: {

            static int_t i = -1;
            trace("codec:dispatch_write:timeout:#:", ++i);

            break;
         }
         case node_t::res_request_con: {

            trace("codec:dispatch_write:connect:request");
            break;
         }
         case node_t::res_confirm_con: {

            trace("codec:dispatch_write:connect:confirm");
            break;
         }
         case node_t::res_last: {

            // custom cleanup here;

            trace("codec:dispatch_write:last:exit");
            break;
         }
         default: {
            trace("codec:dispatch_write:", res, ":unhandled");
            break;
         }
      }
   }

   /* ----------------------------------------------------------------------- */

   int_t cycle_write(cmd_e const &cmd0) {

      cycle_t cycle(cmd0);

      int_t res = 0;

      switch(cmd0) {
         case flush:
         case off: {
            cycle_t cycle(flush);
            msg_t msg(wctrl.nid, writer.nid, cycle_t::cid);
            msg.set(cycle);
            trace("codec:cycle_write:fetch:", cycle.str());
            res = wctrl.fetch(node_t::queue_ctrl, msg);
            trace("codec:cycle_write:fetch:", cycle.str(), -1 == res ? " failed" : " succeeded");
            break;
            //res = writer.stop();
         }
         case on: {
            // init/play pipeline;
            //trace("codec:cycle_write:writer:run");
            //res = writer.run();
         }
      }
      return res;
   }

   /* ----------------------------------------------------------------------- */

   void_t onmsg_write(buffer_t const& buf) {

      out.url = buf.header().url;
      out.handle = buf.header().handle;

      int_t const res = write(buf);
      trace("codec:onmsg_write:res", res);

      static int_t prev_level = 0;

      int_t fill = 0;
      int_t size = 0;
      int_t const level = writer.level(node_t::queue_data, fill, size);

      trace("codec:onmsg_write:buffer_t:level:", level, " fill:", fill, " size:", size);

      int_t const threshold = 20;

      if(level < threshold && threshold <= prev_level ) {

         trace("codec:onmsg_write:buffer_t:cycle");
         cycle_read(on);
      }
      prev_level = level;
   }

   /* ----------------------------------------------------------------------- */
   // state matrix;

   // +-----------------+--------+--------+
   // + state  \  cycle + reader + writer +
   // + ----------------+--------+--------+
   // + stop            + off    + off    +
   // + pause:buffering + on     + off    +
   // + play :streaming + on     + on     +
   // + ----------------+--------+--------+

   /* ----------------------------------------------------------------------- */

   void_t onctrl_read(ctrl_t const &ctrl1) {

      playstate_e ps0 = out.playstate;
      playstate_e ps1 = ctrl1.playstate;

      trace("codec:onctrl_read:playstate:", estr2(ps0), " -> ", estr2(ps1),
                                " reason:", estr2(out.reason));

      /* -------------------------------------------------------------------- */

      if(false == out.logconfig().empty()) {

         trace("codec:onctrl_read:update logconfig");

         state_t slc; // extract state logconfig;
         slc.logconfig = out.logconfig;

         onctrl(slc);

         slc.reason = REASON_ACK_LOGCONFIG;
         branch(slc, true);

         //in.logconfig = out.logconfig = logconfig_t();
      }

      /* -------------------------------------------------------------------- */

      out.playstate = ps1;

      /* -------------------------------------------------------------------- */
      // playstate;

      switch(ps0) {
         case PLAYSTATE_NOP: // catch initial state;
         case PLAYSTATE_STOP: {
            switch(ps1) {
               case PLAYSTATE_PAUSE: {
                  oninit_int();
                  onctrl(out);
                  if(pump) {
                     cycle_write(off);
                     cycle_read(on);
                  }
                  break;
               }
               case PLAYSTATE_PLAY: {
                  oninit_int();
                  onctrl(out);
                  if(pump) {
                     cycle_write(on);
                     cycle_read(on);
                  }
                  break;
               }
               default:case PLAYSTATE_NOP:case PLAYSTATE_STOP: break;
            }
            break;
         }
         case PLAYSTATE_PAUSE: {
            switch(ps1) {
               case PLAYSTATE_STOP: {
                  if(pump) {
                     cycle_write(flush);
                  }
                  onctrl(out);
                  onfini_int();
                  break;
               }
               case PLAYSTATE_SET:
                   onctrl(out);
                   out.playstate = ps0;
                   break;
               case PLAYSTATE_PLAY: {
                  onctrl(out);
                  break;
               }
               default:case PLAYSTATE_NOP:case PLAYSTATE_PAUSE: break;
            }
            break;
         }
         case PLAYSTATE_PLAY: {
            switch(ps1) {
               case PLAYSTATE_STOP: {
                  if(pump) {
                     cycle_read(off);
                     cycle_write(flush);
                  }
                  onctrl(out);
                  onfini_int();
                  break;
               }
               case PLAYSTATE_SET:
                   onctrl(out);
                   out.playstate = ps0;
                   break;
               case PLAYSTATE_PAUSE: {
                  onctrl(out);
                  break;
               }
               default:case PLAYSTATE_NOP:case PLAYSTATE_PLAY: break;
            }
            break;
         }
         //default:case PLAYSTATE_NOP: break;
      }

      if(ps0 != ps1) {

         //branch(out, true);
      }

      /* -------------------------------------------------------------------- */

      trace("codec:onctrl_read:out:", out.str(0), " exit");
   }

   /* ----------------------------------------------------------------------- */

   int_t oncycle_read(msg_t& /*msg*/) {

      trace("codec:oncycle_read:playstate:", estr2(out.playstate));

      if(PLAYSTATE_PLAY != out.playstate) {

         trace("codec:oncycle_read:not playing");
         return -1;
      }

      /* -------------------------------------------------------------------- */

      if(0 == file.fd) {

         if(in.url().size()) {

            file.init(in.url);
            file.open();

         } else {
            return -1; // stop cycling;
         }
      }

      /* -------------------------------------------------------------------- */
      // check buffer level;

      int_t fill = 0;
      int_t size = 0;
      int_t const level = writer.level(node_t::queue_data, fill, size);
      int_t const space = size - fill;

      trace("codec:oncycle_read:level:", level, "% fill:", fill, " size:", size, " space:", space);

      /* -------------------------------------------------------------------- */
      // populate buffer;

      buffer_t buf;

      buf.header().url     = in.url;
      buf.header().handle  = in.handle;
      buf.header().speed() = in.speed();

      int_t const bytes = read(file, space, buf);

      /* -------------------------------------------------------------------- */

      if(REASON_END_OF_FILE == buf.header().reason) {

         trace("codec:oncycle_read:eof");

         file.close();
         in.url().clear();

         if(states.size()) {

            in = states.a->obj;
            states.remove(states.a);

            trace("codec:oncycle_read:states:size:", states.size(), " new url:", in.url);

         } else {

            buf.header().reason = REASON_NEW_TRACK;
         }
      }

      /* -------------------------------------------------------------------- */

      if(0 == bytes) {

         trace("codec:oncycle_read:space:low");
         return -1; // stop cycling;
      }

      /* -------------------------------------------------------------------- */
      // send chunk;

      msg_t msg(reader.nid, writer.nid, buffer_t::cid);
      msg.set(buf);

      if(node_t::res_full == reader.post(node_t::queue_data, msg)) {

          trace("codec:oncycle_read:full");
          return -1; // stop cycling;
      }

      /* -------------------------------------------------------------------- */

      return 0; // next cycle;
   }

   /* -------------------------------------------------------------------- */

   int_t oncycle_write(msg_t& /*msg*/) {

      trace("codec:oncycle_write:playstate:", estr2(out.playstate));

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   int_t ini;

   string_t dll_name;
   string_t codecname;

   state_call_t call;
   void_t *context;

   int_t pump;

   state_t in, out;
   list_t<state_t> states;

   smp_t join;

   /* ----------------------------------------------------------------------- */

   node_t mutable reader;
   node_t mutable rctrl;

   node_t mutable writer;
   node_t mutable wctrl;

   file_t file;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

 #endif // ME_CODEC_HPP_INCLUDED

/* -------------------------------------------------------------------------- */
