/* -------------------------------------------------------------------------- */
/**
 *   @defgroup mebase mebase.hpp
 *   @ingroup  MEngine
 *   @author   Stephan Pieper, 2013
 *
 *   Basework.
 */
/* -------------------------------------------------------------------------- */

#if !defined(ME_BASE_HPP_INCLUDED)
#define ME_BASE_HPP_INCLUDED

/* ------------------------------------------------------------------------- */

#include "mecodec.hpp"

/* -------------------------------------------------------------------------- */

namespace me {

/* -------------------------------------------------------------------------- */

struct dll_t {

   /* ----------------------------------------------------------------------- */

   dll_t() : ini(false), handle(0), codec(0) {
   }
   dll_t(string_t const& path0) : ini(true), handle(0), codec(0) {
      init(path0);
   }
   ~dll_t() {
      if(ini) fini();
   }
   void_t init(string_t const& url0) { // Roadmap 13010;
      url = url0;
      load();
   }
   void_t fini() {
      unload();
   }

   /* ----------------------------------------------------------------------- */

   int_t load() { // Roadmap 13010;

       handle = dlopen(url.at(), RTLD_LAZY);

       byte_t* error = dlerror();

       trace("me:base:dec:load:path:", url, " handle:", handle, " ", error ? error : "");

       if(0 == handle) return -1;

       codec_t *(*instance)() = 0;

       *(void_t**)&instance = dlsym(handle, "instance");

       if((error = dlerror())) {

          trace("me:base:dec:load:", url.name(), " sym:failed:", error);
          fini();
          return -1;
       }

       codec = instance();

       return 0;
   }
   void_t unload() {

      trace("me:base:dec:unload:", str());

      codec = 0;

      if(handle) {

         dlclose(handle);
         handle = 0;
      }

      trace("me:base:dec:unload:exit");
   }

   /* ----------------------------------------------------------------------- */

   string_t str() const {
      string_t s;
      s <<     "ini:" << ini
        << " handle:" << handle
        <<  " codec:" << codec
        <<    " url:" << url;
      return s;
   }

   /* ----------------------------------------------------------------------- */

   int_t ini;
   string_t url;
   void_t *handle;
   codec_t *codec;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

struct format_t {

   /* ----------------------------------------------------------------------- */

   string_t ext;
   dll_t dll;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

struct decs_t {

   /* ----------------------------------------------------------------------- */

   typedef vector_t<dll_t>           dlls_t;
   typedef map_t<string_t, format_t> formats_t;

   /* ----------------------------------------------------------------------- */

   decs_t() : ini(false), call(0), context(0) {
   }
   decs_t(string_t const &name0, string_t const& path0, state_call_t call0, void_t *context0) :
      ini(true), call(0), context(0) {
      init(name0, path0, call0, context0);
   }
   ~decs_t() {
      if(ini) fini();
   }

   /* ----------------------------------------------------------------------- */

   void_t init(string_t const &name0, string_t const& path0, state_call_t call0, void_t *context0) {

      add_path(path0);

      name    = name0;
      call    = call0;
        context = context0;
   }

   /* ----------------------------------------------------------------------- */

   void_t fini() {

      /* -------------------------------------------------------------------- */

      if(0 == dlls.size()) {

         trace("me:base:decs:fini:dlls:empty");
         return;
      }

      /* -------------------------------------------------------------------- */

      // for each dll;
      for(auto r = dlls.range(); r.x; ++r) {

         dll_t &dll = *r.x;

         /* ----------------------------------------------------------------- */
         // finish codec and unload dll;

         if(dll.codec) {

            trace("me:base:decs:fini:dll:codec:", dll.url.name());
            dll.codec->fini();
         }

         trace("me:base:decs:fini:dll:unload:", dll.url);
         dll.unload();

         /* ----------------------------------------------------------------- */
      }

      /* -------------------------------------------------------------------- */

      formats.clear();

      /* -------------------------------------------------------------------- */
   }

   /* ----------------------------------------------------------------------- */

   int_t add_path(string_t const& path0) {

      if(path0.size()) {

         if(NULL == dirs.find_f(path0)) {

             trace("me:base:decs:add:new:path:", path0);

             dirs.append(path0);

         } else {

            trace("me:base:decs:add:exists:path:", path0, " size:", dirs.size());
         }
      } else {

         trace("me:base:decs:add:dirs:size:", dirs.size());
      }
      return 0;
   }

   /* ----------------------------------------------------------------------- */

   int_t update_dlls(string_t const& path0 = null_t<string_t>()) { // Roadmap 13010;

      add_path(path0);

      if(0 == dirs.size()) {

         trace("me:base:decs:update_dlls:no directories");
         return -1;
      }

      // clear result of previous update;
      formats.clear();

      // scan directories for dynamic libraries;
      for(auto r = dirs.range(); r.x; ++r) {

         dir_t dir(*r.x);

         if(0 == dir.fd) {
            trace("me:base:decs:update_dlls:dir:init:failed:", dir.path);
            return -1;
         }

         string_t ent;
         while(0 == dir.read(ent)) {

            string_t const ext = ent.from(ent.size() - 3);
            if(0 == ext.compare(string_t(".so")) && ent.find_f(string_t("libmenginecodec_"))) {

               trace("me:base:decs:update_dlls:dll:found:", ent);

               // dynamic library found;
               string_t fullpath = dir.path;
               if('/' != fullpath[fullpath.size() - 1]) {
                  fullpath << string_t("/");
               }
               fullpath << ent;

               dll_t dll;
               dll.init(fullpath); // load codec dll;

               if(dll.codec) {

                  dlls.append(dll);

                  strings_t exts = dll.codec->exts();

                  // store supported extensions;
                  string_t strexts;
                  for(auto r = exts.range(); r.x; ++r) {

                     string_t const ext =  *r.x;
                     int_t exists;

                     formats_t::node_t *fp = formats.insert(ext, format_t(), exists);
                     asrt(NULL != fp);
                     format_t &format = fp->obj;

                     if(false == exists) {

                        format.ext = ext;
                     }

                     strexts << ext;
                     if(r.x != r.z) strexts << ",";
                  }
                  if(0 == strexts.size()) {
                     strexts << "<none>";
                  }
                  trace("me:base:decs:update_dlls:dll:codec:", dll.url.name(), " supports:", strexts);

               } else {

                  trace("me:base:decs:update_dlls:dll:codec:", dll.url.name(), " no extensions");
               }
            }
         }
      }

      /* -------------------------------------------------------------------- */
      // list dlls;

      trace("me:base:decs:update_dlls:list:size:", dlls.size());

      for(int_t i = 0; i < dlls.size(); ++i) {

          trace("me:base:decs:update_dlls:dll:id:", i, " url", dlls[i].url);
      }

      /* -------------------------------------------------------------------- */
      // initialise codec per format;

      for(auto r = formats.range(); r.x; ++r) {

         format_t const& format = r.x->obj;

         set_codec(format.ext, 0);
      }

      /* -------------------------------------------------------------------- */
      // list format -> dll mappings;

      for(auto r = formats.range(); r.x; ++r) {

          format_t const& format = r.x->obj;

          trace("me:base:decs:update_dlls:format:", format.ext, " -> ", format.dll.url);
      }

      /* -------------------------------------------------------------------- */

      trace("me:base:decs:update_dlls:found:dlls:", dlls.size(), " formats:", formats.size());

      return dlls.size();
   }

   /* ----------------------------------------------------------------------- */

   strings_t get_codecs(string_t ext0) const { // Roadmap 13010;

      ext0.lower();

      trace("me:base:decs:get_codecs:find:ext:", ext0, " in ", dlls.size(), " dlls");

      /* -------------------------------------------------------------------- */

      strings_t codecs;

      for(auto r = dlls.range(); r.x; ++r) {

          dll_t const& dll = *r.x;

          if(dll.codec) {

             strings_t const exts = dll.codec->exts();

             for(auto re = exts.range(); re.x; ++re) {

                string_t const& ext = *re.x;

                if(ext == ext0) {

                   trace("me:base:decs:get_codecs:ext:", ext, " supported by:", dll.url.name());
                   codecs.append(dll.url);
                }
             }
          }
      }
      trace("me:base:decs:get_codecs:ext:", ext0, " found:", codecs.size());
      return codecs;
   }

   /* ----------------------------------------------------------------------- */

   int_t set_codec(string_t ext0, int_t const& id0) {

      ext0.lower();

      trace("me:base:decs:set_codec:ext:", ext0, " id:", id0);

      /* -------------------------------------------------------------------- */

      formats_t::node_t *fp = formats.find(ext0);

      if(0 == fp) {

         trace("me:base:decs:set_codec:ext:", ext0, " not supported");
         return -1;
      }

      //format_t &format = fp->obj;

      /* -------------------------------------------------------------------- */

      strings_t codecs = get_codecs(ext0);

      if(id0 < 0) {

         trace("me:base:decs:setcodec:id:", id0, " not found");
         return -1;
      }
      if(codecs.size() <= id0) {

         trace("me:base:decs:set_codec:id:", id0, " out of range [0:", codecs.size() - 1, "]");
         return -1;
      }

      string_t const url = codecs[id0];

      /* -------------------------------------------------------------------- */

      return set_codec(ext0, url);
   }

   /* ----------------------------------------------------------------------- */

   int_t set_codec(string_t ext0, string_t url0) {

      ext0.lower();

      trace("me:base:decs:set_codec:ext:", ext0, " select url:", url0);

      /* -------------------------------------------------------------------- */
      // get format matching extension;

      formats_t::node_t *fp = formats.find(ext0);

      if(0 == fp) {

         trace("me:base:decs:set_codec:ext:", ext0, " not supported");
         return -1;
      }

      format_t &format = fp->obj;

      /* -------------------------------------------------------------------- */
      // complement url;

      string_t url1 = url0.name();

      if(string_t("libmengine") != url1.upto(9)) {

         url1.insert("libmengine");
         trace("me:base:decs:set_codec:prepend:\"libmengine\" -> ", url1);

      } else if(string_t("lib") != url1.upto(2)) {

         url1.insert("lib");
         trace("me:base:decs:set_codec:prepend:\"lib\" -> ", url1);
      }

      if(url1.from(url1.size() - 6) != string_t("_so.so")) {

         url1.append("_so.so");
         trace("me:base:decs:set_codec:apppend:\"_so.so\" -> ", url1);

      } else if(url1.from(url1.size() - 3) != string_t(".so")) {

         url1.append(".so");
         trace("me:base:decs:set_codec:apppend:\".so\" -> ", url1);
      }

      trace("me:base:decs:set_codec:url:", url0, " -> ", url1);

      /* -------------------------------------------------------------------- */
      // reference dll in format map;

      for(int_t i = 0; i < dlls.size(); ++i) {

         if(url1 == dlls[i].url.name()) {

            format.dll = dlls[i];

            trace("me:base:decs:set_codec:format:", format.ext, " decoded by:", format.dll.str());

            break;
         }
      }

      /* -------------------------------------------------------------------- */
      // configure codec contained in dll;

      dll_t &dll = format.dll;

      if(dll.codec) {

         int_t const local = 1;

         trace("me:base:decs:set_codec:dll:codec:config local:", local);

         dll.codec->config(url1, "~mengine", call, context, local);
      }

      /* -------------------------------------------------------------------- */

      return 0;
   }

   /* ----------------------------------------------------------------------- */

   codec_t *get_codec(string_t ext0) {

      ext0.lower();

      trace("me:base:decs:get_codec:ext:", ext0);

      formats_t::node_t *fp = formats.find(ext0);

      if(0 == fp) {

         trace("me:base:decs:get_codec:ext:", ext0, " not supported");
         return 0;
      }

      format_t &format = fp->obj;

      codec_t *cp = format.dll.codec;

      if(0 == cp) {

         trace("me:base:decs:get_codec:codec:null");
         return 0;
      }

      return cp;
   }

   /* ----------------------------------------------------------------------- */

   string_t str() const {
      string_t s;
      s <<        "name:"   << name
        <<    " dlls:size:" << dlls.size()
        << " formats:size:" << formats.size();
      return s;
   }

   /* ----------------------------------------------------------------------- */

   int_t ini;

   strings_t dirs;
   dlls_t    dlls;
   formats_t formats;

   string_t name;
   state_call_t call;
    void_t *context;

   /* ----------------------------------------------------------------------- */
};

/* -------------------------------------------------------------------------- */

} // me;

/* -------------------------------------------------------------------------- */

 #endif // ME_BASE_HPP_INCLUDED

 /* ------------------------------------------------------------------------- */
