// -----------------------------------------------------------------------------
//
//  Copyright (c) 2013 - 2017 Robert Bosch Car Multimedia GmbH, Hildesheim
//
//  Author: Stephan Pieper
//
//  Permission is hereby granted, free of charge, to any person obtaining a copy
//  of this software and associated documentation files (the "Software"), to deal
//  in the Software without restriction, including without limitation the rights
//  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions:
//
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN No  EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//  THE SOFTWARE.
//
// -----------------------------------------------------------------------------
//
// Multipurpose lib.
//
// Os abstraction:   types, lib, math, time
//
// Base containers:  vector_t, list_t, tree_t
//
// Containers:       string_t, set_t, multiset_t, map_t,
//                   multimap_t, ring_t
//
// Allocators:       store_t, pool_t, ram_t
//
// Parallel:         smp_t, thr_t, lock_t, signal_t, proc_t,
//                   shm_t, suid_t
//
// FS abstraction:   dir_t, file_t, log_t
//
// Communication:    msg_t, node_t
//
// -----------------------------------------------------------------------------

/*lint -save -e808  -e115  -e142  -e30   -e1764 -e1514 -e1404 -e839  -e801  -e728  */
/*lint -save -e1785 -e826  -e641  -e437  -e578  -e712  -e715  -e713  -e1786 -e551  */
/*lint -save -e747  -e64   -e818  -e830  -e1725 -e1704 -e1534 -e1773 -e451  -e668  */
/*lint -save -e704  -e424  -e666  -e1573 -e1509 -e831  -e1511 -e1551 -e1538 -e611  */
/*lint -save -e1579 -e506  -e1401 -e1712 -e1788 -e1516 -e1015 -e1013 -e838  -e530  */
/*lint -save -e1566 -e37   -e18   -e1052 -e10   -e438  -e529  -e514  -e58   -e449  */
/*lint -save -e1502 -e587  -e774  -e776  -e1763 -e820  -e40   -e26   -e1070 -e762  */
/*lint -save -e1556 -e1540 -e722  -e1729 -e717  -e703  -e533  -e1790 -e1730 -e740  */
/*lint -save -e1793 -e613  -e750  -e720  -e647  -e524  -e866  -e1058 -e520  -e1055 */
/*lint -save -e746  -e1762 */

// -----------------------------------------------------------------------------

#if !defined(M1_HPP)
#define M1_HPP

// -----------------------------------------------------------------------------

#define M1_SPREAD (0x0100)

#define M1_ENV_NONE   (M1_SPREAD   * 0x00)
#define M1_ENV_LINUX  (M1_ENV_NONE + 0x01)
#define M1_ENV_WIN    (M1_ENV_NONE + 0x02)
#define M1_ENV_LAST   (M1_ENV_NONE + 0x03)

// -----------------------------------------------------------------------------
// feature choice;

#define M1_LIB_NONE  (M1_SPREAD   * 0x01)
#define M1_LIB_LINUX (M1_LIB_NONE + 0x01)
#define M1_LIB_WIN   (M1_LIB_NONE + 0x02)
#define M1_LIB_NACL  (M1_LIB_NONE + 0x03)
#define M1_LIB_LAST  (M1_LIB_NONE + 0x04)

#define M1_TIME_NONE  (M1_SPREAD    * 0x02)
#define M1_TIME_LINUX (M1_TIME_NONE + 0x01)
#define M1_TIME_WIN   (M1_TIME_NONE + 0x02)
#define M1_TIME_NACL  (M1_TIME_NONE + 0x03)
#define M1_TIME_LAST  (M1_TIME_NONE + 0x04)

#define M1_CONT_NONE  (M1_SPREAD    * 0x03)
#define M1_CONT_M1    (M1_CONT_NONE + 0x01)
#define M1_CONT_LAST  (M1_CONT_NONE + 0x02)

#define M1_PARA_NONE  (M1_SPREAD    * 0x04)
#define M1_PARA_LINUX (M1_PARA_NONE + 0x01)
#define M1_PARA_WIN   (M1_PARA_NONE + 0x02)
#define M1_PARA_LAST  (M1_PARA_NONE + 0x03)

#define M1_FS_NONE  (M1_SPREAD  * 0x07)
#define M1_FS_LINUX (M1_FS_NONE + 0x01)
#define M1_FS_WIN   (M1_FS_NONE + 0x02)
#define M1_FS_LAST  (M1_FS_NONE + 0x03)

#define M1_PROC_NONE  (M1_SPREAD    * 0x08)
#define M1_PROC_LINUX (M1_PROC_NONE + 0x01)
#define M1_PROC_WIN   (M1_PROC_NONE + 0x02)
#define M1_PROC_LAST  (M1_PROC_NONE + 0x03)

#define M1_COM_NONE (M1_SPREAD   * 0x09)
#define M1_COM_M1   (M1_COM_NONE + 0x01)
#define M1_COM_LAST (M1_COM_NONE + 0x02)

#define M1_ORG_NONE (M1_SPREAD   * 0x0a)
#define M1_ORG_M1   (M1_ORG_NONE + 0x01)
#define M1_ORG_LAST (M1_ORG_NONE + 0x02)

// -----------------------------------------------------------------------------
// feature set;

//#define M1_FEAT_PRINTF
//#define M1_FEAT_NONEW
//#define M1_FEAT_DEBUG_HEAP
//#define M1_FEAT_CONT_DEBUG
//#define M1_FEAT_SMP_DEBUG
//#define M1_FEAT_THR_DEBUG
//#define M1_FEAT_SHM_DEBUG
//#define M1_FEAT_RING_DEBUG
//#define M1_FEAT_FS_DEBUG
//#define M1_FEAT_COM_DEBUG
//#define M1_FEAT_PROC_DEBUG

// -----------------------------------------------------------------------------
// predefined environments: linuc;

#if (M1_ENV_LINUX == M1_ENV)
   // feature choices;
   #if !defined(M1_LIB)
      #define M1_LIB    M1_LIB_LINUX
   #endif
   #if !defined(M1_TIME)
      #define M1_TIME   M1_TIME_LINUX
   #endif
   #if !defined(M1_CONT)
      #define M1_CONT   M1_CONT_M1
   #endif
   #if !defined(M1_PARA)
      #define M1_PARA   M1_PARA_LINUX
   #endif
   #if !defined(M1_FS)
      #define M1_FS     M1_FS_LINUX
   #endif
   #if !defined(M1_PROC)
      #define M1_PROC   M1_PROC_LINUX
   #endif
   #if !defined(M1_COM)
      #define M1_COM    M1_COM_M1
   #endif
   #if !defined(M1_ORG)
      #define M1_ORG    M1_ORG_M1
   #endif

   // feature set;
   //#define M1_FEAT  M1_FEAT_PRINTF &\
   //                 M1_FEAT_NONEW
#endif

// -----------------------------------------------------------------------------
// predefined environments: windows;

#if (M1_ENV_WIN == M1_ENV)
   #if !defined(M1_COM)
      #define M1_LIB    M1_LIB_WIN
   #endif
   #if !defined(M1_COM)
      #define M1_TIME   M1_TIME_WIN
   #endif
   #if !defined(M1_COM)
      #define M1_CONT   M1_CONT_M1
   #endif
   #if !defined(M1_COM)
      #define M1_PARA   M1_PARA_WIN
   #endif
   #if !defined(M1_COM)
      #define M1_FS     M1_FS_WIN
   #endif
   #if !defined(M1_COM)
      #define M1_PROC   M1_PROC_WIN
   #endif
   #if !defined(M1_COM)
      #define M1_COM    M1_COM_M1
   #endif
   #if !defined(M1_ORG)
      #define M1_ORG    M1_ORG_M1
   #endif
   // feature set;
   #define M1_FEAT   M1_FEAT_PRINTF &\
                     M1_FEAT_NONEW
#endif

// -----------------------------------------------------------------------------
// fallback to none;

#if !defined(M1_LIB)
   #define M1_LIB    M1_LIB_NONE
#endif
#if !defined(M1_TIME)
   #define M1_TIME   M1_TIME_NONE
#endif
#if !defined(M1_CONT)
   #define M1_CONT   M1_CONT_NONE
#endif
#if !defined(M1_PARA)
   #define M1_PARA   M1_PARA_NONE
#endif
#if !defined(M1_FS)
   #define M1_FS     M1_FS_NONE
#endif
#if !defined(M1_PROC)
   #define M1_PROC   M1_PROC_NONE
#endif
#if !defined(M1_COM)
   #define M1_COM    M1_COM_NONE
#endif
#if !defined(M1_ORG)
   #define M1_ORG    M1_ORG_NONE
#endif

// -----------------------------------------------------------------------------
// default path;

#if (M1_ENV_LINUX == M1_ENV)
   #if !defined(M1_PATH)
      #define M1_PATH "/tmp/"
   #endif
#endif

#if (M1_ENV_WIN == M1_ENV)
   #if !defined(M1_PATH)
      #define M1_PATH "C:\\tmp\\"
   #endif
#endif

// -----------------------------------------------------------------------------
// default scope;

#if !defined(M1_SCOPE)
   #define M1_SCOPE "m1"
#endif

// -----------------------------------------------------------------------------

#define M1_QUOTE(str) #str
#define M1_EXPAND_AND_QUOTE(str) M1_QUOTE(str)

// -----------------------------------------------------------------------------

#pragma message \
                "\n\n"\
       "+--------------------------------------+       \n"\
       "+ m1 configuration                     +       \n"\
       "+--------------------------------------+       \n"\
       "| M1_ENV  : " M1_EXPAND_AND_QUOTE(M1_ENV  ) " |\n"\
       "| M1_LIB  : " M1_EXPAND_AND_QUOTE(M1_LIB  ) " |\n"\
       "| M1_TIME : " M1_EXPAND_AND_QUOTE(M1_TIME ) " |\n"\
       "| M1_CONT : " M1_EXPAND_AND_QUOTE(M1_CONT ) " |\n"\
       "| M1_PARA : " M1_EXPAND_AND_QUOTE(M1_PARA ) " |\n"\
       "| M1_FS   : " M1_EXPAND_AND_QUOTE(M1_FS   ) " |\n"\
       "| M1_PROC : " M1_EXPAND_AND_QUOTE(M1_PROC ) " |\n"\
       "| M1_COM  : " M1_EXPAND_AND_QUOTE(M1_COM  ) " |\n"\
       "| M1_ORG  : " M1_EXPAND_AND_QUOTE(M1_ORG  ) " |\n"\
       "+--------------------------------------+       \n"

// -----------------------------------------------------------------------------
// lib;

#if !(M1_LIB_NONE == M1_LIB)

// -----------------------------------------------------------------------------

#if (M1_LIB_LINUX == M1_LIB) ||\
    (M1_LIB_WIN   == M1_LIB)

   #include <math.h>
   #include <string.h> // mem*
   #include <ctype.h>  // to*
   #include <stdio.h>  // print
   #include <stdlib.h> // alloc;
   #include <stdarg.h> // varg;

   #include <assert.h>

   #include <sys/types.h>
   #include <signal.h>
#endif

#if (M1_LIB_LINUX == M1_LIB)
   #include <unistd.h>
   #include <sys/stat.h>
   #include <dlfcn.h>
   #include <errno.h>
   #include <pwd.h>
   #include <grp.h>
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------
// types;

// 1 byte;  8 bit;
typedef char          byte_t; //                        127 0x7f
                              //                       -128 0x80
// 8 byte; 64 bit;
typedef long long int int_t;  //  9.223.372.036.854.775.807 0x7fffffffffffffff
                              // -9.223.372.036.854.775.808 0x8000000000000000
typedef void          void_t;
// 8 byte; 64 bit;
typedef double        flt_t;  // typically IEEE 754 double-precision format;
                              // 1.7E +/- 308, 15 digits;

//TODO: arbitrary precision float;

// -----------------------------------------------------------------------------
// prnt;

#if defined(M1_FEAT_PRINTF)
   #define prnt(...) printf(__VA_ARGS__)
#else
   #define prnt(...) ((void_t)0)
#endif

// -----------------------------------------------------------------------------
// trace;

typedef void_t (*tracebuf_t)(byte_t const*, int_t const);

static tracebuf_t traceb = 0;

static int_t tracereg(tracebuf_t tb) {
   int_t const res = traceb ? 1 : 0;
   traceb = tb;
   return res; // 0 = first reg; 1 = replace reg;
}

template<typename ... Ts> // forward;
static int_t tracef(Ts ... ts);

template<typename ... Ts> // forward;
static int_t tracec(Ts ... ts); // include linefeed;

// -----------------------------------------------------------------------------
// assert;

#define asrt(cond) if(false == (cond)) { tracec("assert(", M1_EXPAND_AND_QUOTE((cond)), ") ", \
                                         __FILE__, " ", __FUNCTION__, " l.", __LINE__);       \
                                         usleep(300 * 1000); \
                                         abort(); }
// assert(cond); }
// ::raise(SIGABRT); }

// -----------------------------------------------------------------------------

enum fmt_e :  long long int {
  txt,
  hed,
  bin,
  raw
};

// -----------------------------------------------------------------------------
// xenum2;

//TODO: separate def & estr impl; merge reg_t with string_t;

#define X2(a, b) a,
#define Y21(a, b) case a: return b;
#define Y22(a, b) case a: return string_t(pad((int_t)a, len, pat, fmt), ":", b);
#define XENUM2_IMPL(name) \
enum name : int_t {\
   XENUM2(X2)\
};\
static string_t estr1(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM2(Y21)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}\
static string_t estr2(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM2(Y22)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}

// xenum2 template;
//#define XENUM2(X) \
//   X(NAME_NONE, "none")\
//   X(NAME_VAL,  "val")\
//   X(NAME_LAST, "last")
//   XENUM2_IMPL(name)
//#undef XENUM2

// -----------------------------------------------------------------------------
// xenum3;

#define X3(a, b, c) a b,
#define Y31(a, b, c) case a: return c;
#define Y32(a, b, c) case a: return string_t(pad((int_t)a, len, pat, fmt), ":", c);
#define XENUM3_IMPL(name) \
enum name: int_t {\
   XENUM3(X3)\
};\
static string_t estr1(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM3(Y31)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}\
static string_t estr2(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM3(Y32)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}

// xenum3 template;
//#define XENUM3(X) \
//   X(NAME_NONE, = -1, "none")\
//   X(NAME_VAL,      , "val")\
//   X(NAME_LAST,     , "last")
//   XENUM3_IMPL(name)
//#undef XENUM3

// -----------------------------------------------------------------------------
// xenum4;

#define X4(a, b, c, d) a b,
#define Y41(a, b, c, d) case a: return c;
#define Y42(a, b, c, d) case a: return string_t(pad((int_t)a, len, pat, fmt), ":", c);
#define Y43(a, b, c, d) case a: return string_t(pad((int_t)a, len, pat, fmt), ":", d);
#define Y44(a, b, c, d) case a: return d;
#define XENUM4_IMPL(name) \
enum name : int_t {\
   XENUM4(X4)\
};\
static string_t estr1(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM4(Y41)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}\
static string_t estr2(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM4(Y42)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}\
static string_t estr3(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM4(Y43)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}\
static string_t estr4(name const val0, fmt_e const fmt = txt,\
   int_t const len = 1, byte_t const pat = '0') {\
   switch(val0) {\
      XENUM4(Y44)\
      default: return string_t(pad((int_t)val0, len, pat, fmt), ":?");\
   }\
}

// xenum4 template;
//#define XENUM4(X) \
//   X(NAME_NONE, = -1, "none", "text0")\
//   X(NAME_VAL,      , "val",  "text1")\
//   X(NAME_LAST,     , "last", "text2")
//   XENUM4_IMPL(name)
//#undef XENUM4

// -----------------------------------------------------------------------------
// libc;

static int_t mcmp(void_t const* ptr0, void_t const* ptr1, int_t const num0) {
   return ::memcmp(ptr0, ptr1, (size_t)num0);
}
static void_t *mcpy(void_t* dst0, void_t const* src0, int_t const num0) {
   if(0 == dst0 || 0 == src0) return 0;
   return ::memcpy(dst0, src0, (size_t)num0);
}
static void_t *mset(void_t* dst0, int_t val0, int_t const num0) {
   if(0 == dst0) return 0;
   return ::memset(dst0, (int)val0, (size_t)num0);
}
static void_t *mmov(void_t* dst0, void_t const* src0, int_t const num0) {
   if(0 == dst0 || 0 == src0) return 0;
   return ::memmove(dst0, src0, (size_t)num0);
}
static int_t slen(byte_t const *str0) {
   return str0 ? ::strlen((char const*)str0) : 0;
}
static int_t scmp(byte_t const *str0, byte_t const* str1, int_t const num0) {
   return ::strncmp((char const*)str0, (char const*)str1, (size_t)num0);
}

static int_t (*tlow)(int_t) = (int_t (*)(int_t))::tolower;
static int_t (*toup)(int_t) = (int_t (*)(int_t))::toupper;

#if (M1_LIB_WIN == M1_LIB)
static int_t sprnt(byte_t* dst, int_t size0, byte_t const* format, ...) {
    va_list va;
    va_start(va, format);
    int_t const res = vsnprintf_s((char*)dst, (size_t)size0, (size_t)size0 - 1, (char const*)format, va);
    va_end(va);
    return res;
}
#endif

#if (M1_LIB_LINUX == M1_LIB) ||\
    (M1_LIB_NACL  == M1_LIB)
static int_t sprnt(byte_t* dst, int_t size0, byte_t const* format, ...) {
    va_list va;
    va_start(va, format);
    int_t const res = vsnprintf((char*)dst, (size_t)size0, (char const*)format, va);
    va_end(va);
    return res;
}
#endif

// -----------------------------------------------------------------------------
// misc;

struct heap_t {

   byte_t *get(int_t const size0) const {

      // structure of allocated block:
      // 1: size1 of block content (64 bit, 8 byte, int_t); header
      // 2: size1 bytes (multiple of 8)
      // 3: 0 (64 bit, 8 byte, int_t) trailer);

      // layout: | 1 | 2 | 3 |

      // roundup requested size0 to multiple of 8 -> size1;
      int_t const size1 = (size0 + 7) & ~7; // rndup8(size0);

      // allocate block adding 2 * 64 bit for header and trailer;
      int_t* base = (int_t*)::malloc((size_t)size1 + 2 * sizeof(int_t));

      // initialize block to 0;
      mset(base, 0ll, size1 + 2 * sizeof(int_t));

      // store content size1 in 64 bit header;
      base[0] = size1;

      // store 0 in 64 bit trailer;
      base[(size1 >> 3) + 1] = 0;

      // increment content byte counter;
      // do not take header & trailer into account to ease
      // client plausibility check of allocated bytes;

      #if defined(M1_FEAT_DEBUG_HEAP)
         int_t const used0 = used();
      #endif
      //TODO: atomic check;
      used() += size1;
      #if defined(M1_FEAT_DEBUG_HEAP)
         //hint: cannot use tracec here since it requires malloc itself;
         printf("heap_t::get:used:%lld + %lld -> %lld\n", used0, size1, used());
      #endif

      // return pointer to content;
      return (byte_t*)++base;
   }
   int_t ret(byte_t* mem0) const {

      // do not free 0;
      asrt(mem0!=NULL);

      // retrieve pointer to block;
      int_t* base = ((int_t*)mem0) - 1;

      // retrieve content size;
      int_t const size0 = *base;

      #if defined(M1_FEAT_DEBUG_HEAP)
         int_t const used0 = used();
      #endif

      //TODO: atomic check;
      // decrement content byte counter;
      used() -= size0;

      #if defined(M1_FEAT_DEBUG_HEAP)
         //hint: cannot use tracec here since it requires malloc itself;
         printf("heap_t::ret:used:%lld  -  %lld -> %lld\n", used0, size0, used());
      #endif

      // return memory block to system;
      free(base);

      // return content size of deallocated block;
      return size0;
   }
   static int_t &used() {
      static int_t used_ = 0;
      return used_;
   }
};

// -----------------------------------------------------------------------------

template<typename T>
struct null_t {
   static T &def() {
      // call own ctor and dtor to check static init sequence
      // before and proper heap_t cleanup beyond main;
      static null_t self;
      static T null; // = T(); // default constructed client object;
      return null;
   }
   // null_t used in vector_t -> no tracec here to prevent recursion;
   null_t() {
      printf("null_t:ctor:heap:used:%lld\n", heap_t::used());
   };
   ~null_t() {
      printf("null_t:dtor:heap:used:%lld\n", heap_t::used());
   };
};

// -----------------------------------------------------------------------------

template<typename derived_t>
struct single_t {
   static derived_t &inst() {
      //static int_t first = 1;
      //if(first) {
      //   // call ctor and dtor to check static
      //   // init sequence proper heap_t cleanup
      //   // before and beyond main;
      //   //static single_t self;
      //   first = 0;
      //}
      static derived_t object;
      return object;
   }
protected:
   single_t() {
      tracec("single:ctor");
   };
   virtual ~single_t() {
      tracec("single:dtor");
   };
private:
   single_t(single_t const &) = delete;
   single_t& operator=(single_t const &) = delete;
};

// -----------------------------------------------------------------------------

template<typename client_t>
struct scope_t {
   scope_t(client_t& client0, int_t const op0 = true) :
      client(client0), op(op0) {
      if(op) client.enter();
   }
   ~scope_t() {
      if(op) client.exit();
   }
   client_t& client;
   int_t op;
};

// -----------------------------------------------------------------------------

template<typename client_t>
struct auto_t {
   auto_t(client_t& client0) : client(client0) {
      client.init();
   }
   template<typename ... Ts>
   auto_t(client_t& client0, Ts ... ts) :
      client(client0) {
      client.init(ts ...);
   }
   ~auto_t() {
      client.fini();
   }
   auto_t(auto_t const&);
   auto_t& operator= (auto_t const&);
   client_t& client;
};

// -----------------------------------------------------------------------------
// math;

#define M1_MINUINT64 (unsigned long long int)0xull                 // =                          0
#define M1_MAXUINT64 (unsigned long long int)0xffffffffffffffffull // = 18.446.744.073.709.551.615

#define M1_MININT64  (signed long long int)0x8000000000000000ll    // = -9.223.372.036.854.775.807
#define M1_MAXINT64  (signed long long int)0x7fffffffffffffffll    // =  9.223.372.036.854.775.807

#define M1_MINUINT32  (unsigned long int)0xul                      // =                          0
#define M1_MAXUINT32  (unsigned long int)0x7ffffffful              // =              4.294.967.295

#define M1_MININT32  (signed int)0x80000000l                       // =             -2.147.483.647
#define M1_MAXINT32  (signed int)0x7fffffffl                       // =              2.147.483.647

// min, max;
template<typename val_t>
static val_t const& min(val_t const& a, val_t const& b) {
   return a < b ? a : b;
}
#define M1_MIN(a, b)(((a) < (b)) ? (a) : (b))

template<typename val_t>
static val_t const& max(val_t const& a, val_t const& b) {
   return a < b ? b : a;
}
#define M1_MAX(a, b) ((a) < (b) ? (b) : (a))

// abs;
template<typename val_t>
static val_t abs(val_t const& val) {
   return (val_t)0 < val ? val : -val;
}
#define M1_ABS(a) (0 < (a) ? (a) : -(a))

// regula de tribus; régle de tri; rule of three;
// Dreisatz, Verhaeltnisgleichung, Proportionalitaet;
static int_t rdt(int_t const &a, int_t const &b, int_t const &c) {
  return c ? (a * b / c) : 0;
}

// signum;
template<typename val_t>
static int_t sign2(val_t const& val) {
   return (val < (val_t)0) ? -1 : 1;
}
template<typename val_t>
static int_t sign3(val_t const& val) {
   return (val < (val_t)0) ? -1 : (val > (val_t)0);
}
// fmod (remainder of int division; %); modf (separate fract. from int part);
template<typename val_t>
static val_t fmod(val_t const& x, val_t const& y) {
   return ::fmod(x, y);
}
template<typename val_t>
static val_t modf(val_t const& x, val_t* pint) {
   return ::modf(x, pint);
}

// round;
static int_t rndup(int_t const val, int_t const unit) {
   asrt(0 <= val && 0 < unit);
   return val + (unit - 1) - ((val - 1) % unit);
}
static int_t rnddwn(int_t const val, int_t const unit) {
   asrt(0 <= val && 0 < unit);
   return val - (val % unit);
}
#define M1_RNDUP8(x) (((x) +  7) & (~ 7))

static int_t rndup8(int_t const val) {
   asrt(0 <= val);
   return (val + 7) & ~7;
}
static int_t rnddwn8(int_t const val) {
   asrt(0 <= val);
   return val & ~7;
}
#define M1_RNDUP4(x) (((x) +  3) & (~ 3))

static int_t rndup4(int_t const val) {
   asrt(0 <= val);
   return (val + 3) & ~3;
}
static int_t rnddwn4(int_t const val) {
   asrt(0 <= val);
   return val & ~3;
}
// pow; sqrt;
template<typename base_t, typename exp_t>
static base_t pow(base_t const& base, exp_t const& exp0) {
   return ::pow(base, exp0);
   //uint_t Pow(uint_t const& base, uint_t const exp0) {
   //   if(0 == exp0) return 1;
   //   uint_t res = base;
   //   for(uint_t i = 1; i < exp0; ++i)
   //      res *=base;
   //   return res;
   //}
}
template<typename val_t>
static val_t sqrt(val_t const& val) {
   return ::sqrt(val);
}
// sin; asin;
template<typename val_t>
static val_t sin(val_t const& val) {
   return ::sin(val);
}
template<typename val_t>
static val_t asin(val_t const& val) {
   return ::asin(val);
}
// cos; acos;
template<typename val_t>
static val_t cos(val_t const& val) {
   return ::cos(val);
}
template<typename val_t>
static val_t acos(val_t const& val) {
   return ::acos(val);
}
// tan; atan; atan2;
template<typename val_t>
static val_t tan(val_t const& val) {
   return ::tan(val);
}
template<typename val_t> // atan(val) -> -pi/2 .. pi/2
static val_t atan(val_t const& val) {
   return ::atan(val);
}
template<typename val_t> // atan(y/x) -> -pi .. pi
static val_t atan2(val_t const& y, val_t const& x) {
   return ::atan2(y, x);
}
// floor; ceil;
template<typename val_t>
static val_t floor(val_t const& val) {
   return ::floor(val);
}
template<typename val_t>
static val_t ceil(val_t const& val) {
   return ::ceil(val);
}
// sinh; cosh; tanh;
template<typename val_t>
static val_t sinh(val_t const& val) {
   return ::sinh(val);
}
template<typename val_t>
static val_t cosh(val_t const& val) {
   return ::cosh(val);
}
template<typename val_t>
static val_t tanh(val_t const& val) {
   return ::tanh(val);
}
// exp; ldexp; frexp;
template<typename val_t> // e^x;
static val_t exp(val_t const& x) {
   return ::exp(x);
}
template<typename val_t> // x*2^exp;
static val_t ldexp(val_t const& x, int_t const& exp0) {
   return ::ldexp(x, exp0);
}
template<typename val_t> // base-2 int + res = x;
static val_t frexp(val_t const& x, int_t* pint) {
   return ::frexp(x, pint);
}
// log (natural); log10;
template<typename val_t> // e^res = x;
static val_t log(val_t const& x) {
   return ::log(x);
}
template<typename val_t> // 10^res = x;
static val_t log10(val_t const& x) {
   return ::log10(x);
}
//#if !defined(log2)
template<typename val_t> // 2^res = x;
static val_t log2(val_t const& x) {
   return log(x) / log(2);
}

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_LIB_NONE;

// -----------------------------------------------------------------------------
// time;

#if !(M1_TIME_NONE == M1_TIME)

// -----------------------------------------------------------------------------

#if (M1_TIME_LINUX == M1_TIME)
   #include <stdlib.h>
   #include <time.h>
   // link librt;
#endif

#if (M1_TIME_WIN == M1_TIME)
   #include <windows.h>
   #include <mmsystem.h> // timeGetTime;
   //#pragma comment(lib, "winmm.lib")
   #include <stdlib.h> // rand;
#endif

#if (M1_TIME_NACL == M1_TIME)
   #include "ppapi/cpp/module.h"
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------
// time & random;

struct timeval_t {
   timeval_t(int_t const sec0 = 0, int_t const usec0 = 0) :
      sec(sec0), usec(usec0) {
   }
   int_t sec, usec;
};

// -----------------------------------------------------------------------------

struct normal_t : single_t<normal_t> {

   // --------------------------------------------------------------------------

   friend struct single_t<normal_t>;

   // --------------------------------------------------------------------------
   // ctor dtor;

   normal_t() {
      tracec("normal:ctor");
      init();
   }

   ~normal_t() {
      fini();
      tracec("normal:dtor");
   }

   // --------------------------------------------------------------------------

   void_t init() {
      tracec("normal:init");
      // store process startup time;
      up = ticks();
      // default seed;
      seed(-1);
   }
   void_t fini() {
      tracec("normal:fini");
   }

   // --------------------------------------------------------------------------

   void_t seed(int_t const seed0 = -1) {
      #if (M1_TIME_LINUX == M1_TIME)
         // initialise generator;
         initstate(0, state, sizeof(state));
         setstate(state);
         // seeding;
         ::srandom(-1 == seed0 ? ticks() : seed0);
      #endif
      #if (M1_TIME_WIN == M1_TIME)
         //TODO;
      #endif
      #if (M1_TIME_NACL == M1_TIME)
         //TODO;
      #endif
   }

   // --------------------------------------------------------------------------

   int_t ticks() const { // milliseconds;
      #if (M1_TIME_LINUX == M1_TIME)
         timespec tp;
         int_t const res = clock_gettime(CLOCK_REALTIME, &tp);
         asrt(0 == res);
         int_t const  sec = tp.tv_sec;
         int_t const nsec = tp.tv_nsec;
         int_t const msec = 1000 * sec + nsec / 1000000;
         //tracec("ticks:sec:", sec, " nsec:", nsec, " msec:" , ms);
         return msec;
      #endif
      #if (M1_TIME_WIN == M1_TIME)
         return ::timeGetTime();
      #endif
      #if (M1_TIME_NACL == M1_TIME)
         pp::Module* module = pp::Module::Get();
         //TODO: check resolution;
         return (int_t)(PP_TimeTicks)module->core()->GetTimeTicks();
      #endif
   }

   // --------------------------------------------------------------------------

   int_t random() {
      #if (M1_TIME_LINUX == M1_TIME)
         return ::random();
      #endif
      #if (M1_TIME_WIN == M1_TIME)
         return (int_t)::rand();
      #endif
      #if (M1_TIME_NACL == M1_TIME)
          (int_t)::rand();
      #endif
   }

   // --------------------------------------------------------------------------

   timeval_t time() {
      #if (M1_TIME_LINUX == M1_TIME)
         //TODO;
         return timeval_t(); // seconds, microseconds;
      #endif
      #if (M1_TIME_WIN == M1_TIME)
         //TODO: timezone;
         FILETIME ft;
         ::GetSystemTimeAsFileTime(&ft);

         int_t tmp = 0;
         tmp |= ft.dwHighDateTime;
         tmp <<= 32;
         tmp |= ft.dwLowDateTime;

         static const int_t epoch_usecs = 11644473600000000ull;

         tmp -= epoch_usecs; // to unix epoch;
         tmp /= 10;  // to usecs;

         return timeval_t(tmp / 1000000ul, tmp % 1000000ul); // seconds, microseconds;
      #endif
      #if (M1_TIME_NACL == M1_TIME)
         //TODO;
         return timeval_t(0, 0);
      #endif
   }

   // --------------------------------------------------------------------------

   int_t runtime() const {
      return ticks() - up;
   }

   // --------------------------------------------------------------------------
   // data;

   int_t up;

   #if (M1_TIME_LINUX == M1_TIME)
      byte_t state[256]; // random generator array;
   #endif

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// instantiate ahead of client;

static normal_t &t = normal_t::inst();

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_TIME_NONE;

// -----------------------------------------------------------------------------
// cont;

#if !(M1_CONT_NONE == M1_CONT)

// -----------------------------------------------------------------------------

#if (M1_CONT_M1 == M1_CONT)

// -----------------------------------------------------------------------------

#include <stdlib.h>
#include <type_traits>

#if !defined(M1_FEAT_NONEW)
   #include <new>
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------

// lexicographical compare;
template<typename R0, typename R1>
static int_t compare_f(R0 r0, R1 r1) {
   for(r0.x = r0.a, r1.x = r1.a; r0.x && r1.x && *r0.x == *r1.x; ++r0, ++r1);
   if(0 == r0.x)
      if(0 == r1.x)   return   0; // equal;
      else            return  -1; // r1 longer;
   else if(0 == r1.x) return   1; // r0 longer;
   return (*r0.x < *r1.x) ? -1 : 1;
}
template<typename R0, typename R1>
static int_t compare_b(R0 r0, R1 r1) {
   for(r0.x = r0.z, r1.x = r1.z; r0.x && r1.x && *r0.x == *r1.x; --r0, --r1);
   if(0 == r0.x)
      if(0 == r1.x)   return   0; // equal;
      else            return  -1; // r1 longer;
   else if(0 == r1.x) return   1; // r0 longer;
   return (*r0.x < *r1.x) ? -1 : 1;
}

// deep copy;
template<typename R0, typename R1>
static int_t copy_f(R0 dst, R1 src) {
   int_t cnt = 0;
   for(dst.x = dst.a, src.x = src.a; dst.x && src.x; ++dst, ++src, ++cnt)
      new (dst.x) typename R0::obj_t(*src.x);
   return cnt;
}
template<typename R0, typename R1>
static int_t copy_b(R0 dst, R1 src) {
   int_t cnt = 0;
   for(dst.x = dst.z, src.x = src.z; dst.x && src.x; --dst, --src, ++cnt)
      new (dst.x) typename R0::obj_t(*src.x);
   return cnt;
}

// deep assign;
template<typename R0, typename R1>
static int_t assign_f(R0 dst, R1 src) {
   int_t cnt = 0;
   for(dst.x = dst.a, src.x = src.a; dst.x && src.x; ++dst, ++src, ++cnt)
      *dst.x = *src.x;
   return cnt;
}
template<typename R0, typename R1>
static int_t assign_b(R0 dst, R1 src) {
   int_t cnt = 0;
   for(dst.x = dst.z, src.x = src.z; dst.x && src.x; --dst, --src, ++cnt)
      *dst.x = *src.x;
   return cnt;
}

// create range of default objects in place;
template<typename R>
static void_t create(R dst) {
   //for(;dst.x; ++dst) new (dst.x) typename null_t<typename R::obj_t>::def();//typename R::obj_t();
   for(;dst.x; ++dst) {
      new (dst.x) typename R::obj_t;
      *dst.x = null_t<typename R::obj_t>::def();
   }
}
// create range of custom object copies in place;
template<typename R>
static void_t create(R dst, typename R::obj_t const& obj0) {
   //for(;dst.x; ++dst) new (dst.x) typename R::obj_t(obj0);
   for(;dst.x; ++dst) {
      new (dst.x) typename R::obj_t;
      *dst.x = obj0;
   }
}
// call dtor on range of objects in place;
template<typename R>
static void_t destroy(R dst) {
   for(; dst.x; ++dst) dst.x->R::obj_t::~obj_t();
}

// -----------------------------------------------------------------------------
// vector;

template<typename obj_T, byte_t deep = true>
struct vr_t
{
   // --------------------------------------------------------------------------

   typedef obj_T obj_t;

   // --------------------------------------------------------------------------

   vr_t() : a(0), x(0), z(0) {
   }
   vr_t(obj_t *a0, obj_t *z0, obj_t *x0 = 0) :
      a(a0), x(0), z(z0) {
      x = x0 ? (contains(x0) ? x0 : a) : a;
      if(empty()) x = 0;
   }
   vr_t(obj_t *a0, int_t const size0, obj_t *x0 = 0) :
      a(a0), x(0), z(a0 + size0 - 1) {
      x = x0 ? (contains(x0) ? x0 : a) : a;
   }
   ~vr_t() {
      z = x = a = 0;
   }

   // --------------------------------------------------------------------------

   vr_t range(obj_t *x0 = 0) const {
      return vr_t(a, z, x0);
   }
   vr_t range(obj_t *a0, obj_t *z0, obj_t *x0 = 0) const {
      if(contains(a0)) {
         if(contains(z0)) return vr_t(a0, z0, x0);
         else             return from(a0, x0);
      } else {
         if(contains(z0)) return upto(z0, x0);
         else             return range(x0);
      }
   }
   vr_t range(obj_t *a0, int_t const size0, obj_t *x0 = 0) const {
      return range(a0, a0 + size0, x0);
   }
   vr_t range(int_t const id0, int_t const size0, obj_t *x0 = 0) const {
      return range(a + id0, size0, x0);
   }

   // --------------------------------------------------------------------------

   vr_t from(obj_t *a0 = 0, obj_t *x0 = 0) const {
      if(0 == a0) a0 = x;
      return contains(a0) ? vr_t(a0, z, x0) : vr_t();
   }
   vr_t from(int_t const id0, obj_t *x0 = 0) const {
      return from(a + id0, x0);
   }

   vr_t upto(obj_t *z0 = 0, obj_t *x0 = 0) const {
      if(0 == z0) z0 = x;
      return contains(z0) ? vr_t(a, z0, x0) : vr_t();
   }
   vr_t upto(int_t const id0, obj_t *x0 = 0) const {
      return upto(a + id0, x0);
   }

   // --------------------------------------------------------------------------

   int_t empty() const {
      return 0 == z;
   }
   int_t size() const {
      return empty() ? 0 : ((z - a) + 1);
   }
   int_t id(obj_t *x0) const {
      return contains(x0) ? x0 - a : -1;
   }
   obj_t *at(int_t const id0 = 0) const {
      return contains(id0) ? a + id0 : 0;
   }

   // --------------------------------------------------------------------------
   // operators;

   obj_t& operator[] (int_t const id0) {
      return *at(id0);
   }

   vr_t& operator++ () { // prefix;
      x = (x && z && x != z) ? (x + 1) : 0;
      return *this;
   }
   vr_t operator++ (int) { // postfix;
      vr_t that = *this;
      this->operator++();
      return that;
   }
   vr_t& operator-- () { // prefix
      x = (x && z && x != a) ? (x - 1) : 0;
      return *this;
   }
   vr_t operator-- (int) { // postfix
      vr_t that = *this;
      this->operator--();
      return that;
   }

   // --------------------------------------------------------------------------

   obj_t *rewind() {
      return x = a;
   }

   // --------------------------------------------------------------------------

   obj_t *operator-> () {
      return x;
   }
   obj_t &operator* () {
      return *x;
   }

   // --------------------------------------------------------------------------
   // compare;

   template<typename ir_t>
   int_t compare(ir_t const &r0) const {
      if(deep) {
         return compare_f(*this, r0);
      } else {
         int_t const size0 = size();
         int_t const size1 = r0.size();
         if(0 == size0 && 0 == size1) return 0;
         if(size0 < size1) return -1;
         if(size1 < size0) return  1;
         return mcmp(a, r0.a, size0); // equal size > 0;
      }
   }
   int_t compare(obj_t* a0, obj_t* z0) const {
      return compare(vr_t(a0, z0));
   }

   int_t operator==(vr_t const &r0) const {
      return 0 == compare(r0);
   }
   int_t operator<(vr_t const& r0) const {
      return 0 > compare(r0);
   }

   int_t operator!=(vr_t const &r0) const {
      return !(*this == r0);
   }
   int_t operator>(vr_t const& r0) const {
      return r0 < *this;
   }
   int_t operator<=(vr_t const& r0) const {
      return !(r0 < *this);
   }
   int_t operator>=(vr_t const& r0) const {
      return !(*this < r0);
   }

   // --------------------------------------------------------------------------
   // contains;

   int_t contains(obj_t *x0) const {
      if(0 == x0 || empty()) return -1;
      return a <= x0 && x0 <= z;
   }
   int_t contains(int_t const id0) const {
      return contains(a + id0);
   }
   int_t contains(vr_t const &r0) const {
      return contains(r0.a) && contains(r0.z);
   }

   // --------------------------------------------------------------------------
   // create; destroy;

   void_t create(obj_t const& obj0 = null_t<obj_t>::def()) {
      if(deep) m1::create(range(), obj0);
      else mset(a, 0, size() * sizeof(obj_t));
   }
   void_t destroy() {
      if(deep) m1::destroy(range());
      else mset(a, 0, size() * sizeof(obj_t));
   }

   // --------------------------------------------------------------------------
   // copy; assign;

   void_t copy(vr_t const& src) {
      asrt(!empty() && !src.empty());
      asrt(src.size() <= size());
      if(a < src.a) {
         if(deep) m1::copy_f(range(), src.range());
         else mmov(a, src.a, src.size() * sizeof(obj_t));
      } else if(src.a < a)  {
         if(deep) m1::copy_b(range(), src.range());
         else mmov(a, src.a, src.size() * sizeof(obj_t));
      }
   }
   void_t assign(vr_t const& src) {
      asrt(!empty() && !src.empty());
      asrt(src.size() <= size());
      if(a < src.a) {
         if(deep) m1::assign_f(range(), src.range());
         else mmov(a, src.a, src.size() * sizeof(obj_t));
      } else if(src.a < a)  {
         if(deep) m1::assign_b(range(), src.range());
         else mmov(a, src.a, size() * sizeof(obj_t));
      }
   }

   // --------------------------------------------------------------------------
   // find;

   obj_t* find_f(obj_t const& obj, obj_t* x0 = 0) const {
      if(0 == x0) x0 = x;
      vr_t r = from(x0);
      for(; r.x && obj != *r.x; ++r);
      return r.x;
   }
   obj_t *find_f(obj_t const& obj, int_t const ofs) const {
      return find_f(obj, a + ofs);
   }

   obj_t* find_b(obj_t const& obj, obj_t* x0 = 0) const {
      if(0 == x0) x0 = z; //x;
      vr_t r = upto(x0, x0);
      for(; r.x && obj != *r.x; --r);
      return r.x;
   }
   obj_t *find_b(obj_t const& obj, int_t const ofs) const {
      return find_b(obj, a + ofs);
   }

   obj_t* find_f(vr_t const& pat, obj_t* x0 = 0) const {
      //asrt(!pat.empty() && !empty());
      x0 = find_f(*pat.a, x0); // find first obj;
      if(0 == x0) return 0;
      //if(!from(x0).contains(pat)) return 0; // break recursion;
      if(0 == pat.compare(x0, x0 + pat.size() - 1)) return x0; // match;
      else return find_f(pat, x0 + 1); // no match;
   }
   obj_t *find_f(vr_t const& pat, int_t const ofs) const {
      return find_f(pat, a + ofs);
   }

   obj_t* find_b(vr_t const& pat, obj_t* x0 = 0) const {
      //asrt(!pat.empty() && !empty());
      x0 = find_b(*pat.a, x0); // find first obj;
      if(0 == x0) return 0;
      //if(a == x0) return 0; // break recursion;
      if(0 == pat.compare(x0, x0 + pat.size() - 1)) return x0; // match;
      else return find_b(pat, x0 - 1); // no match;
   }
   obj_t *find_b(vr_t const& pat, int_t const ofs) const {
      return find_b(pat, a + ofs);
   }

   // --------------------------------------------------------------------------
   // count;

   int_t count_f(obj_t const& obj, obj_t* x0 = 0) const {
      x0 = find_f(obj, x0);
      return x0 ? count_f(obj, x0 + 1) + 1: 0;
   }

   obj_t* count_b(obj_t const& obj, obj_t* x0 = 0) const {
      x0 = find_b(obj, x0);
      return x0 ? count_b(obj, x0 - 1) + 1 : 0;
   }

   //TODO:count ir_t;
   //TODO: algos for all containers;

   // --------------------------------------------------------------------------
   // findof;

   obj_t* findof_f(vr_t const& pat, obj_t* x0 = 0) const {
      if(pat.empty() || empty()) return 0;
      for(vr_t r = from(x0); r.x; ++r)
         if(pat.find_f(*r.x)) return r.x;
      return 0;
   }
   obj_t* findof_f(vr_t const& pat, int_t const ofs) const {
      return findof_f(pat, a + ofs);
   }

   obj_t* findof_b(vr_t const& pat, obj_t* x0 = 0) const {
      if(pat.empty() || empty()) return 0;
      if(0 == x0) x0 = z;
      for(vr_t r = upto(x0, x0); r.x; --r)
         if(pat.find_f(*r.x)) return r.x;
      return 0;
   }
   obj_t *findof_b(vr_t const& pat, int_t const ofs) const {
      return findof_b(pat, a + ofs);
   }

   // --------------------------------------------------------------------------
   // findnof;

   obj_t *findnof_f(vr_t const& pat, obj_t* x0 = 0) const {
      if(pat.empty() || empty()) return 0;
      for(vr_t r = from(x0); r.x; ++r) {
         if(0 == pat.find_f(*r.x)) return r.x;
      }
      return 0;
   }
   obj_t *findnof_f(vr_t const& pat, int_t const ofs) const {
      return findnof_f(pat, a + ofs);
   }

   obj_t *findnof_b(vr_t const& pat, obj_t* x0 = 0) const {
      if(pat.empty() || empty()) return 0;
      if(0 == x0) x0 = z;
      for(vr_t r = upto(x0, x0); r.x; --r)
         if(0 == pat.find_f(*r.x)) return r.x;
      return 0;
   }
   obj_t *findnof_b(vr_t const& pat, int_t const ofs) const {
      return findnof_b(pat, a + ofs);
   }

   // --------------------------------------------------------------------------
   // replace;

   int_t replace_f(obj_t const& pat, obj_t const& subst, int_t const num = M1_MAXINT64, obj_t* x0 = 0) {
      if(num && (x0 = find_f(pat, x0))) {
         *x0 = subst;
         return replace_f(pat, subst, num - 1, x0 + 1);
      }
      return M1_MAXINT64 - num;
   }
   int_t replace_f(obj_t const& pat, obj_t const& subst, int_t const num, int_t const ofs) {
      return replace_f(pat, subst, num, a + ofs);
   }

   int_t replace_b(obj_t const& pat, obj_t const& subst, int_t const num = M1_MAXINT64, obj_t* x0 = 0) {
      if(num && (x0 = find_b(pat, x0))) {
         *x0 = subst;
         return replace_b(pat, subst, num - 1, x0 - 1);
      }
      return M1_MAXINT64 - num;
   }
   int_t replace_b(obj_t const& pat, obj_t const& subst, int_t const num, int_t const ofs) {
      return replace_b(pat, subst, num, a + ofs);
   }

   // --------------------------------------------------------------------------
   // order;

  //TODO: provide swap function;
   vr_t &rev() {
      for(int_t i = 0; i < (size() >> 1); ++i) {
         obj_t const tmp = *at(i);
         *at(i) = *at(size() - i - 1);
         *at(size() - i - 1) = tmp;
      }
      return *this;
   }

   // --------------------------------------------------------------------------

   void_t trace() {
      tracec("vr_t:size:", size(),
                       " \"", at(), "\"",
                       " a:", id(a), ":[", (void_t*)a, "]",
                       " x:", id(x), ":[", (void_t*)x, "]",
                       " z:", id(z), ":[", (void_t*)z, "]");
   }

   // --------------------------------------------------------------------------

   obj_t *a, *x, *z;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------

struct string_t; // forward;

// -----------------------------------------------------------------------------
// ctor disambiguation wrapper;

struct num_t {
   num_t(int_t const num0) : num(num0) {
   }
   operator int_t() const {
      return num;
   }
   int_t num;
};
// sim4hi: commented following function as it is not used
/*
static num_t number(int_t const num0) {
   return num_t(num0);
}
*/

// -----------------------------------------------------------------------------

template<typename obj_T, byte_t deep = true, typename mem_T = heap_t>
struct vector_t : public vr_t<obj_T, deep>
{
   // --------------------------------------------------------------------------

   typedef obj_T obj_t;
   typedef mem_T mem_t;
   typedef vr_t<obj_t, deep> r_t;

   //TODO: add cat & ext;

   // --------------------------------------------------------------------------
   // ctor - dtor;

   vector_t(mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("vector():%lld\n", heap_t::used());
      #endif
   }
   vector_t(num_t const size0, mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("vector0(%lld):%lld\n", size0, heap_t::used());
      #endif
      resize(size0);
   }
   vector_t(num_t const size0, obj_t const& obj0, mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("vector1(%lld):%lld\n", size0, heap_t::used());
      #endif
      resize(size0, obj0);
   }
   vector_t(r_t const& src, mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("vector(r):%lld\n", heap_t::used());
      #endif
      append(src);
   }
   ~vector_t() {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("~vector:%lld\n", heap_t::used());
      #endif
      clear();
   }

   // --------------------------------------------------------------------------
   // copy - assign;

   vector_t(vector_t const& v0) : mem(v0.mem) {
      reserve(v0.cap());
      if(v0.size()) {
         append(v0);
      }
   }
   vector_t const& operator= (vector_t const& v0) {
      if(this != &v0) {
         clear();
         reserve(v0.cap());
         if(v0.size()) {
            append(v0);
         }
      }
      return *this;
   }

   // --------------------------------------------------------------------------
   // reserve, resize;

   //TODO: reservaton strategy: frequent realloc vs. oversized prealloc;

   int_t reserve(int_t cap1, obj_t const& obj0 = null_t<obj_t>::def()) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("reserve:cap10:%lld\n", cap1);
      #endif
      cap1 = rndup8(cap1);
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("reserve:cap11:%lld\n", cap1);
      #endif
      if(cap1 <= cap()) return cap(); // do not shrink;
      obj_t *a1 = (obj_t*)mem.get(cap1 * sizeof(obj_t));
      asrt(a1!=NULL);
      if(false == r_t::empty()) {
         r_t r1(a1, a1 + r_t::size() - 1);
         r1.assign(r_t::range());
         r_t::destroy();
         r_t::z = r1.z;
      }
      if(r_t::a) mem.ret((byte_t*)r_t::a);
      r_t::a = r_t::x = a1;
      r_t r2(r_t::a + r_t::size(), cap() - r_t::size());
      r2.create(obj0);
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("reserve:exit\n");
      #endif
      return cap1;
   }
   int_t resize(int_t const size1, obj_t const& obj0 = null_t<obj_t>::def()) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("resize:size1:%lld\n", size1);
      #endif
      if(size1 == r_t::size()) return size1;
      int_t const res = reserve(size1, obj0);
      asrt(res);
      int_t const size0 = r_t::size();
      if(size1 < size0) // less;
         r_t::from(r_t::a + size1).destroy();
      if(size1) r_t::z = r_t::a + size1 - 1;
      else r_t::z = 0;
      //if(size0 < size1) // more;
         //r_t::from(r_t::a + size0).create(obj0); // done in reserve;
      return size1;
   }

   void_t clear() {
      if(!r_t::empty()) {
         #if defined(M1_FEAT_CONT_DEBUG)
         printf("vector:clear:range:size:%lld\n", r_t::range().size());
         #endif
         r_t r(r_t::a, cap());
         r.destroy();
         //r_t::range().destroy();
      }
      if(r_t::a) mem.ret((byte_t*)r_t::a);
      r_t::a = r_t::x = r_t::z = 0;
   }

   int_t cap() const {
      if(r_t::a) {
         int_t const bytes = *(((int_t*)r_t::a) - 1);
         asrt(0 == bytes % sizeof(obj_t));
         int_t const cap0 = bytes / sizeof(obj_t);
         return cap0;
      } else return 0;
   }

   // --------------------------------------------------------------------------
   // insert;

   // one;
   obj_t *insert(obj_t const& obj0, int_t const id0) {
      int_t const size0 = r_t::size();
      if(-1 == resize(max(size0, id0) + 1)) return 0;
      obj_t *dst = r_t::a + id0;
      if(id0 < size0) {
         r_t r0(dst,     dst +     size0 - id0);
         r_t r1(dst + 1, dst + 1 + size0 - id0);
         copy_b(r1, r0);
      }
      *dst = obj0;
      return dst;
   }
   obj_t *insert(obj_t const& obj0, obj_t* dst = 0) {
      int_t id0 = dst ? r_t::id(dst) : 0;
      return insert(obj0, id0);
   }

   // range;
   //template<typename ir_t>
   obj_t *insert(r_t const &src, int_t const id0) {
      //asrt(!src.empty());
      if(src.empty()) return r_t::at(id0);
      int_t const size0 = r_t::size();
      if(-1 == resize(max(size0, id0) + src.size())) return 0;
      obj_t *dst = r_t::a + id0;
      if(id0 < size0) {
         r_t r0(dst,              dst +              size0 - id0);
         r_t r1(dst + src.size(), dst + src.size() + size0 - id0);
         copy_b(r1, r0);
      }
      r_t(dst, src.size()).assign(src);
      return dst;
   }
   //template<typename ir_t>
   obj_t *insert(r_t const &src, obj_t* dst = 0) {
      int_t const id0 = dst ? r_t::id(dst) : 0;
      return insert(src, id0);
   }

   // --------------------------------------------------------------------------
   // append;

   // one;
   obj_t *append(obj_t const& obj0, int_t const id0) {
      return insert(obj0, id0 + 1);
   }
   obj_t *append(obj_t const& obj0, obj_t* dst = 0) {
      int_t const id0 = dst ? r_t::id(dst) : (r_t::size() - 1);
      return append(obj0, id0);
   }

   // range;
   obj_t *append(r_t const &src, int_t const id0) {
      return insert(src, id0 + 1);
   }
   obj_t *append(r_t const &src, obj_t* dst = 0) {
      int_t const id0 = dst ? r_t::id(dst) : (r_t::size() - 1);
      return append(src, id0);
   }

   // --------------------------------------------------------------------------
   // remove;

   // remove one @ adr;
   obj_t *remove(obj_t* dst) {
      asrt(r_t::contains(dst));
      if(deep) dst->~obj_t();
      if(dst < r_t::z)
         r_t::from(dst).assign(r_t::from(dst + 1));
      resize(r_t::size() - 1);
      return dst;
   }
   obj_t *remove(int_t const id0) {
      return remove(r_t::at(id0));
   }

   // remove range;
   obj_t *remove(r_t const &dst) {
      asrt(r_t::contains(dst));
      m1::destroy(dst);
      if(dst.z < r_t::z)
         r_t::from(dst.a).assign(r_t::from(dst.z + 1));
      resize(r_t::size() - dst.size());
      return dst.a;
   }
   obj_t *remove(obj_t* dst, int_t const size0) {
      return remove(r_t(dst, size0));
   }
   obj_t *remove(int_t const id0, int_t const size0) {
      return remove(r_t(r_t::at(id0), size0));
   }

   // --------------------------------------------------------------------------
   // access;

   obj_t &operator[] (int_t const id0) {
      if(id0 < r_t::size()) return *r_t::at(id0);
      else return *insert(null_t<obj_t>::def(), id0);
   }

   // --------------------------------------------------------------------------
   // replace range;

   template<typename ir_t>
   void_t replace_f(r_t const& pat, ir_t const& subst, int_t const num = M1_MAXINT64, obj_t* src = 0) {
      if(num && (src = r_t::find_f(pat, src))) {
         src = remove(src, pat.size());
         insert(subst, src);
         return replace_f(pat, subst, num - 1, src + subst.size());
      }
   }
   template<typename ir_t>
   void_t replace_f(r_t const& pat, ir_t const& subst, int_t const num, int_t const ofs) {
      replace_f(pat, subst, num, r_t::a + ofs);
   }

   template<typename ir_t>
   void_t replace_b(r_t const& pat, ir_t const& subst, int_t const num = M1_MAXINT64, obj_t* src = 0) {
      if(num && (src = r_t::find_b(pat, src))) {
         src = remove(src, pat.size());
         insert(subst, src);
         return replace_b(pat, subst, num - 1, src - subst.size());
      }
   }
   template<typename ir_t>
   void_t replace_b(r_t const& pat, ir_t const& subst, int_t const num, int_t const ofs) {
      replace_b(pat, subst, num, r_t::a + ofs);
   }

   // --------------------------------------------------------------------------

   void_t trace() {
      m1::tracef("m1:vector:size:", r_t::size(), " cap:", cap(), ":[");
      for(r_t r = r_t::range(); r.x; ++r) {
         m1::tracef(*r, r.x == r.z ? "" : ",");
      }
      tracec(']');
   }

   // --------------------------------------------------------------------------

   mem_t& mem;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// string;

struct string_t : public vector_t<byte_t, false>
{
   // --------------------------------------------------------------------------
   // ctor - dtor;

   //TODO: allow custom heap (see vector_t ctors);

   string_t() : fmt(txt) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("string:%lld\n", heap_t::used());
      #endif
      reserve(1ll, '\0');
   }
   string_t(num_t const size0, byte_t const fill0 = '\0') : fmt(txt) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("string0(%lld):%lld\n", size0, heap_t::used());
      #endif
      reserve(size0 + 1, '\0');
      //resize(size0, fill0);
      for(int_t i = 0; i < size0; ++i) {
         vector_t::append(fill0);
      }
   }
   string_t(r_t const& src) : fmt(txt) {
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("string(r):%lld\n", heap_t::used());
      #endif
      vector_t::append(src);
   }
   //string_t(char const* src0) : fmt(txt) {
   //   #if defined(M1_FEAT_CONT_DEBUG)
   //   printf("string(c*):%lld\n", heap_t::used());
   //   #endif
   //   if(0 == src0) {
   //      reserve(1ll, '\0');
   //   } else {
   //      int_t const size0 = slen((byte_t*)src0);
   //      reserve(size0 + 1, '\0');
   //      vector_t::append(r_t((byte_t*)src0, size0));
   //   }
   //}
   //string_t(char *src0) : fmt(txt) {
   //   #if defined(M1_FEAT_CONT_DEBUG)
   //   printf("string(*):%lld\n", heap_t::used());
   //   #endif
   //   if(0 == src0) {
   //      reserve(1, '\0');
   //   } else {
   //      int_t const size0 = slen((byte_t*)src0);
   //      reserve(size0 + 1, '\0');
   //      vector_t::append(r_t((byte_t*)src0, size0));
   //   }
   //}
   //string_t(int const size0, byte_t const fill0 /*= '\0'*/) : fmt(txt) {
   //   #if defined(M1_FEAT_CONT_DEBUG)
   //   printf("string1(%lld):%lld\n", heap_t::used());
   //   #endif
   //   reserve((int_t)size0 + 1, '\0');
   //   //resize((int_t)size0, fill0);
   //   for(int_t i = 0; i < size0; ++i) {
   //      vector_t::append(fill0);
   //   }
   //}
   template<typename ... Ts>
   string_t(Ts ... ts) : fmt(txt) { // variadic ctor;
      #if defined(M1_FEAT_CONT_DEBUG)
      printf("string(...):%lld\n", heap_t::used());
      #endif
      cat(ts ...);
   }
   ~string_t() {
      clear();
   }

   // --------------------------------------------------------------------------
   // copy - assign;

   string_t(string_t const& s0) {
      mem = s0.mem;
      fmt = s0.fmt;
      reserve(s0.cap());
      if(s0.size()) {
         vector_t::append(s0);
      }
   }
   string_t const& operator= (string_t const& s0) {
      if(this != &s0) {
         clear();
         fmt = s0.fmt;
         reserve(s0.cap());
         if(s0.size()) {
            vector_t::append(s0);
         }
      }
      return *this;
   }
   //string_t const& operator= (char const* src0) {
   //   if(a != &src0[0]) {
   //      clear();
   //      if(0 == src0) return *this;
   //      fmt = txt;
   //      int_t const size0 = slen((byte_t*)src0);
   //      reserve(size0 + 1, '\0');
   //      vector_t::append(r_t((byte_t*)src0, size0));
   //   }
   //   return *this;
   //}

   // --------------------------------------------------------------------------
   // insert;

   obj_t* insert(char const* src, int_t const id0) {
      return vector_t::insert(string_t(src), id0);
   }
   obj_t* insert(char const* src, obj_t* dst = 0) {
      return vector_t::insert(string_t(src), dst);
   }

   obj_t* insert(string_t const& src, int_t const id0) {
      return vector_t::insert(src, id0);
   }
   obj_t* insert(string_t const& src, obj_t* dst = 0) {
      return vector_t::insert(src, dst);
  }

   // --------------------------------------------------------------------------
   // append;

   obj_t* append(char const* src, int_t const id0) {
      return vector_t::append(string_t(src), id0);
   }
   obj_t* append(char const* src, obj_t* dst = 0) {
      return vector_t::append(string_t(src), dst);
   }

   obj_t* append(string_t const& src, int_t const id0) {
      return vector_t::append(src, id0);
   }
   obj_t* append(string_t const& src, obj_t* dst = 0) {
      return vector_t::append(src, dst);
   }

   // --------------------------------------------------------------------------
   // find;

   obj_t* find_f(string_t const& pat, obj_t* x0 = 0) const {
      return vr_t::find_f(pat.range(), x0);
   }
   obj_t *find_f(string_t const& pat, int_t const ofs) const {
      return vr_t::find_f(pat.range(), ofs);
   }

   obj_t* find_b(string_t const& pat, obj_t* x0 = 0) const {
      return vr_t::find_b(pat.range(), x0);
   }
   obj_t *find_b(string_t const& pat, int_t const ofs) const {
      return vr_t::find_b(pat.range(), ofs);
   }

   // --------------------------------------------------------------------------
   // findof;

   obj_t* findof_f(string_t const& pat, obj_t* x0 = 0) const {
      return vr_t::findof_f(pat.range(), x0);
   }
   obj_t* findof_f(string_t const& pat, int_t const ofs) const {
      return vr_t::findof_f(pat.range(), ofs);
   }

   obj_t* findof_b(string_t const& pat, obj_t* x0 = 0) const {
      return vr_t::findof_b(pat.range(), x0);
   }
   obj_t* findof_b(string_t const& pat, int_t const ofs) const {
      return vr_t::findof_b(pat.range(), ofs);
   }

   // --------------------------------------------------------------------------
   // replace (vr_t);

   int_t replace_f(obj_t const& pat, obj_t const& subst, int_t const num = M1_MAXINT64, obj_t* x0 = 0) {
      return vr_t::replace_f(pat, subst, num, x0);
   }
   int_t replace_f(obj_t const& pat, obj_t const& subst, int_t const num, int_t const ofs) {
      return vr_t::replace_f(pat, subst, num, ofs);
   }

   int_t replace_b(obj_t const& pat, obj_t const& subst, int_t const num = M1_MAXINT64, obj_t* x0 = 0) {
      return vr_t::replace_b(pat, subst, num, x0);
   }
   int_t replace_b(obj_t const& pat, obj_t const& subst, int_t const num, int_t const ofs) {
      return vr_t::replace_b(pat, subst, num, ofs);
   }

   // --------------------------------------------------------------------------
   // replace (vector_t);

   void_t replace_f(char const* pat, char const* subst, int_t const num = M1_MAXINT64, obj_t* src = 0) {
      return vector_t::replace_f(string_t(pat), string_t(subst), num, src);
   }
   void_t replace_f(char const* pat, char const* subst, int_t const num, int_t const ofs) {
      return vector_t::replace_f(string_t(pat), string_t(subst), num, ofs);
   }

   void_t replace_b(char const* pat, char const* subst, int_t const num = M1_MAXINT64, obj_t* src = 0) {
      return vector_t::replace_b(string_t(pat), string_t(subst), num, src);
   }
   void_t replace_b(char const* pat, char const* subst, int_t const num, int_t const ofs) {
      return vector_t::replace_b(string_t(pat), string_t(subst), num, ofs);
   }

   // --------------------------------------------------------------------------
   // conversion;

   string_t& lower(r_t r = r_t()) {
      if(r.empty()) r = range();
      for(; r.x; ++r) *r.x = (byte_t)tlow(*r.x);
      return *this;
   }
   string_t& upper(r_t r = r_t()) {
      if(!empty()) r = range();
      for(; r.x; ++r) *r.x = (byte_t)toup(*r.x);
      return *this;
   }

   string_t& pad(int_t const len, byte_t const pat = ' ') {
      if(abs(len) <= size()) {
         return *this;
      }
      if(len < -1) { // pad left;
         insert(string_t(num_t(-len - size()), pat));
      }
      if(len > 1) { // pad right;
         append(string_t(num_t( len - size()), pat));
      }
      return *this;
   }
   string_t& normalize() {
      vector_t::replace_f(string_t("\\\\"), string_t("/"));
      vr_t::replace_f('\\', '/');
      return *this;
   }
   string_t& trim(string_t const pat = string_t(" \v\t")) {
      // \t  0x09 horizontal tab
      // \n  0x0a line feed - new line
      // \v  0x0b vertical tab
      // \f  0x0c form feed - new page
      // \r  0x0d carriage return
      //TODO: check upto with int arg;
      if(empty()) return *this;
      int_t const firstid = id(findnof_f(pat));
      if(0      < firstid) vector_t::remove(upto(firstid - 1));
      int_t const lastid  = id(findnof_b(pat));
      if(lastid < id(z)  ) vector_t::remove(from(lastid  + 1));
      return *this;
   }
   string_t compact(string_t const pat = string_t(" ")) {
      // remove repetitions;
      if(empty()) return *this;
      //TODO;
      return *this;
   }

   // --------------------------------------------------------------------------
   // path;

   r_t path() const {
      byte_t* x0 = findof_b(string_t("/\\:"), z);
      return x0 && a < x0 ? r_t(a, x0 - 1) : range();
   }
   r_t name() const {
      byte_t *x0 = findof_b(string_t("/\\:"), z);
      return x0 && x0 < z ? r_t(x0 + 1, z) : range();
   }
   r_t ext() const {
      byte_t *x0 = name().find_b((byte_t)'.', z);
      return x0 && x0 < z ? r_t(x0 + 1, z) : r_t();
   }

   // --------------------------------------------------------------------------
   // inspect;

   int_t atoi() const {
      string_t cp = *this;
      //cp.trim();
      if('-' == cp[0]) cp.remove(0ll);
      return cp.isuint() ? (int_t)::atoi(a) : -1;
      //uint_t res = 0;
      //uint_t const maxid = str.size() - 1;
      //for(uint_t i = maxid; 0xffffffff != i; --i)
      //   res += (str[i] - '0') * Pow(10, maxid - i);
      //return res;
   }
   int_t strtol() const {
      return (int_t)::strtol(a, 0, 0);
   }

   // --------------------------------------------------------------------------

   int_t isuint() const {
      // maximum unsigned integer: 4.294.967.295 (10 digits);
      int_t const size0 = size();
      if(10 < size0 || 0 == size0) return false;
      if(size0 < 10) {
         //check digits;
         for(int_t i = 0; i < size0; ++i) {
            if(*at(i) < '0' || '9' < *at(i)) return false;
         }
         return true;
      }
      // size0 is 10;
      string_t maxuint("4294967295");
      for(int_t i = 0; i < 10; ++i) {
         if(maxuint[i] < *at(i)) return false;
         if(*at(i) < maxuint[i]) return true;
      }
      return true;
   }

   // --------------------------------------------------------------------------
   // SI prefix;

   // 8 byte; 64 bit; int_t; signed;
   // [-9.223.372.036.854.775.808 ~ 0x8000000000000000,
   //   9.223.372.036.854.775.807 ~ 0x7fffffffffffffff];

   string_t dec_prefix(int_t const val, int_t const variant = 0) {
      int_t const total = abs(val);
    //if(1000000000000000000000000 <= total) return 0 == variant ? "Y" : "Yotta"; //exceeds 64bit;
    //if(   1000000000000000000000 <= total) return 0 == variant ? "Z" : "Zetta"; //exceeds 64bit;
      if(      1000000000000000000 <= total) return 0 == variant ? "E" :   "Exa";
      if(         1000000000000000 <= total) return 0 == variant ? "P" :  "Peta";
      if(            1000000000000 <= total) return 0 == variant ? "T" :  "Tera";
      if(               1000000000 <= total) return 0 == variant ? "G" :  "Giga";
      if(                  1000000 <= total) return 0 == variant ? "M" :  "Mega";
      if(                     1000 <= total) return 0 == variant ? "k" :  "kilo";
      if(                      100 <= total) return 0 == variant ? "h" : "hekto";
      if(                       10 <= total) return 0 == variant ? "d" :  "deka";
   }
   string_t bin_prefix(int_t const val, int_t const variant = 0) {
      int_t const total = abs(val);
    //if(1ll << 80 <= total) return 0 == variant ? "Yi" : "Yobi"; //exceeds 64bit;
    //if(1ll << 70 <= total) return 0 == variant ? "Zi" : "Zebi"; //exceeds 64bit;
      if(1ll << 60 <= total) return 0 == variant ? "Ei" : "Exbi";
      if(1ll << 50 <= total) return 0 == variant ? "Pi" : "Pebi";
      if(1ll << 40 <= total) return 0 == variant ? "Ti" : "Tebi";
      if(1ll << 30 <= total) return 0 == variant ? "Gi" : "Gibi";
      if(1ll << 20 <= total) return 0 == variant ? "Mi" : "Mebi";
      if(1ll << 10 <= total) return 0 == variant ? "Ki" : "Kibi"; // 1024;
   }

   // --------------------------------------------------------------------------
   // concatenate pod types;

   string_t &operator<<(long int const val) {
      return *this << (int_t)val;
   }
   string_t &operator<<(int const val) {
      return *this << (int_t)val;
   }
   string_t &operator<<(unsigned int const val) {
      return *this << (int_t)val;
   }

   string_t &operator<<(char const *val) {
      return *this << (byte_t*)val;
   }

   // --------------------------------------------------------------------------
   // concatenate m1 types;

   string_t &operator<<(byte_t const val) {
      byte_t buf[8] = { 0 };
      int_t len = 0;
      switch(fmt) {
         case txt:
         default:
            len = sprnt(buf, sizeof(buf), "%c", val);
            break;
         case hed: {
            len = sprnt(buf, sizeof(buf), "%02x.", val);
            break;
         }
         case bin: {
            for(int_t i = 0; i < 8; ++i) {
               buf[7 - i] = (val & (1 << i)) ? '1' : '0';
            }
            len = 8;
            break;
         }
         case raw: {
            buf[0] = val;
            len = 1;
            break;
         }
      }
      vector_t::append(r_t(buf, buf + len - 1));
      return *this;
   }
   string_t &operator<<(int_t val) {
      switch(fmt) {
         case txt:
         default: {
            byte_t buf[32] = { 0 };
            int_t const len = sprnt(buf, sizeof(buf), "%lld", val);
            vector_t::append(r_t(buf, buf + len - 1));
            break;
         }
         case hed: {
            byte_t buf[32] = { 0 };
            //int_t const len = sprnt(buf, sizeof(buf), "%016llx", val);
            int_t const len = sprnt(buf, sizeof(buf), "%llx", val);
            vector_t::append(r_t(buf, buf + len - 1));
            break;
         }
         case bin: {
            byte_t buf[64] = { 0 };
            //int_t ofs = -1;
            for(int_t i = 0; i < 64; ++i) {
               buf[i] = ((val >> (63 -i)) & 1) ? '1' : '0';
               //if(-1 == ofs && buf[i]) ofs = i;
            }
            //vector_t::append(r_t(buf, buf + ofs));
            vector_t::append(r_t(buf, buf + sizeof(buf) - 1));
            break;
         }
         case raw: {
            byte_t buf[8] = { 0 };
            //int_t ofs = -1;
            for(int_t i = 0; i < 8;  ++i) {
               buf[i] = ((byte_t)(val >> (8 * (7 - i)))) & 0xff;
               //printf("buf:%lld:%d\n", i, buf[i]);
               //if(-1 == ofs && buf[i]) ofs = i;
            }
            //vector_t::append(r_t(buf, buf + ofs));
            vector_t::append(r_t(buf, buf + sizeof(buf) - 1));
            break;
         }
      }
      return *this;
   }
   string_t &operator<<(flt_t val) {
      byte_t buf[64] = { 0 };
      int_t const len = sprnt(buf, 64, "%g", val);
      vector_t::append(r_t(buf, buf + len - 1));
      return *this;
   }
   string_t &operator<<(byte_t* val) {
      vector_t::append(r_t(val, val + slen(val) - 1));
      return *this;
   }
   template<typename T>
   string_t &operator<<(T *val) {
      byte_t buf[64] = { 0 };
      int_t const len = sprnt(buf, 64, "%p", val);
      vector_t::append(r_t(buf, buf + len - 1));
      return *this;
   }
   string_t &operator<<(string_t const& val) {
      switch(fmt) {
         case txt:
         case hed:
         default: {
            if(raw == val.fmt) {
               for(auto r = val.range(); r.x; ++r) {
                  cat(hed, *r.x, txt);
               }
            } else {
               vector_t::append(val.range());
            }
            break;
         }
         case bin:
         case raw: {
            vector_t::append(val.range());
            break;
         }
      }
      return *this;
   }

   // --------------------------------------------------------------------------
   // state;

   string_t& operator<< (fmt_e const& fmt0) {
      fmt = fmt0;
      return *this;
   }
   string_t& operator<< (num_t const& size) {
      resize(size);
      return *this;
   }

   template<typename T>
   typename std::enable_if<std::is_enum<T>::value, string_t &>::type
   operator<< (T const &val) {
      return *this << (int_t)val;
   }

   // --------------------------------------------------------------------------
   // concatenate variadic types;

   template<typename T0>
   string_t &cat(T0 const& t0) {
      return *this << t0; // terminate;

   }
   template<typename T0, typename ... Ts>
   string_t &cat(T0 const& t0, Ts ... ts) {
      *this << t0;
      return cat(ts ...); // recurse;
   }

   // --------------------------------------------------------------------------
   // extract pod types;

   string_t &operator>>(int &val) {
      int_t val1;
      *this >> val1;
      if(val1 < 0) { // check overflow;
         asrt(0 == val1 / ((int_t)M1_MININT32 - 1));
      } else {
         asrt(0 == val1 / ((int_t)M1_MAXINT32 + 1));
      }
      val = val1;
      return *this;
   }
   string_t &operator>>(unsigned int &val) {
      int_t val1;
      *this >> val1;
      asrt(0 == val1 / ((int_t)M1_MAXUINT32 + 1));
      val = val1;
      return *this;
   }

   // --------------------------------------------------------------------------
   // extract m1 types;

   string_t &operator>>(byte_t &val) {
      switch(fmt) {
         case txt:
            val = *x;
            ++x;
            break;
         case hed: {
            asrt(0); //TODO;
            break;
         }
         default: asrt(0);
      }
      return *this;
   }
   static byte_t &sep() {
      return null_t<byte_t>::def();
   }

   string_t &operator>>(int_t &val) {
      switch(fmt) {
         case txt:
            if(NULL != x)
            {
                val = strtoll(x, &x, 10);
            }
            break;
         case hed:
            if(NULL != x)
            {
                val = strtoll(x, &x, 16);
            }
            break;
         default: asrt(0);
       }
       return *this;
   }
   string_t &operator>>(flt_t &val) {
      val = strtod(x, &x);
      return *this;
   }
   template<typename T>
   string_t &operator>>(T *&val) {
      val = (T*)strtoll(x, &x, 16);
      return *this;
   }
   string_t &operator>>(string_t &val) {
      val = r_t(x, slen(x) - 1);
      return *this;
   }

   // --------------------------------------------------------------------------
   // extract variadic types;

   template<typename T0>
   string_t &ext(T0 &t0) {
      return *this >> t0; // terminate;
   }
   template<typename T0, typename ... Ts>
   string_t &ext(T0 &t0, Ts &... ts) {
      *this >> t0;
      return ext(ts ...); // recurse;
   }

   // --------------------------------------------------------------------------

   string_t str(int_t const var = 0) const {
      switch(var) {
         case 0:
         default: {
            return string_t("size:", size(),
                            " cap:", cap(),
                            " fmt:", fmt,
                              " \"", at(), "\"",
                              " a:", id(a), ":[", (void_t*)a, "]",
                              " x:", id(x), ":[", (void_t*)x, "]",
                              " z:", id(z), ":[", (void_t*)z, "]");
         }
         case 1: {
            string_t s(hed);
            for(int_t i = 0; i < size(); ++i) {
              if('\0' == *at(i)) {
                 s << '.' << " ";
              } else {
                 s << (int_t)*at(i) << " ";
              }
           }
           return s;
         }
      };
   }

   fmt_e fmt;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// convenience;

template<typename T>
static string_t pad(T const &t, int_t const size, byte_t const pat = ' ',
   fmt_e const fmt = txt) {
   return string_t(fmt, t).pad(size, pat);
}
template<typename T>
static string_t trim(T const &t, string_t const pat = string_t(" \v\t"),
   fmt_e const fmt = txt) {
   return string_t(fmt, t).trim(pat);
}

// -----------------------------------------------------------------------------

typedef vector_t<string_t> strings_t;

// -----------------------------------------------------------------------------
// bits;

#define XENUM3(X) \
   X(inv,  = -1, "inv")\
   X(low,  =  0, "low")\
   X(high, =  1, "high")
   XENUM3_IMPL(bit_e)
#undef XENUM3

#define XENUM3(X) \
   X(GET,  = 0, "get")\
   X(SET,  = 1, "set")\
   X(AND,  = 2, "and")\
   X(OR,   = 3, "or")\
   X(XOR,  = 4, "xor")\
   X(INV,  = 5, "inv")\
   X(CLR,  = 6, "clr")
   XENUM3_IMPL(op_e)
#undef XENUM3

//------------------------------------------------------------------------------
// one bit;

struct bit_t {
   typedef bit_e val_t;
   static val_t const err = inv;
   bit_t(int_t const pos0, op_e const op0 = GET, bit_e const val0 = inv) :
       pos(pos0), op(op0), val(val0) {
   }
   void_t set(int_t const pos0, op_e const op0 = GET, bit_e const val0 = inv) {
      pos = pos0;
      op = op0;
      bit_e val;
   }
   string_t str() const {
      return string_t("pos:", hed, pos,
                      " op:", txt, estr2(op),
                     " val:", txt, estr2(val));
   }
   int_t pos; // bit position (right to left);
   op_e op;
   bit_e val; // high = 1; low == 0;
};

//------------------------------------------------------------------------------
// upto 64 bits;

struct bits_t {
   typedef int_t val_t;
   static val_t const err = -1;
   bits_t(int_t const num0, int_t const pos0, op_e const op0 = GET,
      int_t const val0 = 0) :
      num(num0), pos(pos0), op(op0), val(val0) {
   }
   void_t set(int_t const num0, int_t const pos0, op_e const op0 = GET,
      int_t const val0 = 0) {
      num = num0;
      pos = pos0;
      op = op0;
      bit_e val;
   }
   string_t str() const {
      return string_t("num:", hed, num,
                txt, " pos:", pos,
                      " op:", txt, estr2(op),
                     " val:", val);
   }
   int_t num; // number of bits;
   int_t pos; // bit position (right to left);
   op_e op;
   int_t val;
};

//------------------------------------------------------------------------------
// register;

struct reg_t : public string_t {

   template<typename ...Ts> // delegate to base ctors;
   reg_t(Ts ...ts) : string_t(raw, ts...) {
   }

   // --------------------------------------------------------------------------
   // catch string literals containing zeros; e.g. reg_t("\x00\xff", 2);

   reg_t(byte_t const* val, int const size) : string_t(raw) {
      for(int_t i = 0; i < size; ++i) {
         cat(*(val + i));
      }
   }

   // --------------------------------------------------------------------------
   // modify bit;

   // one;
   bit_e mod(bit_t &bit) {
      asrt(0 <= bit.pos && (bit.pos >> 3) < size());
      // byte and bit numbering in reverse order;
      int_t const id  = size() - 1 - (bit.pos >> 3);
      int_t const pos = bit.pos % 8;
      byte_t &byte = *at(id);
      switch(bit.op) {
         case SET:
            //byte &= ~(      1 << pos); // clear;
            //byte |=  (bit.val << pos); // set;
            switch(bit.val) {
               case high:
                  byte |=  (byte_t)(1 << pos);
                  break;
               case low:
                  byte &=  (byte_t)(~(1 << pos));
                  break;
               case inv:
                  break;
            }
            break;
         case AND:
            byte &=  (byte_t)(bit.val << pos);
            break;
         case OR:
            byte |=  (byte_t)(bit.val << pos);
            break;
         case XOR:
            byte ^=  (byte_t)(bit.val << pos);
            break;
         case INV:
            byte ^=  (byte_t)(1 << pos);
            break;
         case CLR:
            byte &=  (byte_t)(~(1 << pos));
            break;
         case GET:
            break;
      }
      bit_e const val = ((byte >> pos) & 1) ? high : low;
      if(GET == bit.op) {
         bit.val = val;
      }
      return val;
   }
   bit_e mod(bit_t const &bit) { // catch rvalues;
      bit_t bit1 = bit;
      return mod(bit1);
   }

   // multiple non adjacent;
   template<typename ...Ts>
   void_t mod(bit_t const &bit, Ts const &...ts) {
      mod(bit);
      mod(ts...);
   }

   // --------------------------------------------------------------------------
   // modify bits;

   // group of upto 64 adjacent;
   int_t mod(bits_t &bits) {
      vr_t r = subset(bits);
      int_t sub = cast64(r); // cast to 64 bit;
      switch(bits.op) {
         case SET: // clear & set;
            sub &= ~(           ((1 << bits.num) - 1)  << bits.pos);
            sub |=  (bits.val & ((1 << bits.num) - 1)) << bits.pos;
            break;
         case AND:
            sub &=  (bits.val & ((1 << bits.num) - 1)) << bits.pos;
            break;
         case OR:
            sub |=  (bits.val & ((1 << bits.num) - 1)) << bits.pos;
            break;
         case XOR:
            sub ^=  (bits.val & ((1 << bits.num) - 1)) << bits.pos;
            break;
         case INV:
            sub ^=  (((1 << bits.num) - 1) << bits.pos);
            break;
         case CLR:
            sub &= ~(((1 << bits.num) - 1) << bits.pos);
            break;
         case GET:
            break;
      }
      reg_t reg;
      for(int_t i = 0; i < r.size(); ++i) {
         reg.insert((byte_t)((sub >> (8 * i)) & 0xff));
      }
      r.assign(reg);
      int_t const val = (sub >> (bits.pos % 8)) & ((1 << bits.num) - 1);
      if(GET == bits.op) {
         bits.val = val;
      }
      return val;
   }
   int_t mod(bits_t const &bits) { // catch rvalues;
      bits_t bits1 = bits;
      return mod(bits1);
   }

   // multiple groups of adjacent;
   template<typename ...Ts>
   void_t mod(bits_t const &bits, Ts const &...ts) {
      mod(bits);
      mod(ts...);
   }

   // --------------------------------------------------------------------------
   // cast to equivalent pod types;

   operator byte_t() const {
      asrt(1 == size());
      return *at(0);
   }
   operator int_t() const {
      asrt(8 >= size());
      int_t val = 0;
      for(int_t i = 0; i < size(); ++i) {
         val += ((int_t)*at(i)) << ((size() - i - 1) << 3);
      }
      return val;
   }

   byte_t cast8(vr_t const &r) const {
      asrt(1 == r.size());
      return *r.at(0);
   }
   int_t cast64(vr_t const &r) const {
      asrt(8 >= r.size());
      int_t val = 0;
      for(int_t i = 0; i < r.size(); ++i) {
         val += ((int_t)*r.at(i)) << ((r.size() - i - 1) << 3);
      }
      return val;
   }

   // --------------------------------------------------------------------------
   // extract bytes containing upto 64 adjacent bits;

   r_t subset(bits_t const &bits) const {
      asrt(0 < bits.num && 0 <= bits.pos);
      int_t const base = size() - 1;
      int_t const id0 = base - (bits.pos >> 3);
      int_t const id1 = base - ((bits.pos + bits.num - 1) >> 3);
      return range(id1, id0 - id1); // checks containship;
   }

   // --------------------------------------------------------------------------

   int_t narrow(int_t const arch) { // strip some leading zeros;
      int_t leads = 0;
      for(int_t i = 0; i < size(); ++i) {
         if(*at(i)) {
            leads = i;
            break;
         }
      }
      leads = leads ? size() - rndup(size() - leads, arch >> 3) - 1 : 0;
      if(leads) remove(upto(leads));
      return (arch >> 3) >= size() ? 0 : -1;
   }

   // -------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// list;

template<typename obj_T>
struct lr_t
{
   // --------------------------------------------------------------------------

   typedef obj_T obj_t;

   struct node_t {

      // -----------------------------------------------------------------------
      // ctor dtor;

      node_t() : prev(0), next(0) {
      }
      node_t(obj_t const& obj0) :
         obj(obj0), prev(0), next(0) {
      }

   //       // -----------------------------------------------------------------------
      // compare;

      int_t operator==(node_t const &rhs) const {
         return obj == rhs.obj;
      }
      int_t operator<(node_t const& rhs) const {
         return obj < rhs.obj;
      }

      int_t operator!=(node_t const &rhs) const {
         return !(*this == rhs);
      }
      int_t operator>(node_t const& rhs) const {
         return rhs < *this;
      }
      int_t operator<=(node_t const& rhs) const {
         return !(rhs < *this);
      }
      int_t operator>=(node_t const& rhs) const {
         return !(*this < rhs);
      }

      // -----------------------------------------------------------------------
      // data;

      obj_t obj;
      node_t *prev, *next;
   };

   // --------------------------------------------------------------------------
   // ctor dtor;

   lr_t() : a(0), x(0), z(0), size_(-1) {
   }
   lr_t(node_t *a0, node_t *z0, node_t *x0 = 0) :
      a(a0), x(0), z(z0), size_(-1) {
      x = x0 ? (find_f(x0) ? x0 : a) : a;
   }
   ~lr_t() {
   }

   // --------------------------------------------------------------------------

   lr_t range(node_t *x0 = 0) const {
      return lr_t(a, z, x0);
   }
   lr_t range(node_t *a0, node_t *z0, node_t *x0 = 0) const {
      if(find_f(a0)) {
         if(find_b(z0)) return lr_t(a0, z0, x0);
         else           return from(a0, x0);
      } else {
         if(find_b(z0)) return upto(z0, x0);
         else           return range(x0);
      }
   }

   // --------------------------------------------------------------------------

   lr_t from(node_t *a0, node_t *x0 = 0) const {
      return find_f(a0) ? lr_t(a0, z, x0) : lr_t();
   }

   lr_t upto(node_t *z0, node_t *x0 = 0) const {
      return find_b(z0) ? lr_t(a, z0, x0) : lr_t();
   }

   // --------------------------------------------------------------------------

   int_t empty() const {
      return 0 == a || 0 == z;
   }
   int_t size() const {
      if(-1 == size_) {
          size_ = 0;
          for(lr_t r = range(); r.x; ++r) ++size_;
      }
      return size_;
   }

   // --------------------------------------------------------------------------
   // operators;

   lr_t& operator++ () { // prefix;
      x = x && x != z ? x->next : 0;
      return *this;
   }
   lr_t operator++ (int) { // postfix;
      lr_t that = *this;
      this->operator++();
      return that;
   }
   lr_t& operator-- () { // prefix;
      x = x && x != a ? x->prev : 0;
      return *this;
   }
   lr_t operator-- (int) { // postfix;
      lr_t that = *this;
      this->operator--();
      return that;
   }

   node_t *operator-> () {
      return x;
   }
   obj_t &operator* () {
      return x->obj;
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t compare(lr_t const &r0) const {
      return compare_f(*this, r0);
   }
   int_t compare(node_t* a0, node_t* z0) const {
      return compare(lr_t(a0, z0));
   }

   template<typename ir_t>
   int_t operator==(ir_t const &r0) const {
      return 0 == compare(r0);
   }
   template<typename ir_t>
   int_t operator<(ir_t const& r0) const {
      return 0 > compare(r0);
   }

   template<typename ir_t>
   int_t operator!=(ir_t const &r0) const {
      return !(*this == r0);
   }
   template<typename ir_t>
   int_t operator>(ir_t const& r0) const {
      return r0 < *this;
   }
   template<typename ir_t>
   int_t operator<=(ir_t const& r0) const {
      return !(r0 < *this);
   }
   template<typename ir_t>
   int_t operator>=(ir_t const& r0) const {
      return !(*this < r0);
   }

   // --------------------------------------------------------------------------
   // create; destroy;

   void_t create(obj_t const& obj0 = null_t<obj_t>::def()) {
      m1::create(range(), obj0);
   }
   void_t destroy() {
      m1::destroy(range());
   }

   // --------------------------------------------------------------------------
   // copy; assign;

   //template<typename ir_t>
   void_t copy(lr_t const& src) {
      asrt(!empty() && !src.empty());
      asrt(src.size() <= size());
      if(find_f(src.z)) m1::copy_b(*this, src);
      else              m1::copy_f(*this, src);
   }
   //template<typename ir_t>
   void_t assign(lr_t const& src) {
      asrt(src.size() <= size());
      if(find_f(src.z)) m1::assign_b(*this, src);
      else              m1::assign_f(*this, src);
   }

   // --------------------------------------------------------------------------
   // find;

   node_t* find_f(node_t *x0) const { // node
     if(z == x0) return z;
     lr_t r = range();
     for(; r.x && r.x != z; ++r);
     return r.x;
   }
   node_t* find_b(node_t *x0) const {
     if(a == x0) return a;
     lr_t r = range(z);
     for(; r.x && r.x != a; --r);
     return r.x;
   }

   node_t* find_f(obj_t const& obj) const { // object;
      lr_t r = range();
      for(; r.x && obj != *r.x; ++r);
      return r.x;
   }
   node_t* find_b(obj_t const& obj) const {
      lr_t r = range(z);
      for(; r.x && obj != *r.x; --r);
      return r.x;
   }

   // --------------------------------------------------------------------------
   // findof;

   template<typename ir_t>
   obj_t* findof_f(ir_t const& pat, obj_t* x0 = 0) const {
      asrt(!pat.empty() && !empty());
      for(lr_t r = from(x0); r.x; ++r)
         if(x0 = pat.find_f(*r.x)) return x0;
      return 0;
   }

   template<typename ir_t>
   obj_t* findof_b(ir_t const& pat, obj_t* x0 = 0) const {
      asrt(!pat.empty() && !empty());
      for(lr_t r = upto(x0, x0); r.x; --r)
         if(x0 = pat.find_f(*r.x)) return x0;
      return 0;
   }

   // --------------------------------------------------------------------------
   // replace;

   int_t replace_f(obj_t const& pat, obj_t const& subst, obj_t* x0 = 0, int_t num = M1_MAXINT64) {
      for(lr_t r = from(x0); num && r.x; ++r, --num)
         if(pat == *r.x) *r.x = subst;
      return M1_MAXINT64 - num;
   }

   int_t replace_b(obj_t const& pat, obj_t const& subst, obj_t* x0 = 0, int_t num = M1_MAXINT64) {
      for(lr_t r = upto(x0, x0); num && r.x; --r, --num)
         if(pat == *r.x) *r.x = subst;
      return M1_MAXINT64 - num;
   }

   // --------------------------------------------------------------------------

   node_t *a, *x, *z;
   int_t mutable size_;

   // --------------------------------------------------------------------------
};

template<typename obj_T, typename mem_T = heap_t>
struct list_t : public lr_t<obj_T>
{
   // --------------------------------------------------------------------------

   typedef obj_T obj_t;
   typedef mem_T mem_t;
   typedef lr_t<obj_T> r_t;

   typedef typename r_t::node_t node_t;

   // --------------------------------------------------------------------------
   // ctor - dtor;

   list_t(mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
   }
   list_t(int_t const size0, mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      for(int_t i = 0; i < size0; ++i)
         append(obj_t());
   }
   list_t(int_t const size0, obj_t const& obj0, mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      for(int_t i = 0; i < size0; ++i)
         append(obj0);
   }
   ~list_t() {
      clear();
   }

   // --------------------------------------------------------------------------
   // copy - assign;

   list_t(list_t const& l0) : mem(l0.mem) {
      r_t::copy(l0.range());
   }
   list_t& operator= (list_t const& l0) {
      if(this != &l0) {
         clear();
         mem = l0.mem;
         r_t::assign(l0.range());
      }
      return *this;
   }

   // --------------------------------------------------------------------------

   void_t resize(int_t const size0, obj_t const& obj0 = null_t<obj_t>::def()) {
      int_t const diff = size0 - r_t::size();
      if(0 < diff) { // more;
         for(int_t i = 0; i < diff; ++i) {
            append(obj0);
         }
      } else if(0 > diff) { // less;
         for(int_t i = 0; i < -diff; ++i) {
            remove(r_t::z);
         }
      }
   }

   // --------------------------------------------------------------------------

   void_t clear() {
      while(0 < r_t::size_) remove(r_t::a);
   }

   // --------------------------------------------------------------------------
   // insert;

   // one;
   node_t* insert(obj_t const& obj0, node_t* node = 0) {
      if(0 == node) node = r_t::a;
      node_t *p = (node_t*)mem.get(sizeof(node_t));
      asrt(NULL!=p);
      new (p) node_t(obj0);
      if(r_t::size()) {
         p->prev = node->prev;
         p->next = node;
         if(node->prev)
            node->prev->next = p;
         node->prev = p;
         if(r_t::a == node) r_t::a = p;
         if(r_t::x == node) r_t::x = p;
      } else {
         r_t::a = r_t::x = r_t::z = p;
      }
      ++r_t::size_;
      return p;
   }

   // range;
   //template<typename ir_t>
   node_t *insert(r_t const &r0, node_t* node = 0) {
      asrt(!r0.empty());
      asrt(find_f(node));
      if(0 == node) node = r_t::a;
      for(r_t r = r0.range(r0.z); r.x; --r)
         node = insert(r.x->obj, node);
      return node;
   }

   // --------------------------------------------------------------------------
   // append;

   // one;
   node_t* append(obj_t const& obj0, node_t* node = 0) {
      if(0 == node) node = r_t::z;
      node_t *p = (node_t*)mem.get(sizeof(node_t));
      asrt(NULL!=p);
      new (p) node_t(obj0);
      if(r_t::size()) {
         p->next = node->next;
         p->prev = node;
         if(node->next)
            node->next->prev = p;
         node->next = p;
         if(r_t::z == node) r_t::z = p;
         if(r_t::x == node) r_t::x = p;
      } else {
         r_t::a = r_t::x = r_t::z = p;
      }
      ++r_t::size_;
      return p;
   }
   // range;
   //template<typename ir_t>
   obj_t *append(r_t const &r0, node_t* node = 0) {
      asrt(!r0.empty());
      asrt(NULL!=find_f(node));
      if(0 == node) node = r_t::z;
      for(r_t r = r0.range(r0.z); r.x; --r)
         node = append(r.x->obj, node);
      return node;
   }

   // --------------------------------------------------------------------------
   // remove;

   // one;
   node_t * remove(node_t* node) {
      asrt(node && r_t::size());
      node_t *node1 = (r_t::z == node) ? node->prev : node->next;
      if(r_t::x == node) r_t::x = node1;
      if(r_t::z == node) r_t::z = node->prev;
      else               node->next->prev = node->prev;
      if(r_t::a == node) r_t::a = node->next;
      else               node->prev->next = node->next;
      node->~node_t();
      mem.ret((byte_t*)node);
      --r_t::size_;
      return node1;
   }

   // range;
   node_t *remove(r_t const &r0) {
      asrt(!r0.empty());
      node_t *node = r0.a;
      asrt(find_f(node));
      for(int_t i = 0; i < r0.size(); ++i)
         node = remove(node);
      return node;
   }
   node_t *remove(node_t* node, int_t const size0) {
      asrt(NULL!=find_f(node));
      for(int_t i = 0; i < size0; ++i)
         node = remove(node);
      return node;
   }

   // --------------------------------------------------------------------------

   string_t str(int_t const var = 0) const {
      switch(var) {
         case 0:
         default: {
            return string_t("size:", r_t::size(),
                              " a:", (void_t*)r_t::a,
                              " x:", (void_t*)r_t::x,
                              " z:", (void_t*)r_t::z);
         }
         case 1: {
            string_t s;
            for(auto r = r_t::range(); r.x; ++r) {
               s.cat(r.x->obj, ' ');
            }
            return s;
         }
      };
   }

   // --------------------------------------------------------------------------

   void_t trace() {
      tracef("m1:list:size:", r_t::size(), " [");
      for(r_t r = r_t::range(); r.x; ++r) {
         m1::tracef(*r, r.x == r.z ? "" : ",");
      }
      tracec(']');
   }

   // --------------------------------------------------------------------------

   mem_t&  mem;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// tree;

struct treenode_t {

   // --------------------------------------------------------------------------
   // ctor dtor;

   treenode_t() : hl(0), hr(0),
      left(0), right(0), parent(0) {
   }
   ~treenode_t() {
      left = right = parent = 0;
   }

   // --------------------------------------------------------------------------

   int_t scale() const {
      return hr - hl;
   }
   int_t heigth() const {
      return max(hl, hr) + 1;
   }
   string_t str() const {
      return string_t("l:", (void_t*)left, " r:", (void_t*)right, " p:", (void_t*)parent);
   }

   // --------------------------------------------------------------------------
   // data;

   int_t hl, hr;
   treenode_t *left, *right, *parent;
};

// -----------------------------------------------------------------------------

template<typename node_T>
struct tr_t
{
   // --------------------------------------------------------------------------

   typedef node_T node_t;
   typedef typename node_t::key_t key_t;

   // --------------------------------------------------------------------------
   // ctor dtor;

   tr_t() : root(0), a(0), x(0), z(0), size_(-1) {
   }
   tr_t(node_t *root0, node_t *a0, node_t *z0, node_t *x0 = 0) :
      root(root0), a(a0), x(0), z(z0), size_(-1) {
         x = x0 ? (find(x0) ? x0 : a0) : a0;
   }

   // --------------------------------------------------------------------------
   // copy assign;

   //template<typename ir_t>
   void_t copy(tr_t const& src) {
      asrt(!empty() && !src.empty());
      asrt(src.size() <= size());
      if(find(src.z)) m1::copy_b(*this, src);
      else            m1::copy_f(*this, src);
   }
   //template<typename ir_t>
   void_t assign(tr_t const& src) {
      asrt(src.size() <= size());
      if(find(src.z)) m1::assign_b(*this, src);
      else            m1::assign_f(*this, src);
   }

   // --------------------------------------------------------------------------

   tr_t range(node_t *x0 = 0) const {
      return tr_t(root, a, z, x0);
   }
   tr_t range(node_t *a0, node_t *z0, node_t *x0 = 0) const {
      if(find(a0->key)) {
         if(find(z0->key)) return tr_t(root, a0, z0, x0);
         else              return from(a0, x0);
      } else {
         if(find(z0->key)) return upto(z0, x0);
         else              return range(x0);
      }
   }

   tr_t base(node_t *root0, node_t *x0 = 0) const {
      if(0 == root0) return tr_t();
      if(root == root0) return range(x0);
      return tr_t(root0, minimum(root0), maximum(root0), x0);
   }

   // --------------------------------------------------------------------------

   tr_t from(node_t *a0, node_t *x0 = 0) const {
      if(0 == a0 || 0 == find(a0)) return tr_t();
      return tr_t(root, a0, z, x0);
   }
   tr_t upto(node_t *z0, node_t *x0 = 0) const {
      if(0 == z0 || 0 == find(z0)) return tr_t();
      return tr_t(root, a, z0, x0);
   }

   // --------------------------------------------------------------------------

   int_t empty() const {
      return 0 == a || 0 == z;
   }
   int_t size() const {
      if(-1 == size_) {
         size_ = 0;
         for(tr_t r = range(); r.x; ++r) ++size_;
      }
      return size_;
   }

   // --------------------------------------------------------------------------
   // operators;

   tr_t& operator++ () { // prefix;
      if(0 == x || z == x) x = 0;
      else x = x->right ? minimum((node_t*)x->right) : top(x);
      return *this;
   }
   tr_t operator++ (int) { // postfix;
      tr_t that = *this;
      this->operator++();
      return that;
   }

   tr_t& operator-- () { // prefix;
      if(0 == x || a == x) x = 0;
      else x = x->left ? maximum((node_t*)x->left) : sub(x);
      return *this;
   }
   tr_t operator-- (int) { // postfix;
      tr_t that = *this;
      this->operator--();
      return that;
   }

   node_t *operator-> () {
      return x;
   }
   key_t &operator* () {
      return x->key;
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t compare(tr_t const &r0) const {
      return compare_f(*this, r0);
   }

   template<typename ir_t>
   int_t operator==(ir_t const &r0) const {
      return 0 == compare(r0);
   }
   template<typename ir_t>
   int_t operator<(ir_t const& r0) const {
      return 0 > compare(r0);
   }

   template<typename ir_t>
   int_t operator!=(ir_t const &r0) const {
      return !(*this == r0);
   }
   template<typename ir_t>
   int_t operator>(ir_t const& r0) const {
      return *this < r0;
   }
   template<typename ir_t>
   int_t operator<=(ir_t const& r0) const {
      return !(r0 < *this);
   }
   template<typename ir_t>
   int_t operator>=(ir_t const& r0) const {
      return !(*this < r0);
   }

   // --------------------------------------------------------------------------
   // find;

   node_t *find(key_t const& key0) const {
     node_t *node = root;
     while(node) {
        if(key0 == node->key) return node;
        node = key0 < node->key ? (node_t*)node->left : (node_t*)node->right;
     }
     return 0;
   }
   node_t *find(node_t *node0) const {
      return find(node0->key);
   }
   node_t *lower(key_t const& key0) const {
      node_t *node = root;
      while(node) {
         node_t *node1 = key0 < node->key ? (node_t*)node->left : (node_t*)node->right;
         if(0 == node1) {
            if(node->key <= key0) return node;
            else return node->left ? maximum((node_t*)node->left) : sub(node);
         } else node = node1;
      }
      return 0;
   }
   node_t *upper(key_t const& key0) const {
      node_t *node = root;
      while(node) {
         node_t* node1 = key0 < node->key ? (node_t*)node->left : (node_t*)node->right;
         if(0 == node1) {
            if(key0 < node->key) return node;
            else return node->right ? minimum((node_t*)node->right) : top(node);
         } else node = node1;
      }
      return 0;
   }

   // --------------------------------------------------------------------------

   node_t *minimum(node_t *node) const {
      while(node->left) node = (node_t*)node->left;
      return node;
   }
   node_t *maximum(node_t *node) const {
      while(node->right) node = (node_t*)node->right;
      return node;
   }
   node_t* top(node_t* node) const {
      if(0 == node->parent) return 0;
      return (((node_t*)node->parent)->key < node->key) ? top((node_t*)node->parent) : (node_t*)node->parent;
   }
   node_t* sub(node_t* node) const {
      if(0 == node->parent) return 0;
      return (node->key < ((node_t*)node->parent)->key) ? sub((node_t*)node->parent) : (node_t*)node->parent;
   }

   // --------------------------------------------------------------------------

   void_t trace() {
      if(empty()) {
         tracec("tr:t:empty");
      } else {
         tracec("tr_t:size:", size());
         tracec("tr_t:r:", root->str());
         tracec("tr_t:a:", a->str());
         tracec("tr_t:x:", x->str());
         tracec("tr_t:z:", z->str());
      }
   }

   // --------------------------------------------------------------------------

   node_t *root, *a, *x, *z;
   int_t mutable size_;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------

template<typename node_T, typename mem_T = heap_t>
struct tree_t : public tr_t<node_T>
{
   // --------------------------------------------------------------------------

   typedef node_T node_t;
   typedef mem_T mem_t;
   typedef tr_t<node_t> r_t;
   typedef typename r_t::key_t key_t;

   // --------------------------------------------------------------------------

   tree_t(mem_t& mem0 = null_t<mem_t>::def()) :
      mem(mem0) {
      r_t::size_ = 0;
   }
   ~tree_t() {
      clear();
   }

   // --------------------------------------------------------------------------
   // copy - assign;

   tree_t(tree_t const& t0) :
      mem(t0.mem) {
      r_t::copy(t0.range());
   }
    tree_t& operator= (tree_t const& t0) {
      if(this != &t0) {
         clear();
         mem = t0.mem;
         r_t::assign(t0.range());
      }
      return *this;
   }

   // --------------------------------------------------------------------------

   void_t clear() {
      remove(r_t::a);
   }

   // --------------------------------------------------------------------------

   node_t *insert(key_t const& key0, int_t& exists = null_t<int_t>::def()) {
      exists = false;
      if(0 == r_t::root) {
         return r_t::a = r_t::x = r_t::z = r_t::root = create(key0, 0);
      }
      node_t *node = r_t::root;
      while(node) {
         if(key0 < node->key) {
            if(0 == node->left) {
               node_t *left = create(key0, node);
               if(0 == left) return 0;
               node->left = left;
               if(key0 < r_t::a->key)
                  r_t::a = r_t::x = (node_t*)node->left;
               node = fix((node_t*)node->left, 1);
               break;
            } else node = (node_t*)node->left;
         } else if(key0 > node->key) {
            if(0 == node->right) {
               node_t *right = create(key0, node);
               if(0 == right) return 0;
               node->right = right;
               if(key0 > r_t::z->key)
                  r_t::z = (node_t*)node->right;
               node = fix((node_t*)node->right, 1);
               break;
            } else node = (node_t*)node->right;
         } else { // key0 == node->key;
            exists = true;
            break;
         }
      }
      //check();
      return node;
   }
   node_t *append(key_t const& key0, int_t& exists = null_t<int_t>::def()) {
      return insert(key0, exists); // equivalent to insert;
   }

   // --------------------------------------------------------------------------

   int_t remove(node_t *node) {
      if(0 == node) return false;
      node_t *succ = 0;
      if(node->left) {
         if(node->right) {
            int_t const var = ((int_t)node / sizeof(void_t*)) % 2;
            if(var) {
               succ = r_t::minimum((node_t*)node->right);
               if(succ == r_t::z) r_t::z = node;
            } else {
               succ = r_t::maximum((node_t*)node->left);
               if(succ == r_t::a) r_t::a = r_t::x = node;
            }
            node->key = succ->key;
            subst(succ, var ? (node_t*)succ->right : (node_t*)succ->left);
         } else {
            if(node == r_t::z) r_t::z = r_t::maximum((node_t*)node->left);
            subst(node, (node_t*)node->left);
         }
      } else if(node->right) {
         if(node == r_t::a) r_t::a = r_t::x = r_t::minimum((node_t*)node->right);
         subst(node, (node_t*)node->right);
      } else {
         if(node == r_t::root) {
            destroy(r_t::root);
            r_t::a = r_t::x = r_t::z = r_t::root = 0;
         } else {
            if(node == r_t::a) r_t::a = r_t::x = (node_t*)node->parent;
            if(node == r_t::z) r_t::z = (node_t*)node->parent;
            subst(node, 0);
         }
      }
      //check();
      return true;
   }
   int_t remove(key_t const& key0) {
      node_t *node = r_t::find(key0);
      if(0 == node) return false;
      else return remove(node);
   }

   // --------------------------------------------------------------------------

   int_t depth(node_t *node, node_t *root0 = 0) const {
      root0 = root0 ? root0 : r_t::root;
      int_t depth0 = 0;
      while(node != root0 && (node = (node_t*)node->parent)) ++depth0;
      return depth0;
   }
   int_t heigth(node_t *node) const {
      if(0 == node) return 0;
      int_t heigth0 = 0;
      for(r_t r = range(node); r.x; ++r) {
         int_t const depth0 = depth(r.x, node);
         if(depth0 > heigth0) heigth0 = depth0;
      }
      return heigth0;
   }

   // --------------------------------------------------------------------------

   node_t *fix(node_t *node, int_t const add) {
      for(node_t *n = node; n->parent;) {
         if(n == n->parent->right)
              n->parent->hr += add;
         else n->parent->hl += add;
         n = (node_t*)n->parent;
         int_t const scale = n->scale();
         if(add > 0 ? !scale : scale) break;
         if(2 == scale) {
            node_t *core = (node_t*)n->right;
            if(-1 == core->scale()) {
               core->hl = core->left->right ?
                          core->left->right->heigth() : 0;
               rotright(core);
               n->right->hr =   core ? core->heigth() : 0;
            }
            n = rotleft(n);
            core = (node_t*)n->left;
            core->hr = core->right ?
                       core->right->heigth() : 0;
            n->hl = core->heigth();
            break;
         } else if(-2 == scale) {
            node_t *core = (node_t*)n->left;
            if(1 == core->scale()) {
               core->hr = core->right->left ?
                          core->right->left->heigth() : 0;
               rotleft(core);
               n->left->hl =    core ? core->heigth() : 0;
            }
            n = rotright(n);
            core = (node_t*)n->right;
            core->hl = core->left ?
                       core->left->heigth() : 0;
            n->hr = core->heigth();
            break;
         }
      }
      return node;
   }
   node_t *rotleft(node_t* h) {
      node_t *x = (node_t*)h->right;
      h->right = x->left;
      if(h->right)
         h->right->parent = h;
      x->left = h;
      x->parent = h->parent;
      if(x->parent) {
         if(h == h->parent->left)
              h->parent->left  = x;
         else h->parent->right = x;
      }
      if(r_t::root == h) r_t::root = x;
      return (node_t*)(h->parent = x);
   }
   node_t *rotright(node_t* h) {
      node_t *x = (node_t*)h->left;
      h->left = x->right;
      if(h->left)
         h->left->parent = h;
      x->right = h;
      x->parent = h->parent;
      if(x->parent) {
         if(h == h->parent->left)
              h->parent->left  = x;
         else h->parent->right = x;
      }
      if(r_t::root == h) r_t::root = x;
      return (node_t*)(h->parent = x);
   }
   void_t subst(node_t *node, node_t *sub) {
      if(r_t::root == fix(node, -1)) r_t::root = sub;
      else if(node == node->parent->left)
         node->parent->left = sub;
      else node->parent->right = sub;
      if(sub) sub->parent = node->parent;
      destroy(node);
   }

   // --------------------------------------------------------------------------

   node_t *create(key_t const& key0, node_t *parent0) {
      node_t *node = (node_t*)mem.get(sizeof(node_t));
      asrt(node!=NULL);
      new (node) node_t(key0);
      node->parent = parent0;
      ++r_t::size_;
      return node;
   }
   void_t destroy(node_t *node) {
      node->~node_t();
      mem.ret((byte_t*)node);
      --r_t::size_;
   }

   // --------------------------------------------------------------------------

   int_t check() const {
      for(r_t r = r_t::range(); r.x; ++r) {
         int_t const hleft  = r.x->left  ? heigth(r.x->left)  + 1 : 0;
         int_t const hright = r.x->right ? heigth(r.x->right) + 1 : 0;
         if(r.x->hl != hleft || r.x->hr != hright) {
            string_t s;
            s << "check:unbalanced:key:" << r.x->key
              << " hl:" << r.x->hl << " hleft:"  << hleft
              << " hr:" << r.x->hr << " hright:" << hright;
            tracec(&s[0]);
            return false;
         }
      }
      return true;
   }
   void_t dotrace(node_t *node, int_t const var = 2) const {
      if(0 == node) {
         tracec("[]");
         return;
      }
      vector_t<string_t> p;
      int_t x = 0;
      for(r_t r = range(node); r.x; ++r) {
         //for(int_t i = 0; i < p.size(); ++i) {
         //   tracec(&p[i][0]);
         //}
         string_t val;
         switch(var) {
            default:
            case 0: val << '[' << r.x->key     << ']'; break;
            case 1: val << '[' << r.x->key     << ":"
                               << r.x->scale() << ']'; break;
            case 2: val << '[' << r.x->key     << ":"
                               << r.x->hl      << ":"
                               << r.x->hr      << ']'; break;
         }
         int_t const y = depth(r.x, node);
         if(r.x->left) {
            int_t const x1 = p[y + 1].id(p[y + 1].find_f(']')) + 1;
            p[y] << string_t(    x1 - p[y].size(), ' ');
            p[y] << string_t(x - x1              , '_');
         } else
            p[y] << string_t(x - p[y].size(), ' ');
         p[y] << val;
         if(y && r.x->parent && (r.x == r.x->parent->right))
            p[y - 1] << string_t(x - p[y - 1].size(), '_');
         x += val.size();
      }
      for(int_t i = 0; i < p.size(); ++i) {
         tracec("%s", &p[i][0]);
      }
   }
   void_t dotrace() const {
      return dotrace(r_t::root);
   }

   // --------------------------------------------------------------------------

   mem_t&  mem;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// set;

template<typename key_T>
struct setnode_t : public treenode_t {

   typedef key_T key_t;

   // --------------------------------------------------------------------------
   // ctor dtor

   setnode_t(key_t const& key0) : key(key0) {
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t operator==(setnode_t const &rhs) const {
      return key == rhs.key;
   }
   int_t operator<(setnode_t const& rhs) const {
      return key < rhs.key;
   }

   int_t operator!=(setnode_t const &rhs) const {
      return !(*this == rhs);
   }
   int_t operator>(setnode_t const& rhs) const {
      return rhs < *this;
   }
   int_t operator<=(setnode_t const& rhs) const {
      return !(rhs < *this);
   }
   int_t operator>=(setnode_t const& rhs) const {
      return !(*this < rhs);
   }

   // --------------------------------------------------------------------------
   // data;

   key_t key;
};

// -----------------------------------------------------------------------------

template<typename key_T, typename mem_T = heap_t>
struct set_t : public tree_t<setnode_t<key_T>, mem_T>
{
   typedef mem_T mem_t;
   typedef key_T key_t;
   typedef key_t obj_t;
   typedef tree_t<setnode_t<key_T>, mem_t> c_t;

   typedef typename c_t::node_t node_t;
   typedef typename c_t::r_t r_t;

   set_t(mem_t& mem0 = null_t<mem_t>::def()) :
      c_t(mem0) {
   }
};

// -----------------------------------------------------------------------------
// multiset;

template<typename key_T, typename mem_T = heap_t>
struct multiset_t : public tree_t<setnode_t<key_T>, mem_T>
{
   typedef mem_T mem_t;
   typedef key_T key_t;
   typedef key_t obj_t;
   typedef tree_t<setnode_t<key_T>, mem_t> c_t;

   typedef typename c_t::node_t node_t;
   typedef typename c_t::r_t r_t;

   multiset_t(mem_t& mem0 = null_t<mem_t>::def()) :
      c_t(mem0) {
   }

   // --------------------------------------------------------------------------

   node_t *insert(key_t const& key0, int_t& exists = null_t<int_t>::def()) {
      exists = false;
      if(0 == r_t::root)
         return r_t::a = r_t::x = r_t::z = r_t::root = c_t::create(key0, 0);
      node_t *node = r_t::root;
      while(node) {
         if(key0 <= node->key) {
            if(0 == node->left) {
               node_t *left = c_t::create(key0, (node_t*)node);
               if(0 == left) return left;
               node->left = left;
               if(key0 <= r_t::a->key)
                  r_t::a = r_t::x = (node_t*)node->left;
               node = c_t::fix((node_t*)node->left, 1);
               break;
            } else node = (node_t*)node->left;
         } else if(key0 > node->key) {
            if(0 == node->right) {
               node_t *right = c_t::create(key0, (node_t*)node);
               if(0 == right) return 0;
               node->right = right;
               if(key0 > r_t::z->key)
                  r_t::z = (node_t*)node->right;
               node = c_t::fix((node_t*)node->right, 1);
               break;
            } else node = (node_t*)node->right;
         }
      }
      //check();
      return node;
   }
   node_t *append(key_t const& key0, int_t& exists = null_t<int_t>::def()) {
      return insert(key0, exists); // equivalent to insert;
   }

   // --------------------------------------------------------------------------

   r_t find(key_t const& key0) const {
      node_t *low = r_t::lower(key0);
      node_t *up  = r_t::upper(key0);
      return r_t(r_t::root, ++from(low), --upto(up, up));
   }

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// map;

template<typename key_T, typename obj_T>
struct mapnode_t : public treenode_t {

   typedef key_T key_t;
   typedef obj_T obj_t;

   // --------------------------------------------------------------------------
   // ctor dtor;

   mapnode_t(key_t const& key0) : key(key0) {
   }
   mapnode_t(key_t const& key0, obj_t const& obj0) : key(key0), obj(obj0) {
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t operator==(mapnode_t const &rhs) const {
      return key == rhs.key && obj == rhs.obj;
   }
   int_t operator<(mapnode_t const& rhs) const {
      return key < rhs.key || (key == rhs.key && obj < rhs.obj);
   }

   int_t operator!=(mapnode_t const &rhs) const {
      return !(*this == rhs);
   }
   int_t operator>(mapnode_t const& rhs) const {
      return rhs < *this;
   }
   int_t operator<=(mapnode_t const& rhs) const {
      return !(rhs < *this);
   }
   int_t operator>=(mapnode_t const& rhs) const {
      return !(*this < rhs);
   }

   // --------------------------------------------------------------------------
   // data;

   key_t key;
   obj_t obj;
};

template<typename key_T, typename obj_T, typename mem_T = heap_t>
struct map_t : public tree_t<mapnode_t<key_T, obj_T>, mem_T>
{
   typedef obj_T obj_t;
   typedef mem_T mem_t;
   typedef tree_t<mapnode_t<key_T, obj_t>, mem_t> c_t;

   typedef typename c_t::node_t node_t;
   typedef typename c_t::r_t r_t;
   typedef typename r_t::key_t key_t;

   map_t(mem_t& mem0 = null_t<mem_t>::def()) :
      c_t(mem0) {
   }
   node_t *insert(key_t const& key0, obj_t const& obj0, int_t& exists = null_t<int_t>::def()) {
      node_t *node = c_t::insert(key0, exists);
      if(false == exists) {
         node->obj = obj0;
      }
      return node;
   }
   obj_t& operator[] (key_t const key0) {
      return insert(key0, obj_t())->obj;
   }
};

// -----------------------------------------------------------------------------
// multimap;

template<typename key_T, typename obj_T, typename mem_T = heap_t>
struct multimap_t : public tree_t<mapnode_t<key_T, obj_T>, mem_T>
{
   typedef obj_T obj_t;
   typedef mem_T mem_t;
   typedef tree_t<mapnode_t<key_T, obj_t>, mem_t> c_t;

   typedef typename c_t::node_t node_t;
   typedef typename c_t::r_t r_t;
   typedef typename r_t::key_t key_t;

   multimap_t(mem_t& mem0 = null_t<mem_t>::def()) :
      c_t(mem0) {
   }

   // --------------------------------------------------------------------------

   node_t *insert(key_t const& key0, obj_t const& obj0, int_t& exists = null_t<int_t>::def()) {
      exists = false;
      if(0 == r_t::root) {
         r_t::a = r_t::x = r_t::z = r_t::root = c_t::create(key0, 0);
         r_t::root->obj = obj0;
         return r_t::root;
      }
      node_t *node = r_t::root;
      while(node) {
         if(key0 <= node->key) {
            if(key0 == node->key) exists = true;
            if(0 == node->left) {
               node_t *left = c_t::create(key0, node);
               if(0 == left) return 0;
               node->left = left;
               if(key0 <= r_t::a->key) {
                  r_t::a = r_t::x = (node_t*)node->left;
               }
               node = c_t::fix((node_t*)node->left, 1);
               break;
            } else node = (node_t*)node->left;
         } else if(key0 > node->key) {
            if(0 == node->right) {
               node_t *right = c_t::create(key0, node);
               if(0 == right) return 0;
               node->right = right;
               if(key0 > r_t::z->key) {
                  r_t::z = (node_t*)node->right;
               }
               node = c_t::fix((node_t*)node->right, 1);
               break;
            } else node = (node_t*)node->right;
         }
      }
      node->obj = obj0;
      //check();
      return node;
   }
   node_t *append(key_t const& key0, obj_t const& obj0, int_t& exists = null_t<int_t>::def()) {
      return insert(key0, obj0, exists); // equivalent to insert;
   }

   // --------------------------------------------------------------------------

   r_t find(key_t const& key0) const {
      node_t *node = r_t::find(key0);
      r_t fromr = from(node);
      r_t uptor = upto(node, node);
      r_t fromr1 = fromr;
      r_t uptor1 = uptor;
      while(fromr.x && key0 == fromr.x->key) {
         fromr1 = fromr;
         ++fromr;
      };
      while(uptor.x && key0 == uptor.x->key) {
         uptor1 = uptor;
         --uptor;
      };
      return r_t(r_t::root, uptor1.x, fromr1.x);
   }

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// store (allocator);

template<typename obj_T, typename mem_T = heap_t>
struct store_t
{
   typedef obj_T obj_t;
   typedef mem_T mem_t;
   typedef vector_t<obj_t, false> cont_t;

   struct item_t {
      item_t() : next(0) {
      }
      item_t *next;
      obj_t obj;
   };

   store_t(mem_t& mem0 = null_t<mem_t>::def()) :
      ini(false), cont(mem0), first(0), used(0) {
   }
   store_t(int_t const size0, mem_t& mem0 = null_t<mem_t>::def()) :
      ini(true), cont(mem0), first(0), used(0) {
      init(size0);
   }
   ~store_t() {
      if(ini) fini();
   }
   void_t init(int_t const size0) {
      asrt(cont.empty());
      cont.resize(size0);
      typename cont_t::r_t r = cont.range();
      for(; r.x < r.z; ++r)
         r.x->next = r.x + 1;
      first = r.a;
   }
   void_t fini() {
      asrt(false == cont.empty());
      used = 0;
      first = 0;
      cont.clear();
   }

   obj_t *get() {
      asrt(false == cont.empty());
      if(0 == first) return 0; // exhausted;
      item_t *item = first;
      first = first->next;
      item->next = (item_t*)((int_t)this - 1); // mark odd;
      ++used;
      return &item->obj; // got;
   }
   int_t ret(obj_t* mem0) {
      asrt(mem0);
      item_t *item = ((item_t*)mem0) - 1;
      asrt(cont.contains(item));
      asrt((int_t)item->next % 2); // check mark;
      item->next = first;
      item->obj.~obj_t();
      first = item;
      --used;
   }

   int_t ini;
   cont_t cont;
   item_t *first;
   int_t used;
};

// -----------------------------------------------------------------------------
// pool (allocator);

template<typename mem_T = heap_t>
struct pool_t
{
   typedef mem_T mem_t;

   struct slot_t {
      slot_t() : prev(0), size(0) {
      }
      byte_t *mem() const {
         return (byte_t*)(this + 1);
      }
      void_t clear() {
         mset(mem(), 0, size);
      }
      slot_t *prev;
      int_t size;
   };

   pool_t(mem_t& mem0 = null_t<mem_t>::def()) :
      ini(false), v(mem0), used(0) {
   }
   pool_t(int_t const size0, mem_t& mem0 = null_t<mem_t>::def()) :
      ini(true), v(mem0), used(0) {
      init(size0);
   }
   ~pool_t() {
      if(ini) fini();
   }
   void_t init(int_t const size0) {
      asrt(v.empty());
      v.resize(size0);
      slot_t* slot = (slot_t*)v.a;
      slot->prev = 0;
      slot->size = v.size() - sizeof(slot_t);
      used = sizeof(slot_t);
   }
   void_t fini() {
      if(v.empty()) return;
      used = 0;
      v.clear();
   }

   byte_t *get(int_t const size0) {
      asrt(false == v.empty());
      int_t const size1 = rndup8(size0);
      for(slot_t *slot = (slot_t*)v.a; slot; slot = next(slot)) {
         if(avail(slot)) {
            if(size1 == slot->size) { // reuse;
               occupy(slot);
               used += size1;
               return slot->mem();
            } else if((size1 + sizeof(slot_t)) < slot->size) { // split;
               slot_t* rem = ((byte_t*)(slot + 1)) + size1;
               rem->prev  = slot;
               rem->size  = slot->size - (size1 + sizeof(slot_t));
               slot_t* nex = next(slot);
               if(nex) nex->prev = avail(nex->prev) ? rem : occupy(rem);
               occupy(slot->prev);
               slot->size = size1;
               used += size1 + sizeof(slot_t);
               return slot->mem();
            }
         }
      }
      return 0; // exhausted;
   }
   int_t ret(byte_t* mem0) {
      asrt(mem0!=NULL);
      asrt(v.contains(mem0));
      slot_t* slot = ((slot_t*)mem0) - 1;
      asrt(!avail(slot));
      int_t const size0 = slot->size;
      used -= size0;
      if(slot->prev && avail(slot->prev)) { // snap left;
         slot = slot->prev;
         slot->size += sizeof(slot_t) + size0;
      }
      slot_t *nex = next(slot);
      if(nex && avail(next)) { // snap right;
         nex->prev = slot;
         slot->size += sizeof(slot_t) + nex->size;
      }
      vacate(slot);
      slot->clear();
      return size0;
   }
   slot_t* next(slot_t* slot0) const {
      asrt(slot0 && slot0 < v.z);
      return ((byte_t*)(slot0 + 1)) + slot0->size;
   }
   slot_t* occupy(slot_t *&slot0) const {
      asrt(slot0 && !avail(slot0));
      return (slot_t*)(((int_t)slot0) - 1); // mark odd;
   }
   slot_t* vacate(slot_t *&slot0) const {
      asrt(slot0 && avail(slot0));
      return (slot_t*)(((int_t)slot0) + 1); // mark even;
   }
   slot_t* avail(slot_t* slot0) const {
      return ((int_t)slot0) % 2 ? 0 : slot0; // check odd mark;
   }
   int_t ini;
   vector_t<byte_t, false> v;
   slot_t *first;
   int_t used;
};

// -----------------------------------------------------------------------------
// ram (allocator);

struct ram_t {

   ram_t() : ini(false), mem(0) {
   }
   ram_t(int_t const size0, byte_t const val0 = 0) : ini(true), mem(0) {
      init(size0, val0);
   }
   ~ram_t() {
      if(ini) fini();
   }

   void_t init(int_t const size0, byte_t const val0 = 0) {
      asrt(0 == mem);
      resize(size0, val0);
   }
   void_t fini() {
      if(0 == mem) return;
      heap_t().ret(mem);
      mem = 0;
   }
   int_t size() const {
      return mem ? *(((int_t*)mem) - 1) : 0;
   }
   int_t resize(int_t const size0, byte_t const val0 = 0) {
      int_t const size1 = M1_RNDUP8(size0);
      if(size1 == size()) return size1; // equal;
      byte_t* mem1 = heap_t().get(size1);
      asrt(mem1!=NULL);
      mcpy(mem1, mem, M1_MIN(size(), size1));
      if(size() < size1) { // grow;
         mset(mem1 + size(), val0, size1 - size());
      }
      fini();
      mem = mem1;
      return size();
   }
   void_t read(byte_t* tgt, int_t const pos, int_t const bytes) const {
      asrt(mem!=NULL);
      asrt(pos + bytes < size());
      mcpy(tgt, mem + pos, bytes);
   }
   void_t write(int_t const pos, byte_t* src, int_t const bytes) {
      asrt(mem!=NULL);
      asrt(pos + bytes < size());
      mcpy(mem + pos, src, bytes);
   }
   void_t set(int_t const pos, byte_t val, int_t const bytes) {
      asrt(mem!=NULL);
      asrt(pos + bytes < size());
      mset(mem + pos, val, bytes);
   }
   int_t ini;
   byte_t* mem;
};

// -----------------------------------------------------------------------------

#endif

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_CONT;

// -----------------------------------------------------------------------------
// lib;

#if !(M1_LIB_NONE == M1_LIB)

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------
// endianess;

#define XENUM2(X) \
   X(little, "little")\
   X(big,    "big")
   XENUM2_IMPL(endian_e)
#undef XENUM2

static endian_e endianess() {
   int_t const val = 1;
   return *(byte_t*)&val ? little : big;
}

// -----------------------------------------------------------------------------
// err_str;

#if (M1_LIB_LINUX == M1_LIB)
static string_t err_str(int_t err = -1) {
   if(-1 == err) {
      err = errno;
   }
   return string_t(err, ":", strerror(err));
}
#endif

// -----------------------------------------------------------------------------
// exists;

#if (M1_LIB_LINUX == M1_LIB)
static int_t exists(string_t const path) {
   struct stat buf;
   return 0 == stat(path.at(), &buf);
}
#endif

// -----------------------------------------------------------------------------
// rm;

#if (M1_LIB_LINUX == M1_LIB)
static int_t rm(string_t const path, int_t trace = 0) {
   int_t const res = remove(path.at());
   if(res && trace) {
      tracec("rm:err:", err_str());
   }
   return res;
}
#endif

// -----------------------------------------------------------------------------
// cwd;

#if (M1_LIB_LINUX == M1_LIB)
static string_t cwd() {
   string_t cwd0(getcwd(0, 0));
   if(cwd0.empty()) {
      tracec("cwd:err:", err_str());
   }
   return cwd0;
}
#endif

// -----------------------------------------------------------------------------
// execname;

#if (M1_LIB_LINUX == M1_LIB)
static string_t execname() {
   int_t const bufsize = 256;
   byte_t buf[bufsize] = { 0 };
   int_t const res = readlink("/proc/self/exe", buf, bufsize - 1);
   if(-1 == res) {
      tracec("execname:err:", err_str());
   }
   return &buf[0];
}
#endif

#if (M1_LIB_WIN == M1_LIB)
static string_t execname() {
   asrt(0);
   //shModule = NULL -> current process executable;
   //DWORD WINAPI GetModuleFileName(
   //  _In_opt_  HMODULE hModule,
   //  _Out_     LPTSTR lpFilename,
   //  _In_      DWORD nSize
);
#endif

// -----------------------------------------------------------------------------
// dllname;

// -ldl;

#if (M1_LIB_LINUX == M1_LIB)
static string_t dllname() {
   Dl_info dl_info;
   dladdr((void*)dllname, &dl_info);
   return dl_info.dli_fname;
}
#endif

#if (M1_LIB_WIN == M1_LIB)
static string_t dllname() {
   asrt(0);
}
#endif

// -----------------------------------------------------------------------------
// envval;

#if (M1_LIB_LINUX == M1_LIB)
static string_t envvar(string_t const& name0) {
   return getenv(name0.at());
}
#endif

#if (M1_LIB_WIN == M1_LIB)
static string_t envvar(string_t const& name0) {
   asrt(0);
}
#endif

// -----------------------------------------------------------------------------
// curpid;

#if (M1_LIB_LINUX == M1_LIB)
static int_t curpid() {
   return getpid();
}
#endif

#if (M1_LIB_WIN == M1_LIB)
static int_t curpid() {
   asrt(0);
}
#endif

// -----------------------------------------------------------------------------
// usr & grp;

#if (M1_LIB_LINUX == M1_LIB)

static int_t cureuid() {
   return (int_t)geteuid();
}
static string_t curuname() {
   int_t uid = cureuid();
   struct passwd *pw = getpwuid((uid_t)uid);
   return pw ? string_t(pw->pw_name) : string_t("");
}
static int_t euid(string_t const &uname0) {
   struct passwd *pwd = getpwnam(uname0.at());
   return pwd ? pwd->pw_uid : -1;
}

static int_t curegid() {
   return (int_t)getegid();
}
static string_t curgname() {
   int_t gid = curegid();
   struct group *grp = getgrgid((gid_t)gid);
   return grp ? string_t(grp->gr_name) : string_t("");
}
static int_t egid(string_t const &gname0) {
   struct group *grp = getgrnam(gname0.at());
   return grp ? grp->gr_gid : -1;
}

// -----------------------------------------------------------------------------

template<typename ... Ts>
static int_t tracef(Ts ... ts) {
   int_t res = 0;
   #if !defined(M1_FEAT_PRINTF)
      if(traceb) {
   #endif
         string_t s(ts ...);
         #if defined(M1_FEAT_PRINTF)
            prnt("%s", s.at());
         #endif
         if(traceb) {
            traceb(s.at(), s.size());
         }
   #if !defined(M1_FEAT_PRINTF)
      }
   #endif
   return res;
}

template<typename ... Ts>
static int_t tracec(Ts ... ts) {
   static int_t const pid = curpid();
   return tracef(pid, ":", heap_t::used(), ":m1:", ts .../*, '\n'*/);
}

#endif

// -----------------------------------------------------------------------------
// constrain;

static string_t constr(byte_t &x, byte_t const a, byte_t const z) {
   int_t const x0 = x;
   x = x < a ? a : (z < x ? z : x) ;
   return string_t("constr:", x0, '[', (int_t)a, ':', (int_t)z, ']', (int_t)x);
}
static string_t constr(int_t &x, int_t const a, int_t const z) {
   int_t const x0 = x;
   x = x < a ? a : (z < x ? z : x);
   return string_t("constr:", x0, '[', a, ':', z, ']', x);
}

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_LIB;

// -----------------------------------------------------------------------------
// filesystem;

#if !(M1_FS_NONE == M1_FS)

// -----------------------------------------------------------------------------

#if (M1_FS_LINUX == M1_FS)
   #include <sys/types.h>
   #include <dirent.h>
   #include <stdio.h>
   #include <unistd.h>
   #include <sys/stat.h>
   #include <fcntl.h>
   #include <pthread.h>
#endif
#if (M1_FS_WIN == M1_FS)
   //TODO;
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------

#if (M1_FS_LINUX == M1_FS)

static int_t isdir(string_t const &path) {
   struct stat st;
   if(-1 == stat(path.at(), &st)) {
      tracec("isdir:err:", err_str());
   }
   return S_ISDIR(st.st_mode);
}

struct dir_t {
   dir_t() : ini(false), fd(0) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("dir");
      #endif
   }
   dir_t(string_t const& path0, int_t const mode0 = 0777) : ini(true), fd(0), mode_(mode0) {
      init(path0);
   }
   ~dir_t() {
      if(ini) fini();
   }
   void_t init(string_t const& path0, int_t const mode0 = 0777) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("dir:path0:", path0, " mode:", mode0);
      #endif
      asrt(0 == fd);
      path = path0;
      mode_ = mode0;
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("dir:path1:", path, " mode:", mode_);
      #endif
   }
   void_t fini() {
      close();
      path.clear();
   }
   int_t exists() {
      struct stat buf;
      return 0 == stat(path.at(), &buf) && S_ISDIR(buf.st_mode);
   }
   int_t make(int_t const mode0 = 0777) {
      if(path.empty()) {
         tracec("dir:make:path:empty");
         return -1;
      }
      if(mkdir(path.at(), mode0)) {
         int_t const err = errno;
         tracec("dir:make:err:", err_str(err));
         return err;
      }
      mode_ = mode0;
      return 0;
   }
   int_t mode(int_t const mode0 = 0777) {
      if(path.empty()) {
         tracec("dir:mode:path:empty");
         return -1;
      }
      if(chmod(path.at(), mode0)) {
         int_t const err = errno;
         tracec("dir:make:err:", err_str(err));
         return err;
      }
      mode_ = mode0;
      return 0;
   }
   int_t open() {
      if(path.empty()) {
         tracec("dir:open:path:empty");
         return -1;
      }
      if(fd) {
         tracec("dir:open:already opened");
         return -2;
      }
      fd = opendir(path.at());
      if(0 == fd) {
         int_t const err = errno;
         tracec("dir:open:err:", err_str(err));
         return err;
      }
      return 0;
   }
   void_t close() {
      if(fd && -1 == closedir(fd)) {
         tracec("dir:close:err:", err_str());
      }
      fd = 0;
   }
   int_t read(string_t &name) {
      if(0 == fd) {
         int_t const res = open();
         if(0 != res) return res;
      }
      errno = 0;
      dirent* ent = readdir(fd);
      if(ent) {
         name = ent->d_name;
         return 0;
      }
      tracec("dir:read:err:", err_str());
      if(0 == errno) return -1; // eos;
      return -2;
   }
   int_t list(strings_t &ents, int_t const dots = false) {
      if(0 == fd) {
         int_t const res = open();
         if(0 != res) return res;
      }
      int_t ent_cnt = 0;
      string_t ent;
      while(0 == read(ent)) {
         if(!dots && (string_t("." ) == ent ||
                      string_t("..") == ent)) continue;
         string_t full(path);
         if(*full.z != '/') full << '/';
         full << ent;
         ents.append(full);
         ++ent_cnt;
      }
      return ent_cnt;
   }
   int_t ini;
   string_t path;
   DIR* fd;
   int_t mode_;
};

struct file_t {
   enum : int_t {
      default_limit = 1 << 20 // 1024^2 = 1048576 ~ 1MiB;
   };
   enum : int_t {
      origin_set = SEEK_SET,
      origin_cur = SEEK_CUR,
      origin_end = SEEK_END
   };
   file_t() : ini(false), fd(0), size_(-1) {
   }
   // r=read,w=write,a=append,+=update,b=binary,x=exclusive
   file_t(string_t const& url0, string_t const& mode0 = string_t("a+")) :
      ini(true), fd(0), size_(-1) {
      init(url0, mode0);
   }
   ~file_t() {
      if(ini) fini();
   }
   void_t init(string_t const& url0, string_t const& mode0 = string_t("a+")) {
      //asrt(0 == fd);
      url = url0;
      mode = mode0;
      size_ = -1;
      limit(default_limit);
      mutex = PTHREAD_MUTEX_INITIALIZER;
      open();
   }
   void_t fini() {
      close();
      url.clear();
   }
   int_t open(string_t const mode0 = string_t("a+")) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:open:url:", url);
      #endif
      if(mode != mode0) {
         #if defined(M1_FEAT_FS_DEBUG)
            tracec("file:open:name:", url.name(), " mode:", mode, " -> ", mode0);
         #endif
         mode = mode0;
         if(fd) {
            close();
         }
      }
      if(fd) {
         #if defined(M1_FEAT_FS_DEBUG)
            tracec("file:open:fd:", (void_t*)fd, " already open");
         #endif
         return -1;
      }
      string_t const path = url.path();
      dir_t dir(url.path());
      if(!dir.exists() && dir.make()) {
         tracec("file:open:dir:path:", dir.path, " err:", err_str());
         return -2;
      }
      #if defined(M1_FEAT_FS_DEBUG)
      tracec("file:open:dir:path:", dir.path, " err:", err_str());
      #endif
      fd = fopen(url.at(), mode.at());
      if(0 == fd) {
         tracec("file:open:err:", err_str());
         return -3;
      }
      size();
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:open:opened:name:", url.name(), " fd:", (void_t*)fd, " size:", size_);
      #endif
      return 0;
   }
   void_t close() {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:close:url:", url, " mode:", mode, " fd:", (void_t*)fd, " size:", size());
      #endif
      if(0 == fd) return;
      if(-1 == fclose(fd)) {
         tracec("file:close:err:", err_str());
      }
      fd = 0;
   }
   int_t post(int_t &val0 = null_t<int_t>::def()) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:post:", url);
      #endif
      if(0 == fd) {
         tracec("file:post:invalid handle");
         if(url.empty()) {
            url = string_t(M1_PATH, M1_SCOPE, ".lock");
         }
         open();
         asrt(fd!=0);
      }
      static struct flock fl {
         fl.l_type   = F_UNLCK,  // F_RDLCK, F_WRLCK, F_UNLCK;
         fl.l_whence = SEEK_SET, // SEEK_SET, SEEK_CUR, SEEK_END;
         fl.l_start  = 0,        // offset from l_whence;
         fl.l_len    = 0         // length, 0 = to eof;
      };
      fl.l_pid = curpid();
      pthread_mutex_unlock(&mutex);
      // set the region to unlocked;
      if(-1 == fcntl(fileno(fd), F_SETLK, &fl)) {
         tracec("file:post:err:", err_str());
      }
      return 0;
   }
   int_t wait(int_t const /*timeout0 = forever*/ = -1) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:wait:", url);
      #endif
      if(0 == fd) {
         tracec("file:wait:invalid handle");
         if(url.empty()) {
            url = string_t(M1_PATH, M1_SCOPE, ".lock");
         }
         open();
         asrt(fd!=0);
      }
      static struct flock fl = {
         fl.l_type   = F_WRLCK,  // F_RDLCK, F_WRLCK, F_UNLCK;
         fl.l_whence = SEEK_SET, // SEEK_SET, SEEK_CUR, SEEK_END;
         fl.l_start  = 0,        // offset from l_whence;
         fl.l_len    = 0         // length, 0 = to eof;
      };
      fl.l_pid = curpid();
      // F_GETLK, F_SETLK, F_SETLKW;
      if(-1 == fcntl(fileno(fd), F_SETLKW, &fl)) {
         tracec("file:wait:err:", err_str());
      }
      pthread_mutex_lock(&mutex);
      return 0;
   }
   int_t exists() {
      struct stat buf;
      return 0 == stat(url.at(), &buf);
   }
   int_t read(byte_t* buf, int_t const& num) {
      if(0 == fd) return -1;
      int_t num1 = 0;
      if(num != (num1 = fread(buf, sizeof(byte_t), num, fd))) {
         if(feof(fd)) {
            tracec("file:read:eof");
            return num1;
         } else if(ferror(fd)) {
            tracec("file:read:err:stream");
            return -2;
         }
         tracec("file:read:err:undefined");
         return -3;
      }
      return num;
   }
   int_t readline(string_t &line) {
      if(0 == fd) return -1;
      int_t c;
      while(1) {
         c = fgetc(fd);
         if('\n' == c || EOF == c) break;
         //tracec("file:readline:fgetc:b:", (int_t)c);
         line.append((byte_t)c);
      }
      //tracec("file:readline:", line, " size:", line.size());
      return EOF == c ? EOF : line.size();
   }
   int_t readlines(strings_t &lines, int_t num = M1_MAXINT64) {
     if(0 == fd) return -1;
     if(0 == size()) return -2;
     rewind();
     string_t line;
     int_t res = 0;
     while(1) {
        res = readline(line);
        if(EOF == res) break;
        lines.append(line);
        --num;
        line.clear();
     }
     return lines.size();
   }
   int_t writelines(strings_t &lines) {
     if(0 == fd) return -1;
     for(int_t i = 0; i < lines.size(); ++i) {
        writeline(lines[i]);
     }
     return lines.size();
   }
   int_t write(byte_t const* buf, int_t num = 0) {
      if(0 == fd) {
         return -1;
      }
      num = (0 == num ? slen(buf) : num);
      if(size() + num > limit_) {
         tracec("file:write:", str(), " overflow");
         return -2;
      }
      int_t num1 = 0;
      if(num != (num1 = fwrite(buf, sizeof(byte_t), num, fd))) {
         if(feof(fd)) {
            tracec("file:write:eof");
            return num1;
         } else if(ferror(fd)) {
            tracec("file:write:err:stream");
            return -1;
         }
         tracec("file:write:err:undefined");
         return -3;
      }
      size_ += num;
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:write:num:", num, ' ', str());
      #endif
      if(fflush(fd)) {
          tracec("file:fflush:err:", err_str());
      }
      return num;
   }
   int_t write(string_t const &s) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:write:", s);
      #endif
      return write(s.at(), s.size());
   }
   int_t writeline(string_t const &s) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:writeline:", s);
      #endif
      string_t line(s, '\n');
      return write(line.at(), line.size());
   }
   int_t seek(int_t const& offset, int_t const origin) {
      if(0 == fd) return -1;
      asrt(origin_set <= origin && origin <= origin_end);
      if(offset < 0) return -1;
      int_t const pos = fseek(fd, offset, origin);
      if(-1 == pos) {
         tracec("file:seek:err:", err_str());
      }
      return pos;
   }
   int_t rewind() {
      return seek(0, origin_set);
   }
   int_t tell() {
      if(0 == fd) return -1;
      int_t const pos = ftell(fd);
      if(-1 == pos) {
         tracec("file:tell:err:", err_str());
      }
      return pos;
   }
   int_t scan(string_t const& format, ...) {
      if(0 == fd) return -1;
      va_list va;
      va_start(va, format);
      int_t const res = vfscanf(fd, format.at(), va);
      va_end(va);
      return res;
   }
   int_t size() const {
      if(-1 == size_) {
         struct stat stat0;
         int_t const res = stat(url.at(), &stat0);
         if(-1 == res) {
            tracec("file:size:err:", err_str());
         }
         size_ = (-1 != res) ? (int_t)stat0.st_size : -1;
      }
      #if defined(M1_FEAT_FS_DEBUG)
      tracec("file:size:", size_);
      #endif
      return size_;
   }
   int_t limit(int_t const limit0) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:limit:", limit0);
      #endif
      if(limit0 < 0) {
         #if defined(M1_FEAT_FS_DEBUG)
            tracec("file:limit:negative limit:", limit0);
         #endif
         return -2;
      }
      if(limit0 < size_) {
         return resize(limit0);
      }
      limit_ = limit0;
      return 0;
   }
   int_t resize(int_t const size0) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:resize:", size0);
      #endif
      if(size0 < 0) {
         #if defined(M1_FEAT_FS_DEBUG)
            tracec("file:resize:negative size:", size0);
         #endif
         return -1;
      }
      if(truncate(url.at(), size0)) {
         int_t const err = errno;
         tracec("file:resize:err:", err_str(err));
         return err;
      }
      size();
      return 0;
   }
   void_t clear() {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:clear:", str());
      #endif
      if(fd) {
         open(string_t("w"));
         open();
      } else {
         open(string_t("w"));
         close();
      }
   }
   int_t remove() {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("file:remove:", str());
      #endif
      if(fd) {
         #if defined(M1_FEAT_FS_DEBUG)
            tracec("file:remove:opened:url:", url);
         #endif
         close();
      }
      if(::remove(url.at())) {
         int_t const err = errno;
         tracec("file:remove:err:", err_str(err));
         return err;
      }
      return 0;
   }
   string_t str() const {
      return string_t("ini:", ini,
                     " url:", url,
                    " mode:", mode,
                    " size:", size(),
                      " fd:", (void_t*)fd,
                    " limit:", limit_);
   }
   int_t ini;
   string_t url, mode;
   FILE* fd;
   int_t mutable size_;
   int_t limit_;
   pthread_mutex_t mutex;
};

struct log_t {

   log_t() : forward(0) {
   }
   log_t(string_t const& url0, int_t const limit0 = file_t::default_limit) :
      ini(true), forward(0) {
      init(url0, limit0);
   }
   ~log_t() {
      if(ini) fini();
   }
   void_t init(string_t const& url0, int_t const limit0 = file_t::default_limit) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("log:init:url:", url0, " limit:", limit0);
      #endif
      //suid.init("<m1.uid>"); // share with node_t;
      //lock.init(string_t("log.", suid.get()));
      //scope_t<lock_t> sl(lock);
      file.init(url0, "a+");
      file.limit(limit0);
   }
   void_t fini() {
      {
         //scope_t<lock_t> sl(lock);
         file.fini();
      }
      //lock.fini();
      //suid.fini();
   }
   int_t exists() {
      //scope_t<lock_t> sl(lock);
      return file.exists();
   }
   int_t limit(int_t const length) {
      //scope_t<lock_t> sl(lock);
      return file.limit(length);
   }
   int_t open(int_t const forward0 = 0) {
      #if defined(M1_FEAT_FS_DEBUG)
         tracec("log:open:url:", file.url, " forward:", forward0);
      #endif
      forward = forward0;
      return file.open();
   }
   void_t close() {
      file.close();
      forward = 0;
   }
   template<typename ... Ts>
   int_t write(Ts ... ts) {
      if(0 == file.fd) {
         file.open();
      }
      //scope_t<lock_t> sl(lock);
      static int_t const pid = curpid();
      string_t s(pid, ":", ts ..., "\n"); // prepend process id;
      if(forward) {
         tracec(s);
      }
      return file.write(s.at(), s.size());
   }
   void_t clear() {
      file.clear();
   }
   int_t ini;
   //suid_t suid;
   //lock_t lock;
   file_t file;
   int_t forward;
};

//static log_t logger("m1-log.txt");
//
//static void_t log_init(string_t const& url0,
//   int_t const limit0 = log_t::default_limit) {
// logger.init(url0, limit0);
//}
//template<typename ... Ts>
//static int_t log(Ts ... ts) {
//   return logger.write(ts ...);
//}

#endif

#if (M1_FS_WIN == M1_FS)
   //TODO;
#endif

// -----------------------------------------------------------------------------

}  // m1;

// -----------------------------------------------------------------------------

#endif // M1_FS;

// -----------------------------------------------------------------------------
// parallel;

#if !(M1_PARA_NONE == M1_PARA)

// -----------------------------------------------------------------------------

#if (M1_PARA_LINUX == M1_PARA)
   #include <pthread.h>
   #include <unistd.h>    // usleep, ftruncate;
   #include <fcntl.h>     // for O_* constants;
   #include <sys/stat.h>  // for mode constants;
   #include <semaphore.h>
   #include <errno.h>     // for semaphore timeout;
   #include <type_traits> // lock_t<smp_t>; lock_t<file_t>;
   #include <sys/mman.h> // shm_open; mmap; munmap;
   // -lrt; link against real-time library (librt);
   #include <sys/prctl.h> // prctl
#endif

#if (M1_PARA_WIN == M1_PARA)
   #include <windows.h>
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------
// semaphore (named; counting);

struct smp_t {

   #if (M1_PARA_LINUX == M1_PARA)
      typedef sem_t* handle_t;
      static handle_t constexpr invalid_handle = (handle_t)0; //SEM_FAILED
      #define XENUM3(X) \
         X(forever, = -1 , "forever")
         XENUM3_IMPL(timeout_e)
      #undef XENUM3
   #endif
   #if (M1_PARA_WIN == M1_PARA)
      typedef HANDLE handle_t;
      static handle_t constexpr invalid_handle = (handle_t)0;
      #define XENUM3(X) \
         X(forever, = INFINITE , "forever")
         XENUM3_IMPL(timeout_e)
      #undef XENUM3
   #endif

   #define XENUM3(X) \
      X(signal,  = -3 , "signal")\
      X(timeout, = -2 , "timeout")\
      X(fail,    = -1 , "fail")\
      X(ok,      =  0 , "ok")
      XENUM3_IMPL(res_e)
   #undef XENUM3

   smp_t() : ini(false), fd(invalid_handle), created(false) {
   }
   smp_t(string_t const name0, string_t const val00 = string_t("0")) :
   //smp_t(string_t const name0, int_t const val0 = 0) :
      ini(true), fd(invalid_handle), created(false) {
      init(name0, val00);
      //init(name0, val0);
   }
   ~smp_t() {
      if(ini) fini();
   }
   void_t init(string_t const &name0, string_t const val00 = string_t("0")) {
   //void_t init(string_t const &name0, int_t const val0 = 0) {

      int_t val0 = -1;
      string_t s = val00;
      s >> val0;

      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:init:val00:", val00, " val0:", val0);
         if(invalid_handle != fd || name0.empty()) {
            tracec("smp:init:fd:", fd, " name:", name, " val:", val0);
            usleep(1000 * 1000);
         }
      #endif

      asrt(invalid_handle == fd);
      asrt(false == name0.empty());

      //if(name0.empty()) {
      //   static int_t id = 0;
      //   name = string_t("sem");
      //   name << id++;
      //} else {
         name = string_t("smp:", name0);
      //   name << name0;
      //}
      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:init:", str(), " val:", val0);
      #endif

      #if (M1_PARA_LINUX == M1_PARA)
         fd = sem_open(&name[0], O_CREAT | O_EXCL, S_IRWXU | S_IRWXG | S_IRWXO, (unsigned int)val0);
         if(invalid_handle == fd) {
            fd = sem_open(&name[0], 0);
            if(invalid_handle == fd) {
                tracec("smp:init:open:err:", err_str());
                usleep(1000 * 1000);
                asrt(0);
            }
            //asrt(invalid_handle != fd);
         } else {
            created = true;
         }
         #if defined(M1_FEAT_SMP_DEBUG)
            int sval = 0;
            int_t const res  = (int_t)sem_getvalue(fd, &sval);
            int_t const val1 = (int_t)sval;
            tracec("smp:init:val1:", val1, " res:", res);
         #endif
      #endif

      #if (M1_PARA_WIN == M1_PARA)
         SECURITY_ATTRIBUTES attr;
         attr.nLength = sizeof(SECURITY_ATTRIBUTES);
         attr.lpSecurityDescriptor = 0;
         attr.bInheritHandle = true;
         fd = CreateSemaphoreA(&attr, (LONG)val0, 0xffffffff, (LPCSTR)&name[0]); // $2 = init; $3 = max;
         if(invalid_handle == fd) {
            //asrt(ERROR_ALREADY_EXISTS == GetLastError());
            fd = OpenSemaphoreA(SEMAPHORE_ALL_ACCESS, true, (LPCSTR)&name[0]); // $2 = inherit fd;
            asrt(invalid_handle != fd);
         } else {
            created = true;
         }
      #endif

      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:init:", str());
      #endif
   }
   void_t fini() {

      asrt(invalid_handle != fd);

      #if (M1_PARA_LINUX == M1_PARA)
         int_t res = sem_close(fd);
         asrt(-1 != res);
         if(created) {
            res = sem_unlink(&name[0]);
            if(-1 == res) {
               tracec("smp:fini:res,", res, " err:", err_str(res));
            }
         }
      #endif

      #if (M1_PARA_WIN == M1_PARA)
         int_t const res = CloseHandle(fd);
         asrt(res); // ref counted;
      #endif

      created = false;
      fd = invalid_handle;

      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:fini");
      #endif
   }

   int_t get() {
      asrt(invalid_handle != fd);
      #if (M1_PARA_LINUX == M1_PARA)
         int val1 = 0;
         if(0 == sem_getvalue(fd, &val1)) {
            return (int_t)val1;
         } else {
            return -1;
         }
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         asrt(0);
      #endif
   }
   void_t reset() {
       while(0 < get()) {
         wait();
       }
   }
   int_t post(int_t &val0 = null_t<int_t>::def()) {

      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:post:enter:", str());
      #endif

      if(invalid_handle == fd){
          return fail;
      }

      int_t res = ok;

      #if (M1_PARA_LINUX == M1_PARA)
         int val1 = 0;
         if(0 == sem_getvalue(fd, &val1)) {
            val0 = val1;
            res  = ok;
         } else {
            tracec("smp:post:get:err:", err_str());
            val0 = -1;
            res = fail;
         }
         if(ok == res && 0 == sem_post(fd)) {
            ++val0;
         } else {
            tracec("smp:post:err:", err_str());
            val0 = -1;
            res = fail;
         }
      #endif

      #if (M1_PARA_WIN == M1_PARA)
         int_t val1 = 0;
         if(false != ReleaseSemaphore(fd, 1, &val1)) { // $2 = inc; $3 = ref prev val;
            val0 = (int_t)val1;
            res  = ok;
         } else {
            tracec("smp:post:error");
         val0 = -1;
            res  = fail;
         }
      #endif

      #if defined(M1_FEAT_SMP_DEBUG)
          tracec("smp:post:exit:", str(), " val:", val0, " res:", res);
      #endif
      return res;
   }
   int_t wait(int_t const timeout0 = forever) {

      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("smp:wait:enter:", str());
      #endif

      asrt(invalid_handle != fd);

      #if (M1_PARA_LINUX == M1_PARA)
         if(forever == timeout0) {
            #if defined(M1_FEAT_SMP_DEBUG)
               tracec("smp:wait:", str(), " timeout:forever");
            #endif
            if(-1 == sem_wait(fd)) {
               int_t const err = errno;
               if(EINTR == err) {
                  tracec("smp:wait:signal:err:", err_str(err));
                  return signal;
               } else {
                  tracec("smp:wait:fail");
                  return fail;
               }
            } else {
               #if defined(M1_FEAT_SMP_DEBUG)
                  tracec("smp:wait:...received");
               #endif
               return ok;
            }
         } else {
            timespec timeout1;
            int_t const res = clock_gettime(CLOCK_REALTIME, &timeout1);
            asrt(-1 != res);
            timeout1.tv_sec += timeout0 / 1000;
            int_t nsec = timeout1.tv_nsec + ((timeout0 % 1000) * 1000000);
            if(1000000000 < nsec) {
               nsec -= 1000000000;
               ++timeout1.tv_sec;
            }
            timeout1.tv_nsec = nsec;
            #if defined(M1_FEAT_SMP_DEBUG)
               tracec("smp:wait:", str(), " timeout:sec:", (int_t)timeout1.tv_sec, " nsec:", (int_t)timeout1.tv_nsec);
            #endif
            if(-1 == sem_timedwait(fd, &timeout1)) {
               int_t const err = errno;
               if(ETIMEDOUT == err) {
                  tracec("smp:wait:timeout:err:", err_str(err));
                  return timeout;
               } else if(EINTR == err) {
                  tracec("smp:wait:signal:err:", err_str(err));
                  return signal;
               } else {
                  tracec("smp:wait:fail:err:", err_str(err));
                  return fail;
               }
            } else {
               #if defined(M1_FEAT_SMP_DEBUG)
                  tracec("smp:wait:...received");
               #endif
               return ok;
            }
         }
      #endif

      #if (M1_PARA_WIN == M1_PARA)
         if(WAIT_OBJECT_0 != WaitForSingleObject(fd, (DWORD)timeout0)) {
            if(WAIT_TIMEOUT == GetLastError()) return timeout;
            else return fail;
         }  else return ok;
      #endif
   }
   string_t str() const {
      return string_t("ini:", ini,
                    " name:", name,
                      " fd:", fd,
                 " created:", created);
   }
   int_t    ini;
   string_t name;
   handle_t fd;
   int_t    created;
};

// -----------------------------------------------------------------------------
// thread;

struct thr_t {
   #if (M1_PARA_LINUX == M1_PARA)
      typedef pthread_t handle_t;
      static handle_t constexpr invalid_handle = (handle_t)0;
      // range: 1 - 99 for SCHED_FIFO  & SCHED_RR;
      // range: 0      for SCHED_OTHER & SCHED_BATCH;
      enum prio_e : int_t {
         low    =  1,
         normal = 50,
         high   = 99
      };
      static void_t *entry(void_t* arg0) {
         call(arg0);
         return 0;
      }
   #endif
   #if (M1_PARA_WIN == M1_PARA)
      typedef HANDLE handle_t;
      static handle_t constexpr invalid_handle = (handle_t)0;
      enum prio_e : int_t  {
         low    = THREAD_PRIORITY_IDLE,   // 16;
         normal = THREAD_PRIORITY_NORMAL, // 24;
         high   = THREAD_PRIORITY_HIGHEST // 26;
      };
      static DWORD __stdcall entry(void_t* arg0) {
         call(arg0);
         return 0;
      }
   #endif
   typedef void_t (*call_t) (void_t*);
   struct arg_t {
      arg_t() : call(0), thr(0), arg(0) {
      }
      call_t  call;
      thr_t*  thr;
      void_t* arg;
   };
   static void_t call(void_t const* arg0) {
      //asrt(arg0);
      if(0 == arg0) return;
      arg_t &arg0_r = *(arg_t*)arg0;
      #if (M1_PARA_LINUX == M1_PARA)
         pthread_detach(pthread_self());
         setname(arg0_r.thr->name);
      #endif
      if(arg0_r.call) {
         (*arg0_r.call)(arg0_r.arg);
      } else {
         arg0_r.thr->work(arg0_r.arg);
      }
      arg0_r.thr->join.post();
   }

   thr_t() : ini(false), prio(-1), handle(invalid_handle) {
   }
   thr_t(string_t const name0, call_t cb0 = 0, void_t* arg0 = 0, int_t const prio0 = normal) :
      ini(true), prio(-1), handle(invalid_handle) {
      init(name0, cb0, arg0, prio0);
   }
   virtual ~thr_t() {
      if(ini) {
         exit();
         fini();
      }
   }

   void_t init(string_t const name0, call_t cb0 = 0, void_t* arg0 = 0, int_t const prio0 = normal) {
      #if defined(M1_FEAT_THR_DEBUG)
         tracec("thr_t:init:name:", name0, " cb:", (void_t*)cb0, " arg:", (void_t*)arg0, " prio:", prio0);
         tracec("thr_t:init:handle:", (void_t*)handle, " invalid_handle:", (void_t*)invalid_handle);
      #endif

      asrt(invalid_handle == handle);

      name = string_t("<thr:<", name0, ">>.join");

      prio = prio0;

      join.init(name);

      arg.call = cb0;
      arg.thr  = this;
      arg.arg  = arg0;
      #if (M1_PARA_LINUX == M1_PARA)
         //pthread_attr_t attr;
         //asrt(pthread_attr_init(&attr));
         //asrt(pthread_attr_setstacksize(&attr, stack_size));
         //asrt(pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED)); // PTHREAD_CREATE_JOINABLE
         //free after exit: pthread_attr_destroy(&attr);
         //int_t res = pthread_create(&handle, &attr, entry, &arg);
         int_t res = pthread_create(&handle, NULL, entry, &arg);

         if(0 != res || invalid_handle == handle) {

            tracec("thr_t:init:handle:", (void_t*)handle, " res:", res);
            usleep(300 * 1000);
         }

         //priority(prio0);

      #endif
      #if (M1_PARA_WIN == M1_PARA)
         SECURITY_ATTRIBUTES attr;
         mset(&attr, 0, sizeof(SECURITY_ATTRIBUTES);
         attr.nLength = sizeof(SECURITY_ATTRIBUTES);
         attr.lpSecurityDescriptor = 0;
         attr.bInheritHandle = true;
         handle = CreateThread(&attr, 0, entry, &arg, 0, 0); // $2=stack; $5=run; $6=id;
         asrt(invalid_handle != handle);
         priority(prio0);
      #endif
   }
   void_t fini() {
      asrt(invalid_handle != handle);

      join.wait();
      join.fini();

      #if (M1_PARA_LINUX == M1_PARA)
         pthread_cancel(handle);
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         CloseHandle(handle);
      #endif
      handle = invalid_handle;
      prio   = -1;
   }

   virtual void_t work(void_t * /*arg0*/) {
      asrt(false); // override or specify entry on init;
   }

   void_t priority(int_t const prio0 = -1) {
      #if defined(M1_FEAT_THR_DEBUG)
         tracec("thr_t:priority:handle:", (void_t*)handle, " prio:", prio0);
      #endif
      asrt(invalid_handle != handle);
      #if (M1_PARA_LINUX == M1_PARA)
         pthread_attr_t attr;
         pthread_attr_init(&attr);
         int_t res = 0;
         if(-1 == prio0) {
            int policy = 0;
            pthread_attr_getschedpolicy(&attr, &policy);
            int_t const maxprio = sched_get_priority_max(policy);
            #if defined(M1_FEAT_THR_DEBUG)
               tracec("thr_t:priority:maxprio:", maxprio);
            #endif
            res = pthread_setschedprio(handle, (int)maxprio);
            prio = maxprio;
         } else {
            res = pthread_setschedprio(handle, prio0);
            prio = prio0;
         }
         pthread_attr_destroy(&attr);
         #if defined(M1_FEAT_THR_DEBUG)
            tracec("thr_t:priority:prio:", prio, " res:", res);
         #endif
         if(EINVAL == res) {
            tracec("thr_t:priority:err:EINVAL err:", err_str(), " invalid prio:", prio, " res:", res);
            usleep(300 * 1000);
         } else if(EPERM == res) {
            tracec("thr_t:priority:err:EPERM err:", err_str(), " inappropriate privileges");
            usleep(300 * 1000);
         } else if(ESRCH == res) {
            tracec("thr_t:priority:err:ESRCH err:", err_str(), " unknown handle:", (void_t*)handle);
            usleep(300 * 1000);
         }
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         int_t res = 0;
         if(-1 == prio0) {
            res = SetThreadPriority(handle, (int)high);
            prio = high;
         } else {
            int_t const res = SetThreadPriority(handle, (int)prio0);
            prio = prio0;
         }
         asrt(res);
      #endif
   }

   static void_t wait(int_t const msec0) {
      #if (M1_PARA_LINUX == M1_PARA)
         int_t const res = usleep(1000 * msec0);
         asrt(0 == res);
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         Sleep((DWORD)msec0);
      #endif
   }
   static void_t exit() {
      #if (M1_PARA_LINUX == M1_PARA)
         pthread_exit(0);
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         ExitThread(0);
      #endif
   }
   static handle_t whoami() {
      #if (M1_PARA_LINUX == M1_PARA)
         return pthread_self() ;
      #endif
      #if (M1_PARA_WIN == M1_PARA)
         return GetCurrentThread();
      #endif
   }
   static void_t setname(string_t const &name0) {
      #if defined(M1_FEAT_THR_DEBUG)
         tracec("thr_t:setname:name:", name0);
      #endif
      prctl(PR_SET_NAME, name0.at(), 0, 0, 0, 0);
   }
   string_t str() const {
      return string_t("ini:", ini,
                    " name:", name,
                    " prio:", prio,
                    " call:", (void_t*)arg.call,
                     " arg:", arg.arg,
                  " handle:", (int_t)handle);
   }
   int_t    ini;
   string_t name;
   int_t    prio;
   arg_t    arg;
   handle_t handle;
   smp_t    join;
};

// -----------------------------------------------------------------------------
// mutual exclusion lock (named binary semaphore or file; reentrant per thread);

template<typename pivot_T>
struct lock_t {
   lock_t() : owner(thr_t::invalid_handle), depth(0) {
   }
   lock_t(string_t const& name0) :
      owner(thr_t::invalid_handle), depth(0) {
      init(name0);
      pivot.ini = true;
      //sem.ini = true;
   }
   void_t init(string_t const &name0) {
      if(std::is_same<smp_t, pivot_T>::value) {
         pivot.init(string_t(name0, ".lock"), string_t("1"));
      } else if (std::is_same<file_t, pivot_T>::value) {
         pivot.init(name0, string_t("a+"));
      } else {
         asrt(0); // not supported;
      }
   }
   void_t fini() {
      depth = 0;
      owner = thr_t::invalid_handle;
      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("lock:fini");
      #endif
      pivot.fini();
      //sem.fini();
   }

   int_t enter() {
      thr_t::handle_t const tid = thr_t::whoami();
      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("lock:enter:tid:", (void_t*)tid, " owner:", (void_t*)owner, " depth:", depth);
      #endif
      if(thr_t::invalid_handle == owner || tid != owner) {
         pivot.wait();
         //sem.wait();
         owner = tid;
      }
      return ++depth;
   }
   int_t exit() {
      thr_t::handle_t const tid = thr_t::whoami();
      asrt(0 < depth && tid == owner);
      --depth;
      if(0 == depth) {
         owner = thr_t::invalid_handle;
         pivot.post();
         //sem.post();
      }
      #if defined(M1_FEAT_SMP_DEBUG)
         tracec("lock:exit:tid:", (void_t*)tid, " owner:", (void_t*)owner, " depth:", depth);
      #endif
      return depth;
   }
   string_t str() const {
      return string_t("owner:", (void_t*)owner, " depth:", depth, " ", pivot.str());
   }
   //smp_t sem;
   pivot_T pivot;
   thr_t::handle_t owner;
   int_t depth;
};

// -----------------------------------------------------------------------------
// shared memory (not threadsafe);

// layout:         -2|    -1|        0|
//         refcount:8|size:8|data:size|

struct shm_t {

   // --------------------------------------------------------------------------

   #if (M1_ENV_LINUX == M1_ENV)
      typedef int_t handle_t;
      static handle_t constexpr invalid_handle = (handle_t)-1;
   #endif

   #if (M1_ENV_WIN == M1_ENV)
      typedef void_t* handle_t;
      static handle_t constexpr invalid_handle = (handle_t)0;
   #endif

   // --------------------------------------------------------------------------

   shm_t() : handle(invalid_handle) {
   }
   shm_t(string_t const &name0, int_t const size0 = 0) :
      ini(true), handle(invalid_handle) {
      init(name0, size0);
   }
   ~shm_t() {
      if(ini) fini();
   }

   // --------------------------------------------------------------------------

   void_t init(string_t const &name0, int_t const size0 = 0) {
      #if defined(M1_FEAT_SHM_DEBUG)
         tracec("shm:init:name0:", name0, " size:", size0, " handle:", handle);
         thr_t::wait(1000);
      #endif
      asrt(invalid_handle == handle);
      name = string_t("shm:", name0);
      int_t const size1 = M1_RNDUP8(size0) + 2 * sizeof(int_t);
      int_t created = false;

      #if (M1_ENV_LINUX == M1_ENV)
         handle = shm_open(&name[0], O_RDWR, 0666);
         #if defined(M1_FEAT_SHM_DEBUG)
            tracec("shm:init:open:", handle);
         #endif
         if(invalid_handle == handle) {
            if(0 >= size0) { // prevent zero bytes;
               #if defined(M1_FEAT_SHM_DEBUG)
                  tracec("shm:init:open1:name:", name, " err:", err_str());
               #endif
               return;
            }
            handle = shm_open(&name[0], O_CREAT | O_RDWR, 0666);
            if(invalid_handle == handle) {
               #if defined(M1_FEAT_SHM_DEBUG)
                  tracec("shm:init:open2:err:", err_str());
                  thr_t::wait(1000);
               #endif
               asrt(0);
            }
            int_t const res = ftruncate(handle, size1);
            if(-1 == res) {
               #if defined(M1_FEAT_SHM_DEBUG)
                  tracec("shm:init:ftruncate:res:", res, " size:", size1, " err:", err_str());
               #endif
            }
            created = true;
         }
      #endif

      #if (M1_ENV_WIN == M1_ENV)
         handle = OpenFileMappingA(FILE_MAP_ALL_ACCESS, false, (LPCSTR)&name[0]); // inherit
         if(invalid_handle == handle) {
            if(0 >= size0) { // prevent zero bytes;
               #if defined(M1_FEAT_SHM_DEBUG)
                  tracec("shm:init:open1:name:", name, " not created");
               #endif
               return;
            }
            handle = CreateFileMappingA(INVALID_HANDLE_VALUE, // sys paging file;
                                        0,                    // sec;
                                        PAGE_READWRITE,       // protect;
                                        0,                    // size high;
                                        (DWORD)size1,         // size low;
                                        (LPCSTR)&name[0]);
            asrt(invalid_handle != handle);
            created = true;
         }
      #endif

      if(created) {
         val(-2, 1);   // initialise refcount & size;
         val(-1, M1_RNDUP8(size0));
      } else {
         val(-2, val(-2) + 1); // increment refcount;
      }
      #if defined(M1_FEAT_SHM_DEBUG)
         int_t const refcnt0 = refcnt();
         int_t const size2 = size();
         tracec("shm:init:", str(), " refcnt:",refcnt0, " size:", size2);
      #endif
   }
   void_t fini() {
      if(invalid_handle == handle) return;
      val(-2, val(-2) - 1); // decrement refcount;
      int_t const refcnt0 = refcnt();
      #if defined(M1_FEAT_SHM_DEBUG)
         int_t const size2 = size();
         tracec("shm:fini:", str(), " refcnt:", refcnt0, " size:", size2);
      #endif
      #if (M1_ENV_LINUX == M1_ENV)
         int_t res = close(handle);
         asrt(-1 != res);
         if(0 == refcnt0) {
            res = shm_unlink(&name[0]);
            if(-1 == res) {
               tracec("shm:fini:err:", err_str(res));
            }
         }
      #endif
      #if (M1_ENV_WIN == M1_ENV)
         int_t const res = CloseHandle(handle);
         asrt(res); // last closing reference deletes;
      #endif
      handle = invalid_handle;
      ini = false;
   }

   // --------------------------------------------------------------------------

   int_t size() const {
      return val(-1);
   }
   int_t refcnt() const {
      return val(-2);
   }

   int_t resize(int_t const size0) {
      asrt(1 == val(-2)); // guarantee unique reference;
      int_t const size1 = M1_RNDUP8(size0);
      if(size1 == size()) return size1; // equal;
      ram_t ram(size());
      read(ram.mem, 0, size());
      int_t const ini0 = ini;
      fini();
      init(name, size1);
      ini = ini0;
      write(0, ram.mem, M1_MIN(ram.size(), size1));
      if(ram.size() < size1) { // grow;
         // assume zero initialisation of shared memory;
      }
      return size1;
   }

   // --------------------------------------------------------------------------

   void_t read(byte_t* tgt, int_t const pos, int_t const bytes) const {
      #if defined(M1_FEAT_SHM_DEBUG)
         tracec("shm:read:pos:", pos, " bytes:", bytes);
      #endif
      if(0 == bytes) return;
      if(tgt == NULL) return;

      byte_t* src = map(pos, bytes);

     if (src != 0) {
      mcpy(tgt, src, bytes);
      unmap(src, pos, bytes);
     }
   }
   void_t write(int_t const pos, byte_t* src, int_t const bytes) const {
      #if defined(M1_FEAT_SHM_DEBUG)
         tracec("shm:write:pos:", pos, " bytes:", bytes);
      #endif
      if(0 == bytes) return;
      byte_t* tgt = map(pos, bytes);
      if(tgt != 0)  {
      mcpy(tgt, src, bytes);
      unmap(tgt, pos, bytes);
     }
   }
   void_t set(int_t const pos, byte_t val, int_t const bytes) const {
      #if defined(M1_FEAT_SHM_DEBUG)
         tracec("shm:set:pos:", pos, " val:", val, " bytes:", bytes);
      #endif
      byte_t* tgt = map(pos, bytes);
      if(tgt != NULL)  {
         mset(tgt, val, bytes);
         unmap(tgt, pos, bytes);
      }
   }

   // --------------------------------------------------------------------------

   byte_t* map(int_t const pos, int_t const bytes) const {
      if(invalid_handle == handle)
          return 0;
      byte_t* mem = 0;
      #if (M1_ENV_LINUX == M1_ENV)
         int_t const pagesize = getpagesize();
         int_t const pos1   = rnddwn(pos + 2 * sizeof(int_t), pagesize);
         int_t const offset = pos + 2 * sizeof(int_t) - pos1;
         int_t const bytes1 = bytes + offset;
         #if defined(M1_FEAT_SHM_DEBUG)
            tracec("shm:map:", str(), " pagesize:", pagesize, " pos:", pos,
                        " pos1:", pos1,    " offset:", offset, " bytes:", bytes,
                      " bytes1:", bytes1);
         #endif
         mem = (byte_t*)mmap(0, bytes1, PROT_READ | PROT_WRITE, MAP_SHARED, handle, pos1);
         if((void_t*)-1 == mem) { // MAP_FAILED;
            tracec("shm:map:mmap:pagesize:", getpagesize(), " err:", err_str());
            mem = 0;
            //asrt(0); //handled in map function call
         } else {
            mem += offset;
         }
      #endif
      #if (M1_ENV_WIN == M1_ENV)
         mem = (byte_t*)MapViewOfFile(handle, FILE_MAP_ALL_ACCESS, 0, pos + 2 * sizeof(int_t), bytes); // ofsh, ofsl, size;
      #endif
      //asrt(mem);//handled in map function call
      return mem;
   }
   void_t unmap(byte_t* mem, int_t const pos, int_t const bytes) const {
      if(invalid_handle == handle)
          return;
      asrt(mem!=NULL);
      #if (M1_ENV_LINUX == M1_ENV)
         int_t const pagesize = getpagesize();
         int_t const pos1   = rnddwn(pos + 2 * sizeof(int_t), pagesize);
         int_t const offset = pos + 2 * sizeof(int_t) - pos1;
         int_t const bytes1 = bytes + offset;
         mem -= offset;
         int_t const res = munmap(mem, bytes1);
         asrt(-1 != res);
         #if defined(M1_FEAT_SHM_DEBUG)
            tracec("shm:unmap:", str(), " pagesize:", pagesize, " pos:", pos,
                          " pos1:", pos1,    " offset:", offset, " bytes:", bytes, " bytes1:", bytes1);
         #endif
      #endif
      #if (M1_ENV_WIN == M1_ENV)
         int_t const res = UnmapViewOfFile((void_t*)mem);
         asrt(false != res);
      #endif
   }

   // --------------------------------------------------------------------------
   // specialised read,write for integer value at given position;

   int_t val(int_t const id0) const {
      union {
         int_t val;
         byte_t mem[sizeof(int_t)];
      } u0;
      u0.val = 0;
      read(&u0.mem[0], id0 * sizeof(int_t), sizeof(int_t));
      return u0.val;
   }
   void_t val(int_t const id0, int_t const val0) const {
      union {
         int_t val;
         byte_t mem[sizeof(int_t)];
      } u0;
      u0.val = val0;
      write(id0 * sizeof(int_t), &u0.mem[0], sizeof(int_t));
   }

   // --------------------------------------------------------------------------

   string_t str() const {
      return string_t("ini:", ini,
                    " name:", name,
                  " handle:", handle);
   }

   // --------------------------------------------------------------------------

   int_t    ini;
   string_t name;
   handle_t handle;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// shared unique id (named, threadsafe);

struct suid_t {
   suid_t() {
   }
   suid_t(string_t const name0, int_t const id0 = -1) {
      init(name0, id0);
   }
   void_t init(string_t name0, int_t const id0 = -1) {
      tracec("suid:init:name:", name0, " id:", id0);
      name0 << ".suid";
      lock.init(name0);
      scope_t<lock_t<smp_t>> sl(lock);
      shm.init(name0, sizeof(int_t));
      if(1 == shm.refcnt()) {
         shm.val(0, id0); // initialise;
      }
   }
   void_t fini() {
      {
         scope_t<lock_t<smp_t>> sl(lock);
         shm.fini();
      }
      lock.fini();
   }
   int_t get() {
      scope_t<lock_t<smp_t>> sl(lock);
      int_t const next = shm.val(0) + 1;
      shm.val(0, next);
      return next;
   }
   //TODO: ret(.);
   lock_t<smp_t> lock;
   shm_t shm;
};

// -----------------------------------------------------------------------------
// shared ring buffer (message queue; not threadsafe);

// layout:         -1|        0|         1|          2|      3        |
//         ringsize:8|readpos:8|writepos:8|datasize0:8|data0:datasize0|...
// precondition: zero initialised memory; idr == idw ~ empty; 1 spare byte;

template<typename mem_T = ram_t>
struct ring_t {
   typedef mem_T mem_t;
   ring_t() : ini(false) {
   }
   ring_t(int_t const size0) : ini(true) {
      init(size0);
   }
   ring_t(string_t const &name0, int_t const size0 = 0) : ini(true) {
      init(name0, size0);
   }
   ~ring_t() {
      if(ini) fini();
   }
   void_t init(int_t const size0) {
      mem.init(size0);
   }
   void_t init(string_t const &name0, int_t const size0 = 0) {
      mem.init(name0, size0);
      //tracec("ring:init:", str());
   }
   void_t fini() {
      mem.fini();
   }
   int_t write(byte_t* src0, int_t const size0) {
      asrt(src0 && size0);
      int_t const size1 = M1_RNDUP8(size0);
      int_t const total = sizeof(int_t) + size1;
      int_t const idr = 2 * sizeof(int_t) + mem.val(0);
      int_t       idw = 2 * sizeof(int_t) + mem.val(1);
      #if defined(M1_FEAT_RING_DEBUG)
         tracec("ring:write:size0:", size0, " size1:", size1,
                             " total:", total,   " idr:", idr, " idw:", idw);
      #endif
      if(idw < idr) {
         int_t const space = (idr - idw) - 1;
         if(space < total) return 0; // full;
         mem.val(idw >> 3, size0);
         idw += sizeof(int_t);
         #if defined(M1_FEAT_RING_DEBUG)
            tracec("ring:write:space:", space, " idw:", idw);
         #endif
         mem.write(idw, src0, size0);
         idw += size1;
      } else {
         int_t const space = (size() - idw) + idr - 1;
         if(space < total) return 0; // full;
         mem.val(idw >> 3, size0);
         idw += sizeof(int_t);
         int_t const right = size() - idw;
         #if defined(M1_FEAT_RING_DEBUG)
            tracec("ring:write:space:", space, " idw:", idw, " right:", right);
         #endif
         mem.write(idw, src0, min(size0, right));
         if(right < size0) {
            mem.write(2 * sizeof(int_t), src0 + right, size0 - right);
         }
         idw = (idw + size1) % size();
         if(0 == idw) idw = 2 * sizeof(int_t);
      }
      #if defined(M1_FEAT_RING_DEBUG)
         tracec("ring:write:idw:", idw);
      #endif
      mem.val(1, idw - 2 * sizeof(int_t));
      return size0;
   }
   int_t read(byte_t* dst0, int_t const max0, int_t const truncate = false) {
      asrt(dst0!=NULL);
      int_t       idr = mem.val(0) + 2 * sizeof(int_t);
      int_t const idw = mem.val(1) + 2 * sizeof(int_t);
      if(idr == idw) return 0; // empty;
      int_t       size0 = mem.val(idr >> 3);
      int_t const size1 = M1_RNDUP8(size0);
      if(max0 < size0) {
         if(false == truncate) return -1; // no space;
         size0 = max0;
      }
      idr += sizeof(int_t);
      #if defined(M1_FEAT_RING_DEBUG)
         tracec("ring:read:size0:", size0, " size1:", size1, " idr:", idr, " idw:", idw);
      #endif
      if(idr < idw) {
         mem.read(dst0, idr, size0);
         idr += size1;
      } else { // idw < idr;
         int_t const right = size() - idr;
         mem.read(dst0, idr, min(size0, right));
         if(right < size0) {
            mem.read(dst0 + right, 2 * sizeof(int_t), size0 - right);
         }
         idr = (idr + size1) % size();
         if(0 == idr) idr = 2 * sizeof(int_t);
      }
      #if defined(M1_FEAT_RING_DEBUG)
         tracec("ring:read:idr:", idr);
      #endif
      mem.val(0, idr - 2 * sizeof(int_t));
      return size0;
   }
   int_t next() const {
      int_t       idr = mem.val(0) + 2 * sizeof(int_t);
      int_t const idw = mem.val(1) + 2 * sizeof(int_t);
      if(idr == idw) return 0; // empty;
      return mem.val(idr >> 3); // msgsize;
   }
   void_t flush() {
      mem.set(0, 0, mem.size());
   }
   int_t size() const {
      return mem.val(-1) - 2 * sizeof(int_t);
   }
   int_t fill() {
      int_t const idr = 2 * sizeof(int_t) + mem.val(0);
      int_t       idw = 2 * sizeof(int_t) + mem.val(1);
      int_t size0 = size();
      int_t space = 0;
      if(idw < idr) {
         space = (idr - idw) - 1;
      } else {
         space = (size0 - idw) + idr - 1;
      }
      int_t fill0 = size0 - space;
      return fill0;
   }
   int_t level(int_t &fill0 = null_t<int_t>::def(), int_t &size0 = null_t<int_t>::def()) {
      fill0 = fill();
      size0 = size();
      return rdt(100, fill0, size0);
   }
   string_t str() const {
      int_t const size0 = size();
      int_t const idr = mem.val(0) + 2 * sizeof(int_t);
      int_t const idw = mem.val(1) + 2 * sizeof(int_t);
      int_t const msgsize = mem.val(idr >> 3);
      string_t s;
      s <<      "ini:" << ini
        <<    " size:" << size0
        <<     " idr:" << idr
        <<     " idw:" << idw
        << " msgsize:" << msgsize;
      return s;
   }
   int_t ini;
   mem_t mem;
};

#if 0

static int_t test_ring() {

   //ring_t<> ring0(128);
   //tracec("test_ring0:", ring0.str());

   //ring_t<ram_t> ring1(64);
   //tracec("test_ring1:", ring1.str());

   ring_t<shm_t> ring2("ring2", 64);
   tracec("test_ring2:", ring2.str());

   return 0;
}

#endif

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_PARA;

// -----------------------------------------------------------------------------
// signals & processes;

#if !(M1_PROC_NONE == M1_PROC)

// -----------------------------------------------------------------------------

#if (M1_PROC_LINUX == M1_PROC)
   #include <spawn.h>
   #include <signal.h>
#endif
#if (M1_PROC_WIN == M1_PROC)
   #include <signal.h>
   //TODO;
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------

#if (M1_PROC_LINUX == M1_PROC)

// -----------------------------------------------------------------------------
// signals;

#define XENUM4(X) \
   X(M1_SIGHUP   , = SIGHUP   , "SIGHUP",    "hangup (1)")\
   X(M1_SIGINT   , = SIGINT   , "SIGINT",    "keyboard interrupt (2)")\
   X(M1_SIGQUIT  , = SIGQUIT  , "SIGQUIT",   "keyboard quit (3)")\
   X(M1_SIGILL   , = SIGILL   , "SIGILL",    "illegal instruction (4)")\
   X(M1_SIGRRAP  , = SIGTRAP  , "SIGTRAP",   "trace/breakpoint trap (5)")\
   X(M1_SIGABRT  , = SIGABRT  , "SIGABRT",   "abort signal(=SIGIOT)(6)")\
   X(M1_SIGBUS   , = SIGBUS   , "SIGBUS",    "bus error (7)")\
   X(M1_SIGFPE   , = SIGFPE   , "SIGFPE",    "floating point exception (8)")\
   X(M1_SIGKILL  , = SIGKILL  , "SIGKILL",   "kill signal (9)")\
   X(M1_SIGUSR1  , = SIGUSR1  , "SIGUSR1",   "user signal 1 (10)")\
   X(M1_SIGSEGV  , = SIGSEGV  , "SIGSEGV",   "invalid memory reference (11)")\
   X(M1_SIGUSR2  , = SIGUSR2  , "SIGUSR2",   "user signal 2 (12)")\
   X(M1_SIGPIPE  , = SIGPIPE  , "SIGPIPE",   "broken pipe (13)")\
   X(M1_SIGALRM  , = SIGALRM  , "SIGALRM",   "timer signal (14)")\
   X(M1_SIGTERM  , = SIGTERM  , "SIGTERM",   "termination signal (15)")\
   X(M1_SIGSTKFLT, = SIGSTKFLT, "SIGSTKFLT", "stack fault on coprocessor (16)")\
   X(M1_SIGCHLD  , = SIGCHLD  , "SIGCHLD",   "child stopped or terminated (17)")\
   X(M1_SIGCONT  , = SIGCONT  , "SIGCONT",   "continue if stopped (18)")\
   X(M1_SIGSTOP  , = SIGSTOP  , "SIGSTOP",   "stop process (19)")\
   X(M1_SIGTSTP  , = SIGTSTP  , "SIGTSTP",   "terminal stop (20)")\
   X(M1_SIGTTIN  , = SIGTTIN  , "SIGTTIN",   "background terminal input (21)")\
   X(M1_SIGTTOU  , = SIGTTOU  , "SIGTTOU",   "background terminal output (22)")\
   X(M1_SIGURG   , = SIGURG   , "SIGURG",    "urgent socket condition (23)")\
   X(M1_SIGXCPU  , = SIGXCPU  , "SIGXCPU",   "cpu time limit exceeded (24)")\
   X(M1_SIGXFSZ  , = SIGXFSZ  , "SIGXFSZ",   "file size limit exceeded (25)")\
   X(M1_SIGVTALRM, = SIGVTALRM, "SIGVTALRM", "virtual alarm clock (26)")\
   X(M1_SIGPROF  , = SIGPROF  , "SIGPROF",   "profiling timer expired (27)")\
   X(M1_SIGWINCH , = SIGWINCH , "SIGWINCH",  "window resize (28)")\
   X(M1_SIGIO    , = SIGIO    , "SIGIO",     "io now possible(=SIGPOLL=SIGLOST)(29)")\
   X(M1_SIGPWR   , = SIGPWR   , "SIGPWR",    "power failure (30)") \
   X(M1_SIGSYS   , = SIGSYS   , "SIGSYS",    "bad argument to sytemcall(=SIGUNUSED)(31)")
   XENUM4_IMPL(signal_e)
#undef XENUM4

// -----------------------------------------------------------------------------

struct signal_t : public single_t<signal_t> {

   // --------------------------------------------------------------------------

   friend struct single_t<signal_t>;

   // --------------------------------------------------------------------------

   typedef void_t (*handler_t)(int_t, int_t, int_t);

   // --------------------------------------------------------------------------

   signal_t() {
      tracec("signal:ctor");
   }
   ~signal_t() {
      tracec("signal:dtor");
   }

   // --------------------------------------------------------------------------

   static void call(int sig0, siginfo_t *si, void */*arg*/) {
      signal_t &ins = signal_t::inst();
      if(ins.handler) {
         (*ins.handler)((int_t)sig0, (int_t)si->si_pid, (int_t)si->si_uid);
      }
   }
   int_t reg(signal_e const sig0, handler_t handler0) {
      struct sigaction sigact0;
      struct sigaction sigact1;
      mset(&sigact1, 0, sizeof(sigact1));
      sigemptyset(&sigact1.sa_mask);
      sigact1.sa_sigaction = call;
      sigact1.sa_flags     = SA_SIGINFO;
      if(sigaction((int)sig0, &sigact1, &sigact0)) { // e.g. M1_SIGSEGV;
         tracec("signal:err:", err_str());
         return -1;
      }
      tracec("signal:reg:", estr2(sig0));
      handler = handler0;
      return 0;
   }

   // --------------------------------------------------------------------------
   // data;

   handler_t handler;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// process;

struct proc_t {
   enum type_e {
      type_fork,
      type_spawn,
      type_last
   };
   proc_t() : ini(false), pid(-1), time(-1) {
   }
   proc_t(string_t const& proc0, int_t const& prio0, string_t const& arg0) :
      ini(true), pid(-1), prio(-1), time(-1) {
      init(proc0, prio0, arg0);
   }
   ~proc_t() {
      if(ini) fini();
   }
   void_t init(string_t const& proc0, int_t const& prio0, string_t const& arg0) {
      asrt(-1 == pid);
      asrt(proc0.size());
      proc = proc0;
      prio = prio0;
      arg = arg0;
      time = -1;
   }
   void_t fini() const {
      //asrt(-1 != pid);
   }
   int_t find() const {
      string_t name0 = proc.name(); // remove basepath;
      dir_t dir("/proc");
      if(0 == dir.fd) {
         tracec("proc:find:dir:init:failed");
         return -1;
      }
      string_t ent;
      while(0 == dir.read(ent)) {
         if(false == ent.isuint()) continue;
         string_t buf("/proc/");
         buf << ent << "/stat";
         file_t file(buf, "r");
         file.open();
         if(file.fd) {
            int pid1;
            string_t name1(128);
            byte_t state;
            if(3 != (file.scan("%d (%[^)]) %c", &pid1, name1.at(), &state))) {
               continue;
            }
            if(name1.upto(14) == name0.upto(14)) {
               tracec("proc:find:found:name0:", name0, " pid:", pid1);
               // parse commandline;
               string_t buf1("/proc/");
               buf1 << ent << "/cmdline";
               tracec("proc:find:read:", buf1);
               file_t file1(buf1, "r");
               file1.open();
               if(file1.fd) {
                  string_t cmd(256);
                  int_t const len = file1.read(cmd.at(), cmd.size()); (void_t)len;
                  tracec("proc:find:size:", file1.size(), " length:", len);
                  cmd.replace_f('\0', ' ');
                  //cmd.trim(string_t(" "));
                  string_t pat(" ");
                  int_t const lastid = cmd.id(cmd.findnof_b(pat));
                  if(lastid < cmd.id(cmd.z)) cmd.remove(cmd.from(lastid + 1));
                  string_t name2(cmd.upto(cmd.id(cmd.find_b(' ')) - 1));
                  string_t arg2 (cmd.from(cmd.id(cmd.find_b(' ')) + 1));
                  tracec("proc:find:proc:", proc);
                  tracec("proc:find:name2:", name2, " arg2:", arg2);
                  if(proc.name() == name2.name()) {
                     if(arg == arg2) {
                        tracec("proc:find:found:name:", name2, " arg:", arg2);
                        return pid1;
                     } else {
                        tracec("proc:find:arg:mismatch:", arg, " vs. ", arg2);
                     }
                  }
               }
            }
         }
      }
      tracec("proc:find:gone:proc:", name0, " arg:", arg);
      return -1;
   }
   int_t start(type_e const type0 = type_fork, int_t &found0 = null_t<int_t>::def()) {
      asrt(-1 == pid);
      lock_t<smp_t> lock(arg);
      scope_t<lock_t<smp_t>> sl(lock);
      if(-1 != (pid = find())) {
         tracec("proc:start:running:pid:", pid, " proc:", proc.name(), " arg:", arg);
         found0 = true;
         return pid;
      }
      found0 = false;
      tracec("proc:spawn:%s", proc.at());
      smp_t smp(string_t("<", arg, ".start>"));
      tracec("proc:start:smp:", smp.str());
      vector_t<byte_t*> argv1;
      argv1[0] = proc.at();
      argv1[1] = arg.at();
      argv1[2] = 0; // zero terminate;
      //for(int_t i = 0; i < argv.size(); ++i) {
      //   argv1[i] = &argv[i][0];
      //}
      //argv1[argv.size()] = 0; // zero terminate;
      //vector_t<byte_t*> argv1 = {&argv[0][0], 0};
      //TODO: add attr & env;
      time = normal_t::inst().ticks();
      int pid0;
      int_t res = 0;
      if(type_fork == type0) {
         pid0 = fork(); // duplicate this proc;
         if(0 == pid0) { // child;
            tracec("proc:fork:child");
            execvp(proc.at(), &argv1[0]); // execute given binary;
            tracec("proc:execvp:err:", err_str()); // only return on error;
            abort(); // _exit(-1);
         } else if(0 < pid0) { // parent;
            tracec("proc:fork:parent");
         } else {
            tracec("proc:fork:error:", err_str());
         }
      } else {
         res = posix_spawnp(&pid0, proc.at(), 0, 0, &argv1[0], 0);
      }
      pid = (int_t)pid0;
      tracec("proc:start:spawn:pid:", pid, 0 == res ? " success" : " fail");
      if(0 != res) {
         thr_t::wait(1000);
      }
      asrt(0 == res);

      // wait for server start;
      int_t const timeout = 10000;
      res = smp.wait(timeout);
      time = normal_t::inst().ticks() - time;
      if(smp_t::timeout == res) {
         tracec("proc:start:failed timeout:", timeout, " msec");
         return -1;
      } else if(smp_t::ok == res) {
         tracec("proc:start:time:", time, " msec pid:", pid);
         return pid;
      }
      return -2;
   }
   int_t shutdown() {
      lock_t<smp_t> lock(arg);
      scope_t<lock_t<smp_t>> sl(lock);
      if(-1 == (pid = find())) {
         tracec("proc:shutdown:notfound:proc:", proc.name(), " arg:", arg);
         return -1;
      }
      tracec("proc:shutdown:pid:", pid, " proc:", proc.name(), " arg:", arg);
      smp_t smp(string_t("<", arg, ".shut>"));
      tracec("proc:shutdown:smp:", smp.str());
      time = normal_t::inst().ticks();
      int_t res = shut(pid, SIGINT);
      if(0 == res) {
         // wait for server shutdown;
         int_t const timeout = 10000;
         res = smp.wait(timeout);
         time = normal_t::inst().ticks() - time;
         if(smp_t::ok == res) {
            tracec("proc:shutdown:time:", time, " msec pid:", pid);
            return 0;
         }
         if(smp_t::timeout == res) {
            tracec("proc:shutdown:failed timeout:", timeout, " msec");
         }
      } else {
         tracec("proc:shutdown:kill:failed:res:", res, " err:", err_str());
      }
      return -1;
   }

   // --------------------------------------------------------------------------

   static int_t shut(int_t pid0 = -1, int_t sig0 = -1) {
      pid0 = -1 == pid0 ? curpid() : pid0;
      sig0 = -1 == sig0 ? SIGKILL  : sig0;
      int_t const res = kill((int)pid0, (int)sig0);
      if(-1 == res) {
         tracec("proc:shut:err:", err_str());
      }
      return res;
   }

   // --------------------------------------------------------------------------

   int_t ini;
   int_t pid;
   string_t proc;
   int_t prio;
   //vector_t<string_t> argv;
   string_t arg;
   int_t time;

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
// types;

namespace types {

   // --------------------------------------------------------------------------

   // registration state; type ids start from 0 upwards;
   #define XENUM3(X) \
      X(registering,  = -1, "registering")\
      X(unregistered, = -2, "unregistered")
      XENUM3_IMPL(reg_e)
   #undef XENUM3

   // --------------------------------------------------------------------------

   struct fmt_t {
      fmt_t(int_t nfo0 = 3, int_t val0 = 0, int_t lvl0 = M1_MAXINT64) :
         nfo(nfo0), val(val0), lvl(lvl0) {
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            default:
            case 0:
               return string_t(nfo, val, lvl);
               break;
            case 1:
               return string_t("nfo:", nfo, " val:", val, " lvl:", lvl);
               break;
         }
      }
      int_t nfo; // 0 = empty; 1 = ids; 2 = names; 3 = ids & names;
      int_t val; // 0 = off; 1 = on;
      int_t lvl; // 0 = type only; default: M1_MAXINT64 = all levels;
   };

   // --------------------------------------------------------------------------
   // type info;

   struct info_t {
      info_t() : id(-1) {
      }
      info_t(int_t const id0, string_t const &name0) :
         id(id0), name(name0) {
      }
      string_t &cat(string_t &s, fmt_t fmt) const {
         switch(fmt.nfo) {
            case 0: break;
            case 1: s.cat(id); break;
            case 2: s.cat(name); break;
            case 3: s.cat(id, ":", name); break;
            default: asrt(0);
         }
         return s;
      }
      string_t &proc(string_t &s) {
         s.ext(string_t::sep()); // '(';
         s.ext(id, string_t::sep()); // "%lld:";
         name = s.range(s.x, s.find_f(')') - 1);
         s.x += name.size();
         s.ext(string_t::sep()); // ')';
         return s;
      }
      string_t &ext(string_t &s) {
         byte_t *cb = s.find_f('{'); // curly brace;
         byte_t *ab = s.find_b('>'); // angle brace;
         s.ext(id, string_t::sep()); // "%lld:";
         if(0 == cb && 0 == ab) {
            // 0:int_t; native;
            name = s.from();
         } else if(0 == ab || cb < ab) {
            // 2:nest_t{0:int_t}; composite;
            name = s.range(s.x, cb - 1);
         } else if(0 == cb || ab < cb) {
            // 3:vector_t<0:int_t>; predicate;
            name = s.range(s.x, ab);
         } else {
            asrt(0);
         }
         s.x += name.size();
         return s;
      }
      string_t str(fmt_t fmt = fmt_t()) const {
         string_t s;
         return cat(s, fmt);
      }
      int_t id;
      string_t name;
   };

   // --------------------------------------------------------------------------
   // type construction;

   struct face_t; // forward;

   struct collect_t {
      collect_t() {
      }
      virtual void_t regtype() = 0;
      vector_t<face_t*, false> v;
   };

   struct face_t {
      face_t() {
      }
      virtual string_t &cat(string_t &, fmt_t fmt) const = 0;
      virtual string_t &ext(string_t &, fmt_t fmt) = 0;
      virtual void_t copy(face_t *) = 0;
      virtual int_t compare(face_t *) const = 0;
   };

   //---------------------------------------------------------------------------
   // shared type registration;

   struct shared_t : public single_t<shared_t> {

      //------------------------------------------------------------------------

      friend struct single_t<shared_t>;

      //------------------------------------------------------------------------

      shared_t() : scopediver(8, 0) {
         tracec("types:shared:ctor");
         init();
      }
      ~shared_t() {
         fini();
         tracec("types:shared:dtor");
      }

      //------------------------------------------------------------------------

      void_t init() {
         tracec("types:shared:init");
         // initialize lockfile;
         lf.init(string_t(M1_PATH, M1_SCOPE, ".lock"));
         tracec("types:shared:init:lockfile:", lf.str());
         // initialize typefile;
         tf.init(string_t(M1_PATH, M1_SCOPE, ".types"));
         tracec("types:shared:init:typefile:", tf.str());
      }

      void_t fini() {
         while(tf.depth) {
            tf.exit();
         }
         tf.fini();
         while(lf.depth) {
            lf.exit();
         }
         lf.fini();
         tracec("types:shared:fini");
      }

      // -----------------------------------------------------------------------
      // local type registration;

      template<typename T>
      int_t id(T const &t) {
         #if defined(M1_FEAT_PROC_DEBUG)
            tracec("types:id0");
         #endif
         static int_t id_ = unregistered; // -2;
         #if defined(M1_FEAT_PROC_DEBUG)
            tracec("types:id1:", id_);
         #endif
         if(unregistered != id_) {
            #if defined(M1_FEAT_PROC_DEBUG)
               tracec("types:id2:", id_);
            #endif
            return id_; // return temporary;
         }
         id_ = registering;
         #if defined(M1_FEAT_PROC_DEBUG)
            tracec("types:id3:", id_);
         #endif
         id_ = reg(t); // shared or new id;
         #if defined(M1_FEAT_PROC_DEBUG)
            tracec("types:id4:", id_);
         #endif
         return id_;
      }

      // -----------------------------------------------------------------------
      // shared type registration;

      template<typename T>
      int_t reg(T const &t) { // once per type;
         //tracec("types:reg:tf:", shared_t::inst().tf.str(), " whoami:", (int_t)thr_t::whoami());
         asrt(shared_t::inst().tf.pivot.fd!=0); // type file opened;
         scope_t<lock_t<file_t>> sc(shared_t::inst().tf); // share remote;
         info_t pnfo0(curpid(), execname().name()); // local process info;
         tracec("types:reg:pnfo:(", pnfo0.str(), ")");
         string_t type0; // registree type;
         cat(type0, t, fmt_t(3, 0, std::is_base_of<collect_t, T>::value ? 1 : 0));
         info_t nfo0; // registree info;
         nfo0.ext(type0);
         strings_t lines; // shared types;
         shared_t::inst().tf.pivot.readlines(lines);
         for(int_t i = 0; i < lines.size(); ++i) {
            info_t pnfo1; // shared process info;
            pnfo1.proc(lines[i]);
            string_t type1 = lines[i].from(); // registered type;
            info_t nfo1; // registered info;
            nfo1.ext(type1);
            if(nfo0.name == nfo1.name) { // name in use;
               string_t const entry = type0.from();
               if(entry == type1.from()) {
                  tracec("types:reg:", nfo0.name, " id:", nfo1.id, " known");
                  return nfo1.id; // share known;
               } else { // type content conflict;
                  if(pnfo0.name == pnfo1.name) { // same process name;
                     if(pnfo0.id == pnfo1.id) { // same process id;
                        // identically named local type with different content;
                        regtrc(pnfo0, type0, pnfo1, type1, "err:scope clash");
                        asrt(0);
                     } else { // other process id; assume build;
                        string_t entry('(', pnfo0.str(), ')', nfo1.id, ':', nfo0.name, type0.from());
                        lines[i] = entry; // update;
                        shared_t::inst().tf.pivot.clear();
                        shared_t::inst().tf.pivot.writelines(lines);
                        regtrc(pnfo0, type0, pnfo1, type1, "update");
                        return nfo1.id;
                     }
                  } else { // different process name;
                     // identically named remote type with different content;
                     regtrc(pnfo0, type0, pnfo1, type1, "err:remote clash");
                     asrt(0);
                  }
               }
            }
         }
         int_t const id = lines.size(); // share new;
         string_t entry('(', pnfo0.str(), ')', id, ':', nfo0.name, type0.from());
         shared_t::inst().tf.pivot.writeline(entry);
         tracec("types:reg:entry:", entry, " id:", id, " new");
         return id;
      }

      // -----------------------------------------------------------------------

      void_t regtrc(info_t const &pnfo0, string_t const &type0,
                    info_t const &pnfo1, string_t const &type1,
                    string_t const &txt) {
         tracec("types:regtrc:",  txt);
         tracec("types:regtrc:(", pnfo0.str(), ")", type0, " vs.");
         tracec("types:regtrc:(", pnfo1.str(), ")", type1);
      }

      //------------------------------------------------------------------------
      // data;

      vector_t<collect_t*, false> scopediver;

      lock_t<file_t> lf; // lockfile;
      lock_t<file_t> tf; // typefile;

      //------------------------------------------------------------------------
   };

   //---------------------------------------------------------------------------
   // forward;

   template<typename T>
   static string_t name(T const &t);
   template<typename T>
   static string_t &cat(string_t &s, T const &t, fmt_t fmt);
   template<typename T>
   static string_t &ext(string_t &s, T &t, fmt_t fmt);

   // --------------------------------------------------------------------------
   // byte_t;

   static string_t name(byte_t const &) {
      return "byte_t";
   }
   static string_t &cat(string_t &s, byte_t const &t) {
      return s << t;
   }
   static string_t &ext(string_t &s, byte_t &t) {
      return s >> t;
   }

   // --------------------------------------------------------------------------
   // int_t;

   static string_t name(int_t const &) {
      return "int_t";
   }
   static string_t &cat(string_t &s, int_t const &t) {
      return s << t;
   }
   static string_t &ext(string_t &s, int_t &t) {
      return s >> t;
   }

   // --------------------------------------------------------------------------
   // flt_t;

   static string_t name(flt_t const &) {
      return "flt_t";
   }
   static string_t &cat(string_t &s, flt_t const &t) {
      return s << t;
   }
   static string_t &ext(string_t &s, flt_t &t) {
      return s >> t;
   }

   // --------------------------------------------------------------------------
   // base helpers;

   template<typename T, typename = void_t>
   struct name_helper {
      static string_t name(T const &) {
         tracec("types::name_helper:not implemented");
         asrt(0);
      }
   };
   template<typename T, typename = void_t>
   struct cat_helper {
      static string_t &cat(string_t &, T const &t, fmt_t fmt) {
         tracec("types::cat_helper:", name(t), " not implemented");
         asrt(0);
      }
   };
   template<typename T, typename = void_t>
   struct ext_helper {
      static string_t &cat(string_t &, T &t, fmt_t fmt) {
         tracec("types::ext_helper:", name(t), " not implemented");
         asrt(0);
      }
   };

   // --------------------------------------------------------------------------
   // arithmetic helpers;

   template<typename T>
   struct name_helper<T, typename std::enable_if<std::is_arithmetic<T>::value>::type> {
      static string_t name(T const &t) {
         return types::name(t);
      }
   };
   template<typename T>
   struct cat_helper<T, typename std::enable_if<std::is_arithmetic<T>::value>::type> {
      static string_t &cat(string_t &s, T const &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t(shared_t::inst().id(t), types::name(t)).cat(s, fmt);
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.cat(':');
         types::cat(s, t);
         return s;
      }
   };
   template<typename T>
   struct ext_helper<T, typename std::enable_if<std::is_arithmetic<T>::value>::type> {
      static string_t &ext(string_t &s, T &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t info(types::shared_t::inst().id(t), types::name(t));
            s.x += info.str(fmt).size();
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.ext(string_t::sep()); // ':';
         return types::ext(s, t);
      }
   };

   // --------------------------------------------------------------------------
   // enum helpers;

   template<typename T>
   struct name_helper<T, typename std::enable_if<std::is_enum<T>::value>::type> {
      static string_t name(T const &) {
         return "enum_t";
      }
   };
   template<typename T>
   struct cat_helper<T, typename std::enable_if<std::is_enum<T>::value>::type> {
      static string_t &cat(string_t &s, T const &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t(types::shared_t::inst().id(t), types::name(t)).cat(s, fmt);
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.cat(':');
         return types::cat(s, (int_t)t);
      }
   };
   template<typename T>
   struct ext_helper<T, typename std::enable_if<std::is_enum<T>::value>::type> {
      static string_t &ext(string_t &s, T &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            s.x += info_t(types::shared_t::inst().id(t), types::name(t)).str(fmt).size();
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.ext(string_t::sep());
         return types::ext(s, *(int_t*)&t);
      }
   };

   // --------------------------------------------------------------------------
   // pointer helpers;

   template<typename T>
   struct name_helper<T*> {
      static string_t name(T* const &) {
         return "void_t*";
      }
   };
   template<typename T>
   struct cat_helper<T*> {
      static string_t &cat(string_t &s, T* const &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t(types::shared_t::inst().id(t), types::name(t)).cat(s, fmt);
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.cat(':');
         return s.cat((void_t*)t);
      }
   };
   template<typename T>
   struct ext_helper<T*> {
      static string_t &ext(string_t &s, T* &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t info(types::shared_t::inst().id(t), types::name(t));
            s.x += info.str(fmt).size();
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.ext(string_t::sep()); // ':';
         return s.ext(t);
      }
   };

   // --------------------------------------------------------------------------
   // collect_t helpers;

   template<typename T>
   struct name_helper<T, typename std::enable_if<std::is_base_of<collect_t, T>::value>::type> {
      static string_t name(T const &t) {
         return t.name();
      }
   };
   template<typename T>
   struct cat_helper<T, typename std::enable_if<std::is_base_of<collect_t, T>::value>::type> {
      static string_t &cat(string_t &s, T const &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t(types::shared_t::inst().id(t), types::name(t)).cat(s, fmt);
         }
         if(0 > --fmt.lvl) return s;
         if(0 == fmt.nfo && 0 == fmt.val) return s;
         s.cat('{');
         for(auto r = t.v.range(); r.x; ++r) {
            (*r.x)->cat(s, fmt_t(fmt.nfo, fmt.val, fmt.lvl));
            if(r.x != r.z) s.cat(';');
         }
         return s.cat("}");
      }
   };
   template<typename T>
   struct ext_helper<T, typename std::enable_if<std::is_base_of<collect_t, T>::value>::type> {
      static string_t &ext(string_t &s, T &t, fmt_t fmt) {
         t = T();
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            s.x += info_t(types::shared_t::inst().id(t), types::name(t)).str(fmt).size();
         }
         if(0 > --fmt.lvl) return s;
         if(0 == fmt.nfo && 0 == fmt.val) return s;
         s.ext(string_t::sep()); // '{';
         for(vector_t<face_t*, false> r = t.v.range(); r.x; ++r) {
            (*r.x)->ext(s, fmt_t(fmt.nfo, fmt.val, fmt.lvl));
            if(r.x != r.z) s.ext(string_t::sep());
         }
         return s.ext(string_t::sep()); // '}';
      }
   };

   // --------------------------------------------------------------------------
   // unary impl (vector_t, list_t, set_t, multiset_t);

   template<typename T>
   static string_t &cat_unary(string_t &s, T const &t, fmt_t fmt, byte_t const sep0, byte_t const sep1) {
      if(0 > fmt.lvl) return s;
      if(0 < fmt.nfo) {
         info_t(types::shared_t::inst().id(t), types::name(t)).cat(s, fmt_t(fmt.nfo, 0, fmt.lvl));
         s.cat('<');
         types::cat(s, null_t<typename T::obj_t>::def(), fmt_t(fmt.nfo, 0, fmt.lvl)); // recurse;
         s.cat('>');
      }
      if(0 > fmt.lvl) return s;
      if(1 == fmt.val) {
         if(fmt.nfo) {
            s.cat(':');
         }
         s.cat(t.size(), sep0);
         for(auto r = t.range(); r.x; ++r) {
            types::cat(s, *r, fmt_t(0, 1, fmt.lvl));
            if(r.x != r.z) s.cat(',');
         }
         s.cat(sep1);
      }
      return s;
   }

   template<typename T>
   static string_t &ext_unary(string_t &s, T &t, fmt_t fmt) {
      t.clear();
      if(0 > fmt.lvl) return s;
      if(0 < fmt.nfo) {
         string_t type;
         types::cat(type, t, fmt_t(fmt.nfo, 0, fmt.lvl));
         s.x += type.size();
      }
      if(0 > fmt.lvl) return s;
      if(1 == fmt.val) {
         if(fmt.nfo) {
            s.ext(string_t::sep()); // ':';
         }
         int_t size = 0;
         s.ext(size, string_t::sep()); // '%lld[';
         typename T::obj_t obj;
         for(int_t i = 0; i < size; ++i) {
            types::ext(s, obj, fmt_t(0, 1, fmt.lvl));
            t.append(obj);
            if(i < size - 1) s.ext(string_t::sep()); // ',';
         }
         s.ext(string_t::sep()); // ']';
      }
      return s;
   }

   // --------------------------------------------------------------------------
   // vector_t helpers;

   template<typename T, byte_t deep, typename mem_T>
   struct name_helper<vector_t<T, deep, mem_T>> {
      static string_t name(vector_t<T, deep, mem_T> const &) {
         return string_t("vector_t");
      }
   };
   template<typename T, byte_t deep, typename mem_T>
   struct cat_helper<vector_t<T, deep, mem_T>> {
      static string_t &cat(string_t &s, vector_t<T, deep, mem_T> const &t, fmt_t fmt) {
         return cat_unary(s, t, fmt, '[', ']');
      }
   };
   template<typename T, byte_t deep, typename mem_T>
   struct ext_helper<vector_t<T, deep, mem_T>> {
      static string_t &ext(string_t &s, vector_t<T, deep, mem_T> &t, fmt_t fmt) {
         return ext_unary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // list_t helpers;

   template<typename T>
   struct name_helper<list_t<T>> {
      static string_t name(list_t<T> const &) {
         return string_t("list_t");
      }
   };
   template<typename T>
   struct cat_helper<list_t<T>> {
      static string_t &cat(string_t &s, list_t<T> const &t, fmt_t fmt) {
         return cat_unary(s, t, fmt, '(', ')');
      }
   };
   template<typename T>
   struct ext_helper<list_t<T>> {
      static string_t &ext(string_t &s, list_t<T> &t, fmt_t fmt) {
         return ext_unary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // set_t helpers;

   template<typename T>
   struct name_helper<set_t<T>> {
      static string_t name(set_t<T> const &) {
         return string_t("set_t");
      }
   };
   template<typename T>
   struct cat_helper<set_t<T>> {
      static string_t &cat(string_t &s, set_t<T> const &t, fmt_t fmt) {
         return cat_unary(s, t, fmt, '[', ']');
      }
   };
   template<typename T>
   struct ext_helper<set_t<T>> {
      static string_t &ext(string_t &s, set_t<T> &t, fmt_t fmt) {
         return ext_unary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // multiset_t helpers;

   template<typename T>
   struct name_helper<multiset_t<T>> {
      static string_t name(multiset_t<T> const &) {
         return string_t("multiset_t");
      }
   };
   template<typename T>
   struct cat_helper<multiset_t<T>> {
      static string_t &cat(string_t &s, multiset_t<T> const &t, fmt_t fmt) {
         return cat_unary(s, t, fmt, '[', ']');
      }
   };
   template<typename T>
   struct ext_helper<multiset_t<T>> {
      static string_t &ext(string_t &s, multiset_t<T> &t, fmt_t fmt) {
         return ext_unary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // string_t helpers;

   template<>
   struct name_helper<string_t> {
      static string_t name(string_t const &) {
         return string_t("string_t");
      }
   };
   template<>
   struct cat_helper<string_t> {
      static string_t &cat(string_t &s, string_t const &t, fmt_t fmt) {
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t(shared_t::inst().id(t), name(t)).cat(s, fmt);
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.cat(':');
         s.cat(t.size(), ':');
         return s << t;
      }
   };
   template<>
   struct ext_helper<string_t> {
      static string_t &ext(string_t &s, string_t &t, fmt_t fmt) {
         t.clear();
         if(0 > fmt.lvl) return s;
         if(0 < fmt.nfo) {
            info_t info(shared_t::inst().id(t), name(t));
            s.x += info.str(fmt).size();
         }
         if(0 == fmt.val) return s;
         if(fmt.nfo) s.ext(string_t::sep()); // ':';
         int_t size = 0;
         s.ext(size, string_t::sep()); // '%lld[';
         t = s.range(s.x, s.x + size - 1);
         s.x += size;
         return s;
      }
   };

   // --------------------------------------------------------------------------
   // binary impl (map_t, mutimap_t);

   template<typename T>
   static string_t &cat_binary(string_t &s, T const &t, fmt_t fmt, byte_t const sep0, byte_t const sep1) {
      if(0 > fmt.lvl) return s;
      if(0 < fmt.nfo) {
         info_t(types::shared_t::inst().id(t), types::name(t)).cat(s, fmt_t(fmt.nfo, 0, fmt.lvl));
         s.cat('<');
         types::cat(s, null_t<typename T::key_t>::def(), fmt_t(fmt.nfo, 0, fmt.lvl)); // recurse;
         s.cat(',');
         types::cat(s, null_t<typename T::obj_t>::def(), fmt_t(fmt.nfo, 0, fmt.lvl)); // recurse;
         s.cat('>');
      }
      if(0 > fmt.lvl) return s;
      if(1 == fmt.val) {
         if(fmt.nfo) {
            s.cat(':');
         }
         s.cat(t.size(), sep0);
         for(auto r = t.range(); r.x; ++r) {
            types::cat(s, r.x->key, fmt_t(0, 1, fmt.lvl));
            s.cat('~');
            types::cat(s, r.x->obj, fmt_t(0, 1, fmt.lvl));
            if(r.x != r.z) s.cat(',');
         }
         s.cat(sep1);
      }
      return s;
   }

   template<typename T>
   static string_t &ext_binary(string_t &s, T &t, fmt_t fmt) {
      t.clear();
      if(0 > fmt.lvl) return s;
      if(0 < fmt.nfo) {
         string_t type;
         types::cat(type, t, fmt_t(fmt.nfo, 0, fmt.lvl));
         s.x += type.size();
      }
      if(0 > fmt.lvl) return s;
      if(1 == fmt.val) {
         if(fmt.nfo) {
            s.ext(string_t::sep()); // ':';
         }
         int_t size = 0;
         s.ext(size, string_t::sep()); // '%lld[';
         typename T::key_t key;
         typename T::obj_t obj;
         for(int_t i = 0; i < size; ++i) {
            types::ext(s, key, fmt_t(0, 1, fmt.lvl));
            s.ext(string_t::sep());
            types::ext(s, obj, fmt_t(0, 1, fmt.lvl));
            t.insert(key, obj);
            if(i < size - 1) s.ext(string_t::sep()); // ',';
         }
         s.ext(string_t::sep()); // ']';
      }
      return s;
   }

   // --------------------------------------------------------------------------
   // map_t helpers;

   template<typename T, typename U>
   struct name_helper<map_t<T, U>> {
      static string_t name(map_t<T, U> const &) {
         return string_t("map_t");
      }
   };
   template<typename T, typename U>
   struct cat_helper<map_t<T, U>> {
      static string_t &cat(string_t &s, map_t<T, U> const &t, fmt_t fmt) {
         return cat_binary(s, t, fmt, '[', ']');
      }
   };
   template<typename T, typename U>
   struct ext_helper<map_t<T, U>> {
      static string_t &ext(string_t &s, map_t<T, U> &t, fmt_t fmt) {
         return ext_binary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // multimap_t helpers;

   template<typename T, typename U>
   struct name_helper<multimap_t<T, U>> {
      static string_t name(multimap_t<T, U> const &) {
         return string_t("multimap_t");
      }
   };
   template<typename T, typename U>
   struct cat_helper<multimap_t<T, U>> {
      static string_t &cat(string_t &s, multimap_t<T, U> const &t, fmt_t fmt) {
         return cat_binary(s, t, fmt, '[', ']');
      }
   };
   template<typename T, typename U>
   struct ext_helper<multimap_t<T, U>> {
      static string_t &ext(string_t &s, multimap_t<T, U> &t, fmt_t fmt) {
         return ext_binary(s, t, fmt);
      }
   };

   // --------------------------------------------------------------------------
   // base templates;

   template<typename T>
   static string_t name(T const &t) {
      return name_helper<T>::name(t);
   }
   template<typename T>
   static string_t &cat(string_t &s, T const &t, fmt_t fmt) {
      return cat_helper<T>::cat(s, t, fmt);
   }
   template<typename T>
   static string_t &ext(string_t &s, T &t, fmt_t fmt) {
      return ext_helper<T>::ext(s, t, fmt);
   }

   // --------------------------------------------------------------------------

   template<typename T>
   string_t str(T const &t) {
      string_t s;
      cat(s, t, fmt_t(3, 1, M1_MAXINT64));
      return s;
   }
   template<typename T>
   void_t trace(T const &t) {
      tracec(str(t));
   }

   // --------------------------------------------------------------------------

} // types;

// -----------------------------------------------------------------------------

template<typename T = void_t>
struct nif_t : public types::collect_t {

   // --------------------------------------------------------------------------
   // ctor dtor;

   nif_t() : rhs(0) {
      open();
   }
   ~nif_t() {
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("nif:dtor:", types::name(null_t<T>::def()));
      #endif
   }

   // --------------------------------------------------------------------------
   // copy assign;

   nif_t(nif_t const &rhs0) {
      rhs = (nif_t*)&rhs0;
      open();
   }
   nif_t &operator= (nif_t const &rhs0) {
      rhs = (nif_t*)&rhs0;
      copy();
      return *this;
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t compare(nif_t const &rhs0) const {
      for(int_t i = 0; i < v.size(); ++i) {
         int_t const res = (*v.at(i))->compare(*rhs0.v.at(i));
         #if defined(M1_FEAT_PROC_DEBUG)
            tracec("nif:compare:i:", i, " res:", res);
         #endif
         if(0 != res) return res < 0 ? -1 : 1; // less : greater;
      }
      return 0; // equal;
   }
   int_t operator==(nif_t const &rhs0) const {
      return 0 == compare(rhs0);
   }
   int_t operator<(nif_t const &rhs0) const {
      return 0 > compare(rhs0);
   }

   int_t operator!=(nif_t const &rhs0) const {
      return !(*this == rhs0);
   }
   int_t operator>(nif_t const &rhs0) const {
      return rhs0 < *this;
   }
   int_t operator<=(nif_t const &rhs0) const {
      return !(rhs0 < *this);
   }
   int_t operator>=(nif_t const &rhs0) const {
      return !(*this < rhs0);
   }

   // --------------------------------------------------------------------------
   // access;

   T const &operator()() const { // parentheses operator;
      return *(T*)this;
   }
   T &operator()() {
      return *(T*)this;
   }
   operator T const&() const { // casting operator;
      return *(T*)this;
   }
   operator T &() {
      return *(T*)this;
   }

   // --------------------------------------------------------------------------

   void_t open() {
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("nif:open");
      #endif
      if(0 == types::shared_t::inst().tf.pivot.fd) {
         types::shared_t::inst().init();
      }
      int_t const depth = types::shared_t::inst().lf.enter();
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("nif:open:tf:", types::shared_t::inst().tf.str(), " depth:", depth);
      #endif
      collect_t **coll = types::shared_t::inst().scopediver.at(depth);
      if(coll) {
         if(*coll) {
            tracec("nif:wrong nesting");
            return; // wrong nesting;
         }
         *coll = this; // global access;
      } else {
         types::shared_t::inst().scopediver[depth] = this; // global access;
      }
   }
   void_t copy() {
      for(int_t i = 0; i < v.size(); ++i) {
          if(NULL != rhs) v[i]->copy(rhs->v[i]);
      }
      rhs = 0;
   }
   void_t regtype() {
      types::shared_t::inst().id(*(T*)this);
      if(rhs) copy();
   }
   nif_t *rhs;
};

template<>
struct nif_t<void_t> {
   nif_t() {
      close();
   }
   nif_t(nif_t const&) {
      close();
   }
   ~nif_t() {
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("nif:dtor");
      #endif
   }
   void_t close() {
      int_t const depth = types::shared_t::inst().lf.depth;
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("nif:close:depth:", depth);
      #endif
      types::collect_t **coll = types::shared_t::inst().scopediver.at(depth);
      asrt(coll && *coll); // wrong nesting;
      (*coll)->regtype();
      (*coll) = 0; // global restrict;
      types::shared_t::inst().lf.exit();
   }
};

// -----------------------------------------------------------------------------

template<typename T>
struct if_t : public types::face_t {

   // --------------------------------------------------------------------------
   // ctor dtor;

   if_t() {
      add();
   }
   if_t(T const &t0) : t(t0) {
      add();
   }
   ~if_t() {
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("if:dtor:", types::name(t));
      #endif
   }

   // --------------------------------------------------------------------------
   // copy assign;

   if_t(if_t const &) {
      add();
   }
   if_t &operator= (T const &t0) {
      t = t0;
      return *this;
   }

   // --------------------------------------------------------------------------
   // compare;

   int_t compare(face_t *face) const {
      if_t* p = (if_t*)face;
      if(t == p->t) return 0;
      return t < p->t ? -1 : 1;
   }

   // --------------------------------------------------------------------------

   int_t compare(if_t const &rhs) const {
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("if:compare:lhs:", t, " rhs:", rhs.t);
      #endif
      if(t == rhs.t) return 0;
      return t < rhs.t ? -1 : 1;
   }

   template<typename RHS>
   int_t operator==(RHS const &rhs) const {
      return 0 == compare(rhs);
   }
   template<typename RHS>
   int_t operator<(RHS const& rhs) const {
      return 0 > compare(rhs);
   }

   template<typename RHS>
   int_t operator!=(RHS const &rhs) const {
      return !(*this == rhs);
   }
   template<typename RHS>
   int_t operator>(RHS const& rhs) const {
      return rhs < *this;
   }
   template<typename RHS>
   int_t operator<=(RHS const& rhs) const {
      return !(rhs < *this);
   }
   template<typename RHS>
   int_t operator>=(RHS const& rhs) const {
      return !(*this < rhs);
   }

   // --------------------------------------------------------------------------
   // access;

   T const &operator()() const { // parentheses operator;
      return t;
   }
   T &operator()() {
      return t;
   }
   operator T const&() const {  // casting operator;
      return t;
   }
   operator T &() {
      return t;
   }

   // --------------------------------------------------------------------------

   void_t copy(face_t *rhs) {
      if(NULL != rhs){ t = ((if_t*)rhs)->t;}
   }
   void_t add() {
      int_t const depth = types::shared_t::inst().lf.depth;
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("if:add:tf:", types::shared_t::inst().tf.str(), " depth:", depth);
      #endif
      int_t const id = types::shared_t::inst().id(t);
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("if:add:id:got:", id);
      #endif
      types::collect_t **coll = types::shared_t::inst().scopediver.at(depth);
      if(0 == coll || 0 == *coll) {
         tracec("if:add:no collector");
         return; // no collector;
      }
      #if defined(M1_FEAT_PROC_DEBUG)
         tracec("if:add:append");
      #endif
      (*coll)->v.append(this);
   }
   string_t &cat(string_t &s, types::fmt_t fmt) const {
      return types::cat(s, t, fmt);
   }
   string_t &ext(string_t &s, types::fmt_t fmt) {
      return types::ext(s, t, fmt);
   }
   T t;
};

// -----------------------------------------------------------------------------
// specialisations;

//template<typename T>
//struct if_t : base_if_t<T> {
// typedef base_if_t<T> base_t;
// using base_t::base_t; // inherit ctors;
//   //if_t() : base_t() {
// // tracec("if_t");
//   //}
//   //if_t(if_t const &rhs) : base_t(rhs) {
// // tracec("if_t:copy");
//   //}
//   //if_t(T const &t0) : base_t(t0) {
// // tracec("if_t:val");
//   //}
//};
//
//template<typename T>
//struct if_t<vector_t<T>> : base_if_t<vector_t<T>> {
// typedef base_if_t<vector_t<T>> base_t;
// using base_t::base_t; // inherit ctors;
//   //if_t() : base_t() {
//   //}
//   //if_t(if_t const &rhs) : base_t(rhs) {
//   //}
//   //if_t(T const &t0) : base_t(t0) {
//   //}
//   T &operator[] (int_t const id0) {
//    return base_t::t.operator[](id0);
//   }
//};

#endif // M1_PROC_LINUX == M1_PROC;

// -----------------------------------------------------------------------------

#if (M1_PROC_WIN == M1_PROC)
   //TODO;
#endif // M1_PROC_WIN == M1_PROC;

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_PROC_NONE;

// -----------------------------------------------------------------------------
// com;

#if !(M1_COM_NONE == M1_COM)

// -----------------------------------------------------------------------------

#if (M1_LIB_LINUX == M1_LIB)
   #include <termios.h>
#endif

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------

#if (M1_COM_M1 == M1_COM)

struct msg_t : public string_t {
   enum : int_t {
      id_src = 0, // source node id;
      id_dst,     // destination node id;
      id_cid,     // content id;
      id_tim,     // timestamp;
      id_last,
      header_size = id_last * sizeof(int_t)
   };
   msg_t() {
      resize(header_size);
      stamp();
   }
   template<typename T>
   msg_t(T const &t) : msg_t() { // ctor delegation;
      //resize(header_size);
      //stamp();
      set(t);
   }
   msg_t(int_t const src0, int_t const dst0, int_t const cid0 = -1) {
      resize(header_size);
      stamp();
      sign(src0, dst0, cid0);
   }
   ~msg_t() {
   }

   // --------------------------------------------------------------------------

   int_t  src() const           { return ((int_t*)a)[id_src]; }
   void_t src(int_t const src0) { ((int_t*)a)[id_src] = src0; }
   int_t  dst() const           { return ((int_t*)a)[id_dst]; }
   void_t dst(int_t const dst0) { ((int_t*)a)[id_dst] = dst0; }
   int_t  cid() const
   {
       int_t val = 0;
       if(NULL != a)
       {
           val = ((int_t*)a)[id_cid];
       }
       return val;
   }
   void_t cid(int_t const cid0) { ((int_t*)a)[id_cid] = cid0; }
   int_t  tim() const           { return ((int_t*)a)[id_tim]; }
   void_t tim(int_t const tim0) { ((int_t*)a)[id_tim] = tim0; }

   // --------------------------------------------------------------------------

   void_t sign(int_t const src0,      int_t const dst0,
               int_t const cid0 = -1, int_t const tim0 = -1) {
      src(src0);
      dst(dst0);
      cid(cid0);
      tim(tim0);
   }
   void_t stamp() {
      tim(normal_t::inst().runtime());
   }

   // --------------------------------------------------------------------------
   // nif_t type serialisation;

   template<typename T>
   void_t get(T &t) {
      reset();
      types::ext(*this, t, types::fmt_t(3, 1, M1_MAXINT64));
   }
   template<typename T>
   void_t set(T const &t) {
      clear();
      types::cat(*this, t, types::fmt_t(3, 1, M1_MAXINT64));
   }

   // --------------------------------------------------------------------------

   void_t resize(int_t const size0, byte_t const val0 = '\0') {
      asrt(header_size <= size0);
      string_t::resize(size0, val0);
   }
   msg_t& swap() { // exchange sender and receiver (src & dst) in msg header;
      int_t const src0 = src();
      src(dst());
      dst(src0);
      return *this;
   }
   void_t reset() {
      string_t::x = string_t::a + header_size;
   }
   void_t clear() { // remove content
      resize(header_size);
   }

   // --------------------------------------------------------------------------

   string_t str(int_t const var = 0) const {
      switch(var) {
         default:
         case 0: {
            return string_t("size:", size(),
                            " src:", src(),
                            " dst:", dst(),
                            " cid:", cid(),
                            " tim:", tim());
         }
         case 1: {
           return string_t(str(0), " cont:\"", string_t::x + header_size, "\"");
         }
      }
   }

   // --------------------------------------------------------------------------
};

// -----------------------------------------------------------------------------
//// pod-msg serialisation;
//
//template<typename T>
//static void_t get(T &t0, msg_t const &msg) {
//   asrt(msg.size() - msg_t::header_size == sizeof(T));
//   mcpy(&t0, msg.at(msg_t::header_size), sizeof(T));
//}
//template<typename T>
//static void_t set(msg_t &msg, T const& t0) {
//   msg.resize(msg_t::header_size + sizeof(T));
//   mcpy(msg.at(msg_t::header_size), &t0, sizeof(T));
//}
//
// -----------------------------------------------------------------------------
//// string-msg serialisation;
//
//static void_t get(string_t &str0, msg_t const &msg) {
//   str0 = msg.from(msg_t::header_size);
//}
//static void_t set(msg_t &msg, string_t const &str0) {
//   msg.clear();
//   msg.append(str0); // check zero termination;
//}

// -----------------------------------------------------------------------------
// node;

struct node_t : public thr_t
{
   #define XENUM2(X) \
      X(mode_init,   "init")\
      X(mode_work,   "work")\
      X(mode_listen, "listen")
      XENUM2_IMPL(mode_e)
   #undef XENUM2

   #define XENUM3(X) \
      X(res_ok,          =   0, "ok")\
      X(res_fail,        = - 1, "fail")\
      X(res_timeout,     = - 2, "timeout")\
      X(res_full,        = - 3, "full")\
      X(res_empty,       = - 4, "empty")\
      X(res_gone,        = - 5, "gone")\
      X(res_unknown,     = - 6, "unknown")\
      X(res_first,       = - 7, "first")\
      X(res_last,        = - 8, "last")\
      X(res_request_con, = - 9, "request_con")\
      X(res_confirm_con, = -10, "confirm_con")\
      X(res_request_iso, = -11, "request_iso")\
      X(res_confirm_iso, = -12, "confirm_iso")
      XENUM3_IMPL(res_e)
   #undef XENUM3

   #define XENUM2(X) \
      X(queue_ctrl, "ctrl")\
      X(queue_data, "data")\
      X(queue_cnt,  "cnt")
      XENUM2_IMPL(queue_e)
   #undef XENUM2

   struct iqs_t {
      iqs_t() : ini(false), size_data(-1) {
      }
      iqs_t(string_t const& name0, int_t const size0 = -1, int_t const size1 = -1) :
         ini(true), size_data(-1) {
         init(name0, size0, size1);
      }
      ~iqs_t() {
         if(ini) fini();
      }
      void_t init(string_t const& name0, int_t const size0 = -1,
         int_t const size1 = -1) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:init:name:", name0, " size_ctrl:", size0, ", size_data:", size1);
         #endif
         smp.init(string_t("<", name0, ">.iqs"));
         if(-1 != size0 && size0 < msg_t::header_size + sizeof(int_t)) {
            tracec("node:iqs:init:ctrl:size0:", size0, " header:", (int_t)msg_t::header_size);
            thr_t::wait(1000);
            asrt(0);
         }
         qs[queue_ctrl].init(string_t("<", name0, ".iqs.ctrl>"), size0);
         if(size1) { // 2nd queue optional (-1 = connect; 0 = ignore);
            size_data = size1;
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:iqs:init:data");
            #endif
            qs[queue_data].init(string_t("<", name0, ".iqs.data>"), size1);
         }
         flag = 0;
      }
      void_t fini() {
         qs[queue_data].fini();
         qs[queue_ctrl].fini();
         size_data = -1;
         smp.fini();
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:fini:exit");
         #endif
      }
      int_t post(queue_e queue0, msg_t &msg0) {
         if(queue_data == queue0 && 0 == size_data) return -1;
         int_t const ticks0 = normal_t::inst().ticks();
         msg0.tim(ticks0);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:post:msg:", msg0.str());
         #endif
         int_t const bytes = qs[queue0].write(msg0.at(), msg0.size());
         if(0 == bytes) return 0; // full;
         smp.post();
         return bytes;
      }
      int_t wait(msg_t &msg0, int_t const timeout0 = smp_t::forever) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:wait:timeout:", timeout0, "...");
         #endif
         int_t const res = smp.wait(timeout0);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:wait:...received:res:", res);
         #endif
         if(smp_t::ok != res) return res;
         if(res_gone == flag) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:iqs:gone");
            #endif
            return res_gone;
         }
         int_t bytes = qs[queue_ctrl].next();
         if(0 < bytes) { // receive ctrl message;
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:iqs:wait:received:ctrl:bytes:", bytes);
            #endif
            msg0.resize(bytes);
            bytes = qs[queue_ctrl].read(msg0.at(), bytes);
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:iqs:wait:received:ctrl:msg:", msg0.str(), " elapsed:", ticks() - msg0.tim());
            #endif
            return bytes;
         }
         bytes = size_data ? qs[queue_data].next() : -1;
         if(0 < bytes) { // receive data message;
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:iqs:received:data:bytes:", bytes);
            #endif
            msg0.resize(bytes);
            bytes = qs[queue_data].read(&msg0[0], bytes);
            return bytes;
          }
          #if defined(M1_FEAT_NODE_DEBUG)
             tracec("node:iqs:queues:empty");
          #endif
          return res_empty; // queue(s) empty;
      }
      void_t flush() {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:iqs:flush");
         #endif
         qs[queue_ctrl].flush();
         if(0 < size_data) {
            qs[queue_data].flush();
         }
         smp.reset();
      }
      int_t refcnt() const {
         return qs[queue_ctrl].mem.refcnt();
      }
      string_t str() const {
         return string_t("size_data:", size_data,
                             " flag:", flag,
                             " ctrl:", qs[queue_ctrl].str(),
                             " data:", (size_data ? qs[queue_data].str() : string_t("void")));
      }
      int_t ini;
      smp_t smp;
      int_t size_data;
      ring_t<shm_t> qs[queue_cnt];
      int_t flag;
   };
   typedef map_t<int_t, iqs_t> cqs_t;
   typedef void_t (*disp_t)(msg_t&, res_e, void_t*);
   node_t() : nid(-1), disp(0), that(0), timeout(smp_t::forever),
      mode(mode_init), sid(-1) {
   }
   node_t(string_t const name0, int_t const size0, int_t const size1,
      mode_e const mode0, disp_t disp0 = 0, void_t* that0 = 0,
      int_t const timeout0 = smp_t::forever) :
      nid(-1), disp(0), that(0), timeout(smp_t::forever), mode(mode_init), sid(-1) {
      init(name0, size0, size1, mode0, disp0, that0, timeout0);
   }
   node_t(int_t const size0, int_t const size1, mode_e const mode0,
      disp_t disp0 = 0, void_t* that0 = 0,
      int_t const timeout0 = smp_t::forever) :
      nid(-1), disp(0), that(0), timeout(smp_t::forever), mode(mode_init), sid(-1) {
      init(size0, size1, mode0, disp0, that0, timeout0);
   }
   ~node_t() {
      if(ini) fini();
   }
   void_t init(int_t const size0, int_t const size1, mode_e const mode0,
      disp_t disp0 = 0, void_t* that0 = 0,
      int_t const timeout0 = smp_t::forever) {
      init(string_t(), size0, size1, mode0, disp0, that0, timeout0);
   }
   void_t init(string_t const name0, int_t const size0, int_t const size1,
      mode_e const mode0, disp_t disp0 = 0, void_t* that0 = 0,
      int_t const timeout0 = smp_t::forever) {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:init:nid:", nid, " name:", name0, " size0:", size0, " size1:", size1);
         tracec("node:init:mode:", estr2(mode0), " timeout:", timeout0);
      #endif
      if(-1 != nid) {
         thr_t::wait(1000);
         asrt(0); // prevent double init;
      }
      suid.init("<m1.uid>", 0); // deleted automatically when all nodes are gone;
      nid = suid.get(); // generate unique node id;
      if(name0.size()) {
         name = name0; // use given unique name;
      } else {
         name << "node" << nid; // build unique name from nid;
      }
      lock.init(string_t("<", name, ".lock>"));
      asrt(msg_t::header_size + sizeof(int_t) <= size0);
      disp = disp0;
      that = (0 == that0) ? this : that0;
      timeout = timeout0; // use given timeout for work() loop;
      conf.init(string_t("<", name, ".conf>"));
      dam.init(string_t("<", name, ".dam>"));
      iqs.init(name, size0, size1); // create local input queues;
      if(mode_work == mode0) {
         work(0); // in this thread;
      } else if(mode_listen == mode0) {
         listen(); // parallel;
      }
   }
   void_t fini() {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fini:nid:", nid, " name:", name);
      #endif
      if(-1 == nid) return;
      if(mode_listen == mode) {
         run();
         unlock();
         thr_t::fini();
      }
      mode = mode_init;
      dam.fini();
      conf.fini();
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fini:cqs:size0:", cqs.size());
      #endif
      //TODO: inform clients;
      while(false == cqs.empty()) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:fini:cqs:a:name:", cqs.a->obj.smp.name.);
         #endif
         cqs.a->obj.fini();
         cqs.remove(cqs.a);
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fini:cqs:size1:", cqs.size());
      #endif
      iqs.fini();
      timeout = smp_t::forever;
      suid.fini();
      lock.fini();
      if(disp) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:fini:disp:last");
         #endif
         (*disp)(null_t<msg_t>::def(), res_last, that);
      }
      nid = -1;
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fini:exit");
      #endif
   }

   void_t unlock() {
      if(res_gone != iqs.flag) {
         iqs.flag = res_gone;
         iqs.smp.post();
      }
   }
   void_t flush() {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:flush");
      #endif
      scope_t<lock_t<smp_t>> sl(lock);
      iqs.flush();
   }

   int_t connect(string_t const name0, int_t const timeout0 = smp_t::forever) {
      asrt(-1 != nid); // guarantee initialisation;
      msg_t msg(nid, -1, -1);
      {
         scope_t<lock_t<smp_t>> sl(lock);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:connect:", name, " -> ", name0, " timeout:", timeout0);
         #endif
         iqs_t mqs(name0); // remote name; finish on return;
         if(1 == mqs.refcnt()) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:connect:", name0, " unavailable");
            #endif
            return res_gone; // server node down;
         }
         msg.set(name); // local name;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:connect:post:msg:", msg.str(1));
         #endif
         if(0 == mqs.post(queue_ctrl, msg)) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:connect:", name0, " full");
            #endif
            return res_full; // server queue full;
         }
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:connect:", name0, " requested");
         #endif
      }
      if(mode_init == mode) {
         int_t const bytes = wait(msg, timeout0);
         if(0 >= bytes) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:connect:error:", bytes);
            #endif
            return bytes;
         }
         asrt(-1 == msg.src());
         string_t rname;
         msg.get(rname);
         int_t const rnid = msg.dst();
         record(rname, rnid);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:connect:", name, " -> ", rname, " rnid:", rnid, " synchronously confirmed");
         #endif
         return rnid;
      } else { // mode_work or mode_listen;
         // wait for server response;
         int_t const res = conf.wait(timeout0);
         int_t const rnid = 0 > res ? res_fail : sid;
         sid = -1;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:connect:", name, " -> ", name0, " rnid:", rnid, " res:", res, " asynchronously ", -1 == res ? "unknown" : "confirmed");
         #endif
         return rnid; // received in work() loop;
      }
   }
   int_t isolate(int_t const rnid, int_t const timeout0 = smp_t::forever) {
      asrt(-1 != nid); // guarantee initialisation;
      msg_t msg(nid, -2, -1 );
      {
         scope_t<lock_t<smp_t>> sl(lock);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:isolate:nid:", nid, " rnid:", rnid, " name:", name, " cqs:size:", cqs.size());
         #endif
         cqs_t::node_t* node0 = cqs.find(rnid);
         if(0 == node0) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:isolate:unknown");
            #endif
            return res_unknown; // not connected;
         }
         iqs_t &mqs = node0->obj; // reference connected remote node;
         // post isolation  request;
         msg.set(name); // local name;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:isolate:msg:", msg.str(1));
         #endif
         if(0 == mqs.post(queue_ctrl, msg)) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:isolate:", rnid, " full");
            #endif
            return res_full; // server queue full;
         }
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:isolate:", rnid, " requested");
         #endif
      }
      if(mode_init == mode) {
         int_t const bytes = wait(msg, timeout0);
         if(0 >= bytes) {
            #if defined(M1_FEAT_NODE_DEBUG)
               tracec("node:isolate:error:", bytes);
            #endif
            return bytes;
         }
         asrt(-2 == msg.src());
         string_t rname;
         msg.get(rname);
         int_t const rnid = msg.dst();
         remove(rnid);
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:isolate:", name, " -> ", rname, " rnid:", rnid, " synchronously confirmed");
         #endif
         return rnid;
      } else { // mode_work or mode_listen;
         // wait for server response;
         int_t const res = conf.wait(timeout0);
         int_t const rnid = 0 > res ? res_fail : sid;
         sid = -1;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:isolate:", name, " -> rnid:", rnid, " res:", res, " asynchronously ", -1 == res ? "unknown" : "confirmed");
         #endif
         return rnid; // received in work() loop;
      }
   }

   void_t record(string_t const &rname, int_t const &rnid) {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:record:rname:", rname, " rnid:", rnid, " cqs:size:", cqs.size());
      #endif
      scope_t<lock_t<smp_t>> sl(lock);
      cqs_t::node_t* node0 = cqs.find(rnid);
      if(0 == node0) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:record:insert");
         #endif
         cqs[rnid].init(rname);
      } else {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:record:present");
         #endif
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:record:cqs:size:", cqs.size(), " exit");
      #endif
   }
   void_t remove(int_t const &rnid) {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:remove:rnid:", rnid, " cqs:size:", cqs.size());
      #endif
      scope_t<lock_t<smp_t>> sl(lock);
      cqs_t::node_t* node0 = cqs.find(rnid);
      if(0 == node0) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:remove:unknown");
         #endif
      } else {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:remove:remove");
         #endif
         node0->obj.fini();
         cqs.remove(node0);
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:remove:cqs:size:", cqs.size(), " exit");
      #endif
   }

   int_t post(queue_e queue0, msg_t &msg, int_t retries0 = 3,
      int_t const every0 = 100) {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:post:nid:", nid, " queue:", estr2(queue0), " retry:", retries0, " times every:", every0, " ms");
      #endif
      if(-1 == nid) {
         thr_t::wait(1000);
         asrt(0);
      }
      scope_t<lock_t<smp_t>> sl(lock);
      cqs_t::node_t* node0 = cqs.find(msg.dst());
      if(0 == node0) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:post:nid:", msg.dst(), " unknown");
         #endif
         return res_unknown; // not connected;
      }
      msg.src(nid);
      while(retries0--) {
         int_t const bytes = node0->obj.post(queue0, msg);
         if(0 < bytes) return res_ok;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:post:msg:retry");
         #endif
         thr_t::wait(every0); // server queue (temporarily) full;
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:post:msg:full");
      #endif
      return res_full;
   }
   int_t wait(msg_t& msg, int_t const timeout0 = smp_t::forever) {
      asrt(-1 != nid);
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:wait");
      #endif
      scope_t<lock_t<smp_t>> sl(lock);
      int_t const bytes = iqs.wait(msg, timeout0);
      if(0 == bytes) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:wait:empty");
         #endif
         return res_empty; // input queue empty;
      } else if(smp_t::fail == bytes) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:wait:fail");
         #endif
         return res_fail; // semaphore failure;
      } else if(smp_t::timeout == bytes) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:wait:timeout");
         #endif
         return res_timeout; // semaphore timed out;
      } else if(res_gone == bytes) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:wait:gone");
         #endif
         return res_gone; // node shutdown;
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:wait:bytes:", bytes);
      #endif
      return bytes;
   }

   int_t fetch(queue_e queue0, msg_t& msg,
      int_t const timeout0 = smp_t::forever, int_t const retries0 = 3,
      int_t const every0 = 100) {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fetch:queue:", estr2(queue0), " retry:", retries0, " times every:", every0, " ms");
      #endif
      int_t time = normal_t::inst().ticks();
      int_t res = post(queue0, msg, retries0, every0);  // post request;
      if(res_ok == res) {
         res = wait(msg, timeout0);
         time = normal_t::inst().ticks() - time;
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:fetch:elapsed:", time);
         #endif
      }
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:fetch:res:", res);
      #endif
      return res;
   }
   void_t listen() {
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:listen:nid:", nid, " mode:", estr2(mode), " name:", name);
      #endif
      if(-1 == nid || mode_init != mode) {
         thr_t::wait(1000);
      }
      asrt(-1 != nid);
      asrt(mode_init == mode);
      mode = mode_listen;
      thr_t::init(name); // runs work() paralelly;
   }

   int_t run() {
      asrt(-1 != nid);
      asrt(mode_work == mode || mode_listen == mode);
      return dam.post();
   }
   int_t stop() {
      asrt(-1 != nid);
      asrt(mode_work == mode || mode_listen == mode);
      msg_t msg(0, 0, -1);
      return iqs.post(queue_ctrl, msg);
   }

   int_t level(queue_e const &queue0, int_t &fill0 = null_t<int_t>::def(),
      int_t &size0 = null_t<int_t>::def()) {
      return iqs.qs[queue0].level(fill0, size0);
   }

   virtual void_t work(void_t * /* arg0 */) {  // overload from thr_t;
      asrt(-1 != nid);
      asrt(mode_work != mode);
      #if defined(M1_FEAT_NODE_DEBUG)
         tracec("node:work");
      #endif
      if(mode_init == mode) {
         mode = mode_work;
      }
      if(disp) (*disp)(null_t<msg_t>::def(), res_first, that);
      for(struct { msg_t msg; int_t res; } state; res_gone != (state.res = iqs.wait(state.msg, timeout));) {
         #if defined(M1_FEAT_NODE_DEBUG)
            tracec("node:work:received:msg:", state.msg.str());
         #endif
         if(res_ok <= state.res && (0 > state.msg.src() || 0 > state.msg.dst())) {
            string_t rname;
            state.msg.get(rname);
            if(-1 == state.msg.dst()) { // connection request;
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:connect:request:", rname, " -> ", name);
               #endif
               int_t const rnid = state.msg.src();
               record(rname, rnid);
               if(disp) (*disp)(null_t<msg_t>::def(), res_request_con, that);
               state.msg.src(-1); // confirm;
               state.msg.dst(nid); // local nid;
               state.msg.set(name); // local name;
               cqs[rnid].post(queue_ctrl, state.msg);
            } else if(-1 == state.msg.src()) { // connection confirmation;
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:connect:confirm:", name, " -> ", rname);
               #endif
               sid = state.msg.dst();
               record(rname, sid);
               if(disp) (*disp)(null_t<msg_t>::def(), res_confirm_con, that);
               conf.post(); // complete connect;
            } else if(-2 == state.msg.dst()) { // isolation request;
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:isolate:request:", rname, " -> ", name);
               #endif
               int_t const rnid = state.msg.src();
               state.msg.src(-2); // confirm;
               state.msg.dst(nid);  // local nid;
               state.msg.set(name); // local name;
               cqs[rnid].post(queue_ctrl, state.msg);
               if(disp) (*disp)(null_t<msg_t>::def(), res_request_iso, that);
               remove(rnid);
            } else if(-2 == state.msg.src()) { // isolation confirmation;
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:isolate:confirm:", name, " -> ", rname);
               #endif
               sid = state.msg.dst();
               remove(sid);
               if(disp) (*disp)(null_t<msg_t>::def(), res_confirm_iso, that);
               conf.post(); // complete connect;
            }
         } else {
            if(res_gone == iqs.flag) {
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:unlocked:res:", state.res);
               #endif
            } else if(-1 == state.msg.cid()) {
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:stop");
               #endif
               dam.reset();
               dam.wait();
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:run");
               #endif
            } else {
               #if defined(M1_FEAT_NODE_DEBUG)
                  tracec("node:work:disp:res:", state.res);
               #endif
               if(res_ok <= state.res) state.res = res_ok;
               if(disp) (*disp)(state.msg, (res_e)state.res, that);
            }
         }
      }
      //tracec("node:work:disp:last");
      //if(disp)(*disp)(null_t<msg_t>::def(), res_last, that);
      tracec("node:work:exit");
   }
   string_t str() const {
      return string_t("name:", name,
                      " nid:", nid,
                     " disp:", (void*)disp,
                     " that:", (void*)that,
                  " timeout:", timeout,
                      " cqs:", cqs.size());
   }
   string_t name;
   lock_t<smp_t> lock;
   suid_t suid;
   int_t nid; // this nodes id;
   disp_t disp;
   void_t* that;
   int_t timeout;
   smp_t conf;
   smp_t dam;
   mode_e mode;
   int_t sid; // server node id returned on connect;
   iqs_t iqs; // local input queues (created, wait);
   cqs_t cqs; // connected remote queues (post);
};

//------------------------------------------------------------------------------
// serial;

struct serial_t {

   //------------------------------------------------------------------------------

   // https://linux.die.net/man/3/termios;

   // typedef unsigned cc_t;
   // typedef unsigned speed_t;
   // typedef unsigned tcflag_t;

   // struct termios {
   //   cc_t      c_cc[NCCS];
   //   tcflag_t  c_cflag;
   //   tcflag_t  c_iflag;
   //   tcflag_t  c_lflag;
   //   tcflag_t  c_oflag;
   //   speed_t   c_ispeed;
   //   speed_t   c_ospeed;
   // };

   //------------------------------------------------------------------------

   #define XENUM3(X) \
      X(off, = 0, "off")\
      X(on,  = 1, "on" )
      XENUM3_IMPL(flag_e)
   #undef XENUM3

   //------------------------------------------------------------------------
   // baudrate;

   #define XENUM3(X) \
      X(M1_BR_0,       =       B0,       "0")\
      X(M1_BR_50,      =      B50,      "50")\
      X(M1_BR_75,      =      B75,      "75")\
      X(M1_BR_110,     =     B110,     "110")\
      X(M1_BR_134,     =     B134,     "134")\
      X(M1_BR_150,     =     B150,     "150")\
      X(M1_BR_200,     =     B200,     "200")\
      X(M1_BR_300,     =     B300,     "300")\
      X(M1_BR_600,     =     B600,     "600")\
      X(M1_BR_1200,    =    B1200,    "1200")\
      X(M1_BR_1800,    =    B1800,    "1800")\
      X(M1_BR_2400,    =    B2400,    "2400")\
      X(M1_BR_4800,    =    B4800,    "4800")\
      X(M1_BR_9600,    =    B9600,    "9600")\
      X(M1_BR_19200,   =   B19200,   "19200")\
      X(M1_BR_38400,   =   B38400,   "38400")\
      X(M1_BR_57600,   =   B57600,   "57600")\
      X(M1_BR_115200,  =  B115200,  "115200")\
      X(M1_BR_230400,  =  B230400,  "230400")\
      X(M1_BR_460800,  =  B460800,  "460800")\
      X(M1_BR_500000,  =  B500000,  "500000")\
      X(M1_BR_576000,  =  B576000,  "576000")\
      X(M1_BR_921600,  =  B921600,  "921600")\
      X(M1_BR_1000000, = B1000000, "1000000")\
      X(M1_BR_1152000, = B1152000, "1152000")\
      X(M1_BR_1500000, = B1500000, "1500000")\
      X(M1_BR_2000000, = B2000000, "2000000")\
      X(M1_BR_2500000, = B2500000, "2500000")\
      X(M1_BR_3000000, = B3000000, "3000000")\
      X(M1_BR_3500000, = B3500000, "3500000")\
      X(M1_BR_4000000, = B4000000, "4000000")
      XENUM3_IMPL(br_e)
   #undef XENUM3

   static br_e mapbr(int_t const br) {
      switch(br) {
         case       0: return M1_BR_0      ;
         case      50: return M1_BR_50     ;
         case      75: return M1_BR_75     ;
         case     110: return M1_BR_110    ;
         case     134: return M1_BR_134    ;
         case     150: return M1_BR_150    ;
         case     200: return M1_BR_200    ;
         case     300: return M1_BR_300    ;
         case     600: return M1_BR_600    ;
         case    1200: return M1_BR_1200   ;
         case    1800: return M1_BR_1800   ;
         case    2400: return M1_BR_2400   ;
         case    4800: return M1_BR_4800   ;
         case    9600: return M1_BR_9600   ;
         case   19200: return M1_BR_19200  ;
         case   38400: return M1_BR_38400  ;
         case   57600: return M1_BR_57600  ;
         case  115200: return M1_BR_115200 ;
         case  230400: return M1_BR_230400 ;
         case  460800: return M1_BR_460800 ;
         case  500000: return M1_BR_500000 ;
         case  576000: return M1_BR_576000 ;
         case  921600: return M1_BR_921600 ;
         case 1000000: return M1_BR_1000000;
         case 1152000: return M1_BR_1152000;
         case 1500000: return M1_BR_1500000;
         case 2000000: return M1_BR_2000000;
         case 2500000: return M1_BR_2500000;
         case 3000000: return M1_BR_3000000;
         case 3500000: return M1_BR_3500000;
         case 4000000: return M1_BR_4000000;
         default : return (br_e)-1;
      }
   }

   //---------------------------------------------------------------------------
   // c_cflag: baudrate; databits; parity; stopbits; hw flow ctrl;

   #define XENUM4(X) \
     X(M1_CFLAG_PARENB,  = PARENB,  "PARENB",  "parity on/off")\
     X(M1_CFLAG_PARODD,  = PARODD,  "PARODD",  "odd parity")\
     X(M1_CFLAG_CMSPAR,  = CMSPAR,  "CMSPAR",  "parity always PARODD (non POSIX)")\
     X(M1_CFLAG_CSIZE,   = CSIZE,   "CSIZE",   "mask for data bits")\
     X(M1_CFLAG_HUPCL,   = HUPCL,   "HUPCL",   "hangup (drop DTR) on close")\
     X(M1_CFLAG_CSTOPB,  = CSTOPB,  "CSTOPB",  "stopbit on")\
     X(M1_CFLAG_CREAD,   = CREAD,   "CREAD",   "read on")\
     X(M1_CFLAG_CLOCAL,  = CLOCAL,  "CLOCAL",  "ignore ctrl")\
     X(M1_CFLAG_CRTSCTS, = CRTSCTS, "CRTSCTS", "hw flow ctrl on")
     XENUM4_IMPL(cflag_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_CS_5, = CS5, "CS5",     "5 bits")\
     X(M1_CS_6, = CS6, "CS6",     "6 bits")\
     X(M1_CS_7, = CS7, "CS7",     "7 bits")\
     X(M1_CS_8, = CS8, "CS8",     "8 bits")
     XENUM4_IMPL(cs_e)
   #undef XENUM4

   //---------------------------------------------------------------------------
   // c_lflag: local driver input settings: canonical or raw;

   #define XENUM4(X) \
     X(M1_LFLAG_ISIG,    = ISIG,    "ISIG",    "enable SIGINTR,SIGSUSP,SIGDSUSP,SIGQUIT")\
     X(M1_LFLAG_ICANON,  = ICANON,  "ICANON",  "canonical or raw)")\
     X(M1_LFLAG_IEXTEN,  = IEXTEN,  "IEXTEN",  "enable extensions")\
     X(M1_LFLAG_ECHO,    = ECHO,    "ECHO",    "echo input")\
     X(M1_LFLAG_ECHOE,   = ECHOE,   "ECHOE",   "echo erase char as BS-SP-BS")\
     X(M1_LFLAG_ECHOK,   = ECHOK,   "ECHOK",   "echo NL after KILL")\
     X(M1_LFLAG_ECHONL,  = ECHONL,  "ECHONL",  "always echo NL")\
     X(M1_LFLAG_NOFLSH,  = NOFLSH,  "NOFLSH",  "disable buffer flush on INTR or QUIT")\
     X(M1_LFLAG_XCASE,   = XCASE,   "XCASE",   "uppercase only terminal")\
     X(M1_LFLAG_TOSTOP,  = TOSTOP,  "TOSTOP",  "SIGTTOU on background input")\
     X(M1_LFLAG_ECHOPRT, = ECHOPRT, "ECHOPRT", "print erased char on erase")\
     X(M1_LFLAG_ECHOCTL, = ECHOCTL, "ECHOCTL", "echo special chars with offset 0x40")\
     X(M1_LFLAG_ECHOKE,  = ECHOKE,  "ECHOKE",  "erase entire line on KILL")
     XENUM4_IMPL(lflag_e)
   #undef XENUM4

   //---------------------------------------------------------------------------
   // c_iflag: local character input processing;

   #define XENUM4(X) \
     X(M1_IFLAG_IGNBRK,  = IGNBRK,  "IGNBRK",  "ignore break")\
     X(M1_IFLAG_BRKINT,  = BRKINT,  "BRKINT",  "SIGINT on break")\
     X(M1_IFLAG_IGNPAR,  = IGNPAR,  "IGNPAR",  "ignore parity errors")\
     X(M1_IFLAG_PARMRK,  = PARMRK,  "PARMRK",  "mark parity errors")\
     X(M1_IFLAG_INPCK,   = INPCK,   "INPCK",   "enable parity check")\
     X(M1_IFLAG_ISTRIP,  = ISTRIP,  "ISTRIP",  "strip parity bits")\
     X(M1_IFLAG_INLCR,   = INLCR,   "INLCR",   "map NL to CR")\
     X(M1_IFLAG_IGNCR,   = IGNCR,   "IGNCR",   "ignore CR")\
     X(M1_IFLAG_ICRNL,   = ICRNL,   "ICRNL",   "map CR to NL")\
     X(M1_IFLAG_IXON,    = IXON,    "IXON",    "enable outgoing sw flow control")\
     X(M1_IFLAG_IXOFF,   = IXOFF,   "IXOFF",   "enable incoming sw flow control")\
     X(M1_IFLAG_IUCLC,   = IUCLC,   "IUCLC",   "map upper to lower case")\
     X(M1_IFLAG_IXANY,   = IXANY,   "IXANY",   "flow restart by any char")\
     X(M1_IFLAG_IMAXBEL, = IMAXBEL, "IMAXBEL", "echo BEL on queue full")\
     X(M1_IFLAG_IUTF8,   = IUTF8,   "IUTF8",   "correct char erase in raw")
     XENUM4_IMPL(iflag_e)
   #undef XENUM4

   //---------------------------------------------------------------------------
   // c_oflag: output filtering;

   #define XENUM4(X) \
     X(M1_OFLAG_OPOST,  = OPOST,  "OPOST",  "raw (no output pocessing)")\
     X(M1_OFLAG_OLCUC,  = OLCUC,  "OLCUC",  "map lowercase to uppercase")\
     X(M1_OFLAG_OCRNL,  = OCRNL,  "OCRNL",  "map CR to NL")\
     X(M1_OFLAG_ONLCR,  = ONLCR,  "ONLCR",  "map NL to CR-LF")\
     X(M1_OFLAG_ONOCR,  = ONOCR,  "ONOCR",  "no CR output at column 0")\
     X(M1_OFLAG_ONLRET, = ONLRET, "ONLRET", "NL performs CR")\
     X(M1_OFLAG_OFILL,  = OFILL,  "OFILL",  "use fill char for delay")\
     X(M1_OFLAG_OFDEL,  = OFDEL,  "OFDEL",  "use DEL as fill char")\
     X(M1_OFLAG_NLDLY,  = NLDLY,  "NLDLY",  "mask for delay after NL")\
     X(M1_OFLAG_CRDLY,  = CRDLY,  "CRDLY",  "mask for delay after CR")\
     X(M1_OFLAG_TABDLY, = TABDLY, "TABDLY", "mask for delay after TAB")\
     X(M1_OFLAG_BSDLY,  = BSDLY,  "BSDLY",  "mask for delay after BS")\
     X(M1_OFLAG_VTDLY,  = VTDLY,  "VTDLY",  "mask for delay after VT")\
     X(M1_OFLAG_FFDLY,  = FFDLY,  "FFDLY",  "mask for delay after FF")
     XENUM4_IMPL(oflag_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_NL_0, = NL0, "NL0",   "0 ms")\
     X(M1_NL_1, = NL1, "NL1", "100 ms")
     XENUM4_IMPL(nl_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_CR_0, = CR0, "CR0", "  0 ms")\
     X(M1_CR_1, = CR1, "CR1", "column dependent")\
     X(M1_CR_2, = CR2, "CR2", "100 ms")\
     X(M1_CR_3, = CR3, "CR3", "150 ms")
     XENUM4_IMPL(cr_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_TAB_0, = TAB0, "TAB0", "  0 ms")\
     X(M1_TAB_1, = TAB1, "TAB1", "column dependent")\
     X(M1_TAB_2, = TAB2, "TAB2", "100 ms")\
     X(M1_TAB_3, = TAB3, "TAB3", "expand TAB to space")
     XENUM4_IMPL(tab_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_BS_0, = BS0, "BS0",  "0 ms")\
     X(M1_BS_1, = BS1, "BS1", "50 ms")
     XENUM4_IMPL(bs_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_VT_0, = VT0, "VT0", "0 ms")\
     X(M1_VT_1, = VT1, "VT1", "2 sec")
     XENUM4_IMPL(vt_e)
   #undef XENUM4

   #define XENUM4(X) \
     X(M1_FF_0, = FF0, "FF0", "0 msn")\
     X(M1_FF_1, = FF1, "FF1", "2 sec")
     XENUM4_IMPL(ff_e)
   #undef XENUM4

   //---------------------------------------------------------------------------
   // c_cc index: control chars and timeout;

   #define XENUM4(X) \
     X(M1_VID_VINTR,    = VINTR,    "VINTR",    "interrupt (INTR, O03, ETX, Ctrl-C)")\
     X(M1_VID_VQUIT,    = VQUIT,    "VQUIT",    "quit (QUIT, O34, FS, Ctrl-\\)")\
     X(M1_VID_VERASE,   = VERASE,   "VERASE",   "erase (ERASE, O10, BS, Ctrl-H)")\
     X(M1_VID_VKILL,    = VKILL,    "VKILL",    "kill line (O25, NAK, Ctrl-U)")\
     X(M1_VID_VEOF,     = VEOF,     "VEOF",     "end-of-file (EOF, O04, EOT, Ctrl-D)")\
     X(M1_VID_VEOL,     = VEOL,     "VEOL",     "additional LF (EOL, O00, NULL)")\
     X(M1_VID_VEOL2,    = VEOL2,    "VEOL2",    "2nd additional LF (EOL2, O00, NULL, not POSIX)")\
     X(M1_VID_VSWTC,    = VSWTC,    "VSWTC",    "switch (SWITCH, O00, NULL, not POSIX)")\
     X(M1_VID_VSTART,   = VSTART,   "VSTART",   "start (START, O21, 0xDC1, Ctrl-Q)")\
     X(M1_VID_VSTOP,    = VSTOP,    "VSTOP",    "stop (STOP, O23, 0xDC3, Ctrl-S)")\
     X(M1_VID_VSUSP,    = VSUSP,    "VSUSP",    "suspend (SUSP, O32, SUB, Ctrl-Z)")\
     X(M1_VID_VREPRINT, = VREPRINT, "VREPRINT", "reprint unread chars (REPRINT, O22, 0xDC2, Ctrl-R, not POSIX)")\
     X(M1_VID_VWERASE,  = VWERASE,  "VWERASE",  "word erase (WERASE, O27, ETB, Ctrl-W, not POSIX)")\
     X(M1_VID_VLNEXT,   = VLNEXT,   "VLNEXT",   "literal quote (LNEXT, O26, SYN, Ctrl-V, not POSIX)")\
     X(M1_VID_VDISCARD, = VDISCARD, "VDISCARD", "toggle flush (O17, SI, Ctrl-O, not POSIX)")\
     X(M1_VID_VMIN,     = VMIN,     "VMIN",     "min number of non-canonical read chars")\
     X(M1_VID_VTIME,    = VTIME,    "VTIME",    "non-canonical read timeout (deciseconds)")
     XENUM4_IMPL(vid_e)
   #undef XENUM4

   #define XENUM3(X) \
     X(M1_VCHR_CTRL_C,        = 003, "^C")\
     X(M1_VCHR_CTRL_RSOLIDUS, = 034, "^\\")\
     X(M1_VCHR_CTRL_H,        = 010, "^H")\
     X(M1_VCHR_CTRL_U,        = 025, "^U")\
     X(M1_VCHR_CTRL_D,        = 004, "^D")\
     X(M1_VCHR_NULL,          = 000, "^NUL")\
     X(M1_VCHR_CTRL_Q,        = 021, "^Q")\
     X(M1_VCHR_CTRL_S,        = 023, "^S")\
     X(M1_VCHR_CTRL_Z,        = 032, "^Z")\
     X(M1_VCHR_CTRL_R,        = 022, "^R")\
     X(M1_VCHR_CTRL_W,        = 027, "^W")\
     X(M1_VCHR_CTRL_V,        = 026, "^V")\
     X(M1_VCHR_CTRL_O,        = 017, "^O")
     XENUM3_IMPL(vchr_e)
   #undef XENUM3

   //---------------------------------------------------------------------------

   // raw input passed through unprocessed;
   // $ stty -F /dev/ttyS1 raw speed 115200 -echo
   // $ stty -F /dev/ttyS1 -a

   #define XENUM3(X) \
     X(M1_CONF_RAW,   = 0, "raw")\
     X(M1_CONF_CANON, = 1, "canon")
     XENUM3_IMPL(conf_e)
   #undef XENUM3

   //---------------------------------------------------------------------------

   struct cflag_t {
      cflag_t(conf_e const conf = M1_CONF_RAW) {
         switch(conf) {
            M1_CONF_RAW:
            default:
               // -parenb -parodd -cmspar cs8 hupcl -cstopb cread clocal -crtscts
               bri=M1_BR_115200; bro=M1_BR_115200;
               parenb=off; parodd=off; cmspar=off; cs=M1_CS_8;
               hupcl=on; cstopb=off; cread=on; clocal=on; crtscts=off;
               break;
         }
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            case 0: {
               return string_t("i", estr1(bri),
                              " o", estr1(bro),
          ' ', parenb  ? 'P' : 'p', "arenb "  ,
               parodd  ? 'P' : 'p', "arodd "  ,
               cmspar  ? 'C' : 'c', "mspar ");
            }
            case 1: {
               return string_t(estr1(cs) ,
          ' ', hupcl   ? 'H' : 'h', "upcl "   ,
               cstopb  ? 'C' : 'c', "stopb "  ,
               cread   ? 'C' : 'c', "read "   ,
               clocal  ? 'C' : 'c', "local "  ,
               crtscts ? 'C' : 'c', "rtscts " );
            }
         }
      }
      br_e   bri;
      br_e   bro;
      flag_e parenb;
      flag_e parodd;
      flag_e cmspar;
      cs_e   cs;
      flag_e hupcl;
      flag_e cstopb;
      flag_e cread;
      flag_e clocal;
      flag_e crtscts;
   };

   struct  lflag_t {
      // disable ECHO,ECHOE sending to echoing device to prevent recursion;
      lflag_t(conf_e const conf = M1_CONF_RAW) {
         switch(conf) {
            M1_CONF_RAW:
            default:
               // -isig -icanon iexten -echo echoe echok -echonl -noflsh -xcase -tostop -echoprt
               // echoctl echoke
               isig=off; icanon=off; iexten=on; echo=off; echoe=on; echok=on;
               echonl=off; noflsh=off; xcase=off; tostop=off; echoprt=off;
               echoctl=on; echoke=on;
               break;
            M1_CONF_CANON:
               // -isig icanon iexten echo echoe echok -echonl -noflsh -xcase -tostop -echoprt
               // echoctl echoke
               isig=off; icanon=on; iexten=on; echo=on; echoe=on; echok=on;
               echonl=off; noflsh=off; xcase=off; tostop=off; echoprt=off;
               echoctl=on; echoke=on;
               break;
         }
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            case 0: {
               return string_t(isig    ? 'I' : 'i', "sig "   ,
                               icanon  ? 'I' : 'i', "canon " ,
                               iexten  ? 'I' : 'i', "exten " ,
                               echo    ? 'E' : 'e', "cho "   ,
                               echoe   ? 'E' : 'e', "choe "  ,
                               echok   ? 'E' : 'e', "chok "  ,
                               echonl  ? 'E' : 'e', "chonl"  );
            }
            case 1: {
               return string_t(noflsh  ? 'N' : 'n', "oflsh " ,
                               xcase   ? 'X' : 'x', "case "  ,
                               tostop  ? 'T' : 't', "ostop " ,
                               echoprt ? 'E' : 'e', "choprt ",
                               echoctl ? 'E' : 'e', "choctl ",
                               echoke  ? 'E' : 'e', "choke " );
            }
         }
      }
      flag_e isig;
      flag_e icanon;
      flag_e iexten;
      flag_e echo;
      flag_e echoe;
      flag_e echok;
      flag_e echonl;
      flag_e noflsh;
      flag_e xcase;
      flag_e tostop;
      flag_e echoprt;
      flag_e echoctl;
      flag_e echoke;
   };

   struct iflag_t {
      iflag_t(conf_e const conf = M1_CONF_RAW) {
         switch(conf) {
            M1_CONF_RAW:
            default:
               // -ignbrk -brkint -ignpar -parmrk -inpck -istrip -inlcr -igncr -icrnl -ixon -ixoff
               // -iuclc -ixany -imaxbel -iutf8
               ignbrk=off; brkint=off; ignpar=off; parmrk=off; inpck=off;
               istrip=off; inlcr=off; igncr=off; icrnl=off; ixon=off; ixoff=off;
               iuclc=off; ixany=off; imaxbel=off; iutf8=off;
               break;
         }
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            case 0: {
               return string_t(ignbrk  ? 'I' : 'i', "gnbrk " ,
                               brkint  ? 'B' : 'b', "rkint " ,
                               ignpar  ? 'I' : 'i', "gnpar " ,
                               parmrk  ? 'P' : 'p', "armrk " ,
                               inpck   ? 'I' : 'i', "npck "  ,
                               istrip  ? 'I' : 'i', "strip " ,
                               inlcr   ? 'I' : 'i', "nlcr "  ,
                               igncr   ? 'I' : 'i', "gncr");
            }
            case 1: {
               return string_t(icrnl   ? 'I' : 'i', "crnl "  ,
                               ixon    ? 'I' : 'i', "xon "   ,
                               ixoff   ? 'I' : 'i', "xoff "  ,
                               iuclc   ? 'I' : 'i', "uclc "  ,
                               ixany   ? 'I' : 'i', "xany "  ,
                               imaxbel ? 'I' : 'i', "maxbel ",
                               iutf8   ? 'I' : 'i', "utf8"   );
            }
         }
      }
      flag_e ignbrk;
      flag_e brkint;
      flag_e ignpar;
      flag_e parmrk;
      flag_e inpck;
      flag_e istrip;
      flag_e inlcr;
      flag_e igncr;
      flag_e icrnl;
      flag_e ixon;
      flag_e ixoff;
      flag_e iuclc;
      flag_e ixany;
      flag_e imaxbel;
      flag_e iutf8;
   };

   struct oflag_t {
      // when OPOST is unset all other flags are ignored;
      oflag_t(conf_e const conf = M1_CONF_RAW) {
         switch(conf) {
            M1_CONF_RAW:
            default:
               // -opost -olcuc -ocrnl onlcr -onocr -onlret -ofill -ofdel nl0 cr0 tab0 bs0 vt0 ff0
               opost=off; olcuc=off; ocrnl=off; onlcr=on; onocr=off; onlret=off;
               ofill=off; ofdel=off; nl=M1_NL_0; cr=M1_CR_0; tab=M1_TAB_0;
               bs=M1_BS_0; vt=M1_VT_0; ff=M1_FF_0;
               break;
         }
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            case 0: {
               return string_t(opost ? 'O' : 'o', "post " ,
                               olcuc ? 'O' : 'o', "lcuc " ,
                               ocrnl ? 'O' : 'o', "crnl " ,
                               onlcr ? 'O' : 'o', "nlcr " ,
                               onocr ? 'O' : 'o', "nocr " ,
                               onlret? 'O' : 'o', "nlret ",
                               ofill ? 'O' : 'o', "fill " ,
                               ofdel ? 'O' : 'o', "fdel"  );
            }
            case 1: {
               return string_t(estr1(nl),
                         ' ', estr1(cr),
                         ' ', estr1(tab),
                         ' ', estr1(bs),
                         ' ', estr1(vt),
                         ' ', estr1(ff));
            }
         }
      }
      flag_e opost;
      flag_e olcuc;
      flag_e ocrnl;
      flag_e onlcr;
      flag_e onocr;
      flag_e onlret;
      flag_e ofill;
      flag_e ofdel;
      nl_e  nl;
      cr_e  cr;
      tab_e tab;
      bs_e  bs;
      vt_e  vt;
      ff_e  ff;
   };

   struct vchr_t {
      vchr_t(conf_e const conf = M1_CONF_RAW) {
         switch(conf) {
            M1_CONF_RAW:
            default:
               // intr = ^C; quit = ^\; erase = ^?; kill = ^U; eof = ^D; eol = <undef>;
               // eol2 = <undef>; swtch = <undef>; start = ^Q; stop = ^S; susp = ^Z; rprnt = ^R;
               // werase = ^W; lnext = ^V; flush = ^O; min = 1; time = 0;
               // 1;
               intr=M1_VCHR_CTRL_C; quit=M1_VCHR_CTRL_RSOLIDUS; erase=M1_VCHR_CTRL_H;
               kill=M1_VCHR_CTRL_U; eof=M1_VCHR_CTRL_D; eol=M1_VCHR_NULL;
               // 3;
               eol2=M1_VCHR_NULL; swtc=M1_VCHR_NULL;
               // if lflag.iexten is set;
               // 2;
               start=M1_VCHR_CTRL_Q; stop=M1_VCHR_CTRL_S; susp=M1_VCHR_CTRL_Z;
               // 4;
               reprint=M1_VCHR_CTRL_R;werase=M1_VCHR_CTRL_W; lnext=M1_VCHR_CTRL_V;
               discard=M1_VCHR_CTRL_O;
               min = 0;  // 1=block; 0=noblock;
               time = 0; // read timeout (decisec);
               break;
         }
      }
      string_t str(int_t const var = 0) const {
         switch(var) {
            case 0: {
               return string_t("intr", estr1(intr),
                              " quit", estr1(quit),
                             " erase", estr1(erase),
                              " kill", estr1(kill),
                               " eof", estr1(eof),
                               " eol", estr1(eol),
                              " eol2", estr1(eol2),
                              " swtc", estr1(swtc));
            }
            case 1: {
               return string_t("start", estr1(start),
                                "stop", estr1(stop),
                               " susp", estr1(susp),
                            " reprint", estr1(reprint),
                             " werase", estr1(werase),
                              " lnext", estr1(lnext),
                            " discard", estr1(discard));
            }
            case 2 : {
               return string_t("min:", min,
                             " time:", time);
            }
         }
      }
      vchr_e intr;
      vchr_e quit;
      vchr_e erase;
      vchr_e kill;
      vchr_e eof;
      vchr_e eol;
      vchr_e eol2;
      vchr_e swtc;
      vchr_e start;
      vchr_e stop;
      vchr_e susp;
      vchr_e reprint;
      vchr_e werase;
      vchr_e lnext;
      vchr_e discard;
      int_t  min;
      int_t  time;
   };

   //---------------------------------------------------------------------------

   template<typename FS, typename F>
   static void_t setflag(FS &flags, F const flag, flag_e const val) {
      if(on == val) {
         flags |=  flag;
      } else if(off == val) {
         flags &= ~flag;
      }
      //tracec("serial:setflag:", estr2(flag), ":", estr2(val), " check:", estr2(getflag(flags, flag)));
   }

   template<typename FS, typename F>
   static flag_e getflag(FS const &flags, F const flag) {
      //tracec("serial:getflag:", string_t::bin, flags);
      return (flags & flag) ? on : off;
   }

   //------------------------------------------------------------------------

   template<typename FS, typename M, typename F>
   static void_t setmask(FS &flags, M const mask, F const val) {
      flags &= ~mask;
      flags |=   val;
      F val1;
      getmask(flags, mask, val1);
      //tracec("serial:setmask:", estr2(mask), ":", estr2(val), " check:", estr2(val1));
   }
   template<typename FS, typename M, typename F>
   static void_t getmask(FS const &flags, M const mask, F &flag) {
      flag = (F)(flags & mask);
   }

   //------------------------------------------------------------------------

   template<typename CC>
   static void_t setvchr(CC &cc, vid_e const id, vchr_e const val) {
      cc[id] = val;
      //tracec("serial:setvchr:", estr2(id), ":", estr2(val), " check:", estr2(getvchr(cc, id)));
   }
   template<typename CC>
   static vchr_e getvchr(CC const &cc, vid_e const id) {
      return (vchr_e)cc[id];
   }

   //------------------------------------------------------------------------

   template<typename CC>
   static void_t setvval(CC &cc, vid_e const id, int_t const val) {
      cc[id] = val;
      //tracec("serial:setvval:", estr2(id), ":", val, " check:", getvval(cc, id));
   }
   template<typename CC>
   static int_t getvval(CC const &cc, vid_e const id ) {
      return cc[id];
   }

   //---------------------------------------------------------------------------

   struct conf_t {
      conf_t() {
      }
      conf_t(string_t const &dev0, br_e const br = M1_BR_115200) {
         cflag.bri = br;
         cflag.bro = br;
      }
      struct termios tty() { // termios from conf;

         struct termios tty;
         mset(&tty, 0, sizeof(tty));

         //---------------------------------------------------------------------
         // baudrate;

         //tracec("serial:conf:tty:bri:", estr2(cflag.bri), " bro:", estr2(cflag.bro));

         cfsetispeed(&tty, cflag.bri);
         cfsetospeed(&tty, cflag.bro);

         //---------------------------------------------------------------------
         // c_cflag: control modes;

         setflag(tty.c_cflag, M1_CFLAG_PARENB, cflag.parenb);
         setflag(tty.c_cflag, M1_CFLAG_PARODD, cflag.parodd);
         setflag(tty.c_cflag, M1_CFLAG_CMSPAR, cflag.cmspar);

         setmask(tty.c_cflag, M1_CFLAG_CSIZE, cflag.cs);

         setflag(tty.c_cflag, M1_CFLAG_HUPCL,   cflag.hupcl);
         setflag(tty.c_cflag, M1_CFLAG_CSTOPB,  cflag.cstopb);
         setflag(tty.c_cflag, M1_CFLAG_CREAD,   cflag.cread);
         setflag(tty.c_cflag, M1_CFLAG_CLOCAL,  cflag.clocal);
         setflag(tty.c_cflag, M1_CFLAG_CRTSCTS, cflag.crtscts);

         //---------------------------------------------------------------------
         // c_lflag: local modes;

         setflag(tty.c_lflag, M1_LFLAG_ISIG,    lflag.isig);
         setflag(tty.c_lflag, M1_LFLAG_ICANON,  lflag.icanon);
         setflag(tty.c_lflag, M1_LFLAG_ECHO,    lflag.iexten);
         setflag(tty.c_lflag, M1_LFLAG_ECHOE,   lflag.echo);
         setflag(tty.c_lflag, M1_LFLAG_IEXTEN,  lflag.echoe);
         setflag(tty.c_lflag, M1_LFLAG_ECHOK,   lflag.echok);
         setflag(tty.c_lflag, M1_LFLAG_ECHONL,  lflag.echonl);
         setflag(tty.c_lflag, M1_LFLAG_NOFLSH,  lflag.noflsh);
         setflag(tty.c_lflag, M1_LFLAG_XCASE,   lflag.xcase);
         setflag(tty.c_lflag, M1_LFLAG_TOSTOP,  lflag.tostop);
         setflag(tty.c_lflag, M1_LFLAG_ECHOPRT, lflag.echoprt);
         setflag(tty.c_lflag, M1_LFLAG_ECHOCTL, lflag.echoctl);
         setflag(tty.c_lflag, M1_LFLAG_ECHOKE,  lflag.echoke);

         //---------------------------------------------------------------------
         // c_iflag: input modes;

         setflag(tty.c_iflag, M1_IFLAG_IGNBRK,  iflag.ignbrk);
         setflag(tty.c_iflag, M1_IFLAG_BRKINT,  iflag.brkint);
         setflag(tty.c_iflag, M1_IFLAG_IGNPAR,  iflag.ignpar);
         setflag(tty.c_iflag, M1_IFLAG_PARMRK,  iflag.parmrk);
         setflag(tty.c_iflag, M1_IFLAG_INPCK,   iflag.inpck);
         setflag(tty.c_iflag, M1_IFLAG_ISTRIP,  iflag.istrip);
         setflag(tty.c_iflag, M1_IFLAG_INLCR,   iflag.inlcr);
         setflag(tty.c_iflag, M1_IFLAG_IGNCR,   iflag.igncr);
         setflag(tty.c_iflag, M1_IFLAG_ICRNL,   iflag.icrnl);
         setflag(tty.c_iflag, M1_IFLAG_IXON,    iflag.ixon);
         setflag(tty.c_iflag, M1_IFLAG_IXOFF,   iflag.ixoff);
         setflag(tty.c_iflag, M1_IFLAG_IUCLC,   iflag.iuclc);
         setflag(tty.c_iflag, M1_IFLAG_IXANY,   iflag.ixany);
         setflag(tty.c_iflag, M1_IFLAG_IMAXBEL, iflag.imaxbel);
         setflag(tty.c_iflag, M1_IFLAG_IUTF8,   iflag.iutf8);

         //---------------------------------------------------------------------
         // c_oflag: output modes;

         setflag(tty.c_oflag, M1_OFLAG_OPOST,  oflag.opost);
         setflag(tty.c_oflag, M1_OFLAG_OLCUC,  oflag.olcuc);
         setflag(tty.c_oflag, M1_OFLAG_OCRNL,  oflag.ocrnl);
         setflag(tty.c_oflag, M1_OFLAG_ONLCR,  oflag.onlcr);
         setflag(tty.c_oflag, M1_OFLAG_ONOCR,  oflag.onocr);
         setflag(tty.c_oflag, M1_OFLAG_ONLRET, oflag.onlret);
         setflag(tty.c_oflag, M1_OFLAG_OFILL,  oflag.ofill);
         setflag(tty.c_oflag, M1_OFLAG_OFDEL,  oflag.ofdel);

         setmask(tty.c_oflag, M1_OFLAG_NLDLY,  oflag.nl);
         setmask(tty.c_oflag, M1_OFLAG_CRDLY,  oflag.cr);
         setmask(tty.c_oflag, M1_OFLAG_TABDLY, oflag.tab);
         setmask(tty.c_oflag, M1_OFLAG_BSDLY,  oflag.bs);
         setmask(tty.c_oflag, M1_OFLAG_VTDLY,  oflag.vt);
         setmask(tty.c_oflag, M1_OFLAG_FFDLY,  oflag.ff);

         //---------------------------------------------------------------------
         // c_cc: special characters;

         setvchr(tty.c_cc, M1_VID_VINTR,    vchr.intr);
         setvchr(tty.c_cc, M1_VID_VQUIT,    vchr.quit);
         setvchr(tty.c_cc, M1_VID_VERASE,   vchr.erase);
         setvchr(tty.c_cc, M1_VID_VKILL,    vchr.kill);
         setvchr(tty.c_cc, M1_VID_VEOF,     vchr.eof);
         setvchr(tty.c_cc, M1_VID_VEOL,     vchr.eol);
         setvchr(tty.c_cc, M1_VID_VEOL2,    vchr.eol2);
         setvchr(tty.c_cc, M1_VID_VSWTC,    vchr.swtc);
         setvchr(tty.c_cc, M1_VID_VSTART,   vchr.start);
         setvchr(tty.c_cc, M1_VID_VSTOP,    vchr.stop);
         setvchr(tty.c_cc, M1_VID_VSUSP,    vchr.susp);
         setvchr(tty.c_cc, M1_VID_VREPRINT, vchr.reprint);
         setvchr(tty.c_cc, M1_VID_VWERASE,  vchr.werase);
         setvchr(tty.c_cc, M1_VID_VLNEXT,   vchr.lnext);
         setvchr(tty.c_cc, M1_VID_VDISCARD, vchr.discard);

         //---------------------------------------------------------------------

         setvval(tty.c_cc, M1_VID_VMIN,  vchr.min);
         setvval(tty.c_cc, M1_VID_VTIME, vchr.time);

         //---------------------------------------------------------------------

         return tty;
      };
      void_t tty(struct termios const &tty) { // termios to conf;

         //---------------------------------------------------------------------
         // baudrate;

         cflag.bri = (br_e)cfgetispeed(&tty);
         cflag.bro = (br_e)cfgetospeed(&tty);

         //---------------------------------------------------------------------
         // c_cflag: control modes;

         cflag.parenb = getflag(tty.c_cflag, M1_CFLAG_PARENB);
         cflag.parodd = getflag(tty.c_cflag, M1_CFLAG_PARODD);
         cflag.cmspar = getflag(tty.c_cflag, M1_CFLAG_CMSPAR);

         getmask(tty.c_cflag, M1_CFLAG_CSIZE, cflag.cs);

         cflag.hupcl   = getflag(tty.c_cflag, M1_CFLAG_HUPCL);
         cflag.cstopb  = getflag(tty.c_cflag, M1_CFLAG_CSTOPB);
         cflag.cread   = getflag(tty.c_cflag, M1_CFLAG_CREAD);
         cflag.clocal  = getflag(tty.c_cflag, M1_CFLAG_CLOCAL);
         cflag.crtscts = getflag(tty.c_cflag, M1_CFLAG_CRTSCTS);

         //---------------------------------------------------------------------
         // c_lflag: local modes;

         lflag.isig    = getflag(tty.c_lflag, M1_LFLAG_ISIG);
         lflag.icanon  = getflag(tty.c_lflag, M1_LFLAG_ICANON);
         lflag.iexten  = getflag(tty.c_lflag, M1_LFLAG_ECHO);
         lflag.echo    = getflag(tty.c_lflag, M1_LFLAG_ECHOE);
         lflag.echoe   = getflag(tty.c_lflag, M1_LFLAG_IEXTEN);
         lflag.echok   = getflag(tty.c_lflag, M1_LFLAG_ECHOK);
         lflag.echonl  = getflag(tty.c_lflag, M1_LFLAG_ECHONL);
         lflag.noflsh  = getflag(tty.c_lflag, M1_LFLAG_NOFLSH);
         lflag.xcase   = getflag(tty.c_lflag, M1_LFLAG_XCASE);
         lflag.tostop  = getflag(tty.c_lflag, M1_LFLAG_TOSTOP);
         lflag.echoprt = getflag(tty.c_lflag, M1_LFLAG_ECHOPRT);
         lflag.echoctl = getflag(tty.c_lflag, M1_LFLAG_ECHOCTL);
         lflag.echoke  = getflag(tty.c_lflag, M1_LFLAG_ECHOKE);

         //---------------------------------------------------------------------
         // c_iflag: input modes;

         iflag.ignbrk  = getflag(tty.c_iflag, M1_IFLAG_IGNBRK);
         iflag.brkint  = getflag(tty.c_iflag, M1_IFLAG_BRKINT);
         iflag.ignpar  = getflag(tty.c_iflag, M1_IFLAG_IGNPAR);
         iflag.parmrk  = getflag(tty.c_iflag, M1_IFLAG_PARMRK);
         iflag.inpck   = getflag(tty.c_iflag, M1_IFLAG_INPCK);
         iflag.istrip  = getflag(tty.c_iflag, M1_IFLAG_ISTRIP);
         iflag.inlcr   = getflag(tty.c_iflag, M1_IFLAG_INLCR);
         iflag.igncr   = getflag(tty.c_iflag, M1_IFLAG_IGNCR);
         iflag.icrnl   = getflag(tty.c_iflag, M1_IFLAG_ICRNL);
         iflag.ixon    = getflag(tty.c_iflag, M1_IFLAG_IXON);
         iflag.ixoff   = getflag(tty.c_iflag, M1_IFLAG_IXOFF);
         iflag.iuclc   = getflag(tty.c_iflag, M1_IFLAG_IUCLC);
         iflag.ixany   = getflag(tty.c_iflag, M1_IFLAG_IXANY);
         iflag.imaxbel = getflag(tty.c_iflag, M1_IFLAG_IMAXBEL);
         iflag.iutf8   = getflag(tty.c_iflag, M1_IFLAG_IUTF8);

         //---------------------------------------------------------------------
         // c_oflag: output modes;

         oflag.opost  = getflag(tty.c_oflag, M1_OFLAG_OPOST);
         oflag.olcuc  = getflag(tty.c_oflag, M1_OFLAG_OLCUC);
         oflag.ocrnl  = getflag(tty.c_oflag, M1_OFLAG_OCRNL);
         oflag.onlcr  = getflag(tty.c_oflag, M1_OFLAG_ONLCR);
         oflag.onocr  = getflag(tty.c_oflag, M1_OFLAG_ONOCR);
         oflag.onlret = getflag(tty.c_oflag, M1_OFLAG_ONLRET);
         oflag.ofill  = getflag(tty.c_oflag, M1_OFLAG_OFILL);
         oflag.ofdel  = getflag(tty.c_oflag, M1_OFLAG_OFDEL);

         getmask(tty.c_oflag, M1_OFLAG_NLDLY, oflag.nl);
         getmask(tty.c_oflag, M1_OFLAG_CRDLY, oflag.cr);
         getmask(tty.c_oflag, M1_OFLAG_TABDLY, oflag.tab);
         getmask(tty.c_oflag, M1_OFLAG_BSDLY, oflag.bs);
         getmask(tty.c_oflag, M1_OFLAG_VTDLY, oflag.vt);
         getmask(tty.c_oflag, M1_OFLAG_FFDLY, oflag.ff);

         //---------------------------------------------------------------------
         // c_cc: special characters;

         vchr.intr    = getvchr(tty.c_cc, M1_VID_VINTR);
         vchr.quit    = getvchr(tty.c_cc, M1_VID_VQUIT);
         vchr.erase   = getvchr(tty.c_cc, M1_VID_VERASE);
         vchr.kill    = getvchr(tty.c_cc, M1_VID_VKILL);
         vchr.eof     = getvchr(tty.c_cc, M1_VID_VEOF);
         vchr.eol     = getvchr(tty.c_cc, M1_VID_VEOL);
         vchr.eol2    = getvchr(tty.c_cc, M1_VID_VEOL2);
         vchr.swtc    = getvchr(tty.c_cc, M1_VID_VSWTC);
         vchr.start   = getvchr(tty.c_cc, M1_VID_VSTART);
         vchr.stop    = getvchr(tty.c_cc, M1_VID_VSTOP);
         vchr.susp    = getvchr(tty.c_cc, M1_VID_VSUSP);
         vchr.reprint = getvchr(tty.c_cc, M1_VID_VREPRINT);
         vchr.werase  = getvchr(tty.c_cc, M1_VID_VWERASE);
         vchr.lnext   = getvchr(tty.c_cc, M1_VID_VLNEXT);
         vchr.discard = getvchr(tty.c_cc, M1_VID_VDISCARD);

         //---------------------------------------------------------------------

         vchr.min  = getvval(tty.c_cc, M1_VID_VMIN);
         vchr.time = getvval(tty.c_cc, M1_VID_VTIME);

         //---------------------------------------------------------------------
      };

      //------------------------------------------------------------------------

      void_t trace() const {
         tracec("serial:conf:c:", cflag.str(0));
         tracec("serial:conf:c:", cflag.str(1));
         tracec("serial:conf:l:", lflag.str(0));
         tracec("serial:conf:l:", lflag.str(1));
         tracec("serial:conf:i:", iflag.str(0));
         tracec("serial:conf:i:", iflag.str(1));
         tracec("serial:conf:o:", oflag.str(0));
         tracec("serial:conf:o:", oflag.str(1));
         tracec("serial:conf:v:", vchr.str(0));
         tracec("serial:conf:v:", vchr.str(1));
         tracec("serial:conf:v:", vchr.str(2));
      }

      //------------------------------------------------------------------------
      // data;

      cflag_t cflag;
      lflag_t lflag;
      iflag_t iflag;
      oflag_t oflag;
      vchr_t  vchr;

      //------------------------------------------------------------------------
   };

   //---------------------------------------------------------------------------
   // ctor dtor;

   serial_t() : ini(0), fd(-1) {
   }
   serial_t(string_t const &dev0) : ini(1), fd(-1) {
      init(dev0);
   }
   ~serial_t() {
      if(ini) fini();
   }
   void_t init(string_t const &dev0) {
      dev = dev0;
      setconfig();
   }
   // sim4hi 180629: changed return type from int_t to void_t as fix for Coverity CID-148281
   void_t fini() {
      close();
   }

   //---------------------------------------------------------------------------
   // config;

   int_t getconfig() { // retreive from device;

      open(dev);
      flush();

      struct termios tty;
      mset(&tty, 0, sizeof(tty));

      if(tcgetattr(fd, &tty)) {
         tracec("serial:getconfig:tcgetattr:err:", err_str());
         return -1;
      }

      conf.tty(tty); // store locally;

      return 0;
   }
   int_t setconfig() { // apply to device;

      struct termios tty = conf.tty(); // convert;

      open(dev);

      // apply;
      int_t const res = tcsetattr(fd, TCSANOW, &tty); // TCSADRAIN=1; TCSAFLUSH=2; TCSANOW=3;

      if(-1 == res) {
         tracec("serial:setconfig:tcsetattr:err:", err_str());
         return -1;
      }
      return 0;
   }

   //---------------------------------------------------------------------------
   // open;

   int_t open(string_t const &dev0) {

      //------------------------------------------------------------------------

      if(0 <= fd) {
         if(dev0 != dev) {
            close();
            dev = dev0;
         } else {
            return -1; // already open;
         }
      }
      dev = dev0;
      tracec("serial:open:dev:", dev);

      //------------------------------------------------------------------------
      // trace interface settings;

      //conf.trace();

      //------------------------------------------------------------------------
      // open device;

      // no control; do not care for DCD signal line;
      fd = ::open(dev.at(), O_RDWR | O_NOCTTY | O_NDELAY); // O_SYNC;

      if(-1 == fd) {
         tracec("serial:open:err:", err_str());
      }
      if(-1 == ::fcntl(fd, F_SETFL, 0)) {
         tracec("serial:open:fctl:err:", err_str());
      }

      //------------------------------------------------------------------------

      return fd;
   }

   //---------------------------------------------------------------------------

   void_t close() {
      if(0 <= fd) {
         tracec("serial:close");
         ::close(fd);
         fd = -1;
      }
   }

   //---------------------------------------------------------------------------

   int_t write(string_t const &in) {

      //------------------------------------------------------------------------

      asrt(0 <= fd);

      //------------------------------------------------------------------------
      // prettyprint;

      #if 0
      string_t in1(hed);
      for(int_t i = 0; i < in.size(); ++i) {
         in1.cat("0x", *in.at(i), " ");
      }
      tracec("serial:write:in:", in.size(), ":", in1);
      #endif

      //------------------------------------------------------------------------
      // write to device;

      int_t bytes = ::write(fd, in.at(), in.size());

      if(0 > bytes) {

         tracec("serial:write:err:", err_str());
         return -1;
      }

      //------------------------------------------------------------------------

      return 0;
   }

   //---------------------------------------------------------------------------

   string_t read(int_t const max = -1) {

      //------------------------------------------------------------------------

      asrt(0 <= fd);

      //------------------------------------------------------------------------
      // select;

      fd_set fds;
      FD_ZERO(&fds);
      FD_SET(fd, &fds);
      //FD_SET(fd1, &fds);

      struct timeval timeout;
      timeout.tv_sec  = 8;
      timeout.tv_usec = 0;

      int const max_fd = fd;

      int_t const res = ::select(max_fd + 1, &fds, 0, 0, &timeout);

      if(0 > res) {

         tracec("serial:read:select:err:", err_str());
         return "select";

      } else if(0 == res) {

         tracec("serial:read:select:timeout:", (int_t)timeout.tv_sec);
         return "timeout";
      }
      asrt(0 < res);

      //------------------------------------------------------------------------
      // switch source;

      //if(FD_ISSET(fd, &fds));

      //------------------------------------------------------------------------
      // read from device;

      string_t out;
      byte_t buf;
      int_t n = 0, sum = 0;

      while(n = ::read(fd, &buf, 1)) { // '\r'
         //tracec("serial:read:buf:", buf, " n:", n);
         out.cat(buf);
         sum += n;
         if(-1 != max && max <= sum) break;
      }

      if(0 > n) {
         tracec("serial:read:err:", strerror(errno));
         return "err";
      }

      //------------------------------------------------------------------------
      // prettyprint;

      //tracec("serial:read:out:", out, " ", hed, out);

      //------------------------------------------------------------------------

      return out;
   }

   //---------------------------------------------------------------------------

   int_t flush() const {

      asrt(0 <= fd);

      if(0 > ::tcflush(fd, TCIOFLUSH)) { // TCIFLUSH, TCOFLUSH;
         tracec("serial:flush:err:", err_str());
         return -1;
      }
      return 0;
   }

   //---------------------------------------------------------------------------
   // data;

   int_t ini;
   string_t dev;
   int fd;
   conf_t conf;

   //---------------------------------------------------------------------------
};

//------------------------------------------------------------------------------
// keyboard;

struct kbd_t : public serial_t,
               public single_t<kbd_t> {

   friend struct single_t<kbd_t>;

   kbd_t()
   #if (M1_LIB_LINUX == M1_LIB)
   : serial_t("/dev/stdin")
   #endif
   {
      tracec("kbd:ctor");
   }
   ~kbd_t() {
      tracec("kbd:dtor");
   }
   #if (M1_LIB_LINUX == M1_LIB)
      byte_t key() {
         getconfig(); // get current tty settings;
         conf.lflag.icanon = off;
         conf.lflag.echo   = off;
         conf.vchr.min  = 1;
         conf.vchr.time = 0;
         setconfig();
         string_t out = serial_t::read(1);
         conf.lflag.icanon = on;
         conf.lflag.echo   = on;
         setconfig();
         return out[0];
      }
   #else
      byte_t key() {
         return ::getch();
      }
   #endif
};

//------------------------------------------------------------------------------

#endif // M1_COM_M1 == M1_COM

//------------------------------------------------------------------------------

} // m1;

//------------------------------------------------------------------------------

#endif // M1_COM_NONE;

// -----------------------------------------------------------------------------
// org;

#if !(M1_ORG_NONE == M1_ORG)

// -----------------------------------------------------------------------------

#include <execinfo.h>

// -----------------------------------------------------------------------------

namespace m1 {

// -----------------------------------------------------------------------------

#if (M1_ORG_M1 == M1_ORG)

// -----------------------------------------------------------------------------
// startup initialisation comparison;

//                 machine  process  binary   runtime      before  always  inter     portable
//                 reboot   restart  rebuild  modificaion  main    inited  process
// hardcoded defs  +         +        -       +            +        +      + (vers)  +
// compile defs    +         +        -       +            +        -      + (vers)  +
// env vars        -         -        +       -            +        -      -         -
// cmdline args    -         -        +       +            -        -      -         +
// config file     +         +        +       -            +        -      -         +

// -----------------------------------------------------------------------------

struct org_t : single_t<org_t> {

   // --------------------------------------------------------------------------

   friend struct single_t<org_t>;

   // --------------------------------------------------------------------------

   org_t() {
      tracec("org:ctor");
      startup();
   }
   ~org_t() {
      shutdown();
      tracec("org:dtor");
   }

   // --------------------------------------------------------------------------
   // signal handler;

   static void_t signal_cb(int_t sig, int_t pid, int_t uid) {

      tracec("org:signal_cb:sig:", estr2((signal_e)sig), " pid:", pid, " uid:", uid);

      //inst().specialexit(); // final cleanup of static resources that
                              // cannot call their dtors anymore;

      // build with -gdynamic;
      void_t *stack[13];
      int_t const maxlines = sizeof(stack) / sizeof(void_t*);
      int_t const size  = backtrace(stack, maxlines);
      int_t const lines = min(maxlines, size);

      byte_t **symbols = backtrace_symbols(stack, lines);

      for(int_t i = 0; i < lines; ++i) {
         tracec("org:signal_cb:", i, ":", (byte_t*)symbols[i]);
      }
      free(symbols);

      if(M1_SIGINT == sig) {
         tracec("org:signal_cb:", estr2((signal_e)sig), " this is the end.");
         abort();
      }
      exit(0);
   }

   // --------------------------------------------------------------------------
   // called on registered signals;

   void_t specialexit() {

      tracec("org:specialexit");

      // -----------------------------------------------------------------------
      // put abnormal termination cleanup here;
      // explicit call of static singleton dtors;

      //types::shared_t::inst().fini();
      //signal_t::inst().fini();  // no heap data;
      //normal_t::inst().fini();

      // -----------------------------------------------------------------------

      // cleanup self;
      this->~org_t();

      // -----------------------------------------------------------------------
   }

   // --------------------------------------------------------------------------
   // mandatory: called automatically ahead of main;

   #define M1_ORG_REGSIGNALS

   void_t startup() {

      tracec("org:startup");

      // -----------------------------------------------------------------------
      // check double startup;

      static int_t ini = 0;
      if(ini) {
         tracec("org:startup:double init");
         return;
      }
      ini = 1;

      // -----------------------------------------------------------------------
      // register signals;

      #if defined M1_ORG_REGSIGNALS

         signal_t &sig = m1::signal_t::inst();

         int_t const res = sig.reg(M1_SIGSEGV, signal_cb) ||
                           sig.reg(M1_SIGABRT, signal_cb) ||
                           sig.reg(M1_SIGINT,  signal_cb) ||
                           sig.reg(M1_SIGTERM, signal_cb);

         tracec("org:startup:signals:", 0 == res ? "good" : "bad");

      #else

         tracec("org:startup:signals:inactive");

      #endif

      // -----------------------------------------------------------------------
   }

   // --------------------------------------------------------------------------
   // mandatory called on dtor of static singleton;

   void_t shutdown() {

      // unregister signal callback;
      signal_t &sig = m1::signal_t::inst();

      int_t const res = sig.reg(M1_SIGSEGV, 0) ||
                        sig.reg(M1_SIGABRT, 0) ||
                        sig.reg(M1_SIGINT,  0) ||
                        sig.reg(M1_SIGTERM, 0);

      tracec("org:shutdown:signals:0");
   }

   // --------------------------------------------------------------------------
   // optionally: call on main;

   void_t save(int_t const argc, byte_t **argv) {
      for(int_t i = 0; i < argc; ++i) {
         args.append(*argv + i);
         tracec("org:save:[", i, "]:", args[i]);
      }
   }

   // --------------------------------------------------------------------------
   // data;

   strings_t args;

   /* ----------------------------------------------------------------------- */
};

// -----------------------------------------------------------------------------
// instantiate ahead of client;

static org_t &o = org_t::inst();

// -----------------------------------------------------------------------------

#endif // M1_ORG_M1 == M1_ORG;

// -----------------------------------------------------------------------------

} // m1;

// -----------------------------------------------------------------------------

#endif // M1_ORG_NONE;

//------------------------------------------------------------------------------

#endif // M1_HPP;

//------------------------------------------------------------------------------
