/**
 * @defgroup RootDaemon RootDaemon
 * @author Matthias Thömel
 * 
 * @brief Public interface for the RootDaemon to start and stop the process
 * @{
 */

#ifndef _GMPROOTDAEMON_H_
#define _GMPROOTDAEMON_H_

#include "ILocalSPM.h"
#include "GMPCommands.h"
#include "GMPClientProtocol.h"

/**
 * definition of class RootDaemon
 * encapsulate communication to the root daemon
 */
 class RootDaemon : public ILocalSPM
{

public:

     //SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the RootDaemon componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
     RootDaemon(tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~RootDaemon();

    /**
     * This function is used by LocalSPM to create the RootDaemon.
     * @attention: running in SPM thread context
     * Inform LocalSPM that Create is ready -> CreateDone(0) 
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the RootDaemon initialization.
     * @attention: running in SPM thread context
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * This function is used by LocalSPM to start the RootDaemon.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the RootDaemon process.
     *
     * @return < 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function is used by LocalSPM to stop the RootDaemon.
     * @attention: running in SPM thread context
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * This function is used by LocalSPM to cleanup the RootDaemon.
     * @attention: running in SPM thread context
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * transfers a command to the root daemon and waits for the result code
     *
     * @param[in] commandNo command number. This depends on the command
     * implementation for the media player done in the root daemon component
     * @param[in] arguments arguments to the command. format is command specific
     *
     * @return result code of the issued command. depends on the command implementation
     */
    int Command(const int commandNo, const char *arguments, const int timeoutMS = 0);

private:

    GMPCommands mGmpCommands;
    GMPClientProtocol mClient;
};

#endif  //_GMPROOTDAEMON_H_

/** @} */
