#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#ifdef TARGET_BUILD
#include "trcGenProj/Header/PlayerManagerTest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#endif
#endif

#include "PlayerManager.h"
#include "FunctionTracer.h"
#include "VarTrace.h"
#include "LocalSPM.h"
#include "CppUnitDefinitions.h"
#include "Dispatcher.h"
#include "RequestResponseSM.h"
#include "MediaPlayer_ErrorCodes.h"

#include "PlayerManagerTest.h"
static tListID m_ListID = LIST_ID_NONE;

#define WAIT_FOR_HMI_STATE(stateToReach, timeoutInSec) \
    if (CheckPlaybackState((stateToReach), (timeoutInSec)) != MP_NO_ERROR) {\
        tHMIPlaybackState playstatus = HMI_PBS_UNDEFINED;\
        CPPUNIT_ASSERT(stateToReach == playstatus);\
    }

tResult PlayerManagerTest::WaitForPosition(const tPlaytime position)
{
    tResult res = MP_NO_ERROR;

    tPlaytime elapsedPlaytime = 0;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    enum {secMax = 5};
    tUInt sec;

    for(sec=secMax; sec; sec--) {
        res = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        if(res != MP_NO_ERROR) break;

        if(position <= elapsedPlaytime) break;
        sleep(1);
    }

    if(0 == sec) { //requested position not found in time
        res = MP_ERR_MISC_TIMEOUT;
    }
    return res;
}

#ifdef useTestFW
void PlayerManagerTest::SourceActivityCallback(cTestData *objTestData)
{
    ENTRY
    ETG_TRACE_USR2(("Start PlayerManagerTest::SourceActivityCallback, msgType=%d", objTestData->eMSGTYPE));

    if(objTestData !=NULL)
    {
        switch(objTestData->eMSGTYPE)
        {
            case EV_UPDATEPLAYTIME:
                mNumberOfUpdatesCalled++;
                break;
            case EV_UPDATENOWPLAYING:
                mNumberOfUpdatesCalled++;
                break;
            case EV_UPDATEPLAYBACKSTATE:
                nresultObatined = 0;
                break;
            default:
                break;
        }
    }

    ASYNC_WAIT_RELEASE
}

void PlayerManagerTest::testCallback(cTestData *objTestData)
{
    ENTRY
     (this->*funPtr)(objTestData);
}
#endif

PlayerManagerTest::PlayerManagerTest()
{
    funPtr = NULL;
    m_ListID = LIST_ID_NONE;
    this->UcssortFolderCreated = FALSE;
}

void PlayerManagerTest::setUp()
{
#ifdef useTestFW
    pthread_mutex_init(&mutex,NULL);
    pthread_cond_init(&condition,NULL);
#endif

    ticks.begin();
}

void PlayerManagerTest::tearDown()
{
    ticks.elapsed();

    if (FALSE != this->UcssortFolderCreated) {

        tResult ret = MP_NO_ERROR;
        tDeviceID deviceID;
        tDeviceInfo deviceInfo;
        tPath command;

        ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB-2");
        if(MP_NO_ERROR == ret) {

            ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID, 0);
            if(MP_NO_ERROR == ret) {

                /* remove the test directories */
                snprintf(command, sizeof(command), "rm -rf %s/UCSSort_*", deviceInfo.mountPoint);
                system(command);
            }
        }
    }
    this->UcssortFolderCreated = FALSE;

#ifdef useTestFW
    pthread_mutex_destroy(&mutex);
    pthread_cond_destroy(&condition);
#endif
}

/**********************************************************************
 * Test cases
 *********************************************************************/

void PlayerManagerTest::Allocate()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendAllocate */
    tAudioOutputDevice audioOutputDevice;
    strncpy_r(OUT audioOutputDevice, USB_ALSA_DEVICENAME, IN sizeof(audioOutputDevice));

    ret = LocalSPM::GetPlayerManager().SendAllocate(IN audioOutputDevice);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(2); //Wait a little bit until IPC is initialized by first Allocate call

    // test result
    tAllocateState allocateState;
    ret = LocalSPM::GetPlayerManager().GetAllocateState(OUT allocateState, OUT audioOutputDevice);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(ALS_ALLOCATED == allocateState);
    CPPUNIT_ASSERT(!strcmp(audioOutputDevice, USB_ALSA_DEVICENAME));
}

void PlayerManagerTest::DeAllocate()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendDeAllocate */
    ret = LocalSPM::GetPlayerManager().SendDeAllocate();
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    tAllocateState allocateState;
    tAudioOutputDevice audioOutputDevice;
    ret = LocalSPM::GetPlayerManager().GetAllocateState(OUT allocateState, OUT audioOutputDevice);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(ALS_DEALLOCATED == allocateState);
}

/* disable test framework for source activity on/off because this leads to error */
#ifdef useTestFW
#undef useTestFW
#endif

void PlayerManagerTest::SourceActivityOn()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

#ifdef useTestFW
    this->m_strTestcase = "SourceActivityOn";
    this->bLetFWValidateResult = false ;
    nresultObatined = -1;
    //Test Bench call

    CListener *oListener = CListener::getCListenerInstance();
    //CListener Listener;
    //CListener *oListener = &Listener;
    oListener->registerCallback(this);
    oListener->startListen();
    //oListener->registerEvent(EV_UPDATEPLAYTIME);
    //oListener->registerEvent(EV_UPDATENOWPLAYING);
    oListener->registerEvent(EV_UPDATEPLAYBACKSTATE);
    PlayerManagerTest::funPtr =  &PlayerManagerTest::SourceActivityCallback;

    //useTestFw(this);
#endif
    /* Call SendSourceActivity */
    tSourceActivity sourceActivity = SA_ON;
    ret = LocalSPM::GetPlayerManager().SendSourceActivity(IN sourceActivity);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#ifdef useTestFW
    ASYNC_WAIT_START(1);
    CPPUNIT_ASSERT(nresultObatined == 0);
    oListener->stopListen();
    delete oListener;
#else
    sleep(2);
#endif

    // test result
    tBoolean isActive = FALSE;
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(FALSE != isActive);
}

void PlayerManagerTest::SourceActivityOff()
{
    SourceActivityOff(1);
}

#ifdef useTestFW
#undef useTestFW // not used for SourceActivityOn/Off tests
#endif

void PlayerManagerTest::SourceActivityOff(int checkCallbackResult)
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
#ifdef useTestFW
    this->m_strTestcase = "SourceActivityOff";
    this->bLetFWValidateResult = false ;
    nresultObatined = -1;
    //Test Bench call

    CListener *oListener = CListener::getCListenerInstance();
    oListener->registerCallback(this);
    oListener->startListen();
    //oListener->registerEvent(EV_UPDATEPLAYTIME);
    //oListener->registerEvent(EV_UPDATENOWPLAYING);
    oListener->registerEvent(EV_UPDATEPLAYBACKSTATE);
    PlayerManagerTest::funPtr =  &PlayerManagerTest::SourceActivityCallback;

    //useTestFw(this);
#else
    (void)checkCallbackResult;
#endif

    /* Call SendSourceActivity */
    tSourceActivity sourceActivity = SA_OFF;
    ret = LocalSPM::GetPlayerManager().SendSourceActivity(IN sourceActivity);

#ifdef useTestFW
    /* wait for callback to be called */
    ASYNC_WAIT_START(10);

    /* stop th elistener, the callback result is checked later on */
    oListener->stopListen();
    delete oListener;
#else
    sleep(1);
#endif

    // test result
    tBoolean isActive = TRUE;
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(FALSE == isActive);

#ifdef useTestFW
    /* check the callback result */
    if (checkCallbackResult) {
        ETG_TRACE_USR1(("debug: check nresultObatined=%d\n", nresultObatined));
        CPPUNIT_ASSERT(nresultObatined == 0);
    }
#endif
}

void PlayerManagerTest::NewList_CreateMPIndexedList()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Create a new indexed list */
    tListSize listSize = 0;
    tListType listType = LTY_SONG;
    tStreaming streaming = false;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;
    tDeviceID deviceID;

    /* Get deviceID */
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Set device to active */
    ret = LocalSPM::GetDBManager().SetActiveDevice(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check repeat mode and set it to RPT_ALL if not already in this state */
    tRepeatMode oldRepeatMode;
    ret = LocalSPM::GetListControl().GetRepeatMode(OUT oldRepeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    if(RPT_ALL != oldRepeatMode)
    {
        ret = LocalSPM::GetListControl().SetRepeatMode(IN RPT_ALL);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    }

    /* Set the filter tags */
    tag1.tag = 0;
    tag2.artist = 0;
    tag3.composer = 0;
    tag4.album = 0;

    /* Call create list */
    ret = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
            OUT m_ListID,
            OUT listSize,
            IN listType,
            IN streaming,
            IN tag1,
            IN tag2,
            IN tag3,
            IN tag4,
            IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != m_ListID);
    CPPUNIT_ASSERT(0 < listSize);

    /* Call SendNewList */
    tIndex index = INDEX_LAST_ACTIVE_OBJECT;
    tPlaytime offset = 0;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN m_ListID, IN index, IN offset);

    // test result
    tNowPlaying nowPlaying;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (m_ListID == nowPlaying.listID) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(m_ListID == nowPlaying.listID);
}

void PlayerManagerTest::NewList_ActiveMediaDeviceSet()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get last played list of a specific device */
    tListID listID = LIST_ID_NONE;
    tIndex listIndex = INDEX_NONE;
    tPlaytime offset = PLAYTIME_NONE;
    tDeviceID deviceID;

    /* Get device ID */
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(ret == 0);

    ret = LocalSPM::GetListControl().GetLastPlayedList(OUT listID, OUT listIndex, OUT offset, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != listID);
#ifndef TARGET_BUILD
    CPPUNIT_ASSERT(0 < offset);
#endif

    /* Call SendNewList */
    tIndex index = INDEX_LAST_ACTIVE_OBJECT;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN index, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif

    tNowPlaying nowPlaying;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (listID == nowPlaying.listID) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(listID == nowPlaying.listID);

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for (int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (offset < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(offset < elapsedPlaytime);
}

void PlayerManagerTest::NewList_CreateMPFileList()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Create a new file list */
    tListID fileListID;
    tListSize listSize = 0;
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tPath path;
    tDeviceID deviceID;

    /* Get device ID */
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, IN "USB-2");
    CPPUNIT_ASSERT(ret == 0);

    /* Set the path */
/*
#ifndef TARGET_BUILD
    strncpy_r(path, get_current_dir_name(), sizeof(path));
    strncat_r(path, "/Customer/Simulation/CustomControl/stick_3", sizeof(path));
#else
    strncpy_r(path, "/opt/bosch/test/data/GMP/stick_3", sizeof(path));
#endif
*/
    strncpy_r(OUT path, IN "/", IN sizeof(path));

    /* Call create list */
    ret = LocalSPM::GetListControl().CreateMediaPlayerFileList(
            OUT fileListID,
            OUT listSize,
            OUT numberOfMediaObjects,
            OUT numberOfPlayLists,
            OUT numberOfFolders,
            IN LTY_FILELIST,
            IN path,
            IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != fileListID);
    CPPUNIT_ASSERT(0 < numberOfMediaObjects);

    /* Call SendNewList */
    tIndex index = 0;
    tPlaytime offset = 0;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN fileListID, IN index, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(4); //Play at least 4s

#if 0 // todo remove
    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif

    tNowPlaying nowPlaying;
    for (int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (fileListID == nowPlaying.listID) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(fileListID == nowPlaying.listID);
    CPPUNIT_ASSERT(!strcmp((const char *)nowPlaying.object.title, "04_lakshya.mp3"));

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (offset < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(offset < elapsedPlaytime);
}

void PlayerManagerTest::NextBuffer()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    if(!(LocalSPM::GetDataProvider().UseMediaEngine()))
    {
        return; //Test makes no sense because only new PlayerEngine supports buffering of next object
    }

    /* Enable GaplessPlay configuration */
    int gaplessPlay = LocalSPM::GetDataProvider().GaplessPlay();
    if (0 == gaplessPlay)
    {
        LocalSPM::GetDataProvider().GaplessPlay = 1;
    }

    /* Check if an media object is playing */
    tHMIPlaybackState currentPlaystatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT currentPlaystatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == currentPlaystatus);

    /* Skip to next song to store the position which is only done in GaplesPlay mode */
    tNextPrevSkipCount steps = 1;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Simulate buffer request from MediaEngine */
    tPEHandle handle = 1234;
    tPEPlaybackState status = PE_PBS_BUFFERINGSTATE;
    me::reason_e reason = REASON_NEW_TRACK;
    me::speed_e speed = ME_SPEED_NONE;
    ret = LocalSPM::GetPlayerManager().ForwardPlaybackStatusNew(IN handle, IN status, IN reason, IN speed);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(3); //Wait at least 3s

    // test result
    tHMIPlaybackState currentPlaystatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif

    tPosition position;
    ret = LocalSPM::GetPlayerManager().GetBufferIndex(OUT position);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(1 == position);

    /* Reset configuration */
    LocalSPM::GetDataProvider().GaplessPlay = gaplessPlay;
}

void PlayerManagerTest::SeekTo()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMediaObject mediaObject;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(OUT mediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(1 < mediaObject.totalPlaytime);

    tPlaytime position;
    if((PLAYTIME_NONE == mediaObject.totalPlaytime)
       ||
       (PLAYTIME_END_OF_OBJECT == mediaObject.totalPlaytime))
    {
        position = 10000; //10s
    }
    else
    {
        position = mediaObject.totalPlaytime / 2;
    }

    /* Call SendSeekTo */
    ret = LocalSPM::GetPlayerManager().SendSeekTo(IN position, PPF_ABSOLUTE);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    //sleep(1);
    ret = WaitForPosition(IN position);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test result
    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(mediaObject.objectID == objectID);
    ETG_TRACE_USR1(("position=%d, elapsedPlaytime=%d", position, elapsedPlaytime));
    CPPUNIT_ASSERT(position <= elapsedPlaytime);
}

void PlayerManagerTest::PlaybackActionNext()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* wait for the playing state */
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);

    tNowPlaying lastPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT lastPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Call SendPlaybackAction(NEXT) */
    tNextPrevSkipCount steps = 2;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(6); //Play at least 6s

#if 0 // todo remove
    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif

    // TEST
    //while(1) sleep(1);

    tNowPlaying nowPlaying;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(lastPlaying.listID == nowPlaying.listID);
        if (lastPlaying.objectID != nowPlaying.objectID) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(lastPlaying.objectID != nowPlaying.objectID);

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (0 < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(0 < elapsedPlaytime);
}

void PlayerManagerTest::PlaybackActionMultipleNext()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tNowPlaying lastPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT lastPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR1(("PlayerManagerTest::PlaybackActionMultipleNext => LastPlaying objectID : %u, Position : %u", lastPlaying.objectID, lastPlaying.position));

    /* Call SendPlaybackAction(NEXT) for three times to bring multiple Next */
    /* 1st Request => SendPlaybackAction(NEXT)  */
    tNextPrevSkipCount steps = 1;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR1(("PlayerManagerTest::PlaybackActionMultipleNext => 1st Next objectID : %u, Position : %u", nowPlaying.objectID, nowPlaying.position));

    /* 2nd Request => SendPlaybackAction(NEXT)  */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* 3rd Request => SendPlaybackAction(NEXT)  */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    tUInt sleepTime = (LocalSPM::GetDataProvider().MultipleNextWaitTime() / 1000) + 1;
    sleep(sleepTime);

    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR1(("PlayerManagerTest::PlaybackActionMultipleNext => NowPlaying object ID : %u, Position : %u", nowPlaying.objectID, nowPlaying.position));
    CPPUNIT_ASSERT(lastPlaying.listID == nowPlaying.listID);
    CPPUNIT_ASSERT(lastPlaying.objectID != nowPlaying.objectID);
    CPPUNIT_ASSERT((lastPlaying.position)+3 == nowPlaying.position);

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime,OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);

        if (0 < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(0 < elapsedPlaytime);
}

void PlayerManagerTest::PlaybackActionPrev()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tNowPlaying lastPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT lastPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Call SendPlaybackAction(PREV) */
    tNextPrevSkipCount steps = 1;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_PREV, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(3); //Play at least 3s

#if 0 // todo remove
    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif

    tNowPlaying nowPlaying;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(lastPlaying.listID == nowPlaying.listID);
        if (lastPlaying.objectID != nowPlaying.objectID) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(lastPlaying.objectID != nowPlaying.objectID);

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime,OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (0 < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(0 < elapsedPlaytime);
}

void PlayerManagerTest::PlaybackActionMultiplePrev()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tNowPlaying lastPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT lastPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR1(("PlayerManagerTest::PlaybackActionMultiplePrev => LastPlaying objectID : %u, Position : %u", lastPlaying.objectID, lastPlaying.position));

    /* Call SendPlaybackAction(PREV) */
    tNextPrevSkipCount steps = 1;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_PREV, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_PREV, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    tUInt sleepTime = (LocalSPM::GetDataProvider().MultipleNextWaitTime() / 1000) + 1;
    sleep(sleepTime);

    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR1(("PlayerManagerTest::PlaybackActionMultiplePrev => NowPlaying object ID : %u, Position : %u", nowPlaying.objectID, nowPlaying.position));
    CPPUNIT_ASSERT(lastPlaying.listID == nowPlaying.listID);
    CPPUNIT_ASSERT(lastPlaying.objectID != nowPlaying.objectID);
    CPPUNIT_ASSERT((lastPlaying.position) > nowPlaying.position);

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (0 < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(0 < elapsedPlaytime);
}

void PlayerManagerTest::FastNextInALoop()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(NEXT) */
    tNextPrevSkipCount steps = 1;
    for(tUInt i=0; i<10; i++)
    {
        ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        usleep(50000); //50ms
    }

#if 0 // todo remove
    sleep(3); //Play at least 3s

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif
}

void PlayerManagerTest::PlaybackActionPause()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(PAUSE) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_PAUSE);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(3); //Pause at least 3s

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PAUSED == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PAUSED, 30);
#endif
}

void PlayerManagerTest::PlaybackActionResume()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(PLAY) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_PLAY);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(3); //Play at least 3s

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif
}

void PlayerManagerTest::PlaybackActionStop()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(STOP) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_STOP);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(10);

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PAUSED == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PAUSED, 30);
#endif
}

void PlayerManagerTest::PlaybackActionFfwd()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(FFWD_START) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FFWD_START);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(6); //Do Ffwd for 6s to have a big elapsed time

#if 0
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_FFWD == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_FFWD, 30);
#endif

    /* Call SendPlaybackAction(FFWD_STOP) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FFWD_STOP);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(3);

    // test result
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif
}

void PlayerManagerTest::PlaybackActionFrev()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Call SendPlaybackAction(FREV_START) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FREV_START);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(3); //Do Frev for 3s

#if 0 // todo remove
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_FREV == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_FREV, 30);
#endif

    /* Call SendPlaybackAction(FREV_STOP) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FREV_STOP);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0
    sleep(1);

    // test result
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif
}

void PlayerManagerTest::PlaybackActionFFwdInDealloate()
{
    ENTRY

    /*set playermanager as inactive*/
    SourceActivityOff(0);
    DeAllocate();

    tResult ret = MP_NO_ERROR;

    /*read the pre condition for the test
     * like "Active" , "NowPlaying" ,"PlayBackState" , "Time"*/
    tBoolean isActive = FALSE;
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(FALSE == isActive);

    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(nowPlaying);

    tHMIPlaybackState playbackState;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playbackState);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playbackState);

    tPlaytime elapsedTime,totalTime;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedTime, OUT totalTime, OUT objectID);
    VARTRACE(elapsedTime);
    VARTRACE(totalTime);

    /* Call SendPlaybackAction(FFWD_START) when Playermanager is not active*/
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FFWD_START);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(3); //Do Ffwd for 3s

#if 0 // todo remove
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playstatus);
    CPPUNIT_ASSERT(HMI_PBS_FFWD == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_FFWD, 30);
#endif

    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedTime, OUT totalTime,OUT objectID);
    VARTRACE(elapsedTime);
    VARTRACE(totalTime);

    /* Call SendPlaybackAction(FFWD_STOP) when playermanager is not active*/
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FFWD_STOP);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(1);

    // test result
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playstatus);
    CPPUNIT_ASSERT(playstatus == playbackState);
#else
    WAIT_FOR_HMI_STATE(playbackState, 30);
#endif

    /*set playermanager as active again*/
    Allocate();
    SourceActivityOn();
    sleep(2);

    /*@todo :check whether playtime is resumed from the last simulted elapsed time*/

}

void PlayerManagerTest::PlaybackActionFRevInDealloate()
{

    ENTRY

    /*set playermanager as inactive*/
    SourceActivityOff(0);
    DeAllocate();

    tResult ret = MP_NO_ERROR;

    /*read the pre condition for the test
     * like "Active" , "NowPlaying" ,"PlayBackState" , "Time"*/
    tBoolean isActive = FALSE;
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(FALSE == isActive);

    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(nowPlaying);

    tHMIPlaybackState playbackState;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playbackState);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playbackState);

    tPlaytime elapsedTime,totalTime;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedTime, OUT totalTime, OUT objectID);
    VARTRACE(elapsedTime);
    VARTRACE(totalTime);

    /* Call SendPlaybackAction(FFWD_START) when Playermanager is not active*/
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FREV_START);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(3); //Do Frev for 3s

#if 0 // todo remove
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playstatus);
    CPPUNIT_ASSERT(HMI_PBS_FREV == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_FREV, 30);
#endif

    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedTime, OUT totalTime, OUT objectID);
    VARTRACE(elapsedTime);
    VARTRACE(totalTime);

    /* Call SendPlaybackAction(FFWD_STOP) when playermanager is not active*/
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_FREV_STOP);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0 // todo remove
    sleep(1);

    // test result
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    VARTRACE(playstatus);
    CPPUNIT_ASSERT(playstatus == playbackState);
#else
    WAIT_FOR_HMI_STATE(playbackState, 30);
#endif

    /*set playermanager as active again*/
    Allocate();
    SourceActivityOn();
    sleep(2);

    /*@todo :check whether playtime is resumed from the last simulted elapsed time*/


}

void PlayerManagerTest::StreamError_FileNotFound()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Send STREAM_ERROR message to PlayerManagerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    me::reason_e reason = REASON_FORMAT_ERROR;
    ret = LocalSPM::GetPlayerManager().ParameterSTREAM_ERROR(OUT parameterString, IN size, IN reason);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetPlayerManager().SendEventByName("STREAM_ERROR", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

#if 0 // todo remove
    sleep(2); //Wait for error handling at least 2s

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_PLAYING == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);
#endif
}

void PlayerManagerTest::StreamError_DevNotFound()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Send STREAM_ERROR message to PlayerManagerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    me::reason_e reason = REASON_DEVICE_ERROR;
    ret = LocalSPM::GetPlayerManager().ParameterSTREAM_ERROR(OUT parameterString, IN size, IN reason);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetPlayerManager().SendEventByName("STREAM_ERROR", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

#if 0 // todo remove
    sleep(2); //Wait for error handling at least 2s

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_STOPPED == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_STOPPED, 30);
#endif
}

void PlayerManagerTest::NoPlayableContent()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Create a new file list */
    tListID fileListID;
    tListSize listSize = 0;
    tNumberOfMediaObjects numberOfMediaObjects;
    tNumberOfPlayLists numberOfPlayLists;
    tNumberOfFolders numberOfFolders;
    tPath path;
    tDeviceID deviceID;

    /* Enable DBFileBrowsingByDB configuration */
    int browsingByDB = LocalSPM::GetDataProvider().DBFileBrowsingByDB();
    if (0 == browsingByDB)
    {
        LocalSPM::GetDataProvider().DBFileBrowsingByDB = 1;
    }

    /* Set FileErrorHandling configuration to FEH_MARK_OBJECT */
    tFileErrorHandling fileErrorHandling = (tFileErrorHandling)LocalSPM::GetDataProvider().FileErrorHandling();
    if (FEH_NO_ACTION == fileErrorHandling)
    {
        LocalSPM::GetDataProvider().FileErrorHandling = (int)FEH_MARK_OBJECT;
    }

    /* Get device ID */
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(ret == 0);

    /* Check repeat mode and set it to RPT_NONE if not already in this state */
    tRepeatMode oldRepeatMode;
    ret = LocalSPM::GetDBManager().GetRepeatMode(OUT oldRepeatMode, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    if(RPT_NONE != oldRepeatMode)
    {
        ret = LocalSPM::GetDBManager().SetRepeatMode(IN deviceID, IN RPT_NONE);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    }

    /* Set the path */
    strncpy_r(OUT path, IN "/music/Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET/", IN sizeof(path));

    /* Call create list */
    ret = LocalSPM::GetListControl().CreateMediaPlayerFileList(
            OUT fileListID,
            OUT listSize,
            OUT numberOfMediaObjects,
            OUT numberOfPlayLists,
            OUT numberOfFolders,
            IN LTY_FILELIST,
            IN path,
            IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != fileListID);
    CPPUNIT_ASSERT(0 < numberOfMediaObjects);

    /* Call SendNewList */
    tIndex index = 0;
    tPlaytime offset = 0;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN fileListID, IN index, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    /* Get indexed list slice from a file list (This will work because we have a LTY_BROWSELIST */
    vector<tMediaObject> mediaObjectVector;
    ret = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN fileListID, 0, IN listSize);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(mediaObjectVector.size() == 13);

    /* Set all objects to notPlayable */
    for(tUInt i=0; i<mediaObjectVector.size(); i++)
    {
        mediaObjectVector[i].notPlayable = FNP_NOT_PLAYABLE;
        ret = LocalSPM::GetDBManager().UpdateMediaObject(IN mediaObjectVector[i]);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    }

    /* Call SendPlaybackAction(NEXT) */
    tNextPrevSkipCount steps = 1;
    ret = LocalSPM::GetPlayerManager().SendPlaybackAction(PBA_NEXT, IN steps);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);


#if 0 // todo remove
    sleep(1);

    // test result
    tHMIPlaybackState playstatus;
    ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(HMI_PBS_STOPPED == playstatus);
#else
    WAIT_FOR_HMI_STATE(HMI_PBS_STOPPED, 30);
#endif

    /* Set all objects to Playable again */
    for(tUInt i=0; i<mediaObjectVector.size(); i++)
    {
        mediaObjectVector[i].notPlayable = FNP_PLAYABLE;
        ret = LocalSPM::GetDBManager().UpdateMediaObject(IN mediaObjectVector[i]);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    }

    /* Reset repeat mode */
    ret = LocalSPM::GetDBManager().SetRepeatMode(IN deviceID, IN oldRepeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Reset configuration */
    LocalSPM::GetDataProvider().DBFileBrowsingByDB = browsingByDB;
    LocalSPM::GetDataProvider().FileErrorHandling = (int)fileErrorHandling;
}

void PlayerManagerTest::LastModeAfterSourceSwitch()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMediaObject oldMediaObject;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(oldMediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    tPlaytime oldElapsedPlaytime = 0;
    tPlaytime totalPlaytime = 0;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT oldElapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Call SendSourceActivity */
    tSourceActivity sourceActivity = SA_OFF;
    ret = LocalSPM::GetPlayerManager().SendSourceActivity(IN sourceActivity);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Call SendDeAllocate */
    ret = LocalSPM::GetPlayerManager().SendDeAllocate();
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    tBoolean isActive = TRUE;
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(FALSE == isActive);

    /* Call SendAllocate */
    tAudioOutputDevice audioOutputDevice;
    strncpy_r(OUT audioOutputDevice, USB_ALSA_DEVICENAME, IN sizeof(audioOutputDevice));
    ret = LocalSPM::GetPlayerManager().SendAllocate(IN audioOutputDevice);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Call SendSourceActivity */
    sourceActivity = SA_ON;
    ret = LocalSPM::GetPlayerManager().SendSourceActivity(IN sourceActivity);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(2);

    // test result
    ret = LocalSPM::GetPlayerManager().GetActiveState(OUT isActive);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(FALSE != isActive);

    tMediaObject newMediaObject;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(newMediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(oldMediaObject.objectID == newMediaObject.objectID);

    tPlaytime newElapsedPlaytime = 0;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT newElapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(newMediaObject.objectID == objectID);
    CPPUNIT_ASSERT(oldElapsedPlaytime <= newElapsedPlaytime);
}

void PlayerManagerTest::LastPlaytime()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* 1) Setup precondition for test case */
    /* Enable first category type (CTY_VIDEO) of last mode position configuration */
    int lastModePosCategoryTypes = LocalSPM::GetDataProvider().LastModePosCategoryTypes();
#if defined(TARGET_BUILD)
    int newLastModePosCategoryTypes = LocalSPM::GetDataProvider().LastModePosCategoryTypes = 1;
#else
    int newLastModePosCategoryTypes = LocalSPM::GetDataProvider().LastModePosCategoryTypes = 4;
#endif
    if (newLastModePosCategoryTypes > lastModePosCategoryTypes)
    {
        LocalSPM::GetDataProvider().LastModePosCategoryTypes = newLastModePosCategoryTypes;
    }

    /* Enable first device type (DTY_USB) of last mode position configuration */
    int lastModePosDeviceTypes = LocalSPM::GetDataProvider().LastModePosDeviceTypes();
    if (1 > lastModePosDeviceTypes)
    {
        LocalSPM::GetDataProvider().LastModePosDeviceTypes = 1;
    }

    /* Get deviceID */
    tDeviceID deviceID;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

#if 0
    /* Set device to active */
    ret = LocalSPM::GetDBManager().SetActiveDevice(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
#endif

    /* Create a new indexed video episode list */
    tListID listID = LIST_ID_NONE;
    tListSize listSize = 0;
#if defined(TARGET_BUILD)
    tListType listType = LTY_EPISODE_OF_VIDEO;
    tIndex firstObject = 2;
    tIndex secondObject = 3;
#else
    tListType listType = LTY_SONG;
    tIndex firstObject = 0;
    tIndex secondObject = 1;
#endif
    tStreaming streaming = false;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;

    tag1.video = 0;
    tag2.tag = 0;
    tag3.tag = 0;
    tag4.tag = 0;

    ret = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
            OUT listID,
            OUT listSize,
            IN listType,
            IN streaming,
            IN tag1,
            IN tag2,
            IN tag3,
            IN tag4,
            IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != listID);
    CPPUNIT_ASSERT(0 < listSize);

    vector<tMediaObject> mediaObjectVector;
    ret = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, 0, listSize);
    CPPUNIT_ASSERT(ret == 0);
    CPPUNIT_ASSERT(mediaObjectVector.size() > 3);

    /* Check if 1st object is supported for storing last mode position */
    CPPUNIT_ASSERT(OBJECT_ID_NONE != mediaObjectVector[firstObject].objectID);
    CPPUNIT_ASSERT(LocalSPM::GetDataProvider().IsLastModePositionSupported(IN mediaObjectVector[firstObject].deviceType, IN mediaObjectVector[firstObject].catType));

    /* Remove last playtime for this object */
    ret = LocalSPM::GetDBManager().RemoveLastPlaytime(IN mediaObjectVector[firstObject].objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check if 2nd object is supported for storing last mode position */
    CPPUNIT_ASSERT(OBJECT_ID_NONE != mediaObjectVector[secondObject].objectID);
    CPPUNIT_ASSERT(LocalSPM::GetDataProvider().IsLastModePositionSupported(IN mediaObjectVector[secondObject].deviceType, IN mediaObjectVector[secondObject].catType));

    /* Remove last playtime for this object */
    ret = LocalSPM::GetDBManager().RemoveLastPlaytime(IN mediaObjectVector[secondObject].objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);


    /* 2) Start playback of 1st video/song */
    /* Send NewList for 1st object, offset is 50% of total playtime */
    tPlaytime offset = PLAYTIME_NONE;
    if((PLAYTIME_NONE == mediaObjectVector[firstObject].totalPlaytime)
       ||
       (PLAYTIME_END_OF_OBJECT == mediaObjectVector[firstObject].totalPlaytime))
    {
        offset = 10000; //10s
    }
    else
    {
        offset = mediaObjectVector[firstObject].totalPlaytime / 2;
    }
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN firstObject, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for requested object */
    tNowPlaying nowPlaying;
    tUInt wait = 0;
    tUInt sec = 0;
    for(sec=30; sec; sec--) {
        sleep(1);
        wait++;
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (mediaObjectVector[firstObject].objectID == nowPlaying.objectID) break;
    }
    CPPUNIT_ASSERT(sec > 0); //requested object not found in time

    tPlaytime elapsedPlaytime = 0;
    tPlaytime totalPlaytime = 0;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(nowPlaying.objectID == objectID);

    /* Store last playtime for this object */
    ret = LocalSPM::GetPlayerManager().StoreLastPlaytime();
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check if last playtime for this object is stored correctly */
    tPlaytime storedPlaytime;
    tPlaytime deltaTime;
    ret = LocalSPM::GetDBManager().GetLastPlaytime(OUT storedPlaytime, IN mediaObjectVector[firstObject].objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    deltaTime = (elapsedPlaytime > storedPlaytime) ? (elapsedPlaytime - storedPlaytime) : (storedPlaytime - elapsedPlaytime);
    CPPUNIT_ASSERT(deltaTime < (wait*1000 + 1001)); //less than +-1s


    /* 3) Start playback of 2nd video/song */
    /* Send NewList for 2nd object, offset is 30% of total playtime */
    if((PLAYTIME_NONE == mediaObjectVector[secondObject].totalPlaytime)
       ||
       (PLAYTIME_END_OF_OBJECT == mediaObjectVector[secondObject].totalPlaytime))
    {
        offset = 5000; //5s
    }
    else
    {
        offset = mediaObjectVector[secondObject].totalPlaytime / 3;
    }
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN secondObject, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for requested object */
    wait = 0;
    for(sec=30; sec; sec--) {
        sleep(1);
        wait++;
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (mediaObjectVector[secondObject].objectID == nowPlaying.objectID) break;
    }
    CPPUNIT_ASSERT(sec > 0); //requested object not found in time

    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(nowPlaying.objectID == objectID);

    /* Store last playtime for this object */
    ret = LocalSPM::GetPlayerManager().StoreLastPlaytime();
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check if last playtime for this object is stored correctly */
    ret = LocalSPM::GetDBManager().GetLastPlaytime(OUT storedPlaytime, IN mediaObjectVector[secondObject].objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    deltaTime = (elapsedPlaytime > storedPlaytime) ? (elapsedPlaytime - storedPlaytime) : (storedPlaytime - elapsedPlaytime);
    CPPUNIT_ASSERT(deltaTime < (wait*1000 + 1001)); //less than +-1s


    /* 4) Start playback of 1st video/song again */
    /* Send NewList for 3rd object, offset 0 = stored last playtime */
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN firstObject, IN 0);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for requested object */
    wait = 0;
    for(sec=30; sec; sec--) {
        sleep(1);
        wait++;
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (mediaObjectVector[firstObject].objectID == nowPlaying.objectID) break;
    }
    CPPUNIT_ASSERT(sec > 0); //requested object not found in time

    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(nowPlaying.objectID == objectID);

    ret = LocalSPM::GetDBManager().GetLastPlaytime(OUT storedPlaytime, IN mediaObjectVector[firstObject].objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check if playtime is restored to last playtime */
    deltaTime = (elapsedPlaytime > storedPlaytime) ? (elapsedPlaytime - storedPlaytime) : (storedPlaytime - elapsedPlaytime);
    CPPUNIT_ASSERT(deltaTime < (wait*1000 + 1001)); //less than +-1s


    /* 5) Start  playback of 2nd video/song via Next again */
    ret = LocalSPM::GetPlayerManager().SendNext();
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for requested object */
    wait = 0;
    for(sec=30; sec; sec--) {
        sleep(1);
        wait++;
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (mediaObjectVector[secondObject].objectID == nowPlaying.objectID) break;
    }
    CPPUNIT_ASSERT(sec > 0); //requested object not found in time

    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(nowPlaying.objectID == objectID);

    /* Check if playtime is reset to 0 (not to last playtime) */
    CPPUNIT_ASSERT(elapsedPlaytime < (wait*1000 + 1001)); //less than +-1s


    /* Reset configuration */
    LocalSPM::GetDataProvider().LastModePosCategoryTypes = lastModePosCategoryTypes;
    LocalSPM::GetDataProvider().LastModePosDeviceTypes = lastModePosDeviceTypes;
}

void PlayerManagerTest::RemoteActivity()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get device ID */
    tDeviceID deviceID;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_IPOD, IN "IPOD");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send REMOTE_ACTIVITY message to PlayerManagerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterREMOTE_ACTIVITY(OUT parameterString, IN size, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetPlayerManager().SendEventByName("REMOTE_ACTIVITY", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(10);

    // test result
    tStreamingMode streamingMode;
    ret = LocalSPM::GetPlayerManager().PlayerManager::GetStreamingMode(OUT streamingMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(SM_ON == streamingMode);
}

void PlayerManagerTest::NowPlayingStatus()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get current media object */
    tMediaObject mediaObject;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(OUT mediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send NOW_PLAYING_STATUS message to PlayerManagerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    mediaObject.bitRate = 128 << 10;   //128kBit
    mediaObject.sampleRate = 44100; //44,1kHz
    mediaObject.audioChannelFormat = ACF_STEREO;

    ret = LocalSPM::GetPlayerManager().ParameterNOW_PLAYING_STATUS(OUT parameterString,
            IN size,
            IN (tPEHandle)mediaObject.objectID,
            IN mediaObject.MetadataField1,
            IN mediaObject.MetadataField2,
            IN mediaObject.MetadataField3,
            IN mediaObject.MetadataField4,
            IN mediaObject.title,
            IN mediaObject.mediaType,
            IN mediaObject.bitRate,
            IN mediaObject.sampleRate,
            IN mediaObject.audioChannelFormat,
            IN mediaObject.metadataConvertFlag,
            IN NULL,
            IN NULL);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetPlayerManager().SendEventByName("NOW_PLAYING_STATUS", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    //CPPUNIT_ASSERT(mediaObject.objectID == nowPlaying.objectID); changed by FillupMetaDataIDs
    CPPUNIT_ASSERT(strcmp(mediaObject.title, nowPlaying.object.title) == 0);
    CPPUNIT_ASSERT(mediaObject.bitRate == nowPlaying.object.bitRate);
    CPPUNIT_ASSERT(mediaObject.sampleRate == nowPlaying.object.sampleRate);
    CPPUNIT_ASSERT(mediaObject.audioChannelFormat == nowPlaying.object.audioChannelFormat);
}

void PlayerManagerTest::PlaytimeStatus()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get current media object */
    tMediaObject mediaObject;
    tPlaytime elapsedPlaytime;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(OUT mediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    if(PLAYTIME_NONE != mediaObject.totalPlaytime)
    {
        elapsedPlaytime = mediaObject.totalPlaytime / 2;
    }
    else
    {
        elapsedPlaytime = 15000;
        mediaObject.totalPlaytime = 30000;
    }

    tPETimeInfoStruct timeInfoStruct;
    InitPETimeInfoStruct(timeInfoStruct);
    timeInfoStruct.position.ms = (tPEMilliseconds)elapsedPlaytime;
    timeInfoStruct.duration.ms = (tPEMilliseconds)mediaObject.totalPlaytime;

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfoString;
    size_t size = sizeof(timeInfoString);
    SMF::Marshal(OUT timeInfoString,
                 IN size-1,
                 IN DOUBLE_MARSHAL_SEPARATOR,
                 IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
                 IN timeInfoStruct.position.bytes,
                 IN timeInfoStruct.position.pct,
                 IN timeInfoStruct.position.ms,
                 IN timeInfoStruct.duration.bytes,
                 IN timeInfoStruct.duration.ms);

    /* Send PLAYTIME_STATUS message to PlayerManagerSM */
    tAllParameters parameterString;
    size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterPLAYTIME_STATUS(OUT parameterString,
            IN size,
            IN (tPEHandle)mediaObject.objectID,
            IN mediaObject.deviceID,
            IN timeInfoString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetPlayerManager().SendEventByName("PLAYTIME_STATUS", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    tPlaytime newElapsedPlaytime;
    tPlaytime newTotalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT newElapsedPlaytime, OUT newTotalPlaytime, OUT objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(elapsedPlaytime == newElapsedPlaytime);
    CPPUNIT_ASSERT(mediaObject.totalPlaytime == newTotalPlaytime);
    CPPUNIT_ASSERT(mediaObject.objectID == objectID);
}

void PlayerManagerTest::SendPlaybackMode()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get old playback mode */
    tPlaybackMode oldPlaybackMode;
    ret = LocalSPM::GetListControl().GetPlaybackMode(OUT oldPlaybackMode);
    CPPUNIT_ASSERT(ret == 0);
    CPPUNIT_ASSERT(PBM_NORMAL == oldPlaybackMode);

    /* Call SendPlaybackMode */
    tPlaybackMode playbackMode = PBM_RANDOM;
    ret = LocalSPM::GetPlayerManager().SendPlaybackMode(IN playbackMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    tPlaybackMode testPlaybackMode;
    ret = LocalSPM::GetListControl().GetPlaybackMode(OUT testPlaybackMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(testPlaybackMode == playbackMode);

    /* Reset playback mode to old value (PBM_NORMAL) */
    ret = LocalSPM::GetPlayerManager().SendPlaybackMode(IN oldPlaybackMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    ret = LocalSPM::GetListControl().GetPlaybackMode(OUT testPlaybackMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(testPlaybackMode == oldPlaybackMode);
}

void PlayerManagerTest::SendRepeatMode()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Get old repeat mode */
    tRepeatMode oldRepeatMode;
    ret = LocalSPM::GetListControl().GetRepeatMode(OUT oldRepeatMode);
    CPPUNIT_ASSERT(ret == 0);
    CPPUNIT_ASSERT(RPT_ALL == oldRepeatMode);

    /* Call SendRepeatMode */
    tRepeatMode repeatMode = RPT_ONE;
    ret = LocalSPM::GetPlayerManager().SendRepeatMode(IN repeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    tRepeatMode testRepeatMode;
    ret = LocalSPM::GetListControl().GetRepeatMode(OUT testRepeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(testRepeatMode == repeatMode);

    /* Reset repeat mode to old value (RPT_ALL) */
    ret = LocalSPM::GetPlayerManager().SendRepeatMode(IN oldRepeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    ret = LocalSPM::GetListControl().GetRepeatMode(OUT testRepeatMode);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(testRepeatMode == oldRepeatMode);
}

void PlayerManagerTest::AlbumArtAvailability()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    tMediaObject mediaObject;
    ret = LocalSPM::GetPlayerManager().GetMediaObject(OUT mediaObject);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(OBJECT_ID_NONE != mediaObject.objectID);

    tBoolean updateHMI = false;
    ret = LocalSPM::GetPlayerManager().SetAlbumArtAvailability(IN false, IN updateHMI);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test result
    tNowPlaying nowPlaying;
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(0 == strlen_r(nowPlaying.object.albumArtString));

    ret = LocalSPM::GetPlayerManager().SetAlbumArtAvailability(IN true, IN updateHMI);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test result
    ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(0 < strlen_r(nowPlaying.object.albumArtString));
}

void PlayerManagerTest::NewListTwice()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Create a new indexed list */
    tListID listID;
    tListSize listSize = 0;
    tStreaming streaming = false;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;
    tDeviceID deviceID;

    /* Get device ID */
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(ret == 0);

    /* Set the filter tags */
    tag1.tag = 0;
    tag2.artist = 0;
    tag3.composer = 0;
    tag4.album = 0;

    /* Call create list */
    ret = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
            OUT listID,
            OUT listSize,
            IN LTY_SONG,
            IN streaming,
            IN tag1,
            IN tag2,
            IN tag3,
            IN tag4,
            IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(LIST_ID_NONE != listID);
    CPPUNIT_ASSERT(0 < listSize);

    /* Get the list slice */
    vector<tMediaObject> mediaObjectVector;
    ret= LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(OUT mediaObjectVector, IN listID, 0, IN listSize);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(mediaObjectVector.size() == listSize);

    /* Call SendNewList */
    tIndex index = 0;
    tPlaytime offset = 0;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN index, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Force race condition by request 2 START event straight after another */
    index = listSize-1;
    ret = LocalSPM::GetPlayerManager().SendNewList(IN listID, IN index, IN offset);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Check if second object is playing */
    tNowPlaying nowPlaying;
    for (int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetNowPlaying(OUT nowPlaying);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if ((listID == nowPlaying.listID) && (mediaObjectVector[index].objectID == nowPlaying.objectID)) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(listID == nowPlaying.listID);
    CPPUNIT_ASSERT(mediaObjectVector[index].objectID == nowPlaying.objectID);
    CPPUNIT_ASSERT(0 == strcmp(nowPlaying.object.fileName, mediaObjectVector[index].fileName));

    tPlaytime elapsedPlaytime;
    tPlaytime totalPlaytime;
    tObjectID objectID = OBJECT_ID_NONE;
    for(int sec=30; sec; sec--) {
        ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT elapsedPlaytime, OUT totalPlaytime, OUT objectID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(nowPlaying.objectID == objectID);
        if (offset < elapsedPlaytime) break;
        sleep(1);
    }
    CPPUNIT_ASSERT(offset < elapsedPlaytime);
}

#ifndef useTestFW
#define useTestFW
#endif

void PlayerManagerTest::PlayAndIndexingTogetherForPlaytimeUpdates()
{
    ENTRY_TEST
    int ret,res;

    /* create a playable list */
    NewList_CreateMPIndexedList();

#ifdef useTestFW
    /* register playtime update callback */
    this->m_strTestcase = "PlayAndIndexingTogetherForPlaytimeUpdates";
    this->bLetFWValidateResult = false ;
    mNumberOfUpdatesCalled = 0;

    CListener *oListener = CListener::getCListenerInstance();
    oListener->registerCallback(this);
    oListener->startListen();
    oListener->registerEvent(EV_UPDATEPLAYTIME);
    PlayerManagerTest::funPtr =  &PlayerManagerTest::SourceActivityCallback;
#endif

    /* delete the second USB device */
    tDeviceID deviceID;
    res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB-2");
    CPPUNIT_ASSERT(res == 0);
    res = LocalSPM::GetDBManager().RemoveMediaObjects(IN deviceID);
    CPPUNIT_ASSERT(res == 0);
    res = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == res);

    /* start indexing of first USB device */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tNumberOfDevices numberOfDevices;
    res = LocalSPM::GetDBManager().GetConnectedDeviceCount(OUT numberOfDevices);
    CPPUNIT_ASSERT(MP_NO_ERROR == res);

    /* Send DB_DEVICE_INSERTED message to IndexerSM to create scan context */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN numberOfDevices+1, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_CONNECTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN numberOfDevices+1, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* add much more files to testdata */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID, 0);

    /* Commented Below logic of creating multiple directory -To resolve "No space issue" in tartget unittest run(RTC-578579).
     * Though commented,the purpose of this test is still met.
     * i.e This test is to verify- The playtime updates are updated by USBControlSM to user eventhough it is doing a device indexing
     * This scenario is met even without multiple directory content creation.
     *
     * ONLY LIMITATION:
     * Earlier the Playtime update verification may be done for 1 min of track playback (as huge content is there for indexing, it will take more time for indexing completion)
     * But,now the Playtime update verification done for 6 sec of track playback(as comparatively less content will be there for indexing.It is seen ~50 tracks available whose indexing completes in 6 secs)
     */
#if 0 //RTC-578579
    /* copy the directory UCASort in many other directories to extend the indexing time */
    int i;
    enum {noOfCopies = 40};
    tPath command;
    for(i=0; i<noOfCopies; i++) {
        snprintf(command, sizeof(command), "cp -rf %s/UCASort %s/UCSSort_%03d", deviceInfo.mountPoint, deviceInfo.mountPoint, i);
        system(command);
    }
    this->UcssortFolderCreated = TRUE;
#endif

    /* Check if listID is valid */
    CPPUNIT_ASSERT(LIST_ID_NONE != m_ListID);

    /* play the list */
    ret = LocalSPM::GetPlayerManager().SendNewList(IN m_ListID, IN 0, IN 0);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(3); //Play at least 3s

    WAIT_FOR_HMI_STATE(HMI_PBS_PLAYING, 30);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for end of indexing */
    tIndexingState indexingState;
    int retry = 60;
    int seconds = 0;
    int updatesCounted = 0;
    mNumberOfUpdatesCalled = 0;

    //When Indexing of device named "USB-2" ongoing,count no.of playtime updates received for device named "USB"'s playback
    //Counting done for 1 minute.If Indexing of "USB-2" is completed before 1 minute,then counting stopped.
    while(retry) {

        ETG_TRACE_USR2(("PlayerManagerTest::mid:%d retries left, %d seconds since PlayList() call, %d update calls", retry, seconds, updatesCounted));

        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        /* is the indexing complete? */
        if (IDS_COMPLETE == indexingState) break;

        /* let him do the indexing, now the playtime update must be counted */
        sleep(1);

        /* fetch the number of update calles after the sleep */
        updatesCounted = mNumberOfUpdatesCalled;

        retry--;
        seconds++;
    }
    ETG_TRACE_USR2(("PlayerManagerTest::end:%d retries left, %d seconds since PlayList() call, %d update calls", retry, seconds, updatesCounted));

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    /* stop the playing */
    PlaybackActionStop();

#ifdef useTestFW
    /* deactivate testframework listeners */
    oListener->stopListen();
    delete oListener;
#endif

    /* check the number of update calls */
    /* for every second two update calls should have been counted */
    /* if it is less, than the indexer has blocked out the update calls */
    if(LocalSPM::GetDataProvider().UseMediaEngine()) {
        CPPUNIT_ASSERT(updatesCounted >= ((seconds*3) - 3));
    } else {
        CPPUNIT_ASSERT(updatesCounted >= ((seconds*2) - 2));
    }
}

tResult PlayerManagerTest::CheckPlaybackState(const tHMIPlaybackState stateToReach, const int timeoutInSec)
{
    ENTRY_INTERNAL
    tResult ret;
    tHMIPlaybackState playstatus = HMI_PBS_UNDEFINED;

    /* check loop */
    for(int sec=timeoutInSec; sec; sec--)
    {
        ret = LocalSPM::GetPlayerManager().GetPlaybackState(OUT playstatus);
        if (MP_NO_ERROR != ret) return ret;

        if (playstatus == stateToReach)
        {
            ETG_TRACE_USR2(("state reached: playstatus=%d", playstatus));
            return MP_NO_ERROR;
        }
        sleep(1);
    }

    ETG_TRACE_USR2(("state not reached: stateToReach=%d, playstatus=%d", stateToReach, playstatus));

    return MP_ERR_SM_GENERAL_ERROR;
}
void PlayerManagerTest::CheckSetPlaytimeStatus()
{
    ENTRY_TEST
    // Test : ElapsedTime = 1500 TotalPlaytime = 3000
    tResult ret = MP_NO_ERROR;
    tPlaytime elapsedPlaytime = 1500;
    tPlaytime totalPlaytime = 3000;
    tBoolean validLastMode = false;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().SetPlaytime(IN elapsedPlaytime, IN totalPlaytime, IN validLastMode, IN objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test result
    tPlaytime newElapsedPlaytime;
    tPlaytime newTotalPlaytime;
    tObjectID newobjectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT newElapsedPlaytime, OUT newTotalPlaytime, OUT newobjectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(elapsedPlaytime == newElapsedPlaytime);
    CPPUNIT_ASSERT(totalPlaytime == newTotalPlaytime);
    CPPUNIT_ASSERT(objectID == newobjectID);
}
void PlayerManagerTest::CheckSetPlaytimeStatusforPlaytimeNone()
{
    ENTRY_TEST
    // Test : ElapsedTime = -1 TotalPlaytime = -1
    tResult ret = MP_NO_ERROR;
    tPlaytime elapsedPlaytime = PLAYTIME_NONE;
    tMediaObject mediaObject;
    mediaObject.totalPlaytime = PLAYTIME_NONE;
    tBoolean validLastMode = false;
    tObjectID objectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().SetPlaytime(IN elapsedPlaytime, IN mediaObject.totalPlaytime, IN validLastMode, IN objectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test result
    tPlaytime newElapsedPlaytime;
    tPlaytime newTotalPlaytime;
    tObjectID newobjectID = OBJECT_ID_NONE;
    ret = LocalSPM::GetPlayerManager().GetPlaytime(OUT newElapsedPlaytime, OUT newTotalPlaytime, OUT newobjectID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(newElapsedPlaytime != 0);
    CPPUNIT_ASSERT(newTotalPlaytime != 0);
    CPPUNIT_ASSERT(objectID == newobjectID);
}
