/**
 * @defgroup PlayerManager PlayerManager
 * @ingroup Mediaplayer
 * @author Thomas Porsch
 * 
 * Public interface for player manager to control playback
 * @{
 */

#ifndef _PLAYERMANAGER_H_
#define _PLAYERMANAGER_H_

#include "PlayerManagerSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "Lock.h"

class PlayerManager : public PlayerManagerSM, public ILocalSPM , public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the PlayerManager componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    PlayerManager(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~PlayerManager();

    /**
     * This function is used by LocalSPM to create the PlayerManager.
     * @attention: running in SPM thread context
     * Create the PlayerManager state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0) 
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the PlayerManager initialization.
     * @attention: running in SPM thread context
     * Init the PlayerManager state machine.
     * Register PlayerManagerSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0) 
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the PlayerManager.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the PlayerManager thread and the state machine.
     * Register on DB trigger DB_DEVICE_REMOVED at DBManager -> RegisterOnDeviceRemoved(smName,msgName)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the PlayerManager.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0) 
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the PlayerManager.
     * @attention: running in SPM thread context
     * Deregister PlayerManagerSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

#if 0
    /**
     * The error handler is called on a non zero return of a user callback method. If this handler returns
     * zero a internal retry loop calls the user callback method again. If this error handler returns non zero,
     * the retry loop will be finished.
     *
     * @param[in] error error number that was returned by user call back function
     * @param[in] trans pointer to the current transition
     * @param[in] fnName name of the user callback function that failed
     * @return != 0: stop the retry loop, = 0: retry the user callback for the specific transition
     */
    tResult ErrorHandler(const tResult error, const tTransition *trans, const char *fnName);
#endif

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

//Playback control part

    /**
     * Function stores ALSA device name of audio output/input and
      * If GetAudioDevice request is ongoing
      * - Cancel GetAudioDevice timer
      * - OutputWrapper -> SendGetAudioDeviceAnswer(deviceID)
     * RouteMessageAnswer(ALLOCATE_AUDIO_OUTPUT) to DeviceDispatcher (deviceType=0).
     *    -> deviceType "ALLOCATE_AUDIO_OUTPUT audioOutputDevice,audioInputDevice","PLAYER_MANAGER ALLOCATE_AUDIO_OUTPUT_ANSWER".
     * It is needed by PlayerEngine to assign output/input devices in the GStreamer
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult AllocateAudioOutput(const tAudioOutputDevice audioOutputDevice);

    /**
     * Function sends DeAllocateOutDevice to DeviceControl.
     * RouteMessageAnswer(DEALLOCATE_AUDIO_OUTPUT) to DeviceDispatcher (deviceType=0).
     *    -> deviceType "DEALLOCATE_AUDIO_OUTPUT","PLAYER_MANAGER DEALLOCATE_AUDIO_OUTPUT_ANSWER".
     * It is needed by PlayerEngine to disconnect output/input devices in the GStreamer
     *
     * @return != 0: error, = 0: OK
     */
    tResult DeAllocateAudioOutput();

    /**
     * Function sends related playback action event to state machine.
     * if( streaming mode == SM_ON ) //Roadmap 13008
     * - SendMessage(PLAYBACK_ACTION, playbackAction, nextPrevSkipCount, isHMIRequest=true)
     * else
     * - if( playbackAction == PLAY ) -> SendMessage(RESUME)
     *   else                         -> SendMessage(playbackAction)
     *   In case of NEXT and PREV the skip count is also sent.
     *
     * @param[in] playbackAction enum of playback action to set
     * @param[in] nextPrevSkipCount skip count for NEXT and PREV to skip more than one object with one action
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaybackAction(const tPlaybackAction playbackAction, const tNextPrevSkipCount nextPrevSkipCount=1,const tPlaybackSpeed playbackSpeed= PBK_SPEED_1X,const speedstate_e isPlaybackSpeed=ME_SPEEDSTATE_OFF); //Roadmap 13008: 100%

    /**
     * Function sends allocate event to state machine.
     * Call SendMessage(ALLOCATE, audioOutputDevice).
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SendAllocate(const tAudioOutputDevice audioOutputDevice);

    /**
     * Function sends deallocate event to state machine.
     * Call SendMessage(DEALLOCATE).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendDeAllocate();

    /**
     * Function sends source activity event to state machine.
     * Call SendMessage(SOURCE_ACTIVITY_ON or SOURCE_ACTIVITY_OFF).
     *
     * @param[in] sourceActivity ON or OFF
     * @return != 0: error, = 0: OK
     */
    tResult SendSourceActivity(const tSourceActivity sourceActivity);

    /**
     * Function sends START event to state machine.
     * Call SendMessage(START).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendStart();

    /**
     * Function sends STOP event to state machine.
     * Call SendMessage(STOP).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendStop();

    /**
     * Function sends EXIT event to state machine.
     * Call SendMessage(EXIT).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendExit();

    /**
     * Function sends NEXT event to state machine.
     * Call SendMessage(NEXT, steps=1, isHMIRequest=false).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendNext();

    /**
     * Function to reset the list and media object.
     */
    void ResetCurrentListAndMediaObject(void);

    /**
     * Function sends NEXT event to state machine with isHMIRequest=true.
     * Call SendMessage(NEXT, steps=1, isHMIRequest=true).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendHMINext();

    /**
     * Function sends PREV event to state machine.
     * Call SendMessage(PREV, steps=1, isHMIRequest=false).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendPrevious();

    /**
     * Function sends PREV event to state machine with isHMIRequest=true.
     * Call SendMessage(PREV, steps=1, isHMIRequest=true).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendHMIPrevious();

    /**
     * Function sends SEEK_TO event to state machine.
     * Calculate absolute playtime position in case of used percentage format and
     * Call SendMessage(SEEK_TO, position).
     *
     * @param[in] playpointPosition desired position in the object to move the playpoint
     * @param[in] playpointFormat format of PlaypointPosition parameter
     *            (percentage or absolute time value in milliseconds)
     * @return != 0: error, = 0: OK
     */
    tResult SendSeekTo(const tPlaytime playpointPosition, const tPlaypointFormat playpointFormat);

    /**
     * Function sends GET_AUDIO_DEVICE event to state machine.
     * Call SendMessage(GET_AUDIO_DEVICE, deviceID).
     *
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult SendGetAudioDevice(const tDeviceID deviceID);

    /**
     * Function sends GET_AUDIO_DEVICE_ANSWER event to state machine.
     * Call SendMessage(GET_AUDIO_DEVICE_ANSWER, success).
     *
     * @param[in] deviceID device ID of effected device
     * @param[in] success success of request
     * @return != 0: error, = 0: OK
     */
    tResult SendGetAudioDeviceAnswer(const tDeviceID deviceID, const tSuccess success);

    /**
     * Function sets playback mode in ListControl and related device controls.
     * Roadmap 13008
     * Call SendMessage(PLAYBACK_MODE, playbackMode).
     *
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaybackMode(const tPlaybackMode playbackMode); //Roadmap 13008: 100%

    /**
     * Function sets repeat mode in ListControl and related device controls.
     * Roadmap 13008
     * Call SendMessage(REPEAT_MODE, repeatMode).
     *
     * @param[in] repeatMode repeat the current playing track or list or nothing
     * @return != 0: error, = 0: OK
     */
    tResult SendRepeatMode(const tRepeatMode repeatMode); //Roadmap 13008: 100%, 13017: 100%

    /**
     * Function sends NEW_LIST event to state machine.
     * Deactivate all favorites in DB via DBManager -> DeactivateAllFavorites().
     * Call SendMessage(NEW_LIST, listID, listIndex, offset).
     *
     * @param[in] listID id of the list which should be used for playback
     * @param[in] listIndex index of object in list (-1 = last active object)
     * @param[in] offset playtime to start
     * @return != 0: error, = 0: OK
     */
    tResult SendNewList(const tListID listID, const tIndex listIndex, const tPlaytime offset);

    /**
     * Function sends NEXT_LIST event to state machine.
     * Roadmap 13017
     * Call SendMessage(NEXT_LIST, listID).
     *
     * @param[in] listID id of the list which should be used for playback
     * @return != 0: error, = 0: OK
     */
    tResult SendNextList(const tListID listID); //Roadmap 13017: 100%

    /**
     * Function sends NO_OBJECT and NEXT event to state machine.
     *
     * Do action if removed device (deviceID) is the current device.
     * Call SendMessage(NO_OBJECT, NOR_DEVICE_REMOVED).
     * Call SendMessage(NEXT, steps=1, isHMIRequest=false).
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult SendNoObjectAndNext(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Function sends NO_OBJECT event to state machine.
     * Call SendMessage(NO_OBJECT, noObjectReason).
     *
     * @param[in] noObjectReason explanation for no object reason
     * @return != 0: error, = 0: OK
     */
    tResult SendNoObject(const tNoObjectReason noObjectReason);

    /**
     * Function is called during a state tranistion to NoObject
     * Call StopMediaObject
     * Call ListIsEmpty with parameter noObjectReason
     *
     * @param[in] noObjectReason explanation for no object reason
     * @return != 0: error, = 0: OK
     */
    tResult NoObject(const tNoObjectReason noObjectReason);

    /**
     * Function starts playback of media objects of a new list.
     * Get current listID -> GetListID(currentListID).
     * if( listID != currentListID )
     * - Store last mode -> StoreLastPlayedList()
     * - Remove obsolete highlighting on old list in ListControl -> UpdateNowPlaying(currentListID, OBJECT_ID_NONE).
     * - Remove highlighting of parent lists in ListControl -> ResetParentHighlighting(currentListID).
     * - Store listID internally -> SetListID(listID).
     * Set active media object in ListControl -> SetMediaObject(listID, listIndex).
     * if( offset == 0)
     * - Restore last playtime of current media object -> RestoreLastPlaytime() //Roadmap 14017
     * else
     * - Set start playtime (m_StartTime=offset).
     * if( (listInfo.streaming == FALSE) && (currentStreamingMode == SM_ON) ) //Roadmap 13008
     * - Set playtime internally -> SetPlaytime(m_StartTime, totalPlaytime).
     * - SendMessage(STREAMING_MODE_OFF)
     * else if( (listInfo.streaming == TRUE) && (currentStreamingMode == SM_OFF) )
     * - SendMessage(STREAMING_MODE_ON)
     * else
     * - Reset playtime internally -> SetPlaytime(0, 0).
     * - Call SendMessage(START) to trigger state machine.
     * if( listInfo.streaming == TRUE )
     * - RouteMessage(STREAMING_MODE_ON) to specific DeviceControl (deviceType).
     *      -> deviceType "STREAMING_MODE_ON deviceType,deviceID,mountPoint".
     *
     * @param[in] listID id of the list which should be used for playback
     * @param[in] listIndex index of object in list (-1 = last active object)
     * @param[in] offset playtime to start
     * @return != 0: error, = 0: OK
     */
    tResult NewList(const tListID listID, const tIndex listIndex, const tPlaytime offset); //Roadmap 13008: 100%, 14017: 100%

    /**
     * Function skips to next media object in ListControl and sends START event to state machine.
     * Get current listID -> GetListID(currentListID).
     * Get list info from ListControl -> GetListInfo(listInfo).
     * if( listInfo.listType == LTY_SONG )
     * - Deactivate all favorites in DB via DBManager -> DeactivateAllFavorites().
     * Store last playtime of old media object -> StoreLastPlaytime() //Roadmap 14017
     * Skip to next media object in ListControl -> Next(currentListID, steps).
     * if( ret == MP_NO_ERROR)
     * - Check if restoring of last mode position after NEXT is configured by DataProvider -> LastModePosAfterNext()
     * - if( restoring == true ) -> RestoreLastPlaytime() //Roadmap 14017
     * - else                    -> Reset start playtime (m_StartTime=0).
     * - Call SendMessage(START).
     * if( ret == LIST_IS_NOTPLAYABLE) //Roadmap 13012
     * - SendNoObject(NOR_LIST_IS_NOT_PLAYABLE)
     * else if ( ret != MP_NO_ERROR )
     * - SendNoObject(NOR_NO_OBJECT)
     *
     * @param[in] steps skip count to skip more than one media object with one action
     * @param[in] isHMIRequest flag for HMI request
     * @return != 0: error, = 0: OK
     */
    tResult NextMediaObject(const tNextPrevSkipCount steps, const tHMIRequest isHMIRequest); //Roadmap 13012: 100%, 14017: 100%

    /**
     * Function skips to previous media object in ListControl and sends START event to state machine.
     * Get current listID -> GetListID(currentListID).
     * if( listInfo.listType == LTY_SONG )
     * - Deactivate all favorites in DB via DBManager -> DeactivateAllFavorites().
     * Store last playtime of old media object -> StoreLastPlaytime() //Roadmap 14017
     * Decrement steps if media object is not at the beginning -> reset media object to start position
     * Skip to previous media object in ListControl -> Prev(currentListID, steps).
     * if( ret == MP_NO_ERROR)
     * - Check if restoring of last mode position after PREV is configured by DataProvider -> LastModePosAfterPrev()
     * - if( m_StartTime != PLAYTIME_END_OF_OBJECT ) -> Reset start playtime (m_StartTime=0).
     * - else if( restoring == true ) -> RestoreLastPlaytime() //Roadmap 14017
     * - Call SendMessage(START).
     * if( ret == LIST_IS_NOTPLAYABLE) //Roadmap 13012
     * - SendNoObject(NOR_LIST_IS_NOT_PLAYABLE)
     * else if ( ret != MP_NO_ERROR )
     * - SendNoObject(NOR_NO_OBJECT)
     *
     * @param[in] steps skip count to skip more than one media object with one action
     * @param[in] isHMIRequest flag for HMI request
     * @return != 0: error, = 0: OK
     */
    tResult PrevMediaObject(const tNextPrevSkipCount steps, const tHMIRequest isHMIRequest); //Roadmap 13012: 100%, 14017: 100%

    /**
     * Function sets start playtime and send START event to state machine.
     * This is needed to make fast reverse across media object borders.
     *
     * Set start playtime to 0.
     * Call SendMessage(START).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendStartAndResetTime();

    /**
     * Function sets start playtime and send PREV event to state machine.
     * This is needed to make fast reverse across media object borders.
     * Former name: FRevContinueMediaObject
     *
     * Set start playtime to end of object.
     * Call SendMessage(PREV, 1).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendPreviousAndSetToEnd();

    /**
     * Function starts playback of a media object.
     * Get current media object data from ListControl -> GetCurrentMediaObject(&mediaObject, &mediaObjectStatus, m_ListID)
     * In case of no object received -> SendMessage(NO_OBJECT, noObjectReason=NOR_NO_OBJECT).
     * Otherwise store mediaObject internally,
     * check if video layer has to be enable/disable
     *    If current media type = (video or podcast) and new media type != (video or podcast) disable video by VideoControl -> DisableVideo
     *    If current media type != (video or podcast) and new media type = (video or podcast) enable video by VideoControl -> EnableVideo
     * and RouteMessageAnswer(PLAY) to DeviceDispatcher (deviceType=0).
     *    -> deviceType "PLAY m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint,
     *       m_MediaObject.objectID,m_MediaObject.position","PLAYER_MANAGER PLAY_ANSWER". //Roadmap 13010
     *
     * @return != 0: error, = 0: OK
     */
    tResult PlayMediaObject(); //Roadmap 13010: 100%

    /**
     * Function starts playback of a media stream.
     * Roadmap 13008
     * Get deviceID from DBManager::GetActiveDevice(&deviceID).
     * Get device info from DBManager::GetDeviceInfo(&deviceInfo, deviceID).
     * RouteMessageAnswer(PLAY) to specific DeviceControl (deviceType)
     *   -> deviceType "PLAY deviceInfo.deviceType,deviceInfo.deviceID,deviceInfo.mountPoint,position","PLAYER_MANAGER PLAY_ANSWER".
     *
     * @param[in] position absolute playtime position to move playpoint
     * @return != 0: error, = 0: OK
     */
    tResult PlayMediaObjectStreaming(); //Roadmap 13008: 50%

    /**
     * Function stops playback of a media object.
     * Reset playtime internally -> SetPlaytime(0, 0).
     * Disable video by VideoControl -> DisableVideo.
     * RouteMessageAnswer(STOP) to specific DeviceControl (deviceType).
     *    -> deviceType "STOP m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint","PLAYER_MANAGER STOP_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopMediaObject();

    /**
     * Function pauses playback of a media object.
     * RouteMessageAnswer(PAUSE) to specific DeviceControl (deviceType).
     *    -> deviceType "PAUSE m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint","PLAYER_MANAGER PAUSE_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult PauseMediaObject();

    /**
     * Function resumes playback of a media object.
     * RouteMessageAnswer(RESUME) to specific DeviceControl (deviceType).
     *    -> deviceType "RESUME m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint","PLAYER_MANAGER RESUME_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult ResumeMediaObject();

    /**
     * Function starts fast forward action of a media object.
     * Request reduced volume at AudioManagement via OutputWrapper -> RequestVolumeChange(REDUCED). //Roadmap 13018
     * RouteMessageAnswer(FFWD_START) to specific DeviceControl (deviceType).
     *    -> deviceType "FFWD_START m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint,m_MediaObject.position,rate","PLAYER_MANAGER FFWD_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult convertPlaybackSpeed(tPlaybackSpeed curPlaybackSpeed);
    tResult FastForwardStart();

    /**
     * Function stops fast forward action of a media object.
     * Request normal volume at AudioManagement via OutputWrapper -> RequestVolumeChange(NORMAL). //Roadmap 13018
     * RouteMessageAnswer(FFWD_STOP) to specific DeviceControl (deviceType).
     *    -> deviceType "FFWD_STOP m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint","PLAYER_MANAGER FFWD_STOP_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult FastForwardStop();

    /**
     * Function starts fast reverse action of a media object.
     * Request reduced volume at AudioManagement via OutputWrapper -> RequestVolumeChange(REDUCED). //Roadmap 13018
     * RouteMessageAnswer(FREV_START) to specific DeviceControl (deviceType).
     *    -> deviceType "FREV_START m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint,m_MediaObject.position,rate","PLAYER_MANAGER FREV_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult FastReverseStart();

    /**
     * Function stops fast reverse action of a media object.
     * Request normal volume at AudioManagement via OutputWrapper -> RequestVolumeChange(NORMAL). //Roadmap 13018
     * RouteMessageAnswer(FREV_STOP) to specific DeviceControl (deviceType).
     *    -> deviceType "FREV_STOP m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint","PLAYER_MANAGER FREV_STOP_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult FastReverseStop();

    /**
     * Function moves playpoint to a specific position of a media object.
     * RouteMessageAnswer(SEEK_TO) to specific DeviceControl (deviceType).
     *    -> deviceType "SEEK_TO m_MediaObject.deviceType,m_MediaObject.deviceID,m_MediaObject.URL,m_MediaObject.mountPoint,position","PLAYER_MANAGER SEEK_TO_ANSWER".
     *
     * @param[in] position absolute playtime position to move playpoint
     * @return != 0: error, = 0: OK
     */
    tResult ActiveSeekTo(const tPlaytime position);

    /**
     * Function sets playpoint to a specific position for a not played media object.
     * Store last played media object data in ListControl -> StoreLastPlayedList(m_ListID, position).
     * Update Playtime property at OutputWrapper
     *    -> UpdatePlaytime(position, m_MediaObject.TotalPlaytime)
     * Set start playtime.
     *
     * @param[in] position absolute playtime position to move playpoint
     * @return != 0: error, = 0: OK
     */
    tResult InactiveSeekTo(const tPlaytime position);

    /**
     * Function sends playback action to streaming device.
     * Roadmap 13008
     * Get deviceID from DBManager::GetActiveDevice(&deviceID).
     * Get device info from DBManager::GetDeviceInfo(&deviceInfo, deviceID).
     * RouteMessageAnswer(playbackAction e.g. PAUSE) to specific DeviceControl (deviceType)
     *   -> deviceType "playbackAction e.g. PAUSE deviceInfo.deviceType,deviceInfo.deviceID,deviceInfo.mountPoint,position","PLAYER_MANAGER PLAYBACK_ACTION_ANSWER".
     *
     * @param[in] playbackAction playback action to send (e.g. PAUSE)
     * @param[in] nextPrevSkipCount skip count for NEXT and PREV to skip more than one object with one action
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackActionStreaming(const tPlaybackAction playbackAction, const tNextPrevSkipCount nextPrevSkipCount = 1); //Roadmap 13008 100%

    /**
     * Function sets playback mode in ListControl and related device controls.
     * Roadmap 13008
     * if non streaming mode and not configured SetDevicePlaybackModeNonStreaming
     * - Set playback mode in ListControl -> SetPlaybackMode(playbackMode).
     * else
     * - Get deviceID from DBManager::GetActiveDevice(&deviceID).
     * - Get device info from DBManager::GetDeviceInfo(&deviceInfo, deviceID).
     *   if deviceType is able to buffer something //Roadmap 13017
     *   - RouteMessage(DISCARD_BUFFER) to specific DeviceControl (deviceType).
     *     -> deviceType "DISCARD_BUFFER deviceInfo.deviceType".
     *   if( deviceInfo.deviceType != MYMEDIA )
     *   - RouteMessage(SET_PLAYBACK_MODE) to specific DeviceControl (deviceType)
     *     -> deviceType "SET_PLAYBACK_MODE deviceInfo.deviceType,deviceInfo.deviceID,deviceInfo.mountPoint,playbackMode".
     *
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackMode(const tPlaybackMode playbackMode); //Roadmap 13008: 100%, 13017: 90%

    /**
     * Function sets repeat mode in ListControl and related device controls.
     * Roadmap 13008
     * if non streaming mode and not configured SetDeviceRepeatModeNonStreaming
     * - Set playback mode in ListControl -> SetRepeatMode(playbackMode).
     * else
     * - Get deviceID from DBManager::GetActiveDevice(&deviceID).
     * - Get device info from DBManager::GetDeviceInfo(&deviceInfo, deviceID).
     *   if( deviceInfo.deviceType != MYMEDIA )
     *   - RouteMessage(SET_REPEAT_MODE) to specific DeviceControl (deviceInfo.deviceType)
     *     -> deviceType "SET_REPEAT_MODE deviceInfo.deviceType,deviceInfo.deviceID,deviceInfo.mountPoint,repeatMode".
     *
     * @param[in] repeatMode repeat the current playing track or list or nothing
     * @return != 0: error, = 0: OK
     */
    tResult RepeatMode(const tRepeatMode repeatMode); //Roadmap 13008: 100%

    /**
     * Function gets next buffer media object from ListControl and sends it to PlayerEngine to buffer it.
     * Roadmap 13010
     * Get next buffer media object from ListControl -> GetMediaObject(&bufferMediaObject, listID, m_BufferIndex).
     * if( bufferMediaObject is valid && bufferMediaObject.deviceType == m_MediaObject.deviceType)
     * - RouteMessageAnswer(BUFFER) to specific DeviceControl (deviceType).
     *     -> deviceType "BUFFER bufferMediaObject.deviceType,bufferMediaObject.deviceID,bufferMediaObject.URL,
     *        bufferMediaObject.mountPoint,bufferMediaObject.objectID,bufferMediaObject.position","PLAYER_MANAGER BUFFER_ANSWER".
     * else
     * - SendAnswer(not successful)
     *
     * @return != 0: error, = 0: OK
     */
    tResult NextBufferMediaObject(); //Roadmap 13010: 100%

    /**
     * Function stores new listID and discard the buffer in the PlayerEngine.
     * Roadmap 13017
     * - RouteMessage(DISCARD_BUFFER) to specific DeviceControl (deviceType).
     *     -> deviceType "DISCARD_BUFFER deviceType".
     * - Set listID internally SetListID(listID)
     * The new listID will be used when a new song is requested.
     *
     * @param[in] listID id of the list which should be used for playback
     * @return != 0: error, = 0: OK
     */
    tResult NextList(const tListID listID,const tBoolean updateNowPlaying); //Roadmap 13017: 90%

    /**
     * Function delivers answer to condition.
     * In case of start (e.g. after next) in mode "paused" this function is called.
     * If automatic play is configured return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsAutoPlay();

    /**
     * Function delivers answer to guard query.
     * In case of start (e.g. after next) in mode "pausedStop" this function is called.
     * If automatic play is configured return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsAutoPlayInStop();

    /**
     * Function delivers answer to guard query.
     * In case of restart in last playback mode "paused" this function is called.
     * If automatic play is configured return true, otherwise false. 
     *
     * @return true or false
     */
    tResult IsAutoPlayInLastMode();

    /**
     * Function delivers answer to condition.
     * If playtime <= configured period of time return true, otherwise false. 
     *
     * @return true or false
     */
    tResult IsAtBeginningOfPlayback();

    /**
     * Function delivers answer to condition.
     * If continue of cueing action is expected at the end of a media object while
     * ffwd is ongoing return true, otherwise false.
     * It can be configured to play next object when reaching end of media object
     * or when reaching end of list. In case of configuration "end of list"
     * the index of the new media object has to be checked.
     *
     * @return true or false
     */
    tResult IsFFwdContinuousCueing();

    /**
     * Function delivers answer to condition.
     * If continue of cueing action is expected at the begin of a media object while
     * frev is ongoing return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsFRevContinuousCueing();

    /**
     * Function delivers answer to condition.
     * If SourceActivityOn is delayed because wrong source timer is running return true, otherwise false.
     *
     * @return true or false
     */
    tResult IsSourceActivityOnDelay();

    /**
     * Function delivers answer to condition.
     * If allocation was successful return true, otherwise false. 
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return true or false
     */
    tResult IsAllocateSuccessful(const tReturnValue returnValue);

    /**
     * Function delivers answer to condition.
     * If deallocation was successful return true, otherwise false. 
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return true or false
     */
    tResult IsDeAllocateSuccessful(const tReturnValue returnValue);

    /**
     * Function makes a FATAL_M_ASSERT because it should not happen
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult AllocationFailed(const tReturnValue returnValue);

    /**
     * Function makes a FATAL_M_ASSERT because it should not happen
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult DeAllocationFailed(const tReturnValue returnValue);

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Send releasing event to own waiting SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to waiting IIL state machine when SA_OFF is done.
     * Answer to IIL SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleSAOffFinished();

    /**
     * Function sends answer to waiting IIL state machine when SA_ON is done.
     * Answer to IIL SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleSAOnFinished();

    /**
     * Function sends answer to waiting IIL state machine when ALLOCATE is done.
     * Answer to IIL SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAllocateFinished();

    /**
     * Function sends answer to waiting IIL state machine when DEALLOCATE is done.
     * Answer to IIL SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleDeAllocateFinished();

    /**
     * Function handles GetAudioDevice request in state allocated
     * If deviceType is able to support CarPlayMode
     * - If device is another one than the current real device
     *   - Send SET_REMOTE_ACTIVITY message to specific DeviceControlSM
     * - Send SPI MethodAnswer for GetAudioDevice
     * Else
     * - Send SPI MethodError for GetAudioDevice
     *
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAudioDeviceNotAllocated(const tDeviceID deviceID);

    /**
     * Function handles GetAudioDevice request in state deallocated
     * If deviceType is able to support CarPlayMode
     * - Start GetAudioDevice timer
     * - Send SET_REMOTE_ACTIVITY message to specific DeviceControlSM
     * Else
     * - Send SPI MethodError for GetAudioDevice
     *
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAudioDeviceAllocated(const tDeviceID deviceID);

    /**
     * GetAudioDeviceTimer callback
     *
     * @param[in] timerID timer ID
     * @param[in] instance pointer to own (PlayerManager) instance
     * @param[in] userData pointer to possible user data
     * @return true or false
     */
     static bool GetAudioDeviceTimerCallback(timer_t timerID ,void* instance, const void *userData);

     /**
      * Function handles GetAudioDevice answer
      * If GetAudioDevice request is ongoing
      * - Cancel GetAudioDevice timer
      * - If success
      *   - OutputWrapper -> SendGetAudioDeviceAnswer(deviceID)
      *   Else
      *   - OutputWrapper -> SendGetAudioDeviceError(deviceID)
      *
      * @param[in] deviceID device ID of effected device
      * @param[in] success success of request
      * @return != 0: error, = 0: OK
      */
     tResult GetAudioDeviceAnswer(const tDeviceID deviceID, const tSuccess success);

     /**
      * AppleBugWrongSourceTimer callback
      *
      * @param[in] timerID timer ID
      * @param[in] instance pointer to own (PlayerManager) instance
      * @param[in] userData pointer to possible user data
      * @return true or false
      */
      static bool AppleBugWrongSourceTimerCallback(timer_t timerID ,void* instance, const void *userData);

     /**
     * Function updates property status of NowPlaying to HMI.
     * Get current media object data from ListControl -> GetCurrentMediaObject(&mediaObject, &mediaObjectStatus, m_ListID).
     * In case of no object received -> SendMessage(NO_OBJECT, noObjectReason=NOR_NO_OBJECT).
     * Otherwise update NowPlaying property at
     * - OutputWrapper -> UpdateNowPlaying() and
     * - ListControl -> UpdateNowPlaying(m_ListID, m_MediaObjectID)
     *
     * @return != 0: error, = 0: OK
     */
    tResult UpdateNowPlayingFromDB();

    /**
     * Function informs CustomControl that list is empty.
     * Call ListIsEmpty at CustomControl -> ListIsEmpty(m_ListID, noObjectReason).
     *
     * @param[in] noObjectReason Explanation for no object reason
     * @return != 0: error, = 0: OK
     */
    tResult ListIsEmpty(const tNoObjectReason noObjectReason);

    /**
     * Function is called by leaving active state.
     * Deactivate all favorites in DB via DBManager -> DeactivateAllFavorites().
     * Store last last mode for a list.
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreLastPlayed();

    /**
     * Function stores last mode for a list.
     * Get last mode playtime -> GetLastModePlaytime(&playtime)
     * if( elapsedPlaytime != PLAYTIME_NONE ) playtime is not reset
     * - Store last played media object data inclusive playtime in ListControl -> StoreLastPlayedList(m_ListID, playtime).
     * - Store last playtime of old media object -> StoreLastPlaytime() //Roadmap 14017
     * - Update totalPlaytime of current mediaObject if totalPlaytime > 0 at DBManager -> UpdateMediaObject(m_MediaObject);
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreLastPlayedList(); //Roadmap 14017: 100%

    /**
     * Function stores last mode position for an media object.
     * Roadmap 14017
     * Check if last mode position of current media object is supported by DataProvider -> IsLastModePositionSupported(&answer, m_MediaObject.deviceType, m_MediaObject.catType)
     * if(answer == true)
     * - Get last stored playtime for this media object from DBManager -> GetLastPlaytime(&storedPlaytime, m_MediaObject.objectID)
     * - if( |m_MediaObject.elapsedPlaytime - storedPlaytime| > LastModeDeltaTime (5s) ) media object played for a while
     *   - if( |m_MediaObject.totalPlaytime - m_MediaObject.elapsedPlaytime| < LastModeDeltaTimeZero (5s) ) end of media object is reached
     *     - Store 0 as last playtime for that media object in DBManager -> StoreLastPlaytime(m_MediaObject.objectID, 0).
     *   - else if( m_MediaObject.elapsedPlaytime < LastModeDeltaTimeZero (5s) ) beginning of the media object is reached
     *     - Store 0 as last playtime for that media object in DBManager -> StoreLastPlaytime(m_MediaObject.objectID, 0).
     *   - else
     *     - Store last playtime for that media object in DBManager -> StoreLastPlaytime(m_MediaObject.objectID, m_MediaObject.elapsedPlaytime).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreLastPlaytime(); //Roadmap 14017: 100%

    /**
     * Function restores last mode position for an media object.
     * Roadmap 14017
     * Check if last mode position of current media object is supported by DataProvider -> IsLastModePositionSupported(&answer, m_MediaObject.deviceType, m_MediaObject.catType)
     * if(answer == true)
     * - Get last stored playtime for this media object from DBManager -> GetLastPlaytime(&m_StartTime, m_MediaObject.objectID)
     * else
     * - Reset start playtime (m_StartTime=0).
     *
     * @return != 0: error, = 0: OK
     */
    tResult RestoreLastPlaytime(); //Roadmap 14017: 100%

    /**
     * Function sets allocate state to ALLOCATED.
     * Roadmap 13024
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreAllocated(); //Roadmap 13024: 100%

    /**
     * Function sets allocate state to ACTIVE.
     * Roadmap 13024
     *
     * Call SendMessage(START)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreActive(); //Roadmap 13024: 100%

    /**
     * Function sets allocate state to DEALLOCATED.
     * Roadmap 13024
     *
     * @return != 0: error, = 0: OK
     */
    tResult StoreDeAllocated(); //Roadmap 13024: 100%

    /**
     * Function resets the internal NowPlaying member variable
     *
     * @return != 0: error, = 0: OK
     */
    tResult ResetNowPlaying();

    /**
     * A special playing state (e.g. NO_DEVICE, LIST_COMPLETE) is set from another component.
     * PlayerManager sets the state to a new state as soon as a new track is started.
     * Store the playing state in a member variable
     * Call UpdateNowPlaying at OutputWrapper -> UpdateNowPlaying().
     * Call UpdateNowPlaying at ListControl -> UpdateNowPlaying(m_ListID, m_MediaObjectID)
     *
     * @param[in] nowPlayingState new now playing state
     * @return != 0: error, = 0: OK
     */
    tResult SetNowPlayingState(const tNowPlayingState nowPlayingState);

    /**
     * Function returns the current NowPlaying information
     *
     * @param[out] nowPlaying list ID and media object
     * @return != 0: error, = 0: OK
     */
    tResult GetNowPlaying(tNowPlaying &nowPlaying);

    /**
     * Function updates the tNowPlayingState variable:state in the given nowPlaying instance
     *
     * @param[inout] state in nowPlaying
     * @return != 0: error, = 0: OK
     */
    tResult CalculateNowPlayingState(tNowPlaying &nowPlaying);

    /**
     * Function stores the album art availability for a special media object
     * Roadmap 13006
     *
     * @param[in] available the album art of media object is available or not yet
     * @param[in] updateHMI update the NowPlaying property to the HMI
     * @return != 0: error, = 0: OK
     */
    tResult SetAlbumArtAvailability(const tAvailable available, const tBoolean updateHMI); //Roadmap 13006: 100%

    /**
     * Setter for elapsed and total playtime of current media object.
     *
     * @param[in] elapsedPlaytime absolute elapsed playtime
     * @param[in] totalPlaytime absolute total playtime
     * @param[in] validLastMode last mode is valid or not
     * @param[in] objectID ID of the media object to which the param1 and param2 correspond to //Note:objectID is reliable only for Mass Storage devices & MTP Devices
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaytime(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime, const tBoolean validLastMode,const tObjectID objectID);

    /**
     * Getter for elapsed and total playtime of current media object.
     *
     * @param[out] elapsedPlaytime absolute elapsed playtime
     * @param[out] totalPlaytime absolute total playtime
     * @return != 0: error, = 0: OK
     */
    tResult GetPlaytime(tPlaytime &elapsedPlaytime, tPlaytime &totalPlaytime, tObjectID &objectID);

    /**
     * Getter for elapsed of last mode media object.
     *
     * @param[out] elapsedPlaytime absolute elapsed playtime
     * @return != 0: error, = 0: OK
     */
    tResult GetLastModePlaytime(tPlaytime &elapsedPlaytime);

    /**
     * Function updates playback state derived from internal SM state
     *
     * @param[in] status playback state coming from DeviceControl (PlayerEngine)
     * @return != 0: error, = 0: OK
     */
    tResult UpdatePlaybackState(const tPEPlaybackState status);
    tResult UpdatePlaybackState(void);

    /**
     * Function returns current playback state
     * There is a mapping between internal states and external states
     *
     * @param[out] playbackState external (HMI) playback state
     * @return != 0: error, = 0: OK
     */
    tResult GetPlaybackState(tHMIPlaybackState &playbackState);

    /**
     * Function makes mapping of of state machine state to HMI playback state
     *
     * @param[out] playbackState calculated playback state
     * @return != 0: error, = 0: OK
     */
    tResult CalcPlaybackState(tHMIPlaybackState &playbackState);

    /**
     * Function handles the buffer response in waiting SM states.
     * Roadmap 13010
     * In case of success increment current buffer index (m_BufferIndex).
     *
     * @param[in] success success of request
     * @return != 0: error, = 0: OK
     */
    tResult BufferStatus(const tSuccess success); //Roadmap 13010: 100%

    /**
     * Function handles the playback status response in waiting SM states.
     * Additionally it calls ForwardPlaybackStatus for checking the parameters.
     *
     * @param[in] status playback state coming from DeviceControl (PlayerEngine)
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @param[in] metadataTitle coming from PlayerEngine with meta data content
     * @param[in] mediaType current media type sent by PlayerEngine
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusPlay(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tTitle metadataTitle,
        const tMediaType mediaType,
        const tConvertFlag metadataConvertFlag=0,const tUUID uuid=NULL);

    /**
     * Function handles the playback status response in waiting SM states.
     * Additionally it calls ForwardPlaybackStatus for checking the parameters.
     *
     * @param[in] status playback state coming from DeviceControl (PlayerEngine)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status);

    /**
     * Function handles the playback status response in waiting SM states.
     * Roadmap 13010
     * Additionally it calls ForwardPlaybackStatusNew for checking the parameters.
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sets internal player manager member values according to the
     * metadata (for example playback state or song name).
     * Check if indexing is supported for given device type by DataProvider -> GetIndexingPriority(&priority, m_MediaObject.deviceType)
     * if(priority != PRIORITY_NONE)
     * - Fill all ID's from DB or Livetags table at DBManager -> FillupMetaDataIDs(m_MediaObject)
     * Update NowPlaying property at OutputWrapper -> UpdateNowPlaying().
     * Update NowPlaying at ListControl -> UpdateNowPlaying(m_ListID, m_MediaObjectID).
     * Update PlaybackState property at OutputWrapper -> UpdatePlaybackState().
     * For special playback states this functions sends a message
     * to the own state machine -> SendMessage(FINISHED 1)/(STREAM_ERROR metadata2)
     *
     * @param[in] status playback state coming from DeviceControl (PlayerEngine)
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @param[in] metadataTitle coming from PlayerEngine with meta data content
     * @param[in] mediaType current media type sent by PlayerEngine
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tTitle metadataTitle,
        const tMediaType mediaType,
        const tConvertFlag metadataConvertFlag=0,
        const tUUID uuid=NULL,
        const tObjectID objectID = OBJECT_ID_NONE);

#if 0
    /**
     * Function handles the playtime status response in waiting SM states.
     * Additionally it calls ForwardPlaytimeStatus for checking the parameters.
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);
#endif

#if 0
    /**
     * Function handles the tick time elapsed signal.
     * Set internal player manager member values according to the values.
     * Update Playtime property at OutputWrapper -> UpdatePlaytime().
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);
#endif

    /**
     * Function handles the playtime status response in waiting SM states.
     * Additionally it calls ForwardPlaytimeStatusNew for checking the parameters.
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] deviceID device ID of updating device
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatusNew(const tPEHandle handle, const tDeviceID deviceID, const tPETimeInfo timeInfo);

    /**
     * Function handles the PLAYTIME_STATUS signal.
     * Roadmap 13010
     * Set internal player manager member values according to the values.
     * Update Playtime property at OutputWrapper -> UpdatePlaytime().
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] deviceID device ID of updating device
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatusNew(const tPEHandle handle, const tDeviceID deviceID, const tPETimeInfo timeInfo); //Roadmap 13010: 100%

    /**
     * Function handles the PLAYBACK_STATUS signal.
     * Roadmap 13010
     * Update PlaybackState property at OutputWrapper -> UpdatePlaybackState().
     * For special playback states this functions sends a message to the own state machine
     * if( status==PE_PBS_FINISHEDSTATE ) -> SendMessage(FINISHED, 1)
     * if( status==PE_PBS_ERRORSTATE ) -> SendMessage(STREAM_ERROR reason)
     * Check if gapless play is configured by DataProvider -> GaplessPlay()
     * if( status==PE_PBS_BUFFERINGSTATE && gapless== true ) -> SendMessage(NEXT_BUFFER)
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] status playback state coming from DeviceControl (PlayerEngine)
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState status, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function handles the NOW_PLAYING_STATUS signal.
     * Roadmap 13010
     * Set internal player manager member values according to the metadata (for example genre, artist, album and song name).
     * if( handle != m_MediaObjectID )
     * - Set current media object data at ListControl -> SetCurrentMediaObject(m_ListID, handle).
     * - Get current media object data from ListControl -> GetCurrentMediaObject(&m_MediaObject, &mediaObjectStatus, m_ListID)
     * Update meta data in m_MediaObject
     * Check if indexing is supported for given device type by DataProvider -> GetIndexingPriority(&priority, m_MediaObject.deviceType)
     * if(priority != PRIORITY_NONE)
     * - Fill all ID's from DB or Livetags table at DBManager -> FillupMetaDataIDs(m_MediaObject)
     * Update NowPlaying property at OutputWrapper -> UpdateNowPlaying().
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] metadata1 coming from DeviceControl with meta data content
     * @param[in] metadata2 coming from DeviceControl with meta data content
     * @param[in] metadata3 coming from DeviceControl with meta data content
     * @param[in] metadata4 coming from DeviceControl with meta data content
     * @param[in] metadataTitle coming from DeviceControl with meta data content
     * @param[in] mediaType current media type sent by PlayerEngine
     * @param[in] bitRate current media type sent by PlayerEngine //Roadmap 13001
     * @param[in] sampleRate current media type sent by PlayerEngine //Roadmap 13001
     * @param[in] audioChannelFormat current media type sent by PlayerEngine //Roadmap 13001
     * @return != 0: error, = 0: OK
     */
    tResult ForwardNowPlayingStatus(const tPEHandle handle,        //Roadmap 13010: 100%, 13001: 100%, 13008: 100%
                                    const tMetadata metadata1,
                                    const tMetadata metadata2,
                                    const tMetadata metadata3,
                                    const tMetadata metadata4,
                                    const tTitle metadataTitle,
                                    const tMediaType mediaType,
                                    const tBitRate bitRate,
                                    const me::samplerate_i sampleRate,
                                    const tAudioChannelFormat audioChannelFormat,
                                    const tConvertFlag metadataConvertFlag,
                                    const tUUID uuid,
                                    const tAlbumArt albumArtString);

    /**
     * Function handles the ACTIVITY_STATUS signal.
     * Roadmap 13013
     * Set internal player manager member values according to the values.
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] activityStatus activity status of PlayerEngine (idle or busy)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardActivityStatus(const tPEHandle handle, const tPEActivityStatus activityStatus); //Roadmap 13013

    /**
     * Function handles the playback mode signal coming from the device.
     * Roadmap 13008
     * Set playback mode in ListControl -> SetPlaybackMode(deviceID, playbackMode).
     *
     * @param[in] deviceID device ID of effected device
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackModeStatus(const tDeviceID deviceID, const tPlaybackMode playbackMode); //Roadmap 13008: 100%

    /**
     * Function handles the repeat mode signal coming from the device.
     * Roadmap 13008
     * Set repeat mode in ListControl -> SetRepeatMode(deviceID, repeatMode).
     *
     * @param[in] deviceID device ID of effected device
     * @param[in] repeatMode repeat the current playing track or list or nothing
     * @return != 0: error, = 0: OK
     */
    tResult ForwardRepeatModeStatus(const tDeviceID deviceID, const tRepeatMode repeatMode); //Roadmap 13008: 100%

    /**
     * Function handles the ALBUM_ART_STATUS signal.
     * Roadmap 13006
     * Set internal player manager member values according to the values.
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] available the album art of media object is available or not yet
     * @return != 0: error, = 0: OK
     */
    tResult ForwardAlbumArtStatus(const tPEHandle handle, const tAvailable available); //Roadmap 13006: 100%

    /**
     * Function sets active remote device.
     * Roadmap 13008
     * Get deviceID from DBManager::GetActiveDevice(&currentDeviceID).
     * if((currentListType != LTY_CURRENT_SELECTION)
     *    ||
     *    (streamingMode == SM_OFF)
     *    ||
     *    (deviceID != currentDeviceID))
     * - CustomControl::SetActiveStreamingDevice(deviceID).
     * Remark: If the MP switches to streaming mode if a remote activity in a not active device is recognized is project dependent
     *
     * @param[in] deviceID device ID of device which reports remote activity
     * @return != 0: error, = 0: OK
     */
    tResult RemoteActivity(const tDeviceID deviceID); //Roadmap 13008: 100%

    /**
     * Function sets streaming mode to SM_ON.
     * Roadmap 13008
     *
     * @return != 0: error, = 0: OK
     */
    tResult StreamingModeOn(); //Roadmap 13008: 100%

    /**
     * Function sets streaming mode to SM_OFF.
     * Roadmap 13008
     *
     * @return != 0: error, = 0: OK
     */
    tResult StreamingModeOff(); //Roadmap 13008: 100%

    /**
     * Function handles a stream error situation.
     * In case of a file error (reason=FORMAT_ERROR||DRM_ERROR||FATAL_READ_ERROR) //Roadmap 13010/13012
     * - Get file error handling configuration from DataProvider -> FileErrorHandling (FEH_NO_ACTION, FEH_MARK_OBJECT, FEH_DELETE_OBJECT)
     * - if( fileErrorHandling == MARK_OBJECT )
     *   - Check if indexing is supported for given device type by DataProvider -> GetIndexingPriority(&priority, m_MediaObject.deviceType)
     *     if(priority != PRIORITY_NONE)
     *     - the media object is marked as not playable in the DB
     *       -> DBManager::FillupMetaDataIDs(m_MediaObject)
     *       -> DBManager::UpdateMediaObject(m_MediaObject)
     * - else if( fileErrorHandling == FEH_DELETE_OBJECT )
     *   - the media object m_MediaObject will be deleted by the list control.
     *     This is to remove old media object entries in the database.
     * if( reason == UNDERVOLTAGE_ERROR ) -> Call SendMessage(PAUSE). //Roadmap 13024
     * else if(REASON_DEVICE_ERROR != reason) -> Call SendMessage(NEXT, steps=1, isHMIRequest=false).
     * else -> Call SendMessage(NO_OBJECT, NOR_NO_OBJECT).
     *
     * @param[in] errorCode error code sent by the PlayerEngine
     * @return != 0: error, = 0: OK
     */
    tResult HandleStreamError(const me::reason_e reason); //Roadmap 13010: 100%, 13012: 100%, 13024: 100%

    /**
     * Function handles a stream error situation.
     * Call SendMessage(NO_OBJECT, NOR_NO_OBJECT).
     *
     * @param[in] errorCode error code sent by the PlayerEngine
     * @return != 0: error, = 0: OK
     */
    tResult HandleStreamErrorStreaming(const me::reason_e reason); //Roadmap 13010: 100%, 13012: 100%, 13024: 100%

    /**
     * FirstErrorTimer callback
     *
     * @param[in] timerID timer ID
     * @param[in] instance pointer to own (PlayerManager) instance
     * @param[in] userData pointer to possible user data
     * @return true or false
     */
     static bool FirstErrorTimerCallback(timer_t timerID ,void* instance, const void *userData);

     /**
     * Function handles an error after triggering an action.
     * Roadmap 13010
     * Clear release event and disarm timer of own state machine in case of still waiting -> ClearReleaseEvent().
     *
     * @param[in] reason reason for error
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason); //Roadmap 13010: 100%

    /**
     * Restore requested source allocation state after undervoltage.
     * Roadmap 13024
     * if( currentAllocateState == DEALLOCATED )
     * - if( newAllocateState == ALLOCATED || newAllocateState == ACTIVE )      -> Call SendAllocate(audioOutputDevice).
     * - if( newAllocateState == ACTIVE )                                       -> Call SendSourceActivity(SA_ON).
     * if( currentAllocateState == ALLOCATED )
     * - if( newAllocateState == ACTIVE )                                       -> Call SendSourceActivity(SA_ON).
     * - if( newAllocateState == DEALLOCATED )                                  -> Call SendDeAllocate().
     * if( currentAllocateState == ACTIVE )
     * - if( newAllocateState == ALLOCATED || newAllocateState == DEALLOCATED ) -> Call SendSourceActivity(SA_OFF).
     * - if( newAllocateState == DEALLOCATED )                                  -> Call SendDeAllocate().
     *
     * @param[in] allocateState source allocation state (DEALLOCATED, ALLOCATED or ACTIVE)
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult RestoreAllocateState(const tAllocateState allocateState, const tAudioOutputDevice audioOutputDevice); //Roadmap 13024: 100%

    /**
     * Setter for allocate state.
     * Roadmap 13024
     *
     * @param[in] allocateState source allocation state (DEALLOCATED, ALLOCATED or ACTIVE)
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetAllocateState(const tAllocateState allocateState); //Roadmap 13024: 100%

    /**
     * Getter for allocate state and ALSA name for audio output.
     * Roadmap 13024
     *
     * @param[out] allocateState source allocation state (DEALLOCATED, ALLOCATED or ACTIVE)
     * @param[out] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult GetAllocateState(tAllocateState &allocateState, tAudioOutputDevice &audioOutputDevice); //Roadmap 13024: 100%

    /**
     * Getter for active state.
     *
     * @param[out] isActive true or false
     * @return != 0: error, = 0: OK
     */
    tResult GetActiveState(tBoolean &isActive);

    /**
     * Setter for streaming mode (remote control of device e.g. iPod).
     * Roadmap 13008
     *
     * @param[in] newStreamingMode SM_OFF or SM_ON
     * @return != 0: error, = 0: OK
     */
    tResult SetStreamingMode(const tStreamingMode newStreamingMode); //Roadmap 13008: 100%

    /**
     * Getter for streaming mode.
     * Roadmap 13008
     *
     * @param[out] currentStreamingMode SM_OFF or SM_ON
     * @return != 0: error, = 0: OK
     */
    tResult GetStreamingMode(tStreamingMode &currentStreamingMode); //Roadmap 13008: 100%

    /**
     * Setter for current active list ID used for playback.
     *
     * @param[in] listID id of current active list
     * @return != 0: error, = 0: OK
     */
    tResult SetListID(const tListID listID);

    /**
     * Getter for current active list ID used for playback.
     *
     * @param[out] listID id of current active list
     * @return != 0: error, = 0: OK
     */
    tResult GetListID(tListID &listID);

    /**
     * Setter for current media object used for playback.
     *
     * @param[in] mediaObject current active media object
     * @return != 0: error, = 0: OK
     */
    tResult SetMediaObject(const tMediaObject mediaObject);

    /**
     * Getter for current media object used for playback.
     * if( streaming mode == SM_OFF ) //Roadmap 13008
     * - Deliver internal media object
     * else
     * - Get current media object data from ListControl -> GetCurrentMediaObject(&mediaObject, &mediaObjectStatus, m_ListID)
     *
     * @param[out] mediaObject current active media object
     * @return != 0: error, = 0: OK
     */
    tResult GetMediaObject(tMediaObject &mediaObject); //Roadmap 13008: 100%

    /**
     * Getter for device related information for media object.
     * Get current listID -> GetListID(&currentListID).
     * Get list info from ListControl -> GetListInfo(&listInfo, currentListID).
     * Get device info from DBManager -> GetDeviceInfo(&deviceInfo, listInfo.deviceID).
     * Copy device relevant data into the media object.
     *
     * @param[out] mediaObject media object to store the device relevant data
     * @return != 0: error, = 0: OK
     */
    tResult GetDevicePartOfMediaObject(tMediaObject &mediaObject);

    /**
     * Setter for current buffer index.
     * Roadmap 13010
     *
     * @param[in] bufferIndex index of media object in current list which is currently buffered
     * @return != 0: error, = 0: OK
     */
    tResult SetBufferIndex(const tPosition bufferIndex); //Roadmap 13010: 100%

    /**
     * Getter for current buffer index.
     * Roadmap 13010
     *
     * @param[out] bufferIndex index of media object in current list which is currently buffered
     * @return != 0: error, = 0: OK
     */
    tResult GetBufferIndex(tPosition &bufferIndex); //Roadmap 13010: 100%

    /**
     * Function sets switch for auto register on DB trigger while Run
     * *
     * @param[in] autoRegister on or off
     * @return != 0: error, = 0: OK
     */
    tResult AutoRegisterOnDBTrigger(const tTriggerState autoRegister);

    /**
     * Function registers or unregisters DB trigger for the PlayerManager
     * *
     * @param[in] trigger DB trigger of an event for indexer
     * @param[in] requestedState on or off -> register or deregister DB trigger
     * @return != 0: error, = 0: OK
     */
    tResult SwitchDBTrigger(const tTriggerType trigger, const tTriggerState requestedState);//Roadmap 13035_Overtemperature  adjust function to react with REMOVE if CS_OVERTEMP is set in Database
    
    /**
     * Function starts a Timer with the WaitTime Value for Playback
     * *
     * @return != 0: error, = 0: OK
     */
    tResult ReStartMultipleNextWaitTimer(); //Roadmap 14001_Multiple_Next.

    /**
     * MultipleNextWaitTimer callback
     *
     * @param[in] timerID timer ID
     * @param[in] instance pointer to own (PlayerManager) instance
     * @param[in] userData pointer to possible user data
     * @return true or false
     */
     static bool MultipleNextWaitTimerCallback(timer_t timerID ,void* instance, const void *userData); //Roadmap 14001_Multiple_Next.

     /**
      * Function stops the MultipleNextWaitTimer
      * *
      * @return != 0: error, = 0: OK
      */
     tResult StopMultipleNextWaitTimer(); //Roadmap 14001_Multiple_Next.

    /**
     * Roadmap_13032_FFWD_FFRW_in_Inactive
     * Function reads the configuration to support FFWD / FREV simulation in deAllocated state
     * Its checks if Simulation rate is a non-Zero value to enable the feature
     *
     * @return 1 if simulation rate is non-Zero , 0 otherwise
     */
     tResult IsFFwdFRevSimulationEnabled();

     /**
      * Roadmap_13032_FFWD_FFRW_in_Inactive
      * Function simulates the fast forward action of a media object.
      * It increments and updates the elapsed time at constant configured time interval
      *
      * @return != 0: error, = 0: OK
      */
     int FastForwardSimulationStart();

     /**
      * Roadmap_13032_FFWD_FFRW_in_Inactive
      * Function stops the simulated fast forward action of a media object.
      * It sets the m_StartTime to latest simulated elapsedTime value to enable the
      * play back from that position when Playermanager is allocated next time
      *
      * @return != 0: error, = 0: OK
      */
     int FastForwardSimulationStop();

     /**
      * Roadmap_13032_FFWD_FFRW_in_Inactive
      * Function simulates the fast reverse action of a media object.
      * It decrements and updates the elapsed time at constant configured time interval
      *
      * @return != 0: error, = 0: OK
      */
     int FastReverseSimulationStart();

     /**
      * Roadmap_13032_FFWD_FFRW_in_Inactive
      * Function stops the simulated fast reverse action of a media object.
      * It sets the m_StartTime to latest simulated elapsedTime value to enable the
      * play back from that position when Player manager is allocated next time
      *
      * @return != 0: error, = 0: OK
      */
     int FastReverseSimulationStop();

     /**
      * Roadmap_13032_FFWD_FFRW_in_Inactive
      * Timer callback function for FFWD/FREV simulation progress
      *
      * @param[in] timerID timer ID
      * @param[in] instance pointer to own (PlayerManager) instance
      * @param[in] userData pointer to possible user data
      * @return true or false
      */
     static bool FFwdFRevProgressTimerCallBack(timer_t timerID ,void* instance, const void *userData);
     tBool FFwdFRevProgressTimerCallBack();

     tResult IsDeActivateAllFavoritesRequired(const tUUID uuid, tDeviceID deviceid, const tListID currentListID, const tListSize listsize);

    /**
     * Methods returns the current scanmode of the currently active device. 
     * @param[out] current ScanMode value,which is true/false
     * @return < 0: error, = 0: OK
     */
    tResult GetScanMode(tScanMode &scanMode);
    
     /**
     * Methods informs the PlayerManagerSM to send the nelwy requested scanmode 
     * to the currently active device. 
     * @param[in] ScanMode,which is true/false
     * @return < 0: error, = 0: OK
     */
    tResult SendScanMode(const tScanMode scanMode);
    
     /**
     * Methods stores  the current scan mode value informed by the currently active deviceControl.
     * In addition,it informs the OutputWrapper to update the current scan mode value to HMI 
     * @param[in] DeviceID;ScanMode,which is true/false
     * @return < 0: error, = 0: OK
     */
    tResult ForwardScanModeStatus(const tDeviceID deviceID,const tScanMode scanMode);


    /**
     * Roadmap 15027 : Adjust Brightness of Video
     * SendVideoProperty Method to set the video property of the video objects in all devices.
     * Properties like brightness, hue, saturation, contrast, brightness offset , hue offset, saturation offset can be set using this function interface.
     * Properties set will be available for all video objects  for all devices.
     * @param[in] video  property type enum
     * @param[in] video  property value of the corresponding video property type
     * @return < 0: error, = 0: OK
     */
    tResult SendVideoProperty(IN const tVideoProperty videoProperty, IN const tPropertyValue propertyValue);

    /**
     * Roadmap 15027 : Adjust Brightness of Video
     * ValidateVideoProperty Method to validate the input video property with the configured range.ie, if the video property type is a known type and if property value lies within the range of configured min-max values.
     * @param[in] video  property type enum
     * @param[in] video  property value of the corresponding video property type
     * @return true or false
     */
    tBoolean ValidateVideoProperty(IN const tVideoProperty videoProperty, IN const tPropertyValue propertyValue);


    /**
     * Roadmap 15027 : Adjust Brightness of Video
     * HandleVideoProperty Method to validate the input video property with the previously stored value in database and store the new value to database if value is changed.
     * @param[in] video  property type enum
     * @param[in] video  property value of the corresponding video property type
     * @return < 0: error, = 0: OK
     */
    tResult HandleVideoProperty( IN const tPropertyValue propertyValue,IN const tVideoProperty videoProperty);
    
    /**
    * This Method compares the given listID  with the CurrentListID in PlayerManager
    * If same,the given position is updated in Nowplaying info and NowPlaying property's status is updated to Clients,
    * @param[in] listID ID of a List
    * @param[in] position position of the list's Nowplaying item
    * @return != 0: error, = 0: OK
    */
    tResult UpdateNowPlayingWithGivenPosition(const tListID listID,const tPosition position);
    void SetActiveDevice(tDeviceID activeDevice);
    void GetActiveDevice(tDeviceID & activeDevice);
    

    /**
    * This function Clears or Informs to Ignore the lastmode pause.
    *
    * With the given DeviceID,the device is identified as Streaming or Non-Streaming(i.e NormalMode) Device
    * If NormalMode device,it clears LastModePause by setting m_AutoplayFlag to TRUE.
    * If StreamingMode device,It store the DeviceID in DataProvider.So DeviceControl,before beginning Playback of a device,checks for its deviceID
    * in Dataprovider.If found,it resumes Playback.Else,remains paused.
    *
    * @param[in] tDeviceID Device ID
    * @return < 0: error, = 0: OK
    */    
    tResult ClearLastModePause (const tDeviceID deviceID);

    void GetPlaybackSpeed(tPlaybackSpeed &playbackSpeed);
      /**
        This Function returns the source activity pause flag
        */
    tBoolean IsSourceActivityPause();
     /**
     * This method  getStreamingInfo used to send streaming info
     * @param[in]
     * @param[OUT] device : Device type ,
     * @return < 0: error, = 0: OK
     */
    tResult getStreamingInfo(OUT tResponseMsg &info, OUT tDeviceType &type, OUT tDeviceID &deviceId);
    /**
     * HandelNoResponseProperty property trigger from Device control
     */
    tResult HandelNoResponseProperty(tResponseMsg response, tDeviceID id, tDeviceType type);
    /**
     * SetNowPlayingMetadataEditedByUser will copy the latest metadata edited by user from DB and updtae in the now playing object
     */
    tResult SetNowPlayingMetadataEditedByUser(IN const tObjectID editedMediaObjectID);
private:

    tResult PrepAndSendScanMessage(const tDeviceType deviceType, const tDeviceID deviceID, const tScanMode scanMode);
    tResult ScanMode(const tDeviceType deviceType, const tDeviceID deviceID, const tScanMode scanMode);
    tResult CheckConnectionStateMyMedia();

    /*NCG3D-25268:Sets m_AutoPlayFlag to True
     *
     *Reason:If newlist is Non-Streaming list & playerManager is in StreamingMode,then it switches OFF StreamingMode.
     *While switching OFF,the m_AutoPlayFlag set to False.
     *As a result,NewList Playback did not happen in PausedLastMode State.
     *This issue is solved by setting m_AutoPlayFlag to True once Streamingmode is switched OFF
     */
    tResult SetAutoPlayFlag();

    /** CMG3G - 8071
     * This function changes device's RepeatMode to DefaultRepeatMode,if the current Repeatmode being RPT_ALL
     *
     * @return < 0: error, = 0: OK
     */
    tResult DecideSwitchingToDefaultRepeatMode();
    tResult SendNoResponseEvent(const me::reason_e response);

    tTriggerState m_AutoRegisterOnDBTrigger;
    tTriggerID m_TriggerID_DevRemoved;
    Lock m_Mutex;                           /**< lock access to member variables */
    tUInt m_ErrorHandlerRetryCounter;       /**< retry counter for  */

    tStreamingMode m_StreamingMode;         /**< current streaming mode */
    tAllocateState m_AllocateState;         /**< current allocate state (Roadmap 13024) */
    tAudioOutputDevice m_AudioOutputDevice; /**< current audio output device (alsa device name) (Roadmap 13024) */
    tListID m_ListID;                       /**< current active list ID */
    tPlaytime m_StartTime;                  /**< start playtime used by PlayMediaObject */
    tMediaObject m_MediaObject;             /**< current media object */
    tMediaObject m_prevMediaObject;         /**< previous media object */
    tPosition m_BufferIndex;                /**< current buffer index */
    tPlaybackAction m_LastAction;           /**< last HMI playback action */
    tObjectID m_FirstErrorObjectID;         /**< first element of a list which causes an error */
    tPosition m_firstErrorObjectPosition;   /**< position of first element of a list which causes an error */
    tBoolean m_AutoPlayFlag;                /**< auto play flag, set by new selection (NewList) in deallocate state */
    tBoolean m_SAPauseFlag;                 /**< last SA_OFF is a PAUSE due to an interrupt source like TA */
    tBoolean m_NewStreamingObject;          /**< new streaming object flag, set by new selection (NewList) for permanent streaming devices */

    tHMIPlaybackState m_HMIPlaybackState;   /**< last playback state send to HMI */
    tNowPlaying m_HMINowPlaying;            /**< last NowPlaying object send to HMI */
    tMediaObject m_PrevNowPlayingUpdateMediaObject;/**< Previous NowPlaying object  sent to HMI */
    tListID m_PrevNowPlayingUpdateListID;   /**< ListID of m_PrevNowPlayingUpdateMediaObject */
    tAvailable m_AlbumArtAvailable;         /**< last album art availability of NowPlaying object */
    tPlaytime m_HMIElapsedPlaytime;         /**< last ElapsedPlaytime (part of Playtime) send to HMI */
    tPlaytime m_HMITotalPlaytime;           /**< last TotalPlaytime (part of Playtime) send to HMI */
    tObjectID m_HMIObjectID;
    tBoolean m_ValidLastMode;               /**< current last mode is valid? */

    Timer m_MultipleNextWaitTimer;          /**< multiple next timer to wait for start playback after next/prev action */
    timer_t m_MultipleNextWaitTimerID;      /**< multiple next timer index */
    Timer m_FFwdFRevProgressTimer;          /**< progress timer simulation for FFwd/FRev while in deAllocated state*/
    timer_t m_FFwdFRevProgressTimerID;      /**< progress timer simulation index */
    Timer m_GetAudioDeviceTimer;            /**< get audio device timer to wait for Allocate request from audiomanagement */
    timer_t m_GetAudioDeviceTimerID;        /**< get audio device timer index */
    Timer m_AppleBugWrongSourceTimer;       /**< wrong source timer to delay SA_ON because of possible wrong source allocation due to a apple bug */
    timer_t m_AppleBugWrongSourceTimerID;   /**< wrong source timer index */
    Timer m_FirstErrorTimer;                /**< first restart after streaming error is still ongoing timer */
    timer_t m_FirstErrorTimerID;            /**< first restart after streaming error timer index */

    tScanMode mIsScanMode;
    tObjectID m_firstScannedMediaObjectID; 
    tDeviceID m_ActiveDevice;
    tDeviceID mDeviceIdResponse;
    tResponseInfo mResponseMsg;
};

#endif //_PLAYERMANAGER_H_

/** @} */
