#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_PICTURE_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/PictureManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_PICTURE_MANAGER
#endif
#endif

#include "LocalSPM.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "MediaPlayer_ErrorCodes.h"
#include "PictureManager.h"

PictureManager::PictureManager(const tComponentID componentID):ILocalSPM(componentID)
{
    ENTRY_INTERNAL

    /* Early initialization of variables */
    InitMembers();
}

PictureManager::~PictureManager()
{
    ENTRY_INTERNAL
}

void PictureManager::Create()
{
    ENTRY

    /* Create the state machine */
    PictureManagerSM::Create();

    CreateDone(0);
}

tResult PictureManager::Init(tInitReason reason)
{
    ENTRY

    (void)reason;
    /* Init the state machine */
    PictureManagerSM::Init();
    SetAnswerTimeout(PICTURE_MANAGER_SM_ANSWER_TIMEOUT_MS); 

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);

    return InitDone(0);
}

tResult PictureManager::InitSM()
{
    ENTRY

    InitMembers();

    return MP_NO_ERROR;
}

tResult PictureManager::InitMembers()
{
    ENTRY

    /* Initialize member variables */
    m_nowShowing.listID = LIST_ID_NONE;
    m_nowShowing.position = 0;
    InitMediaObject(OUT m_nowShowing.object);
    strncpy_r(m_nowShowing.nextFile, "", sizeof(m_nowShowing.nextFile));
    m_HMIPlaybackState = HMI_PBS_STOPPED;

    return MP_NO_ERROR;
}

tResult PictureManager::Run()
{
    ENTRY

    LocalSPM::GetThreadFactory().Do(this, 0, NULL); //PictureManager inclusive state machine

    return RunDone(0);
}

void PictureManager::Do(int functionID, void *ptr)
{
    ENTRY
    (void)functionID;
    (void)ptr;

    //set the threads name
    LocalSPM::GetThreadFactory().SetName(PictureManagerSM::GetSMNameFull());

    //actual thread
    while(PictureManagerSM::STATE_MACHINE_FINISHED != PictureManagerSM::StateMachine_Main()){}
}

tResult PictureManager::Stop()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Send STOP message to own SM */
    ret = SendForceEvent(STOP_SM, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult PictureManager::StopEventProcessed()
{
    ENTRY

    /* Send stop done to SPM in the transition to final state in state machine */
    return StopDone(0);
}

tResult PictureManager::Done()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Deregister state machine with dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* Send DONE message to own SM */
    ret = SendForceEvent(DONE, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult PictureManager::DoneEventProcessed()
{
    ENTRY

    /* Send done done to SPM in the transition to final state in state machine */
    return DoneDone(0);
}

char *PictureManager::GetSMStateName(OUT tGeneralString stateName, size_t size)
{
    GetCurrentState((char *)stateName, size);
    return stateName;
}

tResult PictureManager::UpdateNowShowing()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    ret = LocalSPM::GetOutputWrapper().UpdateNowShowing();
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while updating NowShowing via OutputWrapper (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult PictureManager::UpdateSlideShowState()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    switch (SMF::GetState()) {
        case Playing:
        case Active:
            SetSlideshowState(HMI_PBS_PLAYING);
            break;
        case Paused:
            SetSlideshowState(HMI_PBS_PAUSED);
            break;
        case Idle:
            SetSlideshowState(HMI_PBS_STOPPED);
            break;
        default:
            break;
    }

    ret = LocalSPM::GetOutputWrapper().UpdateSlideshowState();
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while UpdateSlideshowState via OutputWrapper (ErrorCode:%s)", errorString(ret)));
    }

    return ret;

}


tResult PictureManager::StartTimer()
{

    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_SlideShowTimer.StartTimer(OUT m_SlideShowTimerID,
               IN LocalSPM::GetDataProvider().SlideShowTimeInterval(),
               IN LocalSPM::GetDataProvider().SlideShowTimeInterval(),
               IN &LocalSPM::GetPictureViewer(),
               IN &SlideShowTimerCallBack,
               IN NULL);
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::StopTimer()
{

    ENTRY
    tResult ret = MP_NO_ERROR;
    m_SlideShowTimer.CancelTimer(IN m_SlideShowTimerID);
    return ret;
}

bool PictureManager::SlideShowTimerCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY
    (void)userData;
    tResult ret = MP_NO_ERROR;

    if (NULL == instance)    {
        ETG_TRACE_ERR(("Invalid parameter: instance is NULL (ErrorCode:%s)", errorString(ret)));
        ret = MP_ERR_PM_INVALID_PARAM;
    } else {
        ETG_TRACE_USR3(("PictureManager::SlideShowTimerCallBack timerID:0x%X, instance:0x%X", (int)((long)timerID), instance));

        /* Send timeout event */
        PictureManager *self = (PictureManager*)instance;
        ret = self->SendEventByName("TIMEOUT", (char *)NULL);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return (ret == MP_NO_ERROR);
}


tResult PictureManager::TimeOutAction()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    ret = NextAction(1);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("Error while calling NextAction (ErrorCode:%s)", errorString(ret)));
    }

    return ret;

}

tResult PictureManager::NextAction(const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    if (0 == nextPrevSkipCount) return ret; //error code has to be defined

    tListInfo listInfo;
    m_Mutex.lock();
    tPosition newPosition = m_nowShowing.position + nextPrevSkipCount;
    ret = LocalSPM::GetListControl().GetListInfo(OUT listInfo, IN m_nowShowing.listID, IN TRUE);
    m_Mutex.unlock();

    if (ret) return ret;


    /*For Renault, Slideshow should start over from the beginning of the list if the end of list is reached.(Bug#NCG3D-48980)
    For other variants, if the Slide show should PAUSE when end of the list if reached, appropriate changes has to be made*/
    if ((newPosition + 1) > listInfo.listSize) {
        if (SMF::GetState() == Playing) {
            newPosition = 0;
        }
        else
        {
            return ret; //error code has to be defined
        }
    }

    ret = ProcessAction(PBA_NEXT, newPosition, listInfo.listSize);

    return ret;
}

tResult PictureManager::PreviousAction(const tNextPrevSkipCount nextPrevSkipCount)
{

    ENTRY
    tResult ret = MP_NO_ERROR;

    if (0 == nextPrevSkipCount) return ret; //error code has to be defined

    //check if the new position is valid
     m_Mutex.lock();
     if ((0 == m_nowShowing.position) || (m_nowShowing.position < nextPrevSkipCount)) {
         m_Mutex.unlock();
         if (SMF::GetState() == Playing) {
             ret = RequestSlideshowAction(PBA_PAUSE);
         }
         return ret; //error code has to be defined
     }

    tPosition newPosition = m_nowShowing.position - nextPrevSkipCount;
    m_Mutex.unlock();
    ret = ProcessAction(PBA_PREV, newPosition);
    return ret;
}

tResult PictureManager::ProcessAction(const tPlaybackAction playbackAction, const tPosition position, const tListSize listSize)
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Restart timer only when playackAction is [Next/Prev] & current SM state is Playing */
    if ((PBA_PLAY != playbackAction) && (SMF::GetState() == Playing)) {
        SetSlideshowState(HMI_PBS_PLAYING);
        ret = RestartTimer();
        if (ret) return ret;
    }

    tMediaObject currentMediaObject, nextMediaObject;
    InitMediaObject(OUT currentMediaObject);
    InitMediaObject(OUT nextMediaObject);

    m_Mutex.lock();
    tListID listID = m_nowShowing.listID;
    m_Mutex.unlock();

    ret = LocalSPM::GetListControl().GetMediaObject(OUT currentMediaObject, IN listID, IN position);
    if (ret) return ret;

    switch (playbackAction) {
        case PBA_PLAY: {
            tPosition tmpPosition;
            if ((position + 1) < listSize) {
                tmpPosition = position + 1;

            }
            else
            {
                tmpPosition = 0;//Slideshow should start from the beginning when end of the list is reached.
            }
            ret = StartTimer();
            if (ret) return ret;
            ret = LocalSPM::GetListControl().GetMediaObject(OUT nextMediaObject, IN listID, IN tmpPosition);

            if (ret) return ret;
            ret = UpdateSlideShowState();
            if (ret) return ret;
         }
         break;
         case PBA_PREV: {
             m_Mutex.lock();
             if (1 < (m_nowShowing.position - position)) {
                 m_Mutex.unlock();
                 ret = LocalSPM::GetListControl().GetMediaObject(OUT nextMediaObject, IN listID, IN position + 1);
                 if (ret) return ret;
             } else {
                 if (1 == (m_nowShowing.position - position)) {
                     nextMediaObject = m_nowShowing.object;
                 }
                 m_Mutex.unlock();
             }
         }
         break;
         case PBA_NEXT: {
             tPosition tmpPosition;
             if ((position + 1) < listSize) {
                 tmpPosition = position + 1;

             }
             else
             {
                 tmpPosition = 0;//Slideshow should start from the beginning when end of the list is reached.
             }
             ret = LocalSPM::GetListControl().GetMediaObject(OUT nextMediaObject, IN listID, IN tmpPosition);
             if (ret) return ret;
         }
         break;
         default:
         break;
    }

    m_Mutex.lock();
    m_nowShowing.position = position;
    m_nowShowing.object = currentMediaObject;
    strncpy_r(m_nowShowing.nextFile, nextMediaObject.fileName, sizeof(m_nowShowing.nextFile));

    if (0 == position) {
        m_nowShowing.state = NP_LIST_START;
    } else {
        m_nowShowing.state = (listSize == (position + 1)) ? NP_LIST_COMPLETE : NP_NEW_TRACK;
    }
    m_Mutex.unlock();

    ret = UpdateNowShowing();
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while updating Now showing (ErrorCode:%s)", errorString(ret)));
    }
    return ret;
}

tResult PictureManager::NewSlideShow()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    m_Mutex.lock();
    tPosition position = m_nowShowing.position;
    m_Mutex.unlock();
    tListInfo listInfo;
    ret = LocalSPM::GetListControl().GetListInfo(OUT listInfo, IN m_nowShowing.listID, IN TRUE);
    if (ret) return ret;

    ret = ProcessAction(PBA_PLAY, position,listInfo.listSize);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error from ProcessAction (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}


tResult PictureManager::StartSlideshow(const tListID listID, const tObjectID objectID)
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    //check if already a slideshow is active if so then send error
    if (HMI_PBS_STOPPED != m_HMIPlaybackState) return ret; //error code has to be defined.

    tPosition position = 0;
    ret = LocalSPM::GetListControl().GetPositionInList(OUT position, IN listID, IN objectID);

    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while getting position from ListControl (ErrorCode:%s)", errorString(ret)));
    } else {
        SetNowshowingListID(listID);
        SetNowshowingPosition(position);

        /* Send PLAY message to own SM */
        ret = SendEvent(PLAY, (char *)NULL);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return ret;
}

tResult PictureManager::RequestSlideshowAction(const tPlaybackAction playbackAction, const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY

    tResult ret = MP_NO_ERROR;
    tEvent *event = STOP;

    switch(playbackAction) {
        case PBA_PLAY: {
            event = PLAY;
        }
        break;
        case PBA_PAUSE: {
            event = PAUSE;
        }
        break;
        case PBA_STOP: {
            event = STOP;
        }
        break;
        case PBA_PREV: {
            event = PREV;
        }
        break;
        case PBA_NEXT: {
            event = NEXT;
        }
        break;
        default: {
           ETG_TRACE_ERR(("Invalid SlideShowAction type: %u",playbackAction));
           ret = MP_ERR_PM_INVALID_PARAM;
        }
        break;
    }

    if (MP_NO_ERROR == ret) {
         /* Pack parameter steps for event NEXT and PREV */
         if ((PBA_PREV == playbackAction) ||
             (PBA_NEXT == playbackAction)) {

             tAllParameters parameterString = {0};
             size_t size = sizeof(parameterString);

            ret = ParameterNEXT(OUT parameterString, IN size, IN nextPrevSkipCount);
            if (MP_NO_ERROR != ret) {
                 ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            } else {
                 ret = SendEvent(IN event, IN parameterString);
                 if (MP_NO_ERROR != ret) {
                     ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
                 }
            }
         } else if ((PBA_PLAY == playbackAction) ||
                    (PBA_PAUSE == playbackAction) ||
                    (PBA_STOP == playbackAction)) {

            ret = SendEvent(event, (char *)NULL);
            if (MP_NO_ERROR != ret) {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }

        }
    }
    return ret;
}


tResult PictureManager::SetSlideshowTime(const tSlideshowTime slideshowTime)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    LocalSPM::GetDataProvider().SlideShowTimeInterval = slideshowTime;
    m_Mutex.unlock();
    return ret;
}


tResult PictureManager::GetSlideshowTime(tSlideshowTime &slideshowTime)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    slideshowTime = LocalSPM::GetDataProvider().SlideShowTimeInterval();
    m_Mutex.unlock();
    return ret;
}


tResult PictureManager::GetNowShowing(tNowShowing &nowShowing)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    nowShowing = m_nowShowing;
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::GetSlideshowState(tHMIPlaybackState &playbackState)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    playbackState = m_HMIPlaybackState;
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::SetSlideshowState(const tHMIPlaybackState playbackState)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_HMIPlaybackState = playbackState;
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::SetNowshowingListID(const tListID listID)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_nowShowing.listID = listID;
    m_Mutex.unlock();
    return ret;

}

tResult PictureManager::SetNowshowingPosition(const tPosition position)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_nowShowing.position = position;
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::SetNowshowingObject(const tMediaObject Object)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_nowShowing.object = Object;
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::SetNowshowingNextFileName(const tFilename nextFileName)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    strncpy_r(m_nowShowing.nextFile, nextFileName, sizeof(m_nowShowing.nextFile));
    m_Mutex.unlock();
    return ret;
}

tResult PictureManager::SetNowshowingState(const  tNowPlayingState state)
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    m_Mutex.lock();
    m_nowShowing.state = state;
    m_Mutex.unlock();
    return ret;
}


tResult PictureManager::RestartTimer()
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    ret = StopTimer();
    if (ret == MP_NO_ERROR) {
        ret = StartTimer();
    }
    return ret;
}


