
/**
 * @defgroup OutputWrapper OutputWrapper
 * @ingroup MediaPlayer
 * @author Stefan Scherber
 *
 * Public interface for OutputWrapper
 * @{
 */

#ifndef _OUTPUTWRAPPER_H_
#define _OUTPUTWRAPPER_H_

#include <iostream>
using namespace std;

#include "TypeDefinitions.h"
#include "ILocalSPM.h"


#define MEDIAPLAYERFBLOCK 0xAB
#define MEDIAPLAYERFBLOCKINST 0x1

/**
 * Pure virtual class. Is used by the media player components to send messages to external components.
 */
class OutputWrapper : public ILocalSPM
{

public:
    OutputWrapper(tComponentID componentId):ILocalSPM(componentId),currentSource(0),currentSourceActive(0),audioAllocatedDevice(DEVICE_ID_NOT_SET){} ;
    ~OutputWrapper() {};

    /**
     * This function requests an audio channel from the AVManager of the system.  Must be overwritten by the derived class.
     *
     * @param[in] sourceNr
     * @param[in] logicalAVChannel
     * @return < 0: error, = 0: OK
     */
    virtual tResult RequestAVActivation(const tSourceNr sourceNr, const tLogicalAVChannel logicalAVChannel, const tSource source=0, const tDeviceID deviceID = DEVICE_ID_NOT_SET) = 0;

    /**
     * This function requests for connected devices list with DeviceManager
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult RequestConnectedDevices() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when playtime property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaytime() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when playbackstate property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaybackState() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when nowplaying property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateNowPlaying() {return MP_NO_ERROR;};

    /**
     * This function is called when the active device was changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateActiveMediaDevice() {return MP_NO_ERROR;};

    /**
     * This function is called when any device changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateMediaPlayerDeviceConnections() {return MP_NO_ERROR;};

    /**
     * This function is called when a device connection state has changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDeviceConnection(const tDeviceID deviceID, const tConnectionReason connectionReason) {return MP_NO_ERROR;};

    /**
     * This function is called when the indexing state (or percentage) of a device changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateIndexingState() {return MP_NO_ERROR;};

    /**
     * This function is called when the playback mode (shuffle) of a device changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaybackMode() {return MP_NO_ERROR;};

    /**
     * This function is called when the repeat mode of a device changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateRepeatMode() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when slideshow property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateSlideshowState() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when now showing property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateNowShowing() {return MP_NO_ERROR;};

    /**
     * DEPRECATED
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePhoneInfo() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateBTProfileInfo() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPONowPlaying() {return MP_NO_ERROR;};

    /**
     * DEPRECATED
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOMetaData() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOPlaytime() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOPlaybackStatus() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOPlaybackShuffleMode() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOPlaybackRepeatMode() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOCallState() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when Call duration has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOCallDuration() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOCommunications() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when slide show property changed.
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateSlideshowTime() {return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core when SPI component should start or stop with sending location information .
     * Can be overwritten by the derived class.
     * @param[in] startStopLocationInfo start or stop with updates
     * @param[in] locationInfoType supported location info types
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOLocationInfo(){return MP_NO_ERROR;};

    /**
     * This function is called by the mediaplayer core to update SPI componenet on the GPRMC data status values notified by Apple device.
     * The function calls eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOGPRMCDATASTATUSVALUES. The framework then notifies all registerd clients of the change.
     * @param[in] DeviceID
     * @param[in] GPRMCDataStatusValues(A,V,X)
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOGPRMCDataStatusValues(){return MP_NO_ERROR;};

    /**
     * This function is used to notify optical disc slot status towards HMI. I.e. VD_DeviceManager receives it from VD_MediaManager and forwards it though output wrapper to HMI
     *
     * @param[in] enOpticalDiscEjectCmd specify if command is eject or insert
     * @return != 0: error, = 0: OK
     */
     virtual tResult UpdateOpticalDiscSlotState(){return MP_NO_ERROR;}; //Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * This function is called by the mediaplayer core to ask SPI component if role switch is required.
     * The answer will be received asynchronously
     * Can be overwritten by the derived class.
     * @param[in] deviceID DeviceID
     * @param[in] USBSerialNumber USBSerialNumber of Apple device.
     * @return < 0: error, = 0: OK
     */
    virtual tResult DIPORoleSwitchRequired(const tDeviceID deviceID,const tMountPoint USBSerialNumber,const tBool DeviceCarplayCapability){return MP_NO_ERROR;};
    virtual tResult GetBTLimitationModeFromBTClient(const tDeviceSerialNumber serialNumber, tBTLimitationActionState& btLimitation) {return MP_NO_ERROR; };

    /**
     * This function is called by MediaPlayer to get WiFi credentials from SPI
     * Sends answer message to client
     * @param[in] deviceId Device ID of the device whose SPP connection is to be terminated
     * @param[in] appleDeviceMACAddress MAC address of Apple device
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetWiFiCredentials(const tDeviceID deviceID,const tMACAddress appleDeviceMACAddress){return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer to send Disconnect BT SPP connection request to BT
     * Sends answer message to client
     * @param[in] deviceId Device ID of the device whose SPP connection is to be terminated
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDisconnectSPPService(const tMACAddress deviceAddress){return MP_NO_ERROR;};

    virtual tResult AddOobPairedDevice(const tDeviceID deviceID,const tMACAddress deviceAddress,const tDeviceName deviceName, const tLinkKey linkKey,const tBTOobType oobType){return MP_NO_ERROR; };
    /**
     * This function is called when favorite table changes
     * Can be overwritten by derived class
     * @param[in] listChangeType type of the list changes
     * @param[in] listSize list size
     * @param[in] favID changed favorite ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendFavoritesChanged(tListChangeType listChangeType, tListSize listSize, tFavoriteID favID) {return MP_NO_ERROR;};

    /**
     * Reset all the internal status of favorite changed notifications
     * (e.g.
     *      mFavID = 0;
     *      mFavListChangeType = LCH_CONTENT_CHANGED;
     *      mFavListSize = 0;
     * )
     * @return < 0: error, = 0: OK
     */
    virtual tResult ResetFavoritesChanged() {return MP_NO_ERROR;};

    /**
     * This function is called by framework to get the changed values of the favorite table
     * @param[out] listChangeType type of the list changes
     * @param[out] listSize list size
     * @param[out] favID changed favorite ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetFavoriteChanged(tListChangeType &listChangeType, tListSize &listSize, tFavoriteID &favID) {return MP_NO_ERROR;};

    /**
     * This function is called when a list changes
     * Can be overwritten by derived class
     * @param[in] listID ID of the changed list
     * @param[in] listChangeType type of chaange
     * @param[in] listSize size of list
     * @param[in] changedItems changed items
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendMediaPlayerListChanged(tListID listID, tListChangeType listChangeType, tListSize listSize, vector<tPosition> changedItems) {return MP_NO_ERROR;};

    /**
     * This function is called by framework to get the list changes
     * @param[out] listID ID of the changed list
     * @param[out] listSize size of list
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetMediaPlayerListChanged(tListID &listID, tListChangeType &listChangeType, tListSize &listSize, vector<tPosition> &changedItems) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer awhen the result of GetMediaObjectAlbumArtInfo is calculated
     * Must be overwritte by derived class
     * @param[in] mimeType MIME image subtype
     * @param[in] size size of image in bytes
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtInfoAnswer(tMimeType mimeType, tImageSize size, tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArtInfo cannot provided
     * Must be overwritte by derived class
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtInfoError(tErrorCode errorCode, tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt is calculated
     * Must be overwritte by derived class
     * @param[in] imageData image data
     * @param[in] imageSize size of image in bytes
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtAnswer(tImageData imageData, tImageSize imageSize, tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt cannot provided
     * Must be overwritte by derived class
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtError(tErrorCode errorCode, tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of Command is calculated
     * Can be overwritte by derived class
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] appName a string containing App name of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @param[in] userContext  client information
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendCommandAnswer(const tMountPoint mountPoint, const tAppName appName, const tSessionID sessionID, const tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of DiPORequestAudioDevice is calculated
     * Can be overwritten by derived class
     * @param[in] deviceID ID of the device.
     * @param[in] audioOutputDevice name of the alsa device.
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetAudioDeviceAnswer(const tDeviceID deviceID, const tAudioOutputDevice audioOutputDevice) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of DiPORequestAudioDevice cannot be provided
     * Can be overwritten by derived class
     * @param[in] deviceID ID of the device.
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetAudioDeviceError(const tDeviceID deviceID) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when SPI has to return audio device
     * Can be overwritten by derived class
     * @param[in] deviceID ID of the device.
     * @param[in] diPOStopReason reason for stopping carplay playback.
     * @return < 0: error, = 0: OK
     */
    virtual tResult StopMediaPlayback(const tDeviceID deviceID, const tDiPOStopReason diPOStopReason) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when SPI has to resume audio
     * Can be overwritten by derived class
     * @param[in] deviceID ID of the device.
     * @return < 0: error, = 0: OK
     */
    virtual tResult ResumeMediaPlayback(const tDeviceID deviceID) {return MP_NO_ERROR;};

    /* Functions for IPOD authentication */

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] bInit if TRUE initialize authentication; if true deinitalize authentication
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AuthenticationInit(tBool bInit) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] certLength length of certificate
    * @param[out] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AccessoryCertificate(tU16 *certLength, tU8 *certData) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] certLength length of certificate
    * @param[in] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 IPodCertificate(tU16 certLength, tU8 *certData) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] sigDataLen  length of signature
    * @param[in] sigData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 SetSignature(tU16 sigDataLen, tU8 *sigData) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[in] ResponseBuffer  challange data from the IPOD
    * @param[out] SignatureDataLength length of the signature
    * @param[out] SignatureData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 GenSignature(const tU8  *ResponseBuffer, tU16 *SignatureDataLength, tU8 *SignatureData) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] ChallengeDataLength  challange data length
    * @param[out] ChallengeData challange
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 GetChallenge(tU16 *ChallengeDataLength, tU8 *ChallengeData) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] DeviceID device ID
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 DeviceID (tU32 *DeviceID) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] majorVer  Major version of firmware
    * @param[out] minorVer  Minor version of firmware
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 FirmwareVersion (tU8 *majorVer, tU8 *minorVer) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @param[out] majorVer  Major version of protocol
    * @param[out] minorVer  Minor version of protocol
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AuthenticationProtocolVersion (tU8 *majorVer, tU8 *minorVer) = 0;

    /**
    * Pure virtual function. Must be written by derived class
    * @return IPOD_OK: passed or IPOD_ERROR: failed
    */
    virtual tS32 IPodAuthenticationSelftest (tU8 *certificate, tU8 *private_key, tU8 *ram_check, tU8 *checksum) = 0;

    /**
    * Virtual function. Must be written by derived class if appcontrol interface is used
    * Is called by MediaPlayer when the IAP data session open event is received from the iPod Device control
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlConnect(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID, const tProtocolName protocolName) {return MP_NO_ERROR;};

    /**
    * Virtual function. Must be written by derived class if appcontrol interface is used
    * Is called by MediaPlayer when the response is received from the iPod Device control
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @param[in] sizeOfBuffer size of answer buffer
    * @param[in] commandBuffer pointer to answer buffer from IPOD
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlCommand(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer, const tCommandBufferPtr commandBuffer) {return MP_NO_ERROR;};

    /**
    * Virtual function. Must be written by derived class if appcontrol interface is used
    * Is called by MediaPlayer when the IAP data session close event is received from the iPod Device control
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlClose(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID) {return MP_NO_ERROR;};

    /**
    * The function stores the arguments in member variables and calls eUpdateClients(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTART) of ServiceAppControl class
    * @param[in] iAP2iOSAppIdentifier Identifier of connected iOS app
    * @param[in] sinkEndpoint USB endpoint identifier of communication sink.
    * @param[in] sourceEndpoint USB endpoint identifier of communication source.
    * @return < 0: error, = 0: OK
    */
    virtual tResult NativeTransportStart(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint ) {return MP_NO_ERROR;};

    /**
    * The function stores the arguments in member variables and calls eUpdateClients(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTOP) of ServiceAppControl class
    * @param[in] iAP2iOSAppIdentifier Identifier of connected iOS app
    * @param[in] sinkEndpoint USB endpoint identifier of communication sink.
    * @param[in] sourceEndpoint USB endpoint identifier of communication source.
    * @return < 0: error, = 0: OK
    */
    virtual tResult NativeTransportStop(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint ) {return MP_NO_ERROR;};

    /**
    * Internal function to get dipo location info
    */
    virtual tResult GetDiPOLocationInfo(tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType) {return MP_NO_ERROR;};

    /**
    * function to get app control connect  member variables
    */
    virtual tResult GetAppControlConnect(tMountPoint &deviceID, tAppName &appName, tSessionID &sessionID,tProtocolName &protocolName) {return MP_NO_ERROR;};

    /**
    * function to get app control command member variables
    */
    virtual tResult GetAppControlCommand(tMountPoint &deviceID,  tAppName &appName, tSessionID &sessionID, tSize &sizeOfBuffer,  tCommandBufferPtr &commandBuffer) {return MP_NO_ERROR;};

    /**
    * function to get app control close member variables
    */
    virtual tResult GetAppControlClose(tMountPoint &deviceID, tAppName &appName, tSessionID &sessionID) {return MP_NO_ERROR;};

    /**
    * function to get native transport start member variables
    */
    virtual tResult GetNativeTransportStart(tDeviceID &deviceId, tU8 &iAP2iOSAppIdentifier, tU8 &sinkEndpoint, tU8 &sourceEndpoint ) {return MP_NO_ERROR;};

    /**
    * function to get native transport stop member variables
    */
    virtual tResult GetNativeTransportStop(tDeviceID &deviceId, tU8 &iAP2iOSAppIdentifier, tU8 &sinkEndpoint, tU8 &sourceEndpoint ) {return MP_NO_ERROR;};

    /**
     * Function checks if a internal application is registered to a devices focus application.
     * If yes: it changes the output channel at the av management. enableStreaming will return true
     * if the av channel was not changed.
     *
     * dependencies: AVManager.cpp
     *
     * @param[in] deviceID the device id that changed the focus app
     * @param[in] appName the name of the focus app to search in the internal app table for
     * @return != 0: error, = 0: OK
     */
    virtual tResult OnFocusAppChanges(const tDeviceID deviceID, const tAppName appName); //Roadmap 13008

    /**
     * This function is called by MediaPlayer when the result of DiPOActiveDevice is received
     * Can be overwritte by derived class
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOActiveDeviceAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of DiPORoleSwitchRequest is received
     * Can be overwritte by derived class
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPORoleSwitchRequestAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of DiPORoleSwitchRequest is received
     * Can be overwritte by derived class
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOTransferGPSDataAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext) {return MP_NO_ERROR;};

    /**
     * This function is called by MediaPlayer when the result of DiPOTransferDRData is received
     * Can be overwritte by derived class
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOTransferDRDataAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext) {return MP_NO_ERROR;};


    /**
       * This function is used to send from Mediaplayer Interface to cllientHandler_DeviceManager command eject
       *
       * @param[in] enOpticalDiscEjectCmd specify if command is eject or insert
       * @return != 0: error, = 0: OK
       */
    virtual tResult SendEjectOpticalDisc(IN tDMOpticalDiscEjectCmd enOpticalDiscEjectCmd){return MP_NO_ERROR;}; //Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * This function is used to send asynchronous response for eject command to output wrapper.
     *
     * @param[in] entOpticalDiscEjectMethodResult if command was successfull or not see definition of tDMOpticalDiscEjectMethodResult
     * @return != 0: error, = 0: OK
     */
    virtual tResult SendOpticalDiscEjectMethodResult(IN tDMOpticalDiscEjectMethodResult entOpticalDiscEjectMethodResult){return MP_NO_ERROR;};//Roadmap 15002_CDAudio_DetectionAndDiagnosis


    /**
     * Function is called by interface to set the current active source ID.
     * This information is used to check if a source change is necessary if an external device changes the audio application
     * E.g. if the current active source is Tuner (id x) and the user activates mediaplayer on a connected apple device, the AvManager of the
     * system shall be asked to switch to mediaplayer (behavior can be configured in DataProvider)
     *
     * @param[in] sourceNr current active source
     * @return != 0: error, = 0: OK
     */
    virtual tResult SetCurrentSource(const tSource source); //Roadmap 13008
    virtual tResult SetCurrentSourceActive(const tBool active);

    /**
     * This function requests a mute state at the AVManager of the system.
     * Can be overwritten by the derived class.
     *
     * @param[in] muteState requested mute state (DEMUTED or MUTED)
     * @return != 0: error, = 0: OK
     */
    virtual tResult RequestMuteState(const tMuteState muteState , const tRampType rampType = SLOW_RAMP);//Roadmap 13018

    /**
     * This function requests a volume change at the AVManager of the system.
     * Can be overwritten by the derived class.
     *
     * @param[in] volumeState requested volume change to NORMAL or REDUCED
     * @return != 0: error, = 0: OK
     */
    virtual tResult RequestVolumeChange(const tVolumeState volumeState) {return MP_NO_ERROR;}; //Roadmap 13018

    /**
     * Roadmap 13031-BT_DeviceSourceAllocation
     * This function returns the ID of the FBlock which is currently using the LogicalAVChannel:LC_MAIN_AUDIO
     * @return != 0: error, = 0: OK
     */
    virtual tResult GetCurrentSource(tSource &source);

    /**
     * This function is called when the scan mode of a device changed.
     * This function will call the function eUpdateClients of CCA framework
     * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_SCANMODE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateScanMode(){return MP_NO_ERROR;}; //Roadmap 15008_SCAN
    virtual tResult SetAudioAllocatedDevice(const tDeviceID deviceID); //Fix for CAF-1819,PSA-5691...

    /**
     * This function is used to notify optical disc CD Info towards HMI. I.e. VD_DeviceManager receives it from VD_MediaManager and forwards it though output wrapper to HMI
     *
     * @return != 0: error, = 0: OK
     */
     virtual tResult UpdateCDInfo(){return MP_NO_ERROR;}; //CD Info



     /** //Roadmap 16014 'full text search'
     *  This function calls method return first of fi method 'CreateMediaListWithSearchString' also referred as fc_mediaplayer_tclService_MediaPlayer::onCreateMediaPlayerListSearchString'
     *  it is first return
     *  @param [IN] listID is the ID of the created earch string list
     *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
     *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
     *
     *  @return < 0: error, = 0: OK
     */
     virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnFirst(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID){return MP_NO_ERROR;};

     /** //Roadmap 16014 'full text search'
     *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. And first answer i.e. 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
     *  has been send already. This method is called once or more. I.e. it is the second, third ... answer
     *  @param [IN] listID is the ID of the created earch string list
     *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
     *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
     *
     *  @return < 0: error, = 0: OK
     */
     virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle(IN tListID listIDParent, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID){return MP_NO_ERROR;};

     /** //Roadmap 16014 'full text search'
     *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. It is called after once 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
     * and then after once or more times 'Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle' has been called.
     * It is the last method return expected to be called after once 'CreateMediaListWithSearchString'  has been called.
     *
     *  @param [IN] listID is the ID of the created earch string list
     *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
     *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
     *
     *  @return < 0: error, = 0: OK
     */
     virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnLast(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID){return MP_NO_ERROR;};


     /** //Roadmap 16014 'full text search'
      *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. It is called after once 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
      * and then after once or more times 'Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle' has been called.
      * It is the last method return expected to be called after once 'CreateMediaListWithSearchString'  IF THE SEARCH HAS NOT BEEN FINISHED because it has been interrupted e.g. by the next search (user has typed new character for search).
      *
      *  @param [IN] listID is the ID of the created earch string list
      *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
      *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
      *
      *  @return < 0: error, = 0: OK
      */
      virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnAbort(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID){return MP_NO_ERROR;};


     /**
     * This function is called when there is an error sent by apple device for accessory commands.
     * This function will call the function eUpdateClients of CCA framework.
     * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_IPODCOMMUNICATIONERROR, the framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateIpodCommunicationError(const tDeviceID deviceID, tIpodCommError iPodCommError){return MP_NO_ERROR;}; //CMG3G-8083

     /**
     * function to get iPod Communication Error  member variables
     */
     virtual tResult GetIpodCommunicationError(tDeviceID &deviceID, tIpodCommError &iPodCommError){return MP_NO_ERROR;};

          /**
     * This function is called when there is an error sent by apple device for accessory commands for carplay communication.
     * This function will call the function eUpdateClients of CCA framework.
     * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_DIPOCOMMUNICATIONERROR, the framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDipoCommunicationError(const tDeviceID deviceID, tIpodCommError iPodCommError, const tMountPoint USBSerialNumber){return MP_NO_ERROR;}; //CMG3G-8083

     /**
     * function to get iPod Communication Error  member variables
     */
     virtual tResult GetDipoCommunicationError(tDeviceID &deviceID, tIpodCommError &iPodCommError, tMountPoint &USBSerialNumber){return MP_NO_ERROR;};

      /**
      * This function is called when the video property of video object is changed.
      * This function will call the function eUpdateClients of CCA framework
      * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_VIDEOBRIGHTNESS/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOHUE/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOSATURATION/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOCONTRAST/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOBRIGHTNESSOFFSET/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOHUEOFFSET depend upon the video property changed. The framework then notifies all registerd clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateVideoProperty(const tVideoProperty videoProperty){return MP_NO_ERROR;}; //Roadmap 15027_Adjust Video Brightness

    /**
     * This function is called by the mediaplayer core when PowerUpdate information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPOWER. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateDiPOPower() {return MP_NO_ERROR;};

       /**
       * This function is called by the mediaplayer core when ripping information has changed .
       * @return < 0: error, = 0: OK
       */
       virtual tResult UpdateRippingStatus(){return MP_NO_ERROR;};
       virtual tResult UpdateAutoRipping(){return MP_NO_ERROR;};
       /**
        * This function is called by the mediaplayer core when there is a ripping error .
        * @return < 0: error, = 0: OK
        */
       virtual tResult UpdateRippingErrors(){return MP_NO_ERROR;};
       /**
        * This function is called by the mediaplayer core when gracenote lookup mode has changed .
        * @return < 0: error, = 0: OK
        */
       virtual tResult UpdateGracenoteLookupMode(){return MP_NO_ERROR;};
       /**
       * This function is called by the mediaplayer core when ripping encoding quality has changed .
       * @return < 0: error, = 0: OK
       */
       virtual tResult UpdateRippingEncodingQuality(){return MP_NO_ERROR;};

       /**
        * This function is called by the mediaplayer core when there is gracenote metadata for the particular album .
        * @return < 0: error, = 0: OK
        */

       virtual tResult UpdateGNdataforAlbumOnUserDemand(){return MP_NO_ERROR;};


       /**
        * This function is called when the dvd property of video object is changed.
        * This function will call the function eUpdateClients of CCA framework
        * @return < 0: error, = 0: OK
        */
        virtual tResult UpdateDVDProperty(const tDvdProperty dvdProperty){return MP_NO_ERROR;}

        /**
         * This function is called to get the Gracenote version.
         * @return < 0: error, = 0: OK
         */
        virtual tResult UpdateGracenoteVersion(){return MP_NO_ERROR;};

        virtual tResult UpdateGraceNoteResolveAmbiguity(){return MP_NO_ERROR;};
        /**
         * This function is called when there is an ambiguity in the gracenote.
         * This function will call the function ResolveGracenoteAmbuiguites of Mediaplayer
         * @return < 0: error, = 0: OK
         */

        //>--Roadmap 16003 : 'CD Ripping With Gracenote'

    /**
     * This function is called when the delete media content operation is completed.
     * This function calls method return after fi method 'DeleteMediaContentByMediaObjectUrls' or 'DeleteMediaContentByMediaObjectTags' or  'DeleteMediaContentByFilterTags' is called
     * @param [IN] deleteMediaContentType is type of 'deletemediacontent' interface used
     * @param [IN] fileOperationErrorCodes is type of 'deletemediacontent' error code
     * @param [IN] numofFilesDeleted is number of files got deleted
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_OnDeleteMediaContentMethodResult(IN tDeleteMediaContentType deleteMediaContentType, IN tFileOperationErrorCodes fileOperationErrorCodes,IN tU16 numofFilesDeleted ){return MP_NO_ERROR;}

    /**
     * This function is called when the edit media metadata operation is completed.
     * This function calls method return after fi method 'EditMetaDataByMediaObjectUrl' or 'EditMetaDataByMediaObjectTag'  is called
     * @param [IN] editMetadatatType is type of metadata interface used
     * @param [IN] fileOperationErrorCodes is type of editmetadata result code
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_OnEditMetadataMethodResult(IN tEditMetadatatType editMetadatatType, IN tFileOperationErrorCodes fileOperationErrorCodes ){return MP_NO_ERROR;}

    /**
     * This function is called when the add entries into playlist  operation is completed.
     * This function calls method return after fi method 'AddFilenameToInternalPlaylist'  is called
     * @param [IN] playListOperationErrorCodes is type of playlist operation result code
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_OnAddFileNamesToPlaylistMethodResult(IN tResult playListOperationErrorCodes ){return MP_NO_ERROR;}

    /**
     * This function is called when the delete entries into playlist  operation is completed.
     * This function calls method return after fi method 'DeleteEntriesFromPlaylist'  is called
     * @param [IN] playListOperationErrorCodes is type of playlist operation result code
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_OnDeleteEntriesFromPlaylistMethodResult(IN tResult playListOperationErrorCodes ){return MP_NO_ERROR;}

    /**
     * The below functions(StartFileOperation,isFileOperationCompleted) are helpers for the unit test cases. This will help us to identify an operation done on a thread function completed or not
     */
    virtual void StartFileOperation(){};
    virtual bool isFileOperationCompleted(tResult &res){return MP_NO_ERROR;};



        /**
     * AIVI-50524 Update streaming info to HMI
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateStreamingInfo() {return MP_NO_ERROR;};
    //>--Roadmap CMG3G-10221 : 'Scene Recorder '
     /**
     * This function is called when the edit media metadata operation is completed.
     * This function calls method return after fi method 'Send_SetfilePermissionByMediaobjectTags'
     * @param [IN] fileOperationErrorCodes is type of set file permission result code
     * @param [IN] numofFiles is number of file permision set
     * @return < 0: error, = 0: OK
     */
    virtual tResult Send_SetfilePermissionByMediaobjectTagsMethodResult(IN tResult fileOperationErrorCodes, IN tU16 numofFiles){return MP_NO_ERROR;}
    //<--Roadmap CMG3G-10221 : 'Scene Recorder '
    /**
     * This function is called by the mediaplayer core when ROUTEGUIDANCE information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateDiPORouteGuidance() {return MP_NO_ERROR;};
    /**
     * This function is called by the mediaplayer core when ROUTEGUIDANCEMANEUVER information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateDiPORouteGuidanceManeuver() {return MP_NO_ERROR;};


     /**
     * This function is called to update Videomanager for DVD Source activation status
     */
     virtual tResult UpdateDVDVideoSourceStatus(const bool) {return MP_NO_ERROR;};
     /**
     * This function is called to acknowledge Videomanager for DVD Source request
     */
     virtual tResult vSendVideoStateAcknowledgement(const tVideoSourceState){return MP_NO_ERROR;};
     /**
     * This function is called to send method result for RequestDVDTemperature
     */

     virtual tResult SendRequestDVDTemperatureAnswer(tU16 temperature , tUserContext userContext){return MP_NO_ERROR;};

     virtual tResult UpdatePlaybackSpeed(){return MP_NO_ERROR;};

     virtual void UpdateVolumeToAudio(IN int Volume){};

     virtual void UpdateFolderBrowseStructureCleared(){};

     virtual void GetCurrentVolumeFromAudio(){};

     /**
      * This function is called by the mediaplayer core when DevieTime information has changed .
      * The function calls eUpdateClients with parameter
      * MPLAY_APPCONTROLFI_C_U16_DIPODEVICETIMEUPDATE
      * The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
      virtual tResult UpdateDiPODeviceTime() {return MP_NO_ERROR;};

     /**
     * Updating that the device has updated that it is in playing and Mediaplayer is ready to Play due to remote acivity
     */
     virtual void UpdateReadyToPlay() {};

     /**
     * Get onDeviceNotifiyConnection CCA property from DeviceManager
     */
     virtual void RequestDeviceManagerNotifyConnection() {};

protected:

    tSource currentSource;
    tBool currentSourceActive;
    tDeviceID audioAllocatedDevice; //Fix for CAF-1819,PSA-5691...

};

#endif //_OUTPUTWRAPPER_H_

/** @} */
