/**
 * @addtogroup LocalSPM
 * 
 * @author Matthias Thömel , Dinesh D
 *
 * Public Interface for all Local SPM Clients
 * @{
 */

#ifndef _ILOCALSPM_H_
#define _ILOCALSPM_H_

#include "Timer.h"
#include "TypeDefinitions.h"
#include <string.h>

/** 
 * Class ILocalSPM defines the LocalSPM interface which all the components are expected to implement
 */
class ILocalSPM : public Timer
{

private:

    tComponentID    mComponentID;       /**< component ID assigned by the LocalSPM 		*/
    tComponentState mComponentState;    /**< current Local SPM state of the component   */

public:

    ILocalSPM(tComponentID componentID):mComponentID(componentID),mComponentState(COMPONENT_STATE_UNKNOWN){};

    tComponentState GetState(void)	    {	return mComponentState;	};
    tComponentID    GetComponentId(void){	return mComponentID   ; };
    const char* GetStateName(void) const;
    const char* GetComponentName(void) const;

    void SetState(tComponentState componentState );

    /** 
     * allocate memory, create components, call constructors, create threads but don't let them run
     * no access to other components allowed
     * 
     * @return < 0: error, = 0: OK
     */
    virtual void Create() { CreateDone(0);}
	
    /** 
     * initialize variables, open databases creates queues, don't use foreign components
     *  
     * @return < 0: error, = 0: OK
     */
    virtual int Init(tInitReason reason) {return InitDone(0);}
	
    /** 
     * start threads, start state machines, other components are now usable
     * 
     * @return < 0: error, = 0: OK
     */
    virtual int Run() {return RunDone(0);};
	
    /** 
     * stop state machines, store last mode , stop the running threads
     * 
     * @return < 0: error, = 0: OK
     */
    virtual int Stop() {return StopDone(0);}
	
    /** 
     *  delete message queues, close databases, don't access foreign components
     * 
     * @return < 0: error, = 0: OK
     */
    virtual int Done() {return DoneDone(0);}

    /** 
     * Inform Start State machine with create answer (uses Dispatcher interface)
     * 
     * @return < 0: error, = 0: OKthe name of the curent state the state machine is in or "" if no state machine class
     */
    int CreateDone(int error);

    /** 
     * Inform Start State machine with Init answer (uses Dispatcher interface internally)
     * 
     * @return < 0: error, = 0: OK
     */
    int InitDone(int error);

    /**
     * Answer from the component when it reaches the state RUNNING
     *
     * @return < 0: error, = 0: OK
     */
    int RunDone(int error);
	
    /** 
     * Inform Stop State machine with Stop answer (uses Dispatcher interface internally)
     * 
     * @return < 0: error, = 0: OK
     */
    int StopDone(int error);
	
    /** 
     * Inform Stop StateMachine with Done answer (uses Dispatcher interface internally)
     * 
     * @return < 0: error, = 0: OK
     */
    int DoneDone(int error);

    /**
     * return a statistic string dependent on the component
     *
     * @return < 0: error, = 0: OK
     */
    virtual int Statistics(OUT tStatistics stat) {stat[0]=0; return 0;};

    /**
     * return the name of the current state the state machine is in or "" if no state machine class
     *
     * @return < 0: error, = 0: OK
     */
    virtual char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
     * return answer if the component is a state machine
     *
     * @return true or false
     */
    virtual tBoolean IsComponentSM();

};

#endif	 /*_ILOCALSPM_H_*/

/** @} */
