/**
 * @defgroup ListControl ListControl
 * @author Stefan Scherber
 * @author Matthias Thoemel
 * 
 * Public interface for list control used by components which must retrieve HMI or player lists 
 * @{
 */

#ifndef _LISTCONTROL_H_
#define _LISTCONTROL_H_

#include <iostream>
#include "ThreadFactory.h" //Radded for Roadmap 160014
#include "MessageQueue.h"  //added for Roadmap 160014
#include "ILocalSPM.h"
#include "Lock.h"
#include "TypeDefinitions.h"

using namespace std;





class ListControl : public ILocalSPM, public TFThread //Roadmap 160014 'full text search needs a thread to continue parallel search after first hit'
{

public:    

    //added for Roadmap 160014 ''full text search'
    enum{
        eThreadFct_FullTextSearch       = 0,
        eThreadFct_FullTextSearchNotify = 1,
        eThreadFctSize                  = 2
    };



    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the ListControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    ListControl(tComponentID componentID):ILocalSPM(componentID){} ;

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~ListControl(){};


    /**
       * thread function added first for //Roadmap 160014 'full text search'
       *
       * @return void
       */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to cleanup the ListControl.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * This function is used by LocalSPM to stop the ListControl.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * This function is used by LocalSPM to cleanup the ListControl.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Brief: returns statistic data as a string for the over all media player object statitcs
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    int Statistics(tStatistics stat);

    /**
     * releases a list (regardless of PlayList or IndexedList)
     *
     * calls: DBManager::ReleaseList(listID) 
     * removes the browse hierarchie for this list with DBManager
     * 
     * @param[in] listID ID which list has to be released
     * @param[in] force release list even list reusing strategy is active
     *
     * @return < 0: error, = 0: OK
     */
    tResult ReleaseList(const tListID listID, const tBoolean force = false);
    tResult ReleaseAllLists();

    /**
     * Informs HMI which object is currently playing (in which list)
     *
     * Sends a MediaPlayerListChange(listID, LCH_CONTENT_CHANGED, listSize, 0, NULL) via OutputWrapper
     *
     * The now playing info has to be stored in member vars in order to compare these informations by
     * RequestMediaPlayerIndexedListSlice.
     *
     * @param[in] listID ID for which the object is playing
     * @param[in] objectID ID of played object
     *
     * @param[out] position position of the played object
     * @return < 0: error, = 0: OK
     */
    tResult UpdateNowPlaying(tPosition &position, const tListID listID, const tObjectID objectID);

    /**
     * Informs HMI that the elements of a list slice has changed
     * Roadmap 15009
     * MediaPlayerListChange(listID, LCH_ITEMS_CHANGED, listSize, changedItems) via OutputWrapper
     *
     * @param[in] listID list ID to update list slice
     * @param[in] startIndex begin of the slice in the list
     * @param[in] sliceSize size of the slice
     * @return < 0: error, = 0: OK
     */
    tResult UpdateListSlice(const tListID listID, const tIndex startIndex, const tIndex sliceSize); //Roadmap 15009: 100%

    /**
     * Internal function which checks all indexed list size against current DB contents
     * and sends than: MediaPlayerListChange(listID, LCH_CONTENT_CHANGED, listSize, NULL) via OutputWrapper for each list
     *
     * @param[in] deviceID ID of the affected device to be checked
     * @return < 0: error, = 0: OK
     */
    tResult CheckLists(const tDeviceID deviceID);

    /**
     * Internal function which updates the object highlighting in all existing file lists
     * and sends than: MediaPlayerListChange(listID, LCH_CONTENT_CHANGED, listSize, NULL) via OutputWrapper for each list
     *
     * @param[in] deviceID ID of the affected device to be checked
     * @return < 0: error, = 0: OK
     */
    tResult CheckFileLists(const tDeviceID deviceID);

    /** 
     * Function creates a playable list for an object and list type combination. 
     * Object can be an id for a genre, artist, album playlist, book, .. or song.
     * in case of a id for a song the list type decides if t is an all songs list
     * or only one song. If it is an all songs list the current position in this 
     * list will be set to the given object id. 
     * 
     * @param[in] objectID id which is used to specify the filter for a given list type
     * @param[in] objectType object type decides which kind of list is to be created.
     *
     * @param[out] listID id of the list which was created
     * @return < 0: error, = 0: OK
     */
    tResult CreatePlayList(tListID &listID,
        const tObjectID objectID,
        const tCategoryType objectType,
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID);

    /** 
     * Function creates a playable list for an list type and filter tag combination. 
     * uses : DB manager: CreateList
     *        DB manager: GetListInfo     
     * 
     * @param[in] listType list type to create play list filtered after object ID 
     *                     see special behavior described above for all songs list.
     * @param[in] streaming streaming flag //Roadmap 13008
     * @param[in] tag1,tag2,tag3,tag4  four filter tag object id's to generate the complete list
     *                     which ID is in which tag position depends on the list type.
     *                     max 4 different tags are possible
     * @param[in] playContinuation decides what to do on end of list (@ref tPlayContinuation)
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @param[in] checkVTIPOD check if the new list is a VTIPOD list.
     *                     If this flag is false the new list gets automatically the attribute mIsVTIPODList = false
     * @param[out] listID id of the list which was created
     * @return < 0: error, = 0: OK
     */
    tResult CreatePlayList(tListID &listID, //Roadmap 13008: 100%
        const tListType listType,
        const tStreaming streaming,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID,
        const tBoolean checkVTIPOD = true,
        const char *pSearchText   = NULL,
        const tUserID userID = 0);        //Roadmap 17001 Personalization:

    /** 
     * Function creates a playable list for a list type and filter tag combination. 
     * uses : DB manager: CreateList(&listID,listType,filterTags[0..3],deviceID,playCont=NO_Repeat);
     *        DB manager: GetListInfo
     * 
     * @param[in] listType list type to create play list filtered after object ID
     *                     see special behavior described above for all songs list.
     * @param[in] streaming streaming flag //Roadmap 13008
     * @param[in] tag1,tag2,tag3,tag4     three filter tag object id's to generate the complete list
     *                     which ID is in which tag position depends on the list type.
     *                     ATTENTION: GenMP uses a new tag definition which is different to the GM tag definition
     *                     use MapTagsFromGM2GenMP(...) to convert from GM to GenMP tag positions
     * @param[in] deviceID id of the device for special filtered list depending on a device
     *
     * @param[out] listID id of the list which was created
     * @param[out] listSize length of list
     * @return < 0: error, = 0: OK
     */
    tResult CreateMediaPlayerIndexedList(tListID &listID, //Roadmap 13008: 100%
        tListSize &listSize,
        const tListType listType,
        const tStreaming streaming,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tDeviceID deviceID,
        const tUserID userID = 0);



    /** 
     * Roadmap 160014 'Full text search'
     * 
     *
     */
    tResult CreateMediaPlayerIndexedListWithSearchString(OUT tListID   &listID, //Roadmap 160014: 100%
                                                         OUT tListSize         &listSize,
                                                         OUT tDeviceID         &deviceID,
                                                         IN const tListID       listIDParent,
                                                         IN const tListID       listIDReplacable,
                                                         IN const tSearchString searchString,
                                                         IN tSearchIncrement    searchIncrement = tSearchIncrement_init,
                                                         IN tDelaySec      testDelaySec = tDelaySec_noDelay);

    
    

    
    /** 
     * Function creates a playable list for a path to a playlist. 
     * uses : DB manager: CreateList with a special list type identifying the virtual playlist table module
     *        DB manager: GetListInfo    
     * 
     * @param[in] fileName filename to a playlist on a usb stick for example
     *
     * @param[out] listID id of the list which was created
     * @param[out] listSize length of list
     * @return < 0: error, = 0: OK
     */    
    tResult CreateMediaPlayerPlaylistList(tListID &listID, tListSize &listSize, const tFilename fileName);
    tResult CreateMediaPlayerPlaylistList(tListID &listID, tListSize &listSize, const tFilename fileName, tDeviceID deviceID);

    /** 
     * Function returns a part of a previously created indexed list
     *
     *  why that?: DBManager::GetNowPlaying(&nowPlaying) gets the current media object and the current list
     *    why that?: DBManager::GetListInfo(nowPlaying.listID) to get the type of the currently played list
     *    DBManager::GetListInfo(listID) to get the type of the currently played list
     *  determine the category type from the list type:
     *    rule: The last noun in LTY is the key to cat type:
     *      LTY_GENRE_ARTIST -> CTY_ARTIST
     *  if song list:
     *     for loop n over startIndex count=sliceSize
     *     DBManager::GetMediaObject(&mediaObject, listID, n)  (could be media, album, artist, genre ...)
     *     compare filter tags and list-type  or song id to tag a media object as Playing or genre, album, artist
     *       this behavior should be switchable by DataProvider config switches. For example: One customer 
     *       wants to see always a highlight on a played item even if it is played by another list type.
     *   endwhile
     *
     *  if category list (eg: CTY_ALBUM)
     *   for loop n over startIndex count=sliceSize
     *     DBManager::GetAlbum(&albumTag,&albumName,listID, n)
     *     DBManager::GetArtistOfAlbum(&artistTag,&artistName,albumTag);
     *     DBManager::GetGenreOfAlbum(&genreTag,&genreName,lbumTag);
     *     create media object
     *     compare filter tags and list-type or song id to tag as media object as playing or genre, album, artist, rest see above
     *    endwhile
     * 
     * @param[in] listID list ID for which the slice is requested
     * @param[in] startIndex begin of the slice in the list
     * @param[in] sliceSize size of the slice
     *
     * @param[out] mediaObjectVector vector with found objects for list slice, the vector must be allocated by caller.
     * @return < 0: error, = 0: OK
     */    
    tResult RequestMediaPlayerIndexedListSlice(vector<tMediaObject> &mediaObjectVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize);
    tResult RequestMediaPlayerIndexedUUIDListSlice(tBatchPlaybackList *batchPlaybackListPtr,
            const tListID listID,
            const tIndex startIndex,
            const tIndex sliceSize);

    /** 
     * Function creates a playable list for tag 
     *
     * calles DBManager::CreatePlayList
      *
     * @param[in] tag object id (artist, album, ...) to find a or generate corresponding list 
     *                type for a playable list
     * @param[in] playContinuation decides what to do on end of list (@ref tPlayContinuation)
     *
     * @param[out] listID id of the list which was created
     * @return < 0: error, = 0: OK
     */
    tResult CreatePlayList(tListID &listID, const tFilterTag tag, const tPlayContinuation playContinuation, const tDeviceID deviceID);

    /**
     * Function returns current media object from a playlist. This object is stored in the 
     * list id object and will be set with Next/Prev/SetMediaObject and is not filled 
     * by an access to the DB.
     *
     * special: The album art string in the mdiaObject will be filled by using the device ID and the
     *          media object path -> "deviceID:PathToMediaObjectOnStorrage"
     *
     * file virtual table: if the file VT is the source of a media object, the album, composer, genre tag
     *                     value are "NULL"
     *
     * This is valid for all media object delivering methods
     * 
     * @param[in] listID id of the list from which the current object has to be calculated
     *
     * @param[out] mediaObject the current media object in list
     * @return < 0: error, = 0: OK
     */
    tResult GetCurrentMediaObject(tMediaObject &mediaObject, const tListID listID);

    /**
     * Function sets the current active media object of a list via objectID
     * Roadmap 13017
     *
     * @param[in] listID id of the list to update the current active media object
     * @param[in] objectID id of the object in list
     *
     * @return < 0: error, = 0: OK
     */
    tResult SetCurrentMediaObject(const tListID listID, const tObjectID objectID); //Roadmap 13017: 100%

    /** 
     * Function resets the list position to current active media object
     *
     * @param[in] listID id of the list to update the current active media object
     *
     * @return < 0: error, = 0: OK
     */
    tResult ResetListPosition(const tListID listID);

    /** 
     * Function sets the current media object of a list.
     * AND sets the active device in the database. The device id will be read out of the list id.
     *
     * for use in PlayerManager: if this functions returns an error the user has to send
     * send a NoObject message to the PlayerManager state machine to cover that a list or
     * object is not valid anymore. 
     * 
     * @param[in] listID id of the list from which the current object has to be calculated
     * @param[in] startIndex position of media object in list that has to be set as new media object.
     *            if INDEX_LAST_ACTIVE_OBJECT than no object will be set only the media object
     *            will be filled with current object active in list.
     *
     * @param[out] mediaObject the current media object in list
     * @return = 0: OK < 0: error
     */
    tResult SetMediaObject(tMediaObject &mediaObject, const tListID listID, const tIndex startIndex);
    
    /** 
     * Function checks if a media object or list is valid
     *
     * This is used in the CCA service to get the information if a desired media object
     * from a list is valid and could be played
     * 
     * @param[in] listID id if the list where the object should be searched.
     * @param[in] rowNumber number of row to be set as current
     * @param[in] playtime play position to be set in media object
     *
     * @param[out] mediaObjectInfo contains the status of a media object (@ref tMediaObjectInfo)
     * @return < 0: error, = 0: OK
     */
    tResult CheckMediaObject(tMediaObjectInfo &mediaObjectInfo, const tListID listID, const tIndex rowNumber, const tPlaytime playtime);

    /** 
     * Function returns details about a created playable / hmi list 
     * 
     * @param[in] listID id of the list from which the information will be delivered
     * @param[in] withListSize switch if list size in listInfo should be filled or not (because it's time expensive)
     *
     * @param[out] listInfo pointer to struct where the list info will be stored in 
     *                      (@ref tListInfo)
     * @return < 0: error, = 0: OK
     */
    tResult GetListInfo(tListInfo &listInfo, const tListID listID, const tBoolean withListSize);
    tResult GetPath(tPath &path, const tListID listID);
    
    /** 
     * Function sets the next media object in a playlist. 
     * With (@ref GetNextMediaObject) one can read the media object data
     *
     * Depending on PlayContinuation (@ref tPlayContinuation) (@ref DataProvider) or parameter 
     * to CreatePlayList the current list will be repeated or the next sibling (step up/step down)
     * will be played.
     *
     * Only playable object will be set as next media object. No playlists and no directories
     * Objects which are marked as NotPlayable will be skipped //Roadmap 13012
     * If in a list are only NotPlayable objects function delivers MP_ERR_LC_LIST_IS_NOT_PLAYABLE
     * 
     * @param[inout] listID id of the list from which the next object has to be calculated
     * @param[in] stepCount number of steps for next (default = 1 for step to next song)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Next(tListID &listID, const tNextPrevSkipCount stepCount, const tHMIRequest isHMIRequest); //Roadmap 13012
    tResult NextInternal(tListID &listID, const int stepCount);

    /** 
     * Function sets the previous media object in a playlist. 
     * With (@ref GetNextMediaObject) one can read the media object data. The playtime is 
     * set to 0 in this object.
     *
     * Depending on PlayContinuation (@ref tPlayContinuation) (@ref DataProvider) or parameter 
     * to CreatePlayList the current list will be repeated or the next sibling (step up/step down)
     * will be played.
     * 
     * only playable object will be set as next media object. No playlists and no directories
     * 
     * @param[inout] listID id of the list from which the previous object has to be calculated
     * @param[in] stepCount number of steps for prev (default = 1 for step to prev song)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Prev(tListID &listID, const tNextPrevSkipCount stepCount, const tHMIRequest isHMIRequest);

    /** 
     * Function stores the current media object of a list as last mode object with playtime
     * In case of MyMedia the list can be changed in the meantime
     * - Set position in DB list via DBManager -> SetPosition(&position, listPtr->nowPlayingObjectID, listPtr->dbBrowseListID)
     * Store the last mode of the list at the DBManager if any object is active
     * - StoreLastPlayed(listPtr->dbBrowseListID, listPtr->nowPlayingObjectID, listPtr->nowPlayingListPosition, playtime);
     *
     * @param[in] listID id of the list from which the previous object has to be calculated
     * @param[in] playtime current play position of current media object played
     * @return < 0: error, = 0: OK
     */
    tResult StoreLastPlayedList(const tListID listID, const tPlaytime playtime);

    /** 
     * Function returns the last saved list for a device ID
     * 
     * @param[in] deviceID id for which device the list has to be fetched.
     * @param[out] listID id of the list last played for the device
     * @param[out] startPosition position of last object in the list
     * @param[out] playtime last played position of the current media object in list
     * @return < 0: error, = 0: OK
     */
    tResult GetLastPlayedList(tListID &listID, tIndex &startPosition, tPlaytime &playtime, const tDeviceID deviceID);

    /**
     * Function returns a search keyboard list
     * 
     * @param[in] listID id of the list for which the keyboard should be created
     * @param[out] searchKeyboardList a vector with all letters for the keyboard
     *
     * @return < 0: error, = 0: OK
     */
    tResult SearchKeyboardMediaPlayerList(vector<tSearchKeyboard> &searchKeyboardList, const tListID listID);

    /** 
     * Function returns a media object information based on given listID and  position
     *  @param[in] listID - list from which mediaobject is requested
     *  @param[in] position - position of mediaobject requested
     *
     *  @param[out] mediaObject filled object with all media object data
     *  @return != 0: error, = 0: OK
     */
      tResult GetMediaObject(tMediaObject &mediaObject, const tListID listID, const tPosition position);

    /** 
     * Function creates a list out of files and folders of an USB stick
     *
     * calls DBManager::CreateMediaPlayerFileList(...) accordingly
     * the listSize is numberOfMediaObjects+numberOfPlayLists+numberOfFolders
     * 
     * @param[in] listType type of list (LTY_FILELIST or LTY_FILELIST_UNSORTED)
     * @param[in] path path in which the list should be created
     * @param[in] deviceID id of the desired USB stick
     * @param[in] streaming streaming flag //Roadmap 13008
     * @param[in] fileTypeSelection selection criteria (e.g. only audio files)
     * @param[out] listID id of the created list
     * @param[out] listSize length of the list
     * @param[out] numberOfMediaObjects number of songs in the list
     * @param[out] numberOfPlayLists number of playlists in the list
     * @param[out] numberOfFolders number of sub folders in the list
     * @return < 0: error, = 0: OK
     */
    tResult CreateMediaPlayerFileList(tListID &listID, tListSize &listSize, tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists, tNumberOfFolders &numberOfFolders, tListType listType, const tPath path,
        const tDeviceID deviceID, const tStreaming streaming = false, const tFileTypeSelection fileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);

    /**
     * Function creates a list out of files and folders of an USB stick
     * Roadmap 13017
     * calls DBManager::CreateMediaPlayerFileList(...) accordingly
     * the listSize is numberOfMediaObjects+numberOfPlayLists+numberOfFolders
     *
     * @param[in] path path with filename in which the list should be created
     * @param[in] deviceID id of the desired USB stick
     * @param[in] streaming streaming flag //Roadmap 13008
     * @param[in] fileTypeSelection selection criteria (e.g. only audio files)
     * @param[out] listID id of the created list
     * @param[out] listSize length of the list
     * @param[out] numberOfMediaObjects number of songs in the list
     * @param[out] numberOfPlayLists number of playlists in the list
     * @param[out] numberOfFolders number of sub folders in the list
     * @param[out] position position of the file in the list
     * @return < 0: error, = 0: OK
     */
    tResult CreateMediaPlayerFileList(tListID &listID, tListSize &listSize, tNumberOfMediaObjects &numberOfMediaObjects, //Roadmap 13017: 100%
        tNumberOfPlayLists &numberOfPlayLists, tNumberOfFolders &numberOfFolders, tPosition &position, const tPath path,
        const tDeviceID deviceID, const tStreaming streaming = false, const tFileTypeSelection fileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);

    /**
     * Function creates a slice of files of a given created list (see CreateMediaPlayerFileList)
     *
     * for loop n over startIndex count=sliceSize
       *   DBManager::GetMediaObject(&mediaObject, listID, n)
     *   create a file object and fill out all possible information
     *   also add playable attribute (Roadmap 13012)
     *   compares mediaObject.objectID with listPtr->nowPlaying, if yes: mediaObject is the current playing object
     *   it is marked in file.isPlaying
     *
     * @param[in] listID id of created list 
     * @param[in] startIndex first item in list to be returned
     * @param[in] sliceSize size of slice beginning with startIndex
     * @param[out] mediaFileVector vector of all found files
     * @return < 0: error, = 0: OK
     */
    tResult RequestMediaPlayerFileListSlice(vector<tFiles> &mediaFileVector, const tListID listID,  //Roadmap 13012: 100%
        const tIndex startIndex, const tIndex sliceSize);

    /** 
     * Function returns detailed information about a list pointed to by listID
     * is needed for back browsing
     *
     * does the following:
     *
     * DBManager::GetListInfo(&listInfo, listID)
     * DBManager::GetCurrentMediaObject(&mediaObject, listID)
     * if list type first level (like LTY_GENRE) and listInfo.parent == NULL:
     * then:
     *  read tag or object id to create list for upper level -> objectID
     *  e.g. listType is ALBUM_SONG then read ALBUM id from media object and create a list with LTY_ALBUM == objectID
     *  DBManager::CreateList(&parentListID, objectID, listTypeOfParent== LTY_GENRE, autoPlay=PC_NO_REPEAT)
     *  DBManager::SetListParent(listID, parentListID)
     *  DBManager::GetPositionInList(&position, parentListID, objectID)
     * 
     * @param[in] listID id for the list to find
     * @param[out] listType type of list
     * @param[out] parentListID id of the parent list (0 for parent)
     * @param[out] deviceID device id for which list was created (0 = myMedia)
     * @param[out] position position of this list in the parent list
     * @param[out] listSize length of the list
     * @return < 0: error, = 0: OK
     */
    tResult RequestListInformation(tListType &listType, tListID &parentListID, tDeviceID &deviceID,
        tPosition &position, tListSize &listSize, const tListID listID);
    /**

    * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
    * GMP returns the following position or if the end of list is reached it returns end of list information.
    * The search start position can be defined by client.
    *
    * @param[out] outposition is the position of the searched string
    * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
    * @param[in] listID id for the list to find the string
    * @param[in] startposition is the starting position for quicksearch(can be defined by client)
    * @param[in] searchstring is the string that is searched
    */
    tResult Quicksearch(tPosition &outposition, tSearchResult &quicksearchresult, const tListID listID=0,
            tPosition startposition=0, tSearchString searchstring = NULL);

    /**

    * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
    * GMP returns the following position or if the end of list is reached it returns end of list information.
    * The search start position can be defined by client.
    * This is exclusively used for IAP1 devices.
    *
    * @param[out] outposition is the position of the searched string
    * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
    * @param[in] listID id for the list to find the string
    * @param[in] startposition is the starting position for quicksearch(can be defined by client)
    * @param[in] searchstring is the string that is searched
    */
    tResult QuicksearchForBrowseList(tPosition &outposition, tSearchResult &quicksearchresult, const tListID listID=0,
            tPosition startposition=0, tSearchString searchstring = NULL);

    /**

    * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
    * GMP returns the following position or if the end of list is reached it returns end of list information.
    * The search start position can be defined by client.
    *
    * @param[out] outposition is the position of the searched string
    * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
    * @param[in] listID id for the list to find the string
    * @param[in] startposition is the starting position for quicksearch(can be defined by client)
    * @param[in] searchstring is the string that is searched
    */
    tResult QuicksearchForIndexedList(tPosition &outposition, tSearchResult &quicksearchresult, const tListID listID=0,
            tPosition startposition=0, tSearchString searchstring = NULL);

    /**

    * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
    * GMP returns the following position or if the end of list is reached it returns end of list information.
    * The search start position can be defined by client.
    *
    * @param[out] outposition is the position of the searched string
    * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
    * @param[in] listID id for the list to find the string
    * @param[in] startposition is the starting position for quicksearch(can be defined by client)
    * @param[in] searchstring is the string that is searched
    */
    tResult QuicksearchForFolderList(tPosition &outposition, tSearchResult &quicksearchresult, const tListID listID=0,
            tPosition startposition=0, tSearchString searchstring = NULL);

    /**

     * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
     * GMP returns the following position or if the end of list is reached it returns end of list information.
     * The search start position can be defined by client.
     *
     * @param[out] outposition is the position of the searched string
     * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
     * @param[in] listID id for the list to find the string
     * @param[in] lowerLimitIndex is the starting position for Binary Search
     * @param[in] upperLimitIndex is the ending position for Binary Search
     * @param[in] searchstring is the string that is searched
     */
     tResult BinarySearchForQuicksearch(tPosition &outposition, tSearchResult &quicksearchresult, const tListID listID=0,
             tPosition lowerLimitIndex=0, tPosition upperLimitIndex=0, tSearchString searchstring = NULL);
    /**
     * Function returns the position of an object in a list
     *
     * @param[in] listID id for the list to find
     * @param[in] objectID id of the object in list
     * @param[out] position position of this list in the parent list
     * @return < 0: error, = 0: OK
     */
     tResult GetPositionInList(tPosition &position, const tListID listID, const tObjectID objectID);

     /**
      * Function returns the position of an object in a list
      * @param[in] fileName name of the object to find
      * @param[in] listID:id of the list in which the object to be found
      * @param[out] position: position of the object in the parent list
      * @return < 0: error, = 0: OK
      */
     tResult GetPositionInList(tPosition &position, const tListID &listID, const tPath &fileName);

    /** 
     * Function sets the playback mode
     *
     * calls 
     * DBManager::GetactiveDevice(&deviceID)
     * DBManager::SetPlaybackMode(deviceID, playbackMode)
     * 
     * @param[in] playbackMode playback mode enum
     * @return < 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tPlaybackMode playbackMode);

    /**
     * Set the new playback mode for a device.
     * Roadmap 13008
     * DBManager::SetPlaybackMode(deviceID, playbackMode)
     * If the device is the current device, it calls OutputWrapper::UpdatePlaybackMode
     *
     * @param[in] deviceID id of the device that must be set
     * @param[in] playbackMode new playback mode
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tDeviceID deviceID, const tPlaybackMode playbackMode); //Roadmap 13008: 100%

    /**
     * Function returns the playback mode
     *
     * calls
     * DBManager::GetactiveDevice(&deviceId)
     * DBManager::GetPlaybackMode(&playbackMode, deviceId)
     *
     * @param[out] playbackMode playback mode enum
     * @return < 0: error, = 0: OK
     */
    tResult GetPlaybackMode(tPlaybackMode &playbackMode);

    /**
     * Function sets the repeat mode
     *
     * calls
     * DBManager::GetactiveDevice(&deviceID)
     * DBManager::SetRepeatMode(deviceID, repeatMode)
     *
     * @param[in] repeatMode repeat mode enum
     * @return < 0: error, = 0: OK
     */

    tResult SetRepeatMode(const tRepeatMode repeatMode);
    /**
     * Set the new repeat mode for a device.
     * Roadmap 13008
     * DBManager::SetRepeatMode(deviceID, repeatMode)
     * If the device is the current device, it calls OutputWrapper::UpdateRepeatMode
     *
     * @param[in] deviceID id of the device that must be set
     * @param[in] repeatMode new repeat mode
     * @return != 0: error, = 0: OK
     */
    tResult SetRepeatMode(const tDeviceID deviceID, const tRepeatMode repeatMode); //Roadmap 13008: 100%

    /**
     * Function returns the repeat mode
     *
     * calls
     * DBManager::GetactiveDevice(&deviceId)
     * DBManager::GetRepeatMode(&repeatMode, deviceId)
     *
     * @param[out] repeatMode repeat mode enum
     * @return < 0: error, = 0: OK
     */
    tResult GetRepeatMode(tRepeatMode &repeatMode);

    tResult CreatePlayableFilelist(tListID &listID, tListSize &listSize, tIndex &startPosition, const tDeviceID deviceID, const tStreaming streaming = false);

    /**
     * Function creates a list out of files and folders including sub folders of an USB stick
     * Roadmap 13017
     * calls DBManager::CreateFileListOfMediaObjects(...) accordingly
     *
     * @param[in] path path in which the list should be created
     * @param[in] deviceID ID of the desired device
     * @param[in] withSubFolders with or without subfolders
     * @param[out] listID ID of the created list
     * @return < 0: error, = 0: OK
     */
    tResult CreateFileListOfMediaObjects(tListID &listID, const tPath path, //Roadmap 13008: 100%, 13017: 100%
            const tDeviceID deviceID, const tBoolean withSubFolders);
    
    tResult GetFilelistWithObject(tListID &listID, tListSize &listSize, tNumberOfMediaObjects &numberOfMediaObjects, const tPath path,
            const tDeviceID deviceID, const tStreaming streaming = false);

    /**
     * Function creates an internal playlist 
     * R8976_UC3_LLD3.1_SEQ1
     * @param[in] deviceID device 
     * @param[in] playlistName name of the playlist
     * @param[out] playlistPath  filename of the created playlist.
     * @return != 0: error, = 0: OK
     */
    tResult CreateInternalFileNamePlaylist(tPath &playlistPath, const tDeviceID deviceTag, const tPlaylistName playlistName);
    
    /**
     * Function adds a filename to an  internal playlist 
     * R8976_UC3_LLD3.1_SEQ2
     * @param[in] playlistPath filename of the internal playlist
     * @param[in] filenamePath filename of the song
     * @param[in] position Position in playlist for this new item:
            0: in front of play list 
            -1: at the end of playlist 
            n: for position
     * @return != 0: error, = 0: OK
     */
    tResult AddFileNameToPlaylist(const tPath playlistPath, const tPath filenamePath, const tPosition position);
    
    /**
     * Function deletes an entry from an  internal playlist 
     * R8976_UC3_LLD3.1_SEQ3
     * @param[in] playlistPath filename of the internal playlist
     * @param[in] position Position in playlist 
     * @param[in]  deleteAllEntries if true all the entries will be cleared
     * @return != 0: error, = 0: OK
     */
    tResult DeleteEntryFromPlaylist(const tPath playlistPath, const tPosition position,const bool deleteAllEntries = false);

    /**
     * Function renames an internal playlist 
     * R8976_UC3_LLD3.3_SEQ4
     * @param[in] newPlaylistName filename of the internal playlist to be changed
     * @param[in] deviceID device
     * @param[in] newPlaylistName New name of the playlist 
     * @return != 0: error, = 0: OK
     */
    tResult RenameInternalPlaylist(const tDeviceID deviceTag, const tPath oldPlaylistName,  const tPlaylistName newPlaylistName);
    
    /**
     * R8976_UC3_LLD3.2
     * Function deletes an internal playlist 
     * @param[in] deviceID device
     * @param[in] playlistName filename of the internal playlist
     * @return != 0: error, = 0: OK
     */

    tResult DeleteInternalPlaylist(const tDeviceID deviceTag, const tPath playlistName);
    
    /**
     * Function creates a list with internal playlists 
     *
     * @param[in] deviceTag Device id for which the playlist list shall be created. ID = 0 all internal playlists
     * @param[out] listID ID of the list
     * @param[out] listSize length of the list
     * @return != 0: error, = 0: OK
     */
    tResult CreateMediaPlayerInternalPlaylistList(tListID& listID, tListSize& listSize, const tDeviceID deviceTag);

    /**
     * Recreates the DB with new lanaguage
     * -Clear all the old list handles as cleanup step
     * -Sets the new language into configuration with Dataprovider
     * -asks DBManager to recreate the DB
     *
     * @param[in] language new langauage which will be used by the recreated DB
     *
     * @return < 0: error, = 0: OK
     */
    tResult RecreateDatabase(const tLanguageType language);

    /**
     * Function resets highlighting of parent lists
     *
     * @param[in] listID ID of the current list
     * @return < 0: error, = 0: OK
     */
    tResult ResetParentHighlighting(const tListID listID);

    /**
    * Function updates the Mediaobjects' coverArtString and thumbnailBlob
    * Roadmap 15009
    *
    * @param[inout] mediaObjectVector vector with media objects to update
    * @return < 0: error, = 0: OK
    */
    tResult AddCoverArtAndThumbnail(vector<tMediaObject> &mediaObjectVector); //Roadmap 15009: 100%

    /**
    * Function updates the Mediaobjects coverArtString
    * Roadmap 15009
    *
    * @param[inout] mediafileVector vector with file objects to update
    * @param[in] deviceID device ID of affected device
    * @return < 0: error, = 0: OK
    */
    tResult AddCoverArt(vector<tFiles> &mediaFileVector, const tDeviceID deviceID);

    /**
    * Function updates the Mediaobjects' coverArtString
    * Roadmap 15009
    *
    * @param[inout] mediaObjectVector vector with media objects to update
    * @return < 0: error, = 0: OK
    */
    tResult AddCoverArt(vector<tMediaObject> &mediaObjectVector);

    /**
    * Function updates the Mediaobjects' thumbnailBlob
    * Roadmap 15009
    *
    * @param[inout] mediaObjectVector vector with media objects to update
    * @return < 0: error, = 0: OK
    */
    tResult AddThumbnail(vector<tMediaObject> &mediaObjectVector);

#if USE_IMAGE_OBJECT
    /**
    * Function updates the Mediaobjects' fileName with absolute image path (with mountpoint + filename)
    *
    * @param[inout] mediaObjectVector vector with media objects to update
    * @return < 0: error, = 0: OK
    */
    tResult ReplaceFilenameWithAbsolutePath(vector<tMediaObject> &mediaObjectVector);
#endif

    //<<------Roadmap 16003
    /**
    * Roadmap 16003 : CD ripping with Gracenote
    * R8976_UC3_LLD3.5_SEQ5
    * Methods to re order the entries in the internal playlist.
    * @param[in] playlist path including the filename
    * @param[in] deviceID id of device this playlist is for
    * @param[in] oldPosition the old position of the entry
    * @param[in] newPosition the new position of the entry
    * @return < 0: error, = 0: OK
    */
    tResult ReorderInternalPlaylist(IN const tPath &PlaylistPath, IN const tDeviceID deviceID, IN const tPosition oldPosition,IN const tPosition newPosition);



    /**
     * Roadmap 16003 : CD ripping with Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Function deletes the files specified in the tMediaObject vector
     *
     *  Deletes the files from database and flash device
     * @param[in] deviceid device id of the device
     * @param[in] mediaObjectUrls vector of mediaobject urls to be deleted
     * @param[out] number of files deleted
     * @return true or false
     */
    tResult DeleteMediaContent(IN const tDeviceInfo &deviceInfo,IN  const string &mediaObjectUrl, IN tBoolean isNormalMode = true);

    /**
     * Roadmap 16003 : CD ripping with Gracenote
     * R8972_UC3_LLD4.3_SEQ6
     * Function edit the metadata of  the files specified in the tEditMetaDataByUrl strcuct
     *
     *  Edit the meta data of the files  from database and flash device
     * @param[in] editMetaDataByUrl strcut packed with data having url, metdata, and device id.
     * @return true or false
     */
    tResult EditMetaData(IN tEditMetaDataByUrl &editMetaDataByUrl);

    //>--Roadmap CMG3G-10221 : 'Scene Recorder '

    /**
     * SetFilePermissions File permition for list of file will chnage as per user action
     * @param[in] filePermission strcut of objet id , url flag and readonly Flag
     * @return true or false
     */
    tResult SetFilePermissions(IN const tDeviceInfo deviceInfo,IN  const vector<tFilePermission> &filePermission,OUT tU16 &numofFiles);
    /**
     * This method  MediaplayerFolderInfo to set file permission
     * @param[in] deviceID device tag
     * @param[in] Path "/"
     * @param[in] FileTypeSelection "Type of the file selected"
     * @param[OUT] folderinfo Folder info for
     * total file size , protected file size
     * total file number , protected file number
     * @return < 0: error, = 0: OK
     */
    tResult MediaplayerFolderInfo(IN tDeviceID deviceID, IN tPath Path,
                                  IN tFileTypeSelection fileTypeSelection, OUT tFolderInfo& folderinfo);
    //>--Roadmap CMG3G-10221 : 'Scene Recorder '
    /**
     * ReMountFolder function will remount the specified device as read only or
     * read write .
     * @param[in] deviceInfo for the device
     * @param[in] readPermission is true for read only permission
     * @return < 0: error, = 0: OK
     */
    tResult ReMountFolder(const tDeviceInfo &deviceInfo, const tBool readPermission);

    //<--Roadmap CMG3G-10221 : 'Scene Recorder '
    
    /**
    * Function Reshuffles the ShuffleTable of given listID & ensures to maintain currentRow of the List
    * as the first item in shuffleTable
    *
    * @param[in] listID ID of the list whose shuffleTable need to be reshuffled
    * @return < 0: error, = 0: OK
    */
    tResult ReshuffleShuffleTable(const tListID listID,const tIndex beginIndex = INDEX_NONE);    


    /** //Suzuki CRQ 296694 and Bug 265009
     * Function gets the  position of current list from shufflelist if the playback mode is in random mode
     *
     * @param[in] currentListID list id of the list
     * @param[in] currrentListPosition position of the list
     * @param[out] shuffleListPosition position of the list from the shuffle list
     * @return < 0: error, = 0: OK
     */
    tResult GetPositionFromShuffleList(IN tListID listID, IN tPosition listPosition, OUT tPosition &shuffleListPosition);
    tIndex GetFirstItemFromShuffleList(IN tListID listID);

    /**
     * Function finds the folder path to switch to in case of a folder up/down request.
     *
     * @param[in,out] folderPath Current folder path as input and Folder path to switch to as output
     * @param[in] deviceID device tag
     * @param[in] playbackAction Denotes folder up or down action
     * @param[in] nextPrevSkipCount Number of folders to skip
     * @return < 0: error, = 0: OK
     */
    tResult GetFolderPathToSwitchTo(INOUT tPath& folderPath, IN tDeviceID deviceID, IN tPlaybackAction playbackAction, IN tNextPrevSkipCount nextPrevSkipCount);

    /**
     * Function populates the FolderHierarchy map with the list of folders in the device.
     *
     * @param[in] deviceID device tag
     * @return < 0: error, = 0: OK
     */
    tResult UpdateFolderHierarchyForDeviceID(const IN tDeviceID deviceID);

    /**
     * Function gets the number of media objects in the list
     *
     * @param[in] listID list id of the list
     * @param[out] numberOfMediaObjects number of media objects in the list
     * @return < 0: error, = 0: OK
     */
    tResult GetNumberOfMediaObjects(IN tListID listID, OUT tNumberOfMediaObjects& numberOfMediaObjects);

protected: 
    std::vector<tListID>  listIDsOfFullTextSearch;

    Lock  mLockIt[eThreadFctSize];
    tBool mbLoopThreadFct[eThreadFctSize];

    typedef struct
    {
        tListID          listIDParent;
        tListID          listID;
         tListSize        listSize;
         tDeviceID        deviceID;   

        tSearchIncrement searchIncrement;
        tDelaySec        testDelaySec;

        MessageQueue    *pMsgQ;
        void            *pMsgBuffer;
        unsigned int     uMsgQTimout_sec;
        unsigned int     uMsgLen;
        unsigned int     uMsgPrio;

        tBool            bFullTextSearchThreadExecuting;
        tBool            useNotifyThread; //otherwhise send within first thread
    }tParmsFullText;

    tParmsFullText m_ParmsFullTextSearch;

    /**
    * Roadmap 16014 'full text search'
    *
    * checks is listID belongs to 'full text search list'
    */
    tBool IsSearchList(tListID listID);

    /**
    * Roadmap 16014 'full text search'
    * if listIDParent has child 'FullttextSearchlists' then these are added to vector listsToRelease
    *
    * @param[in]    list id of parent list
    * @param[inout] vector is used to add those child listIDs
    * @return number of child listsIDs which have been added
    */
    tListSize AddChildSearchListsIfExists(IN const tListID listIDParent,INOUT std::vector<tListID> &listsToRelease);

   /**
    * stops thread used for full text search list 'listIDsearchList'
    * Roadmap 16014 'full text search'
    *
    */
    tResult StopFullTextSearch(const tListID listIDsearchList);

    
    //<-- Roadmap 16014  

    

    
    /**
    * ThreadFunction: notitfies every (x) seconds MethodreturnMiddle - same values are possible
    * Roadmap 16014 'full text search'
    *
    */
   void ThreadFctFullTextSearchNotify();
    
      /**
    * threadFunction: searches for next numOfIncrement results. Done per increment to offer abort of search 
    * Roadmap 16014 'full text search'
    *
    */
    
   void ThreadFctFullTextSearch();
      
     /**
    * Function: Stop full text search. To provide methodReturnMiddle after corresponding create list has been done it
    * ThreadFctFullTextSearch is started to search for the next increment of the list. this function is used to stop this search 
    * Roadmap 16014 'full text search'
    *
    */
   void ThreadFctFullTextSearchStop(tBoolean bCalledByFTSThread = false);

   /**
     * Function: Start full text search. I.e. start correspodning thread functions ThreadFctFullTextSearchNotify() and ThreadFctFullTextSearch() see also do() - function of ListControl
     * Roadmap 16014 'full text search'
     *
     * parameters necessary to text based list search. Especiall to create answers in the middle
     * @parm[IN] listID        
     * @parm[IN] listSize
     * @parm[IN] deviceID
     * @parm[IN] searchIncrement
     * 
     */
   void ThreadFctFullTextSearchStart(IN tListID listIDParent, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID,IN tSearchIncrement searchIncrement, IN tDelaySec testDelaySec);
   
    /**
    * Function updates the Mediaobjects' title for searching according to category type
    *
    * @param[in] mediaObject media object to search
    * @param[out] dbSearchMetadata updated search metadata
    * @return < 0: error, = 0: OK
    */
    tResult GetDBSearchMetadata(tMediaObject mediaObject, tMetadata &dbSearchMetadata);

    /**
    * Function removes the special characters(A, An, The) from starting of Metadata
    *
    * @param[inout] dbSearchMetadata updated search metadata
    * @return < 0: error, = 0: OK
    */
    tResult IgnoreSpecialWordsFromBeginning(tMetadata &dbSearchMetadata);

private:

    typedef struct shuffleTableAttributes
    {
        tNumberOfMediaObjects numberOfMediaObjects;
        tNumberOfPlayLists numberOfPlayLists;
        tNumberOfFolders numberOfFolders;
    }tShuffleTableAttributes;

    typedef struct {
        tListID listID;         // ID of list in ListControl
        tListID dbBrowseListID;   // ID of list in DB Manager for the browse list
        tIndex *shuffleTable;    // table to do the shuffling
        tListSize sizeOfShuffleTable;
        tShuffleTableAttributes shuffleTableAttributes;
        int isInShuffleMode;
        tIndex *shuffleSkippingStack;  // Stack for PreviousInShuffledList (Roadmap 13016)
        tListSize sizeOfShuffleSkippingStack;
        int shuffleOnNextForward;
        tListType listType;     // original list type on generation
        tCategoryType objectType; // original object type for which this list was created
        tObjectID objectID;     // original object which is used as filter tag on list creation
        tPlayContinuation playContinuation; // type of play continuation
        tDeviceID deviceID;     // id of device this list is for
        tFilterTag1 tag1;
        tFilterTag2 tag2;
        tFilterTag3 tag3;
        tFilterTag4 tag4;
        tSearchString searchText; //Roadmap 16014 'Full text search'
        tPath path;             // in case of LTY_FILELIST: path of this list in file system w/o mountpoint
        tObjectID nowPlayingObjectID;       // ID of the currently selected object used for object highlighting in any kind of list
        tPosition nowPlayingListPosition;   // position of currently played object in a playable list used for object tracking for list change update
        tListID parent;
        tListID next;
        tStreaming streaming;   // streaming flag Roadmap 13008
        tFileTypeSelection fileTypeSelection;   // selection criteria (e.g. only audio files)
        int isVTIPODList;       //flag if list is a VTiPod list
        int refCounter;
        int createCounter;
        tUserID userID; //Roadmap 17001 Personalization
    } tList;


    class autoDeRefListPtr
    {
    public:
        autoDeRefListPtr(tList *listPtr)
        {
            mListPtr = listPtr;
        }

        ~autoDeRefListPtr()
        {
            if (mListPtr) {
                if (mListPtr->refCounter) {
                    mListPtr->refCounter--;
                }
            }
        }

    private:
        tList *mListPtr;
    };

    vector<tList *> mLists;
    Lock mLock;
    int mStopFlag;

    map <tDeviceID, vector<tObjectID>> mFolderHierarchy; // CRQ 300254 Support for Folder Up/Down command

    tResult CreateList(tListID &listID);
    tList *GetListPointer(const tListID listID);
    tResult CreateMediaPlayerPlaylistParent(const tListID listID, const tPath path, const tDeviceID deviceID, const tStreaming streaming);
    
    tResult SetParent(tList *listPtr);
    tResult SetParentListCriteria(OUT tList &parentListCriteria,IN tList *listPtr);/*Roadmap 160014 full text search:*/
    tResult SetParentHighlighting(const tList *listPtr);
    tResult SetFileListHighlighting(const tList *listPtr);
    tResult GetPosition(tPosition &position, const tListID listID, const tListID parentListID);
    tResult SetPostionDuringIndexing(const tListID listID);
    tResult NextProcessing(tRowNumber &newRow, tListSize &listSize,tRowNumber &lastPlayedRow, const tListID listID, const int stepCount);
    /**
     * Function finds whether the given Folder contains any item  to play
     * and creates a new list for the same
     * @param[in] isEmptyListIDNeeded whether to update the ListID even when the Folder's List is Empty
     * @param[in] stepCount number of steps for next(>0) / prev(<0)
     * @param[in] FolderPath
     * @param[in,out] listID list ID of current list
     * @param[out] newRow new row number of object to play
     * @return < 0: error, = 0: OK
     */
    tResult FolderProcessing(tRowNumber &newRow,tListID &listID,tPath &path, const int stepCount,const tBoolean isEmptyListIDNeeded = false);
    tResult CreateShuffleTable(tList *listObject);
    tResult ReshuffleShuffleTable(tList *listObject,const tIndex beginIndex = INDEX_NONE);
    tResult ReInitShuffleSkippingStack(const tList *listObject);
    tResult SetupFirstRowInShuffleTable(tList* listObject,const tIndex beginIndex);
    tResult SetupUnConsecutiveNumbersInShuffleTable(tList* listObject);
    tBoolean CompareShuffleTables(const tIndex* oldShuffleTable,const tIndex* newShuffleTable,const tListSize size);      
    /**
     * Function finds the next object to play and possibly creats a new list
     *
     * @param[in] stepCount number of steps for next(>0) / prev(<0)
     * @param[in,out] listID list ID of current list
     * @param[out] newRow new row number of object to play
     * @return < 0: error, = 0: OK
     */
    tResult WrapProcessing(tRowNumber &newRow, tListID &listID, const int stepCount);

    tResult CreateFollowingList(tListID &listID, const int stepCount, const tRepeatMode repeatMode);

    /**
     * Function finds the the next Folder to play following the current folder in the parent folder
     * and creates a new list for the same
     *
     * @param[in] stepCount number of steps for next(>0) / prev(<0)
     * @param[in,out] listID list ID of current list
     * @param[out] newRow new row number of object to play
     * @return < 0: error, = 0: OK
     */
    tResult CreateFollowingFileList(tRowNumber &newRow,tListID &listID, const int stepCount);
    //tResult GetNowPlayingObjectID(tObjectID &nowPlayingObjectID, const tListID listID);


    /**
     * GetFileMode function will get file permission or
     * read write .
     * @param[in] deviceInfo for the device
     * @param[in] readPermission is true for read only permission
     * @return < 0: error, = 0: OK
     */
    
    tBoolean GetFileMode(const tDeviceInfo &deviceInfo, const char* path);
    tBoolean GetFileMode(IN const tDeviceID deviceID, IN const tObjectID objectID);
    //<--Roadmap CMG3G-10221 : 'Scene Recorder '
};

#endif  //_LISTCONTROL_H_

inline void GetCompletePlaylistPath(OUT tPath &playlistCompletePath,IN const tPath playlistName);


#define LISTCONTROL_HELPER(a) \
tListType helperSteps[] = {LTY_ARTIST, LTY_ALBUM, LTY_SONG, LTY_GENRE,\
                                       LTY_ALBUM, LTY_SONG, LTY_GENRE,\
                                                  LTY_SONG, LTY_GENRE,\
                                       LTY_ALBUM, LTY_SONG, LTY_GENRE,\
                           LTY_ARTIST, LTY_ALBUM, LTY_SONG, LTY_GENRE,\
                           LTY_ARTIST, LTY_END_OF_LTY};\
static int helperStepsIndex = 0;\
if (helperSteps[helperStepsIndex] == a) {\
    helperStepsIndex++;\
    if (helperSteps[helperStepsIndex] == LTY_END_OF_LTY) {\
        a = LTY_END_OF_LTY;\
        helperStepsIndex = 0;\
    }\
} else {\
    helperStepsIndex = 0;\
}\

/** @} */
