/**
 * @defgroup MediaPlayerInterface MediaPlayerInterface
 * @author Stefan Scherber
 *
 * Public interface for MediaPlayer
 * @{
 */

#ifndef _MEDIAPLAYERINTERFACE_H_
#define _MEDIAPLAYERINTERFACE_H_

#include "TypeDefinitions.h"
#include "MediaPlayer_ErrorCodes.h"
#include "Configuration.h"
#include "CustomControl.h"
#include "OutputWrapper.h"
#include "Lock.h"
#include <taglib/tstring.h>
#include <glib.h>
#include <vector>
#include "FastUTF8.h"

class MediaPlayerInterface
{
    public:

    /**
     * Function to create a singleton for the MediaPlayerInterface
     *
     * @return pointer to instance of MediaPlayerInterface object
     */
    static MediaPlayerInterface &GetInstance(void);

    /**
     * Function to unregister all components of MediaPlayer
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult UnregisterAll();

    /**
     * Function to register custom configuration component with MediaPlayer
     * This function must be called before the function CreateMediaPlayer is called
     *
     * @param[in] configuration pointer to custom configuration object which derives from the Configuration
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(Configuration *configuration);

    /**
     * Function to register custom output wrapper component with MediaPlayer
     * This function must be called before the function CreateMediaPlayer is called
     *
     * @param[in] outputWrapper pointer to custom output wrapper object which derives from the OutputWrapper class
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(OutputWrapper *outputWrapper);

    /**
     * Function to register specialised custom control component with MediaPlayer
     * This function must be called before the function CreateMediaPlayer is called
     *
     * @param[in] customControl pointer to project specific custom Control  object which derives from the CustomControl class
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult Register(CustomControl *customControl);

    /**
     * Function to create all internal Mediaplayer components
     * This function must be called before the function StateChangeNormal is called
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult CreateMediaPlayer(void);

    /**
     * Function to request Mediaplayer transition to NORMAL state
     * This function must be called before the MediaPlayer can be used
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult StateChangeNormal(void);

    /**
     * Function to request Mediaplayer transition to OFF state
     * After calling of this function the MediaPlayer functions cannot be used anymore.
     * If afterwards the MediaPlayer is required again, function StateChangeNormal must be called first again.
     *
     * @return int = 0: ok, !=0: failed
     */
    tResult StateChangeOff(void);

    /**
     * This function must be called by the controller to start audio output
     * Roadmap 13024
     * if (m_CurrentState == UNDERVOLTAGE)
     * - Store allocate state and alsa name in local member variables
     * else
     * - Call PlayerManager::SendAllocate(audioOutputDevice)
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return < 0: error, = 0: OK
     */
    tResult SendAllocate(const tAudioOutputDevice audioOutputDevice); //Roadmap 13024: 100%

    /**
     * This function must be called by the controller to stop audio output
     * Roadmap 13024
     * if (m_CurrentState == UNDERVOLTAGE)
     * - Store allocate state in local member variable
     * else
     * - Call PlayerManager::SendDeAllocate()
     *
     * @return < 0: error, = 0: OK
     */
    tResult SendDeAllocate(); //Roadmap 13024: 100%

    /**
     * Activate or deactivate the MediaPlayer audio output
     * Roadmap 13024
     * if (m_CurrentState == UNDERVOLTAGE)
     * - Store allocate state in local member variable
     * else
     * - Call PlayerManager::SendSourceActivity(sourceActivity)
     *
     * @param[in] sourceActivity ON or OFF
     * @return < 0: error, = 0: OK
     */
    tResult SendSourceActivity(const tSourceActivity sourceActivity); //Roadmap 13024: 100%

    /**
     * Clear last source (Current listID and media object) in PlayerManager
     * @return < 0: error, = 0: OK
     */
    tResult ClearLastSource();

    /**
     * Function must be called if mute status has changed
     * @param[in] muteState Mute status
     * @return < 0: error, = 0: OK
     */
    tResult SendMuteStatus(const tMuteState muteState);

    /**
    *  This method creates a new list that contains the meta-data values of the requested indexed media as follows:
    *
    *  1.   The client can specify up to three filter Tags, dependent on the type of list requested.
    *       The order and contents of the list generated are dependent on the filter criteria.
    *  2.   This method may only be called on a device that supported indexing acording to its DeviceInfo.
    *  3.   This method returns a handle to the created playlist and the number of objects in the list.
    *  4.   A list is considered to be valid until the ReleaseMediaPlayerList function is executed.
    *  5.   Changes to a list are reported by the OutputWrapper::SendMediaPlayerListChanged.
    *
    *  The CreateMediaPlayerIndexedList function assigns a unique ListHandle for each filtered list created and returns this value,
    *  as well as the number of results in the filtered list.
    *
    *  The valid ListType and fitler string combinations are shown below:
    *  | ListType                      |FilterTag1      |FilterTag2      |FilterTag3|
    *  |-------------------------------|----------------|----------------|-----------
    *  |LTY_GENRE                      |Null            |Null            |Null      |
    *  |LTY_GENRE_ARTIST               |Genre Tag       |Null            |Null      |
    *  |LTY_GENRE_ARTIST_ALBUM         |Genre Tag       |Artist Tag      |Null      |
    *  |LTY_GENRE_ARTIST_ALBUM_SONG    |Genre Tag       |Artist Tag      |Album Tag |
    *  |LTY_ARTIST                     |Null            |Null            |Null      |
    *  |LTY_ARTIST_ALBUM               |Null            |Artist Tag      |Null      |
    *  |LTY_ARTIST_ALBUM_SONG          |Null            |Artist Tag      |Album Tag |
    *  |LTY_ALBUM                      |Null            |Null            |Null      |
    *  |LTY_ALBUM_SONG                 |Null            |Null            |Album Tag |
    *  |LTY_SONG                       |Null            |Null            |Null      |
    *  |LTY_PODCAST                    |Null            |Null            |Null      |
    *  |LTY_PODCAST_EPISODE            |Podcast Tag     |Null            |Null      |
    *  |LTY_AUDIOBOOK                  |Null            |Null            |Null      |
    *  |LTY_BOOKTITLE_CHAPTER          |Null            |Book Tag        |Null      |
    *  |LTY_AUTHOR                     |Null            |Null            |Null      |
    *  |LTY_AUTHOR_BOOKTITLE           |Author Tag      |Null            |Null      |
    *  |LTY_AUTHOR_BOOKTITLE_CHAPTER   |Author Tag      |Book Tag        |Null      |
    *  |LTY_COMPOSER                   |Null            |Null            |Null      |
    *  |LTY_COMPOSER_ALBUM             |Composer Tag    |Null            |Null      |
    *  |LTY_COMPOSER_ALBUM_SONG        |Composer Tag    |Album Tag       |Null      |
    *  |LTY_COMPOSER_SONG              |Composer Tag    |Null            |Null      |
    *  |LTY_VIDEO                      |Null            |Null            |Null      |
    *  |LTY_VIDEO_EPISODE              |Video Tag       |Null            |Null      |
    *  |LTY_PLAYLIST                   |Null            |Null            |Null      |
    *  |LTY_PLAYLIST_SONG              |Playlist Tag    |Null            |Null      |
    * @param[in] listType list type to create play list filtered after object ID
    *                     see special behavior described above for all songs list.
    * @param[in] tag1,tag2,tag3     three filter tag object id's to generate the complete list
    *                     which ID is in which tag position depends on the list type.
    *                     max 3 different tags are possible
    * @param[in] deviceID id of the device for special filtered list depending on a device
    *
    * @param[out] listID id of the list which was created
    * @param[out] listSize length of list
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerIndexedList(
        tListID &listID,
        tListSize &listSize,
        const tListType listType,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tDeviceID deviceID,
        tUserID userID = 0);


    /**
       * This method generates a list based on an existing. Tags from that parent list plus searchString are used  //Roadmap 16014 'Full text search'
       *
       * @param[in] listIDParent is of parent list
    * @param[in] listIDReplaceable if the user changes the search string hten he wants to create a new list.
    *                     The previously used listIDReplaceable is handed over to GMP if it is now obsolete.
    *                     GMP cares to delete it
    * @param[in] searchString holds the string which is used in this context for the text search. E.g. if parent is DTY_ALBUM list then resulting list has
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerListSearchString(
        OUT tListID &listID,
        OUT tListSize &listSize,
        OUT tDeviceID &deviceID,
        IN const tListID listIDParent,
        IN const tListID listIDReplaceable,
        IN const tSearchString searchString);


    /**
    * This method generates a list of the items in a playlist.
    *
    * @param[in] fileName filename to a playlist on a usb stick for example
    * @param[out] listID id of the list which was created
    * @param[out] listSize length of list
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerPlaylistList(
        tListID &listID,
        tListSize &listSize,
        const tFilename fileName);

    /**
    * This method generates an ordered list of media objects that match all the specified filter criteria.
    * The order and contents of the list generated are dependent on the filter criteria and the AutoPlay parameter.
    * The list created by this method creates the list of objects to play, in the order to play them as follows:
    *
    *   1.   This method may only be called on a device that supported indexing ccording to its DeviceInfo.
    *   2.   This method returns a handle to the created playlist and the number of objects in the list.
    *   3.   A list is considered to be valid until the ReleaseMediaPlayerList function is executed.
    *   4.   Changes to a list are reported by OutputWrapper::SendMediaPlayerListChanged.
    *
    *  The valid ListType and fitler string combinations are shown below:
    *   |ListType                        |FilterTag1      |FilterTag2      |FilterTag3      |FilterTag4 |
    *   |--------------------------------|----------------|----------------|----------------|-----------|
    *   |LTY_GENRE_SONG                  |Genre Tag       |Null            |Null            |Null       |
    *   |LTY_GENRE_ARTIST_SONG           |Genre Tag       |ArtistTag       |Null            |Null       |
    *   |LTY_GENRE_ARTIST_ALBUM_SONG     |Genre Tag       |Artist Tag      |Album Tag       |Null       |
    *   |LTY_GENRE_ARTIST_ALBUM_SONG     |Genre Tag       |Artist Tag      |Album Tag       |Song Tag   |
    *   |LTY_ARTIST_SONG                 |Null            |Artist Tag      |Null            |Null       |
    *   |LTY_ARTIST_ALBUM_SONG           |Null            |Artist Tag      |Album Tag       |Null       |
    *   |LTY_ARTIST_ALBUM_SONG           |Null            |Artist Tag      |Album Tag       |Song Tag   |
    *   |LTY_ALBUM_SONG                  |Null            |Null            |Album Tag       |Null       |
    *   |LTY_ALBUM_SONG                  |Null            |Null            |Album Tag       |Song Tag   |
    *   |LTY_SONG                        |Null            |Null            |Null            |Song Tag   |
    *   |LTY_PODCAST_EPISODE             |Podcast Tag     |Null            |Null            |Null       |
    *   |LTY_BOOKTITLE_CHAPTER           |Audiobook Tag   |Null            |Null            |Null       |
    *   |LTY_AUTHOR_BOOKTITLE_CHAPTER    |Audiobook Tag   |Author Tag      |Null            |Null       |
    *   |LTY_COMPOSER_ALBUM_SONG         |Composer Tag    |Album Tag       |Null            |Null       |
    *   |LTY_COMPOSER_SONG               |Composer Tag    |Null            |Null            |Null       |
    *   |LTY_VIDEO_EPISODE               |Video Tag       |Null            |Null            |Null       |
    *   |LTY_PLAYLIST                    |Null            |Null            |Null            |Null       |
    *
    * @param[in] listType list type to create play list filtered after object ID
    * @param[in] tag1,tag2,tag3, tag4     four filter tag object id's to generate the complete list
    *                     which ID is in which tag position depends on the list type.
    *                     max 4 different tags are possible
    * @param[in] playContinuation decides what to do on end of list
    * @param[in] deviceID decides what to do on end of list
    * @param[out] listInfo reference to the object storing the information of the created list
    * @return < 0: error, = 0: OK
    */
    tResult CreateQuickPlayList(
        tListInfo &listInfo,
        const tListType listType,
        const tFilterTag1 tag1,
        const tFilterTag2 tag2,
        const tFilterTag3 tag3,
        const tFilterTag4 tag4,
        const tPlayContinuation playContinuation,
        const tDeviceID deviceID);

    /**
    * This method creates a list of all supported files and folders in the directory.
    * The CreateMediaPlayerFileList function assigns a unique
    * ListHandle for each filtered list created and returns this value, as well
    * as the number of results in the filtered list. A list is considered to be
    * valid until the ReleaseMediaPlayerList function is executed. Changes to a
    * list are reported by OutputWrapper::SendMediaPlayerListChanged.
    * This method also returns the total number of MediaObjects, Playlists and Folders
    * in the associated List. These numbers are static for each List at creation time and
    * are included as returned parameters to allow HMI controllers to configure their view accordingly
    * without requiring the controller to pull the entire list and iterate it first.
    * @param[in] deviceID id of the desired USB stick
    * @param[in] path path in which the list should be created
    * @param[in] filetypeSelection selection criteria (e.g. only audio files)
    * @param[out] listId id of the created list
    * @param[out] listSize length of the list
    * @param[out] numberOfMediaObjects number of songs in the list
    * @param[out] numberOfPlayLists number of playlists in the list
    * @param[out] numberOfFolders number of sub folders in the list
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerFileList(
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        const tPath path,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);

    tResult CreateMediaPlayerCDList(
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        const tPath path,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection = FTS_AUDIO);

    /**
    * This method creates a list of all supported files and folders in the directory.
    * The CreateMediaPlayerFileListExt function assigns a unique
    * ListHandle for each filtered list created and returns this value, as well
    * as the number of results in the filtered list. A list is considered to be
    * valid until the ReleaseMediaPlayerList function is executed. Changes to a
    * list are reported by OutputWrapper::SendMediaPlayerListChanged.
    * This method also returns the total number of MediaObjects, Playlists and Folders
    * in the associated List. These numbers are static for each List at creation time and
    * are included as returned parameters to allow HMI controllers to configure their view accordingly
    * without requiring the controller to pull the entire list and iterate it first.
    * Filename conatins the path and a file name. The functions returns the position of this file in teh directory
    * @param[in] deviceID id of the desired USB stick
    * @param[in] filename path including filename in which the list should be created
    * @param[in] filetypeSelection selection criteria (e.g. only audio files)
    * @param[out] listId id of the created list
    * @param[out] listSize length of the list
    * @param[out] numberOfMediaObjects number of songs in the list
    * @param[out] numberOfPlayLists number of playlists in the list
    * @param[out] numberOfFolders number of sub folders in the list
    * @param[out] position index in the created list of the given file (used for highlighting) //Roadmap 13017
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerFileListExt( //Roadmap 13017
        tListID &listId,
        tListSize &listSize,
        tNumberOfMediaObjects &numberOfMediaObjects,
        tNumberOfPlayLists &numberOfPlayLists,
        tNumberOfFolders &numberOfFolders,
        tPosition &position,
        const tPath filename,
        const tDeviceID deviceID,
        tFileTypeSelection filetypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);

    /**
    * The RequestMediaPlayerIndexedListSlice method provides a slice of the filtered list specified by the ListHandle parameter.
    * The list must be created either with method CreateQuickPlayList or method CreateMediaPlayerIndexedList.
    * The slice begins with the item at location WindowStart and includes in sequential order the number of items specified by the parameter WindowSize.
    * The RequestListSlice method returns the slice as a MediaObjects vector.
    * The data contained in each MediaObject is used by the HMI to determine the meta-data and other properties of the media object.
    * If the number of items in the list starting at WindowStart is less than the WindowSize parameter then the number of items
    * provided in the slice will be less than the WindowSize calling parameter.
    *
    * @param[in] listID if for which the slice is requested
    * @param[in] startIndex begin of the slice in the list
    * @param[in] sliceSize size of the slice
    * @param[out] mediaObjectVector vector with found objects for list slice, the vector must be allocated by caller.
    * @return < 0: error, = 0: OK
    */
    tResult RequestMediaPlayerIndexedListSlice(
        vector<tMediaObject> &mediaObjectVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize);

    /**
    * The RequestMediaPlayerFileListSlice method provides a slice of the filtered list specified by the ListHandle parameter.
    * The list must be created either with method CreateMediaPlayerFileList or method CreateMediaPlayerPlaylistList.
    * The slice begins with the item at location WindowStart and includes in sequential order the number of items specified by the parameter WindowSize.
    * The RequestMediaPlayerFileListSlice method returns the slice as a FileList stream.
    * If the number of items in the list starting at WindowStart is less than the WindowSize parameter then the number of items provided in the slice
    * will be less than the WindowSize calling parameter.
    *
    * @param[in] listID id of created list
    * @param[in] startIndex first item in list to be returned
    * @param[in] sliceSize size of slice beginning with startIndex
    * @param[out] mediaFileVector vector of all found files
    * @return < 0: error, = 0: OK
     */
    tResult RequestMediaPlayerFileListSlice(
        vector<tFiles> &mediaFileVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize);

    /**
    * The ReleaseMediaPlayerList function releases a list that has been created by the CreateMediaPlayerFileList, CreateMediaPlayerIndexedList,
    * CreateMediaPlayerQuickPlayList,CreateMediaPlayerPlaylistList methods. The list to release is specified by the ListHandle parameter.
    * The ListHandle becomes invalid by this function.
    *
    * @param[in] listID id which list has to be released
    * @return < 0: error, = 0: OK
    */
    tResult ReleaseList(
        const tListID listID);

    /**
    * The RequestListInformation Method provides a means for HMI controllers to retrieve detailed information on a ListHandle as follows:
    *   1.   This method was added to allow controllers to determine type, size, and parent list information for the ListHandle
    *       transmitted in the GetNowPlaying method.
    *       Only the ListHandle is sent within the GetNowPlaying method, but controllers need type, parent,
    *       and size information to implement the required functionality.
    *   2.   The following information is returned in the ResultAck response.
    *       *   ListType contains the type of the list for the supplied ListHandle.
    *       *   ParentListHandle contains the handle of the parent to this list. If the supplied ListHandle is the 'root' list and does not have a parent,
    *           the returned ListHandle will be Null.
    *       *   DeviceTag points to the device from which the list was generated which will be either 0 for MyMedia
    *           if the list was generated from the MyMedia database across all devices or
    *           a valid device-specific tag if the list only includes information from a specific user device.
    *       *   Position of this list item within the parent's list (the list pointed by the ParentListHandle). If this list does not have a parent (ParentListHandle = NULL, 0x00000000), then the Position parameter shall be set to 0 as a NULL default.
    *       *   ListSize is the current number of items in the list pointed by the supplied ListHandle.
    *
    * @param[in] listID id for the list to find
    * @param[out] listType type of list
    * @param[out] parentListID id of the parent list (0 for parent)
    * @param[out] deviceID device id for which list was created (0 = myMedia)
    * @param[out] position position of this list in the parent list
    * @param[out] listSize length of the list
    * @param[out] path root path of the list
    * @return < 0: error, = 0: OK
    */
    tResult RequestListInformation(
        tListType &listType,
        tListID &parentListID,
        tDeviceID &deviceID,
        tPosition &position,
        tListSize &listSize,
        tPath &path,
        const tListID listID);

    /**
     * Quicksearch function is used to search a string in a list and returns the position of the string in the list. If the string is not found the
     * GMP returns the following position or if the end of list is reached it returns end of list information.
     * The search start position can be defined by client.
     *
     * @param[out] outposition is the position of the searched string
     * @param[out] quicksearchresult is an enum-:FOUND, FOLLOWING, ENDOFLIST
     * @param[in] listID id for the list to find
     * @param[in] startposition is the starting position for quicksearch(can be defined by client)
     * @param[in] searchstring is the string that is searched
     */
    tResult Quicksearch(
        tPosition &outposition,
        tSearchResult &quicksearchresult,
        const tListID listID,
        tPosition startposition,
        tSearchString searchstring);


    /**
    * This function returns the search keyboard for a particular ListHandle.  The returned list identifies which letters of the Vertical Keyboard have at
    * least one Media Player entry that start with that letter. The index of the
    * first entry associated with the letter is also provided. If a
    * letter does not have an index then a Media Player entry that starts with
    * that letter does not exist.
    * Note that the returned search keyboard list may no longer be valid if items in the underlying list have changed.
    * Thus, controllers shall re-request the search keyboard list upon a receipt of OutputWrapper::SendMediaPlayerListChanged for the ListHandle tied to the returned search keyboard.
    *
    * @param[in] listID id of the list for which the keyboard should be created
    * @param[out] searchKeyboardList a vector with all letters for the keyboard
    * @return < 0: error, = 0: OK
    */
    tResult SearchKeyboardMediaPlayerList(
        vector<tSearchKeyboard> &searchKeyboardList,
        const tListID listID);

    /**
    *  Sets the current playback mode of the current device:
    *   * Normal: Navigate through the active playlist sequentially
    *   * Random: Enable shuffle mode by playing items in the active playlist in a random order
    *
    * @param[in] playbackMode playback mode enum
    * @return < 0: error, = 0: OK
    */
    tResult SendPlaybackMode(
        const tPlaybackMode playbackMode);

    /**
    * Method returns the playback mode of the current device
    *
     * @param[out] playbackMode playback mode enum
     * @return < 0: error, = 0: OK
     */
    tResult GetPlaybackMode(
        tPlaybackMode &playbackMode);

    /**
     * Methods sets the repeat mode of the current device
     *
     * @param[in] repeatMode repeat mode enum
     * @return < 0: error, = 0: OK
     */
    tResult SendRepeatMode(
        const tRepeatMode repeatMode);

    /**
     * Function returns the repeat mode of the current device
     *
     * @param[out] repeatMode repeat mode enum
     * @return < 0: error, = 0: OK
     */
    tResult GetRepeatMode(
        tRepeatMode &repeatMode);

    /**
     * Function returns a media object information
     * The CategoryType parameter is optional and can be used to limit the scope of a tag search.
     * If the controller requests for example a genre "media object" this
     * function will return the info of first song found in this genre
     *
     * @param[in] mediaObjectID id of the media object
     * @param[in] categoryType type id for a category (optional) to shorten the search
     * @param[out] mediaObject filled object with all media object data
     *
     * @return < 0: error, = 0: OK
     */
    tResult GetMediaObject(
        tMediaObject &mediaObject,
        const tMediaObjectID mediaObjectID,
        const tCategoryType categoryType);

    /**
    * This Method is used to retrieve the MIME type and size of the album art
    * image associated with a particular media object (if one exists for the
    * requested object). The actual album art image is retrieved by calling the
    * GetMediaObjectAlbumArt function.
    * This is an asynchronous request. The answer is given back to the controller by the method OutputWrapper::SendGetMediaObjectAlbumArtInfoAnswer
    * In case of an error the method OutputWrapper::SendGetMediaObjectAlbumArtInfoError is called.
    *
    * @param[in] albumArtString device ID + URL of image to read
    * @param[in] sizeX number of pixels in X direction
    * @param[in] sizeY number of pixels in Y direction
    * @param[in] mimeType type of picture (MMT_PNG, MMT_JPG, ...)
    * @param[in] userContext user context structure which will be forwarded to the chain in
    * case of an async update must be done after reading the album art.
    * @return < 0: error, = 0: OK
    */
    tResult GetMediaObjectAlbumArtInfo(
        const tAlbumArt albumArtString,
        const tSize sizeX,
        const tSize sizeY,
        const tMimeType mimeType,
        const tUserContext userContext);

    /**
    * The GetMediaObjectAlbumArt method is called to retrieve the album art image, if it exists, for a particular media object.
    * The image is returned in a MIME-typed image format. To get the MIME type, as well as image size, controllers first call the GetMediaObjectAlbumArtInfo Method,
    * retrieve the MIME type and size to set up the retrieval of the album art, then call GetMediaObjectAlbumArt to retrieve the actual image data.
    * The controller shall support the following image file formats. If a particular media object's Album art is not in one of these supported formats,
    * the MediaPlayer either transcode the album art into one of these formats or report no album art
    *       1. PNG Format
    *           * MIME Type = 'image/png'
    *           * PNG 'TrueColor' 24-bit format without Alpha blending
    *           * PNG 'Indexed' with 1, 2, 4, and 8 bits per pixel
    *       2. BMP Format
    *           * MIME Type = 'image/x-ms-bmp'
    *           * Header field = 'BM' (other header types not supported)
    *           * Bits per pixel = 8, 16, or 24
    *           * Number of colors in color palette = 0, full color, as well as paletted support for the above Bits per pixel avalues.
    *           * Compression = RGB, RLE8, and BITFIELDS
    *           * JPEG and PNGs contained within a BMP are intended for printer support only and are not required. JPEG and PNG images shall be transmitted as dedicated JPEG and PNG images instead of containerizing within the BMP format.
    *       3. GIF Format
    *           * MIME Type = 'image/gif'
    *           * LZW compressed support only. No Animated GIF support required.
    *       4. JPEG Format (JPEG File Interchange Format (JFIF) per ISO/IEC FCD 10918-5)
    *           * MIME Type = 'image/jpeg'
    *           * Thumbnail data and JFIF extension segment format not required. i.e. Only one image per each JPEG formatted MHP transmission.
    *       5. For all of the formats above, the maximum size and image resolution are limited by internal calibrations within the MediaPlayer.
    *       For album art above this limit, the MediaPlayer scales the image down to the maximum supported size if possible.
    *       If image re-scaling is not possible, then the MediaPlayer reports no image for the associated Album art.
    *
    * @param[in] albumArtString device ID + URL of image to read
    * @param[in] sizeX number of pixels in X direction
    * @param[in] sizeY number of pixels in Y direction
    * @param[in] mimeType type of picture (MMT_PNG, MMT_JPG, ...)
    * @param[in] userContext user context structure which will be forwarded to the chain in
    * case of an async update must be done after reading the album art.
     * @return < 0: error, = 0: OK
     */
    tResult GetMediaObjectAlbumArt(
        const tAlbumArt albumArtString,
        const tSize sizeX,
        const tSize sizeY,
        const tMimeType mimeType,
        const tUserContext userContext);

    /**
    * Returns the number of devices ever connected that support media playback.
    * For each ever connected device, this method provides specific information about the device.
    * @param[out] numberOfDevices number of connected devices
    * @param[out] deviceInfo array with all device information
    * @param[in] isForDipo Flag to check if call is made from DIPOconnections(Default=FALSE)
    * @return < 0: error, = 0: OK
    */
    tResult GetMediaplayerDeviceConnections(
        tNumberOfDevices &numberOfDevices,
        vector<tDeviceInfo> &deviceInfo,
        tBoolean isForDipo = false
        );
    /**
    * Returns device info of a device
    * @param[out] deviceID Device ID
    * @param[in] mountpoint
    * @return < 0: error, = 0: OK
    */
    tResult GetDeviceID(OUT tDeviceID &deviceID,IN const tMountPoint mountPoint);
    /**
    * Returns device info of a device
    * @param[in] deviceID Device ID
    * @param[out] deviceInfo
    * @return < 0: error, = 0: OK
    */
    tResult GetDeviceInfo(
        tDeviceInfo &deviceInfo,
        const tDeviceID deviceID);
    /**
    * Returns the number of devices connected that support media playback.
    * @param[out] numberOfDevices number of connected devices
    * @return < 0: error, = 0: OK
    */
    tResult GetConnectedDeviceCount(
        tNumberOfDevices &numberOfDevices);

    /**
    * Returns number of image files for a device
    * @param[in] deviceID device ID
    * @param[out] numberOfImageFiles number of image files
    * @return < 0: error, = 0: OK
    */
    tResult GetImageFileCount(tNumberOfMediaObjects &numberOfImageFiles, const tDeviceID deviceID);

    /**
    * Returns number of image files for a device and path
    * @param[in] deviceID device ID
    * @param[in] path path to image folder
    * @param[out] numberOfImages number of image objects
    * @return < 0: error, = 0: OK
    */
    tResult GetNumberOfImagesByPath(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID, const tPath path);

    /**
    * This method returns a vector of the indexing state of all currently connected devices.
    * The controller is notified about a change in indexing on one or more devices by the method OutputWrapper::UpdateIndexingState.
    *
    * @param[out] indexingState struct with indexing state info
    * @return < 0: error, = 0: OK
    */
    tResult GetIndexingState(vector<tIndexingStateResult> &indexingState);

    /**
    * This method returns the connected bluetooth profiles (iAP2 connections, host and device mode) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateBTProfileInfo.
    *
    * @param[out] profileList BT profile list
    * @return < 0: error, = 0: OK
    */
    tResult GetBTProfileInfo(tBTProfileList &profileList);

    /**
    * This method returns the nowPlaying info (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPONowPlaying.
    *
    * @param[out] deviceID device ID
    * @param[out] nowPlaying now playing
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPONowPlaying(tDeviceID &deviceID, tDiPONowPlaying &nowPlaying);

    /**
    * This method returns the playback status (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOPlaybackStatus.
    *
    * @param[out] deviceID device ID
    * @param[out] playbackState playback status
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOPlaybackStatus(tDeviceID &deviceID, tHMIPlaybackState &playbackState);

    /**
    * This method returns the playback shuffle mode (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOPlaybackShuffleMode.
    *
    * @param[out] deviceID device ID
    * @param[out] playbackMode playback shuffle mode
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOPlaybackShuffleMode(tDeviceID &deviceID, tPlaybackMode &playbackMode);

    /**
    * This method returns the playback repeat mode (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOPlaybackRepeatMode.
    *
    * @param[out] deviceID device ID
    * @param[out] repeatMode playback repeat mode
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOPlaybackRepeatMode(tDeviceID &deviceID, tRepeatMode &repeatMode);

    /**
    * This method returns the time elapsed and total duration (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOPlaytime.
    *
    * @param[out] deviceID device ID
    * @param[out] elapsedPlaytime elapsed playtime of current media object
    * @param[out] totalPlaytime total playtime of current media object
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOPlaytime(tDeviceID &deviceID, tPlaytime &elapsedPlaytime, tPlaytime &totalPlaytime);

    /**
    * This method returns the callStateList (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOCallState.
    *
    * @param[out] deviceID device ID
    * @param[out] callStateList list of callstates
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOCallState(tDeviceID &deviceID, tDiPOCallState &callStateList);

    tResult GetDiPOCallDuration(tDeviceID &deviceID, unsigned int &callDuraion);

    /**
    * This method returns the communications update (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOCommunications.
    *
    * @param[out] deviceID device ID
    * @param[out] communications communications
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOCommunications(tDeviceID &deviceID, tDiPOCommunications &communications);

    tResult GetDiPOLocationInfo(tDeviceID &deviceID, tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType);

    tResult GetDiPOGPRMCDataStatusValues(tDeviceID &deviceID, tMountPoint &USBSerialNumber, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues);

    /**
    * This method returns the indexing state for a given device id.
    * The controller is notified about a change in indexing on one or more devices by the method OutputWrapper::UpdateIndexingState.
    *
    * @param[in] deviceID ID of a device object
    * @param[out] indexingState indexing state for the device
    * @return < 0: error, = 0: OK
    */
    tResult GetIndexingState(tIndexingState &indexingState, const tDeviceID deviceID);
    /**
    * Identifies the media device that is currently sourced. The DeviceTag of
    * 0x00 is reserved as the MyMedia device. When this tag is the
    * ActiveMediaSource, the user is presented with an indexed library of
    * aggregated media from all connected devices with DeviceIndexedStates of
    * PARTIAL and COMPLETE.
    *
    * @param[out] deviceInfo reference to the parameter indicating the deviceInfo struct of the active device.
    * @return < 0: error, = 0: OK
    */

    /**
      * This function to get current slot State of optical disc e.g. if it is ejected or inserted
      *
      * @param[out] opticalDiscSlotState :
      * @return < 0: error, = 0: OK
      */

    tResult GetOpticalDiscSlotStatus(OUT tDMOpticalDiscSlotState &opticalDiscSlotState); //Roadmap 15002_CDAudio_DetectionAndDiagnosis



    tResult ActiveMediaDeviceGet(
        tDeviceInfo &deviceInfo,
        tBoolean isForDipo = false);

    /**
    * This function is used to set an active media source by DeviceTag.
    * The DeviceTag of 0x00 is reserved as the MyMedia device.
    *
    * @param[in] deviceID object ID of the device to activate
    * @param[in] activeSource defines if the device becomes inactive (FALSE) or active (TRUE)
    * @return < 0: error, = 0: OK
    */
    tResult ActiveMediaDeviceSet(
        const tDeviceID deviceID,
        const tBoolean activeSource,
         tMediaContext mediaContext = MC_ALL /*default value */);

    /**
    * Requests the playback speed for the currently playing media files. This functionality include:
    *      * Forward / Backward playback
    *      * Fast playback and Slow playback based on playbackspeed value.
    *
    * @param[in] playbackDirection enum of playback Direction to set
    * @param[in] playbackSpeed enum of playback speed to set with absolute and relative speed values option.
    * @return < 0: error, = 0: OK
    */
    tResult SendPlaybackSpeed(
            const tPEPlaybackDirection playbackDirection,
            const tPlaybackSpeed playbackSpeed);
    /**
    * Requests the playback action for the currently playing media files. Playback actions include:
    *      * Previous or next track
    *      * Fast forward or fast reverse
    *      * Play, pause, or stop
    *
    * @param[in] playbackAction enum of playback action to set
    * @param[in] nextPrevSkipCount skip count for NEXT and PREV to skip more than one object with one action
    * @return < 0: error, = 0: OK
    */
    tResult SendPlaybackAction(
        const tPlaybackAction playbackAction,
        const tNextPrevSkipCount nextPrevSkipCount=1);

    /**
    * Requests to move to a specific point within the current track. The desired
    * position can be expressed either as a percentage of the TotalPlaytime or
    * as the absolute desired playpoint position. The two parameters are
    * mutually exclusive.
    *
    * @param[in] playpointPosition desired position in the object to move the playpoint
    * @param[in] playpointFormat format of PlaypointPosition parameter
    *            (percentage or absolute time value in milliseconds)
    * @return < 0: error, = 0: OK
    */
    tResult SendSeekTo(
        const tPlaytime playpointPosition,
        const tPlaypointFormat playpointFormat);

    /**
    * The PlayItemFromList method is called to play an item from a list that has already been created. This method is handled by the MediaPlayer as follows:
    *       1.   If the MediaPlayer is not currently connected as an audio source, then the MediaPlayer sets the item in the list,
    *           and the rest of the list as well.
    *       2.   If the MediaPlayer is already connected and playing audio on another media object, it stops playback of that media object and
    *           starts playing the selected list item automatically.
    *       3.   Playback begins at the position specified by the StartIndex parameter and continues in the order of the items included in the list
    *           defined by the ListHandle parameter.
    *       4.   After playing the selected item, the MediaPlayer plays the rest of the list, if any, according to its rules for play order.
    *       5.   Note that the connect operation to the AV-Management of the system is not performed by the MediaPlayer and
    *           is instead performed by the HMI controller. For example, if the user directly selects an Artist from an HMI menu,
    *           the HMI controller would first call PlayMediaPlayerObject and then would ask the AV-Managementto connect the MediaPlayer over a Logical Audio Channel.
    *
    * @param[out] listSize number of elements in the list
    * @param[in] listID id of the list which should be used for playback
    * @param[in] index index of object in list (-1 = last active object)
    * @param[in] offset playtime to start
    * @return < 0: error, = 0: OK
    */
    tResult PlayItemFromList(
        tListSize &listSize,
        const tListID listID,
        const tIndex index=-1,
        const tPlaytime offset=0);

    /**
    * The PlayItemFromListByTag method is called to play an item from a previously created list. The item is identified by the Tag parameter which must point to an item
    * currently in the list. PlayItemFromListByTag replaces PlayItemFromList 0x229 because PlayItemFromList uses an index to identify the item to play.
    * However, it was discovered late in MY13 development that the index method doesn't work properly when a device is currently indexing because the item pointed by an index
    * will change as updates to the list are posted from the MediaPlayer. This will lead to the MediaPlayer playing a different item than the one the user selected.
    * The PlayItemFromListByTag method is handled by the MediaPlayer as follows:
    *       1.   If the MediaPlayer is not currently connected as an audio source to the MOST network (via the AVManager), then the MediaPlayer sets the item in the list,
    *            and the rest of the list as well, as the next media object to be played when the MediaPlayer is next connected to the AVManager.
    *       2.   If the MediaPlayer is already connected on the network and playing audio on another media object, it stops playback of that media object
    *            and starts playing the selected list item automatically.
    *       3.   Playback begins at the object identifed by the Tag parameter and continues according to the order of the list identified by the ListHandle parameter.
    *       4.   In order to use this method, all items in the list must be "playable". Therefore, this method is mainly used in conjunction with CreateMediaPlayerQuickPlayList
    *            but can also be used with CreateMediaPlayerFile.
    *       5.   After playing the selected item, the MediaPlayer shall play the rest of the list, if any, according to its rules for play order as defined in GIS-351.
    *       6.   The item identified by the Tag must currently exist in the list. If it does not exist, the MediaPlayer shall respond
    *            with an ErrorAck(ErrorCode = 0xC8 ERR_MP_INVALID_INDEX with ErrorInfo = the original invalid Tag received in the StartResultAck.
    *
    * @param[out] listSize number of elements in the list
    * @param[in] listID id of the list which should be used for playback
    * @param[in] objectID object ID to play
    * @return < 0: error, = 0: OK
    */
    tResult PlayItemFromListByTag(
        tListSize &listSize,
        const tListID listID,
        const tObjectID objectID);

    /**
    * The PlayMediaPlayerObject method is called to set the play object within the MediaPlayer. The MediaPlayer handles this function as follows:
    *       1.   If the MediaPlayer is not currently playing audio, it internally stores the new object as the object
    *            to play the next time that the MediaPlayer is connected for audio playback.
    *       2.   If the MediaPlayer is already connected and playing audio on another media object, it stops playback of that media object
    *            and starts playing the new object automatically
    *       3.   Note that the connect operation to the AV-Managmenet of the system is not performed by the MediaPlayer and
    *            is instead performed by the HMI controller. For example, if the user directly selects an Artist from an HMI menu,
    *            the HMI controller would first call PlayMediaPlayerObject and then would ask the AV-Management to connect the MediaPlayer over a Logical Audio Channel.
    *
     * @param[in] objectID object ID to play
     * @param[in] playContinuation  decides what to do on end of list
     * @param[out] listID reference to the parameter containing the listID of the created list
     * @param[out] listSize reference to the parameter containing the listSize of the created list
     * @return < 0: error, = 0: OK
     */
    tResult PlayMediaPlayerObject(tListID &listID,
        tListSize &listSize,
        const tObjectID objectID,
        const tPlayContinuation playContinuation);

    /**
    * This method returns all information about the currently playing item:
    *       1   ListHandle of the active playlist.
    *       2   Position within the list of the active item
    *       3   The Tag value of the active item
    *       4   Metadata for the active item
    *
    * @return < 0: error, = 0: OK
    */
    tResult GetNowPlaying(
        tNowPlaying &nowPlaying);

    /**
    *   Reports the current folder base path when the MediaPlayer is playing from a folder list.
    *   When the MediaPlayer is not playing from a file list, the ListHandle parameter shall be set to NULL = 0x00000000 and the CurrentFolderPath
    *   parameter shall be set to a zero-length null string.
    * @param[out] currentFolderPath Current folder
    * @param[out] listID Current list ID
    * @return < 0: error, = 0: OK
    */
    tResult GetCurrentFolderPath(tPath &currentFolderPath, tListID& listID);

    /**
    * Returns the amount of time elapsed and total duration of a media object. Resolution is in seconds.
    * The MediaPlayer informs the HMI about a change by calling the function OutputWrapper::UpdatePlaytime.
    *
    * @param[out] elapsedPlaytime elapsed playtime of current media object
    * @param[out] totalPlaytime total playtime of current media object
    * @param[out] objectID ID of the media object to which the param1 and param2 correspond to //Note:objectID is reliable only for Mass Storage devices & MTP Devices
    * @return < 0: error, = 0: OK
    */
    tResult GetPlaytime(
        tPlaytime &elapsedPlaytime,
        tPlaytime &totalPlaytime,
        tObjectID &objectID);

    /**
    * The current playback state of the MediaPlayer
    *        * PBA_PLAY: The Media Player is actively playing a media object. The object being played can be obtained using the GetNowPlaying function
    *        * PBA_PAUSE: The Media Player is not actively playing a media object. The item to queued to play can be obtained using the GetNowPlaying function
    *        * PBA_STOP: The Media Player has stopped playback. The GetNowPlaying function does not return an item
    *        * PBA_FFWD_START: The Media Player is fast forwarding.
    *        * PBA_FREV_START: The Media Player is fast reversing.
    *
    * @param[out] playbackState external (HMI) playback state
    * @return < 0: error, = 0: OK
     */
    tResult GetPlaybackState(
        tHMIPlaybackState &playbackState);


    /*
     * This function activates a previous stored mediaplayer favorite.
     * Therefore a playlist will be created and playback started.
     * In case the mediaplayer has no active audio route a RequestAvActivation will be generated
     * The parameter AudioChn and VideoChn are currently ignored and the eMAIN_AUDIO channel is used for RequestAvActivation.
     *
     * Note that the connect operation to the AV-Managmenet of the system is performed by the MediaPlayer by calling the
     * method OutputWarpper::RequestAVActivation. If this method is not overwritten correctly by the controller the HMI hats to ask the AV-Management
     * to connect the MediaPlayer
     *
     * @param[in] favID identifier of the favorite to play
     * @param[in] audioChn defines the AudioChannel to be used for RequestAvActivation
     * @param[in] videoChn defines the VideoChannel to be used for RequestAvActivation
     * @param[out] retFavID identifier of the activated favorite
     * @return < 0: error, = 0: OK
     */
    tResult ActivateFavorite(
        tFavoriteID &retFavID,
        const tFavoriteID favID,
        const tLogicalAVChannel audioChn,
        const tLogicalAVChannel videoChn);

    /**
     * This function stores the given ObjectID under a free FavoritID
     *
     * @param[in] objectID identifier of the favorite object to be stored
     * @param[out] retFavID favorite identifier of the stored object
     * @return < 0: error, = 0: OK
     */
    tResult StoreFavorite(
        tFavoriteID &retFavID,
        const tObjectID objectID);

    /**
     * This function deletes the favorite identified by the FavoriteID parameter
     *
     * @param[in] favID identifies the favorite to be deleted
     * @param[out] retFavID favorite identifier of the deleted object
     * @return < 0: error, = 0: OK
     */
    tResult DeleteFavorite(
        tFavoriteID &retFavID,
        const tFavoriteID favID);

    /*
     * This function gives information about the favorite identified by the FavoriteID parameter.
     *
     * @param[in] favID identifies the favorite which information has to be returned
     * @param[out] favInfo information struct to the requested favorite
     * @return < 0: error, = 0: OK
     */
    tResult GetFavoriteInfo(
        tFavoriteInfo &favInfo,
        const tFavoriteID favID);

    /*
     * This function gives information about all the favorites stored in the MediaPlayer.
     *
     * @param[out] favInfo list of all the favorite items stored .
     *
     * @return < 0: error, = 0: OK
     */
     tResult GetAllFavoriteInfo(vector<tFavoriteInfo> &favInfo);

    /*
     * The ResetFavorites function causes the Favorite Provider to remove all favorites from its Stored Favorites
     * and then rebuild its Stored Favorites with its Default Favorites.
     * This function returns NumStoredFavorites which is the number of favorites in the Stored Favorites.
     *
     * @return < 0: error, = 0: OK
     */
    tResult ResetFavorites(void);

    /**
     * Triggers the clearance of the complete database
     *
     * @return < 0: error, = 0: OK
     */
    tResult ClearMediaPlayerData(void);

    /**
    Sets the current language
    * @param[in] language current language
    * @return < 0: error, = 0: OK
    */
    tResult SetLanguage(const tLanguageType language);

    /**
    Gets the current language
    * @param[OUT] language current language
    * @return < 0: error, = 0: OK
    */
    tResult GetLanguage(tLanguageType &language);
    /** Sets current outside temperature. Store value in DataProvider and inform ipodcontrol
    * @param[in] language current language
    * @return < 0: error, = 0: OK
    */
    tResult SetOutsideTemperature(const tOutsideTemperature outsideTemperature);

    /** Sets reason for stopping carplay playback, see topic "HID play" and Carplay TAKE/BORROW certification issue
    * @param[in] diPOStopReason
    * @return < 0: error, = 0: OK
    */
    tResult SetDiPOStopReason(const tDiPOStopReason diPOStopReason);

    /**
     * Method must be called by controller if a change in the device list is detected
     * Roadmap 13024
     * if (undervoltage == TRUE && m_CurrentState != UNDERVOLTAGE)
     * - Get allocate state and alsa name from PlayerManager -> GetAllocateState(&allocateState, &alsaName).
     * - Store allocate state and alsa name in local member variables and set m_CurrentState to UNDERVOLTAGE.
     * - Trigger state change to UNDERVOLTAGE at LocalSPM -> StateChangeUndervoltage(TRUE).
     * else if (undervoltage == FALSE && m_CurrentState == UNDERVOLTAGE)
     * - Set m_CurrentState to NORMAL
     * - Trigger state change to NORMAL again at LocalSPM -> StateChangeUndervoltage(FALSE).
     * - Set allocate state and alsa name to PlayerManager -> RestoreAllocateState(m_AllocateState, m_AudioOutputDevice).
     * Call DBManager::DeviceChanged(deviceProperties)
     *
     * @param[in] undervoltage undervoltage flag
     * @param[in] deviceInfos list with all attached devices
     * @return < 0: error, = 0: OK
    */
    //tResult DeviceChanged(const tUndervoltage undervoltage, const vector<tDeviceProperty> deviceProperties); //Roadmap 13024: 100%
    tResult DeviceChanged(const tUndervoltage undervoltage, const vector<tDeviceInfo> deviceInfos); //Roadmap 13024: 100%

    tResult UpdateVRSessionStatus(IN tVRSessionStatus status,tSiriStatus siristatus,tVRSupportStatus vrsupport);

    /**
     * Method is called by Controller to transfer tags to a customer apple device
     *
     * @param[in] deviceTag Device ID
     * @param[in] transferTags Tags to transfer to the apple device
     * @param[out] transferStatus status of the transfer
     * @param[out] untransferredTags tags which was not transferred
     * @return < 0: error, = 0: OK
    */
    tResult TransferTags(const tDeviceID deviceTag, const vector<tTagTransfer> transferTags, tTagTransferStatus &transferStatus, vector<tUntransferredTag> &untransferredTags);


    /**
     * The AppControl Connect method provides the interface for the Internet Radio to tell the
     * iPod device control to connect to the Pandora application using the MediaPlayer.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] protocol a string containing protocol of the IAP application.
     * @param[in] bundleSeedID a string containing the bundle seed ID of the IAP application.
     * @param[in] appName a string containing App name of the IAP application.
     * @param[out] sessionID The session ID of the connected IAP application.
     * @return < 0: error, = 0: OK
    */
    tResult SendAppControlConnect(const tMountPoint mountPoint, const tProtocolName protocol, const tBundleSeedID bundleSeedID, const tAppName appName, const tAppLaunchOption launchOption);

    /**
     * The AppControl Command method provides the interface for the Internet Radio to send
     * an IAP command to the MediaPlayer.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] appName a string containing App name of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @param[in] sizeOfBuffer size of the command buffer
     * @param[in] commandBuffer IAP command.
     * @return < 0: error, = 0: OK
    */
    tResult SendAppControlCommand(const tMountPoint mountPoint, const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer, const tCommandBufferPtr commandBuffer, const tUserContext userContext);

    /**
     * The AppControl Close method provides the interface for the Internet Radio to send an IAP
     * command to the MediaPlayer to terminate the IAP session with the Pandora application.
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] appName a string containing App name of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @return < 0: error, = 0: OK
    */
    tResult SendAppControlClose(const tMountPoint mountPoint, const tAppName appName, const tSessionID sessionID);

    /**
     * This function can be called to start audio on apple iAP2 devices. The function is called by Pandora application
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @return < 0: error, = 0: OK
    */
    tResult StartAudio(const tMountPoint mountPoint);

    /**
     * This function can be called to stop audio on apple iAP2 devices. The function is called by Pandora application
     *
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @return < 0: error, = 0: OK
    */
    tResult StopAudio(const tMountPoint mountPoint);


    /**
     * Function triggers an unmount for a device id which is done by the special device control
     *
     * @param[in] deviceID device for which the unmount is to be done.
     * @return != 0: error, = 0: OK
     */
    tResult Umount(tErrorCode &errorCode, const tDeviceID deviceID); //Roadmap 13003

    /**
     * Function plays the song specified with filename on the first found device of type deviceType
     *
     * @param[in] deviceType type of device
     * @param[in] filename  path and filename of song
     * @return != 0: error, = 0: OK
     */
    tResult DiagnosisPlaySong(tDeviceType deviceType, tFilename filename);

    /**
     * Function triggers an reindexing for a device id
     *
     * @param[in] deviceID device for which the reindexing is to be done.
     * @return != 0: error, = 0: OK
     */
    tResult Reindexing(const tDeviceID deviceID);

    /**
     * Function creates an internal playlist
     *
     * @param[in] deviceID device
     * @param[in] playlistName name of the playlist
     * @param[out] playlistPath path including filename of the created playlist.
     * @return != 0: error, = 0: OK
     */
    tResult CreateInternalFileNamePlaylist(tPath &playlistPath, const tDeviceID deviceTag, const tPlaylistName playlistName);

    /**
     * R8976_UC3_LLD3.1_SEQ2
     * Function adds a filenames to an  internal playlist
     *
     * @param[in] playlistPath path including filename of the internal playlist
     * @param[in] deviceID device
     * @param[in] filenamePathVector vector of urls
     * @param[in] positionVector Position vector in playlist for this new item
     * @return != 0: error, = 0: OK
     */
    tResult AddFileNamesToPlaylist(const tPath playlistPath,const tDeviceID deviceTag, const vector<string> filenamePathVector, const vector<tPosition> positionVector);

    /**
     * R8976_UC3_LLD3.1_SEQ3
     * Function deletes  entries from an  internal playlist
     * @param[in] playlistPath path including filename of the internal playlist
     * @param[in] deviceID device
     * @param[in] positions Position in playlist

     * @return != 0: error, = 0: OK
     */
    tResult DeleteEntriesFromPlaylist( const tPath playlistPath, const tDeviceID deviceTag, const vector<tPosition> positionVector);

    /**
     * R8976_UC3_LLD3.3_SEQ4
     * Function renames an internal playlist
     * @param[in] deviceID device
     * @param[in] playlistPath path including filename of the internal playlist
     * @param[in] playlistName New name of the playlist
     * @return != 0: error, = 0: OK
     */
    tResult RenameInternalPlaylist(const tDeviceID deviceTag, const tPath playlistPath,  const tPlaylistName playlistName);

    /**
     * R8976_UC3_LLD3.2
     * Function deletes an internal playlist
     * @param[in] deviceID device
     * @param[in] playlistPath path including filename of the internal playlist
     * @return != 0: error, = 0: OK
     */

    tResult DeleteInternalPlaylist(const tDeviceID deviceTag, const tPath playlistPath);

    /**
     * Function creates a list with internal playlists
     *
     * @param[in] deviceTag Device id for which the playlist list shall be created. ID = 0 all internal playlists
     * @param[out] listID ID of the list
     * @param[out] listID Size of the list
     * @return != 0: error, = 0: OK
     */
    tResult CreateMediaPlayerInternalPlaylistList(tListID& listID, tListSize& listSize, const tDeviceID deviceTag);

    /**
    * The function GetListPositionOfObject searches the position (index) of a media object in a list
    * @param[in] listID ID of the list which should be used
    * @param[in] objectID object ID to search
    * @return < 0: error, = 0: OK
    */
    tResult GetListPositionOfObject(tPosition &position, const tListID listID, const tObjectID objectID);

    /**
    * The function StartSlideshow starts the slideshow of the list at the image specified by objectID
    * @param[in] listID id of the list which should be used for slideshow
    * @param[in] objectID object ID to play
    * @return < 0: error, = 0: OK
    */
    tResult StartSlideshow(
        const tListID listID,
        const tObjectID objectID);

    /**
    * Requests the slideshow action for the currently shown image list. Playback actions include:
    *      * Previous or next track
    *      * Fast forward or fast reverse are not supported
    *      * Play, pause, or stop
    *
    * @param[in] playbackAction enum of playback action to set
    * @param[in] nextPrevSkipCount skip count for NEXT and PREV to skip more than one object with one action
    * @return < 0: error, = 0: OK
    */
    tResult SendSlideshowAction(
        const tPlaybackAction playbackAction,
        const tNextPrevSkipCount nextPrevSkipCount=1);

    /**
    * Method returns the slideshow time
    *
     * @param[out] slideshowTime slideshow time
     * @return < 0: error, = 0: OK
     */
    tResult GetSlideshowTime(tSlideshowTime &slideshowTime);

    /**
     * Methods sets the slideshow time
     * Time in seconds when the GMP shows the next image.
     * If -1 is set the GMP does not show the next image automatically.
     * @param[in] slideshowTime slideshow time
     * @return < 0: error, = 0: OK
     */
    tResult SetSlideshowTime(const tSlideshowTime slideshowTime);

    /**
     *  This functions returns  all information about the currently showing item.
     *  The HMI has to use this information to load, if necessary to convert and to show the current image.
     *  Additionally, the next image is delivered to the HMI to realize a preloading mechanism
     * @param[in] nowShowing nowshowing information
     * @return < 0: error, = 0: OK
     */
    tResult GetNowShowing(tNowShowing &nowShowing);

    /**
    * The current slideshow state of the MediaPlayer
    *        * PBA_PLAY: The Media Player is showing a slideshow.
    *        * PBA_PAUSE: The slideshow is in state pause
    *        * PBA_STOP: The slideshow is in state stop
    *
    * @param[out] playbackState external (HMI) slideshow state
    * @return < 0: error, = 0: OK
     */
    tResult GetSlideshowState(tHMIPlaybackState &playbackState);

    /**
    * MediaPlayer is informed when a device is used as DiPO device.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPOActive "True" if DiPO session is active. "False" if DiPO session is not active
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult DiPOActiveDevice(const tDeviceID deviceID, const tDiPOActive diPOActive, const tUserContext userContext);

    /**
    * Requests a USB role switch: From USB client to USB host or vice versa
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPORoleStatus Role status
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult DiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext,const tAppInfo appInfo);

    tResult DiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext);

    /**
    * Transfers GPS data to DiPO device
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] USBSerialNumber of the apple device.
    * @param[in] diPOGPGGAData GPS data in GPGGA format as specified by apple
    * @param[in] diPOGPRMCData GPS data in GPRMC format as specified by apple
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult DiPOTransferGPSData(tDeviceID deviceID, const tMountPoint USBSerialNumber, const tDiPOGPGGAData diPOGPGGAData,const tDiPOGPRMCData diPOGPRMCData, const tDiPOGPGSVData diPOGPGSVData, const tDiPOGPHDTData diPOGPHDTData, const tUserContext userContext);

    /**
    * Transfers GPS data to DiPO device
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] USBSerialNumber of the apple device.
    * @param[in] diPOPASCDData Vehicle speed data arrang in the format as specified by apple
    * @param[in] diPOPAGCDData vehicle gyro data arrang in the format as specified by apple
    * @param[in] userContext user context structure. Is transmitted to OutputWrapper without changes
    * @return < 0: error, = 0: OK
    */
    tResult DiPOTransferDRData(tDeviceID deviceID, const tMountPoint USBSerialNumber, const tDiPOPASCDData diPOPASCDData, const tDiPOPAGCDData diPOPAGCDData, const tDiPOPAACDData diPOPAACDData,const tUserContext userContext);

    /**
    * Informs MediaPlayer core if role switch is required
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] diPOSwitchReqResponse role switch required or not
    * @return < 0: error, = 0: OK
    */
    tResult DiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse);
    tResult DiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse,const tAppInfo appInfo);

    /**
    * Informs MediaPlayer core if SPI requests an audio device.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @return < 0: error, = 0: OK
    */
    tResult DiPORequestAudioDevice(const tDeviceID deviceID);

    /**
    * Informs MediaPlayer core if SPI releases an audio device.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @return < 0: error, = 0: OK
    */
    tResult DiPOReleaseAudioDevice(const tDeviceID deviceID);

    /**
    * Requests MediaPlayer core to call playback action for specific deviceID.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @param[in] playbackAction enum of playback action to set
    * @param[in] nextPrevSkipCount skip count for NEXT and PREV to skip more than one object with one action
    * @return < 0: error, = 0: OK
    */
    tResult DiPORequestPlaybackAction(
            const tDeviceID deviceID,
            const tPlaybackAction playbackAction,
            const tNextPrevSkipCount nextPrevSkipCount);

    /**
    * Request to start iAP2 session over wireless Carplay
    * @param[in] BT Mac Address of Apple device
    */
    tResult DiPOStartIAP2OverCarplayWiFi(const tGeneralString btMacAddress);

    /**
    * Request to Set iAP2 CallControlsConfiguration in DB.
    * @param[in] CallControls configuration .
    */
    tResult DiPOSetCallControlsConfiguration(const tDiPOCallControlsConfiguration callControlsConfiguration);

    /**
    * Request to Initiate call over iAP2
    * @param[in] USBSerialNumber serial number of the device.
    * @param[in] initiateCall parameters.
    */
    tResult DiPOSendInitiateCall(const tMountPoint USBSerialNumber, const tDiPOInitiateCall initiateCall);

    /**
    * Request to Accept call over iAP2
    * @param[in] USBSerialNumber serial number of the device.
    * @param[in] acceptCall parameters.
    */
    tResult DiPOSendAcceptCall(const tMountPoint USBSerialNumber, const tDiPOAcceptCall acceptCall);

    /**
    * Request to End call over iAP2
    * @param[in] USBSerialNumber serial number of the device.
    * @param[in] endCall parameters.
    */
    tResult DiPOSendEndCall(const tMountPoint USBSerialNumber, const tDiPOEndCall endCall);

    /**
    * Request to swap calls over iAP2
    * @param[in] deviceID Unique identifier of the device.
    */
    tResult SendSwapCalls(const tDeviceID deviceID);

    /**
    * Request to swap calls over iAP2
    * @param[in] USBSerialNumber serial number of the device.
    */
    tResult DiPOSendSwapCalls(const tMountPoint USBSerialNumber);

    /**
    * Informs MediaPlayer core if SPI has released  audio device.
    *
    * @param[in] deviceID Unique identifier of the device.
    * @return < 0: error, = 0: OK
    */
    tResult StopMediaPlaybackResult(const tDeviceID deviceID);

    /**
    * Informs MediaPlayer core about RequestAVActivation result.
    *
    * @param[in] success success of request
    * @return < 0: error, = 0: OK
    */
    tResult RequestAVActivationResult(const tSuccess success);

    /**
    * MyMediaSupported by GMP.
    *
    * @return true or false
    */
    tBoolean MyMediaSupported();


    /**
     * Methods sets the BT address of the device.
     * @param[in] address BT address
     * @return < 0: error, = 0: OK
     */
    tResult SetVehicleBTAddress(const tBTMACAddress address);


    /**
     * Method forwards eject received e.g. from HMI to components VD_DVM
     * @return < 0: error, = 0: OK
     */
    tResult SendEjectOpticalDisc(tDMOpticalDiscEjectCmd opticalDiscEjectCmd);//Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * Method forwards result of eject command
     * @param[in] entOpticalDiscEjectMethodResult: result of eject cmd which has been send before
     * @return < 0: error, = 0: OK
     */
    tResult SendOpticalDiscEjectMethodResult(tDMOpticalDiscEjectMethodResult entOpticalDiscEjectMethodResult); //Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * Method to check if Disc drive is Kenwood(DVD Drive) or MASCA
     * @return true or false
     */
    bool IsDVDDrive();

    /**
     * Method used by clientHandler_devicemanager to notifiy changed state of optical disc e.g. injected inserted etc.
     * @return < 0: error, = 0: OK
     */
    tResult OpticalDiscStatusChanged(const tDMOpticalDiscSlotState opticalDiscSlotState); //Roadmap 15002_CDAudio_DetectionAndDiagnosis

     /**
     * Methods informs the PlayerManager about nelwy requested scanmode by the user
     * for the currently active device.
     * @param[in] ScanMode,which is true/false
     * @return < 0: error, = 0: OK
     */
    tResult SendScanMode(const tScanMode scanMode);

     /**
     * Methods returns the current scanmode from PlayerManager.
     * @param[out] current ScanMode value,which is true/false
     * @return < 0: error, = 0: OK
     */
    tResult GetScanMode(tScanMode &scanMode);

    /**
    * This method creates a list of all folders having at least one image in it.This is a root list i.e without any folder hierarchy.
    * The CreateMediaPlayerIndexedImageFolderList function assigns a unique
    * ListHandle for each filtered list created and returns this value. A list is considered to be
    * valid until the ReleaseMediaPlayerList function is executed. Changes to a
    * list are reported by OutputWrapper::SendMediaPlayerListChanged.
    * This method also returns the total number of Folders
    * in the associated List. These numbers are static for each List at creation time and
    * are included as returned parameters to allow HMI controllers to configure their view accordingly
    * without requiring the controller to pull the entire list and iterate it first.
    * @param[in] deviceID id of the desired USB stick or MyMedia
    * @param[in] listType, Type of the list to be created
    * @param[out] listId id of the created list
    * @param[out] listSize length of the list
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerIndexedImageFolderList(
            tListID &listID,
            tListSize &listSize,
            const tListType listType,
            const tDeviceID deviceID);
    /**
    * This method creates a list of all images inside a given folder.
    * The CreateMediaPlayerIndexedImageFolderItemList function assigns a unique
    * ListHandle for each filtered list created and returns this value. A list is considered to be
    * valid until the ReleaseMediaPlayerList function is executed. Changes to a
    * list are reported by OutputWrapper::SendMediaPlayerListChanged.
    * This method also returns the total number of images
    * in the associated List. These numbers are static for each List at creation time and
    * are included as returned parameters to allow HMI controllers to configure their view accordingly
    * without requiring the controller to pull the entire list and iterate it first.
    * @param[in] deviceID id of the desired USB stick or MyMedia
    * @param[in] listType, Type of the list to be created
    * @param[out] listId id of the created list
    * @param[out] listSize length of the list
    * @return < 0: error, = 0: OK
    */
    tResult CreateMediaPlayerIndexedImageFolderItemList(
        tListID &listId,
        tListSize &listSize,
        const tPath path,
        const tDeviceID deviceID);

#if USE_IMAGE_OBJECT
    /**
    * The RequestMediaPlayerIndexedImageFolderListSlice method provides a slice of the filtered list specified by the ListHandle parameter.
    * The list must be created either with method CreateMediaPlayerIndexedImageFolderList or method CreateMediaPlayerIndexedImageFolderItemList.
    * The slice begins with the item at location WindowStart and includes in sequential order the number of items specified by the parameter WindowSize.
    * The RequestListSlice method returns the slice as a ImageObjects vector.
    * The data contained in each MediaObject is used by the HMI to determine the meta-data and other properties of the image media object.
    * If the number of items in the list starting at WindowStart is less than the WindowSize parameter then the number of items
    * provided in the slice will be less than the WindowSize calling parameter.
    *
    * @param[in] listID if for which the slice is requested
    * @param[in] startIndex begin of the slice in the list
    * @param[in] sliceSize size of the slice
    * @param[out] imageObjectVector vector with found objects for list slice, the vector must be allocated by caller.
    * @return < 0: error, = 0: OK
    */
    tResult RequestMediaPlayerIndexedImageFolderListSlice(
        vector<tImageObject> &ImageObjectVector,
        const tListID listID,
        const tIndex startIndex,
        const tIndex sliceSize);
#endif

    /**
     * Notify CD info will notify the clients regarding changes in CD Drive Version and CD Device State
     * @param[in] trOpticalDiskCDInfo OpticalDiskCDInfo is a Structure for Drive Version and Device State
     * @return < 0: error, = 0: OK
    */
    tResult DeviceManagerNotifyCDInfo(IN trOpticalDiskCDInfo OpticalDiskCDInfo);

    /**
      * This function to get current CD Info of optical disc
      *
     * @param[out] trOpticalDiskCDInfo OpticalDiskCDInfo is a Structure for Drive Version and Device State
     * @return < 0: error, = 0: OK
      */
    tResult GetOpticalDiscCDInfo(OUT trOpticalDiskCDInfo &OpticalDiskCDInfo); // CD Info

    //To Dump the hex values of the mediaobject's metadata entries
    void HexDumpMediaObjectMetadataFields(tMediaObject &mediaObject,FILE* codesetConvLogFile);


    /**
     * Roadmap 15027 : Adjust Brightness of Video
     * Method to set the video property of the video objects in all devices.
     * Properties like brightness, hue, saturation, contrast, brightness offset , hue offset, saturation offset can be set using this function interface.
     * Properties set will be available for all video objects  for all devices.
     * @param[in] video  property type enum
     * @param[in] video  property value of the corresponding video property type
     * @return < 0: error, = 0: OK
     */
    tResult SendVideoProperty(IN const tVideoProperty videoProperty, IN const tPropertyValue propertyValue);


    /**
     * Roadmap 15027 : Adjust Brightness of Video
     * Methods to get the currently set video brightness property of the video objects.
     * @param[out] current video brightness value
     * @return < 0: error, = 0: OK
     */
    tResult GetVideoBrightness(OUT tVideoBrightness &videoBrightness);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video hue property of the video objects.
      * @param[out] current video hue value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoHue(OUT tVideoHue &videoHue);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video saturation property of the video objects.
      * @param[out] current video saturation value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoSaturation(OUT tVideoSaturation &videoSaturation);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video contrast property of the video objects.
      * @param[out] current video contrast value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoContrast(OUT tVideoContrast &videoContrast);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video brightness offset property of the video objects.
      * @param[out] current video brightness offset value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoBrightnessOffset(OUT tVideoBrightnessOffset &videoBrightnessOffset);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video hue offset property of the video objects.
      * @param[out] current video hue offset value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoHueOffset(OUT tVideoHueOffset &videoHueOffset);

    /**
      * Roadmap 15027 : Adjust Brightness of Video
      * Methods to get the currently set video saturation offset property of the video objects.
      * @param[out] current video saturation offset value
      * @return < 0: error, = 0: OK
      */
    tResult GetVideoSaturationOffset(OUT tVideoSaturationOffset &videoSaturationOffset);

    /**
    * This function informs PlayerManager to Clear the lastmode pause for the given DeviceID.
    * @param[in] tDeviceID Device ID
    * @return < 0: error, = 0: OK
    */
    tResult ClearLastModePause (const tDeviceID deviceID);

    /**
     * This function informs IpodControl the Accessory WiFi Credentials to be shared with Apple device.
     * @param[in] tWiFiAPCredentials wifiCredentials
     * @return <0: error, =0: OK
     */
    tResult GetWiFiCredentialsResult(const tDeviceID deviceId, tWiFiAPCredentials wifiCredentials, bool bError);
    /**
    * This method returns the Power update (iAP2 connections, host only) from IPODControl.
    * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPOPower.
    *
    * @param[out] deviceID device ID
    * @param[out] Power Power
    * @return < 0: error, = 0: OK
    */
    tResult GetDiPOPower(tDeviceID &deviceID, tDiPOPower &Power);

    //>--Roadmap 16003 : 'CD Ripping With Gracenote'
    /**
      * R8976_UC3_LLD3.5_SEQ5
      * Roadmap 16003 : CD Ripping With Gracenote
      * Method to reorder the position of already created internal playlist.
      * @param[in] Playlist path
      * @param[in] Device tag
      * @param[in] old position in the playlist
      * @param[in] new position in the playlist
      * @return < 0: error, = 0: OK
      */
    tResult ReorderInternalPlaylist(IN const tPath &PlaylistPath , IN const tDeviceID deviceID, IN const tPosition oldPosition,IN const tPosition newPosition);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] listType list type to create play list filtered after object ID
     *                     see special behavior described above for all songs list.
     * @param[in] tag1,tag2,tag3     three filter tag object id's to generate the complete list
     *                     which ID is in which tag position depends on the list type.
     *                     max 3 different tags are possible
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(IN const tListType listType, const tFilterTag1 tag1, const tFilterTag2 tag2, const tFilterTag3 tag3, const tFilterTag4 tag4, const tDeviceID deviceID);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] mediaObjectIDVector     vector of mediaobjectids
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(IN const vector<tObjectID> mediaObjectIDVector, const tDeviceID deviceID);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.1_SEQ7
     * Method to delete media content
     * @param[in] mediaObjectUrlVector     vector of mediaobject urls
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @return < 0: error, = 0: OK
     */
    tResult DeleteMediaContent(IN const vector<string> mediaObjectUrlVector, const tDeviceID deviceID);


    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8972_UC3_LLD4.3_SEQ6
     * Method to edit the metadata of media object
     * @param[in] tEditMetaDataByID struct with the mediaobjectid and metadata
     * @return < 0: error, = 0: OK
     */
    tResult EditMetaData(IN tEditMetaDataByID editMetaDataByID);


   /**
    * Roadmap 16003 : CD Ripping With Gracenote
    * R8972_UC3_LLD4.3_SEQ6
    * Method to edit the metadata of media object
    * @param[in] tEditMetaDataByUrl struct with the mediaobjecturl and metadata
    * @return < 0: error, = 0: OK
    */
    tResult EditMetaData(IN tEditMetaDataByUrl editMetaDataByUrl);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC1_LLD2.1
     * Method to get the ecoding quality used for recording the ripped files
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult GetRippingEncodingQuality(OUT tEncodingQuality &rippingEncodingQuality);

    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R6392_UC1_LLD2.1
     * Method to set the ecoding quality used for recording the ripped files
     * @param[in] rippingEncodingQuality quality of the encoding used
     * @return < 0: error, = 0: OK
     */
    tResult SetRippingEncodingQuality(IN tEncodingQuality rippingEncodingQuality);



    /**
     * Roadmap 16003 : CD Ripping With Gracenote
     * R8976_UC3_LLD3.1_SEQ2
     * Method to add filenames to internal playlist
     * @param[in] listType list type to create play list filtered after object ID
     *                     see special behavior described above for all songs list.
     * @param[in] tag1,tag2,tag3     three filter tag object id's to generate the complete list
     *                     which ID is in which tag position depends on the list type.
     *                     max 3 different tags are possible
     * @param[in] playlistPath path including filename of the internal playlist
     * @param[in] deviceID id of the device for special filtered list depending on a device
     * @return < 0: error, = 0: OK
     */
    tResult AddFileNamesToPlaylistByTag(IN const tListType listType, const tFilterTag1 tag1, const tFilterTag2 tag2, const tFilterTag3 tag3, const tFilterTag4 tag4, const tPath playlistPath, const tDeviceID deviceID);




    //<--Roadmap 16003 : 'CD Ripping With Gracenote'


    /**
     * Get DVD device error message
     * @param[out] tDeviceID Device ID
     * @return < 0: error, = 0: OK
     */
    tResult GetDiskError(OUT tBool& isError, OUT tU8 & diskError);

    /**
     * Function get mechanical Info for Dvd (mechanicalInfo : FID 0x255)
     *
     * @param[OUT] error , mechanical Info
     * @return != 0: error, = 0: OK
     */
    tResult GetDiskMechanicalInfo(OUT tU8& mechanicalInfo);

    /**
     * Function get skip availability info for (FID 0x256)
     *
     * @param[OUT] ,skip
     * @return != 0: error, = 0: OK
     */
    tResult GetDVDDriveInfo(OUT tU8& discType, OUT tU8& driveMode);
    /**
     * Function get disc type and drive mode info for (FID 0x275)
     *
     * @param[OUT] ,discType,driveMode
     * @return != 0: error, = 0: OK
     */
    tResult GetSkipInfo(OUT tBool& skip);
    /**
     * Function get DVDMenuPlaybackOngoing Info for Dvd (FID 0x274)
     *
     * @param[OUT] ,dvdMenu
     * @return != 0: error, = 0: OK
     */
    tResult GetDVDMenuPlaybackOngoing(OUT tBool& dvdMenu);
    /**
     * Function Get Dvd Direct Select Info for  (Skip info : FID 0x257)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetDirectSelectInfo(OUT tBool& dirceSelect);

    /**
     * Function get dvd dirct select Info for Dvd (Skip info : FID 0x258)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetAngleInfo(OUT tU8& angleTrack, tU8& totalAngle);
    /**
     * Function Get Dvd Subtitle Info for Dvd (Skip info : FID 0x259)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetSubtitleInfo(OUT tBool& state, OUT tU8& current, tU8& total ,OUT tU16& subLang);

    /**
     * Function Get Dvd Subtitle Info for Dvd (Skip info : FID 0x259)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetAudioChannelInfo(OUT tU8& totalChannels, OUT tBool& subWoofer, tU8& Assignment);
    /**
     * Function Get Set diplay mode setting (Skip info : FID 0x262)
     *
     * @param[OUT] , display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetDisplayMode(OUT tU8& displayMode);
    /**
     * Function Get Set diplay mode setting  (Skip info : FID 0x262)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult SetDisplayMode(IN tU8 displayMode);
    /**
     * Function Get Play Status Setting   (Skip info : FID 0x265)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetPlayStatusSetting(OUT tBool& playStatus);
    /**
     * Function Get Play Status Setting (Skip info : FID 0x265)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult SetPlayStatusSetting(IN tBool playStatus);
    /**
     * Function Get Dvd DRC Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetDRCSetting(OUT tU8& drc);
    /**
     * Function Get Dvd DRC Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetDRCSetting(IN tU8 drc);
    /**
     * Function Get Current AudioInfo Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetCurrentAudioInfo(OUT tU8& soundformat, OUT tU8& VcdAudio,
            OUT tU8& DvdAudio,OUT tU8& totalOutput ,OUT tU16& AudioLang);
    /**
     * Function SetAngleMarkSetting(Skip info : FID 0x264)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetAngleMarkSetting(IN tBool AngleMark);
    /**
     * Function GetAngleMarkSetting(Skip info : FID 0x264)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetAngleMarkSetting(OUT tBool& AngleMark);
    /**
     * Function Set Current Language Setting(Skip info : FID 0x263)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetLanguageSetting(IN tU16 Audio, IN tU16 SubTitel, IN tU16 Menu);
    /**
     * Function Get Current Language Setting(Skip info : FID 0x263)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetLanguageSetting(OUT tU16& Audio, OUT tU16& SubTitel, OUT tU16& Menu);
    /**
     * This method  RequestDiscOperation used to ejct dvd drive
     * is used to Eject or Load DVD
     * @param[in] Operation drive Eject = 1 Load = 0
     * @return < 0: error, = 0: OK
     */
    tResult RequestDiscOperation(tU8 Operation);
    /**
     * This method  RequestVCDAudioChannel
     * is used to SelectionType and AudioMode
     * @param[in] SelectionType , AudioMode
     * @return < 0: error, = 0: OK
     */
    tResult RequestVCDAudioChannel(tU8 SelectionType, tU8 AudioMode);
    /**
     * This method  RequestDVDAudioChannel
     * is used to SelectionType and AudioMode
     * @param[in] SelectionType and AudioMode
     * @return < 0: error, = 0: OK
     */
    tResult RequestDVDAudioChannel(tU8 SelectionType, tU8 AudioMode);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] Action
     * @return < 0: error, = 0: OK
     */
    tResult SendNavigationKey(tU8 Action);
    /**
     * This method  SendTouchCommand used to set Touch Actions
     * @param[in] Action,Xcod,Ycod,maxXcod,maxYcod
     * @return < 0: error, = 0: OK
     */
    tResult SendTouchCommand(tU8 Action, tU16 Xcod, tU16 Ycod, tU16 maxXcod, tU16 maxYcod);

    /**
     * This method  RequestDirectSearch used direct search
     * @param[in] titel , chapter
     * @return < 0: error, = 0: OK
     */
    tResult RequestDirectSearch(tU32 titel, tU32 chapter);

    /**
     * This method  RequestAngleChange
     * @param[in] SelectionType , Angle
     * @return < 0: error, = 0: OK
     */
    tResult RequestAngleChange(tU8 SelectionType, tU8 Angle);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] MenuControl
     * @return < 0: error, = 0: OK
     */
    tResult SetMenuPlayBackControl(tU8 MenuControl);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] number
     * @return < 0: error, = 0: OK
     */
    tResult SelectDirectNumber(tU32 number);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] number
     * @return < 0: error, = 0: OK
     */
    tResult RequestTitleSearch(tU32 number);
    /**
     * This method  RequestSubtitleChange
     * @param[in] state , selectionType , Subtitel
     * @return < 0: error, = 0: OK
     */
    tResult RequestSubtitleChange(tBool state, tU8  selectionType , tU8 Subtitel);
    /**
     * This method  RequestCMSkip
     * @param[in] SkipValue
     * @return < 0: error, = 0: OK
     */
    tResult RequestCMSkip(tU8 SkipValue);
    tResult GetKeyCommandResponse(OUT tU8& response);
    /**
     * This method  RequestDVDTemperature
     * @param[in] User Context
     * @return < 0: error, = 0: OK
     */
    tResult RequestDVDTemperature(const tUserContext userContext);

    /**
     * R6392_UC1_LLD2.1
     * This method sets the status of AutoRipping mode.
     *
     * @param[IN] AutoRippingMode Ripping mode from project
     * @return < 0: error, = 0: OK
     */
    tResult SetAutoRipping(tBoolean AutoRippingMode);

    /*
     * This method sets the gracenote lookup mode
    */
    tResult SetGracenoteLookupMode(tBoolean GNlookupMode);
    /**
     * R6392_UC1_LLD2.1
     * stops ripping.
     *
     * @return < 0: error, = 0: OK
     */
    tResult StopRipping(void);

    tResult StartRipping(void);

    /**
        * This method gets the status of ripping mode.
        *R6392_UC1_LLD2.1
        * @param[OUT] RippingState
        * @param[OUT] TotalTracks
        * @param[OUT] CurrentRippingIndex
        * @param[OUT] Percentage
        * @return < 0: error, = 0: OK
        */
    tResult GetRippingStatus(tUInt &RippingState, tUInt &TotalTracks, tUInt &CurrentRippingIndex, tUInt &Percentage);
    tResult GetAutoRippingStatus(tBoolean &AutoRippingMode);
    /**
     * This property  GetRippingError is used to get Ripping error
     */
    tResult GetRippingError(tResult &RippingError);

    /**
     * This property  GetGracenoteLookupMode is used to get the lookup mode for gracenote - online/offline
     */
    tResult GetGracenoteLookupMode(tGracenoteLookupMode& GracenoteLookupMode);
    tResult ExtractGracenoteMetadata(string albumTitle, tBool multipleAlbumSelection);
    /**
     * R6392_UC1_LLD2.1
     * This method  ExtractGracenoteMetadata is used to get the index of the selected album
     * @param[in] albumTitle
     * @param[OUT]
     * @return < 0: error, = 0: OK
     */
    tResult GetGNMetadataExtractionStatus(tGNmetadataExtractionResult& GNmetadataExtractionResult);
    /*
     * This property to get the gracenote metadata extraction status
     */
    tResult GetGracenoteVersion(tGracenoteVersion &GracenoteVersion);
    /**
     * This method  GetGracenoteVersion is used to get the version of Gracenote
     */
    tResult GetAlbumStringsforGracenote (vector<string> &albumStringVector);
    /**
     * R6392_UC1_LLD2.1
     * This method GetAlbumStringsforGracenote is used to populate the vector with the albums from the map
     * @param[in]
     * @param[OUT] albumStringVector
     * @return < 0: error, = 0: OK
     */

    tResult DatabaseStatistics(void);
    /**
     * This method  getStreamingInfo used to send streaming info
     * @param[in]
     * @param[OUT] device : Device type ,
     * @return < 0: error, = 0: OK
     */
    tResult getStreamingInfo(OUT tU8 &device,OUT tU8 &tag, OUT tS16 &info);
    /**
       * This method returns the RouteGuidance update (iAP2 connections, host only) from IPODControl.
       * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPORouteGuidance.
       *
       * @param[out] tDeviceID deviceID
       * @param[out] RGUpdate RGUpdate
       * @return < 0: error, = 0: OK
       */
    tResult GetRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate);

    /**
       * This method returns the RouteGuidanceManeuver update (iAP2 connections, host only) from IPODControl.
       * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPORouteGuidanceManeuver.
       *
       * @param[out] tDeviceID deviceID
       * @param[out] RGManeuverUpdate RGManeuverUpdate
       * @return < 0: error, = 0: OK
       */
    tResult GetRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate);

    /**
       * This method returns the RouteGuidance update (iAP2 connections, host only) from IPODControl.
       * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPORouteGuidance.
       *
       * @param[out] tDeviceID deviceID
       * @param[out] RGUpdate RGUpdate
       * @return < 0: error, = 0: OK
       */
    tResult PullRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate);

    /**
       * This method returns the RouteGuidanceManeuver update (iAP2 connections, host only) from IPODControl.
       * The controller is notified about a change  on one or more devices by the method OutputWrapper::UpdateDiPORouteGuidanceManeuver.
       *
       * @param[out] tDeviceID deviceID
       * @param[out] RGManeuverUpdate RGManeuverUpdate
       * @return < 0: error, = 0: OK
       */
    tResult PullRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate);

    tResult DiPOStartRouteGuidance(const tMountPoint USBSerialNumber, tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs, tBoolean SourceNameAvail, tBoolean SourceSupportsRouteGuidance);
    tResult DiPOStopRouteGuidance(const tMountPoint USBSerialNumber, tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs);
    tResult DiPOSetRGDisplayComponentInfo(const tDiPORGDisplayComponentList RGDisplayComponentList);
    tResult DiPOGetUSBSerialNumber(const tDeviceID deviceID, tMountPoint USBSerialNumber);
    //>--Roadmap CMG3G-10221 : 'Scene Recorder '
    /**
     * This method  SetFilePermissions to set file permission
     * @param[in] media object and mode to set
     * @param[OUT] device : none
     * @return < 0: error, = 0: OK
     */
    tResult SetFilePermissions(IN std::vector<tFilePermission> pfilePermissions, IN tDeviceID const pDeviceID);
    /**
     * This method  MediaplayerFolderInfo to set file permission
     * @param[in] deviceID device tag
     * @param[in] Path "/"
     * @param[in] FileTypeSelection "Type of the file selected"
     * @param[OUT] folderinfo Folder info for
     * total file size , protected file size
     * total file number , protected file number
     * @return < 0: error, = 0: OK
     */
    tResult MediaplayerFolderInfo(IN tDeviceID deviceID, IN tPath Path,
                                         IN tFileTypeSelection fileTypeSelection, OUT tFolderInfo& folderinfo);
    //<--Roadmap CMG3G-10221 : 'Scene Recorder '

    /*
     * To recieve property updates from Video Manager component
     * Currently used for DVD Video Source
     *  0=unknown, 1=inactive, 2=activating, 3=active, 4=blocked, 5=restricted (audio only)
     *  0=NONE, 1=RVCA, 2=RVCD, 3=DTV, 4=SPCX, 5=DVD, 6=USB
     */
    tResult VideoManagerSourceUpdate(tVideoSource videoSource , tVideoSourceState status );

    //<--Roadmap 17001 : 'Personalization'
    /**
    * Roadmap 17001 - Personalization
    * Stores the Object as favorite for the particular user
    * @param[in]  ObjectID
    * @param[in]  UserID
    * @param[out] FavoriteID
    * @return < 0: error, = 0: OK
    */
    tResult StorePersonalizedFavorite(OUT tFavoriteID &retfavID, IN tObjectID objectID, IN tUserID userID);

    /**
    * Roadmap 17001 - Personalization
    * Deletes the favorite for the particular user
    * @param[in]  objectID
    * @param[in]  UserID
    * @return < 0: error, = 0: OK
    */
    tResult DeletePersonalizedFavorite(OUT tObjectID objectID, IN tUserID userID);
    //<--Roadmap 17001 : 'Personalization'

    tResult AddOobPairedDeviceResult(const tDeviceID deviceID,const tBoolean result);
    /*playbackspeed value to display on HMI */

    tResult GetPlaybackSpeed(OUT tPlaybackSpeed &playbackSpeed);
    tResult GetDeviceIDForFolderStructureCleared(OUT tDeviceID &deviceID);
    tResult GetValuesForReadyToPlay(tDeviceSerialNumber &deviceAddress , tBool &PlayingStatus);

    tDeviceID GetAppleDeviceIDUsingUSBSerialNumber(const tMountPoint USBSerialNumber);

    tResult SendAppleHIDCommand(const tPlaybackHIDCommand playbackHIDCommand,const tBTButtonEvent keyEvent);


    tResult SendMediaEnginePlay();

    tResult SendMediaEngineStop();
    /**
     * This method is to get DeviceTimeUpdate from iPodControl
     * @param[out] deviceID, deviceTime
     * @return < 0: error, = 0: OK
     */
    tResult GetDiPODeviceTime(tDeviceID &deviceID, tDiPODeviceTime &deviceTime);

private:

    /**
     * Private Constructor
     *
     */
    MediaPlayerInterface();

    /**
     * Destructor
     *
     */
    virtual ~MediaPlayerInterface();

    protected:

    static MediaPlayerInterface *pMediaPlayer;

    /* MediaPlayer is created */
    tBoolean isCreated;

    // current playermanager state. Must be stored temporary during under voltage
    tAllocateState          mAllocateState;        //Roadmap 13024
    tAudioOutputDevice      mAudioOutputDevice;    //Roadmap 13024
    tDMOpticalDiscSlotState mOpticalDiscSlotState; //Roadmap 15002_CDAudio_DetectionAndDiagnosis
    tOpticalDriveVersion mOpticalDriveVersion;  // CD Info
    tenOpticalDeviceState mOpticalDeviceState;  //CD Info

    Lock mInterfaceLock;


    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * Finds the system language's codeset (via function:FindCharEncodingForSysLanguage)
    * Reconstructs the Latin1 strings as read from metadata header (via function:ReconstructLatin1Metadata & ReconstructTheGivenMetadata)
    * Convert the ReconstructedString to UTF-8 (via function:Convert2UTF8FromGivenCodeSet & ConvertTheGivenMetadata2UTF8)
    * @param[inout] mediaObjectVector
    * @return < 0: error, = 0: OK
    */
    tResult ConvertCodeSetFromSysLang2UTF8(vector<tMediaObject> &mediaObjectVector);

    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * Finds current system language's  most commonly used character encoding codeset
    * @param[inout] fromCodeset
    * @return < 0: error, = 0: OK
    */
    tResult FindCharEncodingForSysLanguage(string &fromCodeset);
    /**
     * ConvertDevicenameCodeSetFromSysLang2UTF8
     * Implemented for Nissan UTF-8 conversion of USB device name. Bug NCG3D-109483.
     * Finds current system language's  most commonly used character encoding codeset
     * @param[inout] str USB device name.
     * @param[in] size usb device name string length
     * @return void
     */
    void ConvertDevicenameCodeSetFromSysLang2UTF8(FastUTF8::tString str, const size_t size);

    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * Finds the MetadaFields to be reconstructed to Latin1.If found,then it invokes the function:ReconstructTheGivenMetadata
    * @param[inout] mediaObject
    * @return < 0: error, = 0: OK
    */
    tResult ReconstructLatin1Metadata(tMediaObject &mediaObject);

    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * The taglib string is reconstructed.This string is same as the taglib string extracted from Metadata header
    * @param[inout] MetadataField
    * @return < 0: error, = 0: OK
    */
    tResult ReconstructTheGivenMetadata(tMetadata &MetadataField);

    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * Finds the MetadaFields to be convert to UTF-8.If found,then it invokes the function:ConvertTheGivenMetadata2UTF8
    * @param[inout] mediaObject
    * @param[in] codeset
    * @return < 0: error, = 0: OK
    */
    tResult Convert2UTF8FromGivenCodeSet(tMediaObject &mediaObject,string &codeset);

    /**
    * Roadmap 15004 - CodeSetConversionForNissan
    * Converts the metadatafield to UTF-8 from the system language's codeset
    * @param[inout] MetadataField
    * @param[in] codeset
    * @return  = false: error, = true: OK
    */
    tBoolean ConvertTheGivenMetadata2UTF8(tMetadata &MetadataField,const gchar* &fromCodeset);



};


#endif  //_MEDIAPLAYERINTERFACE_H_

/** @} */
