#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#ifdef TARGET_BUILD
#include "trcGenProj/Header/IndexerTest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#endif
#endif

#include "Indexer.h"
#include "FunctionTracer.h"
#include "LocalSPM.h"
#include "CppUnitDefinitions.h"
#include "Dispatcher.h"
#include "RequestResponseSM.h"
#include "MediaPlayer_ErrorCodes.h"

#define NO_REQRESP
#include "MTPControlTestUtil.h"

#include "IndexerTest.h"

/*lint -save -e1401 */

void IndexerTest::setUp()
{
    ticks.begin();
}

void IndexerTest::tearDown()
{
    ticks.elapsed();
}


/**********************************************************************
 * Test cases
 *********************************************************************/

void IndexerTest::InitTestsuite()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Recreate the test database without song table */
    ret = LocalSPM::GetDataProvider().RecreateTestDatabase(true, false);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Set MY_MEDIA as active device so we can free the DB entries of other devices */
    ret = LocalSPM::GetDBManager().ActiveMediaDeviceSet(IN MY_MEDIA, true);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Register DB trigger of indexer */
    ret = LocalSPM::GetIndexer().SwitchDBTrigger(IN TT_ALL, IN TS_ON);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Deregister DB_DEVICE_CONNECTED trigger of indexer to be able to test each use case separately */
    ret = LocalSPM::GetIndexer().SwitchDBTrigger(IN TT_DB_DEVICE_CONNECTED, IN TS_OFF);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);
}

void IndexerTest::EndTestsuite()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    /* Deregister DB trigger of indexer */
    ret = LocalSPM::GetIndexer().SwitchDBTrigger(IN TT_ALL, IN TS_OFF);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Recreate the test database */
    ret = LocalSPM::GetDataProvider().RecreateTestDatabase(true, true);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
}

void IndexerTest::DBDeviceInserted()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Disconnect device (mark it as DISCONNECTED in DB) */
    ret = LocalSPM::GetDBManager().DeviceDisconnected(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test intermediate result
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_DISCONNECTED == deviceInfo.connectionState);

    ret = LocalSPM::GetDBManager().DeviceAttached(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_INSERTED message to IndexerSM to create scan context */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test result
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_CONNECTED == deviceInfo.connectionState);
}


void IndexerTest::DBDeviceConnectedUSB()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_CONNECTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for end of indexing */
    tIndexingState indexingState;
    //Reason for Increasing retry count
    //Sometimes,indexing takes ~150 ms more for completion.Hence increased count by 1
    int retry = 51;

    while(retry) {
        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (IDS_COMPLETE == indexingState) break;
        usleep(200000L); //wait 200ms
        retry--;
    }
    ETG_TRACE_USR4(("%d retries left", retry));
    CPPUNIT_ASSERT(retry != 0);

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    // test result
    tFingerprint fingerprint = {0};
    retry = 10;
    while(retry) {
        ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (strcmp(fingerprint, FINGERPRINT_INVALID) != 0) break;
        sleep(1);
        retry--;
    }

    tNumberOfFiles numberOfAudioFiles;
    ret = LocalSPM::GetDBManager().GetNumberOfAudioFiles(OUT numberOfAudioFiles, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(31 <= (int)numberOfAudioFiles);

    tNumberOfMediaObjects number;
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT number);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    //CPPUNIT_ASSERT(numberOfAudioFiles <= number); TODO: Activate it after recalculation of numberOfAudioFiles is complete
    CPPUNIT_ASSERT(46 <= (int)number);
}

void IndexerTest::DBDeviceConnectedMTP()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;

    //--------------------------------------
    //check if real MTP device is connected
    //--------------------------------------
    bool bRealDeviceConnected = false;

    tDeviceName         strModelname;
    tDeviceName         strFriendlyname;
    tDeviceSerialNumber strSerialnumber;
    tDeviceVersion      strDeviceversion;
    tDeviceName         strManufacturername;

    tMemorySize         totalSize = 0;
    tMemorySize         freeSize  = 0;

    util_getConnectedMTPDevice(OUT bRealDeviceConnected,
                                IN strModelname,
                                IN strFriendlyname,
                                IN strSerialnumber,
                                IN strDeviceversion,
                                IN strManufacturername);


    if(true == bRealDeviceConnected)
    {
        //------------------------------------------------------------
        //tell MTPControl to use real device Wrapper instead of stub
        //------------------------------------------------------------
        ETG_TRACE_USR4(("*****************************************************************************"));
        ETG_TRACE_USR4(("IndexerTest::DBDeviceConnectedMTP: FOUND REAL MTP DEVICE"));
        ETG_TRACE_USR4(("*****************************************************************************"));
        //-----------------------------------------------------
        //care that real MTPLib is used
        //-----------------------------------------------------
        LocalSPM::GetMTPControl().TestIF_setWrapper(eMTPWrapper_realLib);
        //--------------------------------
        //add MTP device to database
        //--------------------------------
        tNumberOfDevices numberOfDevices;
        ret = LocalSPM::GetDBManager().GetConnectedDeviceCount(numberOfDevices);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        ETG_TRACE_USR4(("IndexerTest::DBDeviceConnectedMTP: numberOfDevices:%d",numberOfDevices));

        util_InsertNewMTPDevicetoDB(strSerialnumber,
                                    strDeviceversion,
                                    strFriendlyname,
                                    EMPTY_STRING,
                                    totalSize, freeSize);

        util_showdevicesDB(DTY_MTP,strSerialnumber);


        ret = LocalSPM::GetDBManager().GetConnectedDeviceCount(numberOfDevices);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        ETG_TRACE_USR4(("IndexerTest::DBDeviceConnectedMTP: numberOfDevices:%d",numberOfDevices));

        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tDeviceCount deviceCount = numberOfDevices; //last device in DB is MTP device which has been added last

        /* Get deviceID */
        tDeviceID deviceID = DEVICE_ID_NOT_SET;
        ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_MTP);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        ETG_TRACE_USR4(("IndexerTest::DBDeviceConnectedMTP: deviceID:%d",deviceID));

        //-----------------------------------------------------
        //care that indexer triggers insert device connection
        //-----------------------------------------------------
         /* Send DB_DEVICE_INSERTED message to IndexerSM to create scan context */
        ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        sleep(5);



        //-------------------------------------------------------------
        /* Send DB_DEVICE_CONNECTED message to IndexerSM */
        //-------------------------------------------------------------
        ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        /* Wait for end of indexing */
        tIndexingState indexingState;
        int retry = 60;
        while(retry) {
            ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
            CPPUNIT_ASSERT(MP_NO_ERROR == ret);
            if (IDS_COMPLETE == indexingState) break;
            sleep(1);
            retry--;
        }
        ETG_TRACE_USR4(("%d retries left", retry));
        CPPUNIT_ASSERT(retry != 0);

        LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

        // test result
        tFingerprint fingerprint = {0};
        ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(strcmp(fingerprint, FINGERPRINT_INVALID) != 0);

        tNumberOfFiles numberOfAudioFiles;
        ret = LocalSPM::GetDBManager().GetNumberOfAudioFiles(OUT numberOfAudioFiles, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        CPPUNIT_ASSERT(0 < (int)numberOfAudioFiles); //At least one media object on device

        tNumberOfMediaObjects number;
        ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT number);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        //CPPUNIT_ASSERT(numberOfAudioFiles <= number); TODO: Activate it after recalculation of numberOfAudioFiles is complete
        CPPUNIT_ASSERT(0 < (int)number);

        //-----------------------------------------------------
        //care that stub MTPLib is used for stubbed test
        //-----------------------------------------------------
        LocalSPM::GetMTPControl().TestIF_setWrapper(eMTPWrapper_stubLib);
        bRealDeviceConnected = false;

    }
    else
    {
        ETG_TRACE_FATAL(("IndexerTest::DBDeviceConnectedMTP: not executed (no real MTP device is connected)"));
    }
}

void IndexerTest::DBDeviceConnectedFlash()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tDeviceCount deviceCount = 2;

    if( 1 > LocalSPM::GetDataProvider().InternalDevices() ) return;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_FLASH);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Start indexing via SendReindexing call */
    ret = LocalSPM::GetIndexer().SendReindexing(IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for end of indexing */
    tIndexingState indexingState;
    int retry = 60;
    while(retry) {
        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (IDS_COMPLETE == indexingState) break;
        sleep(1);
        retry--;
    }
    ETG_TRACE_USR4(("%d retries left", retry));
    CPPUNIT_ASSERT(retry != 0);

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    // test result
    tFingerprint fingerprint = {0};
    ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(strcmp(fingerprint, FINGERPRINT_INVALID) != 0);

    tNumberOfFiles numberOfAudioFiles;
    ret = LocalSPM::GetDBManager().GetNumberOfAudioFiles(OUT numberOfAudioFiles, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(46 <= (int)numberOfAudioFiles);

    tNumberOfMediaObjects number;
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT number);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    //CPPUNIT_ASSERT(numberOfAudioFiles <= number); TODO: Activate it after recalculation of numberOfAudioFiles is complete
    CPPUNIT_ASSERT(46 <= (int)number);
}


//parametrised test for device configuration
void  IndexerTest::DBDeviceConnectedExtUserConfig_SuppressDeviceTypes()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bIsEqual = FALSE;
    tDeviceType eTyp;

    for(tUInt i = (tUInt)DTY_UNKNOWN;i <= (tUInt)DTY_UNSUPPORTED; i++ )
    {

        eTyp = (tDeviceType)i;
        if(    (eTyp  == DTY_USB ) //not issue of test
            || (eTyp  == DTY_IPOD)
            || (eTyp  == DTY_SD)
            || (eTyp  == DTY_IPHONE )
            || (eTyp  == DTY_MTP )
            || (eTyp  == DTY_CDROM ))
        {
            ETG_TRACE_FATAL(("DBDeviceConnectedExtUserConfig_SuppressDeviceTypes: %d", ETG_CENUM(tDeviceType,eTyp)));
            bIsEqual = util_ExtUserConfigCheckSuppressDeviceType(eTyp);
            CPPUNIT_ASSERT(bIsEqual == TRUE);
        }
    }
    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
}


//parametrised test
void IndexerTest::DBDeviceConnectedExtUserConfig_SuppressFilesystems()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bIsEqual = FALSE;
    tFileSystemType eTyp;

    for(tUInt i = (tUInt)FSTY_UNKNOWN;i <= (tUInt)FSTY_UDF; i++ )
    {
        eTyp = (tFileSystemType)i;
        if(eTyp  != FSTY_UNKNOWN) //not issue of test
        {
            ETG_TRACE_FATAL(("DBDeviceConnectedExtUserConfig_SuppressDeviceTypes: %d", ETG_CENUM(tFileSystemType,eTyp)));
            bIsEqual = util_ExtUserConfigCheckSuppressFilesystems(eTyp);
            CPPUNIT_ASSERT(bIsEqual == TRUE);
        }
    }
    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
}


void IndexerTest::DBDeviceConnectedExtUserConfig_SuppressPartitions()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bIsEqual = FALSE;

    for(tUInt i = 0;i < 5; i++ )
    {
        ETG_TRACE_FATAL(("DBDeviceConnectedExtUserConfig_SuppressPartitions: Partition:%d", i));
        bIsEqual = util_ExtUserConfigCheckSuppressPartitions(i /*partition number*/);
        CPPUNIT_ASSERT(bIsEqual == TRUE);

    }
    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

}

void IndexerTest::DBDeviceIndexing()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_INDEXING message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INDEXING(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INDEXING", IN parameterString);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
}

void IndexerTest::DBDeviceRemoved()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_REMOVED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_REMOVED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_REMOVED", IN parameterString);

    // test result
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);
}

void IndexerTest::ContinueIndexing()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    int flagDBFileListWhileIndexing = LocalSPM::GetDataProvider().DBFileListWhileIndexing();
    LocalSPM::GetDataProvider().DBFileListWhileIndexing = 1;

    /* Reset indexing state and fingerprint to restart indexing for this device */
    tIndexingState indexingState = IDS_NOT_STARTED;
    tIndexingPercentComplete percent = 0;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN deviceID, IN indexingState, IN percent);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    tFingerprint fingerprint = FINGERPRINT_INVALID;
    tNumberOfFiles numberOfAudioFiles = 0;
    tNumberOfFiles numberOfVideoFiles = 0;
    ret = LocalSPM::GetDBManager().SetFingerprint(IN deviceID, IN fingerprint, IN numberOfAudioFiles,IN numberOfVideoFiles);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_INSERTED message to IndexerSM to create scan context */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test intermediate result
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_CONNECTED == deviceInfo.connectionState);

    /* Send DB_DEVICE_CONNECTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait until number of files of new device is available in DB */
    //Reason for increasing RetryCount:
    //"numberOfAudioFiles" field is set when an Audio Indexed.
    //Sometimes,Images in the device are indexed prior to Videos.Hence,videos are indexed late after ~5.5 secs.
    //Hence wait time changed from 2000 ms to 6000 ms
    int retry = 60;
    while(retry) {
        usleep(100000L); //wait 100ms
        ret = LocalSPM::GetDBManager().GetNumberOfAudioFiles(OUT numberOfAudioFiles, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret); //CID 18097 (#1 of 1): Unused value (UNUSED_VALUE)

        ETG_TRACE_USR4(("numberOfFiles:%d", numberOfAudioFiles));
        if(0 != numberOfAudioFiles) break;
        //ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
        //ETG_TRACE_USR4(("fingerprint:%s FINGERPRINT_INVALID:%s", fingerprint, FINGERPRINT_INVALID));
        //if(strcmp(fingerprint, FINGERPRINT_INVALID) != 0) break;
        retry--;
    }
    ETG_TRACE_USR4(("%d retries left", retry));
    CPPUNIT_ASSERT(retry != 0);

    /* Disconnect device (mark it as DISCONNECTED in DB) */
    ret = LocalSPM::GetDBManager().DeviceDisconnected(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test intermediate result
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_DISCONNECTED == deviceInfo.connectionState);

    ret = LocalSPM::GetDBManager().DeviceAttached(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_INSERTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(2);

    // test intermediate result
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_CONNECTED == deviceInfo.connectionState);

    /* Send DB_DEVICE_CONNECTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for end of indexing */
    retry = 60;
    while(retry) {
        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (IDS_COMPLETE == indexingState) break;
        sleep(1);
        retry--;
    }
    CPPUNIT_ASSERT(retry != 0);

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    // test result
    ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(strcmp(fingerprint, FINGERPRINT_INVALID) != 0);

    tNumberOfMediaObjects number;
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT number);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(30 <= number);

    /* Reset limit to old value */
    LocalSPM::GetDataProvider().DBFileListWhileIndexing = flagDBFileListWhileIndexing;
}

void IndexerTest::DBFreeSpace()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID which should be freed */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Disconnect device (mark it as DISCONNECTED in DB) */
    ret = LocalSPM::GetDBManager().DeviceDisconnected(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Get new deviceID */
    deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, "USB-2");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Reduce limit for number of objects in DB to test free space */
    tNumberOfMediaObjects lastNumber = 0;
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT lastNumber);
    if(MP_NO_ERROR != ret)
    {
        lastNumber = 52;
    }

    tNumberOfMediaObjects limitNumberObjectsInDB = (tNumberOfMediaObjects)LocalSPM::GetDataProvider().LimitNumberObjectsInDB();
    LocalSPM::GetDataProvider().LimitNumberObjectsInDB = lastNumber + 10;

    /* Disconnect device (mark it as DISCONNECTED in DB) */
    ret = LocalSPM::GetDBManager().DeviceDisconnected(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    // test intermediate result
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_DISCONNECTED == deviceInfo.connectionState);

    ret = LocalSPM::GetDBManager().DeviceAttached(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send DB_DEVICE_INSERTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_INSERTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_INSERTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    // test intermediate result
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(CS_CONNECTED == deviceInfo.connectionState);

    /* Send DB_DEVICE_CONNECTED message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterDB_DEVICE_CONNECTED(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("DB_DEVICE_CONNECTED", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Wait for end of indexing */
    tIndexingState indexingState;
    int retry = 60;
    while(retry) {
        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);
        if (IDS_COMPLETE == indexingState) break;
        sleep(1);
        retry--;
    }
    ETG_TRACE_USR4(("%d retries left", retry));
    CPPUNIT_ASSERT(retry != 0);

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    // test result
    tFingerprint fingerprint = {0};
    ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(strcmp(fingerprint, FINGERPRINT_INVALID) != 0);

    tNumberOfFiles numberOfAudioFiles;
    ret = LocalSPM::GetDBManager().GetNumberOfAudioFiles(OUT numberOfAudioFiles, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(48 <= (int)numberOfAudioFiles);

    tNumberOfMediaObjects number;
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT number);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT((tNumberOfMediaObjects)LocalSPM::GetDataProvider().LimitNumberObjectsInDB() >= number);

    /* Reset limit to old value */
    LocalSPM::GetDataProvider().LimitNumberObjectsInDB = limitNumberObjectsInDB;
}

void IndexerTest::Reindexing()
{
    ENTRY_TEST

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceCount deviceCount = 3;

    /* Get deviceID */
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN DTY_USB, IN "USB");
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    /* Send REINDEXING message to IndexerSM */
    ret = LocalSPM::GetIndexer().ParameterREINDEXING(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    ret = LocalSPM::GetIndexer().SendEventByName("REINDEXING", IN parameterString);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    sleep(1);

    /* Wait for end of indexing */

    tDeviceInfo deviceInfo;
    int retry = 50;

    while(retry) {
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
        CPPUNIT_ASSERT(MP_NO_ERROR == ret);

        if (IDS_COMPLETE == deviceInfo.indexedState) break;
        usleep(200000L); //wait 200ms
        retry--;
    }
    ETG_TRACE_USR4(("%d retries left", retry));
    CPPUNIT_ASSERT(retry != 0);

    LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);

    // test result
    CPPUNIT_ASSERT(DS_OK == deviceInfo.deviceState);
    CPPUNIT_ASSERT(48 <= (int)deviceInfo.numberOfAudioFiles);

    tFingerprint fingerprint = {0};
    ret = LocalSPM::GetDBManager().GetFingerprint(OUT fingerprint, IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    CPPUNIT_ASSERT(strcmp(fingerprint, FINGERPRINT_INVALID) != 0);
}


//--------------------------------------some helper functions ---------------------------------------------------

void IndexerTest::util_InsertNewDevicetoDB(tDeviceType      f_DeviceType,
                                                tFileSystemType  fileSystemType,
                                                tPartitionNumber f_partitionNumber,
                                                char*            f_serialNumber,
                                                char*            f_deviceVersion,
                                                char*            f_deviceName,
                                                char*            f_accessoryName,
                                                char*            f_mountPoint,
                                                char*            f_UUID
                                                )
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tResult     res;
    tDeviceInfo l_deviceInfo;

    /* set device data */
    InitDeviceInfo(OUT l_deviceInfo);

    l_deviceInfo.partitionNumber = f_partitionNumber;
    l_deviceInfo.fileSystemType  = fileSystemType;

    l_deviceInfo.deviceState     = DS_NONE;
    l_deviceInfo.activeSource    = 0;///@todo check for other tests if 1 is demanded
    l_deviceInfo.connected       = 1;
    l_deviceInfo.deviceType      = f_DeviceType;
    l_deviceInfo.connectionState = CS_ATTACHED;
    l_deviceInfo.totalSize       = 1000;
    l_deviceInfo.freeSize        = 111;
    l_deviceInfo.productID       = 1111;
    l_deviceInfo.connectionType  = DCT_USB;

    snprintf(l_deviceInfo.serialNumber ,  sizeof (l_deviceInfo.serialNumber ), "%s",f_serialNumber);
    snprintf(l_deviceInfo.deviceVersion,  sizeof (l_deviceInfo.deviceVersion ),"%s",f_deviceVersion);
    snprintf(l_deviceInfo.deviceName,     sizeof (l_deviceInfo.deviceName ),   "%s",f_deviceName);
    snprintf(l_deviceInfo.mountPoint ,    sizeof (l_deviceInfo.mountPoint),    "%s",f_mountPoint);
    snprintf(l_deviceInfo.accessoryName , sizeof (l_deviceInfo.accessoryName), "%s",f_accessoryName);
    snprintf(l_deviceInfo.UUID ,          sizeof (l_deviceInfo.UUID ),         "%s",f_UUID);



    //add device to DB
    res = LocalSPM::GetDBManager().AddDevice(INOUT l_deviceInfo);
    CPPUNIT_ASSERT(res == 0);

    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
}



void IndexerTest::util_showdevicesDB(tDeviceType f_deviceType, char* f_serialNumber)
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tResult res = MP_NO_ERROR;
    tNumberOfDevices numberOfDevices;
    std::vector<tDeviceInfo> deviceInfo;

    res = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfo);
    if(res == MP_NO_ERROR) ///@todo check why it works for other tests
    {
         for (std::vector<tDeviceInfo>::iterator it = deviceInfo.begin() ; it != deviceInfo.end(); ++it)
         {
             if((it->deviceType == f_deviceType) && (0 == strcmp(f_serialNumber,it->serialNumber)))
             {
                 ETG_TRACE_USR4(("=======  this device is issue of the test =================="));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): deviceType    : %d",ETG_CENUM(tDeviceType,f_deviceType)));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): deviceID      : %d",it->deviceID));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): deviceName    : %s",it->deviceName));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): serialNumber  : %s",it->serialNumber));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): deviceVersion : %s",it->deviceVersion));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): mountPoint    : %s",it->mountPoint));
                 ETG_TRACE_USR4(("IndexerTest::util_showdevicesDB(): accessoryName : %s",it->accessoryName));

             }
             else
             {
                 ETG_TRACE_USR4(("MTPControlTestUtil::util_showdevicesDB(): deviceType    : %d (not %d)",ETG_CENUM(tDeviceType,it->deviceType),ETG_CENUM(tDeviceType,f_deviceType)));
                 ETG_TRACE_USR4(("MTPControlTestUtil::util_showdevicesDB(): mountPoint    : %s",it->mountPoint));
             }
         }
    }


    ETG_TRACE_USR4(("MTPControlTestUtil::util_showdevicesDB():numberOfDevices:%d ",numberOfDevices));
    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

}





tBool IndexerTest::util_CompareWithNewDeviceInDB(IN tDeviceID          f_deviceID,
                                                          IN tDeviceType        f_DeviceType,
                                                          IN tFileSystemType    f_fileSystemType,
                                                          IN tPartitionNumber   f_partitionNumber,
                                                          IN char*              f_serialNumber,
                                                          IN char*              f_deviceVersion,
                                                          IN char*              f_deviceName,
                                                          IN char*              f_accessoryName,
                                                          IN char*              f_mountPoint,
                                                          IN char*              f_UUID,
                                                          /*configuration dependend*/
                                                          IN tConnectionState   f_connectionState,
                                                          IN tDeviceState       f_deviceState,
                                                          IN tIndexingState     f_indexedState
                                                         )
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bIsEqual = FALSE;

    //-------------------------------
    //check the device
    //-------------------------------
    tDeviceInfo deviceInfo;
    tResult ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN f_deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    int isEq = 0;
    isEq =+strcmp(deviceInfo.UUID,         f_UUID);
    isEq =+strcmp(deviceInfo.serialNumber, f_serialNumber);
    isEq =+strcmp(deviceInfo.deviceVersion,f_deviceVersion);
    isEq =+strcmp(deviceInfo.deviceName,   f_deviceName);
    isEq =+strcmp(deviceInfo.mountPoint,   f_mountPoint);
    isEq =+strcmp(deviceInfo.accessoryName,f_accessoryName);

    if(isEq == 0)
    {
        bIsEqual = FALSE;
        if(deviceInfo.deviceType ==f_DeviceType)
        {
            if((deviceInfo.fileSystemType == f_fileSystemType) )
            {
                if((deviceInfo.partitionNumber == f_partitionNumber) )
                {
                    bIsEqual = TRUE;
                }
            }
        }
    }
    else
    {
        bIsEqual = FALSE;
    }

    //--------------------------------------
    //check parameters influenced after insertion
    //--------------------------------------
    if(bIsEqual)
    {
         bIsEqual = FALSE;
         if(deviceInfo.connectionState == f_connectionState)
         {
             if(deviceInfo.deviceState == f_deviceState)
             {
                 if(deviceInfo.indexedState == f_indexedState)
                 {
                     bIsEqual = TRUE;
                 }
             }
         }
     }


   ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

   return bIsEqual;
}


void IndexerTest::util_OrigConfigurationStore()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    m_iSupportedDeviceTypes   = LocalSPM::GetDataProvider().SupportedDeviceTypes();
    m_iSupportedDeviceType_01 = LocalSPM::GetDataProvider().SupportedDeviceType_01();
    m_iSupportedDeviceType_02 = LocalSPM::GetDataProvider().SupportedDeviceType_02();
    m_iSupportedDeviceType_03 = LocalSPM::GetDataProvider().SupportedDeviceType_03();
    m_iSupportedDeviceType_04 = LocalSPM::GetDataProvider().SupportedDeviceType_04();
    m_iSupportedDeviceType_05 = LocalSPM::GetDataProvider().SupportedDeviceType_05();
    m_iSupportedDeviceType_06 = LocalSPM::GetDataProvider().SupportedDeviceType_06();
    m_iSupportedDeviceType_07 = LocalSPM::GetDataProvider().SupportedDeviceType_07();
    m_iSupportedDeviceType_08 = LocalSPM::GetDataProvider().SupportedDeviceType_08();

    //Roadmap 13001 ExtUserConf - supported filsystems
    m_iSupportedfileSystems   = LocalSPM::GetDataProvider().SupportedFileSystems();
    m_iSupportedfileSystem_01 = LocalSPM::GetDataProvider().SupportedFileSystem_01();
    m_iSupportedfileSystem_02 = LocalSPM::GetDataProvider().SupportedFileSystem_02();
    m_iSupportedfileSystem_03 = LocalSPM::GetDataProvider().SupportedFileSystem_03();
    m_iSupportedfileSystem_04 = LocalSPM::GetDataProvider().SupportedFileSystem_04();
    m_iSupportedfileSystem_05 = LocalSPM::GetDataProvider().SupportedFileSystem_05();
    m_iSupportedfileSystem_06 = LocalSPM::GetDataProvider().SupportedFileSystem_06();
    m_iSupportedfileSystem_07 = LocalSPM::GetDataProvider().SupportedFileSystem_07();
    m_iSupportedfileSystem_08 = LocalSPM::GetDataProvider().SupportedFileSystem_08();

    //Roadmap 13001 ExtUserConf - supported partions:
    m_iSupportedPartitionNumbers   = LocalSPM::GetDataProvider().SupportedPartionNumbers();

    //Roadmap 13023 ExtUserConf - visibility
    m_iUnsupportedDeviceTypeVisible = LocalSPM::GetDataProvider().UnsupportedDeviceTypeVisible();
    m_iUnsupportedFileSystemVisible = LocalSPM::GetDataProvider().UnsupportedFileSystemVisible();
    m_iUnsupportedPartitionVisible  = LocalSPM::GetDataProvider().UnsupportedPartitionVisible();


    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
}
void IndexerTest::util_OrigConfigurationRestore()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    LocalSPM::GetDataProvider().SupportedDeviceTypes   = m_iSupportedDeviceTypes;
    LocalSPM::GetDataProvider().SupportedDeviceType_01 = m_iSupportedDeviceType_01;
    LocalSPM::GetDataProvider().SupportedDeviceType_02 = m_iSupportedDeviceType_02;
    LocalSPM::GetDataProvider().SupportedDeviceType_03 =
m_iSupportedDeviceType_03;
    LocalSPM::GetDataProvider().SupportedDeviceType_04 = m_iSupportedDeviceType_04;
    LocalSPM::GetDataProvider().SupportedDeviceType_05 = m_iSupportedDeviceType_05;
    LocalSPM::GetDataProvider().SupportedDeviceType_06 = m_iSupportedDeviceType_06;
    LocalSPM::GetDataProvider().SupportedDeviceType_07 = m_iSupportedDeviceType_07;
    LocalSPM::GetDataProvider().SupportedDeviceType_08 = m_iSupportedDeviceType_08;

    LocalSPM::GetDataProvider().SupportedFileSystems   = m_iSupportedfileSystems;
    LocalSPM::GetDataProvider().SupportedFileSystem_01 = m_iSupportedfileSystem_01;
    LocalSPM::GetDataProvider().SupportedFileSystem_02 = m_iSupportedfileSystem_02;
    LocalSPM::GetDataProvider().SupportedFileSystem_03 = m_iSupportedfileSystem_03;
    LocalSPM::GetDataProvider().SupportedFileSystem_04 = m_iSupportedfileSystem_04;
    LocalSPM::GetDataProvider().SupportedFileSystem_05 = m_iSupportedfileSystem_05;
    LocalSPM::GetDataProvider().SupportedFileSystem_06 = m_iSupportedfileSystem_06;
    LocalSPM::GetDataProvider().SupportedFileSystem_07 = m_iSupportedfileSystem_07;
    LocalSPM::GetDataProvider().SupportedFileSystem_08 = m_iSupportedfileSystem_08;

    LocalSPM::GetDataProvider().SupportedPartionNumbers   = m_iSupportedPartitionNumbers;

    //visibility
    LocalSPM::GetDataProvider().UnsupportedDeviceTypeVisible = m_iUnsupportedDeviceTypeVisible;
    LocalSPM::GetDataProvider().UnsupportedFileSystemVisible = m_iUnsupportedFileSystemVisible;
    LocalSPM::GetDataProvider().UnsupportedPartitionVisible  = m_iUnsupportedPartitionVisible;


    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
}

void IndexerTest::util_TestConfigurationSet()
{
    //tthis configuration means all are supported
    LocalSPM::GetDataProvider().SupportedDeviceTypes   = 6;
    LocalSPM::GetDataProvider().SupportedDeviceType_01 = DTY_USB;
    LocalSPM::GetDataProvider().SupportedDeviceType_02 = DTY_SD;
    LocalSPM::GetDataProvider().SupportedDeviceType_03 = DTY_IPOD;
    LocalSPM::GetDataProvider().SupportedDeviceType_04 = DTY_IPHONE;
    LocalSPM::GetDataProvider().SupportedDeviceType_05 = DTY_MTP;
    LocalSPM::GetDataProvider().SupportedDeviceType_06 = DTY_CDROM;
    LocalSPM::GetDataProvider().SupportedDeviceType_07 = DTY_UNKNOWN; // DTY_BLUETOOTH;
    LocalSPM::GetDataProvider().SupportedDeviceType_08 = DTY_UNKNOWN;


    //Roadmap 13001 ExtUserConf - supported filsystems
    LocalSPM::GetDataProvider().SupportedFileSystems   = 7;              /* number of registered file systems */
    LocalSPM::GetDataProvider().SupportedFileSystem_01 = FSTY_FAT;
    LocalSPM::GetDataProvider().SupportedFileSystem_02 = FSTY_NTFS;
    LocalSPM::GetDataProvider().SupportedFileSystem_03 = FSTY_EXT4;
    LocalSPM::GetDataProvider().SupportedFileSystem_04 = FSTY_HFS;
    LocalSPM::GetDataProvider().SupportedFileSystem_05 = FSTY_EXFAT;
    LocalSPM::GetDataProvider().SupportedFileSystem_06 = FSTY_ISO9660;
    LocalSPM::GetDataProvider().SupportedFileSystem_07 = FSTY_UDF;
    LocalSPM::GetDataProvider().SupportedFileSystem_08 = FSTY_UNKNOWN;

    LocalSPM::GetDataProvider().SupportedPartionNumbers   = 4;

    LocalSPM::GetDataProvider().UnsupportedDeviceTypeVisible = 0; //not visible
    LocalSPM::GetDataProvider().UnsupportedFileSystemVisible = 0;
    LocalSPM::GetDataProvider().UnsupportedPartitionVisible  = 0;
}


void IndexerTest::util_showConfiguration()
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    //tthis configuration means all are supported
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceTypes()   = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceTypes()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_01() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_01()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_02() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_02()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_03() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_03()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_04() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_04()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_05() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_05()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_06() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_06()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_07() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_07()) ));
    ETG_TRACE_USR4(("LocalSPM::GetDataProvider().SupportedDeviceType_08() = %d",ETG_CENUM(tDeviceType,LocalSPM::GetDataProvider().SupportedDeviceType_08()) ));


    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

}


tBool  IndexerTest::util_ExtUserConfigCheckSuppressDeviceType(tDeviceType eDeviceType)
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bRet = TRUE;
    char strDeviceType[128];
    tFileSystemType eFileSystemType = FSTY_FAT;
    tUInt uPartitionNum             = 0;


    //-------------------------------
    //manipulate configuration
    //-------------------------------
    util_OrigConfigurationStore();
    util_TestConfigurationSet();

    //adjust configuration that devices which should be tested as suppressed devices don't appear in the configuration
    LocalSPM::GetDataProvider().UnsupportedDeviceTypeVisible = 1; //not visible
    LocalSPM::GetDataProvider().SupportedDeviceTypes         = 5;
    LocalSPM::GetDataProvider().SupportedDeviceType_06       = DTY_UNKNOWN;
    switch(eDeviceType)
    {
        case DTY_USB:
            strncpy_r(strDeviceType,"DTY_USB",sizeof(strDeviceType));
            LocalSPM::GetDataProvider().SupportedDeviceType_01 = DTY_CDROM;
            break;
        case DTY_SD:
            strncpy_r(strDeviceType, "DTY_SD", sizeof(strDeviceType));
            LocalSPM::GetDataProvider().SupportedDeviceType_02 = DTY_CDROM;
            break;
        case DTY_IPOD:
            strncpy_r(strDeviceType,"DTY_IPOD",sizeof(strDeviceType));
            LocalSPM::GetDataProvider().SupportedDeviceType_03 = DTY_CDROM;
            eFileSystemType = FSTY_UNKNOWN; //MTP and Appledevs have in their normal more no partitions for the outside world
            break;
        case DTY_IPHONE:
            strncpy_r(strDeviceType,"DTY_IPHONE",sizeof(strDeviceType));
            LocalSPM::GetDataProvider().SupportedDeviceType_04 = DTY_CDROM;
            eFileSystemType = FSTY_UNKNOWN;
            break;
        case DTY_MTP:
            strncpy_r(strDeviceType,"DTY_MTP",sizeof(strDeviceType));
            LocalSPM::GetDataProvider().SupportedDeviceType_05 = DTY_CDROM;
            eFileSystemType = FSTY_UNKNOWN;
            break;
        case DTY_CDROM:
            strncpy_r(strDeviceType,"DTY_CDROM",sizeof(strDeviceType));
            /*above configuration has already no DTY_CDROM*/
            break;
        default:
            bRet = FALSE;
            break;
    }


    util_showConfiguration();

    if(bRet)
    {
       bRet = util_ExtUserConfigInsertDeviceAndCheck(DS_UNSUPPORTED,
                                                     eDeviceType,  //<-------issue of test
                                                     eFileSystemType,
                                                     uPartitionNum,
                                                     strDeviceType);
    }

    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));
    return bRet;

}

tBool  IndexerTest::util_ExtUserConfigCheckSuppressFilesystems(tFileSystemType eFileSystemType)
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bRet = TRUE;

    char strFileSystemType[128];

    //-------------------------------
    //manipulate configuration
    //-------------------------------
    util_OrigConfigurationStore(); //if this test would be shifted to its own testgroup then this could be put in to a fixture
    util_TestConfigurationSet();

    //adjust configuration that devices which should be tested as suppressed devices don't appear in the configuration
    LocalSPM::GetDataProvider().UnsupportedFileSystemVisible = 1;
    LocalSPM::GetDataProvider().SupportedFileSystems         = 6;
    LocalSPM::GetDataProvider().SupportedFileSystem_07       = DTY_UNKNOWN;
    switch(eFileSystemType)
    {
        case FSTY_FAT:
            strncpy_r(strFileSystemType,"FSTY_FAT",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_01 = FSTY_UDF;
            break;
        case FSTY_NTFS:
            strncpy_r(strFileSystemType,"FSTY_NTFS",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_02 = FSTY_UDF;
            break;
        case FSTY_EXT4:
            strncpy_r(strFileSystemType,"FSTY_EXT4",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_03 = FSTY_UDF;
            break;
        case FSTY_HFS:
            strncpy_r(strFileSystemType,"FSTY_HFS",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_04 = FSTY_UDF;
            break;
        case FSTY_EXFAT:
            strncpy_r(strFileSystemType,"FSTY_EXFAT",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_05 = FSTY_UDF;

            break;
        case FSTY_ISO9660:
            strncpy_r(strFileSystemType,"FSTY_ISO9660",sizeof(strFileSystemType));
            LocalSPM::GetDataProvider().SupportedFileSystem_06 = FSTY_UDF;
            break;
        case FSTY_UDF:
            strncpy_r(strFileSystemType,"FSTY_UDF",sizeof(strFileSystemType));
            /*configruation set above already suppresses FSTY_UDF*/
            break;
        default:
            bRet = FALSE;
            break;
    }

    if(bRet)
    {
       bRet = util_ExtUserConfigInsertDeviceAndCheck(DS_UNSUPPORTED_FILESYSTEM,
                                                     DTY_USB,
                                                     eFileSystemType, //<-------issue of test
                                                     1 /*Partition*/,
                                                     strFileSystemType);
    }

    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

    return bRet;

 }

tBool  IndexerTest::util_ExtUserConfigCheckSuppressPartitions(tUInt uPartitionNum)
{
    ENTRY_INTERNAL
    ETG_TRACE_USR3(("Begin: %s", __PRETTY_FUNCTION__));

    tBool bRet = TRUE;
    char strPartition[128];

    //-------------------------------
    //manipulate configuration
    //-------------------------------
    util_OrigConfigurationStore(); //if this test would be shifted to its own testgroup then this could be put in to a fixture
    util_TestConfigurationSet();

    //adjust configuration that devices which should be tested as suppressed devices don't appear in the configuration
    LocalSPM::GetDataProvider().UnsupportedPartitionVisible  = 1;
    LocalSPM::GetDataProvider().SupportedPartionNumbers      = 4;
    switch(uPartitionNum)
    {
    case 0:
        strncpy_r(strPartition,"Partition0", sizeof(strPartition));
        LocalSPM::GetDataProvider().SupportedPartionNumbers  = PARTITION_NUMBER_NONE; //0 are supported hence a device with single partition and partition number=0 which is a device /dev/sda with single partition would be suppressed
        break;
    case 1:
        strncpy_r(strPartition,"Partition1", sizeof(strPartition));
        LocalSPM::GetDataProvider().SupportedPartionNumbers = PARTITION_NUMBER_NONE; //0 are supported hence a device with a single partition and partition number = 1  which is a device /dev/sda1 with single partition would be suppressed
        break;
    case 2:
        strncpy_r(strPartition,"Partition2", sizeof(strPartition));
         LocalSPM::GetDataProvider().SupportedPartionNumbers = 1; //partition 2 of device will be suppressed since number of allowed partitions are 1
        break;
    case 3:
        strncpy_r(strPartition,"Partition3", sizeof(strPartition));
        LocalSPM::GetDataProvider().SupportedPartionNumbers = 2;
        break;
    case 4:
        strncpy_r(strPartition,"Partition4", sizeof(strPartition));
        LocalSPM::GetDataProvider().SupportedPartionNumbers = 3;

        break;
    default:
        bRet = FALSE;
        break;
    }

    if(bRet)
    {
        bRet = util_ExtUserConfigInsertDeviceAndCheck(DS_UNSUPPORTED_PARTITION,
                                                      DTY_USB,
                                                      FSTY_FAT,
                                                      uPartitionNum,//<-------issue of test
                                                      strPartition);
    }

    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

    return bRet;

 }


tBool  IndexerTest::util_ExtUserConfigInsertDeviceAndCheck(IN tDeviceState expectedDeviceState,IN tDeviceType eDeviceType, IN tFileSystemType eFileSystemType, IN tUInt uPartitionNum ,IN char* strIdentifier)
{
    tResult ret;
    char strSerialNumber[128];
    char strDeviceVersion[128];
    char strDeviceName[128];
    char strAccessoryName[128];
    char strMountPoint[128];
    char strUUID[128];

    tDeviceID        deviceID;

    snprintf(strSerialNumber, sizeof(strSerialNumber), "%s:SerialNumber:1234",strIdentifier); //identifier i thought to help reading traces
    snprintf(strDeviceVersion,sizeof(strDeviceVersion),"%s:DeviceVersion:1.0",strIdentifier);
    snprintf(strDeviceName,   sizeof(strDeviceName),   "%s:DeviceName",       strIdentifier);
    snprintf(strAccessoryName,sizeof(strAccessoryName),"%s:AccessoryName",    strIdentifier);
    snprintf(strMountPoint,   sizeof(strMountPoint),   "%s:MountPoint",       strIdentifier);
    snprintf(strUUID,         sizeof(strUUID),         "%s:UUID",             strIdentifier);

    //--------------------------------
    //add device to database
    //--------------------------------
    util_InsertNewDevicetoDB( eDeviceType,
                              eFileSystemType,
                              uPartitionNum,
                              strSerialNumber,
                              strDeviceVersion,
                              strDeviceName,
                              strAccessoryName,
                              strMountPoint,
                              strUUID);

    util_showdevicesDB(eDeviceType,strSerialNumber);

    sleep(1); //is there a better way for test syncronisation?
    //-------------------------------
    //check the device in database
    //-------------------------------
    deviceID = DEVICE_ID_NOT_SET;

    //find the device in data base which is issue of the test
    ret = LocalSPM::GetDBManager().GetDevice(OUT deviceID, IN eDeviceType, IN strDeviceName, true /*get all devices */);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ETG_TRACE_USR4(("IndexerTest::DBDeviceConnectedMTP: deviceID:%d",deviceID));

    //-------------------------------------------------------------------------------------
    // trigger indexer to run  routines to take configuration into account - i.e. trigger insert device connection
    //-------------------------------------------------------------------------------------

    tBool bIsEqual = util_CompareWithNewDeviceInDB(IN deviceID,
                                                   IN eDeviceType,
                                                   IN eFileSystemType,
                                                   IN uPartitionNum,
                                                   IN strSerialNumber,
                                                   IN strDeviceVersion,
                                                   IN strDeviceName,
                                                   IN strAccessoryName,
                                                   IN strMountPoint,
                                                   IN strUUID,
                                                   IN CS_UNSUPPORTED,
                                                   IN expectedDeviceState,
                                                   IN IDS_NOT_STARTED);

    //--------------------------------
    //disconnect device (mark it as DISCONNECTED in DB) so that this device can be freed later on
    //--------------------------------
    ret = LocalSPM::GetDBManager().DeviceDisconnected(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);
    ret = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(IN deviceID);
    CPPUNIT_ASSERT(MP_NO_ERROR == ret);

    //--------------------------------
    //reset configuration values to old ones
    //--------------------------------
    util_OrigConfigurationRestore();

    ETG_TRACE_USR3(("End  : %s", __PRETTY_FUNCTION__));

    return bIsEqual;


}


