#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INDEXER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/Indexer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INDEXER
#endif
#endif

#include "LocalSPM.h"
#include "VarTrace.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "Indexer.h"
#include <sys/syscall.h>

#define FUNCTION_ID_WAIT_FOR_UPDATE_MYMEDIA_FINISHED    1

/*lint -save -e1401 */

Indexer::Indexer(const tComponentID componentID):ILocalSPM(componentID)
{
    ENTRY_INTERNAL

    /* Early initialization of variables */
    m_AutoRegisterOnDBTrigger = TS_ON;
}

Indexer::~Indexer()
{
    ENTRY_INTERNAL
}

void Indexer::Create()
{
    ENTRY

    /* Create the state machine */
    IndexerSM::Create();

    CreateDone(0);
}

tResult Indexer::Init(tInitReason reason)
{
    ENTRY
    (void)reason;

    /* Init the state machine */
    IndexerSM::Init();
    SetAnswerTimeout(INDEXER_SM_ANSWER_TIMEOUT_MS); // Set answer timer

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);

    return InitDone(0);
}

tResult Indexer::InitSM()
{
    ENTRY

    /* Initialize variables */
    m_TriggerID_DevInserted = 0;
    m_TriggerID_DevAttached = 0;
    m_TriggerID_DevConnected = 0;
    m_TriggerID_DevIndexingState = 0;
    m_TriggerID_DevIndexingProgress = 0;
    m_TriggerID_DevRemoved = 0;
    m_TriggerID_DevVersionUpdated = 0;

    /* Add first element (of no device) to indexing context */
    //m_IndexingContext.clear();
    if( 0 < m_IndexingContext.capacity() )
    {
        vector<tIndexingContext>().swap(m_IndexingContext); //used instead of clear to guarantee a memory reallocation
    }
    tIndexingContext indexingContextElem;
    InitIndexingContext(INOUT indexingContextElem);
    m_IndexingContext.push_back(indexingContextElem);
    m_Index = m_IndexingContext.size()-1;
    m_IndexingContext[m_Index].deviceID = DEVICE_ID_NOT_SET;

    m_DevicesToIndex.clear();
    m_StoredIndexingStates.clear();

    m_NumberObjectsInDB = 0;
    m_LimitNumberObjectsInDB = 100000;
    m_LimitNumberObjectsPerDevice = 100000;

    InitMediaObject(INOUT m_MediaObject);

    m_ProgressTimerID = 0;
    m_ProgressTimeout = 3000L;

    m_FingerprintTimerID = 0;
    m_LastFingerprintDeviceID = DEVICE_ID_NOT_SET;

    mSyncID = tIndexSession_init;

    return MP_NO_ERROR;
}

tResult Indexer::Run()
{
    ENTRY
    if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad())
    {
        LocalSPM::GetThreadFactoryLowprio().Do(this, 0, NULL); //Indexer inclusive state machine
    }
    else
    {
        LocalSPM::GetThreadFactory().Do(this, 0, NULL); //Indexer inclusive state machine
    }

    if(TS_ON == m_AutoRegisterOnDBTrigger)
    {
        /* Register on DB trigger */
        SwitchDBTrigger(TT_ALL, TS_ON);
    }
    // set all internal devices to connection state CS_ATTACHED to start indexing. This must be done in Indexer to gurantee that trigger are already available
    LocalSPM::GetDBManager().SetInternalDevicesToAttached();
    return RunDone(0);
}

void Indexer::Do(int functionID, void *ptr)
{
    ENTRY;

    if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
        //set the threads name
        LocalSPM::GetThreadFactoryLowprio().SetName(IN IndexerSM::GetSMNameFull());
    }
    else
    {
        LocalSPM::GetThreadFactory().SetName(IN IndexerSM::GetSMNameFull());
    }

    switch (functionID) {
        case 0:
        {
            if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
                LocalSPM::GetThreadFactoryLowprio().SetName("Indexer_0");
                SetThreadNiceProperty(syscall(SYS_gettid), LocalSPM::GetDataProvider().IndexerThreadNiceValue());
            }
            else
            {
                LocalSPM::GetThreadFactory().SetName("Indexer_0");
            }
            while (IndexerSM::STATE_MACHINE_FINISHED != IndexerSM::StateMachine_Main()) { }
            break;
        }
        case FUNCTION_ID_WAIT_FOR_UPDATE_MYMEDIA_FINISHED:
        {
            if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
                LocalSPM::GetThreadFactoryLowprio().SetName("Indexer_1");
            }
            else
            {
                LocalSPM::GetThreadFactory().SetName("Indexer_1");
            }
            DoWaitForUpdateMyMediaFinished((char *)ptr);
            break;
        }
        default:
            ETG_TRACE_ERR(("Unhandled functionID %d", functionID));
            break;
    }
}

tResult Indexer::Stop()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Deregister on DB trigger */
    ret = SwitchDBTrigger(TT_ALL, TS_OFF);

    /* Send STOP message to own SM */
    ret = SendForceEvent(STOP_SM, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult Indexer::StopEventProcessed()
{
    ENTRY

    /* Send stop done to SPM in the transition to final state in state machine */
    return StopDone(0);
}

tResult Indexer::Done()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Deregister state machine with dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* Send DONE message to own SM */
    ret = SendForceEvent(DONE, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult Indexer::DoneEventProcessed()
{
    ENTRY

    /* Send done done to SPM in the transition to final state in state machine */
    return DoneDone(0);
}

char *Indexer::GetSMStateName(OUT tGeneralString stateName, size_t size)
{
    GetCurrentState((char *)stateName, size);
    return stateName;
}

tResult Indexer::DBInitializeDevice(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceCount);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;

    /* Get device type and mount point from DBManager */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DBManager (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        VARTRACE(deviceInfo);

        tBoolean isSupportedDeviceType = LocalSPM::GetDataProvider().IsDeviceTypeSupported(IN deviceInfo.deviceType);
        tBoolean isSupportedFilesystem = LocalSPM::GetDataProvider().IsFileSystemSupported(IN deviceInfo.fileSystemType);
        tBoolean isSupportedPartitionNum = LocalSPM::GetDataProvider().IsPartitionSupported (IN deviceInfo.partitionNumber);

        tBoolean ignoreFSType = (deviceInfo.deviceType == DTY_IPOD) ||
                (deviceInfo.deviceType == DTY_BLUETOOTH) ||
                (deviceInfo.deviceType == DTY_IPHONE) ||
                (deviceInfo.deviceType == DTY_MTP) ||
                (deviceInfo.deviceType == DTY_CS) ||
                (deviceInfo.deviceType == DTY_CDDA)||
                (deviceInfo.deviceType == DTY_DVD_DRIVE);

        if(!isSupportedDeviceType)
        {
            deviceInfo.deviceState = DS_UNSUPPORTED;
            deviceInfo.connectionState = CS_UNSUPPORTED;
        }
        else if(!ignoreFSType && !isSupportedFilesystem)
        {
            deviceInfo.deviceState = DS_UNSUPPORTED_FILESYSTEM;
            deviceInfo.connectionState = CS_UNSUPPORTED;
        }
        else if (!ignoreFSType && !isSupportedPartitionNum)
        {
            deviceInfo.deviceState = DS_UNSUPPORTED_PARTITION;
            deviceInfo.connectionState = CS_UNSUPPORTED;
        }
        else
        {
            deviceInfo.deviceState = DS_OK;
        }

        if(deviceInfo.deviceState == DS_OK)
        {
            /* Send INIT_DEVICE_CONNECTION message to specific DeviceControlSM */
            char msgToSendString[64];
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);

            strncpy_r(OUT msgToSendString, IN "INIT_DEVICE_CONNECTION", IN sizeof(msgToSendString));
            tDeviceInfoString deviceInfoString;
            DataProvider::MarshalDeviceInfo(deviceInfoString, deviceInfo);

            ret = LocalSPM::GetUSBControl().ParameterINIT_DEVICE_CONNECTION(OUT parameterString,
                    IN size,
                    IN deviceInfoString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }

            ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceInfo.deviceType, IN msgToSendString, IN parameterString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
        else
        {
            //Roadmap 13023 ExtConfiguration
            /* Write device into DB via DataProvider */
            ret = LocalSPM::GetDataProvider().DeviceInitialized(IN deviceID, IN deviceInfo.deviceName, IN deviceInfo.connectionState, IN deviceInfo.deviceState);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while writing device into DB via DataProvider (ErrorCode:%s)", errorString(ret)));
            }
        }
    }

    return ret;
}

tResult Indexer::DeviceInitialized(const tDeviceName deviceName, const tDeviceID deviceID, const tConnectionState connectionState)
{
    ENTRY
    ETG_TRACE_USR3(("Indexer::DeviceInitialized deviceName:%40s, deviceID: %u, connectionState: %u", deviceName, deviceID, connectionState));

    tResult ret = MP_NO_ERROR;
    /* Get device type and mount point from DataProvider */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        if(CS_CONNECTED == connectionState && deviceInfo.connectionState != CS_CONNECTED)
        {
            /* Update indexing state for device via DataProvider */
            tIndexingState indexingState = IDS_NOT_STARTED;
            tIndexingPercentComplete percent = 0;
            ret = LocalSPM::GetDBManager().SetIndexingState(IN deviceID, IN indexingState, IN percent);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
            }

            /* Insert device to indexing state map */
            m_StoredIndexingStates[deviceID] = deviceInfo.indexedState;
        }

        if(connectionState != CS_DISCONNECTED)                          /* for the ticket NCG3D-82012 */
        {
            /* Write device into DB via DataProvider */
            tDeviceState deviceState = DS_OK;
            ret = LocalSPM::GetDataProvider().DeviceInitialized(IN deviceID, IN deviceName, IN connectionState, IN deviceState);
        }

        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while writing device into DB via DataProvider (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult Indexer::DBDeviceConnected(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceCount);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    //database trigger DB_DEVICE_CONNECTED
    //for immediate reaction on DB change place code here...

    //...

    //Forward internal Event to process DB change
    //Note: During waitingFor_indexing this event will be delayed
    // to guarentee indexed metadata consistency (MessageAnswer to DeviceControls)
    if(DEVICE_ID_NOT_SET != deviceID)
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterDEVICE_CONNECTED(OUT parameterString, IN size, IN deviceID);
        ret = SendEvent(DEVICE_CONNECTED, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("SendEvent(DEVICE_CONNECTED) failed (ErrorCode:%s)", errorString(ret)));
        }
    }

    return MP_NO_ERROR;
}

tResult Indexer::DeviceConnected(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;

    if(deviceID != m_IndexingContext[m_Index].deviceID)
    {
        /* Reinsert current deviceID at the beginning of the index queue */
        if(DEVICE_ID_NOT_SET != m_IndexingContext[m_Index].deviceID)
        {
            m_DevicesToIndex.insert(m_DevicesToIndex.begin(), m_IndexingContext[m_Index].deviceID);

            /* Update indexing state for device via DataProvider */
            tIndexingState indexingState = IDS_NOT_STARTED;
            tIndexingPercentComplete percent = 0;
            ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN percent);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
            }

            // Don't change indexing state here to have static condition to start or finish indexing
            //if(!IsAppleDevice(m_IndexingContext[m_Index].deviceType))
            //{
            //    /* Change indexing state map */
            //    m_StoredIndexingStates[m_IndexingContext[m_Index].deviceID] = IDS_PARTIAL;
            //}
        }
    }

    /* Add deviceID to index queue */
    m_DevicesToIndex.push_back(deviceID);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::DBDeviceRemoved(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceCount);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    //database trigger DB_DEVICE_REMOVED
    //for immediate reaction on DB change place code here...

    /* Get device type and mount point from DataProvider */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        /* Send REMOVE_DEVICE_CONNECTION message to specific DeviceControlSM */
        char msgToSendString[64];
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        strncpy_r(OUT msgToSendString, IN "REMOVE_DEVICE_CONNECTION", IN sizeof(msgToSendString));

        ret = LocalSPM::GetUSBControl().ParameterREMOVE_DEVICE_CONNECTION(OUT parameterString,
                IN size,
                IN deviceInfo.mountPoint,
                IN deviceID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceInfo.deviceType, IN msgToSendString, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Forward internal Event to process DB change
    //Note: During waitingFor_indexing this event will be delayed
    // to guarentee indexed metadata consistency (MessageAnswer to DeviceControls)
    if(DEVICE_ID_NOT_SET != deviceID)
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterDEVICE_REMOVED(OUT parameterString, IN size, IN deviceID);
        ret = SendEvent(DEVICE_REMOVED, IN parameterString);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("SendEvent(DEVICE_REMOVED) failed (ErrorCode:%s)", errorString(ret)));
        }
    }

    return MP_NO_ERROR;
}

tReturnValue Indexer::IsSameDevice(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    tReturnValue returnValue = false;
    tResult ret = MP_NO_ERROR;

    if(deviceID == m_IndexingContext[m_Index].deviceID)
    {
        returnValue = true;
    }
    else
    {
        /* Remove deviceID from index queue */
        for(int i = m_DevicesToIndex.size()-1; i>=0; i--)
        {
            if(deviceID == m_DevicesToIndex[i])
            {
                m_DevicesToIndex.erase(m_DevicesToIndex.begin() + i);
            }
        }

        /* Remove device from indexing state map */
        map<tDeviceID, tIndexingState>::iterator iterStoredIndexingStates = m_StoredIndexingStates.find(deviceID);
        if(iterStoredIndexingStates != m_StoredIndexingStates.end())
        {
            m_StoredIndexingStates.erase(iterStoredIndexingStates);
        }
    }

    /* Remove device from MyMedia DB */
    ret = LocalSPM::GetDBManager().RemoveDeviceFromMyMedia(IN deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while removing device from MyMedia DB (ErrorCode:%s)", errorString(ret)));
    }

    /* Remove device from LiveTag Table
       LiveTags table is persistent now to support LastMode and Favorites of partial indexed devices */
    //ret = LocalSPM::GetDBManager().RemoveLiveTags(IN deviceID);
    //if (MP_NO_ERROR != ret)
    //{
    //    ETG_TRACE_ERR(("Error while removing device from  LiveTag Table (ErrorCode:%s)", errorString(ret)));
    //}

    /* Reset variables of device in indexing context */
    for( tUInt i = 0; i < m_IndexingContext.size(); i++ )
    {
        if( deviceID == m_IndexingContext[i].deviceID )
        {
            InitIndexingContext(INOUT m_IndexingContext[i]);
            m_IndexingContext[i].deviceID = deviceID;
        }
    }

    return returnValue;
}

tResult Indexer::DeviceToIndex()
{
    ENTRY

    tResult ret = MP_NO_ERROR;
    bool foundNotYetIndexedDevice = false;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;

    if(1 < m_DevicesToIndex.size())
    {
        /* The order of the devices to index is defined by 3 switches
         * 1) Not yet indexed devices should be handled first
         * 2) The device types have a priority
         * 3) First or last device will serve first
         */
        tBoolean indexingNotYetIndexedPrio = (tBoolean)LocalSPM::GetDataProvider().IndexingNotYetIndexedPrio();
        tBoolean indexingDeviceTypePrio = (tBoolean)LocalSPM::GetDataProvider().IndexingDeviceTypePrio();
        tIndexingOrder indexingOrder = (tIndexingOrder) LocalSPM::GetDataProvider().IndexingOrder();

        /* Fill temporary device to index list based on indexing state (not yet indexed devices should be handled first) */
        vector<tDeviceID> prioDevicesToIndex;
        for (tUInt i = 0; i < m_DevicesToIndex.size(); i++)
        {
            if(indexingNotYetIndexedPrio) //Is feature enabled?
            {
                /* Get stored indexing state from indexing state map for device */
                tIndexingState storedIndexingState = IDS_NOT_STARTED;
                map<tDeviceID, tIndexingState>::iterator iterStoredIndexingStates = m_StoredIndexingStates.find(m_DevicesToIndex[i]);
                if(iterStoredIndexingStates != m_StoredIndexingStates.end())
                {
                    storedIndexingState = (tIndexingState)iterStoredIndexingStates->second;
                }

                /* Is stored indexing state of device NOT_STARTED insert element at the top else at the end*/
                if(IDS_NOT_STARTED == storedIndexingState)
                {
                    prioDevicesToIndex.insert(prioDevicesToIndex.begin(), m_DevicesToIndex[i]);
                    foundNotYetIndexedDevice = true;
                }
                else
                {
                    prioDevicesToIndex.push_back(m_DevicesToIndex[i]);
                }
            }
            else
            {
                prioDevicesToIndex.push_back(m_DevicesToIndex[i]);
            }
        }

        /*NCG3D-87289: When notyetindexed devices are found disable devicetype priority, to index notyetindexed device*/
        if(indexingDeviceTypePrio && !foundNotYetIndexedDevice)
        {
            tDeviceInfo deviceInfo;
            tPriority priority;
            tPriority highestPriority = PRIORITY_NONE;
            tIndex deviceIndex = 0;
            for (tUInt i = 0; i < prioDevicesToIndex.size(); i++)
            {
                if(IO_AS_CONNECTED_REVERSE == indexingOrder)
                {
                    deviceIndex = prioDevicesToIndex.size() -1 - i;
                }
                else
                {
                    deviceIndex = i;
                }

                /* Get device type from DataProvider */
                InitDeviceInfo(INOUT deviceInfo);
                ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN prioDevicesToIndex[deviceIndex]);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
                }
                else
                {
                    /* Get indexing priority from DataProvider */
                    priority = PRIORITY_NONE;
                    ret = LocalSPM::GetDataProvider().GetIndexingPriority(OUT priority, IN deviceInfo.deviceType);
                    if (MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while getting indexing priority from DataProvider (ErrorCode:%s)", errorString(ret)));
                    }
                    else
                    {
                        /* Store device ID of device type with highest priority */
                        if ((PRIORITY_NONE == highestPriority) || (priority < highestPriority))
                        {
                            deviceID = prioDevicesToIndex[deviceIndex];
                            highestPriority = priority;
                        }
                    }
                }
            }
        }
        else
        {
            switch (indexingOrder)
            {
                case IO_AS_CONNECTED:
                {
                    deviceID = prioDevicesToIndex.front();
                    break;
                }
                case IO_AS_CONNECTED_REVERSE:
                {
                    deviceID = prioDevicesToIndex.back();
                    break;
                }
                default:
                {
                    ETG_TRACE_ERR(("Invalid indexingOrder type: %u",indexingOrder));
                    ret = MP_ERR_PM_INVALID_PARAM;
                    break;
                }
            }
        }
    }
    else
    {
        /* Less than 2 devices in queue */
        if(!m_DevicesToIndex.empty())
        {
            deviceID = m_DevicesToIndex.front();
        }
    }
    ETG_TRACE_USR3(("Next device to index: %u",deviceID));

    /* Search for deviceID in indexing context and switch context */
    tBoolean found = false;
    for( tUInt i = 0; i < m_IndexingContext.size(); i++ )
    {
        if( deviceID == m_IndexingContext[i].deviceID )
        {
            found = true;
            m_Index = i;
            break;
        }
    }
    if(false == found)
    {
        tIndexingContext indexingContextElem;
        InitIndexingContext(INOUT indexingContextElem);
        m_IndexingContext.push_back(indexingContextElem);
        m_Index = m_IndexingContext.size()-1;
        m_IndexingContext[m_Index].deviceID = deviceID;
    }

#if 0
    if(m_IndexingContext[m_Index].deviceID != deviceID)
    {
        /* Reset variables of device in indexing context */
        InitIndexingContext(INOUT m_IndexingContext[m_Index]);
        m_IndexingContext[m_Index].deviceID = deviceID;
    }
#endif

    if(DEVICE_ID_NOT_SET != deviceID)
    {
        /* Send START_INDEX message to own SM */
        ret = SendEvent(START_INDEX, (char *)NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tReturnValue Indexer::IsIndexingSupported()
{
    ENTRY

    tReturnValue returnValue = false;
    tResult ret = MP_NO_ERROR;

    /* Remove deviceID from index queue */
    for(int i = m_DevicesToIndex.size()-1; i>=0; i--)
    {
        if(m_IndexingContext[m_Index].deviceID == m_DevicesToIndex[i])
        {
            m_DevicesToIndex.erase(m_DevicesToIndex.begin() + i);
        }
    }

    /* Get device type from DataProvider */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        /* Request indexing support for device type from DataProvider */
        returnValue = LocalSPM::GetDataProvider().IsIndexingSupported(IN deviceInfo.deviceType);
    }

    return returnValue;
}

tResult Indexer::UpdateDeviceNotSupported(const tDeviceID deviceID, const tNumberOfFiles numberOfFiles)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(numberOfFiles);
    tResult ret = MP_NO_ERROR;

    if(m_IndexingContext[m_Index].deviceID == deviceID)
    {

        tNumberOfFiles numberOfVideoFiles = NUMBER_OF_FILES_NONE;
        tNumberOfFiles numberOfImageFiles = NUMBER_OF_FILES_NONE;

        //If totalNumberOfFiles 0,then update Video,Image count also to 0.
        if(0 == numberOfFiles)
        {
            numberOfVideoFiles = 0;
            numberOfImageFiles = 0;
        }

        /* Update number of files */
        SetDeviceFileCounts(IN m_IndexingContext[m_Index].deviceID,IN numberOfFiles,IN numberOfVideoFiles,IN numberOfImageFiles);
    }
    else
    {
        ETG_TRACE_ERR(("Error: invalid deviceID"));
    }

    /* Update indexing state for device via DataProvider */
    tIndexingState indexingState = IDS_NOT_SUPPORTED;
    m_IndexingContext[m_Index].percent = 0;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN m_IndexingContext[m_Index].percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::GetNumberOfFiles()
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

    /* Get device type and mount point from DataProvider */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN m_IndexingContext[m_Index].deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        m_IndexingContext[m_Index].deviceType = deviceInfo.deviceType;
    }

    /* Send GET_NUMBER_OF_FILES message to specific DeviceControlSM */
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    char answerMsgString[64];

    strncpy_r(OUT msgToSendString, IN "GET_NUMBER_OF_FILES", IN sizeof(msgToSendString));
    strncpy_r(OUT answerMsgString, IN "IndexerSM::NUMBER_OF_FILES_ANSWER", IN sizeof(answerMsgString));

    ret = LocalSPM::GetUSBControl().ParameterGET_NUMBER_OF_FILES(OUT parameterString,
            IN size,
            IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = LocalSPM::GetDeviceDispatcher().RouteMessageAnswer(IN m_IndexingContext[m_Index].deviceType,
            IN msgToSendString,
            IN parameterString,
            IN answerMsgString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessageAnswer of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::Check()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    //increment sync ID
    //mSyncID++;  //NCG3D-55131

    /* Get device type and mount point from DataProvider */
    tDeviceInfo deviceInfo;
    VARTRACE(m_IndexingContext[m_Index].deviceID);
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN m_IndexingContext[m_Index].deviceID);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
        /* Send NOT_SUPPORTED message to own SM */
        ret = SendEvent(NOT_SUPPORTED, (char *)NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else    /* Check if fingerprint is already available */
    {
        m_IndexingContext[m_Index].deviceType = deviceInfo.deviceType;
        strncpy_r(OUT m_IndexingContext[m_Index].mountPoint, IN deviceInfo.mountPoint, IN sizeof(m_IndexingContext[m_Index].mountPoint));


        //TODO(pee1cob):
        //Couldn't understand purpose why noOfFiles from DB taken as m_IndexingContext[m_Index].totalNumberOfFiles.?!
        //Instead,m_IndexingContext[m_Index].totalNumberOfFiles should be left "not overwritten".
        //Reason:m_IndexingContext[m_Index].totalNumberOfFiles has the totalItemCOunt found via fingerPrint calculation.This is lost!

        //update indexer context with last numberOfFile, BEST GUESS
        VARTRACE(deviceInfo.numberOfAudioFiles);
        VARTRACE(deviceInfo.numberOfVideoFiles);

        m_IndexingContext[m_Index].totalNumberOfFiles = deviceInfo.numberOfAudioFiles;

        if(!IsAppleDevice(deviceInfo.deviceType) && (0 < strlen_r(m_IndexingContext[m_Index].fingerprint)))
        {
            /* Take data from indexing context and fake device control SM response:
             * Send expected event to waiting state machine immediately*/
            tFingerprintStatus fingerprintStatus = FPS_OK;
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);

            ret = ParameterFINGERPRINT_AVAIL(OUT parameterString,
                    IN size,
                    IN m_IndexingContext[m_Index].fingerprint,
                    IN fingerprintStatus,
                    IN m_IndexingContext[m_Index].totalNumberOfFiles,
                    m_IndexingContext[m_Index].deviceID);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string"));
            }
            else
            {
                ret = SendEvent(FINGERPRINT_AVAIL, IN parameterString);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while sending internal event via SMF"));
                }
            }
        }
        else
        {
            // indexing state independant: Do not filter for IDS_NOT_STARTED in case of iPod dynamic reindexing
            if (deviceInfo.deviceID == m_IndexingContext[m_Index].deviceID /*&& deviceInfo.indexedState == IDS_NOT_STARTED*/)
            {
                //Roadmap 13006 IAP2
                //Get current fingerprint from DB and forward it to DeviceControl via GET_FINGERPRINT
                tFingerprint lastFingerprint = {0};
                ret = LocalSPM::GetDBManager().GetFingerprint(OUT lastFingerprint, IN m_IndexingContext[m_Index].deviceID);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while getting indexing state from DataProvider (ErrorCode:%s)", errorString(ret)));
                }

                /* Send GET_FINGERPRINT message to specific DeviceControlSM */
                char msgToSendString[64];
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);

                strncpy_r(OUT msgToSendString, IN "GET_FINGERPRINT", IN sizeof(msgToSendString));

                ret = LocalSPM::GetUSBControl().ParameterGET_FINGERPRINT(OUT parameterString,
                        IN size,
                        IN m_IndexingContext[m_Index].mountPoint,
                        IN m_IndexingContext[m_Index].deviceID,
                        IN lastFingerprint);

                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }

                ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN m_IndexingContext[m_Index].deviceType,
                        IN msgToSendString,
                        IN parameterString);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
                }

                /* Update indexing state for device via DataProvider */
                tIndexingState indexingState = IDS_PARTIAL;
                tUInt percent = 0;
                ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN percent);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
                }

                /*For PSA RCC variant, Do not start fingerprint timer as fingerprint is calculated even if timer exceeds and updated in DB in OnFingerprintUpdate()*/
                if(LocalSPM::GetDataProvider().MonitorFingerPrintCalculation())
                {
                    StartFingerprintTimer(m_IndexingContext[m_Index].deviceID);
                }
            }
        }
    }
    return ret;
}
//TODO(kmv5cob):since video file and audio file count has been handled separately,fingerprint calculation has to provide video file separately .Once done checkresult has to handle it
tResult Indexer::CheckResult(const tFingerprint fingerprint,
        const tFingerprintStatus fingerprintStatus,
        const tNumberOfFiles numberOfFiles,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(fingerprint);
    VARTRACE(fingerprintStatus);
    VARTRACE(numberOfFiles);
    VARTRACE(deviceID);

    //TODO(pee1cob):Why discarding if the update arrived for device which is not the interested device ?-Discarding leads to performance issue!.
    //Instead,it could be stored to Indexing Context by calling OnFingerPrintUpdate(..)
    if(m_IndexingContext[m_Index].deviceID != deviceID)
    {
        ETG_TRACE_ERR(("Invalid deviceID - fingerprint ignored"));
        return MP_ERR_PM_INVALID_PARAM;
    }

    tResult ret = MP_NO_ERROR;

    if(numberOfFiles != NUMBER_OF_FILES_NONE) { //SUZUKI-22056
        m_IndexingContext[m_Index].totalNumberOfFiles = numberOfFiles;
    }

    switch (fingerprintStatus)
    {
        case FPS_NOT_AVAIL:
        {
            m_IndexingContext[m_Index].indexingMode = IM_FULL_SCAN;

            if ((NUMBER_OF_FILES_NONE != m_IndexingContext[m_Index].totalNumberOfFiles) //JAC2-5073
                    &&
                    //Do not set number of files (except 0) for mass storage devices and file browsing via DB enabled. It is done later with first flush
                    ((!LocalSPM::GetDataProvider().DBFileListWhileIndexing())
                            ||
                            (!IsMassStorageDevice(m_IndexingContext[m_Index].deviceType))
                            ||
                            (0 == m_IndexingContext[m_Index].totalNumberOfFiles)))
            {
                //Set first estimation of File counts for device via DBManager
                tNumberOfFiles numberOfVideoFiles = NUMBER_OF_FILES_NONE;
                tNumberOfFiles numberOfImageFiles = NUMBER_OF_FILES_NONE;
                //If totalNumberOfFiles 0,then update Video,Image count to 0
                if(0 == m_IndexingContext[m_Index].totalNumberOfFiles)
                {
                    numberOfVideoFiles = 0;
                    numberOfImageFiles = 0;
                }

                SetDeviceFileCounts(IN m_IndexingContext[m_Index].deviceID,IN m_IndexingContext[m_Index].totalNumberOfFiles,IN numberOfVideoFiles,IN numberOfImageFiles);
            }

            ResetAlbumArtIndexingComplete();

            //Sync with MyMedia DB Update tasks (running in MyMediaUpdateManager)
            SendSync();
            break;
        }
        case FPS_ERROR:
        {
            /* Send NOT_SUPPORTED message to own SM */
            ret = SendEvent(NOT_SUPPORTED, (char *)NULL);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
            break;
        }
        case FPS_OK_SKIP_INDEXING:
        {
            strncpy_r(OUT m_IndexingContext[m_Index].fingerprint, IN fingerprint, IN sizeof(m_IndexingContext[m_Index].fingerprint));
            ret = SendEvent(FINISHED, (char *)NULL);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
            break;
        }
        case FPS_OK_DELTA:
        case FPS_OK:
        {
            strncpy_r(OUT m_IndexingContext[m_Index].fingerprint, IN fingerprint, IN sizeof(m_IndexingContext[m_Index].fingerprint));

            /* Get stored fingerprint for device from DBManager */
            tFingerprint storedFingerprint = {0};
            ret = LocalSPM::GetDBManager().GetFingerprint(OUT storedFingerprint, IN m_IndexingContext[m_Index].deviceID);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while getting indexing state from DBManager (ErrorCode:%s)", errorString(ret)));
            }

            /* Get stored indexing state from indexing state map for device */
            tIndexingState storedIndexingState = IDS_NOT_STARTED;
            map<tDeviceID, tIndexingState>::iterator iterStoredIndexingStates = m_StoredIndexingStates.find(m_IndexingContext[m_Index].deviceID);
            if(iterStoredIndexingStates != m_StoredIndexingStates.end())
            {
                storedIndexingState = (tIndexingState)iterStoredIndexingStates->second;
            }

            VARTRACE(storedFingerprint);
            VARTRACE(storedIndexingState);
            if((strcmp(fingerprint, storedFingerprint) == 0) && (IDS_COMPLETE == storedIndexingState))
            {
                /* Send FINISHED message to own SM */
                ret = SendEvent(FINISHED, (char *)NULL);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }
            else
            {
                if(strcmp(fingerprint, storedFingerprint) == 0)
                {
                    m_IndexingContext[m_Index].indexingMode = IM_CONTINUE_SCAN;
                }
                else
                {
                    m_IndexingContext[m_Index].indexingMode = IM_FULL_SCAN;

                    ResetAlbumArtIndexingComplete();
                    LocalSPM::GetDataProvider().clearCache();
                    LocalSPM::GetOutputWrapper().UpdateNowPlaying();
                }

                //Do not set number of files (except 0) for mass storage devices and file browsing via DB enabled. It is done later with first flush
                if((!LocalSPM::GetDataProvider().DBFileListWhileIndexing())
                        ||
                        (!IsMassStorageDevice(m_IndexingContext[m_Index].deviceType))
                        ||
                        (0 == m_IndexingContext[m_Index].totalNumberOfFiles))
                {
                    tNumberOfFiles numberOfVideoFiles = NUMBER_OF_FILES_NONE;
                    tNumberOfFiles numberOfImageFiles = NUMBER_OF_FILES_NONE;
                    //If totalNumberOfFiles 0,then update Video,Image count also to 0.
                    if(0 == m_IndexingContext[m_Index].totalNumberOfFiles)
                    {
                        numberOfVideoFiles = 0;
                        numberOfImageFiles = 0;
                    }

                    /* Set first estimation of File counts for device via DBManager */
                    SetDeviceFileCounts(IN m_IndexingContext[m_Index].deviceID,IN m_IndexingContext[m_Index].totalNumberOfFiles,IN numberOfVideoFiles,IN numberOfImageFiles);
                }

                if(FPS_OK_DELTA == fingerprintStatus) {
                    /* Delta indexing: Update MyMedia DB */
                    ret = LocalSPM::GetDBManager().UpdateMyMediaDatabase(IN m_IndexingContext[m_Index].deviceID);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while updating MyMedia DB (ErrorCode:%s)", errorString(ret)));
                    }
                }

                //Sync with MyMedia DB Update tasks (running in MyMediaUpdateManager)
                SendSync();
            }
            break;
        }
        default:
        {
            ETG_TRACE_ERR(("Invalid fingerprintStatus type: %u",fingerprintStatus));
            ret = MP_ERR_PM_INVALID_PARAM;
            break;
        }
    }

    return ret;
}

tResult Indexer::OnFingerprintUpdate(const tFingerprint fingerprint,
        const tFingerprintStatus fingerprintStatus,
        const tNumberOfFiles numberOfFiles,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(fingerprint);
    VARTRACE(fingerprintStatus);
    VARTRACE(numberOfFiles);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    //check for indexing device
    //TODO(pee1cob):Understand why FingerPrint arrived for currently not indexing is discarded?
    //TODO(pee1cob):Why DEVICE_CONNECTED(to decide device to Index) done only for Apple devices?
    if(m_IndexingContext[m_Index].deviceID != deviceID)
    {
        tIndexingState indexingState = IDS_NOT_STARTED;
        ret = LocalSPM::GetDBManager().GetIndexingState(OUT indexingState, IN deviceID);
        if((IDS_COMPLETE == indexingState) && (FPS_OK == fingerprintStatus))
        {
            /* Setting valid fingerprint in DB if indexing gets compeleted before fingerprint */
            ret = LocalSPM::GetDBManager().SetFingerprint(IN deviceID, IN fingerprint);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating fingerprint in Database (ErrorCode:%s)", errorString(ret)));
            }
        }
        tDeviceInfo deviceInfo;
        ret =LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);

        if(MP_NO_ERROR == ret)
        {
            /* Trigger restart indexing for apple devices */
            if((fingerprintStatus != FPS_ERROR)
                    &&
                    (IsAppleDevice(deviceInfo.deviceType)))
            {
                ETG_TRACE_USR1(("FingerprintUpdate for deviceID %d -> Restart indexing", deviceID));

                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterDEVICE_CONNECTED(OUT parameterString, IN size, IN deviceID);
                SendEvent(IN DEVICE_CONNECTED, IN parameterString);
            }
        }
        else
        {
            ETG_TRACE_ERR(("Error getting device info from Database (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        switch (fingerprintStatus) {
            case FPS_OK:
            case FPS_OK_DELTA:
            {
                strncpy_r(OUT m_IndexingContext[m_Index].fingerprint, IN fingerprint, IN sizeof(m_IndexingContext[m_Index].fingerprint));

                if (NUMBER_OF_FILES_NONE != numberOfFiles)
                {
                    m_IndexingContext[m_Index].totalNumberOfFiles = numberOfFiles;

                    //Do not set number of files (except 0) for mass storage devices and file browsing via DB enabled. It is done later with first flush
                    if ((!LocalSPM::GetDataProvider().DBFileListWhileIndexing())
                            ||
                            (!IsMassStorageDevice(m_IndexingContext[m_Index].deviceType))
                            ||
                            (0 == numberOfFiles))
                    {
                        /* Update number of files, trigger update "playable content" for AudioMgr */
                        tNumberOfFiles numberOfVideoFiles = NUMBER_OF_FILES_NONE;
                        tNumberOfFiles numberOfImageFiles = NUMBER_OF_FILES_NONE;
                        //If totalNumberOfFiles 0,then update Video,Image count also to 0.
                        if(0 == m_IndexingContext[m_Index].totalNumberOfFiles)
                        {
                            numberOfVideoFiles = 0;
                            numberOfImageFiles = 0;
                        }

                        SetDeviceFileCounts(IN m_IndexingContext[m_Index].deviceID,IN m_IndexingContext[m_Index].totalNumberOfFiles,IN numberOfVideoFiles,IN numberOfImageFiles);
                    }
                }
                break;
            }
            default:
                //TODO(pee1cob):Why FPS_OK_SKIP_INDEXING,FPS_ERROR,FPS_NOT_AVAIL are not handled as in Indexer::CheckResult(..)?
                break;
        }
    }
    return MP_NO_ERROR;
}

tResult Indexer::CheckingStopped()
{
    ENTRY
#if 0
    tResult ret = MP_NO_ERROR;

    /* Clear release event and disarm timer of own state machine in case of still waiting */
    ret = ClearReleaseEvent();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while clearing release event via SMF (ErrorCode:%s)", errorString(ret)));
    }
#endif
    if(LocalSPM::GetDataProvider().MonitorFingerPrintCalculation())
    {
        StopFingerprintTimer();
    }
    return MP_NO_ERROR;
}

tResult Indexer::UpdateDeviceCurrentlyNotSupported()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Update indexing state for device via DataProvider */
    tIndexingState indexingState = IDS_CURRENTLY_NOT_SUPPORTED;
    m_IndexingContext[m_Index].percent = 0;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN m_IndexingContext[m_Index].percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::FinishedNoReindex()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    VARTRACE(m_IndexingContext[m_Index].totalNumberOfFiles);
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN m_IndexingContext[m_Index].deviceID);

    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error getting device info from Database (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        m_IndexingContext[m_Index].writtenFileCount = deviceInfo.numberOfAudioFiles + deviceInfo.numberOfVideoFiles + deviceInfo.numberOfImageFiles;
        m_IndexingContext[m_Index].writtenFileCountForVideo = deviceInfo.numberOfVideoFiles;
        m_IndexingContext[m_Index].writtenFileCountForImage = deviceInfo.numberOfImageFiles;
    }

    ret = Finished();
    /*SUZUKI-22187*/
    /*PSARCCB-4915, changing number of files recalculation as a generic behaviour, in case of NoReindex */

    ret = LocalSPM::GetDBManager().RecalculateNumberOfFiles(IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating no of files in DB (ErrorCode:%s)", errorString(ret)));
    }


    /* Update MyMedia DB */
    ret = LocalSPM::GetDBManager().UpdateMyMediaDatabase(IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating MyMedia DB (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::InitPrepare()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Update indexing state for device via DataProvider */
    tIndexingState indexingState = IDS_PARTIAL;
    m_IndexingContext[m_Index].percent = 0;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN m_IndexingContext[m_Index].percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    /* Get current number of media objects in DB from DataProvider */
    ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT m_NumberObjectsInDB);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting current number of media objects in DB from DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    /* Calculation of all limits */
    m_LimitNumberObjectsInDB = LocalSPM::GetDataProvider().LimitNumberObjectsInDB();
    m_LimitNumberObjectsPerDevice = LocalSPM::GetDataProvider().LimitNumberObjectsPerDevice();

    /* Inform ListControl that indexing has started */
    ret = LocalSPM::GetCustomControl().StartCheckListTimer(IN m_IndexingContext[m_Index].deviceID);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while removing media object at ListControl (ErrorCode:%s)", errorString(ret)));
    }

    /* Create UpdateIndexingProgressTimer */
    m_ProgressTimer.StartTimer(OUT m_ProgressTimerID,
            IN m_ProgressTimeout,
            IN m_ProgressTimeout,
            IN &LocalSPM::GetIndexer(),
            IN &ProgressTimerCallBack,
            IN NULL);

    return ret;
}

tResult Indexer::GetMetadata()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Send GET_METADATA message to specific DeviceControlSM */
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    char answerMsgString[64];

    strncpy_r(OUT msgToSendString, IN "GET_METADATA", IN sizeof(msgToSendString));
    strncpy_r(OUT answerMsgString, IN "IndexerSM::METADATA_ANSWER", IN sizeof(answerMsgString));

    ret = LocalSPM::GetUSBControl().ParameterGET_METADATA(OUT parameterString,
            IN size,
            IN m_IndexingContext[m_Index].mountPoint,
            IN m_IndexingContext[m_Index].readPosition,
            IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = LocalSPM::GetDeviceDispatcher().RouteMessageAnswer(IN m_IndexingContext[m_Index].deviceType,
            IN msgToSendString,
            IN parameterString,
            IN answerMsgString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessageAnswer of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult Indexer::CheckMetadata(const tMediaObjectPtr mediaObjectPtr, const tMetadataStatus metadataStatus)
{
    ENTRY;
    ETG_TRACE_USR3(("Indexer::CheckMetadata mediaObjectPtr: %p, metadataStatus:%u", mediaObjectPtr, metadataStatus));

    tResult ret = MP_NO_ERROR;
    tEvent *event = NULL;

    m_IndexingContext[m_Index].indexedFileCount++;

    switch(metadataStatus)
    {
        case MDS_SUCCESS:
            /* Store media object temporary */
            if(mediaObjectPtr)
            {
                m_MediaObject = *mediaObjectPtr;
                event = WRITE;
            }
            else //deviceControl timeout (e.g. read of video metadata via TagInfo takes too long)
            {
                /* Do skipping max 100 times */
                if(100 > m_IndexingContext[m_Index].errorSkipCount)
                {
                    m_IndexingContext[m_Index].errorSkipCount++;
                    event = SKIP;
                }
                else
                {
                    ETG_TRACE_FATAL(("Cancel indexing because of several errors!"));
                    event = ERROR;
                }
            }
            break;
        case MDS_REMOVED:
            /* Store media object temporary */
            if(mediaObjectPtr)
            {
                m_MediaObject = *mediaObjectPtr;
                event = REMOVE;
            }
            else
            {
                event = SKIP;
            }
            break;
        case MDS_FINISHED:
        case MDS_FINISHED_RECURSIVE_EMERGENCY_BREAK:
            //Don't do a restart from Indexer side because totalNumberOfFiles is only an approximate value
            //(e.g. USB: assigned by file count via extension).
            //if(m_IndexingContext[m_Index].totalNumberOfFiles <= m_IndexingContext[m_Index].indexedFileCount)
            //{
            event = FINISHED;
            //}
            //else
            //{
            //    m_IndexingContext[m_Index].readPosition[0] = '\0';
            //    event = RESTART;
            //}
            break;
        case MDS_FLUSH:
            event = FLUSH;
            break;
        case MDS_FILE_SKIP:
            event = SKIP;
            break;
        case MDS_FILE_ERROR:
            /* Do skipping max 100 times */
            if(100 > m_IndexingContext[m_Index].errorSkipCount)
            {
                m_IndexingContext[m_Index].errorSkipCount++;
                event = SKIP;
            }
            else // It looks like a general error
            {
                ETG_TRACE_FATAL(("Cancel indexing because of several errors!"));
                event = ERROR;
            }
            break;
        case MDS_DEVICE_ERROR:
        {
            //call additional DBDeviceRemoved manually to cancel indexing loop
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            ParameterDEVICE_REMOVED(OUT parameterString, IN size, IN m_IndexingContext[m_Index].deviceID);
            ret = SendEvent(DEVICE_REMOVED, IN parameterString);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("SendEvent(DEVICE_REMOVED) failed (ErrorCode:%s)", errorString(ret)));
            }

            event = RESTART;
            break;
        }
        case MDS_ERROR_RESTART:
        {
            /* trigger restart indexing */
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            ParameterDEVICE_CONNECTED(OUT parameterString, IN size, IN m_IndexingContext[m_Index].deviceID);
            SendEvent(IN DEVICE_CONNECTED, IN parameterString);

            event = ERROR;
            break;
        }
        default:
            event = ERROR;
            break;
    }

    if(event != NULL) {
        /* Send event message to own SM */
        ret = SendEvent(IN event, IN (char *)NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    delete mediaObjectPtr;

    return MP_NO_ERROR;
}

tResult Indexer::MetadataAnswerNotConsumed(const tMediaObjectPtr mediaObjectPtr, const tMetadataStatus metadataStatus)
{
    ENTRY;
    VARTRACE(metadataStatus);

    ETG_TRACE_ERR(("ERROR: Metadata lost for indexing!"));
    if(mediaObjectPtr) {
        delete mediaObjectPtr;
    }

    return MP_NO_ERROR;
}


tReturnValue Indexer::IsLimitNotReached()
{
    ENTRY

    tReturnValue returnValue = true;

    if((m_LimitNumberObjectsInDB < (m_NumberObjectsInDB + m_IndexingContext[m_Index].writtenFileCount))
            ||
            (m_LimitNumberObjectsPerDevice <= m_IndexingContext[m_Index].writtenFileCount))
    {
        returnValue = false;
    }

    return returnValue;
}

tResult Indexer::PutMetadata()
{
    ENTRY;
    VARTRACE(m_MediaObject.fileName);
    VARTRACE(m_MediaObject.title);
    tResult ret = MP_NO_ERROR;

    string title(m_MediaObject.title);
    size_t pos = title.find_first_not_of(" ._");    //title having leading character " " or "." or "_"

    if(pos){
        title.erase(0, pos);                        //removing leading character " " or "." or "_" from title
        strncpy(m_MediaObject.title, title.c_str(), sizeof(m_MediaObject.title));
        // sim4hi 180629: overide last character in string with termination character as fix for CID-183918
        m_MediaObject.title[sizeof(m_MediaObject.title)-1]=0;
        VARTRACE(m_MediaObject.title);
    }

    m_IndexingContext[m_Index].writtenFileCount++;
    if( CTY_VIDEO == m_MediaObject.catType )
    {
        m_IndexingContext[m_Index].writtenFileCountForVideo++;
    }
    else if( CTY_IMAGE == m_MediaObject.catType)
    {
        m_IndexingContext[m_Index].writtenFileCountForImage++;
    }
    strncpy_r(OUT m_IndexingContext[m_Index].readPosition, IN m_MediaObject.fileName, IN sizeof(m_IndexingContext[m_Index].readPosition));

    /* Store media object in DB */
    ret = LocalSPM::GetDBManager().StoreMediaObject(IN m_MediaObject);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while storing media object in DB (ErrorCode:%s)", errorString(ret)));
    }

    //When First Playable object found in "this" Connection:Audio,Video,Image in a MSD/MTP Device,
    //Flush it to MediaObjects table + set its count as "1" in Devices table.
    //Thus,
    //a)SourceAvailability announced(if not yet done based on filecount of previous connection info).
    //b)DeviceInfo announced with valid audio,video,image item count(NCG3D-103083)(if not yet done based on their count on previous connection info).

    //Note:member:"flushFirstPlayable***Object" denotes first Flush occured in "this" connection.
    //Hence if first flush occurs,***fileCount overwrites its count found in previous connection in "Devices" table.
    if((LocalSPM::GetDataProvider().DBFileListWhileIndexing())
            &&
            ((IsMassStorageDevice(IN m_IndexingContext[m_Index].deviceType)) || (DTY_MTP == m_IndexingContext[m_Index].deviceType))
            &&
            ((((!m_IndexingContext[m_Index].flushFirstPlayableObject) && (CTY_SONG == m_MediaObject.catType))
                    ||
                    ((!m_IndexingContext[m_Index].flushFirstPlayableVideoObject)&& (CTY_VIDEO == m_MediaObject.catType))
                    ||
                    ((!m_IndexingContext[m_Index].flushFirstPlayableImageObject) && (CTY_IMAGE == m_MediaObject.catType)))
                    &&
                    (FNP_PLAYABLE == m_MediaObject.notPlayable)))
    {

        /* Flush the media object cache */
        FlushMediaObjects();

        /* Set first estimation of files for device via DBManager */
        //TODO(pee1cob):Regardless of SeparateMediaContent,Update audioCount,VideoCount in DB in their corresponding Columns

        if(CTY_SONG == m_MediaObject.catType)
        {
            tNumberOfFiles numberOfFiles = 1;

            ret = LocalSPM::GetDBManager().SetNumberOfAudioFiles(IN m_IndexingContext[m_Index].deviceID, IN numberOfFiles);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating number of files at DBManager (ErrorCode:%s)", errorString(ret)));
            }
            m_IndexingContext[m_Index].flushFirstPlayableObject = true;
        }
        else if(CTY_VIDEO == m_MediaObject.catType)
        {
            tNumberOfFiles numberOfVideoFiles = 1;

            ret = LocalSPM::GetDBManager().SetNumberOfVideoFiles(IN m_IndexingContext[m_Index].deviceID, IN numberOfVideoFiles);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating number of video files at DBManager (ErrorCode:%s)", errorString(ret)));
            }
            m_IndexingContext[m_Index].flushFirstPlayableVideoObject = true;
        }


        if(CTY_IMAGE == m_MediaObject.catType)
        {
            tNumberOfFiles numberOfImageFiles = 1;

            ret = LocalSPM::GetDBManager().SetNumberOfImageFiles(IN m_IndexingContext[m_Index].deviceID, IN numberOfImageFiles);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while updating number of image files at DBManager (ErrorCode:%s)", errorString(ret)));
            }
            m_IndexingContext[m_Index].flushFirstPlayableImageObject = true;
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::RemoveMetadata()
{
    ENTRY;
    VARTRACE(m_MediaObject.UUID);

    tResult ret = MP_NO_ERROR;

    /* Remove media object from DB */
    ret = LocalSPM::GetDBManager().RemoveMediaObjectByUUID(IN m_IndexingContext[m_Index].deviceID, IN m_MediaObject.UUID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while removing media object from DB (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::FlushMediaObjects()
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    /* Flush the media object cache */
    ret = LocalSPM::GetDBManager().StoreMediaObjectEnd(false);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while flushing media object cache in DB (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::SendGoOn()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Send GO_ON message to own SM via dispatcher to create external event */
    char msgToSendString[32];
    strncpy_r(OUT msgToSendString, IN "IndexerSM::GO_ON", IN sizeof(msgToSendString));

    ret = Dispatcher::GetInstance().SendMessage(IN msgToSendString, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending external event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}
tResult Indexer::SendReindexing(const tDeviceCount deviceCount, const tDeviceID deviceID)   //Roadmap 13005 SupportOfInternalFlash
{
    ENTRY;
    VARTRACE(deviceCount);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ret = ParameterREINDEXING(OUT parameterString, IN size, IN deviceCount, IN deviceID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    //just send as internal event
    ret = SendEvent(REINDEXING, IN parameterString);

    return ret;
}

tResult Indexer::Reindexing(const tDeviceCount deviceCount, const tDeviceID deviceID)   //Roadmap 13005 SupportOfInternalFlash
{
    ENTRY;
    VARTRACE(deviceCount);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;

#if 0
    //check if device is connected or at least attached
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    } else {

        if(!IsAppleDevice(deviceInfo.deviceType) && LocalSPM::GetDataProvider().IsIndexingSupported(IN deviceInfo.deviceType)) {

            /* Remove device from MyMedia DB */
            ret = LocalSPM::GetDBManager().RemoveDeviceFromMyMedia(IN deviceID);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while removing device from MyMedia DB (ErrorCode:%s)", errorString(ret)));
            }
            /* Reset variables of device in indexing context */
            for( tUInt i = 0; i < m_IndexingContext.size(); i++ )
            {
                if( deviceID == m_IndexingContext[i].deviceID )
                {
                    InitIndexingContext(OUT m_IndexingContext[i]);
                    m_IndexingContext[i].deviceID = deviceID;

                    //resetfingerprint
                    ret = LocalSPM::GetDBManager().SetFingerprint(IN m_IndexingContext[i].deviceID, IN m_IndexingContext[i].fingerprint, IN m_IndexingContext[m_Index].fingerprint, IN m_IndexingContext[m_Index].writtenFileCount - m_IndexingContext[m_Index].writtenFileCountForVideo - m_IndexingContext[m_Index].writtenFileCountForImage,m_IndexingContext[m_Index].writtenFileCountForVideo);
                    if (MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while updating fingerprint at DataProvider (ErrorCode:%s)", errorString(ret)));
                    }

                    //trigger new indexing
                    tAllParameters parameterString;
                    size_t size = sizeof(parameterString);
                    ParameterDEVICE_CONNECTED(OUT parameterString, IN size, IN deviceID);
                    SendEvent(IN DEVICE_CONNECTED, IN parameterString);
                    return MP_NO_ERROR;
                }
            }
        }
    }
#else
    //check if device is connected or at least attached
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    } else {

        if(!IsAppleDevice(deviceInfo.deviceType) && LocalSPM::GetDataProvider().IsIndexingSupported(IN deviceInfo.deviceType)) {

            VARTRACE(deviceInfo.connectionState);
            VARTRACE(deviceInfo.deviceState);
            if (deviceInfo.connectionState != CS_DISCONNECTED) {

                //remove obsolete content, call DeviceControl::RemoveDeviceConnection
                IsSameDevice(IN deviceID);

                //reset fingerprint
                ret = LocalSPM::GetDBManager().SetFingerprint(IN deviceID, IN FINGERPRINT_INVALID, IN NUMBER_OF_FILES_NONE,IN NUMBER_OF_FILES_NONE);
                if (MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("Error while updating fingerprint at DataProvider (ErrorCode:%s)", errorString(ret)));
                }

                //initialize DeviceControl again
                ret = DBInitializeDevice(IN deviceCount, IN deviceID);
                if (MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("Reindexing not possible - DBInitializeDevice failed"));
                } else {
                    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
                    if (MP_NO_ERROR != ret) {
                        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
                    }
                    if(deviceInfo.deviceState != DS_OK) {
                        ETG_TRACE_ERR(("Reindexing not possible - device state not OK"));
                    } else {
                        //success
                        return MP_NO_ERROR;
                    }
                }
            }
        }
    }
#endif
    ETG_TRACE_ERR(("Reindexing not possible"));
    //not connected return to general indexing loop
    SendEvent(IN ERROR, (char *)NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::ReindexingInitialized(const tDeviceName deviceName, const tDeviceID deviceID, const tConnectionState connectionState)
{
    ENTRY;
    VARTRACE(deviceName);
    VARTRACE(deviceID);
    VARTRACE(connectionState);

    tResult ret = MP_NO_ERROR;

    /* Check if device is connected or at least attached */
    bool send_db_trigger = true;
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s)", errorString(ret)));
    } else {

        VARTRACE(deviceInfo.connectionState);
        if(deviceInfo.connectionState != CS_CONNECTED && CS_CONNECTED == connectionState) {
            send_db_trigger = false; //db trigger case, prevent multiple event
        }
    }
    VARTRACE(send_db_trigger);

    /* Update indexing state for device via DataProvider */
    tIndexingState indexingState = IDS_NOT_STARTED;
    tIndexingPercentComplete percent = 0;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN deviceID, IN indexingState, IN percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    /* Insert device to indexing state map but with IDS_NOT_STARTED */
    m_StoredIndexingStates[deviceID] = IDS_NOT_STARTED;

    /* Write device into DB via DataProvider */
    tDeviceState deviceState = DS_OK;
    ret = LocalSPM::GetDataProvider().DeviceInitialized(IN deviceID, IN deviceName, IN connectionState, IN deviceState);
    if(MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while writing device into DB via DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    if(send_db_trigger) {

        //Call DBDeviceConnected manually to enter general indexing loop
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterDEVICE_CONNECTED(OUT parameterString, IN size, IN deviceID);
        ret = SendEvent(DEVICE_CONNECTED, IN parameterString);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("SendEvent(DEVICE_CONNECTED) failed (ErrorCode:%s)", errorString(ret)));
        }
    }

    return MP_NO_ERROR;
}

tResult Indexer::FreeSpace()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Check if another device to free is available in DB */
    tDeviceID oldestDeviceID = DEVICE_ID_NOT_SET;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceID> oldDeviceIDs;
    ret = LocalSPM::GetDBManager().GetOldDevices(OUT numberOfDevices, OUT oldDeviceIDs);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting oldest device from DB (ErrorCode:%s)", errorString(ret)));
    }
    else if (0 < oldDeviceIDs.size())
    {
        oldestDeviceID = oldDeviceIDs[0];
    }

    if((DEVICE_ID_NOT_SET != oldestDeviceID) //found a device to free
            &&
            (m_IndexingContext[m_Index].deviceID != oldestDeviceID) //not the current indexing device
            &&
            (m_LimitNumberObjectsPerDevice > m_IndexingContext[m_Index].writtenFileCount)) //file count is under limit per device
    {
        /* Free space in DB via DBManager */
        ret = LocalSPM::GetDBManager().FreeSpace();
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while freeing space in DB at DBManager (ErrorCode:%s)", errorString(ret)));
        }

        /* Get current number of media objects in DB from DataProvider */
        ret = LocalSPM::GetDBManager().GetNumberOfMediaObjectsInDB(OUT m_NumberObjectsInDB);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while getting current number of media objects in DB from DataProvider (ErrorCode:%s)", errorString(ret)));
        }
        else
        {
            ETG_TRACE_USR3(("Number of media objects in DB after free space: %d", m_NumberObjectsInDB));
        }

        /* Send WRITE message to own SM */
        ret = SendEvent(WRITE, (char *)NULL);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Send FINISHED message to own SM */
        ret = SendEvent(FINISHED, (char *)NULL);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult Indexer::IncrementFileCounter()
{
    ENTRY

    m_IndexingContext[m_Index].skippedFileCount++;
    // Fingerprint calculation does not find playable file in MTP.
    if(m_IndexingContext[m_Index].deviceType != DTY_MTP)
    {
        if(0 < (int)m_IndexingContext[m_Index].totalNumberOfFiles)
        {
            m_IndexingContext[m_Index].totalNumberOfFiles--;
        }
    }

    ETG_TRACE_ERR(("Skip indexing file. deviceID:%d, errorSkipCount/skippedFileCount:%d/%d, behind readPosition:%s",
            m_IndexingContext[m_Index].deviceID, m_IndexingContext[m_Index].errorSkipCount, m_IndexingContext[m_Index].skippedFileCount, m_IndexingContext[m_Index].readPosition));

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::IndexingStopped()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Clear release event and disarm timer of own state machine in case of still waiting */
    ret = ClearReleaseEvent();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while clearing release event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    /* Delete UpdateIndexingProgressTimer */
    m_ProgressTimer.CancelTimer(IN m_ProgressTimerID);

    /* Inform ListControl that indexing has stopped */
    ret = LocalSPM::GetCustomControl().StopCheckListTimer();
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while removing media object at ListControl (ErrorCode:%s)", errorString(ret)));
    }

    /* Flush the media object cache */
    ret = LocalSPM::GetDBManager().StoreMediaObjectEnd();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while flushing media object cache in DB (ErrorCode:%s)", errorString(ret)));
    }

    /* Final list update */
    ret = LocalSPM::GetListControl().CheckLists(IN m_IndexingContext[m_Index].deviceID);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while calling CheckLists at ListControl (ErrorCode:%s)", errorString(ret)));
    }

    /* Calculate indexing percentage value */
    if(0 < (int)m_IndexingContext[m_Index].totalNumberOfFiles)
    {
        m_IndexingContext[m_Index].percent = (m_IndexingContext[m_Index].indexedFileCount*1000/m_IndexingContext[m_Index].totalNumberOfFiles +5)/10; //+5: for correct rounding
    }
    else
    {
        m_IndexingContext[m_Index].percent = 100;
    }

    //clipping
    if( m_IndexingContext[m_Index].percent > 100) {
        m_IndexingContext[m_Index].percent = 100;
    }

    /* Update indexing progress via DataProvider */
    tIndexingState indexingState = IDS_PARTIAL;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN m_IndexingContext[m_Index].percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult Indexer::FinishedComplete()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    ret = Finished();

    /* Update Main DB */
    //Note: It's important to do it after flushing the media object cache in DBManager (StoreMediaObjectEnd)
    ret = LocalSPM::GetDBManager().UpdateMain(IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating Main DB (ErrorCode:%s)", errorString(ret)));
    }

    /* Update lists here twice additional to IndexingStopped for all lists because list size can be decreased (GMMY16-4154) */
    ret = LocalSPM::GetListControl().CheckLists(IN m_IndexingContext[m_Index].deviceID);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while calling CheckLists at ListControl (ErrorCode:%s)", errorString(ret)));
    }

    /* Remove device from MyMedia DB immediately after cleanup Main DB in case of no MyMedia support */
    if (0 == LocalSPM::GetDataProvider().MyMediaSupported())
    {
        ret = LocalSPM::GetDBManager().RemoveDeviceFromMyMedia(IN m_IndexingContext[m_Index].deviceID);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while removing device from MyMedia DB (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::ErrorComplete()
{
    ENTRY

    /* Reset fingerprint */
    strncpy_r(OUT m_IndexingContext[m_Index].fingerprint, IN FINGERPRINT_INVALID, IN sizeof(m_IndexingContext[m_Index].fingerprint));

    FinishedComplete();

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult Indexer::UpdateIndexingProgressTimer()
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

    /* Calculate indexing percentage value */
    tIndexingPercentComplete percent;
    if(0 < (int)m_IndexingContext[m_Index].totalNumberOfFiles)
    {
        percent = (m_IndexingContext[m_Index].indexedFileCount*1000/m_IndexingContext[m_Index].totalNumberOfFiles +5)/10; //+5: for correct rounding
    }
    else
    {
        percent = 1; //percent
    }

    //clipping
    if( percent > 100) {
        percent = 100;
    }

    VARTRACE(percent);
    if((m_IndexingContext[m_Index].percent+1) <= percent) //change is significant (every 1%)
    {
        m_IndexingContext[m_Index].percent = percent;

        /* Update indexing progress via DataProvider */
        tIndexingState indexingState = IDS_PARTIAL;
        ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN percent);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult Indexer::DBDeviceIndexing(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY
    ETG_TRACE_USR3(("Indexer::DBDeviceIndexing deviceCount:%u, deviceID: %u", deviceCount, deviceID));

    tResult ret = MP_NO_ERROR;

    /* Update HMI of property IndexingState */
    ret = LocalSPM::GetOutputWrapper().UpdateIndexingState();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating NowPlaying via OutputWrapper (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult Indexer::HandleAnswerTimeout()
{
    ENTRY
    ETG_TRACE_ERR(("Indexer::HandleAnswerTimeout"));

    tResult ret = MP_NO_ERROR;

    /* Send releasing event to own waiting state machine */
    ret = ReleaseWaiting();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while releasing waiting via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult Indexer::AutoRegisterOnDBTrigger(const tTriggerState autoRegister)
{
    ENTRY
    ETG_TRACE_USR3(("Indexer::AutoRegisterOnDBTrigger autoRegister: %u", autoRegister));

    m_AutoRegisterOnDBTrigger = autoRegister;

    return MP_NO_ERROR;
}

tResult Indexer::SwitchDBTrigger(const tTriggerType trigger, const tTriggerState active)
{
    ENTRY
    ETG_TRACE_USR3(("Indexer::SwitchDBTrigger trigger:%u, active: %u", trigger, active));

    tResult ret = MP_NO_ERROR;

    if(TS_ON == active)
    {
        /* Register on DB trigger */
        if((TT_DB_DEVICE_INSERTED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevInserted)
            {
                ret = LocalSPM::GetDBManager().OnInsertTrigger(OUT m_TriggerID_DevInserted,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_INSERTED",
                        IN "Devices",
                        IN "WHEN NEW.ID>0 AND NEW.ConnectionState=1",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=NEW.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_ATTACHED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevAttached)
            {
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevAttached,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_INSERTED",
                        IN "Devices",
                        IN "ConnectionState",
                        IN "WHEN NEW.ID>0 AND OLD.ConnectionState<>1 AND NEW.ConnectionState=1",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_CONNECTED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevConnected)
            {
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevConnected,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_CONNECTED",
                        IN "Devices",
                        IN "ConnectionState",
                        IN "WHEN NEW.ID>0 AND OLD.ConnectionState<>0 AND NEW.ConnectionState=0",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_INDEXING_STATE == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevIndexingState)
            {
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevIndexingState,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_INDEXING",
                        IN "Devices",
                        IN "IndexingState",
                        IN "WHEN NEW.ID>0 AND OLD.IndexingState<>NEW.IndexingState",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_INDEXING_PROGRESS == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevIndexingProgress)
            {
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevIndexingProgress,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_INDEXING",
                        IN "Devices",
                        IN "IndexingProgress",
                        IN "WHEN NEW.ID>0 AND OLD.IndexingProgress<>NEW.IndexingProgress",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_REMOVED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevRemoved)
            {
                //Roadmap 13035_Overtemperature: ConnectionState 2=CS_DISCONNECTED, 6=CS_OVERTEMP, 7=CS_ON_HOLD
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevRemoved,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_REMOVED",
                        IN "Devices",
                        IN "ConnectionState",
                        IN "WHEN NEW.ID>0 AND OLD.ConnectionState<>2 AND OLD.ConnectionState<>6 AND OLD.ConnectionState<>7 AND (NEW.ConnectionState=2 OR NEW.ConnectionState=6 OR NEW.ConnectionState=7)",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if(LocalSPM::GetDataProvider().AVRCPLowerProfileSupported() &&
                ((TT_DB_DEVICE_VERSION_UPDATED == trigger)||(TT_ALL == trigger)))
        {
            if(0 == m_TriggerID_DevVersionUpdated)
            {
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevVersionUpdated,
                        IN IndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_VERSION_UPDATED",
                        IN "Devices",
                        IN "FirmwareVersion",
                        IN "WHEN NEW.ID>0 AND OLD.FirmwareVersion='1.0' AND (NEW.FirmwareVersion='1.4' OR NEW.FirmwareVersion='1.3')AND (OLD.ConnectionState=1 OR OLD.ConnectionState=0)",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
    }
    else
    {
        /* Deregister on DB trigger */
        if((TT_DB_DEVICE_INSERTED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevInserted)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevInserted);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevInserted = 0;
            }
        }
        if((TT_DB_DEVICE_ATTACHED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevAttached)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevAttached);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevAttached = 0;
            }
        }
        if((TT_DB_DEVICE_CONNECTED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevConnected)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevConnected);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevConnected = 0;
            }
        }
        if((TT_DB_DEVICE_INDEXING_STATE == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevIndexingState)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevIndexingState);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevIndexingState = 0;
            }
        }
        if((TT_DB_DEVICE_INDEXING_PROGRESS == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevIndexingProgress)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevIndexingProgress);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevIndexingProgress = 0;
            }
        }
        if((TT_DB_DEVICE_REMOVED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevRemoved)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevRemoved);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevRemoved = 0;
            }
        }
        if(LocalSPM::GetDataProvider().AVRCPLowerProfileSupported() && ((TT_DB_DEVICE_VERSION_UPDATED == trigger)
                ||
                (TT_ALL == trigger)))
        {
            if(m_TriggerID_DevVersionUpdated)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevVersionUpdated);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevVersionUpdated = 0;
            }
        }
    }

    return ret;
}

tResult Indexer::Finished()
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

    VARTRACE(m_IndexingContext[m_Index].writtenFileCount);
    VARTRACE(m_IndexingContext[m_Index].writtenFileCountForImage);
    VARTRACE(m_IndexingContext[m_Index].writtenFileCountForVideo);

    tFingerprint fingerprint = {0};
    if(0 < strlen_r(m_IndexingContext[m_Index].fingerprint))
    {
        strncpy_r(OUT fingerprint, IN m_IndexingContext[m_Index].fingerprint, IN sizeof(fingerprint));
    }
    else
    {
        /* Set fingerprint as invalid if fingerprint stored in indexing content is empty ie. when indexing gets completed before fingerprint calculation*/
        strncpy_r(OUT fingerprint, IN FINGERPRINT_INVALID, IN sizeof(fingerprint));
    }
    /* Update fingerprint of device via DataProvider */
    ret = LocalSPM::GetDBManager().SetFingerprint(IN m_IndexingContext[m_Index].deviceID, IN fingerprint, IN m_IndexingContext[m_Index].writtenFileCount - m_IndexingContext[m_Index].writtenFileCountForVideo - m_IndexingContext[m_Index].writtenFileCountForImage,m_IndexingContext[m_Index].writtenFileCountForVideo);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating fingerprint at DataProvider (ErrorCode:%s)", errorString(ret)));
    }
    tNumberOfFiles numberOfImageFiles = m_IndexingContext[m_Index].writtenFileCountForImage;
    ret = LocalSPM::GetDBManager().SetNumberOfImageFiles(IN m_IndexingContext[m_Index].deviceID, IN numberOfImageFiles);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating number of Image files at DBManager (ErrorCode:%s)", errorString(ret)));
    }

    // CRQ 300254 Support for Folder Up/Down command
    ret = LocalSPM::GetListControl().UpdateFolderHierarchyForDeviceID(IN m_IndexingContext[m_Index].deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating folder hierarchy for the device (ErrorCode:%s)", errorString(ret)));
    }

    /* Update indexing progress via DataProvider */
    tIndexingState indexingState = IDS_COMPLETE;
    m_IndexingContext[m_Index].percent = 100;
    ret = LocalSPM::GetDBManager().SetIndexingState(IN m_IndexingContext[m_Index].deviceID, IN indexingState, IN m_IndexingContext[m_Index].percent);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while updating indexing state at DataProvider (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}
bool Indexer::ProgressTimerCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY_INTERNAL
    (void)userData;
    tResult ret = MP_NO_ERROR;

    if(NULL == instance)
    {
        ETG_TRACE_ERR(("Invalid parameter: instance is NULL (ErrorCode:%s)", errorString(ret)));
        ret = MP_ERR_PM_INVALID_PARAM;
    }
    else
    {
        ETG_TRACE_USR3(("Indexer::ProgressTimerCallBack timerID:0x%X, instance:0x%X", (int)((long)timerID), instance));

        /* Send timeout event */
        Indexer *self = (Indexer*)instance;
        ret = self->SendEventByName("UPDATE_INDEXING_PROGRESS_TIMER", (char *)NULL);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return (ret == MP_NO_ERROR);
}

tResult Indexer::FileCountCompWithDB(const tNumberOfFiles numberOfPlaylist,const tNumberOfFiles numberOfVideo,const tNumberOfFiles numberOfTrack,const tDeviceID deviceID,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(numberOfPlaylist);
    VARTRACE(numberOfVideo);
    VARTRACE(numberOfTrack);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;
    tFingerprintStatus fingerprintStatus=FPS_OK;

    /* Get stored indexing state from indexing state map for device */
    tIndexingState storedIndexingState = IDS_NOT_STARTED;
    map<tDeviceID, tIndexingState>::iterator iterStoredIndexingStates = m_StoredIndexingStates.find(deviceID);
    if(iterStoredIndexingStates != m_StoredIndexingStates.end())
    {
        storedIndexingState = (tIndexingState)iterStoredIndexingStates->second;
    }
    VARTRACE(storedIndexingState);
    if(storedIndexingState == IDS_COMPLETE){
        fingerprintStatus = FPS_OK_SKIP_INDEXING;
    }
    if( FPS_OK_SKIP_INDEXING == fingerprintStatus ){
        tNumberOfFiles numberOfPlaylistInDB;
        ret=LocalSPM::GetDBManager().GetNumberOfPlaylist(OUT numberOfPlaylistInDB, IN deviceID);
        VARTRACE(numberOfPlaylistInDB);
        if(ret == MP_NO_ERROR){
            if(numberOfPlaylist != numberOfPlaylistInDB){
                fingerprintStatus = FPS_OK;
            }
        }else{
            fingerprintStatus = FPS_OK;
        }
    }
    if( FPS_OK_SKIP_INDEXING == fingerprintStatus ){
        tNumberOfFiles numberOfVideoInDB;
        ret=LocalSPM::GetDBManager().GetNumberOfVideos(OUT numberOfVideoInDB, IN deviceID);
        VARTRACE(numberOfVideoInDB);
        if(ret == MP_NO_ERROR){
            if(numberOfVideo != numberOfVideoInDB){
                fingerprintStatus = FPS_OK;
            }
        }else{
            fingerprintStatus = FPS_OK;
        }
    }
    if( FPS_OK_SKIP_INDEXING == fingerprintStatus ) {
        tNumberOfFiles numberOfTrackInDB;
        ret=LocalSPM::GetDBManager().GetNumberofTracks(OUT numberOfTrackInDB, IN deviceID);
        VARTRACE(numberOfTrackInDB);
        if(ret == MP_NO_ERROR){
            if(numberOfTrack != numberOfTrackInDB){
                fingerprintStatus = FPS_OK;
            }
        }else{
            fingerprintStatus = FPS_OK;
        }
    }
    if( FPS_OK_SKIP_INDEXING == fingerprintStatus ) {
        tNumberOfFiles numberOfFiles=numberOfTrack+numberOfPlaylist;
        ret = LocalSPM::GetDBManager().SetNumberOfAudioFiles(IN deviceID,IN numberOfFiles);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while updating number of files at DBManager (ErrorCode:%s)", errorString(ret)));
        }
        ret = LocalSPM::GetDBManager().SetNumberOfVideoFiles(IN deviceID,IN numberOfVideo);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while updating number of files at DBManager (ErrorCode:%s)", errorString(ret)));
        }
        LocalSPM::GetDBManager().DropPlaylistTempTable();
        LocalSPM::GetDBManager().DropTrackTempTable();
    }
    tDeviceType deviceType=DTY_UNKNOWN;
    if(m_IndexingContext[m_Index].deviceID == deviceID){
        deviceType = m_IndexingContext[m_Index].deviceType;
    }else{
        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error While GetDeviceInfo"));
        }else{
            deviceType = deviceInfo.deviceType;
        }
    }
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    strncpy_r(OUT msgToSendString, IN "FILE_COUNT_COMP_ANSWER", IN sizeof(msgToSendString));
    ret = LocalSPM::GetUSBControl().ParameterFILE_COUNT_COMP_ANSWER(OUT parameterString, IN size,IN deviceID,IN fingerprintStatus,IN indexSession);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceType,
            IN msgToSendString,
            IN parameterString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }
    VARTRACE(ret);
    return MP_NO_ERROR;
}

tResult Indexer::PlaylistNameCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfPlaylist,tVoidPtr playlistRecordsPtr,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(index);
    VARTRACE(chunk);
    VARTRACE(numberOfPlaylist);
    tNumberOfFiles numberOfEntry=0;
    tNumberOfFiles numberOfEntryAfterRemove=0;
    tFingerprintStatus fingerprintStatus=FPS_OK_SKIP_INDEXING;
    tResult ret = MP_NO_ERROR;
    vector<string> *playlistNameVectorPtr=(vector<string>*)playlistRecordsPtr;
    if(NULL != playlistNameVectorPtr){

        if(index == 1){
            ret = LocalSPM::GetDBManager().CreatePlaylistTempTable();
            if(ret == MP_NO_ERROR){
                ret = LocalSPM::GetDBManager().FillPlaylistTempTable(deviceID);
                if(ret != MP_NO_ERROR){
                    ETG_TRACE_ERR(("Error While Fill the PlayList Temp Table"));
                }
            }else{
                ETG_TRACE_ERR(("Error While Creating PlayList Temp Table"));
            }
        }
        if(ret == MP_NO_ERROR){
            ret = LocalSPM::GetDBManager().GetNumberofPlaylistTempEntry(numberOfEntry);
            VARTRACE(numberOfEntry);
        }
        if(ret == MP_NO_ERROR){
            for(tUInt i=0;i<playlistNameVectorPtr->size();i++){
                tPlaylistName playlistName;
                strncpy_r(playlistName,playlistNameVectorPtr->at(i).c_str(),sizeof(playlistName));
                ret=LocalSPM::GetDBManager().RemovePlaylistEntryFromTemp(playlistName);
                if(ret == MP_NO_ERROR){
                    numberOfEntry--;
                    ret = LocalSPM::GetDBManager().GetNumberofPlaylistTempEntry(numberOfEntryAfterRemove);
                    if(ret == MP_NO_ERROR){
                        if(numberOfEntryAfterRemove != numberOfEntry){
                            fingerprintStatus=FPS_OK;
                            break;
                        }
                    }else {
                        ETG_TRACE_ERR(("Error While GetNumberofPlaylistTempEntry"));
                        break;
                    }
                }else{
                    ETG_TRACE_ERR(("Error While RemovePlaylistEntryFromTemp"));
                    break;
                }
            }
        }
        delete playlistNameVectorPtr;
    }

    if( FPS_OK_SKIP_INDEXING != fingerprintStatus || ( numberOfPlaylist == (index+chunk)))
    {
        LocalSPM::GetDBManager().DropPlaylistTempTable();
    }
    tDeviceType deviceType=DTY_UNKNOWN;
    if(m_IndexingContext[m_Index].deviceID == deviceID){
        deviceType = m_IndexingContext[m_Index].deviceType;
    }else{
        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error While GetDeviceInfo"));
        }else{
            deviceType = deviceInfo.deviceType;
        }
    }
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    strncpy_r(OUT msgToSendString, IN "PLAYLIST_NAME_COMP_ANSWER", IN sizeof(msgToSendString));
    ret = LocalSPM::GetUSBControl().ParameterPLAYLIST_NAME_COMP_ANSWER(OUT parameterString, IN size,IN deviceID,IN fingerprintStatus,IN index,IN chunk,IN indexSession);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceType,
            IN msgToSendString,
            IN parameterString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }
    VARTRACE(ret);
    return MP_NO_ERROR;
}

tResult Indexer::VideoUIDCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfVideo,tVoidPtr trackInfosPtr,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(index);
    VARTRACE(chunk);
    VARTRACE(numberOfVideo);
    tNumberOfFiles numberOfEntry=0;
    tNumberOfFiles numberOfEntryAfterRemove=0;
    tFingerprintStatus fingerprintStatus=FPS_OK_SKIP_INDEXING;
    tResult ret = MP_NO_ERROR;
    vector <string> *trackInfoVectorPtr=(vector<string>*)trackInfosPtr;
    if(NULL != trackInfoVectorPtr){

        if(index == 0){
            ret = LocalSPM::GetDBManager().CreateTrackTempTable();
            if(ret == MP_NO_ERROR){
                ret = LocalSPM::GetDBManager().FillTrackTempTableWithVideoUID(deviceID);
                if(ret != MP_NO_ERROR){
                    ETG_TRACE_ERR(("Error While Fill the Track Temp Table with Video UID "));
                }
            }else{
                ETG_TRACE_ERR(("Error While Creating Track Temp Table"));
            }
        }
        if(ret == MP_NO_ERROR){
            ret = LocalSPM::GetDBManager().GetNumberofTrackTempEntry(numberOfEntry);
            VARTRACE(numberOfEntry);
        }
        if(ret == MP_NO_ERROR){
            for(tUInt i=0;i<trackInfoVectorPtr->size();i++){
                tUUID uUID;
                strncpy_r(uUID,trackInfoVectorPtr->at(i).c_str(),sizeof(uUID));
                ret=LocalSPM::GetDBManager().RemoveTrackEntryFromTemp(uUID);
                if(ret == MP_NO_ERROR){
                    numberOfEntry--;
                    ret = LocalSPM::GetDBManager().GetNumberofTrackTempEntry(numberOfEntryAfterRemove);
                    if(ret == MP_NO_ERROR){
                        if(numberOfEntryAfterRemove != numberOfEntry){
                            fingerprintStatus=FPS_OK;
                            break;
                        }
                    }else{
                        ETG_TRACE_ERR(("Error While GetNumberofTrackTempEntry"));
                        break;
                    }
                }else{
                    ETG_TRACE_ERR(("Error While RemoveTrackEntryFromTemp"));
                    break;
                }
            }
        }
        delete trackInfoVectorPtr;
    }

    if( FPS_OK_SKIP_INDEXING != fingerprintStatus || ( numberOfVideo == (index+chunk)))
    {
        LocalSPM::GetDBManager().DropTrackTempTable();
    }
    tDeviceType deviceType=DTY_UNKNOWN;
    if(m_IndexingContext[m_Index].deviceID == deviceID){
        deviceType = m_IndexingContext[m_Index].deviceType;
    }else{
        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error While GetDeviceInfo"));
        }else{
            deviceType = deviceInfo.deviceType;
        }
    }
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    strncpy_r(OUT msgToSendString, IN "VIDEO_UID_COMP_ANSWER", IN sizeof(msgToSendString));
    ret = LocalSPM::GetUSBControl().ParameterVIDEO_UID_COMP_ANSWER(OUT parameterString, IN size,IN deviceID,IN fingerprintStatus,IN index,IN chunk,IN indexSession);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceType,
            IN msgToSendString,
            IN parameterString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }
    VARTRACE(ret);
    return MP_NO_ERROR;
}

tResult Indexer::TrackUIDCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfTrack,tVoidPtr trackInfosPtr,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(index);
    tNumberOfFiles numberOfEntry=0;
    tNumberOfFiles numberOfEntryAfterRemove=0;
    tFingerprintStatus fingerprintStatus=FPS_OK_SKIP_INDEXING;
    tResult ret = MP_NO_ERROR;
    vector <string> *trackInfoVectorPtr=(vector<string>*)trackInfosPtr;
    if(NULL != trackInfoVectorPtr){

        if(index == 0){
            ret = LocalSPM::GetDBManager().CreateTrackTempTable();
            if(ret == MP_NO_ERROR){
                ret = LocalSPM::GetDBManager().FillTrackTempTable(deviceID);
                if(ret != MP_NO_ERROR){
                    ETG_TRACE_ERR(("Error While Fill the Track Temp Table "));
                }
            }else{
                ETG_TRACE_ERR(("Error While Creating Track Temp Table"));
            }
        }
        if(ret == MP_NO_ERROR){
            ret = LocalSPM::GetDBManager().GetNumberofTrackTempEntry(numberOfEntry);
            VARTRACE(numberOfEntry);
        }
        if(ret == MP_NO_ERROR){
            for(tUInt i=0;i<trackInfoVectorPtr->size();i++){
                tUUID uUID;
                strncpy_r(uUID,trackInfoVectorPtr->at(i).c_str(),sizeof(uUID));
                ret=LocalSPM::GetDBManager().RemoveTrackEntryFromTemp(uUID);
                if(ret == MP_NO_ERROR){
                    numberOfEntry--;
                    ret = LocalSPM::GetDBManager().GetNumberofTrackTempEntry(numberOfEntryAfterRemove);
                    if(ret == MP_NO_ERROR){
                        if(numberOfEntryAfterRemove != numberOfEntry){
                            fingerprintStatus=FPS_OK;
                            break;
                        }
                    }else {
                        ETG_TRACE_ERR(("Error While GetNumberofTrackTempEntry"));
                        break;
                    }
                }else{
                    ETG_TRACE_ERR(("Error While RemoveTrackEntryFromTemp"));
                    break;
                }
            }
        }
        delete trackInfoVectorPtr;
    }

    if( FPS_OK_SKIP_INDEXING != fingerprintStatus || ( numberOfTrack == (index+chunk)))
    {
        LocalSPM::GetDBManager().DropTrackTempTable();
    }

    tDeviceType deviceType=DTY_UNKNOWN;
    if(m_IndexingContext[m_Index].deviceID == deviceID){
        deviceType = m_IndexingContext[m_Index].deviceType;
    }else{
        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
        if (MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error While GetDeviceInfo"));
        }else{
            deviceType = deviceInfo.deviceType;
        }
    }
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    strncpy_r(OUT msgToSendString, IN "TRACK_UID_COMP_ANSWER", IN sizeof(msgToSendString));
    ret = LocalSPM::GetUSBControl().ParameterTRACK_UID_COMP_ANSWER(OUT parameterString, IN size,IN deviceID,IN fingerprintStatus,IN index , IN chunk,IN indexSession);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceType,
            IN msgToSendString,
            IN parameterString);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

void Indexer::StartFingerprintTimer(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    /* Set the timeout value */
    long timeoutValue = LocalSPM::GetDataProvider().IndexerFingerprintTimeoutMS();

    //store last fingerprint parameters
    m_LastFingerprintDeviceID = deviceID;

    /* Start the timer, single shot */
    m_FingerprintTimer.StartTimer(OUT m_FingerprintTimerID, IN timeoutValue, 0L, &LocalSPM::GetIndexer(), &FingerprintTimerCallBack, NULL);
}

void Indexer::StopFingerprintTimer()
{
    ENTRY;
    /* Cancel the timer */
    if (m_FingerprintTimerID) {
        m_FingerprintTimer.CancelTimer(IN m_FingerprintTimerID);
        m_FingerprintTimerID = 0;
    }
}

bool Indexer::FingerprintTimerCallBack(timer_t timerID , void* instance ,const void *userData)
{
    ENTRY;
    (void)timerID;
    (void)userData;
    Indexer *self = (Indexer*)instance;
    if(self) {
        //Timeout, no fingerprint available at this time
        self->OnFingerprintTimeout();
    }
    return MP_NO_ERROR;
}

void Indexer::OnFingerprintTimeout()
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    VARTRACE(m_LastFingerprintDeviceID);

    tFingerprint fingerprint = {0};
    tFingerprintStatus fingerprintStatus = FPS_NOT_AVAIL;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterFINGERPRINT_AVAIL(OUT parameterString,
            IN size,
            IN fingerprint,
            IN fingerprintStatus,
            IN NUMBER_OF_FILES_NONE,
            IN m_LastFingerprintDeviceID);
    ret = SendEvent(FINGERPRINT_AVAIL, IN parameterString);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF"));
    }
}

tResult Indexer::DBDeviceVersionUpdated(const tDeviceID deviceID)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    char msgToSendString[64];
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tDeviceInfoString deviceInfoString;
    tDeviceInfo deviceInfo;

    strncpy_r(OUT msgToSendString, IN "DO_MEDIA_PLAYER_APPLICATION_SETTINGS", IN sizeof(msgToSendString));

    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);

    DataProvider::MarshalDeviceInfo(deviceInfoString, deviceInfo);

    ret = LocalSPM::GetBTControl().ParameterDO_MEDIA_PLAYER_APPLICATION_SETTINGS(OUT parameterString,
            IN size,
            IN deviceInfoString,
            IN (tDeviceVersionUpdated)true);

    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    else
    {
        ret = LocalSPM::GetDeviceDispatcher().RouteMessage(IN deviceInfo.deviceType,
                IN msgToSendString,
                IN parameterString);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while calling RouteMessage of DeviceDispatcher (ErrorCode:%s)", errorString(ret)));
        }
    }

    return MP_NO_ERROR;
}

tResult Indexer::DoWaitForUpdateMyMediaFinished(char *parameterStr)
{
    ENTRY;
    VARTRACE(parameterStr);

    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    tIndexSession syncID = tIndexSession_init;

    //unpack deviceID
    SMF::UnMarshal(parameterStr, tDeviceID_format tIndexSession_format, &deviceID, &syncID);
    if (parameterStr) {
        delete[] parameterStr; //new by CheckResult
    }

    VARTRACE(deviceID);
    VARTRACE(syncID);

    /* Wait for the end of update and then go on */
    tResult syncResult = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished(deviceID);
    VARTRACE(syncResult);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterSYNCED(OUT parameterString, IN size, IN deviceID, IN syncID, IN syncResult);

    /* Send SYNC result message to own SM */
    tResult ret = SendEvent(SYNCED, IN parameterString);
    if(MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }
    return MP_NO_ERROR;
}

tResult Indexer::SyncResult(const tDeviceID deviceID, const tIndexSession syncID, const tResult syncResult)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(syncID);
    VARTRACE(syncResult);

    if(m_IndexingContext[m_Index].deviceID != deviceID /*|| syncID != mSyncID*/) { // NCG3D-55131: Remove syncID check.
        //filter out obsolete SYNC results
        ETG_TRACE_USR1(("Invalid deviceID or syncID - SyncResult ignored"));
    } else {
        tResult ret = MP_NO_ERROR;

        if(syncResult != MP_NO_ERROR) {
            ETG_TRACE_ERR(("Syncing MyMedia DB Update tasks failed (ErrorCode:%s)", errorString(syncResult)));
            ret = SendEvent(NOT_SUPPORTED, (char *)NULL);
        } else {
            //SYNC OK, enter indexing loop...
            ret = SendEvent(INDEX, (char *)NULL);
        }

        if(MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return MP_NO_ERROR;
}

tResult Indexer::SendSync()
{
    ENTRY;
    tIndex threadID;
    //(internal - NOT thread safe)

    //Sync with MyMedia DB Update tasks (running in MyMediaUpdateManager)

    /* Send SYNC_MYMEDIA_UPDATES message to own SM */
    tResult ret = SendEvent(SYNC, (char *)NULL);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    const size_t size = sizeof(tAllParameters);
    char *parameterString = new char[size]; //deleted by DoWaitForUpdateMyMediaFinished
    SMF::Marshal(parameterString, size - 1, tDeviceID_format tIndexSession_format, m_IndexingContext[m_Index].deviceID, mSyncID);

    if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
        threadID = LocalSPM::GetThreadFactoryLowprio().Do(IN this, IN FUNCTION_ID_WAIT_FOR_UPDATE_MYMEDIA_FINISHED, IN parameterString);
    }
    else
    {
        threadID = LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_WAIT_FOR_UPDATE_MYMEDIA_FINISHED, IN parameterString);
    }
    VARTRACE(threadID);

    return MP_NO_ERROR;
}

tResult Indexer::ResetAlbumArtIndexingComplete()
{
    ENTRY;
    //(internal - NOT thread safe)

    /* Reset album art indexing complete flag to false */
    tAlbumArtIndexingComplete completeFlag = FALSE;
    tResult ret = LocalSPM::GetDBManager().SetAlbumArtIndexingComplete(IN m_IndexingContext[m_Index].deviceID, IN completeFlag);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while setting album art indexing complete flag at DBManager (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

void Indexer::SetDeviceFileCounts(const tDeviceID deviceID,
        const tNumberOfFiles numberOfFiles/*=NUMBER_OF_FILES_NONE*/,
        const tNumberOfFiles numberOfVideoFiles /*=NUMBER_OF_FILES_NONE*/,
        const tNumberOfFiles numberOfImageFiles/*=NUMBER_OF_FILES_NONE*/)
{
    ENTRY;

    tResult result = MP_NO_ERROR;

    if(deviceID != DEVICE_ID_NOT_SET)
    {
        if(NUMBER_OF_FILES_NONE != numberOfFiles)
        {
            result = LocalSPM::GetDBManager().SetNumberOfAudioFiles(IN deviceID, numberOfFiles);
            if (MP_NO_ERROR != result)
            {
                ETG_TRACE_ERR(("Error while updating number of files at DBManager (ErrorCode:%s)", errorString(result)));
            }
        }

        if(NUMBER_OF_FILES_NONE != numberOfVideoFiles)
        {
            result = LocalSPM::GetDBManager().SetNumberOfVideoFiles(IN deviceID, IN numberOfVideoFiles);
            if(MP_NO_ERROR != result)
            {
                ETG_TRACE_ERR(("Error while updating number of video files at DBManager (ErrorCode:%s)", errorString(result)));
            }
        }

        if(NUMBER_OF_FILES_NONE != numberOfImageFiles)
        {
            result = LocalSPM::GetDBManager().SetNumberOfImageFiles(IN deviceID, IN numberOfImageFiles);
            if(MP_NO_ERROR != result)
            {
                ETG_TRACE_ERR(("Error while updating number of image files at DBManager (ErrorCode:%s)", errorString(result)));
            }
        }
    }
    else
    {
        ETG_TRACE_ERR(("Invalid DeviceID.DeviceFileCounts not Updated!"));
    }
}

