#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INDEXER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/AlbumArtIndexer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_INDEXER
#endif
#endif

#include "LocalSPM.h"
#include "VarTrace.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include <dirent.h>
#include "AlbumArtIndexer.h"
#include <sys/syscall.h>

/*lint -save -e1401 */

AlbumArtIndexer::AlbumArtIndexer(const tComponentID componentID):ILocalSPM(componentID)
{
    ENTRY_INTERNAL

    /* Early initialization of variables */
    m_AutoRegisterOnDBTrigger = TS_ON;
}

AlbumArtIndexer::~AlbumArtIndexer()
{
    ENTRY_INTERNAL
}

void AlbumArtIndexer::Create()
{
    ENTRY

    /* Create the state machine */
    AlbumArtIndexerSM::Create();

    CreateDone(0);
}

tResult AlbumArtIndexer::Init(tInitReason reason)
{
    ENTRY
    (void)reason;

    /* Init the state machine */
    AlbumArtIndexerSM::Init();
    SetAnswerTimeout(ALBUMART_INDEXER_SM_ANSWER_TIMEOUT_MS); // Set answer timer

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);

    return InitDone(0);
}

tResult AlbumArtIndexer::InitSM()
{
    ENTRY

    /* Initialize variables */
    m_TriggerID_DevIDSComplete = 0;
    m_TriggerID_DevRemoved = 0;

    m_CoverArtWidth = 0;
    m_CoverArtHeight = 0;
    m_CoverArtMimeType = MMT_PNG;
    m_CoverArtDirPath[0] = '\0';
    m_LimitCoverArtMemoryOnFlashKB = 0;
    m_CoverArtMemoryOnFlashKB = 0;
    m_ThumbnailWidth = 0;
    m_ThumbnailHeight = 0;
    m_ThumbnailMimeType = MMT_PNG;

    m_DevicesToIndex.clear();
    InitAlbumArtIndexingContext(INOUT m_IndexingContext);
    InitAlbumArtIndexingContext(INOUT m_IndexingVideoContext);
    InitMediaObject(OUT m_MediaObject);
    m_CoverArtObjectPtr = NULL;
    m_ThumbnailObjectPtr = NULL;

    m_VideoThumbnailWidth =0;
    m_VideoThumbnailHeight =0;
    m_VideoThumbnailtMimeType = MMT_JPG;
    m_VideoThumbnailDirPath[0] = '\0';
    m_VideoThumbnailObjectPtr = NULL;
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::Run()
{
    ENTRY
    if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
        LocalSPM::GetThreadFactoryLowprio().Do(this, 0, NULL); //AlbumArtIndexer inclusive state machine
    }
    else
    {
        LocalSPM::GetThreadFactory().Do(this, 0, NULL); //AlbumArtIndexer inclusive state machine
    }

    if(TS_ON == m_AutoRegisterOnDBTrigger)
    {
        /* Register on DB trigger */
        SwitchDBTrigger(TT_ALL, TS_ON);
    }

    return RunDone(0);
}

void AlbumArtIndexer::Do(int functionID, void *ptr)
{
    ENTRY;
    (void)functionID;
    (void)ptr;

    if(LocalSPM::GetDataProvider().ControlCPUThtreadLoad()){
        SetThreadNiceProperty(syscall(SYS_gettid), LocalSPM::GetDataProvider().IndexerThreadNiceValue());
        //set the threads name
        LocalSPM::GetThreadFactoryLowprio().SetName(IN AlbumArtIndexerSM::GetSMNameFull());
    }
    else
    {
        //set the threads name
        LocalSPM::GetThreadFactory().SetName(IN AlbumArtIndexerSM::GetSMNameFull());
    }
    //actual thread
    while(AlbumArtIndexerSM::STATE_MACHINE_FINISHED != AlbumArtIndexerSM::StateMachine_Main()){}
}

tResult AlbumArtIndexer::Stop()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Deregister on DB trigger */
    ret = SwitchDBTrigger(TT_ALL, TS_OFF);

    /* Send STOP message to own SM */
    ret = SendForceEvent(STOP_SM, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult AlbumArtIndexer::StopEventProcessed()
{
    ENTRY

    /* Send stop done to SPM in the transition to final state in state machine */
    return StopDone(0);
}

tResult AlbumArtIndexer::Done()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    /* Deregister state machine with dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* Send DONE message to own SM */
    ret = SendForceEvent(DONE, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult AlbumArtIndexer::DoneEventProcessed()
{
    ENTRY

    /* Send done done to SPM in the transition to final state in state machine */
    return DoneDone(0);
}

char *AlbumArtIndexer::GetSMStateName(OUT tGeneralString stateName, size_t size)
{
    GetCurrentState((char *)stateName, size);
    return stateName;
}

tResult AlbumArtIndexer::DBDeviceIDSComplete(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY
    ETG_TRACE_USR3(("AlbumArtIndexer::DBDeviceIDSComplete deviceCount:%u, deviceID: %u", deviceCount, deviceID));

    if(deviceID != m_IndexingContext.deviceID)
    {
        //Add the current Indexing device to queue only if it is not an interim device.
        //Reason:Only the devices whose metadataIndexing is completed should be added to Queue.If any interim device added to queue,their metadataIndexing may not have completed.
        //Hence,all albums are not identified.This may lead to AlbumArtIndexing done only for few albums but marked as complete in DB - (NCG3D-134745)
        if((DEVICE_ID_NOT_SET != m_IndexingContext.deviceID) && (false == m_IndexingContext.interimList))
        {
            m_DevicesToIndex.insert(m_DevicesToIndex.begin(), m_IndexingContext.deviceID);
        }
    }

    /* Add deviceID to index queue */
    m_DevicesToIndex.push_back(deviceID);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tReturnValue AlbumArtIndexer::IsDBSameDevice(const tDeviceCount deviceCount, const tDeviceID deviceID)
{
    ENTRY
    ETG_TRACE_USR3(("AlbumArtIndexer::IsDBSameDevice deviceCount:%u, deviceID: %u", deviceCount, deviceID));

    tReturnValue returnValue = false;

    if(deviceID == m_IndexingContext.deviceID)
    {
        returnValue = true;
    }
    else
    {
        /* Remove deviceID from index queue */
        for(int i = m_DevicesToIndex.size()-1; i>=0; i--)
        {
            if(deviceID == m_DevicesToIndex[i])
            {
                m_DevicesToIndex.erase(m_DevicesToIndex.begin() + i);
            }
        }
    }

    return returnValue;
}

tResult AlbumArtIndexer::DeviceToIndex()
{
    ENTRY

    tResult ret = MP_NO_ERROR;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;

    if(0 < m_DevicesToIndex.size())
    {
        /* The order of the devices to index is first in first out
         * Only the current active device is preferred
         */
        deviceID = m_DevicesToIndex.front();

        /* Get current active device ID */
        tDeviceID currentDeviceID = DEVICE_ID_NOT_SET;
        ret = LocalSPM::GetDBManager().GetActiveDevice(OUT currentDeviceID);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while getting active device from DBManager (ErrorCode:%s)", errorString(ret)));
        }
        else
        {
            for(int i = m_DevicesToIndex.size()-1; i>=0; i--)
            {
                if(currentDeviceID == m_DevicesToIndex[i])
                {
                    deviceID = m_DevicesToIndex[i];
                    break;
                }
            }
        }
    }
    ETG_TRACE_USR3(("Next device to index: %u",deviceID));

    if(DEVICE_ID_NOT_SET != deviceID)
    {
        InitAlbumArtIndexingContext(INOUT m_IndexingContext);
        m_IndexingContext.deviceID = deviceID;

        /* Send START_ALBUMART_INDEX message to own SM */
        ret = SendEvent(START_ALBUMART_INDEX, (char *)NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tReturnValue AlbumArtIndexer::IsAlbumArtIndexingSupported()
{
    ENTRY

    tReturnValue returnValue = false;
    tResult ret = MP_NO_ERROR;

    /* Remove deviceID from index queue */
    for(int i = m_DevicesToIndex.size()-1; i>=0; i--)
    {
        if(m_IndexingContext.deviceID == m_DevicesToIndex[i])
        {
            m_DevicesToIndex.erase(m_DevicesToIndex.begin() + i);
        }
    }

    /* Get device info from DBManager */
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN m_IndexingContext.deviceID);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting device info from DataProvider (ErrorCode:%s) -> No album art indexing possible", errorString(ret)));
    }
    else
    {
        /* Is feature CoverArtFlow or AlbumArtThumbnail supported, device type supported and at least one file on device? */
        if((LocalSPM::GetDataProvider().CoverArtFlow() || LocalSPM::GetDataProvider().AlbumArtThumbnail())
                &&
                (LocalSPM::GetDataProvider().IsAlbumArtIndexingSupported(IN deviceInfo.deviceType))
                &&
                (0 < (int)deviceInfo.numberOfAudioFiles))
        {
            /* Is albumart indexing not already complete? */
            tAlbumArtIndexingComplete completeFlag;
            ret = LocalSPM::GetDBManager().GetAlbumArtIndexingComplete(OUT completeFlag, IN m_IndexingContext.deviceID);
            if((MP_NO_ERROR != ret)
                    ||
                    (!completeFlag))
            {
                returnValue = true;
            }
        }
    }

    return returnValue;
}

tResult AlbumArtIndexer::SetFocus()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Create a new indexed list of all albums without album art entry */
    tListSize listSize;
    tListSize video_listSize;
    tStreaming streaming = false;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;

    tag1.tag = 0;
    tag2.tag = 0;
    tag3.tag = 0;
    tag4.tag = 0;

    if(LocalSPM::GetDataProvider().VideoThumbnail())
    {
        ret = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
                OUT m_IndexingVideoContext.listID,
                OUT video_listSize,
                LTY_VIDEO,
                IN streaming,
                IN tag1,
                IN tag2,
                IN tag3,
                IN tag4,
                IN m_IndexingContext.deviceID);
    }

    ret = LocalSPM::GetListControl().CreateMediaPlayerIndexedList(
            OUT m_IndexingContext.listID,
            OUT listSize,
            LTY_ALBUM_UNKNOWN_ALBUMART,
            IN streaming,
            IN tag1,
            IN tag2,
            IN tag3,
            IN tag4,
            IN m_IndexingContext.deviceID);

    if(MP_NO_ERROR != ret && MP_ERR_DB_END_OF_LIST != ret)
    {
        ETG_TRACE_ERR(("Error while creating all album without album art list (ErrorCode:%s)", errorString(ret)));
    }

    m_IndexingContext.startIndex = 0;
    if(LocalSPM::GetDataProvider().VideoThumbnail())
    {
        m_IndexingContext.sliceSize = (tIndex)listSize+video_listSize;
        m_IndexingVideoContext.sliceSize = video_listSize;
    }
    else{
        m_IndexingContext.sliceSize = (tIndex)listSize;
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::SendChangeFocus(const tDeviceID deviceID, const tListID listID, const tIndex startIndex, const tIndex sliceSize)
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Parameter check */
    if(MY_MEDIA >= deviceID)
    {
        ETG_TRACE_ERR(("Received deviceID is invalid"));
        return MP_ERR_PM_INVALID_PARAM;
    }
    if(LIST_ID_NONE == listID)
    {
        ETG_TRACE_ERR(("Received listID is invalid"));
        return MP_ERR_PM_INVALID_PARAM;
    }

    /* Send CHANGE_FOCUS message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterCHANGE_FOCUS(OUT parameterString, IN size, IN deviceID, IN listID, IN startIndex, IN sliceSize);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        ret = SendEventByName("CHANGE_FOCUS", IN parameterString);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

tResult AlbumArtIndexer::ChangeFocus(const tDeviceID deviceID, const tListID listID, const tIndex startIndex, const tIndex sliceSize)
{
    ENTRY

    //Add the current Indexing device to queue only if it is not an interim device.
    //Reason:Only the devices whose metadataIndexing is completed should be added to Queue.If any interim device added to queue,their metadataIndexing may not have completed.
    //Hence,all albums are not identified.This may lead to AlbumArtIndexing done only for few albums but marked as complete in DB -(NCG3D-134745)

    //Eventhough ChangeFocus come for current covertArtIndexing device,add it to the list of devices considered for next CoverArtIndexing run(RTC-Bug:340990)
    //Reason: Once ChangeFocus bound slice's covertArtIndexing FINISHED, in AlbumArtIndexer::DeviceToIndex() the Device will not be available to choose for next CoverArtIndexing run.
    //Hence,CoverArtIndexing for the device will not occur anymore. -BUG!
    //if(deviceID != m_IndexingContext.deviceID)
    //{
        /* Reinsert current deviceID at the beginning of the index queue */
        if((DEVICE_ID_NOT_SET != m_IndexingContext.deviceID) && (false == m_IndexingContext.interimList))
        {
            m_DevicesToIndex.insert(m_DevicesToIndex.begin(), m_IndexingContext.deviceID);
        }
    //}

    InitAlbumArtIndexingContext(INOUT m_IndexingContext);
    m_IndexingContext.deviceID = deviceID;
    m_IndexingContext.listID = listID;
    m_IndexingContext.startIndex = startIndex;
    m_IndexingContext.sliceSize = sliceSize;
    m_IndexingContext.interimList = true;

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::PrepareAlbumArtIndexing()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Read configuration */
    m_CoverArtWidth = (tSize)LocalSPM::GetDataProvider().CoverArtWidth();
    m_CoverArtHeight = (tSize)LocalSPM::GetDataProvider().CoverArtHeight();
    m_CoverArtMimeType = (tMimeType)LocalSPM::GetDataProvider().CoverArtMimeType();
    strncpy_r(OUT m_CoverArtDirPath, IN LocalSPM::GetDataProvider().CoverArtDirPath().c_str(), IN sizeof(m_CoverArtDirPath));
    m_LimitCoverArtMemoryOnFlashKB = (tMemorySize)LocalSPM::GetDataProvider().LimitCoverArtMemoryOnFlashKB();
    m_ThumbnailWidth = (tSize)LocalSPM::GetDataProvider().ThumbnailWidth();
    m_ThumbnailHeight = (tSize)LocalSPM::GetDataProvider().ThumbnailHeight();
    m_ThumbnailMimeType = (tMimeType)LocalSPM::GetDataProvider().ThumbnailMimeType();
    m_VideoThumbnailWidth = (tSize)LocalSPM::GetDataProvider().VideoThumbnailWidth();
    m_VideoThumbnailHeight = (tSize)LocalSPM::GetDataProvider().VideoThumbnailHeight();
    m_VideoThumbnailtMimeType = (tMimeType)LocalSPM::GetDataProvider().VideoThumbnailMimeType();
    strncpy_r(OUT m_VideoThumbnailDirPath, IN LocalSPM::GetDataProvider().VideoThumbnailDirPath().c_str(), IN sizeof(m_VideoThumbnailDirPath));

    /* Check if cover art directory is existing */
    if( exists_directory(m_CoverArtDirPath) )
    {
#if 0 //PSARCC21-610
        /* Check if device path is existing */
        tPath devicePath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03"
        snprintf(devicePath, sizeof(devicePath)-1, "%sdev%02d", m_CoverArtDirPath, m_IndexingContext.deviceID);

        if((exists_directory(devicePath))
                &&
                (23 < strlen_r(devicePath))) //do not delete directory recursively with a path length below "/var/opt/bosch/dynamic/"
        {
            /* Remove directory */
            if(0 != remove_directory(devicePath))
            {
                ETG_TRACE_ERR(("remove_directory: error: %d/%s", errno, strerror(errno)));
            }
            else
            {
                ETG_TRACE_COMP(("Delete directory of cover arts in flash: %s", devicePath));
            }
        }
#endif

        /* Get memory size of cover art directory on flash */
        GetCoverArtMemoryOnFlash();
        VARTRACE(m_CoverArtMemoryOnFlashKB);
    }
    else
    {
        /* Create directory with read, write, execute permission for owner and read, execute permission for group = 0750 */
        if(0 != mkdir(m_CoverArtDirPath, S_IRWXU | S_IRGRP | S_IXGRP))
        {
            ETG_TRACE_ERR(("mkdir: error: %d/%s", errno, strerror(errno)));
        }
        else
        {
            ETG_TRACE_COMP(("Make directory for cover arts in flash: %s", m_CoverArtDirPath));
        }

        m_CoverArtMemoryOnFlashKB = 0;
    }
    if(!exists_directory(m_VideoThumbnailDirPath))
    {
        if(0 != mkdir(m_VideoThumbnailDirPath, S_IRWXU | S_IRGRP | S_IXGRP))
        {
            ETG_TRACE_ERR(("mkdir: error: %d/%s", errno, strerror(errno)));
        }
        else
        {
            ETG_TRACE_COMP(("Make directory for cover arts in flash: %s", m_VideoThumbnailDirPath));
        }
        m_CoverArtMemoryOnFlashKB = 0;
    }

    /* Check if album list is valid */
    if((MY_MEDIA < m_IndexingContext.deviceID)
            &&
            (LIST_ID_NONE != m_IndexingContext.listID)
            &&
            (0 < m_IndexingContext.sliceSize)
            &&
            (m_IndexingContext.startIndex < m_IndexingContext.sliceSize))
    {
        /* Get all objects from list and store it into a vector */
        //m_IndexingContext.mediaObjectVector.clear();
        if( 0 < m_IndexingContext.mediaObjectVector.capacity() )
        {
            vector<tMediaObject>().swap(m_IndexingContext.mediaObjectVector); //used instead of clear to guarantee a memory reallocation
        }

        ret = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(
                OUT m_IndexingContext.mediaObjectVector,
                IN m_IndexingContext.listID,
                IN m_IndexingContext.startIndex,
                IN m_IndexingContext.sliceSize);
        if(LocalSPM::GetDataProvider().VideoThumbnail())
        {
            ret = LocalSPM::GetListControl().RequestMediaPlayerIndexedListSlice(
                    OUT m_IndexingVideoContext.mediaObjectVector,
                    IN m_IndexingVideoContext.listID,
                    IN m_IndexingContext.startIndex,
                    IN m_IndexingVideoContext.sliceSize);
            for(vector<tMediaObject>::iterator it = m_IndexingVideoContext.mediaObjectVector.begin();it!=m_IndexingVideoContext.mediaObjectVector.end();it++)
            {
                m_IndexingContext.mediaObjectVector.push_back(*it);
            }
        }
        if((MP_NO_ERROR != ret && MP_ERR_DB_END_OF_LIST != ret)
                ||
                (m_IndexingContext.sliceSize != m_IndexingContext.mediaObjectVector.size()))
        {
            ETG_TRACE_FATAL(("Error while getting all objects from list (ErrorCode:%s)", errorString(ret)));
        }

        m_IndexingContext.objectIndex = 0;
    }

    return ret;
}

tResult AlbumArtIndexer::CheckAlbumArtVector()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Check if media object vector is valid and object index is in range */
    if((1 > m_IndexingContext.mediaObjectVector.size())
            ||
            (m_IndexingContext.objectIndex >= m_IndexingContext.mediaObjectVector.size()))
    {
        /* Send event message to own SM */
        ret = SendEvent(IN FINISHED, IN (char *)NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        SendGoOn();
    }

    return ret;
}

tResult AlbumArtIndexer::GetAlbumArt()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Get current media object from vector */
    m_MediaObject = m_IndexingContext.mediaObjectVector[m_IndexingContext.objectIndex];

    /* Increment media object vector index */
    m_IndexingContext.objectIndex++;

    if(m_IndexingContext.interimList) //Only for interim lists due to focus change
    {
        /* Check if already an entry is available in the DB AlbumArts table */
        tAlbumArt coverArtPath;
        if (m_MediaObject.fileType == FT_VIDEO)
        {
            ret = LocalSPM::GetDBManager().GetCoverArtPath(OUT coverArtPath, IN m_MediaObject.deviceID, IN m_MediaObject.MetadataTag1);
        }
        else
        {
            ret = LocalSPM::GetDBManager().GetCoverArtPath(OUT coverArtPath, IN m_MediaObject.deviceID, IN m_MediaObject.MetadataTag4);
        }
        if(MP_NO_ERROR == ret)
        {
            /* Send event message to own SM */
            ret = SendEvent(IN SKIP, IN (char *)NULL);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }

            return ret;
        }
    }

    tEvent *event;
    ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt m_MediaObject.albumArtString=%s ",m_MediaObject.albumArtString));
    if(0 < strlen_r(m_MediaObject.albumArtString))
    {
        /* Get album art for current object */
        tReturnValue returnValue = FALSE;
        tAlbumArtObjectPtr albumArtObjectPtr = NULL;
        if(m_MediaObject.fileType == FT_VIDEO && LocalSPM::GetDataProvider().VideoThumbnail())
        {
            ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt Video"));
            ret = LocalSPM::GetUSBControl().ReadVideoThumbnailFromFile(OUT albumArtObjectPtr, IN m_MediaObject.albumArtString, IN m_VideoThumbnailWidth, IN m_VideoThumbnailHeight, IN m_VideoThumbnailtMimeType);
            ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt Video returnValue=%d",ret));
            if(!ret && albumArtObjectPtr && albumArtObjectPtr->imageSize)
            {
                if (!m_VideoThumbnailObjectPtr) m_VideoThumbnailObjectPtr = new tAlbumArtObject;
                if(m_VideoThumbnailObjectPtr)
                {
                    m_VideoThumbnailObjectPtr->imageSize = albumArtObjectPtr->imageSize;
                    m_VideoThumbnailObjectPtr->imageData = (tImageData)malloc(m_VideoThumbnailObjectPtr->imageSize);
                    if(m_VideoThumbnailObjectPtr->imageData) memcpy ( m_VideoThumbnailObjectPtr->imageData, albumArtObjectPtr->imageData, m_VideoThumbnailObjectPtr->imageSize ) ;
                    m_VideoThumbnailObjectPtr->sizeX = albumArtObjectPtr->sizeX;
                    m_VideoThumbnailObjectPtr->sizeY = albumArtObjectPtr->sizeY;
                    m_VideoThumbnailObjectPtr->mimeType = albumArtObjectPtr->mimeType; //yts5cob make configurable
                    strncpy_r(OUT m_VideoThumbnailObjectPtr->albumArtString, IN albumArtObjectPtr->albumArtString, IN sizeof(m_VideoThumbnailObjectPtr->albumArtString));
                    ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt m_VideoThumbnailObjectPtr->imageSize =%d albumArtObjectPtr->imageSize=%d",m_VideoThumbnailObjectPtr->imageSize,albumArtObjectPtr->imageSize));
                }
                ETG_TRACE_USR3(("AlbumArtIndexer::VideoThumbnail WRITE_DB_FLASH event"));
                returnValue = TRUE;
            }
            else
            {
                returnValue = FALSE;
            }
        }
        else
        {
            returnValue = LocalSPM::GetUSBControl().ReadAlbumArtFromFile(OUT albumArtObjectPtr, IN m_MediaObject.albumArtString);
            if(returnValue)
            {
                /*Convert the album art to cover art size and format*/
                ret = LocalSPM::GetDataProvider().ConvertAlbumArt(IN albumArtObjectPtr,
                        OUT m_CoverArtObjectPtr,
                        IN m_CoverArtWidth,
                        IN m_CoverArtHeight,
                        IN m_CoverArtMimeType);
                if((MP_NO_ERROR != ret) || (!m_CoverArtObjectPtr))
                {
                    ETG_TRACE_ERR(("Error while converting album art to cover art (ErrorCode:%s)", errorString(ret)));
                    returnValue = FALSE;

                    if(m_CoverArtObjectPtr)
                    {
                        if(m_CoverArtObjectPtr->imageData) free(m_CoverArtObjectPtr->imageData);
                        delete m_CoverArtObjectPtr;
                        m_CoverArtObjectPtr = NULL;
                    }
                }

                /*Convert the album art to thumbnail size and format*/
                ret = LocalSPM::GetDataProvider().ConvertAlbumArt(IN albumArtObjectPtr,
                        OUT m_ThumbnailObjectPtr,
                        IN m_ThumbnailWidth,
                        IN m_ThumbnailHeight,
                        IN m_ThumbnailMimeType);
                if((MP_NO_ERROR != ret) || (!m_ThumbnailObjectPtr))
                {
                    ETG_TRACE_ERR(("Error while converting album art to thumbnail (ErrorCode:%s)", errorString(ret)));
                    //returnValue = FALSE; Even store cover art if thumbnail is not available

                    if(m_ThumbnailObjectPtr)
                    {
                        if(m_ThumbnailObjectPtr->imageData) free(m_ThumbnailObjectPtr->imageData);
                        delete m_ThumbnailObjectPtr;
                        m_ThumbnailObjectPtr = NULL;
                    }
                }
            }
        }
        if(albumArtObjectPtr)
        {
            if(albumArtObjectPtr->imageData) free(albumArtObjectPtr->imageData);
            delete albumArtObjectPtr;
            albumArtObjectPtr = NULL;
        }

        if(returnValue)
        {
            ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt WRITE_DB_FLASH event"));
            event = WRITE_DB_FLASH;
        }
        else
        {
            /*check if read request on device related to this mount point is possible*/
            if(!LocalSPM::GetDataProvider().checkIfDeviceConnected(m_MediaObject.mountPoint))
            {
                //call additional DBDeviceRemoved manually to cancel indexing loop
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterDB_DEVICE_REMOVED(OUT parameterString, IN size, IN 1, IN m_IndexingContext.deviceID);
                ret = SendEvent(DB_DEVICE_REMOVED, IN parameterString);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("SendEvent(DB_DEVICE_REMOVED) failed (ErrorCode:%s)", errorString(ret)));
                }

                event = SKIP;
            }
            else
            {
                ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt WRITE_DB event1"));
                event = WRITE_DB;
            }
        }
    }
    else //No albumArtString
    {
        ETG_TRACE_USR3(("AlbumArtIndexer::GetAlbumArt WRITE_DB event2"));
        event = WRITE_DB;
    }

    /* Send event message to own SM */
    ret = SendEvent(IN event, IN (char *)NULL);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tReturnValue AlbumArtIndexer::IsCoverArtLimitNotReached()
{
    ENTRY

    tReturnValue returnValue = true;

    if((LocalSPM::GetDataProvider().VideoThumbnail() && m_MediaObject.fileType == FT_VIDEO) //feature enabled
            &&
            (m_LimitCoverArtMemoryOnFlashKB < (m_CoverArtMemoryOnFlashKB + ((tMemorySize)m_VideoThumbnailObjectPtr->imageSize / 1024))))
    {
        ETG_TRACE_USR1(("AlbumArtIndexer::IsCoverArtLimitNotReached if 1"));
        returnValue = false;
    }
    else if((LocalSPM::GetDataProvider().CoverArtFlow() && m_MediaObject.fileType != FT_VIDEO) //feature enabled
            &&
            (m_LimitCoverArtMemoryOnFlashKB < (m_CoverArtMemoryOnFlashKB + ((tMemorySize)m_CoverArtObjectPtr->imageSize / 1024))))
    {
        ETG_TRACE_USR1(("AlbumArtIndexer::IsCoverArtLimitNotReached if 2"));
        returnValue = false;
    }
    ETG_TRACE_USR1(("AlbumArtIndexer::IsCoverArtLimitNotReached returnValue=%d",returnValue));
    return returnValue;
}

tResult AlbumArtIndexer::WriteDB()
{
    ENTRY;
    VARTRACE(m_MediaObject.MetadataField4);

    tResult ret = MP_NO_ERROR;

    /* Store (empty) thumbnail and no cover art path to AlbumArts table of DB */
    tImageBlob thumbnailBlob;
    if(m_ThumbnailObjectPtr && LocalSPM::GetDataProvider().AlbumArtThumbnail())
    {
        thumbnailBlob.imageSize = m_ThumbnailObjectPtr->imageSize;
        thumbnailBlob.imageData = m_ThumbnailObjectPtr->imageData;
    }
    else
    {
        thumbnailBlob.imageSize = 0;
        thumbnailBlob.imageData = NULL;
    }

    ret = LocalSPM::GetDBManager().StoreAlbumArt(IN m_MediaObject.deviceID, IN m_MediaObject.MetadataTag4,IN m_MediaObject.title, IN "", IN thumbnailBlob,IN m_MediaObject.dateTime);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while storing album art object in DB (ErrorCode:%s)", errorString(ret)));
    }

    if(m_ThumbnailObjectPtr)
    {
        if(m_ThumbnailObjectPtr->imageData) free(m_ThumbnailObjectPtr->imageData);
        delete m_ThumbnailObjectPtr;
        m_ThumbnailObjectPtr = NULL;
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::WriteDBAndFlash()
{
    ENTRY;
    VARTRACE(m_MediaObject.MetadataField4);
    tResult ret = MP_NO_ERROR;
    // ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() m_MediaObject.MetadataField4=%s m_MediaObject.fileType=%d m_MediaObject.MetadataField1=%s",m_MediaObject.MetadataField4,m_MediaObject.fileType,m_MediaObject.MetadataField1));
    if(m_MediaObject.fileType == FT_VIDEO)
    {
        strncpy_r(OUT m_MediaObject.MetadataField4, IN m_MediaObject.MetadataField1, IN sizeof(m_MediaObject.MetadataField4));
        m_MediaObject.MetadataTag4 = m_MediaObject.objectID;

        /* Create Video Thumbnail path */
        tPath devicePath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03"
        snprintf(devicePath, sizeof(devicePath)-1, "%sdev%02d", m_VideoThumbnailDirPath, m_MediaObject.deviceID);
        tExtension extension;
        GetEndingByMimeType(OUT extension, IN m_VideoThumbnailtMimeType);
        tAlbumArt videothumbnailPath; //e.g. "/var/opt/bosch/dynamic/media/videothumbnail/dev03/ABBA_280_280.png"
        //snprintf(videothumbnailPath, sizeof(videothumbnailPath)-1, "%s/%s_%d_%d%s", devicePath, m_MediaObject.MetadataField4, m_VideoThumbnailWidth, m_VideoThumbnailHeight, extension);
        snprintf(videothumbnailPath, sizeof(videothumbnailPath)-1, "%s/%d_%d_%d%s", devicePath, m_MediaObject.MetadataTag1, m_VideoThumbnailWidth, m_VideoThumbnailHeight, extension);
        if(m_VideoThumbnailObjectPtr)
        {
            ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if m_VideoThumbnailObjectPtr"));
            if(LocalSPM::GetDataProvider().VideoThumbnail()) //feature enabled
            {
                ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if VideoThumbnail()"));
                /* Check if device path is not existing */
                if( !exists_directory(devicePath) )
                {
                    ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if !exists_directory(devicePath)"));
                    /* Create directory with read, write, execute permission for owner and read, execute permission for group = 0750 */
                    if(0 != mkdir(devicePath, S_IRWXU | S_IRGRP | S_IXGRP))
                    {
                        ETG_TRACE_ERR(("mkdir: error: %d/%s", errno, strerror(errno)));
                    }
                    else
                    {
                        ETG_TRACE_COMP(("Make directory for cover arts in flash: %s", devicePath));
                    }
                }

                /* Delete cover art file if it is already existing */
                struct stat statBuffer;
                if(0 == stat(videothumbnailPath, &statBuffer))
                {
                    /* file existing */
                    ETG_TRACE_COMP(("videothumbnail file already existing -> remove it. Size: %d", statBuffer.st_size));

                    if(0 != remove(videothumbnailPath))
                    {
                        ETG_TRACE_ERR(("remove: error: %d/%s", errno, strerror(errno)));
                    }
                    else
                    {
                        /* Decrease used memory on flash */
                        m_CoverArtMemoryOnFlashKB -= (tMemorySize)statBuffer.st_size / 1024;
                    }
                }

                /* Store cover art to flash */
                FILE *fpImage = fopen(videothumbnailPath, "wb");
                if (!fpImage)
                {
                    ETG_TRACE_ERR(("fopen: error: %d/%s", errno, strerror(errno)));
                }
                else
                {
                    ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if fpImage m_VideoThumbnailObjectPtr->imageSize=%d",m_VideoThumbnailObjectPtr->imageSize));
                    fwrite(m_VideoThumbnailObjectPtr->imageData, 1, m_VideoThumbnailObjectPtr->imageSize, fpImage);
                    fclose(fpImage);
                    ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if fpImage File Write Complete"));

                    /* Increase used memory on flash */
                    m_CoverArtMemoryOnFlashKB += (tMemorySize)m_VideoThumbnailObjectPtr->imageSize / 1024;
                }
            }
            else
            {
                videothumbnailPath[0] = '\0';
            }

            if(m_VideoThumbnailObjectPtr->imageData) free(m_VideoThumbnailObjectPtr->imageData);
            delete m_VideoThumbnailObjectPtr;
            m_VideoThumbnailObjectPtr = NULL;
        }

        tImageBlob thumbnailBlob;
        ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() StoreAlbumArt without thumbnailblob"));
        thumbnailBlob.imageSize = 0;
        thumbnailBlob.imageData = NULL;
        ret = LocalSPM::GetDBManager().StoreAlbumArt(IN m_MediaObject.deviceID, IN m_MediaObject.MetadataTag1,IN m_MediaObject.title, IN videothumbnailPath, IN thumbnailBlob,IN m_MediaObject.dateTime);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while storing album art object in DB (ErrorCode:%s)", errorString(ret)));
        }


    }
    else
    {
        tBoolean storeCovertArtInDB = true;//NCG3D-134745

        /* Create cover art path */
        tPath devicePath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03"
        snprintf(devicePath, sizeof(devicePath)-1, "%sdev%02d", m_CoverArtDirPath, m_MediaObject.deviceID);
        tExtension extension;
        GetEndingByMimeType(OUT extension, IN m_CoverArtMimeType);
        tAlbumArt coverArtPath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03/ABBA_280_280.png"
        snprintf(coverArtPath, sizeof(coverArtPath)-1, "%s/%d_%d_%d%s", devicePath, m_MediaObject.MetadataTag4, m_CoverArtWidth, m_CoverArtHeight, extension); //Hotspot fix
        if(m_CoverArtObjectPtr)
        {
            ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if m_CoverArtObjectPtr"));
            if(LocalSPM::GetDataProvider().CoverArtFlow()) //feature enabled
            {
                ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if CoverArtFlow()"));

                tBoolean isDirExist = exists_directory(devicePath);

                /* Check if device path is not existing */
                if(false == isDirExist)
                {
                    ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if !exists_directory(devicePath)"));
                    /* Create directory with read, write, execute permission for owner and read, execute permission for group = 0750 */
                    if(0 != mkdir(devicePath, S_IRWXU | S_IRGRP | S_IXGRP))
                    {
                        ETG_TRACE_ERR(("mkdir: error: %d/%s", errno, strerror(errno)));
                    }
                    else
                    {
                        ETG_TRACE_COMP(("Make directory for cover arts in flash: %s", devicePath));
                        isDirExist = true;
                    }
                }

                if(isDirExist)
                {
                    /* Delete cover art file if it is already existing */
                    struct stat statBuffer;
                    if(0 == stat(coverArtPath, &statBuffer))
                    {
                        /* file existing */
                        ETG_TRACE_COMP(("cover art file already existing -> remove it. Size: %d", statBuffer.st_size));

                        if(0 != remove(coverArtPath)) //todo(pee1cob):why need fopen("wb") will delete and create new file.?
                        {
                            ETG_TRACE_ERR(("remove: error: %d/%s", errno, strerror(errno)));
                        }
                        else
                        {
                            /* Decrease used memory on flash */
                            m_CoverArtMemoryOnFlashKB -= (tMemorySize)statBuffer.st_size / 1024;
                        }
                    }

                    /*todo(pee1cob):If remove() failed,below fopen(..,"wb") will delete the current file.
                     * Hence take care to reduce:m_CoverArtMemoryOnFlashKB
                     */
                    /* Store cover art to flash */
                    FILE *fpImage = fopen(coverArtPath, "wb");
                    if (!fpImage)
                    {
                        ETG_TRACE_ERR(("fopen: error: %d/%s", errno, strerror(errno)));
                        storeCovertArtInDB = false;
                    }
                    else
                    {
                        ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if fpImage m_CoverArtObjectPtr->imageSize=%d",m_CoverArtObjectPtr->imageSize));

                        tSize writtenElementsCount = fwrite(m_CoverArtObjectPtr->imageData, 1, m_CoverArtObjectPtr->imageSize, fpImage);
                        if(!writtenElementsCount)
                        {
                            ETG_TRACE_ERR(("fwrite error: No data written to CovertArtFile"));
                            storeCovertArtInDB = false;
                            //todo remove(coverArtPath)
                            //todo(pee1cob):trace ferror
                        }
                        else
                        {
                            ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() fpImage File Write Complete written Size=%u",writtenElementsCount));

                            /* Increase used memory on flash */
                            m_CoverArtMemoryOnFlashKB += (tMemorySize)writtenElementsCount / 1024;

                            //todo(pee1cob):trace if writtenElementsCount and m_CoverArtObjectPtr->imageSize differ
                        }

                        if(fclose(fpImage))
                        {
                            ETG_TRACE_ERR(("fclose of CovertArtFile failed"));
                        }
                    }
                }
                else
                {
                    storeCovertArtInDB = false;
                }
            }
            else
            {
                storeCovertArtInDB = false;
            }

            if(m_CoverArtObjectPtr->imageData) free(m_CoverArtObjectPtr->imageData);
            delete m_CoverArtObjectPtr;
            m_CoverArtObjectPtr = NULL;
        }
        else
        {
            storeCovertArtInDB = false;
            ETG_TRACE_ERR(("m_CoverArtObjectPtr is NULL"));
        }


        /* Store thumbnail and cover art path to AlbumArts table of DB */
        tImageBlob thumbnailBlob;
        if(m_ThumbnailObjectPtr && LocalSPM::GetDataProvider().AlbumArtThumbnail())
        {
            ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() if AlbumArtThumbnail()"));
            thumbnailBlob.imageSize = m_ThumbnailObjectPtr->imageSize;
            thumbnailBlob.imageData = m_ThumbnailObjectPtr->imageData;
        }
        else
        {
            ETG_TRACE_USR1(("AlbumArtIndexer::WriteDBAndFlash() else AlbumArtThumbnail()"));
            thumbnailBlob.imageSize = 0;
            thumbnailBlob.imageData = NULL;
        }

        if(storeCovertArtInDB != true)
        {
            ETG_TRACE_ERR(("CoverArt not stored to Flash .Storing Empty Url in DB"));
            coverArtPath[0] = '\0';

            //TODO(IMPORTANT):Instead of storing empty url, storing coverArt url should be skipped.Reason:Retry can be done upon next indexing cycle.If store empty,it is treated as indexing comleted but no coverArt found
            //To achieve this,DBManager should offer interfaces 1.StoreThumbnail 2.StoreAlbumArt.This currently not available.(NCG3D-134745)
        }

        ret = LocalSPM::GetDBManager().StoreAlbumArt(IN m_MediaObject.deviceID, IN m_MediaObject.MetadataTag4, IN m_MediaObject.title, IN coverArtPath, IN thumbnailBlob,IN m_MediaObject.dateTime);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while storing album art object in DB (ErrorCode:%s)", errorString(ret)));
        }

        if(m_ThumbnailObjectPtr)
        {
            if(m_ThumbnailObjectPtr->imageData) free(m_ThumbnailObjectPtr->imageData);
            delete m_ThumbnailObjectPtr;
            m_ThumbnailObjectPtr = NULL;
        }
    }
    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::SendGoOn()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Send GO_ON message to own SM via dispatcher to create external event */
    char msgToSendString[32];
    strncpy_r(OUT msgToSendString, IN "AlbumArtIndexerSM::GO_ON", IN sizeof(msgToSendString));

    ret = Dispatcher::GetInstance().SendMessage(IN msgToSendString, (char *)NULL);
    if( MP_NO_ERROR != ret )
    {
        ETG_TRACE_ERR(("Error while sending external event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::FreeCoverArtSpace()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Get vector of old devices*/
    tNumberOfDevices numberOfDevices;
    vector<tDeviceID> oldDeviceIDs;
    ret = LocalSPM::GetDBManager().GetOldDevices(OUT numberOfDevices, OUT oldDeviceIDs);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while getting oldest device from DB (ErrorCode:%s)", errorString(ret)));
    }

    /* Check if another device to free is available in DB */
    tPath devicePath; //e.g. "/var/opt/bosch/dynamic/media/coverart/dev03"
    devicePath[0] = '\0';
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    for(tUInt i=0; i<oldDeviceIDs.size(); i++) //Loop over all old devices
    {
        snprintf(devicePath, sizeof(devicePath)-1, "%sdev%02d", m_CoverArtDirPath, oldDeviceIDs[i]);

        /* Check if device path is existing */
        if( exists_directory(devicePath) )
        {
            if(m_IndexingContext.deviceID != oldDeviceIDs[i]) //not the current indexing device
            {
                deviceID = oldDeviceIDs[i];
                break;
            }
        }
    }

    if((DEVICE_ID_NOT_SET != deviceID) //found a device to free
            &&
            (23 < strlen_r(devicePath))) //do not delete directory recursively with a path length below "/var/opt/bosch/dynamic/"
    {
        /* Free space on flash by remove directory */
        if(0 != remove_directory(devicePath))
        {
            ETG_TRACE_ERR(("remove_directory: error: %d/%s", errno, strerror(errno)));
        }
        else
        {
            ETG_TRACE_COMP(("Free (delete) directory of cover arts in flash: %s", devicePath));
        }

        /* Delete all album art entries for the device from the AlbumArts table */
        ret = LocalSPM::GetDBManager().RemoveAlbumArts(IN deviceID);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while freeing space in DB at DBManager (ErrorCode:%s)", errorString(ret)));
        }

        /* Set album art indexing complete flag to false (NCG3D-134745)*/
        tAlbumArtIndexingComplete completeFlag = false;
        ret = LocalSPM::GetDBManager().SetAlbumArtIndexingComplete(IN deviceID, IN completeFlag);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while setting album art indexing complete flag at DBManager (ErrorCode:%s)", errorString(ret)));
        }

        /* Get memory size of cover art directory on flash */
        GetCoverArtMemoryOnFlash();
        VARTRACE(m_CoverArtMemoryOnFlashKB);

        /* Send WRITE_DB_FLASH message to own SM */
        ret = SendEvent(WRITE_DB_FLASH, (char *)NULL);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Send FINISHED message to own SM */
        ret = SendEvent(FINISHED, (char *)NULL);
        if( MP_NO_ERROR != ret )
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

#if 0
tResult AlbumArtIndexer::IndexingStopped()
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    /* Clear release event and disarm timer of own state machine in case of still waiting */
    ret = ClearReleaseEvent();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while clearing release event via SMF (ErrorCode:%s)", errorString(ret)));
    }

    /* Flush the media object cache */
    ret = LocalSPM::GetDBManager().StoreMediaObjectEnd();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while flushing media object cache in DB (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}
#endif

tResult AlbumArtIndexer::Finished()
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

    if(m_IndexingContext.interimList) //Interim lists due to focus change
    {
        ret = LocalSPM::GetListControl().UpdateListSlice(IN m_IndexingContext.listID, IN m_IndexingContext.startIndex, m_IndexingContext.sliceSize);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while updating list slice at ListControl (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        /* Set album art indexing complete flag to true */
        tAlbumArtIndexingComplete completeFlag = TRUE;
        ret = LocalSPM::GetDBManager().SetAlbumArtIndexingComplete(IN m_IndexingContext.deviceID, IN completeFlag);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while setting album art indexing complete flag at DBManager (ErrorCode:%s)", errorString(ret)));
        }
    }
    /*Reset the IndexingContext as indexing is finished - Bug 326906*/
    InitAlbumArtIndexingContext(INOUT m_IndexingContext);
    InitMediaObject(OUT m_MediaObject);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::HandleAnswerTimeout()
{
    ENTRY
    ETG_TRACE_ERR(("AlbumArtIndexer::HandleAnswerTimeout"));

    tResult ret = MP_NO_ERROR;

    /* Send releasing event to own waiting state machine */
    ret = ReleaseWaiting();
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while releasing waiting via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return ret;
}

tResult AlbumArtIndexer::AutoRegisterOnDBTrigger(const tTriggerState autoRegister)
{
    ENTRY
    ETG_TRACE_USR3(("AlbumArtIndexer::AutoRegisterOnDBTrigger autoRegister: %u", autoRegister));

    m_AutoRegisterOnDBTrigger = autoRegister;

    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::SwitchDBTrigger(const tTriggerType trigger, const tTriggerState active)
{
    ENTRY
    ETG_TRACE_USR3(("AlbumArtIndexer::SwitchDBTrigger trigger:%u, active: %u", trigger, active));

    tResult ret = MP_NO_ERROR;

    if(TS_ON == active)
    {
        /* Register on DB trigger */
        if((TT_DB_DEVICE_IDS_COMPLETE == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevIDSComplete)
            {
                //IndexingState 4=IDS_COMPLETE_FULL_DB, 5=IDS_COMPLETE
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevIDSComplete,
                        IN AlbumArtIndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_IDS_COMPLETE",
                        IN "Devices",
                        IN "IndexingState",
                        IN "WHEN NEW.ID>0 AND OLD.IndexingState<>4 AND OLD.IndexingState<>5 AND (NEW.IndexingState=4 OR NEW.IndexingState=5)",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
        if((TT_DB_DEVICE_REMOVED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(0 == m_TriggerID_DevRemoved)
            {
                //Roadmap 13035_Overtemperature: ConnectionState 2=CS_DISCONNECTED, 6=CS_OVERTEMP, 7=CS_ON_HOLD
                ret = LocalSPM::GetDBManager().OnUpdateTrigger(OUT m_TriggerID_DevRemoved,
                        IN AlbumArtIndexerSM::GetSMNameFull(),
                        IN "DB_DEVICE_REMOVED",
                        IN "Devices",
                        IN "ConnectionState",
                        IN "WHEN NEW.ID>0 AND OLD.ConnectionState<>2 AND OLD.ConnectionState<>6 AND OLD.ConnectionState<>7 AND (NEW.ConnectionState=2 OR NEW.ConnectionState=6 OR NEW.ConnectionState=7)",
                        IN "COUNT()",
                        IN "SELECT ID FROM Devices WHERE ROWID=OLD.ROWID");
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while registering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
            }
        }
    }
    else
    {
        /* Deregister on DB trigger */
        if((TT_DB_DEVICE_IDS_COMPLETE == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevIDSComplete)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevIDSComplete);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevIDSComplete = 0;
            }
        }
        if((TT_DB_DEVICE_REMOVED == trigger)
                ||
                (TT_ALL == trigger))
        {
            if(m_TriggerID_DevRemoved)
            {
                ret = LocalSPM::GetDBManager().UnregisterTrigger(IN m_TriggerID_DevRemoved);
                if( MP_NO_ERROR != ret )
                {
                    ETG_TRACE_ERR(("Error while unregistering DB trigger (ErrorCode:%s)", errorString(ret)));
                }
                m_TriggerID_DevRemoved = 0;
            }
        }
    }

    return ret;
}

tResult AlbumArtIndexer::GetEndingByMimeType(tExtension &extension, const tMimeType mimeType)
{
    ENTRY_INTERNAL;

    switch (mimeType) {
        case MMT_BMP:
            strncpy_r(OUT extension, IN ".bmp", IN sizeof(extension));
            break;
        case MMT_GIF:
            strncpy_r(OUT extension, IN ".gif", IN sizeof(extension));
            break;
        case MMT_JPG:
            strncpy_r(OUT extension, IN ".jpg", IN sizeof(extension));
            break;
        case MMT_PNG:
        default:
            strncpy_r(OUT extension, IN ".png", IN sizeof(extension));
            break;
    }

    return MP_NO_ERROR;
}

tResult AlbumArtIndexer::GetCoverArtMemoryOnFlash()
{
    ENTRY_INTERNAL;
    m_CoverArtMemoryOnFlashKB = 0;

    /* Create temp file name*/
    tFilename tmpFileName = {0};
    if(0 > GenerateTempFileName(OUT tmpFileName, IN "CoverArtMem"))
    {
        ETG_TRACE_ERR(("Cannot generate temp file name: %s", tmpFileName));
    }
    else
    {
        /* Make "du -s" on cover art directory path to estimate file space usage and pipe the result into a temp file */
        char *commandStr[4] = {(char *)"/usr/bin/du",(char *)("-s"),m_CoverArtDirPath,NULL};
        ETG_TRACE_USR4(("AlbumArtIndexer::GetCoverArtMemoryOnFlash: calling executeCommand() for du and CoverArtDirPath:%s",m_CoverArtDirPath));
        int retStatus = executeCommand(commandStr,string(tmpFileName));

        if(0 == retStatus)
        {
            ETG_TRACE_USR3(("AlbumArtIndexer::GetCoverArtMemoryOnFlash: executeCommand() success for du"));
            /* Read temp file to get file space usage */
            FILE *fp;
            fp = fopen(tmpFileName, "r");
            if (!fp)
            {
                ETG_TRACE_ERR(("fopen: error: %d/%s", errno, strerror(errno)));
            }
            else
            {
                char buffer[32];
                if(NULL != fgets(buffer, sizeof(buffer), fp))
                {
                    m_CoverArtMemoryOnFlashKB = atoi(buffer);
                }
                fclose(fp);
            }
        }
        else
        {
            ETG_TRACE_ERR(("AlbumArtIndexer::GetCoverArtMemoryOnFlash: executeCommand() failed for du. Error:%s",strerror(retStatus)));
        }

        if(0 != remove(tmpFileName))
        {
            ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName));
        }
    }

    VARTRACE(m_CoverArtMemoryOnFlashKB);
    return MP_NO_ERROR;
}

