/**
 * @defgroup Indexer Indexer
 * @ingroup Mediaplayer
 * @authors Holger Schwark, Thomas Porsch
 *
 * Public interface for indexer to control indexing from different device types
 * @{
 */

#ifndef _INDEXER_H_
#define _INDEXER_H_

#include "IndexerSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include <vector>
#include <map>
#include "Timer.h"

class Indexer : public IndexerSM, public ILocalSPM , public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the Indexer componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    Indexer(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~Indexer();

    /**
     * This function is used by LocalSPM to create the Indexer.
     * @attention: running in SPM thread context
     * Create the Indexer state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the Indexer initialization.
     * @attention: running in SPM thread context
     * Init the Indexer state machine.
     * Register IndexerSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the Indexer.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the Indexer thread and the state machine.
     * Register on DB trigger
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

        /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the Indexer.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the Indexer.
     * @attention: running in SPM thread context
     * Deregister IndexerSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

//Indexing part

    /**
     * Function starts initialization of device
     * In case of DB trigger DB_DEVICE_INSERTED this function is called.
     *
     * Get device type and mount point from DBManager -> GetDeviceInfo(&deviceInfo, deviceID).
     * Request support for device type from DataProvider -> IsDeviceTypeSupported(&returnValue, deviceInfo.deviceType).
     * Request support for file system from DataProvider -> IsFileSystemSupported(&returnValue, deviceInfo.fileSystemType).
     * Request support for device type from DataProvider -> IsPartitionSupported(&returnValue, deviceInfo.partitionNumber).
     * If one of the returnValues is false //Roadmap 13023
     * - Call DataProvider::DeviceInitialized(deviceName, deviceID, connectionState=CS_UNSUPPORTED,
     *   deviceState=DS_UNSUPPORTED or DS_UNSUPPORTED_FILESYSTEM or DS_UNSUPPORTED_PARTITION)
     * else
     * - RouteMessage(INIT_DEVICE_CONNECTION) to specific DeviceControl (deviceType)
     *     -> deviceType "INIT_DEVICE_CONNECTION deviceType.mountPoint,deviceID".
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult DBInitializeDevice(const tDeviceCount deviceCount, const tDeviceID deviceID); //Roadmap 13023: 100%

    /**
     * Function transfers device to DB after initialization
     * Transition: indexer2indexerInternalDeviceInitialized(DEVICE_INITIALIZED)
     *
     * If (connectionState == CS_CONNECTED)
     * - Get indexing state from GetDBManager::GetIndexingState(&indexingState, deviceID).
     * - If (indexingState == IDS_COMPLETE)
     *   - Update indexing state for device via DBManager::SetIndexingState(deviceID, IDS_PARTIAL, 0).
     * Write device into DB via DataProvider::DeviceInitialized(deviceID, deviceName, connectionState, deviceState). //Roadmap 13023
     *
     * @param[in] deviceName user friendly name of the device
     * @param[in] deviceID device ID
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult DeviceInitialized(const tDeviceName deviceName, const tDeviceID deviceID, const tConnectionState connectionState); //Roadmap 13023: 100%

    /**
     * Function starts decision process to find next device to index.
     * In case of DB_DEVICE_CONNECTED event this function is called.
     * Former name: IsDBDifferentDevice
     *
     * If new device (deviceID) is different to the current device to index (m_DeviceID)
     * reinsert current m_DeviceID at the beginning of the index queue.
     * Add deviceID to index queue
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult DBDeviceConnected(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Internal function called on DEVICE_CONNECTED
     *
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult DeviceConnected(const tDeviceID deviceID);

    /**
     * Function delivers answer to guard query.
     * In case of DB trigger DB_DEVICE_REMOVED this function is called.
     *
     * If removed device (deviceID) is the current device to index (m_DeviceID) return true, otherwise false.
     * In case of false remove deviceID from index queue.
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return true or false
     */
    tResult DBDeviceRemoved(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Internal function called on DEVICE_REMOVED
     *
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tReturnValue IsSameDevice(const tDeviceID deviceID);

    /**
     * Function finds next device to index and starts indexing
     * Entry Function of state: notStarted
     *
     * Create empty indexing context.
     * Find the next device to index and set it in indexing context (m_DeviceID).
     * SendMessage(START_INDEX).
     *
     * @return != 0: error, = 0: OK
     */
    tResult DeviceToIndex();

    /**
     * Function delivers answer to condition.
     * In case of START_INDEX event this function is called.
     *
     * Remove deviceID from index queue.
     * Get device type from DBManager -> GetDeviceInfo(&deviceInfo, m_DeviceID)
     * Request indexing support for device type from DataProvider -> IsIndexingSupported(&returnValue, deviceInfo.deviceType).
     * If indexing for next device is supported return true, otherwise false.
     *
     * @return true or false
     */
    tReturnValue IsIndexingSupported();

    /**
     * Function updates DB that device is not supported
     * Transition: checking2notsupported(NOT_SUPPORTED)
     *
     * Update indexing state for device via DBManager::SetIndexingState(m_DeviceID, IDS_NOT_SUPPORTED, 0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult UpdateDeviceNotSupported(const tDeviceID deviceID, const tNumberOfFiles numberOfFiles);
    tResult GetNumberOfFiles();

    /**
     * Function starts checking of fingerpoint
     * Entry Function of state: checking
     *
     * Get device type and mount point from DBManager -> GetDeviceInfo(&deviceInfo, deviceID).
     * Store deviceType to indexing context -> m_IndexingContext.deviceType = deviceInfo.deviceType;
     * Store mountPoint to indexing context -> m_IndexingContext.mountPoint = deviceInfo.mountPoint;
     * RouteMessageAnswer(GET_FINGERPRINT) to specific DeviceControl (m_DeviceID)
     *   -> deviceType "GET_FINGERPRINT mountPoint","INDEXER FINGERPRINT_AVAIL".
     * GET_FINGERPRINT also counts the number of media files.
     *
     * Roadmap 13006 IAP2: Send old fingerprint to device control
     *
     * @return != 0: error, = 0: OK
     */
    tResult Check(); //Roadmap 13006 IAP2

    /**
     * Function checks answer of fingerprint calculation
     * Transition: waitingForAnswer2waitingForAnswerFingerprintAvail(FINGERPRINT_AVAIL)
     *
     * Store numberOfFiles to indexing context -> m_IndexingContext.totalNumberOfFiles = numberOfFiles;
     * If(fingerprintStatus == NOT_AVAIL) -> m_IndexingContext.indexingMode = IM_FULL_SCAN + SendMessage(INDEX)
     * If(fingerprintStatus == ERROR) -> SendMessage(NOT_SUPPORTED)
     * If(fingerprintStatus == OK)
     * - m_IndexingContext.fingerprint = fingerprint;
     * - Get stored fingerprint for device from DBManager::GetFingerprint(&storedFingerprint, deviceID)
     * - Get last indexing state for device from GetDBManager::GetIndexingState(&indexingState, deviceID)
     * - If(fingerprint == storedFingerprint) AND (indexingState == COMPLETE) -> SendMessage(FINISHED)
     * - Else
     *   - If(fingerprint == storedFingerprint) -> m_IndexingContext.indexingMode = IM_CONTINUE_SCAN;
     *   - Else
     *     - m_IndexingContext.indexingMode = IM_FULL_SCAN;
     *     - Reset album art indexing complete flag to false //Roadmap 15009
     *   - Set first estimation of numberOfFiles for device via DBManager::SetNumberOfAudioFiles(m_DeviceID, numberOfFiles)
     *   - SendMessage(INDEX)
     *
     * Roadmap 13006 IAP2:     When receiving FPS_OK_DELTA only the recent content changes will be transfered from DeviceControl to Indexer
     *                         Instead of calling DBManager::UpdateMain() at the end of indexing the Indexer calls DBManager::UpdateMyMediaDatabase()
     *                         Here the current db content of the indexing device is initially copied to MYMEDIA
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR]
     * @param[in] numberOfFiles number of files to index
     * @param[in] deviceID the device id
     * @return != 0: error, = 0: OK
     */
    tResult CheckResult(const tFingerprint fingerprint, const tFingerprintStatus fingerprintStatus, const tNumberOfFiles numberOfFiles, const tDeviceID deviceID); //Roadmap 13006 IAP2, 15009: 100%

    /**
     * Function processes updates for fingerprint and number of files
     * //Roadmap 13006 IAP2
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR]
     * @param[in] numberOfFiles number of files to index
     * @param[in] deviceID the device id
     * @return != 0: error, = 0: OK
     */
    tResult OnFingerprintUpdate(const tFingerprint fingerprint, const tFingerprintStatus fingerprintStatus, const tNumberOfFiles numberOfFiles, const tDeviceID deviceID); //Roadmap 13006 IAP2

    /**
     * Function is called at the end of checking fingerprint
     * Exit Function of state: checking
     *
     * Clear release event and disarm timer of own state machine in case of still waiting.
     *
     * @return != 0: error, = 0: OK
     */
    tResult CheckingStopped();

    /**
     * Function writes new indexing state CURRENTLY_NOT_SUPPORTED of device to DB
     * Transition: waitingFor_checking2notsupported(NOT_SUPPORTED)
     *
     * Update indexing state for device via DBManager::SetIndexingState(m_DeviceID, IDS_CURRENTLY_NOT_SUPPORTED, 0).
     *
     * @return != 0: error, = 0: OK
     */
    tResult UpdateDeviceCurrentlyNotSupported();

    /**
     * Function completes indexing and updates the Main DB
     * Transition: waitingFor_checking2notsupported(FINISHED)
     *
     * Internal call Finished()
     * Update MyMedia DB via DBManager::UpdateMyMediaDatabase()
     *
     * @return != 0: error, = 0: OK
     */
    tResult FinishedNoReindex();

    /**
     * Function prepares the indexing
     * Entry Function of state: prepareIndexing
     *
     * Update indexing state for device via DBManager::SetIndexingState(m_DeviceID, IDS_PARTIAL, 0).
     * Reset read position in indexing context -> m_IndexingContext.readPosition = 0 (from mount point).
     * Get current number of media objects in DB from DBManager::GetNumberOfMediaObjectsInDB(&m_NumberObjectsInDB).
     * Calculation of all limits.
     * Inform ListControl that indexing has started -> ListControl::IndexingStarted().
     * Create UpdateIndexingProgressTimer.
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitPrepare();

    /**
     * Function requests meta data from device
     * Entry Function of state: indexing
     *
     * RouteMessageAnswer(GET_METADATA) to specific DeviceControl (deviceType)
     *   -> deviceType "GET_METADATA m_IndexingContext.mountPoint,m_IndexingContext.readPosition","INDEXER METADATA_ANSWER".
     *
     * @return != 0: error, = 0: OK
     */
    tResult GetMetadata();

    /**
     * Function checks received media object and defines the next step
     * Transition: waitingForResult2waitingForResultInternalMetadataAnswer(METADATA_ANSWER)
     *
     * Increment indexed file count -> m_IndexingContext.indexedFileCount++.
     * If(metadataStatus == SUCCESS) -> Store media object temporary and SendMessage(WRITE).
     * If(metadataStatus == FINISHED)
     * - If(totalNumberOfFiles <= indexedFileCount) -> SendMessage(FINISHED).
     * - Else
     *   - Set read position to beginning -> m_IndexingContext.readPosition = 0 (from mount point).
     *   - SendMessage(RESTART).
     * If(metadataStatus == FILE_ERROR) -> SendMessage(SKIP).
     * If(metadataStatus == DEVICE_ERROR) -> SendMessage(ERROR).
     *
     * Roadmap 13006 IAP2:     If metadataStatus has value MDS_REMOVED a partial (or delta) indexing is in progress.
     *                      The removed status has to be stored as member, see also usage in Putmetadata()
     *
     * @param[in] mediaObjectPtr pointer to a media object Attention: memory has to be freed after use
     * @param[in] metadataStatus indexing status from DeviceControl [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @return != 0: error, = 0: OK
     */
    tResult CheckMetadata(const tMediaObjectPtr mediaObjectPtr, const tMetadataStatus metadataStatus); //Roadmap 13006 IAP2
    tResult MetadataAnswerNotConsumed(const tMediaObjectPtr mediaObjectPtr, const tMetadataStatus metadataStatus);

    /**
     * Function delivers answer to condition.
     *
     * In case of limit of free DB memory is not reached yet return true, otherwise false.
     * If((m_LimitNumberObjectsInDB < m_NumberObjectsInDB + m_IndexingContext.writtenFileCount)
     *    ||
     *    (m_LimitNumberObjectsPerDevice < m_IndexingContext.writtenFileCount))
     * - Limit is reached
     *
     * @return true or false
     */
    tReturnValue IsLimitNotReached();

    /**
     * Function stores media object in DB
     * Transition: trueCheckSpaceIsLimitReached2indexing
     *
     * Increment written file count -> m_IndexingContext.writtenFileCount++.
     * Store media object in DB via DBManager::StoreMediaObject(mediaObject)
     * Set read position to URL of media in indexing context -> m_IndexingContext.readPosition = mediaObject.fileName;
     *
     * Roadmap 13006 IAP2:     In case of MDS_REMOVED the mediaObject has to be removed from db by calling DBManager::DeleteMediaObjectByUUID()
     *                         After indexing a call of DBManager::UpdateMain is NOT necessary in case of DELTA indexing.
     *
     * @return != 0: error, = 0: OK
     */
    tResult PutMetadata();    //Roadmap 13006 IAP2

    /**
     * Function removed media object from DB
     *
     * Roadmap 13006 IAP2:     In case of MDS_REMOVED the mediaObject has to be removed from db by calling DBManager::DeleteMediaObjectByUUID()
     *                         After indexing a call of DBManager::UpdateMain is NOT necessary in case of DELTA indexing.
     *
     * @return != 0: error, = 0: OK
     */
    tResult RemoveMetadata();    //Roadmap 13006 IAP2
    tResult FlushMediaObjects();  //Roadmap 13006 IAP2

    /**
     * Function sends GO_ON event to state machine.
     * Transition: checkInterrupt2waitingFor_indexingGoOn
     *
     * SendMessage(GO_ON).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendGoOn();

    /**
     * Function sends DB_DEVICE_CONNECTED event to state machine.
     * Roadmap 13005
     *
     * SendMessage(DB_DEVICE_CONNECTED, deviceCount, deviceID).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendReindexing(const tDeviceCount deviceCount, const tDeviceID deviceID); //Roadmap 13005: 100%
    tResult Reindexing(const tDeviceCount deviceCount, const tDeviceID deviceID); //Roadmap 13005: 100%
    tResult ReindexingInitialized(const tDeviceName deviceName, const tDeviceID deviceID, const tConnectionState connectionState);

    /**
     * Function frees DB memory
     * Entry Function of state: checkSpace
     *
     * This function is addicted to the function IsLimiNotReached().
     * Variant controlled freeing of DB memory.
     * - Limit of free DB memory
     * - Other (customer) limits --> setting parameter
     * If no other device to free is available in DB
     * - SendMessage(Error).
     * Else
     * - Request database to free space for n objects via DBManager::FreeSpace().
     * - SendMessage(WRITE).
     *
     * @return != 0: error, = 0: OK
     */
    tResult FreeSpace();

    /**
     * Function increments file count
     * Transition: waitingFor_indexing2waitingFor_indexing(SKIP)
     *
     * Increment indexed file count -> m_IndexingContext.indexedFileCount++.
     * Increment skipped file count -> m_IndexingContext.skippedFileCount++.
     *
     * @return != 0: error, = 0: OK
     */
    tResult IncrementFileCounter();

    /**
     * Function is called at the end of indexing
     * Exit Function of state: prepareIndexing
     *
     * Clear release event and disarm timer of own state machine in case of still waiting.
     * Delete UpdateIndexingProgressTimer.
     * Inform ListControl that indexing has stopped -> ListControl::IndexingStopped().
     * Flush the media object cache -> this implies last commit and start of create index in DB
     * Calculate indexing percentage value: percent = indexedFileCount/totalNumberOfFiles*100.
     * Update indexing progress via DBManager::SetIndexingState(m_DeviceID, IDS_PARTIAL, percent).
     *
     * @return != 0: error, = 0: OK
     */
    tResult IndexingStopped();

    /**
     * Function completes indexing and updates the MyMedia DB
     * Transition: waitingFor_indexing2notStarted(FINISHED)
     *
     * Internal call Finished()
     * Update Main DB via DBManager::UpdateMain(m_DeviceID)
     *
     * @return != 0: error, = 0: OK
     */
    tResult FinishedComplete();

    /**
     * Function resets fingerprint and call FinishedComplete
     * Transition: waitingFor_indexing2notStarted(ERROR)
     *
     * @return != 0: error, = 0: OK
     */
    tResult ErrorComplete();

    /**
     * Function calculates the indexing percentage and update DB/HMI
     *
     * Calculate indexing percentage value: percent = indexedFileCount/totalNumberOfFiles*100.
     * If the change is significant (every 10%):
     * - Store new percentage value: m_IndexingContext.percent = percent.
     * - Update indexing progress via DBManager::SetIndexingState(m_DeviceID, IDS_PARTIAL, percent)
     *
     * @return != 0: error, = 0: OK
     */
    tResult UpdateIndexingProgressTimer();

    /**
     * Function triggers the fetch of all indexing states of the connected devices
     * Is triggered by DB message DB_DEVICE_INDEXING every time the indexing state or progress changes on a device
     *
     * Call UpdateIndexingState at OutputWrapper -> UpdateIndexingState().
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult DBDeviceIndexing(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sets switch for auto register on DB trigger while Run
     * *
     * @param[in] autoRegister on or off
     * @return != 0: error, = 0: OK
     */
    tResult AutoRegisterOnDBTrigger(const tTriggerState autoRegister);

    /**
     * Function registers or unregisters DB trigger for the indexer
     * *
     * @param[in] trigger DB trigger of an event for indexer
     * @param[in] active on or off -> register or deregister DB trigger
     * @return != 0: error, = 0: OK
     */
    tResult SwitchDBTrigger(const tTriggerType trigger, const tTriggerState active); ////Roadmap 13035_Overtemperature  adjust function to react with REMOVE if CS_OVERTEMP is set in Database

    /**
     * Function checks snyc result from waiting threat
     *
     * @param[in] deviceID the device id
     * @param[in] syncID the sync id
     * @param[in] syncResult the sync result
     * @return != 0: error, = 0: OK
     */
    tResult SyncResult(const tDeviceID deviceID, const tIndexSession syncID, const tResult syncResult);

private:

    /**
     * Function writes new indexing state COMPLETE and fingerprint of device to DB
     *
     * Update indexing state for device via DBManager::SetIndexingState(m_DeviceID, IDS_COMPLETE, 100)
     * Update fingerprint of device via DBManager::SetFingerprint(m_DeviceID, m_IndexingContext.fingerprint, IN m_IndexingContext[m_Index].fingerprint, IN m_IndexingContext[m_Index].writtenFileCount - m_IndexingContext[m_Index].writtenFileCountForVideo - m_IndexingContext[m_Index].writtenFileCountForImage,m_IndexingContext[m_Index].writtenFileCountForVideo)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Finished();

    /**
     * Progress timer callback
     * *
     * @param[in] timerID timer ID
     * @param[in] instance pointer to own (Indexer) instance
     * @param[in] userData pointer to possible user data
     * @return true or false
     */
    static bool ProgressTimerCallBack(timer_t timerID , void* instance ,const void *userData);

    tResult FileCountCompWithDB(const tNumberOfFiles numberOfPlaylist,const tNumberOfFiles numberOfVideo,const tNumberOfFiles numberOfTrack,const tDeviceID deviceID,const tIndexSession indexSession);
    tResult PlaylistNameCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfPlaylist,tVoidPtr playListRecordsPtr,const tIndexSession indexSession);
    tResult VideoUIDCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfVideo,tVoidPtr trackInfosPtr,const tIndexSession indexSession);
    tResult TrackUIDCompWithDB(const tDeviceID deviceID,const tIndex index,const tListSize chunk,const tNumberOfFiles numberOfTrack,tVoidPtr trackInfosPtr,const tIndexSession indexSession);

    void StartFingerprintTimer(const tDeviceID deviceID);
    void StopFingerprintTimer();
    static bool FingerprintTimerCallBack(timer_t timerID , void* instance ,const void *userData);
    void OnFingerprintTimeout(); //timer event

    tResult DBDeviceVersionUpdated(const tDeviceID deviceID);

    tResult DoWaitForUpdateMyMediaFinished(char *parameterStr);
    tResult SendSync();

    tResult ResetAlbumArtIndexingComplete();

    /**
     * Function sets Audio,Video,Image FileCount For the Given device in DB
     *
     * IMPORTANT NOTE:Update is done ONLY When the given count != NUMBER_OF_FILES_NONE
     *
     * @param[in] deviceID the device ID
     * @param[in] numberOfFiles  Total number of Files (i.e audio+video+image)
     * @param[in] numberOfVideoFiles Total number of video Files
     * @param[in] numberOfImageFiles Total number of image Files
     */
    void  SetDeviceFileCounts(const tDeviceID deviceID,
            const tNumberOfFiles numberOfFiles=NUMBER_OF_FILES_NONE,
            const tNumberOfFiles numberOfVideoFiles=NUMBER_OF_FILES_NONE,
            const tNumberOfFiles numberOfImageFiles=NUMBER_OF_FILES_NONE);

    tTriggerState m_AutoRegisterOnDBTrigger;
    tTriggerID m_TriggerID_DevInserted;
    tTriggerID m_TriggerID_DevAttached;
    tTriggerID m_TriggerID_DevConnected;
    tTriggerID m_TriggerID_DevIndexingState;
    tTriggerID m_TriggerID_DevIndexingProgress;
    tTriggerID m_TriggerID_DevRemoved;
    tTriggerID m_TriggerID_DevVersionUpdated;

    vector<tIndexingContext> m_IndexingContext;             /**< indexing context */
    tIndex m_Index;                                         /**< current index in indexing context */
    vector<tDeviceID> m_DevicesToIndex;                     /**< index queue of devices to index */
    map<tDeviceID, tIndexingState> m_StoredIndexingStates;  /**< map of stored indexing states of connected devices */

    tNumberOfMediaObjects m_NumberObjectsInDB;              /**< current number of media objects in database */
    tNumberOfMediaObjects m_LimitNumberObjectsInDB;         /**< limit of number of media objects in database */
    tNumberOfMediaObjects m_LimitNumberObjectsPerDevice;    /**< limit of number of media objects per device */

    tMediaObject m_MediaObject;         /**< last delivered media object */
    Timer m_ProgressTimer;              /**< progress timer */
    timer_t m_ProgressTimerID;          /**< progress timer index */
    unsigned long m_ProgressTimeout;    /**< progress timeout value */

    Timer m_FingerprintTimer;               /**< Fingerprint timer */
    timer_t m_FingerprintTimerID;           /**< Fingerprint timer */
    tDeviceID m_LastFingerprintDeviceID;    /**< Fingerprint timer */
    tIndexSession mSyncID;                  /**< sync index session iD */
};

#endif //_INDEXER_H_

/** @} */
