/**
 * @defgroup AlbumArtIndexer AlbumArtIndexer
 * @ingroup Mediaplayer
 * @authors Thomas Porsch
 * 
 * Public interface for album art indexer to control indexing of album arts from mass storage devices
 * Roadmap 15009_CoverArtFlow
 * @{
 */

#ifndef _ALBUMARTINDEXER_H_
#define _ALBUMARTINDEXER_H_

#include "AlbumArtIndexerSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include <vector>

class AlbumArtIndexer : public AlbumArtIndexerSM, public ILocalSPM , public TFThread
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the AlbumArtIndexer componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    AlbumArtIndexer(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~AlbumArtIndexer();

    /**
     * This function is used by LocalSPM to create the AlbumArtIndexer.
     * @attention: running in SPM thread context
     * Create the AlbumArtIndexer state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0) 
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the AlbumArtIndexer initialization.
     * @attention: running in SPM thread context
     * Init the AlbumArtIndexer state machine.
     * Register AlbumArtIndexerSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0) 
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the AlbumArtIndexer.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the AlbumArtIndexer thread and the state machine.
     * Register on DB trigger
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

        /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the AlbumArtIndexer.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Deregister on DB trigger.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the AlbumArtIndexer.
     * @attention: running in SPM thread context
     * Deregister AlbumArtIndexerSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

//Indexing part

    /**
     * Function starts decision process to find next device to index.
     * In case of DB_DEVICE_IDS_COMPLETE event this function is called.
     *
     * If new device (deviceID) is different to the current device to index (m_DeviceID)
     * reinsert current m_DeviceID at the beginning of the index queue.
     * Add deviceID to index queue.
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return != 0: error, = 0: OK
     */
    tResult DBDeviceIDSComplete(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Function delivers answer to guard query.
     * In case of DB trigger DB_DEVICE_REMOVED this function is called.
     *
     * If removed device (deviceID) is the current device to index (m_DeviceID) return true, otherwise false.
     * In case of false remove deviceID from index queue.
     *
     * @param[in] deviceCount number of currently connected devices
     * @param[in] deviceID device ID of effected device
     * @return true or false
     */
    tReturnValue IsDBSameDevice(const tDeviceCount deviceCount, const tDeviceID deviceID);

    /**
     * Function finds next device to index and starts indexing
     *
     * Create empty indexing context.
     * Find the next device to index and set it in indexing context (m_DeviceID).
     * SendMessage(START_ALBUMART_INDEX).
     *
     * @return != 0: error, = 0: OK
     */
    tResult DeviceToIndex();

    /**
     * Function delivers answer to condition.
     * In case of START_ALBUMART_INDEX event this function is called.
     *
     * Remove deviceID from index queue.
     * Get device type from DBManager -> GetDeviceInfo(&deviceInfo, m_DeviceID)
     * If feature CoverArtFlow or AlbumArtThumbnail is supported and device type supported
     * and at least one file on device and album art indexing is not already completed
     * for the requested device return true, otherwise false.
     *
     * @return true or false
     */
    tReturnValue IsAlbumArtIndexingSupported();

    /**
     * Function sets the focus (indexing context) to a LTY_ALBUM_UNKNOWN_ALBUMART list
     *
     * Create a new indexed list of all albums without album art entry
     * and set the members of the indexing context to the created list
     *
     * @return != 0: error, = 0: OK
     */
    tResult SetFocus();

    /**
     * Function sends CHANGE_FOCUS event to state machine.
     * Call SendMessage(CHANGE_FOCUS, deviceID, listID, startIndex, sliceSize).
     *
     * @param[in] deviceID related device ID
     * @param[in] listID list ID of the new list to change the focus to
     * @param[in] startIndex begin of the new list range to focus
     * @param[in] sliceSize size of the new list range to focus
     * @return != 0: error, = 0: OK
     */
    tResult SendChangeFocus(const tDeviceID deviceID, const tListID listID, const tIndex startIndex, const tIndex sliceSize);

    /**
     * Function sets the focus (indexing context) to delivered list from outside
     *
     * Set the members of the indexing context to the new list
     *
     * @param[in] deviceID related device ID
     * @param[in] listID list ID of the new list to change the focus to
     * @param[in] startIndex begin of the new list range to focus
     * @param[in] sliceSize size of the new list range to focus
     * @return != 0: error, = 0: OK
     */
    tResult ChangeFocus(const tDeviceID deviceID, const tListID listID, const tIndex startIndex, const tIndex sliceSize);

    /**
     * Function prepares the album art indexing
     *
     * Read configuration from DataProvider
     * Check if cover art directory is existing
     * If yes
     * - Remove directory if device path is existing
     * - Get memory size of cover art directory on flash
     * If no
     * - Create a new cover art directory
     * Check if album list is valid
     * - Get all objects from list and store it into a vector
     *
     * @return != 0: error, = 0: OK
     */
    tResult PrepareAlbumArtIndexing();

    /**
     * Function checks if media object vector is valid and object index is in range
     *
     * @return != 0: error, = 0: OK
     */
    tResult CheckAlbumArtVector();

    /**
     * Function reads an album art from the device and converts it into the desired resolutions
     *
     * Get current media object from vector.
     * Increment media object vector index.
     * Skip function if already an entry is available in the DB AlbumArts table for a interim list element.
     * Get album art for current object.
     * Convert the album art to cover art size and format and store it in a member.
     * Convert the album art to thumbnail size and format and store it in a member.
     *
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArt();

    /**
     * Function delivers answer to condition.
     *
     * In case of limit of free memory on flash is not reached yet return true, otherwise false.
     *
     * @return true or false
     */
    tReturnValue IsCoverArtLimitNotReached();

    /**
     * Function stores album art object in DB
     *
     * Store (empty) thumbnail and no cover art path to AlbumArts table of DB
     *
     * @return != 0: error, = 0: OK
     */
    tResult WriteDB();

    /**
     * Function stores album art object in DB and a related cover art on the flash
     *
     * Create cover art path.
     * Create directory if device path is not existing.
     * Delete cover art file if it is already existing.
     * Store cover art to flash and increase used memory on flash counter
     * Store thumbnail and cover art path to AlbumArts table of DB.
     *
     * @return != 0: error, = 0: OK
     */
    tResult WriteDBAndFlash();

    /**
     * Function sends GO_ON event to state machine.
     *
     * SendMessage(GO_ON).
     *
     * @return != 0: error, = 0: OK
     */
    tResult SendGoOn();

    /**
     * Function frees cover art space on the flash
     *
     * This function is addicted to the function IsCoverArtLimitNotReached().
     * Get vector of old devices.
     * Check if another device to free is available in DB and in flash.
     * If found device to free
     * - Free space on flash by remove directory.
     * - Delete all album art entries for the device from the AlbumArts table.
     * - Get memory size of cover art directory on flash.
     *
     * @return != 0: error, = 0: OK
     */
    tResult FreeCoverArtSpace();

#if 0
    /**
     * Function is called at the end of indexing
     * Exit Function of state: albumArtIndexing
     *
     * Clear release event and disarm timer of own state machine in case of still waiting.
     * Flush the media object cache -> this implies last commit and start of create index in DB
     *
     * @return != 0: error, = 0: OK
     */
    tResult IndexingStopped();
#endif

    /**
     * Function sets album art indexing complete flag to true
     *
     * Set album art indexing complete flag to true for no interim list
     *
     * @return != 0: error, = 0: OK
     */
    tResult Finished();

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sets switch for auto register on DB trigger while Run
     * *
     * @param[in] autoRegister on or off
     * @return != 0: error, = 0: OK
     */
    tResult AutoRegisterOnDBTrigger(const tTriggerState autoRegister);

    /**
     * Function registers or unregisters DB trigger for the AlbumArtIndexer
     * *
     * @param[in] trigger DB trigger of an event for AlbumArtIndexer
     * @param[in] active on or off -> register or deregister DB trigger
     * @return != 0: error, = 0: OK
     */
    tResult SwitchDBTrigger(const tTriggerType trigger, const tTriggerState active);

private:

    /**
     * Function delivers the related ending for a mime type.
     *
     * @param[in] mimeType mime type like MMT_PNG
     * @param[out] extension related ending like ".png"
     * @return != 0: error, = 0: OK
     */
    tResult GetEndingByMimeType(tExtension &extension, const tMimeType mimeType);

    /**
     * Function calculates the space usage of the cover arts located in the flash.
     *
     * @return != 0: error, = 0: OK
     */
    tResult GetCoverArtMemoryOnFlash();

    tTriggerState m_AutoRegisterOnDBTrigger;
    tTriggerID m_TriggerID_DevIDSComplete;
    tTriggerID m_TriggerID_DevRemoved;

    tAlbumArtIndexingContext m_IndexingContext;     /**< album art indexing context */
    vector<tDeviceID> m_DevicesToIndex;             /**< index queue of devices to index */
    tAlbumArtIndexingContext m_IndexingVideoContext;     /**< VideoThumbnail  indexing context */

    tSize m_CoverArtWidth;
    tSize m_CoverArtHeight;
    tMimeType m_CoverArtMimeType;
    tPath m_CoverArtDirPath;
    tMemorySize m_LimitCoverArtMemoryOnFlashKB;
    tMemorySize m_CoverArtMemoryOnFlashKB;
    tSize m_ThumbnailWidth;
    tSize m_ThumbnailHeight;
    tMimeType m_ThumbnailMimeType;

    tSize m_VideoThumbnailWidth;
    tSize m_VideoThumbnailHeight;
    tMimeType m_VideoThumbnailtMimeType;
    tPath m_VideoThumbnailDirPath;

    tMediaObject m_MediaObject;                             /**< last delivered media object */
    tAlbumArtObjectPtr m_CoverArtObjectPtr;                 /**< pointer to last delivered cover art object */
    tAlbumArtObjectPtr m_ThumbnailObjectPtr;                /**< pointer to last delivered thumbnail object */
    tAlbumArtObjectPtr m_VideoThumbnailObjectPtr;
};

#endif //_ALBUMARTINDEXER_H_

/** @} */
